<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2024 Teclib' and contributors.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

namespace Glpi\Asset\CustomFieldType;

use Glpi\Asset\CustomFieldOption\OptionInterface;

interface TypeInterface
{
    /**
     * Get the field type name.
     */
    public static function getName(): string;

    public function getLabel(): string;

    /**
     * Get the HTML code to use to display the custom field input in the asset form.
     *
     *
     * @param string $name          The field `name` attribute
     * @param mixed $value          The field current value
     * @param string|null $label    The custom label to use, fallback to the label defined in the field definition
     * @param bool $for_default     Indicates whether the field is used to configure the default value of the field definition
     * @return string
     */
    public function getFormInput(string $name, mixed $value, ?string $label = null, bool $for_default = false): string;

    /**
     * Normalize the value submitted using the input generated by `self::getFormInput()`.
     *
     * @param mixed $value
     * @throws \InvalidArgumentException Thrown if the submitted value does not correspond to a valid value.
     */
    public function normalizeValue(mixed $value): mixed;

    /**
     * Format the value before it is saved in the database.
     * Automatically calls `self::normalizeValue()` before formatting the value.
     *
     * @param mixed $value
     * @return mixed
     */
    public function formatValueForDB(mixed $value): mixed;

    /**
     * Format the value after it is read from the database.
     *
     * @param mixed $value
     */
    public function formatValueFromDB(mixed $value): mixed;

    /**
     * Get the field options.
     *
     * @return OptionInterface[]
     */
    public function getOptions(): array;

    /**
     * Defines configured default value.
     */
    public function setDefaultValue(mixed $value): void;

    /**
     * Get the configured default value.
     */
    public function getDefaultValue(): mixed;

    /**
     * Get the HTML code to use to display the default value input in the configuration form.
     */
    public function getDefaultValueFormInput(): string;

    /**
     * Get the field search option specs to be added to the result of `\Glpi\Asset\Asset::rawSearchOptions()`.
     */
    public function getSearchOption(): ?array;
}
