# Response & Headers

Each PIMF controller instance has one response object. The response object is an abstraction of your PIMF application’s
HTTP response that is returned to the HTTP client. Although each PIMF controller includes a default response object,
the \Pimf\Response class is idempotent; you may instantiate the class at will (in middleware or elsewhere in your
PIMF application) without affecting the application as a whole. You can obtain a reference to the PIMF application’s 
response object with:

Standalone using:

```php
  $response = new \Pimf\Response('GET');
  $response->asJSON()->send(array('hello'=>'Barry'));
```

Within the controller/action using:

```php
  // somewhere at your controller/action
  $this->response->asJSON()->send(array('hello'=>'Barry'));
```

### Sending HTTP Cache-Friendly Headers
To take advantage of caches, PHP applications must be made cache friendly. A cache-friendly application understands how
the caching policies in browsers and proxies work and how cacheable its own data is. The application can then be set to
send appropriate cache-related directives with browsers to achieve the desired results.

**NOTE:** some caveats to remember in using external caches:

- Pages that are requested via the POST method cannot be cached with this form of caching.
- This form of caching does not mean that you will serve a page only once. It just means that you will serve it only once to a particular proxy during the cacheability time period.
- Not all proxy servers are RFC compliant. When in doubt, you should err on the side of caution and render your content uncacheable.

### Tell the browser to cache your page
If instead you have a page that has personalization on it (say, for example, the splash page contains local news as well),
you can set a copy to be cached only by the browser.

```php
  $this->response
       ->asTEXT()
       ->cacheBrowser(60)
       ->send('Barry is cached at the browser');
```

### Never ever cache your page
If you want to try as hard as possible to keep a page from being cached anywhere.

```php
  $this->response
       ->asTEXT()
       ->cacheNone()
       ->send('Barry is NOT cached at the browser');
```

### Tell shared proxies to cache your page
If you want to allow a page to be cached by shared proxies for one minute.

```php
  $this->response
       ->asTEXT()
       ->cacheNoValidate(60)
       ->send('Barry is cached at the browser');
```

### Revalidate for freshness by any cache
Handles setting pages that are always to be revalidated for freshness by any cache.

To utilize this function, you need to know the last modification time for a page. For a static page
(such as an image or a **plain** nondynamic HTML page), this is simply the modification time on the file. For a dynamically
generated page (PHP or otherwise), the last modification time is the last time that any of the data used to generate the
page was changed.

```php
  $last_modified = 1800; //comes from your bossiness logic...
  $this->response
       ->asTEXT()
       ->exitIfNotModifiedSince($last_modified)
       ->send('Barry can be modified');
```
