# Routing

The PIMF Framework helps you map resource URIs to specific controller/action. A PIMF application will invoke the
first route that matches the current HTTP request’s URI and method. If the PIMF application does not find routes
with URIs that match the HTTP request URI and method, it will automatically return a **404 Not Found** response.

PIMF just merges variables from request string into the same parameters array where other variables are stored.
Also there is a default routes pattern that maps default routes like this **/:controller/:action/:id**. Basically you
have two options – use defaults and custom routes or use only custom, which means that all other requests will be ignored.

Custom routes can be defined at your [routes.php](https://github.com/gjerokrsteski/pimf-blog/blob/master/app/MyFirstBlog/routes.php) which
lives at your application directory.

## Routing Configuration
All routing configuration lives at [config.app.php](https://github.com/gjerokrsteski/pimf-blog/blob/master/app/config.app.php) at the sector
**Application meta** which you can set your custom URI options for your application:

```php
  /*
  |----------------------------------------------------
  | Application meta
  |----------------------------------------------------
  */
  'app' => array(
    'name'               => 'MyFirstBlog',
     // application key
    'key'                => 'some5secret5key5here',
    // the name of the fallback controller
    'default_controller' => 'blog',
    // get cleaner URLs or not
    'routeable'          => true,
    // URL used to access your application without a trailing slash.
    'url'                => 'http://localhost',
    // using mod_rewrite for cleaner URLs let it empty,
    // otherwise set index.php
    'index'              => '',
    // the base URL used for your application's asset files
    'asset_url'          => '',
  ),
```

Setting **routeable** to **false** you will got URIs like "http://localhost/?controller=blog&action=show" otherwise
setting **routeable** to **true** you will got cleaner URIs like "http://localhost/blog/show"

## Routing restriction
Following route patterns are PIMF Framework restriction, which gives you enough combinations to describe your application:

- /:controller
- /:controller/:action
- /:controller/:action/:id

If your controller class name is **Blog** and your action name is **searchAction** than PIMF will automatically map **/blog/search** to it!

### Route Parameters
You can embed parameters into route resource URIs. In this example, we have one parameter in our route URI **:slug**.

```php

  // ... if $_SERVER['REQUEST_URI'] => '/blog/search/Barry'

  $route = new \Pimf\Route('/blog/search/:slug');

  $route->getParams() === array('slug' => 'Barry')

```

To create a URL parameter, prepend “:” to the parameter name in the route URI pattern. When the route matches the current HTTP request,
the values for each route parameter are extracted from the HTTP request URI and are passed into the associated controller/action in
order of appearance.

### Wildcard route parameters
You may also use wildcard route parameters. These will capture one or many URI segments that correspond to the route pattern’s
wildcard parameter into an array. A wildcard parameter is identified by a “+” suffix; it otherwise acts the same as normal route
parameters shown above. Here’s an example:

```php
  // ... if $_SERVER['REQUEST_URI'] => '/hello/Barry/White'

  $route  = new \Pimf\Route('/hello/:name+', array('controller'=>'hello'));
  $router = new \Pimf\Router();
  $router->map($route);
  $target = $router->find();

  $target->getParams() === array('name' => array('Barry', 'White'));
```

When you invoke this example application with a resource URI “/hello/Barry/White”, the route **name** argument will be equal
to array('Barry', 'White').


### Route Conditions
PIMF lets you assign conditions to your route parameters. If the specified conditions are not met, the route will not run.
For example, if you need a route with a second segment that must be a valid 4-digit year, you could enforce this condition like this:

```php
  $route  = new \Pimf\Route(
   '/blog/list/:year', array(), array('year' => '(19|20)\d\d')
  );
```

The third parameter at \Pimf\Route is **$conditions** which is an associative array with keys that match any of the route’s parameters and values that are
regular expressions.


### Optional route parameters
These are ideal for using one route for a blog archive listing. To declare optional route parameters, specify your route pattern like this:

```php
  $route  = new \Pimf\Route('/blog/list(/:year(/:month(/:day)))');
```

Each subsequent route segment is optional. This route will accept HTTP requests for:

- /list
- /list/2010
- /list/2010/12
- /list/2010/12/05

Currently, you can only use optional route segments in situations like the example above where each route segment is
subsequently optional. You may find this feature unstable when used in scenarios different from the example above.

### Redirect to route
It is easy to redirect the client to another URL with the PIMF application’s controller **redirect()** method. This method accepts two
arguments. The first argument is the route like **controller/action** to which the client will redirect; By default the **redirect()** method
will send a **302 Temporary Redirect** response.

```php
  // somewhere at your controller
  $this->redirect('users/show');
```

Or if you wish to use a permanent redirect, you must specify the destination route as the first parameter and the the second parameter to **true**.

```php
  // somewhere at your controller
  $this->redirect('users/show', true);
```

This method will automatically set the Location: header. The HTTP redirect response will be sent to the HTTP client immediately.

### Route URL Helper
PIMF delivers you a PHP native helper function **url($route = '', array $params = array(), $https = null, $asset = false)** which you can use it
at your controller or within your .phtml templates to compute your desired URL:

```phtml
  <a href="<?php echo url('blog/showentry', array('id' => $this->id)) ?>">
    <?php echo $this->title ?>
  </a>
```
