(function(window) {
  function LinkPreviewer() {
    const _this = this;

    const componentClass = 'link-previewer';
    const styleClass = `${componentClass}-styles`;

    let stylesContainer = document.head.querySelectorAll(`style.${styleClass}`);
    let activeRenderer;
    // ratio 16:9 400x225, 384x216, 368x207, 352x198
    if (stylesContainer.length === 0) {
      stylesContainer = document.createElement('style');
      stylesContainer.className = styleClass;
      stylesContainer.textContent = `
        .${componentClass}-container {
          position:absolute;
          z-index:10000;
          width:368px;
          height:207px;
          border: 1px solid rgba(0,0,0,.2);
          border-radius: 6px;
          outline: 0;
          overflow: clip;
          box-shadow: 0 3px 7px rgba(0,0,0,.2);
          background: #212527;
          background-clip: padding-box;
          box-sizing: border-box;
        }
        .${componentClass}-image-loading {
          padding-top:32px;
          text-align:center;
        }
        .${componentClass}-image-load-loop {
          display: inline-block;
          width: 4em;
          height: 4em;
          --svg: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24'%3E%3Cg fill='none' stroke='%23000' stroke-linecap='round' stroke-width='2'%3E%3Cpath stroke-dasharray='60' stroke-dashoffset='60' stroke-opacity='0.3' d='M12 3C16.9706 3 21 7.02944 21 12C21 16.9706 16.9706 21 12 21C7.02944 21 3 16.9706 3 12C3 7.02944 7.02944 3 12 3Z'%3E%3Canimate fill='freeze' attributeName='stroke-dashoffset' dur='1.3s' values='60;0'/%3E%3C/path%3E%3Cpath stroke-dasharray='15' stroke-dashoffset='15' d='M12 3C16.9706 3 21 7.02944 21 12'%3E%3Canimate fill='freeze' attributeName='stroke-dashoffset' dur='0.3s' values='15;0'/%3E%3CanimateTransform attributeName='transform' dur='1.5s' repeatCount='indefinite' type='rotate' values='0 12 12;360 12 12'/%3E%3C/path%3E%3C/g%3E%3C/svg%3E");
          background-color: currentColor;
          -webkit-mask-image: var(--svg);
          mask-image: var(--svg);
          -webkit-mask-repeat: no-repeat;
          mask-repeat: no-repeat;
          -webkit-mask-size: 100% 100%;
          mask-size: 100% 100%;
        }
        .${componentClass}-image-content {
          width: 100%;
          padding: 8px;
          box-sizing: border-box;
        }
        .${componentClass}-drawing {
        }
        .${componentClass}-text {
          width:368px;
          min-height:207px;
          overflow:hidden;
          padding:10px;
        }
        .${componentClass}-html {
          width:368px;
          min-height:207px;
          padding:1px;
          overflow:hidden;
          border: none;
          transform: scale(0.995);
          transform-origin: 0 0;
        }
        .${componentClass}-video {
          width:100%;
        }
        .${componentClass}-unknown-format {
          align-items: center;
          display:flex;
          color:white;
          flex-direction: column;
          height: 100%;
          justify-content: center;
          padding:10px;
        }
        .${componentClass}-unknown-format h4 {
          font-size:24px;
          line-height:24px;
          margin-top:20px;
        }
        .${componentClass}-unknown-format h5 {
          font-size:14px;
          line-height:20px;
          margin-top:14px;
        }
        .${componentClass}-show {
          transition: opacity 400ms;
          opacity: 1;
        }
        .${componentClass}-hide {
          transition: opacity 800ms;
          opacity: 0;
        }
      `;
      document.head.append(stylesContainer);
    }

    const previewerOverlayTemplate = `
      <div class="${componentClass}-image-loading" style="display:none;">
      <span class="${componentClass}-image-load-loop"></span>
      </div>
      <img class="${componentClass}-image-content ${componentClass}-content" style="display:none;" />
      <div class="${componentClass}-drawing ${componentClass}-content" style="display:none;">
        <canvas class="${componentClass}-drawing-canvas"></canvas>
      </div>
      <pre class="${componentClass}-text ${componentClass}-content" style="display:none;"></pre>
      <div class="${componentClass}-html ${componentClass}-content" style="display:none;"></div>
      <video class="${componentClass}-video ${componentClass}-content" autoplay muted loop style="display:none;"></video>
      <div class="${componentClass}-unknown-format" style="display:none;">
        <h4>Preview not available</h4>
        <h5>Click to view in a pop-up window</h5>
      </div>
    `;

    let cache = {};

    function Renderer(selector, event, params) {
      const _this = this;

      function showContent(newContent) {
        currentContent = newContent;
        reposition();
        currentContent.style.display = '';
        loadingImage.style.display = 'none';
      }

      function renderImage(url) {
        contentImage.src = url;
      }

      function renderVideo(url) {
        contentVideo.src = url;
      }
      function renderHTML(url){
        //contentHtml.src = url;
        let iframeHTML = document.createElement("iframe");
        iframeHTML.setAttribute("width", "366");
        iframeHTML.setAttribute("height", "205");
        iframeHTML.setAttribute("scrolling", "no");
        iframeHTML.setAttribute('frameborder', "0");
        iframeHTML.style.cssText = 'width:366px;height:205px;border: none;overflow:hidden;';
        iframeHTML.src = url;
        contentHtml.appendChild(iframeHTML);
        showContent(contentHtml);
      }
      function renderText(url) {
        fetch(url, {
          signal,
        }).then(function(response) {
          if (response.ok) {
            response.text().then(function(response) {
              contentText.innerText = response;
              showContent(contentText);
            }).catch(function() {
              findRenderer(url, 'unknown');
            });
          } else {
            findRenderer(url, 'unknown');
          }
        }).catch(function(error) {
          findRenderer(url, 'unknown', error);
        });
        window.setTimeout(function() {
          controller.abort();
        }, 10000);
      }

      function renderPdf(url) {
        window.pdfjsLib.getDocument(url).promise.then(function(pdfObj) {
          if (pdfObj.numPages > 0) {
            const pdfTotalPages = 1;
            for (let pdfCurPage = 1; pdfCurPage <= pdfTotalPages; pdfCurPage++) {
              pdfObj.getPage(pdfCurPage).then(function(page) {
                const desiredWidth = parseFloat(getComputedStyle(previewerOverlay, null).width.replace('px', '')) - 2;
                const viewport = page.getViewport({
                  scale: 1,
                });
                const scale = desiredWidth / viewport.width;
                const scaledViewport = page.getViewport({
                  scale: scale,
                });
                contentDrawingCanvas.width = scaledViewport.width;
                contentDrawingCanvas.height = scaledViewport.height;
                const renderContext = {
                  canvasContext: contentDrawingContext,
                  viewport: scaledViewport,
                };
                page.render(renderContext);
                showContent(contentDrawing);
              });
            }
          } else {
            findRenderer(url, 'unknown');
          }
        }).catch(function() {
          findRenderer(url, 'unknown');
        });
      }

      function findRenderer(url, contentType, error) {
        if (error) {
          if (error.name != 'AbortError') {
            cache[url] = {
              url: url,
              contentType: contentType,
            };
          }
        }
        if (contentType.indexOf('image/') != -1) {
          renderImage(url);
        } else if (contentType.indexOf('video/') != -1) {
          renderVideo(url);
        } else if (
          (contentType.indexOf('application/pdf') != -1) &&
          window.pdfjsLib
        ) {
          renderPdf(url);
        } else if (contentType.indexOf('text/html') != -1) {
          renderHTML(url);
        } else if (
          (contentType.indexOf('text/') != -1) ||
          // (contentType.indexOf('text/css') != -1) ||
          // (contentType.indexOf('text/csv') != -1) ||
          // (contentType.indexOf('text/html') != -1) ||
          // (contentType.indexOf('text/javascript') != -1) ||
          // (contentType.indexOf('text/markdown') != -1) ||
          // (contentType.indexOf('text/xml') != -1) ||
          // (contentType.indexOf('text/plain') != -1)
          (contentType.indexOf('application/json') != -1)
        ) {
          renderText(url);
        } else {
          showContent(unknownFormat);
        }
      }

      function identify(url) {
        let cachedResult = cache[url];
        if (cachedResult) {
          findRenderer(url, cachedResult.contentType);
        } else {
          fetch(url, {
            signal,
            method: 'HEAD',
          }).then(function(response) {
            if (response.ok) {
              findRenderer(url, response.headers.get('content-type'));
            } else {
              findRenderer(url, 'unknown');
            }
          }).catch(function(error) {
            findRenderer(url, 'unknown', error);
          });
          window.setTimeout(function() {
            controller.abort();
          }, 3000); // timeout 5000
        }
      }

      function saveMousePos(event) {
        const pageX = event.pageX || event.touches[0].pageX;
        const pageY = event.pageY || event.touches[0].pageY;
        currentLeft = pageX;
        currentTop = pageY;
      }

      function reposition() {
        const documentHeight = parseFloat(getComputedStyle(document.body, null).height.replace('px', ''));
        const documentWidth = parseFloat(getComputedStyle(document.body, null).width.replace('px', ''));
        const contentHeight = previewerOverlay.offsetHeight;
        const contentWidth = previewerOverlay.offsetWidth;
        let deltaTop = 0;
        let deltaLeft = 0;
        if (currentLeft + contentWidth > documentWidth) {
          deltaLeft = contentWidth;
        }
        if (currentTop + contentHeight > documentHeight) {
          deltaTop = contentHeight;
        }
        let newPosition = {
          left: currentLeft - deltaLeft + (deltaLeft > 0 ? -10 : 10),
          top: currentTop - deltaTop + (deltaTop > 0 ? -10 : 10),
        };
        const windowHeight = window.innerHeight;
        const windowScrollTop = window.scrollY;
        const relativeTop = newPosition.top - windowScrollTop;
        if (relativeTop + contentHeight > windowHeight) {
          newPosition.top = newPosition.top - ((relativeTop + contentHeight) - windowHeight) - 10;
        } else if (newPosition.top < windowScrollTop) {
          newPosition.top = windowScrollTop + 10;
        }
        previewerOverlay.style.left = `${newPosition.left}px`;
        previewerOverlay.style.top = `${newPosition.top}px`;
      }

      _this.move = function(event) {
        saveMousePos(event);
        reposition();
      };

      _this.destroy = function() {
        controller.abort();
        previewerOverlay.remove();
      };

      let previewerOverlay = document.createElement('div');
      previewerOverlay.classList.add(`${componentClass}-container`);
      previewerOverlay.classList.add(`${componentClass}-hide`);
      previewerOverlay.style.color = params.fgColor;
      previewerOverlay.style.backgroundColor = params.bgColor;
      previewerOverlay.innerHTML = previewerOverlayTemplate;

      params.beautify(previewerOverlay, selector);

      document.body.appendChild(previewerOverlay);

      const contentImage = previewerOverlay.querySelector(`img.${componentClass}-image-content`);
      contentImage.onload = function() {
        showContent(contentImage);
      };
      contentImage.onerror = function() {
        findRenderer(this.src, 'unknown');
      };

      const contentVideo = previewerOverlay.querySelector(`video.${componentClass}-video`);
      contentVideo.onloadedmetadata = function() {
        showContent(contentVideo);
      };
      contentVideo.onerror = function() {
        findRenderer(this.src, 'unknown');
      };

      const contentHtml = previewerOverlay.querySelector(`div.${componentClass}-html`);
      contentHtml.onload = function() {
        showContent(contentHtml);
      };

      const contentText = previewerOverlay.querySelector(`pre.${componentClass}-text`);
      // const contentHtml = previewerOverlay.querySelector(`div.${componentClass}-html`);
      const contentDrawing = previewerOverlay.querySelector(`div.${componentClass}-drawing`);
      const contentDrawingCanvas = contentDrawing.querySelector(`canvas.${componentClass}-drawing-canvas`);
      const contentDrawingContext = contentDrawingCanvas.getContext('2d');
      const loadingImage = previewerOverlay.querySelector(`div.${componentClass}-image-loading`);
      const unknownFormat = previewerOverlay.querySelector(`div.${componentClass}-unknown-format`);

      const controller = new AbortController();
      const {
        signal,
      } = controller;

      let currentContent = loadingImage;

      let currentTop, currentLeft;

      saveMousePos(event);
      reposition();
      previewerOverlay.classList.add(`${componentClass}-show`);
      previewerOverlay.classList.remove(`${componentClass}-hide`);
      loadingImage.style.display = '';

      identify(selector.getAttribute('href'));

      return _this;
    }

    function createPreviewRenderer(element, event, params) {
      if (!element.classList.contains(componentClass)) {
        element.classList.add(componentClass);
      }
      if (activeRenderer) {
        activeRenderer.destroy();
        activeRenderer = null;
      }
      activeRenderer = new Renderer(element, event, params);
      element.setAttribute(`${componentClass}-attached`, true);
    }

    function syncPreviewRenderer(event) {
      if (activeRenderer) {
        activeRenderer.move(event);
      }
    }

    function destroyPreviewRenderer(element) {
      if (activeRenderer) {
        activeRenderer.destroy();
        activeRenderer = null;
      }
      if (element) {
        element.removeAttribute(`${componentClass}-attached`);
      }
    }

    function delegate(eventName, selector, handler) {
      document.addEventListener(eventName, function(event) {
        for (let target = event.target; target && target != this; target = target.parentNode) {
          if (target.matches(selector)) {
            handler.call(target, event);
            break;
          }
        }
      }, true);
    }

    _this.attach = function(selector, settings) {
      selector = selector || 'a';

      const params = Object.assign({
        bgColor: 'black',
        fgColor: 'white',
        beautify: function() {
        //
        },
      }, settings);

      delegate('mouseenter', selector, function(event) {
        createPreviewRenderer(this, event, params);
      });

      delegate('mousemove', selector, function(event) {
        syncPreviewRenderer(event);
      });

      delegate('mouseleave', selector, function() {
        destroyPreviewRenderer(this);
      });
    };

    return _this;
  }

  if (typeof module !== 'undefined' && module.exports) {
    module.exports = LinkPreviewer;
  } else {
    window.LinkPreviewer = LinkPreviewer;
  }
})(window);
