'use strict';

;
class FramelixBaseTypeDef {
  static toAttrValue(data) {
    return FramelixStringUtils.jsonStringify(data, false, true);
  }
  static fromAttrValue(str) {
    if (!str) {
      return null;
    }
    return JSON.parse(str);
  }
}
;
;
class FramelixColorUtils {
  static inverterCache = {};
  static setColorHtmlAttributesFromColorDef(colorDef, attributes, tagName) {
    if (typeof colorDef === 'string' || colorDef.theme) {
      const theme = colorDef.theme || colorDef;
      if (tagName.startsWith('framelix-')) {
        attributes.set('theme', theme);
      } else {
        attributes.set('data-theme', theme);
        attributes.setStyle('color', 'var(--color-' + theme + '-text)');
        attributes.setStyle('background-color', 'var(--color-' + theme + '-text)');
      }
    }
    if (colorDef instanceof FramelixTypeDefElementColor || typeof colorDef === 'object') {
      if (colorDef.bgColor) {
        if (typeof colorDef.bgColor === 'string') {
          attributes.setStyle('background-color', colorDef.bgColor);
        } else if (colorDef.bgColor instanceof HTMLElement || colorDef.bgColor instanceof cash) {
          attributes.setStyle('background-color', $(colorDef.bgColor).css('background-color'));
        } else {
          const hsla = colorDef.bgColor;
          const colorType = hsla.length === 4 ? 'hsla' : 'hsl';
          if (hsla.length < 2 || hsla[1] === null) {
            hsla[1] = 'var(--color-default-contrast-bg)';
          } else {
            hsla[1] += '%';
          }
          if (hsla.length < 3 || hsla[2] === null) {
            hsla[2] = 'var(--color-default-lightness-bg)';
          } else {
            hsla[2] += '%';
          }
          attributes.setStyle('background-color', colorType + '(' + hsla.join(', ') + ')');
        }
      }
      if (colorDef.textColor) {
        if (colorDef.textColor === 'invert') {
          if (!colorDef.bgColor) {
            console.error('To invert textcolor, a bgcolor must be set');
            return;
          }
          attributes.setStyle('background-color', FramelixColorUtils.invertColor(colorDef.bgColor, true));
        } else if (typeof colorDef.textColor === 'string') {
          attributes.setStyle('color', colorDef.textColor);
        } else if (colorDef.textColor instanceof HTMLElement || colorDef.textColor instanceof cash) {
          attributes.setStyle('color', $(colorDef.textColor).css('color'));
        } else {
          const hsla = colorDef.textColor;
          const colorType = hsla.length === 4 ? 'hsla' : 'hsl';
          if (hsla.length < 3) {
            hsla[2] = 'var(--color-default-lightness-text)';
          }
          if (hsla.length < 2) {
            hsla[1] = 'var(--color-default-contrast-text)';
          }
          hsla[1] += '%';
          hsla[2] += '%';
          attributes.setStyle('color', colorType + '(' + hsla.join(', ') + ')');
        }
      }
    }
  }
  static setColorsFromElementColorDef(colorDef, element) {
    element = $(element);
    if (typeof colorDef === 'string' || colorDef.theme) {
      const theme = colorDef.theme || colorDef;
      if (element[0].tagName.startsWith('FRAMELIX-')) {
        element.attr('theme', theme);
      } else {
        if (typeof element.attr('data-theme') === 'string') {
          element.attr('data-theme', theme);
        } else {
          element.css('color', 'var(--color-' + theme + '-text)').css('background-color', 'var(--color-' + theme + '-text)');
        }
      }
    }
    if (colorDef instanceof FramelixTypeDefElementColor || typeof colorDef === 'object') {
      if (colorDef.bgColor) {
        if (typeof colorDef.bgColor === 'string') {
          element.css('background-color', colorDef.bgColor);
        } else if (colorDef.bgColor instanceof HTMLElement || colorDef.bgColor instanceof cash) {
          element.css('background-color', $(colorDef.bgColor).css('background-color'));
        } else {
          const hsla = colorDef.bgColor;
          const colorType = hsla.length === 4 ? 'hsla' : 'hsl';
          if (hsla.length < 2 || hsla[1] === null) {
            hsla[1] = 'var(--color-default-contrast-bg)';
          } else {
            hsla[1] += '%';
          }
          if (hsla.length < 3 || hsla[2] === null) {
            hsla[2] = 'var(--color-default-lightness-bg)';
          } else {
            hsla[2] += '%';
          }
          element.css('background-color', colorType + '(' + hsla.join(', ') + ')');
        }
      }
      if (colorDef.textColor) {
        if (colorDef.textColor === 'invert') {
          element[0].style.color = FramelixColorUtils.invertColor(element[0].backgroundColor, true);
        } else if (typeof colorDef.textColor === 'string') {
          element.css('color', colorDef.textColor);
        } else if (colorDef.textColor instanceof HTMLElement || colorDef.textColor instanceof cash) {
          element.css('color', $(colorDef.textColor).css('color'));
        } else {
          const hsla = colorDef.textColor;
          const colorType = hsla.length === 4 ? 'hsla' : 'hsl';
          if (hsla.length < 3) {
            hsla[2] = 'var(--color-default-lightness-text)';
          }
          if (hsla.length < 2) {
            hsla[1] = 'var(--color-default-contrast-text)';
          }
          hsla[1] += '%';
          hsla[2] += '%';
          element.css('color', colorType + '(' + hsla.join(', ') + ')');
        }
      }
    }
  }
  static invertColor(colorStr) {
    let blackWhite = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    const cacheKey = colorStr + '_' + (blackWhite ? '1' : '0');
    if (typeof FramelixColorUtils.inverterCache[cacheKey] !== 'undefined') {
      return FramelixColorUtils.inverterCache[cacheKey];
    }
    let rgb = [];
    if (colorStr.startsWith('#') && colorStr.length === 7) {
      rgb = FramelixColorUtils.hexToRgb(colorStr);
    } else if (colorStr.startsWith('rgb')) {
      rgb = colorStr.replace(/[^0-9.,]/g, '').split(',').map(function (value) {
        return parseInt(value);
      });
    } else if (colorStr.startsWith('hsl')) {
      rgb = FramelixColorUtils.hslToRgb(...colorStr.replace(/[^0-9.,]/g, '').split(',').map(function (value, index) {
        if (index === 0) {
          return parseFloat(value) / 360;
        }
        return parseFloat(value) / 100;
      }));
    } else {
      const el = document.createElement('span');
      el.style.color = colorStr;
      document.body.appendChild(el);
      FramelixColorUtils.inverterCache[cacheKey] = FramelixColorUtils.invertColor(getComputedStyle(el).color);
      el.remove();
      return FramelixColorUtils.inverterCache[cacheKey];
    }
    if (blackWhite) {
      FramelixColorUtils.inverterCache[cacheKey] = rgb[0] * 0.299 + rgb[1] * 0.587 + rgb[2] * 0.114 > 186 ? '#000' : '#fff';
      return FramelixColorUtils.inverterCache[cacheKey];
    }
    let r = (255 - rgb[0]).toString(16);
    let g = (255 - rgb[1]).toString(16);
    let b = (255 - rgb[2]).toString(16);
    FramelixColorUtils.inverterCache[cacheKey] = '#' + r.padStart(2, '0') + g.padStart(2, '0') + b.padStart(2, '0');
    return FramelixColorUtils.inverterCache[cacheKey];
  }
  static hslToRgb(h, s, l) {
    let r, g, b;
    if (s === 0) {
      r = g = b = l;
    } else {
      const hue2rgb = function hue2rgb(p, q, t) {
        if (t < 0) {
          t += 1;
        }
        if (t > 1) {
          t -= 1;
        }
        if (t < 1 / 6) {
          return p + (q - p) * 6 * t;
        }
        if (t < 1 / 2) {
          return q;
        }
        if (t < 2 / 3) {
          return p + (q - p) * (2 / 3 - t) * 6;
        }
        return p;
      };
      let q = l < 0.5 ? l * (1 + s) : l + s - l * s;
      let p = 2 * l - q;
      r = hue2rgb(p, q, h + 1 / 3);
      g = hue2rgb(p, q, h);
      b = hue2rgb(p, q, h - 1 / 3);
    }
    return [FramelixNumberUtils.round(r * 255, 0), FramelixNumberUtils.round(g * 255, 0), FramelixNumberUtils.round(b * 255, 0)];
  }
  static rgbToHsl(r, g, b) {
    r /= 255;
    g /= 255;
    b /= 255;
    let max = Math.max(r, g, b),
      min = Math.min(r, g, b);
    let h,
      s,
      l = (max + min) / 2;
    if (max === min) {
      h = s = 0;
    } else {
      let d = max - min;
      s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
      switch (max) {
        case r:
          h = (g - b) / d + (g < b ? 6 : 0);
          break;
        case g:
          h = (b - r) / d + 2;
          break;
        case b:
          h = (r - g) / d + 4;
          break;
      }
      h /= 6;
    }
    return [h, s, l];
  }
  static hexToRgb(hex) {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? [parseInt(result[1], 16), parseInt(result[2], 16), parseInt(result[3], 16)] : null;
  }
  static rgbToHex(r, g, b) {
    return '#' + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
  }
  static cssColorToHex(rgb) {
    const arr = (rgb || null).replace(/[^0-9.,]/g, '').split(',');
    return FramelixColorUtils.rgbToHex(parseFloat(arr[0] || 0), parseFloat(arr[1] || 0), parseFloat(arr[2] || 0));
  }
}
;
;
class FramelixConfig {
  static applicationUrl;
  static modulePublicUrl;
  static modules;
  static compiledFileUrls = {};
  static dateTimeFormatPhp;
  static dateFormatPhp;
  static dateTimeFormatJs;
  static dateFormatJs;
}
;
;
class FramelixDateUtils {
  static compare(dateA, dateB) {
    var _FramelixDateUtils$an, _FramelixDateUtils$an2;
    dateA = parseInt(((_FramelixDateUtils$an = FramelixDateUtils.anyToDayJs(dateA)) === null || _FramelixDateUtils$an === void 0 ? void 0 : _FramelixDateUtils$an.format('YYYYMMDD')) || 0);
    dateB = parseInt(((_FramelixDateUtils$an2 = FramelixDateUtils.anyToDayJs(dateB)) === null || _FramelixDateUtils$an2 === void 0 ? void 0 : _FramelixDateUtils$an2.format('YYYYMMDD')) || 0);
    if (dateA < dateB) {
      return '<';
    }
    if (dateA > dateB) {
      return '>';
    }
    return '=';
  }
  static anyToFormat(value) {
    let outputFormat = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : FramelixConfig.dateFormatJs;
    let expectedInputFormats = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : FramelixConfig.dateFormatJs + ',YYYY-MM-DD';
    const instance = FramelixDateUtils.anyToDayJs(value, expectedInputFormats);
    if (instance === null) return null;
    return instance.format(outputFormat);
  }
  static anyToDayJs(value) {
    let expectedInputFormats = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : FramelixConfig.dateFormatJs + ',YYYY-MM-DD';
    if (value === null || value === undefined) return null;
    if (typeof value === 'number') {
      return dayjs(value);
    }
    const instance = dayjs(value, expectedInputFormats.split(','));
    if (instance.isValid()) {
      return instance;
    }
    return null;
  }
  static anyToUnixtime(value) {
    let expectedInputFormats = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : FramelixConfig.dateFormatJs + ',YYYY-MM-DD';
    const instance = FramelixDateUtils.anyToDayJs(value, expectedInputFormats);
    if (instance === null) return null;
    return instance.unix();
  }
}
;
;
class FramelixDom {
  static domChangesCount = 0;
  static changeListeners = [];
  static observer;
  static init() {
    let observerTimeout = null;
    let consecutiveLoads = 0;
    let lastChange = new Date().getTime();
    FramelixDom.observer = new MutationObserver(function (mutationRecords) {
      FramelixDom.domChangesCount++;
      let valid = false;
      for (let i = 0; i < mutationRecords.length; i++) {
        const record = mutationRecords[i];
        if (record.target && record.target.ignoreDomObserver) {
          continue;
        }
        valid = true;
      }
      if (!valid) return;
      clearTimeout(observerTimeout);
      observerTimeout = setTimeout(function () {
        for (let i = 0; i < FramelixDom.changeListeners.length; i++) {
          FramelixDom.changeListeners[i].callback();
        }
        let currentTime = new Date().getTime();
        if (currentTime - lastChange <= 70) {
          consecutiveLoads++;
        } else {
          consecutiveLoads = 0;
        }
        lastChange = currentTime;
        if (consecutiveLoads > 10) {
          console.warn('Framework->FramelixDom: MutationObserver detected ' + consecutiveLoads + ' consecutiveLoads of a period of  ' + consecutiveLoads * 50 + 'ms - Maybe this point to a loop in dom changes');
        }
      }, 50);
    });
    FramelixDom.observer.observe(document.body, {
      attributes: true,
      childList: true,
      characterData: true,
      subtree: true
    });
  }
  static isInDom(el) {
    if (el instanceof cash) {
      el = el[0];
    }
    return document.body.contains(el);
  }
  static isVisible(el) {
    if (el instanceof cash) {
      el = el[0];
    }
    if (FramelixDom.isInDom(el)) {
      el = el.getBoundingClientRect();
      return el.width > 0 || el.height > 0;
    }
    return false;
  }
  static addChangeListener(id, callback) {
    const row = {
      'id': id,
      'callback': callback
    };
    FramelixDom.changeListeners.push(row);
  }
  static removeChangeListener(id) {
    FramelixDom.changeListeners = FramelixDom.changeListeners.filter(item => item.id !== id);
  }
  static async includeCompiledFile(module, type, id, waitFor) {
    return FramelixDom.includeResource(FramelixConfig.compiledFileUrls[module][type][id], waitFor);
  }
  static async includeResource(fileUrl, waitFor) {
    const id = 'framelix-resource-' + fileUrl.replace(/[^a-z0-9-]/ig, '-');
    if (!document.getElementById(id)) {
      const url = new URL(fileUrl);
      if (url.pathname.endsWith('.css')) {
        $('head').append(`<link rel="stylesheet" media="all" href="${fileUrl}" id="${id}">`);
      } else if (url.pathname.endsWith('.js')) {
        $('head').append(`<script src="${fileUrl}" id="${id}"></script>`);
      }
    }
    if (waitFor) {
      let count = 0;
      while (typeof waitFor === 'string' && typeof window[waitFor] === 'undefined' || typeof waitFor === 'function' && (await waitFor()) !== true) {
        await Framelix.wait(10);
        if (count++ > 1000) {
          break;
        }
      }
    }
  }
}
FramelixInit.late.push(FramelixDom.init);
;
class FramelixHtmlAttributes {
  styles = null;
  classes = null;
  other = null;
  assignToElement(el) {
    if (FramelixObjectUtils.hasKeys(this.styles)) el.css(this.styles);
    if (FramelixObjectUtils.hasKeys(this.classes)) el.addClass(this.classes);
    if (FramelixObjectUtils.hasKeys(this.other)) el.attr(this.other);
  }
  toString() {
    let out = {};
    if (this.styles) {
      let arr = [];
      for (let key in this.styles) {
        arr.push(key + ':' + this.styles[key] + ';');
      }
      out['style'] = arr.join(' ');
      if (out['style'] === '') delete out['style'];
    }
    if (this.classes) {
      out['class'] = this.classes.join(' ');
      if (out['class'] === '') delete out['class'];
    }
    if (this.other) {
      out = FramelixObjectUtils.merge(out, this.other);
    }
    let str = [];
    for (let key in out) {
      str.push(key + '="' + FramelixStringUtils.htmlEscape(out[key]) + '"');
    }
    return str.join(' ');
  }
  addClass(className) {
    if (!this.classes) this.classes = [];
    if (this.classes.indexOf(className) === -1) this.classes.push(className);
  }
  removeClass(className) {
    if (!this.classes) return;
    const index = this.classes.indexOf(className);
    if (index > -1) this.classes.splice(index);
  }
  setStyle(key, value) {
    if (!this.styles) this.styles = {};
    if (value === null) {
      delete this.styles[key];
      return;
    }
    this.styles[key] = value;
  }
  getStyle(key) {
    return this.styles ? this.styles[key] : null;
  }
  set(key, value) {
    if (!this.other) this.other = {};
    if (value === null) {
      delete this.other[key];
      return;
    }
    this.other[key] = value;
  }
  get(key) {
    return this.other ? this.other[key] : null;
  }
}
;
;
class FramelixIntersectionObserver {
  static observer = null;
  static elementMap = new Map();
  static async isIntersecting(element) {
    return new Promise(function (resolve) {
      FramelixIntersectionObserver.observe(element, function (isIntersecting) {
        FramelixIntersectionObserver.unobserve(element);
        resolve(isIntersecting);
      });
    });
  }
  static onGetVisible(element, callback) {
    FramelixIntersectionObserver.observe(element, function (isIntersecting) {
      if (isIntersecting) {
        FramelixIntersectionObserver.unobserve(element);
        callback();
      }
    });
  }
  static onGetInvisible(element, callback) {
    FramelixIntersectionObserver.observe(element, function (isIntersecting) {
      if (!isIntersecting) {
        FramelixIntersectionObserver.unobserve(element);
        callback();
      }
    });
  }
  static observe(element, callback) {
    if (!FramelixIntersectionObserver.observer) FramelixIntersectionObserver.init();
    element = $(element)[0];
    if (!FramelixIntersectionObserver.elementMap.get(element)) {
      FramelixIntersectionObserver.elementMap.set(element, {
        'callbacks': [callback],
        'isIntersecting': false,
        'intersectionRatio': 0,
        'hasValidStatus': false
      });
      FramelixIntersectionObserver.observer.observe(element);
      return;
    }
    const config = FramelixIntersectionObserver.elementMap.get(element);
    config.callbacks.push(callback);
    if (config.hasValidStatus) callback(config.isIntersecting, config.intersectionRatio);
  }
  static unobserve(element, callback) {
    if (!FramelixIntersectionObserver.observer) FramelixIntersectionObserver.init();
    element = $(element)[0];
    if (!callback) {
      FramelixIntersectionObserver.elementMap.delete(element);
      FramelixIntersectionObserver.observer.unobserve(element);
      return;
    }
    const config = FramelixIntersectionObserver.elementMap.get(element);
    let removeIndex = config.callbacks.indexOf(callback);
    if (removeIndex > -1) config.callbacks.splice(removeIndex, 1);
  }
  static init() {
    FramelixIntersectionObserver.observer = new IntersectionObserver(function (observerEntries) {
      observerEntries.forEach(function (observerEntry) {
        FramelixIntersectionObserver.elementMap.forEach(function (config, element) {
          if (element === observerEntry.target) {
            config.hasValidStatus = true;
            config.isIntersecting = observerEntry.isIntersecting;
            config.intersectionRatio = observerEntry.intersectionRatio;
            for (let i = 0; i < config.callbacks.length; i++) {
              config.callbacks[i](config.isIntersecting, config.intersectionRatio);
            }
          }
        });
      });
    }, {
      rootMargin: '0px',
      threshold: 0
    });
  }
}
;
;
class FramelixLang {
  static values = {};
  static loadableLangFiles = {};
  static loadingPromises = {};
  static languagesAvailable;
  static lang;
  static langFallback;
  static async get(key, parameters, lang) {
    if (Array.isArray(key)) {
      return FramelixLang.get.apply(null, key);
    }
    if (!key || typeof key !== 'string' || !key.startsWith('__')) {
      return key;
    }
    const langDefault = lang || FramelixLang.lang;
    const langFallback = FramelixLang.langFallback;
    const languages = [langDefault, langFallback];
    for (let loadLang of languages) {
      if (!FramelixLang.loadingPromises[loadLang]) {
        FramelixLang.loadingPromises[loadLang] = new Promise(async function (resolve) {
          if (!FramelixLang.values[loadLang]) {
            FramelixLang.values[loadLang] = {};
          }
          if (!FramelixLang.loadableLangFiles[loadLang]) {
            resolve();
            return;
          }
          for (let key in FramelixLang.loadableLangFiles[loadLang]) {
            const row = FramelixLang.loadableLangFiles[loadLang][key];
            FramelixLang.values[loadLang] = Object.assign(FramelixLang.values[loadLang], await (await fetch(row.url)).json());
          }
          resolve();
        });
      }
      await FramelixLang.loadingPromises[loadLang];
    }
    let value = null;
    if (FramelixLang.values[langDefault] && FramelixLang.values[langDefault][key] !== undefined) {
      value = FramelixLang.values[langDefault][key];
    }
    if (value === null && FramelixLang.values[langFallback] && FramelixLang.values[langFallback][key] !== undefined) {
      value = FramelixLang.values[langFallback][key];
    }
    if (value === null) {
      return key;
    }
    if (Array.isArray(value)) value = value[0];
    if (parameters) {
      let re = /\{\{(.*?)\}\}/ig;
      let m;
      do {
        m = re.exec(value);
        if (m) {
          let replaceWith = null;
          let conditions = m[1].split('|');
          for (let i = 0; i < conditions.length; i++) {
            const condition = conditions[i];
            const conditionSplit = condition.match(/^([a-z0-9-_]+)([!=<>]+)([0-9*]+):(.*)/i);
            if (conditionSplit) {
              const parameterName = conditionSplit[1];
              const compareOperator = conditionSplit[2];
              const compareNumber = parseInt(conditionSplit[3]);
              const outputValue = conditionSplit[4];
              const parameterValue = parameters[parameterName];
              if (conditionSplit[3] === '*') {
                replaceWith = outputValue;
              } else if (compareOperator === '=' && compareNumber === parameterValue) {
                replaceWith = outputValue;
              } else if (compareOperator === '<' && compareNumber < parameterValue) {
                replaceWith = outputValue;
              } else if (compareOperator === '>' && compareNumber > parameterValue) {
                replaceWith = outputValue;
              } else if (compareOperator === '<=' && compareNumber <= parameterValue) {
                replaceWith = outputValue;
              } else if (compareOperator === '>=' && compareNumber >= parameterValue) {
                replaceWith = outputValue;
              }
              if (replaceWith !== null) {
                break;
              }
            }
          }
          value = FramelixStringUtils.replace(m[0], replaceWith === null ? '' : replaceWith, value);
        }
      } while (m);
      for (let search in parameters) {
        let replace = parameters[search];
        value = FramelixStringUtils.replace('{' + search + '}', replace, value);
      }
    }
    return value;
  }
}
;
;
class FramelixModal {
  static modalsContainer;
  static instances = [];
  static currentInstance = null;
  options = {};
  container;
  contentContainer;
  bodyContainer;
  headerContainer;
  footerContainer;
  closeButton;
  created;
  destroyed;
  confirmed;
  promptResult;
  resolvers;
  static async destroyAll() {
    let promises = [];
    for (let i = 0; i < FramelixModal.instances.length; i++) {
      const instance = FramelixModal.instances[i];
      promises.push(instance.destroy());
    }
    return Promise.all(promises);
  }
  static init() {
    FramelixModal.modalsContainer = $(`<div class="framelix-modals"></div>`);
    $('body').append(FramelixModal.modalsContainer);
  }
  static alert(content, options) {
    if (!options) options = {};
    const html = $(`<div style="text-align: center;">`);
    html.append(content);
    if (!options.maxWidth) options.maxWidth = 600;
    options.bodyContent = html;
    options.footerContent = '<framelix-button icon="718">__framelix_ok__</framelix-button>';
    const modal = FramelixModal.show(options);
    const buttons = modal.footerContainer.find('framelix-button');
    buttons.on('click', function () {
      modal.destroy();
    });
    setTimeout(function () {
      buttons.trigger('focus');
    }, 10);
    return modal;
  }
  static prompt(content, defaultText, options) {
    if (!options) options = {};
    const html = $(`<div style="text-align: center;"></div>`);
    if (content) {
      html.append(content);
      html.append('<div class="framelix-spacer"></div>');
    }
    const input = $('<input type="text" class="framelix-form-field-input">');
    if (defaultText !== undefined) input.val(defaultText);
    html.append($('<div>').append(input));
    let footerContainer = `
        <framelix-button icon="719" theme="light">__framelix_cancel__</framelix-button>
        <framelix-button data-success="1" icon="718" theme="success" style="flex-grow: 4">__framelix_ok__</framelix-button>
    `;
    const close = function (success) {
      if (modal.resolvers['prompt']) {
        modal.resolvers['prompt'](success ? input.val() : null);
        delete modal.resolvers['prompt'];
      }
      modal.destroy();
    };
    input.on('keydown', function (ev) {
      if (ev.key === 'Enter') {
        close(true);
      }
    });
    if (!options.maxWidth) options.maxWidth = 600;
    options.bodyContent = html;
    options.footerContent = footerContainer;
    const modal = FramelixModal.show(options);
    const buttons = modal.footerContainer.find('framelix-button');
    buttons.on('click', function () {
      close($(this).attr('data-success') === '1');
    });
    setTimeout(function () {
      input.trigger('focus');
    }, 10);
    return modal;
  }
  static confirm(content, options) {
    if (!options) options = {};
    const html = $(`<div style="text-align: center;"></div>`);
    html.html(content);
    const bottom = $(`
      <framelix-button theme="light" icon="719">__framelix_cancel__</framelix-button>
      <framelix-button theme="success" data-success="1" icon="718" style="flex-grow: 4">__framelix_ok__</framelix-button>
    `);
    if (!options.maxWidth) options.maxWidth = 600;
    options.bodyContent = html;
    options.footerContent = bottom;
    const modal = FramelixModal.show(options);
    const buttons = modal.footerContainer.find('framelix-button');
    buttons.on('click', function () {
      if (modal.resolvers['confirmed']) {
        modal.resolvers['confirmed']($(this).attr('data-success') === '1');
        delete modal.resolvers['confirmed'];
      }
      modal.destroy();
    });
    setTimeout(function () {
      buttons.last().trigger('focus');
    }, 10);
    return modal;
  }
  static async request(method, urlPath, urlParams, postData, showProgressBar, fetchOptions, modalOptions) {
    if (!modalOptions) modalOptions = {};
    modalOptions.bodyContent = '<div class="framelix-loading"></div>';
    const modal = FramelixModal.show(modalOptions);
    modal.request = FramelixRequest.request(method, urlPath, urlParams, postData, showProgressBar, fetchOptions);
    if ((await modal.request.checkHeaders()) === 0) {
      const json = await modal.request.getJson();
      modal.bodyContainer.html(json === null || json === void 0 ? void 0 : json.content);
    }
    return modal;
  }
  static show(options) {
    const instance = options.instance || new FramelixModal();
    instance.container.append(FramelixToast.container);
    FramelixModal.currentInstance = instance;
    instance.options = options;
    instance.resolvers = {};
    instance.created = new Promise(function (resolve) {
      instance.resolvers['created'] = resolve;
    });
    instance.confirmed = new Promise(function (resolve) {
      instance.resolvers['confirmed'] = resolve;
    });
    instance.promptResult = new Promise(function (resolve) {
      instance.resolvers['prompt'] = resolve;
    });
    instance.destroyed = new Promise(function (resolve) {
      instance.resolvers['destroyed'] = resolve;
    });
    if (!options.instance) {
      FramelixModal.modalsContainer.append(instance.container);
      instance.container[0].showModal();
      instance.closeButton = instance.container.find('.framelix-modal-close framelix-button');
      instance.contentContainer = instance.container.find('.framelix-modal-content');
      instance.headerContainer = instance.container.find('.framelix-modal-header');
      instance.bodyContainer = instance.container.find('.framelix-modal-body');
      instance.footerContainer = instance.container.find('.framelix-modal-footer');
      instance.closeButton.on('click', function () {
        instance.destroy();
      });
      instance.container.on('cancel close', function () {
        instance.destroy();
      });
    }
    instance.container.find('.framelix-modal-inner').attr('class', 'framelix-modal-inner framelix-modal-inner-' + options.color);
    if (typeof options.maxWidth === 'undefined') {
      const content = $('.framelix-content-inner-inner');
      options.maxWidth = content.length ? content.width() : 1800;
    }
    const inner = instance.container.find('.framelix-modal-inner');
    inner.css('max-width', typeof options.maxWidth === 'number' ? options.maxWidth + 'px' : options.maxWidth);
    const writeContainers = {
      'headerContainer': options.headerContent,
      'bodyContainer': options.bodyContent,
      'footerContainer': options.footerContent
    };
    const writePromises = [];
    for (let containerName in writeContainers) {
      const content = writeContainers[containerName];
      if (content === null || content === undefined) {
        continue;
      }
      instance[containerName].removeClass('hidden');
      if (content instanceof FramelixRequest) {
        writePromises.push(new Promise(async function (resolve) {
          instance[containerName].html(`<div class="framelix-loading"></div>`);
          if ((await content.checkHeaders()) > 0) {
            instance.destroy();
            return;
          }
          await content.writeToContainer(instance[containerName]);
          resolve();
        }));
      } else {
        instance[containerName].html(content);
      }
    }
    instance.container.trigger('focus');
    FramelixPopup.destroyTooltips();
    Promise.all(writePromises).then(function () {
      if (instance.resolvers['created']) {
        instance.resolvers['created']();
        delete instance.resolvers['created'];
      }
    });
    return instance;
  }
  constructor() {
    FramelixModal.instances.push(this);
    this.container = $(`<dialog class="framelix-modal">
        <div class="framelix-modal-inner">
            <div class="framelix-modal-close">
              <framelix-button icon="719" title="__framelix_close__" style="font-size: 24px"></framelix-button>
            </div>
            <div class="framelix-modal-content" role="document">
                <div class="framelix-modal-header hidden"></div>
                <div class="framelix-modal-body"></div>
                <div class="framelix-modal-footer hidden"></div>
            </div>
        </div>
    </dialog>`);
    this.container.attr('data-instance-id', FramelixModal.instances.length - 1);
  }
  async destroy() {
    if (!this.resolvers) return;
    for (let key in this.resolvers) {
      this.resolvers[key]();
    }
    this.resolvers = null;
    const prevModal = this.container.prev('dialog');
    this.container[0].close();
    if (!prevModal.length) {
      FramelixModal.currentInstance = null;
      $('body').append(FramelixToast.container);
    } else {
      FramelixModal.currentInstance = FramelixModal.instances[prevModal.attr('data-instance-id')];
      FramelixModal.currentInstance.container.append(FramelixToast.container);
    }
    this.container.remove();
    this.container = null;
  }
}
FramelixInit.late.push(FramelixModal.init);
;
class FramelixNumberUtils {
  static filesizeToUnit(filesize) {
    let maxUnit = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'yb';
    let binary = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    let decimals = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 2;
    const units = ['b', 'kb', 'mb', 'gb', 'tb', 'pb', 'eb', 'zb', 'yb'];
    maxUnit = maxUnit.toLowerCase();
    let unit = '';
    const multi = binary ? 1024 : 1000;
    for (let i = 0; i < units.length; i++) {
      unit = units[i];
      if (filesize < multi) break;
      if (unit === maxUnit) break;
      filesize /= multi;
    }
    unit = unit.toUpperCase();
    if (binary) {
      unit = unit.substr(0, 1) + 'i' + unit.substr(1);
    }
    return FramelixNumberUtils.format(filesize, decimals) + unit;
  }
  static round(value, decimals) {
    if (!('' + value).includes('e')) {
      return +(Math.round(value + 'e+' + decimals) + 'e-' + decimals);
    } else {
      const arr = ('' + value).split('e');
      let sig = '';
      if (+arr[1] + decimals > 0) {
        sig = '+';
      }
      return +(Math.round(+arr[0] + 'e' + sig + (+arr[1] + decimals)) + 'e-' + decimals);
    }
  }
  static toNumber(value) {
    let round = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let commaSeparator = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : ',';
    if (typeof value !== 'number') {
      if (value === null || value === false || value === undefined || typeof value === 'function') {
        return 0.0;
      }
      if (typeof value === 'object') {
        value = value.toString();
      }
      if (typeof value !== 'string') {
        value = value.toString();
      }
      value = value.trim().replace(new RegExp('[^-0-9' + FramelixStringUtils.escapeRegex(commaSeparator) + ']', 'g'), '');
      value = parseFloat(value.replace(new RegExp(commaSeparator, 'g'), '.'));
    }
    if (isNaN(value) || typeof value !== 'number') value = 0;
    return round !== null ? FramelixNumberUtils.round(value, round) : value;
  }
  static format(value) {
    let decimals = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let commaSeparator = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : ',';
    let thousandSeparator = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : '.';
    if (value === '' || value === null || value === undefined) {
      return '';
    }
    let number = value;
    if (typeof value !== 'number') {
      number = FramelixNumberUtils.toNumber(value, decimals, commaSeparator, thousandSeparator);
    } else {
      number = FramelixNumberUtils.round(number, decimals);
    }
    let sign = value < 0 ? '-' : '';
    value = number.toString();
    if (sign === '-') value = value.substr(1);
    value = value.split('.');
    if (thousandSeparator.length) {
      let newInt = '';
      const l = value[0].length;
      for (let i = 0; i < l; i++) {
        newInt += value[0][i];
        if ((i + 1 - l) % 3 === 0 && i + 1 !== l) {
          newInt += thousandSeparator;
        }
      }
      value[0] = newInt;
    }
    if (decimals && decimals > 0) {
      value[1] = value[1] || '';
      if (decimals > value[1].length) value[1] += '0'.repeat(decimals - value[1].length);
    }
    return sign + value.join(commaSeparator);
  }
}
;
;
class FramelixObjectUtils {
  static phpJsonToJs(data) {
    let recursive = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
    if (!data || typeof data !== 'object') return data;
    if (!recursive) {
      return data;
    }
    if (data.jsClass === '') {
      return data.phpProperties;
    }
    if (data.jsClass && data.phpClass && data.phpProperties) {
      const classRef = eval(data.jsClass);
      if (classRef['phpJsonToJs']) {
        return classRef.phpJsonToJs(data.phpProperties, data.phpClass);
      } else {
        const inst = new classRef();
        for (let i in data.phpProperties) {
          if (inst.hasOwnProperty(i)) {
            inst[i] = recursive ? FramelixObjectUtils.phpJsonToJs(data.phpProperties[i], recursive) : data.phpProperties[i];
          }
        }
        if (inst.hasOwnProperty('phpClass')) inst.phpClass = data.phpClass;
        return inst;
      }
    }
    const ret = Array.isArray(data) ? [] : {};
    for (let i in data) {
      ret[i] = FramelixObjectUtils.phpJsonToJs(data[i], recursive);
    }
    return ret;
  }
  static merge() {
    let ret = {};
    for (let i = 0; i < arguments.length; i++) {
      const obj = i < 0 || arguments.length <= i ? undefined : arguments[i];
      if (typeof obj !== 'object' || obj === null) continue;
      for (let key in obj) {
        const v = obj[key];
        if (typeof v === 'object' && v !== null) {
          ret[key] = FramelixObjectUtils.merge(ret[key], v);
        } else if (v !== undefined) {
          ret[key] = v;
        }
      }
    }
    return ret;
  }
  static hasValue(obj, value) {
    if (obj === null || obj === undefined || typeof obj !== 'object') return false;
    if (Array.isArray(obj)) {
      return obj.indexOf(value) > -1;
    }
    for (let i in obj) {
      if (obj[i] === value) {
        return true;
      }
    }
    return false;
  }
  static hasKeys(obj) {
    let minKeys = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 1;
    if (obj === null || obj === undefined || typeof obj !== 'object') return false;
    let count = 0;
    for (let i in obj) {
      if (++count >= minKeys) {
        return true;
      }
    }
    return false;
  }
  static countKeys(obj) {
    if (obj === null || obj === undefined || typeof obj !== 'object') {
      return 0;
    }
    let count = 0;
    for (let i in obj) count++;
    return count;
  }
  static toUrlencodedString(obj, keyPrefix) {
    if (typeof obj !== 'object') {
      return '';
    }
    let str = '';
    for (let i in obj) {
      if (obj[i] === null || obj[i] === undefined) continue;
      let key = typeof keyPrefix === 'undefined' ? i : keyPrefix + '[' + i + ']';
      if (typeof obj[i] === 'object') {
        str += FramelixObjectUtils.toUrlencodedString(obj[i], key) + '&';
      } else {
        str += encodeURIComponent(key) + '=' + encodeURIComponent(obj[i]) + '&';
      }
    }
    return str.substring(0, str.length - 1);
  }
  static async forEach(rows, callback) {
    if (!rows) return;
    if (Array.isArray(rows)) {
      for (let i = 0; i < rows.length; i++) {
        await callback(i, rows[i]);
      }
      return;
    }
    if (typeof rows === 'object') {
      if (rows.type === 'preparedArray' && rows.keys && rows.keys.length) {
        for (let i = 0; i < rows.keys.length; i++) {
          await callback(rows.keys[i], rows.values[i]);
        }
      } else {
        for (let i in rows) {
          await callback(i, rows[i]);
        }
      }
    }
  }
}
;
;
class FramelixPopup {
  static instances = {};
  id;
  target = null;
  popperInstance = null;
  popperEl = null;
  content = null;
  options = {};
  created;
  destroyed;
  boundingRect = null;
  resolvers;
  static init() {
    $(document).on('mouseenter touchstart', '[data-tooltip],[title]', async function (ev) {
      const title = $(this).attr('title');
      if (title !== undefined) {
        $(this).attr('data-tooltip', $(this).attr('title'));
        $(this).removeAttr('title');
      }
      const text = await FramelixLang.get($(this).attr('data-tooltip'));
      if (!text.trim().length) {
        return;
      }
      const instance = FramelixPopup.show(this, text, {
        closeMethods: 'mouseleave-target',
        color: 'dark',
        group: 'tooltip',
        closeButton: false,
        offsetByMouseEvent: ev,
        width: null,
        textAlign: 'center',
        data: {
          tooltip: true
        }
      });
      instance.popperEl.css('z-index', 999);
    });
    $(document).on('click', function (ev) {
      for (let id in FramelixPopup.instances) {
        const instance = FramelixPopup.instances[id];
        if (!instance.popperEl) continue;
        const popperEl = instance.popperEl[0];
        const contains = popperEl.contains(ev.target);
        if (instance.options.closeMethods.indexOf('click-outside') > -1 && !contains) {
          instance.destroy();
        }
        if (instance.options.closeMethods.indexOf('click-inside') > -1 && contains) {
          instance.destroy();
        }
        if (instance.options.closeMethods.indexOf('click') > -1) {
          instance.destroy();
        }
      }
    });
    FramelixDom.addChangeListener('framelix-popup', function () {
      if (!FramelixObjectUtils.hasKeys(FramelixPopup.instances)) return;
      for (let id in FramelixPopup.instances) {
        const instance = FramelixPopup.instances[id];
        if (!instance.popperEl) continue;
        if (!FramelixDom.isVisible(instance.target) || !FramelixDom.isVisible(instance.popperEl)) {
          instance.destroy();
        } else {
          const boundingRect = JSON.stringify(instance.target[0].getBoundingClientRect());
          if (boundingRect !== instance.boundingRect) {
            instance.boundingRect = boundingRect;
            instance.popperInstance.update();
          }
        }
      }
    });
  }
  static show(target, content, options) {
    const lastModal = FramelixModal.modalsContainer.children().last();
    if (!options) options = {};
    if (options.group === undefined) options.group = 'popup';
    if (options.offset === undefined) options.offset = [0, 5];
    if (options.color === undefined) options.color = 'dark';
    if (options.width === undefined) options.width = '300px';
    if (options.textAlign === undefined) options.textAlign = 'center';
    if (options.appendTo === undefined) options.appendTo = lastModal.length ? lastModal : 'body';
    if (options.padding === undefined) options.padding = '5px 10px';
    if (options.closeMethods === undefined) options.closeMethods = 'click-outside';
    if (typeof options.closeMethods === 'string') options.closeMethods = options.closeMethods.replace(/\s/g, '').split(',');
    if (target instanceof cash) target = target[0];
    const instance = new FramelixPopup();
    instance.options = options;
    instance.resolvers = {};
    instance.created = new Promise(function (resolve) {
      instance.resolvers['created'] = resolve;
    });
    instance.destroyed = new Promise(function (resolve) {
      instance.resolvers['destroyed'] = resolve;
    });
    if (options.offsetByMouseEvent) {
      const rect = target.getBoundingClientRect();
      const elCenter = rect.left + rect.width / 2;
      options.offset = [options.offsetByMouseEvent.pageX - elCenter, 5];
    }
    let popperOptions = {};
    popperOptions.placement = options.placement || 'top';
    if (!popperOptions.modifiers) popperOptions.modifiers = [];
    popperOptions.modifiers.push({
      name: 'offset',
      options: {
        offset: options.offset
      }
    });
    popperOptions.modifiers.push({
      name: 'preventOverflow',
      options: {
        padding: 10,
        altAxis: true,
        tether: !options.stickInViewport
      }
    });
    popperOptions.modifiers.push({
      name: 'arrow',
      options: {
        padding: 5
      }
    });
    if (!target.popperInstances) target.popperInstances = {};
    if (target.popperInstances[options.group]) {
      target.popperInstances[options.group].destroy();
    }
    let popperEl = $(`<div class="framelix-popup" data-theme><div data-popper-arrow></div><div class="framelix-popup-inner" style="padding:${options.padding}"></div></div>`);
    $(options.appendTo).append(popperEl);
    FramelixColorUtils.setColorsFromElementColorDef(options.color, popperEl);
    const contentEl = popperEl.children('.framelix-popup-inner');
    if (typeof options.width === 'string') {
      popperEl.css('width', options.width);
    }
    if (typeof options.textAlign === 'string') {
      contentEl.css('text-align', options.textAlign);
    }
    const writePromises = [];
    if (content instanceof FramelixRequest) {
      writePromises.push(new Promise(async function (resolve) {
        contentEl.html(`<div class="framelix-loading"></div>`);
        if ((await content.checkHeaders()) > 0) {
          instance.destroy();
          return;
        }
        await content.writeToContainer(contentEl);
        resolve();
      }));
    } else {
      contentEl.html(content);
    }
    popperEl[0].framelixPopupInstance = instance;
    instance.content = contentEl;
    instance.popperInstance = Popper.createPopper(target, popperEl[0], popperOptions);
    instance.popperEl = popperEl;
    instance.target = $(target);
    instance.id = FramelixRandom.getRandomHtmlId();
    target.popperInstances[options.group] = instance;
    setTimeout(async function () {
      var _instance$popperInsta, _instance$resolvers;
      FramelixPopup.instances[instance.id] = instance;
      (_instance$popperInsta = instance.popperInstance) === null || _instance$popperInsta === void 0 || _instance$popperInsta.update();
      popperEl.attr('data-show-arrow', '1');
      if ((_instance$resolvers = instance.resolvers) !== null && _instance$resolvers !== void 0 && _instance$resolvers.created) {
        Promise.all(writePromises).then(function () {
          instance.resolvers.created();
          delete instance.resolvers.created;
        });
      }
    }, 100);
    if (options.closeMethods.indexOf('mouseleave-target') > -1) {
      $(target).one('mouseleave touchend mousedown', function () {
        FramelixPopup.instances[instance.id] = instance;
        instance.destroy();
      });
    }
    if (options.closeMethods.indexOf('focusout-popup') > -1) {
      instance.popperEl.one('focusin', function () {
        instance.popperEl.on('focusout', function () {
          setTimeout(function () {
            if (!instance.popperEl || !instance.popperEl.has(document.activeElement).length) {
              instance.destroy();
            }
          }, 100);
        });
      });
    }
    $(document).one('swiped-left swiped-right', function () {
      instance.destroy();
    });
    Framelix.addEscapeAction(function () {
      if (!instance.resolvers) return false;
      instance.destroy();
      return true;
    });
    return instance;
  }
  static destroyInstancesOnTarget(el) {
    if (el instanceof cash) {
      el = el[0];
    }
    if (el.popperInstances) {
      for (let group in el.popperInstances) {
        el.popperInstances[group].destroy();
      }
    }
  }
  static destroyTooltips() {
    for (let id in FramelixPopup.instances) {
      var _FramelixPopup$instan;
      if (!FramelixPopup.instances[id].target) {
        continue;
      }
      if ((_FramelixPopup$instan = FramelixPopup.instances[id]) !== null && _FramelixPopup$instan !== void 0 && (_FramelixPopup$instan = _FramelixPopup$instan.options) !== null && _FramelixPopup$instan !== void 0 && (_FramelixPopup$instan = _FramelixPopup$instan.data) !== null && _FramelixPopup$instan !== void 0 && _FramelixPopup$instan.tooltip) {
        FramelixPopup.instances[id].destroy();
      }
    }
  }
  static async destroyAll() {
    for (let id in FramelixPopup.instances) {
      await FramelixPopup.instances[id].destroy();
    }
  }
  async destroy() {
    if (!this.resolvers) return;
    for (let key in this.resolvers) await this.resolvers[key]();
    this.resolvers = null;
    delete FramelixPopup.instances[this.id];
    if (this.popperEl) {
      this.popperEl.remove();
      this.popperEl = null;
    }
    if (this.popperInstance) {
      this.popperInstance.destroy();
      this.popperInstance = null;
    }
  }
}
FramelixInit.late.push(FramelixPopup.init);
;
class FramelixQuickSearch {
  static EVENT_RESULT_LOADED = 'framelix-quicksearch-result-loaded';
  static instances = [];
  placeholder = '__framelix_quick_search_placeholder__';
  optionFields = {};
  container;
  optionsForm = null;
  id;
  searchField;
  resultContainer;
  rememberSearch = true;
  autostartSearch = true;
  forceInitialQuery = null;
  assignedTable = null;
  signedUrlSearch;
  columns;
  static getById(id) {
    for (let i = 0; i < FramelixQuickSearch.instances.length; i++) {
      if (FramelixQuickSearch.instances[i].id === id) {
        return FramelixQuickSearch.instances[i];
      }
    }
    return null;
  }
  constructor() {
    this.id = 'quicksearch-' + FramelixRandom.getRandomHtmlId();
    FramelixQuickSearch.instances.push(this);
    this.container = $('<div>');
    this.container.addClass('framelix-quick-search');
    this.container.attr('data-instance-id', FramelixQuickSearch.instances.length - 1);
  }
  getLocalStorageKey() {
    return 'framelix-quick-search-' + this.id;
  }
  getCleanText() {
    let text = this.searchField[0].innerText;
    text = text.replace(/[\t\r]/g, '');
    return text;
  }
  setSearchQuery(newQuery) {
    newQuery = newQuery ? newQuery + '' : '';
    newQuery = newQuery.substr(0, 200);
    if (newQuery !== this.searchField.text()) {
      this.searchField.text(newQuery);
    }
  }
  async search() {
    const searchValue = this.getCleanText().trim();
    if (this.rememberSearch) FramelixLocalStorage.set(this.getLocalStorageKey(), searchValue);
    if (typeof this.assignedTable === 'string') {
      let tmp = $('#' + this.assignedTable);
      if (tmp.length) this.resultContainer = tmp.closest('.framelix-table');
    } else if (this.assignedTable instanceof FramelixTable && FramelixDom.isInDom(this.assignedTable.container)) {
      this.resultContainer = this.assignedTable.container;
    }
    if (this.resultContainer.children().length) {
      this.resultContainer.toggleClass('framelix-pulse', true);
    } else {
      this.resultContainer.html(`<div class="framelix-loading"></div>`);
    }
    let result = await FramelixRequest.jsCall(this.signedUrlSearch, {
      'query': searchValue,
      'options': this.optionsForm ? this.optionsForm.getValues() : null
    }).getResponseData();
    this.resultContainer.toggleClass('framelix-pulse', false);
    this.resultContainer.html(result);
    this.container.trigger(FramelixQuickSearch.EVENT_RESULT_LOADED);
  }
  async render() {
    const self = this;
    this.searchField = $(`<div class="framelix-quick-search-input-editable" contenteditable="true" data-placeholder="${await FramelixLang.get(this.placeholder)}" spellcheck="false"></div>`);
    this.container.html(`
      <div class="framelix-quick-search-input">
        <framelix-button theme="light" class="framelix-quick-search-help" title="__framelix_quick_search_help__" icon="780" style="cursor:help;"></framelix-button>
        <framelix-icon icon="744"></framelix-icon>
      </div>
      <div class="framelix-quick-search-options hidden"></div>
      <div class="framelix-quick-search-result"></div>
    `);
    let otherForms = $('form');
    if (FramelixObjectUtils.hasKeys(this.optionFields)) {
      const optionsContainer = this.container.find('.framelix-quick-search-options');
      optionsContainer.removeClass('hidden');
      const form = new FramelixForm();
      this.optionsForm = form;
      form.name = this.id + '-options';
      form.fields = this.optionFields;
      form.render();
      await form.rendered;
      optionsContainer.append(form.container);
      optionsContainer.on(FramelixFormField.EVENT_CHANGE_USER, function () {
        self.search();
      });
    }
    this.container.find('.framelix-quick-search-input').append(this.searchField);
    this.resultContainer = this.container.find('.framelix-quick-search-result');
    if (!otherForms.length) {
      setTimeout(function () {
        self.searchField.trigger('focus');
      }, 10);
    }
    let defaultValue = null;
    if (this.rememberSearch) {
      defaultValue = FramelixLocalStorage.get(this.getLocalStorageKey());
      this.setSearchQuery(defaultValue);
    }
    if (this.forceInitialQuery !== null) {
      defaultValue = this.forceInitialQuery;
      this.setSearchQuery(defaultValue);
      this.search();
    } else if (defaultValue !== null && defaultValue.length > 0 && defaultValue !== '*' && this.autostartSearch) {
      this.search();
    }
    this.searchField.on('change input', function (ev) {
      ev.stopPropagation();
      let cleanText = self.getCleanText();
      self.searchField.find('*').not('div,p,span').remove();
      self.searchField.find('[style],[href]').removeAttr('style').removeAttr('href');
      if (self.searchField.text() === cleanText) {
        return;
      }
      self.setSearchQuery(cleanText);
    });
    this.searchField.on('blur paste', function () {
      setTimeout(function () {
        self.setSearchQuery(self.getCleanText());
      }, 10);
    });
    this.searchField.on('keydown', function (ev) {
      if (ev.key === 'Enter') {
        self.search();
        ev.preventDefault();
      }
      if (ev.key === 'Escape') {
        self.setSearchQuery('');
        FramelixLocalStorage.set(self.getLocalStorageKey(), this.value);
      }
    });
  }
}
;
;
class FramelixRandom {
  static CHARSET_ALPHANUMERIC = 1;
  static CHARSET_ALPHANUMERIC_READABILITY = 2;
  static charsets = {
    1: 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789',
    2: 'ABCDEFHKLMNPQRSTUWXYZ0123456789'
  };
  static getRandomHtmlId() {
    return 'id-' + FramelixRandom.getRandomString(10, 13);
  }
  static getRandomString(minLength) {
    let maxLength = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let charset = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : FramelixRandom.CHARSET_ALPHANUMERIC;
    charset = FramelixRandom.charsets[charset] || charset;
    const charsetLength = charset.length;
    maxLength = maxLength !== null ? maxLength : minLength;
    const useLength = FramelixRandom.getRandomInt(minLength, maxLength);
    let str = '';
    for (let i = 1; i <= useLength; i++) {
      str += charset[FramelixRandom.getRandomInt(0, charsetLength - 1)];
    }
    return str;
  }
  static getRandomInt(min, max) {
    const randomBuffer = new Uint32Array(1);
    window.crypto.getRandomValues(randomBuffer);
    let randomNumber = randomBuffer[0] / (0xffffffff + 1);
    min = Math.ceil(min);
    max = Math.floor(max);
    return Math.floor(randomNumber * (max - min + 1)) + min;
  }
}
;
;
class FramelixRequest {
  requestOptions = null;
  progressCallback = null;
  submitRequest;
  finished;
  _responseJson;
  static renderFromRequestOptions(requestOptions, initiatorElement, postData, progressCallback) {
    if (requestOptions.renderTarget && (requestOptions.renderTarget.newTab || requestOptions.renderTarget.selfTab)) {
      const link = $('<a>').attr('href', requestOptions.url).attr('target', requestOptions.renderTarget.newTab ? '_blank' : '_self');
      link.css('display', 'hidden');
      $('body').append();
      link.trigger('click');
      if (progressCallback && requestOptions.renderTarget.newTab) {
        progressCallback(1, null);
      }
      setTimeout(function () {
        link.remove();
      }, 1000);
      return null;
    }
    let method = 'get';
    if (postData || requestOptions.url.includes('/jscv?method=')) {
      method = 'post';
    }
    const request = FramelixRequest.request(method, requestOptions.url, null, postData);
    request.requestOptions = requestOptions;
    request.progressCallback = progressCallback;
    FramelixRequest.renderResponse(request, requestOptions, initiatorElement);
    return request;
  }
  static jsCall(signedUrl, parameters, showProgressBar) {
    const postData = parameters instanceof FormData ? parameters : JSON.stringify(parameters);
    let request;
    request = FramelixRequest.request('post', signedUrl, null, postData, showProgressBar);
    return request;
  }
  static request(method, urlPath, urlParams, postData) {
    let showProgressBar = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : false;
    let fetchOptions = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : null;
    let instance = new FramelixRequest();
    if (typeof urlParams !== 'undefined' && urlParams !== null) {
      if (!urlPath.match(/\?/)) {
        urlPath += '?';
      } else {
        urlPath += '&';
      }
      urlPath += FramelixObjectUtils.toUrlencodedString(urlParams);
    }
    let body = postData;
    if (typeof postData !== 'undefined' && postData !== null) {
      if (typeof postData === 'object' && !(postData instanceof FormData)) {
        body = FramelixRequest.objectToFormData(postData);
      }
    }
    if (!fetchOptions) {
      fetchOptions = {};
    }
    instance.finished = new Promise(function (resolve) {
      instance.submitRequest = new XMLHttpRequest();
      instance.submitRequest.open(method.toUpperCase(), urlPath, true, fetchOptions.username || null, fetchOptions.password || null);
      instance.submitRequest.setRequestHeader('x-requested-with', 'xmlhttprequest');
      instance.submitRequest.setRequestHeader('Cache-Control', 'no-store');
      instance.submitRequest.setRequestHeader('x-browser-url', window.location.href);
      if (typeof body === 'string') {
        instance.submitRequest.setRequestHeader('content-type', 'application/json');
      }
      instance.submitRequest.responseType = 'blob';
      if (fetchOptions.headers) {
        for (let k in fetchOptions.headers) {
          instance.submitRequest.setRequestHeader(k, fetchOptions.headers[k]);
        }
      }
      instance.submitRequest.upload.addEventListener('progress', function (ev) {
        const loaded = 1 / ev.total * ev.loaded;
        if (showProgressBar) {
          Framelix.showProgressBar(loaded, showProgressBar !== true ? showProgressBar : null);
        }
        if (instance.progressCallback) {
          instance.progressCallback(loaded, ev);
        }
      });
      instance.submitRequest.addEventListener('load', async function (ev) {
        if (instance.progressCallback) {
          instance.progressCallback(1, ev);
        }
        resolve();
      });
      instance.submitRequest.addEventListener('error', function (ev) {
        console.error(ev);
        instance.progressCallback(1, ev);
        resolve();
      });
      instance.submitRequest.send(body);
    });
    instance.finished.then(function () {
      if (showProgressBar) {
        Framelix.showProgressBar(null, showProgressBar !== true ? showProgressBar : null);
      }
    });
    return instance;
  }
  static objectToFormData(obj, formData, parentKey) {
    if (!formData) {
      formData = new FormData();
    }
    if (obj) {
      for (let i in obj) {
        let v = obj[i];
        let k = parentKey ? parentKey + '[' + i + ']' : i;
        if (v !== null && v !== undefined) {
          if (typeof v === 'object') {
            FramelixRequest.objectToFormData(v, formData, parentKey);
          } else {
            formData.append(k, v);
          }
        }
      }
    }
    return formData;
  }
  static async renderResponse(request, requestOptions, initiatorElement, overrideResponse) {
    if (typeof requestOptions.renderTarget === 'string') {
      if (requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_MODAL_NEW) {
        requestOptions.renderTarget = {
          modalOptions: {}
        };
      } else if (requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_POPUP) {
        requestOptions.renderTarget = {
          popupOptions: {
            closeMethods: 'click-outside'
          }
        };
      } else if (requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_CURRENT_CONTEXT) {
        requestOptions.renderTarget = requestOptions.renderTarget = {
          modalOptions: {}
        };
        if (!initiatorElement) {
          console.error('Missing initiatorElement for render to current context');
          return null;
        }
        initiatorElement = $(initiatorElement);
        let validTarget = false;
        if (!validTarget) {
          let responseReceiver = initiatorElement.closest('td');
          if (responseReceiver.length) {
            validTarget = true;
            requestOptions.renderTarget = {
              elementSelector: responseReceiver
            };
          }
        }
        if (!validTarget) {
          let responseReceiver = initiatorElement.closest('.framelix-popup');
          if (responseReceiver.length) {
            validTarget = true;
            requestOptions.renderTarget = {
              popupOptions: responseReceiver[0].framelixPopupInstance.options
            };
            initiatorElement = responseReceiver[0].framelixPopupInstance.target;
          }
        }
        if (!validTarget) {
          let responseReceiver = initiatorElement.closest('.framelix-modal');
          if (responseReceiver.length) {
            validTarget = true;
            const modal = FramelixModal.instances[responseReceiver.attr('data-instance-id')];
            if (modal) {
              requestOptions.renderTarget = {
                modalOptions: {
                  instance: modal
                }
              };
            }
          }
        }
        if (!validTarget) {
          let responseReceiver = initiatorElement.closest('[data-request-response-receiver]');
          if (responseReceiver.length) {
            validTarget = true;
            requestOptions.renderTarget = {
              elementSelector: responseReceiver
            };
          }
        }
      }
    }
    const isFixedResponse = !(request instanceof FramelixRequest);
    if (!requestOptions.renderTarget || requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_NONE_AND_CLOSE) {
      if (!isFixedResponse) {
        Framelix.showProgressBar(1);
        request.checkHeaders().then(function () {
          Framelix.showProgressBar(null);
          if (requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_NONE_AND_CLOSE) {
            FramelixPopup.destroyAll();
            FramelixModal.destroyAll();
          }
        });
      } else if (requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_NONE_AND_CLOSE) {
        FramelixPopup.destroyAll();
        FramelixModal.destroyAll();
      }
    } else if (requestOptions.renderTarget.modalOptions) {
      let options = requestOptions.renderTarget.modalOptions;
      options.bodyContent = request;
      await FramelixModal.show(options).created;
    } else if (requestOptions.renderTarget.popupOptions) {
      let options = requestOptions.renderTarget.popupOptions;
      await FramelixPopup.show(initiatorElement, request, options).created;
    } else if (requestOptions.renderTarget.elementSelector) {
      const el = $(requestOptions.renderTarget.elementSelector);
      el.html(`<div class="framelix-loading"></div>`);
      el.html(isFixedResponse ? request : await request.getJson());
    }
  }
  abort() {
    if (this.submitRequest.readyState !== this.submitRequest.DONE) {
      this.submitRequest.abort();
    }
  }
  async checkHeaders() {
    let dispositionHeader = await this.getHeader('content-disposition');
    if (dispositionHeader) {
      let attachmentMatch = dispositionHeader.match(/(attachment|inline)\s*;\s*filename\s*=["'](.*?)["']/);
      if (attachmentMatch) {
        Framelix.downloadBlobAsFile(await this.getBlob(), attachmentMatch[2]);
        return 1;
      }
    }
    let redirectHeader = await this.getHeader('x-redirect');
    if (redirectHeader) {
      Framelix.redirect(redirectHeader);
      return 2;
    }
    if (this.submitRequest.status >= 400) {
      FramelixModal.show({
        bodyContent: await this.getResponseData(false)
      });
      return 3;
    }
    return 0;
  }
  async writeToContainer(container) {
    let showLoadingBar = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
    let checkHeaders = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
    if (showLoadingBar) {
      container.html(`<div class="framelix-loading"></div>`);
    }
    const responseData = await this.getResponseData(checkHeaders);
    container.html(responseData);
  }
  async getResponseData() {
    let checkHeaders = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;
    await this.finished;
    const self = this;
    return new Promise(async function (resolve) {
      if (checkHeaders && (await self.checkHeaders()) !== 0) {
        return;
      }
      if ((await self.getHeader('content-type')) === 'application/json') {
        resolve(self.getJson());
      }
      return resolve(self.getText());
    });
  }
  async getBlob() {
    await this.finished;
    return this.submitRequest.response;
  }
  async getText() {
    await this.finished;
    return this.submitRequest.response.text();
  }
  async getJson() {
    await this.finished;
    if (typeof this._responseJson === 'undefined') {
      try {
        this._responseJson = JSON.parse(await this.getText());
      } catch (e) {}
    }
    return this._responseJson;
  }
  async getHeader(key) {
    await this.finished;
    return this.submitRequest.getResponseHeader(key);
  }
}
;
;
class FramelixResizeObserver {
  static observer = null;
  static observedElements = [];
  static rectMap = new Map();
  static observe(element, callback) {
    let box = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 'content-box';
    if (!FramelixResizeObserver.observer) FramelixResizeObserver.init();
    element = $(element)[0];
    FramelixResizeObserver.observedElements.push([element, callback]);
    FramelixResizeObserver.observer.observe(element, {
      'box': box
    });
  }
  static unobserve(element) {
    if (!FramelixResizeObserver.observer) FramelixResizeObserver.init();
    element = $(element)[0];
    let removeIndex = null;
    for (let i = 0; i < FramelixResizeObserver.observedElements.length; i++) {
      if (FramelixResizeObserver.observedElements[i][0] === element) {
        removeIndex = i;
        break;
      }
    }
    if (removeIndex !== null) {
      FramelixResizeObserver.observedElements.splice(removeIndex, 1);
    }
    FramelixResizeObserver.observer.unobserve(element);
  }
  static init() {
    let observerTimeout = null;
    let consecutiveLoads = 0;
    let lastChange = new Date().getTime();
    FramelixResizeObserver.observer = new ResizeObserver(function (observerEntries) {
      observerEntries.forEach(function (observerEntry) {
        for (let i = 0; i < FramelixResizeObserver.observedElements.length; i++) {
          if (FramelixResizeObserver.observedElements[i][0] === observerEntry.target) {
            FramelixResizeObserver.observedElements[i][1](observerEntry.contentRect);
            break;
          }
        }
      });
      clearTimeout(observerTimeout);
      observerTimeout = setTimeout(function () {
        let currentTime = new Date().getTime();
        if (currentTime - lastChange <= 70) {
          consecutiveLoads++;
        } else {
          consecutiveLoads = 0;
        }
        lastChange = currentTime;
        if (consecutiveLoads > 10) {
          console.warn('Framework->FramelixResizeObserver: ResizeObserver detected ' + consecutiveLoads + ' consecutiveLoads of a period of  ' + consecutiveLoads * 50 + 'ms - Maybe this point to a loop in dom resize changes');
        }
      }, 50);
    });
  }
  static legacyResizeInterval() {
    for (let i = 0; i < FramelixResizeObserver.observedElements.length; i++) {
      const el = FramelixResizeObserver.observedElements[i];
      const boundingBox = el[0].getBoundingClientRect();
      if (FramelixResizeObserver.rectMap.get(el[0]) !== JSON.stringify(boundingBox)) {
        FramelixResizeObserver.rectMap.set(el[0], JSON.stringify(boundingBox));
        el[1](boundingBox);
      }
    }
  }
}
;
;
class FramelixStringUtils {
  static jsonStringify(data) {
    let prettyPrint = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let convertSpecialChars = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    let str = JSON.stringify(data, null, prettyPrint ? 2 : null);
    if (convertSpecialChars) {
      str = str.replace(/\\"/g, '\\u0022');
      str = str.replace(/'/g, '\\u0027');
      str = str.replace(/</g, '\\u003C');
      str = str.replace(/>/g, '\\u003E');
      str = str.replace(/&/g, '\\u0026');
    }
    return str;
  }
  static slugify(str) {
    let replaceSpaces = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
    let replaceDots = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : true;
    let replaceRegex = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : /[^a-z0-9. \-_]/ig;
    let s = ['Ö', 'Ü', 'Ä', 'ö', 'ü', 'ä', 'ß'];
    let r = ['Oe', 'Ue', 'Ae', 'oe', 'ue', 'ae', 'ss'];
    if (replaceSpaces) {
      s.push(' ');
      r.push('-');
    }
    if (replaceDots) {
      s.push('.');
      r.push('-');
    }
    for (let i = 0; i < s.length; i++) {
      str = str.replace(new RegExp(FramelixStringUtils.escapeRegex(s[i]), 'g'), r[i]);
    }
    str = str.replace(replaceRegex, '-');
    return str.replace(/-{2,99}/i, '');
  }
  static stringify(value) {
    let concatChar = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : ', ';
    if (value === null || value === undefined) {
      return '';
    }
    if (typeof value === 'object') {
      const arr = [];
      for (let i in value) {
        arr.push(FramelixStringUtils.stringify(value[i], concatChar));
      }
      return arr.join(concatChar);
    }
    if (typeof value === 'boolean') {
      return value ? '1' : '0';
    }
    if (typeof value !== 'string') {
      return value.toString();
    }
    return value;
  }
  static replace(search, replaceWith, str) {
    return (str + '').replace(new RegExp(FramelixStringUtils.escapeRegex(search), 'i'), replaceWith);
  }
  static htmlEscape(str) {
    let nl2br = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    str = (str + '').replace(/&/g, '&amp;').replace(/'/g, '&apos;').replace(/"/g, '&quot;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
    if (nl2br) str = str.replace(/\n/g, '<br/>');
    return str;
  }
  static escapeRegex(str) {
    return (str + '').replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
  }
  static cut(string, length) {
    let truncateAffix = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '...';
    if (string.length <= length) {
      return string;
    }
    return string.substring(0, length) + truncateAffix;
  }
  static strToHex(str) {
    return str.split('').map(x => x.charCodeAt(0).toString(16).padStart(2, '0')).join('');
  }
  static hexToStr(str) {
    return str.match(/.{1,2}/g).map(x => String.fromCharCode(parseInt(x, 16))).join('');
  }
}
;
;
class FramelixTableCell {
  stringValue = null;
  sortValue = null;
}
;
;
self.onmessage = function (ev) {
  const data = ev.data;
  const rows = data.rows || [];
  rows.sort(function (a, b) {
    for (let i = 0; i < data.sortSettings.length; i++) {
      let sortDirection = data.sortSettings[i].substr(0, 1) === '+' ? 1 : -1;
      let sortValueA = a.sortValues[i];
      let sortValueB = b.sortValues[i];
      if (sortValueA === sortValueB) continue;
      if (sortValueA === '') return sortDirection;
      if (sortValueB === '') return sortDirection * -1;
      if (sortValueA > sortValueB) return sortDirection;
      if (sortValueA < sortValueB) return sortDirection * -1;
    }
    return 0;
  });
  const indexes = [];
  for (let i = 0; i < rows.length; i++) {
    indexes.push(rows[i].rowIndex);
  }
  self.postMessage(indexes);
};
;
class FramelixTable {
  static EVENT_COLUMNSORT_SORT_CHANGED = 'framelix-table-columnsort-sort-changed';
  static EVENT_DRAGSORT_SORT_CHANGED = 'framelix-table-dragsort-sort-changed';
  static EVENT_SORT_CHANGED = 'framelix-table-sort-changed';
  static COLUMNFLAG_DEFAULT = 'default';
  static COLUMNFLAG_ICON = 'icon';
  static COLUMNFLAG_SMALLWIDTH = 'smallwidth';
  static COLUMNFLAG_SMALLFONT = 'smallfont';
  static COLUMNFLAG_IGNORESORT = 'ignoresort';
  static COLUMNFLAG_IGNOREURL = 'ignoreurl';
  static COLUMNFLAG_REMOVE_IF_EMPTY = 'removeifempty';
  static instances = [];
  static sorterWorker = null;
  container;
  table;
  id;
  rows = {};
  columnOrder = [];
  columnFlags = {};
  sortable = true;
  initialSort = null;
  rememberSort = true;
  checkboxColumn = false;
  dragSort = false;
  storableSort = false;
  storableSortJsCallUrl = null;
  storableDeletable = true;
  urlOpenInNewTab = false;
  currentSort = null;
  prependHtml = null;
  appendHtml = null;
  escapeHtml = true;
  rowUrlOpenMethod = 'default';
  rendered;
  _renderedResolve;
  static getById(id) {
    for (let i = 0; i < FramelixTable.instances.length; i++) {
      if (FramelixTable.instances[i].id === id) {
        return FramelixTable.instances[i];
      }
    }
    return null;
  }
  constructor() {
    const self = this;
    this.rendered = new Promise(function (resolve) {
      self._renderedResolve = resolve;
    });
    this.id = 'table-' + FramelixRandom.getRandomHtmlId();
    FramelixTable.instances.push(this);
    this.container = $('<div>');
    this.container.addClass('framelix-table');
    this.container.attr('data-instance-id', FramelixTable.instances.length - 1);
  }
  async sort() {
    const self = this;
    self.updateHeaderSortingInfoLabels();
    self.currentSort = self.currentSort || FramelixLocalStorage.get('framelix-table-user-sort') || self.initialSort;
    if (!self.currentSort) return;
    const thead = self.table ? self.table.children('thead') : null;
    if (thead) {
      thead.addClass('framelix-pulse');
    }
    return new Promise(function (resolve) {
      if (!FramelixTable.sorterWorker) {
        FramelixTable.sorterWorker = new Worker(FramelixConfig.compiledFileUrls['Framelix']['js']['table-sorter-serviceworker']);
      }
      FramelixTable.sorterWorker.onmessage = async function (e) {
        if (self.rows.tbody) {
          const newRows = [];
          for (let i = 0; i < e.data.length; i++) {
            let rowIndex = e.data[i];
            newRows.push(self.rows.tbody[rowIndex]);
          }
          self.rows.tbody = newRows;
          self.updateTbodyDomSort();
          if (self.table) {
            const tbody = self.table.children('tbody')[0];
            for (let i = 0; i < self.rows.tbody.length; i++) {
              const el = self.rows.tbody[i].el;
              if (el) tbody.appendChild(el);
            }
          }
        }
        if (thead) {
          thead.removeClass('framelix-pulse');
        }
        resolve();
      };
      let rows = [];
      if (self.rows.tbody) {
        for (let j = 0; j < self.rows.tbody.length; j++) {
          let sortValues = [];
          const row = self.rows.tbody[j];
          for (let i = 0; i < self.currentSort.length; i++) {
            let sortCellName = self.currentSort[i].substr(1);
            let sortValue = row['sortValues'][sortCellName];
            if (sortValue === null || sortValue === undefined) {
              sortValue = row['cellValues'][sortCellName];
            }
            if (sortValue instanceof FramelixTableCell) {
              sortValue = sortValue.sortValue;
            }
            sortValues.push(sortValue);
          }
          rows.push({
            'rowIndex': j,
            'sortValues': sortValues
          });
        }
      }
      FramelixTable.sorterWorker.postMessage({
        'sortSettings': self.currentSort,
        'rows': rows
      });
    });
  }
  updateHeaderSortingInfoLabels() {
    const theadCells = this.table ? this.table.children('thead').children().first().children('th') : null;
    if (theadCells) {
      theadCells.find('.framelix-table-header-sort-info-number, .framelix-table-header-sort-info-text').empty();
      if (this.currentSort) {
        for (let i = 0; i < this.currentSort.length; i++) {
          const dir = this.currentSort[i].substr(0, 1);
          const cellName = this.currentSort[i].substr(1);
          const cell = theadCells.filter('[data-column-name=\'' + CSS.escape(cellName) + '\']');
          cell.addClass('framelix-table-header-sort');
          cell.html(`<div class="framelix-table-header-sort-info-number">${i + 1}</div><div class="framelix-table-header-sort-info-text">${dir === '+' ? 'A-Z' : 'Z-A'}</div>`);
        }
      }
    }
  }
  updateTbodyDomSort() {
    if (!this.table || !this.rows.tbody) return false;
    let sortHasChanged = false;
    const tbody = this.table.children('tbody')[0];
    const childs = Array.from(tbody.children);
    for (let i = 0; i < this.rows.tbody.length; i++) {
      const el = this.rows.tbody[i].el;
      if (!sortHasChanged && i !== childs.indexOf(el)) {
        sortHasChanged = true;
      }
      if (el) tbody.appendChild(el);
    }
    if (sortHasChanged) {
      this.table.trigger(FramelixTable.EVENT_COLUMNSORT_SORT_CHANGED);
    }
  }
  async openRowUrl(url, forceMethod) {
    let method = forceMethod || this.rowUrlOpenMethod;
    switch (method) {
      case 'newmodal':
      case 'currentmodal':
        {
          let modal;
          if (method === 'currentmodal') {
            const modalContainer = this.container.closest('.framelix-modal');
            if (!modalContainer.length) {
              window.location.href = url;
              return;
            }
            modal = FramelixModal.instances[modalContainer.attr('data-instance-id')];
          }
          const result = await FramelixRequest.request('get', url, null, null, true);
          if (modal) {
            modal.bodyContainer.html((await result.getJson()).content || '');
          } else {
            FramelixModal.request('get', url, null, null, true);
          }
        }
        break;
      case 'currenttab':
        {
          const tab = this.container.closest('.framelix-tab-content');
          if (!tab.length) {
            window.location.href = url;
            return;
          }
          const tabData = FramelixTabs.instances[tab.closest('.framelix-tabs').attr('data-instance-id')].tabs[tab.attr('data-id')];
          if (tabData && tabData.content instanceof FramelixView) {
            tabData.content.url = url;
          }
          const result = await FramelixRequest.request('get', url, null, null, true);
          tab.html((await result.getJson()).content || '');
        }
        break;
      case 'default':
        window.location.href = url;
        break;
      case 'newwindow':
        window.open(url);
        break;
    }
  }
  async render() {
    const self = this;
    if (this.sortable) {
      if (this.rememberSort) {
        const rememberedSort = FramelixLocalStorage.get(this.id + '-table-sort');
        if (rememberedSort) {
          this.initialSort = rememberedSort;
        }
      }
      this.sort();
    }
    let tableHtml = '';
    if (this.prependHtml) tableHtml = this.prependHtml;
    tableHtml += `<table id="${this.id}">`;
    let canDragSort = this.dragSort && FramelixObjectUtils.hasKeys(this.rows.tbody, 2);
    let removeEmptyCells = {};
    for (let i in this.columnFlags) {
      for (let j in this.columnFlags[i]) {
        if (this.columnFlags[i][j] === FramelixTable.COLUMNFLAG_REMOVE_IF_EMPTY) {
          removeEmptyCells[i] = true;
        }
      }
    }
    removeEmptyCells['_deletable'] = true;
    if (this.sortable && this.rows['thead'] && this.rows['thead'][0]) {
      const row = Object.assign({}, this.rows['thead'][0]);
      row.cellValues = {};
      row.htmlAttributes = new FramelixHtmlAttributes();
      row.htmlAttributes.addClass('framelix-table-row-sort-info');
      this.rows['thead'].unshift(row);
    }
    const convertValue = async function (level, cellValue, rowGroup, columnName, cellAttributes, removeEmptyCells) {
      if (cellValue === '' || cellValue === null || cellValue === undefined) {
        cellValue = '';
      }
      if (cellValue instanceof FramelixTableCell) {
        cellValue = cellValue.stringValue;
      } else if (typeof cellValue === 'object') {
        let str = '';
        for (let i in cellValue) {
          str += '<div class="array-value" data-key="' + FramelixStringUtils.htmlEscape(i) + '">' + (await convertValue(level + 1, cellValue[i], rowGroup, columnName, cellAttributes, removeEmptyCells)) + '</div>';
        }
        cellValue = str;
      } else {
        var _cellValue;
        cellValue = rowGroup === 'thead' ? await FramelixLang.get(cellValue) : cellValue;
        if (typeof cellValue !== 'string') cellValue = ((_cellValue = cellValue) !== null && _cellValue !== void 0 ? _cellValue : '').toString();
        if (self.escapeHtml === true || Array.isArray(self.escapeHtml) && self.escapeHtml.indexOf(columnName) > -1) {
          cellValue = FramelixStringUtils.htmlEscape(cellValue).replace(/\n/g, '<br/>');
        }
      }
      if (level === 0) {
        if (rowGroup === 'thead') cellValue = `<div class="framelix-tableheader-text">${cellValue}</div>`;
        if (self.sortable && rowGroup === 'thead' && !cellAttributes.get('data-flag-ignoresort')) {
          cellAttributes.set('tabindex', '0');
        }
        if (rowGroup === 'thead') cellValue = `<div class="framelix-table-cell-header">${cellValue}</div>`;
        if (removeEmptyCells[columnName] && cellValue !== '' && rowGroup === 'tbody') {
          removeEmptyCells[columnName] = false;
        }
      }
      return cellValue;
    };
    for (let rowGroup in this.rows) {
      tableHtml += `<${rowGroup}>`;
      const cellType = rowGroup === 'thead' ? 'th' : 'td';
      for (let i = 0; i < this.rows[rowGroup].length; i++) {
        const row = this.rows[rowGroup][i];
        if (!row.cellValues) row.cellValues = {};
        let rowAttributes = row.htmlAttributes || new FramelixHtmlAttributes();
        rowAttributes.set('data-row-key', i);
        if (rowAttributes.get('data-url')) {
          rowAttributes.set('tabindex', '0');
          if (rowAttributes.get('data-url').replace(/\#.*/g, '') === window.location.href.replace(/\#.*/g, '')) {
            rowAttributes.addClass('framelix-table-row-highlight');
          }
        }
        tableHtml += '<tr ' + rowAttributes.toString() + '>';
        if (canDragSort) {
          let cellAttributes = new FramelixHtmlAttributes();
          cellAttributes.setStyle('width', '0%');
          cellAttributes.set('data-column-name', '_dragsort');
          cellAttributes.set('data-flag-ignoresort', '1');
          cellAttributes.set('data-flag-ignoreurl', '1');
          cellAttributes.set('data-flag-icon', '1');
          let cellValue = '';
          if (rowGroup === 'tbody') {
            cellValue = '<framelix-button icon="70f" title="__framelix_table_dragsort__"></framelix-button>';
          }
          if (rowGroup === 'thead') cellValue = `<div class="framelix-table-cell-header">${cellValue}</div>`;
          tableHtml += '<' + cellType + ' ' + cellAttributes.toString() + '>';
          tableHtml += cellValue;
          tableHtml += '</' + cellType + '>';
        }
        if (this.checkboxColumn) {
          let cellAttributes = new FramelixHtmlAttributes();
          cellAttributes.set('data-column-name', '_checkbox');
          cellAttributes.set('data-flag-ignoresort', '1');
          cellAttributes.set('data-flag-ignoreurl', '1');
          cellAttributes.set('data-flag-icon', '1');
          let cellValue = '<framelix-button theme="transparent"><input type="checkbox" name="_checkbox" value="' + i + '"></framelix-button>';
          if (rowGroup === 'thead') cellValue = `<div class="framelix-table-cell-header">${cellValue}</div>`;
          if (rowGroup === 'thead' && i !== 0) cellValue = '';
          tableHtml += '<' + cellType + ' ' + cellAttributes.toString() + '>';
          tableHtml += cellValue;
          tableHtml += '</' + cellType + '>';
        }
        for (let j = 0; j < this.columnOrder.length; j++) {
          const columnName = this.columnOrder[j];
          let cellAttributes = row.cellAttributes ? row.cellAttributes[columnName] : null;
          if (!cellAttributes) cellAttributes = new FramelixHtmlAttributes();
          cellAttributes.set('data-column-name', columnName);
          if (this.columnFlags[columnName]) {
            for (let i in this.columnFlags[columnName]) {
              const flag = this.columnFlags[columnName][i];
              cellAttributes.set('data-flag-' + flag, '1');
            }
          }
          let cellValue = row.cellValues[columnName] || '';
          if (typeof cellValue === 'string' && cellValue.startsWith('<framelix-button')) {
            cellAttributes.set('data-flag-ignoresort', '1');
            cellAttributes.set('data-flag-ignoreurl', '1');
            cellAttributes.set('data-flag-icon', '1');
          }
          cellValue = await convertValue(0, cellValue, rowGroup, columnName, cellAttributes, removeEmptyCells);
          tableHtml += '<' + cellType + ' ' + cellAttributes.toString() + '>';
          tableHtml += cellValue;
          tableHtml += '</' + cellType + '>';
        }
        tableHtml += '</tr>';
      }
      tableHtml += `</${rowGroup}>`;
    }
    if (this.appendHtml) tableHtml += this.appendHtml;
    self.container[0].innerHTML = tableHtml;
    self.table = self.container.children('table');
    for (let columnName in removeEmptyCells) {
      if (removeEmptyCells[columnName]) {
        self.table.children().children('tr').children('td,th').filter('[data-column-name=\'' + columnName + '\']').remove();
      }
    }
    const tbody = this.table.children('tbody')[0];
    if (tbody) {
      for (let i = 0; i < tbody.childNodes.length; i++) {
        this.rows.tbody[i].el = tbody.childNodes[i];
      }
    }
    this.updateHeaderSortingInfoLabels();
    if (this.checkboxColumn) {
      self.table.on('change', 'thead th[data-column-name=\'_checkbox\'] input', function () {
        $(tbody).children().children('td[data-column-name=\'_checkbox\']').find('input').prop('checked', this.checked);
      });
    }
    let mouseDownRow = null;
    self.table.on('keydown', 'tr[data-url]', function (ev) {
      if (ev.key === 'Enter') {
        ev.preventDefault();
        let url = $(this).attr('data-url');
        if (ev.ctrlKey && self.rowUrlOpenMethod === 'default') {
          self.openRowUrl(url, 'newwindow');
        } else {
          self.openRowUrl(url);
        }
      }
    });
    self.table.on('mousedown', 'tr[data-url]', function (ev) {
      mouseDownRow = this;
      if (ev.which === 2) {
        ev.preventDefault();
      }
    });
    self.table.on('mouseup', 'tr[data-url]', function (ev) {
      if (mouseDownRow !== this) return;
      if (ev.which && ev.which > 2) return;
      const newTab = ev.which === 2 || self.urlOpenInNewTab;
      let target = $(ev.target);
      let url = $(this).attr('data-url');
      if (target.is('a') || target.is('input,select,textarea') || target.attr('data-flag-ignoreurl') === '1' || target.attr('onclick') || target.closest('a').length || target.closest('td').attr('data-flag-ignoreurl') === '1') {
        return;
      }
      if (!ev.touches && !newTab && window.getSelection().toString().length > 0) {
        return;
      }
      if (newTab && self.rowUrlOpenMethod === 'default') {
        self.openRowUrl(url, 'newwindow');
        return;
      }
      self.openRowUrl(url);
    });
    this.table.on(FramelixTable.EVENT_DRAGSORT_SORT_CHANGED + ' ' + FramelixTable.EVENT_COLUMNSORT_SORT_CHANGED, async function () {
      self.table.trigger(FramelixTable.EVENT_SORT_CHANGED);
      if (self.storableSort) {
        if (self.container.children('.framelix-table-savesort').length) return;
        FramelixToast.info(await FramelixLang.get('__framelix_table_savesort_tooltip__'));
        const btn = $(`<framelix-button theme="primary" class="framelix-table-savesort" icon="718">__framelix_table_savesort__</framelix-button>`);
        self.container.append(btn);
        btn.on('click', async function () {
          Framelix.showProgressBar(1);
          btn.addClass('framelix-pulse').attr('disabled', true);
          let data = [];
          self.table.children('tbody').children().each(function () {
            const row = [this.getAttribute('data-id')];
            const connId = this.getAttribute('data-connection-id');
            if (connId) row.push(connId);
            data.push(row);
          });
          const apiResult = await FramelixRequest.jsCall(self.storableSortJsCallUrl, {
            'data': data
          }).getResponseData();
          Framelix.showProgressBar(null);
          if (apiResult === true) {
            btn.remove();
            FramelixToast.success('__framelix_table_savesort_saved__');
          }
        });
      }
    });
    if (canDragSort) {
      FramelixDom.includeCompiledFile('Framelix', 'js', 'sortablejs', 'Sortable').then(function () {
        new Sortable(self.table.children('tbody')[0], {
          'handle': 'td[data-column-name=\'_dragsort\']',
          'onSort': function () {
            self.table.trigger(FramelixTable.EVENT_DRAGSORT_SORT_CHANGED);
          }
        });
      });
    }
    if (this.sortable) {
      self.container.addClass('framelix-table-sortable');
      self.table.on('click keydown', 'th:not([data-flag-ignoresort])', async function (ev) {
        if (ev.type === 'keydown') {
          if (ev.key !== ' ') {
            return;
          }
          ev.preventDefault();
        }
        if (ev.ctrlKey) {
          self.currentSort = null;
          FramelixLocalStorage.remove(self.id + '-table-sort');
          if (self.rows.tbody) {
            self.rows.tbody.sort(function (a, b) {
              return a.rowKeyInitial > b.rowKeyInitial ? 1 : -1;
            });
            self.updateHeaderSortingInfoLabels();
            self.updateTbodyDomSort();
          }
          return;
        }
        if (!self.currentSort) self.currentSort = [];
        const cellName = $(this).attr('data-column-name');
        let flippedCell = null;
        for (let i = 0; i < ((_self$currentSort = self.currentSort) === null || _self$currentSort === void 0 ? void 0 : _self$currentSort.length); i++) {
          var _self$currentSort;
          const sortCellName = self.currentSort[i].substr(1);
          if (sortCellName === cellName) {
            flippedCell = (self.currentSort[i].substr(0, 1) === '+' ? '-' : '+') + sortCellName;
            self.currentSort[i] = flippedCell;
            break;
          }
        }
        if (!ev.shiftKey) {
          self.currentSort = [flippedCell || '+' + cellName];
        } else if (!flippedCell) {
          self.currentSort.push('+' + cellName);
        }
        FramelixLocalStorage.set(self.id + '-table-sort', self.currentSort);
        self.sort();
      });
    }
    if (this._renderedResolve) {
      this._renderedResolve();
      this._renderedResolve = null;
    }
  }
}
;
;
class FramelixTabs {
  static EVENT_TAB_CONTENT_RENDERED = 'framelix-tabs-rendered';
  static instances = [];
  container;
  id;
  tabs = {};
  buttonContainer;
  contentContainer;
  activeTab = null;
  static init() {
    $(window).on('hashchange', function () {
      const currentPath = location.hash.substr(1);
      for (let i = 0; i < FramelixTabs.instances.length; i++) {
        const instance = FramelixTabs.instances[i];
        const path = instance.getFullPath();
        if (currentPath.startsWith(path + ':')) {
          let tabId = currentPath.substr(path.length + 1).split(',')[0];
          instance.setActiveTab(tabId);
        }
      }
    });
  }
  constructor() {
    FramelixTabs.instances.push(this);
    this.container = $('<div>');
    this.container.addClass('framelix-tabs');
    this.container.attr('data-instance-id', FramelixTabs.instances.length - 1);
  }
  getFullPath(addButtonId) {
    let path = [this.id];
    let parent = this.container;
    while (true) {
      parent = parent.parent().closest('.framelix-tab-content');
      if (!parent.length) break;
      path.push(parent.closest('.framelix-tabs').attr('data-id') + ':' + parent.attr('data-id'));
    }
    path.reverse();
    return path.join(',') + (addButtonId ? ':' + addButtonId : '');
  }
  setActiveTab(id) {
    this.activeTab = id;
    if (this.tabs[id] === undefined) return;
    FramelixLocalStorage.set('tabs-active-' + location.pathname + '-' + this.id, this.getFullPath(id));
    const buttons = this.buttonContainer.children();
    const contents = this.contentContainer.children();
    buttons.attr('data-active', '0');
    contents.attr('data-active', '0');
    this.tabs[id].buttonContainer.attr('data-active', '1');
    this.tabs[id].contentContainer.attr('data-active', '1');
  }
  async reloadTab(tabId) {
    const row = this.tabs[tabId];
    if (!row) return;
    const content = $(`<div class="framelix-tab-content"></div>`);
    content.attr('data-id', tabId);
    if (row.optionalContentAttributes) row.optionalContentAttributes.assignToElement(content);
    if (row.content instanceof FramelixView) {
      row.content.urlParameters = row.urlParameters;
      content.html(row.content.container);
      row.content.load();
    } else if (row.url) {
      let request = FramelixRequest.request('get', row.url, row.urlParameters, null, content);
      if ((await request.checkHeaders()) === 0) {
        content.html((await request.getJson()).content);
      }
    } else if (typeof row.content === 'function') {
      content.html(await row.content());
    } else {
      content.html(row.content);
    }
    row.contentContainer = content;
    this.contentContainer.children('[data-id=\'' + tabId + '\']').replaceWith(content);
    if (this.activeTab === tabId) {
      this.setActiveTab(tabId);
    }
    $(document).trigger(FramelixTabs.EVENT_TAB_CONTENT_RENDERED, [this, row]);
  }
  addTab(id, label, content, tabColor) {
    this.tabs[id] = {
      'id': id,
      'label': label,
      'content': content,
      'tabColor': tabColor
    };
  }
  async render() {
    const self = this;
    const basePath = this.getFullPath();
    let matchedHashActiveTabId = null;
    let matchedStoredActiveTabId = null;
    let storedActiveTabId = FramelixLocalStorage.get('tabs-active-' + location.pathname + '-' + this.id);
    let hashTabId = location.hash.substring(1);
    this.buttonContainer = $(`<div class="framelix-tab-buttons"></div>`);
    this.contentContainer = $(`<div class="framelix-tab-contents"></div>`);
    let firstTabId = null;
    let count = 0;
    for (let tabId in this.tabs) {
      const row = this.tabs[tabId];
      if (firstTabId === null) firstTabId = tabId;
      const fullPath = basePath + ':' + tabId;
      const btn = $(`<framelix-button class="framelix-tab-button" theme="light" icon="705"></framelix-button>`);
      btn.attr('data-id', tabId);
      if (row.tabColor) {
        btn.attr('bgcolor', row.tabColor);
      }
      btn.html(row.label);
      this.buttonContainer.append(btn);
      const content = $(`<div></div>`);
      content.addClass('framelix-tab-content');
      content.attr('data-id', tabId);
      FramelixIntersectionObserver.onGetVisible(content, async function () {
        self.reloadTab(tabId);
      });
      if (fullPath === storedActiveTabId) {
        matchedStoredActiveTabId = tabId;
      }
      if (fullPath === hashTabId) {
        matchedHashActiveTabId = tabId;
      }
      row.buttonContainer = btn;
      row.contentContainer = content;
      this.contentContainer.append(row.contentContainer);
      count++;
    }
    this.container.attr('data-id', this.id);
    this.container.attr('data-count', count);
    this.container.append(this.buttonContainer);
    this.container.append(this.contentContainer);
    this.buttonContainer.on('click', '.framelix-tab-button', function () {
      location.hash = '#' + self.getFullPath($(this).attr('data-id'));
    });
    let activeTabId = matchedHashActiveTabId || matchedStoredActiveTabId || firstTabId || '';
    this.setActiveTab(activeTabId);
  }
}
FramelixInit.late.push(FramelixTabs.init);
;
class FramelixTimeUtils {
  static toHours(value) {
    const number = FramelixTimeUtils.toSeconds(value);
    return FramelixNumberUtils.round(number / 3600, 4);
  }
  static toSeconds(value) {
    if (typeof value === 'number') return parseInt(value.toString());
    if (typeof value !== 'string' || !value.length) return 0;
    if (!value.includes(':')) {
      return parseInt(value);
    }
    const spl = value.split(':');
    return parseInt(spl[0]) * 3600 + parseInt(spl[1]) * 60 + parseInt(spl[2] || '0');
  }
  static hoursToTimeString(hours, includeSeconds) {
    return FramelixTimeUtils.secondsToTimeString(FramelixNumberUtils.round(hours * 3600, 0), includeSeconds);
  }
  static secondsToTimeString(seconds, includeSeconds) {
    if (typeof seconds !== 'number') return '';
    const hours = Math.floor(seconds / 3600).toString();
    const minutes = Math.floor(seconds / 60 % 60).toString();
    const restSeconds = Math.floor(seconds % 60).toString();
    return hours.padStart(2, '0') + ':' + minutes.padStart(2, '0') + (includeSeconds ? ':' + restSeconds.padStart(2, '0') : '');
  }
}
;
;
class FramelixToast {
  static container;
  static innerContainer;
  static loaderContainer;
  static countContainer;
  static messageContainer;
  static closeButton;
  static queue = [];
  static showNextTo = null;
  static async init() {
    FramelixToast.container = $(`<div class="framelix-toast hidden" aria-atomic="true" aria-hidden="true">
        <div class="framelix-toast-inner">
          <div class="framelix-toast-loader"></div>
          <div class="framelix-toast-counter"><span class="framelix-toast-count" title="__framelix_toast_count__"></span></div>
          <div class="framelix-toast-message"></div>
          <div class="framelix-toast-close">
            <framelix-button theme="transparent" icon="719"></framelix-button>
          </div>
        </div>
    </div>`);
    $('body').append(FramelixToast.container);
    FramelixToast.innerContainer = FramelixToast.container.children();
    FramelixToast.loaderContainer = FramelixToast.container.find('.framelix-toast-loader');
    FramelixToast.countContainer = FramelixToast.container.find('.framelix-toast-count');
    FramelixToast.messageContainer = FramelixToast.container.find('.framelix-toast-message');
    FramelixToast.closeButton = FramelixToast.container.find('.framelix-toast-close framelix-button');
    FramelixToast.closeButton.on('click', function () {
      FramelixToast.showNext(true);
    });
    $(document).on('keydown', function (ev) {
      if (ev.key === 'Escape') {
        FramelixToast.hideAll();
      }
    });
    FramelixToast.showNext();
  }
  static info(message) {
    let delaySeconds = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'auto';
    FramelixToast.queue.push({
      'message': message,
      'type': 'info',
      'delay': delaySeconds
    });
    FramelixToast.showNext();
  }
  static success(message) {
    let delaySeconds = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'auto';
    FramelixToast.queue.push({
      'message': message,
      'type': 'success',
      'delay': delaySeconds
    });
    FramelixToast.showNext();
  }
  static warning(message) {
    let delaySeconds = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'auto';
    FramelixToast.queue.push({
      'message': message,
      'type': 'warning',
      'delay': delaySeconds
    });
    FramelixToast.showNext();
  }
  static error(message) {
    let delaySeconds = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'auto';
    FramelixToast.queue.push({
      'message': message,
      'type': 'error',
      'delay': delaySeconds
    });
    FramelixToast.showNext();
  }
  static async showNext(force) {
    FramelixToast.updateQueueCount();
    if (force) {
      clearTimeout(FramelixToast.showNextTo);
      FramelixToast.showNextTo = null;
    }
    if (FramelixToast.showNextTo) {
      return;
    }
    if (!FramelixToast.queue.length) {
      FramelixToast.hideAll();
      return;
    }
    const row = FramelixToast.queue.shift();
    let colorClass = ' framelix-toast-' + row.type;
    if (row.type === 'info') colorClass = '';
    let nextMessageDelay = typeof row.delay === 'number' && row.delay > 0 ? row.delay * 1000 : 1000 * 300;
    let messagePromise;
    if (row.message instanceof FramelixRequest) {
      FramelixToast.messageContainer.html('<div class="framelix-loading"></div>');
      if ((await row.message.checkHeaders()) > 0) {
        FramelixToast.showNextTo = null;
        FramelixToast.showNext();
        return;
      }
      messagePromise = row.message.getResponseData();
    } else {
      messagePromise = FramelixLang.get(row.message);
      if (!messagePromise) {
        messagePromise = new Promise(function (resolve) {
          resolve('');
        });
      }
    }
    if (row.delay === 'auto') {
      nextMessageDelay = 5000;
    }
    FramelixToast.loaderContainer.css({
      'width': '0',
      'transition': 'none'
    });
    setTimeout(function () {
      FramelixToast.container.removeClass('hidden');
      FramelixToast.loaderContainer.css({
        'transition': nextMessageDelay + 'ms linear'
      });
      setTimeout(function () {
        FramelixToast.container.addClass('framelix-toast-visible');
        FramelixToast.loaderContainer.css('width', '100%');
      }, 10);
    }, 10);
    FramelixToast.container.attr('role', row.type === 'error' ? 'alert' : 'status').attr('aria-live', row.type === 'error' ? 'assertive' : 'polite');
    FramelixToast.innerContainer.attr('class', 'framelix-toast-inner ' + colorClass);
    messagePromise.then(function (message) {
      FramelixToast.messageContainer.html(message);
    });
    FramelixToast.updateQueueCount();
    FramelixToast.showNextTo = setTimeout(function () {
      FramelixToast.showNextTo = null;
      if (document.visibilityState === 'visible') FramelixToast.showNext();
    }, nextMessageDelay);
  }
  static hideAll() {
    FramelixToast.queue = [];
    FramelixToast.updateQueueCount();
    setTimeout(function () {
      FramelixToast.container.removeClass('framelix-toast-visible');
      setTimeout(function () {
        FramelixToast.container.addClass('hidden');
      }, 200);
    }, 10);
  }
  static updateQueueCount() {
    let queueCount = FramelixToast.queue.length;
    FramelixToast.container.attr('data-count', queueCount);
    FramelixToast.countContainer.text('+' + queueCount);
    FramelixToast.closeButton.attr('title', queueCount > 0 ? '__framelix_toast_next__' : '__framelix_close__').attr('icon', queueCount > 0 ? '713' : '719');
  }
}
FramelixInit.late.push(FramelixToast.init);
;
class FramelixView {
  static instances = [];
  container;
  phpClass;
  url;
  urlParameters;
  loaded = false;
  constructor() {
    FramelixView.instances.push(this);
    this.container = $('<div>');
    this.container.addClass('framelix-view');
  }
  getMergedUrl() {
    let url = this.url;
    if (location.search.length) {
      if (!url.includes('?')) {
        url += '?';
      } else {
        url += '&';
      }
      url += location.search.substring(1);
    }
    if (this.urlParameters) {
      if (!url.match(/\?/)) {
        url += '?';
      } else {
        url += '&';
      }
      url += FramelixObjectUtils.toUrlencodedString(this.urlParameters);
    }
    return url;
  }
  async load() {
    this.loaded = true;
    this.container.html('<div class="framelix-loading"></div> ' + (await FramelixLang.get('__framelix_view_loading__')));
    const result = await FramelixRequest.request('get', this.getMergedUrl(), null, null, false, {
      'headers': {
        'x-tab-id': this.container.closest('.framelix-tab-content').attr('data-id')
      }
    }).getResponseData();
    if (result === undefined) {
      FramelixToast.error(await FramelixLang.get('__framelix_error__', ['Request error']));
      return;
    }
    if (typeof result === 'string') {
      this.container.html(result);
    } else {
      this.container.html(result.content);
    }
  }
  render() {
    const self = this;
    this.container.attr('data-view', this.phpClass);
    FramelixIntersectionObserver.onGetVisible(this.container, function () {
      self.load();
    });
  }
}
;
;
class Framelix {
  static escapeActions = [];
  static initEarly() {
    dayjs.extend(dayjs_plugin_customParseFormat);
    dayjs.extend(dayjs_plugin_isoWeek);
    for (let i = 0; i < FramelixInit.early.length; i++) {
      FramelixInit.early[i]();
    }
  }
  static initLate() {
    for (let i = 0; i < FramelixInit.late.length; i++) {
      FramelixInit.late[i]();
    }
    if (window.location.hash && window.location.hash.startsWith('#scrollto-')) {
      const selector = window.location.hash.substr(10);
      let domChanges = 0;
      let maxDomChanges = 200;
      FramelixDom.addChangeListener('framelix-scrollto', function () {
        const el = $(selector);
        if (domChanges++ > maxDomChanges || el.length) {
          FramelixDom.removeChangeListener('framelix-scrollto');
        }
        if (el.length) {
          Framelix.scrollTo(el);
        }
      });
      Framelix.scrollTo($(window.location.hash.substr(10)));
    }
    const html = $('html');
    let dragTimeout = null;
    $(document).on('dragstart dragover', function (ev) {
      html.toggleClass('dragging', true);
      html.toggleClass('dragging-files', ev.dataTransfer.types.indexOf('Files') > -1);
      clearTimeout(dragTimeout);
      dragTimeout = setTimeout(function () {
        html.toggleClass('dragging', false);
        html.toggleClass('dragging-files', false);
      }, 1000);
    });
    $(document).on('drop dragend', function () {
      clearTimeout(dragTimeout);
      html.toggleClass('dragging', false);
      html.toggleClass('dragging-files', false);
    });
    $(document).on('keydown', '.framelix-space-click', function (ev) {
      if (ev.key === ' ') $(this).trigger('click');
    });
    $(document).on('keydown', function (ev) {
      if (ev.key === 'Escape') {
        while (Framelix.escapeActions.length) {
          if (Framelix.escapeActions.pop()() === true) {
            break;
          }
        }
      }
    });
    if (FramelixInit.initializedResolve) {
      FramelixInit.initializedResolve();
      FramelixInit.initializedResolve = null;
    }
  }
  static addEscapeAction(func) {
    Framelix.escapeActions.push(func);
  }
  static async setPageTitle(title) {
    title = await FramelixLang.get(title);
    document.title = title;
    $('h1').html(title);
  }
  static animate(valuesFrom, valuesTo, duration, stepCallback) {
    const startTime = new Date().getTime();
    const iv = setInterval(function () {
      const currentPosition = new Date().getTime() - startTime;
      let timeDelta = 1 / duration * currentPosition;
      if (timeDelta < 0) timeDelta = 0;
      if (timeDelta >= 1) timeDelta = 1;
      if (typeof valuesFrom === 'number') {
        stepCallback(valuesFrom + (valuesTo - valuesFrom) * timeDelta);
      } else if (typeof valuesFrom === 'object') {
        let values = {};
        for (let i in valuesFrom) {
          values[i] = valuesFrom[i] + (valuesTo[i] - valuesFrom[i]) * timeDelta;
        }
        stepCallback(values);
      }
      if (currentPosition >= duration || !duration) {
        clearInterval(iv);
      }
    }, 1000 / 60);
  }
  static scrollTo(target) {
    let container = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let offset = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 100;
    let duration = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 200;
    let newTop = typeof target === 'number' ? target : $(target).offset().top;
    newTop -= offset;
    if (!container) {
      if (document.body.style.overflow === 'hidden') {
        container = $('body').children().first();
      } else {
        container = $('html, body');
      }
    } else {
      container = $(container);
    }
    if (!duration) {
      container[0].scrollTop = newTop;
      return;
    }
    Framelix.animate(container[0].scrollTop, newTop, duration, function (newScroll) {
      container[0].scrollTop = newScroll;
    });
  }
  static syncScroll(a, b) {
    let direction = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 'a';
    let scrolls = [0, 0];
    if (!a.length || !b.length) return;
    function step() {
      const aScroll = Math.round(a[0].scrollTop);
      const bScroll = Math.round(b[0].scrollTop);
      if (scrolls[0] !== aScroll || scrolls[1] !== bScroll) {
        const offsetA = aScroll - scrolls[0];
        const offsetB = bScroll - scrolls[1];
        if (direction === 'a') a[0].scrollTop += offsetB;
        if (direction === 'b') b[0].scrollTop += offsetA;
        if (direction === 'both') {
          if (offsetA !== 0) b[0].scrollTop += offsetA;
          if (offsetB !== 0) a[0].scrollTop += offsetB;
        }
        scrolls[0] = Math.round(a[0].scrollTop);
        scrolls[1] = Math.round(b[0].scrollTop);
      }
      window.requestAnimationFrame(step);
    }
    window.requestAnimationFrame(step);
  }
  static redirect(url) {
    const urlNow = new URL(window.location.href);
    if (!url.match(/^http/i)) {
      url = window.location.origin + url.trim();
    }
    const urlTarget = new URL(url);
    if (urlTarget.pathname === urlNow.pathname && urlTarget.search === urlNow.search) {
      window.history.pushState('', document.title, urlTarget.toString());
      window.location.reload();
    } else {
      window.location.href = url;
    }
  }
  static showProgressBar(status, container) {
    const type = container ? 'default' : 'top';
    if (!container) {
      container = $(document.body);
    }
    let progressBar = container.children('.framelix-progress');
    if (status === undefined || status === null) {
      progressBar.remove();
      return;
    }
    if (!progressBar.length) {
      progressBar = $(`<div class="framelix-progress" data-type="${type}"><span class="framelix-progress-bar"><span class="framelix-progress-bar-inner"></span></span></div>`);
      container.append(progressBar);
      Framelix.wait(1).then(function () {
        progressBar.addClass('framelix-progress-show');
      });
    }
    if (status < 0) status = 0;
    if (status > 1) status = 1;
    status = Math.min(1, Math.max(0, status));
    if (progressBar.attr('data-status') !== status.toString()) {
      progressBar.children().css('width', status * 100 + '%').attr('data-status', status);
    }
  }
  static async wait(ms) {
    if (!ms) return;
    return new Promise(function (resolve) {
      setTimeout(resolve, ms);
    });
  }
  static downloadBlobAsFile(blob, filename) {
    if (window.navigator.msSaveOrOpenBlob) {
      window.navigator.msSaveOrOpenBlob(blob, filename);
    } else {
      const a = document.createElement('a');
      document.body.appendChild(a);
      const url = window.URL.createObjectURL(blob);
      a.href = url;
      a.download = filename;
      a.click();
      setTimeout(() => {
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      }, 0);
    }
  }
  static recursiveBase64StrToArrayBuffer(obj) {
    let prefix = '=?BINARY?B?';
    let suffix = '?=';
    if (typeof obj === 'object') {
      for (let key in obj) {
        if (typeof obj[key] === 'string') {
          let str = obj[key];
          if (str.substring(0, prefix.length) === prefix && str.substring(str.length - suffix.length) === suffix) {
            str = str.substring(prefix.length, str.length - suffix.length);
            let binary_string = window.atob(str);
            let len = binary_string.length;
            let bytes = new Uint8Array(len);
            for (let i = 0; i < len; i++) {
              bytes[i] = binary_string.charCodeAt(i);
            }
            obj[key] = bytes.buffer;
          }
        } else {
          Framelix.recursiveBase64StrToArrayBuffer(obj[key]);
        }
      }
    }
  }
  static arrayBufferToBase64(buffer) {
    let binary = '';
    let bytes = new Uint8Array(buffer);
    let len = bytes.byteLength;
    for (let i = 0; i < len; i++) {
      binary += String.fromCharCode(bytes[i]);
    }
    return window.btoa(binary);
  }
}
;
;
class FramelixTypeDefElementColor extends FramelixBaseTypeDef {
  static THEME_DEFAULT = "default";
  static THEME_PRIMARY = "primary";
  static THEME_SUCCESS = "success";
  static THEME_WARNING = "warning";
  static THEME_ERROR = "error";
  static toAttrValue(data) {
    return super.toAttrValue(data);
  }
  static fromAttrValue(str) {
    return super.fromAttrValue(str);
  }
  theme = "default";
  bgColor = null;
  textColor = null;
}
;
;
class FramelixTypeDefJsRenderTarget extends FramelixBaseTypeDef {
  static toAttrValue(data) {
    return super.toAttrValue(data);
  }
  static fromAttrValue(str) {
    return super.fromAttrValue(str);
  }
  modalOptions = null;
  popupOptions = null;
  elementSelector = null;
  selfTab = false;
  newTab = false;
}
;
;
class FramelixTypeDefJsRequestOptions extends FramelixBaseTypeDef {
  static RENDER_TARGET_MODAL_NEW = "modalnew";
  static RENDER_TARGET_POPUP = "popup";
  static RENDER_TARGET_CURRENT_CONTEXT = "currentcontext";
  static RENDER_TARGET_NONE_AND_CLOSE = "none-close";
  static toAttrValue(data) {
    return super.toAttrValue(data);
  }
  static fromAttrValue(str) {
    return super.fromAttrValue(str);
  }
  url = "";
  renderTarget = null;
}
;
;
class FramelixTypeDefModalShowOptions extends FramelixBaseTypeDef {
  static toAttrValue(data) {
    return super.toAttrValue(data);
  }
  static fromAttrValue(str) {
    return super.fromAttrValue(str);
  }
  bodyContent = null;
  headerContent = null;
  footerContent = null;
  maxWidth = null;
  color = null;
  instance = null;
  data = null;
}
;
;
class FramelixTypeDefPopupShowOptions extends FramelixBaseTypeDef {
  static CLOSEMETHODS_CLICK_OUTSIDE = "click-outside";
  static CLOSEMETHODS_CLICK_INSIDE = "click-inside";
  static CLOSEMETHODS_CLICK = "click";
  static CLOSEMETHODS_MOUSE_LEAVE_TARGET = "mouseleave-target";
  static CLOSEMETHODS_FOCUSOUT_POPUP = "focusout-popup";
  static CLOSEMETHODS_MANUAL = "manual";
  static toAttrValue(data) {
    return super.toAttrValue(data);
  }
  static fromAttrValue(str) {
    return super.fromAttrValue(str);
  }
  placement = "top";
  stickInViewport = false;
  closeMethods = "click-outside";
  color = "dark";
  group = "popup";
  offset = [0, 5];
  padding = "5px 15px";
  width = "300px";
  textAlign = "center";
  offsetByMouseEvent = null;
  appendTo = "body";
  data = null;
}
;