'use strict';

;
class FramelixLocalStorage {
  static get(key) {
    const v = localStorage.getItem(key);
    if (v === null || v === undefined) return null;
    return JSON.parse(v);
  }
  static set(key, value) {
    if (value === null || value === undefined) {
      FramelixLocalStorage.remove(key);
      return;
    }
    localStorage.setItem(key, JSON.stringify(value));
  }
  static remove(key) {
    localStorage.removeItem(key);
  }
}
;
;
class FramelixSessionStorage {
  static get(key) {
    const v = sessionStorage.getItem(key);
    if (v === null || v === undefined) return null;
    return JSON.parse(v);
  }
  static set(key, value) {
    if (value === null || value === undefined) {
      FramelixSessionStorage.remove(key);
      return;
    }
    sessionStorage.setItem(key, JSON.stringify(value));
  }
  static remove(key) {
    sessionStorage.removeItem(key);
  }
}
;
;
class FramelixDeviceDetection {
  static screenSize;
  static darkMode;
  static colorContrastMore;
  static colorContrastLess;
  static colorContrastCustom;
  static isTouch = null;
  static init() {
    FramelixDeviceDetection.screenSize = window.matchMedia('(max-width: 600px)');
    FramelixDeviceDetection.darkMode = window.matchMedia('(prefers-color-scheme: dark)');
    FramelixDeviceDetection.colorContrastMore = window.matchMedia('(prefers-contrast: more)');
    FramelixDeviceDetection.colorContrastLess = window.matchMedia('(prefers-contrast: less)');
    FramelixDeviceDetection.colorContrastCustom = window.matchMedia('(prefers-contrast: custom)');
    FramelixDeviceDetection.updateAttributes();
    FramelixDeviceDetection.screenSize.addEventListener('change', FramelixDeviceDetection.updateAttributes);
    FramelixDeviceDetection.darkMode.addEventListener('change', FramelixDeviceDetection.updateAttributes);
    FramelixDeviceDetection.colorContrastMore.addEventListener('change', FramelixDeviceDetection.updateAttributes);
    FramelixDeviceDetection.colorContrastLess.addEventListener('change', FramelixDeviceDetection.updateAttributes);
    FramelixDeviceDetection.colorContrastCustom.addEventListener('change', FramelixDeviceDetection.updateAttributes);
    FramelixDeviceDetection.updateTouchFlag(localStorage.getItem('__framelix-touch') === '1' || 'ontouchstart' in document.documentElement && FramelixDeviceDetection.screenSize.matches);
    let nextMousedownIsNoTouch = false;
    document.addEventListener('mousedown', function (ev) {
      if (nextMousedownIsNoTouch) {
        FramelixDeviceDetection.updateTouchFlag(false);
      }
      nextMousedownIsNoTouch = true;
    }, false);
    document.addEventListener('touchstart', function () {
      nextMousedownIsNoTouch = false;
      FramelixDeviceDetection.updateTouchFlag(true);
    }, false);
  }
  static updateTouchFlag(flag) {
    if (flag && !('ontouchstart' in window)) {
      flag = false;
    }
    if (flag !== FramelixDeviceDetection.isTouch) {
      FramelixDeviceDetection.isTouch = flag;
      localStorage.setItem('__framelix-touch', flag ? '1' : '0');
      document.querySelector('html').setAttribute('data-touch', flag ? '1' : '0');
    }
  }
  static updateAttributes() {
    const html = document.querySelector('html');
    html.dataset.screenSize = html.dataset.screenSizeForce || (FramelixDeviceDetection.screenSize.matches ? 's' : 'l');
    html.dataset.colorScheme = html.dataset.colorSchemeForce || (FramelixLocalStorage.get('framelix-darkmode') ? 'dark' : 'light');
    if (html.dataset.colorContrastForce) {
      html.dataset.colorContrast = html.dataset.colorContrastForce;
    } else if (FramelixDeviceDetection.colorContrastLess.matches) {
      html.dataset.colorContrast = 'less';
    } else if (FramelixDeviceDetection.colorContrastMore.matches) {
      html.dataset.colorContrast = 'more';
    } else if (FramelixDeviceDetection.colorContrastCustom.matches) {
      html.dataset.colorContrast = 'custom';
    } else {
      html.dataset.colorContrast = '';
    }
    FramelixDeviceDetection.updateTouchFlag(FramelixDeviceDetection.isTouch);
  }
}
;