'use strict';

;
class FramelixForm {
  static EVENT_SUBMITTED = 'framelix-form-submitted';
  static instances = [];
  container;
  form;
  inputHiddenSubmitFormName;
  inputHiddenSubmitButtonName;
  submitStatusContainer;
  id;
  label;
  htmlAttributes = null;
  fields = {};
  buttons = {};
  submitMethod = 'post';
  submitUrl = null;
  requestOptions = {
    url: null,
    renderTarget: FramelixTypeDefJsRequestOptions.RENDER_TARGET_CURRENT_CONTEXT
  };
  submitAsync = true;
  submitAsyncRaw = true;
  submitWithEnter = true;
  autocomplete = false;
  stickyFormButtons = false;
  fieldGroups = null;
  readOnly = false;
  appendHtml = null;
  customValidation = null;
  validationMessage = null;
  rendered;
  isSubmitting = false;
  submitRequest = null;
  _renderedResolve;
  static init() {
    const searchStartMs = 400;
    const inputSearchMap = new Map();
    const inputSearchValueMap = new Map();
    $(document).on('input keydown', 'input[type=\'search\']', function (ev) {
      if (ev.key === 'Tab') {
        return;
      }
      clearTimeout(inputSearchMap.get(this));
      if (inputSearchValueMap.get(this) === this.value && ev.key !== 'Enter') {
        return;
      }
      inputSearchValueMap.set(this, this.value);
      if (this.getAttribute('data-continuous-search') !== '1' && ev.key !== 'Enter') {
        return;
      }
      if (ev.key === 'Escape') {
        if (this.value === '') {
          $(this).trigger('blur');
        }
        return;
      }
      if (ev.key === 'Enter') {
        ev.preventDefault();
      }
      const el = $(this);
      inputSearchMap.set(this, setTimeout(function () {
        el.trigger('search-start');
        inputSearchMap.delete(this);
        inputSearchValueMap.delete(this);
      }, ev.key !== 'Enter' ? searchStartMs : 0));
    });
  }
  static getById(id) {
    for (let i = FramelixForm.instances.length - 1; i >= 0; i--) {
      if (FramelixForm.instances[i].id === id) {
        return FramelixForm.instances[i];
      }
    }
    return null;
  }
  constructor() {
    const self = this;
    this.rendered = new Promise(function (resolve) {
      self._renderedResolve = resolve;
    });
    this.id = FramelixRandom.getRandomHtmlId();
    FramelixForm.instances.push(this);
    this.container = $('<div>');
    this.container.addClass('framelix-form');
    this.container.attr('data-instance-id', FramelixForm.instances.length - 1);
  }
  addField(field) {
    field.form = this;
    this.fields[field.name] = field;
  }
  removeField(name) {
    if (this.fields[name]) {
      this.fields[name].form = null;
      delete this.fields[name];
    }
  }
  setValues(values) {
    for (let name in this.fields) {
      this.fields[name].setValue(values[name] ? values[name] || null : null);
    }
  }
  getValues() {
    let values = {};
    for (let name in this.fields) {
      values[name] = this.fields[name].getValue();
    }
    return values;
  }
  addButton(actionId, buttonText) {
    let buttonIcon = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '70c';
    let buttonColor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : FramelixTypeDefElementColor.THEME_DEFAULT;
    let buttonTooltip = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : null;
    let additionalAttributes = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : null;
    this.buttons['action-' + actionId] = {
      'type': 'action',
      'action': actionId,
      'color': buttonColor,
      'buttonText': buttonText,
      'buttonIcon': buttonIcon,
      'buttonTooltip': buttonTooltip,
      'additionalAttributes': additionalAttributes
    };
  }
  addLoadUrlButton(url, buttonText) {
    let buttonIcon = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '70c';
    let buttonColor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : FramelixTypeDefElementColor.THEME_DEFAULT;
    let buttonTooltip = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : null;
    let additionalAttributes = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : null;
    this.buttons['url-' + url] = {
      'type': 'url',
      'url': url,
      'color': buttonColor,
      'buttonText': buttonText,
      'buttonIcon': buttonIcon,
      'buttonTooltip': buttonTooltip,
      'additionalAttributes': additionalAttributes
    };
  }
  addSubmitButton(submitFieldName, buttonText) {
    let buttonIcon = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;
    let buttonColor = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : FramelixTypeDefElementColor.THEME_SUCCESS;
    let buttonTooltip = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : null;
    let additionalAttributes = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : null;
    this.buttons['submit-' + submitFieldName] = {
      'type': 'submit',
      'submitFieldName': submitFieldName,
      'color': buttonColor,
      'buttonText': buttonText,
      'buttonIcon': buttonIcon,
      'buttonTooltip': buttonTooltip,
      'additionalAttributes': additionalAttributes
    };
  }
  setSubmitStatus(flag) {
    this.isSubmitting = flag;
    this.container.toggleClass('framelix-form-submitting', flag);
  }
  async showValidationMessage(message) {
    message = await FramelixLang.get(message);
    this.validationMessage = message;
    FramelixToast.error(message);
  }
  hideValidationMessage() {
    this.validationMessage = null;
    FramelixPopup.destroyInstancesOnTarget(this.submitStatusContainer);
  }
  addFieldGroup(id, label, fieldNames) {
    let defaultState = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : true;
    let rememberState = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : true;
    this.fieldGroups[id] = {
      'label': label,
      'fieldNames': fieldNames,
      'defaultState': defaultState,
      'rememberState': rememberState
    };
  }
  removeFieldGroup(id) {
    if (this.fieldGroups) {
      delete this.fieldGroups[id];
    }
  }
  updateFieldVisibility() {
    const formValues = FormDataJson.toJson(this.form[0], {
      'flatList': true,
      'includeDisabled': true
    });
    let formValuesFlatIndexed = {};
    for (let i = 0; i < formValues.length; i++) {
      formValuesFlatIndexed[formValues[i][0]] = formValues[i][1];
    }
    let fieldsWithConditionFlat = [];
    for (let fieldName in this.fields) {
      const field = this.fields[fieldName];
      if (!field.visibilityCondition) {
        field.container.toggleClass('hidden', false);
      } else {
        fieldsWithConditionFlat.push(field);
      }
    }
    for (let i = 0; i < fieldsWithConditionFlat.length; i++) {
      const field = fieldsWithConditionFlat[i];
      let conditionData = field.visibilityCondition.data;
      let isVisible = false;
      conditionLoop: for (let j = 0; j < conditionData.length; j++) {
        const conditionRow = conditionData[j];
        if (conditionRow.type === 'or') {
          if (isVisible) {
            break;
          }
          continue;
        }
        if (conditionRow.type === 'and') {
          if (!isVisible) {
            break;
          }
          continue;
        }
        let conditionFieldValue = typeof formValuesFlatIndexed[conditionRow.field] === 'undefined' ? null : formValuesFlatIndexed[conditionRow.field];
        let requiredValue = conditionRow.value;
        switch (conditionRow.type) {
          case 'equal':
          case 'notEqual':
          case 'like':
          case 'notLike':
            if (requiredValue !== null && typeof requiredValue !== 'object') {
              requiredValue = [requiredValue + ''];
            }
            if (conditionFieldValue !== null && typeof conditionFieldValue !== 'object') {
              conditionFieldValue = [conditionFieldValue + ''];
            }
            for (let requiredValueKey in requiredValue) {
              if (conditionRow.type === 'equal' || conditionRow.type === 'like') {
                for (let conditionFieldValueKey in conditionFieldValue) {
                  const val = conditionFieldValue[conditionFieldValueKey];
                  isVisible = conditionRow.type === 'equal' ? val === requiredValue[requiredValueKey] : val.match(FramelixStringUtils.escapeRegex(requiredValue[requiredValueKey]), 'i');
                  if (isVisible) {
                    continue conditionLoop;
                  }
                }
              } else {
                for (let conditionFieldValueKey in conditionFieldValue) {
                  const val = conditionFieldValue[conditionFieldValueKey];
                  isVisible = conditionRow.type === 'notEqual' ? val !== requiredValue[requiredValueKey] : !val.match(FramelixStringUtils.escapeRegex(requiredValue[requiredValueKey]), 'i');
                  if (isVisible) {
                    continue conditionLoop;
                  }
                }
              }
            }
            break;
          case 'greatherThan':
          case 'greatherThanEqual':
          case 'lowerThan':
          case 'lowerThanEqual':
            if (typeof conditionFieldValue === 'object') {
              conditionFieldValue = FramelixObjectUtils.countKeys(conditionFieldValue);
            } else {
              conditionFieldValue = parseFloat(conditionFieldValue);
            }
            if (conditionRow.type === 'greatherThan') {
              isVisible = conditionFieldValue > requiredValue;
            } else if (conditionRow.type === 'greatherThanEqual') {
              isVisible = conditionFieldValue >= requiredValue;
            } else if (conditionRow.type === 'lowerThan') {
              isVisible = conditionFieldValue < requiredValue;
            } else if (conditionRow.type === 'lowerThanEqual') {
              isVisible = conditionFieldValue <= requiredValue;
            }
            break;
          case 'empty':
          case 'notEmpty':
            isVisible = conditionFieldValue === null || conditionFieldValue === '' || typeof conditionFieldValue === 'object' && !FramelixObjectUtils.countKeys(conditionFieldValue);
            if (conditionRow.type === 'notEmpty') {
              isVisible = !isVisible;
            }
            break;
        }
      }
      field.setVisibilityConditionHiddenStatus(isVisible);
    }
  }
  async validate() {
    let success = true;
    this.hideValidationMessage();
    for (let fieldName in this.fields) {
      const field = this.fields[fieldName];
      field.hideValidationMessage();
    }
    for (let fieldName in this.fields) {
      const field = this.fields[fieldName];
      const validation = await field.validate();
      if (validation !== true) {
        success = false;
        field.showValidationMessage(validation);
      }
    }
    if (success && this.customValidation) {
      const validation = await this.customValidation();
      if (validation !== true) {
        success = false;
        this.showValidationMessage(validation);
      }
    }
    return success;
  }
  async render() {
    const self = this;
    this.form = $(`<form>`);
    if (!this.autocomplete) {
      this.form.attr('autocomplete', 'off');
    }
    this.form.attr('novalidate', true);
    this.container.empty();
    this.container.toggleClass('framelix-form-sticky-form-buttons', this.stickyFormButtons);
    if (this.label) {
      this.container.append($(`<div class="framelix-form-label"></div>`).html(await FramelixLang.get(this.label)));
    }
    this.container.append(this.form);
    this.container.css('display', 'none');
    $(document.body).append(this.container);
    this.form.attr('id', 'framelix-form-' + this.id);
    this.form.attr('name', 'framelix-form-' + this.id);
    this.form.attr('onsubmit', 'return false');
    if (this.htmlAttributes) {
      this.htmlAttributes.assignToElement(this.form);
    }
    this.inputHiddenSubmitFormName = $('<input type="hidden" value="1">');
    this.inputHiddenSubmitButtonName = $('<input type="hidden" value="1">');
    this.form.append(this.inputHiddenSubmitFormName);
    this.form.append(this.inputHiddenSubmitButtonName);
    const fieldRenderPromises = [];
    const positionedFields = [];
    for (let name in this.fields) {
      const field = this.fields[name];
      field.form = this;
      if (this.readOnly) {
        field.disabled = true;
      }
      const row = $('<div class="framelix-form-field-row"></div>').append(field.container);
      row.attr('data-types', field.container.attr('data-types'));
      this.form.append(row);
      field.render();
      fieldRenderPromises.push(field.rendered);
      if (field.positionInForm) {
        positionedFields.push(field);
      }
    }
    if (this.fieldGroups) {
      for (let id in this.fieldGroups) {
        const row = this.fieldGroups[id];
        const storageKey = this.id + '_' + id;
        let state = row.defaultState;
        if (row.rememberState) {
          state = FramelixLocalStorage.get(storageKey);
          if (state === null) {
            state = row.defaultState;
          }
        }
        let groupStartField = null;
        let prevGroupField = null;
        for (let i = 0; i < row.fieldNames.length; i++) {
          const fieldName = row.fieldNames[i];
          const field = this.fields[fieldName];
          if (field) {
            const rowContainer = field.container.parent();
            if (!groupStartField) {
              groupStartField = field;
              rowContainer.before(`<div class="framelix-form-field-group" data-id="${id}" data-storage-key="${storageKey}" data-state="${state ? 1 : 0}" data-remember="${row.rememberState ? '1' : '0'}"><framelix-button theme="light" icon="705">${row.label}</framelix-button></div>`);
            }
            rowContainer.toggleClass('framelix-form-field-group-hidden', !state);
            rowContainer.attr('data-field-group-id', id);
            if (prevGroupField) {
              prevGroupField.container.parent().after(rowContainer);
            }
            prevGroupField = field;
          }
        }
      }
    }
    const bottomRow = $(`<div class="framelix-form-row framelix-form-row-bottom"></div>`);
    bottomRow.attr('id', 'framelix-form-row-bottom-' + this.id);
    this.container.append(bottomRow);
    const buttonsCount = FramelixObjectUtils.countKeys(this.buttons);
    const buttonsRow = $(`<div class="framelix-form-buttons framelix-buttons-wrap" data-buttons="${buttonsCount}"></div>`);
    bottomRow.append(buttonsRow);
    if (buttonsCount) {
      for (let i in this.buttons) {
        const buttonData = this.buttons[i];
        if (this.readOnly && !buttonData.ignoreReadOnly) {
          continue;
        }
        const button = $(`<framelix-button>`);
        FramelixColorUtils.setColorsFromElementColorDef(buttonData.color, button);
        if (buttonData.additionalAttributes) {
          if (!(buttonData.additionalAttributes instanceof FramelixHtmlAttributes)) {
            buttonData.additionalAttributes = FramelixObjectUtils.phpJsonToJs(buttonData.additionalAttributes);
          }
          buttonData.additionalAttributes.assignToElement(button);
        }
        button.attr('data-type', buttonData.type);
        button.attr('data-submit-field-name', buttonData.submitFieldName);
        button.html(buttonData.buttonText);
        if (buttonData.buttonIcon) {
          button.attr('icon', buttonData.buttonIcon);
        }
        if (buttonData.buttonTooltip) {
          button.attr('title', buttonData.buttonTooltip);
        }
        if (buttonData.type === 'submit') {
          button.on('click', function () {
            self.submit($(this).attr('data-submit-field-name'));
          });
        } else if (buttonData.type === 'url') {
          button.on('click', function () {
            if (self.submitRequest) {
              self.submitRequest.abort();
            }
            window.location.href = buttonData.url;
          });
        } else if (buttonData.type === 'action') {
          button.attr('data-action', buttonData.action);
        }
        buttonsRow.append(button);
      }
      this.form.on('keydown', function (ev) {
        if (ev.key === 'Enter' && self.submitWithEnter || ev.key.toLowerCase() === 's' && ev.ctrlKey) {
          buttonsRow.find('[data-type=\'submit\']').first().trigger('click');
          if (ev.ctrlKey) {
            ev.preventDefault();
          }
        }
      });
    }
    this.submitStatusContainer = $(`<div class="framelix-form-submit-status"></div>`);
    bottomRow.append(this.submitStatusContainer);
    if (typeof this.appendHtml === 'string') {
      this.container.append(this.appendHtml);
    }
    this.container.css('display', '');
    if (this.validationMessage !== null) {
      this.showValidationMessage(this.validationMessage);
    }
    this.form.on('focusin', function () {
      self.hideValidationMessage();
    });
    this.form.on('click', '.framelix-form-field-group framelix-button', function () {
      const el = $(this).parent();
      const newState = el.attr('data-state') !== '1';
      const id = el.attr('data-id');
      el.attr('data-state', newState ? '1' : '0');
      self.form.find('.framelix-form-field-row').filter('[data-field-group-id=\'' + id + '\']').toggleClass('framelix-form-field-group-hidden', !newState);
      if (el.attr('data-remember') === '1') {
        FramelixLocalStorage.set(el.attr('data-storage-key'), newState);
      }
    });
    for (let i = 0; i < positionedFields.length; i++) {
      const field = positionedFields[i];
      if (!field.positionInForm || !field.positionInForm.after) {
        continue;
      }
      const afterField = self.fields[field.positionInForm.after];
      if (!afterField) {
        continue;
      }
      const rowToAttach = afterField.container.parent();
      const oldRow = field.container.parent();
      rowToAttach.attr('data-sizing', field.positionInForm.sizing);
      if (field.positionInForm.columnGrowMe) {
        field.container.css('flex-grow', field.positionInForm.columnGrowMe);
      }
      if (field.positionInForm.columnGrowOther) {
        afterField.container.css('flex-grow', field.positionInForm.columnGrowOther);
      }
      rowToAttach.append(field.container);
      if (!oldRow.children('.framelix-form-field').length) {
        oldRow.remove();
      }
    }
    Promise.all(fieldRenderPromises).then(function () {
      if (self._renderedResolve) {
        self._renderedResolve();
      }
      self._renderedResolve = null;
      self.updateFieldVisibility();
      self.form.on(FramelixFormField.EVENT_CHANGE, function () {
        self.updateFieldVisibility();
      });
    });
  }
  async submit(submitButtonName) {
    if (this.isSubmitting) {
      return false;
    }
    if ((await this.validate()) !== true) {
      return false;
    }
    const self = this;
    this.inputHiddenSubmitFormName.attr('name', 'framelix-form-' + this.id);
    this.inputHiddenSubmitButtonName.attr('name', 'framelix-form-button-' + (submitButtonName || this.id));
    if (!this.submitAsync) {
      this.setSubmitStatus(true);
      this.form.removeAttr('onsubmit');
      this.form.attr('method', this.submitMethod);
      this.form.attr('target', this.requestOptions.renderTarget && this.requestOptions.renderTarget.newTab ? '_blank' : '_self');
      this.form.attr('action', this.submitUrl || window.location.href);
      this.form[0].submit();
      this.form.attr('onsubmit', 'return false');
      if (this.form.attr('target') === '_blank') {
        setTimeout(function () {
          self.setSubmitStatus(false);
          self.form.trigger(FramelixForm.EVENT_SUBMITTED, {
            'submitButtonName': submitButtonName
          });
        }, 1000);
      }
      return true;
    }
    self.setSubmitStatus(true);
    let formData;
    if (this.submitAsyncRaw) {
      formData = JSON.stringify(FormDataJson.toJson(this.form[0], {
        'includeDisabled': true
      }));
    } else {
      let values = FormDataJson.toJson(this.form[0], {
        'flatList': true,
        'includeDisabled': true
      });
      formData = new FormData();
      for (let i = 0; i < values.length; i++) {
        formData.append(values[i][0], values[i][1]);
      }
      for (let fieldName in this.fields) {
        const field = this.fields[fieldName];
        if (field instanceof FramelixFormFieldFile) {
          const files = field.getValue();
          if (files) {
            for (let i = 0; i < files.length; i++) {
              formData.append(fieldName + '[]', files[i]);
            }
          }
        }
      }
    }
    this.hideValidationMessage();
    const requestOptions = this.requestOptions ? this.requestOptions : {
      url: null,
      renderTarget: 'currentcontext'
    };
    let submitUrl = this.submitUrl || requestOptions.url;
    if (!submitUrl) {
      const tabContent = this.form.closest('.framelix-tab-content');
      if (tabContent.length) {
        const tabData = FramelixTabs.instances[tabContent.closest('.framelix-tabs').attr('data-instance-id')].tabs[tabContent.attr('data-id')];
        if (tabData && tabData.content instanceof FramelixView) {
          submitUrl = tabData.content.getMergedUrl();
        }
      }
    }
    if (!submitUrl) {
      submitUrl = location.href;
    }
    this.submitRequest = FramelixRequest.request('post', submitUrl, null, formData, this.submitStatusContainer);
    const request = self.submitRequest;
    await request.finished;
    self.setSubmitStatus(false);
    self.form.trigger(FramelixForm.EVENT_SUBMITTED, {
      'submitButtonName': submitButtonName
    });
    for (let fieldName in self.fields) {
      const field = self.fields[fieldName];
      field.hideValidationMessage();
    }
    self.hideValidationMessage();
    const responseCheckHeadersStatus = await request.checkHeaders();
    if (requestOptions && requestOptions.renderTarget === FramelixTypeDefJsRequestOptions.RENDER_TARGET_NONE_AND_CLOSE) {
      FramelixModal.destroyAll();
      FramelixPopup.destroyAll();
    }
    if (responseCheckHeadersStatus !== 0) {
      return true;
    }
    const responseData = await request.getJson();
    if (!responseData) {
      return true;
    }
    if (responseData.errorMessages) {
      if (typeof responseData.errorMessages === 'string') {
        this.showValidationMessage(responseData.errorMessages);
      } else {
        for (let fieldName in self.fields) {
          const field = self.fields[fieldName] || this;
          if (!responseData.errorMessages[fieldName]) {
            continue;
          }
          if (field) {
            field.showValidationMessage(responseData.errorMessages[fieldName]);
          } else {
            this.showValidationMessage(responseData.errorMessages[fieldName]);
          }
        }
      }
    }
    if (FramelixObjectUtils.hasKeys(responseData.toastMessages)) {
      for (let i = 0; i < responseData.toastMessages.length; i++) {
        FramelixToast.queue.push(responseData.toastMessages[i]);
      }
      FramelixToast.showNext();
    }
    if (typeof responseData === 'string') {
      FramelixRequest.renderResponse(responseData, requestOptions, this.container[0]);
    } else if (typeof responseData.content === 'string' && responseData.content.length) {
      FramelixRequest.renderResponse(responseData.content, requestOptions, this.container[0]);
    }
    return true;
  }
}
FramelixInit.late.push(FramelixForm.init);
;
class FramelixFormField {
  static EVENT_CHANGE = 'framelix-form-field-change';
  static EVENT_CHANGE_USER = 'framelix-form-field-change-user';
  static VISIBILITY_HIDDEN = 'hidden';
  static VISIBILITY_TRANSPARENT = 'transparent';
  static classReferences = {};
  static instances = [];
  container;
  field;
  form = null;
  name;
  label = null;
  labelDescription = null;
  minWidth = null;
  maxWidth = null;
  defaultValue = null;
  disabled = false;
  required = false;
  validationMessage = null;
  validationPopup = null;
  visibilityCondition = null;
  visibilityConditionHideMethod = FramelixFormField.VISIBILITY_HIDDEN;
  positionInForm = null;
  rendered;
  _renderedResolve;
  static phpJsonToJs(phpProperties, phpClass) {
    const fieldClass = phpClass.substring(9).replace(/\\/g, '');
    const instance = new this.classReferences[fieldClass]();
    for (let key in phpProperties) {
      instance[key] = FramelixObjectUtils.phpJsonToJs(phpProperties[key], true);
    }
    return instance;
  }
  static getFieldByName(container, name) {
    const fields = $(container instanceof FramelixForm ? container.container : container).find('.framelix-form-field');
    if (!fields.length) return null;
    let field;
    if (!name) {
      field = fields.first();
    } else {
      field = fields.filter('[data-name=\'' + name + '\']');
    }
    if (!field.length) return null;
    return FramelixFormField.instances[field.attr('data-instance-id')] || null;
  }
  static onValueChange(container, fields, onUserChangeOnly, callback) {
    if (!fields) return;
    if (!Array.isArray(fields)) fields = [fields];
    $(document).on(onUserChangeOnly ? this.EVENT_CHANGE_USER : this.EVENT_CHANGE, function (ev) {
      let el = container;
      if (!el) el = $('body');
      if (typeof el === 'string') el = FramelixForm.getById(el).container;
      for (let i in fields) {
        let field = fields[i];
        if (typeof field === 'string') {
          field = FramelixFormField.getFieldByName(el, field);
        }
        const fieldName = $(ev.target).closest('.framelix-form-field').attr('data-name');
        if (fieldName === field.name) callback(field);
      }
    });
  }
  constructor() {
    const self = this;
    this.rendered = new Promise(function (resolve) {
      self._renderedResolve = resolve;
    });
    FramelixFormField.instances.push(this);
    this.container = $(`<div class="framelix-form-field">
        <div class="framelix-form-field-label"></div>
        <div class="framelix-form-field-label-description"></div>
        <div class="framelix-form-field-container"></div>
      </div>`);
    this.container.attr('data-instance-id', FramelixFormField.instances.length - 1);
    let classes = [];
    let types = [];
    let parent = Object.getPrototypeOf(this);
    while (parent && parent.constructor.name !== 'FramelixFormField' && parent.constructor.name.includes('FormField')) {
      classes.push('framelix-form-field-' + parent.constructor.name.substring(parent.constructor.name.indexOf('FormField') + 9).toLowerCase());
      types.push(parent.constructor.name);
      parent = Object.getPrototypeOf(parent);
      if (classes.length > 10) break;
    }
    this.container.addClass(classes.join(' '));
    this.container.attr('data-types', types.join(','));
    this.field = this.container.find('.framelix-form-field-container');
  }
  stringifyValue(value) {
    if (value === null || value === undefined) {
      return '';
    }
    if (typeof value === 'boolean') {
      return value ? '1' : '0';
    }
    if (typeof value !== 'string') {
      return value.toString();
    }
    return value;
  }
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    console.error('setValue need to be implemented in ' + this.constructor.name);
  }
  getValue() {
    console.error('getValue need to be implemented in ' + this.constructor.name);
  }
  triggerChange(el) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    el.trigger(FramelixFormField.EVENT_CHANGE);
    if (isUserChange) {
      el.trigger(FramelixFormField.EVENT_CHANGE_USER);
    }
  }
  async validate() {
    if (!this.isVisible()) return true;
    if (this.required && !(this instanceof FramelixFormFieldHtml) && !(this instanceof FramelixFormFieldHidden)) {
      const value = this.getValue();
      if (value === null || value === undefined || typeof value === 'string' && !value.length || typeof value === 'object' && !FramelixObjectUtils.hasKeys(value)) {
        return await FramelixLang.get('__framelix_form_validation_required__');
      }
    }
    return true;
  }
  async showValidationMessage(message) {
    this.container.toggleClass('framelix-form-field-group-hidden', false);
    if (!this.isVisible()) {
      this.form.showValidationMessage(message);
      return;
    }
    message = await FramelixLang.get(message);
    this.validationMessage = message;
    let container = null;
    this.container.find('[tabindex],input,select,textarea').each(function () {
      if (FramelixDom.isVisible(this)) {
        container = this;
        return false;
      }
    });
    if (!container) container = this.field;
    container = $(container);
    if (this.validationPopup && FramelixDom.isInDom(this.validationPopup.content)) {
      this.validationPopup.content.append($(`<div>`).append(message));
    } else {
      this.validationPopup = FramelixPopup.show(container, message, {
        closeMethods: 'click',
        color: 'error',
        placement: 'bottom-start',
        group: 'field-validation',
        stickInViewport: true
      });
    }
  }
  hideValidationMessage() {
    var _this$validationPopup;
    this.validationMessage = null;
    (_this$validationPopup = this.validationPopup) === null || _this$validationPopup === void 0 || _this$validationPopup.destroy();
  }
  setVisibilityConditionHiddenStatus(flag) {
    const row = this.container.parent();
    const fieldCont = this.container;
    const fieldsInRow = row.children('.framelix-form-field');
    fieldCont.toggleClass('framelix-form-field-hidden', !flag);
    row.attr('data-visible-fields-in-row', fieldsInRow.not('.framelix-form-field-hidden').length.toString());
    if (!flag) {
      row.attr('data-visibility-hidden-method', this.visibilityConditionHideMethod);
      fieldsInRow.attr('data-visibility-hidden-method', this.visibilityConditionHideMethod);
    }
    if (this.visibilityConditionHideMethod === FramelixFormField.VISIBILITY_TRANSPARENT) {
      row.find('[tabindex],input,select,textarea').each(function () {
        if (!flag && this.getAttribute('tabindex') !== null && this.getAttribute('data-tabindex-original') === null) {
          this.setAttribute('data-tabindex-original', this.getAttribute('tabindex'));
        }
        if (!flag) {
          this.setAttribute('tabindex', '-1');
        } else {
          this.setAttribute('tabindex', this.getAttribute('data-tabindex-original'));
        }
      });
    }
  }
  isVisible() {
    return !this.container.hasClass('framelix-form-field-hidden');
  }
  setPositionInForm() {
    let after = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;
    let sizing = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let columnGrowMe = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;
    let columnGrowOther = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : null;
    if (after === null) {
      this.positionInForm = null;
      return;
    }
    if (after instanceof FramelixFormField) {
      after = after.name;
    }
    this.positionInForm = {
      'after': after,
      'sizing': sizing,
      'columnGrowMe': columnGrowMe,
      'columnGrowOther': columnGrowOther
    };
  }
  async renderInternal() {
    const self = this;
    this.container.attr('data-name', this.name);
    let widthContainer = this.field;
    if (!this.form) {
      widthContainer = this.container.closest('.framelix-form-field');
    }
    widthContainer.css('minWidth', this.minWidth !== null ? typeof this.minWidth === 'number' ? this.minWidth + 'px' : this.minWidth : '');
    widthContainer.css('maxWidth', this.maxWidth !== null ? typeof this.maxWidth === 'number' ? this.maxWidth + 'px' : this.maxWidth : '');
    this.container.attr('data-disabled', this.disabled ? 1 : 0);
    let requiredInfoDisplayed = false;
    const labelEl = this.container.find('.framelix-form-field-label');
    if (this.label !== null) {
      requiredInfoDisplayed = true;
      labelEl.html(await FramelixLang.get(this.label));
      if (this.required) {
        labelEl.append(`<span class="framelix-form-field-label-required" title="__framelix_form_validation_required__"></span>`);
      }
    } else {
      labelEl.remove();
    }
    const labelDescEl = this.container.find('.framelix-form-field-label-description');
    if (this.labelDescription !== null) {
      labelDescEl.html(await FramelixLang.get(this.labelDescription));
      if (!requiredInfoDisplayed && this.required) {
        labelDescEl.append(`<span class="framelix-form-field-label-required" title="__framelix_form_validation_required__"></span>`);
      }
    } else {
      labelDescEl.remove();
    }
    this.field.on('focusin change', function () {
      self.hideValidationMessage();
    });
  }
  async render() {
    await this.renderInternal();
    if (this.validationMessage !== null) this.showValidationMessage(this.validationMessage);
    if (this._renderedResolve) {
      this._renderedResolve();
      this._renderedResolve = null;
    }
  }
}
;
;
class FramelixFormFieldText extends FramelixFormField {
  placeholder = null;
  spellcheck = false;
  input;
  minLength = null;
  maxLength = null;
  type = 'text';
  autocompleteSuggestions = null;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let originalVal = this.input.val();
    value = this.stringifyValue(value);
    if (originalVal !== value) {
      this.input.val(value);
      this.triggerChange(this.input, isUserChange);
    }
  }
  getValue() {
    return this.input.val();
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    if (this.minLength !== null) {
      const value = this.getValue();
      if (value.length < this.minLength) {
        return await FramelixLang.get('__framelix_form_validation_minlength__', {
          'number': this.minLength
        });
      }
    }
    if (this.maxLength !== null) {
      const value = this.getValue();
      if (value.length > this.maxLength) {
        return await FramelixLang.get('__framelix_form_validation_maxlength__', {
          'number': this.maxLength
        });
      }
    }
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input = $(`<input type="text" class="framelix-form-field-input">`);
    this.field.html(this.input);
    if (this.autocompleteSuggestions) {
      var _this$form;
      const listId = (((_this$form = this.form) === null || _this$form === void 0 ? void 0 : _this$form.id) || FramelixRandom.getRandomHtmlId()) + '_' + this.name;
      const list = $('<datalist id="' + listId + '">');
      for (let i = 0; i < this.autocompleteSuggestions.length; i++) {
        list.append($('<option>').attr('value', this.autocompleteSuggestions[i]));
      }
      this.field.append(list);
      this.input.attr('list', listId);
    }
    if (this.placeholder !== null) this.input.attr('placeholder', this.placeholder);
    if (this.disabled) this.input.attr('disabled', true);
    if (this.maxLength !== null) this.input.attr('maxlength', this.maxLength);
    this.input.attr('spellcheck', this.spellcheck ? 'true' : 'false');
    this.input.attr('name', this.name);
    this.input.attr('tabindex', '0');
    this.input.attr('type', this.type);
    this.input.on('change input', function () {
      self.triggerChange(self.input, true);
    });
    this.setValue(this.defaultValue || '');
  }
}
FramelixFormField.classReferences['FramelixFormFieldText'] = FramelixFormFieldText;
;
class FramelixFormFieldSelect extends FramelixFormField {
  dropdown = true;
  multiple = false;
  searchable = false;
  showResetButton = null;
  options = [];
  optionsContainer;
  minSelectedItems = null;
  maxSelectedItems = null;
  chooseOptionLabel = '__framelix_form_select_chooseoption_label__';
  noOptionsLabel = '__framelix_form_select_noptions_label__';
  loadUrlOnChange = null;
  loadUrlTarget = '_self';
  optionsPopup = null;
  valueInitialized = false;
  async setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let countChecked = 0;
    if (!this.valueInitialized || this.stringifyValue(value) !== this.stringifyValue(this.getValue())) {
      this.valueInitialized = true;
      let arrValues = [];
      if (value !== null) {
        if (typeof value !== 'object') {
          value = [value];
        }
        for (let i in value) {
          arrValues.push(this.stringifyValue(value[i]));
        }
      }
      this.optionsContainer.html('');
      for (let key in this.options) {
        const optionValue = this.stringifyValue(this.options[key][0]);
        const checked = arrValues.indexOf(optionValue) > -1;
        if (this.dropdown && !checked) continue;
        const el = this.getOptionHtml(key, checked);
        this.optionsContainer.append(el);
        countChecked++;
      }
      this.container.attr('data-checked', countChecked);
      if (!countChecked) {
        this.optionsContainer.html(`<div class="framelix-form-field-select-option">${await FramelixLang.get(this.options.length ? this.chooseOptionLabel : this.noOptionsLabel)}</div>`);
      }
      this.triggerChange(this.field, isUserChange);
    }
  }
  getValue() {
    const values = FormDataJson.toJson(this.optionsContainer[0], {
      'includeDisabled': true,
      'flatList': true
    });
    let arr = [];
    for (let i = 0; i < values.length; i++) {
      arr.push(values[i][1]);
    }
    if (!arr.length) return null;
    return this.multiple ? arr : arr[0];
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    let value = this.getValue();
    if (value !== null) {
      value = this.multiple ? value : [value];
      if (this.minSelectedItems !== null) {
        if (value.length < this.minSelectedItems) {
          return await FramelixLang.get('__framelix_form_validation_minselecteditems__', {
            'number': this.minSelectedItems
          });
        }
      }
      if (this.maxSelectedItems !== null) {
        if (value.length > this.maxSelectedItems) {
          return await FramelixLang.get('__framelix_form_validation_maxselecteditems__', {
            'number': this.maxSelectedItems
          });
        }
      }
    }
    return true;
  }
  addOptions(options) {
    if (options) {
      for (let key in options) this.addOption(key, options[key]);
    }
  }
  addOption(value, label) {
    if (this.indexOfOptionValue(value) === -1) {
      this.options.push([this.stringifyValue(value), label]);
    }
  }
  removeOption(value) {
    let i = this.indexOfOptionValue(value);
    if (i > -1) this.options.splice(i, 1);
  }
  removeOptions(options) {
    if (options) {
      for (let key in options) this.removeOption(options[key]);
    }
  }
  indexOfOptionValue(value) {
    for (let i = 0; i < this.options.length; i++) {
      if (this.options[i][0] === this.stringifyValue(value)) {
        return i;
      }
    }
    return -1;
  }
  getOptionHtml(optionIndex, checked) {
    const optionValue = this.options[optionIndex][0];
    const optionLabel = this.options[optionIndex][1];
    const option = $(`
        <label class="framelix-form-field-select-option">
            <div class="framelix-form-field-select-option-checkbox">
                <input type="checkbox" name="${this.name + (this.multiple ? '[]' : '')}" ${this.disabled ? 'disabled' : ''}>
            </div>
            <div class="framelix-form-field-select-option-label"></div>
        </label>
      `);
    const input = option.find('input');
    input.attr('value', optionValue);
    input.prop('checked', checked);
    FramelixLang.get(optionLabel).then(function (result) {
      option.find('.framelix-form-field-select-option-label').html(result);
    });
    return option;
  }
  async showDropdown() {
    if (this.disabled) {
      return;
    }
    const self = this;
    const values = this.getValue();
    let popupContent = $(`<div class="framelix-form-field-select-popup"><div class="framelix-form-field-input" tabindex="0"></div></div>`);
    let popupContentInner = popupContent.children();
    if (this.searchable) {
      popupContentInner.append(`<div class="framelix-form-field-select-search"><input type="search" placeholder="${await FramelixLang.get('__framelix_form_select_search__')}" class="framelix-form-field-input" data-continuous-search="1" tabindex="0"></div>`);
    }
    const popupOptionsContainer = $(`<div class="framelix-form-field-select-popup-options"></div>`);
    popupContentInner.append(popupOptionsContainer);
    const optionsElementsIndexed = {};
    for (let key in this.options) {
      const optionValue = this.options[key][0];
      const optionElement = this.getOptionHtml(key, values === optionValue || Array.isArray(values) && values.indexOf(optionValue) > -1);
      optionsElementsIndexed[optionValue] = optionElement;
      popupOptionsContainer.append(optionElement);
    }
    this.optionsPopup = FramelixPopup.show(this.field, popupContent, {
      placement: 'bottom-start',
      closeMethods: 'click-outside,focusout-popup',
      appendTo: this.field,
      padding: '',
      offset: [0, 0],
      color: this.optionsContainer.parent()
    });
    this.optionsPopup.destroyed.then(function () {
      let values = [];
      popupContentInner.find('input:checked').each(function () {
        values.push(this.value);
      });
      if (!self.multiple) {
        values = values.shift();
      }
      self.setValue(values, true);
      self.optionsPopup = null;
    });
    this.optionsPopup.popperEl.css('width', Math.max(this.field.width(), 250) + 'px');
    this.initOptionsContainer(popupOptionsContainer);
    popupContentInner.find('.framelix-form-field-select-search input').on('search-start', function (ev) {
      ev.stopPropagation();
      const val = this.value.trim();
      for (let key in self.options) {
        const optionValue = self.options[key][0];
        const optionLabel = self.options[key][1];
        optionsElementsIndexed[optionValue].toggleClass('hidden', val !== '' && !optionLabel.match(new RegExp(val, 'i')));
      }
    });
    setTimeout(function () {
      let input = popupContentInner.find('input:checked').first();
      if (!input.length) {
        input = popupContentInner.find('input').first();
      }
      input.trigger('focus');
    }, 10);
  }
  toggleDropdown() {
    if (this.disabled) {
      return;
    }
    if (this.optionsPopup) {
      this.destroyDropdown();
    } else {
      this.showDropdown();
    }
  }
  destroyDropdown() {
    if (this.optionsPopup) {
      this.optionsPopup.destroy();
    }
  }
  initOptionsContainer(container) {
    const self = this;
    let mouseStartEl = null;
    if (!this.multiple) {
      container.on('change', 'input', function (ev) {
        const checked = ev.target.checked;
        container.find('input').prop('checked', false);
        ev.target.checked = checked;
        if (self.dropdown) {
          self.destroyDropdown();
          setTimeout(function () {
            self.field.children().first().trigger('focus');
          }, 10);
        }
      });
    } else {
      function updateValue() {
        const arr = [];
        container.find('input:checked').each(function () {
          arr.push(this.value);
        });
        self.setValue(arr, true);
      }
      container.on('mousedown', 'label', function (ev) {
        mouseStartEl = this;
        $(document).one('mouseup', function () {
          mouseStartEl = null;
        });
      });
      container.on('mouseenter', 'label', function (ev) {
        if (mouseStartEl && (ev.which || ev.touches)) {
          const input = $(this).find('input')[0];
          input.checked = !input.checked;
          updateValue();
        }
      });
      container.on('click', 'label', function (ev) {
        if (!ev.shiftKey) return;
        const input = $(this).find('input')[0];
        container.find('input').prop('checked', input.checked);
        updateValue();
      });
    }
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.valueInitialized = false;
    this.container.attr('data-multiple', this.multiple ? '1' : '0');
    this.container.attr('data-dropdown', this.dropdown ? '1' : '0');
    this.container.attr('data-options', this.options ? this.options.length : 0);
    this.field.html(`
      <div class="framelix-form-field-input framelix-form-field-select-picker">
          <div class="framelix-form-field-select-options"></div>          
      </div>
    `);
    this.optionsContainer = this.field.find('.framelix-form-field-select-options');
    const pickerEl = this.field.children();
    if (!this.disabled) {
      if (this.showResetButton === true || this.showResetButton === null && !this.required) {
        pickerEl.append($('<framelix-button data-action="unset" icon="719" title="__framelix_form_select_unset__" theme="transparent" textcolor="red"></framelix-button>').on('click', function (ev) {
          ev.stopPropagation();
          self.destroyDropdown();
          self.setValue(null, true);
        }));
      }
      if (this.dropdown) {
        const dropdownBtn = $(`<framelix-button data-action="open" title="__framelix_form_select_open__" icon="711" theme="transparent"  textcolor="var(--color-page-text)"></framelix-button>`).on('click', function (ev) {
          ev.stopPropagation();
          self.toggleDropdown();
        });
        pickerEl.on('click', function (ev) {
          ev.preventDefault();
          dropdownBtn.trigger('click');
        });
        pickerEl.on('keydown', function (ev) {
          if (ev.key === ' ') {
            ev.stopPropagation();
            ev.preventDefault();
            dropdownBtn.trigger('click');
          }
        });
        pickerEl.append(dropdownBtn);
      }
      this.container.on(FramelixFormField.EVENT_CHANGE_USER, function () {
        if (!self.loadUrlOnChange) return;
        const isJsCall = self.loadUrlOnChange.includes('/jscall?phpMethod');
        let target = self.loadUrlTarget;
        if (isJsCall && target !== 'none') target = 'modal';
        let callUrl;
        if (isJsCall) {
          const params = {};
          params[self.name] = self.getValue();
          callUrl = FramelixRequest.jsCall(self.loadUrlOnChange, params);
          if (target === 'modal') {
            FramelixModal.show({
              bodyContent: callUrl
            });
          } else {
            callUrl.checkHeaders();
          }
        } else {
          callUrl = new URL(self.loadUrlOnChange, window.location.href);
          callUrl.searchParams.append(self.name, FramelixStringUtils.stringify(self.getValue(), ','));
          if (target === 'modal' || target === 'none') {
            FramelixModal.show({
              bodyContent: FramelixRequest.request('get', callUrl.href)
            });
          } else if (target === '_blank') {
            window.open(callUrl.href);
          } else {
            window.location.href = callUrl.href;
          }
        }
      });
      this.initOptionsContainer(this.optionsContainer);
    }
    this.setValue(this.defaultValue);
  }
}
FramelixFormField.classReferences['FramelixFormFieldSelect'] = FramelixFormFieldSelect;
;
class FramelixFormFieldBic extends FramelixFormFieldText {
  maxWidth = 200;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let val = this.stringifyValue(value).replace(/[^a-z0-9]/ig, '').toUpperCase();
    super.setValue(val, isUserChange);
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.off('change input').on('change input', function () {
      self.setValue(this.value, true);
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldBic'] = FramelixFormFieldBic;
;
class FramelixFormFieldCaptcha extends FramelixFormField {
  static TYPE_RECAPTCHA_V2 = 'recaptchav2';
  static TYPE_RECAPTCHA_V3 = 'recaptchav3';
  type;
  publicKeys;
  trackingAction = 'framelix';
  renderAfterUserInput = true;
  signedUrlVerifyToken;
  recaptchaWidgetId = null;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
  }
  getValue() {
    return this.field.find('input').val() || '';
  }
  async renderInternal() {
    await super.renderInternal();
    if (this.disabled) {
      return;
    }
    if (!this.renderAfterUserInput || !this.form) {
      await render();
      return;
    }
    const self = this;
    this.form.container.one('change input paste', function () {
      render();
    }).one('mouseenter', '.framelix-form-buttons framelix-button[data-type="submit"]', function () {
      render();
    });
    const messageContainer = $(`<framelix-alert style="visibility: hidden">&nbsp;</framelix-alert>`);
    self.field.append(messageContainer);
    const alert = messageContainer[0];
    let rendered = false;
    async function render() {
      if (rendered) return;
      rendered = true;
      alert.style.visibility = '';
      alert.updateBodyHtml(`<div class="framelix-loading"></div>&nbsp;&nbsp;${await FramelixLang.get('__framelix_form_validation_captcha_loading__')}`);
      if (self.type === FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V2 || self.type === FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V3) {
        await FramelixDom.includeResource('https://www.google.com/recaptcha/api.js?render=' + (self.publicKeys[FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V3] || 'explicit'), function () {
          return typeof grecaptcha !== 'undefined' && typeof grecaptcha.ready === 'function';
        });
      }
      if (self.type === FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V2) {
        grecaptcha.ready(function () {
          let el = document.createElement('div');
          self.field.html(el);
          self.recaptchaWidgetId = grecaptcha.render(el, {
            'sitekey': self.publicKeys[self.type],
            'theme': $('html').attr('data-color-scheme'),
            'callback': async function () {
              const token = self.field.find('textarea').val();
              let apiResponse = await FramelixRequest.jsCall(self.signedUrlVerifyToken, {
                'token': token,
                'type': self.type
              }).getResponseData();
              if (!apiResponse || !apiResponse.hash) {
                grecaptcha.reset(self.recaptchaWidgetId);
                self.showValidationMessage('__framelix_form_validation_captcha_invalid__');
                return;
              }
              self.hideValidationMessage();
              self.field.append($(`<input type="hidden" name="${self.name}">`).val(token + ':' + apiResponse.hash));
              self.triggerChange(self.field, false);
            }
          });
        });
      }
      if (self.type === FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V3) {
        grecaptcha.ready(async function () {
          let token = await grecaptcha.execute(self.publicKeys[self.type], {
            action: self.trackingAction
          });
          let apiResponse = await FramelixRequest.jsCall(self.signedUrlVerifyToken, {
            'token': token,
            'type': self.type
          }).getResponseData();
          if (!apiResponse || !apiResponse.hash) {
            self.type = FramelixFormFieldCaptcha.TYPE_RECAPTCHA_V2;
            self.render();
          } else {
            self.hideValidationMessage();
            alert.setAttribute('theme', 'success');
            alert.updateBodyHtml(await FramelixLang.get('__framelix_form_validation_captcha_verified__'));
            self.field.append($(`<input type="hidden" name="${self.name}">`).val(token + ':' + apiResponse.hash));
            self.triggerChange(self.field, false);
          }
        });
      }
    }
  }
}
FramelixFormField.classReferences['FramelixFormFieldCaptcha'] = FramelixFormFieldCaptcha;
;
class FramelixFormFieldColor extends FramelixFormField {
  maxWidth = 130;
  colorInput;
  textInput;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    value = value || '';
    if (value.length) {
      value = value.toUpperCase();
      value = value.replace(/[^0-9A-F]/g, '');
      value = '#' + value;
    }
    this.textInput.val(value);
    this.colorInput.val(value);
    this.field.attr('data-empty', !value.length ? '1' : '0');
    this.triggerChange(this.textInput, isUserChange);
  }
  getValue() {
    return this.textInput.val();
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.field.html(`
      <div class="framelix-form-field-input framelix-form-field-container-color-wrap">      
        <input type="text" maxlength="7" tabindex="0" ${this.disabled ? 'disabled' : ''}>  
        <label>
            <input type="color" tabindex="0" ${this.disabled ? 'disabled' : ''}>
            <framelix-icon class="framelix-form-field-container-color-pick" icon="7a5"></framelix-icon>
        </label>
      </div>
    `);
    const inputs = this.container.find('input');
    this.colorInput = inputs.last();
    this.textInput = inputs.first();
    this.textInput.attr('name', this.name);
    this.textInput.on('change input', function () {
      self.setValue(self.textInput.val(), true);
    });
    this.colorInput.on('change input', function () {
      self.setValue(self.colorInput.val(), true);
    });
    this.setValue(this.defaultValue || '');
  }
}
FramelixFormField.classReferences['FramelixFormFieldColor'] = FramelixFormFieldColor;
;
class FramelixFormFieldDate extends FramelixFormFieldText {
  maxWidth = 150;
  showDatepickerBtn = true;
  minDate = null;
  maxDate = null;
  datepickerPopup = null;
  datepickerBtn = null;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let originalVal = this.input.val();
    let date = FramelixDateUtils.anyToDayJs(value);
    if (date) {
      date = FramelixDateUtils.anyToFormat(date);
    } else {
      date = '';
    }
    if (originalVal !== date) {
      this.input.val(date);
      this.triggerChange(this.input, isUserChange);
    }
  }
  isDateValid(date) {
    const value = FramelixDateUtils.anyToDayJs(date);
    if (value) {
      if (this.minDate !== null) {
        if (FramelixDateUtils.compare(value, this.minDate) === '<') {
          return false;
        }
      }
      if (this.maxDate !== null) {
        if (FramelixDateUtils.compare(value, this.maxDate) === '>') {
          return false;
        }
      }
    }
    return true;
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = FramelixDateUtils.anyToDayJs(this.getValue());
    if (value) {
      if (this.minDate !== null) {
        if (FramelixDateUtils.compare(value, this.minDate) === '<') {
          return await FramelixLang.get('__framelix_form_validation_mindate__', {
            'date': FramelixDateUtils.anyToFormat(this.minDate)
          });
        }
      }
      if (this.maxDate !== null) {
        if (FramelixDateUtils.compare(value, this.maxDate) === '>') {
          return await FramelixLang.get('__framelix_form_validation_maxdate__', {
            'date': FramelixDateUtils.anyToFormat(this.maxDate)
          });
        }
      }
    }
    return true;
  }
  async showDatepicker() {
    async function renderCalendar() {
      const table = $(`
        <table>
            <thead>
                <tr>
                
                </tr>
            </thead>
            <tbody>
            
            </tbody>
        </table>        
      `);
      container.find('.framelix-form-field-date-popup-calendar').empty().append(table);
      const trHead = table.find('tr');
      const tbody = table.find('tbody');
      for (let i = 1; i <= 7; i++) {
        trHead.append(`<th>${await FramelixLang.get('__framelix_dayshort_' + i + '__')}</th>`);
      }
      let today = dayjs().format('YYYY-MM-DD');
      let monthNow = monthSelected.month();
      let weekCurrent = monthSelected.clone().date(1).isoWeekday(1);
      while (true) {
        const tr = $(`<tr></tr>`);
        tbody.append(tr);
        let dateNow = weekCurrent.clone();
        for (let i = 1; i <= 7; i++) {
          const td = $(`<td data-date="${dateNow.format('YYYY-MM-DD')}">${dateNow.date()}</td>`);
          tr.append(td);
          if (dateNow.format('YYYY-MM-DD') === dateSelected.format('YYYY-MM-DD')) {
            td.attr('data-selected', '1');
          }
          if (monthNow !== dateNow.month()) {
            td.attr('data-other-month', '1');
          }
          if (dateNow.isoWeekday() >= 6) {
            td.attr('data-satsun', '1');
          }
          if (!self.isDateValid(dateNow)) {
            td.attr('data-disabled', '1');
          }
          if (today === dateNow.format('YYYY-MM-DD')) {
            td.attr('data-today', '1');
          }
          dateNow = dateNow.add(1, 'day');
        }
        weekCurrent = weekCurrent.add(1, 'week');
        if (weekCurrent.month() !== monthNow) break;
      }
    }
    async function renderMonthSelect() {
      const monthSelect = new FramelixFormFieldSelect();
      monthSelect.showResetButton = false;
      monthSelect.maxWidth = null;
      monthSelect.container.on(FramelixFormField.EVENT_CHANGE_USER, function () {
        monthSelected = FramelixDateUtils.anyToDayJs(monthSelect.getValue());
        renderCalendar();
        renderMonthSelect();
      });
      for (let i = -12; i <= 12; i++) {
        const date = monthSelected.clone().add(i, 'month');
        monthSelect.addOption(FramelixDateUtils.anyToFormat(date, 'YYYY-MM-01'), (await FramelixLang.get('__framelix_month_' + (date.month() + 1) + '__')) + ' ' + date.year());
      }
      monthSelect.defaultValue = FramelixDateUtils.anyToFormat(monthSelected, 'YYYY-MM-01');
      monthSelect.render();
      container.find('.framelix-form-field-date-popup-monthpicker-select').empty().append(monthSelect.container);
    }
    const self = this;
    this.datepickerPopup = FramelixPopup.show(this.field, '', {
      color: 'light',
      placement: 'bottom',
      width: null
    });
    this.datepickerPopup.destroyed.then(function () {
      self.datepickerPopup = null;
    });
    let dateSelected = FramelixDateUtils.anyToDayJs(this.getValue()) || dayjs();
    let monthSelected = dateSelected.clone();
    renderMonthSelect();
    const container = $(`<div class="framelix-form-field-date-popup">
        <div class="framelix-form-field-date-popup-monthpicker">
            <framelix-button theme="transparent" icon="704" data-action="month_switch" data-dir="-1"></framelix-button>
            <div class="framelix-form-field-date-popup-monthpicker-select"></div>
            <framelix-button theme="transparent" icon="705" data-action="month_switch" data-dir="1"></framelix-button>
        </div>
        <div class="framelix-form-field-date-popup-calendar"></div>
        <div class="framelix-form-field-date-popup-actions">
            <framelix-button theme="primary" icon="719" data-action="delete">Löschen</framelix-button>
            <framelix-button theme="primary" icon="72a" data-action="today">Heute</framelix-button>
        </div>
    </div>`);
    container.on('click', '[data-date]', function () {
      var _self$datepickerPopup, _self$datepickerBtn;
      self.setValue(this.dataset.date, true);
      (_self$datepickerPopup = self.datepickerPopup) === null || _self$datepickerPopup === void 0 || _self$datepickerPopup.destroy();
      (_self$datepickerBtn = self.datepickerBtn) === null || _self$datepickerBtn === void 0 || _self$datepickerBtn.trigger('focus');
    });
    container.on('click', '[data-action]', function () {
      var _self$datepickerPopup2, _self$datepickerBtn2, _self$datepickerPopup3, _self$datepickerBtn3;
      switch (this.dataset.action) {
        case 'delete':
          self.setValue(null, true);
          (_self$datepickerPopup2 = self.datepickerPopup) === null || _self$datepickerPopup2 === void 0 || _self$datepickerPopup2.destroy();
          (_self$datepickerBtn2 = self.datepickerBtn) === null || _self$datepickerBtn2 === void 0 || _self$datepickerBtn2.trigger('focus');
          break;
        case 'today':
          self.setValue(dayjs(), true);
          (_self$datepickerPopup3 = self.datepickerPopup) === null || _self$datepickerPopup3 === void 0 || _self$datepickerPopup3.destroy();
          (_self$datepickerBtn3 = self.datepickerBtn) === null || _self$datepickerBtn3 === void 0 || _self$datepickerBtn3.trigger('focus');
          break;
        case 'month_switch':
          const dir = parseInt(this.dataset.dir);
          monthSelected = monthSelected.add(dir, 'month');
          renderMonthSelect();
          renderCalendar();
          break;
      }
    });
    this.datepickerPopup.content.empty();
    this.datepickerPopup.content.append(container);
    renderCalendar();
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.attr('type', 'text');
    this.input.on('change input', function (ev) {
      if (ev.key === 'Enter' || ev.type === 'change') {
        let val = self.stringifyValue(self.getValue());
        val = val.trim().replace(/[^0-9\.]/ig, '.');
        if (!val.match(/[^0-9]/) && val >= 1) {
          const now = new Date();
          let day = now.getDate();
          let month = now.getMonth() + 1;
          let year = now.getFullYear();
          if (val.length <= 2) {
            day = parseInt(val);
          } else if (val.length === 3) {
            day = parseInt(val.substring(0, 1));
            month = parseInt(val.substring(1));
          } else if (val.length === 4) {
            day = parseInt(val.substring(0, 2));
            month = parseInt(val.substring(2));
          } else if (val.length === 6) {
            day = parseInt(val.substring(0, 2));
            month = parseInt(val.substring(2, 4));
            year = parseInt('20' + val.substring(4));
          } else if (val.length === 8) {
            day = parseInt(val.substring(0, 2));
            month = parseInt(val.substring(2, 4));
            year = parseInt(val.substring(4));
          } else {
            val = null;
          }
          if (val !== null) {
            let date = dayjs();
            val = date.date(day).month(month - 1).year(year);
          }
        }
        self.setValue(val);
      }
    });
    if (this.showDatepickerBtn) {
      this.datepickerBtn = $(`<framelix-button icon="72a" theme="primary"></framelix-button>`);
      this.field.attr('data-field-with-button', '1');
      this.field.append(this.datepickerBtn);
      this.datepickerBtn.on('click', function () {
        self.showDatepicker();
      });
    }
  }
}
FramelixFormField.classReferences['FramelixFormFieldDate'] = FramelixFormFieldDate;
;
class FramelixFormFieldDateTime extends FramelixFormFieldText {
  maxWidth = 200;
  minDateTime = null;
  maxDateTime = null;
  allowSeconds = false;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    super.setValue(this.prepareValue(this.stringifyValue(value)), isUserChange);
  }
  prepareValue(value) {
    if (!this.allowSeconds && value.length > 16) {
      value = value.substr(0, 16);
    }
    return value.replace(/ /, 'T');
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.attr('type', 'datetime-local');
    if (this.minDateTime) this.input.attr('min', this.prepareValue(this.minDateTime));
    if (this.maxDateTime) this.input.attr('max', this.prepareValue(this.maxDateTime));
    if (this.allowSeconds) {
      this.field.css('maxWidth', this.maxWidth !== null ? typeof this.maxWidth === 'number' ? this.maxWidth + 30 + 'px' : this.maxWidth : '');
      this.input.attr('step', 1);
    }
    this.input.off('change input').on('change', function () {
      self.setValue(this.value, true);
    });
    self.setValue(this.defaultValue);
  }
}
FramelixFormField.classReferences['FramelixFormFieldDateTime'] = FramelixFormFieldDateTime;
;
class FramelixFormFieldEditor extends FramelixFormField {
  static tinymceIncluded = false;
  textarea;
  minHeight = null;
  maxHeight = null;
  spellcheck = false;
  minLength = null;
  maxLength = null;
  editor;
  tinymcePath;
  setValue(value) {
    var _this$_editor;
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    if (this.textarea.val() === value) {
      return;
    }
    this.textarea.val(value);
    (_this$_editor = this._editor) === null || _this$_editor === void 0 || _this$_editor.setContent(value);
    this.triggerChange(this.textarea, isUserChange);
  }
  getValue() {
    return this.textarea.val();
  }
  async initializeTinymce(element, height) {
    const self = this;
    return new Promise(async function (resolve) {
      if (!FramelixFormFieldEditor.tinymceIncluded) {
        FramelixFormFieldEditor.tinymceIncluded = true;
        await FramelixDom.includeResource(self.tinymcePath, 'tinymce');
      }
      let plugins = ['advlist', 'autolink', 'lists', 'image', 'link', 'searchreplace', 'visualblocks', 'code', 'table', 'code'];
      const darkMode = $('html').attr('data-color-scheme') === 'dark';
      if (!height) plugins.push('autoresize');
      tinymce.init({
        target: element[0],
        language: FramelixLang.lang,
        browser_spellcheck: self.spellcheck,
        'height': height,
        menubar: 'edit insert view format table tools',
        statusbar: false,
        readonly: self.disabled,
        pagebreak_separator: '<div class="framelix-form-field-editor-pagebreak" pagebreak="true"></div>',
        pagebreak_split_block: true,
        ui_mode: 'split',
        'plugins': plugins,
        contextmenu: 'copy | paste | link image inserttable | cell row column deletetable',
        toolbar: 'insert | undo redo | fontsizeselect | bold italic underline forecolor | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | removeformat | placeholders textconditions pagebreak',
        content_css: darkMode ? 'dark' : 'default',
        skin: darkMode ? 'oxide-dark' : 'oxide',
        autoresize_bottom_margin: 10,
        paste_as_text: true,
        paste_block_drop: true,
        relative_urls: false,
        remove_script_host: false,
        promotion: false,
        min_height: self.minHeight,
        max_height: self.maxHeight,
        init_instance_callback: function (editor) {
          tinymce.triggerSave();
          editor.on('change', function () {
            editor.save();
          });
          self.editor = editor;
          resolve(editor);
        },
        setup: function (editor) {}
      });
    });
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = this.getValue();
    if (this.minLength !== null) {
      if (value.length < this.minLength) {
        return await FramelixLang.get('__framelix_form_validation_minlength__', {
          'number': this.minLength
        });
      }
    }
    if (this.maxLength !== null) {
      if (value.length > this.maxLength) {
        return await FramelixLang.get('__framelix_form_validation_maxlength__', {
          'number': this.maxLength
        });
      }
    }
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.textarea = $(`<textarea></textarea>`);
    this.field.html(this.textarea);
    this.textarea.attr('name', this.name);
    this.textarea.val(this.defaultValue || '');
    await this.initializeTinymce(this.textarea);
  }
}
FramelixFormField.classReferences['FramelixFormFieldEditor'] = FramelixFormFieldEditor;
;
class FramelixFormFieldEmail extends FramelixFormFieldText {
  maxWidth = 400;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let val = this.stringifyValue(value).toLowerCase();
    super.setValue(val, isUserChange);
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = this.getValue();
    if (value.length) {
      if (!value.match(new RegExp('^[a-zA-Z0-9' + FramelixStringUtils.escapeRegex('.!#$%&’*+/=?^_`{|}~-') + ']+@[a-zA-Z0-9\\-]+\\.[a-zA-Z0-9\\-.]{2,}'))) {
        return await FramelixLang.get('__framelix_form_validation_email__');
      }
    }
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.off('change input').on('change input', function () {
      self.setValue(this.value, true);
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldEmail'] = FramelixFormFieldEmail;
;
class FramelixFormFieldFile extends FramelixFormField {
  inputFile;
  multiple = false;
  allowedFileTypes;
  files = {};
  filesContainer;
  minSelectedFiles = null;
  maxSelectedFiles = null;
  buttonLabel = '__framelix_form_file_pick__';
  instantDelete = false;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    for (let filename in this.files) {
      this.removeFile(filename, false, isUserChange);
    }
    if (FramelixObjectUtils.hasKeys(value)) {
      for (let i in value) this.addFile(value[i], false, isUserChange);
    }
    this.triggerChange(this.inputFile, isUserChange);
  }
  getValue() {
    let arr = [];
    for (let fileId in this.files) {
      if (!(this.files[fileId].file instanceof File)) continue;
      arr.push(this.files[fileId].file);
    }
    return arr.length ? arr : null;
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = FramelixObjectUtils.countKeys(this.getValue());
    if (this.minSelectedFiles !== null) {
      if (value < this.minSelectedFiles) {
        return await FramelixLang.get('__framelix_form_validation_minselectedfiles__', {
          'number': this.minSelectedFiles
        });
      }
    }
    if (this.maxSelectedFiles !== null) {
      if (value > this.maxSelectedFiles) {
        return await FramelixLang.get('__framelix_form_validation_maxselectedfiles__', {
          'number': this.maxSelectedFiles
        });
      }
    }
    return true;
  }
  addFile(file, triggerChange) {
    let isUserChange = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    if (!this.multiple) {
      for (let filename in this.files) {
        this.removeFile(filename, false);
      }
    }
    const filename = file.name;
    let fileId = file.name;
    const container = $(`<div class="framelix-form-field-file-file">
        <div class="framelix-form-field-file-file-label">
            <framelix-button theme="light" class="framelix-form-field-file-file-remove" title="__framelix_form_file_delete_queue__" icon="719"></framelix-button>
            <div class="framelix-form-field-file-file-label-text">
                ${file.url ? '<a href="' + file.url + '">' + filename + '</a>' : filename}
            </div>
            <div class="framelix-form-field-file-file-label-size">${FramelixNumberUtils.filesizeToUnit(file.size, 'mb')}</div>
        </div>    
      </div>`);
    if (file.id) {
      fileId = file.id;
      container.find('.framelix-form-field-file-file-remove').attr('title', '__framelix_form_file_delete_existing__');
      container.attr('data-id', file.id);
      container.attr('data-delete-url', file.deleteUrl);
      container.append(`<input type="hidden" name="${this.name}[${file.id}]" value="1">`);
    }
    container.attr('data-file-internal-id', fileId);
    this.files[fileId] = {
      'file': file,
      'container': container
    };
    this.filesContainer.append(container);
    if (triggerChange) this.triggerChange(this.inputFile, isUserChange);
  }
  removeFile(fileId, triggerChange) {
    let isUserChange = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;
    const fileRow = this.files[fileId];
    if (!fileRow) return;
    if (fileRow.uploadRequest) fileRow.uploadRequest.abort();
    fileRow.container.remove();
    delete this.files[fileId];
    if (triggerChange) this.triggerChange(this.inputFile, isUserChange);
  }
  async uploadFiles(jsCallUrl, parameters) {
    const arr = [];
    for (let fileId in this.files) {
      const req = this.uploadFile(jsCallUrl, fileId, parameters);
      if (req) {
        await req.finished;
        arr.push(req);
      }
    }
    return arr;
  }
  uploadFile(jsCallUrl, fileId, parameters) {
    const row = this.files[fileId];
    if (!row.file) return null;
    const formData = new FormData();
    formData.append('file', row.file);
    if (parameters) {
      formData.append('parameters', JSON.stringify(parameters));
    }
    const req = FramelixRequest.jsCall(jsCallUrl, formData, row.container);
    row.uploadRequest = req;
    req.finished.then(function () {
      row.file = null;
      row.container.attr('data-uploaded', '1');
    });
    return req;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.field.html(`      
        <framelix-button block class="framelix-form-field-file-button" icon="708">${this.buttonLabel}</framelix-button>
        <label style="display: none"><input type="file" ${this.disabled ? 'disabled' : ''}></label>
        <div class="framelix-form-field-file-files"></div>
    `);
    if (this.disabled) {
      this.field.children().first().addClass('hidden');
    }
    this.filesContainer = this.field.find('.framelix-form-field-file-files');
    this.inputFile = this.field.find('input[type=\'file\']');
    if (this.allowedFileTypes) this.inputFile.attr('accept', this.allowedFileTypes);
    if (this.multiple) this.inputFile.attr('multiple', true);
    this.inputFile.on('change', function (ev) {
      if (!ev.target.files) return;
      ev.stopPropagation();
      for (let i = 0; i < ev.target.files.length; i++) {
        self.addFile(ev.target.files[i], false);
      }
      self.triggerChange(self.inputFile, true);
    });
    this.filesContainer.on('click', '.framelix-form-field-file-file-remove', async function () {
      const fileEntry = $(this).closest('.framelix-form-field-file-file');
      if (fileEntry.attr('data-id')) {
        const deleteUrl = fileEntry.attr('data-delete-url');
        if (self.instantDelete && deleteUrl) {
          if (!(await FramelixModal.confirm(await FramelixLang.get('__framelix_delete_sure__')).confirmed)) {
            return;
          }
          const response = await FramelixRequest.jsCall(deleteUrl).getResponseData();
          if (response === true) {
            fileEntry.remove();
            FramelixToast.success('__framelix_deleted__');
          } else {
            FramelixToast.error(await FramelixLang.get('__framelix_error__', [response]));
          }
          return;
        }
        fileEntry.toggleClass('framelix-form-field-file-file-strikethrough');
        const deleteFlag = fileEntry.hasClass('framelix-form-field-file-file-strikethrough');
        fileEntry.find('input').val(!deleteFlag ? '1' : '0');
      } else {
        self.removeFile(fileEntry.attr('data-file-internal-id'), true);
      }
    });
    this.container.on('click', '.framelix-form-field-file-button', function () {
      $(this).next().trigger('click');
    });
    this.container.on('dragover', function (ev) {
      ev.preventDefault();
    });
    this.container.on('drop', function (ev) {
      ev.preventDefault();
      for (let i = 0; i < ev.dataTransfer.files.length; i++) {
        self.addFile(ev.dataTransfer.files[i], false);
      }
      self.triggerChange(self.inputFile, true);
    });
    this.setValue(this.defaultValue);
  }
}
FramelixFormField.classReferences['FramelixFormFieldFile'] = FramelixFormFieldFile;
;
class FramelixFormFieldHidden extends FramelixFormField {
  input;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    this.input.val(value);
    this.triggerChange(this.input, isUserChange);
  }
  getValue() {
    return this.input.val();
  }
  async renderInternal() {
    await super.renderInternal();
    this.input = $(`<input type="hidden">`);
    this.input.attr('name', this.name);
    this.field.html(this.input);
    this.setValue(this.defaultValue || '');
  }
}
FramelixFormField.classReferences['FramelixFormFieldHidden'] = FramelixFormFieldHidden;
;
class FramelixFormFieldHtml extends FramelixFormField {
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    this.field.html(value);
    this.triggerChange(this.field, isUserChange);
  }
  getValue() {
    return null;
  }
  async renderInternal() {
    await super.renderInternal();
    this.setValue(this.defaultValue || '');
  }
}
FramelixFormField.classReferences['FramelixFormFieldHtml'] = FramelixFormFieldHtml;
;
class FramelixFormFieldIban extends FramelixFormFieldText {
  maxWidth = 300;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let val = this.stringifyValue(value).replace(/[^a-z0-9]/ig, '').toUpperCase();
    if (val.length) {
      val = val.match(/.{1,4}/g).join(' ');
    }
    super.setValue(val, isUserChange);
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.off('change input').on('change input', function () {
      self.setValue(this.value, true);
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldIban'] = FramelixFormFieldIban;
;
class FramelixFormFieldNumber extends FramelixFormFieldText {
  maxWidth = 150;
  commaSeparator = ',';
  thousandSeparator = ',';
  decimals = 0;
  min = null;
  max = null;
  input;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let originalVal = this.input.val();
    let val = FramelixNumberUtils.format(value, this.decimals, this.commaSeparator, this.thousandSeparator);
    if (val !== originalVal) {
      this.input.val(val);
      this.triggerChange(this.input, isUserChange);
    }
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = FramelixNumberUtils.toNumber(this.getValue(), this.decimals, this.commaSeparator);
    if (this.min !== null) {
      if (value < this.min) {
        return await FramelixLang.get('__framelix_form_validation_min__', {
          'number': FramelixNumberUtils.format(this.min, this.decimals, this.commaSeparator, this.thousandSeparator)
        });
      }
    }
    if (this.max !== null) {
      if (value > this.max) {
        return await FramelixLang.get('__framelix_form_validation_max__', {
          'number': FramelixNumberUtils.format(this.max, this.decimals, this.commaSeparator, this.thousandSeparator)
        });
      }
    }
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.attr('inputmode', 'decimal');
    this.input.on('change', function () {
      self.setValue(this.value, true);
    });
    this.input.on('input', function () {
      self.triggerChange(self.input, true);
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldNumber'] = FramelixFormFieldNumber;
;
class FramelixFormFieldPassword extends FramelixFormFieldText {
  maxWidth = 400;
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.field.attr('data-field-with-button', '1');
    this.field.append(`<framelix-button theme="primary" title="__framelix_form_password_toggle__" icon="733"></framelix-button>`);
    this.field.find('framelix-button').on('click keydown', function (ev) {
      if (ev.key === ' ' || ev.key === 'Enter' || !ev.key) {
        self.input.attr('type', self.input.attr('type') === self.type ? 'text' : 'password');
      }
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldPassword'] = FramelixFormFieldPassword;
;
class FramelixFormFieldSearch extends FramelixFormField {
  maxWidth = 400;
  multiple = false;
  signedUrlSearch;
  continuousSearch = true;
  initialSelectedOptions = null;
  resultPopup = null;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    this.triggerChange(this.field, isUserChange);
  }
  getValue() {
    const values = FormDataJson.toJson(this.field.find('.framelix-form-field-search-selected-options')[0], {
      'includeDisabled': true,
      'flatList': true
    });
    let arr = [];
    for (let i = 0; i < values.length; i++) {
      arr.push(values[i][1]);
    }
    if (!arr.length) return null;
    return this.multiple ? arr : arr[0];
  }
  getOptionHtml(value, label, checked) {
    const option = $(`
        <label class="framelix-form-field-select-option">
            <div class="framelix-form-field-select-option-checkbox">
                <input type="checkbox" name="${this.name + (this.multiple ? '[]' : '')}" ${this.disabled ? 'disabled' : ''}>
            </div>
            <div class="framelix-form-field-select-option-label"></div>
        </label>
      `);
    const input = option.find('input');
    option.find('.framelix-form-field-select-option-label').html(label);
    input.attr('value', value);
    input.prop('checked', checked);
    return option;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.container.attr('data-multiple', this.multiple ? '1' : '0');
    this.field.html(`
      <div class="framelix-form-field-search-container">
        <div class="framelix-form-field-search-input"><div class="framelix-form-field-container" data-field-with-button="1"><input type="search" placeholder="${await FramelixLang.get('__framelix_form_select_search__')}" class="framelix-form-field-input" spellcheck="false" data-continuous-search="${this.continuousSearch ? '1' : '0'}" ${this.disabled ? 'disabled' : ''}><framelix-button theme="primary" icon="744"></framelix-button></div></div>
        <div class="framelix-form-field-search-selected-options framelix-form-field-input"></div>
      </div>
    `);
    const searchInputContainer = this.field.find('.framelix-form-field-search-input');
    const searchInput = searchInputContainer.find('input');
    const searchButton = searchInputContainer.find('framelix-button');
    const selectedOptionsContainer = this.field.find('.framelix-form-field-search-selected-options');
    if (this.initialSelectedOptions && this.initialSelectedOptions.keys.length) {
      for (let i = 0; i < this.initialSelectedOptions.keys.length; i++) {
        const value = this.initialSelectedOptions.keys[i];
        const label = this.initialSelectedOptions.values[i];
        selectedOptionsContainer.append(this.getOptionHtml(value, label, true));
        if (!this.multiple) break;
      }
    }
    if (!this.disabled) {
      searchButton.on('click', function () {
        searchInput.trigger('search-start');
      });
      searchInput.on('search-start', async function () {
        let query = this.value.trim();
        if (query.length) {
          const currentValue = self.getValue();
          if (!self.resultPopup) {
            function updateValue() {
              let existingOptions = {};
              selectedOptionsContainer.find('input').each(function () {
                existingOptions[this.value] = this;
              });
              if (self.resultPopup && self.resultPopup.popperEl) {
                self.resultPopup.popperEl.find('input').each(function () {
                  if (existingOptions[this.value]) {
                    existingOptions[this.value].checked = this.checked;
                    return;
                  }
                  if (!this.checked) return true;
                  let optionEl = $(this).closest('.framelix-form-field-select-option');
                  optionEl.find('input').prop('checked', this.checked);
                  if (!self.multiple) {
                    selectedOptionsContainer.empty();
                    selectedOptionsContainer.append(optionEl);
                    return false;
                  }
                  selectedOptionsContainer.append(optionEl);
                });
              }
            }
            self.resultPopup = FramelixPopup.show(searchInput, `<div class="framelix-form-field-search-popup framelix-form-field-input" data-multiple="${self.multiple ? '1' : '0'}"></div>`, {
              closeMethods: 'click-outside',
              placement: 'bottom-start',
              appendTo: searchInputContainer,
              padding: '',
              offset: [0, 0]
            });
            self.resultPopup.destroyed.then(function () {
              updateValue();
              self.resultPopup = null;
            });
            self.resultPopup.popperEl.on('change', function () {
              updateValue();
              if (!self.multiple) {
                var _self$resultPopup;
                (_self$resultPopup = self.resultPopup) === null || _self$resultPopup === void 0 || _self$resultPopup.destroy();
              }
            });
          }
          searchButton.attr('disabled', '').addClass('framelix-pulse');
          let options = await FramelixRequest.jsCall(self.signedUrlSearch, {
            'query': this.value
          }).getResponseData();
          searchButton.attr('disabled', '').removeClass('framelix-pulse');
          const content = self.resultPopup.popperEl.find('.framelix-popup-inner > .framelix-form-field-input');
          content.html('');
          if (!options.keys.length) {
            content.html(`<div class="framelix-form-field-select-option">${await FramelixLang.get('__framelix_form_search_noresult__')}</div>`);
          } else {
            if (options.keys) {
              for (let i = 0; i < options.keys.length; i++) {
                const value = self.stringifyValue(options.keys[i]);
                content.append(self.getOptionHtml(value, options.values[i], value === currentValue || FramelixObjectUtils.hasValue(currentValue, value)));
              }
            }
          }
        } else {
          if (self.resultPopup) {
            self.resultPopup.destroy();
          }
        }
      });
      searchInput.on('keydown', function (ev) {
        if (self.resultPopup && ev.key === 'Tab' && !ev.shiftKey) {
          ev.preventDefault();
          self.resultPopup.popperEl.find('label').first().trigger('focus');
        }
      });
    }
  }
}
FramelixFormField.classReferences['FramelixFormFieldSearch'] = FramelixFormFieldSearch;
;
class FramelixFormFieldTextarea extends FramelixFormField {
  static autoheightData = {};
  placeholder = null;
  textarea;
  minHeight = null;
  maxHeight = null;
  spellcheck = false;
  minLength = null;
  maxLength = null;
  static initLate() {
    let resizeTo = null;
    $(window).on('resize', function () {
      if (resizeTo) return;
      resizeTo = setTimeout(function () {
        $('.framelix-textarea-autoheight-active').each(function () {
          FramelixFormFieldTextarea.calculateAutoheight(this);
        });
        resizeTo = null;
      }, 500);
    });
    FramelixDom.addChangeListener('framelix-textarea', function () {
      $('textarea').filter('.framelix-textarea-autoheight').not('.framelix-textarea-autoheight-active').each(function () {
        $(this).addClass('framelix-textarea-autoheight-active');
        FramelixFormFieldTextarea.calculateAutoheight(this);
        $(this).on('input change', function () {
          FramelixFormFieldTextarea.calculateAutoheight(this);
        });
      });
    });
  }
  static calculateAutoheight(textarea) {
    textarea.ignoreDomObserver = true;
    const cachedData = FramelixFormFieldTextarea.autoheightData;
    if (!cachedData.helper) {
      const helperContainer = $('<div class="framelix-textarea-autoheight-helper" aria-hidden="true"><textarea></textarea></div>');
      cachedData.helper = helperContainer.children()[0];
      cachedData.helper.ignoreDomObserver = true;
      $('body').append(helperContainer);
    }
    const helperTextarea = cachedData.helper;
    if (cachedData.lastTextarea !== textarea) {
      const styles = window.getComputedStyle(textarea);
      cachedData.lastTextarea = textarea;
      cachedData.lastHeight = parseInt(textarea.style.height.replace(/[^0-9]/g, ''));
      if (styles.cssText !== '') {
        helperTextarea.style.cssText = styles.cssText;
      } else {
        helperTextarea.style.cssText = Object.values(styles).reduce((css, propertyName) => `${css}${propertyName}:${styles.getPropertyValue(propertyName)};`);
      }
    }
    helperTextarea.style.width = textarea.clientWidth + 'px';
    helperTextarea.value = textarea.value;
    helperTextarea.style.height = '5px';
    if (cachedData.lastHeight !== helperTextarea.scrollHeight) {
      cachedData.lastHeight = helperTextarea.scrollHeight;
      textarea.style.height = helperTextarea.scrollHeight + 'px';
    }
  }
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    if (this.textarea.val() === value) {
      return;
    }
    this.textarea.val(value);
    this.triggerChange(this.textarea, isUserChange);
    FramelixFormFieldTextarea.calculateAutoheight(this.textarea[0]);
  }
  getValue() {
    return this.textarea.val();
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    const value = this.getValue();
    if (this.minLength !== null) {
      if (value.length < this.minLength) {
        return await FramelixLang.get('__framelix_form_validation_minlength__', {
          'number': this.minLength
        });
      }
    }
    if (this.maxLength !== null) {
      if (value.length > this.maxLength) {
        return await FramelixLang.get('__framelix_form_validation_maxlength__', {
          'number': this.maxLength
        });
      }
    }
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.textarea = $(`<textarea class="framelix-form-field-input framelix-textarea-autoheight framelix-textarea-autoheight-active"></textarea>`);
    this.field.html(this.textarea);
    if (this.placeholder !== null) this.textarea.attr('placeholder', this.placeholder);
    if (this.disabled) {
      this.textarea.attr('disabled', true);
    }
    if (this.minHeight !== null) this.textarea.css('minHeight', this.minHeight + 'px');
    if (this.maxHeight !== null) this.textarea.css('maxHeight', this.maxHeight + 'px');
    this.textarea.attr('spellcheck', this.spellcheck ? 'true' : 'false');
    this.textarea.attr('name', this.name);
    this.textarea.on('change input', function (ev) {
      ev.stopPropagation();
      self.triggerChange(self.textarea, true);
      FramelixFormFieldTextarea.calculateAutoheight(self.textarea[0]);
    });
    this.textarea.val(this.defaultValue || '');
    FramelixIntersectionObserver.onGetVisible(self.textarea[0], function () {
      self.textarea[0].style.height = '5px';
      self.textarea[0].style.height = self.textarea[0].scrollHeight + 'px';
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldTextarea'] = FramelixFormFieldTextarea;
FramelixInit.late.push(FramelixFormFieldTextarea.initLate);
;
class FramelixFormFieldTime extends FramelixFormFieldText {
  maxWidth = 90;
  allowSeconds = false;
  minTime = null;
  maxTime = null;
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input.attr('type', 'time');
    if (this.minTime) this.input.attr('min', this.minTime);
    if (this.maxTime) this.input.attr('max', this.maxTime);
    if (this.allowSeconds) {
      this.field.css('maxWidth', this.maxWidth !== null ? typeof this.maxWidth === 'number' ? this.maxWidth + 30 + 'px' : this.maxWidth : '');
      this.input.attr('step', 1);
    }
    this.input.off('change input').on('change', function () {
      self.setValue(this.value, true);
    });
    self.setValue(this.defaultValue);
  }
}
FramelixFormField.classReferences['FramelixFormFieldTime'] = FramelixFormFieldTime;
;
class FramelixFormFieldToggle extends FramelixFormField {
  static STYLE_TOGGLE = 'toggle';
  static STYLE_CHECKBOX = 'checkbox';
  style = FramelixFormFieldToggle.STYLE_TOGGLE;
  input;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    this.input.prop('checked', !!value);
    this.triggerChange(this.input, isUserChange);
  }
  getValue() {
    return this.input.prop('checked') ? this.input.val() : null;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.field.html(`<label class="framelix-form-field-input" data-style="${this.style}"><input type="checkbox" ${this.disabled ? 'disabled' : ''} value="1" ${!!this.defaultValue ? 'checked' : ''}></label>`);
    const label = this.field.children();
    this.input = this.field.find('input');
    this.input.attr('name', this.name);
    label.on('focusin', function () {
      if (self.disabled || label.attr('data-user-activated')) return;
      label.attr('data-user-activated', '1');
    });
    this.field.on('keydown', function (ev) {
      if (self.disabled) return;
      if (ev.key === ' ') {
        self.setValue(!self.input.prop('checked'), true);
        ev.stopPropagation();
        ev.preventDefault();
      }
    });
    this.field.on('change', function () {
      self.triggerChange(self.input, true);
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldToggle'] = FramelixFormFieldToggle;
;
class FramelixFormFieldTwoFactorCode extends FramelixFormField {
  input;
  formAutoSubmit = true;
  setValue(value) {
    let isUserChange = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    let i = 1;
    this.field.find('[type=\'text\']').each(function () {
      if (typeof value === 'string' && value.length >= i) {
        this.value = value[i];
      }
      i++;
    });
  }
  getValue() {
    return this.input.val();
  }
  async validate() {
    if (!this.isVisible()) return true;
    const parentValidation = await super.validate();
    if (parentValidation !== true) return parentValidation;
    return true;
  }
  async renderInternal() {
    await super.renderInternal();
    const self = this;
    this.input = $(`<input type="text" inputmode="decimal" autocomplete="one-time-code" class="framelix-form-field-input">`);
    this.input.attr('name', this.name);
    this.container.append(this.input);
    this.field.append(`<div class="framelix-form-field-twofactorcode-label">${await FramelixLang.get('__framelix_form_2fa_enter__')}</div>`);
    this.field.append(this.input);
    this.field.append(`<div class="framelix-form-field-twofactorcode-backup"><framelix-button theme="light">__framelix_form_2fa_usebackup__</framelix-button></div>`);
    this.field.find('.framelix-form-field-twofactorcode-backup framelix-button').on('click', async function () {
      self.field.find('.framelix-form-field-twofactorcode-backup').remove();
      self.field.find('.framelix-form-field-twofactorcode-label').text(await FramelixLang.get('__framelix_form_2fa_enter_backup__'));
      self.input.attr('type', 'text');
      self.input.addClass('framelix-form-field-twofactorcode-backup-input');
      self.input.val('');
    });
    this.field.on('focusin', 'input', function () {
      this.select();
    });
    this.field.on('input', '.framelix-form-field-twofactorcode-backup-input', function (ev) {
      ev.stopPropagation();
      ev.stopImmediatePropagation();
      self.input.val(self.input.val().replace(/[^0-9A-Z]/ig, ''));
      if (self.input.val().length === 10 && self.formAutoSubmit && self.form) {
        self.form.submit();
      }
    });
    this.field.on('input', function () {
      self.input.val(self.input.val().replace(/[^0-9]/ig, ''));
      if (self.input.val().length === 6 && self.formAutoSubmit && self.form) {
        self.form.submit();
      }
    });
  }
}
FramelixFormField.classReferences['FramelixFormFieldTwoFactorCode'] = FramelixFormFieldTwoFactorCode;
;
class FramelixFormFieldVisibilityCondition {
  data = [];
  and() {
    this.data.push({
      'type': 'and'
    });
  }
  or() {
    this.data.push({
      'type': 'or'
    });
    return this;
  }
  empty(fieldName) {
    this.data.push({
      'type': 'empty',
      'field': fieldName
    });
    return this;
  }
  notEmpty(fieldName) {
    this.data.push({
      'type': 'notEmpty',
      'field': fieldName
    });
    return this;
  }
  like(fieldName, value) {
    this.data.push({
      'type': 'like',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  notLike(fieldName, value) {
    this.data.push({
      'type': 'notLike',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  equal(fieldName, value) {
    this.data.push({
      'type': 'equal',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  notEqual(fieldName, value) {
    this.data.push({
      'type': 'notEqual',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  greatherThan(fieldName, value) {
    this.data.push({
      'type': 'greatherThan',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  greatherThanEqual(fieldName, value) {
    this.data.push({
      'type': 'greatherThanEqual',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  lowerThan(fieldName, value) {
    this.data.push({
      'type': 'lowerThan',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  lowerThanEqual(fieldName, value) {
    this.data.push({
      'type': 'lowerThanEqual',
      'field': fieldName,
      'value': value
    });
    return this;
  }
  clear() {
    this.data = [];
  }
}
;