/*
 * Copyright (c) 2024 by FlashInfer team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <gtest/gtest.h>

#include <cstdint>
#include <flashinfer/sampling.cuh>
#include <fstream>
#include <random>

#include "cpu_reference.h"
#include "utils.h"

using namespace flashinfer;

template <typename T, typename IdType>
void _TestTopKSamplingFromProb(size_t batch_size, uint32_t k, size_t vocab_size) {
  std::vector<T> probs_h(batch_size * vocab_size);
  float p = float(k) * 0.1;
  utils::vec_fill_<T>(probs_h, (1 - p) / float((vocab_size - k)));
  std::vector<int32_t> all_token_ids(vocab_size);
  std::iota(all_token_ids.begin(), all_token_ids.end(), 0);
  std::vector<std::set<int32_t>> high_prob_token_ids_sets;
  for (uint32_t i = 0; i < batch_size; ++i) {
    std::vector<int32_t> high_prob_token_ids;
    std::set<int32_t> high_prob_token_ids_set;
    std::sample(all_token_ids.begin(), all_token_ids.end(), std::back_inserter(high_prob_token_ids),
                k, std::mt19937{std::random_device{}()});
    high_prob_token_ids_set.insert(high_prob_token_ids.begin(), high_prob_token_ids.end());
    high_prob_token_ids_sets.emplace_back(high_prob_token_ids_set);
    for (uint32_t j = 0; j < k; ++j) {
      probs_h[i * vocab_size + high_prob_token_ids[j]] = 0.1;
    }
  }

  thrust::device_vector<T> probs_d(probs_h);
  thrust::device_vector<bool> success_d(batch_size);
  thrust::device_vector<IdType> sampled_ids_d(batch_size);
  const int32_t num_samples = 1000;
  const uint32_t max_top_p_rounds = 32;
  std::vector<int32_t> counter(batch_size * vocab_size);
  utils::vec_fill_<int32_t>(counter, 0);
  for (uint32_t draw = 0; draw < num_samples; ++draw) {
    std::vector<T> uniform_samples_h(batch_size * max_top_p_rounds);
    utils::vec_uniform_<T>(uniform_samples_h, 0, 1);
    thrust::device_vector<T> uniform_samples_d(uniform_samples_h);

    auto status = sampling::TopKSamplingFromProb<T, IdType>(
        thrust::raw_pointer_cast(probs_d.data()),
        thrust::raw_pointer_cast(uniform_samples_d.data()),
        thrust::raw_pointer_cast(sampled_ids_d.data()), thrust::raw_pointer_cast(success_d.data()),
        /*top_k_arr=*/nullptr, batch_size, k, vocab_size, max_top_p_rounds, /*deterministic=*/true);

    EXPECT_EQ(status, cudaSuccess) << "TopKSamplingFromProb kernel launch failed, error message: "
                                   << cudaGetErrorString(status);

    thrust::host_vector<bool> success_h(success_d);
    for (uint32_t i = 0; i < batch_size; ++i) {
      EXPECT_TRUE(success_h[i]) << "TopKSamplingFromProb failed for batch " << i;
    }

    thrust::host_vector<IdType> sampled_ids_h(sampled_ids_d);
    for (uint32_t i = 0; i < batch_size; ++i) {
      counter[i * vocab_size + sampled_ids_h[i]]++;
    }
  }

  for (uint32_t i = 0; i < batch_size; ++i) {
    for (uint32_t j = 0; j < vocab_size; ++j) {
      if (counter[i * vocab_size + j] > 0) {
        EXPECT_TRUE(high_prob_token_ids_sets[i].find(j) != high_prob_token_ids_sets[i].end())
            << "high_prob_token_ids_sets[" << i << "] does not contain " << j << std::endl;
      }
    }
  }

  std::cout << "batch_size: " << batch_size << ", k: " << k << ", vocab_size: " << vocab_size
            << ", accuracy test passed." << std::endl;
}

template <typename T, typename IdType>
void _TestTopPSamplingFromProb(size_t batch_size, uint32_t k, size_t vocab_size) {
  std::vector<T> probs_h(batch_size * vocab_size);
  float p = float(k) * 0.1;
  utils::vec_fill_<T>(probs_h, (1 - p) / float((vocab_size - k)));
  std::vector<int32_t> all_token_ids(vocab_size);
  std::iota(all_token_ids.begin(), all_token_ids.end(), 0);
  std::vector<std::set<int32_t>> high_prob_token_ids_sets;
  for (uint32_t i = 0; i < batch_size; ++i) {
    std::vector<int32_t> high_prob_token_ids;
    std::set<int32_t> high_prob_token_ids_set;
    std::sample(all_token_ids.begin(), all_token_ids.end(), std::back_inserter(high_prob_token_ids),
                k, std::mt19937{std::random_device{}()});
    high_prob_token_ids_set.insert(high_prob_token_ids.begin(), high_prob_token_ids.end());
    high_prob_token_ids_sets.emplace_back(high_prob_token_ids_set);
    for (uint32_t j = 0; j < k; ++j) {
      probs_h[i * vocab_size + high_prob_token_ids[j]] = 0.1;
    }
  }

  thrust::device_vector<T> probs_d(probs_h);
  thrust::device_vector<bool> success_d(batch_size);
  thrust::device_vector<IdType> sampled_ids_d(batch_size);
  const int32_t num_samples = 1000;
  const uint32_t max_top_p_rounds = 32;
  std::vector<int32_t> counter(batch_size * vocab_size);
  utils::vec_fill_<int32_t>(counter, 0);
  for (uint32_t draw = 0; draw < num_samples; ++draw) {
    std::vector<T> uniform_samples_h(batch_size * max_top_p_rounds);
    utils::vec_uniform_<T>(uniform_samples_h, 0, 1);
    thrust::device_vector<T> uniform_samples_d(uniform_samples_h);

    auto status = sampling::TopPSamplingFromProb<T, IdType>(
        thrust::raw_pointer_cast(probs_d.data()),
        thrust::raw_pointer_cast(uniform_samples_d.data()),
        thrust::raw_pointer_cast(sampled_ids_d.data()), thrust::raw_pointer_cast(success_d.data()),
        /*top_p_arr=*/nullptr, batch_size, p, vocab_size, max_top_p_rounds, /*deterministic=*/true);

    EXPECT_EQ(status, cudaSuccess) << "TopPSamplingFromProb kernel launch failed, error message: "
                                   << cudaGetErrorString(status);

    thrust::host_vector<bool> success_h(success_d);
    for (uint32_t i = 0; i < batch_size; ++i) {
      EXPECT_TRUE(success_h[i]) << "TopPSamplingFromProb failed for batch " << i;
    }

    thrust::host_vector<IdType> sampled_ids_h(sampled_ids_d);
    for (uint32_t i = 0; i < batch_size; ++i) {
      counter[i * vocab_size + sampled_ids_h[i]]++;
    }
  }

  for (uint32_t i = 0; i < batch_size; ++i) {
    for (uint32_t j = 0; j < vocab_size; ++j) {
      if (counter[i * vocab_size + j] > 0) {
        EXPECT_TRUE(high_prob_token_ids_sets[i].find(j) != high_prob_token_ids_sets[i].end())
            << "high_prob_token_ids_sets[" << i << "] does not contain " << j << std::endl;
      }
    }
  }

  std::cout << "batch_size: " << batch_size << ", p: " << p << ", vocab_size: " << vocab_size
            << ", accuracy test passed." << std::endl;
}

template <typename T, typename IdType>
void _TestSamplingFromProbOneHot(size_t batch_size, size_t vocab_size) {
  std::vector<IdType> sampled_ids_ref_h(batch_size);
  std::vector<T> probs_h(batch_size * vocab_size);
  std::vector<T> uniform_samples_h(batch_size);
  utils::vec_randint_<int32_t>(sampled_ids_ref_h, 0, vocab_size - 1);
  utils::vec_uniform_<T>(uniform_samples_h, 0, 1);
  utils::vec_zero_<T>(probs_h);

  for (uint32_t i = 0; i < batch_size; ++i) {
    probs_h[i * vocab_size + sampled_ids_ref_h[i]] = 1;
  }

  thrust::device_vector<T> probs_d(probs_h);
  thrust::device_vector<T> uniform_samples_d(uniform_samples_h);
  thrust::device_vector<IdType> sampled_ids_d(batch_size);

  auto status = sampling::SamplingFromProb<T>(thrust::raw_pointer_cast(probs_d.data()),
                                              thrust::raw_pointer_cast(uniform_samples_d.data()),
                                              thrust::raw_pointer_cast(sampled_ids_d.data()),
                                              batch_size, vocab_size, /*deterministic=*/true);
  EXPECT_EQ(status, cudaSuccess) << "SamplingFromProb kernel launch failed, error message: "
                                 << cudaGetErrorString(status);

  thrust::host_vector<IdType> sampled_ids_h(sampled_ids_d);

  for (uint32_t i = 0; i < batch_size; ++i) {
    EXPECT_EQ(sampled_ids_h[i], sampled_ids_ref_h[i])
        << "sampled_ids_h[" << i << "] != sampled_ids_ref_h[" << i << "]";
  }

  std::cout << "batch_size: " << batch_size << ", vocab_size: " << vocab_size
            << ", accuracy test passed." << std::endl;
}

template <typename T, typename IdType>
void TestSamplingFromProbOneHot() {
  for (size_t batch_size : {1, 7, 333}) {
    for (size_t d : {24, 97, 1002, 4096, 32000, 128000}) {
      _TestSamplingFromProbOneHot<T, IdType>(batch_size, d);
    }
  }
}

template <typename T, typename IdType>
void TestSamplingFromProbExtremeCase() {
  size_t vocab_size = 5000;
  size_t batch_size = 1;
  // NOTE(Zihao): this test case is contributed by Lequn Chen (https://github.com/abcdabcd987)
  // which could fail sampling algorithm if prefix sum is not monotonic.
  std::vector<T> probs_h{0.00015559792518615723, 0.00048538687406107783, 6.076615318306722e-05,
                         1.0300786925654393e-05, 0.0007201108965091407,  7.774373807478696e-05,
                         6.848858902230859e-05,  0.0001553625479573384,  0.00010724778985604644,
                         4.418972457642667e-05,  0.00017786094394978136, 0.0004431027628015727,
                         3.1250826850737212e-06, 0.00027679704362526536, 7.149282464524731e-05,
                         5.6281915021827444e-05, 3.908606231561862e-05,  0.0003422315348871052,
                         6.0697431763401255e-05, 9.117962326854467e-05,  0.0002959851117338985,
                         4.328760405769572e-05,  0.0005179578438401222,  0.00018602932686917484,
                         0.00016096157196443528, 3.520331665640697e-05,  2.6075373170897365e-05,
                         0.00014143079170025885, 0.00031504034996032715, 8.978219557320699e-05,
                         0.00022176721540745348, 8.185418118955567e-05,  1.3724516065849457e-05,
                         0.00023276064894162118, 4.595539212459698e-05,  7.583790284115821e-05,
                         0.00015678841737098992, 6.0936217778362334e-05, 0.000799499626737088,
                         3.926393037545495e-05,  0.00011619182623689994, 8.124499436235055e-05,
                         0.00038782021147198975, 4.19602838519495e-05,   2.6854606403503567e-05,
                         0.0002459599054418504,  7.132251630537212e-05,  0.00015570450341328979,
                         0.003574109636247158,   0.00013191944162826985, 0.00030993990367278457,
                         4.671538044931367e-05,  0.00012223119847476482, 6.566303636645898e-05,
                         4.4042724766768515e-05, 0.0004951139562763274,  1.1368098967068363e-05,
                         5.659379530698061e-05,  0.0006518339505419135,  0.0002392872702330351,
                         5.788865382783115e-05,  0.00026443233946338296, 6.154876609798521e-05,
                         0.0002908229944296181,  1.8213553630630486e-05, 2.5195698981406167e-05,
                         0.0001355647691525519,  0.00019627073197625577, 0.0001904494856717065,
                         0.0005108538898639381,  0.0001373245904687792,  0.0005138062406331301,
                         3.199433194822632e-05,  0.0001136174178100191,  0.0006180970231071115,
                         8.671481191413477e-05,  0.0004028920957352966,  0.0012885298347100616,
                         7.681828719796613e-05,  1.535531373519916e-05,  8.582916780142114e-05,
                         7.694553642068058e-05,  0.00010334467515349388, 0.00010295744141330943,
                         7.400397589663044e-05,  0.00013254092482384294, 0.0001669518678681925,
                         6.176364695420489e-05,  5.895409412914887e-05,  2.7728778150049038e-05,
                         0.00043061518226750195, 7.880698831286281e-05,  0.0006038321298547089,
                         0.00011174591054441407, 0.00017710828979033977, 9.850953210843727e-05,
                         0.00013414930435828865, 3.590396954677999e-05,  6.402372673619539e-05,
                         0.0019158698851242661,  5.966964818071574e-05,  6.759039388271049e-05,
                         0.00015201824135147035, 0.00012014871754217893, 0.0005053271306678653,
                         3.354372893227264e-05,  0.00017043921980075538, 2.4420391127932817e-05,
                         0.0010445850202813745,  8.231749961851165e-05,  7.600457320222631e-05,
                         4.3014908442273736e-05, 5.203160981182009e-05,  7.767926581436768e-05,
                         8.43061352497898e-05,   9.79983014985919e-05,   0.00032465634285472333,
                         0.00031267126905731857, 1.1368976629455574e-05, 0.00015381975390482694,
                         0.00010642140114214271, 0.00012574950233101845, 6.0344002122292295e-05,
                         0.00018541213648859411, 0.0004064177628606558,  0.00017528065654914826,
                         0.0002162993623642251,  4.486736361286603e-05,  3.051850580959581e-05,
                         1.2214953130751383e-05, 9.849154594121501e-06,  0.00022693387290928513,
                         8.368508133571595e-05,  0.00022313144290819764, 0.0005361074581742287,
                         6.531889812322333e-05,  0.00026537710800766945, 7.883973739808425e-05,
                         0.00013731264334637672, 0.00012267536658328027, 0.00023475776833947748,
                         2.6765605070977472e-05, 0.0005794286262243986,  7.600678509334102e-05,
                         0.00013074715388938785, 0.00011062491830671206, 0.0016951510915532708,
                         0.00011428036668803543, 0.00016306902398355305, 0.00024765843409113586,
                         0.00021350749011617154, 5.599885116680525e-05,  2.2178761355462484e-05,
                         2.258897256979253e-05,  0.00013219831453170627, 1.0895384548348375e-05,
                         0.00010807781654875726, 5.638175571220927e-05,  3.268223736085929e-05,
                         0.00012442580191418529, 9.121948824031278e-05,  0.00011003281542798504,
                         0.00018195620214100927, 4.714807073469274e-05,  2.637806028360501e-05,
                         3.0695387977175415e-05, 9.560988110024482e-05,  4.6540735638700426e-05,
                         0.00014333055878523737, 8.367578993784264e-05,  0.00010767165804281831,
                         6.30511567578651e-05,   0.00010534076136536896, 0.000996411545202136,
                         4.491173240239732e-05,  0.00016698669060133398, 0.00032435034518130124,
                         0.00015383596473839134, 0.000941336271353066,   0.00011833190364995971,
                         3.026474041689653e-05,  0.0007111453451216221,  9.392905485583469e-05,
                         2.4234706870629452e-05, 0.0001761648163665086,  0.0003148177347611636,
                         5.31572186446283e-05,   4.032947981613688e-05,  7.39449606044218e-05,
                         9.737988148117438e-05,  4.605281355907209e-05,  2.4843131541274488e-05,
                         0.0009361918782815337,  0.0003890021180268377,  0.000440780830103904,
                         0.0005261428304947913,  8.830805745674297e-05,  9.97878159978427e-05,
                         6.748830492142588e-05,  0.00011457232903921977, 0.0002132511872332543,
                         0.00027547081117518246, 4.9255355406785384e-05, 0.00010805369674926624,
                         5.617174974759109e-05,  0.0001160655083367601,  4.3972322600893676e-05,
                         3.9643480704398826e-05, 9.633183071855456e-05,  4.1525374399498105e-05,
                         9.785373549675569e-05,  0.00020469269657041878, 3.3718788472469896e-05,
                         0.00035335251595824957, 3.5235018003731966e-05, 5.451041579362936e-05,
                         0.00010164189734496176, 9.195677557727322e-05,  0.0001487918198108673,
                         0.0002954828378278762,  0.00011601152073126286, 0.00023479285300709307,
                         4.635747245629318e-05,  1.4458188161370344e-05, 0.0002287497918587178,
                         0.00011422082025092095, 0.000328084803186357,   0.0004996033385396004,
                         0.00012422981671988964, 0.0002493273059371859,  3.7932801205897704e-05,
                         2.6020647055702284e-05, 4.428106331033632e-05,  0.00012342135596554726,
                         0.00013997737551108003, 0.00039685695082880557, 0.00046508427476510406,
                         3.474571349215694e-05,  4.907420589006506e-05,  0.0001217502576764673,
                         0.0002065811859210953,  0.0002907615853473544,  0.00013230647891759872,
                         4.922816879115999e-05,  0.0007734995451755822,  0.0002600475854706019,
                         1.549326225358527e-05,  0.0001876997121144086,  0.000160562529345043,
                         0.0002167065249523148,  3.8146528822835535e-05, 0.0001246153115062043,
                         2.576084625616204e-05,  7.830333197489381e-05,  8.199543663067743e-05,
                         9.556880831951275e-05,  0.0003665124240797013,  2.3427672203979455e-05,
                         0.0001707724732114002,  2.785152719297912e-05,  0.00015790382167324424,
                         0.00022873631678521633, 7.454463775502518e-05,  0.0003101128386333585,
                         5.830691588926129e-05,  0.0001382564805680886,  0.00015507036005146801,
                         7.245310553116724e-05,  0.00035894865868613124, 0.0002877171500585973,
                         1.3643423699249979e-05, 0.00011919427197426558, 0.00026662630261853337,
                         0.00021767056023236364, 0.00020001850498374552, 0.00010658909741323441,
                         0.00012833565415348858, 8.352331497007981e-05,  3.502327672322281e-05,
                         0.0001555426133563742,  1.9158043869538233e-05, 7.699148409301415e-05,
                         0.00027627000235952437, 0.0003658691421151161,  0.00017679115990176797,
                         0.0003332932246848941,  6.290655437624082e-05,  4.075413744430989e-05,
                         0.00014151599316392094, 5.3265990572981536e-05, 4.2448362364666536e-05,
                         7.432845450239256e-05,  0.0026788238901644945,  0.00012146203516749665,
                         8.36100589367561e-05,   0.0002926172746811062,  8.537942449038383e-06,
                         0.0003237926575820893,  0.00023086390865501016, 0.0002723048673942685,
                         7.854685827624053e-05,  0.0006101232138462365,  7.202927372418344e-05,
                         0.00018056639237329364, 9.36800497584045e-05,   8.691177208675072e-05,
                         0.00017308242968283594, 0.0003942294861190021,  0.0002973210939671844,
                         0.000497720786370337,   3.370467311469838e-05,  7.379592716461048e-05,
                         0.00011221088061574847, 4.6421977458521724e-05, 0.00019633934425655752,
                         0.00011409746366553009, 8.556375541957095e-05,  8.536854875274003e-05,
                         0.00011360019561834633, 4.896915561403148e-05,  4.9637452320894226e-05,
                         0.00024898123228922486, 0.0002568161580711603,  4.3470350647112355e-05,
                         3.154576916131191e-05,  0.00010538587957853451, 0.00038148550083860755,
                         4.1407114622415975e-05, 0.00013379713345784694, 2.479251634213142e-05,
                         0.00020380363275762647, 8.339488704223186e-05,  1.554504706291482e-05,
                         7.948959682835266e-05,  0.0003578632022254169,  5.84406043344643e-05,
                         0.0006477304850704968,  0.00023423055245075375, 0.0001101445741369389,
                         0.0004998904187232256,  9.705793490866199e-05,  0.00039804118569009006,
                         0.0005787852569483221,  0.000137823517434299,   0.00024979040608741343,
                         0.0004093940369784832,  0.00013470760313794017, 0.0003302344703115523,
                         0.0005012860056012869,  5.124416929902509e-05,  8.312642603414133e-05,
                         0.00011316394375171512, 0.0002579018473625183,  5.554896779358387e-05,
                         3.980553083238192e-05,  4.7948811698006466e-05, 4.34733992733527e-05,
                         9.115188004216179e-05,  5.007939034840092e-05,  0.0001531030284240842,
                         0.0002643332991283387,  0.00022473580611404032, 4.5484601287171245e-05,
                         8.760738273849711e-05,  6.671450682915747e-05,  0.00023637525737285614,
                         6.485534686362371e-05,  0.0007353606051765382,  0.00030135296401567757,
                         8.410352893406525e-05,  6.864702299935743e-05,  0.0001246862520929426,
                         9.301433601649478e-05,  0.00011852023453684524, 0.0008643215987831354,
                         0.00026819543563760817, 0.0001097122294595465,  0.00037075550062581897,
                         6.047682745702332e-06,  0.000816835556179285,   3.9730057324049994e-05,
                         0.0001887220423668623,  1.3069349733996205e-05, 0.00022200641978997737,
                         0.00011929087486350909, 0.00016635714564472437, 9.054622205439955e-05,
                         4.499730857787654e-05,  0.00014222279423847795, 3.194831879227422e-05,
                         0.00018416110833641142, 6.606381066376343e-05,  0.00010106898844242096,
                         0.00015943341713864356, 4.467047256184742e-05,  4.614955832948908e-05,
                         4.0429560613119975e-05, 0.00019261401030234993, 0.0002745094825513661,
                         5.291130946716294e-05,  4.2681218474172056e-05, 0.0001626024895813316,
                         0.00010321613808628172, 9.914283873513341e-05,  0.0003242707171011716,
                         7.267262117238715e-05,  4.666902532335371e-05,  2.586897426226642e-05,
                         6.94201880833134e-05,   0.00011036969954147935, 0.00010233406646875665,
                         0.00013464984658639878, 0.00011247427755733952, 0.0003732256591320038,
                         9.56419826252386e-05,   0.00010474747250555083, 0.00020505771681200713,
                         0.00017702190962154418, 0.00018375304352957755, 0.000493236817419529,
                         0.0019721456337720156,  0.00024604590726085007, 0.0001699005952104926,
                         2.6690371669246815e-05, 0.0003413781523704529,  0.00010569044388830662,
                         7.971587183419615e-05,  7.779335282975808e-05,  2.7490237698657438e-05,
                         1.2699446415354032e-05, 3.322170960018411e-05,  0.00014734065916854888,
                         0.00016361403686460108, 0.00012594947475008667, 0.0001620340335648507,
                         6.389318878063932e-05,  0.0001692840305622667,  0.00048616540152579546,
                         7.672212086617947e-05,  0.0001700459251878783,  4.2961244616890326e-05,
                         2.30081204790622e-05,   0.0004427777021192014,  0.00019670020265039057,
                         0.0001996461651287973,  5.058191527496092e-05,  4.207512756693177e-05,
                         0.0007554035983048379,  9.893852984532714e-05,  0.0001641898270463571,
                         8.483658893965185e-05,  0.0002363732346566394,  6.879333523102105e-05,
                         8.954870281741023e-05,  0.00011170643847435713, 4.143284968449734e-05,
                         0.00013402455078903586, 7.800787716405466e-05,  0.00025429032393731177,
                         3.564758662832901e-05,  0.00013965276593808085, 0.0005741685745306313,
                         0.00040312271448783576, 4.561629248200916e-05,  0.0002261712506879121,
                         0.00013987206330057234, 2.3622622393304482e-05, 6.325405411189422e-05,
                         4.1310795495519415e-05, 0.0006344714784063399,  2.7572566978051327e-05,
                         0.00018857362738344818, 3.251176167395897e-05,  7.197193099273136e-06,
                         8.94636832526885e-05,   0.00021009816555306315, 0.0001795701973605901,
                         0.00018413414363749325, 0.0001088934950530529,  0.0002560255234129727,
                         1.710472315608058e-05,  4.709645872935653e-05,  0.0002661642211023718,
                         8.351048018084839e-05,  1.3744044736085925e-05, 5.684133429895155e-05,
                         0.00022355707187671214, 1.4386364455276635e-05, 0.00012571882689371705,
                         0.00022932775027584285, 0.00034291972406208515, 1.7289010429522023e-05,
                         6.661367660854012e-05,  4.1932580643333495e-05, 2.9703875043196604e-05,
                         4.25475554948207e-05,   2.5972733055823483e-05, 0.0004204420547466725,
                         7.943311356939375e-05,  0.0006115012802183628,  0.0009893352398648858,
                         2.182941352657508e-05,  0.0001564600970596075,  1.2626060197362676e-05,
                         6.425273750210181e-05,  0.0009931838139891624,  2.8903656129841693e-05,
                         9.781194239621982e-05,  0.00021819585526827723, 0.0008186994818970561,
                         0.0014882077230140567,  5.642850010190159e-05,  4.077351695741527e-05,
                         0.00010386096982983872, 5.733469515689649e-05,  2.4418666725978255e-05,
                         0.00024478117120452225, 5.921684351051226e-05,  9.846661851042882e-05,
                         4.59248767583631e-05,   0.00010566046694293618, 0.0005999189452268183,
                         0.00024031462089624256, 4.3183717934880406e-05, 0.0002614698314573616,
                         0.00024008085893001407, 0.000272257108008489,   4.767261634697206e-05,
                         3.8318226870615035e-05, 3.651249426184222e-05,  3.6396566429175436e-05,
                         0.0003767017915379256,  0.0002940237754955888,  0.00020608918566722423,
                         1.8588652892503887e-05, 7.243311119964346e-05,  0.00017128430772572756,
                         0.00021366632427088916, 0.00021192466374486685, 7.802543404977769e-05,
                         6.754251808160916e-05,  0.0002119294076692313,  0.0001755374833010137,
                         0.00018566599464975297, 4.901269858237356e-05,  5.7525598094798625e-05,
                         2.6832372896024026e-05, 0.0005854827468283474,  1.6102287190733477e-05,
                         9.187866817228496e-05,  0.0003842117148451507,  2.1562031179200858e-05,
                         0.00028266580193303525, 9.081076132133603e-05,  0.00012283424439374357,
                         4.448431354830973e-05,  2.3291113393497653e-05, 0.00012684571265708655,
                         3.7754984077764675e-05, 7.304675818886608e-05,  9.871365909930319e-05,
                         1.7755119188223034e-05, 3.8783000491093844e-05, 0.00031661460525356233,
                         3.172268043272197e-05,  0.00010269648919347674, 5.071646592114121e-05,
                         0.0003013824170920998,  0.0002757901675067842,  2.9967819500598125e-05,
                         4.1771152609726414e-05, 8.735013398109004e-05,  0.00026187466573901474,
                         7.975644257385284e-05,  0.00010554178152233362, 0.00012852452346123755,
                         0.00025074442964978516, 0.00033813025220297277, 0.0001696775434538722,
                         8.116821118164808e-05,  0.0006903486791998148,  5.8431411162018776e-05,
                         6.401404971256852e-05,  0.00014291857951320708, 2.457187838444952e-05,
                         9.861128637567163e-05,  0.00022907440143171698, 0.0001714319223538041,
                         0.001001330791041255,   5.752441938966513e-05,  0.0001316885172855109,
                         0.00012066806812072173, 0.00013768638018518686, 1.4833054592600092e-05,
                         8.824075484881178e-05,  6.292380567174405e-05,  0.00035924901021644473,
                         1.4713546079292428e-05, 0.00011620717850746587, 8.128350600600243e-06,
                         0.0001532525202492252,  0.00013708847109228373, 3.7714842619607225e-05,
                         6.226765981409699e-05,  0.0005824422696605325,  5.8743462432175875e-05,
                         0.00018369153258390725, 0.00023733322450425476, 0.00026214515673927963,
                         2.5808478312683292e-05, 6.133261194918305e-05,  9.324245183961466e-05,
                         0.0002187991631217301,  5.13420527568087e-05,   6.979522004257888e-05,
                         0.0003501066821627319,  0.00019652141781989485, 0.0003148118848912418,
                         4.743348472402431e-05,  6.704131374135613e-05,  0.0001984016998903826,
                         0.000361420534318313,   0.00016544517711736262, 0.0001489554124418646,
                         0.0002134793030563742,  7.342769094975665e-05,  0.00018094746337737888,
                         9.042588499141857e-05,  0.0006077881553210318,  1.5261273802025244e-05,
                         4.705890023615211e-05,  1.1131795872643124e-05, 0.0004045733076054603,
                         0.0002412096509942785,  0.0001847599633038044,  1.6710118870832957e-05,
                         9.624404628993943e-05,  0.0001415343431290239,  9.52161499299109e-05,
                         6.590667180716991e-05,  5.480563049786724e-05,  0.0003347562742419541,
                         6.141027552075684e-05,  0.00022233590425457805, 4.56510424555745e-05,
                         2.4743185349507257e-05, 2.9874952815589495e-05, 0.0001293796522077173,
                         8.137422264553607e-05,  0.00023711237008683383, 6.000378562021069e-05,
                         2.8769587515853345e-05, 6.422774458769709e-05,  0.000556967977900058,
                         0.0001158292216132395,  5.968549885437824e-05,  6.455664697568864e-05,
                         0.0002614253025967628,  0.00038092202157713473, 0.00020305128418840468,
                         0.0001118495492846705,  0.0002892329648602754,  0.00014266243670135736,
                         9.929545922204852e-05,  5.0587750592967495e-05, 5.968717596260831e-05,
                         8.171433000825346e-05,  9.32486291276291e-05,   2.3733722628094256e-05,
                         4.346228888607584e-05,  3.100894537055865e-05,  0.00019704722217284143,
                         0.00013503145601134747, 0.00020298027084209025, 0.00031269335886463523,
                         0.000463562726508826,   0.00017056023352779448, 0.0004107982967980206,
                         4.833276034332812e-05,  0.000593491131439805,   0.00011447828728705645,
                         0.00039465585723519325, 7.663064025109634e-05,  0.0002470189065206796,
                         0.00016655016224831343, 8.061908010859042e-05,  6.774016947019845e-05,
                         8.284464274765924e-05,  3.6559184081852436e-05, 6.344381108647212e-05,
                         8.741038618609309e-05,  0.00010901869973167777, 5.2305425924714655e-05,
                         3.142548303003423e-05,  5.760492422268726e-05,  8.809957216726616e-05,
                         8.177026029443368e-05,  7.787033246131614e-05,  0.00013298550038598478,
                         0.0003484021872282028,  0.0002832505269907415,  0.00029241025913506746,
                         4.1880823118845e-05,    0.0001270145148737356,  0.00018355995416641235,
                         3.754967838176526e-05,  0.00014275872672442347, 0.00018503327737562358,
                         6.580992339877412e-05,  0.00015100806194823235, 3.0815499485470355e-05,
                         9.941921598510817e-05,  7.471385470125824e-05,  0.0003130143159069121,
                         1.5985360732884146e-05, 0.00024928271886892617, 2.3041453459882177e-05,
                         5.6458942708559334e-05, 0.00022123480448499322, 1.9504694137140177e-05,
                         0.00018281694792676717, 0.00014726656081620604, 6.331942131510004e-05,
                         5.754851008532569e-05,  0.00017139583360403776, 7.563472900073975e-05,
                         0.0001314137043664232,  0.0001709608332021162,  7.328280480578542e-05,
                         0.00010847122030099854, 9.406732715433463e-05,  0.00017434966866858304,
                         4.175108188064769e-05,  0.00019695211085490882, 0.00039001242839731276,
                         0.0007307336200028658,  0.0001752612297423184,  0.000461520190583542,
                         0.00018481242295820266, 0.0001180059407488443,  0.00015153038839343935,
                         8.863888797350228e-05,  3.446882692514919e-05,  0.0003530723333824426,
                         7.067440310493112e-05,  1.4567174730473198e-05, 0.0004010784032288939,
                         0.0005071971681900322,  0.00013422495976556093, 0.0001515313342679292,
                         3.5206536267651245e-05, 0.0001229139743372798,  0.0003173945879098028,
                         8.819817594485357e-05,  0.00016333894745912403, 1.955797415575944e-05,
                         0.0001779502781573683,  0.0001991090102819726,  9.991923434427008e-05,
                         0.00021133538393769413, 0.0002549369528423995,  0.0016187301371246576,
                         8.159640128724277e-05,  0.00027626907103694975, 0.0002089036861434579,
                         0.0001966548152267933,  0.00032023052335716784, 0.00011158250708831474,
                         0.00011702911433530971, 6.600473716389388e-05,  6.009323624311946e-05,
                         5.9142712416360155e-05, 0.0003184590023010969,  8.773396984906867e-05,
                         0.0001289940846618265,  0.0004592814075294882,  0.0001942189410328865,
                         5.802138548460789e-05,  0.0006358650862239301,  0.00013346552441362292,
                         9.836661047302186e-05,  5.873015834367834e-05,  0.00030327466083690524,
                         0.00025960899074561894, 0.0003380058624316007,  9.427361510461196e-05,
                         0.000111241708509624,   6.699826917611063e-05,  5.6827757362043485e-05,
                         0.0001540594530524686,  0.00028247051523067057, 3.6025201552547514e-05,
                         6.778715032851323e-05,  3.360850678291172e-05,  7.360903691733256e-05,
                         0.00018992225523106754, 0.0002859224332496524,  8.623032044852152e-05,
                         5.857826181454584e-05,  0.00015898412675596774, 0.0002753009321168065,
                         0.00013288549962453544, 1.1076121154474095e-05, 0.00010658660903573036,
                         7.421724149025977e-05,  0.00014001411909703165, 0.00015271638403646648,
                         3.287528670625761e-05,  0.00023944747226778418, 5.7532892242306843e-05,
                         3.248808206990361e-05,  6.83421385474503e-05,   0.00011392328451620415,
                         1.736870672175428e-06,  6.779914110666141e-05,  0.000560420157853514,
                         5.322973447619006e-05,  0.001667055650614202,   0.0008942924323491752,
                         5.371628503780812e-05,  0.00023643707390874624, 1.600371797394473e-05,
                         8.002638060133904e-05,  1.4840617950540036e-05, 0.00010750476940302178,
                         0.0006927055073902011,  6.54217874398455e-05,   0.006563655566424131,
                         5.9148689615540206e-05, 0.00015161083138082176, 7.738993008388206e-05,
                         0.0005962318973615766,  0.00017308614042121917, 2.4781442334642634e-05,
                         8.192023960873485e-05,  0.0002623674226924777,  0.000450808962341398,
                         0.00024356352514587343, 0.0003819866105914116,  5.0307917263126e-05,
                         0.00015943539619911462, 0.0001876091118901968,  0.00012100800086045638,
                         0.00011578846169868484, 0.00010832257976289839, 5.603851968771778e-05,
                         0.000176145214936696,   0.00026101822732016444, 0.00011322483624098822,
                         6.698220386169851e-05,  5.395134212449193e-05,  0.00013197951193433255,
                         2.0708861484308727e-05, 0.0002678456366993487,  0.00010778065916383639,
                         1.8195125448983163e-05, 0.0011729249963536859,  0.0001120247834478505,
                         0.0005041929543949664,  0.0001877871254691854,  2.5545825337758288e-05,
                         7.698513218201697e-05,  2.1451101929415017e-05, 0.0006038902211003006,
                         0.0002188646758440882,  2.7578218578128144e-05, 7.811667455825955e-05,
                         8.234316919697449e-05,  0.0003480084997136146,  1.6694857549737208e-05,
                         0.0002407805441180244,  6.203126395121217e-05,  0.0002482785494066775,
                         0.00010118018690263852, 9.411426435690373e-05,  0.00025925302179530263,
                         3.117865344393067e-05,  0.00046672517783008516, 0.00028169230790808797,
                         0.00015800395340193063, 6.08079535595607e-05,   0.00013230433978606015,
                         0.00011858621292049065, 6.34741663816385e-05,   0.00014929809549357742,
                         3.8202942960197106e-05, 0.00016328346100635827, 0.00011361135693732649,
                         5.6313376262551174e-05, 0.00015490669466089457, 9.031871741171926e-05,
                         0.00010319153807358816, 0.0001836253795772791,  0.00019175108172930777,
                         9.384856093674898e-05,  0.000422862998675555,   5.56564555154182e-05,
                         0.00026271803653799,    0.0006756484508514404,  0.00043203355744481087,
                         4.192906271782704e-05,  0.00010222286800853908, 0.00025022297631949186,
                         4.4629359763348475e-05, 0.00021769650629721582, 0.00024355525965802372,
                         0.00042090905481018126, 0.0003648370911832899,  6.750284956069663e-05,
                         0.0015385578153654933,  1.9033916032640263e-05, 7.088085840223357e-05,
                         9.366991434944794e-05,  0.0007104689721018076,  0.00014447337889578193,
                         0.0004253361839801073,  0.00010108600690728053, 0.0001857211609603837,
                         0.0004860791377723217,  0.0005191339296288788,  3.6350309528643265e-05,
                         5.9568243159446865e-05, 0.00013831509568262845, 0.00022012554109096527,
                         5.850872184964828e-05,  0.0005510536138899624,  1.738112587190699e-05,
                         3.925041892216541e-05,  0.0001994546182686463,  5.908897946937941e-05,
                         2.5982111765188165e-05, 2.0408135242178105e-05, 0.00025064460351131856,
                         0.00012890453217551112, 0.00011474377970444039, 0.00032561749685555696,
                         5.774562669103034e-05,  0.0001901814976008609,  3.2673298846930265e-05,
                         0.00023883109679445624, 0.00011325004743412137, 0.0010166543070226908,
                         4.583854752127081e-05,  4.03417507186532e-05,   0.00013094099995214492,
                         0.0001573410554556176,  5.0500289944466203e-05, 0.0004987875581718981,
                         0.00018958111468236893, 0.0006176265305839479,  3.0403554774238728e-05,
                         0.00011086805898230523, 0.0003789713664446026,  0.00010909775301115587,
                         0.00023126373707782477, 2.814389517880045e-05,  2.1147312509128824e-05,
                         2.1730314983869903e-05, 8.611600787844509e-05,  0.0011870376765727997,
                         0.0002749936538748443,  0.0002268434764118865,  0.00016349217912647873,
                         0.00012882122246082872, 0.0003028274222742766,  0.00028906084480695426,
                         5.121053254697472e-05,  6.636606121901423e-05,  4.160336538916454e-05,
                         0.0001950612640939653,  0.00026682927273213863, 7.950715371407568e-05,
                         0.0007722968002781272,  0.00013036573363933712, 0.00010669050243450329,
                         0.00019278234685771167, 7.887868559919298e-05,  1.3600526472146157e-05,
                         9.787342423805967e-05,  5.072342901257798e-05,  0.00014137632388155907,
                         6.01390129304491e-05,   0.00015204089868348092, 2.648025292728562e-05,
                         0.0003127178642898798,  7.442377682309598e-05,  9.070097439689562e-05,
                         0.00014582012954633683, 0.0001946788834175095,  0.00016682509158272296,
                         0.00022763331071473658, 0.0003950117970816791,  3.8822374335723e-05,
                         3.028517858183477e-05,  0.0001053814630722627,  0.0002189933293266222,
                         0.00020094646606594324, 0.0002898294187616557,  0.0003942370822187513,
                         0.00012069281365256757, 5.037938171881251e-05,  0.00015512137906625867,
                         0.00030174426501616836, 0.0002934384683612734,  0.00010534614557400346,
                         0.0005330632557161152,  5.751374555984512e-05,  7.62169947847724e-05,
                         2.5865641873679124e-05, 0.0005154756945557892,  8.076240919763222e-05,
                         0.0002641331811901182,  8.198144496418536e-05,  0.0001343732001259923,
                         0.00010583559196675196, 8.49635835038498e-05,   6.0017518990207464e-05,
                         2.553476588218473e-05,  2.2274783987086266e-05, 2.0723766283481382e-05,
                         0.00029517849907279015, 0.00010217740782536566, 2.5182245735777542e-05,
                         0.00021477309928741306, 0.0001402955240337178,  0.00042716326424852014,
                         0.00040279910899698734, 0.00022710696794092655, 4.5325097744353116e-05,
                         0.0005019521340727806,  0.00016586350102443248, 0.0001874976442195475,
                         0.00043679631198756397, 0.00010179197124671191, 0.0001324173790635541,
                         0.00011504632129799575, 0.0004495907633099705,  9.47994994930923e-05,
                         3.0189514291123487e-05, 0.0001257051044376567,  7.339916191995144e-05,
                         0.0002870942698791623,  5.1706771046156064e-05, 0.0005628988146781921,
                         5.3249106713337824e-05, 9.663561650086194e-05,  0.00016079522902145982,
                         1.465723107685335e-05,  0.000270481628831476,   0.0012578306486830115,
                         0.00011901423567906022, 0.0001703163143247366,  3.81470745196566e-05,
                         0.0001305650221183896,  0.00011320361954858527, 8.740838529774919e-05,
                         0.00014020749949850142, 0.0003367444151081145,  6.70221634209156e-05,
                         0.00021739743533544242, 0.00017294721328653395, 0.00011403204553062096,
                         0.0005225276108831167,  0.00017464537813793868, 9.44212733884342e-05,
                         0.0001259970013052225,  0.00010798843868542463, 6.72188398311846e-05,
                         0.00011598839046200737, 9.82230922090821e-05,   6.98615112924017e-05,
                         0.0002689244574867189,  6.469176878454164e-05,  6.29435817245394e-05,
                         6.815774395363405e-05,  8.780101052252576e-05,  4.198155511403456e-05,
                         0.00023962876002769917, 0.00025822868337854743, 0.0002444647834636271,
                         6.69909204589203e-05,   0.0001929222489707172,  0.00013524484529625624,
                         3.281656972831115e-05,  1.1529854418768082e-05, 0.00022075076412875205,
                         4.851834091823548e-05,  0.00021743593970313668, 5.5809799960115924e-05,
                         0.0003130695258732885,  0.00024990280508063734, 0.00022590196749661118,
                         1.944039286172483e-05,  0.0020953090861439705,  2.620591658342164e-05,
                         2.2279000404523686e-05, 0.00027894575032405555, 0.0005742271896451712,
                         0.00016120410873554647, 0.0002222984767286107,  0.002026567468419671,
                         0.00023443569079972804, 9.539967140881345e-05,  5.260030957288109e-05,
                         3.546233710949309e-05,  0.0001007809623843059,  0.0001229543995577842,
                         0.0001466602843720466,  5.485930523718707e-05,  0.0002878574014175683,
                         0.00017937870870810002, 7.965291297296062e-05,  3.465861300355755e-05,
                         5.793697710032575e-05,  0.00010917554754996672, 5.2279818191891536e-05,
                         0.00014096003724262118, 8.445206913165748e-05,  8.207320206565782e-05,
                         6.594083970412612e-05,  0.0001595662033651024,  0.00016800672165118158,
                         0.0002971050562337041,  7.061585347400978e-05,  0.00010053358710138127,
                         8.708950190339237e-05,  4.871797136729583e-05,  0.00024415706866420805,
                         3.83177466574125e-05,   4.1221243009204045e-05, 0.00030280870851129293,
                         3.352161729708314e-05,  3.8897607737453654e-05, 0.00010806358477566391,
                         1.0182648111367598e-05, 4.838028326048516e-05,  0.0003746096044778824,
                         0.0008229425875470042,  0.00031518377363681793, 0.00010633491183398291,
                         0.00011355512833688408, 0.0006519070011563599,  4.334200275479816e-05,
                         0.00018995098071172833, 4.5399876398732886e-05, 0.00019297102699056268,
                         6.42717641312629e-05,   0.00015014884411357343, 0.0004398639721330255,
                         0.0007624307181686163,  3.255423507653177e-05,  0.00018737379286903888,
                         6.833288352936506e-05,  0.00023505353601649404, 5.7312368880957365e-05,
                         9.659995703259483e-05,  0.00016973046876955777, 1.9328936105011962e-05,
                         7.075566827552393e-05,  8.012374746613204e-05,  2.7962309104623273e-05,
                         0.0007479471969418228,  4.331678064772859e-05,  0.0001855101145338267,
                         0.00016709281771909446, 9.51257607084699e-05,   9.411246537638362e-06,
                         0.00028262013802304864, 8.375455217901617e-05,  4.239580812281929e-05,
                         0.00029889988945797086, 0.0005978887202218175,  0.00011231296957703307,
                         3.642380761448294e-05,  8.459821401629597e-05,  9.48033484746702e-05,
                         0.00033549999352544546, 8.600504224887118e-05,  5.28358286828734e-05,
                         0.00010136597120435908, 0.00024328893050551414, 0.00066184694878757,
                         2.0583604054991156e-05, 0.0010943679371848702,  7.274493691511452e-05,
                         0.00043540948536247015, 0.00013916775060351938, 0.00030395120847970247,
                         8.074015204329044e-05,  1.8845312297344208e-05, 0.0001856759045040235,
                         0.003387338249012828,   0.00010153625044040382, 0.00011452064791228622,
                         0.0002690725086722523,  5.221745595918037e-05,  0.00016835409041959792,
                         0.00023031304590404034, 0.0001443236687919125,  8.03795555839315e-05,
                         3.166174792568199e-05,  0.00042831277824006975, 3.7171364965615794e-05,
                         0.0002021757245529443,  5.348627382772975e-05,  0.00019132994930259883,
                         0.0009247311390936375,  0.00012158670142525807, 8.149514178512618e-05,
                         0.0002252970152767375,  0.00035257998388260603, 1.5480074580409564e-05,
                         2.5330846256110817e-05, 0.00012030740617774427, 9.687367855804041e-05,
                         0.00016405739006586373, 2.6290199457434937e-05, 5.503340798895806e-05,
                         0.0003571743727661669,  0.00011579414422158152, 0.00039547853521071374,
                         0.00020603070151992142, 5.168988718651235e-05,  0.00021551814279519022,
                         0.00044416761375032365, 9.01776147657074e-05,   0.0002051797928288579,
                         0.0004452984139788896,  7.136718340916559e-05,  0.00010171791655011475,
                         0.0001504644169472158,  0.00031668375595472753, 0.00021764080156572163,
                         0.00010771895904326811, 0.00010342735913582146, 5.239354140940122e-05,
                         0.00041076928027905524, 0.0002755543973762542,  0.00011956593516515568,
                         0.00017975876107811928, 4.5128461351851e-05,    0.0001570666063344106,
                         0.0008392389281652868,  0.0002074430522043258,  0.00045189581578597426,
                         0.0008463601698167622,  5.647787111229263e-05,  5.0126433052355424e-05,
                         0.0003391325008124113,  0.0001022394499159418,  0.00015762739349156618,
                         4.52944659627974e-05,   3.583627039915882e-05,  0.0001766391796991229,
                         0.0001327913487330079,  8.551901555620134e-05,  0.00041534178308211267,
                         9.82960409601219e-05,   3.141382694593631e-05,  3.200757419108413e-05,
                         5.2431503718253225e-05, 0.0004186994628980756,  0.00011011060996679589,
                         3.115771323791705e-05,  3.546677908161655e-05,  0.00030780950328335166,
                         0.0007610684260725975,  7.926784019218758e-05,  9.776237129699439e-05,
                         0.00011304685904178768, 0.0001669290941208601,  3.490161907393485e-05,
                         0.00031425736960954964, 7.715583342360333e-05,  0.00025850124075077474,
                         0.0011157040717080235,  0.0011863323161378503,  0.00011534777149790898,
                         0.00032061769161373377, 7.336024282267317e-05,  6.655423931078985e-05,
                         4.753045504912734e-05,  4.855745646636933e-05,  3.4162596421083435e-05,
                         0.00011595559772104025, 0.00014171656221151352, 0.00012817430251743644,
                         0.00018342112889513373, 5.489306204253808e-05,  0.00036049599293619394,
                         0.0001666671159910038,  2.465965189912822e-05,  0.0004019608604721725,
                         0.00029005686519667506, 9.153513383353129e-05,  0.001249448163434863,
                         9.83861245913431e-05,   0.0007527288398705423,  5.6262837460963055e-05,
                         6.281540117925033e-05,  0.00030088875791989267, 0.00012544759374577552,
                         6.933573604328558e-05,  0.00017370385467074811, 4.3880558223463595e-05,
                         0.00022780813742429018, 0.0007960208458825946,  0.0005966481403447688,
                         8.089619223028421e-05,  7.415672007482499e-05,  0.00031270578620024025,
                         0.0003351741179358214,  0.00013610887981485575, 0.00018355995416641235,
                         0.00025389788788743317, 0.0002383477403782308,  9.731266618473455e-05,
                         4.9837599362945184e-05, 0.00018578802701085806, 8.34772436064668e-05,
                         7.920042844489217e-05,  0.00012558764137793332, 0.0004144998674746603,
                         0.00014982999709900469, 0.0005860629607923329,  0.0003788152534980327,
                         0.0001254167582374066,  4.8892165068537e-05,    2.7061914806836285e-05,
                         5.887058068765327e-05,  9.701147791929543e-05,  0.00020205683540552855,
                         8.298677857965231e-05,  1.1118283509858884e-05, 5.884509300813079e-05,
                         5.0137503421865404e-05, 3.52738170477096e-05,   8.783153316471726e-05,
                         0.00017806341929826885, 0.0002109997149091214,  0.0007436333107762039,
                         2.172115728171775e-06,  0.00034580935607664287, 6.69257206027396e-05,
                         8.587616321165115e-05,  0.00014772506256122142, 4.576911305775866e-05,
                         0.00017578293045517057, 0.00010868853132706136, 0.0003657280467450619,
                         0.0001170411633211188,  0.0001119595835916698,  0.00014924733841326088,
                         5.1373368478380144e-05, 0.00015764814452268183, 8.55784528539516e-05,
                         0.00015512715617660433, 3.9504593587480485e-05, 0.001443900866433978,
                         8.6600222857669e-05,    0.00018278101924806833, 0.00010714863310568035,
                         0.00012348197924438864, 0.00016145009431056678, 3.1058221793500707e-05,
                         9.48896849877201e-05,   6.878185376990587e-05,  0.0007470569689758122,
                         0.0002423497207928449,  0.0001342240284429863,  5.7269939134130254e-05,
                         9.668009442975745e-05,  5.097540270071477e-05,  0.00022393227845896035,
                         3.389467383385636e-05,  0.00035787929664365947, 9.88338069873862e-05,
                         4.094785617780872e-05,  7.781918247928843e-05,  2.6073455956066027e-05,
                         6.342766573652625e-05,  0.0002328755654161796,  0.0001011304630083032,
                         0.00012263200187589973, 0.00022151414304971695, 6.992877752054483e-05,
                         7.071330765029415e-05,  0.00026710404199548066, 0.0002477812231518328,
                         2.370940455875825e-05,  6.219813803909346e-05,  0.0001372320402879268,
                         0.00011303171049803495, 2.9244652978377417e-05, 0.000148390608956106,
                         8.378286293009296e-05,  2.5297278625657782e-05, 0.00011201378219993785,
                         0.0006975782453082502,  0.00015541609900537878, 6.99033698765561e-05,
                         0.0001881196949398145,  3.3377731597283855e-05, 8.022390102269128e-05,
                         6.507519719889387e-05,  4.6484285121550784e-05, 0.0002478084643371403,
                         0.0001683757727732882,  0.00019523443188518286, 3.9882743294583634e-05,
                         9.674410102888942e-05,  0.00031605930416844785, 0.0006029257783666253,
                         5.938180402154103e-05,  0.001200594357214868,   8.914013596950099e-05,
                         0.00029375270241871476, 0.00024913394008763134, 5.951098137302324e-05,
                         0.00014666692004539073, 9.677754860604182e-05,  0.00011721369082806632,
                         5.478399179992266e-05,  0.00022526468092110008, 5.0519025535322726e-05,
                         0.0001007522369036451,  7.89188652561279e-06,   8.132841321639717e-05,
                         0.0012426698813214898,  0.0001065628239302896,  0.0001029528139042668,
                         0.0018664664821699262,  0.00017403002129867673, 7.335619011428207e-05,
                         0.00012191146379336715, 0.0008595330291427672,  0.0001457515318179503,
                         5.3788407967658713e-05, 7.070976425893605e-05,  0.0001173050986835733,
                         6.225759716471657e-05,  9.490674710832536e-05,  1.539651202620007e-05,
                         6.943058542674407e-05,  5.05631287524011e-05,   0.00029556878143921494,
                         8.519653783878312e-05,  0.00016102325753308833, 4.030469790450297e-05,
                         0.0004238305555190891,  2.0824130842811428e-05, 0.0001526202104287222,
                         4.411105328472331e-05,  0.0003188262344338,     0.00020536956435535103,
                         0.00022738176630809903, 0.00013593128824140877, 8.652274846099317e-05,
                         5.789876013295725e-06,  0.00019805617921520025, 0.00021265310351736844,
                         3.4985216188943014e-05, 1.390584384353133e-05,  0.00024159502936527133,
                         0.0001917551999213174,  5.8851066569332033e-05, 0.000112533823994454,
                         6.783021672163159e-05,  0.00014599127462133765, 0.0002998743439093232,
                         0.0004520503571256995,  0.00016479502664878964, 0.00022511702263727784,
                         0.0002967696054838598,  3.655447653727606e-05,  0.00010163943079533055,
                         0.00010934998135780916, 0.00011493017518660054, 0.0007057572947815061,
                         1.5091829482116736e-05, 8.946198067860678e-05,  2.6296716896467842e-05,
                         0.000511096091940999,   0.000286930677248165,   0.00017973048670683056,
                         0.0002203059702878818,  1.9992394300061278e-05, 0.00038396683521568775,
                         0.0003305104037281126,  4.873774014413357e-05,  0.0001254052622243762,
                         6.166462844703346e-05,  1.0161834325117525e-05, 0.00013006586232222617,
                         0.00036282636574469507, 0.0003503442567307502,  3.526204454828985e-05,
                         6.548871169798076e-05,  9.706167475087568e-05,  0.0005001531098969281,
                         0.00040321823325939476, 0.00010609631863189861, 0.00015656765026506037,
                         0.00019050179980695248, 8.593965503678191e-06,  2.0882727767457254e-05,
                         0.0001779775629984215,  1.653343679208774e-05,  0.0006659602513536811,
                         0.00015984928177203983, 0.0001840889162849635,  5.194965342525393e-05,
                         7.16019349056296e-05,   0.00011349840497132391, 3.539612589520402e-05,
                         0.00010770220978884026, 6.542803021147847e-05,  5.146023249835707e-05,
                         4.3073876440757886e-05, 1.8633851141203195e-05, 5.538802724913694e-05,
                         2.2095253370935097e-05, 0.00010510058928048238, 0.0002861094253603369,
                         0.0003018790448550135,  0.0013142809038981795,  0.00038179231341928244,
                         4.435840673977509e-05,  0.0002317979815416038,  4.42887467215769e-05,
                         3.4626795240910724e-05, 0.0003396047104615718,  0.00019934670126531273,
                         7.394239219138399e-05,  3.5306296922499314e-05, 1.2526178579719272e-05,
                         0.00021219889458734542, 6.585496885236353e-05,  8.820524089969695e-05,
                         0.0002770806895568967,  0.0005389236030168831,  0.0003717619110830128,
                         7.635445945197716e-05,  0.0002093808288918808,  0.0002916887169703841,
                         0.002024628920480609,   0.0006422822480089962,  0.00013291927461978048,
                         0.00011681151227094233, 0.00013100887008477002, 0.00033737378544174135,
                         9.079720621230081e-05,  0.00022705981973558664, 6.567465607076883e-05,
                         0.000155014538904652,   7.52130727050826e-05,   0.00010956641199300066,
                         0.00020827804110012949, 8.762672950979322e-05,  0.00016137165948748589,
                         2.0437857529032044e-05, 7.764549081912264e-06,  0.00013846237561665475,
                         0.00018613727297633886, 7.998856744961813e-05,  7.000841287663206e-05,
                         0.0009296505013480783,  3.974383434979245e-05,  0.0001542074460303411,
                         7.215653022285551e-05,  0.00040249270386993885, 0.0001809616369428113,
                         0.00016090500866994262, 0.00023366112145595253, 0.0009602919453755021,
                         3.207659028703347e-05,  0.0002654625859577209,  0.00010897291213041171,
                         0.0007254538359120488,  0.0005366267287172377,  3.5645225580083206e-05,
                         0.00014000704686623067, 0.0001466960384277627,  0.00010794869012897834,
                         0.00022617717331741005, 0.00014725896471645683, 4.61384879599791e-05,
                         7.876195013523102e-05,  0.00040669302688911557, 0.00027190291439183056,
                         6.215022585820407e-05,  0.00017845343973021954, 0.00018963882757816464,
                         7.631951302755624e-05,  0.0005389635916799307,  6.99303054716438e-05,
                         7.925649697426707e-05,  0.0001572010514792055,  1.4848801583866589e-05,
                         0.00023188779596239328, 0.0005614859983325005,  0.0002686905500013381,
                         0.00036310500581748784, 0.00026152326609008014, 0.00011899664968950674,
                         9.652831067796797e-05,  1.9365599655429833e-05, 6.285482959356159e-05,
                         9.352013148600236e-05,  6.670502625638619e-05,  0.00016375798441004008,
                         0.0007133501349017024,  1.45094127219636e-05,   6.878704152768478e-05,
                         5.125680036144331e-05,  5.417236752691679e-05,  0.00020560155098792166,
                         0.0001491220318712294,  0.002580054337158799,   0.0001755440898705274,
                         0.00016888449317775667, 8.672047988511622e-05,  3.220488360966556e-05,
                         2.9485157938324846e-05, 6.305686110863462e-05,  0.00018323918629903346,
                         0.00031222577672451735, 8.787254773778841e-05,  4.856608939007856e-05,
                         1.648118814046029e-05,  3.137390376650728e-05,  0.0003467327624093741,
                         0.0002522125723771751,  0.00017260770255234092, 0.00025215002824552357,
                         0.0001795094140106812,  0.0002161081793019548,  0.0004386194341350347,
                         0.00016049438272602856, 1.75921250047395e-05,   0.0005004761042073369,
                         2.4853641662048176e-05, 8.34109159768559e-05,   0.00011110864579677582,
                         1.619803333596792e-05,  0.00012654412421397865, 0.00018790380272548646,
                         0.00010401604231446981, 0.0003589139087125659,  0.000296239071758464,
                         1.4280380128184333e-05, 1.7129952539107762e-05, 0.00024431466590613127,
                         0.0008183593163266778,  0.0001942224334925413,  4.979229925083928e-05,
                         0.0008570400532335043,  5.7434932386968285e-05, 6.204270903253928e-05,
                         0.0014567130710929632,  0.00017721227777656168, 3.6529127100948244e-05,
                         0.0001346016360912472,  1.4453812582360115e-05, 3.4815104299923405e-05,
                         0.00020328423124738038, 0.00011177751730429009, 0.0004488615377340466,
                         0.0004832067643292248,  0.00021482040756382048, 4.441614873940125e-05,
                         4.943321255268529e-05,  3.8064656109781936e-05, 4.800186070497148e-05,
                         0.00010680038394639269, 0.00012951878306921571, 4.3069911043858156e-05,
                         0.00031173895695246756, 0.0002443463890813291,  0.0002732929424382746,
                         5.934667933615856e-05,  5.689429599442519e-05,  0.00015096919378265738,
                         9.454985411139205e-05,  7.232297502923757e-05,  4.5401138777378947e-05,
                         0.00011334216105751693, 0.0002234756830148399,  0.00012407571193762124,
                         0.00019783523748628795, 0.00016779816360212862, 2.3357499230769463e-05,
                         0.00014665677736047655, 7.525109685957432e-05,  3.1107698305277154e-05,
                         8.68195274961181e-06,   0.0002506250166334212,  4.591708420775831e-05,
                         2.116773248417303e-05,  0.00028263762942515314, 7.929966523079202e-05,
                         0.00012713894830085337, 0.00010671644122339785, 0.0003713752084877342,
                         8.789740240899846e-05,  0.00014130177441984415, 0.00016272398352157325,
                         0.00017263577319681644, 0.00013126105477567762, 6.474775000242516e-05,
                         0.00010177530930377543, 7.961739902384579e-05,  0.00040769355837255716,
                         3.309534804429859e-05,  0.00019556157349143177, 8.192040695575997e-05,
                         0.00025798112619668245, 4.1806502849794924e-05, 1.914551467052661e-05,
                         0.00021714827744290233, 0.00030662628705613315, 0.00020313418644946069,
                         9.297899669036269e-05,  0.0001630806946195662,  0.00025681211263872683,
                         9.906798368319869e-05,  0.00024672309518791735, 0.00012811781198251992,
                         6.511212995974347e-05,  2.552536898292601e-05,  0.00010091262811329216,
                         8.578137931181118e-05,  5.104048614157364e-05,  3.559820834198035e-05,
                         8.378162601729855e-05,  8.284021896542981e-05,  6.722614489262924e-05,
                         0.0005280174664221704,  0.00030304293613880873, 9.3683491286356e-05,
                         0.0005426531424745917,  0.00011382309457985684, 0.000266827002633363,
                         1.891865576908458e-05,  5.8862937294179574e-05, 3.257917342125438e-05,
                         0.00037256223731674254, 2.346056317037437e-05,  4.094451651326381e-05,
                         0.0004148189618717879,  3.428912896197289e-05,  9.92833956843242e-05,
                         8.456909563392401e-05,  9.134590072790161e-05,  6.778683018637821e-05,
                         0.00022055064619053155, 0.00015729453298263252, 0.0005975840031169355,
                         7.697878027101979e-05,  0.000907041656319052,   3.243789615225978e-05,
                         7.550000009359792e-05,  0.00022050412371754646, 6.105037027737126e-05,
                         1.3583430700236931e-05, 0.0008851505117490888,  8.044773858273402e-05,
                         0.00011394882312742993, 0.00013865345681551844, 0.0001714011887088418,
                         0.0006040146690793335,  6.915166159160435e-05,  4.0842223825166e-05,
                         2.8383776225382462e-05, 7.959379581734538e-05,  0.0027251180727034807,
                         0.00027672050055116415, 0.00048598810099065304, 0.00020785495871677995,
                         0.00020070603932254016, 0.00010946526890620589, 3.58922261511907e-05,
                         5.73703546251636e-05,   0.00014523754362016916, 3.21788975270465e-05,
                         0.0005839983932673931,  2.9804523364873603e-05, 1.7685897546471097e-05,
                         0.00011028878361685202, 0.00016171069000847638, 8.749700646148995e-05,
                         3.1605304684489965e-05, 0.00015810868353582919, 0.00015857025573495775,
                         8.126145985443145e-05,  0.00011749174154829234, 8.500833064317703e-05,
                         0.00017947805463336408, 0.0004518088826444,     0.0007581960526295006,
                         0.00017219966684933752, 0.00023333706485573202, 3.711136378115043e-05,
                         8.21461871964857e-05,   0.0001912565203383565,  0.0007549673318862915,
                         0.00015461033035535365, 8.815302135189995e-05,  7.292244845302776e-05,
                         9.248039714293554e-05,  0.0005636928253807127,  0.0002520016860216856,
                         5.14469611516688e-05,   4.49273684353102e-05,   1.9814879124169238e-05,
                         0.00013408550876192749, 0.0003374618827365339,  0.00011448876466602087,
                         0.000400249264203012,   3.812052818830125e-05,  8.871880709193647e-05,
                         7.066948455758393e-05,  7.372559048235416e-05,  0.0003234843898098916,
                         2.3568301912746392e-05, 5.173135650693439e-05,  0.00016273820074275136,
                         0.0001307592901866883,  6.310679600574076e-05,  0.0002508949546609074,
                         0.0003941012837458402,  0.00010645743896020576, 1.0138031029782724e-05,
                         6.294598279055208e-05,  0.00011795294267358258, 6.295099592534825e-05,
                         0.00031730925547890365, 8.011919999262318e-05,  0.00027191132539883256,
                         0.00019491622515488416, 8.068376337178051e-05,  0.0004875222803093493,
                         6.149822002043948e-05,  1.6111680452013388e-05, 5.451250035548583e-05,
                         9.224134555552155e-05,  0.0002759138587862253,  0.0011436856584623456,
                         0.0003715237835422158,  6.820827547926456e-05,  0.00015626636741217226,
                         0.00022985308896750212, 8.087713649729267e-05,  0.00019910938863176852,
                         3.560452023521066e-05,  3.561199628165923e-05,  1.6900692571653053e-05,
                         0.00011247910151723772, 0.0005171690718270838,  0.00012046784831909463,
                         0.0021652246359735727,  6.57609780319035e-05,   5.378628338803537e-05,
                         4.457223622011952e-05,  0.00013813276018481702, 0.0004536374472081661,
                         0.0006831474020145833,  6.961570034036413e-05,  0.00027770467568188906,
                         0.0001354862906737253,  0.0001846293016569689,  0.0014323815703392029,
                         2.5536714019835927e-05, 0.0001763847831171006,  0.00011420960072427988,
                         0.00019372315728105605, 7.107359124347568e-05,  0.00010830811515916139,
                         0.0005708023090846837,  0.00014047982404008508, 6.132901762612164e-05,
                         5.496089579537511e-05,  7.450566772604361e-05,  0.0006528660887852311,
                         0.000141966636874713,   0.00013275127275846899, 3.6137858842266724e-05,
                         3.940727037843317e-05,  5.6007105740718544e-05, 4.290688957553357e-05,
                         0.0004726476327050477,  0.00015504016482736915, 4.31606822530739e-05,
                         6.505717465188354e-05,  8.354426245205104e-05,  0.00014087282761465758,
                         3.259328150306828e-05,  4.02731129724998e-05,   9.14744014153257e-05,
                         7.321480370592326e-05,  0.00017757478053681552, 0.0002386211126577109,
                         5.5630483984714374e-05, 3.189903145539574e-05,  8.332926518050954e-05,
                         3.1805848266230896e-05, 0.0002680887409951538,  0.000859190127812326,
                         1.9725453967112117e-05, 0.00012167845125077292, 0.0003293605986982584,
                         0.00022708559117745608, 4.974315743311308e-05,  6.778669921914116e-05,
                         0.00013864960055798292, 0.001818489981815219,   0.00023444941325578839,
                         0.0007745219045318663,  0.00015685553080402315, 0.0001534217008156702,
                         8.753719157539308e-05,  0.0002971464127767831,  3.750554969883524e-05,
                         0.00044311952660791576, 6.00651947024744e-05,   0.00010469429253134876,
                         4.715978502645157e-05,  0.0001882756914710626,  0.0004634473007172346,
                         0.0001177070225821808,  0.00017566022870596498, 6.969189416849986e-05,
                         0.0001258089323528111,  0.00020857786876149476, 9.649834828451276e-05,
                         9.99626427073963e-05,   0.00011814100435003638, 0.00010456646123202518,
                         0.0001486725959694013,  0.0001567298750160262,  0.00021710482542403042,
                         0.00025602238019928336, 0.00016287219477817416, 3.130180266452953e-05,
                         0.00012484852049965411, 2.1929956346866675e-05, 7.807717338437214e-05,
                         0.00011480168177513406, 4.6206456318031996e-05, 0.00013804090849589556,
                         0.00017258884327020496, 9.777937520993873e-06,  0.00011275754513917491,
                         0.00010412829578854144, 9.24041451071389e-05,   8.568240446038544e-05,
                         0.00010173078044317663, 0.00012287122081033885, 0.0008721948834136128,
                         0.00017198392015416175, 1.0655748155841138e-05, 0.0002715447044465691,
                         0.00010078716150019318, 0.0003095149586442858,  0.00023983536812011153,
                         0.0006240447983145714,  0.00010359887528466061, 0.0001507584092905745,
                         2.25153416977264e-05,   0.0002391022862866521,  0.00015262127271853387,
                         0.00020462111569941044, 0.000303775945212692,   3.797442695940845e-05,
                         9.435816173208877e-05,  0.00014111606287769973, 5.7227869547205046e-05,
                         0.00018338441441301256, 2.2615928173763677e-05, 0.0003399057313799858,
                         0.00016479394980706275, 6.69870223646285e-06,   1.6201323887798935e-05,
                         1.732562122924719e-05,  6.887855124659836e-05,  0.0003810444613918662,
                         0.00015707658894825727, 2.0191739167785272e-05, 9.636077447794378e-05,
                         3.733536141226068e-05,  8.805861580185592e-05,  0.0008451701141893864,
                         7.037336035864428e-05,  0.00017186484183184803, 0.0004750758525915444,
                         6.766297883586958e-05,  4.607855589711107e-05,  5.3805568313691765e-05,
                         0.0003330018662381917,  7.441065099555999e-05,  0.00012253827298991382,
                         0.000189158643479459,   7.810740498825908e-05,  0.0003538752207532525,
                         0.00012083608453394845, 0.00010578982619335875, 0.00042909427429549396,
                         2.3496419089497067e-05, 9.715197666082531e-05,  9.565895015839487e-05,
                         1.842359415604733e-05,  3.1468023280467605e-06, 0.0001672137004788965,
                         7.829997048247606e-05,  4.0365375753026456e-05, 0.0005208234651945531,
                         4.263966184225865e-05,  0.00012004603195236996, 0.00016757208504714072,
                         6.257576023926958e-05,  0.00015629669360350817, 1.1765605449909344e-05,
                         0.00020790222333744168, 8.011335012270138e-05,  6.434474926209077e-05,
                         5.511508061317727e-05,  0.0001893245062092319,  9.208411211147904e-05,
                         0.0001183035274152644,  0.00023258253349922597, 0.00023523627896793187,
                         0.0001583625125931576,  0.0011594477109611034,  0.0005543644656427205,
                         4.530824662651867e-05,  2.2108995835878886e-05, 6.45155887468718e-05,
                         0.0002529510820750147,  0.0002674117567948997,  0.0004115962947253138,
                         0.0005227476940490305,  4.389194509712979e-05,  0.00033069553319364786,
                         8.557298133382574e-05,  0.00021469414059538394, 2.8905225917696953e-05,
                         0.00011700700997607782, 0.0002653541450854391,  2.5384180844412185e-05,
                         0.00013091732398606837, 0.00011190919030923396, 0.00023723395133856684,
                         0.00012411332863848656, 0.0004487838305067271,  0.00021011210628785193,
                         8.157297997968271e-05,  0.00020721454347949475, 0.00021250915597192943,
                         0.000170599072589539,   0.0003904977347701788,  2.956251410068944e-05,
                         7.117778295651078e-05,  0.0002154790417989716,  0.00023345164663624018,
                         9.270598093280569e-05,  3.682024180307053e-05,  0.00022465491201728582,
                         0.0006867875345051289,  9.146916272584349e-05,  3.791365452343598e-05,
                         0.0002585322654340416,  0.0010018537286669016,  3.196356919943355e-05,
                         5.401672751759179e-05,  6.451430090237409e-05,  0.0004158598021604121,
                         0.0001955075713340193,  0.00045943824807181954, 0.00034998281626030803,
                         8.453732152702287e-05,  6.105118518462405e-05,  0.0005131803918629885,
                         0.00023909026640467346, 3.1663348636357114e-05, 6.223818490980193e-05,
                         0.0007955190376378596,  0.00016598936053924263, 7.893131987657398e-05,
                         0.00014038792869541794, 0.00019546823750715703, 9.882834274321795e-05,
                         0.00023125301231630147, 0.00012625352246686816, 3.289645610493608e-05,
                         0.00014301158080343157, 0.0002886747824959457,  0.00016856509319040924,
                         5.778159902547486e-05,  4.9716334615368396e-05, 0.00027664846857078373,
                         7.64194774092175e-05,   0.00015060360601637512, 7.578729127999395e-05,
                         9.444716124562547e-05,  2.5735516828717664e-05, 0.00045708008110523224,
                         0.0017155167879536748,  7.711545185884461e-05,  0.00015297364734578878,
                         7.170125900302082e-05,  3.523836494423449e-05,  0.0007761928136460483,
                         0.00032718805596232414, 0.00011268959497101605, 0.00015173507563304156,
                         6.517348083434626e-05,  0.00010386364738224074, 0.00010324814502382651,
                         7.940951763885096e-05,  8.542572322767228e-05,  0.0013744563329964876,
                         4.340773375588469e-05,  6.208269769558683e-05,  3.696944986586459e-05,
                         0.0006501062889583409,  2.21402442548424e-05,   0.000535377417691052,
                         4.999595694243908e-05,  9.448760829400271e-05,  0.00023205758770927787,
                         0.0006253788014873862,  8.152281225193292e-05,  0.001064714277163148,
                         0.0007408222882077098,  2.9641618311870843e-05, 0.0002793521271087229,
                         0.00014513333735521883, 3.403015580261126e-05,  0.00010125007975148037,
                         4.874311343883164e-05,  0.0005183239700272679,  2.0430870790733024e-05,
                         5.819364378112368e-05,  0.000971936562564224,   9.600559860700741e-05,
                         0.000547130941413343,   4.936887853546068e-05,  8.02669528638944e-05,
                         2.144271638826467e-05,  0.0009273247560486197,  0.00011734807776520029,
                         9.052648965734988e-05,  0.00010283114534104243, 0.0002117391995852813,
                         0.0011877184733748436,  0.00016533814778085798, 9.069668158190325e-05,
                         0.00018422870198264718, 2.8384332836139947e-05, 6.419883720809594e-05,
                         0.0003902697644662112,  2.702973688428756e-05,  6.985508662182838e-05,
                         0.00010500686767045408, 0.0001241058052983135,  0.00018457385885994881,
                         2.5939983970602043e-05, 5.131028592586517e-05,  0.00010872465645661578,
                         0.00039042899152264,    5.201864041737281e-05,  3.658907735371031e-05,
                         8.073661592788994e-05,  0.0003920376766473055,  0.0002344016102142632,
                         7.102565723471344e-05,  0.00011663157056318596, 9.962161129806191e-05,
                         0.0002120860735885799,  0.00018475731485523283, 0.00013814697740599513,
                         0.00012014436651952565, 2.4988374207168818e-05, 0.00013576676428783685,
                         2.792162740661297e-05,  0.0001177755111712031,  0.00017231694073416293,
                         3.796698365476914e-05,  8.888357115210965e-05,  0.002048990223556757,
                         0.0003157192550133914,  0.00021480202849488705, 0.00012067725037923083,
                         5.203158798394725e-05,  0.0003259903460275382,  1.0053311598312575e-05,
                         0.0004936119657941163,  6.011393270455301e-05,  0.0002703823847696185,
                         0.0002069915208267048,  3.4904118365375325e-05, 3.61473867087625e-05,
                         0.00015040067955851555, 0.0016496801981702447,  0.00016780129226390272,
                         0.00019799750589299947, 0.00015311814786400646, 9.922919707605615e-05,
                         6.360418046824634e-05,  6.310847675194964e-05,  0.00022588795400224626,
                         0.00026990132755599916, 0.00013193348422646523, 0.0002327274123672396,
                         0.00015308659931179136, 0.0005961228744126856,  0.00043604985694400966,
                         0.00020812793809454888, 0.00010417632438475266, 3.891613960149698e-05,
                         0.0002598004648461938,  2.3972599592525512e-05, 2.264237446070183e-05,
                         0.00030004922882653773, 7.1786816988606e-05,    0.00016233934729825705,
                         0.00013753704843111336, 0.0008343008812516928,  1.2106011126888916e-05,
                         0.0001539957447675988,  0.0007942282827571034,  3.19921164191328e-05,
                         0.0004218457324896008,  8.712637645658106e-05,  0.00027872849022969604,
                         0.0020617321133613586,  6.535210559377447e-05,  0.00040442694444209337,
                         7.81565104261972e-05,   0.00010088039562106133, 0.00033931448706425726,
                         0.00017155162640847266, 0.0005348005215637386,  0.00013885213411413133,
                         2.5215107598342e-05,    0.0010600299574434757,  7.754525722702965e-05,
                         6.728082553308923e-06,  0.00011594349780352786, 3.815711897914298e-05,
                         0.00016582649550400674, 0.00020121395937167108, 6.044603287591599e-05,
                         0.00039762316737324,    0.00021905283210799098, 6.446620682254434e-05,
                         8.098148828139529e-05,  3.5016375477425754e-05, 6.13194570178166e-05,
                         5.557535769185051e-05,  2.6160481866099872e-05, 3.7423018511617556e-05,
                         0.00013071564899291843, 4.652100324165076e-05,  0.0001081921873264946,
                         0.00013098061026539654, 0.00013337192649487406, 0.00015945914492476732,
                         2.302963548572734e-05,  0.00013965938705950975, 0.0002570712531451136,
                         5.9214613429503515e-05, 0.00016779464203864336, 4.033782170154154e-05,
                         0.0002005429851124063,  0.00015491252997890115, 0.0001919539790833369,
                         0.0003232807503081858,  0.0005665213684551418,  0.0003921345341950655,
                         0.00012927678471896797, 6.983050843700767e-05,  0.00017194014799315482,
                         0.00017555807426106185, 0.0005651130923070014,  0.00016045909433159977,
                         7.353943510679528e-05,  3.327442391309887e-05,  0.00010464791557751596,
                         0.00017770145495887846, 0.00019503930525388569, 3.593524525058456e-05,
                         7.299174467334524e-05,  0.00011085447476943955, 0.0015461808070540428,
                         0.00023429669090546668, 0.0005100502748973668,  0.00012134326971136034,
                         1.8172320778830908e-05, 5.7164186728186905e-05, 0.00010796243441291153,
                         0.000616897305008024,   1.7307998859905638e-05, 8.998156772577204e-06,
                         0.00020143587607890368, 2.5742843718035147e-05, 9.421367394679692e-06,
                         0.0004883055225946009,  0.00014328760153148323, 0.0002184668992413208,
                         0.000360303558409214,   4.620299660018645e-05,  0.0003719372325576842,
                         3.119655593764037e-05,  0.0002547638723626733,  3.0411381885642186e-05,
                         0.00042920323903672397, 1.5062071724969428e-05, 2.7930551368626766e-05,
                         8.325288217747584e-05,  0.0001650981866987422,  3.081905379076488e-05,
                         0.0007014295551925898,  0.00010546566045377403, 0.0004353060503490269,
                         8.749144762987271e-05,  4.025379894301295e-05,  0.0001892313885036856,
                         3.116768129984848e-05,  6.95280177751556e-05,   0.0004830912221223116,
                         7.24948913557455e-05,   0.00020982489513698965, 2.3874472390161827e-05,
                         4.299950887798332e-05,  9.177169704344124e-05,  7.825226930435747e-05,
                         0.00021127721993252635, 0.0003538443415891379,  8.039485692279413e-05,
                         8.774982416070998e-05,  0.00048089493066072464, 5.556706673814915e-05,
                         0.00010326187475584447, 6.75350456731394e-05,   0.00010271250357618555,
                         0.0006448521162383258,  0.0012764432467520237,  4.951105893269414e-06,
                         0.0018216428579762578,  0.0002764989039860666,  0.00011429812730057165,
                         0.00014917703811079264, 8.215715206461027e-05,  7.584213017253205e-05,
                         0.00010924851085292175, 3.843596641672775e-05,  0.00013867643428966403,
                         0.0008998126722872257,  0.00036728742998093367, 0.00022987768170423806,
                         0.00031355320243164897, 0.0002601623709779233,  0.00013943643716629595,
                         9.769511962076649e-05,  0.00038833494181744754, 0.00022292204084806144,
                         0.00012875194079242647, 0.00035088081494905055, 0.00032197649125009775,
                         6.109078822191805e-05,  0.00013439766189549118, 0.00015746573626529425,
                         0.00013564918481279165, 0.000316591962473467,   0.0001224081206601113,
                         0.00010214330541202798, 0.00022254687792155892, 4.258806802681647e-05,
                         7.79872716520913e-05,   0.0003081263566855341,  0.00022837449796497822,
                         0.00017036762437783182, 4.2254210711689666e-05, 0.00016245779988821596,
                         9.060370939550921e-05,  3.6105597246205434e-05, 6.272815517149866e-05,
                         0.00027360042440705,    3.1127536203712225e-05, 3.957294029532932e-05,
                         0.00010694485536077991, 9.556152508594096e-05,  4.0095143049256876e-05,
                         3.478180224192329e-05,  0.00014473703049588948, 3.225293767172843e-05,
                         7.817436562618241e-05,  7.285890751518309e-05,  2.108372063958086e-05,
                         0.00031842809403315187, 2.2502452338812873e-05, 0.00011863818508572876,
                         0.0001586342550581321,  0.0007446666131727397,  8.461386460112408e-05,
                         0.00015828527102712542, 0.00010701152496039867, 0.00023815222084522247,
                         6.900433072587475e-05,  0.0011593687813729048,  0.00023200381838250905,
                         6.875578401377425e-05,  8.058057574089617e-05,  2.8973476219107397e-05,
                         3.6234137951396406e-05, 0.0008192755049094558,  5.482709184434498e-06,
                         0.00014004150580149144, 4.748324863612652e-05,  0.00023998435062821954,
                         3.9742564695188776e-05, 8.721258927835152e-05,  0.00017967101302929223,
                         9.10362068680115e-05,   9.675761248217896e-06,  3.395250314497389e-05,
                         0.0002111409412464127,  0.0004061392100993544,  0.00018572612316347659,
                         5.6853750720620155e-05, 0.0005264063365757465,  3.8720801967429e-05,
                         0.00021318667859304696, 0.0003867381310556084,  2.1562926121987402e-05,
                         4.816453656530939e-05,  7.314124377444386e-05,  0.0001765633060131222,
                         0.0017283951165154576,  3.725741044036113e-05,  8.569646161049604e-05,
                         2.4105336706270464e-05, 0.0009160739136859775,  6.546888471348211e-05,
                         0.00034673718619160354, 0.00015701036318205297, 0.00013726695033255965,
                         0.0005293278954923153,  0.0002934798249043524,  1.3650132132170256e-05,
                         0.00010088741692015901, 0.00026066723512485623, 0.00023730640532448888,
                         2.1998850570525974e-05, 0.00010761457087937742, 2.504090116417501e-05,
                         0.00016022597264964134, 0.00017056900833267719, 0.00039251201087608933,
                         0.0001266610634047538,  2.6508589144214056e-05, 2.4509015929652378e-05,
                         0.00020370798301883042, 0.00021059981372673064, 0.000851740303914994,
                         7.49940809328109e-05,   0.00031270369072444737, 0.00011084987636422738,
                         0.0003040836891159415,  0.0005308794206939638,  0.00014148587069939822,
                         0.0006320843822322786,  5.958821202511899e-05,  0.00013719635899178684,
                         8.316861203638837e-05,  7.013979484327137e-05,  0.0004403236380312592,
                         5.855868221260607e-05,  5.6918521295301616e-05, 0.0002685289946384728,
                         0.00014998798724263906, 0.0005522639257833362,  0.00017377830226905644,
                         0.00031938604661263525, 0.00032563117565587163, 4.7231460484908894e-05,
                         0.00016410906391683966, 0.0004875041777268052,  0.00015066609194036573,
                         9.529400995234028e-05,  0.00014458269288297743, 3.066117278649472e-05,
                         3.615943933255039e-05,  0.00041379101458005607, 4.431339766597375e-05,
                         0.00013793019752483815, 0.0003893983375746757,  0.00012110473471693695,
                         4.657755926018581e-05,  0.0003058603033423424,  0.00030485313618555665,
                         0.0004203848075121641,  7.523674867115915e-05,  7.543943502241746e-05,
                         2.6412659281049855e-05, 0.00023626364418305457, 3.057051435462199e-05,
                         0.00024206875241361558, 0.00016420632891822606, 4.935151446261443e-05,
                         2.014169513131492e-05,  3.2344098144676536e-05, 3.117741653113626e-05,
                         0.00021441903663799167, 9.16544595384039e-05,   0.00045733893057331443,
                         5.562366277445108e-05,  8.862963295541704e-05,  5.583929305430502e-05,
                         0.0003744647547136992,  0.0006415965035557747,  0.0004241352144163102,
                         0.00030252078431658447, 2.5181836463161744e-05, 0.00034849546500481665,
                         2.4186947484849952e-05, 0.00020635950204450637, 0.00011212893150513992,
                         0.00010501526412554085, 0.0003522791375871748,  0.0008702671620994806,
                         0.00035974953789263964, 2.5264909709221683e-05, 0.00018547300714999437,
                         0.00022087080287747085, 0.0007862506317906082,  0.00048400775995105505,
                         0.00021333666518330574, 4.470085696084425e-05,  0.00021884808666072786,
                         0.0001426753879059106,  5.951600178377703e-05,  0.0003137350722681731,
                         0.00022577714116778225, 0.0001560543751111254,  0.00034192667226307094,
                         0.00040395493851974607, 0.00020029580628033727, 9.062216122401878e-05,
                         0.0002514974039513618,  4.003926005680114e-05,  0.00010155135532841086,
                         5.141657675267197e-05,  4.0421393350698054e-05, 0.00034386402694508433,
                         0.00010207678860751912, 6.581478373846039e-05,  0.00021267347619868815,
                         0.0001919671194627881,  0.00022929332044441253, 3.3442604035371915e-05,
                         4.859039108850993e-05,  0.0002383730316068977,  0.00014330029080156237,
                         0.00010510539141250774, 4.1138318920275196e-05, 5.310439155437052e-05,
                         1.7457077774452046e-05, 6.22147781541571e-05,   0.001621707808226347,
                         0.00019511079881340265, 5.496019366546534e-05,  7.032130815787241e-05,
                         0.000501804577652365,   1.5032948795123957e-05, 5.388326462707482e-05,
                         0.00018339024973101914, 7.804843335179612e-05,  0.000510411977302283,
                         0.0004190959734842181,  0.00011233106488361955, 0.00010795013076858595,
                         1.4981953427195549e-05, 3.787155219470151e-05,  7.724993338342756e-05,
                         0.00025576844927854836, 0.00036184367490932345, 1.901236282719765e-05,
                         1.932503073476255e-05,  0.00022269708279054612, 0.00033038598485291004,
                         0.0001315536064794287,  0.0001100906592910178,  8.390207949560136e-05,
                         0.00045761888031847775, 1.9380888261366636e-05, 0.00016360073641408235,
                         0.00042095198296010494, 9.635291644372046e-05,  2.6471012461115606e-05,
                         0.00014319352339953184, 1.3652905181515962e-05, 5.176504055270925e-05,
                         0.00019910559058189392, 3.8128964661154896e-05, 8.348246046807617e-05,
                         0.000272914970992133,   0.000735436740797013,   0.0012113762786611915,
                         7.766451744828373e-06,  0.00043874408584088087, 0.00015252370212692767,
                         0.00038665131432935596, 3.4806573239620775e-05, 7.81911367084831e-05,
                         4.490111314225942e-05,  4.5259188482305035e-05, 7.001638732617721e-05,
                         0.00020952269551344216, 4.4170090404804796e-05, 0.00018363782146479934,
                         0.00023442978272214532, 0.00021116540301591158, 8.076595258899033e-05,
                         4.426784289535135e-05,  0.00011613504466367885, 0.0002795183099806309,
                         0.0007279622368514538,  6.449387728935108e-05,  0.0011631244560703635,
                         2.825178307830356e-05,  0.00024276043404825032, 0.00014773986185900867,
                         0.00027502080774866045, 2.2059901311877184e-05, 0.0007576875505037606,
                         0.00019629806047305465, 4.5232394768390805e-05, 0.0005259803147055209,
                         4.997231008019298e-05,  0.00013748949277214706, 0.00018474581884220243,
                         9.283359395340085e-05,  9.317627700511366e-05,  0.0002150877844542265,
                         9.166888048639521e-05,  0.0005352998850867152,  3.1514438887825236e-05,
                         1.8319373339181766e-05, 5.315237649483606e-05,  9.144338400801644e-05,
                         6.866997864563018e-05,  3.847355765174143e-05,  0.0002909055328927934,
                         0.00010121604282176122, 0.0007893646252341568,  0.00013374398986343294,
                         1.6579968360019848e-05, 0.00016697928367648274, 0.00014441966777667403,
                         4.6471097448375076e-05, 9.565794607624412e-05,  3.437884515733458e-05,
                         0.0003384473966434598,  8.032163168536499e-05,  0.00021248873963486403,
                         0.00017067292355932295, 8.184211765183136e-05,  0.00012423668522387743,
                         0.0005360146169550717,  0.00011138522677356377, 7.415916479658335e-05,
                         0.00015270979201886803, 0.00013110313739161938, 3.350977203808725e-05,
                         0.000358151359250769,   7.43953205528669e-05,   0.0008761692442931235,
                         0.00020046081044711173, 0.0001372115220874548,  0.00012035721738357097,
                         0.0004483993398025632,  7.267188630066812e-05,  5.436302672023885e-05,
                         0.0002867841103579849,  6.607171962969005e-05,  6.495875277323648e-05,
                         6.961656617932022e-05,  3.0251148928073235e-05, 0.000107911582745146,
                         9.281287930207327e-05,  5.8111818361794576e-05, 0.0002471228362992406,
                         0.00011217268911423162, 0.0002965976018458605,  2.289410804223735e-05,
                         6.961580220377073e-05,  0.00011440314847277477, 9.397519897902384e-05,
                         2.583071000117343e-05,  0.00023863448586780578, 6.6895350755658e-05,
                         0.0006330314790830016,  0.00014283618656918406, 0.00020838702039327472,
                         6.520070746773854e-05,  7.013875438133255e-05,  0.00048414000775665045,
                         0.00011650051601463929, 0.0002089848421746865,  4.560989691526629e-05,
                         3.320068208267912e-05,  6.4213404584734235e-06, 0.00037897584843449295,
                         1.7891044990392402e-05, 0.00015220383647829294, 7.635194924660027e-05,
                         0.0004819950554519892,  0.0003999024338554591,  0.00013651761400979012,
                         0.00012876579421572387, 5.06916840095073e-05,   0.0009348746971227229,
                         7.2642789746169e-05,    0.00028388475766405463, 0.00014500370889436454,
                         9.033481910591945e-05,  0.00016541406512260437, 5.404602052294649e-05,
                         0.00028254021890461445, 7.253770309034735e-05,  0.0004368486115708947,
                         3.509408998070285e-05,  0.00013894801668357104, 0.00022734160302206874,
                         0.0003123737405985594,  3.775003642658703e-05,  6.020371802151203e-05,
                         0.0003032184031326324,  0.00013342239253688604, 0.00029313855338841677,
                         1.583420453243889e-05,  7.181878754636273e-05,  5.2993495046393946e-05,
                         0.00010562162788119167, 0.00012985219655092806, 0.0003402224974706769,
                         0.00029182014986872673, 9.189273259835318e-05,  9.42964106798172e-05,
                         0.00014373104204423726, 1.0153919902222697e-05, 9.941921598510817e-05,
                         2.287696406710893e-05,  0.00030541373416781425, 2.946111089840997e-05,
                         0.0002902799751609564,  5.088253601570614e-05,  0.00012590545520652086,
                         0.00018491230730433017, 0.00022575694310944527, 2.2898713723407127e-05,
                         4.084479223820381e-05,  6.790724000893533e-05,  0.00010791869135573506,
                         0.0006013797246851027,  0.00036310346331447363, 0.00028316688258200884,
                         0.0009861862054094672,  0.00017488091543782502, 0.00012195256567792967,
                         7.893583824625239e-05,  0.00012024571333313361, 0.00017536031373310834,
                         0.0001906894613057375,  0.0019149798899888992,  8.865791460266337e-05,
                         1.9825087292701937e-05, 0.0005528788897208869,  5.5882737797219306e-05,
                         0.00021143598132766783, 0.00036097478005103767, 2.9773200367344543e-05,
                         5.689488898497075e-05,  0.00020215383847244084, 8.667021029395983e-05,
                         0.00024645228404551744, 0.00011996815737802535, 5.501086889125872e-06,
                         0.0004288528871256858,  0.00021094741532579064, 8.305031951749697e-05,
                         0.0002854943450074643,  0.000540488341357559,   3.368921534274705e-05,
                         1.4870744962536264e-05, 0.00014262465992942452, 0.00017866273992694914,
                         0.00034358573611825705, 0.0003853381786029786,  0.00010048067633761093,
                         7.183485649875365e-06,  5.5467000493081287e-05, 0.00011255346180405468,
                         5.568172855419107e-05,  0.0005831897142343223,  8.517370588378981e-05,
                         0.0003139852196909487,  2.2790502043790184e-05, 1.5359151802840643e-05,
                         9.818735998123884e-05,  0.00019494678417686373, 5.191302625462413e-05,
                         0.00016809372755233198, 0.00013577251229435205, 3.354418004164472e-05,
                         0.00016357345157302916, 0.0004565261479001492,  0.0005452921031974256,
                         0.00010102784290211275, 5.172497549210675e-05,  6.196548201842234e-05,
                         0.00019851229444611818, 0.0002676268632058054,  0.0002744195226114243,
                         3.819862467935309e-05,  0.00016791005327831954, 0.0002330089919269085,
                         0.00014748063404113054, 0.0002554026141297072,  0.00013699280680157244,
                         0.00022270629415288568, 0.00025693082716315985, 8.569825877202675e-05,
                         0.00034637172939255834, 2.659666097315494e-05,  0.00014626895426772535,
                         0.00018190182163380086, 0.00025075546000152826, 3.6274919693823904e-05,
                         4.783398981089704e-05,  0.00037268386222422123, 0.000317509489832446,
                         0.00014837755588814616, 0.00022800557781010866, 9.95214213617146e-05,
                         0.00022293836809694767, 0.002308220136910677,   6.714765913784504e-05,
                         1.3924646736995783e-05, 4.329866715124808e-05,  0.001333769760094583,
                         6.44801402813755e-05,   4.646221350412816e-05,  4.560998786473647e-05,
                         0.00017418339848518372, 3.3252392313443124e-05, 0.0010026250965893269,
                         0.00014432989701163024, 0.00027143475017510355, 6.358264363370836e-05,
                         0.0004582780529744923,  0.00020443413814064115, 0.00013101061631459743,
                         0.0005080280243419111,  4.380320751806721e-05,  1.0372016731707845e-05,
                         0.0010700267739593983,  0.0001356645516352728,  0.00037053084815852344,
                         0.0006415956886485219,  0.00018792781338561326, 9.517684520687908e-05,
                         0.00026670066290535033, 8.134318341035396e-05,  0.0003151346172671765,
                         0.0002707907115109265,  4.7716803237563e-05,    0.00015617477765772492,
                         0.00018202202045358717, 0.00052956014405936,    6.324973946902901e-05,
                         8.832970343064517e-05,  0.00019774463726207614, 0.00048499388503842056,
                         7.439791806973517e-05,  0.00043249866575933993, 4.291339428164065e-05,
                         8.376776531804353e-05,  0.00019290127966087312, 0.00018190730770584196,
                         0.00015980825992301106, 4.320422885939479e-05,  0.0007411557016894221,
                         2.8415710403351113e-05, 9.38435405259952e-05,   0.0003701754321809858,
                         0.0003262649697717279,  0.00011779624765040353, 5.7748846302274615e-05,
                         4.212429121253081e-05,  4.6755114453844726e-05, 2.9624434318975545e-05,
                         0.0003142175264656544,  9.138594032265246e-05,  0.0002178175054723397,
                         0.00010514208406675607, 0.00016715143283363432, 0.0001468153204768896,
                         0.0005054958746768534,  4.934981552651152e-05,  0.00019563696696422994,
                         7.728983473498374e-05,  4.119069126318209e-05,  2.9521957912947983e-05,
                         0.0005898717208765447,  0.0004506756959017366,  0.00047843047650530934,
                         0.0005750490236096084,  2.2184354747878388e-05, 3.035282679775264e-05,
                         4.169788007857278e-05,  7.887154060881585e-05,  0.00013381874305196106,
                         4.653170253732242e-05,  0.0002382630918873474,  1.693891863396857e-05,
                         0.00019936547323595732, 0.00014069020107854158, 0.00015555911522824317,
                         4.127137435716577e-05,  0.00021923234453424811, 0.000431963853770867,
                         6.480619049398229e-05,  7.704092422500253e-05,  4.580924360197969e-05,
                         0.00027658158796839416, 0.0004058564081788063,  9.874134411802515e-05,
                         0.0002640924067236483,  0.0001334174012299627,  3.7288958992576227e-05,
                         0.00011126425670227036, 0.00013871167902834713, 0.00015781624824739993,
                         0.0002403714752290398,  0.00025293120415881276, 0.00017074942297767848,
                         8.978895493783057e-05,  0.00013512976875063032, 2.8363454475766048e-05,
                         0.00020987978496123105, 0.0001789832895155996,  2.609364310046658e-05,
                         0.0001257624535355717,  9.987688099499792e-05,  4.540410009212792e-05,
                         0.00038575069629587233, 7.858810567995533e-05,  0.0003906824567820877,
                         0.0001348896767012775,  2.4997643777169287e-05, 0.00025248885503970087,
                         0.0002619705628603697,  4.341510430094786e-05,  0.0003409428463783115,
                         1.5024936146801338e-05, 0.00010339188884245232, 9.434382081963122e-05,
                         5.4125350288813934e-05, 0.00010561975796008483, 0.00012680821237154305,
                         0.00023915346537251025, 0.00035869356361217797, 2.839059743564576e-05,
                         5.795215110993013e-05,  0.0008740962366573513,  3.07369846268557e-05,
                         0.00044191445340402424, 5.0675051170401275e-05, 8.998744306154549e-05,
                         0.0005470357136800885,  0.0005106227472424507,  9.136980952462181e-05,
                         0.00010507248225621879, 7.241528510348871e-05,  8.472974150208756e-05,
                         0.00013947481056675315, 0.00015488981443922967, 0.0003013249079231173,
                         0.00014629783981945366, 0.00019841798348352313, 0.0002428567677270621,
                         0.00024814941571094096, 9.118166781263426e-05,  4.1923623939510435e-05,
                         0.0001020468189381063,  0.0004332991666160524,  4.7585701395291835e-05,
                         0.00015551051183138043, 0.0001749478542478755,  0.00013986472913529724,
                         2.487747224222403e-05,  0.00018886319594457746, 0.0003319874231237918,
                         0.0005610075895674527,  0.0001318134309258312,  8.887398871593177e-05,
                         6.0237198340473697e-05, 3.135102451778948e-05,  0.00015922971942927688,
                         7.38181042834185e-05,   2.2163736502989195e-05, 0.00010338465654058382,
                         0.0004274311941117048,  0.00022517571051139385, 0.0002176159614464268,
                         0.00020136388775426894, 3.2680618460290134e-05, 5.1813221944030374e-05,
                         0.00010302250302629545, 7.452186400769278e-05,  0.0001670564233791083,
                         1.9071338101639412e-05, 2.0136758394073695e-05, 6.174598092911765e-05,
                         2.4719023713259958e-05, 0.00027632206911221147, 0.00010176531941397116,
                         3.228391142329201e-05,  4.6288205339806154e-05, 0.00019292130309622735,
                         0.00011535789235495031, 0.00011623626778600737, 0.00036970761721022427,
                         0.00011238067236263305, 0.0001974631450138986,  0.00015771675680298358,
                         0.0004181641270406544,  0.00014326297969091684, 6.077701618778519e-05,
                         0.00021568017837125808, 0.0002452194457873702,  0.0007968085119500756,
                         3.923739495803602e-05,  0.00015444285236299038, 6.450434011640027e-05,
                         2.0315519577707164e-05, 0.00020149274496361613, 0.00025341176660731435,
                         2.929470247181598e-05,  0.00018682068912312388, 0.00022096102475188673,
                         0.0003838548145722598,  0.00033454448566772044, 0.0011656780261546373,
                         0.00015327619621530175, 5.9032208810094744e-05, 0.00013602535182144493,
                         0.00010004724026657641, 0.0006984579376876354,  0.0001686085743131116,
                         4.596102371579036e-05,  4.097514465684071e-05,  6.325574213406071e-05,
                         3.727850707946345e-05,  3.062078758375719e-05,  9.618765761842951e-05,
                         0.00044635916128754616, 0.00012337036605458707, 3.29634131048806e-05,
                         0.00018846243619918823, 0.00016655716171953827, 0.0001471884606871754,
                         2.9402335712802596e-05, 0.00011742111382773146, 3.0159066227497533e-05,
                         7.903507503215224e-05,  0.00033486212487332523, 5.373421299736947e-05,
                         0.0003491297538857907,  7.885277591412887e-05,  2.5783287128433585e-05,
                         9.873281669570133e-05,  0.00034179649082943797, 0.00011974161316175014,
                         0.00014062755508348346, 0.0003245485422667116,  5.7397694035898894e-05,
                         0.0004794201231561601,  0.0002117725380230695,  5.7026049034902826e-05,
                         6.468535138992593e-05,  7.89040423114784e-05,   6.510405364679173e-05,
                         0.00016070443962235004, 2.246838812425267e-05,  7.894520967965946e-05,
                         1.6905494703678414e-05, 8.13116566860117e-05,   0.0007815736462362111,
                         6.049914736649953e-05,  0.00014702857879456133, 3.3458378311479464e-05,
                         0.0009841816499829292,  0.00019128464919049293, 0.00017250102246180177,
                         5.932528438279405e-05,  7.408267265418544e-05,  0.0001682812871877104,
                         3.610826752264984e-05,  9.15732016437687e-05,   0.0001755622506607324,
                         0.00044223267468623817, 1.4035344975127373e-05, 0.00035529647720977664,
                         0.00011058605741709471, 1.1110959349025507e-05, 0.00030314555624499917,
                         8.86760899447836e-05,   0.00012309529120102525, 0.00014280238247010857,
                         6.652524461969733e-05,  2.1147665393073112e-05, 7.674082007724792e-05,
                         0.00014968514733482152, 4.1784522181842476e-05, 7.388814992737025e-05,
                         0.000216313885175623,   0.00017158754053525627, 8.499997784383595e-05,
                         0.00014951812045183033, 8.565038297092542e-05,  0.0005359050701372325,
                         0.00043178469059057534, 0.00012446586333680898, 0.00016618770314380527,
                         6.0217698774067685e-05, 5.4182677558856085e-05, 9.38505691010505e-05,
                         0.0005656371358782053,  5.130225690663792e-05,  0.00018776769866235554,
                         2.6035626433440484e-05, 0.00022911024279892445, 4.210922998026945e-05,
                         0.00011825271940324455, 9.161202615359798e-05,  1.7772670616977848e-05,
                         4.092096060048789e-05,  7.973111496539786e-05,  5.612013046629727e-05,
                         0.00010423221829114482, 5.174806574359536e-05,  0.00021455287060234696,
                         2.3163733203546144e-05, 0.00019501225324347615, 0.00019760825671255589,
                         0.00016878597671166062, 0.0006736460491083562,  0.00018208094115834683,
                         3.9034468500176445e-05, 0.00019517244072631001, 0.0001243136648554355,
                         0.0003708635631483048,  8.76858684932813e-05,   0.0004611786862369627,
                         0.00025191460736095905, 0.0001454622979508713,  0.0003678884531836957,
                         2.867782131943386e-05,  4.810109385289252e-05,  0.00046080054016783834,
                         4.589226227835752e-05,  0.0004737415292765945,  7.958264177432284e-05,
                         0.00016916007734835148, 0.00030927080661058426, 0.00018385611474514008,
                         0.00017631381342653185, 0.00029239512514322996, 0.00017253693658858538,
                         0.00021903890592511743, 3.9988794014789164e-05, 7.307477062568069e-05,
                         6.519635644508526e-05,  2.4187154849641956e-05, 3.703513721120544e-05,
                         4.12444060202688e-05,   0.00015632560825906694, 2.9489008738892153e-05,
                         8.458760567009449e-05,  5.1681850891327485e-05, 0.00017121428390964866,
                         0.0005286243977025151,  0.0002100235433317721,  0.00014883000403642654,
                         0.0002625113120302558,  1.5442570656887256e-05, 0.00010646083683241159,
                         0.00020585389574989676, 0.0010848573874682188,  8.104043081402779e-05,
                         9.850987407844514e-05,  4.1899820644175634e-05, 4.2365980334579945e-05,
                         0.0003053968830499798,  4.0004204493016005e-05, 6.185170786920935e-05,
                         8.105561573756859e-05,  0.00025158084463328123, 0.00018902253941632807,
                         2.387060158071108e-05,  8.430239540757611e-05,  3.336280497023836e-05,
                         0.0006841321592219174,  0.0002789630671031773,  5.003428304917179e-05,
                         0.0004716197436209768,  9.554151620250195e-05,  1.2409375813149381e-05,
                         0.0001127787254517898,  4.6010140067664906e-05, 0.00034327624598518014,
                         0.00020285641949158162, 3.5706398193724453e-05, 6.775710062356666e-05,
                         0.00020186365873087198, 3.6212772101862356e-05, 0.00014831566659267992,
                         5.813928146380931e-05,  4.365924178273417e-05,  5.988199700368568e-05,
                         0.00021031817595940083, 0.0009462260641157627,  0.000173068605363369,
                         6.453406240325421e-05,  7.552326133009046e-05,  9.490624506724998e-05,
                         0.0002770734135992825,  0.00018414820078760386, 0.00013941855286248028,
                         7.660195842618123e-05,  0.0002438790979795158,  0.000182909716386348,
                         0.00035567450686357915, 0.00011959763651248068, 3.687675416585989e-05,
                         6.735327770002186e-05,  0.00044259848073124886, 6.443777965614572e-05,
                         4.494425593293272e-05,  4.012231875094585e-05,  0.0004385187348816544,
                         0.00011168816854478791, 0.00022843886108603328, 3.5517874493962154e-05,
                         0.0002059587714029476,  0.00012061939924024045, 9.996016160584986e-05,
                         1.6879164832062088e-05, 8.134524250635877e-05,  0.000585254980251193,
                         0.0003828458720818162,  0.00023764136130921543, 8.693188283359632e-05,
                         5.425915514933877e-05,  4.491169238463044e-05,  9.009289351524785e-05,
                         0.0002065613225568086,  4.719422577181831e-05,  0.00017062400002032518,
                         7.604027632623911e-05,  0.00017637309792917222, 0.0003875918919220567,
                         0.00031081854831427336, 3.1264491553884e-05,    7.504012319259346e-05,
                         0.0001218132529174909,  0.00012926847557537258, 6.77877978887409e-05,
                         0.00012042481102980673, 9.327189764007926e-05,  2.9263221222208813e-05,
                         0.00046661391388624907, 0.0002708836109377444,  0.00018321051902603358,
                         0.00010232430940959603, 0.0006831277278251946,  6.50958318146877e-05,
                         6.972220580792055e-05,  0.0003980605397373438,  1.525890183984302e-05,
                         0.00014876943896524608, 0.00038641472929157317, 5.888225496164523e-05,
                         0.0004287110350560397,  0.0003410264034755528,  4.594938945956528e-05,
                         0.0003745105059351772,  0.0005035099457018077,  7.182219269452617e-05,
                         0.0001360556052532047,  7.284737512236461e-05,  7.433624705299735e-05,
                         4.945690670865588e-05,  0.000301792926620692,   0.0001655136584304273,
                         0.00010501212091185153, 0.00012723886175081134, 6.734874477842823e-05,
                         0.0003081251052208245,  0.00010465719969943166, 0.00047832855489104986,
                         0.00021089939400553703, 2.8531821953947656e-05, 0.00018818100215867162,
                         3.835882671410218e-05,  3.70117268175818e-05,   0.00021348836889956146,
                         0.0002132316294591874,  2.056004996120464e-05,  0.0005345430690795183,
                         0.0001205759181175381,  0.000594727520365268,   0.00032955961069092155,
                         0.00010150381422135979, 0.0007526595727540553,  5.018713272875175e-05,
                         0.000272571574896574,   5.93829681747593e-05,   0.0007037156610749662,
                         9.826388850342482e-05,  3.0209201213438064e-05, 0.00022535794414579868,
                         7.993553299456835e-05,  5.6842582125682384e-05, 0.00034143836819566786,
                         0.00012245614198036492, 6.66226987959817e-05,   0.000198151363292709,
                         0.00010836483124876395, 0.0002435794158373028,  9.681997471489012e-05,
                         0.00011283983621979132, 1.8775530406855978e-05, 0.00037906659417785704,
                         0.000731621403247118,   9.291415335610509e-05,  2.964228406199254e-05,
                         0.00011042471305700019, 0.0001483460800955072,  4.8316214815713465e-05,
                         5.995625178911723e-05,  4.506122786551714e-05,  0.00031987534021027386,
                         7.104526594048366e-05,  0.00019086949760094285, 1.2958852494193707e-05,
                         5.008708467357792e-05,  5.440348104457371e-05,  4.818856541533023e-05,
                         0.000266148301307112,   3.1168201530817896e-05, 0.00027919167769141495,
                         0.00011141349386889488, 9.178269101539627e-05,  6.934063276275992e-05,
                         0.00023993448121473193, 0.00021177707822062075, 2.850750752259046e-05,
                         5.3021401981823146e-05, 0.0016836144495755434,  0.000108134830952622,
                         9.745467104949057e-05,  0.00016762591258157045, 5.656907887896523e-05,
                         0.00028134751482866704, 0.0008920907275751233,  0.00010656540689524263,
                         0.0001402520720148459,  0.00016546713595744222, 3.086167271248996e-05,
                         0.0009617781033739448,  0.00019709303160198033, 2.287101960973814e-05,
                         0.0003333060594741255,  0.00024966560886241496, 0.00022882157645653933,
                         0.00021241376816760749, 0.0001540935190860182,  2.665155625436455e-05,
                         0.0001434819569112733,  0.000441887357737869,   6.701028178213164e-05,
                         0.000480662944028154,   0.00048095366219058633, 6.530637620016932e-05,
                         0.0007472641300410032,  0.00015994373825378716, 3.601920252549462e-05,
                         0.0002904485445469618,  0.00034142410731874406, 2.59219050349202e-05,
                         0.0005756313912570477,  2.191830935771577e-05,  0.00020586216123774648,
                         2.6884164981311187e-05, 0.0007072216249071062,  0.0010692337527871132,
                         6.967740773689002e-05,  0.0004732793604489416,  0.00010738575656432658,
                         0.00017553554789628834, 0.00023822269577067345, 0.00023097651137504727,
                         0.00025334098609164357, 0.00014207090134732425, 0.00026706309290602803,
                         0.00016048802353907377, 0.00012989225797355175, 0.00019568247080314904,
                         8.538654947187752e-05,  0.00024878926342353225, 0.00020330377446953207,
                         0.0002924831642303616,  4.4970856833970174e-05, 0.00019614087068475783,
                         0.0004934894968755543,  2.0650955775636248e-05, 0.00023669998336117715,
                         4.951887603965588e-05,  0.00010327068594051525, 8.384889224544168e-05,
                         5.8942281611962244e-05, 4.4864027586299926e-05, 0.00010745715553639457,
                         0.00014728291716892272, 0.00011273803102085367, 0.00015935189730953425,
                         0.0005964831216260791,  0.00034383582533337176, 0.00012221475481055677,
                         0.00014171165821608156, 0.00025123811792582273, 6.002187365083955e-05,
                         3.5483360989019275e-05, 0.00019583218090701848, 5.336109461495653e-05,
                         4.4800446630688384e-05, 0.00019295408856123686, 0.00028849192312918603,
                         0.00048336907639168203, 0.0001501132792327553,  5.424063419923186e-05,
                         7.544519030489028e-05,  0.00041248227353207767, 0.00026618599076755345,
                         5.5646720284130424e-05, 1.5896986951702274e-05, 0.0002530130441300571,
                         0.00011148097837576643, 0.00018316299247089773, 0.0004210015758872032,
                         0.00012004741438431665, 0.00013710296479985118, 0.00012048861390212551,
                         0.0001625531876925379,  6.0322912759147584e-05, 0.0002987100160680711,
                         7.201168773462996e-05,  0.0004490380233619362,  5.248508387012407e-05,
                         7.143129187170416e-05,  0.0006298095686361194,  0.00010077063780045137,
                         2.8204929549247026e-05, 6.866601324873045e-05,  9.386050805915147e-05,
                         0.0003014087851624936,  0.00024257540644612163, 0.00015814841026440263,
                         0.00019145784608554095, 0.0001560261589474976,  6.327309529297054e-05,
                         0.0005067425663582981,  0.00020802323706448078, 0.00010336034756619483,
                         0.00035326724173501134, 0.0001635705993976444,  0.0003380972193554044,
                         0.000116657443868462,   0.00019055811571888626, 0.00020451487216632813,
                         9.375901572639123e-05,  0.0008298967150039971,  7.768740761093795e-05,
                         6.806790770497173e-05,  0.00010481366189196706, 0.0003129917604383081,
                         7.662611460546032e-05,  7.65363874961622e-05,   0.00015181500930339098,
                         0.0001841377525124699,  9.813915312406607e-06,  2.9785811420879327e-05,
                         2.5014813218149357e-05, 0.00014593357627745718, 2.5846569769782946e-05,
                         1.7292284610448405e-05, 9.400451381225139e-05,  0.00024824930005706847,
                         4.251881910022348e-05,  0.00013836838479619473, 0.00023507549485657364,
                         4.438157702679746e-05,  9.565174696035683e-05,  0.0001886815152829513,
                         6.700580706819892e-05,  5.762905129813589e-05,  1.5193740182439797e-05,
                         7.75998123572208e-05,   0.0003175877209287137,  0.00011644123878795654,
                         0.0002810204168781638,  6.735937495250255e-05,  0.00011400594667065889,
                         0.00011719021858880296, 0.0007369333179667592,  0.0003676634805742651,
                         4.2962408770108595e-05, 8.591761434217915e-05,  4.7220924898283556e-05,
                         0.00029863635427318513, 0.00023779951152391732, 8.436154894297943e-05,
                         5.7942172134062275e-05, 0.0003059529117308557,  1.994693411688786e-05,
                         0.0007048605475574732,  9.034705726662651e-05,  0.00015016058750916272,
                         0.0004742021264974028,  0.0004623237473424524,  7.51186307752505e-05,
                         0.00117829954251647,    4.610834002960473e-05,  1.6235058865277097e-05,
                         0.0003276325296610594,  6.51766822556965e-05,   0.0003243338142056018,
                         0.00019666258594952524, 0.00037508842069655657, 0.0007928177365101874,
                         0.0005732563440687954,  0.00019031860574614257, 7.601117249578238e-05,
                         0.00029026655829511583, 0.0002352957526454702,  0.0002534597006160766,
                         3.38506288244389e-05,   4.940175131196156e-05,  0.00014159591228235513,
                         7.437617023242638e-05,  0.00035636330721899867, 0.00026992757921107113,
                         5.739909101976082e-05,  8.224816701840609e-05,  3.3203006751136854e-05,
                         0.0005245632492005825,  0.00034969724947586656, 0.00012173756113043055,
                         3.234975883970037e-05,  0.00034138033515773714, 0.00027895174571312964,
                         3.3819975215010345e-05, 6.957902951398864e-05,  2.0249917724868283e-05,
                         0.00011152670776937157, 4.7803459892747924e-05, 0.00012813876674044877,
                         0.0002893023192882538,  9.337469236925244e-05,  0.00020188228518236428,
                         0.00021554467093665153, 0.00017964864673558623, 0.00040863925823941827,
                         1.7708624000079e-05,    7.429301331285387e-05,  6.068953371141106e-05,
                         0.00025417032884433866, 5.474696445162408e-05,  4.729368447442539e-05,
                         0.00045028352178633213, 0.00020897705690003932, 0.0003872588276863098,
                         2.1024497982580215e-05, 6.869315984658897e-05,  0.0008463598205707967,
                         0.0003158586914651096,  9.590100671630353e-05,  4.6166242100298405e-05,
                         7.441196794388816e-05,  1.6289563063764945e-05, 0.0002136774710379541,
                         0.0003128043608739972,  7.045280653983355e-05,  0.0001571881293784827,
                         6.678867066511884e-05,  1.5184919902822003e-05, 0.00035715647391043603,
                         3.9897488022688776e-05, 5.013740883441642e-05,  0.00026686349883675575,
                         0.0006622823420912027,  0.0003382636059541255,  0.00011269868991803378,
                         0.00015253062883857638, 0.00025076206657104194, 0.0004904860397800803,
                         0.0006151087582111359,  0.0001431396958651021,  4.149507367401384e-05,
                         0.000465354387415573,   0.00034043315099552274, 0.0002684758510440588,
                         0.00014711760741192847, 0.00011434048064984381, 0.0002464884892106056,
                         3.140487024211325e-05,  0.0006141584017314017,  0.0001582004624651745,
                         0.00010219294927082956, 7.78123940108344e-05,   0.0004318171995691955,
                         5.621137097477913e-05,  0.00017717352602630854, 7.4295538070146e-05,
                         1.8584167264634743e-05, 0.00016389269148930907, 0.00018459353304933757,
                         0.00027655647136271,    0.00010091243166243657, 3.39554826496169e-05,
                         2.5600320441299118e-05, 0.000365215732017532,   8.489239553455263e-05,
                         0.0003975625440943986,  9.198984480462968e-05,  2.933850191766396e-05,
                         0.00024996179854497313, 8.67923372425139e-05,   8.557433466194198e-05,
                         4.822290429729037e-05,  0.0006889996584504843,  9.393626532983035e-05,
                         0.0001832575217122212,  9.140724432654679e-05,  6.492369720945135e-05,
                         0.0002817034546751529,  0.00011661638563964516, 4.966300184605643e-05,
                         0.00021412732894532382, 0.0003135979059152305,  8.916190563468263e-05,
                         0.00026629198691807687, 0.0003983704373240471,  5.515798693522811e-05,
                         0.00013607204891741276, 0.0002253357961308211,  0.0002641736646182835,
                         5.41142508154735e-05,   6.159670738270506e-05,  3.503303014440462e-05,
                         0.00017794738232623786, 0.00012954222620464861, 5.2829978812951595e-05,
                         0.00018100900342687964, 0.00015986867947503924, 0.00025317928520962596,
                         0.00023551148478873074, 6.821333954576403e-05,  0.00015163928037509322,
                         0.00014092048513703048, 3.2303822081303224e-05, 0.0001612719934200868,
                         0.0004437618190422654,  0.0001230662892339751,  8.369294664589688e-05,
                         0.0004367660149000585,  1.4973104043747298e-05, 0.0004431683337315917,
                         9.104749915422872e-05,  0.00017940279212780297, 6.108621892053634e-05,
                         0.0002126757026417181,  0.00057620694860816,    0.0005110570346005261,
                         0.0015509311342611909,  0.0001261240686289966,  4.796891153091565e-05,
                         0.00010514350287849084, 6.954685522941872e-05,  6.160100019769743e-05,
                         0.0004977559437975287,  1.9557275663828477e-05, 2.9178783734096214e-05,
                         1.4989735063863918e-05, 0.0006435141549445689,  8.450935274595395e-05,
                         2.251495789096225e-05,  0.00019738005357794464, 7.108640420483425e-05,
                         0.00013978926290292293, 0.0003272180329076946,  0.00015283924585673958,
                         0.0004309896321501583,  0.0002793716557789594,  7.280514546437189e-05,
                         0.00025441631441935897, 6.030893200659193e-05,  0.00017198696150444448,
                         0.0003458827268332243,  8.59964347910136e-05,   0.00016050405974965543,
                         0.00011022291437257081, 0.0001223636354552582,  0.0001246682513738051,
                         5.4528300097445026e-05, 4.4784534111386165e-05, 6.435696559492499e-05,
                         0.0001997352810576558,  6.883796595502645e-05,  5.6129654694814235e-05,
                         3.7615682231262326e-05, 0.0005689696408808231,  5.269834218779579e-05,
                         5.607973798760213e-05,  0.00014111452037468553, 0.00014602657756768167,
                         0.00012664530368056148, 2.724310070334468e-05,  9.180430060951039e-05,
                         2.7719050194718875e-05, 0.000752603227738291,   2.2733562218490988e-05,
                         0.00043848747736774385, 7.321033626794815e-05,  0.00010681994899641722,
                         0.0007918888004496694,  3.2906278647715226e-05, 0.0001195486038341187,
                         0.0006685711559839547,  0.00024690196732990444, 0.00012701429659500718,
                         5.192072876525344e-06,  0.00023114972282201052, 3.363086943863891e-05,
                         0.00010007142554968596, 0.00011702799383783713, 9.062336903298274e-05,
                         0.00014091430057305843, 4.693123628385365e-05,  0.0005173794343136251,
                         0.00027759181102737784, 3.729333184310235e-05,  0.00014265386562328786,
                         0.00016400503227487206, 4.600250758812763e-05,  9.906420018523932e-05,
                         0.000506692798808217,   4.690045170718804e-05,  0.0005894490168429911,
                         4.402420381666161e-05,  1.765457091096323e-05,  0.0004898069310002029,
                         6.471910455729812e-05,  0.00034561348729766905, 0.0001671925710979849,
                         0.00020491148461587727, 0.00011526716843945906, 0.0001083610623027198,
                         8.923646964831278e-05,  6.144071085145697e-05,  9.055162081494927e-05,
                         0.0005391712766140699,  0.00021693734743166715, 5.26065741723869e-05,
                         8.639362931717187e-05,  0.0006286388379521668,  9.783339191926643e-05,
                         8.181890734704211e-05,  8.294128929264843e-05,  4.088676723768003e-05,
                         3.0188148230081424e-05, 0.0005057562957517803,  0.00024402349663432688,
                         8.043166599236429e-05,  0.0003688243159558624,  6.875208055134863e-05,
                         1.5791969417477958e-05, 0.00012155482545495033, 3.411398938624188e-05,
                         7.78614921728149e-05,   0.0003831380745396018,  1.1288315363344736e-05,
                         0.00024691547150723636, 9.37583899940364e-05,   0.0002550743520259857,
                         9.11384922801517e-05,   7.298788841580972e-05,  0.00022292044013738632,
                         0.0001973304752027616,  8.814515604171902e-05,  3.4147888072766364e-05,
                         6.430913344956934e-05,  9.833190415520221e-05,  0.0002385626285104081,
                         0.00020531038171611726, 0.00040169511339627206, 0.00010705470776883885,
                         4.472140790312551e-05,  8.317400352098048e-05,  1.8523749531595968e-05,
                         0.00022854292183183134, 3.048643520742189e-05,  0.00014190700312610716,
                         0.00025147615815512836, 2.5250301405321807e-05, 5.953567233518697e-05,
                         0.00036043772706761956, 0.00019773351959884167, 0.00023643465829081833,
                         0.00013078445044811815, 2.3936321667861193e-05, 3.520984319038689e-05,
                         0.0001563409750815481,  0.00029047299176454544, 0.0001742268359521404,
                         3.1764808227308095e-05, 5.7970941270468757e-05, 6.0039845266146585e-05,
                         0.0015815000515431166,  1.8106826246366836e-05, 0.0002557407133281231,
                         5.153945676283911e-05,  1.4213105714588892e-05, 0.0002777767076622695,
                         0.00014647092029917985, 9.355251677334309e-05,  0.00020786264212802052,
                         3.838455450022593e-05,  0.00033674182486720383, 6.215223402250558e-05,
                         4.499750048125861e-06,  6.937935540918261e-05,  0.0007747880299575627,
                         0.00016814212722238153, 2.5259261747123674e-05, 9.757668158272281e-05,
                         1.3284722626849543e-05, 0.00012044057075399905, 0.0006557292072102427,
                         0.00012731889728456736, 6.128776294644922e-05,  0.00020056794164702296,
                         0.00011066427396144718, 2.8714794098050334e-05, 7.070905849104747e-05,
                         6.435171235352755e-05,  7.481196371372789e-05,  0.0009514465928077698,
                         4.74567168566864e-05,   2.390582267253194e-05,  0.00014069033204577863,
                         9.379492257721722e-05,  7.109901343937963e-05,  0.0002675101568456739,
                         0.00026454130420461297, 0.00013457682507578284, 0.00039786819252185524,
                         8.130491187330335e-05,  3.943264528061263e-05,  0.0002379871584707871,
                         0.00017338231555186212, 0.000196750697796233,   0.00012360501568764448,
                         7.101305527612567e-05,  0.0002084573934553191,  0.00019644177518785,
                         9.380382107337937e-05,  0.0011446003336459398,  0.00010628771997289732,
                         6.21587605564855e-05,   0.00015112452092580497, 1.1969570550718345e-05,
                         0.00010994218609994277, 7.914373418316245e-05,  3.189186827512458e-05,
                         7.89240948506631e-05,   0.0005345086683519185,  5.4356802138499916e-05,
                         0.0001636738597881049,  9.968390077119693e-05,  5.016698924009688e-05,
                         0.0001930876460392028,  0.00012599078763742,    0.00033093959791585803,
                         0.00010717628174461424, 0.00016754415992181748, 2.51449291681638e-05,
                         0.00010369083611294627, 0.000429959618486464,   0.00010863677016459405,
                         2.672215850907378e-05,  5.2721839892910793e-05, 0.00037598295602947474,
                         0.0005243901978246868,  6.814321386627853e-05,  0.0001864653459051624,
                         8.544536103727296e-05,  8.510051702614874e-05,  0.00020812773436773568,
                         7.91859274613671e-05,   1.8708700736169703e-05, 9.210351709043607e-05,
                         7.651548366993666e-05,  6.923490582266822e-05,  2.818365828716196e-05,
                         6.125565414549783e-05,  0.00010626977746142074, 0.0003589543921407312,
                         0.00017969576583709568, 7.370003731921315e-05,  0.0001537752104923129,
                         0.0005036108195781708,  5.705983494408429e-05,  0.0001488930283812806,
                         4.410614928929135e-05,  9.017992852022871e-05,  4.9663023673929274e-05,
                         0.0002780204813461751,  4.1028990381164476e-05, 0.00019905348017346114,
                         0.0001932330778799951,  0.00011576477845665067, 5.6875280279200524e-05,
                         1.9019800674868748e-05, 4.7703448217362165e-05, 5.698842869605869e-05,
                         0.0006337149534374475,  2.9387227186816745e-05, 0.00023934587079565972,
                         5.09263263666071e-05,   7.593538612127304e-05,  0.0010144885163754225,
                         4.614548379322514e-05,  0.0006329491152428091,  8.514800720149651e-05,
                         3.95661081711296e-05,   0.00018983353220392019, 0.00010423143248772249,
                         2.7323329049977474e-05, 9.14336706046015e-05,   0.00014875404303893447,
                         0.0007051110151223838,  3.0379047530004755e-05, 3.21887455356773e-05,
                         6.294519698712975e-05,  3.085301796090789e-05,  0.00032124100835062563,
                         0.0003766452136915177,  0.0003088836674578488,  9.627038525650278e-05,
                         0.00024172496341634542, 0.0006432119989767671,  0.00011931658082176,
                         7.152788020903245e-05,  0.00011056407674914226, 0.0002563202870078385,
                         4.364558481029235e-05,  0.0003618532791733742,  3.8437025068560615e-05,
                         0.00025337529950775206, 8.549031190341339e-05,  3.4461289033060893e-05,
                         7.295739123946987e-06,  0.0001766279456205666,  0.0001750944065861404,
                         7.278261182364076e-05,  1.4910699064785149e-05, 8.845631236908957e-05,
                         0.00024734847829677165, 0.0002581994340289384,  0.00020895055786240846,
                         8.686860383022577e-05,  0.0007055060123093426,  0.00026050666929222643,
                         0.00029758221353404224, 0.0002368610439589247,  2.9174276278354228e-05,
                         4.7919555072439834e-05, 0.0001239062985405326,  1.1196048944839276e-05,
                         9.693429092294537e-06,  1.2749102097586729e-05, 5.0253922381671146e-05,
                         3.805511005339213e-05,  5.825353218824603e-05,  0.0002599233412183821,
                         3.8388341636164114e-05, 0.00011050423927372321, 0.0020802924409508705,
                         7.50634862924926e-05,   0.00017546687740832567, 0.0001574833004269749,
                         0.0003745815483853221,  0.00011210076627321541, 0.00025134728639386594,
                         0.0009071022504940629,  3.2387481041951105e-05, 0.0001827971573220566,
                         0.00012756070645991713, 0.00016542467346880585, 0.00046108680544421077,
                         0.00028692244086414576, 0.00020703382324427366, 0.00010677115642465651,
                         0.00010626977746142074, 4.362850449979305e-05,  0.000201687405933626,
                         0.00012061960296705365, 6.158684118418023e-05,  1.737455568218138e-05,
                         0.0002411832829238847,  0.00017561655840836465, 8.322133362526074e-05,
                         0.00015360467659775168, 7.6290663855616e-05,    6.487659265985712e-05,
                         9.344319551018998e-05,  9.417584806215018e-05,  0.00027722647064365447,
                         0.00028058222960680723, 5.60661283088848e-05,   0.0004667957837227732,
                         0.00013434466382022947, 4.4745855120709166e-05, 0.0003512725234031677,
                         0.0001351766404695809,  4.063929009134881e-05,  4.1545677959220484e-05,
                         2.2745241949451156e-05, 0.000439307012129575,   0.00015304084809031337,
                         4.5884800783824176e-05, 2.1642046704073437e-05, 0.00012630516721401364,
                         0.00060648302314803,    7.445259689120576e-05,  4.661777711589821e-05,
                         0.0004098880453966558,  9.799755207495764e-05,  0.00018870458006858826,
                         2.891797703341581e-05,  0.00031540318741463125, 0.00047074593021534383,
                         0.00030929449712857604, 4.8702968342695385e-05, 7.54117063479498e-05,
                         7.242710125865415e-05,  4.292854282539338e-05,  0.0001225679588969797,
                         0.00029762720805592835, 0.000166277342941612,   5.9296922700013965e-05,
                         0.00015101561439223588, 7.190864562289789e-05,  0.00015885445463936776,
                         0.00019295849779155105, 0.0001252660877071321,  7.11544998921454e-05,
                         0.0004432793939486146,  3.769283284782432e-05,  0.0002526253811083734,
                         0.00012620443885680288, 1.985905328183435e-05,  2.81801385426661e-05,
                         0.0003524349885992706,  0.0001546697021694854,  0.00010438626486575231,
                         0.000663026818074286,   0.0001500590587966144,  0.00015732485917396843,
                         2.713521826080978e-05,  1.4691643627884332e-05, 0.00012206493556732312,
                         0.0009089525556191802,  0.00010661577834980562, 9.124398275162093e-06,
                         4.969998190063052e-05,  5.217113721300848e-05,  9.936409333022311e-05,
                         0.00014379223284777254, 8.570598583901301e-05,  6.1535116401501e-05,
                         4.199429167783819e-05,  0.00015761781833134592, 0.001501618535257876,
                         2.4530107111786492e-05, 2.280823355249595e-05,  0.00017214070248883218,
                         0.00031316469539888203, 5.968231562292203e-05,  0.00026387471007183194,
                         6.247045530471951e-05,  2.317548933206126e-05,  0.0001769333757692948,
                         7.864696090109646e-05,  0.00046821130672469735, 0.00041301833698526025,
                         0.0001510686706751585,  4.2565559851936996e-05, 9.711079474072903e-05,
                         0.00017850298900157213, 6.903822213644162e-05,  0.00013772475358564407,
                         6.170036067487672e-05,  0.0001500440703239292,  0.00013026059605181217,
                         0.00013515300815925002, 0.00047581346007063985, 1.9914907170459628e-05,
                         0.0003260780358687043,  0.0002930282207671553,  7.9667879617773e-05,
                         6.980553735047579e-05,  0.0001166707297670655,  0.00011259115126449615,
                         0.00011303381324978545, 0.0001989066950045526,  8.898021042114124e-05,
                         0.0004905792302452028,  0.0007432189304381609,  0.00044521293602883816,
                         0.0001913510641315952,  0.0001170646064565517,  1.731002157612238e-05,
                         0.0007720232242718339,  9.625179518479854e-05,  0.0002808795543387532,
                         4.6844310418237e-05,    3.740533793461509e-05,  0.0005274045397527516,
                         0.00010559966904111207, 0.00022991890728008002, 2.2165069822221994e-05,
                         7.735403050901368e-05,  9.57066731643863e-05,   0.0005764247616752982,
                         8.068099850788713e-05,  0.000107927895442117,   0.00031371210934594274,
                         5.03749142808374e-05,   0.0001819927420001477,  0.0004225304292049259,
                         8.180949953384697e-05,  0.00015503961185459048, 0.00010770266817416996,
                         0.000267287454335019,   4.8030138714239e-05,    0.0006655416800640523,
                         0.0001922373048728332,  0.0002080030826618895,  0.00040594322490505874,
                         0.00018624354561325163, 0.0001281022559851408,  7.387215737253428e-05,
                         6.527303048642352e-05,  0.00019082726794295013, 0.00025798456044867635,
                         2.1802386982017197e-05, 0.0002803544921334833,  4.4673903175862506e-05,
                         3.8878970372024924e-05, 5.643867189064622e-05,  5.988581688143313e-05,
                         0.00010815142741193995, 0.00032381736673414707, 0.00013069128908682615,
                         0.00011832305608550087, 0.00013626096188090742, 0.00021772386389784515,
                         2.6973870262736455e-05, 0.001113888924010098,   0.00021063763415440917,
                         4.457081377040595e-05,  9.180054621538147e-05,  1.1883186743943952e-05,
                         4.409704706631601e-05,  1.8699289284995757e-05, 5.806672197650187e-05,
                         0.0010814698180183768,  0.00036806988646276295, 0.0005698917666450143,
                         0.00023745284124743193, 7.18808441888541e-05,   0.00011260071187280118,
                         0.00017481502436567098, 0.0006264873081818223,  4.781736424774863e-05,
                         0.00028814442339353263, 0.0005080040427856147,  0.0006088857189752162,
                         0.0001638085814192891,  0.0004303246969357133,  4.047203765367158e-05,
                         0.00015931717643979937, 0.0005203715991228819,  0.00019552903540898114,
                         0.00018903038289863616, 3.577864481485449e-05,  6.421793659683317e-05,
                         0.0004828061501029879,  0.00025513357832096517, 4.6674143959535286e-05,
                         0.00023013872851151973, 6.468584615504369e-05,  0.0002972084912471473,
                         4.081864608451724e-05,  0.00012394148507155478, 0.00013584499538410455,
                         0.00015631037240382284, 0.00013860705075785518, 4.594949496095069e-05,
                         0.0005482577253133059,  0.0001745454064803198,  5.880332537344657e-05,
                         3.81341997126583e-05,   6.0169451899128035e-05, 4.960901424055919e-05,
                         0.0001572975452290848,  0.00011394100147299469, 0.0001525870757177472,
                         0.0007180342799983919,  0.00014880466915201396, 3.862948142341338e-05,
                         0.0001011974600260146,  0.00023557311214972287, 1.5943134712870233e-05,
                         2.3399841666105203e-05, 9.593375580152497e-05,  0.00012424145825207233,
                         0.00018840374832507223, 5.759921623393893e-05,  0.00016522301302757114,
                         2.011908873100765e-05,  0.00010863873467314988, 5.252579285297543e-05,
                         1.7008498616633005e-05, 6.486255006166175e-05,  0.0003854420792777091,
                         7.561453094240278e-05,  0.0010549785802140832,  0.00011828276910819113,
                         4.24915378971491e-05,   0.0002914096985477954,  3.762840060517192e-05,
                         0.0002477457746863365,  0.00028666152502410114, 9.236185724148527e-05,
                         0.0001815288414945826,  7.093928434187546e-05,  3.709416705532931e-05,
                         7.28237209841609e-05,   2.2864356651552953e-05, 0.00013954313180875033,
                         3.51279741153121e-05,   0.00023959559621289372, 6.222062074812129e-05,
                         0.00010753547394415364, 0.00037070648977532983, 3.139186082989909e-05,
                         0.00018210291455034167, 6.846174073871225e-05,  6.829040648881346e-05,
                         8.572912338422611e-05,  0.0023617795668542385,  0.0011352369328960776,
                         0.0003403531154617667,  3.3944052120205015e-05, 0.000444283417891711,
                         2.294726982654538e-05,  0.000380594574380666,   4.9140497139887884e-05,
                         4.2460058466531336e-05, 0.00012077242718078196, 0.00018034975801128894,
                         4.295352118788287e-05,  0.00020140752894803882, 3.311114778625779e-05,
                         0.00012875063112005591, 3.3104061003541574e-05, 4.013380021206103e-05,
                         2.8543086955323815e-05, 9.062557364813983e-05,  0.00011620239820331335,
                         0.0003364370495546609,  4.357949728728272e-05,  0.00014167230983730406,
                         0.00026003457605838776, 1.5814124708413146e-05, 9.746642899699509e-05,
                         0.00011545964662218466, 0.00018526462372392416, 0.00037749711191281676,
                         0.0003333878412377089,  0.0001566411228850484,  0.0002977294207084924,
                         0.00038576891529373825, 0.0007520199287682772,  4.920849823975004e-05,
                         0.00018407488823868334, 0.00018112944962922484, 8.045330469030887e-05,
                         1.737858292472083e-05,  0.0011009435402229428,  1.4472660041064955e-05,
                         0.000332419469486922,   5.5059896112652496e-05, 9.803663124330342e-05,
                         6.384458538377658e-05,  7.751035445835441e-05,  0.0003189261187799275,
                         2.6156491003348492e-05, 2.8752623620675877e-05, 0.0003734382917173207,
                         5.414553015725687e-05,  1.7037298675859347e-05, 3.595564339775592e-05,
                         7.595780334668234e-05,  0.00011109604383818805, 0.00037215687916614115,
                         0.0014351916033774614,  0.0005010929889976978,  0.00026278020231984556,
                         9.100374882109463e-05,  0.0001663308503339067,  7.175433711381629e-05,
                         0.0006884920876473188,  0.00015885569155216217, 7.745541370240971e-05,
                         0.00023989100009202957, 0.00019794265972450376, 0.00026567833265289664,
                         0.00012376224913168699, 5.259428871795535e-05,  0.00021847222524229437,
                         8.646513742860407e-05,  4.719060234492645e-05,  2.1244310119072907e-05,
                         0.00023851863807067275, 0.00010232441127300262, 0.00030056352261453867,
                         5.088794569019228e-05,  0.00011722966883098707, 0.00034765820601023734,
                         5.1997878472320735e-05, 2.0474855773500167e-05, 2.6946230718749575e-05,
                         0.0003450191579759121,  0.00018332961190026253, 0.0001149064555647783,
                         9.589620458427817e-05,  0.00013873343414161354, 5.324613084667362e-05,
                         0.0001088585559045896,  0.0008074631332419813,  4.9495338316773996e-05,
                         0.00023560068802908063, 0.00018922766321338713, 3.181437205057591e-05,
                         4.316671038395725e-05,  0.00011171837104484439, 0.00015753462503198534,
                         0.00018351784092374146, 0.00013021435006521642, 0.00042768745333887637,
                         7.462925714207813e-05,  0.00012804339348804206, 0.00013237613893579692,
                         0.0001591947948327288,  0.00013299430429469794, 0.00019827995856758207,
                         0.00013611154281534255, 0.00019157963106408715, 0.0005334524903446436,
                         0.0003954379935748875,  0.00011766677926061675, 1.1122801879537292e-05,
                         9.788023817236535e-06,  8.138679550029337e-05,  0.00019320497813168913,
                         0.0001550878514535725,  0.00014252706023398787, 0.00010759518045233563,
                         0.00012834683002438396, 4.88197329104878e-05,   0.00038639723788946867,
                         0.0005906240548938513,  0.0001646239252295345,  8.264018833870068e-05,
                         2.2426740542869084e-05, 0.00016515731113031507, 0.00045976031105965376,
                         0.00010889074474107474, 0.00015862550935707986, 0.000174326152773574,
                         3.664120959001593e-05,  0.00019478292961139232, 6.325435970211402e-05,
                         2.477250927768182e-05,  7.769633521093056e-05,  9.460438013775274e-05,
                         0.0003399173729121685,  0.00012508317013271153, 3.7673791666748e-05,
                         0.0002770417195279151,  0.00018493369861971587, 6.838761328253895e-05,
                         0.0002466413425281644,  0.00011124849697807804, 8.184996113413945e-05,
                         0.001280209282413125,   0.00010264307638863102, 8.603776950621977e-05,
                         9.502076864009723e-05,  3.921218012692407e-05,  6.773057248210534e-05,
                         0.0005949506303295493,  2.8744687369908206e-05, 0.0009507859940640628,
                         0.00010622251284075901, 0.00011447184078861028, 0.00012745041749440134,
                         0.0005413648905232549,  0.00021209717669989914, 0.00016746361507102847,
                         3.091693724854849e-05,  6.080615639803e-05,     0.00013972401211503893,
                         0.00014625501353293657, 0.00013060946366749704, 2.8184169423184358e-05,
                         0.00021252220903988928, 0.000562939909286797,   0.00042435101931914687,
                         9.347523882752284e-05,  0.00010186129657085985, 0.00014573626685887575,
                         8.750976849114522e-05,  0.00044031706056557596, 0.0005072563071735203,
                         0.00016252222121693194, 2.642742583702784e-05,  0.00020893094188068062,
                         8.223525946959853e-05,  4.875606100540608e-05,  0.0011022539110854268,
                         0.0003225487598683685,  0.0003964175411965698,  0.0003181556530762464,
                         0.00021776060748379678, 0.0001343592448392883,  9.833040530793369e-05,
                         9.831081115407869e-05,  0.00019714358495548368, 7.437606836901978e-05,
                         4.963454557582736e-05,  0.00013880110054742545, 0.00010714316158555448,
                         0.00014927808661013842, 0.00013219589891377836, 0.00017132755601778626,
                         0.00015104103658813983, 7.417599408654496e-05,  0.0009324923739768565,
                         0.00013109666178934276, 0.0003037868009414524,  0.00018952257232740521,
                         0.00017838772328104824, 0.00017524413124192506, 0.0010342708555981517,
                         0.0001530025474494323,  0.00011236781574552879, 0.0002183923206757754,
                         0.0004714425012934953,  0.00010931082215392962, 0.000778728339355439,
                         0.00015190568228717893, 8.450644963886589e-05,  0.0004695610550697893,
                         0.0002332372241653502,  0.00015137341688387096, 0.00033155884011648595,
                         3.8648639019811526e-05, 7.989403820829466e-05,  5.976711690891534e-05,
                         5.062093259766698e-05,  6.487430800916627e-05,  0.00013468349061440676,
                         6.15786193520762e-05,   0.00017666786152403802, 0.00012559369497466832,
                         0.0004127885913476348,  0.00017206576012540609, 0.0001168379167211242,
                         8.269135287264362e-05,  0.00016040925402194262, 0.0003306448634248227,
                         0.0003999473701696843,  0.00011116069072159007, 0.000299162813462317,
                         8.325606177095324e-05,  0.00023967477318365127, 0.0002178536233259365,
                         0.00016400009917560965, 0.0006509397644549608,  0.00010284516611136496,
                         4.5965211029397324e-05, 8.82028034538962e-05,   3.965932774008252e-05,
                         3.662033486762084e-05,  0.00021079383441247046, 9.46761283557862e-05,
                         4.887994873570278e-05,  1.939526373462286e-05,  6.702935934299603e-05,
                         0.00015998411981854588, 1.4840397852822207e-05, 2.468702950864099e-05,
                         7.981716771610081e-05,  0.00022422100300900638, 7.896577153587714e-05,
                         7.670796912861988e-05,  0.00022602881654165685, 8.33443264127709e-05,
                         0.00014530740736518055, 4.88675941596739e-05,   4.8040743422461674e-05,
                         0.0009127347148023546,  0.000402635894715786,   0.0003929564554709941,
                         5.363992750062607e-05,  0.00015619548503309488, 3.890765583491884e-05,
                         0.0007284318562597036,  0.00022880551114212722, 9.35172793106176e-05,
                         0.0012846397003158927,  4.073070522281341e-05,  1.326868004980497e-05,
                         5.76237725908868e-05,   0.0004230248450767249,  0.0001756733254296705,
                         7.147160067688674e-05,  5.8933484979206696e-05, 4.381355029181577e-05,
                         9.51657784753479e-05,   4.6801575081190094e-05, 9.972327097784728e-05,
                         0.00017407782434020191, 0.0003239126526750624,  0.00015926992637105286,
                         9.697761561255902e-05,  0.0003942824841942638,  0.00020644464530050755,
                         0.0002428505104035139,  8.663677726872265e-05,  0.00012856940156780183,
                         7.59983086027205e-05,   0.00032508771982975304, 0.00011777826148318127,
                         0.0001250443747267127,  4.393398921820335e-05,  6.858532287878916e-05,
                         3.662157541839406e-05,  0.00015834366786293685, 0.00017737312009558082,
                         9.80736076598987e-05,   4.346506466390565e-05,  0.00028740704874508083,
                         0.00018200220074504614, 0.00013285494060255587, 0.00012278689246159047,
                         9.059463627636433e-05,  6.774020584998652e-05,  0.0001748832146404311,
                         0.00013254971418064088, 0.00012535153655335307, 9.035399125423282e-05,
                         0.0005454723141156137,  4.9749345635063946e-05, 8.450052700936794e-05,
                         2.043595668510534e-05,  9.429290366824716e-05,  6.742200639564544e-05,
                         9.871196380117908e-05,  3.430158903938718e-05,  0.000468904385343194,
                         8.038297528401017e-05,  0.00012832482752855867, 0.00030439317924901843,
                         6.213445885805413e-05,  0.0002923696010839194,  8.891867037164047e-05,
                         0.0002435069327475503,  6.29726710030809e-05,   9.5872521342244e-05,
                         8.161942969309166e-05,  0.00011977500980719924, 0.0009125336655415595,
                         0.0002011920587392524,  0.0008074359502643347,  0.0006924457848072052,
                         8.687444642418995e-05,  0.00029718695441260934, 0.00010474630835233256,
                         7.962173549458385e-05,  0.00015691867156419903, 9.966684592654929e-05,
                         0.0005664451164193451,  2.7236550522502512e-05, 0.00048738642362877727,
                         0.0008525094599463046,  5.0857357564382255e-05, 4.203209391562268e-05,
                         0.0002700215554796159,  5.953541767667048e-05,  7.75999651523307e-05,
                         0.00011352085857652128, 9.116536239162087e-05,  0.00012107379006920382,
                         3.810471753240563e-05,  4.551122401608154e-05,  5.326667451299727e-05,
                         9.037894778884947e-05,  6.487489736173302e-05,  4.36918307968881e-05,
                         0.00029065270791761577, 0.00010018534521805122, 7.95703090261668e-05,
                         7.85843949415721e-05,   0.0004152680339757353,  0.0018965150229632854,
                         0.0001220985723193735,  6.214346649358049e-05,  3.357251625857316e-05,
                         0.00012541189789772034, 9.746596333570778e-05,  0.00032468693098053336,
                         0.0007483326480723917,  0.00015189011173788458, 7.698403351241723e-05,
                         0.00037020660238340497, 0.0010757340351119637,  0.0004979046643711627,
                         9.574159776093438e-05,  8.788176637608558e-05,  6.350075273076072e-05,
                         0.0007387229125015438,  0.0001708695781417191,  7.155616185627878e-05,
                         0.0003234584874007851,  0.00020423159003257751, 0.00010895852756220847,
                         0.00027092243544757366, 1.1215176527912263e-05, 0.00010175905481446534,
                         9.197453618980944e-05,  9.137304004980251e-05,  0.00030576749122701585,
                         0.0003086254873778671,  0.0008101508137769997,  0.00014327299140859395,
                         0.0002398896322119981,  0.0002728633116930723,  1.3294278687681071e-05,
                         0.00024795698118396103, 0.00012035827967338264, 0.00015592567797284573,
                         0.00013207750453148037, 0.00045279867481440306, 0.00013242366549093276,
                         0.0005288170068524778,  0.00015421821444761008, 9.636762115405872e-05,
                         4.219650509185158e-05,  0.0004972798051312566,  0.00025572048616595566,
                         6.439173739636317e-05,  4.814308795175748e-06,  0.00012081058230251074,
                         8.521769632352516e-05,  0.00012677487393375486, 6.460629083449021e-05,
                         0.0001853939174907282,  0.00019731932843569666, 0.00012051217345288023,
                         0.0009332839981652796,  0.0002541230642236769,  0.001248691463842988,
                         2.429229971312452e-05,  0.00021751539316028357, 3.53120340150781e-05,
                         0.00037944960058666766, 0.0001893102889880538,  8.062131382757798e-05,
                         4.141789395362139e-05,  0.0003767870948649943,  1.358789268124383e-05,
                         0.00023865302500780672, 0.00021608859242405742, 0.00035065534757450223,
                         0.00019346449698787183, 0.00013611797476187348, 0.00016690068878233433,
                         2.3259961380972527e-05, 8.979118865681812e-05,  2.0819275960093364e-05,
                         0.00012423814041540027, 0.0001285633334191516,  9.882942686090246e-05,
                         0.00040421882295049727, 0.0003865523904096335,  7.560573430964723e-05,
                         0.00015556163270957768, 6.99083975632675e-05,   0.00010838798334589228,
                         9.171373676508665e-05,  9.853326628217474e-05,  0.00020116043742746115,
                         0.0004202903073746711,  7.182307308539748e-05,  0.00020190395298413932,
                         0.00023384565429296345, 5.669492020388134e-05,  0.00018979294691234827,
                         0.0010612322948873043,  0.00011265597277088091, 0.00033349558361805975,
                         2.374567702645436e-05,  0.0002732102875597775,  0.00010805281635839492,
                         0.0002877278020605445,  0.00111890968400985,    7.348333747358993e-05,
                         0.0002711308770813048,  1.3383682926360052e-05, 0.00021655271120835096,
                         0.0002818830544129014,  3.643261516117491e-05,  0.0009109652019105852,
                         0.00022014969727024436, 9.947526268661022e-05,  0.00018676728359423578,
                         2.6897101633949205e-05, 5.467058508656919e-05,  9.407957986695692e-05,
                         0.00012081496970495209, 0.00025852632825262845, 0.0001600077230250463,
                         8.513679495081306e-05,  0.0007796057616360486,  0.000191116938367486,
                         6.699836376355961e-05,  0.0014805941609665751,  0.00011316610471112654,
                         0.00025210747844539583, 9.729285375215113e-05,  0.00015336641808971763,
                         0.0002316069876542315,  0.00011202835594303906, 7.754639955237508e-05,
                         0.00013658934039995074, 4.8397749196738005e-05, 0.00035813829163089395,
                         0.00013150744780432433, 3.1374875106848776e-05, 0.0022018561139702797,
                         0.0002633944386616349,  0.00027658435283228755, 5.174717080080882e-05,
                         0.00031591294100508094, 0.0006419907440431416,  0.00044318134314380586,
                         3.0605184292653576e-05, 5.7552402722649276e-05, 4.329581497586332e-05,
                         0.00014545119483955204, 0.00010178987577091902, 2.5973909941967577e-05,
                         0.00011373704910511151, 0.0004465636739041656,  0.0005738934851251543,
                         3.479374572634697e-05,  6.599000334972516e-05,  0.00017248383664991707,
                         0.0012805428123101592,  0.0001697942498140037,  6.156755262054503e-05,
                         0.0002653173578437418,  4.251318023307249e-05,  8.940495899878442e-05,
                         4.5272008719621226e-05, 2.289999247295782e-05,  5.459693420561962e-05,
                         0.0001399959292029962,  4.1979212255682796e-05, 5.9825056268891785e-06,
                         0.00016876649169716984, 0.00022111000725999475, 0.00010077890328830108,
                         0.00010402225598227233, 3.705921335495077e-05,  2.7981899620499462e-05,
                         0.0001274178794119507,  0.000611155410297215,   0.0005516264936886728,
                         0.0003363762516528368,  0.00040613036253489554, 0.00012253176828380674,
                         3.951079270336777e-05,  0.0001391753467032686,  3.1030765967443585e-05,
                         0.00024143910559359938, 0.00016234802023973316, 0.00037827875348739326,
                         0.0006087071378715336,  0.0004463918739929795,  0.0004960173973813653,
                         0.00013299040438141674, 5.2586841775337234e-05, 0.00013509228301700205,
                         0.0001404394570272416,  0.0004389164096210152,  0.00011450509919086471,
                         6.037171988282353e-05,  8.413020259467885e-05,  0.0005483124987222254,
                         0.00035119615495204926, 4.52965177828446e-05,   0.00028168613789603114,
                         3.510618626023643e-05,  5.651970423059538e-05,  5.852166941622272e-05,
                         6.289517841651104e-06,  0.00030065482133068144, 0.0002924331638496369,
                         5.1956863899249583e-05, 0.00016533432062715292, 0.0004791320243384689,
                         8.497411909047514e-05,  2.9167667889851145e-05, 3.478750659269281e-05,
                         6.045557529432699e-05,  8.940841507865116e-05,  2.482410127413459e-05,
                         0.00020333875727374107, 0.00011506556620588526, 0.0004955083131790161,
                         0.00033340422669425607, 0.00014328047109302133, 2.478294300090056e-05,
                         0.00043639057548716664, 1.5982785043888725e-05, 0.00035572622437030077,
                         5.9527214034460485e-05, 0.0001617999078007415,  7.046633982099593e-05,
                         0.00045587349450215697, 0.0002508334000594914,  9.254962060367689e-05,
                         6.03883272560779e-05,   3.7565834645647556e-05, 0.0010212842607870698,
                         0.00014635130355600268, 0.0001313114771619439,  0.00012995504948776215,
                         0.0002011679607676342,  0.00015332791372202337, 0.00010797731374623254,
                         4.530472870101221e-05,  0.00018774786440189928, 3.486090645310469e-05,
                         3.227369234082289e-05,  0.0006375828525051475,  0.0009987229714170098,
                         0.0003035009140148759,  0.00022980166249908507, 8.181609155144542e-05,
                         0.00010680873674573377, 0.00011864626867463812, 7.641856791451573e-05,
                         3.4949436667375267e-05, 2.9040947993053123e-05, 5.296686867950484e-05,
                         7.472205470548943e-05,  9.790871263248846e-05,  7.45548095437698e-05,
                         0.00060952955391258,    1.7565653251949698e-05, 0.00046291734906844795,
                         5.675669672200456e-05,  6.779420073144138e-05,  9.075119305634871e-05,
                         5.822065213578753e-05,  9.921887976815924e-05,  0.00017330082482658327,
                         0.00013782430323772132, 0.00013353483518585563, 0.00011434157931944355,
                         0.0003348048485349864,  0.00012816012895200402, 0.00011559428094187751,
                         0.0002710882981773466,  7.693019870202988e-05,  5.836012860527262e-05,
                         0.0006070468225516379,  7.488016854040325e-05,  0.001852008979767561,
                         2.7007741664419882e-05, 5.515764496522024e-05,  0.00010789687803480774,
                         4.617984814103693e-05,  0.00011003785039065406, 7.73250067140907e-05,
                         4.14055357396137e-05,   1.2358126696199179e-05, 3.16738587571308e-05,
                         3.804877997026779e-05,  0.00034552408033050597, 3.085275602643378e-05,
                         0.0001377114822389558,  3.788619869737886e-05,  6.119041063357145e-05,
                         0.00023054488701745868, 4.0189224819187075e-05, 0.0013503119116649032,
                         0.000144677484058775,   9.775593935046345e-05,  2.6450506993569434e-05,
                         2.115545248670969e-05,  0.00010426602239022031, 5.534932279260829e-05,
                         0.0001132215911638923,  1.5253308447427116e-05, 0.0008886965224519372,
                         6.10595743637532e-05,   7.862217898946255e-05,  0.0004363092011772096,
                         0.00012292005703784525, 5.9192872868152335e-05, 0.0003491935203783214,
                         0.00011458549124654382, 0.0005158179556019604,  0.0001681967405602336,
                         0.00015892555529717356, 0.00023763076751492918, 0.00024058549024630338,
                         6.315366772469133e-05,  5.8320543757872656e-05, 0.0001826909719966352,
                         5.818099089083262e-05,  0.0008595928666181862,  9.808245522435755e-05,
                         0.00019563516252674162, 0.0004713562084361911,  0.00020347863028291613,
                         0.00013220014807302505, 0.00024272155133076012, 0.00011764911323552951,
                         6.876415136503056e-05,  0.0005287153762765229,  7.408983947243541e-05,
                         8.22931615402922e-05,   0.00017969509644899517, 6.595279410248622e-05,
                         4.074059324921109e-05,  0.00011744895164156333, 2.8076701710233465e-05,
                         0.00035538154770620167, 0.00018522677419241518, 0.00010062198271043599,
                         0.00012009423517156392, 8.415250340476632e-05,  5.541978316614404e-05,
                         0.00019621288811322302, 0.0004448060935828835};
  std::vector<T> uniform_samples_h{0.9999998807907104};

  thrust::device_vector<T> probs_d(probs_h);
  thrust::device_vector<IdType> sampled_ids_d(batch_size);
  thrust::device_vector<T> uniform_samples_d(uniform_samples_h);

  auto status = sampling::SamplingFromProb<T>(thrust::raw_pointer_cast(probs_d.data()),
                                              thrust::raw_pointer_cast(uniform_samples_d.data()),
                                              thrust::raw_pointer_cast(sampled_ids_d.data()),
                                              batch_size, vocab_size, /*deterministic=*/true);
  EXPECT_EQ(status, cudaSuccess) << "SamplingFromProb kernel launch failed, error message: "
                                 << cudaGetErrorString(status);

  thrust::host_vector<IdType> sampled_ids_h(sampled_ids_d);

  for (uint32_t i = 0; i < batch_size; ++i) {
    EXPECT_GE(sampled_ids_h[i], 0) << "sampled_ids_h[" << i << "] < 0";
    EXPECT_LT(sampled_ids_h[i], vocab_size) << "sampled_ids_h[" << i << "] >= vocab_size";
  }
}

template <typename T, typename IdType>
void TestTopKSamplingFromProb() {
  for (size_t batch_size : {1, 7, 333}) {
    for (size_t k : {1, 2, 3, 4, 5, 6, 7, 8, 9, 10}) {
      for (size_t d : {24, 97, 1002, 4096, 32000, 128000}) {
        _TestTopKSamplingFromProb<T, IdType>(batch_size, k, d);
      }
    }
  }
}

template <typename T, typename IdType>
void TestTopPSamplingFromProb() {
  for (size_t batch_size : {1, 7, 333}) {
    for (size_t k : {1, 2, 3, 4, 5, 6, 7, 8, 9, 10}) {
      for (size_t d : {24, 4096, 1002, 32000, 128000}) {
        _TestTopPSamplingFromProb<T, IdType>(batch_size, k, d);
      }
    }
  }
}

TEST(FlashInferCorrectnessTests, TestTopKSamplingFromProbFP32) {
  TestTopKSamplingFromProb<float, int32_t>();
}

TEST(FlashInferCorrectnessTests, TestTopPSamplingFromProbFP32) {
  TestTopPSamplingFromProb<float, int32_t>();
}

TEST(FlashInferCorrectnessTests, TestSamplingFromProbOneHotFP32) {
  TestSamplingFromProbOneHot<float, int32_t>();
}

TEST(FlashInferCorrectnessTests, TestSamplingFromProbExtremeCaseFP32) {
  TestSamplingFromProbExtremeCase<float, int32_t>();
}
