package main

import (
	_ "embed"
	"fmt"
	"io"
	"os"

	"github.com/chzyer/readline"
	"github.com/cloudson/gitql/lexical"
	"github.com/cloudson/gitql/parser"
	"github.com/cloudson/gitql/runtime"
	"github.com/cloudson/gitql/semantical"
	"github.com/urfave/cli/v2"
)

//go:embed version.txt
var version string

func main() {
	app := &cli.App{
		Name:        "gitql",
		Usage:       "A git query language",
		Version:     version,
		HideVersion: true,
		Flags: []cli.Flag{
			&cli.BoolFlag{
				Name:    "interactive",
				Aliases: []string{"i"},
				Usage:   "Enter to interactive mode",
			},
			&cli.StringFlag{
				Name:    "path",
				Aliases: []string{"p"},
				Value:   ".",
				Usage:   `The (optional) path to run gitql`,
			},
			&cli.StringFlag{
				Name:    "format",
				Aliases: []string{"f"},
				Value:   "table",
				Usage:   "The output type format {table|json}",
			},
			// for backward compatibility
			&cli.BoolFlag{
				Name:    "version",
				Aliases: []string{"v"},
				Hidden:  true,
			},
			&cli.StringFlag{
				Name:   "type",
				Hidden: true,
			},
			&cli.BoolFlag{
				Name:    "show-tables",
				Aliases: []string{"s"},
				Hidden:  true,
			},
		},
		Commands: []*cli.Command{
			{
				Name:    "show-tables",
				Aliases: []string{"s"},
				Usage:   "Show all tables",
				Action:  showTablesCmd,
			},
			{
				Name:    "version",
				Aliases: []string{"v"},
				Usage:   "The version of gitql",
				Action: func(c *cli.Context) error {
					fmt.Printf("Gitql %s\n", version)
					return nil
				},
			},
		},
		Action: func(c *cli.Context) error {
			path, format, interactive := c.String("path"), c.String("format"), c.Bool("interactive")

			// for backward compatibility
			if c.Bool("version") {
				fmt.Printf("Gitql %s\n", version)
				return nil
			}

			if c.Bool("show-tables") {
				return showTablesCmd(c)
			}

			if typ := c.String("type"); typ != "" {
				format = typ
			}
			// ============================

			if c.NArg() == 0 && !interactive {
				return cli.ShowAppHelp(c)
			}

			if interactive {
				return runPrompt(path, format)
			}

			return runQuery(c.Args().First(), path, format)
		},
	}

	if err := app.Run(os.Args); err != nil {
		fmt.Fprintf(os.Stderr, "Error: %s\n", err)
		os.Exit(1)
	}
}

func showTablesCmd(c *cli.Context) error {
	prog := &parser.NodeProgram{
		Child: &parser.NodeShow{
			Tables: true,
		},
	}
	return runtime.RunShow(prog)
}

func runPrompt(folder, typeFormat string) error {
	term, err := readline.NewEx(&readline.Config{
		Prompt:       "gitql> ",
		AutoComplete: readline.SegmentFunc(suggestQuery),
	})
	if err != nil {
		return err
	}
	defer term.Close()

	for {
		query, err := term.Readline()
		if err != nil {
			if err == io.EOF {
				break // Ctrl^D
			}
			return err
		}

		if query == "" {
			continue
		}

		if query == "exit" || query == "quit" {
			break
		}

		if err := runQuery(query, folder, typeFormat); err != nil {
			fmt.Println("Error: " + err.Error())
			continue
		}
	}

	return nil
}

func runQuery(query, folder, typeFormat string) error {
	parser.New(query)
	ast, err := parser.AST()
	if err != nil {
		return err
	}

	ast.Path = &folder
	switch lexical.Command {
	case lexical.T_SELECT:
		if err := semantical.Analysis(ast); err != nil {
			return err
		}
		err = runtime.RunSelect(ast, &typeFormat)
		break
	case lexical.T_SHOW:
		err = runtime.RunShow(ast)
		break
	case lexical.T_USE:
		err = runtime.RunUse(ast)
		break
	}

	return err
}
