﻿using System;
using System.ComponentModel;
using System.Diagnostics.CodeAnalysis;
using System.Drawing.Design;
using System.Globalization;
using System.Web;
using System.Web.Resources;
using System.Web.UI;
using System.Web.UI.WebControls;

namespace BugNET.UserInterfaceLayer.WebControls
{
    public class BugNetPagerField : DataPagerField
    {
        private readonly string _rowXofY = HttpContext.GetGlobalResourceObject("SharedResources", "RowXofY").ToString();
        private readonly string _ShowRows = HttpContext.GetGlobalResourceObject("SharedResources", "ShowRows").ToString();
        private readonly string _PerPage = HttpContext.GetGlobalResourceObject("SharedResources", "PerPage").ToString();

        private int _startRowIndex;
        private int _maximumRows;
        private int _totalRowCount;

        public BugNetPagerField()
        {
        }

        [
        DefaultValue(5),
        Category("Appearance"),
        ]
        public int ButtonCount
        {
            get
            {
                object o = ViewState["ButtonCount"];
                if (o != null)
                {
                    return (int)o;
                }
                return 5;
            }
            set
            {
                if (value < 1)
                {
                    throw new ArgumentOutOfRangeException("value");
                }
                if (value != ButtonCount)
                {
                    ViewState["ButtonCount"] = value;
                    OnFieldChanged();
                }
            }
        }

        /// <devdoc>
        /// <para>Indicates the button type for the field.</para>
        /// </devdoc>
        [
        Category("Appearance"),
        DefaultValue(ButtonType.Link),
        ]
        public ButtonType ButtonType
        {
            get
            {
                object o = ViewState["ButtonType"];
                if (o != null)
                    return (ButtonType)o;
                return ButtonType.Link;
            }
            set
            {
                if (value < ButtonType.Button || value > ButtonType.Link)
                {
                    throw new ArgumentOutOfRangeException("value");
                }
                if (value != ButtonType)
                {
                    ViewState["ButtonType"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        DefaultValue(""),
        CssClassPropertyAttribute
        ]
        public string CurrentPageLabelCssClass
        {
            get
            {
                object o = ViewState["CurrentPageLabelCssClass"];
                if (o != null)
                {
                    return (string)o;
                }
                return String.Empty;
            }
            set
            {
                if (value != CurrentPageLabelCssClass)
                {
                    ViewState["CurrentPageLabelCssClass"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        DefaultValue(""),
        SuppressMessage("Microsoft.Design", "CA1056:UriPropertiesShouldNotBeStrings",
                        Justification = "Required by ASP.NET parser."),
        UrlProperty()
        ]
        public string NextPageImageUrl
        {
            get
            {
                object o = ViewState["NextPageImageUrl"];
                if (o != null)
                {
                    return (string)o;
                }
                return String.Empty;
            }
            set
            {
                if (value != NextPageImageUrl)
                {
                    ViewState["NextPageImageUrl"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        Localizable(true),
        ]
        public string NextPageText
        {
            get
            {
                object o = ViewState["NextPageText"];
                if (o != null)
                {
                    return (string)o;
                }
                return "Next";
            }
            set
            {
                if (value != NextPageText)
                {
                    ViewState["NextPageText"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        DefaultValue(""),
        CssClassPropertyAttribute
        ]
        public string NextPreviousButtonCssClass
        {
            get
            {
                object o = ViewState["NextPreviousButtonCssClass"];
                if (o != null)
                {
                    return (string)o;
                }
                return String.Empty;
            }
            set
            {
                if (value != NextPreviousButtonCssClass)
                {
                    ViewState["NextPreviousButtonCssClass"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        DefaultValue(""),
        CssClassPropertyAttribute
        ]
        public string NumericButtonCssClass
        {
            get
            {
                object o = ViewState["NumericButtonCssClass"];
                if (o != null)
                {
                    return (string)o;
                }
                return String.Empty;
            }
            set
            {
                if (value != NumericButtonCssClass)
                {
                    ViewState["NumericButtonCssClass"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        DefaultValue(""),
        SuppressMessage("Microsoft.Design", "CA1056:UriPropertiesShouldNotBeStrings",
                        Justification = "Required by ASP.NET parser."),
        UrlProperty()
        ]
        public string PreviousPageImageUrl
        {
            get
            {
                object o = ViewState["PreviousPageImageUrl"];
                if (o != null)
                {
                    return (string)o;
                }
                return String.Empty;
            }
            set
            {
                if (value != PreviousPageImageUrl)
                {
                    ViewState["PreviousPageImageUrl"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        Category("Appearance"),
        Localizable(true),
        ]
        public string PreviousPageText
        {
            get
            {
                object o = ViewState["PreviousPageText"];
                if (o != null)
                {
                    return (string)o;
                }
                return "Previous";
            }
            set
            {
                if (value != PreviousPageText)
                {
                    ViewState["PreviousPageText"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        DefaultValue(true),
        Category("Behavior"),
        ]
        public bool RenderNonBreakingSpacesBetweenControls
        {
            get
            {
                object o = ViewState["RenderNonBreakingSpacesBetweenControls"];
                if (o != null)
                {
                    return (bool)o;
                }
                return true;
            }
            set
            {
                if (value != RenderNonBreakingSpacesBetweenControls)
                {
                    ViewState["RenderNonBreakingSpacesBetweenControls"] = value;
                    OnFieldChanged();
                }
            }
        }

        [
        SuppressMessage("Microsoft.Usage", "CA2204:LiteralsShouldBeSpelledCorrectly", MessageId = "nbsp",
                        Justification = "Literal is HTML escape sequence."),
        SuppressMessage("Microsoft.Globalization", "CA1303:DoNotPassLiteralsAsLocalizedParameters",
                        MessageId = "System.Web.UI.LiteralControl.#ctor(System.String)",
                        Justification = "Literal is HTML escape sequence.")
        ]
        private void AddNonBreakingSpace(DataPagerFieldItem container)
        {
            if (RenderNonBreakingSpacesBetweenControls)
            {
                container.Controls.Add(new LiteralControl(" "));
            }
        }

        protected override void CopyProperties(DataPagerField newField)
        {
            ((NumericPagerField)newField).ButtonCount = ButtonCount;
            ((NumericPagerField)newField).ButtonType = ButtonType;
            ((NumericPagerField)newField).CurrentPageLabelCssClass = CurrentPageLabelCssClass;
            ((NumericPagerField)newField).NextPageImageUrl = NextPageImageUrl;
            ((NumericPagerField)newField).NextPageText = NextPageText;
            ((NumericPagerField)newField).NextPreviousButtonCssClass = NextPreviousButtonCssClass;
            ((NumericPagerField)newField).NumericButtonCssClass = NumericButtonCssClass;
            ((NumericPagerField)newField).PreviousPageImageUrl = PreviousPageImageUrl;
            ((NumericPagerField)newField).PreviousPageText = PreviousPageText;

            base.CopyProperties(newField);
        }

        protected override DataPagerField CreateField()
        {
            return new NumericPagerField();
        }

        [SuppressMessage("Microsoft.Security", "CA2109:ReviewVisibleEventHandlers", MessageId = "0#")]
        public override void HandleEvent(CommandEventArgs e)
        {
            if (String.IsNullOrEmpty(DataPager.QueryStringField))
            {
                int newStartRowIndex = -1;
                int currentPageIndex = _startRowIndex / DataPager.PageSize;
                int firstButtonIndex = (_startRowIndex / (ButtonCount * DataPager.PageSize)) * ButtonCount;
                int lastButtonIndex = firstButtonIndex + ButtonCount - 1;
                int lastRecordIndex = ((lastButtonIndex + 1) * DataPager.PageSize) - 1;

                if (String.Equals(e.CommandName, DataControlCommands.PreviousPageCommandArgument))
                {
                    newStartRowIndex = (firstButtonIndex - 1) * DataPager.PageSize;
                    if (newStartRowIndex < 0)
                    {
                        newStartRowIndex = 0;
                    }
                }
                else if (String.Equals(e.CommandName, DataControlCommands.NextPageCommandArgument))
                {
                    newStartRowIndex = lastRecordIndex + 1;
                    if (newStartRowIndex > _totalRowCount)
                    {
                        newStartRowIndex = _totalRowCount - DataPager.PageSize;
                    }
                }
                else if (string.Equals(e.CommandName, "UpdatePageSize"))
                {
                    DataPager.PageSize = Int32.Parse(e.CommandArgument.ToString());
                    DataPager.SetPageProperties(_startRowIndex, DataPager.PageSize, true);
                }
                else
                {
                    int pageIndex = Convert.ToInt32(e.CommandName, CultureInfo.InvariantCulture);
                    newStartRowIndex = pageIndex * DataPager.PageSize;
                }

                if (newStartRowIndex != -1)
                {
                    DataPager.SetPageProperties(newStartRowIndex, DataPager.PageSize, true);
                }
            }
        }

        private Control CreateNumericButton(string buttonText, string commandArgument, string commandName)
        {
            IButtonControl button;

            switch (ButtonType)
            {
                case ButtonType.Button:
                    button = new Button();
                    break;
                case ButtonType.Link:
                default:
                    button = new LinkButton();
                    break;
            }

            button.Text = buttonText;
            button.CommandName = commandName;
            button.CommandArgument = commandArgument;
            button.CausesValidation = false;

            WebControl webControl = button as WebControl;
            if (webControl != null && !String.IsNullOrEmpty(NumericButtonCssClass))
            {
                webControl.CssClass = NumericButtonCssClass;
            }

            return button as Control;
        }

        private HyperLink CreateNumericLink(int pageIndex)
        {
            int pageNumber = pageIndex + 1;
            HyperLink link = new HyperLink();
            link.Text = pageNumber.ToString(CultureInfo.InvariantCulture);
            link.NavigateUrl = GetQueryStringNavigateUrl(pageNumber);

            if (!String.IsNullOrEmpty(NumericButtonCssClass))
            {
                link.CssClass = NumericButtonCssClass;
            }

            return link;
        }

        private Control CreateNextPrevButton(string buttonText, string commandName, string commandArgument, string imageUrl)
        {
            IButtonControl button;

            switch (ButtonType)
            {
                case ButtonType.Link:
                    button = new LinkButton();
                    break;
                case ButtonType.Button:
                    button = new Button();
                    break;
                case ButtonType.Image:
                default:
                    button = new ImageButton();
                    ((ImageButton)button).ImageUrl = imageUrl;
                    ((ImageButton)button).AlternateText = HttpUtility.HtmlDecode(buttonText);
                    break;
            }
            button.Text = buttonText;
            button.CommandName = commandName;
            button.CommandArgument = commandArgument;

            WebControl webControl = button as WebControl;
            if (webControl != null && !String.IsNullOrEmpty(NextPreviousButtonCssClass))
            {
                webControl.CssClass = NextPreviousButtonCssClass;
            }

            return button as Control;
        }

        private HyperLink CreateNextPrevLink(string buttonText, int pageIndex, string imageUrl)
        {
            int pageNumber = pageIndex + 1;
            HyperLink link = new HyperLink();
            link.Text = buttonText;
            link.NavigateUrl = GetQueryStringNavigateUrl(pageNumber);
            link.ImageUrl = imageUrl;
            if (!String.IsNullOrEmpty(NextPreviousButtonCssClass))
            {
                link.CssClass = NextPreviousButtonCssClass;
            }
            return link;
        }

        /// <summary>
        /// Creates the label record control.
        /// </summary>
        /// <param name="container">The container.</param>
        private void CreateLabelRecordControl(DataPagerFieldItem container)
        {
            int endRowIndex = _startRowIndex + DataPager.PageSize;

            if (endRowIndex > _totalRowCount)
                endRowIndex = _totalRowCount;

            container.Controls.Add(new LiteralControl("<span>"));
            container.Controls.Add(new LiteralControl(string.Format(_rowXofY, _startRowIndex + 1, endRowIndex, _totalRowCount)));
            container.Controls.Add(new LiteralControl("</span>"));
        }

        /// <summary>
        /// Creates the page size control.
        /// </summary>
        /// <param name="container">The container.</param>
        private void CreatePageSizeControl(DataPagerFieldItem container)
        {
            var pageSizeDropDownList = new ButtonDropDownList { CommandName = "UpdatePageSize", CssClass = "form-control"};

            pageSizeDropDownList.Items.Add(new ListItem("5", "5"));
            pageSizeDropDownList.Items.Add(new ListItem("10", "10"));
            pageSizeDropDownList.Items.Add(new ListItem("15", "15"));
            pageSizeDropDownList.Items.Add(new ListItem("25", "25"));
            pageSizeDropDownList.Items.Add(new ListItem("50", "50"));
            pageSizeDropDownList.Items.Add(new ListItem("75", "75"));
            pageSizeDropDownList.Items.Add(new ListItem("100", "100"));

            var pageSizeItem = pageSizeDropDownList.Items.FindByValue(DataPager.PageSize.ToString());

            if (pageSizeItem == null)
            {
                pageSizeItem = new ListItem(DataPager.PageSize.ToString(), DataPager.PageSize.ToString());
                pageSizeDropDownList.Items.Insert(0, pageSizeItem);
            }

            pageSizeItem.Selected = true;
            container.Controls.Add(pageSizeDropDownList);
            
            container.Controls.Add(new LiteralControl(string.Format("<span>{0}</span>", _PerPage)));
        }

        public override void CreateDataPagers(DataPagerFieldItem container, int startRowIndex, int maximumRows, int totalRowCount, int fieldIndex)
        {
            _startRowIndex = startRowIndex;
            _maximumRows = maximumRows;
            _totalRowCount = totalRowCount;

            if (String.IsNullOrEmpty(DataPager.QueryStringField))
            {
                CreateDataPagersForCommand(container, fieldIndex);
            }
            else
            {
                CreateDataPagersForQueryString(container, fieldIndex);
            }
        }

        private void CreateDataPagersForCommand(DataPagerFieldItem container, int fieldIndex)
        {
            int currentPageIndex = _startRowIndex / _maximumRows;
            int firstButtonIndex = (_startRowIndex / (ButtonCount * _maximumRows)) * ButtonCount;
            int lastButtonIndex = firstButtonIndex + ButtonCount - 1;
            int lastRecordIndex = ((lastButtonIndex + 1) * _maximumRows) - 1;


            container.Controls.Add(new LiteralControl("<div class=\"grid-footer-left pull-left\">"));

            //Set of records - total records
            CreateLabelRecordControl(container);

            //Control used to set the page size.
            CreatePageSizeControl(container);

            container.Controls.Add(new LiteralControl("</div>"));

            container.Controls.Add(new LiteralControl("<ul class=\"pagination pull-right\">"));

            if (firstButtonIndex != 0)
            {
                container.Controls.Add(new LiteralControl("<li>"));
                container.Controls.Add(CreateNextPrevButton(PreviousPageText, DataControlCommands.PreviousPageCommandArgument, fieldIndex.ToString(CultureInfo.InvariantCulture), PreviousPageImageUrl));
                container.Controls.Add(new LiteralControl("</li>"));
            }

            for (int i = 0; i < ButtonCount && _totalRowCount > ((i + firstButtonIndex) * _maximumRows); i++)
            {
                if (i + firstButtonIndex == currentPageIndex)
                {
                    //Label pageNumber = new Label();
                    //pageNumber.Text = (i + firstButtonIndex + 1).ToString(CultureInfo.InvariantCulture);
                    //if (!String.IsNullOrEmpty(CurrentPageLabelCssClass))
                    //{
                    //    pageNumber.CssClass = CurrentPageLabelCssClass;
                    //}
                    container.Controls.Add(new LiteralControl("<li class=\"active\">"));
                    container.Controls.Add(new LiteralControl(string.Format("<span>{0}<span class=\"sr-only\">(current)</span></span>", i + firstButtonIndex + 1)));
                    container.Controls.Add(new LiteralControl("</li>"));
                }
                else
                {
                    container.Controls.Add(new LiteralControl("<li>"));
                    container.Controls.Add(CreateNumericButton((i + firstButtonIndex + 1).ToString(CultureInfo.InvariantCulture), fieldIndex.ToString(CultureInfo.InvariantCulture), (i + firstButtonIndex).ToString(CultureInfo.InvariantCulture)));
                    container.Controls.Add(new LiteralControl("</li>")); 
                }
            }

            if (lastRecordIndex < _totalRowCount - 1)
            {
                container.Controls.Add(new LiteralControl("<li>"));
                container.Controls.Add(CreateNextPrevButton(NextPageText, DataControlCommands.NextPageCommandArgument, fieldIndex.ToString(CultureInfo.InvariantCulture), NextPageImageUrl));
                container.Controls.Add(new LiteralControl("</li>"));
            }

            container.Controls.Add(new LiteralControl("</ul>"));

           
        }

        private void CreateDataPagersForQueryString(DataPagerFieldItem container, int fieldIndex)
        {
            int currentPageIndex = _startRowIndex / _maximumRows;
            bool resetProperties = false;
            if (!QueryStringHandled)
            {
                int currentQSPageIndex;
                QueryStringHandled = true;
                bool parsed = Int32.TryParse(QueryStringValue, out currentQSPageIndex);
                if (parsed)
                {
                    currentQSPageIndex--;//convert page number to page index.
                    int highestPageIndex = (_totalRowCount - 1) / _maximumRows;
                    if ((currentQSPageIndex >= 0) && (currentQSPageIndex <= highestPageIndex))
                    {
                        currentPageIndex = currentQSPageIndex;
                        _startRowIndex = (currentPageIndex * _maximumRows);
                        resetProperties = true;
                    }
                }
            }

            int firstButtonIndex = (_startRowIndex / (ButtonCount * _maximumRows)) * ButtonCount;
            int lastButtonIndex = firstButtonIndex + ButtonCount - 1;
            int lastRecordIndex = ((lastButtonIndex + 1) * _maximumRows) - 1;

            if (firstButtonIndex != 0)
            {
                container.Controls.Add(CreateNextPrevLink(PreviousPageText, firstButtonIndex - 1, PreviousPageImageUrl));
                AddNonBreakingSpace(container);
            }

            for (int i = 0; i < ButtonCount && _totalRowCount > ((i + firstButtonIndex) * _maximumRows); i++)
            {
                if (i + firstButtonIndex == currentPageIndex)
                {
                    Label pageNumber = new Label();
                    pageNumber.Text = (i + firstButtonIndex + 1).ToString(CultureInfo.InvariantCulture);
                    if (!String.IsNullOrEmpty(CurrentPageLabelCssClass))
                    {
                        pageNumber.CssClass = CurrentPageLabelCssClass;
                    }
                    container.Controls.Add(pageNumber);
                }
                else
                {
                    container.Controls.Add(CreateNumericLink(i + firstButtonIndex));
                }
                AddNonBreakingSpace(container);
            }

            if (lastRecordIndex < _totalRowCount - 1)
            {
                AddNonBreakingSpace(container);
                container.Controls.Add(CreateNextPrevLink(NextPageText, firstButtonIndex + ButtonCount, NextPageImageUrl));
                AddNonBreakingSpace(container);
            }

            if (resetProperties)
            {
                DataPager.SetPageProperties(_startRowIndex, _maximumRows, true);
            }
        }

        // Required for design-time support (DesignerPagerStyle)
        [SuppressMessage("Microsoft.Security", "CA2123:OverrideLinkDemandsShouldBeIdenticalToBase")]
        public override bool Equals(object o)
        {
            NumericPagerField field = o as NumericPagerField;
            if (field != null)
            {
                if (String.Equals(field.ButtonCount, this.ButtonCount) &&
                    field.ButtonType == this.ButtonType &&
                    String.Equals(field.CurrentPageLabelCssClass, this.CurrentPageLabelCssClass) &&
                    String.Equals(field.NextPageImageUrl, this.NextPageImageUrl) &&
                    String.Equals(field.NextPageText, this.NextPageText) &&
                    String.Equals(field.NextPreviousButtonCssClass, this.NextPreviousButtonCssClass) &&
                    String.Equals(field.NumericButtonCssClass, this.NumericButtonCssClass) &&
                    String.Equals(field.PreviousPageImageUrl, this.PreviousPageImageUrl) &&
                    String.Equals(field.PreviousPageText, this.PreviousPageText))
                {
                    return true;
                }
            }
            return false;
        }

        [SuppressMessage("Microsoft.Security", "CA2123:OverrideLinkDemandsShouldBeIdenticalToBase")]
        public override int GetHashCode()
        {
            return
                this.ButtonCount.GetHashCode() |
                this.ButtonType.GetHashCode() |
                this.CurrentPageLabelCssClass.GetHashCode() |
                this.NextPageImageUrl.GetHashCode() |
                this.NextPageText.GetHashCode() |
                this.NextPreviousButtonCssClass.GetHashCode() |
                this.NumericButtonCssClass.GetHashCode() |
                this.PreviousPageImageUrl.GetHashCode() |
                this.PreviousPageText.GetHashCode();
        }

    }
}