#
#  Parse JMDICT and output SQL source files for Kanji Koohii's dictionary related features.
#
#  (This was originally written around 2007. Back then the built in XML parsing functions
#   seemed really slow, which is why the xml file is parsed like a regular text file.)
#
#
#  BUILD
#
#    $ php split_readings.php > split_readings.utf8
#    $ perl parse_jdict.pl > table_jdict.utf8
#
#
#  OUTPUT
#
#    Data to be loaded straight into SQL tables. See /data/schemas/jdict_schema.sql
#
#    These SQL tables are referenced, read-only. They are never written to.
#
#      table_jdict.utf8
#      table_dictprons.utf8
#      table_dictsplit.utf8
#
#
#  SOURCES
#
#    jmdict.xml.utf8
#
#      Jim Breen / EDRDG's Multilingual Japanese Dictionary (XML format)
#
#
#    split_readings.utf8
#
#      Generated by split_readings.php
#
#      (I didn't know how to parse utf8 in Perl back then so the "splitting" of
#      readings is done by a php script. A unique key <compound+reading> is
#      used to match the split_readings data with the dictid's generated here.)
#
#
#  IMPORTANT
#
#    bitfields
#
#      The bitmasks (pos, verb, etc) have to match exactly with
#
#        apps/koohii/lib/rtkLabs.php
#
#
#
#  GOTCHAS
#
#    <ent_seq>
#
#      jmdict uses a unique index called "ent_seq". However since jmdict can contain
#      multiple compound + reading that share the same gloss, this script generates
#      additional intermediate IDs for each unique compoudn + reading.
#      
#      That's just how I did it then. The intention with intermidate dictid's was to
#      solve how to update to a newer JMDICT version while maintaining integrity
#      with the user data (ie. vocab entries). Should this be necessary, we can create
#      a map of unique <compound+reading> in a new table. Re-generate dict tables,
#      then update the ids in user data with the map table.
#  
#
#  HISTORY
#
#    2008/01/12
#    - optimize: pronid(SMALLINT) instead of pron(CHAR(5)) in dictsplit
#    - dictprons table
#
#

use strict;

use Encode;

use constant FILENAME_JMDICT                => '../../datafiles/download/jmdict.xml.utf8';
use constant FILENAME_FURIGANA              => 'split_readings.utf8';
#use constant FILENAME_GLOSSDATA            => 'table_dictgloss.utf8';

# output files
use constant FILENAME_TABLE_DICTPRONS       => 'table_dictprons.utf8';
use constant FILENAME_TABLE_DICTSPLIT       => 'table_dictsplit.utf8';

# currently unused
#use constant FILENAME_PHP_INCLUDE_DICTPRONS => './dictprons.inc.php';


# Part of speech markings, eash hash correspond to a 32 bit field in the database
# They also correspond to different xml tags (<pos>, <misc>, <re_inf> and <ke_inf>), <pos> needed to be split in two groups

# Relative priority of entries as a bitfield, higher priority = higher bit value, should fit into a TINYINT
my %pricodes = ( 'ichi1',0x80,'news1',0x40,'news2',0x20,'ichi2',0x10,'spec1',8,'spec2',4,'gai1',2,'gai2',1 );


# Part of Speech markings 
my %dictpos = (
# <pos>
  'adj',     0x00000001,  # adjective (keiyoushi)
  'adj-na',  0x00000002,  # adjectival nouns or quasi-adjectives (keiyodoshi)
  'adj-no',  0x00000004,  # nouns which may take the genitive case particle `no'
  'adj-pn',  0x00000008,  # pre-noun adjectival (rentaishi)
  'adj-t',   0x00000010,  # `taru' adjective
  'adv',     0x00000020,  # adverb (fukushi)
  'adv-n',   0x00000040,  # adverbial noun
  'adv-to',  0x00000080,  # adverb taking the `to' particle
  'aux',     0x00000100,  # auxiliary
  'aux-v',   0x00000200,  # auxiliary verb
  'aux-adj', 0x00000400,  # auxiliary adjective
  'conj',    0x00000800,  # conjunction
  'ctr',     0x00001000,  # counter
  'exp',     0x00002000,  # Expressions (phrases, clauses, etc.)
  'int',     0x00004000,  # interjection (kandoushi)
  'iv',      0x00010000,  # irregular verb
  'n',       0x00020000,  # noun (common) (futsuumeishi)
  'n-adv',   0x00040000,  # adverbial noun (fukushitekimeishi)
  'n-pref',  0x00080000,  # noun, used as a prefix
  'n-suf',   0x00100000,  # noun, used as a suffix
  'n-t',     0x00200000,  # noun (temporal) (jisoumeishi)
  'neg',     0x00400000,  # negative (in a negative sentence, or with negative verb)
  'neg-v',   0x00800000,  # negative verb (when used with)
  'num',     0x01000000,  # numeric
  'pref',    0x02000000,  # prefix
  'prt',     0x04000000,  # particle
  'suf',     0x08000000   # suffix
);

my %dictverb = (
# <pos>
  'v1',      0x00000001,  # Ichidan verb
  'v5',      0x00000002,  # Godan verb (not completely classified)
  'v5aru',   0x00000004,  # Godan verb - -aru special class
  'v5b',     0x00000008,  # Godan verb with `bu' ending
  'v5g',     0x00000010,  # Godan verb with `gu' ending
  'v5k',     0x00000020,  # Godan verb with `ku' ending
  'v5k-s',   0x00000040,  # Godan verb - iku/yuku special class
  'v5m',     0x00000080,  # Godan verb with `mu' ending
  'v5n',     0x00000100,  # Godan verb with `nu' ending
  'v5r',     0x00000200,  # Godan verb with `ru' ending
  'v5r-i',   0x00000400,  # Godan verb with `ru' ending (irregular verb)
  'v5s',     0x00000800,  # Godan verb with `su' ending
  'v5t',     0x00001000,  # Godan verb with `tsu' ending
  'v5u',     0x00002000,  # Godan verb with `u' ending
  'v5u-s',   0x00004000,  # Godan verb with `u' ending (special class)
  'v5uru',   0x00008000,  # Godan verb - uru old class verb (old form of Eru)
  'vi',      0x00010000,  # intransitive verb
  'vk',      0x00020000,  # kuru verb - special class
  'vs',      0x00040000,  # noun or participle which takes the aux. verb suru
  'vs-i',    0x00080000,  # suru verb - irregular
  'vs-s',    0x00100000,  # suru verb - special class
  'vt',      0x00200000,  # transitive verb
  'vz',      0x00400000   # zuru verb - (alternative form of -jiru verbs)
);

my %dictfield = (
# <field>
  'Buddh',   0x00000001,  # Buddhist term
  'MA',      0x00000002,  # martial arts term
  'comp',    0x00000004,  # computer terminology
  'food',    0x00000008,  # food term
  'geom',    0x00000010,  # geometry term
  'gram',    0x00000020,  # grammatical term
  'ling',    0x00000040,  # linguistics terminology
  'math',    0x00000080,  # mathematics
  'mil',     0x00000100,  # military
  'physics', 0x00000200   # physics terminology
);

my %dictmisc = (
# <misc>
  'X',       0x00000001,  # rude or X-rated term
  'abbr',    0x00000002,  # abbreviation
  'arch',    0x00000004,  # archaism
  'ateji',   0x00000008,  # ateji (phonetic) reading
  'chn',     0x00000010,  # children's language
  'col',     0x00000020,  # colloquialism
  'derog',   0x00000040,  # derogatory
  'eK',      0x00000080,  # exclusively kanji
  'fam',     0x00000100,  # familiar language
  'fem',     0x00000200,  # female term or language
  'gikun',   0x00000400,  # gikun (meaning) reading
  'hon',     0x00000800,  # honorific or respectful (sonkeigo) language
  'hum',     0x00001000,  # humble (kenjougo) language
  'id',      0x00002000,  # idiomatic expression
  'm-sl',    0x00004000,  # manga slang
  'male',    0x00008000,  # male term or language
  'male-sl', 0x00010000,  # male slang
  'ng',      0x00020000,  # neuter gender
  'obs',     0x00040000,  # obsolete term
  'obsc',    0x00080000,  # obscure term
  'pol',     0x00100000,  # polite (teineigo) language
  'rare',    0x00200000,  # rare
  'sens',    0x00400000,  # sensitive
  'sl',      0x00800000,  # slang
  'uK',      0x01000000,  # word usually written using kanji alone
  'uk',      0x02000000,  # word usually written using kana alone
  'vulg',    0x04000000,  # vulgar expression or word
# <re_inf>
  'ik',      0x08000000,  # word containing irregular kana usage
  'ok',      0x10000000,  # out-dated or obsolete kana usage
# <ke_inf>
  'iK',      0x20000000,  # word containing irregular kanji usage
  'io',      0x40000000,  # irregular okurigana usage
  'oK',      0x80000000   # word containing out-dated kanji
);


use Sort::ArbBiLex (
    'hiragana_sort',   # ask for a &xxxxx_sort to be defined
    "
あ
い
う
え
お
か
が
き
ぎ
きゃ
ぎゃ
きゅ
ぎゅ
きょ
ぎょ
く
ぐ
け
げ
こ
ご
さ
ざ
し
じ
しゃ
じゃ
しゅ
じゅ
しょ
じょ
す
ず
せ
ぜ
そ
ぞ
た
だ
ち
ぢ
ちゃ
ぢゃ
ちゅ
ぢゅ
ちょ
ぢょ
つ
っ
づ
て
で
と
ど
な
に
にゃ
にゅ
にょ
ぬ
ね
の
は
ば
ぱ
ひ
び
ぴ
ひゃ
びゃ
ぴゃ
ひゅ
びゅ
ぴゅ
ひょ
びょ
ぴょ
ふ
ぶ
ぷ
へ
べ
ぺ
ほ
ぼ
ぽ
ま
み
みゃ
みゅ
みょ
む
め
も
や
ゃ
ゆ
ゅ
よ
ょ
ら
り
りゃ
りゅ
りょ
る
れ
ろ
わ
を
ん
ア
ァ
イ
ィ
ウ
ゥ
ヴ
エ
ェ
オ
ォ
カ
ガ
キ
ギ
キャ
ギャ
キュ
ギュ
キョ
ギョ
ク
グ
ケ
ゲ
コ
ゴ
サ
ザ
シ
ジ
シャ
ジャ
シュ
ジュ
ショ
ジョ
ス
ズ
セ
ゼ
ソ
ゾ
タ
ダ
チ
ヂ
チャ
ヂャ
チュ
ヂュ
チョ
ヂョ
ツ
ッ
ヅ
テ
デ
ト
ド
ナ
ニ
ニャ
ニュ
ニョ
ヌ
ネ
ノ
ハ
バ
パ
ヒ
ビ
ピ
ヒャ
ビャ
ピャ
ヒュ
ビュ
ピュ
ヒョ
ビョ
ピョ
フ
ブ
プ
ヘ
ベ
ペ
ホ
ボ
ポ
マ
ミ
ミャ
ミュ
ミョ
ム
メ
モ
ヤ
ャ
ユ
ュ
ヨ
ョ
ラ
リ
リャ
リュ
リョ
ル
レ
ロ
ワ
ヲ
ン
    "
);



#------------------------------
#  functions
#------------------------------

sub openit
{
  my ($filename) = @_;
  local *FH;
    open (FH, $filename) or die "Cannot open file '$filename': $!";
    return *FH;
}

sub openWrite
{
  my ($filename) = @_;
  local *FH;
  open (FH, ">:utf8", $filename) or die "Cannot open file for writing '$filename': $!";
  return *FH;
}


# parse the xml dictionary, one entry at a time
#
#
sub parse_entry
{
  my ($fh, $e) = @_;
  my %kebs;     # possible compounds for that glossary entry
  my $keb;
  my $reb;      # reading for current <r_ele>
  my @restr;    # compounds to which current reading is restricted, if present
                # if not specified, the reading will apply to each compound listed in <k_ele>'s
  my $numsense=0;
          
  while (<$fh>)
  {
    if ( /<entry>/ ) 
    {
      $e->{'ent_seq'} = 0;    # unique glossary index number
      $e->{'keb'} = ();       # will hold assoc.array of compounds, and array of readings for each
      $e->{'gloss'} = '';
      $e->{'numkeb'}=0;
      $e->{'numreb'}=0;
      $e->{'pos'}=0;
      $e->{'verb'}=0;
      $e->{'misc'}=0;
      $e->{'field'}=0;
      $e->{'kpri'} = ();
      $e->{'rpri'} = ();
    }
    elsif ( /<ent_seq>([^<]+)</ ) {
      $e->{'ent_seq'} = $1;
    }
    
    elsif ( /<keb>([^<]+)</ ) {
      $keb=$1;
      $kebs{$1}=();
      $e->{'numkeb'}++;
    }
    elsif ( /<ke_pri>([^<]+)</ ) {
      my $bit = exists $pricodes{$1} ? $pricodes{$1} : 0;
      if (exists ${$e->{'kpri'}}{$keb}) {
        ${$e->{'kpri'}}{$keb} |= $bit;
      }else{
        ${$e->{'kpri'}}{$keb} = $bit;
      }
    }
    
    elsif ( /<r_ele>/ ) {
      @restr=();
      $e->{'numreb'}++;
    }
    elsif ( /<reb>(.*)<\// ) {
      $reb=$1;
      # no k_ele (i.e. all hiragana compound), then create compound same as reading
      if (!scalar(%kebs)){
        $kebs{$reb}=();
      }
    }
    elsif ( /<re_restr>(.*)<\// ) {
      # reading applies only to restricted compounds
      push (@restr, $1);
    }
    elsif ( /<re_pri>([^<]+)</ ) {
      my $bit = exists $pricodes{$1} ? $pricodes{$1} : 0;
      if (exists ${$e->{'rpri'}}{$reb}) {
        ${$e->{'rpri'}}{$reb} |= $bit;
      }else{
        ${$e->{'rpri'}}{$reb} = $bit;
      }
    }
    
    elsif ( /<\/r_ele>/ )
    {
      my $k_ele;
      if (scalar(@restr))
      {
        # reading applies to a restricted subset of the compounds
        foreach $k_ele (@restr)
        {
          if (!exists($kebs{$k_ele}))
          {
            # dict file error : <re_restr> points to unspecified compound (probably a typo)
            print STDERR "WARNING #1311 in ent_seq#".$e->{'ent_seq'}."\n";
            next;
          }
          push (@{$kebs{$k_ele}}, $reb);
        }
      }
      elsif (scalar(%kebs)){
        # reading applies to all given compound in this entry
        foreach $k_ele (keys %kebs)
        {
          push ( @{$kebs{$k_ele}}, $reb);
        }
      }
    }  
    elsif ( /<sense>/ ) {
      $numsense++;
      if ($numsense>1) {
        $e->{'gloss'}.=' ('.$numsense.') ';
      }
    }
    
    # Parts of speech
    elsif ( /<pos>&([^;]+)/ ) {
      if (exists $dictpos{$1}) {
        $e->{'pos'} |= $dictpos{$1};
      } elsif (exists $dictverb{$1}) {
        $e->{'verb'} |= $dictverb{$1};
      }
    }
    elsif ( /<field>&([^;]+)/ ) {
      $e->{'field'} |= (exists $dictfield{$1} ? $dictfield{$1} : 0 );
    }
    elsif ( /<misc>&([^;]+)/ ) {
      $e->{'misc'} |= (exists $dictmisc{$1} ? $dictmisc{$1} : 0 );
    }
    elsif ( /<[rk]e_inf>&([^;]+)/ ) {
      # this one matches <re_inf> and <ke_inf> tags
      $e->{'misc'} |= (exists $dictmisc{$1} ? $dictmisc{$1} : 0 );
    }
    
    elsif ( /<gloss>(.*)<\// ) {
      if ($e->{'gloss'}) {
        $e->{'gloss'}.='; '
      }
      $e->{'gloss'} .=$1;
    }
    elsif ( /<\/entry>/ )
    {
      # jmdict error : one compound in entry missing reading(s) due to <re_restr>
      foreach (keys %kebs) {
        if ( !defined($kebs{$_}) ) {
          delete $kebs{$_};
          print STDERR "MISSING readings in ent_seq#".$e->{'ent_seq'}."\n";
        }
      }
  
    #  if ($numsense>1) {
    #    print STDERR "Entry ".$e->{'ent_seq'}." multiple sense (".$numsense.")\n";
    #  }


      $e->{'keb'} = \%kebs;
      return 1;
    }
  }

  return 0;
}

#------------------------------
#  main
#------------------------------

  #
  #  Get the split readings
  #
  my $fh1 = openit(FILENAME_FURIGANA);
  my %splitR;
  my %uniqueprons;  # unique on and kun readings
  while (<$fh1>){
    chomp;
    my @data = split /\t/, $_;

    # <entseq> <compound> <full reading> <split-reading> (eg. 1582840 入水 じゅすい 1.じゅ.1.すい )
    if ($data[0] =~ /^[0-9]+$/)
    {
      # collect all possible on/kun readings for the 'dictprons' table
      my $pron;
      my @splitreb = split /\./, $data[3];   #decode_utf8($data[3]);
      while(scalar @splitreb > 0)
      {
        shift(@splitreb);
        $pron = shift(@splitreb);
        if (!exists $uniqueprons{$pron}){
          $uniqueprons{$pron} = 1;
        }
      }
      
      # create a hash to match dict entries later to their split readings, using the unique key (compound, reading)
      # the key is not unique (eg."ebisu"), but it works as far as furigana is concerned
      my $combo = $data[1].$data[2];
      if (!exists $splitR{$combo}){
        $splitR{$combo} = $data[3];
      }
      else{
        print STDERR "Note: Non-unique Compound>Reading combo: ".$data[1]." : ".$data[2]."\n";
      }
    }
  }
  close ($fh1);
  print STDERR "Parsed ".scalar(keys %splitR)." split-readings\n";
  print STDERR "Found ".scalar (keys %uniqueprons)." unique on/kun readings\n";

  #
  #  Output table of unique pronunciations and ids
  #

  # sort the pronunciations by hiragana, then katakana
  my @pron_array;
  my $i = 0;
  foreach (keys %uniqueprons) {
    push (@pron_array, $_);
  }
  my @pron_sorted = hiragana_sort(@pron_array);

  # output dictpron file
  print STDERR "\nGenerate DICTPRONS data ...\n";
  my $fh_dictprons = openWrite (FILENAME_TABLE_DICTPRONS);
  my $pronid = 1;
  print $fh_dictprons "0\t\n";
  foreach (@pron_sorted)
  {
    $uniqueprons{decode_utf8($_)} = $pronid;   # remember the id
    
    print $fh_dictprons $pronid."\t".decode_utf8($_)."\n";
    $pronid++;
  }
  close ($fh_dictprons);


  #
  #  Output include files for PHP, to save some lookups (where we don't need joins)
  #
  #use constant PHP_INCLUDE_MAX_LINE_LENGTH => 512;
  #my $fhphpinc = openWrite (FILENAME_PHP_INCLUDE_DICTPRONS);
  #my $s_arraydata = '';
  #my $s_line = '';
  #$pronid = 1; # same as above!
  #print $fhphpinc "<?php\n".
  #      "\$DICTPRONS_LOOKUP_PRON = array(\n''=>0,";
  #foreach (@pron_sorted)
  #{
  #  my $comma = $pronid < scalar @pron_sorted ? ',' : '';
  #  my $pron = decode_utf8($_);
  #  $s_line .= "'$pron'=>".$pronid.$comma;
  #  if (length($s_line)>=PHP_INCLUDE_MAX_LINE_LENGTH){
  #    $s_arraydata .= $s_line."\n";
  #    $s_line = '';
  #  }
  #  $pronid++;
  #}
  #if ($s_line ne ''){
  #  $s_arraydata .= $s_line."\n";
  #}
  #print $fhphpinc $s_arraydata.");\n?>\n";
  #close ($fhphpinc);


  #
  #  Parse jmdict, match split readings to each entry,
  #  output jdict(dictionary) and dictsplit(splitted entries) tables.
  #
  my $fh = openit (FILENAME_JMDICT);
  #my $fh2 = openWrite (FILENAME_GLOSSDATA);
  my $fh_dictsplit = openWrite (FILENAME_TABLE_DICTSPLIT);

  print STDERR "\nGenerate JDICT & DICTSPLIT data ... \n"; 

#  print <<EOD;
#  tab delimited output from JMDICT for insertion in database
#
#EOD

  my %e = ();
  my %entseqs;
  my $numentries = 0;
  my %u8ucs = ();

  while (parse_entry($fh, \%e))
  {
    $numentries++;

    my %kebs = %{ $e{'keb'} };
    my $k_ele;
    my $entseq = $e{'ent_seq'};
    my $entseq_sub=0;

    foreach $k_ele (keys %kebs)
    {
      my ($r_ele, $furigana);
      my @rebs = @{ $kebs{$k_ele} };
      foreach $r_ele (@rebs)
      {
        if($entseq_sub>=10) {
          last;
        }
        
        my $entseq_id = $entseq + $entseq_sub;
        
        if (exists($entseqs{$entseq_id})) {
          print STDERR "ENTSEQ double! ".$entseq_id."\n";
        }
        $entseqs{$entseq_id}=1;
        
        my $frequency = exists ${$e{'kpri'}}{$k_ele} ? ${$e{'kpri'}}{$k_ele} : 0;
        $frequency &= ( exists ${$e{'rpri'}}{$r_ele} ? ${$e{'rpri'}}{$r_ele} : 0 );
        
        print $entseq_id."\t".$frequency."\t".$e{'pos'}."\t".$e{'verb'}."\t".$e{'misc'}."\t".$e{'field'}."\t".$k_ele."\t".$r_ele."\t".$e{'gloss'}."\n";
        # problem with unsigned, and cant get hexadecimal to work with LOAD DATA
        #printf "%d\t%d\t%d\t%d\t%d\t%s\t%s\t%s\n", $entseq_id, $e{'pos'}, $e{'verb'}, $e{'misc'}, $e{'field'}, $k_ele, $r_ele, $e{'gloss'};

        #
        # dict split table
        #
        my $splitted = exists($splitR{$k_ele.$r_ele}) ? $splitR{$k_ele.$r_ele} : '';
        my @splitkeb = split //, decode_utf8($k_ele);
        my @splitreb = split /\./, decode_utf8($splitted);
        my ($kanji, $kanji_id,  $pronid, $type);
        my $pos=0;
        foreach $kanji (@splitkeb) {
          if (length($splitted)>0) {
            $type = shift(@splitreb);
            $pronid = $uniqueprons{shift(@splitreb)};
          }else{
            $type = 0;
            $pronid = 0;  # 0 is empty pron
          }

          # convert kanji to unicode id
          if (exists($u8ucs{$kanji})){
            $kanji_id = $u8ucs{$kanji};
          }
          else{
            my @chars = split //, $kanji;
            # ord works here because we already did decode_utf8
            $kanji_id = $u8ucs{$kanji} = ord($chars[0]); 
          }
          
          # output dictsplit
          print $fh_dictsplit $entseq_id."\t".$kanji_id."\t".$pronid."\t".$type."\t".$pos."\n";
          $pos++;
        }

        $entseq_sub++;
      }
    }
    
    # table of glossaries
    #print $fh2 $e{'ent_seq'}."\t".$e{'gloss'}."\n";

#    if ($e{'ent_seq'}%10) {
#      print STDERR "woops $e{'ent_seq'}\n";
#    }

#    if ($numentries >= 100) { last; }
  }

  close ($fh);
  #close ($fh2);
  close ($fh_dictsplit);

  print STDERR " ... output '".FILENAME_TABLE_DICTSPLIT."'.\n";

  print STDERR $numentries." parsed entries\n";
