<?php
/**
 * Pre-parse the Vite manifest.json file into a php include file for
 * production environment to quickly match entries to css/js dependencies.
 *
 * Uses SF_ROOT_DIR/lib/Vite.php helper to parse manifest.json into an
 * array of "entries". This array is then formatted into a php include
 * file to skip the parsing step in production env.
 *
 * Example:
 *
 *   $ vite build
 *   $ php batch/build_app.php -v --vite web/build/dist/manifest.json -o config/vite-build.inc.php
 */
require_once realpath(dirname(__FILE__).'/..').'/lib/batch/Command_CLI.php';

class BuildViteManifest extends Command_CLI
{
  public function __construct()
  {
    parent::__construct([
      'vite' => 'Include vite entries, parse manifest.json',
      'out|o=s' => 'Output filename (required)',
    ]);

    if (null === $this->getFlag('vite'))
    {
      $this->throwError('Required --vite flag.');
    }

    $outfile = $this->getFlag('o');
    if (null === $outfile)
    {
      $this->throwError('Required --out flag. Type --help for help.');
    }

    // $contents = $this->build($files);
    $generatorTime = date('F j, Y G:i:s');
    $generatorFile = basename(__FILE__);
    $contents = <<<EOD
<?php
/**
 * This file was generated by script "{$generatorFile}"
 * 
 * @date    {$generatorTime}
 */
EOD;

    if (null !== ($manifest = $this->getFlag('vite')))
    {
      $entries = Vite::getManifestJson();

      $contents = $contents."\n\n".$this->formatEntries($entries);

      $this->verbose("Vite manifest parsed.\n");
    }

    if (false === file_put_contents($outfile, $contents))
    {
      $this->throwError('Error writing to outfile "%s".', $outfile);
    }

    $this->verbose('Success! (output file "%s").', $outfile);
  }

  /**
   * Replace any Windows-style backslashes with slashes.
   *
   * @param string $path Fully qualified path name
   *
   * @return string
   */
  protected function normalizeForwardSlashes($path)
  {
    return preg_replace('/[\/\\\]/', '/', $path);
  }

  /**
   * PHP var_export() with short array syntax (square brackets) indented 2 spaces.
   *
   * NOTE: The only issue is when a string value has `=>\n[`, it will get converted to `=> [`
   *
   * @see https://www.php.net/manual/en/function.var-export.php
   *
   * @author  https://gist.github.com/stemar/f4cc560e2bb96b99d1e3af292fb3ee20
   *
   * @param mixed $expression
   */
  private function varexport($expression, bool $return = true)
  {
    $export = var_export($expression, true);
    $patterns = [
      '/array \\(/' => '[',
      '/^([ ]*)\\)(,?)$/m' => '$1]$2',
      "/=>[ ]?\n[ ]+\\[/" => '=> [',
      "/([ ]*)(\\'[^\\']+\\') => ([\\[\\'])/" => '$1$2 => $3',
    ];
    $export = preg_replace(array_keys($patterns), array_values($patterns), $export);

    if ((bool) $return)
    {
      return $export;
    }
    else
    {
      echo $export;
    }
  }

  private function formatEntries(array $entries)
  {
    ob_start();

    echo <<<'EOD'
    // Vite build entries & dependencies.
    EOD;

    echo "\nreturn ";

    echo $this->varexport($entries);

    echo ";\n";

    return ob_get_clean();
  }
}

$cmd = new BuildViteManifest();
