﻿// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

using Fetitor.Properties;
using ScintillaNET;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Xml;

namespace Fetitor
{
	public partial class FrmMain : Form
	{
		private readonly string _windowTitle;
		private string _openedFilePath;
		private bool _fileChanged;
		private List<ListViewItem> _features = new List<ListViewItem>();
		private FrmSearch _searchForm;

		/// <summary>
		/// Creates new instance.
		/// </summary>
		public FrmMain()
		{
			this.InitializeComponent();
			this.SetUpEditor();
			this.UpdateFeatureNames();

			_windowTitle = this.Text;
			this.LblKnownFeatureCount.Text = "";
			this.ToolStrip.Renderer = new ToolStripRendererNL();

			this.UpdateSaveButtons();
		}

		/// <summary>
		/// Updates feature names list.
		/// </summary>
		private void UpdateFeatureNames()
		{
			var featuresPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "features.txt");
			if (File.Exists(featuresPath))
				FeaturesFile.LoadFeatureNames(featuresPath);
		}

		/// <summary>
		/// Called when the form loads.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void FrmMain_Load(object sender, EventArgs e)
		{
			var args = Environment.GetCommandLineArgs();
			if (args.Length > 1)
			{
				var filePath = args[1];
				this.OpenFile(filePath);
			}

			if (Settings.Default.WindowMaximized)
			{
				this.WindowState = FormWindowState.Maximized;
			}
			else if (Settings.Default.WindowLocation.X != -1 || Settings.Default.WindowLocation.Y != -1)
			{
				this.WindowState = FormWindowState.Normal;
				this.Location = Settings.Default.WindowLocation;
				this.Size = Settings.Default.WindowSize;
			}
		}

		/// <summary>
		/// Saves settings when the form is closing.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void FrmMain_FormClosing(object sender, FormClosingEventArgs e)
		{
			Settings.Default.WindowMaximized = (this.WindowState == FormWindowState.Maximized);
			if (this.WindowState == FormWindowState.Normal)
			{
				Settings.Default.WindowLocation = this.Location;
				Settings.Default.WindowSize = this.Size;
			}

			Settings.Default.Save();
		}

		/// <summary>
		/// Sets up the editor for XML code and subscribes to its events.
		/// </summary>
		public void SetUpEditor()
		{
			this.TxtEditor.Styles[Style.Default].Font = "Courier New";
			this.TxtEditor.Styles[Style.Default].Size = 10;
			this.TxtEditor.Styles[Style.Xml.XmlStart].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.XmlEnd].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.TagEnd].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.Tag].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.TagEnd].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.Attribute].ForeColor = Color.Red;
			this.TxtEditor.Styles[Style.Xml.DoubleString].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.Xml.SingleString].ForeColor = Color.Blue;
			this.TxtEditor.Styles[Style.IndentGuide].ForeColor = Color.LightGray;
			this.TxtEditor.Margins[0].Width = 40;
			this.TxtEditor.Lexer = Lexer.Xml;
			this.TxtEditor.TextChanged += this.TxtEditor_OnTextChanged;
			this.TxtEditor.CtrlS += this.TxtEditor_OnCtrlS;
			this.TxtEditor.CtrlF += this.TxtEditor_OnCtrlF;
			this.TxtEditor.UpdateUI += this.TxtEditor_OnUpdateUI;
		}

		/// <summary>
		/// Called if Ctrl+F is pressed in the editor.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtEditor_OnCtrlF(object sender, EventArgs e)
		{
			this.BtnSearch_Click(null, null);
		}

		/// <summary>
		/// Called if the editor's text or styling have changed.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtEditor_OnUpdateUI(object sender, UpdateUIEventArgs e)
		{
			if ((e.Change & UpdateChange.Selection) != 0 || (e.Change & UpdateChange.Content) != 0)
			{
				var pos = this.TxtEditor.CurrentPosition;
				var line = this.TxtEditor.LineFromPosition(pos);
				var col = (pos - this.TxtEditor.Lines[line].Position);

				this.LblCurLine.Text = "Line " + (line + 1);
				this.LblCurCol.Text = "Col " + (col + 1);
			}
		}

		/// <summary>
		/// Called when text in the editor changes.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtEditor_OnTextChanged(object sender, EventArgs e)
		{
			_fileChanged = true;

			this.UpdateUndo();
			this.UpdateSaveButtons();
		}

		/// <summary>
		/// Called when Ctrl+S is pressed in the editor.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtEditor_OnCtrlS(object sender, EventArgs e)
		{
			if (this.BtnSave.Enabled)
				this.BtnSave.PerformClick();
		}

		/// <summary>
		/// Updated Undo and Redo buttons, based on the editor's state.
		/// </summary>
		private void UpdateUndo()
		{
			this.BtnUndo.Enabled = this.TxtEditor.CanUndo;
			this.BtnRedo.Enabled = this.TxtEditor.CanRedo;
		}

		/// <summary>
		/// Resets Undo and Redo in the editor and updates the buttons.
		/// </summary>
		private void ResetUndo()
		{
			this.TxtEditor.EmptyUndoBuffer();
			this.UpdateUndo();
		}

		/// <summary>
		/// Toggles save buttons, based on whether saving is possible right
		/// now.
		/// </summary>
		private void UpdateSaveButtons()
		{
			var enabled = (_fileChanged && !string.IsNullOrWhiteSpace(_openedFilePath));
			this.MnuSave.Enabled = this.BtnSave.Enabled = enabled;

			var fileOpen = (_openedFilePath != null);
			this.MnuSaveAsXml.Enabled = fileOpen;
		}

		/// <summary>
		/// Updates feature name jump list.
		/// </summary>
		private void UpdateFeatureList()
		{
			var text = this.TxtEditor.Text;

			var index = text.IndexOf("<Features>");
			if (index == -1)
				return;

			var list = new List<ListViewItem>();

			while ((index = text.IndexOf("Name=", index)) != -1)
			{
				var nameStartIndex = index + "Name=\"".Length;
				var nameEndIndex = text.IndexOf("\"", nameStartIndex);
				var length = nameEndIndex - nameStartIndex;

				index = nameEndIndex;

				var name = text.Substring(nameStartIndex, length);
				if (name == "?")
					continue;

				var lvi = new ListViewItem(name);
				lvi.Tag = new Tuple<int, int>(nameStartIndex, nameEndIndex);

				list.Add(lvi);
			}

			_features = list;

			this.PopulateFeatureList(_features);
		}

		/// <summary>
		/// Called when feature list is double clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void LstFeatures_DoubleClick(object sender, EventArgs e)
		{
			if (this.LstFeatures.SelectedItems.Count == 0)
				return;

			var selectedItem = this.LstFeatures.SelectedItems[0];
			if (selectedItem == null || selectedItem.Tag == null)
				return;

			if (!(selectedItem.Tag is Tuple<int, int> indices))
				return;

			this.TxtEditor.SetSelection(indices.Item1, indices.Item2);
			this.TxtEditor.ScrollRange(indices.Item1, indices.Item2);
			this.TxtEditor.Focus();
		}

		/// <summary>
		/// Called when one of the Open buttons is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnOpen_Click(object sender, EventArgs e)
		{
			var result = this.OpenFileDialog.ShowDialog();
			if (result != DialogResult.OK)
				return;

			this.OpenFile(OpenFileDialog.FileName);
		}

		/// <summary>
		/// Called if the Undo button is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnUndo_Click(object sender, EventArgs e)
		{
			this.TxtEditor.Undo();

			this.UpdateUndo();
			this.UpdateSaveButtons();
		}

		/// <summary>
		/// Called if the Redo button is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnRedo_Click(object sender, EventArgs e)
		{
			this.TxtEditor.Redo();

			this.UpdateUndo();
			this.UpdateSaveButtons();
		}

		/// <summary>
		/// Called if the Exit menu item is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MenuExit_Click(object sender, EventArgs e)
		{
			this.Close();
		}

		/// <summary>
		/// Called if a file is dragged onto the form.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void FrmMain_DragEnter(object sender, DragEventArgs e)
		{
			if (e.Data.GetDataPresent(DataFormats.FileDrop))
				e.Effect = DragDropEffects.Copy;
		}

		/// <summary>
		/// Called if a file is dropped on the form. 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void FrmMain_DragDrop(object sender, DragEventArgs e)
		{
			var files = (string[])e.Data.GetData(DataFormats.FileDrop);
			if (files.Length == 0)
				return;

			this.OpenFile(files[0]);
		}

		/// <summary>
		/// Tries to open compiled XML file at the given path in the editor.
		/// </summary>
		/// <param name="filePath"></param>
		private void OpenFile(string filePath)
		{
			// Check file's existence
			if (!File.Exists(filePath))
			{
				MessageBox.Show("File not found: " + filePath, _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}

			// Check file extension
			if (!Path.GetFileName(filePath).EndsWith(".xml.compiled"))
			{
				MessageBox.Show("Unknown file type, expected: *.xml.compiled", _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}

			// Read file
			try
			{
				string xml;
				using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
					xml = FeaturesFile.CompiledToXml(fs);

				this.TxtEditor.Text = xml;
				this.Text = _windowTitle + " - " + filePath;

				_openedFilePath = filePath;

				var known = Regex.Matches(xml, @"Name=""[^\?\""]+""").Count;
				var total = Regex.Matches(xml, @"Name=""").Count;
				this.LblKnownFeatureCount.Text = string.Format("Known features: {0}/{1}", known, total);

				_fileChanged = false;
				this.ResetUndo();
				this.UpdateSaveButtons();
				this.UpdateFeatureList();
			}
			catch (InvalidDataException)
			{
				MessageBox.Show("Invalid file format.", _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (EndOfStreamException)
			{
				MessageBox.Show("Corrupted file.", _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (NotSupportedException)
			{
				MessageBox.Show("Unsupported file.", _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (Exception ex)
			{
				MessageBox.Show("Error: " + ex.Message, _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		/// <summary>
		/// Called if one of the Save buttons is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnSave_Click(object sender, EventArgs e)
		{
			if (string.IsNullOrWhiteSpace(_openedFilePath))
				return;

			try
			{
				this.SaveFile(_openedFilePath);

				_fileChanged = false;
				this.UpdateSaveButtons();
			}
			catch (XmlException ex)
			{
				MessageBox.Show("XML Error: " + ex.Message, _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (Exception ex)
			{
				MessageBox.Show("Error: " + ex.ToString(), _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		/// <summary>
		/// Saves file to given path.
		/// </summary>
		/// <param name="filePath"></param>
		private void SaveFile(string filePath)
		{
			using (var fs = new FileStream(filePath, FileMode.OpenOrCreate, FileAccess.Write, FileShare.None))
				FeaturesFile.SaveXmlAsCompiled(TxtEditor.Text, fs);
		}

		/// <summary>
		/// Called if the Save as XML menu item is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MenuSaveAsXml_Click(object sender, EventArgs e)
		{
			this.SaveFileDialog.InitialDirectory = Path.GetDirectoryName(_openedFilePath);
			this.SaveFileDialog.FileName = "features.xml";

			if (this.SaveFileDialog.ShowDialog() != DialogResult.OK)
				return;

			try
			{
				using (var fs = this.SaveFileDialog.OpenFile())
				using (var sw = new StreamWriter(fs))
					sw.Write(this.TxtEditor.Text);
			}
			catch (Exception ex)
			{
				MessageBox.Show("Error: " + ex.ToString(), _windowTitle, MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		/// <summary>
		/// Called if the About menu item is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MenuAbout_Click(object sender, EventArgs e)
		{
			var form = new FrmAbout();
			form.ShowDialog();
		}

		/// <summary>
		/// Called if TxtFeatureFilter gets the focus.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtFeatureFilter_Enter(object sender, EventArgs e)
		{
			if (this.TxtFeatureFilter.ForeColor == Color.Silver)
			{
				this.TxtFeatureFilter.ForeColor = Color.Black;
				this.TxtFeatureFilter.Text = "";
			}
		}

		/// <summary>
		/// Called if TxtFeatureFilter loses the focus.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtFeatureFilter_Leave(object sender, EventArgs e)
		{
			if (string.IsNullOrWhiteSpace(this.TxtFeatureFilter.Text.Trim()))
			{
				this.TxtFeatureFilter.ForeColor = Color.Silver;
				this.TxtFeatureFilter.Text = "Filter";
				this.FilterFeatures("");
			}
		}

		/// <summary>
		/// Filters the feature list, only showing those that contain
		/// the given string.
		/// </summary>
		/// <param name="filter"></param>
		private void FilterFeatures(string filter)
		{
			this.PopulateFeatureList(_features.Where(a => a.Text.ToUpper().Contains(filter.ToUpper())));
		}

		/// <summary>
		/// Fills feature list with the given items.
		/// </summary>
		/// <param name="items"></param>
		private void PopulateFeatureList(IEnumerable<ListViewItem> items)
		{
			this.LstFeatures.BeginUpdate();
			this.LstFeatures.Items.Clear();

			// Add names alphabetically
			foreach (var item in items.OrderBy(a => a.Text))
				this.LstFeatures.Items.Add(item);

			// Autosize column header
			this.LstFeatures.Columns[0].AutoResize(ColumnHeaderAutoResizeStyle.HeaderSize);
			this.LstFeatures.Columns[0].Width -= 5;

			this.LstFeatures.EndUpdate();
		}

		/// <summary>
		/// Called if a key is let go of in TxtFeatureFilter.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void TxtFeatureFilter_KeyUp(object sender, KeyEventArgs e)
		{
			var filter = this.TxtFeatureFilter.Text;
			this.FilterFeatures(filter);
		}

		/// <summary>
		/// Called when Clear Filter button is clicked.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnClearFilter_Click(object sender, EventArgs e)
		{
			if (!string.IsNullOrWhiteSpace(this.TxtFeatureFilter.Text.Trim()))
			{
				this.TxtFeatureFilter.Text = "";
				this.TxtFeatureFilter_Leave(null, null);
			}
		}

		/// <summary>
		/// Jumps to the next occurance of given text in editor.
		/// </summary>
		/// <param name="searchText">Text to search for, starting at current position.</param>
		/// <returns>True if text was found, False if not.</returns>
		public bool SearchFor(string searchText)
		{
			var text = this.TxtEditor.Text;
			var currentPos = this.TxtEditor.CurrentPosition;
			var selectLength = searchText.Length;

			if (string.IsNullOrEmpty(text))
				return false;

			currentPos++;
			if (currentPos > text.Length - 1)
				currentPos = 0;

			var nextIndex = text.IndexOf(searchText, currentPos, StringComparison.CurrentCultureIgnoreCase);
			if (nextIndex == -1)
			{
				nextIndex = text.IndexOf(searchText, 0, StringComparison.CurrentCultureIgnoreCase);
				if (nextIndex == -1)
					return false;
			}

			this.TxtEditor.SetSelection(nextIndex + selectLength, nextIndex);
			this.TxtEditor.ScrollRange(nextIndex + selectLength, nextIndex);
			//this.TxtEditor.Focus();

			return true;
		}

		/// <summary>
		/// Called if Search button is clicked, opens Search form.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnSearch_Click(object sender, EventArgs e)
		{
			if (_searchForm == null)
			{
				_searchForm = new FrmSearch(this);

				var x = this.Location.X + this.Width / 2 - _searchForm.Width / 2;
				var y = this.Location.Y + this.Height / 2 - _searchForm.Height / 2;
				_searchForm.Location = new Point(x, y);
			}

			if (!_searchForm.Visible)
				_searchForm.Show(this);
			else
				_searchForm.Focus();

			_searchForm.SelectSearchText();
		}

		/// <summary>
		/// Opens form to updates feature names.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void BtnUpdateFeatureNames_Click(object sender, EventArgs e)
		{
			var form = new FrmUpdateFeatureNames();
			form.ShowDialog();

			this.UpdateFeatureNames();
		}

		/// <summary>
		/// Enables the feature in the line the cursor is in.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void MnuEnableSelected_Click(object sender, EventArgs e)
		{
			var pos = this.TxtEditor.CurrentPosition;
			var lineIndex = this.TxtEditor.LineFromPosition(pos);
			var line = this.TxtEditor.Lines[lineIndex];
			var start = line.Position;
			var end = line.EndPosition;
			var text = line.Text;

			text = Regex.Replace(text, @"<Feature\s+Hash\s*=\s*""([^""]*)""\s+Name\s*=\s*""([^""]*)""\s+Default\s*=\s*""([^""]*)""\s+Enable\s*=\s*""([^""]*)""\s+Disable\s*=\s*""([^""]*)""", @"<Feature Hash=""$1"" Name=""$2"" Default=""G0S0"" Enable="""" Disable=""""");

			this.TxtEditor.SetSelection(end, start);
			this.TxtEditor.ReplaceSelection(text);
			this.TxtEditor.GotoPosition(pos);
		}
	}
}
