<?php
/**
 *
 * This file is part of phpFastCache.
 *
 * @license MIT License (MIT)
 *
 * For full copyright and license information, please see the docs/CREDITS.txt file.
 *
 * @author Khoa Bui (khoaofgod)  <khoaofgod@gmail.com> https://www.phpfastcache.com
 * @author Georges.L (Geolim4)  <contact@geolim4.com>
 *
 */
declare(strict_types=1);

namespace Phpfastcache\Drivers\Riak;

use Basho\Riak\Riak;
use Phpfastcache\Core\Pool\{
    DriverBaseTrait, ExtendedCacheItemPoolInterface
};
use Phpfastcache\Entities\DriverStatistic;
use Phpfastcache\Exceptions\{
    PhpfastcacheInvalidArgumentException, PhpfastcacheLogicException
};
use Psr\Cache\CacheItemInterface;

/**
 * Class Driver
 * @package phpFastCache\Drivers
 * @property Riak $instance Instance of driver service
 * @property Config $config Config object
 * @method Config getConfig() Return the config object
 */
class Driver implements ExtendedCacheItemPoolInterface
{
    const RIAK_DEFAULT_BUCKET_NAME = 'phpfastcache';

    /**
     * @var string
     */
    protected $bucketName = self::RIAK_DEFAULT_BUCKET_NAME;

    use DriverBaseTrait;

    /**
     * @return bool
     */
    public function driverCheck(): bool
    {
        return \class_exists('Basho\Riak\Riak');
    }


    /**
     * @return bool
     * @throws PhpfastcacheLogicException
     */
    protected function driverConnect(): bool
    {
        if ($this->instance instanceof Riak) {
            throw new PhpfastcacheLogicException('Already connected to Riak server');
        }

        $this->bucketName = $this->getConfig()->getBucketName();

        $this->instance = new Riak($this->getConfig()->getHost(), $this->getConfig()->getPort(), $this->getConfig()->getPrefix());

        return true;
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return null|array
     */
    protected function driverRead(CacheItemInterface $item)
    {
        return $this->decode($this->instance->bucket($this->bucketName)->getBinary($item->getKey())->getData());
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return mixed
     * @throws PhpfastcacheInvalidArgumentException
     */
    protected function driverWrite(CacheItemInterface $item): bool
    {
        /**
         * Check for Cross-Driver type confusion
         */
        if ($item instanceof Item) {
            $this->instance
                ->bucket($this->bucketName)
                ->newBinary($item->getKey(), $this->encode($this->driverPreWrap($item)))
                ->store();
            return true;
        }

        throw new PhpfastcacheInvalidArgumentException('Cross-Driver type confusion detected');
    }

    /**
     * @param \Psr\Cache\CacheItemInterface $item
     * @return bool
     * @throws PhpfastcacheInvalidArgumentException
     */
    protected function driverDelete(CacheItemInterface $item): bool
    {
        /**
         * Check for Cross-Driver type confusion
         */
        if ($item instanceof Item) {
            $this->instance->bucket($this->bucketName)->get($item->getKey())->delete();
            return true;
        }

        throw new PhpfastcacheInvalidArgumentException('Cross-Driver type confusion detected');
    }

    /**
     * @return bool
     */
    protected function driverClear(): bool
    {
        $bucket = $this->instance->bucket($this->bucketName);
        foreach ($bucket->getKeys() as $key) {
            $bucket->get($key)->delete();
        }
        return true;
    }

    /********************
     *
     * PSR-6 Extended Methods
     *
     *******************/

    /**
     * @return DriverStatistic
     */
    public function getStats(): DriverStatistic
    {
        $info = $this->instance->bucket($this->bucketName)->getProperties();

        return (new DriverStatistic())
            ->setData(\implode(', ', \array_keys($this->itemInstances)))
            ->setRawData($info)
            ->setSize(false)
            ->setInfo('Riak does not provide size/date information att all :(');
    }
}