import { type Context } from "@/trpc/context";
import { TRPCError, initTRPC } from "@trpc/server";
import superjson from "superjson";

const t = initTRPC.context<Context>().create({
  transformer: superjson,
  errorFormatter({ shape }) {
    return shape;
  },
});

export const createRouter = t.router;

/**
 * Unprotected procedure
 **/
export const withoutAuth = t.procedure;

/**
 * Reusable middleware to ensure
 * users are logged in
 */
const isAuthenticated = t.middleware(({ ctx, next }) => {
  if (!ctx.session || !ctx.session.user) {
    throw new TRPCError({ code: "UNAUTHORIZED" });
  }

  if (ctx.session.user.locked) {
    throw new TRPCError({
      code: "FORBIDDEN",
      message:
        "Your account has been locked. Please contact support for assistance.",
    });
  }

  return next({
    ctx: {
      // infers the `session` as non-nullable
      session: { ...ctx.session, user: ctx.session.user },
    },
  });
});

/**
 * Protected procedure
 **/
export const withAuth = t.procedure.use(isAuthenticated);
