# Internationalization

Interface X contains a basic system to help you internationalize your application for different
languages and cultures. To do so, it uses
[VueI18n](https://kazupon.github.io/vue-i18n/guide/formatting.html) library, which is the standard
de-facto library for internationalizing Vue apps.

## Translations

If your app is going to be used by users with different languages, you may need to translate texts.
If not, it is also considered a good practice to have texts isolated from components. Interface X
Archetype provides a way to easily do this with the
[VueI18n](https://kazupon.github.io/vue-i18n/guide/formatting.html) library.

### Defining your messages

The first thing to do is to define the messages that you are going to need for your app using a
TypeScript interface. This will help you write and maintain then the messages in every language.
Think of this interface as a contract that every messages object has to fulfill. If you then need to
add, rename, or remove a message key, always start by modifying the interface. TypeScript will then
be gentle enough to remind you all the interface implementations that you need to modify.

To do so, open the [src/i18n/messages.types.ts](../src/i18n/messages.types.ts) file. There you will
find the `Messages` interface, where you should add or modify messages keys. For example, lets
define the following messages for the next steps.

```typescript
// File src/i18n/messages.types.ts

export interface Messages {
  historyQueries: {
    clear: string
    removeLabel: string
    title: string
  }
}
```

### Adding messages for a new locale

Once you have defined your interface, it's time to implement a new locale. The archetype already
includes english and spanish messages, so let's add a new german locale.

1. Create a new `de.messages.ts` file inside the [src/i18n](../src/i18n) directory.
2. Export a default object containing the messages. Use the `createMessages` helper method, to
   remind you from adding all the messages and removing the not used ones. Following the `Messages`
   interface that we had defined before, you can write:

   ```typescript
   // File src/i18n/messages/de.messages.ts
   export default createMessages({
     historyQueries: {
       clear: 'Löschen Sie Verlaufsabfragen',
       removeLabel: 'Entfernen Sie {suggestion} aus Verlaufsabfragen',
       title: 'Verlaufsabfragen',
     },
   })
   ```

   Have you noticed the `{suggestion}` inside the `removeLabel` message? This is to interpolate a
   value when using this message. When using this message you need to provide a value for the
   suggestion, which will replace the `{suggestion}` placeholder. This is explained more in depth
   below, in the [Translating content](#markdown-header-translating-content) section.

3. Re-export the new messages object in the [src/i18n/index.ts](../src/i18n/index.ts) file, giving
   it the locale name. For example:

```typescript
// File src/i18n/messages.index.ts
export { default as de } from './de.messages.ts'
```

Congratulations! Now you should be able to set `de` as a locale, and all the messages should be
translated.

### Translating content

To translate the messages or attributes inside the components, use the
[\$t](https://kazupon.github.io/vue-i18n/guide/formatting.html) method which is available in every
component. This method receives the path of the message, and an object to interpolate with the text
message.

For example, do you remember the german message for the `removeLabel`? It contained a
`{suggestion}`, property, so, to translate it you should write something like
`$t('historyQueries.removeLabel', { suggestion: <The value for the suggestion> })`. For example, if
you were extending the `HistoryQueries` component you can write:

```vue
// File: src/components/custom-history-queries.vue
<template>
  <HistoryQueries>
    <template #suggestion="{ suggestion }">
      <HistoryQuery :suggestion="suggestion">
        <template #remove-button-content>
          <span :aria-label="$t('historyQueries.removeLabel', { suggestion: suggestion.query })">
            x
          </span>
        </template>
      </HistoryQuery>
    </template>
  </HistoryQueries>
</template>

<script lang="ts">
import Vue from 'vue'
import { Component } from 'vue-property-decorator'
import { HistoryQueries, HistoryQuery } from '@empathyco/x-components/history-queries'

@Component({
  components: {
    HistoryQueries,
    HistoryQuery,
  },
})
export default class CustomHistoryQueries extends Vue {}
</script>
```

In this example, the `suggestion` message property takes the value of the `suggestion.query`, which
is retrieved from the `#suggestion` scoped slot.

### Making messages for a locale lazy loaded

If you have to support lots of locales, this would increase unnecessarily the initial bundle size.
Imagine that you need to translate the app for 20 different languages. Why would the german users
want to download the spanish, french, chinese languages... This would just slow down the download
and bootstrap of the app.

To do so, it is recommended to only load one set of messages synchronously. The rest of them is
better to have them loaded asynchronously, on demand. To make this work, just refactor the export
you wrote to a function with the name of the locale that performs a dynamic import. So if you
previously wrote:

```typescript
// File src/i18n/messages.index.ts
export { default as de } from './de.messages.ts'
```

Simply change it to:

```typescript
export const de = () => import('./de.messages')
```

Now the german language will be split to another chunk in the build, and will not increase the
initial bundle size. It will only be downloaded when it is needed.

### Changing the locale

There are some options in the reactive config where you can set the `lang` and the `uiLang`.

- `lang` is the language used for the API requests and the interface messages. It is a required
  parameter.
  - `window.initX = { ..., lang: "es" }`
- `uiLang` is the language for only the interface messages. It is an optional parameter that
  defaults to the `lang` value.
  - `window.initX = { ..., lang: "es", uiLang: "en" }`

Both values can be changed independently. For example, it is possible to search in English
(`lang="en"`) and show the static messages in Spanish (`uiLang="es"`).

### Managing i18n translations

We created a library for managing the i18n translations easier. It transforms the application i18n
files in .csv files that can be opened as _spreadsheet_ files which are more user-friendly; and vice
versa. This library is within the `empathyco` GitHub as
[x-translations](https://github.com/empathyco/x/tree/main/packages/x-translations).

It provides two commands to:

- Export translations from i18n files into a .csv files.
- Import translations from .csv files to i18n files.

The X Archetype includes two script commands (see [package.json](../package.json)):

- Exports each language file from `src/i18n/messages` into .csv files within the `output` folder by
  default.

  ```
  "json:csv": "json-csv ./src/i18n/messages",
  ```

- Imports .csv files from the same `output` folder and includes the translations in the
  `src/i18n/messages` files.

  ```
  "csv:json": "csv-json ./output ./src/i18n/messages"
  ```
