
/**
\brief Nearest Neighbours

k-Nearest-Neighbours (kNN) classification and regression,
as well as outlier detection / anomaly detection / novelty detection

Supports dynamic growth of the model data,
in order to support on-device learning.
*/

#include <eml_common.h>
#include <eml_log.h>
#include <eml_qsort.h>

#include <stdint.h>
#include <string.h>

// Configuration options
#ifndef EML_NEIGHBORS_LOG_LEVEL
#define EML_NEIGHBORS_LOG_LEVEL 0
#endif // EML_NEIGHBORS_LOG_LEVEL

#ifndef EML_NEIGHBORS_MAX_CLASSES
#define EML_NEIGHBORS_MAX_CLASSES 10
#endif // EML_NEIGHBORS_MAX_CLASSES


int32_t eml_isqrt(int32_t x)
{
    int32_t q = 1, r = 0;
    while (q <= x) {
        q <<= 2;
    }
    while (q > 1) {
        int32_t t;
        q >>= 2;
        t = x - r - q;
        r >>= 1;
        if (t >= 0) {
            x = t;
            r += q;
        }
    }
    return r;
}

uint32_t
eml_distance_euclidean_int16(const int16_t *a, const int16_t *b, int length)
{
    // FIXME: test this with large values
    uint32_t ret = 0;
    for (int i=0; i<length; i++) {
        const int32_t diff = a[i] - b[i];        
        const int32_t d = diff * diff;
        ret += d;
    }
    ret = eml_isqrt(ret); 
    return ret;
}

/** @typedef EmlNeighborsDistanceItem
* \brief Distance between input datapoint and a training datapoint
*
* Used during inference.
*/
typedef struct _EmlNeighborsDistanceItem {
    int16_t index;
    uint32_t distance;
} EmlNeighborsDistanceItem;

EmlError
eml_neighbors_sort_distances(EmlNeighborsDistanceItem *distances, size_t length)
{
    EmlNeighborsDistanceItem *A = distances;
    EmlNeighborsDistanceItem tmp;

#define EML_LESS(i, j) A[i].distance < A[j].distance
#define EML_SWAP(i, j) tmp = A[i], A[i] = A[j], A[j] = tmp

    QSORT(length, EML_LESS, EML_SWAP);

#undef EML_LESS
#undef EML_SWAP

    return EmlOk;
}

/** @typedef EmlNeighborsModel
* \brief Nearest Neighbors Model
*
* Handle used to do inference.
* Normally the initialization code is generated by emlearn.
*/
typedef struct _EmlNeighborsModel {

    uint16_t n_features;
    int16_t n_items;
    int16_t max_items;

    int16_t *data; // (n_items * n_features)
    int16_t *labels; // n_items

    int16_t k_neighbors;

} EmlNeighborsModel;

EmlError
eml_neighbors_check(EmlNeighborsModel *self,
        int16_t data_length, int16_t labels_length, int16_t distances_length)
{
    const int32_t expect_data_length = self->max_items * self->n_features;
    if (data_length < expect_data_length) {
        return EmlSizeMismatch;
    }
    if (labels_length < self->max_items) {
        return EmlSizeMismatch;
    }
    if (distances_length < self->max_items) {
        return EmlSizeMismatch;
    }
    return EmlOk;
}

/**
* \brief Add a datapoint to the model / training set
*
* \param self EmlNeighborsModel instance
* \param values Feature values for this datapoint
* \param values_length Length of feature data. Must equal model->n_features
* \param label The label to associate with this datapoint
*
* \return EmlOk on success, or -EmlError on failure
*/
EmlError
eml_neighbors_add_item(EmlNeighborsModel *self,
        const int16_t *values, int16_t values_length,
        int16_t label)
{
    EML_PRECONDITION(values_length == self->n_features, EmlSizeMismatch);
    EML_PRECONDITION(self->n_items < self->max_items, EmlSizeMismatch);

    const int index = self->n_items++;
    int16_t *data = self->data + (self->n_features * index);
    memcpy(data, values, sizeof(int16_t)*values_length);
    self->labels[index] = label;    

#if EML_NEIGHBORS_LOG_LEVEL > 2
    EML_LOG_BEGIN("eml_neighbors_add_item");
    EML_LOG_ADD_INTEGER("index", index);
    EML_LOG_ADD_INTEGER("label", label);
    EML_LOG_END();
#endif

    return EmlOk;
}

EmlError
eml_neighbors_infer(EmlNeighborsModel *self,
            const int16_t *features, int features_length,
            EmlNeighborsDistanceItem *distances, int distances_length)
{
    EML_PRECONDITION(distances_length >= self->n_items, EmlSizeMismatch);
    EML_PRECONDITION(features_length == self->n_features, EmlSizeMismatch);

    // compute distances to all items
    for (int i=0; i<self->n_items; i++) {
        int16_t *item = self->data + (self->n_features * i);
        uint32_t distance = eml_distance_euclidean_int16(features, item, features_length);

        distances[i].index = i;
        distances[i].distance = distance;

#if EML_NEIGHBORS_LOG_LEVEL > 2
        EML_LOG_BEGIN("eml_neighbors_infer_iter");
        EML_LOG_ADD_INTEGER("index", i);
        EML_LOG_ADD_INTEGER("distance", distance);
        EML_LOG_ADD_INTEGER("label", self->labels[i]);
        EML_LOG_END();
#endif

    }

    return EmlOk;
}

// FIXME: avoid hardcoding length
int16_t eml_neighbors_votes[EML_NEIGHBORS_MAX_CLASSES];

EmlError
eml_neighbors_find_nearest(EmlNeighborsModel *self,
        EmlNeighborsDistanceItem *distances, int distances_length,
        int k, int16_t *out)
{
    EML_PRECONDITION(k <= distances_length, EmlSizeMismatch);

#if EML_NEIGHBORS_LOG_LEVEL > 1
        EML_LOG_BEGIN("eml_neighbors_find_nearest_start");
        EML_LOG_ADD_INTEGER("distances", distances_length);
        EML_LOG_ADD_INTEGER("items", distances_length);
        EML_LOG_ADD_INTEGER("k", k);
        EML_LOG_END();
#endif

    // argsort by distance. NOTE: sorts in-place
    eml_neighbors_sort_distances(distances, distances_length);

    int16_t *votes = eml_neighbors_votes;
    memset(votes, 0, sizeof(int16_t) * EML_NEIGHBORS_MAX_CLASSES);

    // merge predictions for top-k matches
    for (int i=0; i<k; i++) {
        EmlNeighborsDistanceItem d = distances[i];
        const int16_t label = self->labels[d.index];         
        if (label < 0 || label >= EML_NEIGHBORS_MAX_CLASSES) {
            return EmlUnknownError;
        }
        votes[label] += 1;

#if EML_NEIGHBORS_LOG_LEVEL > 2
        EML_LOG_BEGIN("eml_neighbors_find_nearest_k_iter");
        EML_LOG_ADD_INTEGER("index", i);
        EML_LOG_ADD_INTEGER("distance", d.distance);
        EML_LOG_ADD_INTEGER("label", label);
        EML_LOG_END();
#endif
    }

    int32_t most_voted_class = -1;
    int32_t most_voted_votes = 0;
    for (int32_t i=0; i<EML_NEIGHBORS_MAX_CLASSES; i++) {

#if EML_NEIGHBORS_LOG_LEVEL > 1
        EML_LOG_BEGIN("eml_neighbors_find_nearest_votes_iter");
        EML_LOG_ADD_INTEGER("index", i);
        EML_LOG_ADD_INTEGER("votes", votes[i]);
        EML_LOG_END();
#endif

        if (votes[i] > most_voted_votes) {
            most_voted_class = i;
            most_voted_votes = votes[i];
        }
    }
    *out = most_voted_class;

    return EmlOk;
}

/**
* \brief Run inference and return most probable class
*
* \param self EmlNeighborsModel instance
* \param features Input data values
* \param features_length Length of input data
* \param distances Array to use for storing distances
* \param distances_length Length of distance array. Must be larger than model->n_items
* \param out Location to return predicted class label
*
* \return EmlOk on success, or -EmlError on failure
*/
EmlError
eml_neighbors_predict(EmlNeighborsModel *self,
        const int16_t *features, int features_length,
        EmlNeighborsDistanceItem *distances, int distances_length,
        int16_t *out)
{
    // NOTE: Preconditions checked inside _infer()
    // Compute distances
    const EmlError infer_err = \
        eml_neighbors_infer(self, features, features_length, distances, distances_length);
    if (infer_err != EmlOk) {
        return infer_err;
    }

    // Find kNN predictions 
    int16_t label = -1;
    EmlError find_err = \
        eml_neighbors_find_nearest(self, distances, self->n_items, self->k_neighbors, &label);
    if (find_err != EmlOk) {
        return find_err;
    }
    *out = label;

    return EmlOk;
}

