namespace DotNet.Sdk.Extensions.Tests.Polly.Http.CircuitBreaker.Extensions;

// For now, temporarily ignore the error about using the obsolete `HttpClientBuilderExtensions.ConfigureHttpMessageHandlerBuilder`
// method. I'm likely to drop support for most of the HttpClient resilience extensions since .net 8 has that built in.
#pragma warning disable CS0618 // Type or member is obsolete

/// <summary>
/// Tests for the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions"/> class.
///
/// NOTE: when debugging sometimes the tests might not behave as expected because the circuit breaker
/// is time sensitive in its nature as shown by the duration of break and sampling duration properties.
/// If required try to increase the CircuitBreakerOptions.SamplingDurationInSecs to a greater value to
/// allow the tests to run successfully when the debugger is attached. For instance, set it to 2 instead of 0.2.
///
/// This is not ideal but at the moment it's the only suggested workaround because these tests are triggering
/// the circuit breaker policy and I don't know how to manipulate/fake time for the policy.
/// </summary>
[Trait("Category", XUnitCategories.Polly)]
public class AddCircuitBreakerPolicyTests
{
    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy(IHttpClientBuilder,Action{CircuitBreakerOptions})"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy1()
    {
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        await httpClient
            .CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler)
            .HttpClientShouldContainCircuitBreakerPolicyAsync();
    }

    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy(IHttpClientBuilder,string)"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy2()
    {
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        const string optionsName = "GitHubOptions";
        var services = new ServiceCollection();
        services
            .AddHttpClientCircuitBreakerOptions(optionsName)
            .Configure(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            });
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy(optionsName)
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        await httpClient
            .CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler)
            .HttpClientShouldContainCircuitBreakerPolicyAsync();
    }

    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy{TPolicyEventHandler}(IHttpClientBuilder,Action{CircuitBreakerOptions})"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    ///
    /// This also tests that the <see cref="ICircuitBreakerPolicyEventHandler"/> events are triggered with the correct values.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy3()
    {
        var circuitBreakerPolicyEventHandlerCalls = new CircuitBreakerPolicyEventHandlerCalls();
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services.AddSingleton(circuitBreakerPolicyEventHandlerCalls);
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy<TestCircuitBreakerPolicyEventHandler>(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        var circuitBreakerAsserter = httpClient.CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler);
        await circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync();
        circuitBreakerAsserter.EventHandlerShouldReceiveExpectedEvents(
            count: 15, // the circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync triggers the circuit breaker 15 times
            httpClientName: httpClientName,
            eventHandlerCalls: circuitBreakerPolicyEventHandlerCalls);
    }

    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy{TPolicyEventHandler}(IHttpClientBuilder,string)"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    ///
    /// This also tests that the <see cref="ICircuitBreakerPolicyEventHandler"/> events are triggered with the correct values.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy4()
    {
        var circuitBreakerPolicyEventHandlerCalls = new CircuitBreakerPolicyEventHandlerCalls();
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        const string optionsName = "GitHubOptions";

        var services = new ServiceCollection();
        services.AddSingleton(circuitBreakerPolicyEventHandlerCalls);
        services
            .AddHttpClientCircuitBreakerOptions(optionsName)
            .Configure(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            });
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy<TestCircuitBreakerPolicyEventHandler>(optionsName)
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        var circuitBreakerAsserter = httpClient.CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler);
        await circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync();
        circuitBreakerAsserter.EventHandlerShouldReceiveExpectedEvents(
            count: 15, // the circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync triggers the circuit breaker 15 times
            httpClientName: httpClientName,
            eventHandlerCalls: circuitBreakerPolicyEventHandlerCalls);
    }

    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy(IHttpClientBuilder,string,Func{IServiceProvider,ICircuitBreakerPolicyEventHandler})"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    ///
    /// This also tests that the <see cref="ICircuitBreakerPolicyEventHandler"/> events are triggered with the correct values.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy5()
    {
        var circuitBreakerPolicyEventHandlerCalls = new CircuitBreakerPolicyEventHandlerCalls();
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        const string optionsName = "GitHubOptions";

        var services = new ServiceCollection();
        services
            .AddHttpClientCircuitBreakerOptions(optionsName)
            .Configure(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            });
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy(optionsName, _ =>
            {
                return new TestCircuitBreakerPolicyEventHandler(circuitBreakerPolicyEventHandlerCalls);
            })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        var circuitBreakerAsserter = httpClient.CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler);
        await circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync();
        circuitBreakerAsserter.EventHandlerShouldReceiveExpectedEvents(
            count: 15, // the circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync triggers the circuit breaker 15 times
            httpClientName: httpClientName,
            eventHandlerCalls: circuitBreakerPolicyEventHandlerCalls);
    }

    /// <summary>
    /// Tests that the <see cref="CircuitBreakerPolicyHttpClientBuilderExtensions.AddCircuitBreakerPolicy(IHttpClientBuilder,Action{CircuitBreakerOptions},Func{IServiceProvider,ICircuitBreakerPolicyEventHandler})"/>
    /// overload method adds a <see cref="DelegatingHandler"/> with a circuit break to the <see cref="HttpClient"/>.
    ///
    /// This also tests that the <see cref="ICircuitBreakerPolicyEventHandler"/> events are triggered with the correct values.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy6()
    {
        var circuitBreakerPolicyEventHandlerCalls = new CircuitBreakerPolicyEventHandlerCalls();
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy(
                configureOptions: options =>
                {
                    options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                    options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                    options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                    options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
                },
                eventHandlerFactory: _ =>
                {
                    return new TestCircuitBreakerPolicyEventHandler(circuitBreakerPolicyEventHandlerCalls);
                })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        var circuitBreakerAsserter = httpClient.CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler);
        await circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync();
        circuitBreakerAsserter.EventHandlerShouldReceiveExpectedEvents(
            count: 15, // the circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync triggers the circuit breaker 15 times
            httpClientName: httpClientName,
            eventHandlerCalls: circuitBreakerPolicyEventHandlerCalls);
    }

    /// <summary>
    /// This tests that all the <see cref="ICircuitBreakerPolicyEventHandler"/> events are triggered with the correct values.
    /// Previous tests would only assert the break and reset events from the circuit breaker event handler.
    /// This is because to check the on half open events takes time and instead of having all the circuit breaker related
    /// tests be slow, only this one is.
    ///
    /// This test is able to check the on half open events because the reset type on the circuit breaker is set to CircuitBreakerPolicyExecutorResetTypes.Normal.
    /// </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicy7()
    {
        var circuitBreakerPolicyEventHandlerCalls = new CircuitBreakerPolicyEventHandlerCalls();
        var testHttpMessageHandler = new TestHttpMessageHandler();
        const string httpClientName = "GitHub";
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services.AddSingleton(circuitBreakerPolicyEventHandlerCalls);
        services
            .AddHttpClient(httpClientName)
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy<TestCircuitBreakerPolicyEventHandler>(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient(httpClientName);
        var circuitBreakerAsserter = httpClient
            .CircuitBreakerPolicyAsserter(circuitBreakerOptions, testHttpMessageHandler)
            .WithReset(CircuitBreakerPolicyExecutorResetTypes.Normal);
        await circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync();
        circuitBreakerAsserter.EventHandlerShouldReceiveExpectedEvents(
            count: 15, // the circuitBreakerAsserter.HttpClientShouldContainCircuitBreakerPolicyAsync triggers the circuit breaker 15 times
            httpClientName: httpClientName,
            eventHandlerCalls: circuitBreakerPolicyEventHandlerCalls);
    }

    /// <summary>
    /// This tests that the policies added to the <see cref="HttpClient"/> by the
    /// CircuitBreakerHttpClientBuilderExtensions.AddCircuitBreakerPolicy methods are unique.
    ///
    /// Policies should NOT be the same between HttpClients or else when one HttpClient triggers
    /// the policy it would trigger for all.
    /// </summary>
    [Fact]
    public void AddCircuitBreakerPolicyUniquePolicyPerHttpClient()
    {
        AsyncPolicyWrap<HttpResponseMessage>? circuitBreakerPolicy1 = null;
        AsyncPolicyWrap<HttpResponseMessage>? circuitBreakerPolicy2 = null;
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services
            .AddHttpClient("GitHub")
            .AddCircuitBreakerPolicy(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigureHttpMessageHandlerBuilder(httpMessageHandlerBuilder =>
            {
                circuitBreakerPolicy1 = httpMessageHandlerBuilder
                    .GetPolicies<AsyncPolicyWrap<HttpResponseMessage>>()
                    .FirstOrDefault();
            });
        services
            .AddHttpClient("Microsoft")
            .AddCircuitBreakerPolicy(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigureHttpMessageHandlerBuilder(httpMessageHandlerBuilder =>
            {
                circuitBreakerPolicy2 = httpMessageHandlerBuilder
                    .GetPolicies<AsyncPolicyWrap<HttpResponseMessage>>()
                    .FirstOrDefault();
            });

        var serviceProvider = services.BuildServiceProvider();
        serviceProvider.InstantiateNamedHttpClient("GitHub");
        serviceProvider.InstantiateNamedHttpClient("Microsoft");
        circuitBreakerPolicy1.ShouldNotBeNull();
        circuitBreakerPolicy2.ShouldNotBeNull();
        ReferenceEquals(circuitBreakerPolicy1, circuitBreakerPolicy2).ShouldBeFalse();
        circuitBreakerPolicy1.PolicyKey.ShouldNotBe(circuitBreakerPolicy2.PolicyKey);
    }

    /// <summary>
    /// This tests that the policies added to the <see cref="HttpClient"/> by the
    /// CircuitBreakerHttpClientBuilderExtensions.AddCircuitBreakerPolicy methods do not trigger
    /// an exception if the circuit state is open or isolated.
    ///
    /// This is because the policy added is a wrapped policy which joins an <see cref="AsyncCircuitBreakerPolicy{T}"/>
    /// and a <see cref="CircuitBreakerCheckerAsyncPolicy{T}"/>.
    /// The <see cref="CircuitBreakerCheckerAsyncPolicy{T}"/>. will check if the circuit is open/isolated and
    /// if so it will return <see cref="CircuitBrokenHttpResponseMessage"/> which is an http response message
    /// with 500 status code and some extra properties.
    ///
    /// This is to optimize the performance of the code by reducing the exceptions thrown as indicated by
    /// https://github.com/App-vNext/Polly/wiki/Circuit-Breaker#reducing-thrown-exceptions-when-the-circuit-is-broken.
    ///  </summary>
    [Fact]
    public async Task AddCircuitBreakerPolicyDoesNotThrowExceptionWhenCircuitIsOpen()
    {
        var testHttpMessageHandler = new TestHttpMessageHandler();
        var circuitBreakerOptions = new CircuitBreakerOptions
        {
            DurationOfBreakInSecs = 0.15,
            SamplingDurationInSecs = 0.3,
            FailureThreshold = 0.6,
            MinimumThroughput = 5,
        };
        var services = new ServiceCollection();
        services
            .AddHttpClient("GitHub")
            .ConfigureHttpClient(client => client.BaseAddress = new Uri("https://github.com"))
            .AddCircuitBreakerPolicy(options =>
            {
                options.DurationOfBreakInSecs = circuitBreakerOptions.DurationOfBreakInSecs;
                options.SamplingDurationInSecs = circuitBreakerOptions.SamplingDurationInSecs;
                options.FailureThreshold = circuitBreakerOptions.FailureThreshold;
                options.MinimumThroughput = circuitBreakerOptions.MinimumThroughput;
            })
            .ConfigurePrimaryHttpMessageHandler(() => testHttpMessageHandler);

        var serviceProvider = services.BuildServiceProvider();
        var httpClient = serviceProvider.InstantiateNamedHttpClient("GitHub");
        await using var circuitBreaker = httpClient.CircuitBreakerExecutor(circuitBreakerOptions, testHttpMessageHandler);
        await circuitBreaker.TriggerFromTransientHttpStatusCodeAsync(HttpStatusCode.ServiceUnavailable);
        // after the above the circuit state is now open which means the following http request, if it hit the
        // circuit breaker policy, would throw a BrokenCircuitException/IsolatedCircuitException; however,
        // because we wrapped the circuit breaker policy with a CircuitBreakerCheckerAsyncPolicy what we get
        // instead is a CircuitBrokenHttpResponseMessage instance
        var response = await httpClient.GetAsync($"/circuit-breaker/transient-http-status-code/{nameof(HttpStatusCode.ServiceUnavailable)}");
        var circuitBrokenHttpResponseMessage = response as CircuitBrokenHttpResponseMessage;
        circuitBrokenHttpResponseMessage.ShouldNotBeNull();
        circuitBrokenHttpResponseMessage.StatusCode.ShouldBe(HttpStatusCode.InternalServerError);
        circuitBrokenHttpResponseMessage.CircuitBreakerState.ShouldBe(CircuitBreakerState.Open);
        // the Exception property should be null because the circuit breaker should NOT be throwing an exception.
        // The CircuitBreakerCheckerAsyncPolicy should check that the circuit is open and return the
        // CircuitBrokenHttpResponseMessage without any exception
        circuitBrokenHttpResponseMessage.Exception.ShouldBeNull();
    }
}

#pragma warning restore CS0618 // Type or member is obsolete
