<?php

namespace EFrane\ConsoleAdditions\Batch;

use Symfony\Component\Console\Command\Command;

/**
 * Return Code Stack
 *
 * Batches generate a return value for each step.
 * To make it possible to reason over the return value
 * of a Batch, the default behaviour is to return the
 * one generated by the last executed action.
 *
 * After a `Batch::run()` or `Batch::runSilent()`,
 * `Batch::getReturnValueStack()` will contain
 * all collected return values in order.
 */
final class ReturnCodeStack
{
    const NO_CODE_YET = -1;
    /**
     * @var int[]
     */
    protected array $returnCodes = [];

    public function push(int $returnCode): void
    {
        $this->returnCodes[] = $returnCode;
    }

    public function getLastReturnCode(): int
    {
        if (count($this->returnCodes) === 0) {
            return self::NO_CODE_YET;
        }

        return $this->returnCodes[array_key_last($this->returnCodes)];
    }

    public function anyErrored(): bool
    {
        return !$this->allSuccessful();
    }

    public function allSuccessful(): bool
    {
        $successfulCodes = array_filter($this->returnCodes, fn(int $returnCode) => $returnCode === Command::SUCCESS);

        return count($successfulCodes) === count($this->returnCodes);
    }

    /**
     * @return int[]
     */
    public function all(): array
    {
        return $this->returnCodes;
    }
}
