// -----------------------------------------------------------------------
// Licensed to The .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// -----------------------------------------------------------------------

using System;
using System.Linq;
using System.Security.Cryptography.Pkcs;
using Kerberos.NET.Crypto;
using Kerberos.NET.Entities;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Tests.Kerberos.NET
{
    [TestClass]
    public class PKInitTests
    {
        [TestMethod]
        public void ParsePaPkAsReq()
        {
            KrbPaPkAsReq asreq = KrbPaPkAsReq.Decode(this.signedPkAsReq);

            Assert.IsNotNull(asreq);
        }

        [TestMethod]
        public void ParsePaPkAsReq_SignedAuthPack()
        {
            KrbPaPkAsReq asreq = KrbPaPkAsReq.Decode(this.signedPkAsReq);

            SignedCms signedCms = new SignedCms();
            signedCms.Decode(asreq.SignedAuthPack.ToArray());
            signedCms.CheckSignature(verifySignatureOnly: true);
        }

        [TestMethod]
        public void ParsePaPkAsReq_SignedAuthPack_ParseAuthPack()
        {
            KrbPaPkAsReq asreq = KrbPaPkAsReq.Decode(this.signedPkAsReq);

            SignedCms signedCms = new SignedCms();
            signedCms.Decode(asreq.SignedAuthPack.ToArray());
            signedCms.CheckSignature(verifySignatureOnly: true);

            KrbAuthPack authPack = KrbAuthPack.Decode(signedCms.ContentInfo.Content);

            Assert.IsNotNull(authPack);

            var param = authPack.ClientPublicValue.Algorithm.Parameters.Value;

            var domainParams = KrbDiffieHellmanDomainParameters.DecodeSpecial(param);

            Assert.IsNotNull(domainParams);

            var special = domainParams.EncodeSpecial();

            Assert.IsTrue(special.Span.SequenceEqual(param.ToArray()));
            _ = CryptEncode.CryptDecodePublicParameter(authPack.ClientPublicValue.SubjectPublicKey).Slice(16);
        }

        [TestMethod]
        public void ParsePaPkAsRep()
        {
            KrbPaPkAsRep asrep = KrbPaPkAsRep.Decode(this.signedPkAsRep);

            Assert.IsNotNull(asrep);
        }

        [TestMethod]
        public void ParsePaPkAsRep_SignedDHRep()
        {
            KrbPaPkAsRep asrep = KrbPaPkAsRep.Decode(this.signedPkAsRep);

            Assert.IsNotNull(asrep);

            SignedCms signed = new SignedCms();
            signed.Decode(asrep.DHInfo.DHSignedData.ToArray());
            signed.CheckSignature(verifySignatureOnly: true);
        }

        [TestMethod]
        public void ParsePaPkAsRep_SignedDHRep_KDCDHKeyInfo()
        {
            KrbPaPkAsRep asrep = KrbPaPkAsRep.Decode(this.signedPkAsRep);

            Assert.IsNotNull(asrep);

            SignedCms signed = new SignedCms();
            signed.Decode(asrep.DHInfo.DHSignedData.ToArray());
            signed.CheckSignature(verifySignatureOnly: true);

            KrbKdcDHKeyInfo keyInfo = KrbKdcDHKeyInfo.Decode(signed.ContentInfo.Content);

            Assert.IsNotNull(keyInfo);
        }

        private readonly byte[] signedPkAsRep = new byte[]
        {
0xa0, 0x82, 0x09, 0x06, 0x30, 0x82, 0x09, 0x02, 0x80, 0x82, 0x08, 0xda, 0x30, 0x82, 0x08, 0xd6,
0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x02, 0xa0, 0x82, 0x08, 0xc7, 0x30,
0x82, 0x08, 0xc3, 0x02, 0x01, 0x03, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02,
0x1a, 0x05, 0x00, 0x30, 0x81, 0xb8, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x02, 0xa0,
0x81, 0xac, 0x04, 0x81, 0xa9, 0x30, 0x81, 0xa6, 0xa0, 0x81, 0x88, 0x03, 0x81, 0x85, 0x00, 0x02,
0x81, 0x81, 0x00, 0x93, 0x9e, 0x04, 0xe2, 0x24, 0x79, 0xe9, 0xee, 0xe6, 0x7d, 0xe6, 0x06, 0xec,
0x7d, 0xcc, 0x66, 0xef, 0x4c, 0x49, 0xdd, 0xf8, 0x3b, 0x17, 0x0a, 0x4d, 0xb3, 0xef, 0xef, 0x27,
0x39, 0xb2, 0xf2, 0xcc, 0x15, 0x3c, 0x50, 0xec, 0x29, 0xe3, 0xbd, 0x69, 0x19, 0xd0, 0x1d, 0x23,
0x17, 0xe5, 0x2a, 0x0c, 0x6d, 0x22, 0x9b, 0x9a, 0xb5, 0x9c, 0xbd, 0x19, 0x9e, 0xbc, 0xb8, 0xd0,
0xc3, 0x23, 0x6a, 0x1b, 0x99, 0x21, 0x35, 0x0b, 0x87, 0xb9, 0x77, 0x79, 0x5b, 0xd8, 0x07, 0x1c,
0x82, 0x22, 0x9a, 0xb1, 0xe4, 0x99, 0xd4, 0x42, 0x91, 0xa0, 0x5f, 0x64, 0xea, 0xbb, 0x8c, 0xfa,
0xeb, 0xb9, 0x68, 0x16, 0x18, 0xf3, 0x98, 0x0d, 0xa5, 0x44, 0xe7, 0x46, 0xa8, 0x53, 0xdd, 0x6f,
0x34, 0x74, 0x76, 0xb4, 0xc6, 0x81, 0xd3, 0x2b, 0xb3, 0xdf, 0xc1, 0xaa, 0x45, 0xe9, 0x61, 0x26,
0x16, 0x11, 0x2b, 0xa1, 0x06, 0x02, 0x04, 0x44, 0x1d, 0x51, 0xa9, 0xa2, 0x11, 0x18, 0x0f, 0x32,
0x30, 0x31, 0x39, 0x31, 0x32, 0x32, 0x30, 0x31, 0x38, 0x34, 0x34, 0x34, 0x30, 0x5a, 0xa0, 0x82,
0x06, 0x0b, 0x30, 0x82, 0x06, 0x07, 0x30, 0x82, 0x04, 0xef, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02,
0x13, 0x2c, 0x00, 0x00, 0x00, 0x10, 0x91, 0x7d, 0x43, 0x02, 0x20, 0x63, 0x9c, 0xae, 0x00, 0x00,
0x00, 0x00, 0x00, 0x10, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
0x0b, 0x05, 0x00, 0x30, 0x68, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93,
0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f, 0x6d, 0x31, 0x24, 0x30, 0x22, 0x06, 0x0a,
0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x14, 0x69, 0x64, 0x65, 0x6e,
0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e,
0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19,
0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
0x0c, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43, 0x41, 0x30, 0x31, 0x2d, 0x43, 0x41, 0x30, 0x1e, 0x17,
0x0d, 0x31, 0x39, 0x31, 0x31, 0x32, 0x37, 0x32, 0x31, 0x31, 0x31, 0x35, 0x36, 0x5a, 0x17, 0x0d,
0x32, 0x30, 0x31, 0x31, 0x32, 0x36, 0x32, 0x31, 0x31, 0x31, 0x35, 0x36, 0x5a, 0x30, 0x00, 0x30,
0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00,
0xae, 0xac, 0xe4, 0xe5, 0x36, 0xff, 0x35, 0x38, 0xb8, 0x3e, 0x66, 0xf9, 0x5b, 0x95, 0xf5, 0xed,
0xe7, 0x83, 0x80, 0xfc, 0x01, 0x82, 0xbb, 0x36, 0x7b, 0x5c, 0x8b, 0x13, 0x34, 0x7e, 0x0e, 0x60,
0xb2, 0xe3, 0x8d, 0x57, 0xf8, 0xbe, 0x7a, 0xc2, 0x16, 0xa4, 0x6d, 0x6a, 0xd5, 0x6e, 0x2e, 0x38,
0x8d, 0xd9, 0x15, 0x83, 0x24, 0x60, 0x3e, 0xdb, 0xb3, 0xbc, 0x4f, 0xd8, 0x4d, 0x9f, 0x7d, 0xb2,
0x39, 0x4b, 0xe7, 0x20, 0x74, 0xe3, 0xcd, 0x18, 0xe6, 0x1c, 0xb1, 0x60, 0xb1, 0xac, 0x12, 0x4b,
0xf6, 0x44, 0xab, 0xed, 0x94, 0xfb, 0x1b, 0xb5, 0x53, 0x9f, 0x65, 0x50, 0x92, 0x92, 0x62, 0x8d,
0x3c, 0xe7, 0x4f, 0xe7, 0x18, 0x29, 0x49, 0x1f, 0x85, 0x30, 0xce, 0xde, 0x28, 0x26, 0x41, 0x91,
0x98, 0x31, 0x20, 0x6e, 0x9d, 0x60, 0x99, 0xef, 0x2a, 0xbd, 0x56, 0x57, 0xc6, 0x03, 0xd0, 0x7c,
0xe4, 0x32, 0xe0, 0x1c, 0xfe, 0xd6, 0x83, 0xf2, 0x2a, 0x86, 0x26, 0x2b, 0x25, 0x73, 0x96, 0x63,
0x99, 0x06, 0x0c, 0x16, 0xa0, 0x0f, 0x22, 0xad, 0xc3, 0xd4, 0x37, 0x1d, 0xd5, 0xb3, 0xb4, 0x18,
0x1b, 0x6e, 0xd5, 0xfa, 0x8c, 0x5c, 0xda, 0xa8, 0xa1, 0x63, 0xf3, 0xdd, 0x03, 0x4c, 0xdd, 0xdb,
0xfa, 0x80, 0x69, 0xea, 0x6d, 0xdb, 0xb9, 0xad, 0xc5, 0xcf, 0x0b, 0xce, 0xe4, 0x4e, 0xd0, 0xaa,
0x5f, 0x2d, 0x16, 0x93, 0x17, 0xd0, 0xa8, 0xe6, 0x84, 0xb3, 0xe0, 0x62, 0xaa, 0xeb, 0x1f, 0x26,
0xde, 0x8b, 0x23, 0x17, 0xe3, 0xc7, 0x5e, 0xcc, 0x34, 0x20, 0x89, 0x2f, 0x68, 0x0e, 0x69, 0x61,
0x79, 0x5e, 0xfa, 0xbe, 0xd4, 0xa5, 0x80, 0x8e, 0x56, 0x1d, 0x97, 0x77, 0x48, 0x0b, 0x56, 0xfe,
0xe9, 0xea, 0x4a, 0xa6, 0xd4, 0x0e, 0x00, 0x1c, 0x49, 0x1c, 0x0b, 0x08, 0x8b, 0x97, 0x63, 0x2d,
0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x03, 0x10, 0x30, 0x82, 0x03, 0x0c, 0x30, 0x38, 0x06,
0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x07, 0x04, 0x2b, 0x30, 0x29, 0x06, 0x21,
0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x08, 0x82, 0xc6, 0xb2, 0x56, 0x85, 0xd5, 0xc3,
0x44, 0x83, 0xa9, 0x95, 0x18, 0x84, 0xdd, 0x91, 0x66, 0x83, 0x88, 0xb4, 0x4e, 0x81, 0x52, 0x01,
0x21, 0x02, 0x01, 0x6e, 0x02, 0x01, 0x00, 0x30, 0x32, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x04, 0x2b,
0x30, 0x29, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x06, 0x08, 0x2b, 0x06,
0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14,
0x02, 0x02, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x05, 0x30, 0x0e, 0x06, 0x03, 0x55,
0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x05, 0xa0, 0x30, 0x40, 0x06, 0x09, 0x2b,
0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x0a, 0x04, 0x33, 0x30, 0x31, 0x30, 0x0a, 0x06, 0x08,
0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05,
0x05, 0x07, 0x03, 0x01, 0x30, 0x0c, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14,
0x02, 0x02, 0x30, 0x09, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x05, 0x30, 0x1d, 0x06,
0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x08, 0x5e, 0x15, 0x64, 0x96, 0x56, 0x6d, 0x3a,
0x20, 0x5a, 0xe7, 0x90, 0xb8, 0xb2, 0x57, 0x11, 0x95, 0x3e, 0x13, 0x12, 0x30, 0x1f, 0x06, 0x03,
0x55, 0x1d, 0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0xed, 0x80, 0x47, 0xbd, 0xb1, 0x11, 0xf3,
0x80, 0x20, 0x18, 0x37, 0xd6, 0xa3, 0xc5, 0x38, 0xb2, 0x2f, 0xf2, 0x15, 0x05, 0x30, 0x81, 0xdc,
0x06, 0x03, 0x55, 0x1d, 0x1f, 0x04, 0x81, 0xd4, 0x30, 0x81, 0xd1, 0x30, 0x81, 0xce, 0xa0, 0x81,
0xcb, 0xa0, 0x81, 0xc8, 0x86, 0x81, 0xc5, 0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x2f, 0x43,
0x4e, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43, 0x41, 0x30, 0x31, 0x2d, 0x43, 0x41, 0x2c, 0x43,
0x4e, 0x3d, 0x63, 0x61, 0x30, 0x31, 0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x44, 0x50, 0x2c, 0x43, 0x4e,
0x3d, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x25, 0x32, 0x30, 0x4b, 0x65, 0x79, 0x25, 0x32, 0x30,
0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x53, 0x65, 0x72, 0x76,
0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72,
0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2c, 0x44, 0x43,
0x3d, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65,
0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x63, 0x65, 0x72,
0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x52, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x74, 0x69,
0x6f, 0x6e, 0x4c, 0x69, 0x73, 0x74, 0x3f, 0x62, 0x61, 0x73, 0x65, 0x3f, 0x6f, 0x62, 0x6a, 0x65,
0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x63, 0x52, 0x4c, 0x44, 0x69, 0x73, 0x74, 0x72,
0x69, 0x62, 0x75, 0x74, 0x69, 0x6f, 0x6e, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x30, 0x81, 0xd3, 0x06,
0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x81, 0xc6, 0x30, 0x81, 0xc3, 0x30,
0x81, 0xc0, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x81, 0xb3, 0x6c,
0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x2f, 0x43, 0x4e, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43,
0x41, 0x30, 0x31, 0x2d, 0x43, 0x41, 0x2c, 0x43, 0x4e, 0x3d, 0x41, 0x49, 0x41, 0x2c, 0x43, 0x4e,
0x3d, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x25, 0x32, 0x30, 0x4b, 0x65, 0x79, 0x25, 0x32, 0x30,
0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x53, 0x65, 0x72, 0x76,
0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72,
0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2c, 0x44, 0x43,
0x3d, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65,
0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x63, 0x41, 0x43,
0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x3f, 0x62, 0x61, 0x73, 0x65, 0x3f,
0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x63, 0x65, 0x72, 0x74,
0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69,
0x74, 0x79, 0x30, 0x55, 0x06, 0x03, 0x55, 0x1d, 0x11, 0x01, 0x01, 0xff, 0x04, 0x4b, 0x30, 0x49,
0x82, 0x22, 0x64, 0x63, 0x30, 0x32, 0x2e, 0x63, 0x6f, 0x72, 0x70, 0x2e, 0x69, 0x64, 0x65, 0x6e,
0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e,
0x2e, 0x63, 0x6f, 0x6d, 0x82, 0x1d, 0x63, 0x6f, 0x72, 0x70, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74,
0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x2e,
0x63, 0x6f, 0x6d, 0x82, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0xf2, 0x5c, 0xb8,
0xfa, 0x95, 0x52, 0xed, 0xd7, 0xa9, 0x6c, 0x02, 0xdc, 0x18, 0xd9, 0xee, 0x41, 0x19, 0x73, 0xa4,
0xde, 0x33, 0x89, 0xfa, 0xbe, 0x57, 0x45, 0x67, 0x4e, 0x50, 0xe5, 0xca, 0x85, 0x5d, 0x12, 0x1a,
0xdc, 0xb5, 0x9f, 0x9f, 0x35, 0x44, 0x07, 0x9b, 0x96, 0x37, 0x91, 0x8d, 0x09, 0x23, 0x5c, 0xf9,
0x72, 0xa2, 0x1a, 0x32, 0xfd, 0xf4, 0x6c, 0xa0, 0x9f, 0x51, 0xbb, 0xbd, 0x91, 0x51, 0x04, 0x62,
0x78, 0xd1, 0x1e, 0xf2, 0x62, 0x88, 0x7c, 0x9e, 0x47, 0xe1, 0xb7, 0x01, 0x37, 0x73, 0x0d, 0x1b,
0x01, 0x3a, 0x16, 0x65, 0x4f, 0xe9, 0x19, 0x8b, 0xdb, 0x58, 0xbe, 0xd7, 0x9d, 0xbc, 0xcb, 0xa3,
0xc2, 0x38, 0x13, 0x56, 0x31, 0x82, 0xde, 0x49, 0xd0, 0x18, 0x02, 0x09, 0xaa, 0xf1, 0xf5, 0xbb,
0x43, 0xe7, 0x1e, 0xea, 0x02, 0xba, 0x37, 0x55, 0xde, 0x04, 0x4d, 0x1c, 0x06, 0xd2, 0xa9, 0x7f,
0xd5, 0xd2, 0x56, 0x52, 0x7d, 0x63, 0x0e, 0x07, 0xa0, 0xb9, 0xab, 0xeb, 0xef, 0x4a, 0xa4, 0xa3,
0x84, 0xe9, 0x2b, 0x0c, 0xea, 0xcb, 0xa4, 0xba, 0xcd, 0x77, 0x7a, 0xce, 0x73, 0x34, 0xeb, 0xcd,
0x31, 0x48, 0xd9, 0x04, 0x93, 0xf0, 0x67, 0xf3, 0xd8, 0x0b, 0xe5, 0x16, 0x55, 0x57, 0x14, 0xf8,
0xe6, 0x52, 0x62, 0x57, 0x53, 0xe8, 0x30, 0x9e, 0x38, 0x91, 0x65, 0xec, 0x78, 0x7e, 0xf1, 0x64,
0x48, 0x3d, 0x8f, 0x31, 0x7f, 0x40, 0xca, 0xeb, 0xd3, 0x2c, 0xde, 0x50, 0xb0, 0xf7, 0x7d, 0x52,
0xdf, 0xd2, 0xde, 0x49, 0xdd, 0x24, 0xc5, 0x00, 0x3e, 0x5f, 0x25, 0x60, 0x7f, 0xff, 0x4c, 0xc9,
0x0f, 0x9a, 0x31, 0x5c, 0x50, 0xbf, 0x4d, 0xe5, 0xe3, 0x1c, 0xe6, 0xc1, 0xa3, 0xae, 0x89, 0xc6,
0x7f, 0xa9, 0x98, 0x7e, 0xb8, 0x6f, 0x3f, 0x2b, 0x2a, 0xc6, 0x50, 0x2c, 0xf6, 0x31, 0x82, 0x01,
0xe5, 0x30, 0x82, 0x01, 0xe1, 0x02, 0x01, 0x01, 0x30, 0x7f, 0x30, 0x68, 0x31, 0x13, 0x30, 0x11,
0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f,
0x6d, 0x31, 0x24, 0x30, 0x22, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
0x19, 0x16, 0x14, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72,
0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92, 0x26,
0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x15, 0x30,
0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0c, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43, 0x41, 0x30,
0x31, 0x2d, 0x43, 0x41, 0x02, 0x13, 0x2c, 0x00, 0x00, 0x00, 0x10, 0x91, 0x7d, 0x43, 0x02, 0x20,
0x63, 0x9c, 0xae, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03,
0x02, 0x1a, 0x05, 0x00, 0xa0, 0x3d, 0x30, 0x16, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
0x01, 0x09, 0x03, 0x31, 0x09, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x02, 0x30, 0x23,
0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04, 0x31, 0x16, 0x04, 0x14, 0xf9,
0xa0, 0xde, 0xdd, 0xfe, 0xa6, 0xe2, 0x09, 0xe4, 0x75, 0x92, 0xb6, 0x96, 0xc6, 0x41, 0xc6, 0x15,
0xbb, 0x8c, 0xc0, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0x78, 0x93, 0x43, 0x49, 0x2e, 0x92, 0xe5, 0xc9, 0xf6, 0xab,
0x54, 0xcc, 0xb0, 0xfc, 0xc4, 0x62, 0xda, 0xee, 0x89, 0x86, 0x21, 0xf2, 0xa7, 0x4b, 0xb9, 0xf4,
0x07, 0x05, 0x45, 0x1d, 0xa4, 0x93, 0x39, 0xe9, 0x6c, 0x0c, 0x04, 0x55, 0xee, 0xd3, 0x42, 0xd1,
0x03, 0xbd, 0xc3, 0xc1, 0x76, 0x6c, 0xbe, 0x33, 0x7d, 0x38, 0x52, 0x33, 0xce, 0xee, 0x42, 0x6a,
0x9a, 0x9f, 0x9c, 0xac, 0xd2, 0xa7, 0x0a, 0x87, 0xa9, 0x78, 0x93, 0x3a, 0xef, 0xc0, 0x55, 0x91,
0x3b, 0x27, 0x9d, 0x6c, 0x03, 0x2c, 0x7a, 0xb5, 0xdd, 0xe6, 0xff, 0xd0, 0xe3, 0x6a, 0x07, 0x59,
0x83, 0x36, 0xdb, 0x74, 0xea, 0x8c, 0xb3, 0xa3, 0xc6, 0x2a, 0xef, 0x47, 0x27, 0x53, 0xc7, 0xd2,
0xef, 0x89, 0x66, 0xb7, 0x91, 0x46, 0x56, 0x66, 0x2f, 0x54, 0xbb, 0xfb, 0xb8, 0x29, 0x07, 0x74,
0x23, 0x97, 0x69, 0xff, 0x9f, 0x8a, 0xa3, 0xd9, 0xf2, 0xb2, 0x26, 0x41, 0x1b, 0x94, 0x79, 0xb7,
0x64, 0x86, 0x85, 0x72, 0x2b, 0x9d, 0xc7, 0x7f, 0x8f, 0x40, 0x15, 0xa5, 0x56, 0x89, 0x6c, 0x7f,
0x54, 0xd6, 0x81, 0xfd, 0xf2, 0x13, 0x40, 0x13, 0x0c, 0x2a, 0x58, 0x89, 0x79, 0x63, 0x04, 0xb6,
0x5a, 0x3b, 0xf8, 0x21, 0x4d, 0xac, 0xa3, 0xf7, 0xa4, 0x89, 0x0d, 0x05, 0x4a, 0xdb, 0x47, 0xfe,
0xf3, 0x2c, 0x09, 0xd8, 0x18, 0xff, 0xa9, 0x26, 0x1a, 0x57, 0x0a, 0x81, 0x5f, 0x02, 0x87, 0xcb,
0xef, 0x31, 0x10, 0xec, 0x40, 0xa8, 0x14, 0x9a, 0xc5, 0xfa, 0x27, 0x26, 0x40, 0x2d, 0xa0, 0x5c,
0xeb, 0xe3, 0xd5, 0x0d, 0xc9, 0x9b, 0x6a, 0x91, 0xf1, 0xda, 0xe8, 0xb5, 0x4f, 0xa9, 0xa0, 0x77,
0x46, 0x09, 0x7b, 0x32, 0x24, 0xff, 0x43, 0x42, 0x5d, 0xb5, 0x69, 0xd2, 0x1f, 0x32, 0x7f, 0xe4,
0x36, 0x97, 0x6a, 0x8a, 0xaa, 0x4e, 0xa1, 0x22, 0x04, 0x20, 0xa3, 0x99, 0xbd, 0x7c, 0x3f, 0x3d,
0xe3, 0xd6, 0x7a, 0x16, 0x39, 0xd6, 0xb6, 0xa7, 0xe2, 0xe8, 0x01, 0x54, 0x98, 0xa7, 0x22, 0x12,
0x8b, 0x46, 0xeb, 0xd4, 0x68, 0x7b, 0x8d, 0xbd, 0xbb, 0x23
        };

        private readonly byte[] signedPkAsReq = new byte[]
        {
0x30, 0x82, 0x0a, 0x82, 0x80, 0x82, 0x0a, 0x7e, 0x30, 0x82, 0x0a, 0x7a, 0x06, 0x09, 0x2a, 0x86,
0x48, 0x86, 0xf7, 0x0d, 0x01, 0x07, 0x02, 0xa0, 0x82, 0x0a, 0x6b, 0x30, 0x82, 0x0a, 0x67, 0x02,
0x01, 0x03, 0x31, 0x0b, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0x00, 0x30,
0x82, 0x02, 0x26, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x01, 0xa0, 0x82, 0x02, 0x19,
0x04, 0x82, 0x02, 0x15, 0x30, 0x82, 0x02, 0x11, 0xa0, 0x3c, 0x30, 0x3a, 0xa0, 0x05, 0x02, 0x03,
0x0b, 0xe0, 0x03, 0xa1, 0x11, 0x18, 0x0f, 0x32, 0x30, 0x31, 0x39, 0x31, 0x32, 0x32, 0x30, 0x31,
0x36, 0x34, 0x34, 0x34, 0x30, 0x5a, 0xa2, 0x06, 0x02, 0x04, 0x44, 0x1d, 0x51, 0xa9, 0xa3, 0x16,
0x04, 0x14, 0xfd, 0x95, 0xab, 0x30, 0x2b, 0x1d, 0xb0, 0xc1, 0x3d, 0x91, 0x58, 0x75, 0xed, 0x60,
0x2d, 0x47, 0x7b, 0x2c, 0x80, 0xf9, 0xa1, 0x82, 0x01, 0xa7, 0x30, 0x82, 0x01, 0xa3, 0x30, 0x82,
0x01, 0x17, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3e, 0x02, 0x01, 0x30, 0x82, 0x01, 0x0a, 0x02,
0x81, 0x81, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc9, 0x0f, 0xda, 0xa2, 0x21,
0x68, 0xc2, 0x34, 0xc4, 0xc6, 0x62, 0x8b, 0x80, 0xdc, 0x1c, 0xd1, 0x29, 0x02, 0x4e, 0x08, 0x8a,
0x67, 0xcc, 0x74, 0x02, 0x0b, 0xbe, 0xa6, 0x3b, 0x13, 0x9b, 0x22, 0x51, 0x4a, 0x08, 0x79, 0x8e,
0x34, 0x04, 0xdd, 0xef, 0x95, 0x19, 0xb3, 0xcd, 0x3a, 0x43, 0x1b, 0x30, 0x2b, 0x0a, 0x6d, 0xf2,
0x5f, 0x14, 0x37, 0x4f, 0xe1, 0x35, 0x6d, 0x6d, 0x51, 0xc2, 0x45, 0xe4, 0x85, 0xb5, 0x76, 0x62,
0x5e, 0x7e, 0xc6, 0xf4, 0x4c, 0x42, 0xe9, 0xa6, 0x37, 0xed, 0x6b, 0x0b, 0xff, 0x5c, 0xb6, 0xf4,
0x06, 0xb7, 0xed, 0xee, 0x38, 0x6b, 0xfb, 0x5a, 0x89, 0x9f, 0xa5, 0xae, 0x9f, 0x24, 0x11, 0x7c,
0x4b, 0x1f, 0xe6, 0x49, 0x28, 0x66, 0x51, 0xec, 0xe6, 0x53, 0x81, 0xff, 0xff, 0xff, 0xff, 0xff,
0xff, 0xff, 0xff, 0x02, 0x01, 0x02, 0x02, 0x81, 0x80, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
0xff, 0xe4, 0x87, 0xed, 0x51, 0x10, 0xb4, 0x61, 0x1a, 0x62, 0x63, 0x31, 0x45, 0xc0, 0x6e, 0x0e,
0x68, 0x94, 0x81, 0x27, 0x04, 0x45, 0x33, 0xe6, 0x3a, 0x01, 0x05, 0xdf, 0x53, 0x1d, 0x89, 0xcd,
0x91, 0x28, 0xa5, 0x04, 0x3c, 0xc7, 0x1a, 0x02, 0x6e, 0xf7, 0xca, 0x8c, 0xd9, 0xe6, 0x9d, 0x21,
0x8d, 0x98, 0x15, 0x85, 0x36, 0xf9, 0x2f, 0x8a, 0x1b, 0xa7, 0xf0, 0x9a, 0xb6, 0xb6, 0xa8, 0xe1,
0x22, 0xf2, 0x42, 0xda, 0xbb, 0x31, 0x2f, 0x3f, 0x63, 0x7a, 0x26, 0x21, 0x74, 0xd3, 0x1b, 0xf6,
0xb5, 0x85, 0xff, 0xae, 0x5b, 0x7a, 0x03, 0x5b, 0xf6, 0xf7, 0x1c, 0x35, 0xfd, 0xad, 0x44, 0xcf,
0xd2, 0xd7, 0x4f, 0x92, 0x08, 0xbe, 0x25, 0x8f, 0xf3, 0x24, 0x94, 0x33, 0x28, 0xf6, 0x73, 0x29,
0xc0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x03, 0x81, 0x85, 0x00, 0x02, 0x81, 0x81,
0x00, 0xf4, 0x4d, 0x68, 0x81, 0x65, 0x45, 0x56, 0x12, 0x1f, 0x84, 0x0a, 0x1a, 0x08, 0xce, 0x62,
0xc8, 0x09, 0xdc, 0x36, 0x09, 0x37, 0x5e, 0x54, 0x93, 0xf1, 0x94, 0x93, 0x4d, 0x2e, 0xcd, 0x60,
0x99, 0x1f, 0x37, 0xed, 0xcf, 0x92, 0x9d, 0xde, 0x30, 0x3e, 0x21, 0x3d, 0xfc, 0x1b, 0xb3, 0x94,
0x62, 0xc8, 0x39, 0x1e, 0x7d, 0xda, 0xb3, 0x98, 0xa8, 0x01, 0x53, 0xf7, 0x15, 0x37, 0x08, 0xd8,
0x94, 0x2e, 0xcd, 0x7e, 0xcf, 0x63, 0xea, 0x1e, 0xc4, 0x21, 0x5c, 0xeb, 0x09, 0x31, 0x89, 0xee,
0x22, 0x2e, 0xaa, 0xbd, 0xa6, 0xea, 0xa6, 0xc9, 0xb5, 0xce, 0x4f, 0x71, 0x3d, 0x17, 0x2f, 0xd5,
0x38, 0x1c, 0x9c, 0x0c, 0x4b, 0x63, 0x28, 0xfb, 0x16, 0xa7, 0x6f, 0x36, 0xb1, 0x9b, 0xab, 0xb1,
0x9c, 0x96, 0x8d, 0x16, 0x0a, 0xa5, 0xb7, 0x3c, 0xbe, 0x53, 0x16, 0xee, 0x3a, 0x71, 0xb3, 0xc3,
0xc2, 0xa2, 0x02, 0x30, 0x00, 0xa3, 0x22, 0x04, 0x20, 0x39, 0x47, 0xd3, 0x6a, 0x33, 0x0c, 0x5f,
0x4d, 0x73, 0xe7, 0x28, 0xc4, 0x08, 0x12, 0x2c, 0x9a, 0x1f, 0x33, 0x7e, 0x69, 0xa4, 0xd1, 0x94,
0xef, 0xd0, 0x9f, 0x83, 0x5c, 0xde, 0xb3, 0xb1, 0xe3, 0xa0, 0x82, 0x06, 0x40, 0x30, 0x82, 0x06,
0x3c, 0x30, 0x82, 0x05, 0x24, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x13, 0x2c, 0x00, 0x00, 0x00,
0x0f, 0x22, 0xa8, 0xc5, 0x22, 0xf1, 0x93, 0x94, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x30,
0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x68,
0x31, 0x13, 0x30, 0x11, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19,
0x16, 0x03, 0x63, 0x6f, 0x6d, 0x31, 0x24, 0x30, 0x22, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93,
0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x14, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69,
0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06,
0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72,
0x70, 0x31, 0x15, 0x30, 0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0c, 0x63, 0x6f, 0x72, 0x70,
0x2d, 0x43, 0x41, 0x30, 0x31, 0x2d, 0x43, 0x41, 0x30, 0x1e, 0x17, 0x0d, 0x31, 0x39, 0x31, 0x31,
0x32, 0x37, 0x32, 0x30, 0x33, 0x32, 0x30, 0x39, 0x5a, 0x17, 0x0d, 0x32, 0x31, 0x31, 0x31, 0x32,
0x37, 0x32, 0x30, 0x34, 0x32, 0x30, 0x39, 0x5a, 0x30, 0x75, 0x31, 0x13, 0x30, 0x11, 0x06, 0x0a,
0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f, 0x6d, 0x31,
0x24, 0x30, 0x22, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16,
0x14, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65,
0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93,
0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x0e, 0x30, 0x0c, 0x06,
0x03, 0x55, 0x04, 0x03, 0x13, 0x05, 0x55, 0x73, 0x65, 0x72, 0x73, 0x31, 0x12, 0x30, 0x10, 0x06,
0x03, 0x55, 0x04, 0x03, 0x13, 0x09, 0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x65, 0x72, 0x30,
0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00,
0xbf, 0x4a, 0xcc, 0x89, 0xe7, 0x58, 0x7c, 0xcb, 0xb3, 0x12, 0x4a, 0x98, 0x01, 0x79, 0x94, 0xad,
0x2d, 0xec, 0x08, 0x48, 0x90, 0xb5, 0x29, 0x5e, 0xc1, 0x12, 0x9a, 0x68, 0x27, 0xcf, 0x3b, 0xfb,
0x2a, 0xbe, 0xa2, 0xa9, 0xf6, 0xd3, 0x83, 0x39, 0xb1, 0xff, 0x9e, 0x2b, 0x42, 0x01, 0xa4, 0xac,
0xbe, 0xe0, 0x84, 0xce, 0x55, 0x54, 0x5a, 0xa7, 0x3d, 0x44, 0xbf, 0x6e, 0x3b, 0xf2, 0x53, 0x8c,
0x6d, 0xcf, 0xb8, 0x84, 0x08, 0x79, 0x9a, 0x54, 0xa5, 0x46, 0x7a, 0x67, 0xd3, 0x60, 0xec, 0xb7,
0x97, 0x46, 0xe8, 0xe5, 0x47, 0xc5, 0x3e, 0x6d, 0x18, 0x3b, 0x57, 0x78, 0xb0, 0x3c, 0xbd, 0xe6,
0x93, 0x1c, 0xea, 0x93, 0x09, 0x3a, 0x87, 0xf1, 0x18, 0x35, 0xd1, 0xc8, 0x62, 0xd4, 0x3f, 0x58,
0xc3, 0x0e, 0x65, 0x55, 0x81, 0x7c, 0x86, 0x16, 0xe2, 0x5b, 0x88, 0x63, 0x9e, 0x5d, 0xfa, 0x55,
0xaf, 0x11, 0xcf, 0x12, 0xb4, 0x09, 0xe1, 0x6b, 0xe5, 0x9b, 0x6f, 0xf9, 0x0f, 0xf6, 0x57, 0x05,
0xfa, 0x96, 0xc6, 0xc0, 0xf3, 0x1a, 0xb7, 0x18, 0xd9, 0xa9, 0x09, 0x1e, 0xb4, 0x9e, 0xff, 0xd4,
0xf5, 0x73, 0xba, 0x06, 0xf0, 0x08, 0x42, 0x54, 0x94, 0xc0, 0xeb, 0x71, 0x6f, 0xe4, 0xda, 0x04,
0xe5, 0xf8, 0xe0, 0xf2, 0x09, 0xaf, 0x0e, 0xee, 0xd3, 0xbd, 0x70, 0xfe, 0xfc, 0x4d, 0x21, 0x75,
0x7b, 0xcf, 0x03, 0x16, 0x02, 0x86, 0x14, 0xbf, 0x1e, 0x9f, 0x34, 0x5f, 0xec, 0x2a, 0x13, 0xee,
0xc7, 0x39, 0x8a, 0x2b, 0x86, 0x50, 0xe0, 0x90, 0xdd, 0x2b, 0xa3, 0xe8, 0x12, 0x9a, 0xb7, 0x72,
0x2c, 0x80, 0x8d, 0x14, 0xa5, 0xcc, 0xd9, 0x76, 0x03, 0x2d, 0xfd, 0xab, 0x7d, 0x7e, 0x1c, 0x46,
0x49, 0xb2, 0x40, 0xd4, 0xae, 0x5e, 0x89, 0x9b, 0x32, 0xe0, 0x99, 0x24, 0x63, 0x0d, 0x5e, 0x09,
0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x82, 0x02, 0xd0, 0x30, 0x82, 0x02, 0xcc, 0x30, 0x3e, 0x06,
0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x07, 0x04, 0x31, 0x30, 0x2f, 0x06, 0x27,
0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x08, 0x82, 0xc6, 0xb2, 0x56, 0x85, 0xd5, 0xc3,
0x44, 0x83, 0xa9, 0x95, 0x18, 0x84, 0xdd, 0x91, 0x66, 0x83, 0x88, 0xb4, 0x4e, 0x81, 0x52, 0x82,
0xdc, 0xcb, 0x52, 0x87, 0x85, 0xe1, 0x47, 0x02, 0x01, 0x64, 0x02, 0x01, 0x07, 0x30, 0x1f, 0x06,
0x03, 0x55, 0x1d, 0x25, 0x04, 0x18, 0x30, 0x16, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
0x03, 0x02, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x02, 0x30, 0x0e,
0x06, 0x03, 0x55, 0x1d, 0x0f, 0x01, 0x01, 0xff, 0x04, 0x04, 0x03, 0x02, 0x05, 0xa0, 0x30, 0x29,
0x06, 0x09, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x15, 0x0a, 0x04, 0x1c, 0x30, 0x1a, 0x30,
0x0a, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x0c, 0x06, 0x0a, 0x2b,
0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x02, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e,
0x04, 0x16, 0x04, 0x14, 0x08, 0x50, 0x69, 0x26, 0x6a, 0x31, 0x36, 0x97, 0x24, 0x66, 0x27, 0x7b,
0x69, 0x06, 0xbc, 0xc2, 0xd2, 0xc4, 0xd5, 0xd9, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04,
0x18, 0x30, 0x16, 0x80, 0x14, 0xed, 0x80, 0x47, 0xbd, 0xb1, 0x11, 0xf3, 0x80, 0x20, 0x18, 0x37,
0xd6, 0xa3, 0xc5, 0x38, 0xb2, 0x2f, 0xf2, 0x15, 0x05, 0x30, 0x81, 0xdc, 0x06, 0x03, 0x55, 0x1d,
0x1f, 0x04, 0x81, 0xd4, 0x30, 0x81, 0xd1, 0x30, 0x81, 0xce, 0xa0, 0x81, 0xcb, 0xa0, 0x81, 0xc8,
0x86, 0x81, 0xc5, 0x6c, 0x64, 0x61, 0x70, 0x3a, 0x2f, 0x2f, 0x2f, 0x43, 0x4e, 0x3d, 0x63, 0x6f,
0x72, 0x70, 0x2d, 0x43, 0x41, 0x30, 0x31, 0x2d, 0x43, 0x41, 0x2c, 0x43, 0x4e, 0x3d, 0x63, 0x61,
0x30, 0x31, 0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x44, 0x50, 0x2c, 0x43, 0x4e, 0x3d, 0x50, 0x75, 0x62,
0x6c, 0x69, 0x63, 0x25, 0x32, 0x30, 0x4b, 0x65, 0x79, 0x25, 0x32, 0x30, 0x53, 0x65, 0x72, 0x76,
0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73,
0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f,
0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2c, 0x44, 0x43, 0x3d, 0x69, 0x64, 0x65,
0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f,
0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69,
0x63, 0x61, 0x74, 0x65, 0x52, 0x65, 0x76, 0x6f, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x4c, 0x69,
0x73, 0x74, 0x3f, 0x62, 0x61, 0x73, 0x65, 0x3f, 0x6f, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x43, 0x6c,
0x61, 0x73, 0x73, 0x3d, 0x63, 0x52, 0x4c, 0x44, 0x69, 0x73, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74,
0x69, 0x6f, 0x6e, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x30, 0x81, 0xd3, 0x06, 0x08, 0x2b, 0x06, 0x01,
0x05, 0x05, 0x07, 0x01, 0x01, 0x04, 0x81, 0xc6, 0x30, 0x81, 0xc3, 0x30, 0x81, 0xc0, 0x06, 0x08,
0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30, 0x02, 0x86, 0x81, 0xb3, 0x6c, 0x64, 0x61, 0x70, 0x3a,
0x2f, 0x2f, 0x2f, 0x43, 0x4e, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43, 0x41, 0x30, 0x31, 0x2d,
0x43, 0x41, 0x2c, 0x43, 0x4e, 0x3d, 0x41, 0x49, 0x41, 0x2c, 0x43, 0x4e, 0x3d, 0x50, 0x75, 0x62,
0x6c, 0x69, 0x63, 0x25, 0x32, 0x30, 0x4b, 0x65, 0x79, 0x25, 0x32, 0x30, 0x53, 0x65, 0x72, 0x76,
0x69, 0x63, 0x65, 0x73, 0x2c, 0x43, 0x4e, 0x3d, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, 0x73,
0x2c, 0x43, 0x4e, 0x3d, 0x43, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f,
0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x72, 0x70, 0x2c, 0x44, 0x43, 0x3d, 0x69, 0x64, 0x65,
0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f,
0x6e, 0x2c, 0x44, 0x43, 0x3d, 0x63, 0x6f, 0x6d, 0x3f, 0x63, 0x41, 0x43, 0x65, 0x72, 0x74, 0x69,
0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x3f, 0x62, 0x61, 0x73, 0x65, 0x3f, 0x6f, 0x62, 0x6a, 0x65,
0x63, 0x74, 0x43, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x63, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63,
0x61, 0x74, 0x69, 0x6f, 0x6e, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x30, 0x39,
0x06, 0x03, 0x55, 0x1d, 0x11, 0x04, 0x32, 0x30, 0x30, 0xa0, 0x2e, 0x06, 0x0a, 0x2b, 0x06, 0x01,
0x04, 0x01, 0x82, 0x37, 0x14, 0x02, 0x03, 0xa0, 0x20, 0x0c, 0x1e, 0x74, 0x65, 0x73, 0x74, 0x75,
0x73, 0x65, 0x72, 0x40, 0x62, 0x6c, 0x61, 0x68, 0x2e, 0x63, 0x6f, 0x72, 0x70, 0x2e, 0x69, 0x6e,
0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c, 0x32, 0x32, 0x32, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0xea, 0x42, 0xa0,
0xfa, 0xf6, 0xdd, 0x43, 0x5e, 0x1b, 0x44, 0x4a, 0xd6, 0xc1, 0x51, 0x78, 0x60, 0x67, 0xf0, 0xb6,
0x3a, 0x33, 0x92, 0x3c, 0xdd, 0xa3, 0xc6, 0x34, 0x6d, 0x47, 0x4b, 0x19, 0x43, 0x9f, 0x7b, 0xc6,
0x14, 0x1a, 0x40, 0xef, 0x3a, 0xcd, 0x4f, 0xdc, 0x23, 0xa8, 0x51, 0x12, 0x0d, 0x6a, 0x5f, 0x0f,
0x7b, 0xc7, 0x8c, 0x28, 0x2a, 0xa9, 0xb4, 0x91, 0x5c, 0x02, 0x72, 0x64, 0xe0, 0x8b, 0x2d, 0x3a,
0xc5, 0xac, 0xa7, 0x1a, 0xa2, 0xa7, 0x74, 0xe8, 0x5c, 0xd8, 0xe2, 0x42, 0x72, 0xcd, 0xc0, 0x64,
0xd7, 0xad, 0x85, 0x06, 0xe0, 0x51, 0x6d, 0x34, 0xc4, 0xb5, 0x88, 0x58, 0x6c, 0xe2, 0x8b, 0x0c,
0x7a, 0x84, 0xcb, 0x46, 0x21, 0x75, 0x4c, 0xe1, 0xa7, 0xbd, 0xd9, 0x30, 0xb1, 0xdd, 0x85, 0x48,
0xb3, 0x23, 0x10, 0x31, 0xa9, 0x67, 0x7b, 0xd6, 0x83, 0xaa, 0xe0, 0x2f, 0x48, 0x5f, 0xb0, 0x62,
0x3a, 0xc5, 0xe6, 0x83, 0x93, 0xf0, 0x3a, 0x02, 0xe5, 0xb4, 0xf8, 0x74, 0x35, 0x1c, 0x1b, 0x42,
0x69, 0xe4, 0xe9, 0xd2, 0x77, 0x8b, 0xa4, 0x4a, 0x53, 0x7e, 0x0d, 0x64, 0xfb, 0x67, 0x94, 0xc9,
0xa7, 0x76, 0x50, 0x75, 0x03, 0x5e, 0xb0, 0xba, 0x07, 0x69, 0x03, 0x28, 0x90, 0x6e, 0x7f, 0x00,
0xf7, 0x8d, 0xa8, 0x9d, 0x49, 0x34, 0xcb, 0xa9, 0x6d, 0x4d, 0x6a, 0x48, 0x94, 0x1b, 0xd2, 0x00,
0xeb, 0xa5, 0xe8, 0x9d, 0x14, 0x14, 0x49, 0xc7, 0x0a, 0x69, 0x39, 0x08, 0xb7, 0x75, 0x1d, 0xf7,
0xeb, 0x8e, 0xfc, 0xe2, 0xbf, 0x25, 0x65, 0xeb, 0x6b, 0xe3, 0x4a, 0x08, 0x18, 0x22, 0xb8, 0x30,
0xf7, 0xd5, 0x42, 0x26, 0x6f, 0xa6, 0x95, 0x7d, 0xd9, 0xa6, 0x29, 0xc2, 0xfb, 0xa8, 0x5a, 0xfe,
0xff, 0x29, 0xa6, 0x20, 0x34, 0x24, 0x64, 0x57, 0x43, 0xb8, 0x1f, 0x6b, 0xad, 0x31, 0x82, 0x01,
0xe5, 0x30, 0x82, 0x01, 0xe1, 0x02, 0x01, 0x01, 0x30, 0x7f, 0x30, 0x68, 0x31, 0x13, 0x30, 0x11,
0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x03, 0x63, 0x6f,
0x6d, 0x31, 0x24, 0x30, 0x22, 0x06, 0x0a, 0x09, 0x92, 0x26, 0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01,
0x19, 0x16, 0x14, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x69, 0x6e, 0x74, 0x65, 0x72,
0x76, 0x65, 0x6e, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0x14, 0x30, 0x12, 0x06, 0x0a, 0x09, 0x92, 0x26,
0x89, 0x93, 0xf2, 0x2c, 0x64, 0x01, 0x19, 0x16, 0x04, 0x63, 0x6f, 0x72, 0x70, 0x31, 0x15, 0x30,
0x13, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0c, 0x63, 0x6f, 0x72, 0x70, 0x2d, 0x43, 0x41, 0x30,
0x31, 0x2d, 0x43, 0x41, 0x02, 0x13, 0x2c, 0x00, 0x00, 0x00, 0x0f, 0x22, 0xa8, 0xc5, 0x22, 0xf1,
0x93, 0x94, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x30, 0x09, 0x06, 0x05, 0x2b, 0x0e, 0x03,
0x02, 0x1a, 0x05, 0x00, 0xa0, 0x3d, 0x30, 0x16, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
0x01, 0x09, 0x03, 0x31, 0x09, 0x06, 0x07, 0x2b, 0x06, 0x01, 0x05, 0x02, 0x03, 0x01, 0x30, 0x23,
0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x09, 0x04, 0x31, 0x16, 0x04, 0x14, 0x85,
0x88, 0x09, 0x0e, 0x2e, 0xc7, 0x6d, 0x56, 0x2c, 0x3f, 0x18, 0x43, 0x2a, 0x88, 0x3e, 0x2b, 0x94,
0xeb, 0x5a, 0x1a, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
0x05, 0x00, 0x04, 0x82, 0x01, 0x00, 0x10, 0x92, 0x4d, 0x6a, 0x67, 0x20, 0xc2, 0x60, 0x68, 0xd2,
0xb5, 0xba, 0x55, 0x21, 0xf3, 0x06, 0x42, 0x2c, 0xf7, 0xc6, 0xe4, 0x31, 0x4a, 0xec, 0xa9, 0xf7,
0x2e, 0x5d, 0xb7, 0x74, 0xa7, 0xda, 0x8f, 0xb0, 0x28, 0x16, 0xed, 0x53, 0x90, 0x4f, 0x19, 0x1d,
0x3f, 0x4d, 0x8d, 0x6f, 0x55, 0x73, 0xa0, 0x22, 0xc2, 0x80, 0x5e, 0x84, 0xf8, 0xce, 0xf7, 0x08,
0x3f, 0x1f, 0xc0, 0x3a, 0xc4, 0x21, 0x6e, 0x55, 0xa7, 0x70, 0x5b, 0xd3, 0x86, 0xfd, 0xd5, 0xcb,
0x8b, 0xe9, 0xe4, 0x07, 0xed, 0xfa, 0x69, 0x20, 0x97, 0x4e, 0x0f, 0x86, 0xd7, 0xda, 0x99, 0xd2,
0xe8, 0xf0, 0xc8, 0xb4, 0x4f, 0xfa, 0xa5, 0xbe, 0x77, 0x3e, 0xdc, 0x27, 0xde, 0x4b, 0x61, 0x4c,
0xa9, 0x50, 0xd4, 0xc8, 0xb3, 0xaf, 0x7e, 0x9b, 0xaa, 0x47, 0x6a, 0xfa, 0xe5, 0x0b, 0x2a, 0x4e,
0xf2, 0x60, 0x1f, 0x14, 0x06, 0x7f, 0x4d, 0x9c, 0xe1, 0x96, 0x60, 0xcd, 0x43, 0xc2, 0x66, 0xc2,
0x35, 0xbd, 0x8e, 0x55, 0x2c, 0xed, 0x0d, 0x91, 0xfe, 0xfc, 0x16, 0x82, 0xe6, 0xad, 0x6d, 0xb3,
0xb5, 0x8f, 0xf0, 0x31, 0xb1, 0x47, 0x8f, 0xd1, 0xa9, 0x11, 0x9e, 0xbb, 0x0b, 0xa2, 0x07, 0x07,
0x7a, 0xaf, 0x92, 0xbd, 0x65, 0xbb, 0xc9, 0x2a, 0x75, 0x7e, 0xa6, 0x28, 0x04, 0x32, 0x1a, 0x73,
0x78, 0x6e, 0x1e, 0x5c, 0xc0, 0x46, 0x6b, 0xb8, 0xc7, 0x65, 0x80, 0x99, 0x64, 0x5b, 0x67, 0x32,
0x5b, 0xd5, 0x14, 0x8b, 0x4b, 0xf4, 0xdb, 0x87, 0xbb, 0x66, 0x23, 0xb3, 0x07, 0x60, 0x72, 0x09,
0xc0, 0x96, 0xcd, 0xf6, 0x56, 0x44, 0xc5, 0xb4, 0x7f, 0x89, 0x5f, 0xbc, 0x2c, 0xda, 0xc5, 0x56,
0x9b, 0x08, 0xfd, 0x04, 0x14, 0xf4, 0x00, 0x6d, 0xe8, 0x76, 0xc6, 0x15, 0xbe, 0x60, 0x59, 0x5f,
0xdd, 0xe8, 0x01, 0xfe, 0x4c, 0x10
        };
    }
}