
:::moniker range="= aspnetcore-7.0"

[View or download sample code](https://github.com/dotnet/AspNetCore.Docs.Samples/tree/main/mvc/action-return-types/7.x/WebApiSample) ([how to download](xref:index#how-to-download-a-sample))

ASP.NET Core provides the following options for web API controller action return types:

* [Specific type](#specific-type)
* [IActionResult](#iactionresult-type)
* [ActionResult\<T>](#actionresultt-type)
* [HttpResults](#httpresults-type)

This article explains when it's most appropriate to use each return type.

## Specific type

The most basic action returns a primitive or complex data type, for example, `string` or a custom object. Consider the following action, which returns a collection of custom `Product` objects:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ProductsController.cs" id="snippet_Get":::

Without known conditions to safeguard against, returning a specific type could suffice. The preceding action accepts no parameters, so parameter constraints validation isn't needed.

When multiple return types are possible, it's common to mix an <xref:Microsoft.AspNetCore.Mvc.ActionResult> return type with the primitive or complex return type. Either [IActionResult](#iactionresult-type) or [ActionResult\<T>](#actionresultt-type) are necessary to accommodate this type of action. Several samples of multiple return types are provided in this article.

### Return IEnumerable\<T> or IAsyncEnumerable\<T>

See [Return `IEnumerable<T>` or `IAsyncEnumerable<T>`](/aspnet/core/fundamentals/best-practices#return-ienumerablet-or-iasyncenumerablet) for performance considerations.

ASP.NET Core buffers the result of actions that return <xref:System.Collections.Generic.IEnumerable%601> before writing them to the response. Consider declaring the action signature's return type as <xref:System.Collections.Generic.IAsyncEnumerable%601> to guarantee asynchronous iteration. Ultimately, the iteration mode is based on the underlying concrete type being returned and the selected formatter affects how the result is processed:

* When using `System.Text.Json` formatter, MVC relies on the support that `System.Text.Json` added to **stream** the result.
* When using `Newtonsoft.Json` or with `XML-based` formatters the result is buffered.

Consider the following action, which returns sale-priced product records as `IEnumerable<Product>`:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ProductsController.cs" id="snippet_GetOnSaleProducts":::

The `IAsyncEnumerable<Product>` equivalent of the preceding action is:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ProductsController.cs" id="snippet_GetOnSaleProductsAsync":::

## IActionResult type

The <xref:Microsoft.AspNetCore.Mvc.IActionResult> return type is appropriate when multiple `ActionResult` return types are possible in an action. The `ActionResult` types represent various HTTP status codes. Any non-abstract class deriving from `ActionResult` qualifies as a valid return type. Some common return types in this category are <xref:Microsoft.AspNetCore.Mvc.BadRequestResult> (400), <xref:Microsoft.AspNetCore.Mvc.NotFoundResult> (404), and <xref:Microsoft.AspNetCore.Mvc.OkObjectResult> (200). Alternatively, convenience methods in the <xref:Microsoft.AspNetCore.Mvc.ControllerBase> class can be used to return `ActionResult` types from an action. For example, `return BadRequest();` is a shorthand form of `return new BadRequestResult();`.

Because there are multiple return types and paths in this type of action, liberal use of the [`[ProducesResponseType]`](xref:Microsoft.AspNetCore.Mvc.ProducesResponseTypeAttribute) attribute is necessary. This attribute produces more descriptive response details for web API help pages generated by tools like [Swagger](xref:tutorials/web-api-help-pages-using-swagger). `[ProducesResponseType]` indicates the known types and HTTP status codes to be returned by the action.

### Synchronous action

Consider the following synchronous action in which there are two possible return types:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ActionResultProductsController.cs" id="snippet_GetByIdIActionResult" highlight="7":::

In the preceding action:

* A 404 status code is returned when the product represented by `id` doesn't exist in the underlying data store. The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.NotFound%2A> convenience method is invoked as shorthand for `return new NotFoundResult();`.
* A 200 status code is returned with the `Product` object when the product does exist. The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.Ok%2A> convenience method is invoked as shorthand for `return new OkObjectResult(product);`.

### Asynchronous action

Consider the following asynchronous action in which there are two possible return types:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ActionResultProductsController.cs" id="snippet_CreateAsyncIActionResult" highlight="9,15":::

In the preceding action:

* A 400 status code is returned when the product description contains "XYZ Widget". The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.BadRequest%2A> convenience method is invoked as shorthand for `return new BadRequestResult();`.
* A 201 status code is generated by the <xref:Microsoft.AspNetCore.Mvc.ControllerBase.CreatedAtAction%2A> convenience method when a product is created. The following code is an alternative to calling `CreatedAtAction`:

  ```csharp
  return new CreatedAtActionResult(nameof(GetById), 
                                  "Products", 
                                  new { id = product.Id }, 
                                  product);
  ```

  In the preceding code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

For example, the following model indicates that requests must include the `Name` and `Description` properties. Failure to provide `Name` and `Description` in the request causes model validation to fail.

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Models/Product.cs" id="snippet_ProductClass" highlight="5-6,8-9":::

If the [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute is applied, model validation errors result in a 400 status code. For more information, see [Automatic HTTP 400 responses](xref:web-api/index#automatic-http-400-responses).

## ActionResult vs IActionResult

The following section compares `ActionResult` to  `IActionResult`

### ActionResult\<T> type

ASP.NET Core includes the [ActionResult\<T>](xref:Microsoft.AspNetCore.Mvc.ActionResult%601) return type for web API controller actions. It enables returning a type deriving from <xref:Microsoft.AspNetCore.Mvc.ActionResult> or return a [specific type](#specific-type). `ActionResult<T>` offers the following benefits over the [IActionResult type](#iactionresult-type):

* The [`[ProducesResponseType]`](xref:Microsoft.AspNetCore.Mvc.ProducesResponseTypeAttribute) attribute's `Type` property can be excluded. For example, `[ProducesResponseType(200, Type = typeof(Product))]` is simplified to `[ProducesResponseType(200)]`. The action's expected return type is inferred from the `T` in `ActionResult<T>`.
* [Implicit cast operators](/dotnet/csharp/language-reference/keywords/implicit) support the conversion of both `T` and `ActionResult` to `ActionResult<T>`. `T` converts to <xref:Microsoft.AspNetCore.Mvc.ObjectResult>, which means `return new ObjectResult(T);` is simplified to `return T;`.

C# doesn't support implicit cast operators on interfaces. Consequently, conversion of the interface to a concrete type is necessary to use `ActionResult<T>`. For example, use of `IEnumerable` in the following example doesn't work:

```csharp
[HttpGet]
public ActionResult<IEnumerable<Product>> Get() =>
    _repository.GetProducts();
```

One option to fix the preceding code is to return `_repository.GetProducts().ToList();`.

Most actions have a specific return type. Unexpected conditions can occur during action execution, in which case the specific type isn't returned. For example, an action's input parameter may fail model validation. In such a case, it's common to return the appropriate `ActionResult` type instead of the specific type.

### Synchronous action

Consider a synchronous action in which there are two possible return types:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ActionResultOfTProductsController.cs" id="snippet_GetByIdActionResultOfT" highlight="7":::

In the preceding action:

* A 404 status code is returned when the product doesn't exist in the database.
* A 200 status code is returned with the corresponding `Product` object when the product does exist.

### Asynchronous action

Consider an asynchronous action in which there are two possible return types:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ActionResultOfTProductsController.cs" id="snippet_CreateAsyncActionResultOfT" highlight="9,15":::

In the preceding action:

* A 400 status code (<xref:Microsoft.AspNetCore.Mvc.ControllerBase.BadRequest%2A>) is returned by the ASP.NET Core runtime when:
  * The [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute has been applied and model validation fails.
  * The product description contains "XYZ Widget".
* A 201 status code is generated by the <xref:Microsoft.AspNetCore.Mvc.ControllerBase.CreatedAtAction%2A> method when a product is created. In this code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

## HttpResults type

In addition to the MVC-specific built-in result types (<xref:Microsoft.AspNetCore.Mvc.IActionResult> and [ActionResult\<T>](xref:Microsoft.AspNetCore.Mvc.ActionResult%601)), ASP.NET Core includes the [HttpResults](xref:Microsoft.AspNetCore.Http.HttpResults) types that can be used in both [Minimal APIs](/aspnet/core/fundamentals/minimal-apis) and Web API.

Different than the MVC-specific result types, the `HttpResults`:

* Are a results implementation that is processed by a call to [IResult.ExecuteAsync](xref:Microsoft.AspNetCore.Http.IResult.ExecuteAsync%2A).
* Does ***not*** leverage the configured [Formatters](/aspnet/core/web-api/advanced/formatting#format-specific-action-results). Not leveraging the configured formatters means:

  * Some features like `Content negotiation` aren't available.
  * The produced `Content-Type` is decided by the `HttpResults` implementation.

The `HttpResults` can be useful when sharing code between Minimal APIs and Web API. 

### IResult type

The <xref:Microsoft.AspNetCore.Http.HttpResults> namespace contains classes that implement the <xref:Microsoft.AspNetCore.Http.IResult> interface. The `IResult` interface defines a contract that represents the result of an HTTP endpoint. The static [Results](<xref:Microsoft.AspNetCore.Http.Results>) class is used to create varying `IResult` objects that represent different types of responses.

The [Built-in results](/aspnet/core/fundamentals/minimal-apis?view=aspnetcore-7.0&preserve-view=true#binr7) table shows the common result helpers.

Consider the following code:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/IResultProductsController.cs" id="snippet_GetByIdIResult" highlight="7":::

In the preceding action:

* A 404 status code is returned when the product doesn't exist in the database.
* A 200 status code is returned with the corresponding `Product` object when the product does exist, generated by the [Results.Ok\<T>()](/dotnet/api/microsoft.aspnetcore.http.results.ok).

Consider the following code:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/IResultProductsController.cs" id="snippet_CreateAsyncIResult" highlight="9,16":::

In the preceding action:

* A 400 status code is returned when:
  * The [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute has been applied and model validation fails.
  * The product description contains "XYZ Widget".
* A 201 status code is generated by the [`Results.Create`](/dotnet/api/microsoft.aspnetcore.http.results.created) method when a product is created. In this code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

### Results\<TResult1, TResultN> type

The static [TypedResults](<xref:Microsoft.AspNetCore.Http.TypedResults>) class returns the concrete `IResult` implementation that allows using `IResult` as return type. The usage of the concrete `IResult` implementation offers the following benefit over the [IResult type](#iresult-type):

* All the [`[ProducesResponseType]`](xref:Microsoft.AspNetCore.Mvc.ProducesResponseTypeAttribute) attributes can be excluded, since the `HttpResult` implementation contributes automatically to the endpoint metadata.

When multiple `IResult` return types are needed, returning [`Results<TResult1, TResultN>`](/dotnet/api/microsoft.aspnetcore.http.httpresults.results-2) is preferred over returning `IResult`. Returning `Results<TResult1, TResultN>` is preferred because generic union types automatically retain the endpoint metadata.

The `Results<TResult1, TResultN>` union types implement implicit cast operators so that the compiler can automatically convert the types specified in the generic arguments to an instance of the union type. This has the added benefit of providing compile-time checking that a route handler actually only returns the results that it declares it does. Attempting to return a type that isn’t declared as one of the generic arguments to `Results<>` results in a compilation error.

Consider the following code:

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ResultsOfTProductsController.cs" id="snippet_GetByIdResultsOfT" highlight="5":::

In the preceding action:

* A 404 status code is returned when the product doesn't exist in the database.
* A 200 status code is returned with the corresponding `Product` object when the product does exist, generated by the [TypedResults.Ok\<T>](<xref:Microsoft.AspNetCore.Http.TypedResults.Ok>).

:::code language="csharp" source="~/../AspNetCore.Docs.Samples/mvc/action-return-types/7.x/WebApiSample/Controllers/ResultsOfTProductsController.cs" id="snippet_CreateAsyncResultsOfT" highlight="6,13":::

In the preceding action:

* A 400 status code is returned when:
  * The [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute was applied and model validation fails.
  * The product description contains "XYZ Widget".
* A 201 status code is generated by the [`TypedResults.Create`](/dotnet/api/microsoft.aspnetcore.http.typedresults.created) method when a product is created. In this code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

## Additional resources

* <xref:mvc/controllers/actions>
* <xref:mvc/models/validation>
* <xref:tutorials/web-api-help-pages-using-swagger>

:::moniker-end

:::moniker range="< aspnetcore-7.0"

[View or download sample code](https://github.com/dotnet/AspNetCore.Docs/tree/main/aspnetcore/web-api/action-return-types/samples) ([how to download](xref:index#how-to-download-a-sample))

ASP.NET Core offers the following options for web API controller action return types:

* [Specific type](#specific-type)
* [IActionResult](#iactionresult-type)
* [ActionResult\<T>](#actionresultt-type)

This document explains when it's most appropriate to use each return type.

## Specific type

The simplest action returns a primitive or complex data type (for example, `string` or a custom object type). Consider the following action, which returns a collection of custom `Product` objects:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_Get":::

Without known conditions to safeguard against during action execution, returning a specific type could suffice. The preceding action accepts no parameters, so parameter constraints validation isn't needed.

When multiple return types are possible, it's common to mix an <xref:Microsoft.AspNetCore.Mvc.ActionResult> return type with the primitive or complex return type. Either [IActionResult](#iactionresult-type) or [ActionResult\<T>](#actionresultt-type) are necessary to accommodate this type of action. Several samples of multiple return types are provided in this document.

### Return IEnumerable\<T> or IAsyncEnumerable\<T>

ASP.NET Core buffers the result of actions that return <xref:System.Collections.Generic.IEnumerable%601> before writing them to the response. Consider declaring the action signature's return type as <xref:System.Collections.Generic.IAsyncEnumerable%601> to guarantee asynchronous iteration. Ultimately, the iteration mode is based on the underlying concrete type being returned. MVC automatically buffers any concrete type that implements `IAsyncEnumerable<T>`.

Consider the following action, which returns sale-priced product records as `IEnumerable<Product>`:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_GetOnSaleProducts":::

The `IAsyncEnumerable<Product>` equivalent of the preceding action is:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_GetOnSaleProductsAsync":::

## IActionResult type

The <xref:Microsoft.AspNetCore.Mvc.IActionResult> return type is appropriate when multiple `ActionResult` return types are possible in an action. The `ActionResult` types represent various HTTP status codes. Any non-abstract class deriving from `ActionResult` qualifies as a valid return type. Some common return types in this category are <xref:Microsoft.AspNetCore.Mvc.BadRequestResult> (400), <xref:Microsoft.AspNetCore.Mvc.NotFoundResult> (404), and <xref:Microsoft.AspNetCore.Mvc.OkObjectResult> (200). Alternatively, convenience methods in the <xref:Microsoft.AspNetCore.Mvc.ControllerBase> class can be used to return `ActionResult` types from an action. For example, `return BadRequest();` is a shorthand form of `return new BadRequestResult();`.

Because there are multiple return types and paths in this type of action, liberal use of the [`[ProducesResponseType]`](xref:Microsoft.AspNetCore.Mvc.ProducesResponseTypeAttribute) attribute is necessary. This attribute produces more descriptive response details for web API help pages generated by tools like [Swagger](xref:tutorials/web-api-help-pages-using-swagger). `[ProducesResponseType]` indicates the known types and HTTP status codes to be returned by the action.

### Synchronous action

Consider the following synchronous action in which there are two possible return types:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_GetByIdIActionResult" highlight="8,11":::

In the preceding action:

* A 404 status code is returned when the product represented by `id` doesn't exist in the underlying data store. The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.NotFound%2A> convenience method is invoked as shorthand for `return new NotFoundResult();`.
* A 200 status code is returned with the `Product` object when the product does exist. The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.Ok%2A> convenience method is invoked as shorthand for `return new OkObjectResult(product);`.

### Asynchronous action

Consider the following asynchronous action in which there are two possible return types:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_CreateAsyncIActionResult" highlight="9,14":::

In the preceding action:

* A 400 status code is returned when the product description contains "XYZ Widget". The <xref:Microsoft.AspNetCore.Mvc.ControllerBase.BadRequest%2A> convenience method is invoked as shorthand for `return new BadRequestResult();`.
* A 201 status code is generated by the <xref:Microsoft.AspNetCore.Mvc.ControllerBase.CreatedAtAction%2A> convenience method when a product is created. An alternative to calling `CreatedAtAction` is `return new CreatedAtActionResult(nameof(GetById), "Products", new { id = product.Id }, product);`. In this code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

For example, the following model indicates that requests must include the `Name` and `Description` properties. Failure to provide `Name` and `Description` in the request causes model validation to fail.

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.DataAccess/Models/Product.cs" id="snippet_ProductClass" highlight="5-6,8-9":::

If the [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute is applied, model validation errors result in a 400 status code. For more information, see [Automatic HTTP 400 responses](xref:web-api/index#automatic-http-400-responses).

## ActionResult vs IActionResult

The following section compares `ActionResult` to  `IActionResult`

### ActionResult\<T> type

ASP.NET Core includes the [ActionResult\<T>](xref:Microsoft.AspNetCore.Mvc.ActionResult%601) return type for web API controller actions. It enables you to return a type deriving from <xref:Microsoft.AspNetCore.Mvc.ActionResult> or return a [specific type](#specific-type). `ActionResult<T>` offers the following benefits over the [IActionResult type](#iactionresult-type):

* The [`[ProducesResponseType]`](xref:Microsoft.AspNetCore.Mvc.ProducesResponseTypeAttribute) attribute's `Type` property can be excluded. For example, `[ProducesResponseType(200, Type = typeof(Product))]` is simplified to `[ProducesResponseType(200)]`. The action's expected return type is instead inferred from the `T` in `ActionResult<T>`.
* [Implicit cast operators](/dotnet/csharp/language-reference/keywords/implicit) support the conversion of both `T` and `ActionResult` to `ActionResult<T>`. `T` converts to <xref:Microsoft.AspNetCore.Mvc.ObjectResult>, which means `return new ObjectResult(T);` is simplified to `return T;`.

C# doesn't support implicit cast operators on interfaces. Consequently, conversion of the interface to a concrete type is necessary to use `ActionResult<T>`. For example, use of `IEnumerable` in the following example doesn't work:

```csharp
[HttpGet]
public ActionResult<IEnumerable<Product>> Get() =>
    _repository.GetProducts();
```

One option to fix the preceding code is to return `_repository.GetProducts().ToList();`.

Most actions have a specific return type. Unexpected conditions can occur during action execution, in which case the specific type isn't returned. For example, an action's input parameter may fail model validation. In such a case, it's common to return the appropriate `ActionResult` type instead of the specific type.

### Synchronous action

Consider a synchronous action in which there are two possible return types:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_GetByIdActionResultOfT" highlight="8,11":::

In the preceding action:

* A 404 status code is returned when the product doesn't exist in the database.
* A 200 status code is returned with the corresponding `Product` object when the product does exist.

### Asynchronous action

Consider an asynchronous action in which there are two possible return types:

:::code language="csharp" source="~/web-api/action-return-types/samples/3.x/WebApiSample.Api.31/Controllers/ProductsController.cs" id="snippet_CreateAsyncActionResultOfT" highlight="9,14":::

In the preceding action:

* A 400 status code (<xref:Microsoft.AspNetCore.Mvc.ControllerBase.BadRequest%2A>) is returned by the ASP.NET Core runtime when:
  * The [`[ApiController]`](xref:Microsoft.AspNetCore.Mvc.ApiControllerAttribute) attribute has been applied and model validation fails.
  * The product description contains "XYZ Widget".
* A 201 status code is generated by the <xref:Microsoft.AspNetCore.Mvc.ControllerBase.CreatedAtAction%2A> method when a product is created. In this code path, the `Product` object is provided in the response body. A `Location` response header containing the newly created product's URL is provided.

## Additional resources

* <xref:mvc/controllers/actions>
* <xref:mvc/models/validation>
* <xref:tutorials/web-api-help-pages-using-swagger>

:::moniker-end
