---
title: Choose an identity management solution
description: A guide to help choose the right identity management solution for a secure .NET web app.
author: JeremyLikness
ms.author: jeliknes
ms.topic: how-to 
ms.date: 07/25/2023
uid: security/how-to-choose-identity
---

# Choose an identity management solution

Most web apps support authentication to ensure that users are who they claim to be. A *user* might be a person or another app. Management of access ensures users are only able to see and modify the information they're authorized to see and modify. For example, an end user shouldn't have access to the administrative section of a website. Identity management solutions are built to handle the requirements of authentication and authorization-related tasks. To learn more about identity management, see [What is identity and access management?](/azure/active-directory/fundamentals/introduction-identity-access-management). Many [identity management solutions for .NET web apps](xref:security/identity-management-solutions) are available, each with different capabilities and requirements to use or install. This article provides guidance on how to choose the right solution.

## Basic identity management with ASP.NET Core Identity

ASP.NET Core ships with a built-in authentication provider: [ASP.NET Core Identity](xref:security/authentication/identity). The provider includes the APIs, UI, and backend database configuration to support managing user identities, storing user credentials, and granting or revoking permissions. Other features it supports include:

* [External logins](xref:security/authentication/social/index)
* [Two-factor authentication (2FA)](xref:security/authentication/2fa)
* [Password management](xref:security/authentication/accconfirm)
* Account lockout and reactivation
* Authenticator apps

For most scenarios, this may be the only provider needed.

To learn more:

* Read the [Introduction to Identity on ASP.NET Core](xref:security/authentication/identity)
* Follow a tutorial to build your own secure .NET web app: [Secure a .NET web app with the ASP.NET Core Identity framework](/training/modules/secure-aspnet-core-identity/).

In other scenarios, a server or service that manages authentication and identity may be beneficial.

## Determine if an OIDC server is needed

Web apps require a way to *remember* past actions because the web, by default, is stateless. Otherwise, users would be forced to enter their credentials every time they navigated to a new page. The common solution for remembering state is *cookies*, a browser-based mechanism for storing data. The web server sends the initial cookie, then the browser stores it and sends it back with each request. This is done automatically without the need for the developer to write any code. Cookies are easy to use and built into the browser but are designed for use within a single website or web domain. The default solution that is built into ASP.NET Core uses cookie-based authentication.

Tokens are containers with metadata that are explicitly passed through the headers or body of HTTP requests. The main advantage of tokens over cookies is that they are not tied to a specific app or domain. Instead, tokens are usually *signed* with asymmetric cryptography. For example, OIDC servers issue tokens with information about identity using the [JSON Web Token (JWT)](https://en.wikipedia.org/wiki/JSON_Web_Token) format which includes signing. Asymmetric cryptography uses a combination of a private key known only to the signer, and a public key which everyone can know. Tokens may also be encrypted. 

The signed token can't be tampered with due to the private key. The public key:

* Makes it possible to validate the token to ensure it hasn't been changed.
* Guarantees that it was generated by the entity holding the private key.

The main disadvantage to using tokens is that they require a service (typically an OIDC server) to both issue and provide validation for tokens. The service must be installed, configured, and maintained.

A common reason an OIDC server is required is for applications that expose web-based APIs that are consumed by other apps. For exposed web-based APIs, client UIs such as Single Page Applications (SPA), mobile clients, and desktop clients are considered to be part of the same app. SPA examples include Angular, React, and Blazor WebAssembly. If apps other than your web app or any client UIs must make a secure API call to your app, you'll likely want to use tokens. If you only have client UIs, ASP.NET Core Identity provides the option to acquire a token during authentication. The authentication token issued by ASP.NET Core Identity:

* Can be used by mobile and desktop clients. Cookies are preferred over tokens for both security and simplicity.
* Isn't suitable for managing access from third-party apps.

Another reason an OIDC server is required is for sharing sign-ins with other apps. Commonly referred to as *single sign on*, this feature enables users to:

* Sign in once with a web app's form.
* Use the resulting credentials to authenticate with other apps without having to sign-in again or choose a different password.

An OIDC server is typically preferred to provide a secure and scalable solution for single sign on. 

For apps that don't share logins with other apps, the simplest way to quickly secure an app is to use the built-in ASP.NET Core Identity provider. Otherwise, an OIDC server provided by a third-party identity management solution is needed. OIDC servers are available as:

* Products you install on your server, called *self-host*.
* Containers run in a host like Docker.
* Web-based services you integrate with to manage identity.

Some solutions are free and open source, while others are commercially licensed. See [identity management solutions](xref:security/identity-management-solutions) for a list of available options. It's possible that your organization already uses an identity provider. In that case, it may make sense to use the existing provider instead of going with a different solution. All of the major solutions provide documentation for configuring ASP.NET Core to use their product or service.

## Disconnected scenarios

Many solutions, such as [Microsoft Entra ID](/azure/active-directory/fundamentals/active-directory-whatis), are cloud-based and require an Internet connection to work. If your environment doesn't allow Internet connectivity, you won't be able to use the service. 

ASP.NET Core Identity works perfectly well in disconnected scenarios, such as:

* The app can't access the Internet.
* The app must still function on the local network even if the Internet is disconnected.

If you require a full OIDC server for a disconnected scenario, choose one of the following options:

* A solution that allows you to install and run the service on your own machines.
* Run the authentication service locally as a container.

## Decide where user data such as sign-ins are stored

Another important factor to consider is where the user sign-in data is stored. Many developers choose external, cloud-based services like Microsoft Entra ID to manage identity. A cloud-based service provider:

* Takes on the responsibilities of securely storing data.
* keeps the software up to date with the latest security patches and releases.
* Complies with privacy regulations.

Others prefer to store data on their own servers due to regulations, compliance, policy, or other reasons.

If the data is stored on your servers, you'll most likely need to choose an installable or container-based solution.

## Identity vs OIDC server

Use the following diagram to help you decide whether to use the ASP.NET Core Identity system or an OIDC server for authentication and authorization:

![Identity management decision flow](./_static/identity-management-decision-flow.png)

The following table lists some of the things to consider when choosing your identity management solution.

|**Feature**  |**Self-host (infrastructure or container)**  |**Cloud** |
|--------------|--------------------------------|------------|
|**App integration**|Local solutions that are implemented as libraries or frameworks can often be integrated directly in your own app. Container-based solutions require a hand-off to occur between your web app and the container-based service.|Cloud-based solutions typically integrate at specific points in your sign-in flow and provide configuration to update the UI to match your theme, but the level of customization available is limited.|
|**Configuration**|Self host solutions require configuring the software for the environment in addition to setting up how you want to manage identities. Container-based solutions typically provide a web-based UI for configuration.|Cloud-based solutions typically provide a web-based UI for configuration.|
|**Customization**|Self-host solutions are usually highly customizable, including code-based changes. Although containerized solutions provide extensibility options, they are often more limited.|Cloud-based services allow customization, but it's typically limited to configuration-based changes.|
|**Maintenance**|Installed products require a dedicated resource to ensure all security patches are applied in a timely fashion and to manage upgrades. The upgrade and patch process for containers is usually lower-friction and involves simply installing the provided container image.|The service provider maintains their cloud-based solution, including applying needed patches and handling upgrades.|
|**User credentials storage**|You are responsible for data governance and handling breaches.|Managing the risks associated with handling user credentials, and complying with regulations. is delegated to the service provider.|

For more information about available options, see [Identity management solutions for ASP.NET Core](xref:security/identity-management-solutions).

## Next steps

* Learn about [JSON Web Tokens](https://auth0.com/docs/secure/tokens/json-web-tokens)
* Browse [sample apps with authentication/authorization and identity for ASP.NET Core](/aspnet/core/security/authentication/samples).
* Follow a tutorial to [secure a .NET web app using built-in ASP.NET Core Identity](/training/modules/secure-aspnet-core-identity).
* Learn more about how to [protect web APIs](/azure/active-directory/develop/index-web-api).
