---
title: Consumer APIs overview for ASP.NET Core
author: tdykstra
description: Receive a brief overview of the various consumer APIs available within the ASP.NET Core data protection library.
ms.author: tdykstra
ms.date: 06/11/2019
uid: security/data-protection/consumer-apis/overview
---
# Consumer APIs overview for ASP.NET Core

The `IDataProtectionProvider` and `IDataProtector` interfaces are the basic interfaces through which consumers use the data protection system. They're located in the [Microsoft.AspNetCore.DataProtection.Abstractions](https://www.nuget.org/packages/Microsoft.AspNetCore.DataProtection.Abstractions/) package.

## IDataProtectionProvider

The provider interface represents the root of the data protection system. It cannot directly be used to protect or unprotect data. Instead, the consumer must get a reference to an `IDataProtector` by calling `IDataProtectionProvider.CreateProtector(purpose)`, where purpose is a string that describes the intended consumer use case. See [Purpose Strings](xref:security/data-protection/consumer-apis/purpose-strings) for much more information on the intent of this parameter and how to choose an appropriate value.

## IDataProtector

The protector interface is returned by a call to `CreateProtector`, and it's this interface which consumers can use to perform protect and unprotect operations.

To protect a piece of data, pass the data to the `Protect` method. The basic interface defines a method which converts byte[] -> byte[], but there's also an overload (provided as an extension method) which converts string -> string. The security offered by the two methods is identical; the developer should choose whichever overload is most convenient for their use case. Irrespective of the overload chosen, the value returned by the Protect method is now protected (enciphered and tamper-proofed), and the application can send it to an untrusted client.

To unprotect a previously-protected piece of data, pass the protected data to the `Unprotect` method. (There are byte[]-based and string-based overloads for developer convenience.) If the protected payload was generated by an earlier call to `Protect` on this same `IDataProtector`, the `Unprotect` method will return the original unprotected payload. If the protected payload has been tampered with or was produced by a different `IDataProtector`, the `Unprotect` method will throw CryptographicException.

The concept of same vs. different `IDataProtector` ties back to the concept of purpose. If two `IDataProtector` instances were generated from the same root `IDataProtectionProvider` but via different purpose strings in the call to `IDataProtectionProvider.CreateProtector`, then they're considered [different protectors](xref:security/data-protection/consumer-apis/purpose-strings), and one won't be able to unprotect payloads generated by the other.

## Consuming these interfaces

For a DI-aware component, the intended usage is that the component takes an `IDataProtectionProvider` parameter in its constructor and that the DI system automatically provides this service when the component is instantiated.

> [!NOTE]
> Some applications (such as console applications or ASP.NET 4.x applications) might not be DI-aware so cannot use the mechanism described here. For these scenarios consult the [Non DI Aware Scenarios](xref:security/data-protection/configuration/non-di-scenarios) document for more information on getting an instance of an `IDataProtection` provider without going through DI.

The following sample demonstrates three concepts:

1. [Add the data protection system](xref:security/data-protection/configuration/overview) to the service container,

2. Using DI to receive an instance of an `IDataProtectionProvider`, and

3. Creating an `IDataProtector` from an `IDataProtectionProvider` and using it to protect and unprotect data.

## Console app

[!code-csharp[](../using-data-protection/samples/protectunprotect.cs?highlight=26,34,35,36,37,38,39,40)]

## Web app
Call <xref:Microsoft.Extensions.DependencyInjection.DataProtectionServiceCollectionExtensions.AddDataProtection(Microsoft.Extensions.DependencyInjection.IServiceCollection,System.Action{Microsoft.AspNetCore.DataProtection.DataProtectionOptions})> in `Program.cs`:

[!code-csharp[](~/security/data-protection/using-data-protection/samples/webappexample1.cs?highlight=5)]

The following highlighted code shows how to use <xref:Microsoft.AspNetCore.DataProtection.IDataProtector> in a controller:

[!code-csharp[](~/security/data-protection/using-data-protection/samples/webappexample2.cs?highlight=3,4,5,6,7,8,18,22)]

The package `Microsoft.AspNetCore.DataProtection.Abstractions` contains an extension method <xref:Microsoft.AspNetCore.DataProtection.DataProtectionCommonExtensions.GetDataProtector%2A> as a developer convenience. It encapsulates as a single operation both retrieving an <xref:Microsoft.AspNetCore.DataProtection.IDataProtectionProvider> from the service provider and calling [`IDataProtectionProvider.CreateProtector`](xref:Microsoft.AspNetCore.DataProtection.IDataProtectionProvider.CreateProtector%2A). The following sample demonstrates its usage:

[!code-csharp[](./overview/samples/getdataprotector.cs?highlight=15)]

>[!TIP]
> Instances of `IDataProtectionProvider` and `IDataProtector` are thread-safe for multiple callers. It's intended that once a component gets a reference to an `IDataProtector` via a call to `CreateProtector`, it will use that reference for multiple calls to `Protect` and `Unprotect`. A call to `Unprotect` will throw CryptographicException if the protected payload cannot be verified or deciphered. Some components may wish to ignore errors during unprotect operations; a component which reads authentication cookies might handle this error and treat the request as if it had no cookie at all rather than fail the request outright. Components which want this behavior should specifically catch CryptographicException instead of swallowing all exceptions.
