### Intellisense completion enhancements for OpenAPI package

ASP.NET Core's OpenAPI support is now more accessible and user-friendly. The OpenAPI APIs are shipped as an independent package, separate from the shared framework. Until now, this meant that developers didn't have the convenience of code-completion features like Intellisense for OpenAPI APIs.

Recognizing this gap, we have introduced a new completion provider and code fixer. These tools are designed to facilitate the discovery and use of OpenAPI APIs, making it easier for developers to integrate OpenAPI into their projects. The completion provider offers real-time code suggestions, while the code fixer assists in correcting common mistakes and improving API usage. This enhancement is part of our ongoing commitment to improve the developer experience and streamline API-related workflows.

When a user types a statement where an OpenAPI-related API is available, the completion provider displays a recommendation for the API. For example, in the following screenshots, completions for [AddOpenApi](https://source.dot.net/#Microsoft.AspNetCore.OpenApi/Extensions/OpenApiServiceCollectionExtensions.cs,61) and [MapOpenApi](https://source.dot.net/#Microsoft.AspNetCore.OpenApi/Extensions/OpenApiEndpointRouteBuilderExtensions.cs,da6124cfb6e2f8d8) are provided when a user is entering an invocation statement on a supported type, such as [IEndpointConventionBuilder](/dotnet/api/microsoft.aspnetcore.builder.iendpointconventionbuilder):

<img width="508" alt="OpenAPI completions" src="https://github.com/dotnet/AspNetCore.Docs/assets/1857993/1ed9fcb9-7ab7-4239-b7c8-d323724e7222">

When the completion is accepted and the `Microsoft.AspNetCore.OpenApi` package is not installed, a codefixer provides a shortcut for automatically installing the dependency in the project.

<img width="586" alt="Automatic package installation" src="https://github.com/dotnet/AspNetCore.Docs/assets/1857993/675baf6f-147f-48a3-bd51-18a0b4eebdf4">

### Support for `[Required]` and `[DefaultValue]` attributes on parameters and properties

When [`[Required]`](/dotnet/api/system.componentmodel.dataannotations.requiredattribute) and [`[DefaultValue]`](/dotnet/api/system.componentmodel.defaultvalueattribute) attributes are applied on parameters or properties within complex types, the OpenAPI implementation maps these to the `required` and `default` properties in the OpenAPI document associated with the parameter or type schema.

For example, the following API produces the accompanying schema for the `Todo` type.

```csharp
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;

var builder = WebApplication.CreateBuilder();

builder.Services.AddOpenApi();

var app = builder.Build();

if (app.Environment.IsDevelopment())
{
    app.MapOpenApi();
}

app.MapPost("/todos", (Todo todo) => { });

app.Run();

class Todo
{
	public int Id { get; init; }
	public required string Title { get; init; }
	[DefaultValue("A new todo")]
	public required string Description { get; init; }
	[Required]
	public DateTime CreatedOn { get; init; }
}
```

```json
{
	"required": [
	  "title",
	  "description",
	  "createdOn"
	],
	"type": "object",
	"properties": {
	  "id": {
	    "type": "integer",
	    "format": "int32"
	  },
	  "title": {
	    "type": "string"
	  },
	  "description": {
	    "type": "string",
	    "default": "A new todo"
	  },
	  "createdOn": {
	    "type": "string",
	    "format": "date-time"
	  }
	}
}
```

### Support for schema transformers on OpenAPI documents

Built-in OpenAPI support now ships with support for schema transformers that can be used to modify schemas generated by [System.Text.Json](/dotnet/api/system.text.json) and the OpenAPI implementation. Like document and operation transformers, schema transformers can be registered on the [OpenApiOptions](https://source.dot.net/#Microsoft.AspNetCore.OpenApi/Services/OpenApiOptions.cs,c0a8b420f4ce6918) object. For example, the following code sample demonstrates using a schema transformer to add an example to a type's schema.

```csharp
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using Microsoft.OpenApi.Any;

var builder = WebApplication.CreateBuilder();

builder.Services.AddOpenApi(options =>
{
    options.AddSchemaTransformer((schema, context, cancellationToken) =>
    {
        if (context.JsonTypeInfo.Type == typeof(Todo))
        {
            schema.Example = new OpenApiObject
            {
                ["id"] = new OpenApiInteger(1),
                ["title"] = new OpenApiString("A short title"),
                ["description"] = new OpenApiString("A long description"),
                ["createdOn"] = new OpenApiDateTime(DateTime.Now)
            };
        }
        return Task.CompletedTask;
    });
});

var app = builder.Build();

if (app.Environment.IsDevelopment())
{
    app.MapOpenApi();
}

app.MapPost("/todos", (Todo todo) => { });

app.Run();

class Todo
{
	public int Id { get; init; }
	public required string Title { get; init; }
	[DefaultValue("A new todo")]
	public required string Description { get; init; }
	[Required]
	public DateTime CreatedOn { get; init; }
}
```
