---
title: Migrate from ASP.NET Core in .NET 8 to ASP.NET Core in .NET 9
author: rick-anderson
description: Learn how to migrate an ASP.NET Core in .NET 8 to ASP.NET Core in .NET 9
ms.author: riande
ms.date: 2/11/2024
uid: migration/80-to-90
---

<!-- New content should be added to ~/migration/includes/aspnetcore-9/includes/{FILE}.md files. This will help prevent merge conflicts in this file. -->

# Migrate from ASP.NET Core in .NET 8 to ASP.NET Core in .NET 9

This article explains how to update an ASP.NET Core in .NET 8 to ASP.NET Core in .NET 9.

## Prerequisites

# [Visual Studio](#tab/visual-studio)

[!INCLUDE[](~/includes/net-prereqs-vs-9.0.md)]

# [Visual Studio Code](#tab/visual-studio-code)

[!INCLUDE[](~/includes/net-prereqs-vsc-9.0.md)]

---

## Update the .NET SDK version in `global.json`

If you rely on a [`global.json`](/dotnet/core/tools/global-json) file to target a specific .NET Core SDK version, update the `version` property to the .NET 9.0 SDK version that's installed. For example:

```diff
{
  "sdk": {
-    "version": "8.0.100"
+    "version": "9.0.100"
  }
}
```

## Update the target framework

Update the project file's [Target Framework Moniker (TFM)](/dotnet/standard/frameworks) to `net9.0`:

```diff
<Project Sdk="Microsoft.NET.Sdk.Web">

  <PropertyGroup>
-    <TargetFramework>net8.0</TargetFramework>
+    <TargetFramework>net9.0</TargetFramework>
  </PropertyGroup>

</Project>
```

## Update package references

In the project file, update each [`Microsoft.AspNetCore.*`](https://www.nuget.org/packages?q=Microsoft.AspNetCore.*), [`Microsoft.EntityFrameworkCore.*`](https://www.nuget.org/packages?q=Microsoft.EntityFrameworkCore.*), [`Microsoft.Extensions.*`](https://www.nuget.org/packages?q=Microsoft.Extensions.*), and [`System.Net.Http.Json`](https://www.nuget.org/packages/System.Net.Http.Json) package reference's `Version` attribute to 9.0.0 or later. For example:

```diff
<ItemGroup>
-   <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="8.0.2" />
-   <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="8.0.2" />
-   <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="8.0.0" />
-   <PackageReference Include="System.Net.Http.Json" Version="8.0.0" />
+   <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="9.0.0" />
+   <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="9.0.0" />
+   <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="9.0.0" />
+   <PackageReference Include="System.Net.Http.Json" Version="9.0.0" />
</ItemGroup>
```

## Replace `UseStaticFiles` with `MapStaticAssets`

Optimize the handling of static files in your web apps by replacing <xref:Microsoft.AspNetCore.Builder.StaticFileExtensions.UseStaticFiles%2A> with <xref:Microsoft.AspNetCore.Builder.StaticAssetsEndpointRouteBuilderExtensions.MapStaticAssets%2A> in the app's `Program` file:
  
```diff
- app.UseStaticFiles();
+ app.MapStaticAssets();
```

In MVC & Razor Pages apps you additionally need to chain a call to `.WithStaticAssets` after `MapRazorPages` or `MapControllerRoute` in `Program.cs`. For an example, see the <xref:fundamentals/static-files?view=aspnetcore-9.0&preserve-view=true>. 

ASP.NET Core automatically fingerprints and precompresses your static files at build and publish time, and then <xref:Microsoft.AspNetCore.Builder.StaticAssetsEndpointRouteBuilderExtensions.MapStaticAssets%2A> surfaces the optimized files as endpoints using endpoint routing with appropriate caching headers.

To resolve the fingerprinted file names from your app:

* In Blazor apps, use the <xref:Microsoft.AspNetCore.Components.ComponentBase.Assets?displayProperty=nameWithType> property. Update explicit references to static assets in Razor component files (`.razor`) to use `@Assets["{ASSET PATH}"]`, where the `{ASSET PATH}` placeholder is the path to the asset. Note that this should ***NOT*** be done for the Blazor framework scripts (`blazor.*.js`). In the following example, Bootstrap, the Blazor project template app stylesheet (`app.css`), and the [CSS isolation stylesheet](xref:blazor/components/css-isolation) (based on an app's namespace of `BlazorSample`) are linked in a root component, typically the `App` component (`Components/App.razor`):

    ```razor
    <link rel="stylesheet" href="@Assets["bootstrap/bootstrap.min.css"]" />
    <link rel="stylesheet" href="@Assets["app.css"]" />
    <link rel="stylesheet" href="@Assets["BlazorSample.styles.css"]" />
    ```

* In MVC & Razor Pages apps, the script and link tag helpers will automatically resolve the fingerprinted file names.

To resolve the fingerprinted file names when importing JavaScript modules, add a generated [import map](https://developer.mozilla.org/docs/Web/HTML/Element/script/type/importmap):

* In Blazor apps, add the (<xref:Microsoft.AspNetCore.Components.ImportMap>) component to the `<head>` content of the app's root component, typically in the `App` component (`App.razor`):

   ```razor
   <ImportMap />
   ```

* In MVC & Razor pages apps, add `<script type="importmap"></script>` to the head of the main layout file, which is updated by the Import Map Tag Helper.
 
For more information, see the following resources:

* <xref:aspnetcore-9#static-asset-delivery-optimization>
* <xref:blazor/fundamentals/static-files>

## Blazor

[!INCLUDE[](~/migration/80-to-90/includes/blazor.md)]

## Additional resources
