---
title: Migrate from ASP.NET Core 6.0 to 7.0
author: rick-anderson
description: Learn how to migrate an ASP.NET Core 6 project to ASP.NET Core 7.0.
ms.author: riande
ms.date: 10/14/2022
uid: migration/60-to-70
---
# Migrate from ASP.NET Core 6.0 to 7.0

This article explains how to update an existing ASP.NET Core 6.0 project to ASP.NET Core 7.0.

## Prerequisites

# [Visual Studio](#tab/visual-studio)

[!INCLUDE[](~/includes/net-prereqs-vs-7.0.md)]

# [Visual Studio Code](#tab/visual-studio-code)

[!INCLUDE[](~/includes/net-prereqs-vsc-7.0.md)]

# [Visual Studio for Mac](#tab/visual-studio-mac)

[!INCLUDE[](~/includes/net-prereqs-mac-7.0.md)]

---

## Update .NET Core SDK version in global.json

If you rely on a [global.json](/dotnet/core/tools/global-json) file to target a specific .NET Core SDK version, update the `version` property to the .NET 7.0 SDK version that's installed. For example:

```diff
{
  "sdk": {
-    "version": "6.0.200"
+    "version": "7.0.100"
  }
}
```

## Update the target framework

Update the project file's [Target Framework Moniker (TFM)](/dotnet/standard/frameworks) to `net7.0`:

```diff
<Project Sdk="Microsoft.NET.Sdk.Web">

  <PropertyGroup>
-    <TargetFramework>net6.0</TargetFramework>
+    <TargetFramework>net7.0</TargetFramework>
  </PropertyGroup>

</Project>
```

## Update package references

In the project file, update each [Microsoft.AspNetCore.*](https://www.nuget.org/packages?q=Microsoft.AspNetCore.*), [Microsoft.EntityFrameworkCore.*](https://www.nuget.org/packages?q=Microsoft.EntityFrameworkCore.*), [Microsoft.Extensions.*](https://www.nuget.org/packages?q=Microsoft.Extensions.*), and [System.Net.Http.Json](https://www.nuget.org/packages/System.Net.Http.Json) package reference's `Version` attribute to 7.0.0 or later. For example:

```diff
<ItemGroup>
-    <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="6.0.9" />
-    <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="6.0.9" />
-    <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="6.0.9" />
-    <PackageReference Include="System.Net.Http.Json" Version="6.0.0" />
+    <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="7.0.0" />
+    <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="7.0.0" />
+    <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="7.0.0" />
+    <PackageReference Include="System.Net.Http.Json" Version="7.0.0" />
</ItemGroup>
```

## Blazor

### Adopt .NET 7 features

After following the guidance earlier in this article to update an app to 7.0, adopt specific features by following the links in <xref:aspnetcore-7#blazor>.

To adopt all of the [new 7.0 features for Blazor apps](xref:aspnetcore-7#blazor), we recommend the following process:

* Create a new 7.0 Blazor project from one of the Blazor project templates. For more information, see <xref:blazor/tooling>.
* Move the app's components and code to the 7.0 app making modifications to adopt the new 7.0 features.

### Simplify component parameter binding

In prior Blazor releases, binding across multiple components required binding to properties with `get`/`set` accessors.

In .NET 6 and earlier:

```razor
<NestedGrandchild @bind-GrandchildMessage="BoundValue" />

@code {
    ...

    private string BoundValue
    {
        get => ChildMessage ?? string.Empty;
        set => ChildMessageChanged.InvokeAsync(value);
    }
}
```

In .NET 7, you can use the new `@bind:get` and `@bind:set` modifiers to support two-way data binding and simplify the binding syntax:

```razor
<NestedGrandchild @bind-GrandchildMessage:get="ChildMessage" 
    @bind-GrandchildMessage:set="ChildMessageChanged" />
```

For more information, see the following content in the *Data binding* article:

* [Introduction](xref:blazor/components/data-binding?view=aspnetcore-7.0)
* [Bind across more than two components](xref:blazor/components/data-binding?view=aspnetcore-7.0#bind-across-more-than-two-components)

### Migrate unmarshalled JavaScript interop

Unmarshalled interop using the <xref:Microsoft.JSInterop.IJSUnmarshalledRuntime> interface is obsolete and should be replaced with JavaScript `[JSImport]`/`[JSExport]` interop.

For more information, see <xref:blazor/js-interop/import-export-interop>.

### Blazor WebAssembly authentication uses history state for redirects

The support for authentication in Blazor WebAssembly apps changed to rely on [navigation history state](xref:blazor/fundamentals/routing?view=aspnetcore-7.0#navigation-history-state) instead of query strings in the URL. As a result, passing the return URL through the query string fails to redirect back to the original page after a successful login in .NET 7.

The following example demonstrates **the prior redirection approach for apps that target .NET 6 or earlier** in `RedirectToLogin.razor`, which is based on a redirect URL (`?returnUrl=`) with <xref:Microsoft.AspNetCore.Components.NavigationManager.NavigateTo%2A>:

```razor
@inject NavigationManager Navigation
@using Microsoft.AspNetCore.Components.WebAssembly.Authentication
@code {
    protected override void OnInitialized()
    {
        Navigation.NavigateTo(
            $"authentication/login?returnUrl={Uri.EscapeDataString(Navigation.Uri)}");
    }
}
```

The following example demonstrates **the new redirection approach for apps that target .NET 7 or later** in `RedirectToLogin.razor`, which is based on [navigation history state](xref:blazor/fundamentals/routing?view=aspnetcore-7.0#navigation-history-state) with <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.NavigationManagerExtensions.NavigateToLogin%2A>:

```razor
@inject NavigationManager Navigation
@using Microsoft.AspNetCore.Components.WebAssembly.Authentication
@using Microsoft.Extensions.Options

@inject IOptionsSnapshot<RemoteAuthenticationOptions<ApiAuthorizationProviderOptions>> OptionsSnapshot
@code {
    protected override void OnInitialized()
    {
        Navigation.NavigateToLogin(OptionsSnapshot.Get(Options.DefaultName).AuthenticationPaths.LogInPath);
    }
}
```

As part of this change, <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.SignOutSessionStateManager> is obsolete in .NET 7 or later and replaced with <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.NavigationManagerExtensions.NavigateToLogout%2A>.

The following example demonstrates **the prior approach** in `Shared/LoginDisplay.razor` of an app generated from the Blazor WebAssembly project template:

```razor
@inject SignOutSessionStateManager SignOutManager

...

@code{
    private async Task BeginLogout(MouseEventArgs args)
    {
        await SignOutManager.SetSignOutState();
        Navigation.NavigateTo("authentication/logout");
    }
}
```

The following example demonstrates **the new approach** in `Shared/LoginDisplay.razor` that calls <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.NavigationManagerExtensions.NavigateToLogout%2A>. The injection (`@inject`) of the <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.SignOutSessionStateManager> is removed from the component's directives at the top of the file, and the `BeginLogOut` method is updated to the following code:

```razor
@code{
    public void BeginLogOut()
    {
        Navigation.NavigateToLogout("authentication/logout");
    }
}
```

The <xref:Microsoft.AspNetCore.Components.WebAssembly.Authentication.SignOutSessionStateManager> service registration is removed in `Program.cs`:

```diff
- builder.Services.AddScoped<SignOutSessionStateManager>();
```

For more information, see the following resources:

* [[Breaking change]: Updates to Authentication in webassembly applications](https://github.com/aspnet/Announcements/issues/497)
* <xref:blazor/fundamentals/routing?view=aspnetcore-7.0#navigation-history-state>
* <xref:blazor/security/webassembly/index?view=aspnetcore-7.0#customize-authorization>
* <xref:blazor/security/index?view=aspnetcore-7.0#authorizeview-component>

### .NET WebAssembly build tools for .NET 6 projects

You can now use the .NET WebAssembly build tools with a .NET 6 project when working with the .NET 7 SDK. The new `wasm-tools-net6` workload includes the .NET WebAssembly build tools for .NET 6 projects so that they can be used with the .NET 7 SDK. The existing `wasm-tools` workload installs the .NET WebAssembly build tools for .NET 7 projects. However, the .NET 7 version of the .NET WebAssembly build tools are incompatible with existing projects built with .NET 6. Projects using the .NET WebAssembly build tools that need to support both .NET 6 and .NET 7 must use multi-targeting.

## Update Docker images

For apps using Docker, update your *Dockerfile* `FROM` statements and scripts. Use a base image that includes the ASP.NET Core 7.0 runtime. Consider the following `docker pull` command difference between ASP.NET Core 6.0 and 7.0:

```diff
- docker pull mcr.microsoft.com/dotnet/aspnet:6.0
+ docker pull mcr.microsoft.com/dotnet/aspnet:7.0
```

## Review breaking changes

For breaking changes from .NET Core 6.0 to .NET 7.0, see [Breaking changes in .NET 7](/dotnet/core/compatibility/7.0). ASP.NET Core and Entity Framework Core are included in the list.
