---
title: Migrate from ASP.NET Core 3.1 to 6.0
author: rick-anderson
description: Learn how to migrate an ASP.NET Core 3.1 project to ASP.NET Core 6.0.
ms.author: riande
monikerRange: '>= aspnetcore-3.1'
ms.date: 10/25/2021
uid: migration/31-to-60
---
# Migrate from ASP.NET Core 3.1 to 6.0

This article explains how to update an existing ASP.NET Core 3.1 project to ASP.NET Core 6.0. To upgrade from ASP.NET Core 5.0 to 6.0, see <xref:migration/50-to-60>.

## Prerequisites

# [Visual Studio](#tab/visual-studio)

[!INCLUDE[](~/includes/net-prereqs-vs-6.0.md)]

# [Visual Studio Code](#tab/visual-studio-code)

[!INCLUDE[](~/includes/net-prereqs-vsc-6.0.md)]

# [Visual Studio for Mac](#tab/visual-studio-mac)

[!INCLUDE[](~/includes/net-prereqs-mac-6.0.md)]

---

## Update .NET SDK version in `global.json`

If you rely upon a [`global.json`](/dotnet/core/tools/global-json) file to target a specific .NET SDK version, update the `version` property to the .NET 6.0 SDK version that's installed. For example:

```diff
{
  "sdk": {
-    "version": "3.1.200"
+    "version": "6.0.100"
  }
}
```

<a name="tf"></a>

## Update the target framework

Update the project file's [Target Framework Moniker (TFM)](/dotnet/standard/frameworks) to `net6.0`:

```diff
<Project Sdk="Microsoft.NET.Sdk.Web">

  <PropertyGroup>
-    <TargetFramework>netcoreapp3.1</TargetFramework>
+    <TargetFramework>net6.0</TargetFramework>
  </PropertyGroup>

</Project>
```

## Update package references

In the project file, update each [`Microsoft.AspNetCore.*`](https://www.nuget.org/packages?q=Microsoft.AspNetCore.%2A), [`Microsoft.EntityFrameworkCore.*`](https://www.nuget.org/packages?q=Microsoft.EntityFrameworkCore.%2A), [`Microsoft.Extensions.*`](https://www.nuget.org/packages?q=Microsoft.Extensions.%2A), and [`System.Net.Http.Json`](https://www.nuget.org/packages/System.Net.Http.Json) package reference's `Version` attribute to 6.0.0 or later. For example:

```diff
<ItemGroup>
-    <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="3.1.6" />
-    <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="3.1.6" />
-    <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="3.1.6" />
-    <PackageReference Include="System.Net.Http.Json" Version="3.2.1" />
+    <PackageReference Include="Microsoft.AspNetCore.JsonPatch" Version="6.0.0" />
+    <PackageReference Include="Microsoft.EntityFrameworkCore.Tools" Version="6.0.0" />
+    <PackageReference Include="Microsoft.Extensions.Caching.Abstractions" Version="6.0.0" />
+    <PackageReference Include="System.Net.Http.Json" Version="6.0.0" />
</ItemGroup>
```

## Delete `bin` and `obj` folders

You may need to delete the `bin` and `obj` folders. Run `dotnet nuget locals --clear all` to clear the NuGet package cache.

## Minimal hosting model

The ASP.NET Core templates generate code using the new [minimal hosting model](xref:migration/50-to-60#new-hosting-model). The minimal hosting model unifies `Startup.cs` and `Program.cs` into a single `Program.cs` file. `ConfigureServices` and `Configure` are no longer used. Apps migrating from ASP.NET Core 3.1 to 6.0 don't need to use the minimal hosting model, using `Startup` and the [Generic Host](xref:fundamentals/host/generic-host) used by the ASP.NET Core 3.1 templates is fully supported.

To use `Startup` with the new minimal hosting model, see [Use Startup with the new minimal hosting model](xref:migration/50-to-60#smhm).

To migrate to the new minimal hosting model using the following pattern used by the ASP.NET Core 6.0 templates, see [
Code samples migrated to the new minimal hosting model in ASP.NET Core 6.0](xref:migration/50-to-60-samples) and [Migrate from ASP.NET Core 5.0 to 6.0](xref:migration/50-to-60)

## Update Razor class libraries (RCLs)

Migrate Razor class libraries (RCLs) to take advantage of new APIs or features that are introduced as part of ASP.NET Core 6.0.

To update a RCL that targets components:

1. Update the following properties in the project file:

   ```diff
   <Project Sdk="Microsoft.NET.Sdk.Razor">
     <PropertyGroup>
   -     <TargetFramework>netstandard2.0</TargetFramework>
   -     <RazorLangVersion>3.0</RazorLangVersion>
   +     <TargetFramework>net6.0</TargetFramework>
     </PropertyGroup>
   ```

1. Update other packages to their latest versions. The latest versions can be found at [NuGet.org](https://www.nuget.org).

To update an RCL targeting MVC, update the following properties in the project file:

```diff
<Project Sdk="Microsoft.NET.Sdk.Razor">

  <PropertyGroup>
-    <TargetFramework>netcoreapp3.1</TargetFramework>
+    <TargetFramework>net6.0</TargetFramework>
    <AddRazorSupportForMvc>true</AddRazorSupportForMvc>
  </PropertyGroup>
```

## Blazor

To adopt all of the [5.0 features](xref:aspnetcore-5.0) and [6.0 features](xref:aspnetcore-6.0#blazor) for Blazor apps, we recommend the following process:

* Create a new 6.0 Blazor project from one of the Blazor project templates. For more information, see <xref:blazor/tooling>.
* Move the app's components and code to the 6.0 app making modifications to adopt the new 5.0 and 6.0 features.

## Update Docker images

For apps using Docker, update your *Dockerfile* `FROM` statements and scripts. Use a base image that includes the ASP.NET Core 6.0 runtime. Consider the following `docker pull` command difference between ASP.NET Core 3.1 and 6.0:

```diff
- docker pull mcr.microsoft.com/dotnet/core/aspnet:3.1
+ docker pull mcr.microsoft.com/dotnet/aspnet:6.0
```

As part of the move to ".NET" as the product name, the Docker images moved from the `mcr.microsoft.com/dotnet/core` repositories to `mcr.microsoft.com/dotnet`. For more information, see [.NET 5.0 - Docker Repo Name Change (dotnet/dotnet-docker #1939)](https://github.com/dotnet/dotnet-docker/issues/1939).

## Model binding changes in ASP.NET Core MVC and Razor Pages

### `DateTime` values are model bound as UTC times

In ASP.NET Core 3.1 and earlier, `DateTime` values were model-bound as local time, where the timezone was determined by the server. `DateTime` values bound from input formatting (JSON) and `DateTimeOffset` values were bound as UTC timezones.

In ASP.NET Core 5.0 and later, model binding consistently binds `DateTime` values with the UTC timezone.

To retain the previous behavior, remove the `DateTimeModelBinderProvider` in `Startup.ConfigureServices`:

```csharp
services.AddControllersWithViews(options =>
    options.ModelBinderProviders.RemoveType<DateTimeModelBinderProvider>());
```

### `ComplexObjectModelBinderProvider` \ `ComplexObjectModelBinder` replace `ComplexTypeModelBinderProvider` \ `ComplexTypeModelBinder`

To add support for model binding [C# 9 record types](/dotnet/csharp/whats-new/csharp-9#record-types), the <xref:Microsoft.AspNetCore.Mvc.ModelBinding.Binders.ComplexTypeModelBinderProvider> is:

* Annotated as obsolete.
* No longer registered by default.

Apps that rely on the presence of the `ComplexTypeModelBinderProvider` in the `ModelBinderProviders` collection need to reference the new binder provider:

```diff
- var complexModelBinderProvider = options.ModelBinderProviders.OfType<ComplexTypeModelBinderProvider>();
+ var complexModelBinderProvider = options.ModelBinderProviders.OfType<ComplexObjectModelBinderProvider>();
```

## `UseDatabaseErrorPage` obsolete

The ASP.NET Core 3.1 templates that include an option for individual user accounts generate a call to <xref:Microsoft.AspNetCore.Builder.DatabaseErrorPageExtensions.UseDatabaseErrorPage%2A>. `UseDatabaseErrorPage` is now obsolete and should be replaced with a combination of `AddDatabaseDeveloperPageExceptionFilter` and `UseMigrationsEndPoint`, as shown in the following code:

```diff
public void ConfigureServices(IServiceCollection services)
{
    services.AddDbContext<ApplicationDbContext>(options =>
        options.UseSqlServer(
            Configuration.GetConnectionString("DefaultConnection")));
+   services.AddDatabaseDeveloperPageExceptionFilter();
    services.AddDefaultIdentity<IdentityUser>(options => options.SignIn.RequireConfirmedAccount = true)
        .AddEntityFrameworkStores<ApplicationDbContext>();
    services.AddRazorPages();
}

public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
{
    if (env.IsDevelopment())
    {
        app.UseDeveloperExceptionPage();
+       app.UseMigrationsEndPoint();
-       app.UseDatabaseErrorPage();
    }
    else
    {
        app.UseExceptionHandler("/Error");
        app.UseHsts();
    }
```

For more information, see [Obsoleting DatabaseErrorPage middleware (dotnet/aspnetcore #24987)](https://github.com/dotnet/aspnetcore/issues/24987).

## ASP.NET Core Module (ANCM)

[!INCLUDE[](~/includes/hosting-bundle.md)]

[!INCLUDE[](~/includes/appname6.md)]

## Review breaking changes

See the following resources:

* [Identity: Default Bootstrap version of UI changed](/dotnet/core/compatibility/aspnet-core/6.0/identity-bootstrap4-to-5)
* [Breaking changes for migration from version 3.1 to 5.0](/dotnet/core/compatibility/3.1-5.0). ASP.NET Core and Entity Framework Core are also included in the list.
* [Breaking changes for migration from version 5.0 to 6.0](/dotnet/core/compatibility/6.0): Includes ASP.NET Core and Entity Framework Core.
* [Announcements GitHub repository (aspnet/Announcements, `6.0.0` label)](https://github.com/aspnet/Announcements/issues?q=is%3Aissue+label%3A6.0.0+is%3Aopen): Includes breaking and non-breaking information.
* [Announcements GitHub repository (aspnet/Announcements, `5.0.0` label)](https://github.com/aspnet/Announcements/issues?q=is%3Aissue+label%3A5.0.0+is%3Aopen): Includes breaking and non-breaking information.
