---
title: Migrate from ASP.NET Core 2.1 to 2.2
author: rick-anderson
description: This article outlines the prerequisites and most common steps for migrating an ASP.NET Core 2.1 project to ASP.NET Core 2.2.
ms.author: wpickett
ms.custom: mvc
ms.date: 05/28/2019
uid: migration/21-to-22
---
# Migrate from ASP.NET Core 2.1 to 2.2

By [Scott Addie](https://github.com/scottaddie)

This article explains how to update an existing ASP.NET Core 2.1 project to ASP.NET Core 2.2.

## Prerequisites

# [Visual Studio](#tab/visual-studio)

[!INCLUDE[](~/includes/net-core-prereqs-vs2019-2.2.md)]

# [Visual Studio Code](#tab/visual-studio-code)

[!INCLUDE[](~/includes/net-core-prereqs-vsc-2.2.md)]

# [Visual Studio for Mac](#tab/visual-studio-mac)

[!INCLUDE[](~/includes/net-core-prereqs-mac-2.2.md)]

---

## Update Target Framework Moniker (TFM)

Projects targeting .NET Core should use the [TFM](/dotnet/standard/frameworks) of a version greater than or equal to .NET Core 2.2. In the project file, update the `<TargetFramework>` node's inner text with `netcoreapp2.2`:

```xml
<TargetFramework>netcoreapp2.2</TargetFramework>
```

Projects targeting .NET Framework may continue to use the TFM of a version greater than or equal to .NET Framework 4.6.1:

```xml
<TargetFramework>net461</TargetFramework>
```

## Adopt the IIS in-process hosting model

To adopt the [in-process hosting model for IIS](xref:host-and-deploy/iis/index#in-process-hosting-model), add the `<AspNetCoreHostingModel>` property with a value of `InProcess` to a `<PropertyGroup>` in the project file:

```xml
<AspNetCoreHostingModel>InProcess</AspNetCoreHostingModel>
```

The in-process hosting model isn't supported for ASP.NET Core apps targeting .NET Framework.

For more information, see <xref:host-and-deploy/aspnet-core-module#hosting-models>.

## Update a custom web.config file

For projects that use a custom *web.config* file in the project root to generate their published *web.config* file:

* In the `<handlers>` entry that adds the ASP.NET Core Module (`name="aspNetCore"`), change the `modules` attribute value from `AspNetCoreModule` to `AspNetCoreModuleV2`.
* In the `<aspNetCore>` element, add the hosting model attribute (`hostingModel="InProcess"`).

For more information and example *web.config* files, see <xref:host-and-deploy/aspnet-core-module#configuration-with-webconfig>.

## Update package references

If targeting .NET Core, remove the metapackage reference's `Version` attribute in the project file. Inclusion of a `Version` attribute results in the following warning:

```console
A PackageReference to 'Microsoft.AspNetCore.App' specified a Version of `2.2.0`. Specifying the version of this package is not recommended. For more information, see https://aka.ms/sdkimplicitrefs
```

For more information, see <xref:fundamentals/metapackage-app>.

The metapackage reference should resemble the following `<PackageReference />` node:

```xml
<ItemGroup>
  <PackageReference Include="Microsoft.AspNetCore.App" />
</ItemGroup>
```

If targeting .NET Framework, update each package reference's `Version` attribute to 2.2.0 or later. Here are the package references in a typical ASP.NET Core 2.2 project targeting .NET Framework:

```xml
<ItemGroup>
  <PackageReference Include="Microsoft.AspNetCore" Version="2.2.0" />
  <PackageReference Include="Microsoft.AspNetCore.CookiePolicy" Version="2.2.0" />
  <PackageReference Include="Microsoft.AspNetCore.HttpsPolicy" Version="2.2.0" />
  <PackageReference Include="Microsoft.AspNetCore.Mvc" Version="2.2.0" />
  <PackageReference Include="Microsoft.AspNetCore.StaticFiles" Version="2.2.0" />
</ItemGroup>
```

If referencing the [Microsoft.AspNetCore.Razor.Design](https://www.nuget.org/packages/Microsoft.AspNetCore.Razor.Design/) package, update its `Version` attribute to 2.2.0 or later. Failure to do so results in the following error:

```console
Detected package downgrade: Microsoft.AspNetCore.Razor.Design from 2.2.0 to 2.1.2. Reference the package directly from the project to select a different version.
```

## Update .NET Core SDK version in global.json

If your solution relies upon a [global.json](/dotnet/core/tools/global-json) file to target a specific .NET Core SDK version, update its `version` property to the 2.2 version installed on your machine:

```json
{
  "sdk": {
    "version": "2.2.100"
  }
}
```

## Update launch settings

If using Visual Studio Code, update the project's launch settings file (`.vscode/launch.json`). The `program` path should reference the new TFM:

[!code-json[](21-to-22/samples/launch.json?highlight=9)]

## Update Kestrel configuration

If the app calls <xref:Microsoft.AspNetCore.Hosting.WebHostBuilderKestrelExtensions.UseKestrel*> by calling `CreateDefaultBuilder` in the [CreateWebHostBuilder method](xref:fundamentals/host/web-host#set-up-a-host) of the `Program` class, call `ConfigureKestrel` to configure Kestrel server instead of `UseKestrel` in order to avoid conflicts with the [IIS in-process hosting model](xref:host-and-deploy/iis/index#in-process-hosting-model):

```csharp
public static IWebHostBuilder CreateWebHostBuilder(string[] args) =>
    WebHost.CreateDefaultBuilder(args)
        .UseStartup<Startup>()
        .ConfigureKestrel((context, options) =>
        {
            // Set properties and call methods on options
        });
```

If the app doesn't call `CreateDefaultBuilder` and builds the host manually in the `Program` class, call <xref:Microsoft.AspNetCore.Hosting.WebHostBuilderKestrelExtensions.UseKestrel*> **before** calling `ConfigureKestrel`:

```csharp
public static void Main(string[] args)
{
    var host = new WebHostBuilder()
        .UseContentRoot(Directory.GetCurrentDirectory())
        .UseKestrel()
        .UseIISIntegration()
        .UseStartup<Startup>()
        .ConfigureKestrel((context, options) =>
        {
            // Set properties and call methods on options
        })
        .Build();

    host.Run();
}
```

For more information, see <xref:fundamentals/servers/kestrel>.

## Update compatibility version

Update the compatibility version in `Startup.ConfigureServices` to `Version_2_2`:

```csharp
services.AddMvc()
        .SetCompatibilityVersion(CompatibilityVersion.Version_2_2);
```

## Update CORS policy

In ASP.NET Core 2.2, the CORS middleware responds with a wildcard origin (`*`) if a policy allows any origin and allows credentials. Credentials aren't supported when a wildcard origin (`*`) is specified, and browsers will disallow the CORS request. For more information, including options for correcting the problem on the client, see the [MDN web docs](https://developer.mozilla.org/docs/Web/HTTP/CORS/Errors/CORSNotSupportingCredentials).

To correct this problem on the server, take one of the following actions:

* Modify the CORS policy to no longer allow credentials. That is, remove the call to <xref:Microsoft.AspNetCore.Cors.Infrastructure.CorsPolicyBuilder.AllowCredentials*> when configuring the policy.
* If credentials are required for the CORS request to succeed, modify the policy to specify allowed hosts. For example, use `builder.WithOrigins("https://api.example1.com", "https://example2.com")` instead of using <xref:Microsoft.AspNetCore.Cors.Infrastructure.CorsPolicyBuilder.AllowAnyOrigin*>.

## Update Docker images

The following table shows the Docker image tag changes:

| 2.1                                       | 2.2                                        |
| ----------------------------------------- | -----------------------------------------  |
| `microsoft/dotnet:2.1-aspnetcore-runtime` | `mcr.microsoft.com/dotnet/core/aspnet:2.2` |
| `microsoft/dotnet:2.1-sdk`                | `mcr.microsoft.com/dotnet/core/sdk:2.2`    |

Change the `FROM` lines in your *Dockerfile* to use the new image tags in the preceding table's 2.2 column.

## Build manually in Visual Studio when using IIS in-process hosting

Visual Studio's **Auto build on browser request** experience doesn't function with the [IIS in-process hosting model](xref:host-and-deploy/iis/index#in-process-hosting-model). You must manually rebuild the project when using in-process hosting. Improvements to this experience are planned for a future release of Visual Studio.

## Update logging code

Recommended logging configuration code didn't change from 2.1 to 2.2, but some 1.x coding patterns that still worked in 2.1 no longer work in 2.2.

If your app does logging provider initialization, filtering, and configuration loading in the `Startup` class, move that code to `Program.Main`:

* Provider initialization:

  1.x example:

  ```csharp
  public void Configure(IApplicationBuilder app, ILoggerFactory loggerFactory)
  {
      loggerFactory.AddConsole();
  }
  ```

  2.2 example:

  ```csharp

  public static void Main(string[] args)
  {
      var webHost = new WebHostBuilder()
          // ...
          .ConfigureLogging((hostingContext, logging) =>
          {
              logging.AddConsole();
          })
          // ...
  }
  ```

* Filtering:

  1.x example:

  ```csharp
  public void Configure(IApplicationBuilder app, ILoggerFactory loggerFactory)
  {
      loggerFactory.AddConsole(LogLevel.Information);
      // or
      loggerFactory.AddConsole((category, level) => 
          category == "A" || level == LogLevel.Critical);
  }
  ```

  2.2 example:

  ```csharp
  public static void Main(string[] args)
  {
      var webHost = new WebHostBuilder()
          // ...
          .ConfigureLogging((hostingContext, logging) =>
          {
              logging.AddConsole()
                     .AddFilter<ConsoleLoggerProvider>
                         (category: null, level: LogLevel.Information)
                     // or
                     .AddFilter<ConsoleLoggerProvider>
                         ((category, level) => category == "A" ||
                             level == LogLevel.Critical)
              );
          })
          // ...
  }
  ```

* Configuration loading:

  1.x example:

  ```csharp
  public void Configure(IApplicationBuilder app, ILoggerFactory loggerFactory)
  {
      loggerFactory.AddConsole(Configuration);
  }
  ```

  2.2 example:

  ```csharp
  public static void Main(string[] args)
  {
      var webHost = new WebHostBuilder()
          // ...
          .ConfigureLogging((hostingContext, logging) =>
          {
              logging.AddConfiguration(hostingContext.Configuration.GetSection("Logging"));
              logging.AddConsole();
          })
          // ...
  }
  ```

For more information, see <xref:fundamentals/logging/index>

## ASP.NET Core Module (ANCM)

[!INCLUDE[](~/includes/hosting-bundle.md)]

## Additional resources

* <xref:mvc/compatibility-version>
* <xref:fundamentals/metapackage-app>
* [Implicit package references](/dotnet/core/tools/csproj#implicit-package-references)
