---
title: Logging in .NET Core and ASP.NET Core
author: tdykstra
description: Learn how to use the logging framework provided by the Microsoft.Extensions.Logging NuGet package.
monikerRange: '>= aspnetcore-3.1'
ms.author: tdykstra
ms.custom: mvc
ms.date: 9/5/2023
uid: fundamentals/logging/index
---

# Logging in .NET Core and ASP.NET Core

[!INCLUDE[](~/includes/not-latest-version.md)]

:::moniker range=">= aspnetcore-6.0"

By [Kirk Larkin](https://twitter.com/serpent5), [Juergen Gutsch](https://github.com/JuergenGutsch), and [Rick Anderson](https://twitter.com/RickAndMSFT)

This article describes logging in .NET as it applies to ASP.NET Core apps. For detailed information on logging in .NET, see [Logging in .NET](/dotnet/core/extensions/logging).

For Blazor logging guidance, which adds to or supersedes the guidance in this node, see <xref:blazor/fundamentals/logging>.

<a name="lp"></a>

## Logging providers

Logging providers store logs, except for the `Console` provider which displays logs. For example, the Azure Application Insights provider stores logs in [Azure Application Insights](/azure/azure-monitor/app/app-insights-overview). Multiple providers can be enabled.

The default ASP.NET Core web app templates call <xref:Microsoft.AspNetCore.Builder.WebApplication.CreateBuilder%2A?displayProperty=nameWithType>, which adds the following logging providers:

* [Console](#console)
* [Debug](#debug)
* [EventSource](#event-source)
* [EventLog](#welog): Windows only

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet1&highlight=1)]

The preceding code shows the `Program.cs` file created with the ASP.NET Core web app templates. The next several sections provide samples based on the ASP.NET Core web app templates.

The following code overrides the default set of logging providers added by `WebApplication.CreateBuilder`:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet2&highlight=2-3)]

Alternatively, the preceding logging code can be written as follows:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet3)]

For additional providers, see:

* [Built-in logging providers](#bilp)
* [Third-party logging providers](#third-party-logging-providers).

## Create logs

To create logs, use an <xref:Microsoft.Extensions.Logging.ILogger%601> object from [dependency injection (DI)](xref:fundamentals/dependency-injection).

The following example:

* Creates a logger, `ILogger<AboutModel>`, which uses a log *category* of the fully qualified name of the type `AboutModel`. The log category is a string that is associated with each log.
* Calls <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogInformation%2A> to log at the <xref:Microsoft.Extensions.Logging.LogLevel.Information> level. The Log *level* indicates the severity of the logged event.

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/TodoApiDTO/Pages/About.cshtml.cs?name=snippet_CallLogMethods&highlight=5,12-13)]

[Levels](#log-level) and [categories](#log-category) are explained in more detail later in this document.

For information on Blazor, see <xref:blazor/fundamentals/logging>.

## Configure logging

Logging configuration is commonly provided by the `Logging` section of `appsettings.{ENVIRONMENT}.json` files, where the `{ENVIRONMENT}` placeholder is the [environment](xref:fundamentals/environments). The following `appsettings.Development.json` file is generated by the ASP.NET Core web app templates:

[!code-json[](~/fundamentals/logging/index/samples/6.x/TodoApiDTO/appsettings.Development.json)]

In the preceding JSON:

* The `"Default"` and `"Microsoft.AspNetCore"` categories are specified.
* The `"Microsoft.AspNetCore"` category applies to all categories that start with `"Microsoft.AspNetCore"`. For example, this setting applies to the `"Microsoft.AspNetCore.Routing.EndpointMiddleware"` category.
* The `"Microsoft.AspNetCore"` category logs at log level `Warning` and higher.
* A specific log provider is not specified, so `LogLevel` applies to all the enabled logging providers except for the [Windows EventLog](#welog).

The `Logging` property can have <xref:Microsoft.Extensions.Logging.LogLevel> and log provider properties. The `LogLevel` specifies the minimum [level](#log-level) to log for selected categories. In the preceding JSON, `Information` and `Warning` log levels are specified. `LogLevel` indicates the severity of the log and ranges from 0 to 6:

`Trace` = 0, `Debug` = 1, `Information` = 2, `Warning` = 3, `Error` = 4, `Critical` = 5, and `None` = 6.

When a `LogLevel` is specified, logging is enabled for messages at the specified level and higher. In the preceding JSON, the `Default` category is logged for `Information` and higher. For example, `Information`, `Warning`, `Error`, and `Critical` messages are logged. If no `LogLevel` is specified, logging defaults to the `Information` level. For more information, see [Log levels](#llvl).

A provider property can specify a `LogLevel` property. `LogLevel` under a provider specifies levels to log for that provider, and overrides the non-provider log settings. Consider the following `appsettings.json` file:

[!code-json[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/appsettings.Prod2.json)]

Settings in `Logging.{PROVIDER NAME}.LogLevel` override settings in `Logging.LogLevel`, where the `{PROVIDER NAME}` placeholder is the provider name. In the preceding JSON, the `Debug` provider's default log level is set to `Information`:

`Logging:Debug:LogLevel:Default:Information`

The preceding setting specifies the `Information` log level for every `Logging:Debug:` category except `Microsoft.Hosting`. When a specific category is listed, the specific category overrides the default category. In the preceding JSON, the `Logging:Debug:LogLevel` categories `"Microsoft.Hosting"` and `"Default"` override the settings in `Logging:LogLevel`.

The minimum log level can be specified for any of:

* Specific providers: For example, `Logging:EventSource:LogLevel:Default:Information`
* Specific categories: For example, `Logging:LogLevel:Microsoft:Warning`
* All providers and all categories: `Logging:LogLevel:Default:Warning`

Any logs below the minimum level are ***not***:

* Passed to the provider.
* Logged or displayed.

To suppress all logs, specify <xref:Microsoft.Extensions.Logging.LogLevel.None?displayProperty=nameWithType>. `LogLevel.None` has a value of 6, which is higher than `LogLevel.Critical` (5).

If a provider supports [log scopes](#logscopes), `IncludeScopes` indicates whether they're enabled. For more information, see [log scopes](#logscopes).

The following `appsettings.json` file contains all the providers enabled by default:

[!code-json[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/appsettings.Production.json)]

In the preceding sample:

* The categories and levels aren't suggested values. The sample is provided to show all of the default providers.
* Settings in `Logging.{PROVIDER NAME}.LogLevel` override settings in `Logging.LogLevel`, where the `{PROVIDER NAME}` placeholder is the provider name. For example, the level in `Debug.LogLevel.Default` overrides the level in `LogLevel.Default`.
* Each default provider *alias* is used. Each provider defines an *alias* that can be used in configuration in place of the fully qualified type name. The built-in providers aliases are:
  * `Console`
  * `Debug`
  * `EventSource`
  * `EventLog`
  * `AzureAppServicesFile`
  * `AzureAppServicesBlob`
  * `ApplicationInsights`

## Log in `Program.cs`

The following example calls [Builder.WebApplication.Logger](xref:Microsoft.AspNetCore.Builder.WebApplication.Logger) in `Program.cs` and logs informational messages:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebLog/Program.cs?name=snippet3)]

The following example calls <xref:Microsoft.Extensions.Logging.ConsoleLoggerExtensions.AddConsole%2A> in `Program.cs` and logs the `/Test` endpoint:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebLog/Program.cs?name=snippet1)]

The following example calls <xref:Microsoft.Extensions.Logging.ConsoleLoggerExtensions.AddSimpleConsole%2A> in `Program.cs`, disables color output, and logs the `/Test` endpoint:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebLog/Program.cs?name=snippet2)]

## Set log level by command line, environment variables, and other configuration

Log level can be set by any of the [configuration providers](xref:fundamentals/configuration/index).

[!INCLUDE[](~/includes/environmentVarableColon.md)]

The following commands:

* Set the environment key `Logging:LogLevel:Microsoft` to a value of `Information` on Windows.
* Test the settings when using an app created with the ASP.NET Core web application templates. The `dotnet run` command must be run in the project directory after using `set`.

```dotnetcli
set Logging__LogLevel__Microsoft=Information
dotnet run
```

The preceding environment setting:

* Is only set in processes launched from the command window they were set in.
* Isn't read by browsers launched with Visual Studio.

The following [setx](/windows-server/administration/windows-commands/setx) command also sets the environment key and value on Windows. Unlike `set`, `setx` settings are persisted. The `/M` switch sets the variable in the system environment. If `/M` isn't used, a user environment variable is set.

```console
setx Logging__LogLevel__Microsoft Information /M
```

Consider the following `appsettings.json` file:

```json
"Logging": {
  "Console": {
    "LogLevel": {
      "Microsoft.Hosting.Lifetime": "Trace"
    }
  }
}
```

The following command sets the preceding configuration in the environment:

```console
setx Logging__Console__LogLevel__Microsoft.Hosting.Lifetime Trace /M
```

> [!NOTE]
> When configuring environment variables with names that contain `.` (periods) in macOS and Linux, consider the "Exporting a variable with a dot (.) in it" question on **Stack Exchange** and its corresponding [accepted answer](https://unix.stackexchange.com/a/93533).

On [Azure App Service](https://azure.microsoft.com/services/app-service/), select **New application setting** on the **Settings > Configuration** page. Azure App Service application settings are:

* Encrypted at rest and transmitted over an encrypted channel.
* Exposed as environment variables.

For more information, see [Azure Apps: Override app configuration using the Azure portal](xref:host-and-deploy/azure-apps/index#override-app-configuration-using-the-azure-portal).

For more information on setting ASP.NET Core configuration values using environment variables, see [environment variables](xref:fundamentals/configuration/index#environment-variables). For information on using other configuration sources, including the command line, Azure Key Vault, Azure App Configuration, other file formats, and more, see <xref:fundamentals/configuration/index>.

## How filtering rules are applied

When an <xref:Microsoft.Extensions.Logging.ILogger%601> object is created, the <xref:Microsoft.Extensions.Logging.ILoggerFactory> object selects a single rule per provider to apply to that logger. All messages written by an `ILogger` instance are filtered based on the selected rules. The most specific rule for each provider and category pair is selected from the available rules.

The following algorithm is used for each provider when an `ILogger` is created for a given category:

* Select all rules that match the provider or its alias. If no match is found, select all rules with an empty provider.
* From the result of the preceding step, select rules with longest matching category prefix. If no match is found, select all rules that don't specify a category.
* If multiple rules are selected, take the **last** one.
* If no rules are selected, use `MinimumLevel`.

<a name="dnrvs"></a>

## Logging output from dotnet run and Visual Studio

Logs created with the [default logging providers](#lp) are displayed:

* In Visual Studio
  * In the Debug output window when debugging.
  * In the ASP.NET Core Web Server window.
* In the console window when the app is run with `dotnet run`.

Logs that begin with "Microsoft" categories are from .NET. .NET and application code use the same [logging API and providers](/dotnet/core/extensions/logging).

<a name="lcat"></a>

## Log category

When an `ILogger` object is created, a *category* is specified. That category is included with each log message created by that instance of `ILogger`. The category string is arbitrary, but the convention is to use the fully qualified class name. For example, in a controller the name might be `"TodoApi.Controllers.TodoController"`. The ASP.NET Core web apps use `ILogger<T>` to automatically get an `ILogger` instance that uses the fully qualified type name of `T` as the category:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Pages/Privacy.cshtml.cs?name=snippet)]

If further categorization is desired, the convention is to use a hierarchical name by appending a subcategory to the fully qualified class name, and explicitly specify the category using `ILoggerFactory.CreateLogger`:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Pages/Contact.cshtml.cs?name=snippet)]

Calling `CreateLogger` with a fixed name can be useful when used in multiple methods so the events can be organized by category.

`ILogger<T>` is equivalent to calling `CreateLogger` with the fully qualified type name of `T`.

<a name="llvl"></a>

## Log level

The following table lists the <xref:Microsoft.Extensions.Logging.LogLevel> values, the convenience `Log{LogLevel}` extension method, and the suggested usage:

| LogLevel | Value | Method | Description |
| -------- | ----- | ------ | ----------- |
| <xref:Microsoft.Extensions.Logging.LogLevel.Trace> | 0 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogTrace%2A> | Contain the most detailed messages. These messages may contain sensitive app data. These messages are disabled by default and should ***not*** be enabled in production. |
| <xref:Microsoft.Extensions.Logging.LogLevel.Debug> | 1 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogDebug%2A> | For debugging and development. Use with caution in production due to the high volume. |
| <xref:Microsoft.Extensions.Logging.LogLevel.Information> | 2 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogInformation%2A> | Tracks the general flow of the app. May have long-term value. |
| <xref:Microsoft.Extensions.Logging.LogLevel.Warning> | 3 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogWarning%2A> | For abnormal or unexpected events. Typically includes errors or conditions that don't cause the app to fail. |
| <xref:Microsoft.Extensions.Logging.LogLevel.Error> | 4 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogError%2A> | For errors and exceptions that cannot be handled. These messages indicate a failure in the current operation or request, not an app-wide failure. |
| <xref:Microsoft.Extensions.Logging.LogLevel.Critical> | 5 | <xref:Microsoft.Extensions.Logging.LoggerExtensions.LogCritical%2A> | For failures that require immediate attention. Examples: data loss scenarios, out of disk space. |
| <xref:Microsoft.Extensions.Logging.LogLevel.None> | 6 | | Specifies that a logging category shouldn't write messages. |

In the previous table, the `LogLevel` is listed from lowest to highest severity.

The <xref:Microsoft.Extensions.Logging.LoggerExtensions.Log%2A> method's first parameter, <xref:Microsoft.Extensions.Logging.LogLevel>, indicates the severity of the log. Rather than calling `Log(LogLevel, ...)`, most developers call the [`Log{LOG LEVEL}`](xref:Microsoft.Extensions.Logging.LoggerExtensions) extension methods, where the `{LOG LEVEL}` placeholder is the log level. For example, the following two logging calls are functionally equivalent and produce the same log:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TestController.cs?name=snippet0&highlight=6-7)]

`MyLogEvents.TestItem` is the event ID. `MyLogEvents` is part of the sample app and is displayed in the [Log event ID](#leid) section.

[!INCLUDE[](~/includes/MyDisplayRouteInfoBoth.md)]

The following code creates `Information` and `Warning` logs:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TodoItemsController.cs?name=snippet_CallLogMethods&highlight=4,10)]

In the preceding code, the first `Log{LOG LEVEL}` parameter,`MyLogEvents.GetItem`, is the [Log event ID](#leid). The second parameter is a message template with placeholders for argument values provided by the remaining method parameters. The method parameters are explained in the [message template](#lmt) section later in this document.

Call the appropriate `Log{LOG LEVEL}` method to control how much log output is written to a particular storage medium. For example:

* In production:
  * Logging at the `Trace`, `Debug`, or `Information` levels produces a high-volume of detailed log messages. To control costs and not exceed data storage limits, log `Trace`, `Debug`, or `Information` level messages to a high-volume, low-cost data store. Consider limiting `Trace`, `Debug`, or `Information` to specific categories.
  * Logging at `Warning` through `Critical` levels should produce few log messages.
    * Costs and storage limits usually aren't a concern.
    * Few logs allow more flexibility in data store choices.
* In development:
  * Set to `Warning`.
  * Add `Trace`, `Debug`, or `Information` messages when troubleshooting. To limit output, set `Trace`, `Debug`, or `Information` only for the categories under investigation.

ASP.NET Core writes logs for framework events. For example, consider the log output for:

* A Razor Pages app created with the ASP.NET Core templates.
* Logging set to `Logging:Console:LogLevel:Microsoft:Information`.
* Navigation to the Privacy page:

```console
info: Microsoft.AspNetCore.Hosting.Diagnostics[1]
      Request starting HTTP/2 GET https://localhost:5001/Privacy
info: Microsoft.AspNetCore.Routing.EndpointMiddleware[0]
      Executing endpoint '/Privacy'
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[3]
      Route matched with {page = "/Privacy"}. Executing page /Privacy
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[101]
      Executing handler method DefaultRP.Pages.PrivacyModel.OnGet - ModelState is Valid
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[102]
      Executed handler method OnGet, returned result .
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[103]
      Executing an implicit handler method - ModelState is Valid
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[104]
      Executed an implicit handler method, returned result Microsoft.AspNetCore.Mvc.RazorPages.PageResult.
info: Microsoft.AspNetCore.Mvc.RazorPages.Infrastructure.PageActionInvoker[4]
      Executed page /Privacy in 74.5188ms
info: Microsoft.AspNetCore.Routing.EndpointMiddleware[1]
      Executed endpoint '/Privacy'
info: Microsoft.AspNetCore.Hosting.Diagnostics[2]
      Request finished in 149.3023ms 200 text/html; charset=utf-8
```

The following JSON sets `Logging:Console:LogLevel:Microsoft:Information`:

[!code-json[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/appsettings.MSFT.json)]

<a name="leid"></a>

## Log event ID

Each log can specify an *event ID*. The sample app uses the `MyLogEvents` class to define event IDs:

<!-- Review: to bad there is no way to use an enum for event ID's -->
[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Models/MyLogEvents.cs?name=snippet_LoggingEvents)]

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TodoItemsController.cs?name=snippet_CallLogMethods&highlight=4,10)]

An event ID associates a set of events. For example, all logs related to displaying a list of items on a page might be 1001.

The logging provider may store the event ID in an ID field, in the logging message, or not at all. The Debug provider doesn't show event IDs. The console provider shows event IDs in brackets after the category:

```console
info: TodoApi.Controllers.TodoItemsController[1002]
      Getting item 1
warn: TodoApi.Controllers.TodoItemsController[4000]
      Get(1) NOT FOUND
```

Some logging providers store the event ID in a field, which allows for filtering on the ID.

<a name="lmt"></a>

## Log message template
<!-- Review, Each log API uses a message template. -->
Each log API uses a message template. The message template can contain placeholders for which arguments are provided. Use names for the placeholders, not numbers.

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TodoItemsController.cs?name=snippet_CallLogMethods&highlight=4,10)]

The *order of the parameters*, not their placeholder names, determines which parameters are used to provide placeholder values in log messages. In the following code, the parameter names are out of sequence in the placeholders of the message template:

```csharp
var apples = 1;
var pears = 2;
var bananas = 3;

_logger.LogInformation("Parameters: {Pears}, {Bananas}, {Apples}", apples, pears, bananas);
```

However, the parameters are assigned to the placeholders in the order: `apples`, `pears`, `bananas`. The log message reflects the *order of the parameters*:

```text
Parameters: 1, 2, 3
```

This approach allows logging providers to implement [semantic or structured logging](https://github.com/NLog/NLog/wiki/How-to-use-structured-logging). The arguments themselves are passed to the logging system, not just the formatted message template. This enables logging providers to store the parameter values as fields. For example, consider the following logger method:

```csharp
_logger.LogInformation("Getting item {Id} at {RequestTime}", id, DateTime.Now);
```

For example, when logging to Azure Table Storage:

* Each Azure Table entity can have `ID` and `RequestTime` properties.
* Tables with properties simplify queries on logged data. For example, a query can find all logs within a particular `RequestTime` range without having to parse the time out of the text message.

## Log exceptions

The logger methods have overloads that take an exception parameter:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TestController.cs?name=snippet_Exp)]

[!INCLUDE[](~/includes/MyDisplayRouteInfoBoth.md)]

Exception logging is provider-specific.

### Default log level

If the default log level is not set, the default log level value is `Information`.

For example, consider the following web app:

* Created with the ASP.NET web app templates.
* `appsettings.json` and `appsettings.Development.json` deleted or renamed.

With the preceding setup, navigating to the privacy or home page produces many `Trace`, `Debug`, and `Information` messages with `Microsoft` in the category name.

The following code sets the default log level when the default log level is not set in configuration:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet_MIN)]

Generally, log levels should be specified in configuration and not code.

### Filter function

A filter function is invoked for all providers and categories that don't have rules assigned to them by configuration or code:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet_FF)]

The preceding code displays console logs when the category contains `Controller` or `Microsoft` and the log level is `Information` or higher.

Generally, log levels should be specified in configuration and not code.

## ASP.NET Core categories

The following table contains some categories used by ASP.NET Core.

| Category                              | Notes |
| ------------------------------------- | ----- |
| `Microsoft.AspNetCore`                | General ASP.NET Core diagnostics. |
| `Microsoft.AspNetCore.DataProtection` | Which keys were considered, found, and used. |
| `Microsoft.AspNetCore.HostFiltering`  | Hosts allowed. |
| `Microsoft.AspNetCore.Hosting`        | How long HTTP requests took to complete and what time they started. Which hosting startup assemblies were loaded. |
| `Microsoft.AspNetCore.Mvc`            | MVC and Razor diagnostics. Model binding, filter execution, view compilation, action selection. |
| `Microsoft.AspNetCore.Routing`        | Route matching information. |
| `Microsoft.AspNetCore.Server`         | Connection start, stop, and keep alive responses. HTTPS certificate information. |
| `Microsoft.AspNetCore.StaticFiles`    | Files served. |


To view more categories in the console window, set **`appsettings.Development.json`** to the following:

[!code-json[](~/fundamentals/logging/index/samples/3.x/MyMain/appsettings.Trace.json)]

<!-- Review: What other providers support scopes? Console is not generally used in staging/production  -->

For a list of Entity Framework categories, see [EF Message categories](/ef/core/logging-events-diagnostics/simple-logging#message-categories).

<a name="logscopes"></a>

## Log scopes

 A *scope* can group a set of logical operations. This grouping can be used to attach the same data to each log that's created as part of a set. For example, every log created as part of processing a transaction can include the transaction ID.

A scope:

* Is an <xref:System.IDisposable> type that's returned by the <xref:Microsoft.Extensions.Logging.ILogger.BeginScope%2A> method.
* Lasts until it's disposed.

The following providers support scopes:

* `Console`
* [`AzureAppServicesFile` and `AzureAppServicesBlob`](xref:Microsoft.Extensions.Logging.AzureAppServices.BatchingLoggerOptions.IncludeScopes)

Use a scope by wrapping logger calls in a `using` block:

[!code-csharp[](~/fundamentals/logging/index/samples/3.x/TodoApiDTO/Controllers/TestController.cs?name=snippet_Scopes)]

<a name="bilp"></a>

## Built-in logging providers

ASP.NET Core includes the following logging providers as part of the shared framework:

* [`Console`](#console)
* [`Debug`](#debug)
* [`EventSource`](#event-source)
* [`EventLog`](#welog)

The following logging providers are shipped by Microsoft, but not as part of the 
shared framework. They must be installed as additional nuget.

* [`AzureAppServicesFile` and `AzureAppServicesBlob`](#azure-app-service)
* [`ApplicationInsights`](#azure-application-insights)

ASP.NET Core doesn't include a logging provider for writing logs to files. To write logs to files from an ASP.NET Core app, consider using a [third-party logging provider](#third-party-logging-providers).

For information on `stdout` and debug logging with the ASP.NET Core Module, see <xref:test/troubleshoot-azure-iis> and <xref:host-and-deploy/aspnet-core-module#log-creation-and-redirection>.

### Console

The `Console` provider logs output to the console. For more information on viewing `Console` logs in development, see [Logging output from dotnet run and Visual Studio](#dnrvs).

### Debug

The `Debug` provider writes log output by using the <xref:System.Diagnostics.Debug?displayProperty=fullName> class. Calls to `System.Diagnostics.Debug.WriteLine` write to the `Debug` provider.

On Linux, the `Debug` provider log location is distribution-dependent and may be one of the following:

* `/var/log/message`
* `/var/log/syslog`

### Event Source

The `EventSource` provider writes to a cross-platform event source with the name `Microsoft-Extensions-Logging`. On Windows, the provider uses [ETW](/windows/win32/etw/event-tracing-portal).

#### dotnet-trace tooling

The [`dotnet-trace`](/dotnet/core/diagnostics/dotnet-trace) tool is a cross-platform CLI global tool that enables the collection of .NET Core traces of a running process. The tool collects <xref:Microsoft.Extensions.Logging.EventSource> provider data using a <xref:Microsoft.Extensions.Logging.EventSource.LoggingEventSource>.

For installation instructions, see [`dotnet-trace`](/dotnet/core/diagnostics/dotnet-trace).

Use the `dotnet-trace` tooling to collect a trace from an app:

1. Run the app with the `dotnet run` command.
1. Determine the process identifier (PID) of the .NET Core app:

    ```dotnetcli
    dotnet-trace ps
    ```

   Find the PID for the process that has the same name as the app's assembly.

1. Execute the `dotnet-trace` command.

   General command syntax:

   ```dotnetcli
   dotnet-trace collect -p {PID} 
       --providers Microsoft-Extensions-Logging:{Keyword}:{Provider Level}
           :FilterSpecs=\"
               {Logger Category 1}:{Category Level 1};
               {Logger Category 2}:{Category Level 2};
               ...
               {Logger Category N}:{Category Level N}\"
   ```

   When using a PowerShell command shell, enclose the `--providers` value in single quotes (`'`):

   ```dotnetcli
   dotnet-trace collect -p {PID} 
       --providers 'Microsoft-Extensions-Logging:{Keyword}:{Provider Level}
           :FilterSpecs=\"
               {Logger Category 1}:{Category Level 1};
               {Logger Category 2}:{Category Level 2};
               ...
               {Logger Category N}:{Category Level N}\"'
   ```

   On non-Windows platforms, add the `-f speedscope` option to change the format of the output trace file to `speedscope`.

   The following table defines the Keyword:

   | Keyword | Description |
   | :-----: | ----------- |
   | 1       | Log meta events about the `LoggingEventSource`. Doesn't log events from `ILogger`. |
   | 2       | Turns on the `Message` event when `ILogger.Log()` is called. Provides information in a programmatic (not formatted) way. |
   | 4       | Turns on the `FormatMessage` event when `ILogger.Log()` is called. Provides the formatted string version of the information. |
   | 8       | Turns on the `MessageJson` event when `ILogger.Log()` is called. Provides a JSON representation of the arguments. |

   The following table lists the provider levels:

   | Provider Level | Description     |
   | :------------: | --------------- |
   | 0              | `LogAlways`     |
   | 1              | `Critical`      |
   | 2              | `Error`         |
   | 3              | `Warning`       |
   | 4              | `Informational` |
   | 5              | `Verbose`       |

   The parsing for a category level can be either a string or a number:

   | Category named value | Numeric value |
   | :------------------: | ------------- |
   | `Trace`              | 0 |
   | `Debug`              | 1 |
   | `Information`        | 2 |
   | `Warning`            | 3 |
   | `Error`              | 4 |
   | `Critical`           | 5 |

   The provider level and category level:

   * Are in reverse order.
   * The string constants aren't all identical.

   If no `FilterSpecs` are specified then the `EventSourceLogger` implementation attempts to convert the provider level to a category level and applies it to all categories.

   | Provider Level     | Category Level   |
   | :----------------: | ---------------- |
   | `Verbose`(5)       | `Debug`(1)       |
   | `Informational`(4) | `Information`(2) |
   | `Warning`(3)       | `Warning`(3)     |
   | `Error`(2)         | `Error`(4)       |
   | `Critical`(1)      | `Critical`(5)    |

   If `FilterSpecs` are provided, any category that is included in the list uses the category level encoded there, all other categories are filtered out.

   The following examples assume:
   
   * An app is running and calling `logger.LogDebug("12345")`.
   * The process ID (PID) has been set via `set PID=12345`, where `12345` is the actual PID.
   
   Consider the following command:

   ```dotnetcli
   dotnet-trace collect -p %PID% --providers Microsoft-Extensions-Logging:4:5
   ```

   The preceding command:

   * Captures debug messages.
   * Doesn't apply a `FilterSpecs`.
   * Specifies level 5 which maps category Debug.

   Consider the following command:

   ```dotnetcli
   dotnet-trace collect -p %PID%  --providers Microsoft-Extensions-Logging:4:5:\"FilterSpecs=*:5\"
   ```

   The preceding command:

   * Doesn't capture debug messages because the category level 5 is `Critical`.
   * Provides a `FilterSpecs`.

   The following command captures debug messages because category level 1 specifies `Debug`.

   ```dotnetcli
   dotnet-trace collect -p %PID%  --providers Microsoft-Extensions-Logging:4:5:\"FilterSpecs=*:1\"
   ```

   The following command captures debug messages because category specifies `Debug`.

   ```dotnetcli
   dotnet-trace collect -p %PID%  --providers Microsoft-Extensions-Logging:4:5:\"FilterSpecs=*:Debug\"
   ```

   `FilterSpecs` entries for `{Logger Category}` and `{Category Level}` represent additional log filtering conditions. Separate `FilterSpecs` entries with the `;` semicolon character.

   Example using a Windows command shell:

   ```dotnetcli
   dotnet-trace collect -p %PID% --providers Microsoft-Extensions-Logging:4:2:FilterSpecs=\"Microsoft.AspNetCore.Hosting*:4\"
   ```

   The preceding command activates:

   * The Event Source logger to produce formatted strings (`4`) for errors (`2`).
   * `Microsoft.AspNetCore.Hosting` logging at the `Informational` logging level (`4`).

1. Stop the `dotnet-trace` tooling by pressing the Enter key or <kbd>Ctrl</kbd>+<kbd>C</kbd>.

   The trace is saved with the name `trace.nettrace` in the folder where the `dotnet-trace` command is executed.

1. Open the trace with [Perfview](#perfview). Open the `trace.nettrace` file and explore the trace events.

If the app doesn't build the host with <xref:Microsoft.AspNetCore.Builder.WebApplication.CreateBuilder%2A?displayProperty=nameWithType>, add the Event Source provider to the app's logging configuration.

For more information, see:

* [Trace for performance analysis utility (`dotnet-trace`)](/dotnet/core/diagnostics/dotnet-trace) (.NET Core documentation)
* [Trace for performance analysis utility (`dotnet-trace`)](https://github.com/dotnet/diagnostics/blob/main/documentation/dotnet-trace-instructions.md) (dotnet/diagnostics GitHub repository documentation)
* <xref:Microsoft.Extensions.Logging.EventSource.LoggingEventSource>
* <xref:System.Diagnostics.Tracing.EventLevel>
* [Perfview](#perfview): Useful for viewing Event Source traces.

#### Perfview

Use the [PerfView utility](https://github.com/Microsoft/perfview) to collect and view logs. There are other tools for viewing ETW logs, but PerfView provides the best experience for working with the ETW events emitted by ASP.NET Core.

To configure PerfView for collecting events logged by this provider, add the string `*Microsoft-Extensions-Logging` to the **Additional Providers** list. Don't miss the `*` at the start of the string.

<a name="welog"></a>

### Windows EventLog

The `EventLog` provider sends log output to the Windows Event Log. Unlike the other providers, the `EventLog` provider does ***not*** inherit the default non-provider settings. If `EventLog` log settings aren't specified, they default to <xref:Microsoft.Extensions.Logging.LogLevel.Warning?displayProperty=nameWithType>.

To log events lower than <xref:Microsoft.Extensions.Logging.LogLevel.Warning?displayProperty=nameWithType>, explicitly set the log level. The following example sets the Event Log default log level to <xref:Microsoft.Extensions.Logging.LogLevel.Information?displayProperty=nameWithType>:

```json
"Logging": {
  "EventLog": {
    "LogLevel": {
      "Default": "Information"
    }
  }
}
```

<xref:Microsoft.Extensions.Logging.EventLoggerFactoryExtensions.AddEventLog%2A> overloads can pass in <xref:Microsoft.Extensions.Logging.EventLog.EventLogSettings>. If `null` or not specified, the following default settings are used:

* `LogName`: "Application"
* `SourceName`: ".NET Runtime"
* `MachineName`: The local machine name is used.

The following code changes the `SourceName` from the default value of `".NET Runtime"` to `MyLogs`:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet_WEL)]

### Azure App Service

The [`Microsoft.Extensions.Logging.AzureAppServices`](https://www.nuget.org/packages/Microsoft.Extensions.Logging.AzureAppServices) provider package writes logs to text files in an Azure App Service app's file system and to [blob storage](/azure/storage/blobs/storage-quickstart-blobs-dotnet#what-is-blob-storage) in an Azure Storage account.

The provider package isn't included in the shared framework. To use the provider, add the provider package to the project.

To configure provider settings, use <xref:Microsoft.Extensions.Logging.AzureAppServices.AzureFileLoggerOptions> and <xref:Microsoft.Extensions.Logging.AzureAppServices.AzureBlobLoggerOptions>, as shown in the following example:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet_AAS)]

When deployed to Azure App Service, the app uses the settings in the [App Service logs](/azure/app-service/troubleshoot-diagnostic-logs) section of the **App Service** page of the Azure portal. When the following settings are updated, the changes take effect immediately without requiring a restart or redeployment of the app.

* **Application Logging (Filesystem)**
* **Application Logging (Blob)**

The default location for log files is in the `D:\\home\\LogFiles\\Application` folder, and the default file name is `diagnostics-yyyymmdd.txt`. The default file size limit is 10 MB, and the default maximum number of files retained is 2. The default blob name is `{app-name}{timestamp}/yyyy/mm/dd/hh/{guid}-applicationLog.txt`.

This provider only logs when the project runs in the Azure environment.

#### Azure log streaming

Azure log streaming supports viewing log activity in real time from:

* The app server
* The web server
* Failed request tracing

To configure Azure log streaming:

* Navigate to the **App Service logs** page from the app's portal page.
* Set **Application Logging (Filesystem)** to **On**.
* Choose the log **Level**. This setting only applies to Azure log streaming.

Navigate to the **Log Stream** page to view logs. The logged messages are logged with the `ILogger` interface.

### Azure Application Insights

The [`Microsoft.Extensions.Logging.ApplicationInsights`](https://www.nuget.org/packages/Microsoft.Extensions.Logging.ApplicationInsights) provider package writes logs to [Azure Application Insights](/azure/azure-monitor/app/cloudservices). Application Insights is a service that monitors a web app and provides tools for querying and analyzing the telemetry data. If you use this provider, you can query and analyze your logs by using the Application Insights tools.

The logging provider is included as a dependency of [`Microsoft.ApplicationInsights.AspNetCore`](https://www.nuget.org/packages/Microsoft.ApplicationInsights.AspNetCore), which is the package that provides all available telemetry for ASP.NET Core. If you use this package, you don't have to install the provider package.

The [`Microsoft.ApplicationInsights.Web`](https://www.nuget.org/packages/Microsoft.ApplicationInsights.Web) package is for ASP.NET 4.x, not ASP.NET Core.

For more information, see the following resources:

* [Application Insights overview](/azure/application-insights/app-insights-overview)
* [Application Insights for ASP.NET Core applications](/azure/azure-monitor/app/asp-net-core): Start here if you want to implement the full range of Application Insights telemetry along with logging.
* [ApplicationInsightsLoggerProvider for .NET Core ILogger logs](/azure/azure-monitor/app/ilogger): Start here if you want to implement the logging provider without the rest of Application Insights telemetry.
* [Application Insights logging adapters](/azure/azure-monitor/app/asp-net-trace-logs)
* [Install, configure, and initialize the Application Insights SDK](/training/modules/instrument-web-app-code-with-application-insights) interactive tutorial.

## Third-party logging providers

Third-party logging frameworks that work with ASP.NET Core:

* [elmah.io](https://elmah.io/) ([GitHub repo](https://github.com/elmahio/Elmah.Io.Extensions.Logging))
* [Gelf](https://go2docs.graylog.org/5-0/getting_in_log_data/gelf.html) ([GitHub repo](https://github.com/mattwcole/gelf-extensions-logging))
* [JSNLog](https://jsnlog.com/) ([GitHub repo](https://github.com/mperdeck/jsnlog))
* [KissLog.net](https://kisslog.net/) ([GitHub repo](https://github.com/catalingavan/KissLog-net))
* [Log4Net](https://logging.apache.org/log4net/) ([GitHub repo](https://github.com/huorswords/Microsoft.Extensions.Logging.Log4Net.AspNetCore))
* [NLog](https://nlog-project.org/) ([GitHub repo](https://github.com/NLog/NLog.Extensions.Logging))
* [PLogger](https://www.nuget.org/packages/InvertedSoftware.PLogger.Core/) ([GitHub repo](https://github.com/invertedsoftware/InvertedSoftware.PLogger.Core))
* [Sentry](https://sentry.io/welcome/) ([GitHub repo](https://github.com/getsentry/sentry-dotnet))
* [Serilog](https://serilog.net/) ([GitHub repo](https://github.com/serilog/serilog-aspnetcore))
* [Stackdriver](https://cloud.google.com/dotnet/docs/stackdriver#logging) ([Github repo](https://github.com/googleapis/google-cloud-dotnet))

Some third-party frameworks can perform [semantic logging, also known as structured logging](https://softwareengineering.stackexchange.com/questions/312197/benefits-of-structured-logging-vs-basic-logging).

Using a third-party framework is similar to using one of the built-in providers:

1. Add a NuGet package to your project.
1. Call an `ILoggerFactory` extension method provided by the logging framework.

For more information, see each provider's documentation. Third-party logging providers aren't supported by Microsoft.

### No asynchronous logger methods

Logging should be so fast that it isn't worth the performance cost of asynchronous code. If a logging data store is slow, don't write to it directly. Consider writing the log messages to a fast store initially, then moving them to the slow store later. For example, when logging to SQL Server, don't do so directly in a `Log` method, since the `Log` methods are synchronous. Instead, synchronously add log messages to an in-memory queue and have a background worker pull the messages out of the queue to do the asynchronous work of pushing data to SQL Server. For more information, see [Guidance on how to log to a message queue for slow data stores (dotnet/AspNetCore.Docs #11801)](https://github.com/dotnet/AspNetCore.Docs/issues/11801).

<a name="clib"></a>

## Change log levels in a running app

The Logging API doesn't include a scenario to change log levels while an app is running. However, some configuration providers are capable of reloading configuration, which takes immediate effect on logging configuration. For example, the [File Configuration Provider](xref:fundamentals/configuration/index#file-configuration-provider), reloads logging configuration by default. If configuration is changed in code while an app is running, the app can call <xref:Microsoft.Extensions.Configuration.IConfigurationRoot.Reload%2A?displayProperty=nameWithType> to update the app's logging configuration.

## ILogger and ILoggerFactory

The <xref:Microsoft.Extensions.Logging.ILogger%601> and <xref:Microsoft.Extensions.Logging.ILoggerFactory> interfaces and implementations are included in the .NET Core SDK. They are also available in the following NuGet packages:  

* The interfaces are in [`Microsoft.Extensions.Logging.Abstractions`](https://www.nuget.org/packages/Microsoft.Extensions.Logging.Abstractions/).
* The default implementations are in [`Microsoft.Extensions.Logging`](https://www.nuget.org/packages/microsoft.extensions.logging/).

<!-- review. Why would you want to hard code filtering rules in code? -->
<a name="fric"></a>

## Apply log filter rules in code

The preferred approach for setting log filter rules is by using [Configuration](xref:fundamentals/configuration/index).

The following example shows how to register filter rules in code:

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebApp/Program.cs?name=snippet_FR)]

`logging.AddFilter("System", LogLevel.Debug)` specifies the `System` category and log level `Debug`. The filter is applied to all providers because a specific provider was not configured.

`AddFilter<DebugLoggerProvider>("Microsoft", LogLevel.Information)` specifies:

* The `Debug` logging provider.
* Log level `Information` and higher.
* All categories starting with `"Microsoft"`.

## Automatically log scope with `SpanId`, `TraceId`, `ParentId`, `Baggage`, and `Tags`.

The logging libraries implicitly create a scope object with `SpanId`, `TraceId`, `ParentId`,`Baggage`, and `Tags`. This behavior is configured via <xref:Microsoft.Extensions.Logging.LoggerFactoryOptions.ActivityTrackingOptions>.

[!code-csharp[](~/fundamentals/logging/index/samples/6.x/WebLog/Program.cs?name=snippet4)]

If the `traceparent` http request header is set, the `ParentId` in the log scope shows the W3C `parent-id` from in-bound `traceparent` header and the `SpanId` in the log scope shows the updated `parent-id` for the next out-bound step/span. For more information, see [Mutating the traceparent Field](https://www.w3.org/TR/trace-context/#mutating-the-traceparent-field).

## Create a custom logger

To create a custom logger, see [Implement a custom logging provider in .NET](/dotnet/core/extensions/custom-logging-provider).

<!-- See https://github.com/dotnet/AspNetCore.Docs/pull/30282#issuecomment-1712135965
## LoggerFactory.Create
***NOTE:*** [LoggerFactory.Create](/dotnet/api/microsoft.extensions.logging.loggerfactory.create) Creates a new instance of <xref:Microsoft.Extensions.Logging.ILoggerFactory> that's configured using the provided `configure` delegate. The new instance of `ILoggerFactory` creates a new logging pipeline, different from the default logging pipeline. The new logging pipeline is not configured by the `Logging` section of `appsettings.json` or `appsettings.{ENVIRONMENT}.json`. For more information, see [Logging configuration](xref:fundamentals/logging/index#configure-logging).
-->

## Additional resources

* [Improving logging performance with source generators](https://andrewlock.net/exploring-dotnet-6-part-8-improving-logging-performance-with-source-generators/)
* [Behind `[LogProperties]` and the new telemetry logging source generator](https://andrewlock.net/behind-logproperties-and-the-new-telemetry-logging-source-generator/)
* [Microsoft.Extensions.Logging source on GitHub](https://github.com/dotnet/runtime/tree/main/src/libraries/Microsoft.Extensions.Logging)
* [View or download sample code](https://github.com/dotnet/AspNetCore.Docs/tree/main/aspnetcore/fundamentals/logging/index/samples/6.x) ([how to download](xref:index#how-to-download-a-sample)).
* [High performance logging](/dotnet/core/extensions/high-performance-logging)
* Logging bugs should be created in the [`dotnet/runtime`](https://github.com/dotnet/runtime/issues) GitHub repository.
* <xref:blazor/fundamentals/logging>

:::moniker-end

[!INCLUDE[](~/fundamentals/logging/index/includes/index5.md)]
