unit Ntapi.ntioapi.fsctl;

{
  The file provides definitions for accessing volume information and issuing
  FSCTL requests to the file system.
}

interface

{$MINENUMSIZE 4}

uses
  Ntapi.WinNt, Ntapi.ntdef, Ntapi.ntioapi, Ntapi.ntwow64, Ntapi.Versions,
  DelphiApi.Reflection;

const
  // WDK::wdm.h - device characteristics
  FILE_REMOVABLE_MEDIA = $00000001;
  FILE_READ_ONLY_DEVICE = $00000002;
  FILE_FLOPPY_DISKETTE = $00000004;
  FILE_WRITE_ONCE_MEDIA = $00000008;
  FILE_REMOTE_DEVICE = $00000010;
  FILE_DEVICE_IS_MOUNTED = $00000020;
  FILE_VIRTUAL_VOLUME = $00000040;
  FILE_AUTOGENERATED_DEVICE_NAME = $00000080;
  FILE_DEVICE_SECURE_OPEN = $00000100;
  FILE_CHARACTERISTIC_PNP_DEVICE = $00000800;
  FILE_CHARACTERISTIC_TS_DEVICE = $00001000;
  FILE_CHARACTERISTIC_WEBDAV_DEVICE = $00002000;
  FILE_CHARACTERISTIC_CSV = $00010000;
  FILE_DEVICE_ALLOW_APPCONTAINER_TRAVERSAL = $00020000;
  FILE_PORTABLE_DEVICE = $00040000;

  // WDK::ntifs.h - file system attributes
  FILE_CASE_SENSITIVE_SEARCH = $00000001;
  FILE_CASE_PRESERVED_NAMES = $00000002;
  FILE_UNICODE_ON_DISK = $00000004;
  FILE_PERSISTENT_ACLS = $00000008;
  FILE_FILE_COMPRESSION = $00000010;
  FILE_VOLUME_QUOTAS = $00000020;
  FILE_SUPPORTS_SPARSE_FILES = $00000040;
  FILE_SUPPORTS_REPARSE_POINTS = $00000080;
  FILE_SUPPORTS_REMOTE_STORAGE = $00000100;
  FILE_RETURNS_CLEANUP_RESULT_INFO = $00000200;
  FILE_SUPPORTS_POSIX_UNLINK_RENAME = $00000400;
  FILE_VOLUME_IS_COMPRESSED = $00008000;
  FILE_SUPPORTS_OBJECT_IDS = $00010000;
  FILE_SUPPORTS_ENCRYPTION = $00020000;
  FILE_NAMED_STREAMS = $00040000;
  FILE_READ_ONLY_VOLUME = $00080000;
  FILE_SEQUENTIAL_WRITE_ONCE = $00100000;
  FILE_SUPPORTS_TRANSACTIONS = $00200000;
  FILE_SUPPORTS_HARD_LINKS = $00400000;
  FILE_SUPPORTS_EXTENDED_ATTRIBUTES = $00800000;
  FILE_SUPPORTS_OPEN_BY_FILE_ID = $01000000;
  FILE_SUPPORTS_USN_JOURNAL = $02000000;
  FILE_SUPPORTS_INTEGRITY_STREAMS = $04000000;
  FILE_SUPPORTS_BLOCK_REFCOUNTING = $08000000;
  FILE_SUPPORTS_SPARSE_VDL = $10000000;
  FILE_DAX_VOLUME = $20000000;
  FILE_SUPPORTS_GHOSTING = $40000000;

  // WDK::ntifs.h - fs control flags
  FILE_VC_QUOTA_NONE = $00000000;
  FILE_VC_QUOTA_TRACK = $00000001;
  FILE_VC_QUOTA_ENFORCE = $00000002;
  FILE_VC_CONTENT_INDEX_DISABLED = $00000008;
  FILE_VC_LOG_QUOTA_THRESHOLD = $00000010;
  FILE_VC_LOG_QUOTA_LIMIT = $00000020;
  FILE_VC_LOG_VOLUME_THRESHOLD = $00000040;
  FILE_VC_LOG_VOLUME_LIMIT = $00000080;
  FILE_VC_QUOTAS_INCOMPLETE = $00000100;
  FILE_VC_QUOTAS_REBUILDING = $00000200;

  // WDK::ntddk.h - sector size fs flags
  SSINFO_FLAGS_ALIGNED_DEVICE = $00000001;
  SSINFO_FLAGS_PARTITION_ALIGNED_ON_DEVICE = $00000002;
  SSINFO_FLAGS_NO_SEEK_PENALTY = $00000004;
  SSINFO_FLAGS_TRIM_ENABLED = $00000008;
  SSINFO_FLAGS_BYTE_ADDRESSABLE = $00000010; // Win 10 TH2+

  // reactos::ntfs.h (from enum) - NTFS attribute type codes
  ATTRIBUTE_TYPE_CODE_UNUSED = $0;
  ATTRIBUTE_TYPE_CODE_STANDARD_INFORMATION = $10;
  ATTRIBUTE_TYPE_CODE_ATTRIBUTE_LIST = $20;
  ATTRIBUTE_TYPE_CODE_FILE_NAME = $30;
  ATTRIBUTE_TYPE_CODE_OBJECT_ID = $40;
  ATTRIBUTE_TYPE_CODE_SECURITY_DESCRIPTOR = $50;
  ATTRIBUTE_TYPE_CODE_VOLUME_NAME = $60;
  ATTRIBUTE_TYPE_CODE_VOLUME_INFORMATION = $70;
  ATTRIBUTE_TYPE_CODE_DATA = $80;
  ATTRIBUTE_TYPE_CODE_INDEX_ROOT = $90;
  ATTRIBUTE_TYPE_CODE_INDEX_ALLOCATION = $A0;
  ATTRIBUTE_TYPE_CODE_BITMAP = $B0;
  ATTRIBUTE_TYPE_CODE_REPARSE_POINT = $C0;
  ATTRIBUTE_TYPE_CODE_EA_INFORMATION = $D0;
  ATTRIBUTE_TYPE_CODE_EA = $E0;
  ATTRIBUTE_TYPE_CODE_LOGGED_UTILITY_STREAM = $100;
  ATTRIBUTE_TYPE_CODE_END = $FFFFFFFF;

  // WDK::ntifs.h - maximum size FSCTL 41, 42, 43
  MAXIMUM_REPARSE_DATA_BUFFER_SIZE = 16384;

  // WDK::ntifs.h - opportunistic lock flags (FSCTL 144)
  OPLOCK_LEVEL_CACHE_READ = $00000001;
  OPLOCK_LEVEL_CACHE_HANDLE = $00000002;
  OPLOCK_LEVEL_CACHE_WRITE = $00000004;

  // WDK::ntifs.h - opportunistic lock request input flags (FSCTL 144)
  REQUEST_OPLOCK_INPUT_FLAG_REQUEST = $00000001;
  REQUEST_OPLOCK_INPUT_FLAG_ACK = $00000002;
  REQUEST_OPLOCK_INPUT_FLAG_COMPLETE_ACK_ON_CLOSE = $00000004;

  // WDK::ntifs.h - opportunistic lock request output flags (FSCTL 144)
  REQUEST_OPLOCK_OUTPUT_FLAG_ACK_REQUIRED = $00000001;
  REQUEST_OPLOCK_OUTPUT_FLAG_MODES_PROVIDED = $00000002;

  // WDK::ntifs.h - opportunistic lock version (FSCTL 144)
  REQUEST_OPLOCK_CURRENT_VERSION = 1;

  // WDK::ntifs.h - file layout input flags (FSCTL 157)
  QUERY_FILE_LAYOUT_RESTART = $00000001;
  QUERY_FILE_LAYOUT_INCLUDE_NAMES = $00000002;
  QUERY_FILE_LAYOUT_INCLUDE_STREAMS = $00000004;
  QUERY_FILE_LAYOUT_INCLUDE_EXTENTS = $00000008;
  QUERY_FILE_LAYOUT_INCLUDE_EXTRA_INFO = $00000010;
  QUERY_FILE_LAYOUT_INCLUDE_STREAMS_WITH_NO_CLUSTERS_ALLOCATED = $00000020;
  QUERY_FILE_LAYOUT_INCLUDE_FULL_PATH_IN_NAMES = $00000040;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION = $00000080;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_DSC_ATTRIBUTE = $00000100;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_TXF_ATTRIBUTE = $00000200;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_EFS_ATTRIBUTE = $00000400;
  QUERY_FILE_LAYOUT_INCLUDE_ONLY_FILES_WITH_SPECIFIC_ATTRIBUTES = $00000800;
  QUERY_FILE_LAYOUT_INCLUDE_FILES_WITH_DSC_ATTRIBUTE = $00001000;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_DATA_ATTRIBUTE = $00002000;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_REPARSE_ATTRIBUTE = $00004000;
  QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_EA_ATTRIBUTE = $00008000;
  QUERY_FILE_LAYOUT_ALL = $0000E7FE;

  // WDK::ntifs.h - file name flags (FSCTL 157)
  FILE_LAYOUT_NAME_ENTRY_PRIMARY = $00000001;
  FILE_LAYOUT_NAME_ENTRY_DOS = $00000002;

  // WDK::ntifs.h - stream entry flags (FSCTL 157)
  STREAM_LAYOUT_ENTRY_IMMOVABLE = $00000001;
  STREAM_LAYOUT_ENTRY_PINNED = $00000002;
  STREAM_LAYOUT_ENTRY_RESIDENT = $00000004;
  STREAM_LAYOUT_ENTRY_NO_CLUSTERS_ALLOCATED = $00000008;
  STREAM_LAYOUT_ENTRY_HAS_INFORMATION = $00000010;

  // WDK::ntifs.h - stream extent entry flags (FSCTL 157)
  STREAM_EXTENT_ENTRY_AS_RETRIEVAL_POINTERS = $00000001;
  STREAM_EXTENT_ENTRY_ALL_EXTENTS = $00000002;

  // WDK::ntifs.h - stream reparse information flags (FSCTL 157)
  QUERY_FILE_LAYOUT_REPARSE_DATA_INVALID = $0001;
  QUERY_FILE_LAYOUT_REPARSE_TAG_INVALID = $0002;

  // WDK::ntifs.h - file layout output flags (FSCTL 157)
  QUERY_FILE_LAYOUT_SINGLE_INSTANCED = $00000001;

  // WDK::ntifs.h - windows overlay filter version (FSCTL 195 & 196)
  WOF_CURRENT_VERSION = 1;

  // WDK::ntifs.h - WOF file provider version (FSCTL 195 & 196)
  FILE_PROVIDER_CURRENT_VERSION = 1;

  // WDK::ntifs.h - extended reparse tag set flag
  REPARSE_DATA_EX_FLAG_GIVEN_TAG_OR_NONE = $00000001;

  // PHNT::ntioapi.h
  DEVICE_NAMED_PIPE = '\Device\NamedPipe\';

  // WDK::ntifs.h - pipe symlink flags
  FILE_PIPE_SYMLINK_FLAG_GLOBAL = $0001;
  FILE_PIPE_SYMLINK_FLAG_RELATIVE = $0002;

  // WDK::ntifs.h - pipe client computer name
  FILE_PIPE_COMPUTER_NAME_LENGTH = 15;

  // PHNT::ntioapi.h
  FLT_SYMLINK_NAME = '\Global??\FltMgr';
  FLT_MSG_SYMLINK_NAME = '\Global??\FltMgrMsg';
  FLT_DEVICE_NAME = '\FileSystem\Filters\FltMgr';
  FLT_MSG_DEVICE_NAME = '\FileSystem\Filters\FltMgrMsg';
  FLT_PORT_EA_NAME = 'FLTPORT';

  // SDK::fltUserStructures.h - filter volume flag (renamed)
  FLTFL_DETACHED_VOLUME = $00000001;

  // WDK::ntifs.h - filesystem features
  SUPPORTED_FS_FEATURES_OFFLOAD_READ = $00000001;  // Windows 8+
  SUPPORTED_FS_FEATURES_OFFLOAD_WRITE = $00000002; // Windows 8+
  SUPPORTED_FS_FEATURES_QUERY_OPEN = $00000004;    // Windows 10 RS2+
  SUPPORTED_FS_FEATURES_BYPASS_IO = $00000008;     // Windows 11+

type
  { Volume Information }

  // WDK::wdm.h
  [SDKName('FS_INFORMATION_CLASS')]
  [NamingStyle(nsCamelCase, 'FileFs'), Range(1)]
  TFsInfoClass = (
    [Reserved] FileFsReserved = 0,
    FileFsVolumeInformation = 1,        // q: TFileFsVolumeInformation
    FileFsLabelInformation = 2,         // s: TFileFsLabelInformation
    FileFsSizeInformation = 3,          // q: TFileFsSizeInformation
    FileFsDeviceInformation = 4,        // q: TFileFsDeviceInformation
    FileFsAttributeInformation = 5,     // q: TFileFsAttributeInformation
    FileFsControlInformation = 6,       // q, s: TFileFsControlInformation
    FileFsFullSizeInformation = 7,      // q: TFileFsFullSizeInformation
    FileFsObjectIdInformation = 8,      // q, s: TFileFsObjectIdInformation
    FileFsDriverPathInformation = 9,    // q: TFileFsDriverPathInformation
    FileFsVolumeFlagsInformation = 10,  // q, s: Cardinal
    FileFsSectorSizeInformation = 11,   // q: TFileFsSectorSizeInformation, Win 8+
    FileFsDataCopyInformation = 12,     // q: Cardinal (NumberOfCopies)
    FileFsMetadataSizeInformation = 13, // q: , Win 10 TH1+
    FileFsFullSizeInformationEx = 14    // q: TFileFsFullSizeInformationEx, Win 10 RS5+
  );

  // WDK::ntddk.h - info class 1
  [SDKName('FILE_FS_VOLUME_INFORMATION')]
  TFileFsVolumeInformation = record
    VolumeCreationTime: TLargeInteger;
    VolumeSerialNumber: Cardinal;
    [Counter(ctBytes)] VolumeLabelLength: Cardinal;
    SupportsObjects: Boolean;
    VolumeLabel: TAnysizeArray<WideChar>;
  end;
  PFileFsVolumeInformation = ^TFileFsVolumeInformation;

  // WDK::ntddk.h - info class 2
  [SDKName('FILE_FS_LABEL_INFORMATION')]
  TFileFsLabelInformation = record
    [Counter(ctBytes)] VolumeLabelLength: Cardinal;
    VolumeLabel: TAnysizeArray<WideChar>;
  end;
  PFileFsLabelInformation = ^TFileFsLabelInformation;

  // WDK::ntddk.h - info class 3
  [SDKName('FILE_FS_SIZE_INFORMATION')]
  TFileFsSizeInformation = record
    TotalAllocationUnits: UInt64;
    AvailableAllocationUnits: UInt64;
    SectorsPerAllocationUnit: Cardinal;
    [Bytes] BytesPerSector: Cardinal;
  end;
  PFileFsSizeInformation = ^TFileFsSizeInformation;

  // WDK::ntifs.h
  {$SCOPEDENUMS ON}
  [SDKName('DEVICE_TYPE')]
  [NamingStyle(nsSnakeCase, 'FILE_DEVICE')]
  TDeviceType = (
    FILE_DEVICE_BEEP = $00000001,
    FILE_DEVICE_CD_ROM = $00000002,
    FILE_DEVICE_CD_ROM_FILE_SYSTEM = $00000003,
    FILE_DEVICE_CONTROLLER = $00000004,
    FILE_DEVICE_DATALINK = $00000005,
    FILE_DEVICE_DFS = $00000006,
    FILE_DEVICE_DISK = $00000007,
    FILE_DEVICE_DISK_FILE_SYSTEM = $00000008,
    FILE_DEVICE_FILE_SYSTEM = $00000009,
    FILE_DEVICE_INPORT_PORT = $0000000a,
    FILE_DEVICE_KEYBOARD = $0000000b,
    FILE_DEVICE_MAILSLOT = $0000000c,
    FILE_DEVICE_MIDI_IN = $0000000d,
    FILE_DEVICE_MIDI_OUT = $0000000e,
    FILE_DEVICE_MOUSE = $0000000f,
    FILE_DEVICE_MULTI_UNC_PROVIDER = $00000010,
    FILE_DEVICE_NAMED_PIPE = $00000011,
    FILE_DEVICE_NETWORK = $00000012,
    FILE_DEVICE_NETWORK_BROWSER = $00000013,
    FILE_DEVICE_NETWORK_FILE_SYSTEM = $00000014,
    FILE_DEVICE_NULL = $00000015,
    FILE_DEVICE_PARALLEL_PORT = $00000016,
    FILE_DEVICE_PHYSICAL_NETCARD = $00000017,
    FILE_DEVICE_PRINTER = $00000018,
    FILE_DEVICE_SCANNER = $00000019,
    FILE_DEVICE_SERIAL_MOUSE_PORT = $0000001a,
    FILE_DEVICE_SERIAL_PORT = $0000001b,
    FILE_DEVICE_SCREEN = $0000001c,
    FILE_DEVICE_SOUND = $0000001d,
    FILE_DEVICE_STREAMS = $0000001e,
    FILE_DEVICE_TAPE = $0000001f,
    FILE_DEVICE_TAPE_FILE_SYSTEM = $00000020,
    FILE_DEVICE_TRANSPORT = $00000021,
    FILE_DEVICE_UNKNOWN = $00000022,
    FILE_DEVICE_VIDEO = $00000023,
    FILE_DEVICE_VIRTUAL_DISK = $00000024,
    FILE_DEVICE_WAVE_IN = $00000025,
    FILE_DEVICE_WAVE_OUT = $00000026,
    FILE_DEVICE_8042_PORT = $00000027,
    FILE_DEVICE_NETWORK_REDIRECTOR = $00000028,
    FILE_DEVICE_BATTERY = $00000029,
    FILE_DEVICE_BUS_EXTENDER = $0000002a,
    FILE_DEVICE_MODEM = $0000002b,
    FILE_DEVICE_VDM = $0000002c,
    FILE_DEVICE_MASS_STORAGE = $0000002d,
    FILE_DEVICE_SMB = $0000002e,
    FILE_DEVICE_KS = $0000002f,
    FILE_DEVICE_CHANGER = $00000030,
    FILE_DEVICE_SMARTCARD = $00000031,
    FILE_DEVICE_ACPI = $00000032,
    FILE_DEVICE_DVD = $00000033,
    FILE_DEVICE_FULLSCREEN_VIDEO = $00000034,
    FILE_DEVICE_DFS_FILE_SYSTEM = $00000035,
    FILE_DEVICE_DFS_VOLUME = $00000036,
    FILE_DEVICE_SERENUM = $00000037,
    FILE_DEVICE_TERMSRV = $00000038,
    FILE_DEVICE_KSEC = $00000039,
    FILE_DEVICE_FIPS = $0000003a,
    FILE_DEVICE_INFINIBAND = $0000003b,
    FILE_DEVICE_TYPE_60 = $0000003c,
    FILE_DEVICE_TYPE_61 = $0000003d,
    FILE_DEVICE_VMBUS = $0000003e,
    FILE_DEVICE_CRYPT_PROVIDER = $0000003f,
    FILE_DEVICE_WPD = $00000040,
    FILE_DEVICE_BLUETOOTH = $00000041,
    FILE_DEVICE_MT_COMPOSITE = $00000042,
    FILE_DEVICE_MT_TRANSPORT = $00000043,
    FILE_DEVICE_BIOMETRIC = $00000044,
    FILE_DEVICE_PMI = $00000045,
    FILE_DEVICE_EHSTOR = $00000046,
    FILE_DEVICE_DEVAPI = $00000047,
    FILE_DEVICE_GPIO = $00000048,
    FILE_DEVICE_USBEX = $00000049,
    FILE_DEVICE_CONSOLE = $00000050,
    FILE_DEVICE_NFP = $00000051,
    FILE_DEVICE_SYSENV = $00000052,
    FILE_DEVICE_VIRTUAL_BLOCK = $00000053,
    FILE_DEVICE_POINT_OF_SERVICE = $00000054,
    FILE_DEVICE_STORAGE_REPLICATION = $00000055,
    FILE_DEVICE_TRUST_ENV = $00000056,
    FILE_DEVICE_UCM = $00000057,
    FILE_DEVICE_UCMTCPCI = $00000058,
    FILE_DEVICE_PERSISTENT_MEMORY = $00000059,
    FILE_DEVICE_NVDIMM = $0000005a,
    FILE_DEVICE_HOLOGRAPHIC = $0000005b,
    FILE_DEVICE_SDFXHCI = $0000005c,
    FILE_DEVICE_UCMUCSI = $0000005d
  );
  {$SCOPEDENUMS OFF}

  // WDK::ntifs.h
  [NamingStyle(nsSnakeCase, 'METHOD')]
  TIoControlMethod = (
    METHOD_BUFFERED = 0,
    METHOD_IN_DIRECT = 1,
    METHOD_OUT_DIRECT = 2,
    METHOD_NEITHER = 3
  );

  [FlagName(FILE_REMOVABLE_MEDIA, 'Removable Media')]
  [FlagName(FILE_READ_ONLY_DEVICE, 'Read-only Device')]
  [FlagName(FILE_FLOPPY_DISKETTE, 'Floppy Disk')]
  [FlagName(FILE_WRITE_ONCE_MEDIA, 'Write-once Media')]
  [FlagName(FILE_REMOTE_DEVICE, 'Remote Device')]
  [FlagName(FILE_DEVICE_IS_MOUNTED, 'Device Is Mounted')]
  [FlagName(FILE_VIRTUAL_VOLUME, 'Virtual Volume')]
  [FlagName(FILE_AUTOGENERATED_DEVICE_NAME, 'Autogenerated Device Name')]
  [FlagName(FILE_DEVICE_SECURE_OPEN, 'Device Secure Open')]
  [FlagName(FILE_CHARACTERISTIC_PNP_DEVICE, 'PnP Device')]
  [FlagName(FILE_CHARACTERISTIC_TS_DEVICE, 'TS Device')]
  [FlagName(FILE_CHARACTERISTIC_WEBDAV_DEVICE, 'WebDav Device')]
  [FlagName(FILE_CHARACTERISTIC_CSV, 'CSV')]
  [FlagName(FILE_DEVICE_ALLOW_APPCONTAINER_TRAVERSAL, 'Allow AppContainer Traversal')]
  [FlagName(FILE_PORTABLE_DEVICE, 'Portable Device')]
  TDeviceCharacteristics = type Cardinal;

  // WDK::wdm.h - info class 4
  [SDKName('FILE_FS_DEVICE_INFORMATION')]
  TFileFsDeviceInformation = record
    DeviceType: TDeviceType;
    Characteristics: TDeviceCharacteristics;
  end;
  PFileFsDeviceInformation = ^TFileFsDeviceInformation;

  [FlagName(FILE_CASE_SENSITIVE_SEARCH, 'Case-sensitive Search')]
  [FlagName(FILE_CASE_PRESERVED_NAMES, 'Case-preserved Names')]
  [FlagName(FILE_UNICODE_ON_DISK, 'Unicode On Disk')]
  [FlagName(FILE_PERSISTENT_ACLS, 'Persistent ACLs')]
  [FlagName(FILE_FILE_COMPRESSION, 'Supports Compression')]
  [FlagName(FILE_VOLUME_QUOTAS, 'Volume Quotas')]
  [FlagName(FILE_SUPPORTS_SPARSE_FILES, 'Supports Sparse Files')]
  [FlagName(FILE_SUPPORTS_REPARSE_POINTS, 'Supports Reparse Points')]
  [FlagName(FILE_SUPPORTS_REMOTE_STORAGE, 'Supports Remote Storage')]
  [FlagName(FILE_RETURNS_CLEANUP_RESULT_INFO, 'Returns Cleanup Result Info')]
  [FlagName(FILE_SUPPORTS_POSIX_UNLINK_RENAME, 'Supports Posix Unlink Rename')]
  [FlagName(FILE_VOLUME_IS_COMPRESSED, 'Volume Is Compressed')]
  [FlagName(FILE_SUPPORTS_OBJECT_IDS, 'Supports Object IDs')]
  [FlagName(FILE_SUPPORTS_ENCRYPTION, 'Supports Encryption')]
  [FlagName(FILE_NAMED_STREAMS, 'Named Streams')]
  [FlagName(FILE_READ_ONLY_VOLUME, 'Read-only Volume')]
  [FlagName(FILE_SEQUENTIAL_WRITE_ONCE, 'Sequential Write Once')]
  [FlagName(FILE_SUPPORTS_TRANSACTIONS, 'Supports Transactions')]
  [FlagName(FILE_SUPPORTS_HARD_LINKS, 'Supports Hardlinks')]
  [FlagName(FILE_SUPPORTS_EXTENDED_ATTRIBUTES, 'Supports EA')]
  [FlagName(FILE_SUPPORTS_OPEN_BY_FILE_ID, 'Supports Open By ID')]
  [FlagName(FILE_SUPPORTS_USN_JOURNAL, 'Supports USN Journal')]
  [FlagName(FILE_SUPPORTS_INTEGRITY_STREAMS, 'Supports Integrity Streams')]
  [FlagName(FILE_SUPPORTS_BLOCK_REFCOUNTING, 'Supports Block Ref. Counting')]
  [FlagName(FILE_SUPPORTS_SPARSE_VDL, 'Supports Sparse VDL')]
  [FlagName(FILE_DAX_VOLUME, 'DAX Volume')]
  [FlagName(FILE_SUPPORTS_GHOSTING, 'Supports Ghosting')]
  TFileSystemAttributes = type Cardinal;

  // WDK::ntifs.h - info class 5
  [SDKName('FILE_FS_ATTRIBUTE_INFORMATION')]
  TFileFsAttributeInformation = record
    FileSystemAttributes: TFileSystemAttributes;
    MaximumComponentNameLength: Integer;
    [Counter(ctBytes)] FileSystemNameLength: Cardinal;
    FileSystemName: TAnysizeArray<WideChar>;
  end;
  PFileFsAttributeInformation = ^TFileFsAttributeInformation;

  [FlagName(FILE_VC_QUOTA_NONE, 'No Quota')]
  [FlagName(FILE_VC_QUOTA_TRACK, 'Track Quota')]
  [FlagName(FILE_VC_QUOTA_ENFORCE, 'Enforce Quota')]
  [FlagName(FILE_VC_CONTENT_INDEX_DISABLED, 'Content Index Disabled')]
  [FlagName(FILE_VC_LOG_QUOTA_THRESHOLD, 'Quota Threshold')]
  [FlagName(FILE_VC_LOG_QUOTA_LIMIT, 'Log Quota Limit')]
  [FlagName(FILE_VC_LOG_VOLUME_THRESHOLD, 'Log Volume Threshold')]
  [FlagName(FILE_VC_LOG_VOLUME_LIMIT, 'Log Volume Limit')]
  [FlagName(FILE_VC_QUOTAS_INCOMPLETE, 'Quotas Incomplete')]
  [FlagName(FILE_VC_QUOTAS_REBUILDING, 'Quotas Rebuilding')]
  TFsControlFlags = type Cardinal;

  // WDK::ntifs.h - info class 6
  [SDKName('FILE_FS_CONTROL_INFORMATION')]
  TFileFsControlInformation = record
    FreeSpaceStartFiltering: UInt64;
    FreeSpaceThreshold: UInt64;
    FreeSpaceStopFiltering: UInt64;
    DefaultQuotaThreshold: UInt64;
    DefaultQuotaLimit: UInt64;
    FileSystemControlFlags: TFsControlFlags;
  end;
  PFileFsControlInformation = ^TFileFsControlInformation;

  // WDK::ntddk.h - info class 7
  [SDKName('FILE_FS_FULL_SIZE_INFORMATION')]
  TFileFsFullSizeInformation = record
    TotalAllocationUnits: UInt64;
    CallerAvailableAllocationUnits: UInt64;
    ActualAvailableAllocationUnits: UInt64;
    SectorsPerAllocationUnit: Cardinal;
    [Bytes] BytesPerSector: Cardinal;
  end;
  PFileFsFullSizeInformation = ^TFileFsFullSizeInformation;

  // WDK::ntddk.h - info class 8
  [SDKName('FILE_FS_OBJECTID_INFORMATION')]
  TFileFsObjectIdInformation = record
    ObjectID: TGuid;
    BirthVolumeId: TGuid;
    BirthObjectId: TGuid;
    DomainId: TGuid;
  end;
  PFileFsObjectIdInformation = ^TFileFsObjectIdInformation;

  // WDK::ntifs.h - info class 9
  [SDKName('FILE_FS_DRIVER_PATH_INFORMATION')]
  TFileFsDriverPathInformation = record
    DriverInPath: Boolean;
    [Counter(ctBytes)] DriverNameLength: Cardinal;
    DriverName: TAnysizeArray<WideChar>;
  end;
  PFileFsDriverPathInformation = ^TFileFsDriverPathInformation;

  [FlagName(SSINFO_FLAGS_ALIGNED_DEVICE, 'Aligned Device')]
  [FlagName(SSINFO_FLAGS_PARTITION_ALIGNED_ON_DEVICE, 'Partition Aligned On Device')]
  [FlagName(SSINFO_FLAGS_NO_SEEK_PENALTY, 'No Seek Penalty')]
  [FlagName(SSINFO_FLAGS_TRIM_ENABLED, 'Trim Enabled')]
  [FlagName(SSINFO_FLAGS_BYTE_ADDRESSABLE, 'Byte-addressable')]
  TFileFsSectorSizeFlags = type Cardinal;

  // WDK::ntddk.h - info class 11
  [MinOSVersion(OsWin8)]
  [SDKName('FILE_FS_SECTOR_SIZE_INFORMATION')]
  TFileFsSectorSizeInformation = record
    LogicalBytesPerSector: Cardinal;
    PhysicalBytesPerSectorForAtomicity: Cardinal;
    PhysicalBytesPerSectorForPerformance: Cardinal;
    FileSystemEffectivePhysicalBytesPerSectorForAtomicity: Cardinal;
    Flags: TFileFsSectorSizeFlags;
    ByteOffsetForSectorAlignment: Cardinal;
    ByteOffsetForPartitionAlignment: Cardinal;
  end;
  PFileFsSectorSizeInformation = ^TFileFsSectorSizeInformation;

  // WDK::ntddk.h - info class 14
  [MinOSVersion(OsWin10RS5)]
  [SDKName('FILE_FS_FULL_SIZE_INFORMATION_EX')]
  TFileFsFullSizeInformationEx = record
    ActualTotalAllocationUnits: UInt64;
    ActualAvailableAllocationUnits: UInt64;
    ActualPoolUnavailableAllocationUnits: UInt64;
    CallerTotalAllocationUnits: UInt64;
    CallerAvailableAllocationUnits: UInt64;
    CallerPoolUnavailableAllocationUnits: UInt64;
    UsedAllocationUnits: UInt64;
    TotalReservedAllocationUnits: UInt64;
    VolumeStorageReserveAllocationUnits: UInt64;
    AvailableCommittedAllocationUnits: UInt64;
    PoolAvailableAllocationUnits: UInt64;
    SectorsPerAllocationUnit: Cardinal;
    [Bytes] BytesPerSector: Cardinal;
  end;
  PFileFsFullSizeInformationEx = ^TFileFsFullSizeInformationEx;

  { NTFS structres }

  [SDKName('ATTRIBUTE_TYPE_CODE')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_UNUSED, 'Unused')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_STANDARD_INFORMATION, 'Standard Information')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_ATTRIBUTE_LIST, 'Attribute List')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_FILE_NAME, 'File Name')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_OBJECT_ID, 'Object ID')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_SECURITY_DESCRIPTOR, 'Security Descriptor')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_VOLUME_NAME, 'Volume Name')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_VOLUME_INFORMATION, 'Volume Information')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_DATA, 'Data')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_INDEX_ROOT, 'Index Root')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_INDEX_ALLOCATION, 'Index Allocation')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_BITMAP, 'Bitmap')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_REPARSE_POINT, 'Reparse Point')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_EA_INFORMATION, 'EA Information')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_EA, 'EA')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_LOGGED_UTILITY_STREAM, 'Logged Utility Stream')]
  [SubEnum(MAX_UINT, ATTRIBUTE_TYPE_CODE_END, 'End')]
  TAttributeTypeCode = type Cardinal;

  { FSCTLs }

  // WDK::ntifs.h - function numbers for corresponding FSCTL_* codes
  {$SCOPEDENUMS ON}
  TFsCtlFunction = (
    FSCTL_REQUEST_OPLOCK_LEVEL_1 = 0,    // void
    FSCTL_REQUEST_OPLOCK_LEVEL_2 = 1,    // void
    FSCTL_REQUEST_BATCH_OPLOCK = 2,      // void
    FSCTL_OPLOCK_BREAK_ACKNOWLEDGE = 3,  // void
    FSCTL_OPBATCH_ACK_CLOSE_PENDING = 4, // void
    FSCTL_OPLOCK_BREAK_NOTIFY = 5,       // void
    FSCTL_LOCK_VOLUME = 6,               // void
    FSCTL_UNLOCK_VOLUME = 7,             // void
    FSCTL_DISMOUNT_VOLUME = 8,           // void
    FSCTL_9,
    FSCTL_IS_VOLUME_MOUNTED = 10,        // void
    FSCTL_IS_PATHNAME_VALID = 11,
    FSCTL_MARK_VOLUME_DIRTY = 12,        // void
    FSCTL_13,
    FSCTL_QUERY_RETRIEVAL_POINTERS = 14,
    FSCTL_GET_COMPRESSION = 15,          // out: TCompressionFormat
    FSCTL_SET_COMPRESSION = 16,          // in: TCompressionFormat
    FSCTL_17,
    FSCTL_18,
    FSCTL_MARK_AS_SYSTEM_HIVE = 19,
    FSCTL_OPLOCK_BREAK_ACK_NO_2 = 20,    // void
    FSCTL_INVALIDATE_VOLUMES = 21,
    FSCTL_QUERY_FAT_BPB = 22,
    FSCTL_REQUEST_FILTER_OPLOCK = 23,    // void
    FSCTL_FILESYSTEM_GET_STATISTICS = 24,
    FSCTL_GET_NTFS_VOLUME_DATA = 25,
    FSCTL_GET_NTFS_FILE_RECORD = 26,     // in: TFileId, out: TNtfsFileRecordOutputBuffer
    FSCTL_GET_VOLUME_BITMAP = 27,
    FSCTL_GET_RETRIEVAL_POINTERS = 28,   // in: UInt64 (StartingVcn), out: TRetrievalPointersBuffer
    FSCTL_MOVE_FILE = 29,
    FSCTL_IS_VOLUME_DIRTY = 30,
    FSCTL_31,
    FSCTL_ALLOW_EXTENDED_DASD_IO = 32,
    FSCTL_33,
    FSCTL_34,
    FSCTL_FIND_FILES_BY_SID = 35,    // requires quota block
    FSCTL_36,
    FSCTL_37,
    FSCTL_SET_OBJECT_ID = 38,        // in: TFileObjectIdBuffer
    FSCTL_GET_OBJECT_ID = 39,        // out: TFileObjectIdBuffer
    FSCTL_DELETE_OBJECT_ID = 40,     // void
    FSCTL_SET_REPARSE_POINT = 41,    // in: TReparseDataBuffer
    FSCTL_GET_REPARSE_POINT = 42,    // out: TReparseDataBuffer
    FSCTL_DELETE_REPARSE_POINT = 43, // in: TReparseDataBuffer
    FSCTL_ENUM_USN_DATA = 44,
    FSCTL_SECURITY_ID_CHECK = 45,
    FSCTL_READ_USN_JOURNAL = 46,
    FSCTL_SET_OBJECT_ID_EXTENDED = 47,  // in: TFileObjectIdBuffer
    FSCTL_CREATE_OR_GET_OBJECT_ID = 48, // out: TFileObjectIdBuffer
    FSCTL_SET_SPARSE = 49,              // in: Boolean
    FSCTL_SET_ZERO_DATA = 50,           // in: TFileZeroDataInformation
    FSCTL_QUERY_ALLOCATED_RANGES = 51,
    FSCTL_ENABLE_UPGRADE = 52,
    FSCTL_SET_ENCRYPTION = 53,
    FSCTL_ENCRYPTION_FSCTL_IO = 54,
    FSCTL_WRITE_RAW_ENCRYPTED = 55,
    FSCTL_READ_RAW_ENCRYPTED = 56,
    FSCTL_CREATE_USN_JOURNAL = 57,
    FSCTL_READ_FILE_USN_DATA = 58,
    FSCTL_WRITE_USN_CLOSE_RECORD = 59,
    FSCTL_EXTEND_VOLUME = 60,           // in: UInt64 (number of sectors)
    FSCTL_QUERY_USN_JOURNAL = 61,
    FSCTL_DELETE_USN_JOURNAL = 62,
    FSCTL_MARK_HANDLE = 63,
    FSCTL_SIS_COPYFILE = 64,
    FSCTL_SIS_LINK_FILES = 65,
    FSCTL_66,
    FSCTL_67,
    FSCTL_68,
    FSCTL_RECALL_FILE = 69,   // void
    FSCTL_70,
    FSCTL_READ_FROM_PLEX = 71,
    FSCTL_FILE_PREFETCH = 72,
    FSCTL_73,
    FSCTL_74,
    FSCTL_75,
    FSCTL_MAKE_MEDIA_COMPATIBLE = 76,
    FSCTL_SET_DEFECT_MANAGEMENT = 77,
    FSCTL_QUERY_SPARING_INFO = 78,
    FSCTL_QUERY_ON_DISK_VOLUME_INFO = 79,
    FSCTL_SET_VOLUME_COMPRESSION_STATE = 80,
    FSCTL_TXFS_MODIFY_RM = 81,
    FSCTL_TXFS_QUERY_RM_INFORMATION = 82, // out: TTxfsQueryRmInformation
    FSCTL_83,
    FSCTL_TXFS_ROLLFORWARD_REDO = 84,
    FSCTL_TXFS_ROLLFORWARD_UNDO = 85,
    FSCTL_TXFS_START_RM = 86,
    FSCTL_TXFS_SHUTDOWN_RM = 87,
    FSCTL_TXFS_READ_BACKUP_INFORMATION = 88,
    FSCTL_TXFS_WRITE_BACKUP_INFORMATION = 89,
    FSCTL_TXFS_CREATE_SECONDARY_RM = 90,
    FSCTL_TXFS_GET_METADATA_INFO = 91,      // out: TTxfsGetMetadataInfoOut
    FSCTL_TXFS_GET_TRANSACTED_VERSION = 92, // out: TTxfsGetTransactedVersion
    FSCTL_93,
    FSCTL_TXFS_SAVEPOINT_INFORMATION = 94,  // not supported
    FSCTL_TXFS_CREATE_MINIVERSION = 95,     // not supported
    FSCTL_96,
    FSCTL_97,
    FSCTL_98,
    FSCTL_TXFS_TRANSACTION_ACTIVE = 99,   // out: Boolean
    FSCTL_100,
    FSCTL_SET_ZERO_ON_DEALLOCATION = 101, // void
    FSCTL_SET_REPAIR = 102,
    FSCTL_GET_REPAIR = 103,
    FSCTL_WAIT_FOR_REPAIR = 104,
    FSCTL_105,
    FSCTL_INITIATE_REPAIR = 106,
    FSCTL_CSC_INTERNAL = 107,
    FSCTL_SHRINK_VOLUME = 108,
    FSCTL_SET_SHORT_NAME_BEHAVIOR = 109,
    FSCTL_DFSR_SET_GHOST_HANDLE_STATE = 110,
    FSCTL_111,
    FSCTL_112,
    FSCTL_113,
    FSCTL_114,
    FSCTL_115,
    FSCTL_116,
    FSCTL_117,
    FSCTL_118,
    FSCTL_119,
    FSCTL_TXFS_LIST_TRANSACTION_LOCKED_FILES = 120, // in/out: TTxfsListTransactionLockedFiles
    FSCTL_TXFS_LIST_TRANSACTIONS = 121,
    FSCTL_QUERY_PAGEFILE_ENCRYPTION = 122,
    FSCTL_RESET_VOLUME_ALLOCATION_HINTS = 123,
    FSCTL_QUERY_DEPENDENT_VOLUME = 124,
    FSCTL_SD_GLOBAL_CHANGE = 125, // in: TSdGlobalChangeInput, out: TSdGlobalChangeOutput
    FSCTL_TXFS_READ_BACKUP_INFORMATION2 = 126,
    FSCTL_LOOKUP_STREAM_FROM_CLUSTER = 127,
    FSCTL_TXFS_WRITE_BACKUP_INFORMATION2 = 128,
    FSCTL_FILE_TYPE_NOTIFICATION = 129,
    FSCTL_FILE_LEVEL_TRIM = 130,
    FSCTL_131,
    FSCTL_132,
    FSCTL_133,
    FSCTL_134,
    FSCTL_135,
    FSCTL_136,
    FSCTL_137,
    FSCTL_138,
    FSCTL_139,
    FSCTL_GET_BOOT_AREA_INFO = 140,
    FSCTL_GET_RETRIEVAL_POINTER_BASE = 141,
    FSCTL_SET_PERSISTENT_VOLUME_STATE = 142,
    FSCTL_QUERY_PERSISTENT_VOLUME_STATE = 143,
    FSCTL_REQUEST_OPLOCK = 144,                // in: TRequestOplockInputBuffer, out: TRequestOplockOutputBuffer
    FSCTL_CSV_TUNNEL_REQUEST = 145,
    FSCTL_IS_CSV_FILE = 146,
    FSCTL_QUERY_FILE_SYSTEM_RECOGNITION = 147,
    FSCTL_CSV_GET_VOLUME_PATH_NAME = 148,
    FSCTL_CSV_GET_VOLUME_NAME_FOR_VOLUME_MOUNT_POINT = 149,
    FSCTL_CSV_GET_VOLUME_PATH_NAMES_FOR_VOLUME_NAME = 150,
    FSCTL_IS_FILE_ON_CSV_VOLUME = 151,
    FSCTL_CORRUPTION_HANDLING = 152,
    FSCTL_OFFLOAD_READ = 153,
    FSCTL_OFFLOAD_WRITE = 154,
    FSCTL_CSV_INTERNAL = 155,
    FSCTL_SET_PURGE_FAILURE_MODE = 156,
    FSCTL_QUERY_FILE_LAYOUT = 157,         // in: TQueryFileLayoutInput, out: TQueryFileLayoutOutput, Win 8+
    FSCTL_IS_VOLUME_OWNED_BYCSVFS = 158,
    FSCTL_GET_INTEGRITY_INFORMATION = 159,
    FSCTL_SET_INTEGRITY_INFORMATION = 160,
    FSCTL_QUERY_FILE_REGIONS = 161,
    FSCTL_162,
    FSCTL_163,
    FSCTL_164,
    FSCTL_165,
    FSCTL_166,
    FSCTL_167,
    FSCTL_168,
    FSCTL_169,
    FSCTL_170,
    FSCTL_RKF_INTERNAL = 171,
    FSCTL_SCRUB_DATA = 172,
    FSCTL_REPAIR_COPIES = 173,
    FSCTL_DISABLE_LOCAL_BUFFERING = 174,
    FSCTL_CSV_MGMT_LOCK = 175,
    FSCTL_CSV_QUERY_DOWN_LEVEL_FILE_SYSTEM_CHARACTERISTICS = 176,
    FSCTL_ADVANCE_FILE_ID = 177,
    FSCTL_CSV_SYNC_TUNNEL_REQUEST = 178,
    FSCTL_CSV_QUERY_VETO_FILE_DIRECT_IO = 179,
    FSCTL_WRITE_USN_REASON = 180,
    FSCTL_CSV_CONTROL = 181,
    FSCTL_GET_REFS_VOLUME_DATA = 182,
    FSCTL_SET_BREAK_ON_STATUS = 183, // private
    FSCTL_CBAFILT_IGNORE_ADS_CHANGES = 184, // private
    FSCTL_CSV_H_BREAKING_SYNC_TUNNEL_REQUEST = 185,
    FSCTL_MPFILTER_QUERY_FILE_CHANGE = 186, // private
    FSCTL_QUERY_STORAGE_CLASSES = 187,
    FSCTL_QUERY_REGION_INFO = 188,
    FSCTL_USN_TRACK_MODIFIED_RANGES = 189,
    FSCTL_USN_SUBMIT_MODIFIED_RANGES = 190, // private
    FSCTL_191,
    FSCTL_QUERY_SHARED_VIRTUAL_DISK_SUPPORT = 192,
    FSCTL_SVHDX_SYNC_TUNNEL_REQUEST = 193,
    FSCTL_SVHDX_SET_INITIATOR_INFORMATION = 194,
    FSCTL_SET_EXTERNAL_BACKING = 195, // in: TFileProviderExternalInfoV1
    FSCTL_GET_EXTERNAL_BACKING = 196, // out: TFileProviderExternalInfoV1
    FSCTL_DELETE_EXTERNAL_BACKING = 197,
    FSCTL_ENUM_EXTERNAL_BACKING = 198,
    FSCTL_ENUM_OVERLAY = 199,
    FSCTL_START_OVERLAY_INTEGRITY = 200, // private
    FSCTL_STOP_OVERLAY_INTEGRITY = 201, // private
    FSCTL_CONTROL_OVERLAY_INTEGRITY = 202, // private
    FSCTL_QUERY_OVERLAY_INTEGRITY = 203, // private
    FSCTL_ADD_OVERLAY = 204,
    FSCTL_REMOVE_OVERLAY = 205,
    FSCTL_UPDATE_OVERLAY = 206,
    FSCTL_207,
    FSCTL_SHUFFLE_FILE = 208, // private
    FSCTL_DUPLICATE_EXTENTS_TO_FILE = 209,
    FSCTL_CHECK_FOR_SECTION = 210,   // void // private
    FSCTL_SPARSE_OVERALLOCATE = 211,
    FSCTL_STORAGE_QOS_CONTROL = 212,
    FSCTL_CLOUD_DATA_TRANSFER = 213, // private
    FSCTL_CLOUD_COMMAND = 214, // private
    FSCTL_INITIATE_FILE_METADATA_OPTIMIZATION = 215,
    FSCTL_QUERY_FILE_METADATA_OPTIMIZATION = 216,
    FSCTL_SVHDX_ASYNC_TUNNEL_REQUEST = 217,
    FSCTL_GET_WOF_VERSION = 218,
    FSCTL_HCS_SYNC_TUNNEL_REQUEST = 219,
    FSCTL_HCS_ASYNC_TUNNEL_REQUEST = 220,
    FSCTL_QUERY_EXTENT_READ_CACHE_INFO = 221,
    FSCTL_QUERY_REFS_VOLUME_COUNTER_INFO = 222,
    FSCTL_CLEAN_VOLUME_METADATA = 223,
    FSCTL_SET_INTEGRITY_INFORMATION_EX = 224,
    FSCTL_SUSPEND_OVERLAY = 225,
    FSCTL_VIRTUAL_STORAGE_QUERY_PROPERTY = 226,
    FSCTL_FILESYSTEM_GET_STATISTICS_EX = 227,
    FSCTL_QUERY_VOLUME_CONTAINER_STATE = 228,
    FSCTL_SET_LAYER_ROOT = 229,
    FSCTL_QUERY_DIRECT_ACCESS_EXTENTS = 230,
    FSCTL_NOTIFY_STORAGE_SPACE_ALLOCATION = 231,
    FSCTL_SSDI_STORAGE_REQUEST = 232,
    FSCTL_QUERY_DIRECT_IMAGE_ORIGINAL_BASE = 233,
    FSCTL_READ_UNPRIVILEGED_USN_JOURNAL = 234,
    FSCTL_GHOST_FILE_EXTENTS = 235,
    FSCTL_QUERY_GHOSTED_FILE_EXTENTS = 236,
    FSCTL_UNMAP_SPACE = 237,
    FSCTL_HCS_SYNC_NO_WRITE_TUNNEL_REQUEST = 238,
    FSCTL_HSM_CONTROL = 239, // private
    FSCTL_START_VIRTUALIZATION_INSTANCE = 240,
    FSCTL_GET_FILTER_FILE_IDENTIFIER = 241,
    FSCTL_STREAMS_ASSOCIATE_ID = 242,
    FSCTL_STREAMS_QUERY_ID = 243,
    FSCTL_GET_RETRIEVAL_POINTERS_AND_REFCOUNT = 244,
    FSCTL_QUERY_VOLUME_NUMA_INFO = 245,
    FSCTL_REFS_DEALLOCATE_RANGES = 246,
    FSCTL_QUERY_REFS_SMR_VOLUME_INFO = 247,
    FSCTL_SET_REFS_SMR_VOLUME_GC_PARAMETERS = 248,
    FSCTL_SET_REFS_FILE_STRICTLY_SEQUENTIAL = 249,
    FSCTL_DUPLICATE_EXTENTS_TO_FILE_EX = 250,
    FSCTL_QUERY_BAD_RANGES = 251,
    FSCTL_SET_DAX_ALLOC_ALIGNMENT_HINT = 252,
    FSCTL_DELETE_CORRUPTED_REFS_CONTAINER = 253,
    FSCTL_SCRUB_UNDISCOVERABLE_ID = 254,
    FSCTL_NOTIFY_DATA_CHANGE = 255,
    FSCTL_START_VIRTUALIZATION_INSTANCE_EX = 256,
    FSCTL_ENCRYPTION_KEY_CONTROL = 257,
    FSCTL_VIRTUAL_STORAGE_SET_BEHAVIOR = 258,
    FSCTL_SET_REPARSE_POINT_EX = 259,   // in: TReparseDataBufferEx, Win 10 RS5+
    FSCTL_DEFINE_STORAGE_RESERVE = 260, // rev
    FSCTL_QUERY_STORAGE_RESERVE = 261,  // rev
    FSCTL_DELETE_STORAGE_RESERVE = 262, // rev
    FSCTL_REPAIR_STORAGE_RESERVE = 263, // rev
    FSCTL_REARRANGE_FILE = 264,
    FSCTL_VIRTUAL_STORAGE_PASSTHROUGH = 265,
    FSCTL_GET_RETRIEVAL_POINTER_COUNT = 266,
    FSCTL_ENABLE_PER_IO_FLAGS = 267,
    FSCTL_QUERY_ASYNC_DUPLICATE_EXTENTS_STATUS = 268,
    FSCTL_269,
    FSCTL_270,
    FSCTL_SMB_SHARE_FLUSH_AND_PURGE = 271,
    FSCTL_REFS_STREAM_SNAPSHOT_MANAGEMENT = 272,
    FSCTL_QUERY_CACHED_RUNS = 273, // rev
    FSCTL_MANAGE_BYPASS_IO = 274,
    FSCTL_REFS_DEALLOCATE_RANGES_EX = 275,
    FSCTL_SET_CACHED_RUNS_STATE = 276,
    FSCTL_REFS_SET_VOLUME_COMPRESSION_INFO = 277,
    FSCTL_REFS_QUERY_VOLUME_COMPRESSION_INFO = 278,
    FSCTL_DUPLICATE_CLUSTER = 279,
    FSCTL_CREATE_LCN_WEAK_REFERENCE = 280,
    FSCTL_DELETE_LCN_WEAK_REFERENCE = 281,
    FSCTL_QUERY_LCN_WEAK_REFERENCE = 282,
    FSCTL_DELETE_LCN_WEAK_REFERENCES = 283,
    FSCTL_REFS_SET_VOLUME_DEDUP_INFO = 284,
    FSCTL_REFS_QUERY_VOLUME_DEDUP_INFO = 285
  );
  {$SCOPEDENUMS OFF}

  // WDK::ntifs.h - FSCTL 26 (output)
  [SDKName('NTFS_FILE_RECORD_OUTPUT_BUFFER')]
  TNtfsFileRecordOutputBuffer = record
    FileReferenceNumber: TFileId;
    [NumberOfBytes] FileRecordLength: Cardinal;
    FileRecordBuffer: TPlaceholder;
  end;
  PNtfsFileRecordOutputBuffer = ^TNtfsFileRecordOutputBuffer;

  // WDK::ntifs.h (unnamed)
  TRetrievalPointersBufferExtents = record
    NextVcn: UInt64;
    Lcn: UInt64;
  end;

  // WDK::ntifs.h - FSCTL 28 (output)
  [SDKName('RETRIEVAL_POINTERS_BUFFER')]
  TRetrievalPointersBuffer = record
    [Counter] ExtentCount: Cardinal;
    StartingVcn: UInt64;
    Extents: TAnysizeArray<TRetrievalPointersBufferExtents>;
  end;
  PRetrievalPointersBuffer = ^TRetrievalPointersBuffer;

  // WDK::ntifs.h - FSCTLs  38, 39, 47, 48
  [SDKName('FILE_OBJECTID_BUFFER')]
  TFileObjectIdBuffer = record
    ObjectId: TFileId128;
    BirthVolumeId: TGuid;
    BirthObjectId: TFileId128;
    DomainId: TGuid;
  end;
  PFileObjectIdBuffer = ^TFileObjectIdBuffer;

  // WDK::ntifs.h - FSCTL 41, 42, 43
  [SDKName('REPARSE_DATA_BUFFER')]
  TReparseDataBuffer = record
    ReparseTag: TReparseTag;
    [NumberOfBytes] ReparseDataLength: Word;
    [Unlisted] Reserved: Word;
    DataBuffer: TPlaceholder;
  end;
  PReparseDataBuffer = ^TReparseDataBuffer;

  // WDK::ntifs.h - FSCTL 50
  [SDKName('FILE_ZERO_DATA_INFORMATION')]
  TFileZeroDataInformation = record
    [Offset] FileOffset: UInt64;
    [Offset] BeyondFinalZero: UInt64;
  end;
  PFileZeroDataInformation = ^TFileZeroDataInformation;

  // WDK::ntifs.h
  TSdGlobalChangeType = (
    SD_GLOBAL_CHANGE_TYPE_MACHINE_SID = $00000001,
    SD_GLOBAL_CHANGE_TYPE_QUERY_STATS = $00010000,
    SD_GLOBAL_CHANGE_TYPE_ENUM_SDS = $00020000
  );

  // WDK::ntifs.h
  [SDKName('SD_CHANGE_MACHINE_SID_INPUT')]
  TSdChangeMachineSidInput = record
    [Offset] CurrentMachineSIDOffset: Word;
    [Bytes] CurrentMachineSIDLength: Word;
    [Offset] NewMachineSIDOffset: Word;
    [Bytes] NewMachineSIDLength: Word;
  end;
  PSdChangeMachineSidInput = ^TSdChangeMachineSidInput;

  // WDK::ntifs.h
  [SDKName('SD_CHANGE_MACHINE_SID_OUTPUT')]
  TSdChangeMachineSidOutput = record
    NumSDChangedSuccess: UInt64;
    NumSDChangedFail: UInt64;
    NumSDUnused: UInt64;
    NumSDTotal: UInt64;
    NumMftSDChangedSuccess: UInt64;
    NumMftSDChangedFail: UInt64;
    NumMftSDTotal: UInt64;
  end;
  PSdChangeMachineSidOutput = ^TSdChangeMachineSidOutput;

  // WDK::ntifs.h
  [SDKName('SD_QUERY_STATS_OUTPUT')]
  TSdQueryStatsOutput = record
    [Bytes] SdsStreamSize: UInt64;
    [Bytes] SdsAllocationSize: UInt64;
    [Bytes] SiiStreamSize: UInt64;
    [Bytes] SiiAllocationSize: UInt64;
    [Bytes] SdhStreamSize: UInt64;
    [Bytes] SdhAllocationSize: UInt64;
    NumSDTotal: UInt64;
    NumSDUnused: UInt64;
  end;

  // WDK::ntifs.h
  [SDKName('SD_ENUM_SDS_INPUT')]
  TSdEnumSDsInput = record
    [Hex] StartingOffset: UInt64;
    MaxSDEntriesToReturn: UInt64;
  end;

  // WDK::ntifs.h
  [SDKName('SD_ENUM_SDS_ENTRY')]
  TSdEnumSDsEntry = record
    [Hex] Hash: Cardinal;
    SecurityId: Cardinal;
    [Hex] Offset: UInt64;
    [Bytes] Length: Cardinal;
    Descriptor: TPlaceholder<TSecurityDescriptor>;
  end;
  PSdEnumSDsEntry = ^TSdEnumSDsEntry;

  // WDK::ntifs.h
  [SDKName('SD_ENUM_SDS_OUTPUT')]
  TSdEnumSDsOutput = record
    [Offset] NextOffset: UInt64;
    NumSDEntriesReturned: UInt64;
    [Bytes] NumSDBytesReturned: UInt64;
    SDEntry: TSdEnumSDsEntry;
  end;

  // WDK::ntifs.h - function 125 (input)
  [SDKName('SD_GLOBAL_CHANGE_INPUT')]
  TSdGlobalChangeInput = record
    [Reserved(0)] Flags: Cardinal;
  case ChangeType: TSdGlobalChangeType of
    SD_GLOBAL_CHANGE_TYPE_MACHINE_SID: (
      SdChange: TSdChangeMachineSidInput;
    );

    SD_GLOBAL_CHANGE_TYPE_QUERY_STATS: (
      [Unlisted] Reserved: Cardinal;
    );

    SD_GLOBAL_CHANGE_TYPE_ENUM_SDS: (
      SdEnumSds: TSdEnumSDsInput;
    );
  end;
  PSdGlobalChangeInput = ^TSdGlobalChangeInput;

  // WDK::ntifs.h - function 125 (output)
  [SDKName('SD_GLOBAL_CHANGE_OUTPUT')]
  TSdGlobalChangeOutput = record
    [Unlisted] Flags: Cardinal;
  case ChangeType: TSdGlobalChangeType of
    SD_GLOBAL_CHANGE_TYPE_MACHINE_SID: (
      SdChange: TSdChangeMachineSidOutput;
    );

    SD_GLOBAL_CHANGE_TYPE_QUERY_STATS: (
      SdQueryStats: TSdQueryStatsOutput;
    );

    SD_GLOBAL_CHANGE_TYPE_ENUM_SDS: (
      SdEnumSds: TSdEnumSDsOutput;
    );
  end;
  PSdGlobalChangeOutput = ^TSdGlobalChangeOutput;

  [FlagName(OPLOCK_LEVEL_CACHE_READ, 'Cache Read')]
  [FlagName(OPLOCK_LEVEL_CACHE_HANDLE, 'Cache Handle')]
  [FlagName(OPLOCK_LEVEL_CACHE_WRITE, 'Cache Write')]
  TOpLockLevel = type Cardinal;

  [FlagName(REQUEST_OPLOCK_INPUT_FLAG_REQUEST, 'Request')]
  [FlagName(REQUEST_OPLOCK_INPUT_FLAG_ACK, 'Acknowledge')]
  [FlagName(REQUEST_OPLOCK_INPUT_FLAG_COMPLETE_ACK_ON_CLOSE, 'Complete Acknowledge On Close')]
  TOpLockInputFlags = type Cardinal;

  // WDK::ntifs.h - FSCTL 144 (input)
  [SDKName('REQUEST_OPLOCK_INPUT_BUFFER')]
  TRequestOplockInputBuffer = record
    [Reserved(REQUEST_OPLOCK_CURRENT_VERSION)] StructureVersion: Word;
    StructureLength: Word;
    RequestedOplockLevel: TOpLockLevel;
    Flags: TOpLockInputFlags;
  end;

  [FlagName(REQUEST_OPLOCK_OUTPUT_FLAG_ACK_REQUIRED, 'Acknowledge Required')]
  [FlagName(REQUEST_OPLOCK_OUTPUT_FLAG_MODES_PROVIDED, 'Modes Provided')]
  TOpLockOutputFlags = type Cardinal;

  // WDK::ntifs.h - FSCTL 144 (output)
  [SDKName('REQUEST_OPLOCK_OUTPUT_BUFFER')]
  TRequestOplockOutputBuffer = record
    [Reserved(REQUEST_OPLOCK_CURRENT_VERSION)] StructureVersion: Word;
    StructureLength: Word;
    OriginalOplockLevel: TOpLockLevel;
    NewOplockLevel: TOpLockLevel;
    Flags: TOpLockOutputFlags;
    AccessMode: TAccessMask;
    ShareMode: Word;
  end;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('CLUSTER_RANGE')]
  TClusterRange = record
    StartingCluster: UInt64;
    ClusterCount: UInt64;
  end;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('FILE_REFERENCE_RANGE')]
  TFileReferenceRange = record
    StartingFileReferenceNumber: UInt64;
    EndingFileReferenceNumber: UInt64;
  end;

  [FlagName(QUERY_FILE_LAYOUT_RESTART, 'Restart')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_NAMES, 'Names')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAMS, 'Streams')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_EXTENTS, 'Extents')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_EXTRA_INFO, 'Extra Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAMS_WITH_NO_CLUSTERS_ALLOCATED, 'No-cluster Streams')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_FULL_PATH_IN_NAMES, 'Full Paths')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION, 'Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_DSC_ATTRIBUTE, 'Storage Class Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_TXF_ATTRIBUTE, 'TxF Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_EFS_ATTRIBUTE, 'EFS Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_ONLY_FILES_WITH_SPECIFIC_ATTRIBUTES, 'Only Specific Attributes')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_FILES_WITH_DSC_ATTRIBUTE, 'Files with Storage Class')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_DATA_ATTRIBUTE, 'Data Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_REPARSE_ATTRIBUTE, 'Reparse Stream Info')]
  [FlagName(QUERY_FILE_LAYOUT_INCLUDE_STREAM_INFORMATION_FOR_EA_ATTRIBUTE, 'EA Stream Info')]
  TQueryFileLayoutInputFlags = type Cardinal;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('QUERY_FILE_LAYOUT_FILTER_TYPE')]
  [NamingStyle(nsSnakeCase, 'QUERY_FILE_LAYOUT_FILTER_TYPE')]
  TQueryFileLayoutFilterType = (
    QUERY_FILE_LAYOUT_FILTER_TYPE_NONE = 0,
    QUERY_FILE_LAYOUT_FILTER_TYPE_CLUSTERS = 1,
    QUERY_FILE_LAYOUT_FILTER_TYPE_FILEID = 2,
    QUERY_FILE_LAYOUT_FILTER_TYPE_STORAGE_RESERVE_ID = 3 // Win 10 RS5+
  );

  // WDK::ntifs.h - FSCTL 157 (input)
  [MinOSVersion(OsWin8)]
  [SDKName('QUERY_FILE_LAYOUT_INPUT')]
  TQueryFileLayoutInput = record
    [Counter] FilterEntryCount: Cardinal;
    Flags: TQueryFileLayoutInputFlags;
    FilterType: TQueryFileLayoutFilterType;
    [Unlisted] Reserved: Cardinal;
  case TQueryFileLayoutFilterType of
    QUERY_FILE_LAYOUT_FILTER_TYPE_CLUSTERS:
      (ClusterRanges: TAnysizeArray<TClusterRange>);
    QUERY_FILE_LAYOUT_FILTER_TYPE_FILEID:
      (FileReferenceRanges: TAnysizeArray<TFileReferenceRange>);
    QUERY_FILE_LAYOUT_FILTER_TYPE_STORAGE_RESERVE_ID:
      (StorageReserveIds: TAnysizeArray<TStorageReserveId>);
  end;
  PQueryFileLayoutInput = ^TQueryFileLayoutInput;

  [FlagName(FILE_LAYOUT_NAME_ENTRY_PRIMARY, 'Primary Name')]
  [FlagName(FILE_LAYOUT_NAME_ENTRY_DOS, 'DOS Name')]
  TFileLayoutNameFlags = type Cardinal;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('FILE_LAYOUT_NAME_ENTRY')]
  TFileLayoutNameEntry = record
    [Offset] NextNameOffset: Cardinal; // from this struct
    Flags: TFileLayoutNameFlags;
    ParentFileReferenceNumber: TFileId;
    [Counter(ctBytes)] FileNameLength: Cardinal;
    [Unlisted] Reserved: Cardinal;
    FileName: TAnysizeArray<WideChar>;
  end;
  PFileLayoutNameEntry = ^TFileLayoutNameEntry;

  [FlagName(STREAM_EXTENT_ENTRY_AS_RETRIEVAL_POINTERS, 'As Retrieval Pointers')]
  [FlagName(STREAM_EXTENT_ENTRY_ALL_EXTENTS, 'All Extents')]
  TStreamExtentEntryFlags = type Cardinal;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('STREAM_EXTENT_ENTRY')]
  TStreamExtentEntry = record
    Flags: TStreamExtentEntryFlags;
    RetrievalPointers: TRetrievalPointersBuffer;
  end;
  PStreamExtentEntry = ^TStreamExtentEntry;

  // WDK::ntifs.h (unnamed)
  TStreamInformationDataEntry = record
    [RecordSize] Length: Word;
    [Hex] Flags: Word;
    [Unlisted] Reserved: Cardinal;
    [Bytes] Vdl: UInt64;
    Data: TPlaceholder;
  end;

  [FlagName(QUERY_FILE_LAYOUT_REPARSE_DATA_INVALID, 'Reparse Data Invalid')]
  [FlagName(QUERY_FILE_LAYOUT_REPARSE_TAG_INVALID, 'Reparse Tag Invalid')]
  TStreamInformationReparseFlags = type Word;

  // WDK::ntifs.h (unnamed)
  TStreamInformationReparseEntry = record
    [RecordSize] Length: Word;
    Flags: TStreamInformationReparseFlags;
    [Bytes] ReparseDataSize: Cardinal;
    [Offset] ReparseDataOffset: Cardinal;
  end;

  // WDK::ntifs.h (unnamed)
  TStreamInformationEAEntry = record
    [RecordSize] Length: Word;
    [Hex] Flags: Word;
    EaSize: Cardinal;
    [Offset] EaInformationOffset: Cardinal; // to TFileFullEaInformation
  end;

  // WDK::ntifs.h
  [SDKName('STREAM_INFORMATION_ENTRY')]
  TStreamInformationEntry = record
    [Reserved(1)] Version: Cardinal;
    Flags: Cardinal;
  case TAttributeTypeCode of
    ATTRIBUTE_TYPE_CODE_LOGGED_UTILITY_STREAM: (
      DesiredStorageClass: TFileDesiredStorageClassInformation
    );
    ATTRIBUTE_TYPE_CODE_DATA: (
      Data: TStreamInformationDataEntry;
    );
    ATTRIBUTE_TYPE_CODE_REPARSE_POINT: (
      ReparsePoint: TStreamInformationReparseEntry;
    );
    ATTRIBUTE_TYPE_CODE_EA_INFORMATION: (
      ExtendedAttribute: TStreamInformationEAEntry;
    );
  end;
  PStreamInformationEntry = ^TStreamInformationEntry;

  [FlagName(STREAM_LAYOUT_ENTRY_IMMOVABLE, 'Immovable')]
  [FlagName(STREAM_LAYOUT_ENTRY_PINNED, 'Pinned')]
  [FlagName(STREAM_LAYOUT_ENTRY_RESIDENT, 'Resident')]
  [FlagName(STREAM_LAYOUT_ENTRY_NO_CLUSTERS_ALLOCATED, 'No Clusters Allocated')]
  [FlagName(STREAM_LAYOUT_ENTRY_HAS_INFORMATION, 'Has Information')]
  TFileLayoutStreamFlags = type Cardinal;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('STREAM_LAYOUT_ENTRY')]
  TStreamLayoutEntry = record
    [Reserved(2)] Version: Cardinal;
    [Offset] NextStreamOffset: Cardinal; // from this struct
    Flags: TFileLayoutStreamFlags;
    [Offset] ExtentInformationOffset: Cardinal; // to TStreamExtentEntry from this struct
    [Bytes] AllocationSize: UInt64;
    [Bytes] EndOfFile: UInt64;
    [Offset] StreamInformationOffset: Cardinal; // to TStreamInformationEntry from this struct
    AttributeTypeCode: TAttributeTypeCode;
    AttributeFlags: TFileAttributes;
    [Counter(ctBytes)] StreamIdentifierLength: Cardinal;
    StreamIdentifier: TAnysizeArray<WideChar>;
  end;
  PStreamLayoutEntry = ^TStreamLayoutEntry;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('FILE_LAYOUT_INFO_ENTRY')]
  TFileLayoutInfoEntry = record
    BasicInformation: TFileBasicInformation;
    OwnerId: Cardinal;
    SecurityId: Cardinal;
    Usn: TUsn;
    [MinOSVersion(OsWin10RS5)] StorageReserveId: TStorageReserveId;
  end;
  PFileLayoutInfoEntry = ^TFileLayoutInfoEntry;

  // WDK::ntifs.h
  [MinOSVersion(OsWin8)]
  [SDKName('FILE_LAYOUT_ENTRY')]
  TFileLayoutEntry = record
    [Reserved(1)] Version: Cardinal;
    [Offset] NextFileOffset: Cardinal;
    [Hex] Flags: Cardinal;
    FileAttributes: TFileAttributes;
    FileReferenceNumber: TFileId;
    [Offset] FirstNameOffset: Cardinal;   // to TFileLayoutNameEntry from this struct
    [Offset] FirstStreamOffset: Cardinal; // to TStreamLayoutEntry from this struct
    [Offset] ExtraInfoOffset: Cardinal;   // to TFileLayoutInfoEntry
    ExtraInfoLength: Cardinal;   // Win 10 RS5+
  end;
  PFileLayoutEntry = ^TFileLayoutEntry;

  [FlagName(QUERY_FILE_LAYOUT_SINGLE_INSTANCED, 'Single Instanced')]
  TQueryFileLayoutOutputFlags = type Cardinal;

  // WDK::ntifs.h - FSCTL 157 (output)
  [MinOSVersion(OsWin8)]
  [SDKName('QUERY_FILE_LAYOUT_OUTPUT')]
  TQueryFileLayoutOutput = record
    [Counter] FileEntryCount: Cardinal;
    [Offset] FirstFileOffset: Cardinal; // to TFileLayoutEntry
    Flags: TQueryFileLayoutOutputFlags;
    [Unlisted] Reserved: Cardinal;
  end;
  PQueryFileLayoutOutput = ^TQueryFileLayoutOutput;

  // WDK::ntifs.h
  [NamingStyle(nsSnakeCase, 'WOF_PROVIDER'), Range(1)]
  TWofProvider = (
    [Reserved] WOF_PROVIDER_UNKNOWN = 0,
    WOF_PROVIDER_WIM = 1,
    WOF_PROVIDER_FILE = 2,
    WOF_PROVIDER_CLOUD = 3
  );

  // WDK::ntifs.h
  [NamingStyle(nsSnakeCase, 'FILE_PROVIDER_COMPRESSION')]
  TFileProviderCompression = (
    FILE_PROVIDER_COMPRESSION_XPRESS4K = 0,
    FILE_PROVIDER_COMPRESSION_LZX = 1,
    FILE_PROVIDER_COMPRESSION_XPRESS8K = 2,
    FILE_PROVIDER_COMPRESSION_XPRESS16K = 3
  );

  // WDK::ntifs.h
  [SDKName('WOF_EXTERNAL_INFO')]
  TWofExternalInfo = record
    [Reserved(WOF_CURRENT_VERSION)] Version: Cardinal;
    Provider: TWofProvider;
  end;

  // WDK::ntifs.h - FSCTL 195 (input) & FSCTL 196 (output)
  [SDKName('FILE_PROVIDER_EXTERNAL_INFO_V1')]
  TFileProviderExternalInfoV1 = record
    WofInfo: TWofExternalInfo; // Embedded for convenience
    [Reserved(FILE_PROVIDER_CURRENT_VERSION)] Version: Cardinal;
    Algorithm: TFileProviderCompression;
    Flags: Cardinal;
  end;
  PFileProviderExternalInfoV1 = ^TFileProviderExternalInfoV1;

  [FlagName(REPARSE_DATA_EX_FLAG_GIVEN_TAG_OR_NONE, 'Given Tag Or None')]
  TReparseDataBufferExFlag = type Cardinal;

  // WDK::ntifs.h - FSCTL 259
  [MinOSVersion(OsWin10RS5)]
  [SDKName('REPARSE_DATA_BUFFER_EX')]
  TReparseDataBufferEx = record
    Flags: TReparseDataBufferExFlag;
    ExistingReparseTag: TReparseTag;
    ExistingReparseGuid: TGuid;
    [Unlisted] Reserved: UInt64;
    ReparseDataBuffer: TReparseDataBuffer;
  end;
  PReparseDataBufferEx = ^TReparseDataBufferEx;

  { Pipes }

  // WDK::ntifs.h - function numbers corresponding FSCTL_PIPE_* codes
  {$SCOPEDENUMS ON}
  TFsCtlPipeFunction = (
    FSCTL_PIPE_ASSIGN_EVENT = 0,
    FSCTL_PIPE_DISCONNECT = 1,           // No input/output
    FSCTL_PIPE_LISTEN = 2,               // No input/output
    FSCTL_PIPE_PEEK = 3,                 // out: TFilePipePeekBuffer
    FSCTL_PIPE_QUERY_EVENT = 4,
    FSCTL_PIPE_TRANSCEIVE = 5,           // User-provided input/output
    FSCTL_PIPE_WAIT = 6,                 // in: TFilePipeWaitForBuffer
    FSCTL_PIPE_IMPERSONATE = 7,          // No input/output
    FSCTL_PIPE_SET_CLIENT_PROCESS = 8,
    FSCTL_PIPE_QUERY_CLIENT_PROCESS = 9,
    FSCTL_PIPE_GET_PIPE_ATTRIBUTE = 10,
    FSCTL_PIPE_SET_PIPE_ATTRIBUTE = 11,
    FSCTL_PIPE_GET_CONNECTION_ATTRIBUTE = 12,
    FSCTL_PIPE_SET_CONNECTION_ATTRIBUTE = 13,
    FSCTL_PIPE_GET_HANDLE_ATTRIBUTE = 14,
    FSCTL_PIPE_SET_HANDLE_ATTRIBUTE = 15,
    FSCTL_PIPE_FLUSH = 16,
    FSCTL_PIPE_DISABLE_IMPERSONATE = 17, // No input/output, Win 10 RS1+
    FSCTL_PIPE_SILO_ARRIVAL = 18,        // Win 10 RS3+
    FSCTL_PIPE_CREATE_SYMLINK = 19, // in: TFilePipeCreateSymlinkInput
    FSCTL_PIPE_DELETE_SYMLINK = 20, // in: TFilePipeDeleteSymlinkInput
    FSCTL_PIPE_QUERY_CLIENT_PROCESS_V2 = 21 // Win 10 19H1+
  );
  {$SCOPEDENUMS OFF}

  // WDK::ntifs.h - pipe FSCTL 3
  [SDKName('FILE_PIPE_PEEK_BUFFER')]
  TFilePipePeekBuffer = record
     NamedPipeState: Cardinal;
     [Bytes] ReadDataAvailable: Cardinal;
     NumberOfMessages: Cardinal;
     [Bytes] MessageLength: Cardinal;
     Data: TPlaceholder;
  end;
  PFilePipePeekBuffer = ^TFilePipePeekBuffer;

  // WDK::ntifs.h - pipe FSCTL 6
  [SDKName('FILE_PIPE_WAIT_FOR_BUFFER')]
  TFilePipeWaitForBuffer = record
    Timeout: TLargeInteger;
    [NumberOfBytes] NameLength: Cardinal;
    TimeoutSpecified: Boolean;
    Name: TAnysizeArray<WideChar>;
  end;
  PFilePipeWaitForBuffer = ^TFilePipeWaitForBuffer;

  [FlagName(FILE_PIPE_SYMLINK_FLAG_GLOBAL, 'Global')]
  [FlagName(FILE_PIPE_SYMLINK_FLAG_RELATIVE, 'Relative')]
  TFilePipeSymlinkFlags = type Cardinal;

  // WDK::ntifs.h - pipe FSCTL 19
  [MinOSVersion(OsWin10RS3)]
  [SDKName('FILE_PIPE_CREATE_SYMLINK_INPUT')]
  TFilePipeCreateSymlinkInput = record
    [Offset] NameOffset: Word; // to PWideChar
    [NumberOfBytes] NameLength: Word;
    [Offset] SubstituteNameOffset: Word; // to PWideChar
    [NumberOfBytes] SubstituteNameLength: Word;
    Flags: TFilePipeSymlinkFlags;
  end;
  PFilePipeCreateSymlinkInput = ^TFilePipeCreateSymlinkInput;

  // WDK::ntifs.h - pipe FSCTL 20
  [MinOSVersion(OsWin10RS3)]
  [SDKName('FILE_PIPE_DELETE_SYMLINK_INPUT')]
  TFilePipeDeleteSymlinkInput = record
    [Offset] NameOffset: Word;
    [NumberOfBytes] NameLength: Word;
  end;
  PFilePipeDeleteSymlinkInput = ^TFilePipeDeleteSymlinkInput;

  TFilePipeClientComputerBuffer = array [0..FILE_PIPE_COMPUTER_NAME_LENGTH] of WideChar;

const
  FSCTL_REQUEST_OPLOCK_LEVEL_1 = $00090000;
  FSCTL_REQUEST_OPLOCK_LEVEL_2 = $00090004;
  FSCTL_REQUEST_BATCH_OPLOCK = $00090008;
  FSCTL_OPLOCK_BREAK_ACKNOWLEDGE = $0009000C;
  FSCTL_OPBATCH_ACK_CLOSE_PENDING = $00090010;
  FSCTL_OPLOCK_BREAK_NOTIFY = $00090014;
  FSCTL_LOCK_VOLUME = $00090018;
  FSCTL_UNLOCK_VOLUME = $0009001C;
  FSCTL_DISMOUNT_VOLUME = $00090020;
  FSCTL_IS_VOLUME_MOUNTED = $00090028;
  FSCTL_GET_COMPRESSION = $0009003C;
  FSCTL_SET_COMPRESSION = $0009C040;
  FSCTL_OPLOCK_BREAK_ACK_NO_2 = $00090050;
  FSCTL_REQUEST_FILTER_OPLOCK = $0009005C;
  FSCTL_GET_NTFS_FILE_RECORD = $00090068;
  FSCTL_GET_RETRIEVAL_POINTERS = $00090073;
  FSCTL_SET_OBJECT_ID = $00090098;
  FSCTL_GET_OBJECT_ID = $0009009C;
  FSCTL_DELETE_OBJECT_ID = $000900A0;
  FSCTL_SET_REPARSE_POINT = $000900A4;
  FSCTL_GET_REPARSE_POINT = $000900A8;
  FSCTL_DELETE_REPARSE_POINT = $000900AC;
  FSCTL_SET_OBJECT_ID_EXTENDED = $000900BC;
  FSCTL_CREATE_OR_GET_OBJECT_ID = $000900C0;
  FSCTL_SET_SPARSE = $000900C4;
  FSCTL_SET_ZERO_DATA = $000980C8;
  FSCTL_EXTEND_VOLUME = $000900F0;
  FSCTL_SET_ZERO_ON_DEALLOCATION = $00090194;
  FSCTL_SD_GLOBAL_CHANGE = $000901F4;
  FSCTL_REQUEST_OPLOCK = $00090240;
  FSCTL_QUERY_FILE_LAYOUT = $00090277;
  FSCTL_SET_EXTERNAL_BACKING = $0009030C;
  FSCTL_GET_EXTERNAL_BACKING = $00090310;
  FSCTL_CHECK_FOR_SECTION = $00090348;
  FSCTL_SET_REPARSE_POINT_EX = $09040C;
  FSCTL_PIPE_DISCONNECT = $110004;
  FSCTL_PIPE_LISTEN = $110008;
  FSCTL_PIPE_PEEK = $11400C;
  FSCTL_PIPE_WAIT = $110018;
  FSCTL_PIPE_IMPERSONATE = $11001C;
  FSCTL_PIPE_DISABLE_IMPERSONATE = $110044;
  FSCTL_PIPE_CREATE_SYMLINK = $11004C;
  FSCTL_PIPE_DELETE_SYMLINK = $110050;

  { Filter manager }

type
  // PHNT::ntioapi.h
  [SDKName('FLT_CONNECT_CONTEXT')]
  TFltConnectContext = record
    PortName: PNtUnicodeString;
    PortName64: PNtUnicodeString64;
    [NumberOfBytes] SizeOfContext: Word;
    [Reserved] Padding: array [0..5] of Byte;
    Context: TPlaceholder;
  end;
  PFltConnectContext = ^TFltConnectContext;

  // PHNT::ntioapi.h
  {$SCOPEDENUMS ON}
  [NamingStyle(nsSnakeCase, 'FLT_CTL'), Range(1)]
  TFltCtlFunction = (
    [Reserved] FLT_CTL_UNKNOWN = 0,
    FLT_CTL_LOAD = 1,            // in: TFltLoadParameters // requires SeDebugPrivilege
    FLT_CTL_UNLOAD = 2,          // in: TFltLoadParameters // requires SeDebugPrivilege
    FLT_CTL_LINK_HANDLE = 3,     // in: TFltLink
    FLT_CTL_ATTACH = 4,          // in: TFltAttach
    FLT_CTL_DETACH = 5,         // in: TFltInstanceParameters
    FLT_CTL_SEND_MESSAGE = 6,    // in, out: port-specific
    FLT_CTL_GET_MESSAGE = 7,     // out: port-specific
    FLT_CTL_REPLY_MESSAGE = 8,   // in: port-specific
    FLT_CTL_FIND_FIRST = 9,      // in: filter/instance/volume info class; out: buffer
    FLT_CTL_FIND_NEXT = 10,      // in: filter/instance/volume info class; out: buffer
    FLT_CTL_GET_INFORMATION = 11 // in: filter/instance info class; out: buffer
  );
  {$SCOPEDENUMS OFF}

const
  // IOCTL values
  FLT_CTL_LOAD = $88004;
  FLT_CTL_UNLOAD = $88008;
  FLT_CTL_LINK_HANDLE = $8400C;
  FLT_CTL_ATTACH = $88010;
  FLT_CTL_DETACH = $88014;
  FLT_CTL_SEND_MESSAGE = $8801B;
  FLT_CTL_GET_MESSAGE = $8401F;
  FLT_CTL_REPLY_MESSAGE = $88023;
  FLT_CTL_FIND_FIRST = $84024;
  FLT_CTL_FIND_NEXT = $84028;
  FLT_CTL_GET_INFORMATION = $8402C;

type
  // PHNT::ntioapi.h - CTLs 1, 2
  [SDKName('FLT_LOAD_PARAMETERS')]
  TFltLoadParameters = record
    [NumberOfBytes] FilterNameSize: Word;
    FilterName: TAnysizeArray<WideChar>;
  end;
  PFltLoadParameters = ^TFltLoadParameters;

  // PHNT::ntioapi.h
  [SDKName('FLT_LINK_TYPE')]
  [NamingStyle(nsSnakeCase)]
  TFltLinkType = (
    FILTER = 0,               // link: TFltFilterParameters; first/next: instance
    FILTER_INSTANCE = 1,      // link: TFltInstanceParameters
    FILTER_VOLUME = 2,        // link: TFltVolumeParameters; first/next: instance
    FILTER_MANAGER = 3,       // link: nothing; first/next: filter
    FILTER_MANAGER_VOLUME = 4 // link: nothing; first/next: volume
  );

  // PHNT::ntioapi.h - CTL 3
  [SDKName('FLT_LINK')]
  TFltLink = record
    LinkType: TFltLinkType;
    ParametersOffset: Cardinal;
  end;
  PFltLink = ^TFltLink;

  // PHNT::ntioapi.h
  [SDKName('FLT_FILTER_PARAMETERS')]
  TFltFilterParameters = record
    FilterNameSize: Word;
    FilterNameOffset: Word; // to WideChar[] from this struct
  end;
  PFltFilterParameters = ^TFltFilterParameters;

  // PHNT::ntioapi.h
  [SDKName('FLT_VOLUME_PARAMETERS')]
  TFltVolumeParameters = record
    VolumeNameSize: Word;
    VolumeNameOffset: Word; // to WideChar[] from this struct
  end;
  PFltVolumeParameters = ^TFltVolumeParameters;

  // PHNT::ntioapi.h
  [SDKName('ATTACH_TYPE')]
  [NamingStyle(nsSnakeCase)]
  TAttachType = (
    AltitudeBased = 0,
    InstanceNameBased = 1
  );

  // PHNT::ntioapi.h - CTLs 4
  [SDKName('FLT_ATTACH')]
  TFltAttach = record
    [NumberOfBytes] FilterNameSize: Word;
    FilterNameOffset: Word;   // to WideChar[] from this struct
    [NumberOfBytes] VolumeNameSize: Word;
    VolumeNameOffset: Word;   // to WideChar[] from this struct
    AttachType: TAttachType;
    [NumberOfBytes] InstanceNameSize: Word;
    InstanceNameOffset: Word; // to WideChar[] from this struct
    [NumberOfBytes] AltitudeSize: Word;
    AltitudeOffset: Word;     // to WideChar[] from this struct
  end;
  PFltAttach = ^TFltAttach;

  // PHNT::ntioapi.h - CTL 5
  [SDKName('FLT_INSTANCE_PARAMETERS')]
  TFltInstanceParameters = record
    [NumberOfBytes] FilterNameSize: Word;
    FilterNameOffset: Word;   // to WideChar[] from this struct
    [NumberOfBytes] VolumeNameSize: Word;
    VolumeNameOffset: Word;   // to WideChar[] from this struct
    [NumberOfBytes] InstanceNameSize: Word;
    InstanceNameOffset: Word; // to WideChar[] from this struct
  end;
  PFltInstanceParameters = ^TFltInstanceParameters;

  // SDK::fltUserStructures.h
  [SDKName('FILTER_INFORMATION_CLASS')]
  [NamingStyle(nsCamelCase, 'Filter')]
  TFilterInformationClass = (
    FilterFullInformation = 0,             // q: TFilterFullInformation
    FilterAggregateBasicInformation = 1,   // q: TFilterAggregateBasicInformation
    FilterAggregateStandardInformation = 2 // q: TFilterAggregateStandardInformation
  );

  // SDK::fltUserStructures.h - filter info class 0
  [SDKName('FILTER_FULL_INFORMATION')]
  TFilterFullInformation = record
    NextEntryOffset: Cardinal;
    FrameID: Cardinal;
    NumberOfInstances: Cardinal;
    [NumberOfBytes] FilterNameLength: Word;
    FilterNameBuffer: TAnysizeArray<WideChar>;
  end;
  PFilterFullInformation = ^TFilterFullInformation;

  // SDK::fltUserStructures.h (renamed)
  [NamingStyle(nsSnakeCase, 'FLTFL_IS')]
  TFltFilterType = (
    FLTFL_IS_UNKWNOWN = 0,
    FLTFL_IS_MINIFILTER = 1,
    FLTFL_IS_LEGACY_FILTER = 2
  );

  // SDK::fltUserStructures.h - filter info class 1
  [SDKName('FILTER_AGGREGATE_BASIC_INFORMATION')]
  TFilterAggregateBasicInformation = record
    NextEntryOffset: Cardinal;
  case Flags: TFltFilterType of
    FLTFL_IS_MINIFILTER: (
      FrameID: Cardinal;
      NumberOfInstances: Cardinal;
      [NumberOfBytes] FilterNameLength: Word;
      FilterNameBufferOffset: Word;     // to WideChar[] from this struct
      [NumberOfBytes] FilterAltitudeLength: Word;
      FilterAltitudeBufferOffset: Word; // to WideChar[] from this struct
    );
    FLTFL_IS_LEGACY_FILTER: (
      [NumberOfBytes] LegacyFilterNameLength: Word;
      LegacyFilterNameBufferOffset: Word;
    );
  end;
  PFilterAggregateBasicInformation = ^TFilterAggregateBasicInformation;

  // SDK::fltUserStructures.h - filter info class 2
  [SDKName('FILTER_AGGREGATE_STANDARD_INFORMATION')]
  TFilterAggregateStandardInformation = record
    NextEntryOffset: Cardinal;
  case Flags: TFltFilterType of
    FLTFL_IS_MINIFILTER: (
      [Hex] MinifilterFlags: Cardinal;
      FrameID: Cardinal;
      NumberOfInstances: Cardinal;
      [NumberOfBytes] FilterNameLength: Word;
      FilterNameBufferOffset: Word;     // to WideChar[] from this struct
      [NumberOfBytes] FilterAltitudeLength: Word;
      FilterAltitudeBufferOffset: Word; // to WideChar[] from this struct
    );
    FLTFL_IS_LEGACY_FILTER: (
      [Hex] LegacyFilterFlags: Cardinal;
      [NumberOfBytes] LegacyFilterNameLength: Word;
      LegacyFilterNameBufferOffset: Word;     // to WideChar[] from this struct
      [NumberOfBytes] LegacyFilterAltitudeLength: Word;
      LegacyFilterAltitudeBufferOffset: Word; // to WideChar[] from this struct
    );
  end;
  PFilterAggregateStandardInformation = ^TFilterAggregateStandardInformation;

  // SDK::fltUserStructures.h
  [SDKName('INSTANCE_INFORMATION_CLASS')]
  [NamingStyle(nsCamelCase, 'Instance')]
  TInstanceInformationClass = (
    InstanceBasicInformation = 0,            // q: TInstanceBasicInformation
    InstancePartialInformation = 1,          // q: TInstancePartialInformation
    InstanceFullInformation = 2,             // q: TInstanceFullInformation
    InstanceAggregateStandardInformation = 3 // q: TInstanceAggregateStandardInformation
  );

  // SDK::fltUserStructures.h - instance info class 0
  [SDKName('INSTANCE_BASIC_INFORMATION')]
  TInstanceBasicInformation = record
    NextEntryOffset: Cardinal;
    [NumberOfBytes] InstanceNameLength: Word;
    InstanceNameBufferOffset: Word; // to WideChar[] from this struct
  end;
  PInstanceBasicInformation = ^TInstanceBasicInformation;

  // SDK::fltUserStructures.h - instance info class 1
  [SDKName('INSTANCE_PARTIAL_INFORMATION')]
  TInstancePartialInformation = record
    NextEntryOffset: Cardinal;
    [NumberOfBytes] InstanceNameLength: Word;
    InstanceNameBufferOffset: Word; // to WideChar[] from this struct
    [NumberOfBytes] AltitudeLength: Word;
    AltitudeBufferOffset: Word;     // to WideChar[] from this struct
  end;
  PInstancePartialInformation = ^TInstancePartialInformation;

  // SDK::fltUserStructures.h - instance info class 2
  [SDKName('INSTANCE_FULL_INFORMATION')]
  TInstanceFullInformation = record
    NextEntryOffset: Cardinal;
    [NumberOfBytes] InstanceNameLength: Word;
    InstanceNameBufferOffset: Word; // to WideChar[] from this struct
    [NumberOfBytes] AltitudeLength: Word;
    AltitudeBufferOffset: Word;     // to WideChar[] from this struct
    [NumberOfBytes] VolumeNameLength: Word;
    VolumeNameBufferOffset: Word;   // to WideChar[] from this struct
    [NumberOfBytes] FilterNameLength: Word;
    FilterNameBufferOffset: Word;   // to WideChar[] from this struct
  end;
  PInstanceFullInformation = ^TInstanceFullInformation;

  // SDK::fltUserStructures.h
  [SDKName('FLT_FILESYSTEM_TYPE')]
  [NamingStyle(nsSnakeCase, 'FLT_FSTYPE')]
  TFltFilesystemType = (
    FLT_FSTYPE_UNKNOWN = 0,
    FLT_FSTYPE_RAW = 1,
    FLT_FSTYPE_NTFS = 2,
    FLT_FSTYPE_FAT = 3,
    FLT_FSTYPE_CDFS = 4,
    FLT_FSTYPE_UDFS = 5,
    FLT_FSTYPE_LANMAN = 6,
    FLT_FSTYPE_WEBDAV = 7,
    FLT_FSTYPE_RDPDR = 8,
    FLT_FSTYPE_NFS = 9,
    FLT_FSTYPE_MS_NETWARE = 10,
    FLT_FSTYPE_NETWARE = 11,
    FLT_FSTYPE_BSUDF = 12,
    FLT_FSTYPE_MUP = 13,
    FLT_FSTYPE_RSFX = 14,
    FLT_FSTYPE_ROXIO_UDF1 = 15,
    FLT_FSTYPE_ROXIO_UDF2 = 16,
    FLT_FSTYPE_ROXIO_UDF3 = 17,
    FLT_FSTYPE_TACIT = 18,
    FLT_FSTYPE_FS_REC = 19,
    FLT_FSTYPE_INCD = 20,
    FLT_FSTYPE_INCD_FAT = 21,
    FLT_FSTYPE_EXFAT = 22,
    FLT_FSTYPE_PSFS = 23,
    FLT_FSTYPE_GPFS = 24,
    FLT_FSTYPE_NPFS = 25,
    FLT_FSTYPE_MSFS = 26,
    FLT_FSTYPE_CSVFS = 27,
    FLT_FSTYPE_REFS = 28,
    FLT_FSTYPE_OPENAFS = 29,
    FLT_FSTYPE_CIMFS = 30
  );

  [FlagName(FLTFL_DETACHED_VOLUME, 'Detached')]
  TFilterVolumeFlags = type Cardinal;

  [MinOSVersion(OsWin8)]
  [FlagName(SUPPORTED_FS_FEATURES_OFFLOAD_READ, 'Offload Read')]
  [FlagName(SUPPORTED_FS_FEATURES_OFFLOAD_WRITE, 'Offload Write')]
  [FlagName(SUPPORTED_FS_FEATURES_QUERY_OPEN, 'Query Open')]
  [FlagName(SUPPORTED_FS_FEATURES_BYPASS_IO, 'Bypass I/O')]
  TSupportedFsFeatures = type Cardinal;

  // SDK::fltUserStructures.h - instance info class 3
  [SDKName('INSTANCE_AGGREGATE_STANDARD_INFORMATION')]
  TInstanceAggregateStandardInformation = record
    NextEntryOffset: Cardinal;
  case Flags: TFltFilterType of
    FLTFL_IS_MINIFILTER: (
      VolumeFlags: TFilterVolumeFlags;
      FrameID: Cardinal;
      VolumeFileSystemType: TFltFilesystemType;
      [NumberOfBytes] InstanceNameLength: Word;
      InstanceNameBufferOffset: Word; // to WideChar[] from this struct
      [NumberOfBytes] AltitudeLength: Word;
      AltitudeBufferOffset: Word;     // to WideChar[] from this struct
      [NumberOfBytes] VolumeNameLength: Word;
      VolumeNameBufferOffset: Word;   // to WideChar[] from this struct
      [NumberOfBytes] FilterNameLength: Word;
      FilterNameBufferOffset: Word;   // to WideChar[] from this struct
      [MinOSVersion(OsWin8)] SupportedFeatures: TSupportedFsFeatures;
    );
    FLTFL_IS_LEGACY_FILTER: (
      LegacyVolumeFlags: TFilterVolumeFlags;
      [NumberOfBytes] LegacyAltitudeLength: Word;
      LegacyAltitudeBufferOffset: Word;   // to WideChar[] from this struct
      [NumberOfBytes] LegacyVolumeNameLength: Word;
      LegacyVolumeNameBufferOffset: Word; // to WideChar[] from this struct
      [NumberOfBytes] LegacyFilterNameLength: Word;
      LegacyFilterNameBufferOffset: Word; // to WideChar[] from this struct
      [MinOSVersion(OsWin8)] LegacySupportedFeatures: TSupportedFsFeatures;
    );
  end;
  PInstanceAggregateStandardInformation = ^TInstanceAggregateStandardInformation;

  // SDK::fltUserStructures.h
  [SDKName('FILTER_VOLUME_INFORMATION_CLASS')]
  [NamingStyle(nsCamelCase, 'FilterVolume')]
  TFilterVolumeInformationClass = (
    FilterVolumeBasicInformation = 0,   // q: TFilterVolumeBasicInformation
    FilterVolumeStandardInformation = 1 // q: TFilterVolumeStandardInformation
  );

  // SDK::fltUserStructures.h - volume info class 0
  [SDKName('FILTER_VOLUME_BASIC_INFORMATION')]
  TFilterVolumeBasicInformation = record
   [NumberOfBytes] FilterVolumeNameLength: Word;
   FilterVolumeName: TAnysizeArray<WideChar>;
  end;
  PFilterVolumeBasicInformation = ^TFilterVolumeBasicInformation;

  // SDK::fltUserStructures.h - volume info class 1
  [SDKName('FILTER_VOLUME_STANDARD_INFORMATION')]
  TFilterVolumeStandardInformation = record
    NextEntryOffset: Cardinal;
    Flags: TFilterVolumeFlags;
    FrameID: Cardinal;
    FileSystemType: TFltFilesystemType;
    [NumberOfBytes] FilterVolumeNameLength: Word;
    FilterVolumeName: TAnysizeArray<WideChar>;
  end;
  PFilterVolumeStandardInformation = ^TFilterVolumeStandardInformation;

  { Mupltiple UNC Provider }

type
  {$SCOPEDENUMS ON}
  [NamingStyle(nsSnakeCase, 'FSCTL'), Range(10)]
  TMupFsctlFunction = (
    Unused0, Unused1, Unused2, Unused3, Unused4,
    Unused5, Unused6, Unused7, Unused8, Unused9,
    FSCTL_MUP_FLUSH_UNC_HARDENING_CONFIGURATION = 10, // in: WideChar[]
    FSCTL_MUP_GET_UNC_CACHE_INFO = 11,                // out: TMupFsctlUncCacheInformation
    FSCTL_MUP_GET_UNC_PROVIDER_LIST = 12,             // out: TMupFsctlUncProviderInformation
    FSCTL_MUP_GET_SURROGATE_PROVIDER_LIST = 13,       // out: TMupFsctlSurrogateProviderInformation
    FSCTL_MUP_GET_UNC_HARDENING_CONFIGURATION = 14,   // out: TMupFsctlUncHardeningPrefixTableEntry
    FSCTL_MUP_GET_UNC_HARDENING_CONFIGURATION_FOR_PATH = 15 // in: TMupFsctlQueryUncHardeningConfigurationIn; out: TMupFsctlQueryUncHardeningConfigurationOut
  );
  {$SCOPEDENUMS OFF}

const
  // PHNT::ntioapi.h
  FSCTL_MUP_FLUSH_UNC_HARDENING_CONFIGURATION = $108028;
  FSCTL_MUP_GET_UNC_CACHE_INFO = $10002C;
  FSCTL_MUP_GET_UNC_PROVIDER_LIST = $100030;
  FSCTL_MUP_GET_SURROGATE_PROVIDER_LIST = $100034;
  FSCTL_MUP_GET_UNC_HARDENING_CONFIGURATION = $100038;
  FSCTL_MUP_GET_UNC_HARDENING_CONFIGURATION_FOR_PATH = $10003C;

  // bits extracted from MUP_FSCTL_UNC_HARDENING_PREFIX_TABLE_ENTRY
  MUP_FSCTL_UNC_HARDENING_REQUIRES_MUTUAL_AUTH = $1;
  MUP_FSCTL_UNC_HARDENING_REQUIRES_INTEGRITY = $2;
  MUP_FSCTL_UNC_HARDENING_REQUIRES_PRIVACY = $4;

type
  // PHNT::ntioapi.h
  [SDKName('MUP_FSCTL_UNC_CACHE_ENTRY')]
  TMupFsctlUncCacheEntry = record
    [RecordSize] TotalLength: Cardinal;
    [Offset] UncNameOffset: Cardinal;
    [NumberOfBytes] UncNameLength: Word;
    [Offset] ProviderNameOffset: Cardinal;
    [NumberOfBytes] ProviderNameLength: Word;
    [Offset] SurrogateNameOffset: Cardinal;
    [NumberOfBytes] SurrogateNameLength: Word;
    [Hex] Flags: Word;
    ProviderPriority: Cardinal;
    EntryTtl: Cardinal;
    Strings: TAnysizeArray<WideChar>;
  end;
  PMupFsctlUncCacheEntry = ^TMupFsctlUncCacheEntry;

  // PHNT::ntioapi.h - MUP FSCTL 11
  [SDKName('MUP_FSCTL_UNC_CACHE_INFORMATION')]
  TMupFsctlUncCacheInformation = record
    MaxCacheSize: Cardinal;
    CurrentCacheSize: Cardinal;
    EntryTimeout: Cardinal;
    [NumberOfElements] TotalEntries: Cardinal;
    CacheEntry: TPlaceholder<TMupFsctlUncCacheEntry>;
  end;
  PMupFsctlUncCacheInformation = ^TMupFsctlUncCacheInformation;

  // PHNT::ntioapi.h
  [SDKName('MUP_FSCTL_UNC_PROVIDER_ENTRY')]
  TMupFsctlUncProviderEntry = record
    [RecordSize] TotalLength: Cardinal;
    ReferenceCount: Integer;
    ProviderPriority: Cardinal;
    ProviderState: Cardinal;
    ProviderId: Cardinal;
    [NumberOfBytes] ProviderNameLength: Word;
    ProviderName: TAnysizeArray<WideChar>;
  end;
  PMupFsctlUncProviderEntry = ^TMupFsctlUncProviderEntry;

  // PHNT::ntioapi.h - MUP FSCTL 12
  [SDKName('MUP_FSCTL_UNC_PROVIDER_INFORMATION')]
  TMupFsctlUncProviderInformation = record
    [NumberOfElements] TotalEntries: Cardinal;
    ProviderEntry: TPlaceholder<TMupFsctlUncProviderEntry>;
  end;
  PMupFsctlUncProviderInformation = ^TMupFsctlUncProviderInformation;

  // PHNT::ntioapi.h
  [SDKName('MUP_FSCTL_SURROGATE_PROVIDER_ENTRY')]
  TMupFsctlSurrogateProviderEntry = record
    [RecordSize] TotalLength: Cardinal;
    ReferenceCount: Integer;
    SurrogateType: Cardinal;
    SurrogateState: Cardinal;
    SurrogatePriority: Cardinal;
    [NumberOfBytes] SurrogateNameLength: Word;
    SurrogateName: TAnysizeArray<WideChar>;
  end;
  PMupFsctlSurrogateProviderEntry = ^TMupFsctlSurrogateProviderEntry;

  // PHNT::ntioapi.h - MUP FSCTL 13
  [SDKName('MUP_FSCTL_SURROGATE_PROVIDER_INFORMATION')]
  TMupFsctlSurrogateProviderInformation = record
    [NumberOfElements] TotalEntries: Cardinal;
    SurrogateEntry: TPlaceholder<TMupFsctlSurrogateProviderEntry>;
  end;
  PMupFsctlSurrogateProviderInformation = ^TMupFsctlSurrogateProviderInformation;

  [FlagName(MUP_FSCTL_UNC_HARDENING_REQUIRES_MUTUAL_AUTH, 'Requires Mutual Auth')]
  [FlagName(MUP_FSCTL_UNC_HARDENING_REQUIRES_INTEGRITY, 'Requires Integrity')]
  [FlagName(MUP_FSCTL_UNC_HARDENING_REQUIRES_PRIVACY, 'Requires Privacy')]
  TMupHardeningCapabilities = type Cardinal;

  // PHNT::ntioapi.h - MUP FSCTL 14
  [SDKName('MUP_FSCTL_UNC_HARDENING_PREFIX_TABLE_ENTRY')]
  TMupFsctlUncHardeningPrefixTableEntry = record
    [Offset] NextOffset: Cardinal;
    [Offset] PrefixNameOffset: Cardinal;
    [NumberOfBytes] PrefixNameCbLength: Word;
    RequiredHardeningCapabilities: TMupHardeningCapabilities;
    OpenCount: UInt64;
  end;
  PMupFsctlUncHardeningPrefixTableEntry = ^TMupFsctlUncHardeningPrefixTableEntry;

  // PHNT::ntioapi.h - MUP FSCTL 15 (input)
  [SDKName('MUP_FSCTL_QUERY_UNC_HARDENING_CONFIGURATION_IN')]
  TMupFsctlQueryUncHardeningConfigurationIn = record
    [RecordSize] Size: Cardinal;
    [Offset] UncPathOffset: Cardinal;
    [NumberOfBytes] UncPathCbLength: Word;
  end;
  PMupFsctlQueryUncHardeningConfigurationIn = ^TMupFsctlQueryUncHardeningConfigurationIn;

  // PHNT::ntioapi.h - MUP FSCTL 15 (input)
  [SDKName('MUP_FSCTL_QUERY_UNC_HARDENING_CONFIGURATION_OUT')]
  TMupFsctlQueryUncHardeningConfigurationOut = record
    [RecordSize] Size: Cardinal;
    RequiredHardeningCapabilities: TMupHardeningCapabilities;
  end;
  PMupFsctlQueryUncHardeningConfigurationOut = ^TMupFsctlQueryUncHardeningConfigurationOut;

// WDK::ntifs.h
function NtQueryVolumeInformationFile(
  [in] FileHandle: THandle;
  [out] out IoStatusBlock: TIoStatusBlock;
  [out, WritesTo] FsInformation: Pointer;
  [in, NumberOfBytes] Length: Cardinal;
  [in] FsInformationClass: TFsInfoClass
): NTSTATUS; stdcall; external ntdll;

// WDK::ntifs.h
function NtSetVolumeInformationFile(
  [in] FileHandle: THandle;
  [out] out IoStatusBlock: TIoStatusBlock;
  [in, ReadsFrom] FsInformation: Pointer;
  [in, NumberOfBytes] Length: Cardinal;
  [in] FsInformationClass: TFsInfoClass
): NTSTATUS; stdcall; external ntdll;

// WDK::ntifs.h
function NtFsControlFile(
  [in] FileHandle: THandle;
  [in, opt] Event: THandle;
  [in, opt] ApcRoutine: TIoApcRoutine;
  [in, opt] ApcContext: Pointer;
  [out] IoStatusBlock: PIoStatusBlock;
  [in] FsControlCode: Cardinal;
  [in, ReadsFrom] InputBuffer: Pointer;
  [in, NumberOfBytes] InputBufferLength: Cardinal;
  [out, WritesTo] OutputBuffer: Pointer;
  [in, NumberOfBytes] OutputBufferLength: Cardinal
): NTSTATUS; stdcall; external ntdll;

{ Helper functions / macros }

// WDK::ntifs.h
function CTL_FS_CODE(
  [in] Func: TFsCtlFunction;
  [in] Method: TIoControlMethod;
  [in] Access: Cardinal
): Cardinal;

function CTL_PIPE_CODE(
  [in] Func: TFsCtlPipeFunction;
  [in] Method: TIoControlMethod;
  [in] Access: Cardinal
): Cardinal;

function DEVICE_TYPE_FSCTL(
  [in] FsControlCode: Cardinal
): TDeviceType;

function FUNCTION_FROM_FS_FSCTL(
  [in] FsControlCode: Cardinal
): TFsCtlFunction;

function FUNCTION_FROM_IOCTL(
  [in] FsControlCode: Cardinal
): Word;

function FUNCTION_FROM_PIPE_FSCTL(
  [in] FsControlCode: Cardinal
): TFsCtlPipeFunction;

function FUNCTION_FROM_MUP_FSCTL(
  [in] FsControlCode: Cardinal
): TMupFsctlFunction;

function ExpectedFsQueryAccess(
  [in] InfoClass: TFsInfoClass
): TFileAccessMask;

function ExpectedFsSetAccess(
  [in] InfoClass: TFsInfoClass
): TFileAccessMask;

implementation

{$BOOLEVAL OFF}
{$IFOPT R+}{$DEFINE R+}{$ENDIF}
{$IFOPT Q+}{$DEFINE Q+}{$ENDIF}

function CTL_FS_CODE;
begin
  Result := (Cardinal(TDeviceType.FILE_DEVICE_FILE_SYSTEM) shl 16) or
    (Access shl 14) or (Cardinal(Func) shl 2) or Cardinal(Method);
end;

function CTL_PIPE_CODE;
begin
  Result := (Cardinal(TDeviceType.FILE_DEVICE_NAMED_PIPE) shl 16) or
    (Access shl 14) or (Cardinal(Func) shl 2) or Cardinal(Method);
end;

function DEVICE_TYPE_FSCTL;
begin
  Result := TDeviceType((FsControlCode shr 16) and $FFFF);
end;

function FUNCTION_FROM_FS_FSCTL;
begin
  Result := TFsCtlFunction((FsControlCode shr 2) and $FFF);
end;

function FUNCTION_FROM_PIPE_FSCTL;
begin
  Result := TFsCtlPipeFunction((FsControlCode shr 2) and $FFF);
end;

function FUNCTION_FROM_MUP_FSCTL;
begin
  Result := TMupFsctlFunction((FsControlCode shr 2) and $FFF);
end;

function FUNCTION_FROM_IOCTL;
begin
  Result := (FsControlCode shr 2) and $FFF;
end;

function ExpectedFsQueryAccess;
begin
  case InfoClass of
    FileFsControlInformation:
      Result := FILE_READ_DATA;

    FileFsVolumeFlagsInformation:
      Result := FILE_READ_ATTRIBUTES;

  else
    Result := 0; // Either no access check or not supported for query
  end;
end;

function ExpectedFsSetAccess;
begin
  case InfoClass of
    FileFsLabelInformation, FileFsControlInformation, FileFsObjectIdInformation:
      Result := FILE_WRITE_DATA;

    FileFsVolumeFlagsInformation:
      Result := FILE_WRITE_ATTRIBUTES;

  else
    Result := 0; // Either no access check or not supported for setting
  end;
end;

end.
