import { getTrialDetails } from 'services/api';
import { V1TrialLogsResponse } from 'services/api-ts-sdk';
import { detApi } from 'services/apiConfig';
import { readStream } from 'services/utils';
import { BrandingType } from 'stores/determinedInfo';
import { routeToExternalUrl } from 'utils/routes';

/*
 * In mobile view the definition of viewport height varies between
 * whether it should include the chrome's vertical space or not.
 * This function dynamically calculates the available height on the
 * browser view minus the chrome height.
 * https://css-tricks.com/the-trick-to-viewport-units-on-mobile/
 */
export const correctViewportHeight = (): void => {
  const vh = window.innerHeight * 0.01;
  document.documentElement.style.setProperty('--vh', `${vh}px`);
};

const downloadBlob = (filename: string, data: Blob): void => {
  const url = window.URL.createObjectURL(data);
  const element = document.createElement('a');
  element.setAttribute('download', filename);
  element.style.display = 'none';
  element.href = url;
  document.body.appendChild(element);
  element.click();
  window.URL.revokeObjectURL(url);
  document.body.removeChild(element);
};

// Different JS engines impose different maximum string lenghts thus for downloading
// large text we split it into different parts and then stich them together as a Blob
export const downloadText = (filename: string, parts: BlobPart[]): void => {
  const data = new Blob(parts, { type: 'text/plain' });
  downloadBlob(filename, data);
};

export const downloadTrialLogs = async (trialId: number): Promise<void> => {
  // String concatnation is fater than array.join https://jsben.ch/OJ3vo
  // characters are utf16 encoded. v8 has different internal representations.
  const MAX_PART_SIZE = 128 * Math.pow(2, 20); // 128m * CHAR_SIZE
  const parts: BlobPart[] = [];
  let downloadStringBuffer = '';
  await readStream<V1TrialLogsResponse>(detApi.StreamingExperiments.trialLogs(trialId), (ev) => {
    downloadStringBuffer += ev.message;
    if (downloadStringBuffer.length > MAX_PART_SIZE) {
      parts.push(downloadStringBuffer);
      downloadStringBuffer = '';
    }
  });
  if (downloadStringBuffer !== '') parts.push(downloadStringBuffer);
  const trial = await getTrialDetails({ id: trialId });
  downloadText(`experiment_${trial.experimentId}_trial_${trialId}_logs.txt`, parts);
};

const generateLogStringBuffer = (count: number, avgLength: number): string => {
  const msg = new Array(avgLength).fill('a').join('') + '\n';
  let stringBuffer = '';
  for (let i = 0; i < count; i++) {
    stringBuffer += i + msg;
  }
  return stringBuffer;
};

export const getCookie = (name: string): string | null => {
  const regex = new RegExp(`(?:(?:^|.*;\\s*)${name}\\s*\\=\\s*([^;]*).*$)|^.*$`);
  const value = document.cookie.replace(regex, '$1');
  return value ? value : null;
};

export const setCookie = (name: string, value: string): void => {
  const date = new Date();
  date.setTime(date.getTime() + 7 * 24 * 60 * 60 * 1000);
  document.cookie = name + '=' + value + '; expires=' + date.toUTCString() + '; path=/';
};

/*
 * The method of cache busting here is to send a query string as most
 * modern browsers treat different URLs as different files, causing a
 * request of a fresh copy. The previous method of using `location.reload`
 * with a `forceReload` boolean has been deprecated and not reliable.
 */
export const refreshPage = (): void => {
  const now = Date.now();
  const url = new URL(window.location.href);
  const params = url.searchParams;

  params.set('ts', now.toString());

  routeToExternalUrl(url.toString());
};

export const simulateLogsDownload = (numCharacters: number): number => {
  const start = Date.now();
  const MAX_PART_SIZE = 128 * Math.pow(2, 20); // 128m * CHAR_SIZE
  const chunkCount = Math.ceil(numCharacters / MAX_PART_SIZE);
  const parts = new Array(chunkCount)
    .fill(0)
    .map(() => generateLogStringBuffer(Math.pow(2, 20), 128));
  downloadText('simulated-logs.txt', parts);
  return Date.now() - start;
};

const updateFavicon = (iconPath: string): void => {
  const linkEl: HTMLLinkElement | null = document.querySelector("link[rel*='shortcut icon']");
  if (!linkEl) return;
  linkEl.type = 'image/png';
  linkEl.href = iconPath;
};

export const updateFaviconType = (active: boolean, branding: BrandingType): void => {
  const suffixDev = process.env.IS_DEV ? '-dev' : '';
  const suffixActive = active ? '-active' : '';
  updateFavicon(`${process.env.PUBLIC_URL}/${branding}/favicon${suffixDev}${suffixActive}.png`);
};
