/**
 * Determined API (Beta)
 * Determined helps deep learning teams train models more quickly, easily share GPU resources, and effectively collaborate. Determined allows deep learning engineers to focus on building and training models at scale, without needing to worry about DevOps or writing custom code for common tasks like fault tolerance or experiment tracking.  You can think of Determined as a platform that bridges the gap between tools like TensorFlow and PyTorch --- which work great for a single researcher with a single GPU --- to the challenges that arise when doing deep learning at scale, as teams, clusters, and data sets all increase in size.
 *
 * OpenAPI spec version: 0.1
 * Contact: community@determined.ai
 *
 * NOTE: Do not edit the class manually.
 */

import { DateString } from "ioTypes";
import { Configuration } from "./configuration";

type ValueOf<T> = T[keyof T];
const BASE_PATH = "http://localhost".replace(/\/+$/, "");

const convert = (v: unknown): string => {
    switch (typeof v) {
        case 'string':
        case 'boolean': {
            return v.toString();
        }
        case 'bigint': {
            return '' + v
        }
        case 'number': {
            if (Number.isFinite(v))  {
                return encodeURIComponent(v);
            }
            return '';
        }
        default: {
            return '';
        }
    }
}

const objToSearchParams = (obj: {}, searchParams: URLSearchParams) => {
    Object.entries(obj).forEach(([key, value]) => {
        if (Array.isArray(value) && value.length > 0) {
            searchParams.set(key, convert(value[0]))
            value.slice(1).forEach((subValue) => searchParams.append(key, convert(subValue)))
        } else if (!Array.isArray(value)) {
            searchParams.set(key, convert(value))
        }
    });
};

/**
 *
 * @export
 */
export const COLLECTION_FORMATS = {
    csv: ",",
    ssv: " ",
    tsv: "\t",
    pipes: "|",
};

/**
 *
 * @export
 * @interface FetchAPI
 */
export interface FetchAPI {
    (url: string, init?: any): Promise<Response>;
}

/**
 *
 * @export
 * @interface FetchArgs
 */
export interface FetchArgs {
    url: string;
    options: any;
}

/**
 *
 * @export
 * @class BaseAPI
 */
export class BaseAPI {
    protected configuration: Configuration;

    constructor(configuration?: Configuration, protected basePath: string = BASE_PATH, protected fetch: FetchAPI = window.fetch) {
        if (configuration) {
            this.configuration = configuration;
            this.basePath = configuration.basePath || this.basePath;
        } else {
            this.configuration = new Configuration()
        }
    }
};

/**
 *
 * @export
 * @class RequiredError
 * @extends {Error}
 */
export class RequiredError extends Error {
    override name: "RequiredError"
    constructor(public field: string, msg?: string) {
        super(msg);
        this.name = "RequiredError"
    }
}


/**
 * Sorts options for checkpoints by the given field.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
 * @export
 * @enum {string}
 */
export const Checkpointv1SortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    UUID: 'SORT_BY_UUID',
    TRIALID: 'SORT_BY_TRIAL_ID',
    BATCHNUMBER: 'SORT_BY_BATCH_NUMBER',
    ENDTIME: 'SORT_BY_END_TIME',
    STATE: 'SORT_BY_STATE',
    SEARCHERMETRIC: 'SORT_BY_SEARCHER_METRIC',
} as const
export type Checkpointv1SortBy = ValueOf<typeof Checkpointv1SortBy>
/**
 * The current state of the checkpoint.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
 * @export
 * @enum {string}
 */
export const Checkpointv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    ACTIVE: 'STATE_ACTIVE',
    COMPLETED: 'STATE_COMPLETED',
    ERROR: 'STATE_ERROR',
    DELETED: 'STATE_DELETED',
    PARTIALLYDELETED: 'STATE_PARTIALLY_DELETED',
} as const
export type Checkpointv1State = ValueOf<typeof Checkpointv1State>
/**
 * The current state of the container.   - STATE_UNSPECIFIED: The container state is unknown.  - STATE_ASSIGNED: The container has been assigned to an agent but has not started yet.  - STATE_PULLING: The container's base image is being pulled from the Docker registry.  - STATE_STARTING: The image has been built and the container is being started, but the service in the container is not ready yet.  - STATE_RUNNING: The service in the container is able to accept requests.  - STATE_TERMINATED: The container has completely exited or the container has been aborted prior to getting assigned.
 * @export
 * @enum {string}
 */
export const Containerv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    ASSIGNED: 'STATE_ASSIGNED',
    PULLING: 'STATE_PULLING',
    STARTING: 'STATE_STARTING',
    RUNNING: 'STATE_RUNNING',
    TERMINATED: 'STATE_TERMINATED',
} as const
export type Containerv1State = ValueOf<typeof Containerv1State>
/**
 * The type of the Device.   - TYPE_UNSPECIFIED: An unspecified device type.  - TYPE_CPU: A CPU device.  - TYPE_CUDA: CUDA device.  - TYPE_ROCM: ROCM.
 * @export
 * @enum {string}
 */
export const Devicev1Type = {
    UNSPECIFIED: 'TYPE_UNSPECIFIED',
    CPU: 'TYPE_CPU',
    CUDA: 'TYPE_CUDA',
    ROCM: 'TYPE_ROCM',
} as const
export type Devicev1Type = ValueOf<typeof Devicev1Type>
/**
 * The current state of the experiment.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
 * @export
 * @enum {string}
 */
export const Experimentv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    ACTIVE: 'STATE_ACTIVE',
    PAUSED: 'STATE_PAUSED',
    STOPPINGCOMPLETED: 'STATE_STOPPING_COMPLETED',
    STOPPINGCANCELED: 'STATE_STOPPING_CANCELED',
    STOPPINGERROR: 'STATE_STOPPING_ERROR',
    COMPLETED: 'STATE_COMPLETED',
    CANCELED: 'STATE_CANCELED',
    ERROR: 'STATE_ERROR',
    DELETED: 'STATE_DELETED',
    DELETING: 'STATE_DELETING',
    DELETEFAILED: 'STATE_DELETE_FAILED',
    STOPPINGKILLED: 'STATE_STOPPING_KILLED',
    QUEUED: 'STATE_QUEUED',
    PULLING: 'STATE_PULLING',
    STARTING: 'STATE_STARTING',
    RUNNING: 'STATE_RUNNING',
} as const
export type Experimentv1State = ValueOf<typeof Experimentv1State>
/**
 * Different kinds of Determined Cloud offerings - PRODUCT_UNSPECIFIED: Not a Cloud Community offering  - PRODUCT_COMMUNITY: Determined Cloud, Community Edition
 * @export
 * @enum {string}
 */
export const GetMasterResponseProduct = {
    UNSPECIFIED: 'PRODUCT_UNSPECIFIED',
    COMMUNITY: 'PRODUCT_COMMUNITY',
} as const
export type GetMasterResponseProduct = ValueOf<typeof GetMasterResponseProduct>
/**
 * Filter workloads with training, validation, and checkpoint information.   - FILTER_OPTION_UNSPECIFIED: Any workload.  - FILTER_OPTION_CHECKPOINT: Only workloads with an associated checkpoint.  - FILTER_OPTION_VALIDATION: Only validation workloads.  - FILTER_OPTION_CHECKPOINT_OR_VALIDATION: Only validation workloads or ones with an associated checkpoint.
 * @export
 * @enum {string}
 */
export const GetTrialWorkloadsRequestFilterOption = {
    UNSPECIFIED: 'FILTER_OPTION_UNSPECIFIED',
    CHECKPOINT: 'FILTER_OPTION_CHECKPOINT',
    VALIDATION: 'FILTER_OPTION_VALIDATION',
    CHECKPOINTORVALIDATION: 'FILTER_OPTION_CHECKPOINT_OR_VALIDATION',
} as const
export type GetTrialWorkloadsRequestFilterOption = ValueOf<typeof GetTrialWorkloadsRequestFilterOption>
/**
 * 
 * @export
 * @interface HealthCheck
 */
export interface HealthCheck {
    /**
     * 
     * @type {HealthStatus}
     * @memberof HealthCheck
     */
    database?: HealthStatus;
    /**
     * 
     * @type {Array<ResourceManagerHealth>}
     * @memberof HealthCheck
     */
    resource_managers?: Array<ResourceManagerHealth>;
    /**
     * 
     * @type {HealthStatus}
     * @memberof HealthCheck
     */
    status?: HealthStatus;
}
/**
 * 
 * @export
 * @enum {string}
 */
export const HealthStatus = {
    up: 'up',
    down: 'down',
} as const
export type HealthStatus = ValueOf<typeof HealthStatus>
/**
 * Job state.   - STATE_UNSPECIFIED: Unspecified state.  - STATE_QUEUED: Job is queued and waiting to be schedlued.  - STATE_SCHEDULED: Job is scheduled.  - STATE_SCHEDULED_BACKFILLED: Job is scheduled as a backfill.
 * @export
 * @enum {string}
 */
export const Jobv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    QUEUED: 'STATE_QUEUED',
    SCHEDULED: 'STATE_SCHEDULED',
    SCHEDULEDBACKFILLED: 'STATE_SCHEDULED_BACKFILLED',
} as const
export type Jobv1State = ValueOf<typeof Jobv1State>
/**
 * Job type.   - TYPE_UNSPECIFIED: Unspecified state.  - TYPE_EXPERIMENT: Experiement Job.  - TYPE_NOTEBOOK: Jupyter Notebook Job.  - TYPE_TENSORBOARD: TensorBoard Job.  - TYPE_SHELL: Shell Job.  - TYPE_COMMAND: Command Job.  - TYPE_CHECKPOINT_GC: CheckpointGC Job.  - TYPE_EXTERNAL: External Job.  - TYPE_GENERIC: Generic Job.
 * @export
 * @enum {string}
 */
export const Jobv1Type = {
    UNSPECIFIED: 'TYPE_UNSPECIFIED',
    EXPERIMENT: 'TYPE_EXPERIMENT',
    NOTEBOOK: 'TYPE_NOTEBOOK',
    TENSORBOARD: 'TYPE_TENSORBOARD',
    SHELL: 'TYPE_SHELL',
    COMMAND: 'TYPE_COMMAND',
    CHECKPOINTGC: 'TYPE_CHECKPOINT_GC',
    EXTERNAL: 'TYPE_EXTERNAL',
    GENERIC: 'TYPE_GENERIC',
} as const
export type Jobv1Type = ValueOf<typeof Jobv1Type>
/**
 * Gets around not being able to do "Optional map<string, int64>". Not ideal but this API is marked internal for now.
 * @export
 * @interface PatchCheckpointOptionalResources
 */
export interface PatchCheckpointOptionalResources {
    /**
     * Resources.
     * @type {{ [key: string]: string; }}
     * @memberof PatchCheckpointOptionalResources
     */
    resources?: { [key: string]: string; };
}
/**
 * Nested object for checkpoint_storage field patch.
 * @export
 * @interface PatchExperimentPatchCheckpointStorage
 */
export interface PatchExperimentPatchCheckpointStorage {
    /**
     * Experiment config checkpoint_storage.save_experiment_best.
     * @type {number}
     * @memberof PatchExperimentPatchCheckpointStorage
     */
    saveExperimentBest?: number;
    /**
     * Experiment config checkpoint_storage.save_trial_best.
     * @type {number}
     * @memberof PatchExperimentPatchCheckpointStorage
     */
    saveTrialBest?: number;
    /**
     * Experiment config checkpoint_storage.save_trial_latest.
     * @type {number}
     * @memberof PatchExperimentPatchCheckpointStorage
     */
    saveTrialLatest?: number;
}
/**
 * Nested object for resources field patch.
 * @export
 * @interface PatchExperimentPatchResources
 */
export interface PatchExperimentPatchResources {
    /**
     * Experiment config resources.max_slots.
     * @type {number}
     * @memberof PatchExperimentPatchResources
     */
    maxSlots?: number;
    /**
     * Experiment config resources.weight.
     * @type {number}
     * @memberof PatchExperimentPatchResources
     */
    weight?: number;
    /**
     * Experiment config resources.priority.
     * @type {number}
     * @memberof PatchExperimentPatchResources
     */
    priority?: number;
}
/**
 * https://developers.google.com/protocol-buffers/docs/reference/java/com/google/protobuf/Any
 * @export
 * @interface ProtobufAny
 */
export interface ProtobufAny {
    /**
     * https://developers.google.com/protocol-buffers/docs/reference/java/com/google/protobuf/Any
     * @type {string}
     * @memberof ProtobufAny
     */
    typeUrl?: string;
    /**
     * Must be a valid serialized protocol buffer of the above specified type.
     * @type {string}
     * @memberof ProtobufAny
     */
    value?: string;
}
/**
 * https://protobuf.dev/reference/java/api-docs/com/google/protobuf/FieldMask
 * @export
 * @interface ProtobufFieldMask
 */
export interface ProtobufFieldMask {
    /**
     * The set of field mask paths.
     * @type {Array<string>}
     * @memberof ProtobufFieldMask
     */
    paths?: Array<string>;
}
/**
 * `NullValue` is a singleton enumeration to represent the null value for the `Value` type union.   The JSON representation for `NullValue` is JSON `null`.   - NULL_VALUE: Null value.
 * @export
 * @enum {string}
 */
export const ProtobufNullValue = {
    NULLVALUE: 'NULL_VALUE',
} as const
export type ProtobufNullValue = ValueOf<typeof ProtobufNullValue>
/**
 * 
 * @export
 * @interface ResourceManagerHealth
 */
export interface ResourceManagerHealth {
    /**
     * 
     * @type {string}
     * @memberof ResourceManagerHealth
     */
    cluster_name?: string;
    /**
     * 
     * @type {HealthStatus}
     * @memberof ResourceManagerHealth
     */
    status?: HealthStatus;
}
/**
 * A wrapper message of a list of devices.
 * @export
 * @interface ResourcesSummaryDevices
 */
export interface ResourcesSummaryDevices {
    /**
     * The devices on an agent.
     * @type {Array<V1Device>}
     * @memberof ResourcesSummaryDevices
     */
    devices?: Array<V1Device>;
}
/**
 * 
 * @export
 * @interface RuntimeError
 */
export interface RuntimeError {
    /**
     * 
     * @type {string}
     * @memberof RuntimeError
     */
    error?: string;
    /**
     * 
     * @type {number}
     * @memberof RuntimeError
     */
    code?: number;
    /**
     * 
     * @type {string}
     * @memberof RuntimeError
     */
    message?: string;
    /**
     * 
     * @type {Array<ProtobufAny>}
     * @memberof RuntimeError
     */
    details?: Array<ProtobufAny>;
}
/**
 * 
 * @export
 * @interface RuntimeStreamError
 */
export interface RuntimeStreamError {
    /**
     * 
     * @type {number}
     * @memberof RuntimeStreamError
     */
    grpcCode?: number;
    /**
     * 
     * @type {number}
     * @memberof RuntimeStreamError
     */
    httpCode?: number;
    /**
     * 
     * @type {string}
     * @memberof RuntimeStreamError
     */
    message?: string;
    /**
     * 
     * @type {string}
     * @memberof RuntimeStreamError
     */
    httpStatus?: string;
    /**
     * 
     * @type {Array<ProtobufAny>}
     * @memberof RuntimeStreamError
     */
    details?: Array<ProtobufAny>;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1ExpMetricNamesResponse
 */
export interface StreamResultOfV1ExpMetricNamesResponse {
    /**
     * 
     * @type {V1ExpMetricNamesResponse}
     * @memberof StreamResultOfV1ExpMetricNamesResponse
     */
    result?: V1ExpMetricNamesResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1ExpMetricNamesResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1GetMetricsResponse
 */
export interface StreamResultOfV1GetMetricsResponse {
    /**
     * 
     * @type {V1GetMetricsResponse}
     * @memberof StreamResultOfV1GetMetricsResponse
     */
    result?: V1GetMetricsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1GetMetricsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1GetTrainingMetricsResponse
 */
export interface StreamResultOfV1GetTrainingMetricsResponse {
    /**
     * 
     * @type {V1GetTrainingMetricsResponse}
     * @memberof StreamResultOfV1GetTrainingMetricsResponse
     */
    result?: V1GetTrainingMetricsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1GetTrainingMetricsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1GetTrialProfilerAvailableSeriesResponse
 */
export interface StreamResultOfV1GetTrialProfilerAvailableSeriesResponse {
    /**
     * 
     * @type {V1GetTrialProfilerAvailableSeriesResponse}
     * @memberof StreamResultOfV1GetTrialProfilerAvailableSeriesResponse
     */
    result?: V1GetTrialProfilerAvailableSeriesResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1GetTrialProfilerAvailableSeriesResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1GetTrialProfilerMetricsResponse
 */
export interface StreamResultOfV1GetTrialProfilerMetricsResponse {
    /**
     * 
     * @type {V1GetTrialProfilerMetricsResponse}
     * @memberof StreamResultOfV1GetTrialProfilerMetricsResponse
     */
    result?: V1GetTrialProfilerMetricsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1GetTrialProfilerMetricsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1GetValidationMetricsResponse
 */
export interface StreamResultOfV1GetValidationMetricsResponse {
    /**
     * 
     * @type {V1GetValidationMetricsResponse}
     * @memberof StreamResultOfV1GetValidationMetricsResponse
     */
    result?: V1GetValidationMetricsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1GetValidationMetricsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1MasterLogsResponse
 */
export interface StreamResultOfV1MasterLogsResponse {
    /**
     * 
     * @type {V1MasterLogsResponse}
     * @memberof StreamResultOfV1MasterLogsResponse
     */
    result?: V1MasterLogsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1MasterLogsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1MetricBatchesResponse
 */
export interface StreamResultOfV1MetricBatchesResponse {
    /**
     * 
     * @type {V1MetricBatchesResponse}
     * @memberof StreamResultOfV1MetricBatchesResponse
     */
    result?: V1MetricBatchesResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1MetricBatchesResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TaskLogsFieldsResponse
 */
export interface StreamResultOfV1TaskLogsFieldsResponse {
    /**
     * 
     * @type {V1TaskLogsFieldsResponse}
     * @memberof StreamResultOfV1TaskLogsFieldsResponse
     */
    result?: V1TaskLogsFieldsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TaskLogsFieldsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TaskLogsResponse
 */
export interface StreamResultOfV1TaskLogsResponse {
    /**
     * 
     * @type {V1TaskLogsResponse}
     * @memberof StreamResultOfV1TaskLogsResponse
     */
    result?: V1TaskLogsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TaskLogsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TrialLogsFieldsResponse
 */
export interface StreamResultOfV1TrialLogsFieldsResponse {
    /**
     * 
     * @type {V1TrialLogsFieldsResponse}
     * @memberof StreamResultOfV1TrialLogsFieldsResponse
     */
    result?: V1TrialLogsFieldsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TrialLogsFieldsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TrialLogsResponse
 */
export interface StreamResultOfV1TrialLogsResponse {
    /**
     * 
     * @type {V1TrialLogsResponse}
     * @memberof StreamResultOfV1TrialLogsResponse
     */
    result?: V1TrialLogsResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TrialLogsResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TrialsSampleResponse
 */
export interface StreamResultOfV1TrialsSampleResponse {
    /**
     * 
     * @type {V1TrialsSampleResponse}
     * @memberof StreamResultOfV1TrialsSampleResponse
     */
    result?: V1TrialsSampleResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TrialsSampleResponse
     */
    error?: RuntimeStreamError;
}
/**
 * 
 * @export
 * @interface StreamResultOfV1TrialsSnapshotResponse
 */
export interface StreamResultOfV1TrialsSnapshotResponse {
    /**
     * 
     * @type {V1TrialsSnapshotResponse}
     * @memberof StreamResultOfV1TrialsSnapshotResponse
     */
    result?: V1TrialsSnapshotResponse;
    /**
     * 
     * @type {RuntimeStreamError}
     * @memberof StreamResultOfV1TrialsSnapshotResponse
     */
    error?: RuntimeStreamError;
}
/**
 * The current state of the task.   - STATE_UNSPECIFIED: The task state is unknown.  - STATE_PULLING: The task's base image is being pulled from the Docker registry.  - STATE_STARTING: The image has been pulled and the task is being started, but the task is not ready yet.  - STATE_RUNNING: The service in the task is running.  - STATE_TERMINATED: The task has exited or has been aborted.  - STATE_TERMINATING: The task has begun to exit.  - STATE_WAITING: The task is waiting on something to complete.  - STATE_QUEUED: Additional state to cover queueing operations.
 * @export
 * @enum {string}
 */
export const Taskv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    PULLING: 'STATE_PULLING',
    STARTING: 'STATE_STARTING',
    RUNNING: 'STATE_RUNNING',
    TERMINATED: 'STATE_TERMINATED',
    TERMINATING: 'STATE_TERMINATING',
    WAITING: 'STATE_WAITING',
    QUEUED: 'STATE_QUEUED',
} as const
export type Taskv1State = ValueOf<typeof Taskv1State>
/**
 * To distinguish the 2 different categories of metrics.   - PROFILER_METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - PROFILER_METRIC_TYPE_SYSTEM: For systems metrics, like GPU utilization or memory.  - PROFILER_METRIC_TYPE_TIMING: For timing metrics, like how long a backwards pass or getting a batch from the dataloader took.  - PROFILER_METRIC_TYPE_MISC: For other miscellaneous metrics.
 * @export
 * @enum {string}
 */
export const TrialProfilerMetricLabelsProfilerMetricType = {
    UNSPECIFIED: 'PROFILER_METRIC_TYPE_UNSPECIFIED',
    SYSTEM: 'PROFILER_METRIC_TYPE_SYSTEM',
    TIMING: 'PROFILER_METRIC_TYPE_TIMING',
    MISC: 'PROFILER_METRIC_TYPE_MISC',
} as const
export type TrialProfilerMetricLabelsProfilerMetricType = ValueOf<typeof TrialProfilerMetricLabelsProfilerMetricType>
/**
 * The current state of the trial. see \dT+ trial_state in db - STATE_UNSPECIFIED: The trial is in an unspecified state.  - STATE_ACTIVE: The trial is in an active state.  - STATE_PAUSED: The trial is in a paused state  - STATE_STOPPING_CANCELED: The trial is canceled and is shutting down.  - STATE_STOPPING_KILLED: The trial is killed and is shutting down.  - STATE_STOPPING_COMPLETED: The trial is completed and is shutting down.  - STATE_STOPPING_ERROR: The trial is errored and is shutting down.  - STATE_CANCELED: The trial is canceled and is shut down.  - STATE_COMPLETED: The trial is completed and is shut down.  - STATE_ERROR: The trial is errored and is shut down.  - STATE_QUEUED: The trial is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The trial is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The trial is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The trial's allocation is actively running. Running is a substate of the Active state.
 * @export
 * @enum {string}
 */
export const Trialv1State = {
    UNSPECIFIED: 'STATE_UNSPECIFIED',
    ACTIVE: 'STATE_ACTIVE',
    PAUSED: 'STATE_PAUSED',
    STOPPINGCANCELED: 'STATE_STOPPING_CANCELED',
    STOPPINGKILLED: 'STATE_STOPPING_KILLED',
    STOPPINGCOMPLETED: 'STATE_STOPPING_COMPLETED',
    STOPPINGERROR: 'STATE_STOPPING_ERROR',
    CANCELED: 'STATE_CANCELED',
    COMPLETED: 'STATE_COMPLETED',
    ERROR: 'STATE_ERROR',
    QUEUED: 'STATE_QUEUED',
    PULLING: 'STATE_PULLING',
    STARTING: 'STATE_STARTING',
    RUNNING: 'STATE_RUNNING',
} as const
export type Trialv1State = ValueOf<typeof Trialv1State>
/**
 * Trial is a set of workloads and are exploring a determined set of hyperparameters.
 * @export
 * @interface Trialv1Trial
 */
export interface Trialv1Trial {
    /**
     * The id of the trial.
     * @type {number}
     * @memberof Trialv1Trial
     */
    id: number;
    /**
     * The id of the parent experiment.
     * @type {number}
     * @memberof Trialv1Trial
     */
    experimentId: number;
    /**
     * The time the trial was started.
     * @type {Date | DateString}
     * @memberof Trialv1Trial
     */
    startTime: Date | DateString;
    /**
     * The time the trial ended if the trial is stopped.
     * @type {Date | DateString}
     * @memberof Trialv1Trial
     */
    endTime?: Date | DateString;
    /**
     * The current state of the trial.
     * @type {Trialv1State}
     * @memberof Trialv1Trial
     */
    state: Trialv1State;
    /**
     * Number times the trial restarted.
     * @type {number}
     * @memberof Trialv1Trial
     */
    restarts: number;
    /**
     * Trial hyperparameters.
     * @type {any}
     * @memberof Trialv1Trial
     */
    hparams: any;
    /**
     * The current processed batches.
     * @type {number}
     * @memberof Trialv1Trial
     */
    totalBatchesProcessed: number;
    /**
     * Best validation.
     * @type {V1MetricsWorkload}
     * @memberof Trialv1Trial
     */
    bestValidation?: V1MetricsWorkload;
    /**
     * Latest validation.
     * @type {V1MetricsWorkload}
     * @memberof Trialv1Trial
     */
    latestValidation?: V1MetricsWorkload;
    /**
     * Best checkpoint.
     * @type {V1CheckpointWorkload}
     * @memberof Trialv1Trial
     */
    bestCheckpoint?: V1CheckpointWorkload;
    /**
     * The last reported state of the trial runner (harness code).
     * @type {string}
     * @memberof Trialv1Trial
     */
    runnerState?: string;
    /**
     * The wall clock time is all active time of the cluster for the trial, inclusive of everything (restarts, initiailization, etc), in seconds.
     * @type {number}
     * @memberof Trialv1Trial
     */
    wallClockTime?: number;
    /**
     * UUID of checkpoint that this trial started from.
     * @type {string}
     * @memberof Trialv1Trial
     */
    warmStartCheckpointUuid?: string;
    /**
     * Id of the first task associated with this trial. This field is deprecated since trials can have multiple tasks.
     * @type {string}
     * @memberof Trialv1Trial
     */
    taskId?: string;
    /**
     * The sum of sizes of all resources in all checkpoints for the trial.
     * @type {string}
     * @memberof Trialv1Trial
     */
    totalCheckpointSize?: string;
    /**
     * The count of checkpoints.
     * @type {number}
     * @memberof Trialv1Trial
     */
    checkpointCount?: number;
    /**
     * summary metrics
     * @type {any}
     * @memberof Trialv1Trial
     */
    summaryMetrics?: any;
    /**
     * Task IDs of tasks associated with this trial. Length of task_ids will always be greater or equal to one when TaskID is sent. For example CompareTrial we will send a reduced Trial object, without TaskID or TaskIDs fileld in. The first element of task_ids will be the same as task_id. task_ids is sorted ascending by task_run_id.
     * @type {Array<string>}
     * @memberof Trialv1Trial
     */
    taskIds?: Array<string>;
    /**
     * Signed searcher metrics value.
     * @type {number}
     * @memberof Trialv1Trial
     */
    searcherMetricValue?: number;
    /**
     * Number of days to retain logs for.
     * @type {number}
     * @memberof Trialv1Trial
     */
    logRetentionDays?: number;
    /**
     * metadata associated with the trial (based off the metadata stored in the run).
     * @type {any}
     * @memberof Trialv1Trial
     */
    metadata?: any;
    /**
     * Log Policy Matched.
     * @type {string}
     * @memberof Trialv1Trial
     */
    logPolicyMatched?: string;
}
/**
 * 
 * @export
 * @interface V1AcceleratorData
 */
export interface V1AcceleratorData {
    /**
     * The id of the container.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    containerId?: string;
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    allocationId?: string;
    /**
     * The name of the node the allocation is on.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    nodeName?: string;
    /**
     * The id of the agent associated with the allocation.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    taskId?: string;
    /**
     * The type of accelerator.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    acceleratorType?: string;
    /**
     * An array of UUIDs of the accelerators associated with the allocation.
     * @type {Array<string>}
     * @memberof V1AcceleratorData
     */
    acceleratorUuids?: Array<string>;
    /**
     * The name of the resource pool.
     * @type {string}
     * @memberof V1AcceleratorData
     */
    resourcePool?: string;
}
/**
 * Acknowledge the receipt of some stop signal.
 * @export
 * @interface V1AckAllocationPreemptionSignalRequest
 */
export interface V1AckAllocationPreemptionSignalRequest {
    /**
     * The allocation that is acknowledging the request.
     * @type {string}
     * @memberof V1AckAllocationPreemptionSignalRequest
     */
    allocationId: string;
}
/**
 * Response to AckAllocationPreemptionSignalRequest.
 * @export
 * @interface V1AckAllocationPreemptionSignalResponse
 */
export interface V1AckAllocationPreemptionSignalResponse {
}
/**
 * Response to ActivateExperimentRequest.
 * @export
 * @interface V1ActivateExperimentResponse
 */
export interface V1ActivateExperimentResponse {
}
/**
 * Activate multiple experiments.
 * @export
 * @interface V1ActivateExperimentsRequest
 */
export interface V1ActivateExperimentsRequest {
    /**
     * Select experiments by id.
     * @type {Array<number>}
     * @memberof V1ActivateExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1ActivateExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1ActivateExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to ActivateExperimentsRequest.
 * @export
 * @interface V1ActivateExperimentsResponse
 */
export interface V1ActivateExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1ActivateExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * ActivityType represents a user activity - ACTIVITY_TYPE_UNSPECIFIED: Default activity type.  - ACTIVITY_TYPE_GET: Represents a get request.
 * @export
 * @enum {string}
 */
export const V1ActivityType = {
    UNSPECIFIED: 'ACTIVITY_TYPE_UNSPECIFIED',
    GET: 'ACTIVITY_TYPE_GET',
} as const
export type V1ActivityType = ValueOf<typeof V1ActivityType>
/**
 * Response to AddProjectNoteRequest.
 * @export
 * @interface V1AddProjectNoteResponse
 */
export interface V1AddProjectNoteResponse {
    /**
     * The complete list of notes on a project.
     * @type {Array<V1Note>}
     * @memberof V1AddProjectNoteResponse
     */
    notes: Array<V1Note>;
}
/**
 * Address represents an exposed port on a container.
 * @export
 * @interface V1Address
 */
export interface V1Address {
    /**
     * ContainerIP is the IP address from inside the container.
     * @type {string}
     * @memberof V1Address
     */
    containerIp?: string;
    /**
     * ContainerPort is the port from inside the container.
     * @type {number}
     * @memberof V1Address
     */
    containerPort?: number;
    /**
     * HostIP is the IP address from outside the container. This can be different than the ContainerIP because of network forwarding on the host machine.
     * @type {string}
     * @memberof V1Address
     */
    hostIp?: string;
    /**
     * HostPort is the IP port from outside the container. This can be different than the ContainerPort because of network forwarding on the host machine.
     * @type {number}
     * @memberof V1Address
     */
    hostPort?: number;
}
/**
 * Agent is a pool of resources where containers are run.
 * @export
 * @interface V1Agent
 */
export interface V1Agent {
    /**
     * The unique id of the agent.
     * @type {string}
     * @memberof V1Agent
     */
    id: string;
    /**
     * The time when the agent registered with the master.
     * @type {Date | DateString}
     * @memberof V1Agent
     */
    registeredTime?: Date | DateString;
    /**
     * A map of slot id to each slot of this agent.
     * @type {{ [key: string]: V1Slot; }}
     * @memberof V1Agent
     */
    slots?: { [key: string]: V1Slot; };
    /**
     * A map of container id to all containers assigned to this agent.
     * @type {{ [key: string]: V1Container; }}
     * @memberof V1Agent
     */
    containers?: { [key: string]: V1Container; };
    /**
     * The addresses of the agent.
     * @type {Array<string>}
     * @memberof V1Agent
     */
    addresses?: Array<string>;
    /**
     * Flag notifying if containers can be scheduled on this agent.
     * @type {boolean}
     * @memberof V1Agent
     */
    enabled?: boolean;
    /**
     * Flag notifying if this agent is in the draining mode: current containers will be allowed to finish but no new ones will be scheduled.
     * @type {boolean}
     * @memberof V1Agent
     */
    draining?: boolean;
    /**
     * The Determined version that this agent was built from.
     * @type {string}
     * @memberof V1Agent
     */
    version?: string;
    /**
     * The name of the resource pools the agent is in. Only slurm can contain multiples.
     * @type {Array<string>}
     * @memberof V1Agent
     */
    resourcePools?: Array<string>;
    /**
     * The slot stats for this agent.
     * @type {V1SlotStats}
     * @memberof V1Agent
     */
    slotStats: V1SlotStats;
}
/**
 * AgentUserGroup represents a username and primary group for a user on an agent host machine.
 * @export
 * @interface V1AgentUserGroup
 */
export interface V1AgentUserGroup {
    /**
     * The user id on the agent.
     * @type {number}
     * @memberof V1AgentUserGroup
     */
    agentUid?: number;
    /**
     * The group id on the agent.
     * @type {number}
     * @memberof V1AgentUserGroup
     */
    agentGid?: number;
    /**
     * User name.
     * @type {string}
     * @memberof V1AgentUserGroup
     */
    agentUser?: string;
    /**
     * Group name.
     * @type {string}
     * @memberof V1AgentUserGroup
     */
    agentGroup?: string;
}
/**
 * Aggregate statistics for a queue.
 * @export
 * @interface V1AggregateQueueStats
 */
export interface V1AggregateQueueStats {
    /**
     * The date of this entry.
     * @type {string}
     * @memberof V1AggregateQueueStats
     */
    periodStart: string;
    /**
     * The total number of seconds queued.
     * @type {number}
     * @memberof V1AggregateQueueStats
     */
    seconds: number;
}
/**
 * Allocation tracks a specific instance of a Task.
 * @export
 * @interface V1Allocation
 */
export interface V1Allocation {
    /**
     * Unique ID of task associated with the allocation.
     * @type {string}
     * @memberof V1Allocation
     */
    taskId: string;
    /**
     * The current state of the allocation.
     * @type {Taskv1State}
     * @memberof V1Allocation
     */
    state: Taskv1State;
    /**
     * Whether the allocation is ready to access.
     * @type {boolean}
     * @memberof V1Allocation
     */
    isReady?: boolean;
    /**
     * Start timestamp.
     * @type {string}
     * @memberof V1Allocation
     */
    startTime?: string;
    /**
     * End timestamp if completed.
     * @type {string}
     * @memberof V1Allocation
     */
    endTime?: string;
    /**
     * Unique ID of the allocation.
     * @type {string}
     * @memberof V1Allocation
     */
    allocationId: string;
    /**
     * The number of slots associated with the allocation.
     * @type {number}
     * @memberof V1Allocation
     */
    slots: number;
    /**
     * The exit reason for the allocation.
     * @type {string}
     * @memberof V1Allocation
     */
    exitReason?: string;
    /**
     * The status code the allocation exits with.
     * @type {number}
     * @memberof V1Allocation
     */
    statusCode?: number;
}
/**
 * Arguments to an all gather.
 * @export
 * @interface V1AllocationAllGatherRequest
 */
export interface V1AllocationAllGatherRequest {
    /**
     * The ID of the allocation.
     * @type {string}
     * @memberof V1AllocationAllGatherRequest
     */
    allocationId: string;
    /**
     * The UUID of the participant in an all gather.
     * @type {string}
     * @memberof V1AllocationAllGatherRequest
     */
    requestUuid?: string;
    /**
     * The number of process to wait for.
     * @type {number}
     * @memberof V1AllocationAllGatherRequest
     */
    numPeers?: number;
    /**
     * The data from this process.
     * @type {any}
     * @memberof V1AllocationAllGatherRequest
     */
    data: any;
}
/**
 * 
 * @export
 * @interface V1AllocationAllGatherResponse
 */
export interface V1AllocationAllGatherResponse {
    /**
     * The data for all the processes.
     * @type {Array<any>}
     * @memberof V1AllocationAllGatherResponse
     */
    data: Array<any>;
}
/**
 * 
 * @export
 * @interface V1AllocationPendingPreemptionSignalRequest
 */
export interface V1AllocationPendingPreemptionSignalRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1AllocationPendingPreemptionSignalRequest
     */
    allocationId: string;
}
/**
 * 
 * @export
 * @interface V1AllocationPendingPreemptionSignalResponse
 */
export interface V1AllocationPendingPreemptionSignalResponse {
}
/**
 * Response to AllocationPreemptionSignalRequest.
 * @export
 * @interface V1AllocationPreemptionSignalResponse
 */
export interface V1AllocationPreemptionSignalResponse {
    /**
     * True if signaling preempt, otherwise just a synchronization marker.
     * @type {boolean}
     * @memberof V1AllocationPreemptionSignalResponse
     */
    preempt?: boolean;
}
/**
 * Mark the given task as ready.
 * @export
 * @interface V1AllocationReadyRequest
 */
export interface V1AllocationReadyRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1AllocationReadyRequest
     */
    allocationId?: string;
}
/**
 * Response to AllocationReadyRequest.
 * @export
 * @interface V1AllocationReadyResponse
 */
export interface V1AllocationReadyResponse {
}
/**
 * 
 * @export
 * @interface V1AllocationRendezvousInfoResponse
 */
export interface V1AllocationRendezvousInfoResponse {
    /**
     * The rendezvous information.
     * @type {V1RendezvousInfo}
     * @memberof V1AllocationRendezvousInfoResponse
     */
    rendezvousInfo: V1RendezvousInfo;
}
/**
 * AllocationSummary contains information about a task for external display.
 * @export
 * @interface V1AllocationSummary
 */
export interface V1AllocationSummary {
    /**
     * TaskID is the unique ID of a task among all tasks.
     * @type {string}
     * @memberof V1AllocationSummary
     */
    taskId?: string;
    /**
     * AllocationID is the ID of an allocation of a task. It is usually of the form TaskID.allocation_number, maybe with some other metadata if different types of allocations run.
     * @type {string}
     * @memberof V1AllocationSummary
     */
    allocationId?: string;
    /**
     * The name of the task.
     * @type {string}
     * @memberof V1AllocationSummary
     */
    name?: string;
    /**
     * The registered time of the task.
     * @type {Date | DateString}
     * @memberof V1AllocationSummary
     */
    registeredTime?: Date | DateString;
    /**
     * The name of the resource pool.
     * @type {string}
     * @memberof V1AllocationSummary
     */
    resourcePool?: string;
    /**
     * The number of slots that are needed.
     * @type {number}
     * @memberof V1AllocationSummary
     */
    slotsNeeded?: number;
    /**
     * ResourcesSummary provides a summary of the resources comprising what we know at the time the allocation is granted.
     * @type {Array<V1ResourcesSummary>}
     * @memberof V1AllocationSummary
     */
    resources?: Array<V1ResourcesSummary>;
    /**
     * The type of the scheduler. Either 'FAIR_SHARE', 'PRIORITY', or 'ROUND_ROBIN'.
     * @type {string}
     * @memberof V1AllocationSummary
     */
    schedulerType?: string;
    /**
     * THe priority of the task.
     * @type {number}
     * @memberof V1AllocationSummary
     */
    priority?: number;
    /**
     * ProxyPortConfig configures a proxy the allocation should start.
     * @type {Array<V1ProxyPortConfig>}
     * @memberof V1AllocationSummary
     */
    proxyPorts?: Array<V1ProxyPortConfig>;
}
/**
 * Mark the given task as waiting.
 * @export
 * @interface V1AllocationWaitingRequest
 */
export interface V1AllocationWaitingRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1AllocationWaitingRequest
     */
    allocationId?: string;
}
/**
 * Response to AllocationWaitingRequest.
 * @export
 * @interface V1AllocationWaitingResponse
 */
export interface V1AllocationWaitingResponse {
}
/**
 * Response to ArchiveExperimentRequest.
 * @export
 * @interface V1ArchiveExperimentResponse
 */
export interface V1ArchiveExperimentResponse {
}
/**
 * Archive multiple experiments.
 * @export
 * @interface V1ArchiveExperimentsRequest
 */
export interface V1ArchiveExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1ArchiveExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1ArchiveExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1ArchiveExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to ArchiveExperimentsRequest.
 * @export
 * @interface V1ArchiveExperimentsResponse
 */
export interface V1ArchiveExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1ArchiveExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * 
 * @export
 * @interface V1ArchiveModelResponse
 */
export interface V1ArchiveModelResponse {
}
/**
 * Response to ArchiveProjectRequest.
 * @export
 * @interface V1ArchiveProjectResponse
 */
export interface V1ArchiveProjectResponse {
}
/**
 * 
 * @export
 * @interface V1ArchiveRunsRequest
 */
export interface V1ArchiveRunsRequest {
    /**
     * The ids of the runs being archived.
     * @type {Array<number>}
     * @memberof V1ArchiveRunsRequest
     */
    runIds: Array<number>;
    /**
     * The id of the current parent project.
     * @type {number}
     * @memberof V1ArchiveRunsRequest
     */
    projectId: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1ArchiveRunsRequest
     */
    filter?: string;
}
/**
 * Response to ArchiveRunsRequest.
 * @export
 * @interface V1ArchiveRunsResponse
 */
export interface V1ArchiveRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1ArchiveRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * Response to ArchiveWorkspaceRequest.
 * @export
 * @interface V1ArchiveWorkspaceResponse
 */
export interface V1ArchiveWorkspaceResponse {
}
/**
 * Add and remove multiple users from multiple groups.
 * @export
 * @interface V1AssignMultipleGroupsRequest
 */
export interface V1AssignMultipleGroupsRequest {
    /**
     * The user ids of users to edit group associations.
     * @type {Array<number>}
     * @memberof V1AssignMultipleGroupsRequest
     */
    userIds: Array<number>;
    /**
     * The ids of groups to associate with users.
     * @type {Array<number>}
     * @memberof V1AssignMultipleGroupsRequest
     */
    addGroups: Array<number>;
    /**
     * The ids of groups to disassociate from users.
     * @type {Array<number>}
     * @memberof V1AssignMultipleGroupsRequest
     */
    removeGroups: Array<number>;
}
/**
 * Response to AssignMultipleGroupsRequest.
 * @export
 * @interface V1AssignMultipleGroupsResponse
 */
export interface V1AssignMultipleGroupsResponse {
}
/**
 * AssignRolesRequest is the body of the request for the call to grant a user or group a role. It requires group_id, role_id, and either scope_workspace_id or scope_project_id.
 * @export
 * @interface V1AssignRolesRequest
 */
export interface V1AssignRolesRequest {
    /**
     * the set of groups being assigned to a role.
     * @type {Array<V1GroupRoleAssignment>}
     * @memberof V1AssignRolesRequest
     */
    groupRoleAssignments?: Array<V1GroupRoleAssignment>;
    /**
     * the set of users being assigned to a role.
     * @type {Array<V1UserRoleAssignment>}
     * @memberof V1AssignRolesRequest
     */
    userRoleAssignments?: Array<V1UserRoleAssignment>;
}
/**
 * AssignRolesResponse is the body of the request for the call to grant a user or group a role.
 * @export
 * @interface V1AssignRolesResponse
 */
export interface V1AssignRolesResponse {
}
/**
 * 
 * @export
 * @interface V1AwsCustomTag
 */
export interface V1AwsCustomTag {
    /**
     * The key of the custom tag
     * @type {string}
     * @memberof V1AwsCustomTag
     */
    key: string;
    /**
     * The value of the custom tag
     * @type {string}
     * @memberof V1AwsCustomTag
     */
    value: string;
}
/**
 * 
 * @export
 * @interface V1BindRPToWorkspaceRequest
 */
export interface V1BindRPToWorkspaceRequest {
    /**
     * The resource pool name.
     * @type {string}
     * @memberof V1BindRPToWorkspaceRequest
     */
    resourcePoolName: string;
    /**
     * The workspace IDs to be bound to the resource pool.
     * @type {Array<number>}
     * @memberof V1BindRPToWorkspaceRequest
     */
    workspaceIds?: Array<number>;
    /**
     * The workspace names to be bound to the resource pool.
     * @type {Array<string>}
     * @memberof V1BindRPToWorkspaceRequest
     */
    workspaceNames?: Array<string>;
}
/**
 * Bind a resource pool to workspaces response.
 * @export
 * @interface V1BindRPToWorkspaceResponse
 */
export interface V1BindRPToWorkspaceResponse {
}
/**
 * Request for binding the given workpaces to new auto-created namespaces.
 * @export
 * @interface V1BulkAutoCreateWorkspaceNamespaceBindingsRequest
 */
export interface V1BulkAutoCreateWorkspaceNamespaceBindingsRequest {
    /**
     * The list of workspace ids.
     * @type {Array<number>}
     * @memberof V1BulkAutoCreateWorkspaceNamespaceBindingsRequest
     */
    workspaceIds?: Array<number>;
}
/**
 * Response to BulkAutoCreateWorkspaceNamespaceBindingsRequest.
 * @export
 * @interface V1BulkAutoCreateWorkspaceNamespaceBindingsResponse
 */
export interface V1BulkAutoCreateWorkspaceNamespaceBindingsResponse {
}
/**
 * Filters to apply actions to multiple experiments.
 * @export
 * @interface V1BulkExperimentFilters
 */
export interface V1BulkExperimentFilters {
    /**
     * Limit experiments to those that match the description.
     * @type {string}
     * @memberof V1BulkExperimentFilters
     */
    description?: string;
    /**
     * Limit experiments to those that match the name.
     * @type {string}
     * @memberof V1BulkExperimentFilters
     */
    name?: string;
    /**
     * Limit experiments to those that match the provided labels.
     * @type {Array<string>}
     * @memberof V1BulkExperimentFilters
     */
    labels?: Array<string>;
    /**
     * Limit experiments to those that are archived.
     * @type {boolean}
     * @memberof V1BulkExperimentFilters
     */
    archived?: boolean;
    /**
     * Limit experiments to those that match the provided state.
     * @type {Array<Experimentv1State>}
     * @memberof V1BulkExperimentFilters
     */
    states?: Array<Experimentv1State>;
    /**
     * Limit experiments to those that are owned by users with the specified userIds.
     * @type {Array<number>}
     * @memberof V1BulkExperimentFilters
     */
    userIds?: Array<number>;
    /**
     * Limit experiments to those within a specified project, or 0 for all projects.
     * @type {number}
     * @memberof V1BulkExperimentFilters
     */
    projectId?: number;
    /**
     * Experiment ids to exclude when filters are used.
     * @type {Array<number>}
     * @memberof V1BulkExperimentFilters
     */
    excludedExperimentIds?: Array<number>;
}
/**
 * Response to CancelExperimentRequest.
 * @export
 * @interface V1CancelExperimentResponse
 */
export interface V1CancelExperimentResponse {
}
/**
 * Cancel multiple experiments.
 * @export
 * @interface V1CancelExperimentsRequest
 */
export interface V1CancelExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1CancelExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1CancelExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1CancelExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to CancelExperimentsRequest.
 * @export
 * @interface V1CancelExperimentsResponse
 */
export interface V1CancelExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1CancelExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * Checkpoint a collection of files saved by a task.
 * @export
 * @interface V1Checkpoint
 */
export interface V1Checkpoint {
    /**
     * ID of the task which generated this checkpoint.
     * @type {string}
     * @memberof V1Checkpoint
     */
    taskId?: string;
    /**
     * ID of the allocation which generated this checkpoint.
     * @type {string}
     * @memberof V1Checkpoint
     */
    allocationId?: string;
    /**
     * UUID of the checkpoint.
     * @type {string}
     * @memberof V1Checkpoint
     */
    uuid: string;
    /**
     * Timestamp when the checkpoint was reported.
     * @type {Date | DateString}
     * @memberof V1Checkpoint
     */
    reportTime?: Date | DateString;
    /**
     * Dictionary of file paths to file sizes in bytes of all files in the checkpoint.
     * @type {{ [key: string]: string; }}
     * @memberof V1Checkpoint
     */
    resources: { [key: string]: string; };
    /**
     * User defined metadata associated with the checkpoint.
     * @type {any}
     * @memberof V1Checkpoint
     */
    metadata: any;
    /**
     * The state of the underlying checkpoint.
     * @type {Checkpointv1State}
     * @memberof V1Checkpoint
     */
    state: Checkpointv1State;
    /**
     * Training-related data for this checkpoint.
     * @type {V1CheckpointTrainingMetadata}
     * @memberof V1Checkpoint
     */
    training: V1CheckpointTrainingMetadata;
    /**
     * Optional ID that describes where this checkpoint is stored. It will be null on every checkpoint pre 0.27.1. It can also be null when a user does not specify the storageID calling the report API themselves or when users don't provide a storage config to core_context.
     * @type {number}
     * @memberof V1Checkpoint
     */
    storageId?: number;
}
/**
 * Request to delete files matching globs in checkpoints.
 * @export
 * @interface V1CheckpointsRemoveFilesRequest
 */
export interface V1CheckpointsRemoveFilesRequest {
    /**
     * The list of checkpoint_uuids for the requested checkpoints.
     * @type {Array<string>}
     * @memberof V1CheckpointsRemoveFilesRequest
     */
    checkpointUuids: Array<string>;
    /**
     * The list of checkpoint_globs for the requested checkpoints. If a value is set to the empty string the checkpoint will only have its metadata refreshed.
     * @type {Array<string>}
     * @memberof V1CheckpointsRemoveFilesRequest
     */
    checkpointGlobs: Array<string>;
}
/**
 * Response to CheckpointRemoveFilesRequest.
 * @export
 * @interface V1CheckpointsRemoveFilesResponse
 */
export interface V1CheckpointsRemoveFilesResponse {
}
/**
 * CheckpointTrainingMetadata is specifically metadata about training.
 * @export
 * @interface V1CheckpointTrainingMetadata
 */
export interface V1CheckpointTrainingMetadata {
    /**
     * The ID of the trial that created this checkpoint.
     * @type {number}
     * @memberof V1CheckpointTrainingMetadata
     */
    trialId?: number;
    /**
     * The ID of the experiment that created this checkpoint.
     * @type {number}
     * @memberof V1CheckpointTrainingMetadata
     */
    experimentId?: number;
    /**
     * The configuration of the experiment that created this checkpoint.
     * @type {any}
     * @memberof V1CheckpointTrainingMetadata
     */
    experimentConfig?: any;
    /**
     * Hyperparameter values for the trial that created this checkpoint.
     * @type {any}
     * @memberof V1CheckpointTrainingMetadata
     */
    hparams?: any;
    /**
     * Training metrics reported at the same steps_completed as the checkpoint.
     * @type {V1Metrics}
     * @memberof V1CheckpointTrainingMetadata
     */
    trainingMetrics?: V1Metrics;
    /**
     * Validation metrics reported at the same steps_completed as the checkpoint.
     * @type {V1Metrics}
     * @memberof V1CheckpointTrainingMetadata
     */
    validationMetrics?: V1Metrics;
    /**
     * Searcher metric (as specified by the expconf) at the same steps_completed of the checkpoint.
     * @type {number}
     * @memberof V1CheckpointTrainingMetadata
     */
    searcherMetric?: number;
}
/**
 * CheckpointWorkload is an artifact created by a trial during training.
 * @export
 * @interface V1CheckpointWorkload
 */
export interface V1CheckpointWorkload {
    /**
     * UUID of the checkpoint.
     * @type {string}
     * @memberof V1CheckpointWorkload
     */
    uuid?: string;
    /**
     * The time the workload finished or was stopped.
     * @type {Date | DateString}
     * @memberof V1CheckpointWorkload
     */
    endTime?: Date | DateString;
    /**
     * The state of the checkpoint.
     * @type {Checkpointv1State}
     * @memberof V1CheckpointWorkload
     */
    state: Checkpointv1State;
    /**
     * Dictionary of file paths to file sizes in bytes of all files in the checkpoint.
     * @type {{ [key: string]: string; }}
     * @memberof V1CheckpointWorkload
     */
    resources?: { [key: string]: string; };
    /**
     * Total number of batches as of this workload's completion.
     * @type {number}
     * @memberof V1CheckpointWorkload
     */
    totalBatches: number;
    /**
     * User defined metadata associated with the checkpoint.
     * @type {any}
     * @memberof V1CheckpointWorkload
     */
    metadata?: any;
}
/**
 * Response to CleanupLogsRequest.
 * @export
 * @interface V1CleanupLogsResponse
 */
export interface V1CleanupLogsResponse {
    /**
     * How many row of logs were removed.
     * @type {string}
     * @memberof V1CleanupLogsResponse
     */
    removedCount: string;
}
/**
 * Active notice from the server admin.
 * @export
 * @interface V1ClusterMessage
 */
export interface V1ClusterMessage {
    /**
     * Text content of message.
     * @type {string}
     * @memberof V1ClusterMessage
     */
    message: string;
    /**
     * Time to begin showing message.
     * @type {Date | DateString}
     * @memberof V1ClusterMessage
     */
    startTime: Date | DateString;
    /**
     * Time to stop showing message.
     * @type {Date | DateString}
     * @memberof V1ClusterMessage
     */
    endTime?: Date | DateString;
    /**
     * Time message was created.
     * @type {Date | DateString}
     * @memberof V1ClusterMessage
     */
    createdTime?: Date | DateString;
}
/**
 * ColumnType indicates the type of data under the column - COLUMN_TYPE_UNSPECIFIED: data type is unknown/mixed  - COLUMN_TYPE_TEXT: data type is textual  - COLUMN_TYPE_NUMBER: data type is numeric  - COLUMN_TYPE_DATE: data type is a date  - COLUMN_TYPE_ARRAY: data type is an array
 * @export
 * @enum {string}
 */
export const V1ColumnType = {
    UNSPECIFIED: 'COLUMN_TYPE_UNSPECIFIED',
    TEXT: 'COLUMN_TYPE_TEXT',
    NUMBER: 'COLUMN_TYPE_NUMBER',
    DATE: 'COLUMN_TYPE_DATE',
    ARRAY: 'COLUMN_TYPE_ARRAY',
} as const
export type V1ColumnType = ValueOf<typeof V1ColumnType>
/**
 * Command is a single container running the configured command.
 * @export
 * @interface V1Command
 */
export interface V1Command {
    /**
     * The id of the command.
     * @type {string}
     * @memberof V1Command
     */
    id: string;
    /**
     * The description of the command.
     * @type {string}
     * @memberof V1Command
     */
    description: string;
    /**
     * The state of the command.
     * @type {Taskv1State}
     * @memberof V1Command
     */
    state: Taskv1State;
    /**
     * The time the command was started.
     * @type {Date | DateString}
     * @memberof V1Command
     */
    startTime: Date | DateString;
    /**
     * The container running the command.
     * @type {V1Container}
     * @memberof V1Command
     */
    container?: V1Container;
    /**
     * The display name of the user that created the command.
     * @type {string}
     * @memberof V1Command
     */
    displayName?: string;
    /**
     * The id of the user that created the command.
     * @type {number}
     * @memberof V1Command
     */
    userId?: number;
    /**
     * The username of the user that created the command.
     * @type {string}
     * @memberof V1Command
     */
    username: string;
    /**
     * The name of the resource pool the command was created in
     * @type {string}
     * @memberof V1Command
     */
    resourcePool: string;
    /**
     * The exit status;
     * @type {string}
     * @memberof V1Command
     */
    exitStatus?: string;
    /**
     * The associated job id.
     * @type {string}
     * @memberof V1Command
     */
    jobId: string;
    /**
     * The workspace id.
     * @type {number}
     * @memberof V1Command
     */
    workspaceId: number;
}
/**
 * Container for a requested trial and its metrics.
 * @export
 * @interface V1ComparableTrial
 */
export interface V1ComparableTrial {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1ComparableTrial
     */
    trial: Trialv1Trial;
    /**
     * The downsampled datapoints.
     * @type {Array<V1DownsampledMetrics>}
     * @memberof V1ComparableTrial
     */
    metrics: Array<V1DownsampledMetrics>;
}
/**
 * Response to CompareTrialsRequest.
 * @export
 * @interface V1CompareTrialsResponse
 */
export interface V1CompareTrialsResponse {
    /**
     * A list of objects containing trial and metrics information.
     * @type {Array<V1ComparableTrial>}
     * @memberof V1CompareTrialsResponse
     */
    trials: Array<V1ComparableTrial>;
}
/**
 * The config to be patched into Master Config.
 * @export
 * @interface V1Config
 */
export interface V1Config {
    /**
     * The log config to be patched into Master Config.
     * @type {V1LogConfig}
     * @memberof V1Config
     */
    log?: V1LogConfig;
}
/**
 * Container is a Docker container that is either scheduled to run or is currently running on a set of slots.
 * @export
 * @interface V1Container
 */
export interface V1Container {
    /**
     * The id of the task that is currently managing this container.
     * @type {string}
     * @memberof V1Container
     */
    parent?: string;
    /**
     * The unique id of this instance of a container.
     * @type {string}
     * @memberof V1Container
     */
    id: string;
    /**
     * The current state that the container is currently in.
     * @type {Containerv1State}
     * @memberof V1Container
     */
    state: Containerv1State;
    /**
     * A list of devices that is being used by this container.
     * @type {Array<V1Device>}
     * @memberof V1Container
     */
    devices?: Array<V1Device>;
    /**
     * User has insufficient permissions to view this container's details. If true, we obfuscate: (1) parent (2) id (4) devices
     * @type {boolean}
     * @memberof V1Container
     */
    permissionDenied?: boolean;
}
/**
 * Request to continue an experiment.
 * @export
 * @interface V1ContinueExperimentRequest
 */
export interface V1ContinueExperimentRequest {
    /**
     * Experiment ID to continue.
     * @type {number}
     * @memberof V1ContinueExperimentRequest
     */
    id: number;
    /**
     * Experiment config (YAML) to merge with the experiment's config.
     * @type {string}
     * @memberof V1ContinueExperimentRequest
     */
    overrideConfig?: string;
}
/**
 * Request to continue an experiment.
 * @export
 * @interface V1ContinueExperimentResponse
 */
export interface V1ContinueExperimentResponse {
    /**
     * The created experiment.
     * @type {V1Experiment}
     * @memberof V1ContinueExperimentResponse
     */
    experiment: V1Experiment;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1ContinueExperimentResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Request to create a new experiment.
 * @export
 * @interface V1CreateExperimentRequest
 */
export interface V1CreateExperimentRequest {
    /**
     * Experiment context.
     * @type {Array<V1File>}
     * @memberof V1CreateExperimentRequest
     */
    modelDefinition?: Array<V1File>;
    /**
     * Experiment config (YAML).
     * @type {string}
     * @memberof V1CreateExperimentRequest
     */
    config?: string;
    /**
     * Only validate instead of creating the experiment. A dry run.
     * @type {boolean}
     * @memberof V1CreateExperimentRequest
     */
    validateOnly?: boolean;
    /**
     * Parent experiment id.
     * @type {number}
     * @memberof V1CreateExperimentRequest
     */
    parentId?: number;
    /**
     * Request to auto-activate the experiment.
     * @type {boolean}
     * @memberof V1CreateExperimentRequest
     */
    activate?: boolean;
    /**
     * Project id to contain the experiment.
     * @type {number}
     * @memberof V1CreateExperimentRequest
     */
    projectId?: number;
    /**
     * Template to use for the experiment.
     * @type {string}
     * @memberof V1CreateExperimentRequest
     */
    template?: string;
    /**
     * Unmanaged experiments are detached.
     * @type {boolean}
     * @memberof V1CreateExperimentRequest
     */
    unmanaged?: boolean;
}
/**
 * Response to CreateExperimentRequest.
 * @export
 * @interface V1CreateExperimentResponse
 */
export interface V1CreateExperimentResponse {
    /**
     * The created experiment.
     * @type {V1Experiment}
     * @memberof V1CreateExperimentResponse
     */
    experiment: V1Experiment;
    /**
     * The created experiment config.
     * @type {any}
     * @memberof V1CreateExperimentResponse
     */
    config: any;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1CreateExperimentResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Request to create a new generic task.
 * @export
 * @interface V1CreateGenericTaskRequest
 */
export interface V1CreateGenericTaskRequest {
    /**
     * Generic task context.
     * @type {Array<V1File>}
     * @memberof V1CreateGenericTaskRequest
     */
    contextDirectory: Array<V1File>;
    /**
     * Generic task config (YAML).
     * @type {string}
     * @memberof V1CreateGenericTaskRequest
     */
    config: string;
    /**
     * Project id to contain the experiment.
     * @type {number}
     * @memberof V1CreateGenericTaskRequest
     */
    projectId?: number;
    /**
     * Parent ID of new task
     * @type {string}
     * @memberof V1CreateGenericTaskRequest
     */
    parentId?: string;
    /**
     * If True inherits the context directory from the paren task (requires parent_id)
     * @type {boolean}
     * @memberof V1CreateGenericTaskRequest
     */
    inheritContext?: boolean;
    /**
     * Id of the task that this task is forked from
     * @type {string}
     * @memberof V1CreateGenericTaskRequest
     */
    forkedFrom?: string;
    /**
     * Flag for whether task can be paused or not.
     * @type {boolean}
     * @memberof V1CreateGenericTaskRequest
     */
    noPause?: boolean;
}
/**
 * Response to CreateExperimentRequest.
 * @export
 * @interface V1CreateGenericTaskResponse
 */
export interface V1CreateGenericTaskResponse {
    /**
     * The created generic taskID.
     * @type {string}
     * @memberof V1CreateGenericTaskResponse
     */
    taskId: string;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1CreateGenericTaskResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * CreateGroupRequest is the body of the request for the call to create a group.
 * @export
 * @interface V1CreateGroupRequest
 */
export interface V1CreateGroupRequest {
    /**
     * The name the new group should have
     * @type {string}
     * @memberof V1CreateGroupRequest
     */
    name: string;
    /**
     * The ids of users that should be added to the new group
     * @type {Array<number>}
     * @memberof V1CreateGroupRequest
     */
    addUsers?: Array<number>;
}
/**
 * CreateGroupResponse is the body of the response for the call to update a group and its members.
 * @export
 * @interface V1CreateGroupResponse
 */
export interface V1CreateGroupResponse {
    /**
     * Info about the group after the update succeeded.
     * @type {V1GroupDetails}
     * @memberof V1CreateGroupResponse
     */
    group: V1GroupDetails;
}
/**
 * Create a trial.
 * @export
 * @interface V1CreateTrialRequest
 */
export interface V1CreateTrialRequest {
    /**
     * The id of the parent experiment.
     * @type {number}
     * @memberof V1CreateTrialRequest
     */
    experimentId?: number;
    /**
     * Trial hyperparameters.
     * @type {any}
     * @memberof V1CreateTrialRequest
     */
    hparams?: any;
    /**
     * Currently only unmanaged trials are supported, must be true.
     * @type {boolean}
     * @memberof V1CreateTrialRequest
     */
    unmanaged?: boolean;
}
/**
 * Response to CreateTrialRequest.
 * @export
 * @interface V1CreateTrialResponse
 */
export interface V1CreateTrialResponse {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1CreateTrialResponse
     */
    trial: Trialv1Trial;
}
/**
 * Response to CurrentUserRequest.
 * @export
 * @interface V1CurrentUserResponse
 */
export interface V1CurrentUserResponse {
    /**
     * The currently logged in user.
     * @type {V1User}
     * @memberof V1CurrentUserResponse
     */
    user: V1User;
}
/**
 * Event data for custom trigger.
 * @export
 * @interface V1CustomWebhookEventData
 */
export interface V1CustomWebhookEventData {
    /**
     * The level at which the event data is logged.
     * @type {V1LogLevel}
     * @memberof V1CustomWebhookEventData
     */
    level?: V1LogLevel;
    /**
     * The title for the event data.
     * @type {string}
     * @memberof V1CustomWebhookEventData
     */
    title?: string;
    /**
     * The description for the event data.
     * @type {string}
     * @memberof V1CustomWebhookEventData
     */
    description?: string;
}
/**
 * One datapoint in a series of metrics from a trial in batch.
 * @export
 * @interface V1DataPoint
 */
export interface V1DataPoint {
    /**
     * Total batches processed by the time this measurement is taken.
     * @type {number}
     * @memberof V1DataPoint
     */
    batches: number;
    /**
     * Values of the requested metrics at this point in the trial.
     * @type {any}
     * @memberof V1DataPoint
     */
    values?: any;
    /**
     * The time the measurement is taken.
     * @type {Date | DateString}
     * @memberof V1DataPoint
     */
    time: Date | DateString;
    /**
     * The epoch this measurement is taken.
     * @type {number}
     * @memberof V1DataPoint
     */
    epoch?: number;
}
/**
 * 
 * @export
 * @interface V1DeleteCheckpointsRequest
 */
export interface V1DeleteCheckpointsRequest {
    /**
     * The list of checkpoint_uuids for the requested checkpoint.
     * @type {Array<string>}
     * @memberof V1DeleteCheckpointsRequest
     */
    checkpointUuids: Array<string>;
}
/**
 * 
 * @export
 * @interface V1DeleteCheckpointsResponse
 */
export interface V1DeleteCheckpointsResponse {
}
/**
 * Response to DeleteClusterMessageRequest.
 * @export
 * @interface V1DeleteClusterMessageResponse
 */
export interface V1DeleteClusterMessageResponse {
}
/**
 * Response to DeleteExperimentLabelRequest.
 * @export
 * @interface V1DeleteExperimentLabelResponse
 */
export interface V1DeleteExperimentLabelResponse {
    /**
     * The complete list of labels associated with the experiment.
     * @type {Array<string>}
     * @memberof V1DeleteExperimentLabelResponse
     */
    labels: Array<string>;
}
/**
 * Response to DeleteExperimentRequest.
 * @export
 * @interface V1DeleteExperimentResponse
 */
export interface V1DeleteExperimentResponse {
}
/**
 * Delete multiple experiments.
 * @export
 * @interface V1DeleteExperimentsRequest
 */
export interface V1DeleteExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1DeleteExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1DeleteExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1DeleteExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to DeleteExperimentsRequest.
 * @export
 * @interface V1DeleteExperimentsResponse
 */
export interface V1DeleteExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1DeleteExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * Response to DeleteGlobalConfigPoliciesRequest.
 * @export
 * @interface V1DeleteGlobalConfigPoliciesResponse
 */
export interface V1DeleteGlobalConfigPoliciesResponse {
}
/**
 * DeleteGroupResponse is the body of the response for the call to delete a group.
 * @export
 * @interface V1DeleteGroupResponse
 */
export interface V1DeleteGroupResponse {
}
/**
 * 
 * @export
 * @interface V1DeleteModelResponse
 */
export interface V1DeleteModelResponse {
}
/**
 * 
 * @export
 * @interface V1DeleteModelVersionResponse
 */
export interface V1DeleteModelVersionResponse {
}
/**
 * Response to DeleteProjectRequest.
 * @export
 * @interface V1DeleteProjectResponse
 */
export interface V1DeleteProjectResponse {
    /**
     * Status of deletion.
     * @type {boolean}
     * @memberof V1DeleteProjectResponse
     */
    completed: boolean;
}
/**
 * Delete runs.
 * @export
 * @interface V1DeleteRunsRequest
 */
export interface V1DeleteRunsRequest {
    /**
     * The ids of the runs being deleted.
     * @type {Array<number>}
     * @memberof V1DeleteRunsRequest
     */
    runIds: Array<number>;
    /**
     * Project id of the runs being deleted.
     * @type {number}
     * @memberof V1DeleteRunsRequest
     */
    projectId?: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1DeleteRunsRequest
     */
    filter?: string;
}
/**
 * Response to DeleteRunsResponse.
 * @export
 * @interface V1DeleteRunsResponse
 */
export interface V1DeleteRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1DeleteRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * Response to DeleteTemplateRequest.
 * @export
 * @interface V1DeleteTemplateResponse
 */
export interface V1DeleteTemplateResponse {
}
/**
 * Response to DeleteTensorboardRequest.
 * @export
 * @interface V1DeleteTensorboardFilesResponse
 */
export interface V1DeleteTensorboardFilesResponse {
}
/**
 * Response to DeleteWebhookRequest.
 * @export
 * @interface V1DeleteWebhookResponse
 */
export interface V1DeleteWebhookResponse {
}
/**
 * Response to DeleteWorkspaceConfigPoliciesRequest.
 * @export
 * @interface V1DeleteWorkspaceConfigPoliciesResponse
 */
export interface V1DeleteWorkspaceConfigPoliciesResponse {
}
/**
 * Response to DeleteWorkspaceNamespaceBindingsRequest.
 * @export
 * @interface V1DeleteWorkspaceNamespaceBindingsResponse
 */
export interface V1DeleteWorkspaceNamespaceBindingsResponse {
}
/**
 * Response to DeleteWorkspaceRequest.
 * @export
 * @interface V1DeleteWorkspaceResponse
 */
export interface V1DeleteWorkspaceResponse {
    /**
     * Status of deletion.
     * @type {boolean}
     * @memberof V1DeleteWorkspaceResponse
     */
    completed: boolean;
}
/**
 * Device represents a single computational device on an agent.
 * @export
 * @interface V1Device
 */
export interface V1Device {
    /**
     * The index of the device.
     * @type {number}
     * @memberof V1Device
     */
    id?: number;
    /**
     * The brand name of the device.
     * @type {string}
     * @memberof V1Device
     */
    brand?: string;
    /**
     * The unique UUID of the device.
     * @type {string}
     * @memberof V1Device
     */
    uuid?: string;
    /**
     * The type of the Device.
     * @type {Devicev1Type}
     * @memberof V1Device
     */
    type?: Devicev1Type;
}
/**
 * DeviceStats contains statistics about a single device group.
 * @export
 * @interface V1DeviceStats
 */
export interface V1DeviceStats {
    /**
     * The number of slots in each state if there's an associated container.
     * @type {{ [key: string]: number; }}
     * @memberof V1DeviceStats
     */
    states?: { [key: string]: number; };
    /**
     * the number of draining slots.
     * @type {number}
     * @memberof V1DeviceStats
     */
    draining: number;
    /**
     * the number of disabled slots.
     * @type {number}
     * @memberof V1DeviceStats
     */
    disabled: number;
    /**
     * the total number of slots.
     * @type {number}
     * @memberof V1DeviceStats
     */
    total: number;
}
/**
 * Disable the agent.
 * @export
 * @interface V1DisableAgentRequest
 */
export interface V1DisableAgentRequest {
    /**
     * The id of the agent.
     * @type {string}
     * @memberof V1DisableAgentRequest
     */
    agentId?: string;
    /**
     * If true, wait for running tasks to finish.
     * @type {boolean}
     * @memberof V1DisableAgentRequest
     */
    drain?: boolean;
}
/**
 * Response to DisableAgentRequest.
 * @export
 * @interface V1DisableAgentResponse
 */
export interface V1DisableAgentResponse {
    /**
     * The disabled agent.
     * @type {V1Agent}
     * @memberof V1DisableAgentResponse
     */
    agent?: V1Agent;
}
/**
 * Disable the slot.
 * @export
 * @interface V1DisableSlotRequest
 */
export interface V1DisableSlotRequest {
    /**
     * The id of the agent.
     * @type {string}
     * @memberof V1DisableSlotRequest
     */
    agentId?: string;
    /**
     * The id of the slot.
     * @type {string}
     * @memberof V1DisableSlotRequest
     */
    slotId?: string;
    /**
     * If true, wait for running task to finish.
     * @type {boolean}
     * @memberof V1DisableSlotRequest
     */
    drain?: boolean;
}
/**
 * Response to DisableSlotRequest.
 * @export
 * @interface V1DisableSlotResponse
 */
export interface V1DisableSlotResponse {
    /**
     * The disabled slot.
     * @type {V1Slot}
     * @memberof V1DisableSlotResponse
     */
    slot?: V1Slot;
}
/**
 * Double filters.
 * @export
 * @interface V1DoubleFieldFilter
 */
export interface V1DoubleFieldFilter {
    /**
     * Less than.
     * @type {number}
     * @memberof V1DoubleFieldFilter
     */
    lt?: number;
    /**
     * Less than or equal.
     * @type {number}
     * @memberof V1DoubleFieldFilter
     */
    lte?: number;
    /**
     * Greater than.
     * @type {number}
     * @memberof V1DoubleFieldFilter
     */
    gt?: number;
    /**
     * Greater than or equal.
     * @type {number}
     * @memberof V1DoubleFieldFilter
     */
    gte?: number;
}
/**
 * DownsampledMetrics captures a metric's name and downsampled data points.
 * @export
 * @interface V1DownsampledMetrics
 */
export interface V1DownsampledMetrics {
    /**
     * A possibly down-sampled series of metrics' readings through the progress of the trial.
     * @type {Array<V1DataPoint>}
     * @memberof V1DownsampledMetrics
     */
    data: Array<V1DataPoint>;
    /**
     * Metric group (training, validation, or unset).
     * @type {V1MetricType}
     * @memberof V1DownsampledMetrics
     */
    type: V1MetricType;
    /**
     * Metric group (training, validation, etc).
     * @type {string}
     * @memberof V1DownsampledMetrics
     */
    group: string;
}
/**
 * Response to EnableAgentRequest.
 * @export
 * @interface V1EnableAgentResponse
 */
export interface V1EnableAgentResponse {
    /**
     * The enabled agent.
     * @type {V1Agent}
     * @memberof V1EnableAgentResponse
     */
    agent?: V1Agent;
}
/**
 * Response to EnableSlotRequest.
 * @export
 * @interface V1EnableSlotResponse
 */
export interface V1EnableSlotResponse {
    /**
     * The enabled slot.
     * @type {V1Slot}
     * @memberof V1EnableSlotResponse
     */
    slot?: V1Slot;
}
/**
 * EntityType represents an entity - ENTITY_TYPE_UNSPECIFIED: Default entity type.  - ENTITY_TYPE_PROJECT: Represents a project.
 * @export
 * @enum {string}
 */
export const V1EntityType = {
    UNSPECIFIED: 'ENTITY_TYPE_UNSPECIFIED',
    PROJECT: 'ENTITY_TYPE_PROJECT',
} as const
export type V1EntityType = ValueOf<typeof V1EntityType>
/**
 * Experiment is a collection of one or more trials that are exploring a user-defined hyperparameter space.
 * @export
 * @interface V1Experiment
 */
export interface V1Experiment {
    /**
     * The id of the experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    id: number;
    /**
     * The description of the experiment.
     * @type {string}
     * @memberof V1Experiment
     */
    description?: string;
    /**
     * Labels attached to the experiment.
     * @type {Array<string>}
     * @memberof V1Experiment
     */
    labels?: Array<string>;
    /**
     * The time the experiment was started.
     * @type {Date | DateString}
     * @memberof V1Experiment
     */
    startTime: Date | DateString;
    /**
     * The time the experiment ended if the experiment is stopped.
     * @type {Date | DateString}
     * @memberof V1Experiment
     */
    endTime?: Date | DateString;
    /**
     * The current state of the experiment.
     * @type {Experimentv1State}
     * @memberof V1Experiment
     */
    state: Experimentv1State;
    /**
     * Boolean denoting whether the experiment was archived.
     * @type {boolean}
     * @memberof V1Experiment
     */
    archived: boolean;
    /**
     * The number of trials linked to the experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    numTrials: number;
    /**
     * The ids of trials linked to the experiment.
     * @type {Array<number>}
     * @memberof V1Experiment
     */
    trialIds?: Array<number>;
    /**
     * The display name of the user that created the experiment.
     * @type {string}
     * @memberof V1Experiment
     */
    displayName?: string;
    /**
     * The id of the user that created the experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    userId?: number;
    /**
     * The username of the user that created the experiment.
     * @type {string}
     * @memberof V1Experiment
     */
    username: string;
    /**
     * The resource pool the experiment was created in
     * @type {string}
     * @memberof V1Experiment
     */
    resourcePool?: string;
    /**
     * The type of searcher for the experiment
     * @type {string}
     * @memberof V1Experiment
     */
    searcherType: string;
    /**
     * The searcher metric name for the experiment
     * @type {string}
     * @memberof V1Experiment
     */
    searcherMetric?: string;
    /**
     * The hyperparameters for the experiment
     * @type {any}
     * @memberof V1Experiment
     */
    hyperparameters?: any;
    /**
     * The experiment name.
     * @type {string}
     * @memberof V1Experiment
     */
    name: string;
    /**
     * The experiment notes.
     * @type {string}
     * @memberof V1Experiment
     */
    notes?: string;
    /**
     * Associated job's id.
     * @type {string}
     * @memberof V1Experiment
     */
    jobId: string;
    /**
     * Original id of a forked or continued experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    forkedFrom?: number;
    /**
     * The current progress of a running experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    progress?: number;
    /**
     * The id of the project associated with this experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    projectId: number;
    /**
     * The name of the project associated with this experiment.
     * @type {string}
     * @memberof V1Experiment
     */
    projectName?: string;
    /**
     * The id of the workspace associated with this experiment.
     * @type {number}
     * @memberof V1Experiment
     */
    workspaceId?: number;
    /**
     * The name of the workspace associated with this experiment.
     * @type {string}
     * @memberof V1Experiment
     */
    workspaceName?: string;
    /**
     * The archived status of the parent project (can be inherited from workspace).
     * @type {boolean}
     * @memberof V1Experiment
     */
    parentArchived?: boolean;
    /**
     * The configuration of the experiment. Is deprecated for performance reasons on the listing experiment route. Use GetExperimentResponse.config instead.
     * @type {any}
     * @memberof V1Experiment
     */
    config: any;
    /**
     * The original configuration that the user submitted.
     * @type {string}
     * @memberof V1Experiment
     */
    originalConfig: string;
    /**
     * The id of the user who created the parent project.
     * @type {number}
     * @memberof V1Experiment
     */
    projectOwnerId: number;
    /**
     * The total size of checkpoints.
     * @type {string}
     * @memberof V1Experiment
     */
    checkpointSize?: string;
    /**
     * The count of checkpoints.
     * @type {number}
     * @memberof V1Experiment
     */
    checkpointCount?: number;
    /**
     * The metrics and hyperparameters associated with the best trial by searcher metric.
     * @type {number}
     * @memberof V1Experiment
     */
    bestTrialSearcherMetric?: number;
    /**
     * Id of experiment's best trial, calculated by the best searcher metrics value of trial's best validation.
     * @type {number}
     * @memberof V1Experiment
     */
    bestTrialId?: number;
    /**
     * Unmanaged experiments are detached.
     * @type {boolean}
     * @memberof V1Experiment
     */
    unmanaged?: boolean;
    /**
     * Time in seconds which experiment ran or has been running.
     * @type {number}
     * @memberof V1Experiment
     */
    duration?: number;
    /**
     * The id of external experiment
     * @type {string}
     * @memberof V1Experiment
     */
    externalExperimentId?: string;
    /**
     * The id of external trial
     * @type {string}
     * @memberof V1Experiment
     */
    externalTrialId?: string;
    /**
     * Size of model definition file, for unmanaged experiments this should be 0.
     * @type {number}
     * @memberof V1Experiment
     */
    modelDefinitionSize?: number;
    /**
     * The experiment pachyderm integration config.
     * @type {any}
     * @memberof V1Experiment
     */
    pachydermIntegration?: any;
}
/**
 * Message for results of individual experiments in a multi-experiment action.
 * @export
 * @interface V1ExperimentActionResult
 */
export interface V1ExperimentActionResult {
    /**
     * Optional error message.
     * @type {string}
     * @memberof V1ExperimentActionResult
     */
    error: string;
    /**
     * Experiment ID.
     * @type {number}
     * @memberof V1ExperimentActionResult
     */
    id: number;
}
/**
 * Response to ExpMetricNamesRequest.
 * @export
 * @interface V1ExpMetricNamesResponse
 */
export interface V1ExpMetricNamesResponse {
    /**
     * The names of the searcher metrics.
     * @type {Array<string>}
     * @memberof V1ExpMetricNamesResponse
     */
    searcherMetrics?: Array<string>;
    /**
     * List of training metric names.
     * @type {Array<string>}
     * @memberof V1ExpMetricNamesResponse
     */
    trainingMetrics?: Array<string>;
    /**
     * List of validation metric names.
     * @type {Array<string>}
     * @memberof V1ExpMetricNamesResponse
     */
    validationMetrics?: Array<string>;
    /**
     * List of metric group-name pairs.
     * @type {Array<V1MetricIdentifier>}
     * @memberof V1ExpMetricNamesResponse
     */
    metricNames?: Array<V1MetricIdentifier>;
}
/**
 * The failure type of a resource.   - FAILURE_TYPE_UNSPECIFIED: UNSPECIFIED denotes an error that is not defined below.  - FAILURE_TYPE_RESOURCES_FAILED: ResourcesFailed denotes that the container ran but failed with a non-zero exit code.  - FAILURE_TYPE_RESOURCES_ABORTED: ResourcesAborted denotes the container was canceled before it was started.  - FAILURE_TYPE_RESOURCES_MISSING: ResourcesMissing denotes the resources were missing when the master asked about it.  - FAILURE_TYPE_TASK_ABORTED: TaskAborted denotes that the task was canceled before it was started.  - FAILURE_TYPE_TASK_ERROR: TaskError denotes that the task failed without an associated exit code.  - FAILURE_TYPE_AGENT_FAILED: AgentFailed denotes that the agent failed while the container was running.  - FAILURE_TYPE_AGENT_ERROR: AgentError denotes that the agent failed to launch the container.  - FAILURE_TYPE_RESTORE_ERROR: RestoreError denotes a failure to restore a running allocation on master blip.  - FAILURE_TYPE_UNKNOWN_ERROR: UnknownError denotes an internal error that did not map to a know failure type.
 * @export
 * @enum {string}
 */
export const V1FailureType = {
    UNSPECIFIED: 'FAILURE_TYPE_UNSPECIFIED',
    RESOURCESFAILED: 'FAILURE_TYPE_RESOURCES_FAILED',
    RESOURCESABORTED: 'FAILURE_TYPE_RESOURCES_ABORTED',
    RESOURCESMISSING: 'FAILURE_TYPE_RESOURCES_MISSING',
    TASKABORTED: 'FAILURE_TYPE_TASK_ABORTED',
    TASKERROR: 'FAILURE_TYPE_TASK_ERROR',
    AGENTFAILED: 'FAILURE_TYPE_AGENT_FAILED',
    AGENTERROR: 'FAILURE_TYPE_AGENT_ERROR',
    RESTOREERROR: 'FAILURE_TYPE_RESTORE_ERROR',
    UNKNOWNERROR: 'FAILURE_TYPE_UNKNOWN_ERROR',
} as const
export type V1FailureType = ValueOf<typeof V1FailureType>
/**
 * File is a Unix file.
 * @export
 * @interface V1File
 */
export interface V1File {
    /**
     * Path to the file.
     * @type {string}
     * @memberof V1File
     */
    path: string;
    /**
     * File type.
     * @type {number}
     * @memberof V1File
     */
    type: number;
    /**
     * base64 encoded contents of the file.
     * @type {string}
     * @memberof V1File
     */
    content: string;
    /**
     * Modified time (Unix timestamp).
     * @type {string}
     * @memberof V1File
     */
    mtime: string;
    /**
     * File mode.
     * @type {number}
     * @memberof V1File
     */
    mode: number;
    /**
     * User ID.
     * @type {number}
     * @memberof V1File
     */
    uid: number;
    /**
     * Group ID.
     * @type {number}
     * @memberof V1File
     */
    gid: number;
}
/**
 * File node is one node of file in experiment model definition file tree.
 * @export
 * @interface V1FileNode
 */
export interface V1FileNode {
    /**
     * Path of file.
     * @type {string}
     * @memberof V1FileNode
     */
    path?: string;
    /**
     * Name of file.
     * @type {string}
     * @memberof V1FileNode
     */
    name?: string;
    /**
     * Modification time of file.
     * @type {Date | DateString}
     * @memberof V1FileNode
     */
    modifiedTime?: Date | DateString;
    /**
     * Number of bytes in file content.
     * @type {number}
     * @memberof V1FileNode
     */
    contentLength?: number;
    /**
     * Is this a directory.
     * @type {boolean}
     * @memberof V1FileNode
     */
    isDir?: boolean;
    /**
     * MIME type of file.
     * @type {string}
     * @memberof V1FileNode
     */
    contentType?: string;
    /**
     * Subdirectory files.
     * @type {Array<V1FileNode>}
     * @memberof V1FileNode
     */
    files?: Array<V1FileNode>;
}
/**
 * The fitting policy of the scheduler.   - FITTING_POLICY_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - FITTING_POLICY_BEST: Best fit. Tasks are preferentially “packed” together on the smallest number of agents  - FITTING_POLICY_WORST: Worst fit. Tasks are placed on under-utilized agents, spreading out the tasks.  - FITTING_POLICY_KUBERNETES: A kubernetes placeholder. In k8s, the task placement is delegated to the k8s scheduler so the fitting policy is not relevant.  - FITTING_POLICY_SLURM: A slurm placeholder. When running on slurm, task placement is delegated.  - FITTING_POLICY_PBS: A PBS placeholder. When running on PBS, task placement is delegated.
 * @export
 * @enum {string}
 */
export const V1FittingPolicy = {
    UNSPECIFIED: 'FITTING_POLICY_UNSPECIFIED',
    BEST: 'FITTING_POLICY_BEST',
    WORST: 'FITTING_POLICY_WORST',
    KUBERNETES: 'FITTING_POLICY_KUBERNETES',
    SLURM: 'FITTING_POLICY_SLURM',
    PBS: 'FITTING_POLICY_PBS',
} as const
export type V1FittingPolicy = ValueOf<typeof V1FittingPolicy>
/**
 * Flat run respresentation. Used for the rows of the Run Table.
 * @export
 * @interface V1FlatRun
 */
export interface V1FlatRun {
    /**
     * The id of the run.
     * @type {number}
     * @memberof V1FlatRun
     */
    id: number;
    /**
     * The time the run was started.
     * @type {Date | DateString}
     * @memberof V1FlatRun
     */
    startTime: Date | DateString;
    /**
     * The time the run ended.
     * @type {Date | DateString}
     * @memberof V1FlatRun
     */
    endTime?: Date | DateString;
    /**
     * The current state of the run(trial).
     * @type {Trialv1State}
     * @memberof V1FlatRun
     */
    state: Trialv1State;
    /**
     * The tags of the associated experiment. TODO(aaron.amanuel): Create add/remove tags for runs.
     * @type {Array<string>}
     * @memberof V1FlatRun
     */
    labels?: Array<string>;
    /**
     * The total size of checkpoints.
     * @type {string}
     * @memberof V1FlatRun
     */
    checkpointSize: string;
    /**
     * The count of checkpoints.
     * @type {number}
     * @memberof V1FlatRun
     */
    checkpointCount: number;
    /**
     * Signed searcher metrics value.
     * @type {number}
     * @memberof V1FlatRun
     */
    searcherMetricValue?: number;
    /**
     * The id of external run
     * @type {string}
     * @memberof V1FlatRun
     */
    externalRunId?: string;
    /**
     * Trial hyperparameters.
     * @type {any}
     * @memberof V1FlatRun
     */
    hyperparameters?: any;
    /**
     * summary metrics.
     * @type {any}
     * @memberof V1FlatRun
     */
    summaryMetrics?: any;
    /**
     * The id of the user who created the run.
     * @type {number}
     * @memberof V1FlatRun
     */
    userId?: number;
    /**
     * Time in seconds which the run ran or has been running.
     * @type {number}
     * @memberof V1FlatRun
     */
    duration?: number;
    /**
     * The id of the project associated with this run.
     * @type {number}
     * @memberof V1FlatRun
     */
    projectId: number;
    /**
     * The name of the project associated with this run.
     * @type {string}
     * @memberof V1FlatRun
     */
    projectName: string;
    /**
     * The id of the workspace associated with this run.
     * @type {number}
     * @memberof V1FlatRun
     */
    workspaceId: number;
    /**
     * The name of the workspace associated with this run.
     * @type {string}
     * @memberof V1FlatRun
     */
    workspaceName: string;
    /**
     * The archived status of the parent project (can be inherited from workspace).
     * @type {boolean}
     * @memberof V1FlatRun
     */
    parentArchived: boolean;
    /**
     * Data related the the experiment associated with this run.
     * @type {V1FlatRunExperiment}
     * @memberof V1FlatRun
     */
    experiment?: V1FlatRunExperiment;
    /**
     * The arbitrary metadata of the run.
     * @type {any}
     * @memberof V1FlatRun
     */
    metadata?: any;
    /**
     * The archived status of this run. This is only looking at the archived status at the run level and not taking into account whether the experiment is archived or not.
     * @type {boolean}
     * @memberof V1FlatRun
     */
    archived: boolean;
    /**
     * Project level local id of run.
     * @type {string}
     * @memberof V1FlatRun
     */
    localId?: string;
    /**
     * Log policy matched.
     * @type {string}
     * @memberof V1FlatRun
     */
    logPolicyMatched?: string;
}
/**
 * 
 * @export
 * @interface V1FlatRunExperiment
 */
export interface V1FlatRunExperiment {
    /**
     * The id of the experiment linked to the run.
     * @type {number}
     * @memberof V1FlatRunExperiment
     */
    id: number;
    /**
     * The type of searcher for the experiment.
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    searcherType: string;
    /**
     * The searcher metric name for the experiment.
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    searcherMetric: string;
    /**
     * Original id of a forked or continued experiment.
     * @type {number}
     * @memberof V1FlatRunExperiment
     */
    forkedFrom?: number;
    /**
     * The id of external experiment
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    externalExperimentId?: string;
    /**
     * The resource pool the experiment was created in.
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    resourcePool: string;
    /**
     * The current progress of a running experiment.
     * @type {number}
     * @memberof V1FlatRunExperiment
     */
    progress: number;
    /**
     * The description of the experiment.
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    description: string;
    /**
     * The experiment name.
     * @type {string}
     * @memberof V1FlatRunExperiment
     */
    name: string;
    /**
     * Unmanaged experiments are detached.
     * @type {boolean}
     * @memberof V1FlatRunExperiment
     */
    unmanaged: boolean;
    /**
     * True if the associated experiment is a multitrial experiment
     * @type {boolean}
     * @memberof V1FlatRunExperiment
     */
    isMultitrial: boolean;
    /**
     * The experiment pachyderm integration config.
     * @type {any}
     * @memberof V1FlatRunExperiment
     */
    pachydermIntegration?: any;
}
/**
 * State of a Generic task - GENERIC_TASK_STATE_UNSPECIFIED: The task state unknown  - GENERIC_TASK_STATE_ACTIVE: The task state unknown  - GENERIC_TASK_STATE_CANCELED: The task state unknown  - GENERIC_TASK_STATE_COMPLETED: The task state unknown  - GENERIC_TASK_STATE_ERROR: The task state unknown  - GENERIC_TASK_STATE_PAUSED: The task state unknown  - GENERIC_TASK_STATE_STOPPING_PAUSED: The task state unknown  - GENERIC_TASK_STATE_STOPPING_CANCELED: The task state unknown  - GENERIC_TASK_STATE_STOPPING_COMPLETED: The task state unknown  - GENERIC_TASK_STATE_STOPPING_ERROR: The task state unknown
 * @export
 * @enum {string}
 */
export const V1GenericTaskState = {
    UNSPECIFIED: 'GENERIC_TASK_STATE_UNSPECIFIED',
    ACTIVE: 'GENERIC_TASK_STATE_ACTIVE',
    CANCELED: 'GENERIC_TASK_STATE_CANCELED',
    COMPLETED: 'GENERIC_TASK_STATE_COMPLETED',
    ERROR: 'GENERIC_TASK_STATE_ERROR',
    PAUSED: 'GENERIC_TASK_STATE_PAUSED',
    STOPPINGPAUSED: 'GENERIC_TASK_STATE_STOPPING_PAUSED',
    STOPPINGCANCELED: 'GENERIC_TASK_STATE_STOPPING_CANCELED',
    STOPPINGCOMPLETED: 'GENERIC_TASK_STATE_STOPPING_COMPLETED',
    STOPPINGERROR: 'GENERIC_TASK_STATE_STOPPING_ERROR',
} as const
export type V1GenericTaskState = ValueOf<typeof V1GenericTaskState>
/**
 * Sort token info by the given field.   - SORT_BY_UNSPECIFIED: Returns token info in an unsorted list.  - SORT_BY_USER_ID: Returns token info sorted by user id.  - SORT_BY_EXPIRY: Returns token info sorted by expiry.  - SORT_BY_CREATED_AT: Returns token info sorted by created at.  - SORT_BY_TOKEN_TYPE: Returns token info sorted by token type.  - SORT_BY_REVOKED: Returns token info sorted by if it is revoked.  - SORT_BY_DESCRIPTION: Returns token info sorted by description of token.
 * @export
 * @enum {string}
 */
export const V1GetAccessTokensRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    USERID: 'SORT_BY_USER_ID',
    EXPIRY: 'SORT_BY_EXPIRY',
    CREATEDAT: 'SORT_BY_CREATED_AT',
    TOKENTYPE: 'SORT_BY_TOKEN_TYPE',
    REVOKED: 'SORT_BY_REVOKED',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
} as const
export type V1GetAccessTokensRequestSortBy = ValueOf<typeof V1GetAccessTokensRequestSortBy>
/**
 * Response to GetAccessTokensRequest.
 * @export
 * @interface V1GetAccessTokensResponse
 */
export interface V1GetAccessTokensResponse {
    /**
     * List of token information.
     * @type {Array<V1TokenInfo>}
     * @memberof V1GetAccessTokensResponse
     */
    tokenInfo: Array<V1TokenInfo>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetAccessTokensResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetActiveTasksCountRequest.
 * @export
 * @interface V1GetActiveTasksCountResponse
 */
export interface V1GetActiveTasksCountResponse {
    /**
     * The count of commands.
     * @type {number}
     * @memberof V1GetActiveTasksCountResponse
     */
    commands: number;
    /**
     * The count of notebooks.
     * @type {number}
     * @memberof V1GetActiveTasksCountResponse
     */
    notebooks: number;
    /**
     * The count of shells.
     * @type {number}
     * @memberof V1GetActiveTasksCountResponse
     */
    shells: number;
    /**
     * The count of TensorBoards.
     * @type {number}
     * @memberof V1GetActiveTasksCountResponse
     */
    tensorboards: number;
}
/**
 * Response to GetAgentRequest.
 * @export
 * @interface V1GetAgentResponse
 */
export interface V1GetAgentResponse {
    /**
     * The requested agent.
     * @type {V1Agent}
     * @memberof V1GetAgentResponse
     */
    agent: V1Agent;
}
/**
 * Sorts agents by the given field.   - SORT_BY_UNSPECIFIED: Returns agents in an unsorted list.  - SORT_BY_ID: Returns agents sorted by id.  - SORT_BY_TIME: Returns agents sorted by time.
 * @export
 * @enum {string}
 */
export const V1GetAgentsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    TIME: 'SORT_BY_TIME',
} as const
export type V1GetAgentsRequestSortBy = ValueOf<typeof V1GetAgentsRequestSortBy>
/**
 * Response to GetAgentsRequest.
 * @export
 * @interface V1GetAgentsResponse
 */
export interface V1GetAgentsResponse {
    /**
     * The list of returned agents.
     * @type {Array<V1Agent>}
     * @memberof V1GetAgentsResponse
     */
    agents: Array<V1Agent>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetAgentsResponse
     */
    pagination?: V1Pagination;
}
/**
 * 
 * @export
 * @interface V1GetAllocationResponse
 */
export interface V1GetAllocationResponse {
    /**
     * The id of the allocation.
     * @type {V1Allocation}
     * @memberof V1GetAllocationResponse
     */
    allocation: V1Allocation;
}
/**
 * Response to GetBestSearcherValidationMetricRequest.
 * @export
 * @interface V1GetBestSearcherValidationMetricResponse
 */
export interface V1GetBestSearcherValidationMetricResponse {
    /**
     * The value of the metric.
     * @type {number}
     * @memberof V1GetBestSearcherValidationMetricResponse
     */
    metric?: number;
}
/**
 * Response to GetCheckpointRequest.
 * @export
 * @interface V1GetCheckpointResponse
 */
export interface V1GetCheckpointResponse {
    /**
     * The requested checkpoint.
     * @type {V1Checkpoint}
     * @memberof V1GetCheckpointResponse
     */
    checkpoint: V1Checkpoint;
}
/**
 * GetClusterMessageResponse is the response that contains the current cluster message.
 * @export
 * @interface V1GetClusterMessageResponse
 */
export interface V1GetClusterMessageResponse {
    /**
     * cluster_message is the current cluster message.
     * @type {V1ClusterMessage}
     * @memberof V1GetClusterMessageResponse
     */
    clusterMessage?: V1ClusterMessage;
}
/**
 * Response to GetCommandRequest.
 * @export
 * @interface V1GetCommandResponse
 */
export interface V1GetCommandResponse {
    /**
     * The requested command.
     * @type {V1Command}
     * @memberof V1GetCommandResponse
     */
    command: V1Command;
    /**
     * The command config.
     * @type {any}
     * @memberof V1GetCommandResponse
     */
    config: any;
}
/**
 * Sorts commands by the given field.   - SORT_BY_UNSPECIFIED: Returns commands in an unsorted list.  - SORT_BY_ID: Returns commands sorted by id.  - SORT_BY_DESCRIPTION: Returns commands sorted by description.  - SORT_BY_START_TIME: Return commands sorted by start time.  - SORT_BY_WORKSPACE_ID: Return commands sorted by workspace_id.
 * @export
 * @enum {string}
 */
export const V1GetCommandsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    STARTTIME: 'SORT_BY_START_TIME',
    WORKSPACEID: 'SORT_BY_WORKSPACE_ID',
} as const
export type V1GetCommandsRequestSortBy = ValueOf<typeof V1GetCommandsRequestSortBy>
/**
 * Response to GetCommandsRequest.
 * @export
 * @interface V1GetCommandsResponse
 */
export interface V1GetCommandsResponse {
    /**
     * The list of returned commands.
     * @type {Array<V1Command>}
     * @memberof V1GetCommandsResponse
     */
    commands: Array<V1Command>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetCommandsResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetExperimentCheckpointsRequest.
 * @export
 * @interface V1GetExperimentCheckpointsResponse
 */
export interface V1GetExperimentCheckpointsResponse {
    /**
     * The list of returned checkpoints.
     * @type {Array<V1Checkpoint>}
     * @memberof V1GetExperimentCheckpointsResponse
     */
    checkpoints: Array<V1Checkpoint>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetExperimentCheckpointsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetExperimentsLabelsRequest.
 * @export
 * @interface V1GetExperimentLabelsResponse
 */
export interface V1GetExperimentLabelsResponse {
    /**
     * The list of labels used across all experiments.
     * @type {Array<string>}
     * @memberof V1GetExperimentLabelsResponse
     */
    labels?: Array<string>;
}
/**
 * Response to GetExperimentRequest.
 * @export
 * @interface V1GetExperimentResponse
 */
export interface V1GetExperimentResponse {
    /**
     * The requested experiment.
     * @type {V1Experiment}
     * @memberof V1GetExperimentResponse
     */
    experiment: V1Experiment;
    /**
     * Associated job summary.
     * @type {V1JobSummary}
     * @memberof V1GetExperimentResponse
     */
    jobSummary?: V1JobSummary;
    /**
     * The experiment's config.
     * @type {any}
     * @memberof V1GetExperimentResponse
     */
    config?: any;
}
/**
 * Sorts experiments by the given field.   - SORT_BY_UNSPECIFIED: Returns experiments in an unsorted list.  - SORT_BY_ID: Returns experiments sorted by id.  - SORT_BY_DESCRIPTION: Returns experiments sorted by description.  - SORT_BY_START_TIME: Return experiments sorted by start time.  - SORT_BY_END_TIME: Return experiments sorted by end time. Experiments without end_time are returned after the ones with end_time.  - SORT_BY_STATE: Return experiments sorted by state.  - SORT_BY_NUM_TRIALS: Return experiments sorted by number of trials.  - SORT_BY_PROGRESS: Return experiments sorted by progress.  - SORT_BY_USER: Return experiments sorted by user.  - SORT_BY_NAME: Returns experiments sorted by name.  - SORT_BY_FORKED_FROM: Returns experiments sorted by originating model.  - SORT_BY_RESOURCE_POOL: Returns experiments sorted by resource pool.  - SORT_BY_PROJECT_ID: Returns experiments sorted by project.  - SORT_BY_CHECKPOINT_SIZE: Returns experiments sorted by checkpoint size.  - SORT_BY_CHECKPOINT_COUNT: Returns experiments sorted by checkpoint count.  - SORT_BY_SEARCHER_METRIC_VAL: Returns experiments sorted by searcher metric value..
 * @export
 * @enum {string}
 */
export const V1GetExperimentsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    STARTTIME: 'SORT_BY_START_TIME',
    ENDTIME: 'SORT_BY_END_TIME',
    STATE: 'SORT_BY_STATE',
    NUMTRIALS: 'SORT_BY_NUM_TRIALS',
    PROGRESS: 'SORT_BY_PROGRESS',
    USER: 'SORT_BY_USER',
    NAME: 'SORT_BY_NAME',
    FORKEDFROM: 'SORT_BY_FORKED_FROM',
    RESOURCEPOOL: 'SORT_BY_RESOURCE_POOL',
    PROJECTID: 'SORT_BY_PROJECT_ID',
    CHECKPOINTSIZE: 'SORT_BY_CHECKPOINT_SIZE',
    CHECKPOINTCOUNT: 'SORT_BY_CHECKPOINT_COUNT',
    SEARCHERMETRICVAL: 'SORT_BY_SEARCHER_METRIC_VAL',
} as const
export type V1GetExperimentsRequestSortBy = ValueOf<typeof V1GetExperimentsRequestSortBy>
/**
 * Response to GetExperimentsRequest.
 * @export
 * @interface V1GetExperimentsResponse
 */
export interface V1GetExperimentsResponse {
    /**
     * The list of returned experiments.
     * @type {Array<V1Experiment>}
     * @memberof V1GetExperimentsResponse
     */
    experiments: Array<V1Experiment>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetExperimentsResponse
     */
    pagination: V1Pagination;
}
/**
 * Sorts trials by the given field.   - SORT_BY_UNSPECIFIED: Returns trials in an unsorted list.  - SORT_BY_ID: Returns trials sorted by id.  - SORT_BY_START_TIME: Return trials sorted by start time.  - SORT_BY_END_TIME: Return trials sorted by end time. Trials without end times are returned after trials that are.  - SORT_BY_STATE: Return trials sorted by state.  - SORT_BY_BEST_VALIDATION_METRIC: Return the trials sorted by the best metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_LATEST_VALIDATION_METRIC: Return the trials sorted by the latest metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_BATCHES_PROCESSED: Return the trials sorted by the number of batches completed.  - SORT_BY_DURATION: Return the trials sorted by the total duration.  - SORT_BY_RESTARTS: Return the trials sorted by the number of restarts.  - SORT_BY_CHECKPOINT_SIZE: Return the trials sorted by checkpoint size.  - SORT_BY_LOG_RETENTION_DAYS: Return the trials sorted by number of log retention days.
 * @export
 * @enum {string}
 */
export const V1GetExperimentTrialsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    STARTTIME: 'SORT_BY_START_TIME',
    ENDTIME: 'SORT_BY_END_TIME',
    STATE: 'SORT_BY_STATE',
    BESTVALIDATIONMETRIC: 'SORT_BY_BEST_VALIDATION_METRIC',
    LATESTVALIDATIONMETRIC: 'SORT_BY_LATEST_VALIDATION_METRIC',
    BATCHESPROCESSED: 'SORT_BY_BATCHES_PROCESSED',
    DURATION: 'SORT_BY_DURATION',
    RESTARTS: 'SORT_BY_RESTARTS',
    CHECKPOINTSIZE: 'SORT_BY_CHECKPOINT_SIZE',
    LOGRETENTIONDAYS: 'SORT_BY_LOG_RETENTION_DAYS',
} as const
export type V1GetExperimentTrialsRequestSortBy = ValueOf<typeof V1GetExperimentTrialsRequestSortBy>
/**
 * Response to GetExperimentTrialsRequest.
 * @export
 * @interface V1GetExperimentTrialsResponse
 */
export interface V1GetExperimentTrialsResponse {
    /**
     * The list of returned trials.
     * @type {Array<Trialv1Trial>}
     * @memberof V1GetExperimentTrialsResponse
     */
    trials: Array<Trialv1Trial>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetExperimentTrialsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetExperimentValidationHistoryRequest.
 * @export
 * @interface V1GetExperimentValidationHistoryResponse
 */
export interface V1GetExperimentValidationHistoryResponse {
    /**
     * validation_history is a collection of zero or more validation metrics for an experiment, describing the best metrics as they were reported through the lifetime of an experiment. The historical list of best validations.
     * @type {Array<V1ValidationHistoryEntry>}
     * @memberof V1GetExperimentValidationHistoryResponse
     */
    validationHistory?: Array<V1ValidationHistoryEntry>;
}
/**
 * 
 * @export
 * @interface V1GetGenericTaskConfigResponse
 */
export interface V1GetGenericTaskConfigResponse {
    /**
     * The config of the task
     * @type {string}
     * @memberof V1GetGenericTaskConfigResponse
     */
    config: string;
}
/**
 * Response to GetGlobalConfigPoliciesRequest.
 * @export
 * @interface V1GetGlobalConfigPoliciesResponse
 */
export interface V1GetGlobalConfigPoliciesResponse {
    /**
     * The global current config policies saved. Contains both invariant configs and constraints in yaml or json format.
     * @type {any}
     * @memberof V1GetGlobalConfigPoliciesResponse
     */
    configPolicies?: any;
}
/**
 * GetGroupResponse is the body of the response for the call to get a group by id.
 * @export
 * @interface V1GetGroupResponse
 */
export interface V1GetGroupResponse {
    /**
     * The group info
     * @type {V1GroupDetails}
     * @memberof V1GetGroupResponse
     */
    group: V1GroupDetails;
}
/**
 * Response object for GetGroupsAndUsersAssignedToWorkspace.
 * @export
 * @interface V1GetGroupsAndUsersAssignedToWorkspaceResponse
 */
export interface V1GetGroupsAndUsersAssignedToWorkspaceResponse {
    /**
     * Groups with a role assigned to the given workspace scope. Contains user membership of each group.
     * @type {Array<V1GroupDetails>}
     * @memberof V1GetGroupsAndUsersAssignedToWorkspaceResponse
     */
    groups: Array<V1GroupDetails>;
    /**
     * Only contains users assigned directly to roles on the workspace scope.
     * @type {Array<V1User>}
     * @memberof V1GetGroupsAndUsersAssignedToWorkspaceResponse
     */
    usersAssignedDirectly: Array<V1User>;
    /**
     * Roles assigned to workspace with associations between groups and users_assigned_directly with roles.
     * @type {Array<V1RoleWithAssignments>}
     * @memberof V1GetGroupsAndUsersAssignedToWorkspaceResponse
     */
    assignments: Array<V1RoleWithAssignments>;
}
/**
 * GetGroupsRequest is the body of the request for the call to search for groups.
 * @export
 * @interface V1GetGroupsRequest
 */
export interface V1GetGroupsRequest {
    /**
     * The id of the user to use to find groups to which the user belongs.
     * @type {number}
     * @memberof V1GetGroupsRequest
     */
    userId?: number;
    /**
     * The group name to use when searching.
     * @type {string}
     * @memberof V1GetGroupsRequest
     */
    name?: string;
    /**
     * Skip the number of groups before returning results. Negative values denote number of groups to skip from the end before returning results.
     * @type {number}
     * @memberof V1GetGroupsRequest
     */
    offset?: number;
    /**
     * Limit the number of groups. Required and must be must be <= 500.
     * @type {number}
     * @memberof V1GetGroupsRequest
     */
    limit: number;
}
/**
 * GetGroupsResponse is the body of the response for the call to search for groups.
 * @export
 * @interface V1GetGroupsResponse
 */
export interface V1GetGroupsResponse {
    /**
     * The found groups
     * @type {Array<V1GroupSearchResult>}
     * @memberof V1GetGroupsResponse
     */
    groups?: Array<V1GroupSearchResult>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetGroupsResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetJobQueueStatsRequest.
 * @export
 * @interface V1GetJobQueueStatsResponse
 */
export interface V1GetJobQueueStatsResponse {
    /**
     * List of queue stats per resource pool.
     * @type {Array<V1RPQueueStat>}
     * @memberof V1GetJobQueueStatsResponse
     */
    results: Array<V1RPQueueStat>;
}
/**
 * Response to GetJobsRequest.
 * @export
 * @interface V1GetJobsResponse
 */
export interface V1GetJobsResponse {
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetJobsResponse
     */
    pagination: V1Pagination;
    /**
     * List of the request jobs.
     * @type {Array<V1Job>}
     * @memberof V1GetJobsResponse
     */
    jobs: Array<V1Job>;
}
/**
 * Response to GetJobsV2Request.
 * @export
 * @interface V1GetJobsV2Response
 */
export interface V1GetJobsV2Response {
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetJobsV2Response
     */
    pagination: V1Pagination;
    /**
     * List of the requested jobs.
     * @type {Array<V1RBACJob>}
     * @memberof V1GetJobsV2Response
     */
    jobs: Array<V1RBACJob>;
}
/**
 * Response to GetKubernetesResourceManagersRequest.
 * @export
 * @interface V1GetKubernetesResourceManagersResponse
 */
export interface V1GetKubernetesResourceManagersResponse {
    /**
     * List of clusters
     * @type {Array<string>}
     * @memberof V1GetKubernetesResourceManagersResponse
     */
    resourceManagers: Array<string>;
}
/**
 * Response for listing namespaces bound to a workspace.
 * @export
 * @interface V1GetKubernetesResourceQuotasResponse
 */
export interface V1GetKubernetesResourceQuotasResponse {
    /**
     * List of workspace-namespace bindings for a given workspace.
     * @type {{ [key: string]: number; }}
     * @memberof V1GetKubernetesResourceQuotasResponse
     */
    resourceQuotas: { [key: string]: number; };
}
/**
 * Response to GetMasterRequest.
 * @export
 * @interface V1GetMasterConfigResponse
 */
export interface V1GetMasterConfigResponse {
    /**
     * The config file loaded by the master.
     * @type {any}
     * @memberof V1GetMasterConfigResponse
     */
    config: any;
}
/**
 * Response to GetMasterRequest.
 * @export
 * @interface V1GetMasterResponse
 */
export interface V1GetMasterResponse {
    /**
     * The current version of the master.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    version: string;
    /**
     * The current instance id of the master.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    masterId: string;
    /**
     * The global cluster id of the master.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    clusterId: string;
    /**
     * The cluster name.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    clusterName: string;
    /**
     * Telemetry status.
     * @type {boolean}
     * @memberof V1GetMasterResponse
     */
    telemetryEnabled?: boolean;
    /**
     * SSO providers.
     * @type {Array<V1SSOProvider>}
     * @memberof V1GetMasterResponse
     */
    ssoProviders?: Array<V1SSOProvider>;
    /**
     * Redirect for starting internal sessions..
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    externalLoginUri?: string;
    /**
     * Redirect for ending external sessions.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    externalLogoutUri?: string;
    /**
     * Branding style to use on front-end.
     * @type {string}
     * @memberof V1GetMasterResponse
     */
    branding?: string;
    /**
     * Feature flag for RBAC and user groups.
     * @type {boolean}
     * @memberof V1GetMasterResponse
     */
    rbacEnabled?: boolean;
    /**
     * What kind of product offering the cluster is part of, if any
     * @type {GetMasterResponseProduct}
     * @memberof V1GetMasterResponse
     */
    product?: GetMasterResponseProduct;
    /**
     * List of features that is on.
     * @type {Array<string>}
     * @memberof V1GetMasterResponse
     */
    featureSwitches?: Array<string>;
    /**
     * Feature flag for user management.
     * @type {boolean}
     * @memberof V1GetMasterResponse
     */
    userManagementEnabled?: boolean;
    /**
     * Feature flag for strict job queue control.
     * @type {boolean}
     * @memberof V1GetMasterResponse
     */
    strictJobQueueControl: boolean;
    /**
     * Active server cluster-wide message if any.
     * @type {V1ClusterMessage}
     * @memberof V1GetMasterResponse
     */
    clusterMessage?: V1ClusterMessage;
    /**
     * Whether the cluster has a user-submitted custom logo.
     * @type {boolean}
     * @memberof V1GetMasterResponse
     */
    hasCustomLogo: boolean;
}
/**
 * Response to GetMeRequest.
 * @export
 * @interface V1GetMeResponse
 */
export interface V1GetMeResponse {
    /**
     * The requested user.
     * @type {V1User}
     * @memberof V1GetMeResponse
     */
    user: V1User;
}
/**
 * Response to GetMetadataValuesRequest.
 * @export
 * @interface V1GetMetadataValuesResponse
 */
export interface V1GetMetadataValuesResponse {
    /**
     * A list of metadata values
     * @type {Array<string>}
     * @memberof V1GetMetadataValuesResponse
     */
    values?: Array<string>;
}
/**
 * Response to GetMetricsRequest.
 * @export
 * @interface V1GetMetricsResponse
 */
export interface V1GetMetricsResponse {
    /**
     * Metric response.
     * @type {Array<V1MetricsReport>}
     * @memberof V1GetMetricsResponse
     */
    metrics: Array<V1MetricsReport>;
}
/**
 * Request to get a file of model definition.
 * @export
 * @interface V1GetModelDefFileRequest
 */
export interface V1GetModelDefFileRequest {
    /**
     * The id of the experiment.
     * @type {number}
     * @memberof V1GetModelDefFileRequest
     */
    experimentId?: number;
    /**
     * The path of file.
     * @type {string}
     * @memberof V1GetModelDefFileRequest
     */
    path?: string;
}
/**
 * Response to GetModelDefFileRequest.
 * @export
 * @interface V1GetModelDefFileResponse
 */
export interface V1GetModelDefFileResponse {
    /**
     * Content of file.
     * @type {string}
     * @memberof V1GetModelDefFileResponse
     */
    file?: string;
}
/**
 * Response to GetModelDefRequest.
 * @export
 * @interface V1GetModelDefResponse
 */
export interface V1GetModelDefResponse {
    /**
     * The base64-encoded, gzipped, tarball.
     * @type {string}
     * @memberof V1GetModelDefResponse
     */
    b64Tgz: string;
}
/**
 * Response to GetModelDefTreeRequest.
 * @export
 * @interface V1GetModelDefTreeResponse
 */
export interface V1GetModelDefTreeResponse {
    /**
     * File tree of an experiment.
     * @type {Array<V1FileNode>}
     * @memberof V1GetModelDefTreeResponse
     */
    files?: Array<V1FileNode>;
}
/**
 * Response to GetModelLabelsRequest.
 * @export
 * @interface V1GetModelLabelsResponse
 */
export interface V1GetModelLabelsResponse {
    /**
     * List of labels used across all models.
     * @type {Array<string>}
     * @memberof V1GetModelLabelsResponse
     */
    labels: Array<string>;
}
/**
 * Response to GetModelRequest.
 * @export
 * @interface V1GetModelResponse
 */
export interface V1GetModelResponse {
    /**
     * The model requested.
     * @type {V1Model}
     * @memberof V1GetModelResponse
     */
    model: V1Model;
}
/**
 * Sort models by the given field.   - SORT_BY_UNSPECIFIED: Returns models in an unsorted list.  - SORT_BY_NAME: Returns models sorted by name.  - SORT_BY_DESCRIPTION: Returns models sorted by description.  - SORT_BY_CREATION_TIME: Returns models sorted by creation time.  - SORT_BY_LAST_UPDATED_TIME: Returns models sorted by last updated time.  - SORT_BY_NUM_VERSIONS: Returns models sorted by number of version.  - SORT_BY_WORKSPACE: Returns models sorted by workspace name.
 * @export
 * @enum {string}
 */
export const V1GetModelsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    NAME: 'SORT_BY_NAME',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    CREATIONTIME: 'SORT_BY_CREATION_TIME',
    LASTUPDATEDTIME: 'SORT_BY_LAST_UPDATED_TIME',
    NUMVERSIONS: 'SORT_BY_NUM_VERSIONS',
    WORKSPACE: 'SORT_BY_WORKSPACE',
} as const
export type V1GetModelsRequestSortBy = ValueOf<typeof V1GetModelsRequestSortBy>
/**
 * Response to GetModelsRequest.
 * @export
 * @interface V1GetModelsResponse
 */
export interface V1GetModelsResponse {
    /**
     * The list of returned models.
     * @type {Array<V1Model>}
     * @memberof V1GetModelsResponse
     */
    models: Array<V1Model>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetModelsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response for GetModelVersionRequest.
 * @export
 * @interface V1GetModelVersionResponse
 */
export interface V1GetModelVersionResponse {
    /**
     * The model version requested.
     * @type {V1ModelVersion}
     * @memberof V1GetModelVersionResponse
     */
    modelVersion: V1ModelVersion;
}
/**
 * Sort models by the given field.   - SORT_BY_UNSPECIFIED: Returns model versions in an unsorted list.  - SORT_BY_VERSION: Returns model versions sorted by version number.  - SORT_BY_CREATION_TIME: Returns model versions sorted by creation_time.
 * @export
 * @enum {string}
 */
export const V1GetModelVersionsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    VERSION: 'SORT_BY_VERSION',
    CREATIONTIME: 'SORT_BY_CREATION_TIME',
} as const
export type V1GetModelVersionsRequestSortBy = ValueOf<typeof V1GetModelVersionsRequestSortBy>
/**
 * Response for GetModelVersionRequest.
 * @export
 * @interface V1GetModelVersionsResponse
 */
export interface V1GetModelVersionsResponse {
    /**
     * The model requested.
     * @type {V1Model}
     * @memberof V1GetModelVersionsResponse
     */
    model: V1Model;
    /**
     * The list of returned model versions.
     * @type {Array<V1ModelVersion>}
     * @memberof V1GetModelVersionsResponse
     */
    modelVersions: Array<V1ModelVersion>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetModelVersionsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetNotebookRequest.
 * @export
 * @interface V1GetNotebookResponse
 */
export interface V1GetNotebookResponse {
    /**
     * The requested notebook.
     * @type {V1Notebook}
     * @memberof V1GetNotebookResponse
     */
    notebook: V1Notebook;
    /**
     * The notebook config.
     * @type {any}
     * @memberof V1GetNotebookResponse
     */
    config: any;
}
/**
 * Sorts notebooks by the given field.   - SORT_BY_UNSPECIFIED: Returns notebooks in an unsorted list.  - SORT_BY_ID: Returns notebooks sorted by id.  - SORT_BY_DESCRIPTION: Returns notebooks sorted by description.  - SORT_BY_START_TIME: Return notebooks sorted by start time.  - SORT_BY_WORKSPACE_ID: Return notebooks sorted by workspace_id
 * @export
 * @enum {string}
 */
export const V1GetNotebooksRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    STARTTIME: 'SORT_BY_START_TIME',
    WORKSPACEID: 'SORT_BY_WORKSPACE_ID',
} as const
export type V1GetNotebooksRequestSortBy = ValueOf<typeof V1GetNotebooksRequestSortBy>
/**
 * Response to GetNotebooksRequest.
 * @export
 * @interface V1GetNotebooksResponse
 */
export interface V1GetNotebooksResponse {
    /**
     * The list of returned notebooks.
     * @type {Array<V1Notebook>}
     * @memberof V1GetNotebooksResponse
     */
    notebooks: Array<V1Notebook>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetNotebooksResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetPermissionsSummaryRequest.
 * @export
 * @interface V1GetPermissionsSummaryResponse
 */
export interface V1GetPermissionsSummaryResponse {
    /**
     * A group of roles in cluster and other scopes.
     * @type {Array<V1Role>}
     * @memberof V1GetPermissionsSummaryResponse
     */
    roles: Array<V1Role>;
    /**
     * Lists of assignments for the cluster and other scopes.
     * @type {Array<V1RoleAssignmentSummary>}
     * @memberof V1GetPermissionsSummaryResponse
     */
    assignments: Array<V1RoleAssignmentSummary>;
}
/**
 * Response to GetProjectByKeyRequest.
 * @export
 * @interface V1GetProjectByKeyResponse
 */
export interface V1GetProjectByKeyResponse {
    /**
     * The project requested.
     * @type {V1Project}
     * @memberof V1GetProjectByKeyResponse
     */
    project: V1Project;
}
/**
 * 
 * @export
 * @interface V1GetProjectColumnsResponse
 */
export interface V1GetProjectColumnsResponse {
    /**
     * List of columns.
     * @type {Array<V1ProjectColumn>}
     * @memberof V1GetProjectColumnsResponse
     */
    columns: Array<V1ProjectColumn>;
}
/**
 * 
 * @export
 * @interface V1GetProjectNumericMetricsRangeResponse
 */
export interface V1GetProjectNumericMetricsRangeResponse {
    /**
     * List of ranges.
     * @type {Array<V1MetricsRange>}
     * @memberof V1GetProjectNumericMetricsRangeResponse
     */
    ranges?: Array<V1MetricsRange>;
}
/**
 * Response to GetProjectRequest.
 * @export
 * @interface V1GetProjectResponse
 */
export interface V1GetProjectResponse {
    /**
     * The project requested.
     * @type {V1Project}
     * @memberof V1GetProjectResponse
     */
    project: V1Project;
}
/**
 * Response to GetProjectsByUserActivityRequest.
 * @export
 * @interface V1GetProjectsByUserActivityResponse
 */
export interface V1GetProjectsByUserActivityResponse {
    /**
     * A list of projects
     * @type {Array<V1Project>}
     * @memberof V1GetProjectsByUserActivityResponse
     */
    projects?: Array<V1Project>;
}
/**
 * Response to GetResourcePoolsRequest.
 * @export
 * @interface V1GetResourcePoolsResponse
 */
export interface V1GetResourcePoolsResponse {
    /**
     * The list of returned resource pools.
     * @type {Array<V1ResourcePool>}
     * @memberof V1GetResourcePoolsResponse
     */
    resourcePools?: Array<V1ResourcePool>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetResourcePoolsResponse
     */
    pagination?: V1Pagination;
}
/**
 * 
 * @export
 * @interface V1GetRolesAssignedToGroupResponse
 */
export interface V1GetRolesAssignedToGroupResponse {
    /**
     * The roles assigned to the requested groups.
     * @type {Array<V1Role>}
     * @memberof V1GetRolesAssignedToGroupResponse
     */
    roles: Array<V1Role>;
    /**
     * What scope each role is assigned to.
     * @type {Array<V1RoleAssignmentSummary>}
     * @memberof V1GetRolesAssignedToGroupResponse
     */
    assignments: Array<V1RoleAssignmentSummary>;
}
/**
 * 
 * @export
 * @interface V1GetRolesAssignedToUserResponse
 */
export interface V1GetRolesAssignedToUserResponse {
    /**
     * The roles assigned to the requested user with assignment information.
     * @type {Array<V1RoleWithAssignments>}
     * @memberof V1GetRolesAssignedToUserResponse
     */
    roles: Array<V1RoleWithAssignments>;
}
/**
 * 
 * @export
 * @interface V1GetRolesByIDRequest
 */
export interface V1GetRolesByIDRequest {
    /**
     * The ids of the roles to be returned
     * @type {Array<number>}
     * @memberof V1GetRolesByIDRequest
     */
    roleIds?: Array<number>;
}
/**
 * 
 * @export
 * @interface V1GetRolesByIDResponse
 */
export interface V1GetRolesByIDResponse {
    /**
     * The roles requested
     * @type {Array<V1RoleWithAssignments>}
     * @memberof V1GetRolesByIDResponse
     */
    roles?: Array<V1RoleWithAssignments>;
}
/**
 * Response to get the metadata of a run.
 * @export
 * @interface V1GetRunMetadataResponse
 */
export interface V1GetRunMetadataResponse {
    /**
     * The arbitrary metadata of the run.
     * @type {any}
     * @memberof V1GetRunMetadataResponse
     */
    metadata?: any;
}
/**
 * Response to GetShellRequest.
 * @export
 * @interface V1GetShellResponse
 */
export interface V1GetShellResponse {
    /**
     * The requested shell.
     * @type {V1Shell}
     * @memberof V1GetShellResponse
     */
    shell: V1Shell;
    /**
     * The shell config.
     * @type {any}
     * @memberof V1GetShellResponse
     */
    config: any;
}
/**
 * Sorts shells by the given field.   - SORT_BY_UNSPECIFIED: Returns shells in an unsorted list.  - SORT_BY_ID: Returns shells sorted by id.  - SORT_BY_DESCRIPTION: Returns shells sorted by description.  - SORT_BY_START_TIME: Return shells sorted by start time.  - SORT_BY_WORKSPACE_ID: Return shells sorted by workspace_id.
 * @export
 * @enum {string}
 */
export const V1GetShellsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    STARTTIME: 'SORT_BY_START_TIME',
    WORKSPACEID: 'SORT_BY_WORKSPACE_ID',
} as const
export type V1GetShellsRequestSortBy = ValueOf<typeof V1GetShellsRequestSortBy>
/**
 * Response to GetShellsRequest.
 * @export
 * @interface V1GetShellsResponse
 */
export interface V1GetShellsResponse {
    /**
     * The list of returned shells.
     * @type {Array<V1Shell>}
     * @memberof V1GetShellsResponse
     */
    shells: Array<V1Shell>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetShellsResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetSlotRequest.
 * @export
 * @interface V1GetSlotResponse
 */
export interface V1GetSlotResponse {
    /**
     * The requested slot.
     * @type {V1Slot}
     * @memberof V1GetSlotResponse
     */
    slot?: V1Slot;
}
/**
 * Response to GetSlotsRequest.
 * @export
 * @interface V1GetSlotsResponse
 */
export interface V1GetSlotsResponse {
    /**
     * The requested slots.
     * @type {Array<V1Slot>}
     * @memberof V1GetSlotsResponse
     */
    slots?: Array<V1Slot>;
}
/**
 * 
 * @export
 * @interface V1GetTaskAcceleratorDataResponse
 */
export interface V1GetTaskAcceleratorDataResponse {
    /**
     * The accelerator data for each allocation associated with the task.
     * @type {Array<V1AcceleratorData>}
     * @memberof V1GetTaskAcceleratorDataResponse
     */
    acceleratorData: Array<V1AcceleratorData>;
}
/**
 * Response to GetTaskContextDirectoryRequest.
 * @export
 * @interface V1GetTaskContextDirectoryResponse
 */
export interface V1GetTaskContextDirectoryResponse {
    /**
     * The base64-encoded, gzipped, tarball.
     * @type {string}
     * @memberof V1GetTaskContextDirectoryResponse
     */
    b64Tgz: string;
}
/**
 * Response to GetTaskRequest.
 * @export
 * @interface V1GetTaskResponse
 */
export interface V1GetTaskResponse {
    /**
     * The requested task.
     * @type {V1Task}
     * @memberof V1GetTaskResponse
     */
    task: V1Task;
}
/**
 * Response to GetTasksRequest.
 * @export
 * @interface V1GetTasksResponse
 */
export interface V1GetTasksResponse {
    /**
     * Information about a task for external display.
     * @type {{ [key: string]: V1AllocationSummary; }}
     * @memberof V1GetTasksResponse
     */
    allocationIdToSummary?: { [key: string]: V1AllocationSummary; };
}
/**
 * Response to GetTelemetryRequest.
 * @export
 * @interface V1GetTelemetryResponse
 */
export interface V1GetTelemetryResponse {
    /**
     * Whether telemetry is enabled or not.
     * @type {boolean}
     * @memberof V1GetTelemetryResponse
     */
    enabled: boolean;
    /**
     * The key used for analytics in the UI.
     * @type {string}
     * @memberof V1GetTelemetryResponse
     */
    segmentKey?: string;
}
/**
 * Response to GetTemplateRequest.
 * @export
 * @interface V1GetTemplateResponse
 */
export interface V1GetTemplateResponse {
    /**
     * The requested template.
     * @type {V1Template}
     * @memberof V1GetTemplateResponse
     */
    template: V1Template;
}
/**
 * Sorts templates by the given field.   - SORT_BY_UNSPECIFIED: Returns templates in an unsorted list.  - SORT_BY_NAME: Returns templates sorted by name.
 * @export
 * @enum {string}
 */
export const V1GetTemplatesRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    NAME: 'SORT_BY_NAME',
} as const
export type V1GetTemplatesRequestSortBy = ValueOf<typeof V1GetTemplatesRequestSortBy>
/**
 * Response to GetTemplatesRequest.
 * @export
 * @interface V1GetTemplatesResponse
 */
export interface V1GetTemplatesResponse {
    /**
     * the list of requested templates.
     * @type {Array<V1Template>}
     * @memberof V1GetTemplatesResponse
     */
    templates: Array<V1Template>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetTemplatesResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetTensorboardRequest.
 * @export
 * @interface V1GetTensorboardResponse
 */
export interface V1GetTensorboardResponse {
    /**
     * The requested tensorboard.
     * @type {V1Tensorboard}
     * @memberof V1GetTensorboardResponse
     */
    tensorboard: V1Tensorboard;
    /**
     * The config;
     * @type {any}
     * @memberof V1GetTensorboardResponse
     */
    config: any;
}
/**
 * Sorts tensorboards by the given field.   - SORT_BY_UNSPECIFIED: Returns tensorboards in an unsorted list.  - SORT_BY_ID: Returns tensorboards sorted by id.  - SORT_BY_DESCRIPTION: Returns tensorboards sorted by description.  - SORT_BY_START_TIME: Return tensorboards sorted by start time.  - SORT_BY_WORKSPACE_ID: Return tensorboards sorted by workspace_id.
 * @export
 * @enum {string}
 */
export const V1GetTensorboardsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    STARTTIME: 'SORT_BY_START_TIME',
    WORKSPACEID: 'SORT_BY_WORKSPACE_ID',
} as const
export type V1GetTensorboardsRequestSortBy = ValueOf<typeof V1GetTensorboardsRequestSortBy>
/**
 * Response to GetTensorboardsRequest.
 * @export
 * @interface V1GetTensorboardsResponse
 */
export interface V1GetTensorboardsResponse {
    /**
     * The list of returned tensorboards.
     * @type {Array<V1Tensorboard>}
     * @memberof V1GetTensorboardsResponse
     */
    tensorboards: Array<V1Tensorboard>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetTensorboardsResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetTrainingMetricsRequest.
 * @export
 * @interface V1GetTrainingMetricsResponse
 */
export interface V1GetTrainingMetricsResponse {
    /**
     * Metric response.
     * @type {Array<V1MetricsReport>}
     * @memberof V1GetTrainingMetricsResponse
     */
    metrics: Array<V1MetricsReport>;
}
/**
 * Response to GetTrialByExternalIDRequest.
 * @export
 * @interface V1GetTrialByExternalIDResponse
 */
export interface V1GetTrialByExternalIDResponse {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1GetTrialByExternalIDResponse
     */
    trial: Trialv1Trial;
}
/**
 * Response to GetTrialCheckpointsRequest.
 * @export
 * @interface V1GetTrialCheckpointsResponse
 */
export interface V1GetTrialCheckpointsResponse {
    /**
     * The list of returned checkpoints.
     * @type {Array<V1Checkpoint>}
     * @memberof V1GetTrialCheckpointsResponse
     */
    checkpoints: Array<V1Checkpoint>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetTrialCheckpointsResponse
     */
    pagination: V1Pagination;
}
/**
 * 
 * @export
 * @interface V1GetTrialMetricsByCheckpointResponse
 */
export interface V1GetTrialMetricsByCheckpointResponse {
    /**
     * All the related trials and their metrics
     * @type {Array<V1MetricsReport>}
     * @memberof V1GetTrialMetricsByCheckpointResponse
     */
    metrics: Array<V1MetricsReport>;
}
/**
 * 
 * @export
 * @interface V1GetTrialMetricsByModelVersionResponse
 */
export interface V1GetTrialMetricsByModelVersionResponse {
    /**
     * All the related trials and their metrics
     * @type {Array<V1MetricsReport>}
     * @memberof V1GetTrialMetricsByModelVersionResponse
     */
    metrics: Array<V1MetricsReport>;
}
/**
 * Response to TrialProfilerAvailableSeriesRequest.
 * @export
 * @interface V1GetTrialProfilerAvailableSeriesResponse
 */
export interface V1GetTrialProfilerAvailableSeriesResponse {
    /**
     * The labels for the series.
     * @type {Array<V1TrialProfilerMetricLabels>}
     * @memberof V1GetTrialProfilerAvailableSeriesResponse
     */
    labels: Array<V1TrialProfilerMetricLabels>;
}
/**
 * 
 * @export
 * @interface V1GetTrialProfilerMetricsResponse
 */
export interface V1GetTrialProfilerMetricsResponse {
    /**
     * A batch matching the series requested.
     * @type {V1TrialProfilerMetricsBatch}
     * @memberof V1GetTrialProfilerMetricsResponse
     */
    batch: V1TrialProfilerMetricsBatch;
}
/**
 * Response to GetTrialRemainingLogRetentionDaysRequest.
 * @export
 * @interface V1GetTrialRemainingLogRetentionDaysResponse
 */
export interface V1GetTrialRemainingLogRetentionDaysResponse {
    /**
     * The remaining log retention days for the trial id.
     * @type {number}
     * @memberof V1GetTrialRemainingLogRetentionDaysResponse
     */
    remainingDays?: number;
}
/**
 * Response to GetTrialRequest.
 * @export
 * @interface V1GetTrialResponse
 */
export interface V1GetTrialResponse {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1GetTrialResponse
     */
    trial: Trialv1Trial;
}
/**
 * Response to GetTrialWorkloadsRequest.
 * @export
 * @interface V1GetTrialWorkloadsResponse
 */
export interface V1GetTrialWorkloadsResponse {
    /**
     * The list of returned workloads.
     * @type {Array<V1WorkloadContainer>}
     * @memberof V1GetTrialWorkloadsResponse
     */
    workloads: Array<V1WorkloadContainer>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetTrialWorkloadsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetUserByUsernameRequest.
 * @export
 * @interface V1GetUserByUsernameResponse
 */
export interface V1GetUserByUsernameResponse {
    /**
     * The requested user.
     * @type {V1User}
     * @memberof V1GetUserByUsernameResponse
     */
    user: V1User;
}
/**
 * Response to GetUserRequest.
 * @export
 * @interface V1GetUserResponse
 */
export interface V1GetUserResponse {
    /**
     * The requested user.
     * @type {V1User}
     * @memberof V1GetUserResponse
     */
    user: V1User;
}
/**
 * Response to GetUserSettingRequest.
 * @export
 * @interface V1GetUserSettingResponse
 */
export interface V1GetUserSettingResponse {
    /**
     * List of user settings.
     * @type {Array<V1UserWebSetting>}
     * @memberof V1GetUserSettingResponse
     */
    settings: Array<V1UserWebSetting>;
}
/**
 * Sort users by the given field.   - SORT_BY_UNSPECIFIED: Returns users in an unsorted list.  - SORT_BY_DISPLAY_NAME: Returns users sorted by display name.  - SORT_BY_USER_NAME: Returns users sorted by user name.  - SORT_BY_ADMIN: Returns users sorted by if they are admin.  - SORT_BY_ACTIVE: Returns users sorted by if they are active.  - SORT_BY_MODIFIED_TIME: Returns users sorted by modified time.  - SORT_BY_NAME: Returns users sorted by username unless display name exist.  - SORT_BY_LAST_AUTH_TIME: Returns users sorted by last authenticated time.  - SORT_BY_REMOTE: Returns users sorted by local or remote auth.
 * @export
 * @enum {string}
 */
export const V1GetUsersRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    DISPLAYNAME: 'SORT_BY_DISPLAY_NAME',
    USERNAME: 'SORT_BY_USER_NAME',
    ADMIN: 'SORT_BY_ADMIN',
    ACTIVE: 'SORT_BY_ACTIVE',
    MODIFIEDTIME: 'SORT_BY_MODIFIED_TIME',
    NAME: 'SORT_BY_NAME',
    LASTAUTHTIME: 'SORT_BY_LAST_AUTH_TIME',
    REMOTE: 'SORT_BY_REMOTE',
} as const
export type V1GetUsersRequestSortBy = ValueOf<typeof V1GetUsersRequestSortBy>
/**
 * Response to GetUsersRequest.
 * @export
 * @interface V1GetUsersResponse
 */
export interface V1GetUsersResponse {
    /**
     * The list of requested users.
     * @type {Array<V1User>}
     * @memberof V1GetUsersResponse
     */
    users?: Array<V1User>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetUsersResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response to GetTrainingMetricsRequest.
 * @export
 * @interface V1GetValidationMetricsResponse
 */
export interface V1GetValidationMetricsResponse {
    /**
     * Metric response.
     * @type {Array<V1MetricsReport>}
     * @memberof V1GetValidationMetricsResponse
     */
    metrics: Array<V1MetricsReport>;
}
/**
 * Response to GetWebhooksRequest.
 * @export
 * @interface V1GetWebhooksResponse
 */
export interface V1GetWebhooksResponse {
    /**
     * The list of returned webhooks.
     * @type {Array<V1Webhook>}
     * @memberof V1GetWebhooksResponse
     */
    webhooks: Array<V1Webhook>;
}
/**
 * Response to GetWorkspaceConfigPoliciesRequest.
 * @export
 * @interface V1GetWorkspaceConfigPoliciesResponse
 */
export interface V1GetWorkspaceConfigPoliciesResponse {
    /**
     * The current config policies saved for the workspace. Contains both invariant configs and constraints in yaml or json format.
     * @type {any}
     * @memberof V1GetWorkspaceConfigPoliciesResponse
     */
    configPolicies?: any;
}
/**
 * Sort associated projects by the given field.   - SORT_BY_UNSPECIFIED: Returns projects in an unsorted list.  - SORT_BY_CREATION_TIME: Returns projects sorted by time that they were created.  - SORT_BY_LAST_EXPERIMENT_START_TIME: Returns projects sorted by most recent start of an experiment.  - SORT_BY_NAME: Returns projects sorted by name.  - SORT_BY_DESCRIPTION: Returns projects sorted by description.  - SORT_BY_ID: Returns projects sorted by ID.
 * @export
 * @enum {string}
 */
export const V1GetWorkspaceProjectsRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    CREATIONTIME: 'SORT_BY_CREATION_TIME',
    LASTEXPERIMENTSTARTTIME: 'SORT_BY_LAST_EXPERIMENT_START_TIME',
    NAME: 'SORT_BY_NAME',
    DESCRIPTION: 'SORT_BY_DESCRIPTION',
    ID: 'SORT_BY_ID',
} as const
export type V1GetWorkspaceProjectsRequestSortBy = ValueOf<typeof V1GetWorkspaceProjectsRequestSortBy>
/**
 * Response to GetWorkspaceProjectsRequest.
 * @export
 * @interface V1GetWorkspaceProjectsResponse
 */
export interface V1GetWorkspaceProjectsResponse {
    /**
     * The projects associated with the workspace.
     * @type {Array<V1Project>}
     * @memberof V1GetWorkspaceProjectsResponse
     */
    projects: Array<V1Project>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetWorkspaceProjectsResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetWorkspaceRequest.
 * @export
 * @interface V1GetWorkspaceResponse
 */
export interface V1GetWorkspaceResponse {
    /**
     * The workspace requested.
     * @type {V1Workspace}
     * @memberof V1GetWorkspaceResponse
     */
    workspace: V1Workspace;
}
/**
 * Sort workspaces by the given field.   - SORT_BY_UNSPECIFIED: Returns workspaces in an unsorted list.  - SORT_BY_ID: Returns workspaces sorted by id.  - SORT_BY_NAME: Returns workspaces sorted by name.
 * @export
 * @enum {string}
 */
export const V1GetWorkspacesRequestSortBy = {
    UNSPECIFIED: 'SORT_BY_UNSPECIFIED',
    ID: 'SORT_BY_ID',
    NAME: 'SORT_BY_NAME',
} as const
export type V1GetWorkspacesRequestSortBy = ValueOf<typeof V1GetWorkspacesRequestSortBy>
/**
 * Response to GetWorkspacesRequest.
 * @export
 * @interface V1GetWorkspacesResponse
 */
export interface V1GetWorkspacesResponse {
    /**
     * The list of returned workspaces.
     * @type {Array<V1Workspace>}
     * @memberof V1GetWorkspacesResponse
     */
    workspaces: Array<V1Workspace>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1GetWorkspacesResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to GetUnboundWorkspacesRequest.
 * @export
 * @interface V1GetWorkspacesWithDefaultNamespaceBindingsResponse
 */
export interface V1GetWorkspacesWithDefaultNamespaceBindingsResponse {
    /**
     * The list of returned workspace ids.
     * @type {Array<number>}
     * @memberof V1GetWorkspacesWithDefaultNamespaceBindingsResponse
     */
    workspaceIds?: Array<number>;
}
/**
 * 
 * @export
 * @interface V1Group
 */
export interface V1Group {
    /**
     * The id of the group being detailed
     * @type {number}
     * @memberof V1Group
     */
    groupId?: number;
    /**
     * The name of the group
     * @type {string}
     * @memberof V1Group
     */
    name?: string;
}
/**
 * GroupDetails contains detailed information about a specific Group including which users belong to the group.
 * @export
 * @interface V1GroupDetails
 */
export interface V1GroupDetails {
    /**
     * The id of the group being detailed
     * @type {number}
     * @memberof V1GroupDetails
     */
    groupId?: number;
    /**
     * The name of the group
     * @type {string}
     * @memberof V1GroupDetails
     */
    name?: string;
    /**
     * The members of the group
     * @type {Array<V1User>}
     * @memberof V1GroupDetails
     */
    users?: Array<V1User>;
}
/**
 * GroupRoleAssignment contains information about the groups belonging to a role.
 * @export
 * @interface V1GroupRoleAssignment
 */
export interface V1GroupRoleAssignment {
    /**
     * The group id of the role assignment
     * @type {number}
     * @memberof V1GroupRoleAssignment
     */
    groupId: number;
    /**
     * The role and scope of the assignment.
     * @type {V1RoleAssignment}
     * @memberof V1GroupRoleAssignment
     */
    roleAssignment: V1RoleAssignment;
}
/**
 * GroupSearchResult is the representation of groups as they're returned by the search endpoint.
 * @export
 * @interface V1GroupSearchResult
 */
export interface V1GroupSearchResult {
    /**
     * A group matching the search criteria
     * @type {V1Group}
     * @memberof V1GroupSearchResult
     */
    group: V1Group;
    /**
     * The number of users that are in this group
     * @type {number}
     * @memberof V1GroupSearchResult
     */
    numMembers: number;
}
/**
 * Kill the requested notebook if idle.
 * @export
 * @interface V1IdleNotebookRequest
 */
export interface V1IdleNotebookRequest {
    /**
     * The id of the notebook.
     * @type {string}
     * @memberof V1IdleNotebookRequest
     */
    notebookId?: string;
    /**
     * The value of idle timeout
     * @type {boolean}
     * @memberof V1IdleNotebookRequest
     */
    idle?: boolean;
}
/**
 * Response to IdleNotebookRequest.
 * @export
 * @interface V1IdleNotebookResponse
 */
export interface V1IdleNotebookResponse {
}
/**
 * Int32 filters.
 * @export
 * @interface V1Int32FieldFilter
 */
export interface V1Int32FieldFilter {
    /**
     * Less than.
     * @type {number}
     * @memberof V1Int32FieldFilter
     */
    lt?: number;
    /**
     * Less than or equal.
     * @type {number}
     * @memberof V1Int32FieldFilter
     */
    lte?: number;
    /**
     * Greater than.
     * @type {number}
     * @memberof V1Int32FieldFilter
     */
    gt?: number;
    /**
     * Greater than or equal.
     * @type {number}
     * @memberof V1Int32FieldFilter
     */
    gte?: number;
    /**
     * In a set. `in` is a reserved word in python.
     * @type {Array<number>}
     * @memberof V1Int32FieldFilter
     */
    incl?: Array<number>;
    /**
     * Not in a set.
     * @type {Array<number>}
     * @memberof V1Int32FieldFilter
     */
    notIn?: Array<number>;
}
/**
 * Job represents a user submitted work that is not in a terminal state.
 * @export
 * @interface V1Job
 */
export interface V1Job {
    /**
     * Job summary.
     * @type {V1JobSummary}
     * @memberof V1Job
     */
    summary?: V1JobSummary;
    /**
     * Job type.
     * @type {Jobv1Type}
     * @memberof V1Job
     */
    type: Jobv1Type;
    /**
     * The time when the job was submitted by the user.
     * @type {Date | DateString}
     * @memberof V1Job
     */
    submissionTime: Date | DateString;
    /**
     * The username of the user who submitted the job.
     * @type {string}
     * @memberof V1Job
     */
    username: string;
    /**
     * The id of the user who submitted the job.
     * @type {number}
     * @memberof V1Job
     */
    userId?: number;
    /**
     * Associated resource pool.
     * @type {string}
     * @memberof V1Job
     */
    resourcePool: string;
    /**
     * Whether the job is preemptible.
     * @type {boolean}
     * @memberof V1Job
     */
    isPreemptible: boolean;
    /**
     * The job priority in priority scheduler.
     * @type {number}
     * @memberof V1Job
     */
    priority?: number;
    /**
     * The job weight in fairshare scheduler.
     * @type {number}
     * @memberof V1Job
     */
    weight?: number;
    /**
     * Entity ID.
     * @type {string}
     * @memberof V1Job
     */
    entityId: string;
    /**
     * Job type.
     * @type {string}
     * @memberof V1Job
     */
    jobId: string;
    /**
     * Number of requested slots.
     * @type {number}
     * @memberof V1Job
     */
    requestedSlots: number;
    /**
     * Number of allocated slots.
     * @type {number}
     * @memberof V1Job
     */
    allocatedSlots: number;
    /**
     * Job name.
     * @type {string}
     * @memberof V1Job
     */
    name: string;
    /**
     * Job's progress from 0 to 1.
     * @type {number}
     * @memberof V1Job
     */
    progress?: number;
    /**
     * Job's workspace id.
     * @type {number}
     * @memberof V1Job
     */
    workspaceId: number;
}
/**
 * Job summary.
 * @export
 * @interface V1JobSummary
 */
export interface V1JobSummary {
    /**
     * The scheduling state of the job.
     * @type {Jobv1State}
     * @memberof V1JobSummary
     */
    state: Jobv1State;
    /**
     * The number of jobs ahead of this one in the queue.
     * @type {number}
     * @memberof V1JobSummary
     */
    jobsAhead: number;
}
/**
 * 
 * @export
 * @interface V1K8PriorityClass
 */
export interface V1K8PriorityClass {
    /**
     * Priority class name.
     * @type {string}
     * @memberof V1K8PriorityClass
     */
    priorityClass?: string;
    /**
     * Priority class value.
     * @type {number}
     * @memberof V1K8PriorityClass
     */
    priorityValue?: number;
}
/**
 * Response to KillCommandRequest.
 * @export
 * @interface V1KillCommandResponse
 */
export interface V1KillCommandResponse {
    /**
     * The requested command.
     * @type {V1Command}
     * @memberof V1KillCommandResponse
     */
    command?: V1Command;
}
/**
 * Response to KillExperimentRequest.
 * @export
 * @interface V1KillExperimentResponse
 */
export interface V1KillExperimentResponse {
}
/**
 * Kill multiple experiments.
 * @export
 * @interface V1KillExperimentsRequest
 */
export interface V1KillExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1KillExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1KillExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1KillExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to KillExperimentsRequest.
 * @export
 * @interface V1KillExperimentsResponse
 */
export interface V1KillExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1KillExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * 
 * @export
 * @interface V1KillGenericTaskRequest
 */
export interface V1KillGenericTaskRequest {
    /**
     * The id of the task.
     * @type {string}
     * @memberof V1KillGenericTaskRequest
     */
    taskId: string;
    /**
     * Kill entire task tree
     * @type {boolean}
     * @memberof V1KillGenericTaskRequest
     */
    killFromRoot?: boolean;
}
/**
 * 
 * @export
 * @interface V1KillGenericTaskResponse
 */
export interface V1KillGenericTaskResponse {
}
/**
 * Response to KillNotebookRequest.
 * @export
 * @interface V1KillNotebookResponse
 */
export interface V1KillNotebookResponse {
    /**
     * The requested notebook.
     * @type {V1Notebook}
     * @memberof V1KillNotebookResponse
     */
    notebook?: V1Notebook;
}
/**
 * Kill runs.
 * @export
 * @interface V1KillRunsRequest
 */
export interface V1KillRunsRequest {
    /**
     * The ids of the runs being killed.
     * @type {Array<number>}
     * @memberof V1KillRunsRequest
     */
    runIds: Array<number>;
    /**
     * Project id of the runs being killed.
     * @type {number}
     * @memberof V1KillRunsRequest
     */
    projectId?: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1KillRunsRequest
     */
    filter?: string;
}
/**
 * Response to KillRunsResponse.
 * @export
 * @interface V1KillRunsResponse
 */
export interface V1KillRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1KillRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * Response to KillShellRequest.
 * @export
 * @interface V1KillShellResponse
 */
export interface V1KillShellResponse {
    /**
     * The requested shell.
     * @type {V1Shell}
     * @memberof V1KillShellResponse
     */
    shell?: V1Shell;
}
/**
 * Response to KillTensorboardRequest.
 * @export
 * @interface V1KillTensorboardResponse
 */
export interface V1KillTensorboardResponse {
    /**
     * The requested tensorboard.
     * @type {V1Tensorboard}
     * @memberof V1KillTensorboardResponse
     */
    tensorboard?: V1Tensorboard;
}
/**
 * Response to KillTrialRequest.
 * @export
 * @interface V1KillTrialResponse
 */
export interface V1KillTrialResponse {
}
/**
 * Request to launch a command.
 * @export
 * @interface V1LaunchCommandRequest
 */
export interface V1LaunchCommandRequest {
    /**
     * Command config (JSON).
     * @type {any}
     * @memberof V1LaunchCommandRequest
     */
    config?: any;
    /**
     * Template name.
     * @type {string}
     * @memberof V1LaunchCommandRequest
     */
    templateName?: string;
    /**
     * The files to run with the command.
     * @type {Array<V1File>}
     * @memberof V1LaunchCommandRequest
     */
    files?: Array<V1File>;
    /**
     * Additional data.
     * @type {string}
     * @memberof V1LaunchCommandRequest
     */
    data?: string;
    /**
     * Workspace ID. Defaults to the 'Uncategorized' workspace if not specified.
     * @type {number}
     * @memberof V1LaunchCommandRequest
     */
    workspaceId?: number;
}
/**
 * Response to LaunchCommandRequest.
 * @export
 * @interface V1LaunchCommandResponse
 */
export interface V1LaunchCommandResponse {
    /**
     * The requested command.
     * @type {V1Command}
     * @memberof V1LaunchCommandResponse
     */
    command: V1Command;
    /**
     * The config;
     * @type {any}
     * @memberof V1LaunchCommandResponse
     */
    config: any;
    /**
     * If the requested slots exceeded the current max available.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1LaunchCommandResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Request to launch a notebook.
 * @export
 * @interface V1LaunchNotebookRequest
 */
export interface V1LaunchNotebookRequest {
    /**
     * Notebook config (JSON).
     * @type {any}
     * @memberof V1LaunchNotebookRequest
     */
    config?: any;
    /**
     * Template name.
     * @type {string}
     * @memberof V1LaunchNotebookRequest
     */
    templateName?: string;
    /**
     * The files to run with the command.
     * @type {Array<V1File>}
     * @memberof V1LaunchNotebookRequest
     */
    files?: Array<V1File>;
    /**
     * Preview a launching request without actually creating a Notebook.
     * @type {boolean}
     * @memberof V1LaunchNotebookRequest
     */
    preview?: boolean;
    /**
     * Workspace ID. Defaults to 'Uncategorized' workspace if not specified.
     * @type {number}
     * @memberof V1LaunchNotebookRequest
     */
    workspaceId?: number;
}
/**
 * Response to LaunchNotebookRequest.
 * @export
 * @interface V1LaunchNotebookResponse
 */
export interface V1LaunchNotebookResponse {
    /**
     * The requested notebook.
     * @type {V1Notebook}
     * @memberof V1LaunchNotebookResponse
     */
    notebook: V1Notebook;
    /**
     * The config;
     * @type {any}
     * @memberof V1LaunchNotebookResponse
     */
    config: any;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1LaunchNotebookResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Request to launch a shell.
 * @export
 * @interface V1LaunchShellRequest
 */
export interface V1LaunchShellRequest {
    /**
     * Shell config (JSON).
     * @type {any}
     * @memberof V1LaunchShellRequest
     */
    config?: any;
    /**
     * Template name.
     * @type {string}
     * @memberof V1LaunchShellRequest
     */
    templateName?: string;
    /**
     * The files to run with the command.
     * @type {Array<V1File>}
     * @memberof V1LaunchShellRequest
     */
    files?: Array<V1File>;
    /**
     * Deprecated: Do not use.
     * @type {string}
     * @memberof V1LaunchShellRequest
     */
    data?: string;
    /**
     * Workspace ID. Defaults to 'Uncategorized' workspace if not specified.
     * @type {number}
     * @memberof V1LaunchShellRequest
     */
    workspaceId?: number;
}
/**
 * Response to LaunchShellRequest.
 * @export
 * @interface V1LaunchShellResponse
 */
export interface V1LaunchShellResponse {
    /**
     * The requested shell.
     * @type {V1Shell}
     * @memberof V1LaunchShellResponse
     */
    shell: V1Shell;
    /**
     * The config;
     * @type {any}
     * @memberof V1LaunchShellResponse
     */
    config: any;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1LaunchShellResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Request to launch a tensorboard.
 * @export
 * @interface V1LaunchTensorboardRequest
 */
export interface V1LaunchTensorboardRequest {
    /**
     * List of source experiment ids.
     * @type {Array<number>}
     * @memberof V1LaunchTensorboardRequest
     */
    experimentIds?: Array<number>;
    /**
     * List of source trial ids.
     * @type {Array<number>}
     * @memberof V1LaunchTensorboardRequest
     */
    trialIds?: Array<number>;
    /**
     * Tensorboard config (JSON).
     * @type {any}
     * @memberof V1LaunchTensorboardRequest
     */
    config?: any;
    /**
     * Tensorboard template name.
     * @type {string}
     * @memberof V1LaunchTensorboardRequest
     */
    templateName?: string;
    /**
     * The files to run with the command.
     * @type {Array<V1File>}
     * @memberof V1LaunchTensorboardRequest
     */
    files?: Array<V1File>;
    /**
     * Workspace in which to launch tensorboard. Defaults to 'Uncategorized'.
     * @type {number}
     * @memberof V1LaunchTensorboardRequest
     */
    workspaceId?: number;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1LaunchTensorboardRequest
     */
    filters?: V1BulkExperimentFilters;
}
/**
 * Response to LaunchTensorboardRequest.
 * @export
 * @interface V1LaunchTensorboardResponse
 */
export interface V1LaunchTensorboardResponse {
    /**
     * The requested tensorboard.
     * @type {V1Tensorboard}
     * @memberof V1LaunchTensorboardResponse
     */
    tensorboard: V1Tensorboard;
    /**
     * The config;
     * @type {any}
     * @memberof V1LaunchTensorboardResponse
     */
    config: any;
    /**
     * List of any related warnings.
     * @type {Array<V1LaunchWarning>}
     * @memberof V1LaunchTensorboardResponse
     */
    warnings?: Array<V1LaunchWarning>;
}
/**
 * Enum values for warnings when launching commands.   - LAUNCH_WARNING_UNSPECIFIED: Default value  - LAUNCH_WARNING_CURRENT_SLOTS_EXCEEDED: For a default webhook
 * @export
 * @enum {string}
 */
export const V1LaunchWarning = {
    UNSPECIFIED: 'LAUNCH_WARNING_UNSPECIFIED',
    CURRENTSLOTSEXCEEDED: 'LAUNCH_WARNING_CURRENT_SLOTS_EXCEEDED',
} as const
export type V1LaunchWarning = ValueOf<typeof V1LaunchWarning>
/**
 * LimitedJob is a Job with omitted fields.
 * @export
 * @interface V1LimitedJob
 */
export interface V1LimitedJob {
    /**
     * Job summary.
     * @type {V1JobSummary}
     * @memberof V1LimitedJob
     */
    summary?: V1JobSummary;
    /**
     * Job type.
     * @type {Jobv1Type}
     * @memberof V1LimitedJob
     */
    type: Jobv1Type;
    /**
     * Associated resource pool.
     * @type {string}
     * @memberof V1LimitedJob
     */
    resourcePool: string;
    /**
     * Whether the job is preemptible.
     * @type {boolean}
     * @memberof V1LimitedJob
     */
    isPreemptible: boolean;
    /**
     * The job priority in priority scheduler.
     * @type {number}
     * @memberof V1LimitedJob
     */
    priority?: number;
    /**
     * The job weight in fairshare scheduler.
     * @type {number}
     * @memberof V1LimitedJob
     */
    weight?: number;
    /**
     * Job type.
     * @type {string}
     * @memberof V1LimitedJob
     */
    jobId: string;
    /**
     * Number of requested slots.
     * @type {number}
     * @memberof V1LimitedJob
     */
    requestedSlots: number;
    /**
     * Number of allocated slots.
     * @type {number}
     * @memberof V1LimitedJob
     */
    allocatedSlots: number;
    /**
     * Job's progress from 0 to 1.
     * @type {number}
     * @memberof V1LimitedJob
     */
    progress?: number;
    /**
     * Job's workspace id.
     * @type {number}
     * @memberof V1LimitedJob
     */
    workspaceId: number;
}
/**
 * ListRolesRequest is the body of the request for the call to search for a role.
 * @export
 * @interface V1ListRolesRequest
 */
export interface V1ListRolesRequest {
    /**
     * the offset for pagination.
     * @type {number}
     * @memberof V1ListRolesRequest
     */
    offset?: number;
    /**
     * the limit for pagination.
     * @type {number}
     * @memberof V1ListRolesRequest
     */
    limit: number;
}
/**
 * ListRolesResponse is the body of the response for the call to search for a role.
 * @export
 * @interface V1ListRolesResponse
 */
export interface V1ListRolesResponse {
    /**
     * a set of roles and all assignments belonging to it.
     * @type {Array<V1Role>}
     * @memberof V1ListRolesResponse
     */
    roles: Array<V1Role>;
    /**
     * pagination information.
     * @type {V1Pagination}
     * @memberof V1ListRolesResponse
     */
    pagination: V1Pagination;
}
/**
 * Response to ListWorkspaceRPsRequest.
 * @export
 * @interface V1ListRPsBoundToWorkspaceResponse
 */
export interface V1ListRPsBoundToWorkspaceResponse {
    /**
     * List of resource pools bound to the workspace.
     * @type {Array<string>}
     * @memberof V1ListRPsBoundToWorkspaceResponse
     */
    resourcePools?: Array<string>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1ListRPsBoundToWorkspaceResponse
     */
    pagination?: V1Pagination;
}
/**
 * Response for listing namespaces bound to a workspace.
 * @export
 * @interface V1ListWorkspaceNamespaceBindingsResponse
 */
export interface V1ListWorkspaceNamespaceBindingsResponse {
    /**
     * List of workspace-namespace bindings for a given workspace.
     * @type {{ [key: string]: V1WorkspaceNamespaceBinding; }}
     * @memberof V1ListWorkspaceNamespaceBindingsResponse
     */
    namespaceBindings: { [key: string]: V1WorkspaceNamespaceBinding; };
}
/**
 * Response to ListWorkspacesBoundToRPRequest.
 * @export
 * @interface V1ListWorkspacesBoundToRPResponse
 */
export interface V1ListWorkspacesBoundToRPResponse {
    /**
     * List of workspace IDs.
     * @type {Array<number>}
     * @memberof V1ListWorkspacesBoundToRPResponse
     */
    workspaceIds?: Array<number>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1ListWorkspacesBoundToRPResponse
     */
    pagination?: V1Pagination;
}
/**
 * LocationType indicates where a column comes from - LOCATION_TYPE_UNSPECIFIED: Location unknown  - LOCATION_TYPE_EXPERIMENT: Column is located on the experiment  - LOCATION_TYPE_HYPERPARAMETERS: Column is located in the hyperparameter config of the experiment  - LOCATION_TYPE_VALIDATIONS: Column is located on the experiment's validation metrics  - LOCATION_TYPE_TRAINING: Column is located on the experiment's training steps  - LOCATION_TYPE_CUSTOM_METRIC: Column is located on the experiment's custom metric  - LOCATION_TYPE_RUN: Column is located on the run  - LOCATION_TYPE_RUN_HYPERPARAMETERS: Column is located in the hyperparameter of the run  - LOCATION_TYPE_RUN_METADATA: Column is located on the run's arbitrary metadata
 * @export
 * @enum {string}
 */
export const V1LocationType = {
    UNSPECIFIED: 'LOCATION_TYPE_UNSPECIFIED',
    EXPERIMENT: 'LOCATION_TYPE_EXPERIMENT',
    HYPERPARAMETERS: 'LOCATION_TYPE_HYPERPARAMETERS',
    VALIDATIONS: 'LOCATION_TYPE_VALIDATIONS',
    TRAINING: 'LOCATION_TYPE_TRAINING',
    CUSTOMMETRIC: 'LOCATION_TYPE_CUSTOM_METRIC',
    RUN: 'LOCATION_TYPE_RUN',
    RUNHYPERPARAMETERS: 'LOCATION_TYPE_RUN_HYPERPARAMETERS',
    RUNMETADATA: 'LOCATION_TYPE_RUN_METADATA',
} as const
export type V1LocationType = ValueOf<typeof V1LocationType>
/**
 * 
 * @export
 * @interface V1LogConfig
 */
export interface V1LogConfig {
    /**
     * The log level for Master Config.
     * @type {V1LogLevel}
     * @memberof V1LogConfig
     */
    level?: V1LogLevel;
    /**
     * The color setting for log in Master Config.
     * @type {boolean}
     * @memberof V1LogConfig
     */
    color?: boolean;
}
/**
 * LogEntry is a log event.
 * @export
 * @interface V1LogEntry
 */
export interface V1LogEntry {
    /**
     * The id.
     * @type {number}
     * @memberof V1LogEntry
     */
    id: number;
    /**
     * The message.
     * @type {string}
     * @memberof V1LogEntry
     */
    message: string;
    /**
     * The timestamp.
     * @type {Date | DateString}
     * @memberof V1LogEntry
     */
    timestamp: Date | DateString;
    /**
     * The log level.
     * @type {V1LogLevel}
     * @memberof V1LogEntry
     */
    level: V1LogLevel;
}
/**
 * Login the user.
 * @export
 * @interface V1LoginRequest
 */
export interface V1LoginRequest {
    /**
     * The username of the user.
     * @type {string}
     * @memberof V1LoginRequest
     */
    username: string;
    /**
     * The password of the user.
     * @type {string}
     * @memberof V1LoginRequest
     */
    password: string;
    /**
     * Indicate whether the provided password is pre-salted & hashed or not.
     * @type {boolean}
     * @memberof V1LoginRequest
     */
    isHashed?: boolean;
}
/**
 * Response to LoginRequest.
 * @export
 * @interface V1LoginResponse
 */
export interface V1LoginResponse {
    /**
     * The token to be used when sending results.
     * @type {string}
     * @memberof V1LoginResponse
     */
    token: string;
    /**
     * The logged in user.
     * @type {V1User}
     * @memberof V1LoginResponse
     */
    user: V1User;
}
/**
 * LogLevel specifies the level for a log.   - LOG_LEVEL_UNSPECIFIED: Unspecified log level.  - LOG_LEVEL_TRACE: A log level of TRACE.  - LOG_LEVEL_DEBUG: A log level of DEBUG.  - LOG_LEVEL_INFO: A log level of INFO.  - LOG_LEVEL_WARNING: A log level of WARNING.  - LOG_LEVEL_ERROR: A log level of ERROR.  - LOG_LEVEL_CRITICAL: A log level of CRITICAL.
 * @export
 * @enum {string}
 */
export const V1LogLevel = {
    UNSPECIFIED: 'LOG_LEVEL_UNSPECIFIED',
    TRACE: 'LOG_LEVEL_TRACE',
    DEBUG: 'LOG_LEVEL_DEBUG',
    INFO: 'LOG_LEVEL_INFO',
    WARNING: 'LOG_LEVEL_WARNING',
    ERROR: 'LOG_LEVEL_ERROR',
    CRITICAL: 'LOG_LEVEL_CRITICAL',
} as const
export type V1LogLevel = ValueOf<typeof V1LogLevel>
/**
 * Response to LogoutRequest.
 * @export
 * @interface V1LogoutResponse
 */
export interface V1LogoutResponse {
}
/**
 * Mark some reservation as a daemon.
 * @export
 * @interface V1MarkAllocationResourcesDaemonRequest
 */
export interface V1MarkAllocationResourcesDaemonRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1MarkAllocationResourcesDaemonRequest
     */
    allocationId: string;
    /**
     * The id of the clump of resources to mark as daemon.
     * @type {string}
     * @memberof V1MarkAllocationResourcesDaemonRequest
     */
    resourcesId?: string;
}
/**
 * Response to MarkAllocationResourcesDaemonRequest.
 * @export
 * @interface V1MarkAllocationResourcesDaemonResponse
 */
export interface V1MarkAllocationResourcesDaemonResponse {
}
/**
 * Response to MasterLogsRequest.
 * @export
 * @interface V1MasterLogsResponse
 */
export interface V1MasterLogsResponse {
    /**
     * The log entry.
     * @type {V1LogEntry}
     * @memberof V1MasterLogsResponse
     */
    logEntry: V1LogEntry;
}
/**
 * Response to MetricBatchesRequest.
 * @export
 * @interface V1MetricBatchesResponse
 */
export interface V1MetricBatchesResponse {
    /**
     * Milestones (in batches processed) at which the specified metric is recorded.
     * @type {Array<number>}
     * @memberof V1MetricBatchesResponse
     */
    batches?: Array<number>;
}
/**
 * MetricIdentifier packages metric name and group.
 * @export
 * @interface V1MetricIdentifier
 */
export interface V1MetricIdentifier {
    /**
     * The group of the metric.
     * @type {string}
     * @memberof V1MetricIdentifier
     */
    group: string;
    /**
     * The name of the metric.
     * @type {string}
     * @memberof V1MetricIdentifier
     */
    name: string;
}
/**
 * 
 * @export
 * @interface V1Metrics
 */
export interface V1Metrics {
    /**
     * Aggregate user-generated metrics
     * @type {any}
     * @memberof V1Metrics
     */
    avgMetrics: any;
    /**
     * User-generated metrics for each batch
     * @type {Array<any>}
     * @memberof V1Metrics
     */
    batchMetrics?: Array<any>;
}
/**
 * MetricsRange represents the range of a metrics. Range is a in the format of [min, max].
 * @export
 * @interface V1MetricsRange
 */
export interface V1MetricsRange {
    /**
     * The name of metrics formatted as <type>.<name>.
     * @type {string}
     * @memberof V1MetricsRange
     */
    metricsName: string;
    /**
     * The min of metrics values.
     * @type {number}
     * @memberof V1MetricsRange
     */
    min: number;
    /**
     * The max of metrics values.
     * @type {number}
     * @memberof V1MetricsRange
     */
    max: number;
}
/**
 * Metrics report.
 * @export
 * @interface V1MetricsReport
 */
export interface V1MetricsReport {
    /**
     * ID of the trial.
     * @type {number}
     * @memberof V1MetricsReport
     */
    trialId: number;
    /**
     * End time of when metric was reported.
     * @type {Date | DateString}
     * @memberof V1MetricsReport
     */
    endTime: Date | DateString;
    /**
     * Struct of the reported metrics.
     * @type {any}
     * @memberof V1MetricsReport
     */
    metrics: any;
    /**
     * batches completed in the report.
     * @type {number}
     * @memberof V1MetricsReport
     */
    totalBatches: number;
    /**
     * If metric is archived.
     * @type {boolean}
     * @memberof V1MetricsReport
     */
    archived: boolean;
    /**
     * ID of metric in table.
     * @type {number}
     * @memberof V1MetricsReport
     */
    id: number;
    /**
     * Run ID of trial when metric was reported.
     * @type {number}
     * @memberof V1MetricsReport
     */
    trialRunId: number;
    /**
     * Name of the Metric Group ("training", "validation", anything else)
     * @type {string}
     * @memberof V1MetricsReport
     */
    group: string;
}
/**
 * MetricsWorkload is a workload generating metrics.
 * @export
 * @interface V1MetricsWorkload
 */
export interface V1MetricsWorkload {
    /**
     * The time the workload finished or was stopped.
     * @type {Date | DateString}
     * @memberof V1MetricsWorkload
     */
    endTime?: Date | DateString;
    /**
     * Metrics.
     * @type {V1Metrics}
     * @memberof V1MetricsWorkload
     */
    metrics: V1Metrics;
    /**
     * Number of inputs processed.
     * @type {number}
     * @memberof V1MetricsWorkload
     */
    numInputs: number;
    /**
     * Total number of batches as of this workload's completion.
     * @type {number}
     * @memberof V1MetricsWorkload
     */
    totalBatches: number;
}
/**
 * To distinguish the different categories of metrics.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
 * @export
 * @enum {string}
 */
export const V1MetricType = {
    UNSPECIFIED: 'METRIC_TYPE_UNSPECIFIED',
    TRAINING: 'METRIC_TYPE_TRAINING',
    VALIDATION: 'METRIC_TYPE_VALIDATION',
    PROFILING: 'METRIC_TYPE_PROFILING',
} as const
export type V1MetricType = ValueOf<typeof V1MetricType>
/**
 * Model is a named collection of model versions.
 * @export
 * @interface V1Model
 */
export interface V1Model {
    /**
     * The name of the model.
     * @type {string}
     * @memberof V1Model
     */
    name: string;
    /**
     * The description of the model.
     * @type {string}
     * @memberof V1Model
     */
    description?: string;
    /**
     * The user-defined metadata of the model.
     * @type {any}
     * @memberof V1Model
     */
    metadata: any;
    /**
     * The time the model was created.
     * @type {Date | DateString}
     * @memberof V1Model
     */
    creationTime: Date | DateString;
    /**
     * The time the model was last updated.
     * @type {Date | DateString}
     * @memberof V1Model
     */
    lastUpdatedTime: Date | DateString;
    /**
     * The id of this model.
     * @type {number}
     * @memberof V1Model
     */
    id: number;
    /**
     * The number of versions associated with this model.
     * @type {number}
     * @memberof V1Model
     */
    numVersions: number;
    /**
     * Labels associated with this model.
     * @type {Array<string>}
     * @memberof V1Model
     */
    labels?: Array<string>;
    /**
     * Username of the user who created this model.
     * @type {string}
     * @memberof V1Model
     */
    username: string;
    /**
     * The id of the workspace associated with this model.
     * @type {number}
     * @memberof V1Model
     */
    workspaceId: number;
    /**
     * Id of the user who created this model.
     * @type {number}
     * @memberof V1Model
     */
    userId: number;
    /**
     * Whether this model is archived or not.
     * @type {boolean}
     * @memberof V1Model
     */
    archived: boolean;
    /**
     * Notes associated with this model.
     * @type {string}
     * @memberof V1Model
     */
    notes?: string;
}
/**
 * A version of a model containing a checkpoint. Users can label checkpoints as a version of a model and use the model name and version to locate a checkpoint.
 * @export
 * @interface V1ModelVersion
 */
export interface V1ModelVersion {
    /**
     * The model the version is related to.
     * @type {V1Model}
     * @memberof V1ModelVersion
     */
    model: V1Model;
    /**
     * The checkpoint of the model version.
     * @type {V1Checkpoint}
     * @memberof V1ModelVersion
     */
    checkpoint: V1Checkpoint;
    /**
     * The version number.
     * @type {number}
     * @memberof V1ModelVersion
     */
    version: number;
    /**
     * The time the model version was created.
     * @type {Date | DateString}
     * @memberof V1ModelVersion
     */
    creationTime: Date | DateString;
    /**
     * Unique id for each model version.
     * @type {number}
     * @memberof V1ModelVersion
     */
    id: number;
    /**
     * Name for this model version.
     * @type {string}
     * @memberof V1ModelVersion
     */
    name?: string;
    /**
     * Metadata associated with this model version.
     * @type {any}
     * @memberof V1ModelVersion
     */
    metadata?: any;
    /**
     * The time this model version was last updated.
     * @type {Date | DateString}
     * @memberof V1ModelVersion
     */
    lastUpdatedTime: Date | DateString;
    /**
     * Comment associated with this model version.
     * @type {string}
     * @memberof V1ModelVersion
     */
    comment?: string;
    /**
     * Username of the user who created this model version.
     * @type {string}
     * @memberof V1ModelVersion
     */
    username?: string;
    /**
     * Id of the user who created this model version.
     * @type {number}
     * @memberof V1ModelVersion
     */
    userId?: number;
    /**
     * Labels associated with this model version.
     * @type {Array<string>}
     * @memberof V1ModelVersion
     */
    labels?: Array<string>;
    /**
     * Notes associated with this model version.
     * @type {string}
     * @memberof V1ModelVersion
     */
    notes?: string;
}
/**
 * Request to move an experiment into a project.
 * @export
 * @interface V1MoveExperimentRequest
 */
export interface V1MoveExperimentRequest {
    /**
     * The id of the experiment being moved.
     * @type {number}
     * @memberof V1MoveExperimentRequest
     */
    experimentId: number;
    /**
     * The id of the new parent project.
     * @type {number}
     * @memberof V1MoveExperimentRequest
     */
    destinationProjectId: number;
}
/**
 * Response to MoveExperimentRequest.
 * @export
 * @interface V1MoveExperimentResponse
 */
export interface V1MoveExperimentResponse {
}
/**
 * Request to move an experiment into a project.
 * @export
 * @interface V1MoveExperimentsRequest
 */
export interface V1MoveExperimentsRequest {
    /**
     * The ids of the experiments being moved.
     * @type {Array<number>}
     * @memberof V1MoveExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * The id of the new parent project.
     * @type {number}
     * @memberof V1MoveExperimentsRequest
     */
    destinationProjectId: number;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1MoveExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1MoveExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to MoveExperimentsRequest.
 * @export
 * @interface V1MoveExperimentsResponse
 */
export interface V1MoveExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1MoveExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * Request to move a model to a workspace.
 * @export
 * @interface V1MoveModelRequest
 */
export interface V1MoveModelRequest {
    /**
     * The target model name.
     * @type {string}
     * @memberof V1MoveModelRequest
     */
    modelName: string;
    /**
     * The workspace id that the model will be stored.
     * @type {number}
     * @memberof V1MoveModelRequest
     */
    destinationWorkspaceId: number;
}
/**
 * 
 * @export
 * @interface V1MoveModelResponse
 */
export interface V1MoveModelResponse {
}
/**
 * Request to move a project into a workspace.
 * @export
 * @interface V1MoveProjectRequest
 */
export interface V1MoveProjectRequest {
    /**
     * The id of the project being moved.
     * @type {number}
     * @memberof V1MoveProjectRequest
     */
    projectId: number;
    /**
     * The id of the new parent workspace.
     * @type {number}
     * @memberof V1MoveProjectRequest
     */
    destinationWorkspaceId: number;
}
/**
 * Response to MoveProjectRequest.
 * @export
 * @interface V1MoveProjectResponse
 */
export interface V1MoveProjectResponse {
}
/**
 * Request to move the run to a different project.
 * @export
 * @interface V1MoveRunsRequest
 */
export interface V1MoveRunsRequest {
    /**
     * The ids of the runs being moved.
     * @type {Array<number>}
     * @memberof V1MoveRunsRequest
     */
    runIds: Array<number>;
    /**
     * The id of the current parent project.
     * @type {number}
     * @memberof V1MoveRunsRequest
     */
    sourceProjectId: number;
    /**
     * The id of the new parent project.
     * @type {number}
     * @memberof V1MoveRunsRequest
     */
    destinationProjectId: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1MoveRunsRequest
     */
    filter?: string;
    /**
     * If true, skip multi-trial experiments for move.
     * @type {boolean}
     * @memberof V1MoveRunsRequest
     */
    skipMultitrial?: boolean;
}
/**
 * Response to MoveRunsRequest.
 * @export
 * @interface V1MoveRunsResponse
 */
export interface V1MoveRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1MoveRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * Note is a user comment connected to a project.
 * @export
 * @interface V1Note
 */
export interface V1Note {
    /**
     * The name or title of the note.
     * @type {string}
     * @memberof V1Note
     */
    name: string;
    /**
     * The text contents of the note.
     * @type {string}
     * @memberof V1Note
     */
    contents: string;
}
/**
 * Notebook is a Jupyter notebook in a containerized environment.
 * @export
 * @interface V1Notebook
 */
export interface V1Notebook {
    /**
     * The id of the notebook.
     * @type {string}
     * @memberof V1Notebook
     */
    id: string;
    /**
     * The description of the notebook.
     * @type {string}
     * @memberof V1Notebook
     */
    description: string;
    /**
     * The state of the notebook.
     * @type {Taskv1State}
     * @memberof V1Notebook
     */
    state: Taskv1State;
    /**
     * The time the notebook was started.
     * @type {Date | DateString}
     * @memberof V1Notebook
     */
    startTime: Date | DateString;
    /**
     * The container running the notebook.
     * @type {V1Container}
     * @memberof V1Notebook
     */
    container?: V1Container;
    /**
     * The display name of the user that created the notebook.
     * @type {string}
     * @memberof V1Notebook
     */
    displayName?: string;
    /**
     * The id of the user that created the notebook.
     * @type {number}
     * @memberof V1Notebook
     */
    userId?: number;
    /**
     * The username of the user that created the notebook.
     * @type {string}
     * @memberof V1Notebook
     */
    username: string;
    /**
     * The service address.
     * @type {string}
     * @memberof V1Notebook
     */
    serviceAddress?: string;
    /**
     * The name of the resource pool the Notebook was created in.
     * @type {string}
     * @memberof V1Notebook
     */
    resourcePool: string;
    /**
     * The exit status.
     * @type {string}
     * @memberof V1Notebook
     */
    exitStatus?: string;
    /**
     * The associated job id.
     * @type {string}
     * @memberof V1Notebook
     */
    jobId: string;
    /**
     * Workspace ID.
     * @type {number}
     * @memberof V1Notebook
     */
    workspaceId: number;
}
/**
 * Arguments to a notify container running.
 * @export
 * @interface V1NotifyContainerRunningRequest
 */
export interface V1NotifyContainerRunningRequest {
    /**
     * The ID of the allocation.
     * @type {string}
     * @memberof V1NotifyContainerRunningRequest
     */
    allocationId: string;
    /**
     * The UUID of the participant in a notify container running message.
     * @type {string}
     * @memberof V1NotifyContainerRunningRequest
     */
    requestUuid?: string;
    /**
     * The number of process to wait for.
     * @type {number}
     * @memberof V1NotifyContainerRunningRequest
     */
    numPeers?: number;
    /**
     * The container's rank.
     * @type {number}
     * @memberof V1NotifyContainerRunningRequest
     */
    rank?: number;
    /**
     * The name of the node who sent the request
     * @type {string}
     * @memberof V1NotifyContainerRunningRequest
     */
    nodeName?: string;
    /**
     * The data from this process.
     * @type {any}
     * @memberof V1NotifyContainerRunningRequest
     */
    data: any;
}
/**
 * 
 * @export
 * @interface V1NotifyContainerRunningResponse
 */
export interface V1NotifyContainerRunningResponse {
    /**
     * The data for all the processes.
     * @type {Array<any>}
     * @memberof V1NotifyContainerRunningResponse
     */
    data: Array<any>;
}
/**
 * Order records in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
 * @export
 * @enum {string}
 */
export const V1OrderBy = {
    UNSPECIFIED: 'ORDER_BY_UNSPECIFIED',
    ASC: 'ORDER_BY_ASC',
    DESC: 'ORDER_BY_DESC',
} as const
export type V1OrderBy = ValueOf<typeof V1OrderBy>
/**
 * Overwrite and replace the workspaces bound to an RP request.
 * @export
 * @interface V1OverwriteRPWorkspaceBindingsRequest
 */
export interface V1OverwriteRPWorkspaceBindingsRequest {
    /**
     * The resource pool name.
     * @type {string}
     * @memberof V1OverwriteRPWorkspaceBindingsRequest
     */
    resourcePoolName: string;
    /**
     * The new workspace IDs to bind to the resource_pool.
     * @type {Array<number>}
     * @memberof V1OverwriteRPWorkspaceBindingsRequest
     */
    workspaceIds?: Array<number>;
    /**
     * The new workspace names to bind to the resource_pool.
     * @type {Array<string>}
     * @memberof V1OverwriteRPWorkspaceBindingsRequest
     */
    workspaceNames?: Array<string>;
}
/**
 * Overwrite and replace the workspaces bound to an RP response.
 * @export
 * @interface V1OverwriteRPWorkspaceBindingsResponse
 */
export interface V1OverwriteRPWorkspaceBindingsResponse {
}
/**
 * Pagination provides information about the offset, limit, and total number of records returned in the results.
 * @export
 * @interface V1Pagination
 */
export interface V1Pagination {
    /**
     * The number of records skipped before returning results.
     * @type {number}
     * @memberof V1Pagination
     */
    offset?: number;
    /**
     * The amount of records limited in the results.
     * @type {number}
     * @memberof V1Pagination
     */
    limit?: number;
    /**
     * The index of the first record in the dataset.
     * @type {number}
     * @memberof V1Pagination
     */
    startIndex?: number;
    /**
     * The index+1 of the last record in the dataset.
     * @type {number}
     * @memberof V1Pagination
     */
    endIndex?: number;
    /**
     * The total number of values that match the filter.
     * @type {number}
     * @memberof V1Pagination
     */
    total?: number;
}
/**
 * Patch user's access token info.
 * @export
 * @interface V1PatchAccessTokenRequest
 */
export interface V1PatchAccessTokenRequest {
    /**
     * The id of the token.
     * @type {number}
     * @memberof V1PatchAccessTokenRequest
     */
    tokenId: number;
    /**
     * The requested updated token description.
     * @type {string}
     * @memberof V1PatchAccessTokenRequest
     */
    description?: string;
    /**
     * The requested updated token revoke status.
     * @type {boolean}
     * @memberof V1PatchAccessTokenRequest
     */
    setRevoked?: boolean;
}
/**
 * Response to PatchAccessTokenRequest.
 * @export
 * @interface V1PatchAccessTokenResponse
 */
export interface V1PatchAccessTokenResponse {
    /**
     * The updated token information.
     * @type {V1TokenInfo}
     * @memberof V1PatchAccessTokenResponse
     */
    tokenInfo?: V1TokenInfo;
}
/**
 * Request to change checkpoint database information.
 * @export
 * @interface V1PatchCheckpoint
 */
export interface V1PatchCheckpoint {
    /**
     * The uuid of the checkpoint.
     * @type {string}
     * @memberof V1PatchCheckpoint
     */
    uuid: string;
    /**
     * Dictionary of file paths to file sizes in bytes of all files in the checkpoint. This won't update actual checkpoint files. If len(resources) == 0 => the checkpoint is considered deleted Otherwise if resources are updated the checkpoint is considered partially deleted.
     * @type {PatchCheckpointOptionalResources}
     * @memberof V1PatchCheckpoint
     */
    resources?: PatchCheckpointOptionalResources;
}
/**
 * Request to patch database info about a checkpoint.
 * @export
 * @interface V1PatchCheckpointsRequest
 */
export interface V1PatchCheckpointsRequest {
    /**
     * List of checkpoints to patch.
     * @type {Array<V1PatchCheckpoint>}
     * @memberof V1PatchCheckpointsRequest
     */
    checkpoints: Array<V1PatchCheckpoint>;
}
/**
 * Intentionally don't send the updated response for performance reasons.
 * @export
 * @interface V1PatchCheckpointsResponse
 */
export interface V1PatchCheckpointsResponse {
}
/**
 * PatchExperiment is a partial update to an experiment with only id required.
 * @export
 * @interface V1PatchExperiment
 */
export interface V1PatchExperiment {
    /**
     * The id of the experiment.
     * @type {number}
     * @memberof V1PatchExperiment
     */
    id: number;
    /**
     * The description of the experiment.
     * @type {string}
     * @memberof V1PatchExperiment
     */
    description?: string;
    /**
     * Labels attached to the experiment.
     * @type {Array<string>}
     * @memberof V1PatchExperiment
     */
    labels?: Array<string>;
    /**
     * The experiment name.
     * @type {string}
     * @memberof V1PatchExperiment
     */
    name?: string;
    /**
     * The experiment notes.
     * @type {string}
     * @memberof V1PatchExperiment
     */
    notes?: string;
    /**
     * Experiment config resources.
     * @type {PatchExperimentPatchResources}
     * @memberof V1PatchExperiment
     */
    resources?: PatchExperimentPatchResources;
    /**
     * Experiment config checkpoint_storage.
     * @type {PatchExperimentPatchCheckpointStorage}
     * @memberof V1PatchExperiment
     */
    checkpointStorage?: PatchExperimentPatchCheckpointStorage;
}
/**
 * Response to PatchExperimentRequest.
 * @export
 * @interface V1PatchExperimentResponse
 */
export interface V1PatchExperimentResponse {
    /**
     * Patched experiment.
     * @type {V1Experiment}
     * @memberof V1PatchExperimentResponse
     */
    experiment?: V1Experiment;
}
/**
 * Patch master config.
 * @export
 * @interface V1PatchMasterConfigRequest
 */
export interface V1PatchMasterConfigRequest {
    /**
     * The new config that the user wants to patch into the master config.
     * @type {V1Config}
     * @memberof V1PatchMasterConfigRequest
     */
    config?: V1Config;
    /**
     * The fields from the master config that the user wants to patch.
     * @type {ProtobufFieldMask}
     * @memberof V1PatchMasterConfigRequest
     */
    fieldMask?: ProtobufFieldMask;
}
/**
 * Response to PatchMasterConfigRequest.
 * @export
 * @interface V1PatchMasterConfigResponse
 */
export interface V1PatchMasterConfigResponse {
}
/**
 * PatchModel is a partial update to a model with only name required.
 * @export
 * @interface V1PatchModel
 */
export interface V1PatchModel {
    /**
     * An updated name for the model.
     * @type {string}
     * @memberof V1PatchModel
     */
    name?: string;
    /**
     * An updated description for the model.
     * @type {string}
     * @memberof V1PatchModel
     */
    description?: string;
    /**
     * An updated metadata object for the model.
     * @type {any}
     * @memberof V1PatchModel
     */
    metadata?: any;
    /**
     * An updated label list for the model.
     * @type {Array<string>}
     * @memberof V1PatchModel
     */
    labels?: Array<string>;
    /**
     * Updated notes associated with this model.
     * @type {string}
     * @memberof V1PatchModel
     */
    notes?: string;
    /**
     * The name of the workspace associated with this model.
     * @type {string}
     * @memberof V1PatchModel
     */
    workspaceName?: string;
    /**
     * The id of the workspace associated with this model.
     * @type {number}
     * @memberof V1PatchModel
     */
    workspaceId?: number;
}
/**
 * Response to PatchModelRequest.
 * @export
 * @interface V1PatchModelResponse
 */
export interface V1PatchModelResponse {
    /**
     * The model created.
     * @type {V1Model}
     * @memberof V1PatchModelResponse
     */
    model: V1Model;
}
/**
 * 
 * @export
 * @interface V1PatchModelVersion
 */
export interface V1PatchModelVersion {
    /**
     * An updated checkpoint to associate with the model version.
     * @type {V1Checkpoint}
     * @memberof V1PatchModelVersion
     */
    checkpoint?: V1Checkpoint;
    /**
     * An updated name for the model version.
     * @type {string}
     * @memberof V1PatchModelVersion
     */
    name?: string;
    /**
     * An updated metadata object for the model version.
     * @type {any}
     * @memberof V1PatchModelVersion
     */
    metadata?: any;
    /**
     * An updated comment for the model version.
     * @type {string}
     * @memberof V1PatchModelVersion
     */
    comment?: string;
    /**
     * An updated label list for the model version.
     * @type {Array<string>}
     * @memberof V1PatchModelVersion
     */
    labels?: Array<string>;
    /**
     * Updated text notes for the model version.
     * @type {string}
     * @memberof V1PatchModelVersion
     */
    notes?: string;
}
/**
 * Response to PatchModelVersionRequest.
 * @export
 * @interface V1PatchModelVersionResponse
 */
export interface V1PatchModelVersionResponse {
    /**
     * The model version created.
     * @type {V1ModelVersion}
     * @memberof V1PatchModelVersionResponse
     */
    modelVersion: V1ModelVersion;
}
/**
 * PatchProject is a partial update to a project with all optional fields.
 * @export
 * @interface V1PatchProject
 */
export interface V1PatchProject {
    /**
     * The new name for the project.
     * @type {string}
     * @memberof V1PatchProject
     */
    name?: string;
    /**
     * The new description for the project.
     * @type {string}
     * @memberof V1PatchProject
     */
    description?: string;
    /**
     * The new key for the project.
     * @type {string}
     * @memberof V1PatchProject
     */
    key?: string;
}
/**
 * Response to PatchProjectRequest.
 * @export
 * @interface V1PatchProjectResponse
 */
export interface V1PatchProjectResponse {
    /**
     * The updated project.
     * @type {V1Project}
     * @memberof V1PatchProjectResponse
     */
    project: V1Project;
}
/**
 * Response to PatchTemplateConfigRequest.
 * @export
 * @interface V1PatchTemplateConfigResponse
 */
export interface V1PatchTemplateConfigResponse {
    /**
     * The updated template.
     * @type {V1Template}
     * @memberof V1PatchTemplateConfigResponse
     */
    template: V1Template;
}
/**
 * 
 * @export
 * @interface V1PatchTemplateNameRequest
 */
export interface V1PatchTemplateNameRequest {
    /**
     * The current name.
     * @type {string}
     * @memberof V1PatchTemplateNameRequest
     */
    oldName: string;
    /**
     * The updated name.
     * @type {string}
     * @memberof V1PatchTemplateNameRequest
     */
    newName: string;
}
/**
 * 
 * @export
 * @interface V1PatchTemplateNameResponse
 */
export interface V1PatchTemplateNameResponse {
    /**
     * The updated template.
     * @type {V1Template}
     * @memberof V1PatchTemplateNameResponse
     */
    template: V1Template;
}
/**
 * Patch a trial.
 * @export
 * @interface V1PatchTrialRequest
 */
export interface V1PatchTrialRequest {
    /**
     * Trial id.
     * @type {number}
     * @memberof V1PatchTrialRequest
     */
    trialId: number;
    /**
     * The state of the trial.
     * @type {Trialv1State}
     * @memberof V1PatchTrialRequest
     */
    state?: Trialv1State;
}
/**
 * Response to PatchTrialRequest.
 * @export
 * @interface V1PatchTrialResponse
 */
export interface V1PatchTrialResponse {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1PatchTrialResponse
     */
    trial: Trialv1Trial;
}
/**
 * Request to edit fields for a user.
 * @export
 * @interface V1PatchUser
 */
export interface V1PatchUser {
    /**
     * String denoting the username of the user.
     * @type {string}
     * @memberof V1PatchUser
     */
    username?: string;
    /**
     * String denoting the password of the user.
     * @type {string}
     * @memberof V1PatchUser
     */
    password?: string;
    /**
     * Bool denoting whether the account is an admin account.
     * @type {boolean}
     * @memberof V1PatchUser
     */
    admin?: boolean;
    /**
     * Bool denoting whether the account is active.
     * @type {boolean}
     * @memberof V1PatchUser
     */
    active?: boolean;
    /**
     * Name to display in the web UI.
     * @type {string}
     * @memberof V1PatchUser
     */
    displayName?: string;
    /**
     * The user and group on the agent host machine.
     * @type {V1AgentUserGroup}
     * @memberof V1PatchUser
     */
    agentUserGroup?: V1AgentUserGroup;
    /**
     * Indicate whether the provided password is pre-salted & hashed or not.
     * @type {boolean}
     * @memberof V1PatchUser
     */
    isHashed?: boolean;
    /**
     * Bool denoting whether the user should be able to login with or change a password.
     * @type {boolean}
     * @memberof V1PatchUser
     */
    remote?: boolean;
}
/**
 * Response to PatchUserRequest.
 * @export
 * @interface V1PatchUserResponse
 */
export interface V1PatchUserResponse {
    /**
     * The updated user.
     * @type {V1User}
     * @memberof V1PatchUserResponse
     */
    user: V1User;
}
/**
 * Update activation status for multiple users.
 * @export
 * @interface V1PatchUsersRequest
 */
export interface V1PatchUsersRequest {
    /**
     * A list of user IDs to update.
     * @type {Array<number>}
     * @memberof V1PatchUsersRequest
     */
    userIds: Array<number>;
    /**
     * Intended status (true to activate, false to deactivate).
     * @type {boolean}
     * @memberof V1PatchUsersRequest
     */
    activate: boolean;
    /**
     * Option to filter to users with these properties.
     * @type {V1UserFilters}
     * @memberof V1PatchUsersRequest
     */
    filters?: V1UserFilters;
}
/**
 * Response to PatchUsersRequest.
 * @export
 * @interface V1PatchUsersResponse
 */
export interface V1PatchUsersResponse {
    /**
     * Details on success or error for each user.
     * @type {Array<V1UserActionResult>}
     * @memberof V1PatchUsersResponse
     */
    results: Array<V1UserActionResult>;
}
/**
 * 
 * @export
 * @interface V1PatchWebhook
 */
export interface V1PatchWebhook {
    /**
     * The new url of the webhook.
     * @type {string}
     * @memberof V1PatchWebhook
     */
    url: string;
}
/**
 * Response to PatchWebhookRequest.
 * @export
 * @interface V1PatchWebhookResponse
 */
export interface V1PatchWebhookResponse {
}
/**
 * PatchWorkspace is a partial update to a workspace with all optional fields.
 * @export
 * @interface V1PatchWorkspace
 */
export interface V1PatchWorkspace {
    /**
     * The new name for the workspace.
     * @type {string}
     * @memberof V1PatchWorkspace
     */
    name?: string;
    /**
     * Optional agent host uid and gid override.
     * @type {V1AgentUserGroup}
     * @memberof V1PatchWorkspace
     */
    agentUserGroup?: V1AgentUserGroup;
    /**
     * Optional checkpoint storage config. Expects same format as experiment config's checkpoint storage.
     * @type {any}
     * @memberof V1PatchWorkspace
     */
    checkpointStorageConfig?: any;
    /**
     * Name of the default compute pool.
     * @type {string}
     * @memberof V1PatchWorkspace
     */
    defaultComputePool?: string;
    /**
     * Name of the default compute pool can be optional.
     * @type {string}
     * @memberof V1PatchWorkspace
     */
    defaultComputeResourcePool?: string;
    /**
     * Name of the default aux pool.
     * @type {string}
     * @memberof V1PatchWorkspace
     */
    defaultAuxPool?: string;
    /**
     * Name of the default aux pool can be optional.
     * @type {string}
     * @memberof V1PatchWorkspace
     */
    defaultAuxResourcePool?: string;
    /**
     * The map of clusters to workspace-namespace metadata (including the workspace-namespace binding and the resource quota placed on the namespace bound to the workspace for a given cluster).
     * @type {{ [key: string]: V1WorkspaceNamespaceMeta; }}
     * @memberof V1PatchWorkspace
     */
    clusterNamespaceMeta?: { [key: string]: V1WorkspaceNamespaceMeta; };
}
/**
 * Response to PatchWorkspaceRequest.
 * @export
 * @interface V1PatchWorkspaceResponse
 */
export interface V1PatchWorkspaceResponse {
    /**
     * The updated workspace.
     * @type {V1Workspace}
     * @memberof V1PatchWorkspaceResponse
     */
    workspace: V1Workspace;
    /**
     * The workspace-namespace bindings and resource quotas created.
     * @type {{ [key: string]: V1WorkspaceNamespaceBinding; }}
     * @memberof V1PatchWorkspaceResponse
     */
    namespaceBindings?: { [key: string]: V1WorkspaceNamespaceBinding; };
}
/**
 * Response to PauseExperimentRequest.
 * @export
 * @interface V1PauseExperimentResponse
 */
export interface V1PauseExperimentResponse {
}
/**
 * Pause multiple experiments.
 * @export
 * @interface V1PauseExperimentsRequest
 */
export interface V1PauseExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1PauseExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1PauseExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1PauseExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to PauseExperimentsRequest.
 * @export
 * @interface V1PauseExperimentsResponse
 */
export interface V1PauseExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1PauseExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * 
 * @export
 * @interface V1PauseGenericTaskResponse
 */
export interface V1PauseGenericTaskResponse {
}
/**
 * Request to pause the experiment associated witha run.
 * @export
 * @interface V1PauseRunsRequest
 */
export interface V1PauseRunsRequest {
    /**
     * The ids of the runs being moved.
     * @type {Array<number>}
     * @memberof V1PauseRunsRequest
     */
    runIds: Array<number>;
    /**
     * The id of the project of the runs being paused.
     * @type {number}
     * @memberof V1PauseRunsRequest
     */
    projectId: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1PauseRunsRequest
     */
    filter?: string;
}
/**
 * Response to PauseRunsRequest.
 * @export
 * @interface V1PauseRunsResponse
 */
export interface V1PauseRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1PauseRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * 
 * @export
 * @interface V1Permission
 */
export interface V1Permission {
    /**
     * The id of the permission
     * @type {V1PermissionType}
     * @memberof V1Permission
     */
    id: V1PermissionType;
    /**
     * The name of the permission
     * @type {string}
     * @memberof V1Permission
     */
    name?: string;
    /**
     * Allowed scope types.
     * @type {V1ScopeTypeMask}
     * @memberof V1Permission
     */
    scopeTypeMask?: V1ScopeTypeMask;
}
/**
 * List of permissions types. Value of the enum has 9xxxx for global only permissions. Permissions on the same object share the thousands place value like 2001 and 2002.   - PERMISSION_TYPE_UNSPECIFIED: The permission type is unknown.  - PERMISSION_TYPE_ADMINISTRATE_USER: Can create and update other users. Allows updating other users passwords making this permission give all other permissions effectively.  - PERMISSION_TYPE_ADMINISTRATE_OAUTH: Ability to manage OAuth clients and settings.  - PERMISSION_TYPE_CREATE_EXPERIMENT: Ability to create experiments.  - PERMISSION_TYPE_VIEW_EXPERIMENT_ARTIFACTS: Ability to view experiment's model code, checkpoints, trials.  - PERMISSION_TYPE_VIEW_EXPERIMENT_METADATA: Ability to view experiment's metadata such as experiment config, progress.  - PERMISSION_TYPE_UPDATE_EXPERIMENT: Ability to update experiment and experiment's lifecycle.  - PERMISSION_TYPE_UPDATE_EXPERIMENT_METADATA: Ability to update experiment's metadata.  - PERMISSION_TYPE_DELETE_EXPERIMENT: Ability to delete experiment.  - PERMISSION_TYPE_CREATE_NSC: Ability to create Notebooks, Shells, and Commands.  - PERMISSION_TYPE_VIEW_NSC: Ability to view Notebooks, Shells, and Commands.  - PERMISSION_TYPE_UPDATE_NSC: Ability to terminate Notebooks, Shells, and Commands.  - PERMISSION_TYPE_UPDATE_GROUP: Ability to create, update, and add / remove users from groups.  - PERMISSION_TYPE_CREATE_WORKSPACE: Ability to create workspaces.  - PERMISSION_TYPE_VIEW_WORKSPACE: Ability to view workspace.  - PERMISSION_TYPE_UPDATE_WORKSPACE: Ability to update workspace.  - PERMISSION_TYPE_DELETE_WORKSPACE: Ability to delete workspace.  - PERMISSION_TYPE_SET_WORKSPACE_AGENT_USER_GROUP: Ability to set workspace agent user group config.  - PERMISSION_TYPE_SET_WORKSPACE_CHECKPOINT_STORAGE_CONFIG: Ability to set workspace checkpoint storage config.  - PERMISSION_TYPE_SET_WORKSPACE_DEFAULT_RESOURCE_POOL: Ability to set workspace default resource pool.  - PERMISSION_TYPE_CREATE_PROJECT: Ability to create projects.  - PERMISSION_TYPE_VIEW_PROJECT: Ability to view projects.  - PERMISSION_TYPE_UPDATE_PROJECT: Ability to update projects.  - PERMISSION_TYPE_DELETE_PROJECT: Ability to delete projects.  - PERMISSION_TYPE_ASSIGN_ROLES: Ability to assign roles to groups / users. If assigned at a workspace scope, can only assign roles to that workspace scope.  - PERMISSION_TYPE_VIEW_MODEL_REGISTRY: Ability to view model registry.  - PERMISSION_TYPE_EDIT_MODEL_REGISTRY: Ability to edit model registry.  - PERMISSION_TYPE_CREATE_MODEL_REGISTRY: Ability to create model registry.  - PERMISSION_TYPE_DELETE_MODEL_REGISTRY: Ability to delete model registry.  - PERMISSION_TYPE_DELETE_MODEL_VERSION: Ability to delete model version.  - PERMISSION_TYPE_DELETE_OTHER_USER_MODEL_REGISTRY: Ability to delete another user's model registry.  - PERMISSION_TYPE_DELETE_OTHER_USER_MODEL_VERSION: Ability to delete another user's model version.  - PERMISSION_TYPE_VIEW_MASTER_LOGS: Ability to view master logs.  - PERMISSION_TYPE_VIEW_CLUSTER_USAGE: Ability to view detailed cluster usage info.  - PERMISSION_TYPE_UPDATE_AGENTS: Ability to update agents.  - PERMISSION_TYPE_VIEW_SENSITIVE_AGENT_INFO: Ability to view sensitive subset of agent info.  - PERMISSION_TYPE_VIEW_MASTER_CONFIG: Ability to view master configs.  - PERMISSION_TYPE_UPDATE_MASTER_CONFIG: Ability to update master configs.  - PERMISSION_TYPE_VIEW_EXTERNAL_JOBS: Ability to view external jobs.  - PERMISSION_TYPE_CONTROL_STRICT_JOB_QUEUE: Ability to control strict job queue.  - PERMISSION_TYPE_VIEW_TEMPLATES: Ability to view templates.  - PERMISSION_TYPE_UPDATE_TEMPLATES: Ability to update templates.  - PERMISSION_TYPE_CREATE_TEMPLATES: Ability to create templates.  - PERMISSION_TYPE_DELETE_TEMPLATES: Ability to delete templates.  - PERMISSION_TYPE_UPDATE_ROLES: Ability to create and update role definitions.  - PERMISSION_TYPE_EDIT_WEBHOOKS: Ability to create and delete webhooks.  - PERMISSION_TYPE_VIEW_WEBHOOKS: Ability to view webhooks.  - PERMISSION_TYPE_MODIFY_RP_WORKSPACE_BINDINGS: Ability to bind, unbind or overwrite resource pool workspace bindings.  - PERMISSION_TYPE_SET_WORKSPACE_NAMESPACE_BINDINGS: Ability to bind, unbind, or overwrite namespace workspace bindings.  - PERMISSION_TYPE_SET_RESOURCE_QUOTAS: Ability to set resource quotas on workspaces.  - PERMISSION_TYPE_VIEW_RESOURCE_QUOTAS: Ability to view resource quotas on workspaces.  - PERMISSION_TYPE_MODIFY_GLOBAL_CONFIG_POLICIES: Ability to modify global config policies.  - PERMISSION_TYPE_MODIFY_WORKSPACE_CONFIG_POLICIES: Ability to modify workspace config policies.  - PERMISSION_TYPE_VIEW_GLOBAL_CONFIG_POLICIES: Ability to view global config policies.  - PERMISSION_TYPE_VIEW_WORKSPACE_CONFIG_POLICIES: Ability to view workspace config policies.  - PERMISSION_TYPE_ADMINISTRATE_TOKEN: Ability to administrate other users' tokens.  - PERMISSION_TYPE_UPDATE_TOKEN: Ability to update one's own token.  - PERMISSION_TYPE_CREATE_TOKEN: Ability to create one's own token  - PERMISSION_TYPE_CREATE_OTHER_TOKEN: Ability to create another user's token  - PERMISSION_TYPE_VIEW_OTHER_TOKEN: Ability to view another user's token  - PERMISSION_TYPE_VIEW_TOKEN: Ability to view one's own token
 * @export
 * @enum {string}
 */
export const V1PermissionType = {
    UNSPECIFIED: 'PERMISSION_TYPE_UNSPECIFIED',
    ADMINISTRATEUSER: 'PERMISSION_TYPE_ADMINISTRATE_USER',
    ADMINISTRATEOAUTH: 'PERMISSION_TYPE_ADMINISTRATE_OAUTH',
    CREATEEXPERIMENT: 'PERMISSION_TYPE_CREATE_EXPERIMENT',
    VIEWEXPERIMENTARTIFACTS: 'PERMISSION_TYPE_VIEW_EXPERIMENT_ARTIFACTS',
    VIEWEXPERIMENTMETADATA: 'PERMISSION_TYPE_VIEW_EXPERIMENT_METADATA',
    UPDATEEXPERIMENT: 'PERMISSION_TYPE_UPDATE_EXPERIMENT',
    UPDATEEXPERIMENTMETADATA: 'PERMISSION_TYPE_UPDATE_EXPERIMENT_METADATA',
    DELETEEXPERIMENT: 'PERMISSION_TYPE_DELETE_EXPERIMENT',
    CREATENSC: 'PERMISSION_TYPE_CREATE_NSC',
    VIEWNSC: 'PERMISSION_TYPE_VIEW_NSC',
    UPDATENSC: 'PERMISSION_TYPE_UPDATE_NSC',
    UPDATEGROUP: 'PERMISSION_TYPE_UPDATE_GROUP',
    CREATEWORKSPACE: 'PERMISSION_TYPE_CREATE_WORKSPACE',
    VIEWWORKSPACE: 'PERMISSION_TYPE_VIEW_WORKSPACE',
    UPDATEWORKSPACE: 'PERMISSION_TYPE_UPDATE_WORKSPACE',
    DELETEWORKSPACE: 'PERMISSION_TYPE_DELETE_WORKSPACE',
    SETWORKSPACEAGENTUSERGROUP: 'PERMISSION_TYPE_SET_WORKSPACE_AGENT_USER_GROUP',
    SETWORKSPACECHECKPOINTSTORAGECONFIG: 'PERMISSION_TYPE_SET_WORKSPACE_CHECKPOINT_STORAGE_CONFIG',
    SETWORKSPACEDEFAULTRESOURCEPOOL: 'PERMISSION_TYPE_SET_WORKSPACE_DEFAULT_RESOURCE_POOL',
    CREATEPROJECT: 'PERMISSION_TYPE_CREATE_PROJECT',
    VIEWPROJECT: 'PERMISSION_TYPE_VIEW_PROJECT',
    UPDATEPROJECT: 'PERMISSION_TYPE_UPDATE_PROJECT',
    DELETEPROJECT: 'PERMISSION_TYPE_DELETE_PROJECT',
    ASSIGNROLES: 'PERMISSION_TYPE_ASSIGN_ROLES',
    VIEWMODELREGISTRY: 'PERMISSION_TYPE_VIEW_MODEL_REGISTRY',
    EDITMODELREGISTRY: 'PERMISSION_TYPE_EDIT_MODEL_REGISTRY',
    CREATEMODELREGISTRY: 'PERMISSION_TYPE_CREATE_MODEL_REGISTRY',
    DELETEMODELREGISTRY: 'PERMISSION_TYPE_DELETE_MODEL_REGISTRY',
    DELETEMODELVERSION: 'PERMISSION_TYPE_DELETE_MODEL_VERSION',
    DELETEOTHERUSERMODELREGISTRY: 'PERMISSION_TYPE_DELETE_OTHER_USER_MODEL_REGISTRY',
    DELETEOTHERUSERMODELVERSION: 'PERMISSION_TYPE_DELETE_OTHER_USER_MODEL_VERSION',
    VIEWMASTERLOGS: 'PERMISSION_TYPE_VIEW_MASTER_LOGS',
    VIEWCLUSTERUSAGE: 'PERMISSION_TYPE_VIEW_CLUSTER_USAGE',
    UPDATEAGENTS: 'PERMISSION_TYPE_UPDATE_AGENTS',
    VIEWSENSITIVEAGENTINFO: 'PERMISSION_TYPE_VIEW_SENSITIVE_AGENT_INFO',
    VIEWMASTERCONFIG: 'PERMISSION_TYPE_VIEW_MASTER_CONFIG',
    UPDATEMASTERCONFIG: 'PERMISSION_TYPE_UPDATE_MASTER_CONFIG',
    VIEWEXTERNALJOBS: 'PERMISSION_TYPE_VIEW_EXTERNAL_JOBS',
    CONTROLSTRICTJOBQUEUE: 'PERMISSION_TYPE_CONTROL_STRICT_JOB_QUEUE',
    VIEWTEMPLATES: 'PERMISSION_TYPE_VIEW_TEMPLATES',
    UPDATETEMPLATES: 'PERMISSION_TYPE_UPDATE_TEMPLATES',
    CREATETEMPLATES: 'PERMISSION_TYPE_CREATE_TEMPLATES',
    DELETETEMPLATES: 'PERMISSION_TYPE_DELETE_TEMPLATES',
    UPDATEROLES: 'PERMISSION_TYPE_UPDATE_ROLES',
    EDITWEBHOOKS: 'PERMISSION_TYPE_EDIT_WEBHOOKS',
    VIEWWEBHOOKS: 'PERMISSION_TYPE_VIEW_WEBHOOKS',
    MODIFYRPWORKSPACEBINDINGS: 'PERMISSION_TYPE_MODIFY_RP_WORKSPACE_BINDINGS',
    SETWORKSPACENAMESPACEBINDINGS: 'PERMISSION_TYPE_SET_WORKSPACE_NAMESPACE_BINDINGS',
    SETRESOURCEQUOTAS: 'PERMISSION_TYPE_SET_RESOURCE_QUOTAS',
    VIEWRESOURCEQUOTAS: 'PERMISSION_TYPE_VIEW_RESOURCE_QUOTAS',
    MODIFYGLOBALCONFIGPOLICIES: 'PERMISSION_TYPE_MODIFY_GLOBAL_CONFIG_POLICIES',
    MODIFYWORKSPACECONFIGPOLICIES: 'PERMISSION_TYPE_MODIFY_WORKSPACE_CONFIG_POLICIES',
    VIEWGLOBALCONFIGPOLICIES: 'PERMISSION_TYPE_VIEW_GLOBAL_CONFIG_POLICIES',
    VIEWWORKSPACECONFIGPOLICIES: 'PERMISSION_TYPE_VIEW_WORKSPACE_CONFIG_POLICIES',
    ADMINISTRATETOKEN: 'PERMISSION_TYPE_ADMINISTRATE_TOKEN',
    UPDATETOKEN: 'PERMISSION_TYPE_UPDATE_TOKEN',
    CREATETOKEN: 'PERMISSION_TYPE_CREATE_TOKEN',
    CREATEOTHERTOKEN: 'PERMISSION_TYPE_CREATE_OTHER_TOKEN',
    VIEWOTHERTOKEN: 'PERMISSION_TYPE_VIEW_OTHER_TOKEN',
    VIEWTOKEN: 'PERMISSION_TYPE_VIEW_TOKEN',
} as const
export type V1PermissionType = ValueOf<typeof V1PermissionType>
/**
 * Response to PinWorkspaceRequest.
 * @export
 * @interface V1PinWorkspaceResponse
 */
export interface V1PinWorkspaceResponse {
}
/**
 * 
 * @export
 * @interface V1PolymorphicFilter
 */
export interface V1PolymorphicFilter {
    /**
     * metric or column name for the filter
     * @type {string}
     * @memberof V1PolymorphicFilter
     */
    name?: string;
    /**
     * double value range for the query
     * @type {V1DoubleFieldFilter}
     * @memberof V1PolymorphicFilter
     */
    doubleRange?: V1DoubleFieldFilter;
    /**
     * integer value range for the query
     * @type {V1Int32FieldFilter}
     * @memberof V1PolymorphicFilter
     */
    integerRange?: V1Int32FieldFilter;
    /**
     * time value range for the query
     * @type {V1TimestampFieldFilter}
     * @memberof V1PolymorphicFilter
     */
    timeRange?: V1TimestampFieldFilter;
}
/**
 * Create the requested user's accessToken.
 * @export
 * @interface V1PostAccessTokenRequest
 */
export interface V1PostAccessTokenRequest {
    /**
     * The id of the user.
     * @type {number}
     * @memberof V1PostAccessTokenRequest
     */
    userId: number;
    /**
     * Lifespan expressing how long the token should last. Should be a Go-format duration (e.g. "2s", "4m", "72h".)
     * @type {string}
     * @memberof V1PostAccessTokenRequest
     */
    lifespan?: string;
    /**
     * Description of the token.
     * @type {string}
     * @memberof V1PostAccessTokenRequest
     */
    description?: string;
}
/**
 * Response to PostAccessTokenRequest.
 * @export
 * @interface V1PostAccessTokenResponse
 */
export interface V1PostAccessTokenResponse {
    /**
     * token value string.
     * @type {string}
     * @memberof V1PostAccessTokenResponse
     */
    token?: string;
    /**
     * token id.
     * @type {number}
     * @memberof V1PostAccessTokenResponse
     */
    tokenId?: number;
}
/**
 * Set the accelerator data for some allocation.
 * @export
 * @interface V1PostAllocationAcceleratorDataRequest
 */
export interface V1PostAllocationAcceleratorDataRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1PostAllocationAcceleratorDataRequest
     */
    allocationId: string;
    /**
     * The accelerator data used by the allocation.
     * @type {V1AcceleratorData}
     * @memberof V1PostAllocationAcceleratorDataRequest
     */
    acceleratorData: V1AcceleratorData;
}
/**
 * 
 * @export
 * @interface V1PostAllocationAcceleratorDataResponse
 */
export interface V1PostAllocationAcceleratorDataResponse {
}
/**
 * Set the proxy address for some allocation.
 * @export
 * @interface V1PostAllocationProxyAddressRequest
 */
export interface V1PostAllocationProxyAddressRequest {
    /**
     * The id of the allocation.
     * @type {string}
     * @memberof V1PostAllocationProxyAddressRequest
     */
    allocationId?: string;
    /**
     * The address of the host where the service is, w.r.t. the master.
     * @type {string}
     * @memberof V1PostAllocationProxyAddressRequest
     */
    proxyAddress?: string;
}
/**
 * 
 * @export
 * @interface V1PostAllocationProxyAddressResponse
 */
export interface V1PostAllocationProxyAddressResponse {
}
/**
 * Request for updating a checkpoints metadata.
 * @export
 * @interface V1PostCheckpointMetadataRequest
 */
export interface V1PostCheckpointMetadataRequest {
    /**
     * The desired checkpoint fields and values.
     * @type {V1Checkpoint}
     * @memberof V1PostCheckpointMetadataRequest
     */
    checkpoint?: V1Checkpoint;
}
/**
 * Response to PostCheckpointRequest.
 * @export
 * @interface V1PostCheckpointMetadataResponse
 */
export interface V1PostCheckpointMetadataResponse {
    /**
     * The updated checkpoint.
     * @type {V1Checkpoint}
     * @memberof V1PostCheckpointMetadataResponse
     */
    checkpoint?: V1Checkpoint;
}
/**
 * Request for creating a model in the registry.
 * @export
 * @interface V1PostModelRequest
 */
export interface V1PostModelRequest {
    /**
     * The name of the model.
     * @type {string}
     * @memberof V1PostModelRequest
     */
    name: string;
    /**
     * The description of the model.
     * @type {string}
     * @memberof V1PostModelRequest
     */
    description?: string;
    /**
     * The user-defined metadata of the model.
     * @type {any}
     * @memberof V1PostModelRequest
     */
    metadata?: any;
    /**
     * Labels associated with this model.
     * @type {Array<string>}
     * @memberof V1PostModelRequest
     */
    labels?: Array<string>;
    /**
     * The name of the workspace associated with this model.
     * @type {string}
     * @memberof V1PostModelRequest
     */
    workspaceName?: string;
    /**
     * The id of the workspace associated with this model.
     * @type {number}
     * @memberof V1PostModelRequest
     */
    workspaceId?: number;
    /**
     * Notes associated with this model.
     * @type {string}
     * @memberof V1PostModelRequest
     */
    notes?: string;
}
/**
 * Response to PostModelRequest.
 * @export
 * @interface V1PostModelResponse
 */
export interface V1PostModelResponse {
    /**
     * The model created.
     * @type {V1Model}
     * @memberof V1PostModelResponse
     */
    model: V1Model;
}
/**
 * Request for creating a model version.
 * @export
 * @interface V1PostModelVersionRequest
 */
export interface V1PostModelVersionRequest {
    /**
     * The name of the model to add this version to.
     * @type {string}
     * @memberof V1PostModelVersionRequest
     */
    modelName: string;
    /**
     * UUID of the checkpoint.
     * @type {string}
     * @memberof V1PostModelVersionRequest
     */
    checkpointUuid: string;
    /**
     * User-friendly name for the model version.
     * @type {string}
     * @memberof V1PostModelVersionRequest
     */
    name?: string;
    /**
     * User-written comment for the model version.
     * @type {string}
     * @memberof V1PostModelVersionRequest
     */
    comment?: string;
    /**
     * The user-defined metadata of the model version.
     * @type {any}
     * @memberof V1PostModelVersionRequest
     */
    metadata?: any;
    /**
     * Labels associated with this model version.
     * @type {Array<string>}
     * @memberof V1PostModelVersionRequest
     */
    labels?: Array<string>;
    /**
     * Notes associated with this model version.
     * @type {string}
     * @memberof V1PostModelVersionRequest
     */
    notes?: string;
}
/**
 * Response for PostModelVersionRequest.
 * @export
 * @interface V1PostModelVersionResponse
 */
export interface V1PostModelVersionResponse {
    /**
     * The model version requested.
     * @type {V1ModelVersion}
     * @memberof V1PostModelVersionResponse
     */
    modelVersion: V1ModelVersion;
}
/**
 * Request for creating a project.
 * @export
 * @interface V1PostProjectRequest
 */
export interface V1PostProjectRequest {
    /**
     * The name of the project.
     * @type {string}
     * @memberof V1PostProjectRequest
     */
    name: string;
    /**
     * Description of the project.
     * @type {string}
     * @memberof V1PostProjectRequest
     */
    description?: string;
    /**
     * Id of the associated workspace.
     * @type {number}
     * @memberof V1PostProjectRequest
     */
    workspaceId: number;
    /**
     * Key for the project.
     * @type {string}
     * @memberof V1PostProjectRequest
     */
    key?: string;
}
/**
 * Response to PostProjectRequest.
 * @export
 * @interface V1PostProjectResponse
 */
export interface V1PostProjectResponse {
    /**
     * The project created.
     * @type {V1Project}
     * @memberof V1PostProjectResponse
     */
    project: V1Project;
}
/**
 * Request to post metadata for a run.
 * @export
 * @interface V1PostRunMetadataRequest
 */
export interface V1PostRunMetadataRequest {
    /**
     * The ID of the run to post metadata for.
     * @type {number}
     * @memberof V1PostRunMetadataRequest
     */
    runId?: number;
    /**
     * The arbitrary metadata to post.
     * @type {any}
     * @memberof V1PostRunMetadataRequest
     */
    metadata: any;
}
/**
 * Response to post metadata for a run.
 * @export
 * @interface V1PostRunMetadataResponse
 */
export interface V1PostRunMetadataResponse {
    /**
     * The new metadata of the run.
     * @type {any}
     * @memberof V1PostRunMetadataResponse
     */
    metadata?: any;
}
/**
 * Request to PostTaskLogs.
 * @export
 * @interface V1PostTaskLogsRequest
 */
export interface V1PostTaskLogsRequest {
    /**
     * The logs to persist.
     * @type {Array<V1TaskLog>}
     * @memberof V1PostTaskLogsRequest
     */
    logs: Array<V1TaskLog>;
}
/**
 * Response to PostTaskLogs.
 * @export
 * @interface V1PostTaskLogsResponse
 */
export interface V1PostTaskLogsResponse {
}
/**
 * Response to PostTemplateRequest.
 * @export
 * @interface V1PostTemplateResponse
 */
export interface V1PostTemplateResponse {
    /**
     * The created template.
     * @type {V1Template}
     * @memberof V1PostTemplateResponse
     */
    template: V1Template;
}
/**
 * Create a batch of trial profiler metrics.
 * @export
 * @interface V1PostTrialProfilerMetricsBatchRequest
 */
export interface V1PostTrialProfilerMetricsBatchRequest {
    /**
     * The batches to create.
     * @type {Array<V1TrialProfilerMetricsBatch>}
     * @memberof V1PostTrialProfilerMetricsBatchRequest
     */
    batches?: Array<V1TrialProfilerMetricsBatch>;
}
/**
 * 
 * @export
 * @interface V1PostTrialProfilerMetricsBatchResponse
 */
export interface V1PostTrialProfilerMetricsBatchResponse {
}
/**
 * 
 * @export
 * @interface V1PostTrialRunnerMetadataResponse
 */
export interface V1PostTrialRunnerMetadataResponse {
}
/**
 * Update user activity.
 * @export
 * @interface V1PostUserActivityRequest
 */
export interface V1PostUserActivityRequest {
    /**
     * The type of the activity.
     * @type {V1ActivityType}
     * @memberof V1PostUserActivityRequest
     */
    activityType: V1ActivityType;
    /**
     * The type of the entity.
     * @type {V1EntityType}
     * @memberof V1PostUserActivityRequest
     */
    entityType: V1EntityType;
    /**
     * The unique id for the entity
     * @type {number}
     * @memberof V1PostUserActivityRequest
     */
    entityId: number;
}
/**
 * Response to PostUserActivityRequest.
 * @export
 * @interface V1PostUserActivityResponse
 */
export interface V1PostUserActivityResponse {
}
/**
 * Create a new user.
 * @export
 * @interface V1PostUserRequest
 */
export interface V1PostUserRequest {
    /**
     * The user to create.
     * @type {V1User}
     * @memberof V1PostUserRequest
     */
    user?: V1User;
    /**
     * The password of the user.
     * @type {string}
     * @memberof V1PostUserRequest
     */
    password?: string;
    /**
     * Indicate whether the provided password is pre-salted & hashed or not.
     * @type {boolean}
     * @memberof V1PostUserRequest
     */
    isHashed?: boolean;
}
/**
 * Response to PostUserRequest.
 * @export
 * @interface V1PostUserResponse
 */
export interface V1PostUserResponse {
    /**
     * The created user.
     * @type {V1User}
     * @memberof V1PostUserResponse
     */
    user?: V1User;
}
/**
 * Update user settings.
 * @export
 * @interface V1PostUserSettingRequest
 */
export interface V1PostUserSettingRequest {
    /**
     * Setting key value pair.
     * @type {Array<V1UserWebSetting>}
     * @memberof V1PostUserSettingRequest
     */
    settings: Array<V1UserWebSetting>;
}
/**
 * Response to PostUserSettingRequest.
 * @export
 * @interface V1PostUserSettingResponse
 */
export interface V1PostUserSettingResponse {
}
/**
 * Request for triggering custom trigger.
 * @export
 * @interface V1PostWebhookEventDataRequest
 */
export interface V1PostWebhookEventDataRequest {
    /**
     * The event data for custom webhook trigger.
     * @type {V1CustomWebhookEventData}
     * @memberof V1PostWebhookEventDataRequest
     */
    data: V1CustomWebhookEventData;
    /**
     * The id of the experiment.
     * @type {number}
     * @memberof V1PostWebhookEventDataRequest
     */
    experimentId: number;
    /**
     * The id of the trial.
     * @type {number}
     * @memberof V1PostWebhookEventDataRequest
     */
    trialId?: number;
}
/**
 * Response to PostWebhookEventDataRequest.
 * @export
 * @interface V1PostWebhookEventDataResponse
 */
export interface V1PostWebhookEventDataResponse {
}
/**
 * Response to PostWebhookRequest.
 * @export
 * @interface V1PostWebhookResponse
 */
export interface V1PostWebhookResponse {
    /**
     * The webhook created.
     * @type {V1Webhook}
     * @memberof V1PostWebhookResponse
     */
    webhook: V1Webhook;
}
/**
 * Request for creating a workspace.
 * @export
 * @interface V1PostWorkspaceRequest
 */
export interface V1PostWorkspaceRequest {
    /**
     * The name of the workspace.
     * @type {string}
     * @memberof V1PostWorkspaceRequest
     */
    name: string;
    /**
     * Optional agent host uid and gid override.
     * @type {V1AgentUserGroup}
     * @memberof V1PostWorkspaceRequest
     */
    agentUserGroup?: V1AgentUserGroup;
    /**
     * Optional checkpoint storage config.
     * @type {any}
     * @memberof V1PostWorkspaceRequest
     */
    checkpointStorageConfig?: any;
    /**
     * The name of the default compute pool.
     * @type {string}
     * @memberof V1PostWorkspaceRequest
     */
    defaultComputePool?: string;
    /**
     * The name of the default aux pool.
     * @type {string}
     * @memberof V1PostWorkspaceRequest
     */
    defaultAuxPool?: string;
    /**
     * The map of clusters to workspace-namespace metadata (including the workspace-namespace binding and the resource quota placed on the namespace bound to the workspace for a given cluster).
     * @type {{ [key: string]: V1WorkspaceNamespaceMeta; }}
     * @memberof V1PostWorkspaceRequest
     */
    clusterNamespaceMeta?: { [key: string]: V1WorkspaceNamespaceMeta; };
    /**
     * The map of clusters to the resource quota placed on the auto-generated namespace within that cluster (inherently capping the slots available to a workspace).
     * @type {{ [key: string]: number; }}
     * @memberof V1PostWorkspaceRequest
     */
    clusterQuotaPairs?: { [key: string]: number; };
}
/**
 * Response to PostWorkspaceRequest.
 * @export
 * @interface V1PostWorkspaceResponse
 */
export interface V1PostWorkspaceResponse {
    /**
     * The workspace created.
     * @type {V1Workspace}
     * @memberof V1PostWorkspaceResponse
     */
    workspace: V1Workspace;
    /**
     * The workspace-namespace bindings and resource quotas created.
     * @type {{ [key: string]: V1WorkspaceNamespaceBinding; }}
     * @memberof V1PostWorkspaceResponse
     */
    namespaceBindings?: { [key: string]: V1WorkspaceNamespaceBinding; };
}
/**
 * Preview hyperparameter search.
 * @export
 * @interface V1PreviewHPSearchRequest
 */
export interface V1PreviewHPSearchRequest {
    /**
     * The experiment config to simulate.
     * @type {any}
     * @memberof V1PreviewHPSearchRequest
     */
    config?: any;
    /**
     * The searcher simulation seed.
     * @type {number}
     * @memberof V1PreviewHPSearchRequest
     */
    seed?: number;
}
/**
 * Response to PreviewSearchRequest.
 * @export
 * @interface V1PreviewHPSearchResponse
 */
export interface V1PreviewHPSearchResponse {
    /**
     * The resulting summary.
     * @type {V1SearchSummary}
     * @memberof V1PreviewHPSearchResponse
     */
    summary?: V1SearchSummary;
}
/**
 * Project is a named collection of experiments.
 * @export
 * @interface V1Project
 */
export interface V1Project {
    /**
     * The unique id of the project.
     * @type {number}
     * @memberof V1Project
     */
    id: number;
    /**
     * The unique name of the project.
     * @type {string}
     * @memberof V1Project
     */
    name: string;
    /**
     * The id of the associated workspace.
     * @type {number}
     * @memberof V1Project
     */
    workspaceId: number;
    /**
     * The description of the project.
     * @type {string}
     * @memberof V1Project
     */
    description?: string;
    /**
     * Time of most recently started experiment within this project.
     * @type {Date | DateString}
     * @memberof V1Project
     */
    lastExperimentStartedAt?: Date | DateString;
    /**
     * Notes associated with this project.
     * @type {Array<V1Note>}
     * @memberof V1Project
     */
    notes: Array<V1Note>;
    /**
     * Count of experiments associated with this project.
     * @type {number}
     * @memberof V1Project
     */
    numExperiments: number;
    /**
     * Count of active experiments associated with this project.
     * @type {number}
     * @memberof V1Project
     */
    numActiveExperiments: number;
    /**
     * Whether this project is archived or not.
     * @type {boolean}
     * @memberof V1Project
     */
    archived: boolean;
    /**
     * User who created this project.
     * @type {string}
     * @memberof V1Project
     */
    username: string;
    /**
     * Whether this project is immutable (default uncategorized project).
     * @type {boolean}
     * @memberof V1Project
     */
    immutable: boolean;
    /**
     * ID of the user who created this project.
     * @type {number}
     * @memberof V1Project
     */
    userId: number;
    /**
     * The name of the associated workspace.
     * @type {string}
     * @memberof V1Project
     */
    workspaceName?: string;
    /**
     * State of project during deletion.
     * @type {V1WorkspaceState}
     * @memberof V1Project
     */
    state: V1WorkspaceState;
    /**
     * Message stored from errors on async-deleting a project.
     * @type {string}
     * @memberof V1Project
     */
    errorMessage: string;
    /**
     * The key of the project.
     * @type {string}
     * @memberof V1Project
     */
    key: string;
    /**
     * Count of runs associated with this project.
     * @type {number}
     * @memberof V1Project
     */
    numRuns: number;
}
/**
 * Project Column is a description of a column used on experiments in the project.
 * @export
 * @interface V1ProjectColumn
 */
export interface V1ProjectColumn {
    /**
     * Raw column name.
     * @type {string}
     * @memberof V1ProjectColumn
     */
    column: string;
    /**
     * Where the column comes from.
     * @type {V1LocationType}
     * @memberof V1ProjectColumn
     */
    location: V1LocationType;
    /**
     * Type of data in the column.
     * @type {V1ColumnType}
     * @memberof V1ProjectColumn
     */
    type: V1ColumnType;
    /**
     * Human-friendly name.
     * @type {string}
     * @memberof V1ProjectColumn
     */
    displayName?: string;
}
/**
 * ProxyPortConfig configures a proxy the allocation should start.
 * @export
 * @interface V1ProxyPortConfig
 */
export interface V1ProxyPortConfig {
    /**
     * The service ID of the proxy port config.
     * @type {string}
     * @memberof V1ProxyPortConfig
     */
    serviceId?: string;
    /**
     * The port of the proxy port config.
     * @type {number}
     * @memberof V1ProxyPortConfig
     */
    port?: number;
    /**
     * True if proxy uses TCP.
     * @type {boolean}
     * @memberof V1ProxyPortConfig
     */
    proxyTcp?: boolean;
    /**
     * True if the proxy is unauthenticated.
     * @type {boolean}
     * @memberof V1ProxyPortConfig
     */
    unauthenticated?: boolean;
}
/**
 * Response to PutExperimentLabelRequest.
 * @export
 * @interface V1PutExperimentLabelResponse
 */
export interface V1PutExperimentLabelResponse {
    /**
     * The complete list of labels associated with the experiment.
     * @type {Array<string>}
     * @memberof V1PutExperimentLabelResponse
     */
    labels: Array<string>;
}
/**
 * Response to PutExperimentRequest.
 * @export
 * @interface V1PutExperimentResponse
 */
export interface V1PutExperimentResponse {
    /**
     * The created experiment.
     * @type {V1Experiment}
     * @memberof V1PutExperimentResponse
     */
    experiment: V1Experiment;
    /**
     * The created experiment config.
     * @type {any}
     * @memberof V1PutExperimentResponse
     */
    config: any;
}
/**
 * Request for changing the log retention policy for the an experiment.
 * @export
 * @interface V1PutExperimentRetainLogsRequest
 */
export interface V1PutExperimentRetainLogsRequest {
    /**
     * The ID of the experiment.
     * @type {number}
     * @memberof V1PutExperimentRetainLogsRequest
     */
    experimentId: number;
    /**
     * The number of days to retain logs, starting from the end time of the task.
     * @type {number}
     * @memberof V1PutExperimentRetainLogsRequest
     */
    numDays: number;
}
/**
 * Response to PutExperimentRetainLogsRequest.
 * @export
 * @interface V1PutExperimentRetainLogsResponse
 */
export interface V1PutExperimentRetainLogsResponse {
}
/**
 * Request for changing the log retention policy for the an experiment.
 * @export
 * @interface V1PutExperimentsRetainLogsRequest
 */
export interface V1PutExperimentsRetainLogsRequest {
    /**
     * The ids of the experiments being moved.
     * @type {Array<number>}
     * @memberof V1PutExperimentsRetainLogsRequest
     */
    experimentIds: Array<number>;
    /**
     * The number of days to retain logs, starting from the end time of the task.
     * @type {number}
     * @memberof V1PutExperimentsRetainLogsRequest
     */
    numDays: number;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1PutExperimentsRetainLogsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1PutExperimentsRetainLogsRequest
     */
    projectId: number;
}
/**
 * Response to PutExperimentRetainLogsRequest.
 * @export
 * @interface V1PutExperimentsRetainLogsResponse
 */
export interface V1PutExperimentsRetainLogsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1PutExperimentsRetainLogsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * PutGlobalConfigPoliciesRequest sets global config policies for the workload type.
 * @export
 * @interface V1PutGlobalConfigPoliciesRequest
 */
export interface V1PutGlobalConfigPoliciesRequest {
    /**
     * The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @type {string}
     * @memberof V1PutGlobalConfigPoliciesRequest
     */
    workloadType: string;
    /**
     * The config policies to use. Contains both invariant configs and constraints in yaml or json format.
     * @type {string}
     * @memberof V1PutGlobalConfigPoliciesRequest
     */
    configPolicies: string;
}
/**
 * Response to PutGlobalConfigPoliciesRequest.
 * @export
 * @interface V1PutGlobalConfigPoliciesResponse
 */
export interface V1PutGlobalConfigPoliciesResponse {
    /**
     * The config policies saved. Contains both invariant configs and constraints in yaml or json format.
     * @type {any}
     * @memberof V1PutGlobalConfigPoliciesResponse
     */
    configPolicies?: any;
}
/**
 * Request for setting project notes.
 * @export
 * @interface V1PutProjectNotesRequest
 */
export interface V1PutProjectNotesRequest {
    /**
     * The complete list of notes.
     * @type {Array<V1Note>}
     * @memberof V1PutProjectNotesRequest
     */
    notes: Array<V1Note>;
    /**
     * The id of the project.
     * @type {number}
     * @memberof V1PutProjectNotesRequest
     */
    projectId: number;
}
/**
 * Response to PutProjectNotesRequest.
 * @export
 * @interface V1PutProjectNotesResponse
 */
export interface V1PutProjectNotesResponse {
    /**
     * The complete list of notes on a project.
     * @type {Array<V1Note>}
     * @memberof V1PutProjectNotesResponse
     */
    notes: Array<V1Note>;
}
/**
 * Response to PutTemplateRequest.
 * @export
 * @interface V1PutTemplateResponse
 */
export interface V1PutTemplateResponse {
    /**
     * The updated or created template.
     * @type {V1Template}
     * @memberof V1PutTemplateResponse
     */
    template?: V1Template;
}
/**
 * Put a trial.
 * @export
 * @interface V1PutTrialRequest
 */
export interface V1PutTrialRequest {
    /**
     * CreateTrialRequest payload.
     * @type {V1CreateTrialRequest}
     * @memberof V1PutTrialRequest
     */
    createTrialRequest?: V1CreateTrialRequest;
    /**
     * External trial id.
     * @type {string}
     * @memberof V1PutTrialRequest
     */
    externalTrialId?: string;
}
/**
 * Response to PutTrialRequest.
 * @export
 * @interface V1PutTrialResponse
 */
export interface V1PutTrialResponse {
    /**
     * The requested trial.
     * @type {Trialv1Trial}
     * @memberof V1PutTrialResponse
     */
    trial: Trialv1Trial;
}
/**
 * Request for changing the log retention policy for the an experiment.
 * @export
 * @interface V1PutTrialRetainLogsRequest
 */
export interface V1PutTrialRetainLogsRequest {
    /**
     * The ID of the trial.
     * @type {number}
     * @memberof V1PutTrialRetainLogsRequest
     */
    trialId?: number;
    /**
     * The number of days to retain logs, starting from the end time of the task.
     * @type {number}
     * @memberof V1PutTrialRetainLogsRequest
     */
    numDays: number;
}
/**
 * Response to PutExperimentRetainLogsRequest.
 * @export
 * @interface V1PutTrialRetainLogsResponse
 */
export interface V1PutTrialRetainLogsResponse {
}
/**
 * PutWorkspaceConfigPoliciesRequest sets config policies for the workspace and workload type.
 * @export
 * @interface V1PutWorkspaceConfigPoliciesRequest
 */
export interface V1PutWorkspaceConfigPoliciesRequest {
    /**
     * The workspace the config policies apply to. Use global API for global config policies.
     * @type {number}
     * @memberof V1PutWorkspaceConfigPoliciesRequest
     */
    workspaceId: number;
    /**
     * The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @type {string}
     * @memberof V1PutWorkspaceConfigPoliciesRequest
     */
    workloadType: string;
    /**
     * The config policies to use. Contains both invariant configs and constraints in yaml or json format.
     * @type {string}
     * @memberof V1PutWorkspaceConfigPoliciesRequest
     */
    configPolicies: string;
}
/**
 * Response to PutWorkspaceConfigPoliciesRequest.
 * @export
 * @interface V1PutWorkspaceConfigPoliciesResponse
 */
export interface V1PutWorkspaceConfigPoliciesResponse {
    /**
     * The config policies saved. Contains both invariant configs and constraints in yaml or json format.
     * @type {any}
     * @memberof V1PutWorkspaceConfigPoliciesResponse
     */
    configPolicies?: any;
}
/**
 * Describes a message to control jobs in a queue.
 * @export
 * @interface V1QueueControl
 */
export interface V1QueueControl {
    /**
     * Job id.
     * @type {string}
     * @memberof V1QueueControl
     */
    jobId: string;
    /**
     * Deprecated; do not use.
     * @type {string}
     * @memberof V1QueueControl
     */
    aheadOf?: string;
    /**
     * Deprecated; do not use.
     * @type {string}
     * @memberof V1QueueControl
     */
    behindOf?: string;
    /**
     * Name of the target resource_pool to move the job to.
     * @type {string}
     * @memberof V1QueueControl
     */
    resourcePool?: string;
    /**
     * The desired job priority in priority scheduler.
     * @type {number}
     * @memberof V1QueueControl
     */
    priority?: number;
    /**
     * The desired job weight in fairshare scheduler.
     * @type {number}
     * @memberof V1QueueControl
     */
    weight?: number;
}
/**
 * Statistics for a queue.
 * @export
 * @interface V1QueueStats
 */
export interface V1QueueStats {
    /**
     * Number of queued jobs in the queue.
     * @type {number}
     * @memberof V1QueueStats
     */
    queuedCount: number;
    /**
     * Number of scheduled jobs in the queue.
     * @type {number}
     * @memberof V1QueueStats
     */
    scheduledCount: number;
}
/**
 * RBACJob is a job that can have either a limited or a full representation of a job.
 * @export
 * @interface V1RBACJob
 */
export interface V1RBACJob {
    /**
     * Full representation.
     * @type {V1Job}
     * @memberof V1RBACJob
     */
    full?: V1Job;
    /**
     * Limited representation for lower access levels.
     * @type {V1LimitedJob}
     * @memberof V1RBACJob
     */
    limited?: V1LimitedJob;
}
/**
 * RemoveAssignmentsRequest is the body of the request for the call to remove a user or group from a role.
 * @export
 * @interface V1RemoveAssignmentsRequest
 */
export interface V1RemoveAssignmentsRequest {
    /**
     * the set of groups being removed from a role.
     * @type {Array<V1GroupRoleAssignment>}
     * @memberof V1RemoveAssignmentsRequest
     */
    groupRoleAssignments?: Array<V1GroupRoleAssignment>;
    /**
     * the set of users being removed from a role.
     * @type {Array<V1UserRoleAssignment>}
     * @memberof V1RemoveAssignmentsRequest
     */
    userRoleAssignments?: Array<V1UserRoleAssignment>;
}
/**
 * RemoveAssignmentsResponse is the body of the response for teh call to remove a user or group from a role.
 * @export
 * @interface V1RemoveAssignmentsResponse
 */
export interface V1RemoveAssignmentsResponse {
}
/**
 * The rendezvous info for the trial to rendezvous with sibling containers.
 * @export
 * @interface V1RendezvousInfo
 */
export interface V1RendezvousInfo {
    /**
     * The rendezvous addresses of the other containers.
     * @type {Array<string>}
     * @memberof V1RendezvousInfo
     */
    addresses: Array<string>;
    /**
     * The container rank.
     * @type {number}
     * @memberof V1RendezvousInfo
     */
    rank: number;
    /**
     * The slots for each address, respectively.
     * @type {Array<number>}
     * @memberof V1RendezvousInfo
     */
    slots: Array<number>;
}
/**
 * Response to ReportCheckpointRequest.
 * @export
 * @interface V1ReportCheckpointResponse
 */
export interface V1ReportCheckpointResponse {
}
/**
 * Persist the given metrics for the trial.
 * @export
 * @interface V1ReportTrialMetricsRequest
 */
export interface V1ReportTrialMetricsRequest {
    /**
     * The metrics to persist.
     * @type {V1TrialMetrics}
     * @memberof V1ReportTrialMetricsRequest
     */
    metrics: V1TrialMetrics;
    /**
     * The type of metrics to persist eg 'training', 'validation', etc.
     * @type {string}
     * @memberof V1ReportTrialMetricsRequest
     */
    group: string;
}
/**
 * 
 * @export
 * @interface V1ReportTrialMetricsResponse
 */
export interface V1ReportTrialMetricsResponse {
}
/**
 * For bookkeeping, updates the progress of the trial as a percent torwards the training length requested of it by the searcher.
 * @export
 * @interface V1ReportTrialProgressRequest
 */
export interface V1ReportTrialProgressRequest {
    /**
     * The id of the trial.
     * @type {number}
     * @memberof V1ReportTrialProgressRequest
     */
    trialId: number;
    /**
     * Total units completed by the trial, in terms of the unit used to configure the searcher.
     * @type {number}
     * @memberof V1ReportTrialProgressRequest
     */
    progress: number;
    /**
     * Indicator for if the passed progress is raw and we want to skip any calculation by searcher. Default to false.
     * @type {boolean}
     * @memberof V1ReportTrialProgressRequest
     */
    isRaw?: boolean;
}
/**
 * 
 * @export
 * @interface V1ReportTrialProgressResponse
 */
export interface V1ReportTrialProgressResponse {
}
/**
 * 
 * @export
 * @interface V1ReportTrialSearcherEarlyExitResponse
 */
export interface V1ReportTrialSearcherEarlyExitResponse {
}
/**
 * 
 * @export
 * @interface V1ReportTrialSourceInfoRequest
 */
export interface V1ReportTrialSourceInfoRequest {
    /**
     * Type of the TrialSourceInfo
     * @type {V1TrialSourceInfo}
     * @memberof V1ReportTrialSourceInfoRequest
     */
    trialSourceInfo: V1TrialSourceInfo;
}
/**
 * 
 * @export
 * @interface V1ReportTrialSourceInfoResponse
 */
export interface V1ReportTrialSourceInfoResponse {
    /**
     * Trial ID of the created
     * @type {number}
     * @memberof V1ReportTrialSourceInfoResponse
     */
    trialId: number;
    /**
     * UUID of the checkpoint.
     * @type {string}
     * @memberof V1ReportTrialSourceInfoResponse
     */
    checkpointUuid: string;
}
/**
 * 
 * @export
 * @interface V1ReportTrialTrainingMetricsResponse
 */
export interface V1ReportTrialTrainingMetricsResponse {
}
/**
 * 
 * @export
 * @interface V1ReportTrialValidationMetricsResponse
 */
export interface V1ReportTrialValidationMetricsResponse {
}
/**
 * Response to ResetUserSettingRequest.
 * @export
 * @interface V1ResetUserSettingResponse
 */
export interface V1ResetUserSettingResponse {
}
/**
 * One instance of slots in the cluster being allocated to a task during a period (aggregated).
 * @export
 * @interface V1ResourceAllocationAggregatedEntry
 */
export interface V1ResourceAllocationAggregatedEntry {
    /**
     * The date of this entry.
     * @type {string}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    periodStart: string;
    /**
     * The period over which aggregation occurred.
     * @type {V1ResourceAllocationAggregationPeriod}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    period: V1ResourceAllocationAggregationPeriod;
    /**
     * The total number of seconds included in this allocation entry.
     * @type {number}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    seconds: number;
    /**
     * The seconds in the cluster used by experiments belonging to each user.
     * @type {{ [key: string]: number; }}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    byUsername: { [key: string]: number; };
    /**
     * The seconds in the cluster used by experiments labeled with each label.
     * @type {{ [key: string]: number; }}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    byExperimentLabel: { [key: string]: number; };
    /**
     * The seconds in the cluster used by experiments assigned to each resource pool.
     * @type {{ [key: string]: number; }}
     * @memberof V1ResourceAllocationAggregatedEntry
     */
    byResourcePool: { [key: string]: number; };
}
/**
 * Response to ResourceAllocationAggregatedRequest.
 * @export
 * @interface V1ResourceAllocationAggregatedResponse
 */
export interface V1ResourceAllocationAggregatedResponse {
    /**
     * An entry summarizing one workload.
     * @type {Array<V1ResourceAllocationAggregatedEntry>}
     * @memberof V1ResourceAllocationAggregatedResponse
     */
    resourceEntries: Array<V1ResourceAllocationAggregatedEntry>;
}
/**
 * The period over which to perform aggregation.   - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY: Aggregation by day.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY: Aggregation by month.
 * @export
 * @enum {string}
 */
export const V1ResourceAllocationAggregationPeriod = {
    UNSPECIFIED: 'RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED',
    DAILY: 'RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY',
    MONTHLY: 'RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY',
} as const
export type V1ResourceAllocationAggregationPeriod = ValueOf<typeof V1ResourceAllocationAggregationPeriod>
/**
 * One instance of slots in the cluster being allocated to a task.
 * @export
 * @interface V1ResourceAllocationRawEntry
 */
export interface V1ResourceAllocationRawEntry {
    /**
     * The kind of workload being run during this allocation (training, checkpoint, or validation).
     * @type {string}
     * @memberof V1ResourceAllocationRawEntry
     */
    kind?: string;
    /**
     * The time at which the allocation began.
     * @type {Date | DateString}
     * @memberof V1ResourceAllocationRawEntry
     */
    startTime?: Date | DateString;
    /**
     * The time at which the allocation ended.
     * @type {Date | DateString}
     * @memberof V1ResourceAllocationRawEntry
     */
    endTime?: Date | DateString;
    /**
     * The ID of the experiment the allocation is a part of.
     * @type {number}
     * @memberof V1ResourceAllocationRawEntry
     */
    experimentId?: number;
    /**
     * The username of the user who ran the experiment.
     * @type {string}
     * @memberof V1ResourceAllocationRawEntry
     */
    username?: string;
    /**
     * The username of the user who ran the experiment.
     * @type {number}
     * @memberof V1ResourceAllocationRawEntry
     */
    userId?: number;
    /**
     * The labels assigned to the experiment.
     * @type {Array<string>}
     * @memberof V1ResourceAllocationRawEntry
     */
    labels?: Array<string>;
    /**
     * The number of seconds for which the allocation was active during the requested period.
     * @type {number}
     * @memberof V1ResourceAllocationRawEntry
     */
    seconds?: number;
    /**
     * The number of slots used by the allocation.
     * @type {number}
     * @memberof V1ResourceAllocationRawEntry
     */
    slots?: number;
}
/**
 * Response to ResourceAllocationRawRequest.
 * @export
 * @interface V1ResourceAllocationRawResponse
 */
export interface V1ResourceAllocationRawResponse {
    /**
     * An entry summarizing one workload.
     * @type {Array<V1ResourceAllocationRawEntry>}
     * @memberof V1ResourceAllocationRawResponse
     */
    resourceEntries?: Array<V1ResourceAllocationRawEntry>;
}
/**
 * A Resource Pool is a pool of resources where containers are run.
 * @export
 * @interface V1ResourcePool
 */
export interface V1ResourcePool {
    /**
     * The unique name of the resource pool.
     * @type {string}
     * @memberof V1ResourcePool
     */
    name: string;
    /**
     * The description of the resource pool
     * @type {string}
     * @memberof V1ResourcePool
     */
    description: string;
    /**
     * The type of the resource pool (AWS/GCP/STATIC)
     * @type {V1ResourcePoolType}
     * @memberof V1ResourcePool
     */
    type: V1ResourcePoolType;
    /**
     * The number of agents running in the resource pool
     * @type {number}
     * @memberof V1ResourcePool
     */
    numAgents: number;
    /**
     * The total number of slots that exist in the resource pool
     * @type {number}
     * @memberof V1ResourcePool
     */
    slotsAvailable: number;
    /**
     * The number of slots that are actively running workloads
     * @type {number}
     * @memberof V1ResourcePool
     */
    slotsUsed: number;
    /**
     * Slot device type: cpu, gpu, ...
     * @type {Devicev1Type}
     * @memberof V1ResourcePool
     */
    slotType: Devicev1Type;
    /**
     * The max number of aux containers that can run in this resource pool
     * @type {number}
     * @memberof V1ResourcePool
     */
    auxContainerCapacity: number;
    /**
     * The current number of aux containers running in this resource pool
     * @type {number}
     * @memberof V1ResourcePool
     */
    auxContainersRunning: number;
    /**
     * Is this resource pool the default compute pool?
     * @type {boolean}
     * @memberof V1ResourcePool
     */
    defaultComputePool: boolean;
    /**
     * Is this resource pool the default auxiliary pool?
     * @type {boolean}
     * @memberof V1ResourcePool
     */
    defaultAuxPool: boolean;
    /**
     * Is this resource pool using preemptible/spot instances? Only meaningful in an AWS or GCP resource pool.
     * @type {boolean}
     * @memberof V1ResourcePool
     */
    preemptible: boolean;
    /**
     * When using dynamic agents, the minimum number of agents that can exist in the resource pool.
     * @type {number}
     * @memberof V1ResourcePool
     */
    minAgents: number;
    /**
     * When using dynamic agents, the maximum number of agents that can exist in the resource pool.
     * @type {number}
     * @memberof V1ResourcePool
     */
    maxAgents: number;
    /**
     * The number of slots that exists on an dynamic agent.
     * @type {number}
     * @memberof V1ResourcePool
     */
    slotsPerAgent?: number;
    /**
     * The maximum number of auxiliary containers that can run on an individual agent
     * @type {number}
     * @memberof V1ResourcePool
     */
    auxContainerCapacityPerAgent: number;
    /**
     * The type of the scheduler. Either 'FAIR_SHARE', 'PRIORITY', or 'ROUND_ROBIN'
     * @type {V1SchedulerType}
     * @memberof V1ResourcePool
     */
    schedulerType: V1SchedulerType;
    /**
     * The fitting policy of the scheduler.
     * @type {V1FittingPolicy}
     * @memberof V1ResourcePool
     */
    schedulerFittingPolicy: V1FittingPolicy;
    /**
     * The location of the resource pool. For AWS this returns the region and for GCP this return the zone.
     * @type {string}
     * @memberof V1ResourcePool
     */
    location: string;
    /**
     * The VM image used for the agents when using dynamic agents.
     * @type {string}
     * @memberof V1ResourcePool
     */
    imageId: string;
    /**
     * The instance type of the agents when using dynamic agents. For AWS this is the Instance Type. For GCP this is the machine type combined with the number and types of GPUs. To work with this data programattically, we recommend working with the ResourcePool.details.aws.instanceType and ResourcePool.details.gcp.machineType/gpuType/gpuNum.
     * @type {string}
     * @memberof V1ResourcePool
     */
    instanceType: string;
    /**
     * The url of the Determined master
     * @type {string}
     * @memberof V1ResourcePool
     */
    masterUrl: string;
    /**
     * A hostname for which the master’s TLS certificate is valid, if the host specified by the master_url option is an IP address or is not contained in the certificate
     * @type {string}
     * @memberof V1ResourcePool
     */
    masterCertName: string;
    /**
     * The startup script for the agent. This runs on the node the agent runs on.
     * @type {string}
     * @memberof V1ResourcePool
     */
    startupScript: string;
    /**
     * The startup script for the agent's container. This runs in the container determined-agent runs in.
     * @type {string}
     * @memberof V1ResourcePool
     */
    containerStartupScript: string;
    /**
     * The Docker network to use for the agent when using dynamic agents.
     * @type {string}
     * @memberof V1ResourcePool
     */
    agentDockerNetwork: string;
    /**
     * The docker runtime to use for the agent when using dynamic agents
     * @type {string}
     * @memberof V1ResourcePool
     */
    agentDockerRuntime: string;
    /**
     * The docker image to use for the agent when using dynamic agents
     * @type {string}
     * @memberof V1ResourcePool
     */
    agentDockerImage: string;
    /**
     * the Fluent docker image to use
     * @type {string}
     * @memberof V1ResourcePool
     */
    agentFluentImage: string;
    /**
     * The maximum idle period of agents in seconds. The master waits for this period of time before shutting down idle agents.
     * @type {number}
     * @memberof V1ResourcePool
     */
    maxIdleAgentPeriod: number;
    /**
     * The maximum starting period of agents in seconds. The master waits for this period of time for starting agents before retrying.
     * @type {number}
     * @memberof V1ResourcePool
     */
    maxAgentStartingPeriod: number;
    /**
     * GCP, AWS and Priority Scheduler details
     * @type {V1ResourcePoolDetail}
     * @memberof V1ResourcePool
     */
    details: V1ResourcePoolDetail;
    /**
     * GCP, AWS accelerator information
     * @type {string}
     * @memberof V1ResourcePool
     */
    accelerator?: string;
    /**
     * Job queue stats
     * @type {V1QueueStats}
     * @memberof V1ResourcePool
     */
    stats?: V1QueueStats;
    /**
     * Resource manager's metadata.
     * @type {{ [key: string]: string; }}
     * @memberof V1ResourcePool
     */
    resourceManagerMetadata: { [key: string]: string; };
    /**
     * Resource manager's associated cluster name.
     * @type {string}
     * @memberof V1ResourcePool
     */
    clusterName: string;
}
/**
 * 
 * @export
 * @interface V1ResourcePoolAwsDetail
 */
export interface V1ResourcePoolAwsDetail {
    /**
     * The region the resource pool exists in
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    region: string;
    /**
     * Size of the root volume of the Determined agent in GB
     * @type {number}
     * @memberof V1ResourcePoolAwsDetail
     */
    rootVolumeSize: number;
    /**
     * The AMI ID of the Determined agent
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    imageId: string;
    /**
     * Key for tagging the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    tagKey: string;
    /**
     * Value for tagging the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    tagValue: string;
    /**
     * Name to set for the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    instanceName: string;
    /**
     * The name of the SSH key registered with AWS for SSH key access to the agent instances
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    sshKeyName: string;
    /**
     * Whether to use public IP addresses for the Determined agent
     * @type {boolean}
     * @memberof V1ResourcePoolAwsDetail
     */
    publicIp: boolean;
    /**
     * The ID of the subnet to run the Determined agents in
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    subnetId?: string;
    /**
     * The ID of the security group to run the Determined agents as
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    securityGroupId: string;
    /**
     * The Amazon Resource Name (ARN) of the IAM instance profile to attach to the agent instances.
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    iamInstanceProfileArn: string;
    /**
     * AWS instance type to use for dynamic agents
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    instanceType?: string;
    /**
     * The log group
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    logGroup?: string;
    /**
     * The log stream
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    logStream?: string;
    /**
     * Whether to use spot instances
     * @type {boolean}
     * @memberof V1ResourcePoolAwsDetail
     */
    spotEnabled: boolean;
    /**
     * The maximum price per hour to pay for a spot instance
     * @type {string}
     * @memberof V1ResourcePoolAwsDetail
     */
    spotMaxPrice?: string;
    /**
     * List of arbitrary user-defined tags that are added to the Determined agent instances
     * @type {Array<V1AwsCustomTag>}
     * @memberof V1ResourcePoolAwsDetail
     */
    customTags?: Array<V1AwsCustomTag>;
}
/**
 * 
 * @export
 * @interface V1ResourcePoolDetail
 */
export interface V1ResourcePoolDetail {
    /**
     * AWS-specific details
     * @type {V1ResourcePoolAwsDetail}
     * @memberof V1ResourcePoolDetail
     */
    aws?: V1ResourcePoolAwsDetail;
    /**
     * GCP-specific details
     * @type {V1ResourcePoolGcpDetail}
     * @memberof V1ResourcePoolDetail
     */
    gcp?: V1ResourcePoolGcpDetail;
    /**
     * Priority scheduler-specific details
     * @type {V1ResourcePoolPrioritySchedulerDetail}
     * @memberof V1ResourcePoolDetail
     */
    priorityScheduler?: V1ResourcePoolPrioritySchedulerDetail;
}
/**
 * 
 * @export
 * @interface V1ResourcePoolGcpDetail
 */
export interface V1ResourcePoolGcpDetail {
    /**
     * The project ID of the GCP resources used by Determined
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    project: string;
    /**
     * The zone of the GCP resources used by Determined
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    zone: string;
    /**
     * Size of the root volume of the Determined agent in GB
     * @type {number}
     * @memberof V1ResourcePoolGcpDetail
     */
    bootDiskSize: number;
    /**
     * The boot disk source image of the Determined agent
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    bootDiskSourceImage: string;
    /**
     * Key for labeling the Determined agent instances.
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    labelKey: string;
    /**
     * Value for labeling the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    labelValue: string;
    /**
     * Name prefix to set for the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    namePrefix: string;
    /**
     * Network resource for the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    network: string;
    /**
     * Subnetwork resource for the Determined agent instances
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    subnetwork?: string;
    /**
     * Whether to use external IP addresses for the Determined agent instances
     * @type {boolean}
     * @memberof V1ResourcePoolGcpDetail
     */
    externalIp: boolean;
    /**
     * The network tags to set firewalls for the Determined agent instances
     * @type {Array<string>}
     * @memberof V1ResourcePoolGcpDetail
     */
    networkTags?: Array<string>;
    /**
     * Email of the service account for the Determined agent instances.
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    serviceAccountEmail: string;
    /**
     * List of scopes authorized for the Determined agent instances
     * @type {Array<string>}
     * @memberof V1ResourcePoolGcpDetail
     */
    serviceAccountScopes: Array<string>;
    /**
     * Type of machine for the Determined agents
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    machineType: string;
    /**
     * Type of GPU for the Determined agents
     * @type {string}
     * @memberof V1ResourcePoolGcpDetail
     */
    gpuType: string;
    /**
     * Number of GPUs for the Determined agents
     * @type {number}
     * @memberof V1ResourcePoolGcpDetail
     */
    gpuNum: number;
    /**
     * Whether to use preemptible instances
     * @type {boolean}
     * @memberof V1ResourcePoolGcpDetail
     */
    preemptible: boolean;
    /**
     * The timeout period for tracking a GCP operation in seconds
     * @type {number}
     * @memberof V1ResourcePoolGcpDetail
     */
    operationTimeoutPeriod: number;
}
/**
 * 
 * @export
 * @interface V1ResourcePoolPrioritySchedulerDetail
 */
export interface V1ResourcePoolPrioritySchedulerDetail {
    /**
     * Whether lower priority tasks should be preempted to schedule higher priority tasks
     * @type {boolean}
     * @memberof V1ResourcePoolPrioritySchedulerDetail
     */
    preemption: boolean;
    /**
     * The priority that is assigned to tasks that do not explicitly specify a priority.
     * @type {number}
     * @memberof V1ResourcePoolPrioritySchedulerDetail
     */
    defaultPriority: number;
    /**
     * List of available priorities for K8 (if applicable).
     * @type {Array<V1K8PriorityClass>}
     * @memberof V1ResourcePoolPrioritySchedulerDetail
     */
    k8Priorities?: Array<V1K8PriorityClass>;
}
/**
 * The type of the ResourcePool.   - RESOURCE_POOL_TYPE_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_POOL_TYPE_AWS: An AWS resource pool.  - RESOURCE_POOL_TYPE_GCP: A GCP resource pool.  - RESOURCE_POOL_TYPE_STATIC: A static resource pool.  - RESOURCE_POOL_TYPE_K8S: The kubernetes resource pool.
 * @export
 * @enum {string}
 */
export const V1ResourcePoolType = {
    UNSPECIFIED: 'RESOURCE_POOL_TYPE_UNSPECIFIED',
    AWS: 'RESOURCE_POOL_TYPE_AWS',
    GCP: 'RESOURCE_POOL_TYPE_GCP',
    STATIC: 'RESOURCE_POOL_TYPE_STATIC',
    K8S: 'RESOURCE_POOL_TYPE_K8S',
} as const
export type V1ResourcePoolType = ValueOf<typeof V1ResourcePoolType>
/**
 * ResourcesFailure contains information about restored resources' failure.
 * @export
 * @interface V1ResourcesFailure
 */
export interface V1ResourcesFailure {
    /**
     * FailureType denotes the type of failure that resulted in the container stopping.
     * @type {V1FailureType}
     * @memberof V1ResourcesFailure
     */
    failureType?: V1FailureType;
    /**
     * The error message of the failure.
     * @type {string}
     * @memberof V1ResourcesFailure
     */
    errMsg?: string;
    /**
     * The exit code of the failure.
     * @type {number}
     * @memberof V1ResourcesFailure
     */
    exitCode?: number;
}
/**
 * ResourcesStarted contains the information needed by tasks from container started.
 * @export
 * @interface V1ResourcesStarted
 */
export interface V1ResourcesStarted {
    /**
     * Addresses represents the exposed ports on a container.
     * @type {Array<V1Address>}
     * @memberof V1ResourcesStarted
     */
    addresses?: Array<V1Address>;
    /**
     * NativeResourcesID is the native Docker hex container ID of the Determined container.
     * @type {string}
     * @memberof V1ResourcesStarted
     */
    nativeResourcesId?: string;
}
/**
 * ResourcesStopped contains the information needed by tasks from container stopped.
 * @export
 * @interface V1ResourcesStopped
 */
export interface V1ResourcesStopped {
    /**
     * ResourcesFailure contains information about restored resources' failure.
     * @type {V1ResourcesFailure}
     * @memberof V1ResourcesStopped
     */
    failure?: V1ResourcesFailure;
}
/**
 * ResourcesSummary provides a summary of the resources comprising what we know at the time the allocation is granted, but for k8s it is granted before being scheduled so it isn't really much and `agent_devices` are missing for k8s.
 * @export
 * @interface V1ResourcesSummary
 */
export interface V1ResourcesSummary {
    /**
     * ResourcesID is the ID of some set of resources.
     * @type {string}
     * @memberof V1ResourcesSummary
     */
    resourcesId?: string;
    /**
     * ResourcesType is the type of some set of resources. This should be purely informational.
     * @type {string}
     * @memberof V1ResourcesSummary
     */
    resourcesType?: string;
    /**
     * AllocationID is the ID of an allocation of a task.
     * @type {string}
     * @memberof V1ResourcesSummary
     */
    allocationId?: string;
    /**
     * ID, an identifier for an agent, maps to the associated devices.
     * @type {{ [key: string]: ResourcesSummaryDevices; }}
     * @memberof V1ResourcesSummary
     */
    agentDevices?: { [key: string]: ResourcesSummaryDevices; };
    /**
     * Available if the RM can give information on the container level.
     * @type {string}
     * @memberof V1ResourcesSummary
     */
    containerId?: string;
    /**
     * Available if the RM knows the resource is already started / exited.
     * @type {V1ResourcesStarted}
     * @memberof V1ResourcesSummary
     */
    started?: V1ResourcesStarted;
    /**
     * ResourcesStopped contains the information needed by tasks from container stopped.
     * @type {V1ResourcesStopped}
     * @memberof V1ResourcesSummary
     */
    exited?: V1ResourcesStopped;
}
/**
 * Request to unpause the experiment associated witha run.
 * @export
 * @interface V1ResumeRunsRequest
 */
export interface V1ResumeRunsRequest {
    /**
     * The ids of the runs being moved.
     * @type {Array<number>}
     * @memberof V1ResumeRunsRequest
     */
    runIds: Array<number>;
    /**
     * The id of the project of the runs being unpaused.
     * @type {number}
     * @memberof V1ResumeRunsRequest
     */
    projectId: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1ResumeRunsRequest
     */
    filter?: string;
}
/**
 * Response to ResumeRunsRequest.
 * @export
 * @interface V1ResumeRunsResponse
 */
export interface V1ResumeRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1ResumeRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * 
 * @export
 * @interface V1Role
 */
export interface V1Role {
    /**
     * The id of the role being detailed
     * @type {number}
     * @memberof V1Role
     */
    roleId: number;
    /**
     * The string of the role being detailed
     * @type {string}
     * @memberof V1Role
     */
    name?: string;
    /**
     * The permissions granted to the role
     * @type {Array<V1Permission>}
     * @memberof V1Role
     */
    permissions?: Array<V1Permission>;
    /**
     * Allowed scope types.
     * @type {V1ScopeTypeMask}
     * @memberof V1Role
     */
    scopeTypeMask?: V1ScopeTypeMask;
}
/**
 * RoleAssignment contains information about the scope of the role.
 * @export
 * @interface V1RoleAssignment
 */
export interface V1RoleAssignment {
    /**
     * The role of the assignment.
     * @type {V1Role}
     * @memberof V1RoleAssignment
     */
    role: V1Role;
    /**
     * The id of the workspace the role belongs to. Empty for cluster-wide scope.
     * @type {number}
     * @memberof V1RoleAssignment
     */
    scopeWorkspaceId?: number;
    /**
     * Whether the role is assigned cluster-wide.
     * @type {boolean}
     * @memberof V1RoleAssignment
     */
    scopeCluster?: boolean;
}
/**
 * RoleAssignmentSummary is used to describe permissions a user has.
 * @export
 * @interface V1RoleAssignmentSummary
 */
export interface V1RoleAssignmentSummary {
    /**
     * The id of the role being detailed
     * @type {number}
     * @memberof V1RoleAssignmentSummary
     */
    roleId: number;
    /**
     * List of workspace IDs to apply the role.
     * @type {Array<number>}
     * @memberof V1RoleAssignmentSummary
     */
    scopeWorkspaceIds?: Array<number>;
    /**
     * Whether the role is assigned cluster-wide.
     * @type {boolean}
     * @memberof V1RoleAssignmentSummary
     */
    scopeCluster?: boolean;
}
/**
 * RoleWithAssignments contains a detailed description of a role and the groups and users belonging to it.
 * @export
 * @interface V1RoleWithAssignments
 */
export interface V1RoleWithAssignments {
    /**
     * The embedded Role.
     * @type {V1Role}
     * @memberof V1RoleWithAssignments
     */
    role?: V1Role;
    /**
     * The embedded GroupRoleAssignment.
     * @type {Array<V1GroupRoleAssignment>}
     * @memberof V1RoleWithAssignments
     */
    groupRoleAssignments?: Array<V1GroupRoleAssignment>;
    /**
     * The embedded UserRoleAssignment.
     * @type {Array<V1UserRoleAssignment>}
     * @memberof V1RoleWithAssignments
     */
    userRoleAssignments?: Array<V1UserRoleAssignment>;
}
/**
 * Job stats for a resource pool.
 * @export
 * @interface V1RPQueueStat
 */
export interface V1RPQueueStat {
    /**
     * Job queue stats.
     * @type {V1QueueStats}
     * @memberof V1RPQueueStat
     */
    stats: V1QueueStats;
    /**
     * Resource pool.
     * @type {string}
     * @memberof V1RPQueueStat
     */
    resourcePool: string;
    /**
     * Aggregate stats.
     * @type {Array<V1AggregateQueueStats>}
     * @memberof V1RPQueueStat
     */
    aggregates?: Array<V1AggregateQueueStats>;
}
/**
 * Message for results of individual runs in a multi-run action.
 * @export
 * @interface V1RunActionResult
 */
export interface V1RunActionResult {
    /**
     * Optional error message.
     * @type {string}
     * @memberof V1RunActionResult
     */
    error: string;
    /**
     * run ID.
     * @type {number}
     * @memberof V1RunActionResult
     */
    id: number;
}
/**
 * Request to prepare to start reporting to a run.
 * @export
 * @interface V1RunPrepareForReportingRequest
 */
export interface V1RunPrepareForReportingRequest {
    /**
     * RunID to sync to.
     * @type {number}
     * @memberof V1RunPrepareForReportingRequest
     */
    runId: number;
    /**
     * Checkpoint storage config.
     * @type {any}
     * @memberof V1RunPrepareForReportingRequest
     */
    checkpointStorage?: any;
}
/**
 * Response to prepare to start reporting to a run.
 * @export
 * @interface V1RunPrepareForReportingResponse
 */
export interface V1RunPrepareForReportingResponse {
    /**
     * The storage_id to be used when creating new checkpoints. This will be returned always when checkpoint storage is set in the request.
     * @type {number}
     * @memberof V1RunPrepareForReportingResponse
     */
    storageId?: number;
}
/**
 * The type of the Scheduler.   - SCHEDULER_TYPE_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - SCHEDULER_TYPE_PRIORITY: The priority scheduler.  - SCHEDULER_TYPE_FAIR_SHARE: The fair share scheduler.  - SCHEDULER_TYPE_ROUND_ROBIN: The round robin scheduler  - SCHEDULER_TYPE_KUBERNETES: The kubernetes scheduler.  - SCHEDULER_TYPE_SLURM: A slurm placeholder. When running on slurm, all scheduling behavior is delegated.  - SCHEDULER_TYPE_PBS: A PBS placeholder. When running on PBS, all scheduling behavior is delegated.
 * @export
 * @enum {string}
 */
export const V1SchedulerType = {
    UNSPECIFIED: 'SCHEDULER_TYPE_UNSPECIFIED',
    PRIORITY: 'SCHEDULER_TYPE_PRIORITY',
    FAIRSHARE: 'SCHEDULER_TYPE_FAIR_SHARE',
    ROUNDROBIN: 'SCHEDULER_TYPE_ROUND_ROBIN',
    KUBERNETES: 'SCHEDULER_TYPE_KUBERNETES',
    SLURM: 'SCHEDULER_TYPE_SLURM',
    PBS: 'SCHEDULER_TYPE_PBS',
} as const
export type V1SchedulerType = ValueOf<typeof V1SchedulerType>
/**
 * 
 * @export
 * @interface V1ScopeTypeMask
 */
export interface V1ScopeTypeMask {
    /**
     * Whether this permission or role can be assigned globally, i.e. cluster-wide. Currently, all permissions can be assigned globally, so this is always true.
     * @type {boolean}
     * @memberof V1ScopeTypeMask
     */
    cluster?: boolean;
    /**
     * Whether this permission or role can be assigned on a particular workspace. For example, `ADMINISTRATE_USER` permission will have this field set to false, since user creation can only be done at a cluster level, and it doesn't make sense for a single workspace.
     * @type {boolean}
     * @memberof V1ScopeTypeMask
     */
    workspace?: boolean;
}
/**
 * 
 * @export
 * @interface V1SearchExperimentExperiment
 */
export interface V1SearchExperimentExperiment {
    /**
     * The experiment in question
     * @type {V1Experiment}
     * @memberof V1SearchExperimentExperiment
     */
    experiment: V1Experiment;
    /**
     * The best performing trial associated with the experiment
     * @type {Trialv1Trial}
     * @memberof V1SearchExperimentExperiment
     */
    bestTrial?: Trialv1Trial;
}
/**
 * 
 * @export
 * @interface V1SearchExperimentsRequest
 */
export interface V1SearchExperimentsRequest {
    /**
     * ID of the project to look at
     * @type {number}
     * @memberof V1SearchExperimentsRequest
     */
    projectId?: number;
    /**
     * How many experiments to skip before including in the results
     * @type {number}
     * @memberof V1SearchExperimentsRequest
     */
    offset?: number;
    /**
     * How many results to show
     * @type {number}
     * @memberof V1SearchExperimentsRequest
     */
    limit?: number;
    /**
     * Sort parameters in the format <col1>=(asc|desc),<col2>=(asc|desc)
     * @type {string}
     * @memberof V1SearchExperimentsRequest
     */
    sort?: string;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1SearchExperimentsRequest
     */
    filter?: string;
}
/**
 * 
 * @export
 * @interface V1SearchExperimentsResponse
 */
export interface V1SearchExperimentsResponse {
    /**
     * The list of returned experiments.
     * @type {Array<V1SearchExperimentExperiment>}
     * @memberof V1SearchExperimentsResponse
     */
    experiments: Array<V1SearchExperimentExperiment>;
    /**
     * Pagination information of the full dataset
     * @type {V1Pagination}
     * @memberof V1SearchExperimentsResponse
     */
    pagination: V1Pagination;
}
/**
 * 
 * @export
 * @interface V1SearchRolesAssignableToScopeRequest
 */
export interface V1SearchRolesAssignableToScopeRequest {
    /**
     * The maximum number of results to return
     * @type {number}
     * @memberof V1SearchRolesAssignableToScopeRequest
     */
    limit: number;
    /**
     * The offset to use with pagination
     * @type {number}
     * @memberof V1SearchRolesAssignableToScopeRequest
     */
    offset?: number;
    /**
     * The id of the workspace to use if searching for a workspace-assignable role
     * @type {number}
     * @memberof V1SearchRolesAssignableToScopeRequest
     */
    workspaceId?: number;
}
/**
 * 
 * @export
 * @interface V1SearchRolesAssignableToScopeResponse
 */
export interface V1SearchRolesAssignableToScopeResponse {
    /**
     * pagination information.
     * @type {V1Pagination}
     * @memberof V1SearchRolesAssignableToScopeResponse
     */
    pagination?: V1Pagination;
    /**
     * the set of roles and all assignments belonging to it.
     * @type {Array<V1Role>}
     * @memberof V1SearchRolesAssignableToScopeResponse
     */
    roles?: Array<V1Role>;
}
/**
 * Get a list of runs.
 * @export
 * @interface V1SearchRunsRequest
 */
export interface V1SearchRunsRequest {
    /**
     * ID of the project to look at
     * @type {number}
     * @memberof V1SearchRunsRequest
     */
    projectId?: number;
    /**
     * How many experiments to skip before including in the results
     * @type {number}
     * @memberof V1SearchRunsRequest
     */
    offset?: number;
    /**
     * How many results to show
     * @type {number}
     * @memberof V1SearchRunsRequest
     */
    limit?: number;
    /**
     * Sort parameters in the format <col1>=(asc|desc),<col2>=(asc|desc)
     * @type {string}
     * @memberof V1SearchRunsRequest
     */
    sort?: string;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1SearchRunsRequest
     */
    filter?: string;
}
/**
 * Response to SearchRunsResponse.
 * @export
 * @interface V1SearchRunsResponse
 */
export interface V1SearchRunsResponse {
    /**
     * The list of returned runs.
     * @type {Array<V1FlatRun>}
     * @memberof V1SearchRunsResponse
     */
    runs: Array<V1FlatRun>;
    /**
     * Pagination information of the full dataset.
     * @type {V1Pagination}
     * @memberof V1SearchRunsResponse
     */
    pagination: V1Pagination;
}
/**
 * SearchSummary contains the estimated trials and training lengths that a search plans to execute.
 * @export
 * @interface V1SearchSummary
 */
export interface V1SearchSummary {
    /**
     * The searcher config from which the summary is generated.
     * @type {any}
     * @memberof V1SearchSummary
     */
    config: any;
    /**
     * A list of planned number of trials to their training lengths.
     * @type {Array<V1TrialSummary>}
     * @memberof V1SearchSummary
     */
    trials?: Array<V1TrialSummary>;
}
/**
 * SearchUnit describes a length unit used by some searchers to manage training.
 * @export
 * @interface V1SearchUnit
 */
export interface V1SearchUnit {
    /**
     * Name of the length unit (if max_length is false).
     * @type {string}
     * @memberof V1SearchUnit
     */
    name?: string;
    /**
     * Value of the length unit (if max_length is false).
     * @type {number}
     * @memberof V1SearchUnit
     */
    value?: number;
    /**
     * Bool indicating whether the training length is defined in code.
     * @type {boolean}
     * @memberof V1SearchUnit
     */
    maxLength: boolean;
}
/**
 * Set the cluster-wide message.
 * @export
 * @interface V1SetClusterMessageRequest
 */
export interface V1SetClusterMessageRequest {
    /**
     * Text content of message.
     * @type {string}
     * @memberof V1SetClusterMessageRequest
     */
    message: string;
    /**
     * Time to begin showing message.
     * @type {Date | DateString}
     * @memberof V1SetClusterMessageRequest
     */
    startTime: Date | DateString;
    /**
     * Time to stop showing message.
     * @type {Date | DateString}
     * @memberof V1SetClusterMessageRequest
     */
    endTime?: Date | DateString;
    /**
     * Duration expressing how long the message should last. Should be a Go-format duration (e.g. 24h, 2w, 5d)
     * @type {string}
     * @memberof V1SetClusterMessageRequest
     */
    duration?: string;
}
/**
 * Response to SetClusterMessageRequest.
 * @export
 * @interface V1SetClusterMessageResponse
 */
export interface V1SetClusterMessageResponse {
}
/**
 * Set the priority of the requested command.
 * @export
 * @interface V1SetCommandPriorityRequest
 */
export interface V1SetCommandPriorityRequest {
    /**
     * The id of the command.
     * @type {string}
     * @memberof V1SetCommandPriorityRequest
     */
    commandId?: string;
    /**
     * The new priority.
     * @type {number}
     * @memberof V1SetCommandPriorityRequest
     */
    priority?: number;
}
/**
 * Response to SetCommandPriorityRequest.
 * @export
 * @interface V1SetCommandPriorityResponse
 */
export interface V1SetCommandPriorityResponse {
    /**
     * The requested command.
     * @type {V1Command}
     * @memberof V1SetCommandPriorityResponse
     */
    command?: V1Command;
}
/**
 * Set the priority of the requested notebook.
 * @export
 * @interface V1SetNotebookPriorityRequest
 */
export interface V1SetNotebookPriorityRequest {
    /**
     * The id of the notebook.
     * @type {string}
     * @memberof V1SetNotebookPriorityRequest
     */
    notebookId?: string;
    /**
     * The new priority.
     * @type {number}
     * @memberof V1SetNotebookPriorityRequest
     */
    priority?: number;
}
/**
 * Response to SetNotebookPriorityRequest.
 * @export
 * @interface V1SetNotebookPriorityResponse
 */
export interface V1SetNotebookPriorityResponse {
    /**
     * The requested notebook.
     * @type {V1Notebook}
     * @memberof V1SetNotebookPriorityResponse
     */
    notebook?: V1Notebook;
}
/**
 * Request to set a resource quota on a workspace (for a specific cluster).
 * @export
 * @interface V1SetResourceQuotasRequest
 */
export interface V1SetResourceQuotasRequest {
    /**
     * The id of the workspace.
     * @type {number}
     * @memberof V1SetResourceQuotasRequest
     */
    id: number;
    /**
     * A map of cluster to resource-quota pairs used to set GPU request limits on the workspace (for a specific cluster).
     * @type {{ [key: string]: number; }}
     * @memberof V1SetResourceQuotasRequest
     */
    clusterQuotaPairs?: { [key: string]: number; };
}
/**
 * Response for setting the resource quota for a workspace (for a specific cluster).
 * @export
 * @interface V1SetResourceQuotasResponse
 */
export interface V1SetResourceQuotasResponse {
}
/**
 * Set the priority of the requested shell.
 * @export
 * @interface V1SetShellPriorityRequest
 */
export interface V1SetShellPriorityRequest {
    /**
     * The id of the shell.
     * @type {string}
     * @memberof V1SetShellPriorityRequest
     */
    shellId?: string;
    /**
     * The new priority.
     * @type {number}
     * @memberof V1SetShellPriorityRequest
     */
    priority?: number;
}
/**
 * Response to SetShellPriorityRequest.
 * @export
 * @interface V1SetShellPriorityResponse
 */
export interface V1SetShellPriorityResponse {
    /**
     * The requested shell.
     * @type {V1Shell}
     * @memberof V1SetShellPriorityResponse
     */
    shell?: V1Shell;
}
/**
 * Set the priority of the requested TensorBoard.
 * @export
 * @interface V1SetTensorboardPriorityRequest
 */
export interface V1SetTensorboardPriorityRequest {
    /**
     * The id of the TensorBoard.
     * @type {string}
     * @memberof V1SetTensorboardPriorityRequest
     */
    tensorboardId?: string;
    /**
     * The new priority.
     * @type {number}
     * @memberof V1SetTensorboardPriorityRequest
     */
    priority?: number;
}
/**
 * Response to SetTensorboardPriorityRequest.
 * @export
 * @interface V1SetTensorboardPriorityResponse
 */
export interface V1SetTensorboardPriorityResponse {
    /**
     * The requested Tensorboard.
     * @type {V1Tensorboard}
     * @memberof V1SetTensorboardPriorityResponse
     */
    tensorboard?: V1Tensorboard;
}
/**
 * Response to SetUserPasswordRequest.
 * @export
 * @interface V1SetUserPasswordResponse
 */
export interface V1SetUserPasswordResponse {
    /**
     * The updated user.
     * @type {V1User}
     * @memberof V1SetUserPasswordResponse
     */
    user?: V1User;
}
/**
 * Request for setting workspace-namespace bindings.
 * @export
 * @interface V1SetWorkspaceNamespaceBindingsRequest
 */
export interface V1SetWorkspaceNamespaceBindingsRequest {
    /**
     * The unique id of the workspace.
     * @type {number}
     * @memberof V1SetWorkspaceNamespaceBindingsRequest
     */
    workspaceId: number;
    /**
     * The cluster-namespace pairs used to create workspace-namespace bindings.
     * @type {{ [key: string]: V1WorkspaceNamespaceMeta; }}
     * @memberof V1SetWorkspaceNamespaceBindingsRequest
     */
    clusterNamespaceMeta?: { [key: string]: V1WorkspaceNamespaceMeta; };
}
/**
 * Response for setting workspace-namespace bindings.
 * @export
 * @interface V1SetWorkspaceNamespaceBindingsResponse
 */
export interface V1SetWorkspaceNamespaceBindingsResponse {
    /**
     * The added workspace-namespace bindings.
     * @type {{ [key: string]: V1WorkspaceNamespaceBinding; }}
     * @memberof V1SetWorkspaceNamespaceBindingsResponse
     */
    namespaceBindings: { [key: string]: V1WorkspaceNamespaceBinding; };
}
/**
 * Shell is an ssh server in a containerized environment.
 * @export
 * @interface V1Shell
 */
export interface V1Shell {
    /**
     * The id of the shell.
     * @type {string}
     * @memberof V1Shell
     */
    id: string;
    /**
     * The description of the shell.
     * @type {string}
     * @memberof V1Shell
     */
    description: string;
    /**
     * The state of the shell.
     * @type {Taskv1State}
     * @memberof V1Shell
     */
    state: Taskv1State;
    /**
     * The time the shell was started.
     * @type {Date | DateString}
     * @memberof V1Shell
     */
    startTime: Date | DateString;
    /**
     * The container running the shell.
     * @type {V1Container}
     * @memberof V1Shell
     */
    container?: V1Container;
    /**
     * The private key for this shell.
     * @type {string}
     * @memberof V1Shell
     */
    privateKey?: string;
    /**
     * The public key for this shell.
     * @type {string}
     * @memberof V1Shell
     */
    publicKey?: string;
    /**
     * The display name of the user that created the shell.
     * @type {string}
     * @memberof V1Shell
     */
    displayName?: string;
    /**
     * The id of the user that created the shell.
     * @type {number}
     * @memberof V1Shell
     */
    userId?: number;
    /**
     * The username of the user that created the shell.
     * @type {string}
     * @memberof V1Shell
     */
    username: string;
    /**
     * The name of the resource pool the Shell was created in
     * @type {string}
     * @memberof V1Shell
     */
    resourcePool: string;
    /**
     * The exit status;
     * @type {string}
     * @memberof V1Shell
     */
    exitStatus?: string;
    /**
     * The addresses;
     * @type {Array<any>}
     * @memberof V1Shell
     */
    addresses?: Array<any>;
    /**
     * The agent user group;
     * @type {any}
     * @memberof V1Shell
     */
    agentUserGroup?: any;
    /**
     * The associated job id.
     * @type {string}
     * @memberof V1Shell
     */
    jobId: string;
    /**
     * The workspace id.
     * @type {number}
     * @memberof V1Shell
     */
    workspaceId: number;
}
/**
 * Slot wraps a single device on the agent.
 * @export
 * @interface V1Slot
 */
export interface V1Slot {
    /**
     * The unqiue id of the slot for a given agent.
     * @type {string}
     * @memberof V1Slot
     */
    id?: string;
    /**
     * The individual resource this slot wraps.
     * @type {V1Device}
     * @memberof V1Slot
     */
    device?: V1Device;
    /**
     * Flag notifying if containers can be scheduled on this slot.
     * @type {boolean}
     * @memberof V1Slot
     */
    enabled?: boolean;
    /**
     * Container that is currently running on this agent. It is unset if there is no container currently running on this slot.
     * @type {V1Container}
     * @memberof V1Slot
     */
    container?: V1Container;
    /**
     * Flag notifying if this slot is in the draining mode: current containers will be allowed to finish but no new ones will be scheduled.
     * @type {boolean}
     * @memberof V1Slot
     */
    draining?: boolean;
}
/**
 * SlotStats contains statistics about a set of slots.
 * @export
 * @interface V1SlotStats
 */
export interface V1SlotStats {
    /**
     * Map of device type to device stats.
     * @type {{ [key: string]: V1DeviceStats; }}
     * @memberof V1SlotStats
     */
    typeStats: { [key: string]: V1DeviceStats; };
    /**
     * Map of device brands to device stats.
     * @type {{ [key: string]: V1DeviceStats; }}
     * @memberof V1SlotStats
     */
    brandStats: { [key: string]: V1DeviceStats; };
}
/**
 * Describe one SSO provider.
 * @export
 * @interface V1SSOProvider
 */
export interface V1SSOProvider {
    /**
     * A descriptive name for this provider.
     * @type {string}
     * @memberof V1SSOProvider
     */
    name: string;
    /**
     * The URL to use for SSO with this provider.
     * @type {string}
     * @memberof V1SSOProvider
     */
    ssoUrl: string;
    /**
     * The type of SSO (such as SAML, OIDC).
     * @type {string}
     * @memberof V1SSOProvider
     */
    type: string;
    /**
     * The flag to indicate if this provider should be always be redirected to, instead of showing the standard sign-in page.
     * @type {boolean}
     * @memberof V1SSOProvider
     */
    alwaysRedirect?: boolean;
}
/**
 * Start a trial.
 * @export
 * @interface V1StartTrialRequest
 */
export interface V1StartTrialRequest {
    /**
     * Trial id.
     * @type {number}
     * @memberof V1StartTrialRequest
     */
    trialId: number;
    /**
     * Whether resume is allowed.
     * @type {boolean}
     * @memberof V1StartTrialRequest
     */
    resume?: boolean;
}
/**
 * Response to StartTrialRequest.
 * @export
 * @interface V1StartTrialResponse
 */
export interface V1StartTrialResponse {
    /**
     * Trial run id.
     * @type {number}
     * @memberof V1StartTrialResponse
     */
    trialRunId: number;
    /**
     * Latest checkpoint.
     * @type {string}
     * @memberof V1StartTrialResponse
     */
    latestCheckpoint?: string;
    /**
     * Steps completed.
     * @type {number}
     * @memberof V1StartTrialResponse
     */
    stepsCompleted: number;
}
/**
 * Project Table type.   - TABLE_TYPE_UNSPECIFIED: Unspecified table type.  - TABLE_TYPE_EXPERIMENT: experiment table.  - TABLE_TYPE_RUN: run table.
 * @export
 * @enum {string}
 */
export const V1TableType = {
    UNSPECIFIED: 'TABLE_TYPE_UNSPECIFIED',
    EXPERIMENT: 'TABLE_TYPE_EXPERIMENT',
    RUN: 'TABLE_TYPE_RUN',
} as const
export type V1TableType = ValueOf<typeof V1TableType>
/**
 * Task is the model for a task in the database.
 * @export
 * @interface V1Task
 */
export interface V1Task {
    /**
     * Unique ID of task.
     * @type {string}
     * @memberof V1Task
     */
    taskId: string;
    /**
     * Type of Task.
     * @type {V1TaskType}
     * @memberof V1Task
     */
    taskType: V1TaskType;
    /**
     * List of Allocations.
     * @type {Array<V1Allocation>}
     * @memberof V1Task
     */
    allocations: Array<V1Allocation>;
    /**
     * Start timestamp.
     * @type {Date | DateString}
     * @memberof V1Task
     */
    startTime: Date | DateString;
    /**
     * End timestamp if completed.
     * @type {Date | DateString}
     * @memberof V1Task
     */
    endTime?: Date | DateString;
    /**
     * The configuration of the task.
     * @type {string}
     * @memberof V1Task
     */
    config?: string;
    /**
     * ID of parent task (empty if root task).
     * @type {string}
     * @memberof V1Task
     */
    parentId?: string;
    /**
     * State of task execution.
     * @type {V1GenericTaskState}
     * @memberof V1Task
     */
    taskState?: V1GenericTaskState;
    /**
     * ID of task this is forked from (If task is a forked task)
     * @type {string}
     * @memberof V1Task
     */
    forkedFrom?: string;
    /**
     * Flag for whether task can be paused or not.
     * @type {boolean}
     * @memberof V1Task
     */
    noPause?: boolean;
}
/**
 * 
 * @export
 * @interface V1TaskLog
 */
export interface V1TaskLog {
    /**
     * The ID of the log.
     * @type {number}
     * @memberof V1TaskLog
     */
    id?: number;
    /**
     * The ID of the task.
     * @type {string}
     * @memberof V1TaskLog
     */
    taskId: string;
    /**
     * The ID of the allocation.
     * @type {string}
     * @memberof V1TaskLog
     */
    allocationId?: string;
    /**
     * The agent the logs came from.
     * @type {string}
     * @memberof V1TaskLog
     */
    agentId?: string;
    /**
     * The ID of the container or, in the case of k8s, the pod name.
     * @type {string}
     * @memberof V1TaskLog
     */
    containerId?: string;
    /**
     * The rank ID.
     * @type {number}
     * @memberof V1TaskLog
     */
    rankId?: number;
    /**
     * The timestamp of the log.
     * @type {Date | DateString}
     * @memberof V1TaskLog
     */
    timestamp?: Date | DateString;
    /**
     * The level of this log.
     * @type {V1LogLevel}
     * @memberof V1TaskLog
     */
    level?: V1LogLevel;
    /**
     * The text of the log entry.
     * @type {string}
     * @memberof V1TaskLog
     */
    log: string;
    /**
     * The source of the log entry.
     * @type {string}
     * @memberof V1TaskLog
     */
    source?: string;
    /**
     * The output stream (e.g. stdout, stderr).
     * @type {string}
     * @memberof V1TaskLog
     */
    stdtype?: string;
}
/**
 * Response to TaskLogsFieldsRequest.
 * @export
 * @interface V1TaskLogsFieldsResponse
 */
export interface V1TaskLogsFieldsResponse {
    /**
     * The distint allocation IDs present in the logs.
     * @type {Array<string>}
     * @memberof V1TaskLogsFieldsResponse
     */
    allocationIds?: Array<string>;
    /**
     * The distinct agent IDs present in the logs.
     * @type {Array<string>}
     * @memberof V1TaskLogsFieldsResponse
     */
    agentIds?: Array<string>;
    /**
     * The distinct container IDs present in the logs.
     * @type {Array<string>}
     * @memberof V1TaskLogsFieldsResponse
     */
    containerIds?: Array<string>;
    /**
     * The distinct rank IDs present in the logs.
     * @type {Array<number>}
     * @memberof V1TaskLogsFieldsResponse
     */
    rankIds?: Array<number>;
    /**
     * The distinct stdtypes present in the logs.
     * @type {Array<string>}
     * @memberof V1TaskLogsFieldsResponse
     */
    stdtypes?: Array<string>;
    /**
     * The distinct sources present in the logs.
     * @type {Array<string>}
     * @memberof V1TaskLogsFieldsResponse
     */
    sources?: Array<string>;
}
/**
 * Response to TaskLogsRequest.
 * @export
 * @interface V1TaskLogsResponse
 */
export interface V1TaskLogsResponse {
    /**
     * The ID of the log.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    id: string;
    /**
     * The timestamp of the log.
     * @type {Date | DateString}
     * @memberof V1TaskLogsResponse
     */
    timestamp: Date | DateString;
    /**
     * The flat version of the log that UIs have shown historically.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    message: string;
    /**
     * The level of the log.
     * @type {V1LogLevel}
     * @memberof V1TaskLogsResponse
     */
    level: V1LogLevel;
    /**
     * The ID of the task.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    taskId: string;
    /**
     * The ID of the allocation.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    allocationId?: string;
    /**
     * The agent the logs came from.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    agentId?: string;
    /**
     * The ID of the container or, in the case of k8s, the pod name.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    containerId?: string;
    /**
     * The rank ID.
     * @type {number}
     * @memberof V1TaskLogsResponse
     */
    rankId?: number;
    /**
     * The text of the log entry.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    log: string;
    /**
     * The source of the log entry.
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    source?: string;
    /**
     * The output stream (e.g. stdout, stderr).
     * @type {string}
     * @memberof V1TaskLogsResponse
     */
    stdtype?: string;
}
/**
 * Type of the task - TASK_TYPE_UNSPECIFIED: The task type is unknown  - TASK_TYPE_TRIAL: "TRIAL" task type for the enum public.task_type in Postgres.  - TASK_TYPE_NOTEBOOK: "NOTEBOOK" task type for the enum public.task_type in Postgres.  - TASK_TYPE_SHELL: "SHELL" task type for the enum public.task_type in Postgres.  - TASK_TYPE_COMMAND: "COMMAND" task type for the enum public.task_type in Postgres.  - TASK_TYPE_TENSORBOARD: "TENSORBOARD" task type for the enum public.task_type in Postgres.  - TASK_TYPE_CHECKPOINT_GC: "CHECKPOINT_GC" task type for the enum public.task_type in Postgres.  - TASK_TYPE_GENERIC: "GENERIC" task type for the enum public.task_type in Postgres.
 * @export
 * @enum {string}
 */
export const V1TaskType = {
    UNSPECIFIED: 'TASK_TYPE_UNSPECIFIED',
    TRIAL: 'TASK_TYPE_TRIAL',
    NOTEBOOK: 'TASK_TYPE_NOTEBOOK',
    SHELL: 'TASK_TYPE_SHELL',
    COMMAND: 'TASK_TYPE_COMMAND',
    TENSORBOARD: 'TASK_TYPE_TENSORBOARD',
    CHECKPOINTGC: 'TASK_TYPE_CHECKPOINT_GC',
    GENERIC: 'TASK_TYPE_GENERIC',
} as const
export type V1TaskType = ValueOf<typeof V1TaskType>
/**
 * Templates move settings that are shared by many experiments into a single YAML file.
 * @export
 * @interface V1Template
 */
export interface V1Template {
    /**
     * The name of the template.
     * @type {string}
     * @memberof V1Template
     */
    name: string;
    /**
     * The template value.
     * @type {any}
     * @memberof V1Template
     */
    config: any;
    /**
     * The id of the workspace associated with this model.
     * @type {number}
     * @memberof V1Template
     */
    workspaceId: number;
}
/**
 * Tensorboard is a tensorboard instance in a containerized environment.
 * @export
 * @interface V1Tensorboard
 */
export interface V1Tensorboard {
    /**
     * The id of the tensorboard.
     * @type {string}
     * @memberof V1Tensorboard
     */
    id: string;
    /**
     * The description of the tensorboard.
     * @type {string}
     * @memberof V1Tensorboard
     */
    description: string;
    /**
     * The state of the tensorboard.
     * @type {Taskv1State}
     * @memberof V1Tensorboard
     */
    state: Taskv1State;
    /**
     * The time the tensorboard was started.
     * @type {Date | DateString}
     * @memberof V1Tensorboard
     */
    startTime: Date | DateString;
    /**
     * The container running the tensorboard.
     * @type {V1Container}
     * @memberof V1Tensorboard
     */
    container?: V1Container;
    /**
     * The experiment ids loaded into this tensorboard instance.
     * @type {Array<number>}
     * @memberof V1Tensorboard
     */
    experimentIds?: Array<number>;
    /**
     * The trial ids loaded into this tensorboard instance.
     * @type {Array<number>}
     * @memberof V1Tensorboard
     */
    trialIds?: Array<number>;
    /**
     * The display name of the user that created the tensorboard.
     * @type {string}
     * @memberof V1Tensorboard
     */
    displayName?: string;
    /**
     * The id of the user that created the tensorboard.
     * @type {number}
     * @memberof V1Tensorboard
     */
    userId?: number;
    /**
     * The username of the user that created the tensorboard.
     * @type {string}
     * @memberof V1Tensorboard
     */
    username: string;
    /**
     * The service address.
     * @type {string}
     * @memberof V1Tensorboard
     */
    serviceAddress?: string;
    /**
     * The name of the resource pool the command was created in
     * @type {string}
     * @memberof V1Tensorboard
     */
    resourcePool: string;
    /**
     * The exit status;
     * @type {string}
     * @memberof V1Tensorboard
     */
    exitStatus?: string;
    /**
     * The associated job id.
     * @type {string}
     * @memberof V1Tensorboard
     */
    jobId: string;
    /**
     * The workspace id.
     * @type {number}
     * @memberof V1Tensorboard
     */
    workspaceId: number;
}
/**
 * Response to TestWebhookRequest.
 * @export
 * @interface V1TestWebhookResponse
 */
export interface V1TestWebhookResponse {
    /**
     * Status of test.
     * @type {boolean}
     * @memberof V1TestWebhookResponse
     */
    completed: boolean;
}
/**
 * Timestamp filters.
 * @export
 * @interface V1TimestampFieldFilter
 */
export interface V1TimestampFieldFilter {
    /**
     * Less than.
     * @type {Date | DateString}
     * @memberof V1TimestampFieldFilter
     */
    lt?: Date | DateString;
    /**
     * Less than or equal.
     * @type {Date | DateString}
     * @memberof V1TimestampFieldFilter
     */
    lte?: Date | DateString;
    /**
     * Greater than.
     * @type {Date | DateString}
     * @memberof V1TimestampFieldFilter
     */
    gt?: Date | DateString;
    /**
     * Greater than or equal.
     * @type {Date | DateString}
     * @memberof V1TimestampFieldFilter
     */
    gte?: Date | DateString;
}
/**
 * TokenInfo represents a token entry in the database.
 * @export
 * @interface V1TokenInfo
 */
export interface V1TokenInfo {
    /**
     * The token ID.
     * @type {number}
     * @memberof V1TokenInfo
     */
    id: number;
    /**
     * The id of the user the token belongs to.
     * @type {number}
     * @memberof V1TokenInfo
     */
    userId: number;
    /**
     * Timestamp of when the token expires.
     * @type {Date | DateString}
     * @memberof V1TokenInfo
     */
    expiry?: Date | DateString;
    /**
     * Tiemstamp of when the token was created.
     * @type {Date | DateString}
     * @memberof V1TokenInfo
     */
    createdAt?: Date | DateString;
    /**
     * Type of token this entry represents.
     * @type {V1TokenType}
     * @memberof V1TokenInfo
     */
    tokenType?: V1TokenType;
    /**
     * Flag denoting if this token is revoked.
     * @type {boolean}
     * @memberof V1TokenInfo
     */
    revoked?: boolean;
    /**
     * Description of the token.
     * @type {string}
     * @memberof V1TokenInfo
     */
    description?: string;
}
/**
 * Token type.   - TOKEN_TYPE_UNSPECIFIED: Default token type.  - TOKEN_TYPE_USER_SESSION: User Session token.  - TOKEN_TYPE_ACCESS_TOKEN: Access token.
 * @export
 * @enum {string}
 */
export const V1TokenType = {
    UNSPECIFIED: 'TOKEN_TYPE_UNSPECIFIED',
    USERSESSION: 'TOKEN_TYPE_USER_SESSION',
    ACCESSTOKEN: 'TOKEN_TYPE_ACCESS_TOKEN',
} as const
export type V1TokenType = ValueOf<typeof V1TokenType>
/**
 * Signals to the experiment the trial early exited.
 * @export
 * @interface V1TrialEarlyExit
 */
export interface V1TrialEarlyExit {
    /**
     * The reason for the exit.
     * @type {V1TrialEarlyExitExitedReason}
     * @memberof V1TrialEarlyExit
     */
    reason: V1TrialEarlyExitExitedReason;
}
/**
 * The reason for an early exit.   - EXITED_REASON_UNSPECIFIED: Zero-value (not allowed).  - EXITED_REASON_INVALID_HP: Indicates the trial exited due to an invalid hyperparameter.  - EXITED_REASON_INIT_INVALID_HP: Indicates the trial exited due to an invalid hyperparameter in the trial init.
 * @export
 * @enum {string}
 */
export const V1TrialEarlyExitExitedReason = {
    UNSPECIFIED: 'EXITED_REASON_UNSPECIFIED',
    INVALIDHP: 'EXITED_REASON_INVALID_HP',
    INITINVALIDHP: 'EXITED_REASON_INIT_INVALID_HP',
} as const
export type V1TrialEarlyExitExitedReason = ValueOf<typeof V1TrialEarlyExitExitedReason>
/**
 * Response to TrialLogFieldsRequest.
 * @export
 * @interface V1TrialLogsFieldsResponse
 */
export interface V1TrialLogsFieldsResponse {
    /**
     * The distinct agent IDs present in the logs.
     * @type {Array<string>}
     * @memberof V1TrialLogsFieldsResponse
     */
    agentIds?: Array<string>;
    /**
     * The distinct container IDs present in the logs.
     * @type {Array<string>}
     * @memberof V1TrialLogsFieldsResponse
     */
    containerIds?: Array<string>;
    /**
     * The distinct rank IDs present in the logs.
     * @type {Array<number>}
     * @memberof V1TrialLogsFieldsResponse
     */
    rankIds?: Array<number>;
    /**
     * The distinct stdtypes present in the logs.
     * @type {Array<string>}
     * @memberof V1TrialLogsFieldsResponse
     */
    stdtypes?: Array<string>;
    /**
     * The distinct sources present in the logs.
     * @type {Array<string>}
     * @memberof V1TrialLogsFieldsResponse
     */
    sources?: Array<string>;
}
/**
 * Response to TrialLogsRequest.
 * @export
 * @interface V1TrialLogsResponse
 */
export interface V1TrialLogsResponse {
    /**
     * The ID of the trial log.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    id: string;
    /**
     * The timestamp of the log.
     * @type {Date | DateString}
     * @memberof V1TrialLogsResponse
     */
    timestamp: Date | DateString;
    /**
     * The flat version of the log that UIs have shown historically.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    message: string;
    /**
     * The level of the log.
     * @type {V1LogLevel}
     * @memberof V1TrialLogsResponse
     */
    level: V1LogLevel;
    /**
     * The ID of the trial associated with this log entry.
     * @type {number}
     * @memberof V1TrialLogsResponse
     */
    trialId: number;
    /**
     * The ID of the agent that logged this.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    agentId?: string;
    /**
     * The ID of the container or, in the case of k8s, the pod name.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    containerId?: string;
    /**
     * The rank ID.
     * @type {number}
     * @memberof V1TrialLogsResponse
     */
    rankId?: number;
    /**
     * The text of the log entry.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    log?: string;
    /**
     * The source of the log entry.
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    source?: string;
    /**
     * The output stream (e.g. stdout, stderr).
     * @type {string}
     * @memberof V1TrialLogsResponse
     */
    stdtype?: string;
}
/**
 * Metrics from the trial some duration of training.
 * @export
 * @interface V1TrialMetrics
 */
export interface V1TrialMetrics {
    /**
     * The trial associated with these metrics.
     * @type {number}
     * @memberof V1TrialMetrics
     */
    trialId: number;
    /**
     * The trial run associated with these metrics.
     * @type {number}
     * @memberof V1TrialMetrics
     */
    trialRunId: number;
    /**
     * The number of batches trained on when these metrics were reported.
     * @type {number}
     * @memberof V1TrialMetrics
     */
    stepsCompleted?: number;
    /**
     * The client-reported time associated with these metrics.
     * @type {Date | DateString}
     * @memberof V1TrialMetrics
     */
    reportTime?: Date | DateString;
    /**
     * The metrics for this bit of training, including: - avg_metrics: metrics reduced over the reporting period). - batch_metrics: (optional) per-batch metrics.
     * @type {V1Metrics}
     * @memberof V1TrialMetrics
     */
    metrics: V1Metrics;
}
/**
 * 
 * @export
 * @interface V1TrialProfilerMetricLabels
 */
export interface V1TrialProfilerMetricLabels {
    /**
     * The ID of the trial.
     * @type {number}
     * @memberof V1TrialProfilerMetricLabels
     */
    trialId: number;
    /**
     * The name of the metric.
     * @type {string}
     * @memberof V1TrialProfilerMetricLabels
     */
    name: string;
    /**
     * The agent ID associated with the metric.
     * @type {string}
     * @memberof V1TrialProfilerMetricLabels
     */
    agentId?: string;
    /**
     * The GPU UUID associated with the metric.
     * @type {string}
     * @memberof V1TrialProfilerMetricLabels
     */
    gpuUuid?: string;
    /**
     * The type of the metric.
     * @type {TrialProfilerMetricLabelsProfilerMetricType}
     * @memberof V1TrialProfilerMetricLabels
     */
    metricType?: TrialProfilerMetricLabelsProfilerMetricType;
}
/**
 * TrialProfilerMetricsBatch is a batch of trial profiler metrics. A batch will contain metrics pertaining to a single series. The fields values, batches and timestamps will be equal length arrays with each index corresponding to a reading.
 * @export
 * @interface V1TrialProfilerMetricsBatch
 */
export interface V1TrialProfilerMetricsBatch {
    /**
     * The measurement for a reading, repeated for the batch of metrics.
     * @type {Array<number>}
     * @memberof V1TrialProfilerMetricsBatch
     */
    values: Array<number>;
    /**
     * The batch at which a reading occurred, repeated for the batch of metrics.
     * @type {Array<number>}
     * @memberof V1TrialProfilerMetricsBatch
     */
    batches: Array<number>;
    /**
     * The timestamp at which a reading occurred, repeated for the batch of metrics.
     * @type {Array<Date | DateString>}
     * @memberof V1TrialProfilerMetricsBatch
     */
    timestamps: Array<Date | DateString>;
    /**
     * The labels for this series.
     * @type {V1TrialProfilerMetricLabels}
     * @memberof V1TrialProfilerMetricsBatch
     */
    labels: V1TrialProfilerMetricLabels;
}
/**
 * The metadata pertaining to the current running task for a trial.
 * @export
 * @interface V1TrialRunnerMetadata
 */
export interface V1TrialRunnerMetadata {
    /**
     * The state of the trial runner.
     * @type {string}
     * @memberof V1TrialRunnerMetadata
     */
    state: string;
}
/**
 * 
 * @export
 * @interface V1TrialSourceInfo
 */
export interface V1TrialSourceInfo {
    /**
     * ID of the trial.
     * @type {number}
     * @memberof V1TrialSourceInfo
     */
    trialId: number;
    /**
     * UUID of the checkpoint.
     * @type {string}
     * @memberof V1TrialSourceInfo
     */
    checkpointUuid: string;
    /**
     * Source `id`` for the model which generated the checkpoint (if applicable)
     * @type {number}
     * @memberof V1TrialSourceInfo
     */
    modelId?: number;
    /**
     * Source `version` in the model_version version field which generated the checkpoint (if applicable)
     * @type {number}
     * @memberof V1TrialSourceInfo
     */
    modelVersion?: number;
    /**
     * Type for this trial_source_info
     * @type {V1TrialSourceInfoType}
     * @memberof V1TrialSourceInfo
     */
    trialSourceInfoType: V1TrialSourceInfoType;
}
/**
 * TrialSourceInfoType is the type of the TrialSourceInfo, which serves as a link between a trial and a checkpoint or model version - TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED: The type is unspecified  - TRIAL_SOURCE_INFO_TYPE_INFERENCE: "Inference" Trial Source Info Type, used for batch inference  - TRIAL_SOURCE_INFO_TYPE_FINE_TUNING: "Fine Tuning" Trial Source Info Type, used in model hub
 * @export
 * @enum {string}
 */
export const V1TrialSourceInfoType = {
    UNSPECIFIED: 'TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED',
    INFERENCE: 'TRIAL_SOURCE_INFO_TYPE_INFERENCE',
    FINETUNING: 'TRIAL_SOURCE_INFO_TYPE_FINE_TUNING',
} as const
export type V1TrialSourceInfoType = ValueOf<typeof V1TrialSourceInfoType>
/**
 * 
 * @export
 * @interface V1TrialsSampleResponse
 */
export interface V1TrialsSampleResponse {
    /**
     * A historical or incremental series of data points for the trials.
     * @type {Array<V1TrialsSampleResponseTrial>}
     * @memberof V1TrialsSampleResponse
     */
    trials: Array<V1TrialsSampleResponseTrial>;
    /**
     * IDs of trials that are newly included in the data.
     * @type {Array<number>}
     * @memberof V1TrialsSampleResponse
     */
    promotedTrials: Array<number>;
    /**
     * IDs of trials that are no longer included in the top N trials.
     * @type {Array<number>}
     * @memberof V1TrialsSampleResponse
     */
    demotedTrials: Array<number>;
}
/**
 * Metadata and metrics stream from a trial.
 * @export
 * @interface V1TrialsSampleResponseTrial
 */
export interface V1TrialsSampleResponseTrial {
    /**
     * The id of the trial.
     * @type {number}
     * @memberof V1TrialsSampleResponseTrial
     */
    trialId: number;
    /**
     * Hyperparamters values for this specific trial.
     * @type {any}
     * @memberof V1TrialsSampleResponseTrial
     */
    hparams: any;
    /**
     * A possibly down-sampled series of metric readings through the progress of the trial.
     * @type {Array<V1DataPoint>}
     * @memberof V1TrialsSampleResponseTrial
     */
    data: Array<V1DataPoint>;
}
/**
 * 
 * @export
 * @interface V1TrialsSnapshotResponse
 */
export interface V1TrialsSnapshotResponse {
    /**
     * A list of trials.
     * @type {Array<V1TrialsSnapshotResponseTrial>}
     * @memberof V1TrialsSnapshotResponse
     */
    trials: Array<V1TrialsSnapshotResponseTrial>;
}
/**
 * Metric value and metadata for a trial that has progress this far.
 * @export
 * @interface V1TrialsSnapshotResponseTrial
 */
export interface V1TrialsSnapshotResponseTrial {
    /**
     * The id of the trial.
     * @type {number}
     * @memberof V1TrialsSnapshotResponseTrial
     */
    trialId: number;
    /**
     * A dictionary of hyperparameter values for this trial.
     * @type {any}
     * @memberof V1TrialsSnapshotResponseTrial
     */
    hparams: any;
    /**
     * The value of the metric in this trial at this point.
     * @type {number}
     * @memberof V1TrialsSnapshotResponseTrial
     */
    metric: number;
    /**
     * The number of batches processed for this particular datapoint.
     * @type {number}
     * @memberof V1TrialsSnapshotResponseTrial
     */
    batchesProcessed: number;
}
/**
 * TrialSummary describes the runs that are estimated to train for a certain length.
 * @export
 * @interface V1TrialSummary
 */
export interface V1TrialSummary {
    /**
     * Number of trials.
     * @type {number}
     * @memberof V1TrialSummary
     */
    count: number;
    /**
     * Training length for the trials.
     * @type {V1SearchUnit}
     * @memberof V1TrialSummary
     */
    unit: V1SearchUnit;
}
/**
 * 
 * @export
 * @interface V1Trigger
 */
export interface V1Trigger {
    /**
     * The id of the trigger.
     * @type {number}
     * @memberof V1Trigger
     */
    id?: number;
    /**
     * The type of the trigger.
     * @type {V1TriggerType}
     * @memberof V1Trigger
     */
    triggerType?: V1TriggerType;
    /**
     * The trigger condition. For TRIGGER_TYPE_TASK_LOG needs {"regex": "abcd"}
     * @type {any}
     * @memberof V1Trigger
     */
    condition?: any;
    /**
     * The parent webhook of the trigger.
     * @type {number}
     * @memberof V1Trigger
     */
    webhookId?: number;
}
/**
 * Enum values for expected trigger types.   - TRIGGER_TYPE_UNSPECIFIED: Default value  - TRIGGER_TYPE_EXPERIMENT_STATE_CHANGE: For an experiment changing state  - TRIGGER_TYPE_METRIC_THRESHOLD_EXCEEDED: For metrics emitted during training.  - TRIGGER_TYPE_TASK_LOG: For task logs.  - TRIGGER_TYPE_CUSTOM: For custom alert.
 * @export
 * @enum {string}
 */
export const V1TriggerType = {
    UNSPECIFIED: 'TRIGGER_TYPE_UNSPECIFIED',
    EXPERIMENTSTATECHANGE: 'TRIGGER_TYPE_EXPERIMENT_STATE_CHANGE',
    METRICTHRESHOLDEXCEEDED: 'TRIGGER_TYPE_METRIC_THRESHOLD_EXCEEDED',
    TASKLOG: 'TRIGGER_TYPE_TASK_LOG',
    CUSTOM: 'TRIGGER_TYPE_CUSTOM',
} as const
export type V1TriggerType = ValueOf<typeof V1TriggerType>
/**
 * Response to UnarchiveExperimentRequest.
 * @export
 * @interface V1UnarchiveExperimentResponse
 */
export interface V1UnarchiveExperimentResponse {
}
/**
 * Unarchive multiple experiments.
 * @export
 * @interface V1UnarchiveExperimentsRequest
 */
export interface V1UnarchiveExperimentsRequest {
    /**
     * Selecting experiments by id.
     * @type {Array<number>}
     * @memberof V1UnarchiveExperimentsRequest
     */
    experimentIds: Array<number>;
    /**
     * Targets all experiments matching filters.
     * @type {V1BulkExperimentFilters}
     * @memberof V1UnarchiveExperimentsRequest
     */
    filters?: V1BulkExperimentFilters;
    /**
     * Project id that the experiments belong to.
     * @type {number}
     * @memberof V1UnarchiveExperimentsRequest
     */
    projectId: number;
}
/**
 * Response to UnarchiveExperimentsRequest.
 * @export
 * @interface V1UnarchiveExperimentsResponse
 */
export interface V1UnarchiveExperimentsResponse {
    /**
     * Details on success or error for each experiment.
     * @type {Array<V1ExperimentActionResult>}
     * @memberof V1UnarchiveExperimentsResponse
     */
    results: Array<V1ExperimentActionResult>;
}
/**
 * 
 * @export
 * @interface V1UnarchiveModelResponse
 */
export interface V1UnarchiveModelResponse {
}
/**
 * Response to UnarchiveProjectRequest.
 * @export
 * @interface V1UnarchiveProjectResponse
 */
export interface V1UnarchiveProjectResponse {
}
/**
 * 
 * @export
 * @interface V1UnarchiveRunsRequest
 */
export interface V1UnarchiveRunsRequest {
    /**
     * The ids of the runs being unarchived.
     * @type {Array<number>}
     * @memberof V1UnarchiveRunsRequest
     */
    runIds: Array<number>;
    /**
     * The id of the current parent project.
     * @type {number}
     * @memberof V1UnarchiveRunsRequest
     */
    projectId: number;
    /**
     * Filter expression
     * @type {string}
     * @memberof V1UnarchiveRunsRequest
     */
    filter?: string;
}
/**
 * Response to UnarchiveRunsRequest.
 * @export
 * @interface V1UnarchiveRunsResponse
 */
export interface V1UnarchiveRunsResponse {
    /**
     * Details on success or error for each run.
     * @type {Array<V1RunActionResult>}
     * @memberof V1UnarchiveRunsResponse
     */
    results: Array<V1RunActionResult>;
}
/**
 * Response to UnarchiveWorkspaceRequest.
 * @export
 * @interface V1UnarchiveWorkspaceResponse
 */
export interface V1UnarchiveWorkspaceResponse {
}
/**
 * Unbind a resource pool to workspaces.
 * @export
 * @interface V1UnbindRPFromWorkspaceRequest
 */
export interface V1UnbindRPFromWorkspaceRequest {
    /**
     * The resource pool name.
     * @type {string}
     * @memberof V1UnbindRPFromWorkspaceRequest
     */
    resourcePoolName: string;
    /**
     * The workspace IDs to be unbound.
     * @type {Array<number>}
     * @memberof V1UnbindRPFromWorkspaceRequest
     */
    workspaceIds?: Array<number>;
    /**
     * The workspace names to be unbound.
     * @type {Array<string>}
     * @memberof V1UnbindRPFromWorkspaceRequest
     */
    workspaceNames?: Array<string>;
}
/**
 * Unbind a resource pool to workspaces response.
 * @export
 * @interface V1UnbindRPFromWorkspaceResponse
 */
export interface V1UnbindRPFromWorkspaceResponse {
}
/**
 * 
 * @export
 * @interface V1UnpauseGenericTaskResponse
 */
export interface V1UnpauseGenericTaskResponse {
}
/**
 * Response to UnpinWorkspaceRequest.
 * @export
 * @interface V1UnpinWorkspaceResponse
 */
export interface V1UnpinWorkspaceResponse {
}
/**
 * UpdateGroupRequest is the body of the request for the call to update a group and its members.
 * @export
 * @interface V1UpdateGroupRequest
 */
export interface V1UpdateGroupRequest {
    /**
     * The id of the group
     * @type {number}
     * @memberof V1UpdateGroupRequest
     */
    groupId: number;
    /**
     * The name of the group
     * @type {string}
     * @memberof V1UpdateGroupRequest
     */
    name?: string;
    /**
     * The user ids of users to add to the group
     * @type {Array<number>}
     * @memberof V1UpdateGroupRequest
     */
    addUsers?: Array<number>;
    /**
     * The user ids of users to delete from the group
     * @type {Array<number>}
     * @memberof V1UpdateGroupRequest
     */
    removeUsers?: Array<number>;
}
/**
 * UpdateGroupResponse is the body of the response for the call to update a group and its members.
 * @export
 * @interface V1UpdateGroupResponse
 */
export interface V1UpdateGroupResponse {
    /**
     * Info about the group after the update succeeded.
     * @type {V1GroupDetails}
     * @memberof V1UpdateGroupResponse
     */
    group: V1GroupDetails;
}
/**
 * Request to update the job queue.
 * @export
 * @interface V1UpdateJobQueueRequest
 */
export interface V1UpdateJobQueueRequest {
    /**
     * List of job queue control requests.
     * @type {Array<V1QueueControl>}
     * @memberof V1UpdateJobQueueRequest
     */
    updates: Array<V1QueueControl>;
}
/**
 * Response to UpdateJobQueueRequest.
 * @export
 * @interface V1UpdateJobQueueResponse
 */
export interface V1UpdateJobQueueResponse {
}
/**
 * User is an account in the determined cluster.
 * @export
 * @interface V1User
 */
export interface V1User {
    /**
     * The user ID.
     * @type {number}
     * @memberof V1User
     */
    id?: number;
    /**
     * The user login name of the user.
     * @type {string}
     * @memberof V1User
     */
    username: string;
    /**
     * Bool denoting whether the account is an admin account.
     * @type {boolean}
     * @memberof V1User
     */
    admin: boolean;
    /**
     * Bool denoting whether the account is active.
     * @type {boolean}
     * @memberof V1User
     */
    active: boolean;
    /**
     * The user and group on the agent host machine.
     * @type {V1AgentUserGroup}
     * @memberof V1User
     */
    agentUserGroup?: V1AgentUserGroup;
    /**
     * Name to display in the web UI.
     * @type {string}
     * @memberof V1User
     */
    displayName?: string;
    /**
     * The version of the user object for caching purposes.
     * @type {Date | DateString}
     * @memberof V1User
     */
    modifiedAt?: Date | DateString;
    /**
     * Bool denoting whether the user should be able to login with or change a password.
     * @type {boolean}
     * @memberof V1User
     */
    remote?: boolean;
    /**
     * when the user last authenticated
     * @type {Date | DateString}
     * @memberof V1User
     */
    lastAuthAt?: Date | DateString;
}
/**
 * Message for results of individual users in a multi-user action.
 * @export
 * @interface V1UserActionResult
 */
export interface V1UserActionResult {
    /**
     * Optional error message.
     * @type {string}
     * @memberof V1UserActionResult
     */
    error: string;
    /**
     * User ID.
     * @type {number}
     * @memberof V1UserActionResult
     */
    id: number;
}
/**
 * Options to filter a subset of users.
 * @export
 * @interface V1UserFilters
 */
export interface V1UserFilters {
    /**
     * Case-insensitive partial match of string to username or display name.
     * @type {string}
     * @memberof V1UserFilters
     */
    name?: string;
    /**
     * Matches users with or without an admin flag.
     * @type {boolean}
     * @memberof V1UserFilters
     */
    admin?: boolean;
}
/**
 * UserRoleAssignment contains information about the users belonging to a role.
 * @export
 * @interface V1UserRoleAssignment
 */
export interface V1UserRoleAssignment {
    /**
     * the user id of the role assignment
     * @type {number}
     * @memberof V1UserRoleAssignment
     */
    userId: number;
    /**
     * the role and scope of the assignment
     * @type {V1RoleAssignment}
     * @memberof V1UserRoleAssignment
     */
    roleAssignment: V1RoleAssignment;
}
/**
 * UserWebSetting represents user web setting.
 * @export
 * @interface V1UserWebSetting
 */
export interface V1UserWebSetting {
    /**
     * The key of setting.
     * @type {string}
     * @memberof V1UserWebSetting
     */
    key: string;
    /**
     * The storage path of setting.
     * @type {string}
     * @memberof V1UserWebSetting
     */
    storagePath?: string;
    /**
     * The value of setting.
     * @type {string}
     * @memberof V1UserWebSetting
     */
    value?: string;
}
/**
 * ValidationHistoryEntry is a single entry for a validation history for an experiment.
 * @export
 * @interface V1ValidationHistoryEntry
 */
export interface V1ValidationHistoryEntry {
    /**
     * The id for the trial associated with this validation entry.
     * @type {number}
     * @memberof V1ValidationHistoryEntry
     */
    trialId: number;
    /**
     * The time at which the completed validation was reported.
     * @type {Date | DateString}
     * @memberof V1ValidationHistoryEntry
     */
    endTime: Date | DateString;
    /**
     * The value of the `searcher.metric`, indicated by the experiment config, for the validation.
     * @type {number}
     * @memberof V1ValidationHistoryEntry
     */
    searcherMetric: number;
}
/**
 * 
 * @export
 * @interface V1Webhook
 */
export interface V1Webhook {
    /**
     * The id of the webhook.
     * @type {number}
     * @memberof V1Webhook
     */
    id?: number;
    /**
     * The url of the webhook.
     * @type {string}
     * @memberof V1Webhook
     */
    url: string;
    /**
     * The triggers of the webhook.
     * @type {Array<V1Trigger>}
     * @memberof V1Webhook
     */
    triggers?: Array<V1Trigger>;
    /**
     * The type of the webhook.
     * @type {V1WebhookType}
     * @memberof V1Webhook
     */
    webhookType: V1WebhookType;
    /**
     * The name of the webhook.
     * @type {string}
     * @memberof V1Webhook
     */
    name: string;
    /**
     * The workspace of the webhook.
     * @type {number}
     * @memberof V1Webhook
     */
    workspaceId?: number;
    /**
     * The mode of the webhook.
     * @type {V1WebhookMode}
     * @memberof V1Webhook
     */
    mode: V1WebhookMode;
}
/**
 * Enum values for webhook mode.   - WEBHOOK_MODE_UNSPECIFIED: Default value  - WEBHOOK_MODE_WORKSPACE: Webhook will be triggered by all experiment in the workspace  - WEBHOOK_MODE_SPECIFIC: Webhook will only be triggered by experiment with matching configuration in the same workspace as the web hook
 * @export
 * @enum {string}
 */
export const V1WebhookMode = {
    UNSPECIFIED: 'WEBHOOK_MODE_UNSPECIFIED',
    WORKSPACE: 'WEBHOOK_MODE_WORKSPACE',
    SPECIFIC: 'WEBHOOK_MODE_SPECIFIC',
} as const
export type V1WebhookMode = ValueOf<typeof V1WebhookMode>
/**
 * Enum values for expected webhook types.   - WEBHOOK_TYPE_UNSPECIFIED: Default value  - WEBHOOK_TYPE_DEFAULT: For a default webhook  - WEBHOOK_TYPE_SLACK: For a slack webhook.
 * @export
 * @enum {string}
 */
export const V1WebhookType = {
    UNSPECIFIED: 'WEBHOOK_TYPE_UNSPECIFIED',
    DEFAULT: 'WEBHOOK_TYPE_DEFAULT',
    SLACK: 'WEBHOOK_TYPE_SLACK',
} as const
export type V1WebhookType = ValueOf<typeof V1WebhookType>
/**
 * WorkloadContainer is a wrapper for Determined workloads to allow repeated oneof types.
 * @export
 * @interface V1WorkloadContainer
 */
export interface V1WorkloadContainer {
    /**
     * Training workload.
     * @type {V1MetricsWorkload}
     * @memberof V1WorkloadContainer
     */
    training?: V1MetricsWorkload;
    /**
     * Validation workload.
     * @type {V1MetricsWorkload}
     * @memberof V1WorkloadContainer
     */
    validation?: V1MetricsWorkload;
    /**
     * Checkpoint workload.
     * @type {V1CheckpointWorkload}
     * @memberof V1WorkloadContainer
     */
    checkpoint?: V1CheckpointWorkload;
}
/**
 * Workspace is a named collection of projects.
 * @export
 * @interface V1Workspace
 */
export interface V1Workspace {
    /**
     * The unique id of the workspace.
     * @type {number}
     * @memberof V1Workspace
     */
    id: number;
    /**
     * The unique name of the workspace.
     * @type {string}
     * @memberof V1Workspace
     */
    name: string;
    /**
     * Whether this workspace is archived or not.
     * @type {boolean}
     * @memberof V1Workspace
     */
    archived: boolean;
    /**
     * User who created this workspace.
     * @type {string}
     * @memberof V1Workspace
     */
    username: string;
    /**
     * Whether this workspace is immutable (default uncategorized workspace).
     * @type {boolean}
     * @memberof V1Workspace
     */
    immutable: boolean;
    /**
     * Number of projects associated with this workspace.
     * @type {number}
     * @memberof V1Workspace
     */
    numProjects: number;
    /**
     * Pin status of this workspace for the current user.
     * @type {boolean}
     * @memberof V1Workspace
     */
    pinned: boolean;
    /**
     * ID of the user who created this project.
     * @type {number}
     * @memberof V1Workspace
     */
    userId: number;
    /**
     * Number of experiments associated with this workspace.
     * @type {number}
     * @memberof V1Workspace
     */
    numExperiments: number;
    /**
     * State of workspace during deletion.
     * @type {V1WorkspaceState}
     * @memberof V1Workspace
     */
    state: V1WorkspaceState;
    /**
     * Message stored from errors on async-deleting a workspace.
     * @type {string}
     * @memberof V1Workspace
     */
    errorMessage: string;
    /**
     * Optional agent host uid and gid override.
     * @type {V1AgentUserGroup}
     * @memberof V1Workspace
     */
    agentUserGroup?: V1AgentUserGroup;
    /**
     * Optional checkpoint storage config. Expects same format as experiment config's checkpoint storage.
     * @type {any}
     * @memberof V1Workspace
     */
    checkpointStorageConfig?: any;
    /**
     * Optional date when workspace was pinned.
     * @type {Date | DateString}
     * @memberof V1Workspace
     */
    pinnedAt?: Date | DateString;
    /**
     * Name of the default compute pool.
     * @type {string}
     * @memberof V1Workspace
     */
    defaultComputePool?: string;
    /**
     * Name of the default aux pool.
     * @type {string}
     * @memberof V1Workspace
     */
    defaultAuxPool?: string;
    /**
     * Optional auto-created namespace bound to the workspace.
     * @type {string}
     * @memberof V1Workspace
     */
    autoCreatedNamespace?: string;
}
/**
 * WorkspaceNamespace represents a workspace-namespace binding for a given workspace and cluster.
 * @export
 * @interface V1WorkspaceNamespaceBinding
 */
export interface V1WorkspaceNamespaceBinding {
    /**
     * The id of the workspace.
     * @type {number}
     * @memberof V1WorkspaceNamespaceBinding
     */
    workspaceId?: number;
    /**
     * The Kubernetes namespace.
     * @type {string}
     * @memberof V1WorkspaceNamespaceBinding
     */
    namespace?: string;
    /**
     * User cluster name.
     * @type {string}
     * @memberof V1WorkspaceNamespaceBinding
     */
    clusterName?: string;
    /**
     * Whether the namespace was auto-created for a workspace-namespace binding.
     * @type {boolean}
     * @memberof V1WorkspaceNamespaceBinding
     */
    autoCreateNamespace?: boolean;
}
/**
 * WorkspaceNamespaceMeta is the metadata associated with a workspace-namespace binding.
 * @export
 * @interface V1WorkspaceNamespaceMeta
 */
export interface V1WorkspaceNamespaceMeta {
    /**
     * The optional namespace of the workspace-namespace binding.
     * @type {string}
     * @memberof V1WorkspaceNamespaceMeta
     */
    namespace?: string;
    /**
     * Whether we want to auto-create a namespace for a workspace-namespace binding.
     * @type {boolean}
     * @memberof V1WorkspaceNamespaceMeta
     */
    autoCreateNamespace?: boolean;
    /**
     * Whether we want to auto-create a namespace for each cluster's workspace-namespace binding.
     * @type {boolean}
     * @memberof V1WorkspaceNamespaceMeta
     */
    autoCreateNamespaceAllClusters?: boolean;
    /**
     * The cluster to which we apply the workspace-namespace binding or resource quota.
     * @type {string}
     * @memberof V1WorkspaceNamespaceMeta
     */
    clusterName?: string;
    /**
     * The optional resource quota placed on the namespace (and consequentially, the workspace). TODO (CM-495): Remove this field and use GetKubernetesResourceQuotasRequest instead.
     * @type {number}
     * @memberof V1WorkspaceNamespaceMeta
     */
    resourceQuota?: number;
}
/**
 * WorkspaceState is used to track progress during a deletion.   - WORKSPACE_STATE_UNSPECIFIED: Object deletion is not in progress.  - WORKSPACE_STATE_DELETING: The object is being deleted.  - WORKSPACE_STATE_DELETE_FAILED: The object failed to delete.  - WORKSPACE_STATE_DELETED: The object finished deleting.
 * @export
 * @enum {string}
 */
export const V1WorkspaceState = {
    UNSPECIFIED: 'WORKSPACE_STATE_UNSPECIFIED',
    DELETING: 'WORKSPACE_STATE_DELETING',
    DELETEFAILED: 'WORKSPACE_STATE_DELETE_FAILED',
    DELETED: 'WORKSPACE_STATE_DELETED',
} as const
export type V1WorkspaceState = ValueOf<typeof V1WorkspaceState>
/**
 * AlphaApi - fetch parameter creator
 * @export
 */
export const AlphaApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Delete global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteGlobalConfigPolicies(workloadType: string, options: any = {}): FetchArgs {
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling deleteGlobalConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/global/{workloadType}`
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options: any = {}): FetchArgs {
            // verify required parameter 'workspaceId' is not null or undefined
            if (workspaceId === null || workspaceId === undefined) {
                throw new RequiredError('workspaceId','Required parameter workspaceId was null or undefined when calling deleteWorkspaceConfigPolicies.');
            }
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling deleteWorkspaceConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/workspaces/{workspaceId}/{workloadType}`
                .replace(`{${"workspaceId"}}`, encodeURIComponent(String(workspaceId)))
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGlobalConfigPolicies(workloadType: string, options: any = {}): FetchArgs {
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling getGlobalConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/global/{workloadType}`
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options: any = {}): FetchArgs {
            // verify required parameter 'workspaceId' is not null or undefined
            if (workspaceId === null || workspaceId === undefined) {
                throw new RequiredError('workspaceId','Required parameter workspaceId was null or undefined when calling getWorkspaceConfigPolicies.');
            }
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling getWorkspaceConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/workspaces/{workspaceId}/{workloadType}`
                .replace(`{${"workspaceId"}}`, encodeURIComponent(String(workspaceId)))
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Add or update global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutGlobalConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putGlobalConfigPolicies(workloadType: string, body: V1PutGlobalConfigPoliciesRequest, options: any = {}): FetchArgs {
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling putGlobalConfigPolicies.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putGlobalConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/global/{workloadType}`
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Add or update workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutWorkspaceConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putWorkspaceConfigPolicies(workspaceId: number, workloadType: string, body: V1PutWorkspaceConfigPoliciesRequest, options: any = {}): FetchArgs {
            // verify required parameter 'workspaceId' is not null or undefined
            if (workspaceId === null || workspaceId === undefined) {
                throw new RequiredError('workspaceId','Required parameter workspaceId was null or undefined when calling putWorkspaceConfigPolicies.');
            }
            // verify required parameter 'workloadType' is not null or undefined
            if (workloadType === null || workloadType === undefined) {
                throw new RequiredError('workloadType','Required parameter workloadType was null or undefined when calling putWorkspaceConfigPolicies.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putWorkspaceConfigPolicies.');
            }
            const localVarPath = `/api/v1/config-policies/workspaces/{workspaceId}/{workloadType}`
                .replace(`{${"workspaceId"}}`, encodeURIComponent(String(workspaceId)))
                .replace(`{${"workloadType"}}`, encodeURIComponent(String(workloadType)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * AlphaApi - functional programming interface
 * @export
 */
export const AlphaApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Delete global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteGlobalConfigPolicies(workloadType: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteGlobalConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).deleteGlobalConfigPolicies(workloadType, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteWorkspaceConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).deleteWorkspaceConfigPolicies(workspaceId, workloadType, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGlobalConfigPolicies(workloadType: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetGlobalConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).getGlobalConfigPolicies(workloadType, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetWorkspaceConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).getWorkspaceConfigPolicies(workspaceId, workloadType, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Add or update global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutGlobalConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putGlobalConfigPolicies(workloadType: string, body: V1PutGlobalConfigPoliciesRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutGlobalConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).putGlobalConfigPolicies(workloadType, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Add or update workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutWorkspaceConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putWorkspaceConfigPolicies(workspaceId: number, workloadType: string, body: V1PutWorkspaceConfigPoliciesRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutWorkspaceConfigPoliciesResponse> {
            const localVarFetchArgs = AlphaApiFetchParamCreator(configuration).putWorkspaceConfigPolicies(workspaceId, workloadType, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * AlphaApi - factory interface
 * @export
 */
export const AlphaApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Delete global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteGlobalConfigPolicies(workloadType: string, options?: any) {
            return AlphaApiFp(configuration).deleteGlobalConfigPolicies(workloadType, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any) {
            return AlphaApiFp(configuration).deleteWorkspaceConfigPolicies(workspaceId, workloadType, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGlobalConfigPolicies(workloadType: string, options?: any) {
            return AlphaApiFp(configuration).getGlobalConfigPolicies(workloadType, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any) {
            return AlphaApiFp(configuration).getWorkspaceConfigPolicies(workspaceId, workloadType, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Add or update global task config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutGlobalConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putGlobalConfigPolicies(workloadType: string, body: V1PutGlobalConfigPoliciesRequest, options?: any) {
            return AlphaApiFp(configuration).putGlobalConfigPolicies(workloadType, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Add or update workspace task config policies.
         * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
         * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
         * @param {V1PutWorkspaceConfigPoliciesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putWorkspaceConfigPolicies(workspaceId: number, workloadType: string, body: V1PutWorkspaceConfigPoliciesRequest, options?: any) {
            return AlphaApiFp(configuration).putWorkspaceConfigPolicies(workspaceId, workloadType, body, options)(fetch, basePath);
        },
    }
};

/**
 * AlphaApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class AlphaApi extends BaseAPI {
    /**
     * 
     * @summary Delete global task config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public deleteGlobalConfigPolicies(workloadType: string, options?: any) {
        return AlphaApiFp(this.configuration).deleteGlobalConfigPolicies(workloadType, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete workspace task config policies.
     * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public deleteWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any) {
        return AlphaApiFp(this.configuration).deleteWorkspaceConfigPolicies(workspaceId, workloadType, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get global task config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public getGlobalConfigPolicies(workloadType: string, options?: any) {
        return AlphaApiFp(this.configuration).getGlobalConfigPolicies(workloadType, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get workspace task config policies.
     * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public getWorkspaceConfigPolicies(workspaceId: number, workloadType: string, options?: any) {
        return AlphaApiFp(this.configuration).getWorkspaceConfigPolicies(workspaceId, workloadType, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Add or update global task config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {V1PutGlobalConfigPoliciesRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public putGlobalConfigPolicies(workloadType: string, body: V1PutGlobalConfigPoliciesRequest, options?: any) {
        return AlphaApiFp(this.configuration).putGlobalConfigPolicies(workloadType, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Add or update workspace task config policies.
     * @param {number} workspaceId The workspace the config policies apply to. Use global API for global config policies.
     * @param {string} workloadType The workload type the config policies apply to: EXPERIMENT or NTSC.
     * @param {V1PutWorkspaceConfigPoliciesRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AlphaApi
     */
    public putWorkspaceConfigPolicies(workspaceId: number, workloadType: string, body: V1PutWorkspaceConfigPoliciesRequest, options?: any) {
        return AlphaApiFp(this.configuration).putWorkspaceConfigPolicies(workspaceId, workloadType, body, options)(this.fetch, this.basePath)
    }
    
}

/**
 * AuthenticationApi - fetch parameter creator
 * @export
 */
export const AuthenticationApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Get the current user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        currentUser(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/auth/user`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Login the user.
         * @param {V1LoginRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        login(body: V1LoginRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling login.');
            }
            const localVarPath = `/api/v1/auth/login`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Logout the user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        logout(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/auth/logout`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * AuthenticationApi - functional programming interface
 * @export
 */
export const AuthenticationApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Get the current user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        currentUser(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CurrentUserResponse> {
            const localVarFetchArgs = AuthenticationApiFetchParamCreator(configuration).currentUser(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Login the user.
         * @param {V1LoginRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        login(body: V1LoginRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1LoginResponse> {
            const localVarFetchArgs = AuthenticationApiFetchParamCreator(configuration).login(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Logout the user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        logout(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1LogoutResponse> {
            const localVarFetchArgs = AuthenticationApiFetchParamCreator(configuration).logout(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * AuthenticationApi - factory interface
 * @export
 */
export const AuthenticationApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Get the current user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        currentUser(options?: any) {
            return AuthenticationApiFp(configuration).currentUser(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Login the user.
         * @param {V1LoginRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        login(body: V1LoginRequest, options?: any) {
            return AuthenticationApiFp(configuration).login(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Logout the user.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        logout(options?: any) {
            return AuthenticationApiFp(configuration).logout(options)(fetch, basePath);
        },
    }
};

/**
 * AuthenticationApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class AuthenticationApi extends BaseAPI {
    /**
     * 
     * @summary Get the current user.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AuthenticationApi
     */
    public currentUser(options?: any) {
        return AuthenticationApiFp(this.configuration).currentUser(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Login the user.
     * @param {V1LoginRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AuthenticationApi
     */
    public login(body: V1LoginRequest, options?: any) {
        return AuthenticationApiFp(this.configuration).login(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Logout the user.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof AuthenticationApi
     */
    public logout(options?: any) {
        return AuthenticationApiFp(this.configuration).logout(options)(this.fetch, this.basePath)
    }
    
}

/**
 * CheckpointsApi - fetch parameter creator
 * @export
 */
export const CheckpointsApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Remove files from checkpoints.
         * @param {V1CheckpointsRemoveFilesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        checkpointsRemoveFiles(body: V1CheckpointsRemoveFilesRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling checkpointsRemoveFiles.');
            }
            const localVarPath = `/api/v1/checkpoints/rm`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete Checkpoints.
         * @param {V1DeleteCheckpointsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteCheckpoints(body: V1DeleteCheckpointsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling deleteCheckpoints.');
            }
            const localVarPath = `/api/v1/checkpoints`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the requested checkpoint.
         * @param {string} checkpointUuid The uuid for the requested checkpoint.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCheckpoint(checkpointUuid: string, options: any = {}): FetchArgs {
            // verify required parameter 'checkpointUuid' is not null or undefined
            if (checkpointUuid === null || checkpointUuid === undefined) {
                throw new RequiredError('checkpointUuid','Required parameter checkpointUuid was null or undefined when calling getCheckpoint.');
            }
            const localVarPath = `/api/v1/checkpoints/{checkpointUuid}`
                .replace(`{${"checkpointUuid"}}`, encodeURIComponent(String(checkpointUuid)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Update checkpoint metadata.
         * @param {string} checkpointUuid UUID of the checkpoint.
         * @param {V1PostCheckpointMetadataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postCheckpointMetadata(checkpointUuid: string, body: V1PostCheckpointMetadataRequest, options: any = {}): FetchArgs {
            // verify required parameter 'checkpointUuid' is not null or undefined
            if (checkpointUuid === null || checkpointUuid === undefined) {
                throw new RequiredError('checkpointUuid','Required parameter checkpointUuid was null or undefined when calling postCheckpointMetadata.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postCheckpointMetadata.');
            }
            const localVarPath = `/api/v1/checkpoints/{checkpointUuid}/metadata`
                .replace(`{${"checkpointUuid"}}`, encodeURIComponent(String(checkpointUuid)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * CheckpointsApi - functional programming interface
 * @export
 */
export const CheckpointsApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Remove files from checkpoints.
         * @param {V1CheckpointsRemoveFilesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        checkpointsRemoveFiles(body: V1CheckpointsRemoveFilesRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CheckpointsRemoveFilesResponse> {
            const localVarFetchArgs = CheckpointsApiFetchParamCreator(configuration).checkpointsRemoveFiles(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete Checkpoints.
         * @param {V1DeleteCheckpointsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteCheckpoints(body: V1DeleteCheckpointsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteCheckpointsResponse> {
            const localVarFetchArgs = CheckpointsApiFetchParamCreator(configuration).deleteCheckpoints(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the requested checkpoint.
         * @param {string} checkpointUuid The uuid for the requested checkpoint.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCheckpoint(checkpointUuid: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetCheckpointResponse> {
            const localVarFetchArgs = CheckpointsApiFetchParamCreator(configuration).getCheckpoint(checkpointUuid, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Update checkpoint metadata.
         * @param {string} checkpointUuid UUID of the checkpoint.
         * @param {V1PostCheckpointMetadataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postCheckpointMetadata(checkpointUuid: string, body: V1PostCheckpointMetadataRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostCheckpointMetadataResponse> {
            const localVarFetchArgs = CheckpointsApiFetchParamCreator(configuration).postCheckpointMetadata(checkpointUuid, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * CheckpointsApi - factory interface
 * @export
 */
export const CheckpointsApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Remove files from checkpoints.
         * @param {V1CheckpointsRemoveFilesRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        checkpointsRemoveFiles(body: V1CheckpointsRemoveFilesRequest, options?: any) {
            return CheckpointsApiFp(configuration).checkpointsRemoveFiles(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete Checkpoints.
         * @param {V1DeleteCheckpointsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteCheckpoints(body: V1DeleteCheckpointsRequest, options?: any) {
            return CheckpointsApiFp(configuration).deleteCheckpoints(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the requested checkpoint.
         * @param {string} checkpointUuid The uuid for the requested checkpoint.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCheckpoint(checkpointUuid: string, options?: any) {
            return CheckpointsApiFp(configuration).getCheckpoint(checkpointUuid, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Update checkpoint metadata.
         * @param {string} checkpointUuid UUID of the checkpoint.
         * @param {V1PostCheckpointMetadataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postCheckpointMetadata(checkpointUuid: string, body: V1PostCheckpointMetadataRequest, options?: any) {
            return CheckpointsApiFp(configuration).postCheckpointMetadata(checkpointUuid, body, options)(fetch, basePath);
        },
    }
};

/**
 * CheckpointsApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class CheckpointsApi extends BaseAPI {
    /**
     * 
     * @summary Remove files from checkpoints.
     * @param {V1CheckpointsRemoveFilesRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CheckpointsApi
     */
    public checkpointsRemoveFiles(body: V1CheckpointsRemoveFilesRequest, options?: any) {
        return CheckpointsApiFp(this.configuration).checkpointsRemoveFiles(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete Checkpoints.
     * @param {V1DeleteCheckpointsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CheckpointsApi
     */
    public deleteCheckpoints(body: V1DeleteCheckpointsRequest, options?: any) {
        return CheckpointsApiFp(this.configuration).deleteCheckpoints(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the requested checkpoint.
     * @param {string} checkpointUuid The uuid for the requested checkpoint.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CheckpointsApi
     */
    public getCheckpoint(checkpointUuid: string, options?: any) {
        return CheckpointsApiFp(this.configuration).getCheckpoint(checkpointUuid, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Update checkpoint metadata.
     * @param {string} checkpointUuid UUID of the checkpoint.
     * @param {V1PostCheckpointMetadataRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CheckpointsApi
     */
    public postCheckpointMetadata(checkpointUuid: string, body: V1PostCheckpointMetadataRequest, options?: any) {
        return CheckpointsApiFp(this.configuration).postCheckpointMetadata(checkpointUuid, body, options)(this.fetch, this.basePath)
    }
    
}

/**
 * ClusterApi - fetch parameter creator
 * @export
 */
export const ClusterApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Clear the cluster-wide message shown to all users.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteClusterMessage(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master/cluster_message`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Disable the agent.
         * @param {string} agentId The id of the agent.
         * @param {V1DisableAgentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableAgent(agentId: string, body: V1DisableAgentRequest, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling disableAgent.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling disableAgent.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/disable`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Disable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {V1DisableSlotRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableSlot(agentId: string, slotId: string, body: V1DisableSlotRequest, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling disableSlot.');
            }
            // verify required parameter 'slotId' is not null or undefined
            if (slotId === null || slotId === undefined) {
                throw new RequiredError('slotId','Required parameter slotId was null or undefined when calling disableSlot.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling disableSlot.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/slots/{slotId}/disable`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)))
                .replace(`{${"slotId"}}`, encodeURIComponent(String(slotId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Enable the agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableAgent(agentId: string, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling enableAgent.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/enable`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Enable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableSlot(agentId: string, slotId: string, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling enableSlot.');
            }
            // verify required parameter 'slotId' is not null or undefined
            if (slotId === null || slotId === undefined) {
                throw new RequiredError('slotId','Required parameter slotId was null or undefined when calling enableSlot.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/slots/{slotId}/enable`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)))
                .replace(`{${"slotId"}}`, encodeURIComponent(String(slotId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the requested agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgent(agentId: string, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling getAgent.');
            }
            const localVarPath = `/api/v1/agents/{agentId}`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a set of agents from the cluster.
         * @param {V1GetAgentsRequestSortBy} [sortBy] Sort agents by the given field.   - SORT_BY_UNSPECIFIED: Returns agents in an unsorted list.  - SORT_BY_ID: Returns agents sorted by id.  - SORT_BY_TIME: Returns agents sorted by time.
         * @param {V1OrderBy} [orderBy] Order agents in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of agents before returning results. Negative values denote number of agents to skip from the end before returning results.
         * @param {number} [limit] Limit the number of agents. A value of 0 denotes no limit.
         * @param {boolean} [excludeSlots] exclude slots.
         * @param {boolean} [excludeContainers] exclude containers.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgents(sortBy?: V1GetAgentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, excludeSlots?: boolean, excludeContainers?: boolean, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/agents`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortBy !== undefined) {
                localVarQueryParameter['sortBy'] = sortBy
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (excludeSlots !== undefined) {
                localVarQueryParameter['excludeSlots'] = excludeSlots
            }
            
            if (excludeContainers !== undefined) {
                localVarQueryParameter['excludeContainers'] = excludeContainers
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the currently configured cluster-wide message.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getClusterMessage(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master/cluster_message`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get master information.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMaster(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get master config.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMasterConfig(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master/config`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the requested slot for an agent.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlot(agentId: string, slotId: string, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling getSlot.');
            }
            // verify required parameter 'slotId' is not null or undefined
            if (slotId === null || slotId === undefined) {
                throw new RequiredError('slotId','Required parameter slotId was null or undefined when calling getSlot.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/slots/{slotId}`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)))
                .replace(`{${"slotId"}}`, encodeURIComponent(String(slotId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get all the slots for an agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlots(agentId: string, options: any = {}): FetchArgs {
            // verify required parameter 'agentId' is not null or undefined
            if (agentId === null || agentId === undefined) {
                throw new RequiredError('agentId','Required parameter agentId was null or undefined when calling getSlots.');
            }
            const localVarPath = `/api/v1/agents/{agentId}/slots`
                .replace(`{${"agentId"}}`, encodeURIComponent(String(agentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get health of Determined and the dependencies.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        health(options: any = {}): FetchArgs {
            const localVarPath = `/health`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Stream master logs.
         * @param {number} [offset] Skip the number of master logs before returning results. Negative values denote number of master logs to skip from the end before returning results.
         * @param {number} [limit] Limit the number of master logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the master stops or the limit is reached.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        masterLogs(offset?: number, limit?: number, follow?: boolean, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master/logs`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (follow !== undefined) {
                localVarQueryParameter['follow'] = follow
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Patch master config.
         * @param {V1PatchMasterConfigRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchMasterConfig(body: V1PatchMasterConfigRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling patchMasterConfig.');
            }
            const localVarPath = `/api/v1/master/config`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get an aggregated view of resource allocation during the given time period.
         * @param {string} startDate The first day to consider (the exact time is midnight UTC at the beginning of the day).
         * @param {string} endDate The last day to consider (the exact time is midnight UTC at the end of the day).
         * @param {V1ResourceAllocationAggregationPeriod} period The period over which to perform aggregation.   - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY: Aggregation by day.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY: Aggregation by month.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationAggregated(startDate: string, endDate: string, period: V1ResourceAllocationAggregationPeriod, options: any = {}): FetchArgs {
            // verify required parameter 'startDate' is not null or undefined
            if (startDate === null || startDate === undefined) {
                throw new RequiredError('startDate','Required parameter startDate was null or undefined when calling resourceAllocationAggregated.');
            }
            // verify required parameter 'endDate' is not null or undefined
            if (endDate === null || endDate === undefined) {
                throw new RequiredError('endDate','Required parameter endDate was null or undefined when calling resourceAllocationAggregated.');
            }
            // verify required parameter 'period' is not null or undefined
            if (period === null || period === undefined) {
                throw new RequiredError('period','Required parameter period was null or undefined when calling resourceAllocationAggregated.');
            }
            const localVarPath = `/api/v1/resources/allocation/aggregated`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (startDate !== undefined) {
                localVarQueryParameter['startDate'] = startDate
            }
            
            if (endDate !== undefined) {
                localVarQueryParameter['endDate'] = endDate
            }
            
            if (period !== undefined) {
                localVarQueryParameter['period'] = period
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a detailed view of resource allocation during the given time period.
         * @param {Date | DateString} timestampAfter The start of the period to consider.
         * @param {Date | DateString} timestampBefore The end of the period to consider.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationRaw(timestampAfter: Date | DateString, timestampBefore: Date | DateString, options: any = {}): FetchArgs {
            // verify required parameter 'timestampAfter' is not null or undefined
            if (timestampAfter === null || timestampAfter === undefined) {
                throw new RequiredError('timestampAfter','Required parameter timestampAfter was null or undefined when calling resourceAllocationRaw.');
            }
            // verify required parameter 'timestampBefore' is not null or undefined
            if (timestampBefore === null || timestampBefore === undefined) {
                throw new RequiredError('timestampBefore','Required parameter timestampBefore was null or undefined when calling resourceAllocationRaw.');
            }
            const localVarPath = `/api/v1/resources/allocation/raw`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (timestampAfter) {
                localVarQueryParameter['timestampAfter'] = typeof timestampAfter === "string" ? timestampAfter : timestampAfter.toISOString()
            }
            
            if (timestampBefore) {
                localVarQueryParameter['timestampBefore'] = typeof timestampBefore === "string" ? timestampBefore : timestampBefore.toISOString()
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Set the cluster-wide message shown to users. Only one can be set at at time, so any existing message will be disabled.
         * @param {V1SetClusterMessageRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setClusterMessage(body: V1SetClusterMessageRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling setClusterMessage.');
            }
            const localVarPath = `/api/v1/master/cluster_message`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * ClusterApi - functional programming interface
 * @export
 */
export const ClusterApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Clear the cluster-wide message shown to all users.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteClusterMessage(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteClusterMessageResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).deleteClusterMessage(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Disable the agent.
         * @param {string} agentId The id of the agent.
         * @param {V1DisableAgentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableAgent(agentId: string, body: V1DisableAgentRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DisableAgentResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).disableAgent(agentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Disable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {V1DisableSlotRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableSlot(agentId: string, slotId: string, body: V1DisableSlotRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DisableSlotResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).disableSlot(agentId, slotId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Enable the agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableAgent(agentId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1EnableAgentResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).enableAgent(agentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Enable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableSlot(agentId: string, slotId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1EnableSlotResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).enableSlot(agentId, slotId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the requested agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgent(agentId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetAgentResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getAgent(agentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a set of agents from the cluster.
         * @param {V1GetAgentsRequestSortBy} [sortBy] Sort agents by the given field.   - SORT_BY_UNSPECIFIED: Returns agents in an unsorted list.  - SORT_BY_ID: Returns agents sorted by id.  - SORT_BY_TIME: Returns agents sorted by time.
         * @param {V1OrderBy} [orderBy] Order agents in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of agents before returning results. Negative values denote number of agents to skip from the end before returning results.
         * @param {number} [limit] Limit the number of agents. A value of 0 denotes no limit.
         * @param {boolean} [excludeSlots] exclude slots.
         * @param {boolean} [excludeContainers] exclude containers.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgents(sortBy?: V1GetAgentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, excludeSlots?: boolean, excludeContainers?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetAgentsResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getAgents(sortBy, orderBy, offset, limit, excludeSlots, excludeContainers, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the currently configured cluster-wide message.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getClusterMessage(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetClusterMessageResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getClusterMessage(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get master information.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMaster(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetMasterResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getMaster(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get master config.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMasterConfig(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetMasterConfigResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getMasterConfig(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the requested slot for an agent.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlot(agentId: string, slotId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetSlotResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getSlot(agentId, slotId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get all the slots for an agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlots(agentId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetSlotsResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).getSlots(agentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get health of Determined and the dependencies.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        health(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<HealthCheck> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).health(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Stream master logs.
         * @param {number} [offset] Skip the number of master logs before returning results. Negative values denote number of master logs to skip from the end before returning results.
         * @param {number} [limit] Limit the number of master logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the master stops or the limit is reached.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        masterLogs(offset?: number, limit?: number, follow?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<StreamResultOfV1MasterLogsResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).masterLogs(offset, limit, follow, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Patch master config.
         * @param {V1PatchMasterConfigRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchMasterConfig(body: V1PatchMasterConfigRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PatchMasterConfigResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).patchMasterConfig(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get an aggregated view of resource allocation during the given time period.
         * @param {string} startDate The first day to consider (the exact time is midnight UTC at the beginning of the day).
         * @param {string} endDate The last day to consider (the exact time is midnight UTC at the end of the day).
         * @param {V1ResourceAllocationAggregationPeriod} period The period over which to perform aggregation.   - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY: Aggregation by day.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY: Aggregation by month.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationAggregated(startDate: string, endDate: string, period: V1ResourceAllocationAggregationPeriod, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ResourceAllocationAggregatedResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).resourceAllocationAggregated(startDate, endDate, period, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a detailed view of resource allocation during the given time period.
         * @param {Date | DateString} timestampAfter The start of the period to consider.
         * @param {Date | DateString} timestampBefore The end of the period to consider.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationRaw(timestampAfter: Date | DateString, timestampBefore: Date | DateString, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ResourceAllocationRawResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).resourceAllocationRaw(timestampAfter, timestampBefore, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Set the cluster-wide message shown to users. Only one can be set at at time, so any existing message will be disabled.
         * @param {V1SetClusterMessageRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setClusterMessage(body: V1SetClusterMessageRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1SetClusterMessageResponse> {
            const localVarFetchArgs = ClusterApiFetchParamCreator(configuration).setClusterMessage(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * ClusterApi - factory interface
 * @export
 */
export const ClusterApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Clear the cluster-wide message shown to all users.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteClusterMessage(options?: any) {
            return ClusterApiFp(configuration).deleteClusterMessage(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Disable the agent.
         * @param {string} agentId The id of the agent.
         * @param {V1DisableAgentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableAgent(agentId: string, body: V1DisableAgentRequest, options?: any) {
            return ClusterApiFp(configuration).disableAgent(agentId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Disable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {V1DisableSlotRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        disableSlot(agentId: string, slotId: string, body: V1DisableSlotRequest, options?: any) {
            return ClusterApiFp(configuration).disableSlot(agentId, slotId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Enable the agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableAgent(agentId: string, options?: any) {
            return ClusterApiFp(configuration).enableAgent(agentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Enable the slot.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        enableSlot(agentId: string, slotId: string, options?: any) {
            return ClusterApiFp(configuration).enableSlot(agentId, slotId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the requested agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgent(agentId: string, options?: any) {
            return ClusterApiFp(configuration).getAgent(agentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a set of agents from the cluster.
         * @param {V1GetAgentsRequestSortBy} [sortBy] Sort agents by the given field.   - SORT_BY_UNSPECIFIED: Returns agents in an unsorted list.  - SORT_BY_ID: Returns agents sorted by id.  - SORT_BY_TIME: Returns agents sorted by time.
         * @param {V1OrderBy} [orderBy] Order agents in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of agents before returning results. Negative values denote number of agents to skip from the end before returning results.
         * @param {number} [limit] Limit the number of agents. A value of 0 denotes no limit.
         * @param {boolean} [excludeSlots] exclude slots.
         * @param {boolean} [excludeContainers] exclude containers.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAgents(sortBy?: V1GetAgentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, excludeSlots?: boolean, excludeContainers?: boolean, options?: any) {
            return ClusterApiFp(configuration).getAgents(sortBy, orderBy, offset, limit, excludeSlots, excludeContainers, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the currently configured cluster-wide message.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getClusterMessage(options?: any) {
            return ClusterApiFp(configuration).getClusterMessage(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get master information.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMaster(options?: any) {
            return ClusterApiFp(configuration).getMaster(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get master config.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMasterConfig(options?: any) {
            return ClusterApiFp(configuration).getMasterConfig(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the requested slot for an agent.
         * @param {string} agentId The id of the agent.
         * @param {string} slotId The id of the slot.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlot(agentId: string, slotId: string, options?: any) {
            return ClusterApiFp(configuration).getSlot(agentId, slotId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get all the slots for an agent.
         * @param {string} agentId The id of the agent.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getSlots(agentId: string, options?: any) {
            return ClusterApiFp(configuration).getSlots(agentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get health of Determined and the dependencies.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        health(options?: any) {
            return ClusterApiFp(configuration).health(options)(fetch, basePath);
        },
        /**
         * 
         * @summary Stream master logs.
         * @param {number} [offset] Skip the number of master logs before returning results. Negative values denote number of master logs to skip from the end before returning results.
         * @param {number} [limit] Limit the number of master logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the master stops or the limit is reached.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        masterLogs(offset?: number, limit?: number, follow?: boolean, options?: any) {
            return ClusterApiFp(configuration).masterLogs(offset, limit, follow, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Patch master config.
         * @param {V1PatchMasterConfigRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchMasterConfig(body: V1PatchMasterConfigRequest, options?: any) {
            return ClusterApiFp(configuration).patchMasterConfig(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get an aggregated view of resource allocation during the given time period.
         * @param {string} startDate The first day to consider (the exact time is midnight UTC at the beginning of the day).
         * @param {string} endDate The last day to consider (the exact time is midnight UTC at the end of the day).
         * @param {V1ResourceAllocationAggregationPeriod} period The period over which to perform aggregation.   - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY: Aggregation by day.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY: Aggregation by month.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationAggregated(startDate: string, endDate: string, period: V1ResourceAllocationAggregationPeriod, options?: any) {
            return ClusterApiFp(configuration).resourceAllocationAggregated(startDate, endDate, period, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a detailed view of resource allocation during the given time period.
         * @param {Date | DateString} timestampAfter The start of the period to consider.
         * @param {Date | DateString} timestampBefore The end of the period to consider.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resourceAllocationRaw(timestampAfter: Date | DateString, timestampBefore: Date | DateString, options?: any) {
            return ClusterApiFp(configuration).resourceAllocationRaw(timestampAfter, timestampBefore, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Set the cluster-wide message shown to users. Only one can be set at at time, so any existing message will be disabled.
         * @param {V1SetClusterMessageRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setClusterMessage(body: V1SetClusterMessageRequest, options?: any) {
            return ClusterApiFp(configuration).setClusterMessage(body, options)(fetch, basePath);
        },
    }
};

/**
 * ClusterApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class ClusterApi extends BaseAPI {
    /**
     * 
     * @summary Clear the cluster-wide message shown to all users.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public deleteClusterMessage(options?: any) {
        return ClusterApiFp(this.configuration).deleteClusterMessage(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Disable the agent.
     * @param {string} agentId The id of the agent.
     * @param {V1DisableAgentRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public disableAgent(agentId: string, body: V1DisableAgentRequest, options?: any) {
        return ClusterApiFp(this.configuration).disableAgent(agentId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Disable the slot.
     * @param {string} agentId The id of the agent.
     * @param {string} slotId The id of the slot.
     * @param {V1DisableSlotRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public disableSlot(agentId: string, slotId: string, body: V1DisableSlotRequest, options?: any) {
        return ClusterApiFp(this.configuration).disableSlot(agentId, slotId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Enable the agent.
     * @param {string} agentId The id of the agent.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public enableAgent(agentId: string, options?: any) {
        return ClusterApiFp(this.configuration).enableAgent(agentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Enable the slot.
     * @param {string} agentId The id of the agent.
     * @param {string} slotId The id of the slot.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public enableSlot(agentId: string, slotId: string, options?: any) {
        return ClusterApiFp(this.configuration).enableSlot(agentId, slotId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the requested agent.
     * @param {string} agentId The id of the agent.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getAgent(agentId: string, options?: any) {
        return ClusterApiFp(this.configuration).getAgent(agentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a set of agents from the cluster.
     * @param {V1GetAgentsRequestSortBy} [sortBy] Sort agents by the given field.   - SORT_BY_UNSPECIFIED: Returns agents in an unsorted list.  - SORT_BY_ID: Returns agents sorted by id.  - SORT_BY_TIME: Returns agents sorted by time.
     * @param {V1OrderBy} [orderBy] Order agents in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of agents before returning results. Negative values denote number of agents to skip from the end before returning results.
     * @param {number} [limit] Limit the number of agents. A value of 0 denotes no limit.
     * @param {boolean} [excludeSlots] exclude slots.
     * @param {boolean} [excludeContainers] exclude containers.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getAgents(sortBy?: V1GetAgentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, excludeSlots?: boolean, excludeContainers?: boolean, options?: any) {
        return ClusterApiFp(this.configuration).getAgents(sortBy, orderBy, offset, limit, excludeSlots, excludeContainers, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the currently configured cluster-wide message.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getClusterMessage(options?: any) {
        return ClusterApiFp(this.configuration).getClusterMessage(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get master information.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getMaster(options?: any) {
        return ClusterApiFp(this.configuration).getMaster(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get master config.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getMasterConfig(options?: any) {
        return ClusterApiFp(this.configuration).getMasterConfig(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the requested slot for an agent.
     * @param {string} agentId The id of the agent.
     * @param {string} slotId The id of the slot.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getSlot(agentId: string, slotId: string, options?: any) {
        return ClusterApiFp(this.configuration).getSlot(agentId, slotId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get all the slots for an agent.
     * @param {string} agentId The id of the agent.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public getSlots(agentId: string, options?: any) {
        return ClusterApiFp(this.configuration).getSlots(agentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get health of Determined and the dependencies.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public health(options?: any) {
        return ClusterApiFp(this.configuration).health(options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Stream master logs.
     * @param {number} [offset] Skip the number of master logs before returning results. Negative values denote number of master logs to skip from the end before returning results.
     * @param {number} [limit] Limit the number of master logs. A value of 0 denotes no limit.
     * @param {boolean} [follow] Continue following logs until the master stops or the limit is reached.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public masterLogs(offset?: number, limit?: number, follow?: boolean, options?: any) {
        return ClusterApiFp(this.configuration).masterLogs(offset, limit, follow, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Patch master config.
     * @param {V1PatchMasterConfigRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public patchMasterConfig(body: V1PatchMasterConfigRequest, options?: any) {
        return ClusterApiFp(this.configuration).patchMasterConfig(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get an aggregated view of resource allocation during the given time period.
     * @param {string} startDate The first day to consider (the exact time is midnight UTC at the beginning of the day).
     * @param {string} endDate The last day to consider (the exact time is midnight UTC at the end of the day).
     * @param {V1ResourceAllocationAggregationPeriod} period The period over which to perform aggregation.   - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_UNSPECIFIED: Unspecified. This value will never actually be returned by the API, it is just an artifact of using protobuf.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_DAILY: Aggregation by day.  - RESOURCE_ALLOCATION_AGGREGATION_PERIOD_MONTHLY: Aggregation by month.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public resourceAllocationAggregated(startDate: string, endDate: string, period: V1ResourceAllocationAggregationPeriod, options?: any) {
        return ClusterApiFp(this.configuration).resourceAllocationAggregated(startDate, endDate, period, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a detailed view of resource allocation during the given time period.
     * @param {Date | DateString} timestampAfter The start of the period to consider.
     * @param {Date | DateString} timestampBefore The end of the period to consider.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public resourceAllocationRaw(timestampAfter: Date | DateString, timestampBefore: Date | DateString, options?: any) {
        return ClusterApiFp(this.configuration).resourceAllocationRaw(timestampAfter, timestampBefore, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Set the cluster-wide message shown to users. Only one can be set at at time, so any existing message will be disabled.
     * @param {V1SetClusterMessageRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ClusterApi
     */
    public setClusterMessage(body: V1SetClusterMessageRequest, options?: any) {
        return ClusterApiFp(this.configuration).setClusterMessage(body, options)(this.fetch, this.basePath)
    }
    
}

/**
 * CommandsApi - fetch parameter creator
 * @export
 */
export const CommandsApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Get the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommand(commandId: string, options: any = {}): FetchArgs {
            // verify required parameter 'commandId' is not null or undefined
            if (commandId === null || commandId === undefined) {
                throw new RequiredError('commandId','Required parameter commandId was null or undefined when calling getCommand.');
            }
            const localVarPath = `/api/v1/commands/{commandId}`
                .replace(`{${"commandId"}}`, encodeURIComponent(String(commandId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of commands.
         * @param {V1GetTensorboardsRequestSortBy} [sortBy] Sort commands by the given field.   - SORT_BY_UNSPECIFIED: Returns commands in an unsorted list.  - SORT_BY_ID: Returns commands sorted by id.  - SORT_BY_DESCRIPTION: Returns commands sorted by description.  - SORT_BY_START_TIME: Return commands sorted by start time.  - SORT_BY_WORKSPACE_ID: Return commands sorted by workspace_id.
         * @param {V1OrderBy} [orderBy] Order commands in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of commands before returning results. Negative values denote number of commands to skip from the end before returning results.
         * @param {number} [limit] Limit the number of commands. A value of 0 denotes no limit.
         * @param {Array<string>} [users] Limit commands to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit commands to those that are owned by users with the specified userIds.
         * @param {number} [workspaceId] Limit commands to those within a specific workspace, or 0 for all accessible workspaces.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommands(sortBy?: V1GetTensorboardsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, users?: Array<string>, userIds?: Array<number>, workspaceId?: number, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/commands`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortBy !== undefined) {
                localVarQueryParameter['sortBy'] = sortBy
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (users) {
                localVarQueryParameter['users'] = users
            }
            
            if (userIds) {
                localVarQueryParameter['userIds'] = userIds
            }
            
            if (workspaceId !== undefined) {
                localVarQueryParameter['workspaceId'] = workspaceId
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killCommand(commandId: string, options: any = {}): FetchArgs {
            // verify required parameter 'commandId' is not null or undefined
            if (commandId === null || commandId === undefined) {
                throw new RequiredError('commandId','Required parameter commandId was null or undefined when calling killCommand.');
            }
            const localVarPath = `/api/v1/commands/{commandId}/kill`
                .replace(`{${"commandId"}}`, encodeURIComponent(String(commandId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Launch a command.
         * @param {V1LaunchCommandRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        launchCommand(body: V1LaunchCommandRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling launchCommand.');
            }
            const localVarPath = `/api/v1/commands`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Set the priority of the requested command.
         * @param {string} commandId The id of the command.
         * @param {V1SetCommandPriorityRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setCommandPriority(commandId: string, body: V1SetCommandPriorityRequest, options: any = {}): FetchArgs {
            // verify required parameter 'commandId' is not null or undefined
            if (commandId === null || commandId === undefined) {
                throw new RequiredError('commandId','Required parameter commandId was null or undefined when calling setCommandPriority.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling setCommandPriority.');
            }
            const localVarPath = `/api/v1/commands/{commandId}/set_priority`
                .replace(`{${"commandId"}}`, encodeURIComponent(String(commandId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * CommandsApi - functional programming interface
 * @export
 */
export const CommandsApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Get the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommand(commandId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetCommandResponse> {
            const localVarFetchArgs = CommandsApiFetchParamCreator(configuration).getCommand(commandId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of commands.
         * @param {V1GetTensorboardsRequestSortBy} [sortBy] Sort commands by the given field.   - SORT_BY_UNSPECIFIED: Returns commands in an unsorted list.  - SORT_BY_ID: Returns commands sorted by id.  - SORT_BY_DESCRIPTION: Returns commands sorted by description.  - SORT_BY_START_TIME: Return commands sorted by start time.  - SORT_BY_WORKSPACE_ID: Return commands sorted by workspace_id.
         * @param {V1OrderBy} [orderBy] Order commands in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of commands before returning results. Negative values denote number of commands to skip from the end before returning results.
         * @param {number} [limit] Limit the number of commands. A value of 0 denotes no limit.
         * @param {Array<string>} [users] Limit commands to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit commands to those that are owned by users with the specified userIds.
         * @param {number} [workspaceId] Limit commands to those within a specific workspace, or 0 for all accessible workspaces.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommands(sortBy?: V1GetTensorboardsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, users?: Array<string>, userIds?: Array<number>, workspaceId?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetCommandsResponse> {
            const localVarFetchArgs = CommandsApiFetchParamCreator(configuration).getCommands(sortBy, orderBy, offset, limit, users, userIds, workspaceId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killCommand(commandId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillCommandResponse> {
            const localVarFetchArgs = CommandsApiFetchParamCreator(configuration).killCommand(commandId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Launch a command.
         * @param {V1LaunchCommandRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        launchCommand(body: V1LaunchCommandRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1LaunchCommandResponse> {
            const localVarFetchArgs = CommandsApiFetchParamCreator(configuration).launchCommand(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Set the priority of the requested command.
         * @param {string} commandId The id of the command.
         * @param {V1SetCommandPriorityRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setCommandPriority(commandId: string, body: V1SetCommandPriorityRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1SetCommandPriorityResponse> {
            const localVarFetchArgs = CommandsApiFetchParamCreator(configuration).setCommandPriority(commandId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * CommandsApi - factory interface
 * @export
 */
export const CommandsApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Get the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommand(commandId: string, options?: any) {
            return CommandsApiFp(configuration).getCommand(commandId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a list of commands.
         * @param {V1GetTensorboardsRequestSortBy} [sortBy] Sort commands by the given field.   - SORT_BY_UNSPECIFIED: Returns commands in an unsorted list.  - SORT_BY_ID: Returns commands sorted by id.  - SORT_BY_DESCRIPTION: Returns commands sorted by description.  - SORT_BY_START_TIME: Return commands sorted by start time.  - SORT_BY_WORKSPACE_ID: Return commands sorted by workspace_id.
         * @param {V1OrderBy} [orderBy] Order commands in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of commands before returning results. Negative values denote number of commands to skip from the end before returning results.
         * @param {number} [limit] Limit the number of commands. A value of 0 denotes no limit.
         * @param {Array<string>} [users] Limit commands to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit commands to those that are owned by users with the specified userIds.
         * @param {number} [workspaceId] Limit commands to those within a specific workspace, or 0 for all accessible workspaces.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getCommands(sortBy?: V1GetTensorboardsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, users?: Array<string>, userIds?: Array<number>, workspaceId?: number, options?: any) {
            return CommandsApiFp(configuration).getCommands(sortBy, orderBy, offset, limit, users, userIds, workspaceId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Kill the requested command.
         * @param {string} commandId The id of the command.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killCommand(commandId: string, options?: any) {
            return CommandsApiFp(configuration).killCommand(commandId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Launch a command.
         * @param {V1LaunchCommandRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        launchCommand(body: V1LaunchCommandRequest, options?: any) {
            return CommandsApiFp(configuration).launchCommand(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Set the priority of the requested command.
         * @param {string} commandId The id of the command.
         * @param {V1SetCommandPriorityRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        setCommandPriority(commandId: string, body: V1SetCommandPriorityRequest, options?: any) {
            return CommandsApiFp(configuration).setCommandPriority(commandId, body, options)(fetch, basePath);
        },
    }
};

/**
 * CommandsApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class CommandsApi extends BaseAPI {
    /**
     * 
     * @summary Get the requested command.
     * @param {string} commandId The id of the command.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CommandsApi
     */
    public getCommand(commandId: string, options?: any) {
        return CommandsApiFp(this.configuration).getCommand(commandId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a list of commands.
     * @param {V1GetTensorboardsRequestSortBy} [sortBy] Sort commands by the given field.   - SORT_BY_UNSPECIFIED: Returns commands in an unsorted list.  - SORT_BY_ID: Returns commands sorted by id.  - SORT_BY_DESCRIPTION: Returns commands sorted by description.  - SORT_BY_START_TIME: Return commands sorted by start time.  - SORT_BY_WORKSPACE_ID: Return commands sorted by workspace_id.
     * @param {V1OrderBy} [orderBy] Order commands in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of commands before returning results. Negative values denote number of commands to skip from the end before returning results.
     * @param {number} [limit] Limit the number of commands. A value of 0 denotes no limit.
     * @param {Array<string>} [users] Limit commands to those that are owned by users with the specified usernames.
     * @param {Array<number>} [userIds] Limit commands to those that are owned by users with the specified userIds.
     * @param {number} [workspaceId] Limit commands to those within a specific workspace, or 0 for all accessible workspaces.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CommandsApi
     */
    public getCommands(sortBy?: V1GetTensorboardsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, users?: Array<string>, userIds?: Array<number>, workspaceId?: number, options?: any) {
        return CommandsApiFp(this.configuration).getCommands(sortBy, orderBy, offset, limit, users, userIds, workspaceId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Kill the requested command.
     * @param {string} commandId The id of the command.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CommandsApi
     */
    public killCommand(commandId: string, options?: any) {
        return CommandsApiFp(this.configuration).killCommand(commandId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Launch a command.
     * @param {V1LaunchCommandRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CommandsApi
     */
    public launchCommand(body: V1LaunchCommandRequest, options?: any) {
        return CommandsApiFp(this.configuration).launchCommand(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Set the priority of the requested command.
     * @param {string} commandId The id of the command.
     * @param {V1SetCommandPriorityRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof CommandsApi
     */
    public setCommandPriority(commandId: string, body: V1SetCommandPriorityRequest, options?: any) {
        return CommandsApiFp(this.configuration).setCommandPriority(commandId, body, options)(this.fetch, this.basePath)
    }
    
}

/**
 * ExperimentsApi - fetch parameter creator
 * @export
 */
export const ExperimentsApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Activate an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling activateExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/activate`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Activate multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ActivateExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiments(projectId: number, body: V1ActivateExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling activateExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling activateExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/activate`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Archive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling archiveExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/archive`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Archive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ArchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiments(projectId: number, body: V1ArchiveExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling archiveExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling archiveExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/archive`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Cancel an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling cancelExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/cancel`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Cancel multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1CancelExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiments(projectId: number, body: V1CancelExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling cancelExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling cancelExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/cancel`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Return a downsampled time series of metrics from multiple trials to compare them side-by-side.
         * @param {Array<number>} [trialIds] The requested trial ids.
         * @param {number} [maxDatapoints] The maximum number of data points to return after downsampling.
         * @param {Array<string>} [metricNames] The names of selected metrics.
         * @param {number} [startBatches] Sample from metrics after this batch number.
         * @param {number} [endBatches] Sample from metrics before this batch number.
         * @param {V1MetricType} [metricType] Metric group.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {Array<string>} [metricIds] metric ids for the query. must be in the form group.metric_name.
         * @param {string} [timeSeriesFilterName] metric or column name for the filter.
         * @param {number} [timeSeriesFilterDoubleRangeLt] Less than.
         * @param {number} [timeSeriesFilterDoubleRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterDoubleRangeGt] Greater than.
         * @param {number} [timeSeriesFilterDoubleRangeGte] Greater than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeLt] Less than.
         * @param {number} [timeSeriesFilterIntegerRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeGt] Greater than.
         * @param {number} [timeSeriesFilterIntegerRangeGte] Greater than or equal.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeNotIn] Not in a set.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLt] Less than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLte] Less than or equal.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGt] Greater than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGte] Greater than or equal.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        compareTrials(trialIds?: Array<number>, maxDatapoints?: number, metricNames?: Array<string>, startBatches?: number, endBatches?: number, metricType?: V1MetricType, group?: string, metricIds?: Array<string>, timeSeriesFilterName?: string, timeSeriesFilterDoubleRangeLt?: number, timeSeriesFilterDoubleRangeLte?: number, timeSeriesFilterDoubleRangeGt?: number, timeSeriesFilterDoubleRangeGte?: number, timeSeriesFilterIntegerRangeLt?: number, timeSeriesFilterIntegerRangeLte?: number, timeSeriesFilterIntegerRangeGt?: number, timeSeriesFilterIntegerRangeGte?: number, timeSeriesFilterIntegerRangeIncl?: Array<number>, timeSeriesFilterIntegerRangeNotIn?: Array<number>, timeSeriesFilterTimeRangeLt?: Date | DateString, timeSeriesFilterTimeRangeLte?: Date | DateString, timeSeriesFilterTimeRangeGt?: Date | DateString, timeSeriesFilterTimeRangeGte?: Date | DateString, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/trials/time-series`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (trialIds) {
                localVarQueryParameter['trialIds'] = trialIds
            }
            
            if (maxDatapoints !== undefined) {
                localVarQueryParameter['maxDatapoints'] = maxDatapoints
            }
            
            if (metricNames) {
                localVarQueryParameter['metricNames'] = metricNames
            }
            
            if (startBatches !== undefined) {
                localVarQueryParameter['startBatches'] = startBatches
            }
            
            if (endBatches !== undefined) {
                localVarQueryParameter['endBatches'] = endBatches
            }
            
            if (metricType !== undefined) {
                localVarQueryParameter['metricType'] = metricType
            }
            
            if (group !== undefined) {
                localVarQueryParameter['group'] = group
            }
            
            if (metricIds) {
                localVarQueryParameter['metricIds'] = metricIds
            }
            
            if (timeSeriesFilterName !== undefined) {
                localVarQueryParameter['timeSeriesFilter.name'] = timeSeriesFilterName
            }
            
            if (timeSeriesFilterDoubleRangeLt !== undefined) {
                localVarQueryParameter['timeSeriesFilter.doubleRange.lt'] = timeSeriesFilterDoubleRangeLt
            }
            
            if (timeSeriesFilterDoubleRangeLte !== undefined) {
                localVarQueryParameter['timeSeriesFilter.doubleRange.lte'] = timeSeriesFilterDoubleRangeLte
            }
            
            if (timeSeriesFilterDoubleRangeGt !== undefined) {
                localVarQueryParameter['timeSeriesFilter.doubleRange.gt'] = timeSeriesFilterDoubleRangeGt
            }
            
            if (timeSeriesFilterDoubleRangeGte !== undefined) {
                localVarQueryParameter['timeSeriesFilter.doubleRange.gte'] = timeSeriesFilterDoubleRangeGte
            }
            
            if (timeSeriesFilterIntegerRangeLt !== undefined) {
                localVarQueryParameter['timeSeriesFilter.integerRange.lt'] = timeSeriesFilterIntegerRangeLt
            }
            
            if (timeSeriesFilterIntegerRangeLte !== undefined) {
                localVarQueryParameter['timeSeriesFilter.integerRange.lte'] = timeSeriesFilterIntegerRangeLte
            }
            
            if (timeSeriesFilterIntegerRangeGt !== undefined) {
                localVarQueryParameter['timeSeriesFilter.integerRange.gt'] = timeSeriesFilterIntegerRangeGt
            }
            
            if (timeSeriesFilterIntegerRangeGte !== undefined) {
                localVarQueryParameter['timeSeriesFilter.integerRange.gte'] = timeSeriesFilterIntegerRangeGte
            }
            
            if (timeSeriesFilterIntegerRangeIncl) {
                localVarQueryParameter['timeSeriesFilter.integerRange.incl'] = timeSeriesFilterIntegerRangeIncl
            }
            
            if (timeSeriesFilterIntegerRangeNotIn) {
                localVarQueryParameter['timeSeriesFilter.integerRange.notIn'] = timeSeriesFilterIntegerRangeNotIn
            }
            
            if (timeSeriesFilterTimeRangeLt) {
                localVarQueryParameter['timeSeriesFilter.timeRange.lt'] = typeof timeSeriesFilterTimeRangeLt === "string" ? timeSeriesFilterTimeRangeLt : timeSeriesFilterTimeRangeLt.toISOString()
            }
            
            if (timeSeriesFilterTimeRangeLte) {
                localVarQueryParameter['timeSeriesFilter.timeRange.lte'] = typeof timeSeriesFilterTimeRangeLte === "string" ? timeSeriesFilterTimeRangeLte : timeSeriesFilterTimeRangeLte.toISOString()
            }
            
            if (timeSeriesFilterTimeRangeGt) {
                localVarQueryParameter['timeSeriesFilter.timeRange.gt'] = typeof timeSeriesFilterTimeRangeGt === "string" ? timeSeriesFilterTimeRangeGt : timeSeriesFilterTimeRangeGt.toISOString()
            }
            
            if (timeSeriesFilterTimeRangeGte) {
                localVarQueryParameter['timeSeriesFilter.timeRange.gte'] = typeof timeSeriesFilterTimeRangeGte === "string" ? timeSeriesFilterTimeRangeGte : timeSeriesFilterTimeRangeGte.toISOString()
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete the requested experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiment(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling deleteExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete a label from the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to delete.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperimentLabel(experimentId: number, label: string, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling deleteExperimentLabel.');
            }
            // verify required parameter 'label' is not null or undefined
            if (label === null || label === undefined) {
                throw new RequiredError('label','Required parameter label was null or undefined when calling deleteExperimentLabel.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/labels/{label}`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)))
                .replace(`{${"label"}}`, encodeURIComponent(String(label)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1DeleteExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiments(projectId: number, body: V1DeleteExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling deleteExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling deleteExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/delete`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete tensorboard files.
         * @param {number} experimentId ID of experiment that the tensorboard files are linked to.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteTensorboardFiles(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling deleteTensorboardFiles.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/tensorboard-files`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the requested experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiment(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of checkpoints for an experiment.
         * @param {number} id The experiment id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling getExperimentCheckpoints.');
            }
            const localVarPath = `/api/v1/experiments/{id}/checkpoints`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortByAttr !== undefined) {
                localVarQueryParameter['sortByAttr'] = sortByAttr
            }
            
            if (sortByMetric !== undefined) {
                localVarQueryParameter['sortByMetric'] = sortByMetric
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of unique experiment labels (sorted by popularity).
         * @param {number} [projectId] Filter experiments by project.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentLabels(projectId?: number, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/experiment/labels`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (projectId !== undefined) {
                localVarQueryParameter['projectId'] = projectId
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of experiments.
         * @param {V1GetExperimentsRequestSortBy} [sortBy] Sort experiments by the given field.   - SORT_BY_UNSPECIFIED: Returns experiments in an unsorted list.  - SORT_BY_ID: Returns experiments sorted by id.  - SORT_BY_DESCRIPTION: Returns experiments sorted by description.  - SORT_BY_START_TIME: Return experiments sorted by start time.  - SORT_BY_END_TIME: Return experiments sorted by end time. Experiments without end_time are returned after the ones with end_time.  - SORT_BY_STATE: Return experiments sorted by state.  - SORT_BY_NUM_TRIALS: Return experiments sorted by number of trials.  - SORT_BY_PROGRESS: Return experiments sorted by progress.  - SORT_BY_USER: Return experiments sorted by user.  - SORT_BY_NAME: Returns experiments sorted by name.  - SORT_BY_FORKED_FROM: Returns experiments sorted by originating model.  - SORT_BY_RESOURCE_POOL: Returns experiments sorted by resource pool.  - SORT_BY_PROJECT_ID: Returns experiments sorted by project.  - SORT_BY_CHECKPOINT_SIZE: Returns experiments sorted by checkpoint size.  - SORT_BY_CHECKPOINT_COUNT: Returns experiments sorted by checkpoint count.  - SORT_BY_SEARCHER_METRIC_VAL: Returns experiments sorted by searcher metric value..
         * @param {V1OrderBy} [orderBy] Order experiments in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of experiments before returning results. Negative values denote number of experiments to skip from the end before returning results.
         * @param {number} [limit] Limit the number of experiments. 0 or Unspecified - returns a default of 100. -1               - returns everything. -2               - returns pagination info but no experiments.
         * @param {string} [description] Limit experiments to those that match the description.
         * @param {string} [name] Limit experiments to those that match the name.
         * @param {Array<string>} [labels] Limit experiments to those that match the provided labels.
         * @param {boolean} [archived] Limit experiments to those that are archived.
         * @param {Array<Experimentv1State>} [states] Limit experiments to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {Array<string>} [users] Limit experiments to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit experiments to those that are owned by users with the specified userIds.
         * @param {number} [projectId] Limit experiments to those within a specified project, or 0 for all projects.
         * @param {number} [experimentIdFilterLt] Less than.
         * @param {number} [experimentIdFilterLte] Less than or equal.
         * @param {number} [experimentIdFilterGt] Greater than.
         * @param {number} [experimentIdFilterGte] Greater than or equal.
         * @param {Array<number>} [experimentIdFilterIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [experimentIdFilterNotIn] Not in a set.
         * @param {boolean} [showTrialData] whether to surface trial specific data from the best trial.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiments(sortBy?: V1GetExperimentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, description?: string, name?: string, labels?: Array<string>, archived?: boolean, states?: Array<Experimentv1State>, users?: Array<string>, userIds?: Array<number>, projectId?: number, experimentIdFilterLt?: number, experimentIdFilterLte?: number, experimentIdFilterGt?: number, experimentIdFilterGte?: number, experimentIdFilterIncl?: Array<number>, experimentIdFilterNotIn?: Array<number>, showTrialData?: boolean, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/experiments`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortBy !== undefined) {
                localVarQueryParameter['sortBy'] = sortBy
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (description !== undefined) {
                localVarQueryParameter['description'] = description
            }
            
            if (name !== undefined) {
                localVarQueryParameter['name'] = name
            }
            
            if (labels) {
                localVarQueryParameter['labels'] = labels
            }
            
            if (archived !== undefined) {
                localVarQueryParameter['archived'] = archived
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            if (users) {
                localVarQueryParameter['users'] = users
            }
            
            if (userIds) {
                localVarQueryParameter['userIds'] = userIds
            }
            
            if (projectId !== undefined) {
                localVarQueryParameter['projectId'] = projectId
            }
            
            if (experimentIdFilterLt !== undefined) {
                localVarQueryParameter['experimentIdFilter.lt'] = experimentIdFilterLt
            }
            
            if (experimentIdFilterLte !== undefined) {
                localVarQueryParameter['experimentIdFilter.lte'] = experimentIdFilterLte
            }
            
            if (experimentIdFilterGt !== undefined) {
                localVarQueryParameter['experimentIdFilter.gt'] = experimentIdFilterGt
            }
            
            if (experimentIdFilterGte !== undefined) {
                localVarQueryParameter['experimentIdFilter.gte'] = experimentIdFilterGte
            }
            
            if (experimentIdFilterIncl) {
                localVarQueryParameter['experimentIdFilter.incl'] = experimentIdFilterIncl
            }
            
            if (experimentIdFilterNotIn) {
                localVarQueryParameter['experimentIdFilter.notIn'] = experimentIdFilterNotIn
            }
            
            if (showTrialData !== undefined) {
                localVarQueryParameter['showTrialData'] = showTrialData
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the list of trials for an experiment.
         * @param {number} experimentId Limit trials to those that are owned by the specified experiments.
         * @param {V1GetExperimentTrialsRequestSortBy} [sortBy] Sort trials by the given field.   - SORT_BY_UNSPECIFIED: Returns trials in an unsorted list.  - SORT_BY_ID: Returns trials sorted by id.  - SORT_BY_START_TIME: Return trials sorted by start time.  - SORT_BY_END_TIME: Return trials sorted by end time. Trials without end times are returned after trials that are.  - SORT_BY_STATE: Return trials sorted by state.  - SORT_BY_BEST_VALIDATION_METRIC: Return the trials sorted by the best metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_LATEST_VALIDATION_METRIC: Return the trials sorted by the latest metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_BATCHES_PROCESSED: Return the trials sorted by the number of batches completed.  - SORT_BY_DURATION: Return the trials sorted by the total duration.  - SORT_BY_RESTARTS: Return the trials sorted by the number of restarts.  - SORT_BY_CHECKPOINT_SIZE: Return the trials sorted by checkpoint size.  - SORT_BY_LOG_RETENTION_DAYS: Return the trials sorted by number of log retention days.
         * @param {V1OrderBy} [orderBy] Order trials in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of trials before returning results. Negative values denote number of trials to skip from the end before returning results.
         * @param {number} [limit] Limit the number of trials. A value of 0 denotes no limit.
         * @param {Array<Experimentv1State>} [states] Limit trials to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentTrials(experimentId: number, sortBy?: V1GetExperimentTrialsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Experimentv1State>, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getExperimentTrials.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/trials`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortBy !== undefined) {
                localVarQueryParameter['sortBy'] = sortBy
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the validation history for an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentValidationHistory(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getExperimentValidationHistory.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/validation-history`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDef(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getModelDef.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/model_def`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get one file content of model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {V1GetModelDefFileRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefFile(experimentId: number, body: V1GetModelDefFileRequest, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getModelDefFile.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling getModelDefFile.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/file`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the model definition file tree of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefTree(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getModelDefTree.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/file_tree`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a single trial.
         * @param {number} trialId The requested trial's id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrial(trialId: number, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling getTrial.');
            }
            const localVarPath = `/api/v1/trials/{trialId}`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of checkpoints for a trial.
         * @param {number} id The trial id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling getTrialCheckpoints.');
            }
            const localVarPath = `/api/v1/trials/{id}/checkpoints`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (sortByAttr !== undefined) {
                localVarQueryParameter['sortByAttr'] = sortByAttr
            }
            
            if (sortByMetric !== undefined) {
                localVarQueryParameter['sortByMetric'] = sortByMetric
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling killExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/kill`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1KillExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiments(projectId: number, body: V1KillExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling killExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling killExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/kill`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill a trial.
         * @param {number} id The trial id
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killTrial(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling killTrial.');
            }
            const localVarPath = `/api/v1/trials/{id}/kill`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Move an experiment into a project.
         * @param {number} experimentId The id of the experiment being moved.
         * @param {V1MoveExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiment(experimentId: number, body: V1MoveExperimentRequest, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling moveExperiment.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling moveExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/move`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Move multiple experiments into a project.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1MoveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiments(projectId: number, body: V1MoveExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling moveExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling moveExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/move`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Patch an experiment's fields.
         * @param {number} experimentId The id of the experiment.
         * @param {V1PatchExperiment} body Patched experiment attributes.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchExperiment(experimentId: number, body: V1PatchExperiment, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling patchExperiment.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling patchExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Pause an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling pauseExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/pause`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Pause multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PauseExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiments(projectId: number, body: V1PauseExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling pauseExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling pauseExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/pause`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Preview hyperparameter search.
         * @param {V1PreviewHPSearchRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        previewHPSearch(body: V1PreviewHPSearchRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling previewHPSearch.');
            }
            const localVarPath = `/api/v1/preview-hp-search`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Put a new label on the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to add.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentLabel(experimentId: number, label: string, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling putExperimentLabel.');
            }
            // verify required parameter 'label' is not null or undefined
            if (label === null || label === undefined) {
                throw new RequiredError('label','Required parameter label was null or undefined when calling putExperimentLabel.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/labels/{label}`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)))
                .replace(`{${"label"}}`, encodeURIComponent(String(label)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {V1PutExperimentRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentRetainLogs(experimentId: number, body: V1PutExperimentRetainLogsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling putExperimentRetainLogs.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putExperimentRetainLogs.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/retain_logs`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PutExperimentsRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentsRetainLogs(projectId: number, body: V1PutExperimentsRetainLogsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling putExperimentsRetainLogs.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putExperimentsRetainLogs.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/retain_logs`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get experiments with grouping and search syntax
         * @param {V1SearchExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        searchExperiments(body: V1SearchExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling searchExperiments.');
            }
            const localVarPath = `/api/v1/experiments-search`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Stream trial logs.
         * @param {number} trialId The id of the trial.
         * @param {number} [limit] Limit the number of trial logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the trial stops.
         * @param {Array<string>} [agentIds] Limit the trial logs to a subset of agents.
         * @param {Array<string>} [containerIds] Limit the trial logs to a subset of containers.
         * @param {Array<number>} [rankIds] Limit the trial logs to a subset of ranks.
         * @param {Array<V1LogLevel>} [levels] Limit the trial logs to a subset of agents.   - LOG_LEVEL_UNSPECIFIED: Unspecified log level.  - LOG_LEVEL_TRACE: A log level of TRACE.  - LOG_LEVEL_DEBUG: A log level of DEBUG.  - LOG_LEVEL_INFO: A log level of INFO.  - LOG_LEVEL_WARNING: A log level of WARNING.  - LOG_LEVEL_ERROR: A log level of ERROR.  - LOG_LEVEL_CRITICAL: A log level of CRITICAL.
         * @param {Array<string>} [stdtypes] Limit the trial logs to a subset of output streams.
         * @param {Array<string>} [sources] Limit the trial logs to a subset of sources.
         * @param {Date | DateString} [timestampBefore] Limit the trial logs to ones with a timestamp before a given time.
         * @param {Date | DateString} [timestampAfter] Limit the trial logs to ones with a timestamp after a given time.
         * @param {V1OrderBy} [orderBy] Order logs in either ascending or descending order by timestamp.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {string} [searchText] Search the logs by whether the text contains a substring.
         * @param {boolean} [enableRegex] Search text is regex. Default to false.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogs(trialId: number, limit?: number, follow?: boolean, agentIds?: Array<string>, containerIds?: Array<string>, rankIds?: Array<number>, levels?: Array<V1LogLevel>, stdtypes?: Array<string>, sources?: Array<string>, timestampBefore?: Date | DateString, timestampAfter?: Date | DateString, orderBy?: V1OrderBy, searchText?: string, enableRegex?: boolean, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling trialLogs.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/logs`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (follow !== undefined) {
                localVarQueryParameter['follow'] = follow
            }
            
            if (agentIds) {
                localVarQueryParameter['agentIds'] = agentIds
            }
            
            if (containerIds) {
                localVarQueryParameter['containerIds'] = containerIds
            }
            
            if (rankIds) {
                localVarQueryParameter['rankIds'] = rankIds
            }
            
            if (levels) {
                localVarQueryParameter['levels'] = levels
            }
            
            if (stdtypes) {
                localVarQueryParameter['stdtypes'] = stdtypes
            }
            
            if (sources) {
                localVarQueryParameter['sources'] = sources
            }
            
            if (timestampBefore) {
                localVarQueryParameter['timestampBefore'] = typeof timestampBefore === "string" ? timestampBefore : timestampBefore.toISOString()
            }
            
            if (timestampAfter) {
                localVarQueryParameter['timestampAfter'] = typeof timestampAfter === "string" ? timestampAfter : timestampAfter.toISOString()
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (searchText !== undefined) {
                localVarQueryParameter['searchText'] = searchText
            }
            
            if (enableRegex !== undefined) {
                localVarQueryParameter['enableRegex'] = enableRegex
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Stream trial log fields.
         * @param {number} trialId The ID of the trial.
         * @param {boolean} [follow] Continue following fields until the trial stops.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogsFields(trialId: number, follow?: boolean, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling trialLogsFields.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/logs/fields`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (follow !== undefined) {
                localVarQueryParameter['follow'] = follow
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unarchive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiment(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling unarchiveExperiment.');
            }
            const localVarPath = `/api/v1/experiments/{id}/unarchive`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unarchive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1UnarchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiments(projectId: number, body: V1UnarchiveExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling unarchiveExperiments.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling unarchiveExperiments.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/experiments/unarchive`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * ExperimentsApi - functional programming interface
 * @export
 */
export const ExperimentsApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Activate an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ActivateExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).activateExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Activate multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ActivateExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiments(projectId: number, body: V1ActivateExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ActivateExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).activateExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Archive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ArchiveExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).archiveExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Archive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ArchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiments(projectId: number, body: V1ArchiveExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ArchiveExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).archiveExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Cancel an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CancelExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).cancelExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Cancel multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1CancelExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiments(projectId: number, body: V1CancelExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CancelExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).cancelExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Return a downsampled time series of metrics from multiple trials to compare them side-by-side.
         * @param {Array<number>} [trialIds] The requested trial ids.
         * @param {number} [maxDatapoints] The maximum number of data points to return after downsampling.
         * @param {Array<string>} [metricNames] The names of selected metrics.
         * @param {number} [startBatches] Sample from metrics after this batch number.
         * @param {number} [endBatches] Sample from metrics before this batch number.
         * @param {V1MetricType} [metricType] Metric group.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {Array<string>} [metricIds] metric ids for the query. must be in the form group.metric_name.
         * @param {string} [timeSeriesFilterName] metric or column name for the filter.
         * @param {number} [timeSeriesFilterDoubleRangeLt] Less than.
         * @param {number} [timeSeriesFilterDoubleRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterDoubleRangeGt] Greater than.
         * @param {number} [timeSeriesFilterDoubleRangeGte] Greater than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeLt] Less than.
         * @param {number} [timeSeriesFilterIntegerRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeGt] Greater than.
         * @param {number} [timeSeriesFilterIntegerRangeGte] Greater than or equal.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeNotIn] Not in a set.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLt] Less than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLte] Less than or equal.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGt] Greater than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGte] Greater than or equal.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        compareTrials(trialIds?: Array<number>, maxDatapoints?: number, metricNames?: Array<string>, startBatches?: number, endBatches?: number, metricType?: V1MetricType, group?: string, metricIds?: Array<string>, timeSeriesFilterName?: string, timeSeriesFilterDoubleRangeLt?: number, timeSeriesFilterDoubleRangeLte?: number, timeSeriesFilterDoubleRangeGt?: number, timeSeriesFilterDoubleRangeGte?: number, timeSeriesFilterIntegerRangeLt?: number, timeSeriesFilterIntegerRangeLte?: number, timeSeriesFilterIntegerRangeGt?: number, timeSeriesFilterIntegerRangeGte?: number, timeSeriesFilterIntegerRangeIncl?: Array<number>, timeSeriesFilterIntegerRangeNotIn?: Array<number>, timeSeriesFilterTimeRangeLt?: Date | DateString, timeSeriesFilterTimeRangeLte?: Date | DateString, timeSeriesFilterTimeRangeGt?: Date | DateString, timeSeriesFilterTimeRangeGte?: Date | DateString, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CompareTrialsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).compareTrials(trialIds, maxDatapoints, metricNames, startBatches, endBatches, metricType, group, metricIds, timeSeriesFilterName, timeSeriesFilterDoubleRangeLt, timeSeriesFilterDoubleRangeLte, timeSeriesFilterDoubleRangeGt, timeSeriesFilterDoubleRangeGte, timeSeriesFilterIntegerRangeLt, timeSeriesFilterIntegerRangeLte, timeSeriesFilterIntegerRangeGt, timeSeriesFilterIntegerRangeGte, timeSeriesFilterIntegerRangeIncl, timeSeriesFilterIntegerRangeNotIn, timeSeriesFilterTimeRangeLt, timeSeriesFilterTimeRangeLte, timeSeriesFilterTimeRangeGt, timeSeriesFilterTimeRangeGte, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete the requested experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiment(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).deleteExperiment(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete a label from the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to delete.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperimentLabel(experimentId: number, label: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteExperimentLabelResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).deleteExperimentLabel(experimentId, label, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1DeleteExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiments(projectId: number, body: V1DeleteExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).deleteExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete tensorboard files.
         * @param {number} experimentId ID of experiment that the tensorboard files are linked to.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteTensorboardFiles(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteTensorboardFilesResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).deleteTensorboardFiles(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the requested experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiment(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperiment(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of checkpoints for an experiment.
         * @param {number} id The experiment id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentCheckpointsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperimentCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of unique experiment labels (sorted by popularity).
         * @param {number} [projectId] Filter experiments by project.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentLabels(projectId?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentLabelsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperimentLabels(projectId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of experiments.
         * @param {V1GetExperimentsRequestSortBy} [sortBy] Sort experiments by the given field.   - SORT_BY_UNSPECIFIED: Returns experiments in an unsorted list.  - SORT_BY_ID: Returns experiments sorted by id.  - SORT_BY_DESCRIPTION: Returns experiments sorted by description.  - SORT_BY_START_TIME: Return experiments sorted by start time.  - SORT_BY_END_TIME: Return experiments sorted by end time. Experiments without end_time are returned after the ones with end_time.  - SORT_BY_STATE: Return experiments sorted by state.  - SORT_BY_NUM_TRIALS: Return experiments sorted by number of trials.  - SORT_BY_PROGRESS: Return experiments sorted by progress.  - SORT_BY_USER: Return experiments sorted by user.  - SORT_BY_NAME: Returns experiments sorted by name.  - SORT_BY_FORKED_FROM: Returns experiments sorted by originating model.  - SORT_BY_RESOURCE_POOL: Returns experiments sorted by resource pool.  - SORT_BY_PROJECT_ID: Returns experiments sorted by project.  - SORT_BY_CHECKPOINT_SIZE: Returns experiments sorted by checkpoint size.  - SORT_BY_CHECKPOINT_COUNT: Returns experiments sorted by checkpoint count.  - SORT_BY_SEARCHER_METRIC_VAL: Returns experiments sorted by searcher metric value..
         * @param {V1OrderBy} [orderBy] Order experiments in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of experiments before returning results. Negative values denote number of experiments to skip from the end before returning results.
         * @param {number} [limit] Limit the number of experiments. 0 or Unspecified - returns a default of 100. -1               - returns everything. -2               - returns pagination info but no experiments.
         * @param {string} [description] Limit experiments to those that match the description.
         * @param {string} [name] Limit experiments to those that match the name.
         * @param {Array<string>} [labels] Limit experiments to those that match the provided labels.
         * @param {boolean} [archived] Limit experiments to those that are archived.
         * @param {Array<Experimentv1State>} [states] Limit experiments to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {Array<string>} [users] Limit experiments to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit experiments to those that are owned by users with the specified userIds.
         * @param {number} [projectId] Limit experiments to those within a specified project, or 0 for all projects.
         * @param {number} [experimentIdFilterLt] Less than.
         * @param {number} [experimentIdFilterLte] Less than or equal.
         * @param {number} [experimentIdFilterGt] Greater than.
         * @param {number} [experimentIdFilterGte] Greater than or equal.
         * @param {Array<number>} [experimentIdFilterIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [experimentIdFilterNotIn] Not in a set.
         * @param {boolean} [showTrialData] whether to surface trial specific data from the best trial.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiments(sortBy?: V1GetExperimentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, description?: string, name?: string, labels?: Array<string>, archived?: boolean, states?: Array<Experimentv1State>, users?: Array<string>, userIds?: Array<number>, projectId?: number, experimentIdFilterLt?: number, experimentIdFilterLte?: number, experimentIdFilterGt?: number, experimentIdFilterGte?: number, experimentIdFilterIncl?: Array<number>, experimentIdFilterNotIn?: Array<number>, showTrialData?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperiments(sortBy, orderBy, offset, limit, description, name, labels, archived, states, users, userIds, projectId, experimentIdFilterLt, experimentIdFilterLte, experimentIdFilterGt, experimentIdFilterGte, experimentIdFilterIncl, experimentIdFilterNotIn, showTrialData, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the list of trials for an experiment.
         * @param {number} experimentId Limit trials to those that are owned by the specified experiments.
         * @param {V1GetExperimentTrialsRequestSortBy} [sortBy] Sort trials by the given field.   - SORT_BY_UNSPECIFIED: Returns trials in an unsorted list.  - SORT_BY_ID: Returns trials sorted by id.  - SORT_BY_START_TIME: Return trials sorted by start time.  - SORT_BY_END_TIME: Return trials sorted by end time. Trials without end times are returned after trials that are.  - SORT_BY_STATE: Return trials sorted by state.  - SORT_BY_BEST_VALIDATION_METRIC: Return the trials sorted by the best metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_LATEST_VALIDATION_METRIC: Return the trials sorted by the latest metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_BATCHES_PROCESSED: Return the trials sorted by the number of batches completed.  - SORT_BY_DURATION: Return the trials sorted by the total duration.  - SORT_BY_RESTARTS: Return the trials sorted by the number of restarts.  - SORT_BY_CHECKPOINT_SIZE: Return the trials sorted by checkpoint size.  - SORT_BY_LOG_RETENTION_DAYS: Return the trials sorted by number of log retention days.
         * @param {V1OrderBy} [orderBy] Order trials in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of trials before returning results. Negative values denote number of trials to skip from the end before returning results.
         * @param {number} [limit] Limit the number of trials. A value of 0 denotes no limit.
         * @param {Array<Experimentv1State>} [states] Limit trials to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentTrials(experimentId: number, sortBy?: V1GetExperimentTrialsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Experimentv1State>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentTrialsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperimentTrials(experimentId, sortBy, orderBy, offset, limit, states, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the validation history for an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentValidationHistory(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetExperimentValidationHistoryResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getExperimentValidationHistory(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDef(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetModelDefResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getModelDef(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get one file content of model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {V1GetModelDefFileRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefFile(experimentId: number, body: V1GetModelDefFileRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetModelDefFileResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getModelDefFile(experimentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the model definition file tree of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefTree(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetModelDefTreeResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getModelDefTree(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a single trial.
         * @param {number} trialId The requested trial's id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrial(trialId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getTrial(trialId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of checkpoints for a trial.
         * @param {number} id The trial id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialCheckpointsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).getTrialCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).killExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1KillExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiments(projectId: number, body: V1KillExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).killExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill a trial.
         * @param {number} id The trial id
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killTrial(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillTrialResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).killTrial(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Move an experiment into a project.
         * @param {number} experimentId The id of the experiment being moved.
         * @param {V1MoveExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiment(experimentId: number, body: V1MoveExperimentRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1MoveExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).moveExperiment(experimentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Move multiple experiments into a project.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1MoveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiments(projectId: number, body: V1MoveExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1MoveExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).moveExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Patch an experiment's fields.
         * @param {number} experimentId The id of the experiment.
         * @param {V1PatchExperiment} body Patched experiment attributes.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchExperiment(experimentId: number, body: V1PatchExperiment, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PatchExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).patchExperiment(experimentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Pause an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PauseExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).pauseExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Pause multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PauseExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiments(projectId: number, body: V1PauseExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PauseExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).pauseExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Preview hyperparameter search.
         * @param {V1PreviewHPSearchRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        previewHPSearch(body: V1PreviewHPSearchRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PreviewHPSearchResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).previewHPSearch(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Put a new label on the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to add.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentLabel(experimentId: number, label: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutExperimentLabelResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).putExperimentLabel(experimentId, label, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {V1PutExperimentRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentRetainLogs(experimentId: number, body: V1PutExperimentRetainLogsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutExperimentRetainLogsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).putExperimentRetainLogs(experimentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PutExperimentsRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentsRetainLogs(projectId: number, body: V1PutExperimentsRetainLogsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutExperimentsRetainLogsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).putExperimentsRetainLogs(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get experiments with grouping and search syntax
         * @param {V1SearchExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        searchExperiments(body: V1SearchExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1SearchExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).searchExperiments(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Stream trial logs.
         * @param {number} trialId The id of the trial.
         * @param {number} [limit] Limit the number of trial logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the trial stops.
         * @param {Array<string>} [agentIds] Limit the trial logs to a subset of agents.
         * @param {Array<string>} [containerIds] Limit the trial logs to a subset of containers.
         * @param {Array<number>} [rankIds] Limit the trial logs to a subset of ranks.
         * @param {Array<V1LogLevel>} [levels] Limit the trial logs to a subset of agents.   - LOG_LEVEL_UNSPECIFIED: Unspecified log level.  - LOG_LEVEL_TRACE: A log level of TRACE.  - LOG_LEVEL_DEBUG: A log level of DEBUG.  - LOG_LEVEL_INFO: A log level of INFO.  - LOG_LEVEL_WARNING: A log level of WARNING.  - LOG_LEVEL_ERROR: A log level of ERROR.  - LOG_LEVEL_CRITICAL: A log level of CRITICAL.
         * @param {Array<string>} [stdtypes] Limit the trial logs to a subset of output streams.
         * @param {Array<string>} [sources] Limit the trial logs to a subset of sources.
         * @param {Date | DateString} [timestampBefore] Limit the trial logs to ones with a timestamp before a given time.
         * @param {Date | DateString} [timestampAfter] Limit the trial logs to ones with a timestamp after a given time.
         * @param {V1OrderBy} [orderBy] Order logs in either ascending or descending order by timestamp.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {string} [searchText] Search the logs by whether the text contains a substring.
         * @param {boolean} [enableRegex] Search text is regex. Default to false.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogs(trialId: number, limit?: number, follow?: boolean, agentIds?: Array<string>, containerIds?: Array<string>, rankIds?: Array<number>, levels?: Array<V1LogLevel>, stdtypes?: Array<string>, sources?: Array<string>, timestampBefore?: Date | DateString, timestampAfter?: Date | DateString, orderBy?: V1OrderBy, searchText?: string, enableRegex?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<StreamResultOfV1TrialLogsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).trialLogs(trialId, limit, follow, agentIds, containerIds, rankIds, levels, stdtypes, sources, timestampBefore, timestampAfter, orderBy, searchText, enableRegex, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Stream trial log fields.
         * @param {number} trialId The ID of the trial.
         * @param {boolean} [follow] Continue following fields until the trial stops.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogsFields(trialId: number, follow?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<StreamResultOfV1TrialLogsFieldsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).trialLogsFields(trialId, follow, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Unarchive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiment(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1UnarchiveExperimentResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).unarchiveExperiment(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Unarchive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1UnarchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiments(projectId: number, body: V1UnarchiveExperimentsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1UnarchiveExperimentsResponse> {
            const localVarFetchArgs = ExperimentsApiFetchParamCreator(configuration).unarchiveExperiments(projectId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
    }
};

/**
 * ExperimentsApi - factory interface
 * @export
 */
export const ExperimentsApiFactory = function (configuration?: Configuration, fetch?: FetchAPI, basePath?: string) {
    return {
        /**
         * 
         * @summary Activate an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).activateExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Activate multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ActivateExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        activateExperiments(projectId: number, body: V1ActivateExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).activateExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Archive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).archiveExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Archive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1ArchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveExperiments(projectId: number, body: V1ArchiveExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).archiveExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Cancel an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).cancelExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Cancel multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1CancelExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cancelExperiments(projectId: number, body: V1CancelExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).cancelExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Return a downsampled time series of metrics from multiple trials to compare them side-by-side.
         * @param {Array<number>} [trialIds] The requested trial ids.
         * @param {number} [maxDatapoints] The maximum number of data points to return after downsampling.
         * @param {Array<string>} [metricNames] The names of selected metrics.
         * @param {number} [startBatches] Sample from metrics after this batch number.
         * @param {number} [endBatches] Sample from metrics before this batch number.
         * @param {V1MetricType} [metricType] Metric group.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {Array<string>} [metricIds] metric ids for the query. must be in the form group.metric_name.
         * @param {string} [timeSeriesFilterName] metric or column name for the filter.
         * @param {number} [timeSeriesFilterDoubleRangeLt] Less than.
         * @param {number} [timeSeriesFilterDoubleRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterDoubleRangeGt] Greater than.
         * @param {number} [timeSeriesFilterDoubleRangeGte] Greater than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeLt] Less than.
         * @param {number} [timeSeriesFilterIntegerRangeLte] Less than or equal.
         * @param {number} [timeSeriesFilterIntegerRangeGt] Greater than.
         * @param {number} [timeSeriesFilterIntegerRangeGte] Greater than or equal.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [timeSeriesFilterIntegerRangeNotIn] Not in a set.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLt] Less than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeLte] Less than or equal.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGt] Greater than.
         * @param {Date | DateString} [timeSeriesFilterTimeRangeGte] Greater than or equal.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        compareTrials(trialIds?: Array<number>, maxDatapoints?: number, metricNames?: Array<string>, startBatches?: number, endBatches?: number, metricType?: V1MetricType, group?: string, metricIds?: Array<string>, timeSeriesFilterName?: string, timeSeriesFilterDoubleRangeLt?: number, timeSeriesFilterDoubleRangeLte?: number, timeSeriesFilterDoubleRangeGt?: number, timeSeriesFilterDoubleRangeGte?: number, timeSeriesFilterIntegerRangeLt?: number, timeSeriesFilterIntegerRangeLte?: number, timeSeriesFilterIntegerRangeGt?: number, timeSeriesFilterIntegerRangeGte?: number, timeSeriesFilterIntegerRangeIncl?: Array<number>, timeSeriesFilterIntegerRangeNotIn?: Array<number>, timeSeriesFilterTimeRangeLt?: Date | DateString, timeSeriesFilterTimeRangeLte?: Date | DateString, timeSeriesFilterTimeRangeGt?: Date | DateString, timeSeriesFilterTimeRangeGte?: Date | DateString, options?: any) {
            return ExperimentsApiFp(configuration).compareTrials(trialIds, maxDatapoints, metricNames, startBatches, endBatches, metricType, group, metricIds, timeSeriesFilterName, timeSeriesFilterDoubleRangeLt, timeSeriesFilterDoubleRangeLte, timeSeriesFilterDoubleRangeGt, timeSeriesFilterDoubleRangeGte, timeSeriesFilterIntegerRangeLt, timeSeriesFilterIntegerRangeLte, timeSeriesFilterIntegerRangeGt, timeSeriesFilterIntegerRangeGte, timeSeriesFilterIntegerRangeIncl, timeSeriesFilterIntegerRangeNotIn, timeSeriesFilterTimeRangeLt, timeSeriesFilterTimeRangeLte, timeSeriesFilterTimeRangeGt, timeSeriesFilterTimeRangeGte, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete the requested experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiment(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).deleteExperiment(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete a label from the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to delete.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperimentLabel(experimentId: number, label: string, options?: any) {
            return ExperimentsApiFp(configuration).deleteExperimentLabel(experimentId, label, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1DeleteExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteExperiments(projectId: number, body: V1DeleteExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).deleteExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Delete tensorboard files.
         * @param {number} experimentId ID of experiment that the tensorboard files are linked to.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteTensorboardFiles(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).deleteTensorboardFiles(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the requested experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiment(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).getExperiment(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a list of checkpoints for an experiment.
         * @param {number} id The experiment id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any) {
            return ExperimentsApiFp(configuration).getExperimentCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a list of unique experiment labels (sorted by popularity).
         * @param {number} [projectId] Filter experiments by project.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentLabels(projectId?: number, options?: any) {
            return ExperimentsApiFp(configuration).getExperimentLabels(projectId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a list of experiments.
         * @param {V1GetExperimentsRequestSortBy} [sortBy] Sort experiments by the given field.   - SORT_BY_UNSPECIFIED: Returns experiments in an unsorted list.  - SORT_BY_ID: Returns experiments sorted by id.  - SORT_BY_DESCRIPTION: Returns experiments sorted by description.  - SORT_BY_START_TIME: Return experiments sorted by start time.  - SORT_BY_END_TIME: Return experiments sorted by end time. Experiments without end_time are returned after the ones with end_time.  - SORT_BY_STATE: Return experiments sorted by state.  - SORT_BY_NUM_TRIALS: Return experiments sorted by number of trials.  - SORT_BY_PROGRESS: Return experiments sorted by progress.  - SORT_BY_USER: Return experiments sorted by user.  - SORT_BY_NAME: Returns experiments sorted by name.  - SORT_BY_FORKED_FROM: Returns experiments sorted by originating model.  - SORT_BY_RESOURCE_POOL: Returns experiments sorted by resource pool.  - SORT_BY_PROJECT_ID: Returns experiments sorted by project.  - SORT_BY_CHECKPOINT_SIZE: Returns experiments sorted by checkpoint size.  - SORT_BY_CHECKPOINT_COUNT: Returns experiments sorted by checkpoint count.  - SORT_BY_SEARCHER_METRIC_VAL: Returns experiments sorted by searcher metric value..
         * @param {V1OrderBy} [orderBy] Order experiments in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of experiments before returning results. Negative values denote number of experiments to skip from the end before returning results.
         * @param {number} [limit] Limit the number of experiments. 0 or Unspecified - returns a default of 100. -1               - returns everything. -2               - returns pagination info but no experiments.
         * @param {string} [description] Limit experiments to those that match the description.
         * @param {string} [name] Limit experiments to those that match the name.
         * @param {Array<string>} [labels] Limit experiments to those that match the provided labels.
         * @param {boolean} [archived] Limit experiments to those that are archived.
         * @param {Array<Experimentv1State>} [states] Limit experiments to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {Array<string>} [users] Limit experiments to those that are owned by users with the specified usernames.
         * @param {Array<number>} [userIds] Limit experiments to those that are owned by users with the specified userIds.
         * @param {number} [projectId] Limit experiments to those within a specified project, or 0 for all projects.
         * @param {number} [experimentIdFilterLt] Less than.
         * @param {number} [experimentIdFilterLte] Less than or equal.
         * @param {number} [experimentIdFilterGt] Greater than.
         * @param {number} [experimentIdFilterGte] Greater than or equal.
         * @param {Array<number>} [experimentIdFilterIncl] In a set. `in` is a reserved word in python.
         * @param {Array<number>} [experimentIdFilterNotIn] Not in a set.
         * @param {boolean} [showTrialData] whether to surface trial specific data from the best trial.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperiments(sortBy?: V1GetExperimentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, description?: string, name?: string, labels?: Array<string>, archived?: boolean, states?: Array<Experimentv1State>, users?: Array<string>, userIds?: Array<number>, projectId?: number, experimentIdFilterLt?: number, experimentIdFilterLte?: number, experimentIdFilterGt?: number, experimentIdFilterGte?: number, experimentIdFilterIncl?: Array<number>, experimentIdFilterNotIn?: Array<number>, showTrialData?: boolean, options?: any) {
            return ExperimentsApiFp(configuration).getExperiments(sortBy, orderBy, offset, limit, description, name, labels, archived, states, users, userIds, projectId, experimentIdFilterLt, experimentIdFilterLte, experimentIdFilterGt, experimentIdFilterGte, experimentIdFilterIncl, experimentIdFilterNotIn, showTrialData, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the list of trials for an experiment.
         * @param {number} experimentId Limit trials to those that are owned by the specified experiments.
         * @param {V1GetExperimentTrialsRequestSortBy} [sortBy] Sort trials by the given field.   - SORT_BY_UNSPECIFIED: Returns trials in an unsorted list.  - SORT_BY_ID: Returns trials sorted by id.  - SORT_BY_START_TIME: Return trials sorted by start time.  - SORT_BY_END_TIME: Return trials sorted by end time. Trials without end times are returned after trials that are.  - SORT_BY_STATE: Return trials sorted by state.  - SORT_BY_BEST_VALIDATION_METRIC: Return the trials sorted by the best metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_LATEST_VALIDATION_METRIC: Return the trials sorted by the latest metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_BATCHES_PROCESSED: Return the trials sorted by the number of batches completed.  - SORT_BY_DURATION: Return the trials sorted by the total duration.  - SORT_BY_RESTARTS: Return the trials sorted by the number of restarts.  - SORT_BY_CHECKPOINT_SIZE: Return the trials sorted by checkpoint size.  - SORT_BY_LOG_RETENTION_DAYS: Return the trials sorted by number of log retention days.
         * @param {V1OrderBy} [orderBy] Order trials in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of trials before returning results. Negative values denote number of trials to skip from the end before returning results.
         * @param {number} [limit] Limit the number of trials. A value of 0 denotes no limit.
         * @param {Array<Experimentv1State>} [states] Limit trials to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentTrials(experimentId: number, sortBy?: V1GetExperimentTrialsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Experimentv1State>, options?: any) {
            return ExperimentsApiFp(configuration).getExperimentTrials(experimentId, sortBy, orderBy, offset, limit, states, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the validation history for an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getExperimentValidationHistory(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).getExperimentValidationHistory(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDef(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).getModelDef(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get one file content of model definition of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {V1GetModelDefFileRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefFile(experimentId: number, body: V1GetModelDefFileRequest, options?: any) {
            return ExperimentsApiFp(configuration).getModelDefFile(experimentId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get the model definition file tree of an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getModelDefTree(experimentId: number, options?: any) {
            return ExperimentsApiFp(configuration).getModelDefTree(experimentId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a single trial.
         * @param {number} trialId The requested trial's id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrial(trialId: number, options?: any) {
            return ExperimentsApiFp(configuration).getTrial(trialId, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get a list of checkpoints for a trial.
         * @param {number} id The trial id.
         * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
         * @param {string} [sortByMetric] Sort by custom validation metric name.
         * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
         * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
         * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any) {
            return ExperimentsApiFp(configuration).getTrialCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Kill an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).killExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Kill multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1KillExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killExperiments(projectId: number, body: V1KillExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).killExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Kill a trial.
         * @param {number} id The trial id
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killTrial(id: number, options?: any) {
            return ExperimentsApiFp(configuration).killTrial(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Move an experiment into a project.
         * @param {number} experimentId The id of the experiment being moved.
         * @param {V1MoveExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiment(experimentId: number, body: V1MoveExperimentRequest, options?: any) {
            return ExperimentsApiFp(configuration).moveExperiment(experimentId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Move multiple experiments into a project.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1MoveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveExperiments(projectId: number, body: V1MoveExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).moveExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Patch an experiment's fields.
         * @param {number} experimentId The id of the experiment.
         * @param {V1PatchExperiment} body Patched experiment attributes.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchExperiment(experimentId: number, body: V1PatchExperiment, options?: any) {
            return ExperimentsApiFp(configuration).patchExperiment(experimentId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Pause an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).pauseExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Pause multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PauseExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseExperiments(projectId: number, body: V1PauseExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).pauseExperiments(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Preview hyperparameter search.
         * @param {V1PreviewHPSearchRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        previewHPSearch(body: V1PreviewHPSearchRequest, options?: any) {
            return ExperimentsApiFp(configuration).previewHPSearch(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Put a new label on the experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {string} label The label to add.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentLabel(experimentId: number, label: string, options?: any) {
            return ExperimentsApiFp(configuration).putExperimentLabel(experimentId, label, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} experimentId The ID of the experiment.
         * @param {V1PutExperimentRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentRetainLogs(experimentId: number, body: V1PutExperimentRetainLogsRequest, options?: any) {
            return ExperimentsApiFp(configuration).putExperimentRetainLogs(experimentId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Retain logs for an experiment.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1PutExperimentsRetainLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperimentsRetainLogs(projectId: number, body: V1PutExperimentsRetainLogsRequest, options?: any) {
            return ExperimentsApiFp(configuration).putExperimentsRetainLogs(projectId, body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Get experiments with grouping and search syntax
         * @param {V1SearchExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        searchExperiments(body: V1SearchExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).searchExperiments(body, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Stream trial logs.
         * @param {number} trialId The id of the trial.
         * @param {number} [limit] Limit the number of trial logs. A value of 0 denotes no limit.
         * @param {boolean} [follow] Continue following logs until the trial stops.
         * @param {Array<string>} [agentIds] Limit the trial logs to a subset of agents.
         * @param {Array<string>} [containerIds] Limit the trial logs to a subset of containers.
         * @param {Array<number>} [rankIds] Limit the trial logs to a subset of ranks.
         * @param {Array<V1LogLevel>} [levels] Limit the trial logs to a subset of agents.   - LOG_LEVEL_UNSPECIFIED: Unspecified log level.  - LOG_LEVEL_TRACE: A log level of TRACE.  - LOG_LEVEL_DEBUG: A log level of DEBUG.  - LOG_LEVEL_INFO: A log level of INFO.  - LOG_LEVEL_WARNING: A log level of WARNING.  - LOG_LEVEL_ERROR: A log level of ERROR.  - LOG_LEVEL_CRITICAL: A log level of CRITICAL.
         * @param {Array<string>} [stdtypes] Limit the trial logs to a subset of output streams.
         * @param {Array<string>} [sources] Limit the trial logs to a subset of sources.
         * @param {Date | DateString} [timestampBefore] Limit the trial logs to ones with a timestamp before a given time.
         * @param {Date | DateString} [timestampAfter] Limit the trial logs to ones with a timestamp after a given time.
         * @param {V1OrderBy} [orderBy] Order logs in either ascending or descending order by timestamp.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {string} [searchText] Search the logs by whether the text contains a substring.
         * @param {boolean} [enableRegex] Search text is regex. Default to false.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogs(trialId: number, limit?: number, follow?: boolean, agentIds?: Array<string>, containerIds?: Array<string>, rankIds?: Array<number>, levels?: Array<V1LogLevel>, stdtypes?: Array<string>, sources?: Array<string>, timestampBefore?: Date | DateString, timestampAfter?: Date | DateString, orderBy?: V1OrderBy, searchText?: string, enableRegex?: boolean, options?: any) {
            return ExperimentsApiFp(configuration).trialLogs(trialId, limit, follow, agentIds, containerIds, rankIds, levels, stdtypes, sources, timestampBefore, timestampAfter, orderBy, searchText, enableRegex, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Stream trial log fields.
         * @param {number} trialId The ID of the trial.
         * @param {boolean} [follow] Continue following fields until the trial stops.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialLogsFields(trialId: number, follow?: boolean, options?: any) {
            return ExperimentsApiFp(configuration).trialLogsFields(trialId, follow, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Unarchive an experiment.
         * @param {number} id The experiment id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiment(id: number, options?: any) {
            return ExperimentsApiFp(configuration).unarchiveExperiment(id, options)(fetch, basePath);
        },
        /**
         * 
         * @summary Unarchive multiple experiments.
         * @param {number} projectId Project id that the experiments belong to.
         * @param {V1UnarchiveExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveExperiments(projectId: number, body: V1UnarchiveExperimentsRequest, options?: any) {
            return ExperimentsApiFp(configuration).unarchiveExperiments(projectId, body, options)(fetch, basePath);
        },
    }
};

/**
 * ExperimentsApi - object-oriented interface
 * @export
 * @class
 * @extends {BaseAPI}
 */
export class ExperimentsApi extends BaseAPI {
    /**
     * 
     * @summary Activate an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public activateExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).activateExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Activate multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1ActivateExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public activateExperiments(projectId: number, body: V1ActivateExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).activateExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Archive an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public archiveExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).archiveExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Archive multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1ArchiveExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public archiveExperiments(projectId: number, body: V1ArchiveExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).archiveExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Cancel an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public cancelExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).cancelExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Cancel multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1CancelExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public cancelExperiments(projectId: number, body: V1CancelExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).cancelExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Return a downsampled time series of metrics from multiple trials to compare them side-by-side.
     * @param {Array<number>} [trialIds] The requested trial ids.
     * @param {number} [maxDatapoints] The maximum number of data points to return after downsampling.
     * @param {Array<string>} [metricNames] The names of selected metrics.
     * @param {number} [startBatches] Sample from metrics after this batch number.
     * @param {number} [endBatches] Sample from metrics before this batch number.
     * @param {V1MetricType} [metricType] Metric group.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
     * @param {string} [group] Metric group (training, validation, etc).
     * @param {Array<string>} [metricIds] metric ids for the query. must be in the form group.metric_name.
     * @param {string} [timeSeriesFilterName] metric or column name for the filter.
     * @param {number} [timeSeriesFilterDoubleRangeLt] Less than.
     * @param {number} [timeSeriesFilterDoubleRangeLte] Less than or equal.
     * @param {number} [timeSeriesFilterDoubleRangeGt] Greater than.
     * @param {number} [timeSeriesFilterDoubleRangeGte] Greater than or equal.
     * @param {number} [timeSeriesFilterIntegerRangeLt] Less than.
     * @param {number} [timeSeriesFilterIntegerRangeLte] Less than or equal.
     * @param {number} [timeSeriesFilterIntegerRangeGt] Greater than.
     * @param {number} [timeSeriesFilterIntegerRangeGte] Greater than or equal.
     * @param {Array<number>} [timeSeriesFilterIntegerRangeIncl] In a set. `in` is a reserved word in python.
     * @param {Array<number>} [timeSeriesFilterIntegerRangeNotIn] Not in a set.
     * @param {Date | DateString} [timeSeriesFilterTimeRangeLt] Less than.
     * @param {Date | DateString} [timeSeriesFilterTimeRangeLte] Less than or equal.
     * @param {Date | DateString} [timeSeriesFilterTimeRangeGt] Greater than.
     * @param {Date | DateString} [timeSeriesFilterTimeRangeGte] Greater than or equal.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public compareTrials(trialIds?: Array<number>, maxDatapoints?: number, metricNames?: Array<string>, startBatches?: number, endBatches?: number, metricType?: V1MetricType, group?: string, metricIds?: Array<string>, timeSeriesFilterName?: string, timeSeriesFilterDoubleRangeLt?: number, timeSeriesFilterDoubleRangeLte?: number, timeSeriesFilterDoubleRangeGt?: number, timeSeriesFilterDoubleRangeGte?: number, timeSeriesFilterIntegerRangeLt?: number, timeSeriesFilterIntegerRangeLte?: number, timeSeriesFilterIntegerRangeGt?: number, timeSeriesFilterIntegerRangeGte?: number, timeSeriesFilterIntegerRangeIncl?: Array<number>, timeSeriesFilterIntegerRangeNotIn?: Array<number>, timeSeriesFilterTimeRangeLt?: Date | DateString, timeSeriesFilterTimeRangeLte?: Date | DateString, timeSeriesFilterTimeRangeGt?: Date | DateString, timeSeriesFilterTimeRangeGte?: Date | DateString, options?: any) {
        return ExperimentsApiFp(this.configuration).compareTrials(trialIds, maxDatapoints, metricNames, startBatches, endBatches, metricType, group, metricIds, timeSeriesFilterName, timeSeriesFilterDoubleRangeLt, timeSeriesFilterDoubleRangeLte, timeSeriesFilterDoubleRangeGt, timeSeriesFilterDoubleRangeGte, timeSeriesFilterIntegerRangeLt, timeSeriesFilterIntegerRangeLte, timeSeriesFilterIntegerRangeGt, timeSeriesFilterIntegerRangeGte, timeSeriesFilterIntegerRangeIncl, timeSeriesFilterIntegerRangeNotIn, timeSeriesFilterTimeRangeLt, timeSeriesFilterTimeRangeLte, timeSeriesFilterTimeRangeGt, timeSeriesFilterTimeRangeGte, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete the requested experiment.
     * @param {number} experimentId The ID of the experiment.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public deleteExperiment(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).deleteExperiment(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete a label from the experiment.
     * @param {number} experimentId The ID of the experiment.
     * @param {string} label The label to delete.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public deleteExperimentLabel(experimentId: number, label: string, options?: any) {
        return ExperimentsApiFp(this.configuration).deleteExperimentLabel(experimentId, label, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1DeleteExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public deleteExperiments(projectId: number, body: V1DeleteExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).deleteExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Delete tensorboard files.
     * @param {number} experimentId ID of experiment that the tensorboard files are linked to.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public deleteTensorboardFiles(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).deleteTensorboardFiles(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the requested experiment.
     * @param {number} experimentId The id of the experiment.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperiment(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperiment(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a list of checkpoints for an experiment.
     * @param {number} id The experiment id.
     * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
     * @param {string} [sortByMetric] Sort by custom validation metric name.
     * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
     * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
     * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperimentCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperimentCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a list of unique experiment labels (sorted by popularity).
     * @param {number} [projectId] Filter experiments by project.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperimentLabels(projectId?: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperimentLabels(projectId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a list of experiments.
     * @param {V1GetExperimentsRequestSortBy} [sortBy] Sort experiments by the given field.   - SORT_BY_UNSPECIFIED: Returns experiments in an unsorted list.  - SORT_BY_ID: Returns experiments sorted by id.  - SORT_BY_DESCRIPTION: Returns experiments sorted by description.  - SORT_BY_START_TIME: Return experiments sorted by start time.  - SORT_BY_END_TIME: Return experiments sorted by end time. Experiments without end_time are returned after the ones with end_time.  - SORT_BY_STATE: Return experiments sorted by state.  - SORT_BY_NUM_TRIALS: Return experiments sorted by number of trials.  - SORT_BY_PROGRESS: Return experiments sorted by progress.  - SORT_BY_USER: Return experiments sorted by user.  - SORT_BY_NAME: Returns experiments sorted by name.  - SORT_BY_FORKED_FROM: Returns experiments sorted by originating model.  - SORT_BY_RESOURCE_POOL: Returns experiments sorted by resource pool.  - SORT_BY_PROJECT_ID: Returns experiments sorted by project.  - SORT_BY_CHECKPOINT_SIZE: Returns experiments sorted by checkpoint size.  - SORT_BY_CHECKPOINT_COUNT: Returns experiments sorted by checkpoint count.  - SORT_BY_SEARCHER_METRIC_VAL: Returns experiments sorted by searcher metric value..
     * @param {V1OrderBy} [orderBy] Order experiments in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of experiments before returning results. Negative values denote number of experiments to skip from the end before returning results.
     * @param {number} [limit] Limit the number of experiments. 0 or Unspecified - returns a default of 100. -1               - returns everything. -2               - returns pagination info but no experiments.
     * @param {string} [description] Limit experiments to those that match the description.
     * @param {string} [name] Limit experiments to those that match the name.
     * @param {Array<string>} [labels] Limit experiments to those that match the provided labels.
     * @param {boolean} [archived] Limit experiments to those that are archived.
     * @param {Array<Experimentv1State>} [states] Limit experiments to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
     * @param {Array<string>} [users] Limit experiments to those that are owned by users with the specified usernames.
     * @param {Array<number>} [userIds] Limit experiments to those that are owned by users with the specified userIds.
     * @param {number} [projectId] Limit experiments to those within a specified project, or 0 for all projects.
     * @param {number} [experimentIdFilterLt] Less than.
     * @param {number} [experimentIdFilterLte] Less than or equal.
     * @param {number} [experimentIdFilterGt] Greater than.
     * @param {number} [experimentIdFilterGte] Greater than or equal.
     * @param {Array<number>} [experimentIdFilterIncl] In a set. `in` is a reserved word in python.
     * @param {Array<number>} [experimentIdFilterNotIn] Not in a set.
     * @param {boolean} [showTrialData] whether to surface trial specific data from the best trial.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperiments(sortBy?: V1GetExperimentsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, description?: string, name?: string, labels?: Array<string>, archived?: boolean, states?: Array<Experimentv1State>, users?: Array<string>, userIds?: Array<number>, projectId?: number, experimentIdFilterLt?: number, experimentIdFilterLte?: number, experimentIdFilterGt?: number, experimentIdFilterGte?: number, experimentIdFilterIncl?: Array<number>, experimentIdFilterNotIn?: Array<number>, showTrialData?: boolean, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperiments(sortBy, orderBy, offset, limit, description, name, labels, archived, states, users, userIds, projectId, experimentIdFilterLt, experimentIdFilterLte, experimentIdFilterGt, experimentIdFilterGte, experimentIdFilterIncl, experimentIdFilterNotIn, showTrialData, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the list of trials for an experiment.
     * @param {number} experimentId Limit trials to those that are owned by the specified experiments.
     * @param {V1GetExperimentTrialsRequestSortBy} [sortBy] Sort trials by the given field.   - SORT_BY_UNSPECIFIED: Returns trials in an unsorted list.  - SORT_BY_ID: Returns trials sorted by id.  - SORT_BY_START_TIME: Return trials sorted by start time.  - SORT_BY_END_TIME: Return trials sorted by end time. Trials without end times are returned after trials that are.  - SORT_BY_STATE: Return trials sorted by state.  - SORT_BY_BEST_VALIDATION_METRIC: Return the trials sorted by the best metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_LATEST_VALIDATION_METRIC: Return the trials sorted by the latest metric so far, where the metric is specified by `searcher.metric` in the experiment configuration.  - SORT_BY_BATCHES_PROCESSED: Return the trials sorted by the number of batches completed.  - SORT_BY_DURATION: Return the trials sorted by the total duration.  - SORT_BY_RESTARTS: Return the trials sorted by the number of restarts.  - SORT_BY_CHECKPOINT_SIZE: Return the trials sorted by checkpoint size.  - SORT_BY_LOG_RETENTION_DAYS: Return the trials sorted by number of log retention days.
     * @param {V1OrderBy} [orderBy] Order trials in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of trials before returning results. Negative values denote number of trials to skip from the end before returning results.
     * @param {number} [limit] Limit the number of trials. A value of 0 denotes no limit.
     * @param {Array<Experimentv1State>} [states] Limit trials to those that match the provided state.   - STATE_UNSPECIFIED: The state of the experiment is unknown.  - STATE_ACTIVE: The experiment is in an active state.  - STATE_PAUSED: The experiment is in a paused state  - STATE_STOPPING_COMPLETED: The experiment is completed and is shutting down.  - STATE_STOPPING_CANCELED: The experiment is canceled and is shutting down.  - STATE_STOPPING_ERROR: The experiment is errored and is shutting down.  - STATE_COMPLETED: The experiment is completed and is shut down.  - STATE_CANCELED: The experiment is canceled and is shut down.  - STATE_ERROR: The experiment is errored and is shut down.  - STATE_DELETED: The experiment has been deleted.  - STATE_DELETING: The experiment is deleting.  - STATE_DELETE_FAILED: The experiment failed to delete.  - STATE_STOPPING_KILLED: The experiment is killed and is shutting down.  - STATE_QUEUED: The experiment is queued (waiting to be run, or job state is still queued). Queued is a substate of the Active state.  - STATE_PULLING: The experiment is pulling the image. Pulling is a substate of the Active state.  - STATE_STARTING: The experiment is preparing the environment after finishing pulling the image. Starting is a substate of the Active state.  - STATE_RUNNING: The experiment has an allocation actively running. Running is a substate of the Active state.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperimentTrials(experimentId: number, sortBy?: V1GetExperimentTrialsRequestSortBy, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Experimentv1State>, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperimentTrials(experimentId, sortBy, orderBy, offset, limit, states, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the validation history for an experiment.
     * @param {number} experimentId The id of the experiment.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getExperimentValidationHistory(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getExperimentValidationHistory(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the model definition of an experiment.
     * @param {number} experimentId The id of the experiment.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getModelDef(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getModelDef(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get one file content of model definition of an experiment.
     * @param {number} experimentId The id of the experiment.
     * @param {V1GetModelDefFileRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getModelDefFile(experimentId: number, body: V1GetModelDefFileRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).getModelDefFile(experimentId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get the model definition file tree of an experiment.
     * @param {number} experimentId The id of the experiment.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getModelDefTree(experimentId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getModelDefTree(experimentId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a single trial.
     * @param {number} trialId The requested trial's id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getTrial(trialId: number, options?: any) {
        return ExperimentsApiFp(this.configuration).getTrial(trialId, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get a list of checkpoints for a trial.
     * @param {number} id The trial id.
     * @param {Checkpointv1SortBy} [sortByAttr] Sort by preset checkpoint attribute.   - SORT_BY_UNSPECIFIED: Returns checkpoints in an unsorted list.  - SORT_BY_UUID: Returns checkpoints sorted by UUID.  - SORT_BY_TRIAL_ID: Returns checkpoints sorted by trial id.  - SORT_BY_BATCH_NUMBER: Returns checkpoints sorted by batch number.  - SORT_BY_END_TIME: Returns checkpoints sorted by end time.  - SORT_BY_STATE: Returns checkpoints sorted by state.  - SORT_BY_SEARCHER_METRIC: Returns checkpoints sorted by the experiment's `searcher.metric` configuration setting.
     * @param {string} [sortByMetric] Sort by custom validation metric name.
     * @param {V1OrderBy} [orderBy] Order checkpoints in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {number} [offset] Skip the number of checkpoints before returning results. Negative values denote number of checkpoints to skip from the end before returning results.
     * @param {number} [limit] Limit the number of checkpoints. A value of 0 denotes no limit.
     * @param {Array<Checkpointv1State>} [states] Limit the checkpoints to those that match the states.   - STATE_UNSPECIFIED: The state of the checkpoint is unknown.  - STATE_ACTIVE: The checkpoint is in an active state.  - STATE_COMPLETED: The checkpoint is persisted to checkpoint storage.  - STATE_ERROR: The checkpoint errored.  - STATE_DELETED: The checkpoint has been deleted.  - STATE_PARTIALLY_DELETED: The checkpoint has been partially deleted.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public getTrialCheckpoints(id: number, sortByAttr?: Checkpointv1SortBy, sortByMetric?: string, orderBy?: V1OrderBy, offset?: number, limit?: number, states?: Array<Checkpointv1State>, options?: any) {
        return ExperimentsApiFp(this.configuration).getTrialCheckpoints(id, sortByAttr, sortByMetric, orderBy, offset, limit, states, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Kill an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public killExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).killExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Kill multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1KillExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public killExperiments(projectId: number, body: V1KillExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).killExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Kill a trial.
     * @param {number} id The trial id
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public killTrial(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).killTrial(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Move an experiment into a project.
     * @param {number} experimentId The id of the experiment being moved.
     * @param {V1MoveExperimentRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public moveExperiment(experimentId: number, body: V1MoveExperimentRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).moveExperiment(experimentId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Move multiple experiments into a project.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1MoveExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public moveExperiments(projectId: number, body: V1MoveExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).moveExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Patch an experiment's fields.
     * @param {number} experimentId The id of the experiment.
     * @param {V1PatchExperiment} body Patched experiment attributes.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public patchExperiment(experimentId: number, body: V1PatchExperiment, options?: any) {
        return ExperimentsApiFp(this.configuration).patchExperiment(experimentId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Pause an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public pauseExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).pauseExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Pause multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1PauseExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public pauseExperiments(projectId: number, body: V1PauseExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).pauseExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Preview hyperparameter search.
     * @param {V1PreviewHPSearchRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public previewHPSearch(body: V1PreviewHPSearchRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).previewHPSearch(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Put a new label on the experiment.
     * @param {number} experimentId The ID of the experiment.
     * @param {string} label The label to add.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public putExperimentLabel(experimentId: number, label: string, options?: any) {
        return ExperimentsApiFp(this.configuration).putExperimentLabel(experimentId, label, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Retain logs for an experiment.
     * @param {number} experimentId The ID of the experiment.
     * @param {V1PutExperimentRetainLogsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public putExperimentRetainLogs(experimentId: number, body: V1PutExperimentRetainLogsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).putExperimentRetainLogs(experimentId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Retain logs for an experiment.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1PutExperimentsRetainLogsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public putExperimentsRetainLogs(projectId: number, body: V1PutExperimentsRetainLogsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).putExperimentsRetainLogs(projectId, body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Get experiments with grouping and search syntax
     * @param {V1SearchExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public searchExperiments(body: V1SearchExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).searchExperiments(body, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Stream trial logs.
     * @param {number} trialId The id of the trial.
     * @param {number} [limit] Limit the number of trial logs. A value of 0 denotes no limit.
     * @param {boolean} [follow] Continue following logs until the trial stops.
     * @param {Array<string>} [agentIds] Limit the trial logs to a subset of agents.
     * @param {Array<string>} [containerIds] Limit the trial logs to a subset of containers.
     * @param {Array<number>} [rankIds] Limit the trial logs to a subset of ranks.
     * @param {Array<V1LogLevel>} [levels] Limit the trial logs to a subset of agents.   - LOG_LEVEL_UNSPECIFIED: Unspecified log level.  - LOG_LEVEL_TRACE: A log level of TRACE.  - LOG_LEVEL_DEBUG: A log level of DEBUG.  - LOG_LEVEL_INFO: A log level of INFO.  - LOG_LEVEL_WARNING: A log level of WARNING.  - LOG_LEVEL_ERROR: A log level of ERROR.  - LOG_LEVEL_CRITICAL: A log level of CRITICAL.
     * @param {Array<string>} [stdtypes] Limit the trial logs to a subset of output streams.
     * @param {Array<string>} [sources] Limit the trial logs to a subset of sources.
     * @param {Date | DateString} [timestampBefore] Limit the trial logs to ones with a timestamp before a given time.
     * @param {Date | DateString} [timestampAfter] Limit the trial logs to ones with a timestamp after a given time.
     * @param {V1OrderBy} [orderBy] Order logs in either ascending or descending order by timestamp.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
     * @param {string} [searchText] Search the logs by whether the text contains a substring.
     * @param {boolean} [enableRegex] Search text is regex. Default to false.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public trialLogs(trialId: number, limit?: number, follow?: boolean, agentIds?: Array<string>, containerIds?: Array<string>, rankIds?: Array<number>, levels?: Array<V1LogLevel>, stdtypes?: Array<string>, sources?: Array<string>, timestampBefore?: Date | DateString, timestampAfter?: Date | DateString, orderBy?: V1OrderBy, searchText?: string, enableRegex?: boolean, options?: any) {
        return ExperimentsApiFp(this.configuration).trialLogs(trialId, limit, follow, agentIds, containerIds, rankIds, levels, stdtypes, sources, timestampBefore, timestampAfter, orderBy, searchText, enableRegex, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Stream trial log fields.
     * @param {number} trialId The ID of the trial.
     * @param {boolean} [follow] Continue following fields until the trial stops.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public trialLogsFields(trialId: number, follow?: boolean, options?: any) {
        return ExperimentsApiFp(this.configuration).trialLogsFields(trialId, follow, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Unarchive an experiment.
     * @param {number} id The experiment id.
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public unarchiveExperiment(id: number, options?: any) {
        return ExperimentsApiFp(this.configuration).unarchiveExperiment(id, options)(this.fetch, this.basePath)
    }
    
    /**
     * 
     * @summary Unarchive multiple experiments.
     * @param {number} projectId Project id that the experiments belong to.
     * @param {V1UnarchiveExperimentsRequest} body
     * @param {*} [options] Override http request option.
     * @throws {RequiredError}
     * @memberof ExperimentsApi
     */
    public unarchiveExperiments(projectId: number, body: V1UnarchiveExperimentsRequest, options?: any) {
        return ExperimentsApiFp(this.configuration).unarchiveExperiments(projectId, body, options)(this.fetch, this.basePath)
    }
    
}

/**
 * InternalApi - fetch parameter creator
 * @export
 */
export const InternalApiFetchParamCreator = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Acknowledge the receipt of a signal to stop the given allocation early. This is used indicate and exit 0 isn't final; specifically, it is used for HP search directed early stops and preemption signals (not necessarily just scheduler preemption).
         * @param {string} allocationId The allocation that is acknowledging the request.
         * @param {V1AckAllocationPreemptionSignalRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        ackAllocationPreemptionSignal(allocationId: string, body: V1AckAllocationPreemptionSignalRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling ackAllocationPreemptionSignal.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling ackAllocationPreemptionSignal.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/signals/ack_preemption`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary AllocationAllGather performs an all gather through the master. An allocation can only perform once all gather at a time.
         * @param {string} allocationId The ID of the allocation.
         * @param {V1AllocationAllGatherRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationAllGather(allocationId: string, body: V1AllocationAllGatherRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationAllGather.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling allocationAllGather.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/all_gather`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Report the receipt of a signal to stop the given allocation early. This is used to communicate back from a SLURM job that it has been notified of a pending preememption. Upon a call to this API the RM should then trigger a checkpoint and immediate exit.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationPendingPreemptionSignalRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationPendingPreemptionSignal(allocationId: string, body: V1AllocationPendingPreemptionSignalRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationPendingPreemptionSignal.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling allocationPendingPreemptionSignal.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/signals/pending_preemption`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Long poll preemption signals for the given allocation. If the allocation has been preempted when called, it will return so immediately. Otherwise, the connection will be kept open until the timeout is reached or the allocation is preempted.
         * @param {string} allocationId The id of the allocation.
         * @param {number} [timeoutSeconds] The timeout in seconds.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationPreemptionSignal(allocationId: string, timeoutSeconds?: number, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationPreemptionSignal.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/signals/preemption`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (timeoutSeconds !== undefined) {
                localVarQueryParameter['timeoutSeconds'] = timeoutSeconds
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Set allocation to ready state.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationReadyRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationReady(allocationId: string, body: V1AllocationReadyRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationReady.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling allocationReady.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/ready`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Gather an allocation's rendezvous info. Blocks until all trial containers connect to gather their rendezvous information and responds to them all at once.
         * @param {string} allocationId The id of the allocation.
         * @param {string} resourcesId The id of the clump of resources.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationRendezvousInfo(allocationId: string, resourcesId: string, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationRendezvousInfo.');
            }
            // verify required parameter 'resourcesId' is not null or undefined
            if (resourcesId === null || resourcesId === undefined) {
                throw new RequiredError('resourcesId','Required parameter resourcesId was null or undefined when calling allocationRendezvousInfo.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/resources/{resourcesId}/rendezvous`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)))
                .replace(`{${"resourcesId"}}`, encodeURIComponent(String(resourcesId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Set allocation to waiting state.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationWaitingRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationWaiting(allocationId: string, body: V1AllocationWaitingRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling allocationWaiting.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling allocationWaiting.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/waiting`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Archive runs.
         * @param {V1ArchiveRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveRuns(body: V1ArchiveRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling archiveRuns.');
            }
            const localVarPath = `/api/v1/runs/archive`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Assign multiple users to multiple groups.
         * @param {V1AssignMultipleGroupsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        assignMultipleGroups(body: V1AssignMultipleGroupsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling assignMultipleGroups.');
            }
            const localVarPath = `/api/v1/users/assignments`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Bind resource pool to workspace
         * @param {string} resourcePoolName The resource pool name.
         * @param {V1BindRPToWorkspaceRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        bindRPToWorkspace(resourcePoolName: string, body: V1BindRPToWorkspaceRequest, options: any = {}): FetchArgs {
            // verify required parameter 'resourcePoolName' is not null or undefined
            if (resourcePoolName === null || resourcePoolName === undefined) {
                throw new RequiredError('resourcePoolName','Required parameter resourcePoolName was null or undefined when calling bindRPToWorkspace.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling bindRPToWorkspace.');
            }
            const localVarPath = `/api/v1/resource-pools/{resourcePoolName}/workspace-bindings`
                .replace(`{${"resourcePoolName"}}`, encodeURIComponent(String(resourcePoolName)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Cleanup task logs according to the retention policy.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cleanupLogs(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/cleanup_logs`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Continues an experiment either to make the existing experiment train longer or to retry it.
         * @param {V1ContinueExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        continueExperiment(body: V1ContinueExperimentRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling continueExperiment.');
            }
            const localVarPath = `/api/v1/experiments/continue`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Create an experiment.
         * @param {V1CreateExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createExperiment(body: V1CreateExperimentRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling createExperiment.');
            }
            const localVarPath = `/api/v1/experiments`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Create an experiment.
         * @param {V1CreateGenericTaskRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createGenericTask(body: V1CreateGenericTaskRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling createGenericTask.');
            }
            const localVarPath = `/api/v1/generic-tasks`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Create a group with optional members on creation.
         * @param {V1CreateGroupRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createGroup(body: V1CreateGroupRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling createGroup.');
            }
            const localVarPath = `/api/v1/groups`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Create unmanaged trial.
         * @param {V1CreateTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createTrial(body: V1CreateTrialRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling createTrial.');
            }
            const localVarPath = `/api/v1/trials`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Remove a group.
         * @param {number} groupId The id of the group that should be deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteGroup(groupId: number, options: any = {}): FetchArgs {
            // verify required parameter 'groupId' is not null or undefined
            if (groupId === null || groupId === undefined) {
                throw new RequiredError('groupId','Required parameter groupId was null or undefined when calling deleteGroup.');
            }
            const localVarPath = `/api/v1/groups/{groupId}`
                .replace(`{${"groupId"}}`, encodeURIComponent(String(groupId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Delete a list of runs.
         * @param {V1DeleteRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteRuns(body: V1DeleteRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling deleteRuns.');
            }
            const localVarPath = `/api/v1/runs/delete`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the set of metric names recorded for a list of experiments.
         * @param {Array<number>} ids The ids for the experiments.
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        expMetricNames(ids: Array<number>, periodSeconds?: number, options: any = {}): FetchArgs {
            // verify required parameter 'ids' is not null or undefined
            if (ids === null || ids === undefined) {
                throw new RequiredError('ids','Required parameter ids was null or undefined when calling expMetricNames.');
            }
            const localVarPath = `/api/v1/experiments/metrics-stream/metric-names`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (ids) {
                localVarQueryParameter['ids'] = ids
            }
            
            if (periodSeconds !== undefined) {
                localVarQueryParameter['periodSeconds'] = periodSeconds
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get details about an Allocation.
         * @param {string} allocationId The id of the allocation.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAllocation(allocationId: string, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling getAllocation.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the best searcher validation for an experiment by the given metric.
         * @param {number} experimentId The ID of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getBestSearcherValidationMetric(experimentId: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling getBestSearcherValidationMetric.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/searcher/best_searcher_validation_metric`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get task config
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGenericTaskConfig(taskId: string, options: any = {}): FetchArgs {
            // verify required parameter 'taskId' is not null or undefined
            if (taskId === null || taskId === undefined) {
                throw new RequiredError('taskId','Required parameter taskId was null or undefined when calling getGenericTaskConfig.');
            }
            const localVarPath = `/api/v1/tasks/{taskId}/config`
                .replace(`{${"taskId"}}`, encodeURIComponent(String(taskId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a group by id.
         * @param {number} groupId The id of the group to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGroup(groupId: number, options: any = {}): FetchArgs {
            // verify required parameter 'groupId' is not null or undefined
            if (groupId === null || groupId === undefined) {
                throw new RequiredError('groupId','Required parameter groupId was null or undefined when calling getGroup.');
            }
            const localVarPath = `/api/v1/groups/{groupId}`
                .replace(`{${"groupId"}}`, encodeURIComponent(String(groupId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Search for groups with optional filters.
         * @param {V1GetGroupsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGroups(body: V1GetGroupsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling getGroups.');
            }
            const localVarPath = `/api/v1/groups/search`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get job queue stats for a resource pool.
         * @param {Array<string>} [resourcePools] Filter the results based on a set of resource pools.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobQueueStats(resourcePools?: Array<string>, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/job-queues/stats`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (resourcePools) {
                localVarQueryParameter['resourcePools'] = resourcePools
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of jobs in queue.
         * @param {number} [offset] Pagination offset.
         * @param {number} [limit] Pagination limit.
         * @param {string} [resourcePool] The target resource-pool for agent resource manager.
         * @param {V1OrderBy} [orderBy] Order results in either ascending or descending order by the number of jobs ahead.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {Array<Jobv1State>} [states] Filter to jobs with states among those given.   - STATE_UNSPECIFIED: Unspecified state.  - STATE_QUEUED: Job is queued and waiting to be schedlued.  - STATE_SCHEDULED: Job is scheduled.  - STATE_SCHEDULED_BACKFILLED: Job is scheduled as a backfill.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobs(offset?: number, limit?: number, resourcePool?: string, orderBy?: V1OrderBy, states?: Array<Jobv1State>, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/job-queues`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (resourcePool !== undefined) {
                localVarQueryParameter['resourcePool'] = resourcePool
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of jobs in queue.
         * @param {number} [offset] Pagination offset.
         * @param {number} [limit] Pagination limit.
         * @param {string} [resourcePool] The target resource-pool for agent resource manager.
         * @param {V1OrderBy} [orderBy] Order results in either ascending or descending order by the number of jobs ahead.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {Array<Jobv1State>} [states] Filter to jobs with states among those given.   - STATE_UNSPECIFIED: Unspecified state.  - STATE_QUEUED: Job is queued and waiting to be schedlued.  - STATE_SCHEDULED: Job is scheduled.  - STATE_SCHEDULED_BACKFILLED: Job is scheduled as a backfill.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobsV2(offset?: number, limit?: number, resourcePool?: string, orderBy?: V1OrderBy, states?: Array<Jobv1State>, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/job-queues-v2`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (resourcePool !== undefined) {
                localVarQueryParameter['resourcePool'] = resourcePool
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (states) {
                localVarQueryParameter['states'] = states
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of all Kubernetes cluster names.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getKubernetesResourceManagers(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/k8s-resource-managers`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Retrieve run metadata values for a given string typed key within a given project.
         * @param {number} projectId Project id
         * @param {string} key Metadata key
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMetadataValues(projectId: number, key: string, options: any = {}): FetchArgs {
            // verify required parameter 'projectId' is not null or undefined
            if (projectId === null || projectId === undefined) {
                throw new RequiredError('projectId','Required parameter projectId was null or undefined when calling getMetadataValues.');
            }
            // verify required parameter 'key' is not null or undefined
            if (key === null || key === undefined) {
                throw new RequiredError('key','Required parameter key was null or undefined when calling getMetadataValues.');
            }
            const localVarPath = `/api/v1/projects/{projectId}/metadata/{key}`
                .replace(`{${"projectId"}}`, encodeURIComponent(String(projectId)))
                .replace(`{${"key"}}`, encodeURIComponent(String(key)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of columns for experiment list table.
         * @param {number} id The id of the project.
         * @param {V1TableType} [tableType] type of table for project columns.   - TABLE_TYPE_UNSPECIFIED: Unspecified table type.  - TABLE_TYPE_EXPERIMENT: experiment table.  - TABLE_TYPE_RUN: run table.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getProjectColumns(id: number, tableType?: V1TableType, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling getProjectColumns.');
            }
            const localVarPath = `/api/v1/projects/{id}/columns`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (tableType !== undefined) {
                localVarQueryParameter['tableType'] = tableType
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get metrics range for a project.
         * @param {number} id The id of the project.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getProjectNumericMetricsRange(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling getProjectNumericMetricsRange.');
            }
            const localVarPath = `/api/v1/projects/{id}/experiments/metric-ranges`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of all resource pools from the cluster.
         * @param {number} [offset] Skip the number of resource pools before returning results. Negative values denote number of resource pools to skip from the end before returning results.
         * @param {number} [limit] Limit the number of resource pools. A value of 0 denotes no limit.
         * @param {boolean} [unbound] Indicate whether or not to return unbound pools only.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getResourcePools(offset?: number, limit?: number, unbound?: boolean, options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/resource-pools`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (unbound !== undefined) {
                localVarQueryParameter['unbound'] = unbound
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get run metadata.
         * @param {number} runId The ID of the run to get metadata for.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getRunMetadata(runId: number, options: any = {}): FetchArgs {
            // verify required parameter 'runId' is not null or undefined
            if (runId === null || runId === undefined) {
                throw new RequiredError('runId','Required parameter runId was null or undefined when calling getRunMetadata.');
            }
            const localVarPath = `/api/v1/runs/{runId}/metadata`
                .replace(`{${"runId"}}`, encodeURIComponent(String(runId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary GetTaskAcceleratorData gets the accelerator data for each allocation associated with a task.
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTaskAcceleratorData(taskId: string, options: any = {}): FetchArgs {
            // verify required parameter 'taskId' is not null or undefined
            if (taskId === null || taskId === undefined) {
                throw new RequiredError('taskId','Required parameter taskId was null or undefined when calling getTaskAcceleratorData.');
            }
            const localVarPath = `/api/v1/tasks/{taskId}/acceleratorData`
                .replace(`{${"taskId"}}`, encodeURIComponent(String(taskId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get telemetry information.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTelemetry(options: any = {}): FetchArgs {
            const localVarPath = `/api/v1/master/telemetry`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a single trial by external id.
         * @param {string} externalExperimentId External experiment id.
         * @param {string} externalTrialId External trial id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialByExternalID(externalExperimentId: string, externalTrialId: string, options: any = {}): FetchArgs {
            // verify required parameter 'externalExperimentId' is not null or undefined
            if (externalExperimentId === null || externalExperimentId === undefined) {
                throw new RequiredError('externalExperimentId','Required parameter externalExperimentId was null or undefined when calling getTrialByExternalID.');
            }
            // verify required parameter 'externalTrialId' is not null or undefined
            if (externalTrialId === null || externalTrialId === undefined) {
                throw new RequiredError('externalTrialId','Required parameter externalTrialId was null or undefined when calling getTrialByExternalID.');
            }
            const localVarPath = `/api/v1/trials/by-external-id/{externalExperimentId}/{externalTrialId}`
                .replace(`{${"externalExperimentId"}}`, encodeURIComponent(String(externalExperimentId)))
                .replace(`{${"externalTrialId"}}`, encodeURIComponent(String(externalTrialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Gets the metrics for all trials associated with this checkpoint
         * @param {string} checkpointUuid UUID of the checkpoint.
         * @param {V1TrialSourceInfoType} [trialSourceInfoType] Type of the TrialSourceInfo.   - TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED: The type is unspecified  - TRIAL_SOURCE_INFO_TYPE_INFERENCE: "Inference" Trial Source Info Type, used for batch inference  - TRIAL_SOURCE_INFO_TYPE_FINE_TUNING: "Fine Tuning" Trial Source Info Type, used in model hub
         * @param {string} [metricGroup] Metric Group string ("training", "validation", or anything else) (nil means all groups).
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialMetricsByCheckpoint(checkpointUuid: string, trialSourceInfoType?: V1TrialSourceInfoType, metricGroup?: string, options: any = {}): FetchArgs {
            // verify required parameter 'checkpointUuid' is not null or undefined
            if (checkpointUuid === null || checkpointUuid === undefined) {
                throw new RequiredError('checkpointUuid','Required parameter checkpointUuid was null or undefined when calling getTrialMetricsByCheckpoint.');
            }
            const localVarPath = `/api/v1/checkpoints/{checkpointUuid}/metrics`
                .replace(`{${"checkpointUuid"}}`, encodeURIComponent(String(checkpointUuid)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (trialSourceInfoType !== undefined) {
                localVarQueryParameter['trialSourceInfoType'] = trialSourceInfoType
            }
            
            if (metricGroup !== undefined) {
                localVarQueryParameter['metricGroup'] = metricGroup
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Gets the metrics for all trials associated with this model version
         * @param {string} modelName The name of the model associated with the model version.
         * @param {number} modelVersionNum Sequential model version number.
         * @param {V1TrialSourceInfoType} [trialSourceInfoType] Type of the TrialSourceInfo.   - TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED: The type is unspecified  - TRIAL_SOURCE_INFO_TYPE_INFERENCE: "Inference" Trial Source Info Type, used for batch inference  - TRIAL_SOURCE_INFO_TYPE_FINE_TUNING: "Fine Tuning" Trial Source Info Type, used in model hub
         * @param {string} [metricGroup] Metric Group string ("training", "validation", or anything else) (nil means all groups).
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialMetricsByModelVersion(modelName: string, modelVersionNum: number, trialSourceInfoType?: V1TrialSourceInfoType, metricGroup?: string, options: any = {}): FetchArgs {
            // verify required parameter 'modelName' is not null or undefined
            if (modelName === null || modelName === undefined) {
                throw new RequiredError('modelName','Required parameter modelName was null or undefined when calling getTrialMetricsByModelVersion.');
            }
            // verify required parameter 'modelVersionNum' is not null or undefined
            if (modelVersionNum === null || modelVersionNum === undefined) {
                throw new RequiredError('modelVersionNum','Required parameter modelVersionNum was null or undefined when calling getTrialMetricsByModelVersion.');
            }
            const localVarPath = `/api/v1/models/{modelName}/versions/{modelVersionNum}/metrics`
                .replace(`{${"modelName"}}`, encodeURIComponent(String(modelName)))
                .replace(`{${"modelVersionNum"}}`, encodeURIComponent(String(modelVersionNum)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (trialSourceInfoType !== undefined) {
                localVarQueryParameter['trialSourceInfoType'] = trialSourceInfoType
            }
            
            if (metricGroup !== undefined) {
                localVarQueryParameter['metricGroup'] = metricGroup
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the list of trials for an experiment.
         * @param {number} id The trial id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialRemainingLogRetentionDays(id: number, options: any = {}): FetchArgs {
            // verify required parameter 'id' is not null or undefined
            if (id === null || id === undefined) {
                throw new RequiredError('id','Required parameter id was null or undefined when calling getTrialRemainingLogRetentionDays.');
            }
            const localVarPath = `/api/v1/trials/{id}/remaining_log_retention_days`
                .replace(`{${"id"}}`, encodeURIComponent(String(id)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the list of workloads for a trial.
         * @param {number} trialId Limit workloads to those that are owned by the specified trial.
         * @param {V1OrderBy} [orderBy] Order workloads in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of workloads before returning results. Negative values denote number of workloads to skip from the end before returning results.
         * @param {number} [limit] Limit the number of workloads. A value of 0 denotes no limit.
         * @param {string} [sortKey] Sort workloads by batches, a training metric, or a validation metric.
         * @param {GetTrialWorkloadsRequestFilterOption} [filter] Filter workloads with validation and/or checkpoint information.   - FILTER_OPTION_UNSPECIFIED: Any workload.  - FILTER_OPTION_CHECKPOINT: Only workloads with an associated checkpoint.  - FILTER_OPTION_VALIDATION: Only validation workloads.  - FILTER_OPTION_CHECKPOINT_OR_VALIDATION: Only validation workloads or ones with an associated checkpoint.
         * @param {boolean} [includeBatchMetrics] Include per-batch metrics.
         * @param {V1MetricType} [metricType] When sorting workloads by sort_key, specify training or validation form of a metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {boolean} [removeDeletedCheckpoints] Remove deleted checkpoints.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialWorkloads(trialId: number, orderBy?: V1OrderBy, offset?: number, limit?: number, sortKey?: string, filter?: GetTrialWorkloadsRequestFilterOption, includeBatchMetrics?: boolean, metricType?: V1MetricType, group?: string, removeDeletedCheckpoints?: boolean, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling getTrialWorkloads.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/workloads`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (orderBy !== undefined) {
                localVarQueryParameter['orderBy'] = orderBy
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            if (sortKey !== undefined) {
                localVarQueryParameter['sortKey'] = sortKey
            }
            
            if (filter !== undefined) {
                localVarQueryParameter['filter'] = filter
            }
            
            if (includeBatchMetrics !== undefined) {
                localVarQueryParameter['includeBatchMetrics'] = includeBatchMetrics
            }
            
            if (metricType !== undefined) {
                localVarQueryParameter['metricType'] = metricType
            }
            
            if (group !== undefined) {
                localVarQueryParameter['group'] = group
            }
            
            if (removeDeletedCheckpoints !== undefined) {
                localVarQueryParameter['removeDeletedCheckpoints'] = removeDeletedCheckpoints
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Send notebook idle data to master
         * @param {string} notebookId The id of the notebook.
         * @param {V1IdleNotebookRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        idleNotebook(notebookId: string, body: V1IdleNotebookRequest, options: any = {}): FetchArgs {
            // verify required parameter 'notebookId' is not null or undefined
            if (notebookId === null || notebookId === undefined) {
                throw new RequiredError('notebookId','Required parameter notebookId was null or undefined when calling idleNotebook.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling idleNotebook.');
            }
            const localVarPath = `/api/v1/notebooks/{notebookId}/report_idle`
                .replace(`{${"notebookId"}}`, encodeURIComponent(String(notebookId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill generic task
         * @param {string} taskId The id of the task.
         * @param {V1KillGenericTaskRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killGenericTask(taskId: string, body: V1KillGenericTaskRequest, options: any = {}): FetchArgs {
            // verify required parameter 'taskId' is not null or undefined
            if (taskId === null || taskId === undefined) {
                throw new RequiredError('taskId','Required parameter taskId was null or undefined when calling killGenericTask.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling killGenericTask.');
            }
            const localVarPath = `/api/v1/tasks/{taskId}/kill`
                .replace(`{${"taskId"}}`, encodeURIComponent(String(taskId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Kill runs.
         * @param {V1KillRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killRuns(body: V1KillRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling killRuns.');
            }
            const localVarPath = `/api/v1/runs/kill`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary List all resource pools, bound and unbound, available to a specific workspace
         * @param {number} workspaceId Workspace ID.
         * @param {number} [offset] The offset to use with pagination.
         * @param {number} [limit] The maximum number of results to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        listRPsBoundToWorkspace(workspaceId: number, offset?: number, limit?: number, options: any = {}): FetchArgs {
            // verify required parameter 'workspaceId' is not null or undefined
            if (workspaceId === null || workspaceId === undefined) {
                throw new RequiredError('workspaceId','Required parameter workspaceId was null or undefined when calling listRPsBoundToWorkspace.');
            }
            const localVarPath = `/api/v1/workspaces/{workspaceId}/available-resource-pools`
                .replace(`{${"workspaceId"}}`, encodeURIComponent(String(workspaceId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary List all workspaces bound to a specific resource pool
         * @param {string} resourcePoolName Resource pool name.
         * @param {number} [offset] The offset to use with pagination.
         * @param {number} [limit] The maximum number of results to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        listWorkspacesBoundToRP(resourcePoolName: string, offset?: number, limit?: number, options: any = {}): FetchArgs {
            // verify required parameter 'resourcePoolName' is not null or undefined
            if (resourcePoolName === null || resourcePoolName === undefined) {
                throw new RequiredError('resourcePoolName','Required parameter resourcePoolName was null or undefined when calling listWorkspacesBoundToRP.');
            }
            const localVarPath = `/api/v1/resource-pools/{resourcePoolName}/workspace-bindings`
                .replace(`{${"resourcePoolName"}}`, encodeURIComponent(String(resourcePoolName)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (offset !== undefined) {
                localVarQueryParameter['offset'] = offset
            }
            
            if (limit !== undefined) {
                localVarQueryParameter['limit'] = limit
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Mark the given reservation (container, pod, etc) within an allocation as a daemon reservation. In the exit of a successful exit, Determined will wait for all resources to exit - unless they are marked as daemon resources, in which case Determined will clean them up regardless of exit status after all non-daemon resources have exited.
         * @param {string} allocationId The id of the allocation.
         * @param {string} resourcesId The id of the clump of resources to mark as daemon.
         * @param {V1MarkAllocationResourcesDaemonRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        markAllocationResourcesDaemon(allocationId: string, resourcesId: string, body: V1MarkAllocationResourcesDaemonRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling markAllocationResourcesDaemon.');
            }
            // verify required parameter 'resourcesId' is not null or undefined
            if (resourcesId === null || resourcesId === undefined) {
                throw new RequiredError('resourcesId','Required parameter resourcesId was null or undefined when calling markAllocationResourcesDaemon.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling markAllocationResourcesDaemon.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/resources/{resourcesId}/daemon`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)))
                .replace(`{${"resourcesId"}}`, encodeURIComponent(String(resourcesId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get the milestones (in batches processed) at which a metric is recorded by an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {string} metricName A metric name.
         * @param {V1MetricType} [metricType] The type of metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        metricBatches(experimentId: number, metricName: string, metricType?: V1MetricType, group?: string, periodSeconds?: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling metricBatches.');
            }
            // verify required parameter 'metricName' is not null or undefined
            if (metricName === null || metricName === undefined) {
                throw new RequiredError('metricName','Required parameter metricName was null or undefined when calling metricBatches.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/metrics-stream/batches`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (metricName !== undefined) {
                localVarQueryParameter['metricName'] = metricName
            }
            
            if (metricType !== undefined) {
                localVarQueryParameter['metricType'] = metricType
            }
            
            if (group !== undefined) {
                localVarQueryParameter['group'] = group
            }
            
            if (periodSeconds !== undefined) {
                localVarQueryParameter['periodSeconds'] = periodSeconds
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Move runs.
         * @param {V1MoveRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveRuns(body: V1MoveRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling moveRuns.');
            }
            const localVarPath = `/api/v1/runs/move`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary NotifyContainterRunning is used to notify the master that the container is running.  On HPC, the launcher will report a state of "Running" as soon as Slurm starts the job, but the container may be in the process of getting pulled down from the Internet, so the experiment is not really considered to be in a "Running" state until all the containers that are part of the experiment are running and not being pulled.
         * @param {string} allocationId The ID of the allocation.
         * @param {V1NotifyContainerRunningRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        notifyContainerRunning(allocationId: string, body: V1NotifyContainerRunningRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling notifyContainerRunning.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling notifyContainerRunning.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/notify_container_running`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Overwrite resource pool - workspace bindings
         * @param {string} resourcePoolName The resource pool name.
         * @param {V1OverwriteRPWorkspaceBindingsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        overwriteRPWorkspaceBindings(resourcePoolName: string, body: V1OverwriteRPWorkspaceBindingsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'resourcePoolName' is not null or undefined
            if (resourcePoolName === null || resourcePoolName === undefined) {
                throw new RequiredError('resourcePoolName','Required parameter resourcePoolName was null or undefined when calling overwriteRPWorkspaceBindings.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling overwriteRPWorkspaceBindings.');
            }
            const localVarPath = `/api/v1/resource-pools/{resourcePoolName}/workspace-bindings`
                .replace(`{${"resourcePoolName"}}`, encodeURIComponent(String(resourcePoolName)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Update checkpoints. Won't modify checkpoint files.
         * @param {V1PatchCheckpointsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchCheckpoints(body: V1PatchCheckpointsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling patchCheckpoints.');
            }
            const localVarPath = `/api/v1/checkpoints`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Patch (an unmanaged) trial.
         * @param {number} trialId Trial id.
         * @param {V1PatchTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchTrial(trialId: number, body: V1PatchTrialRequest, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling patchTrial.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling patchTrial.');
            }
            const localVarPath = `/api/v1/trials/{trialId}`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Patch multiple users' activation status.
         * @param {V1PatchUsersRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchUsers(body: V1PatchUsersRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling patchUsers.');
            }
            const localVarPath = `/api/v1/users`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PATCH', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Pause generic task
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseGenericTask(taskId: string, options: any = {}): FetchArgs {
            // verify required parameter 'taskId' is not null or undefined
            if (taskId === null || taskId === undefined) {
                throw new RequiredError('taskId','Required parameter taskId was null or undefined when calling pauseGenericTask.');
            }
            const localVarPath = `/api/v1/tasks/{taskId}/pause`
                .replace(`{${"taskId"}}`, encodeURIComponent(String(taskId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Pause experiment associated with provided runs.
         * @param {V1PauseRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseRuns(body: V1PauseRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling pauseRuns.');
            }
            const localVarPath = `/api/v1/runs/pause`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary PostAllocationAcceleratorData sets the accelerator for a given allocation.
         * @param {string} allocationId The id of the allocation.
         * @param {V1PostAllocationAcceleratorDataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postAllocationAcceleratorData(allocationId: string, body: V1PostAllocationAcceleratorDataRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling postAllocationAcceleratorData.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postAllocationAcceleratorData.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/acceleratorData`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary PostAllocationProxyAddress sets the proxy address to use when proxying to services provided by an allocation. Upon receipt, the master will also register any proxies specified by the task.
         * @param {string} allocationId The id of the allocation.
         * @param {V1PostAllocationProxyAddressRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postAllocationProxyAddress(allocationId: string, body: V1PostAllocationProxyAddressRequest, options: any = {}): FetchArgs {
            // verify required parameter 'allocationId' is not null or undefined
            if (allocationId === null || allocationId === undefined) {
                throw new RequiredError('allocationId','Required parameter allocationId was null or undefined when calling postAllocationProxyAddress.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postAllocationProxyAddress.');
            }
            const localVarPath = `/api/v1/allocations/{allocationId}/proxy_address`
                .replace(`{${"allocationId"}}`, encodeURIComponent(String(allocationId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Update run metadata.
         * @param {number} runId The ID of the run to post metadata for.
         * @param {V1PostRunMetadataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postRunMetadata(runId: number, body: V1PostRunMetadataRequest, options: any = {}): FetchArgs {
            // verify required parameter 'runId' is not null or undefined
            if (runId === null || runId === undefined) {
                throw new RequiredError('runId','Required parameter runId was null or undefined when calling postRunMetadata.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postRunMetadata.');
            }
            const localVarPath = `/api/v1/runs/{runId}/metadata`
                .replace(`{${"runId"}}`, encodeURIComponent(String(runId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Persist the given task logs.
         * @param {V1PostTaskLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTaskLogs(body: V1PostTaskLogsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postTaskLogs.');
            }
            const localVarPath = `/api/v1/task/logs`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Persist the given TrialProfilerMetricsBatch. The trial ID is in the labels.
         * @param {V1PostTrialProfilerMetricsBatchRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTrialProfilerMetricsBatch(body: V1PostTrialProfilerMetricsBatchRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postTrialProfilerMetricsBatch.');
            }
            const localVarPath = `/api/v1/trials/profiler/metrics`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary For bookkeeping, update trial runner metadata (currently just state).
         * @param {number} trialId The id of the trial.
         * @param {V1TrialRunnerMetadata} body The state for the trial runner.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTrialRunnerMetadata(trialId: number, body: V1TrialRunnerMetadata, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling postTrialRunnerMetadata.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling postTrialRunnerMetadata.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/runner/metadata`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Put an experiment by external id.
         * @param {string} externalExperimentId External experiment id.
         * @param {V1CreateExperimentRequest} body CreateExperimentRequest payload.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperiment(externalExperimentId: string, body: V1CreateExperimentRequest, options: any = {}): FetchArgs {
            // verify required parameter 'externalExperimentId' is not null or undefined
            if (externalExperimentId === null || externalExperimentId === undefined) {
                throw new RequiredError('externalExperimentId','Required parameter externalExperimentId was null or undefined when calling putExperiment.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putExperiment.');
            }
            const localVarPath = `/api/v1/experiments/by-external-id/{externalExperimentId}`
                .replace(`{${"externalExperimentId"}}`, encodeURIComponent(String(externalExperimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Put a trial.
         * @param {V1PutTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putTrial(body: V1PutTrialRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling putTrial.');
            }
            const localVarPath = `/api/v1/trials`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Record a checkpoint.
         * @param {V1Checkpoint} body The training metrics to persist.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportCheckpoint(body: V1Checkpoint, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportCheckpoint.');
            }
            const localVarPath = `/api/v1/checkpoints`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Record metrics for specified trial.
         * @param {number} metricsTrialId The trial associated with these metrics.
         * @param {V1ReportTrialMetricsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialMetrics(metricsTrialId: number, body: V1ReportTrialMetricsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'metricsTrialId' is not null or undefined
            if (metricsTrialId === null || metricsTrialId === undefined) {
                throw new RequiredError('metricsTrialId','Required parameter metricsTrialId was null or undefined when calling reportTrialMetrics.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialMetrics.');
            }
            const localVarPath = `/api/v1/trials/{metricsTrialId}/metrics`
                .replace(`{${"metricsTrialId"}}`, encodeURIComponent(String(metricsTrialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary For bookkeeping, updates the progress towards to current requested searcher training length.
         * @param {number} trialId The id of the trial.
         * @param {V1ReportTrialProgressRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialProgress(trialId: number, body: V1ReportTrialProgressRequest, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling reportTrialProgress.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialProgress.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/progress`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Reports to the searcher that the trial has completed the current requested amount of training with the given searcher validation metric.
         * @param {number} trialId The id of the trial.
         * @param {V1TrialEarlyExit} body The exit reason.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialSearcherEarlyExit(trialId: number, body: V1TrialEarlyExit, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling reportTrialSearcherEarlyExit.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialSearcherEarlyExit.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/early_exit`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Reports a TrialSourceInfo entry for tracking inference or fine-tuning runs
         * @param {V1ReportTrialSourceInfoRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialSourceInfo(body: V1ReportTrialSourceInfoRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialSourceInfo.');
            }
            const localVarPath = `/api/v1/trial-source-info`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Record training metrics for specified training.
         * @param {number} trainingMetricsTrialId The trial associated with these metrics.
         * @param {V1TrialMetrics} body The training metrics to persist.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialTrainingMetrics(trainingMetricsTrialId: number, body: V1TrialMetrics, options: any = {}): FetchArgs {
            // verify required parameter 'trainingMetricsTrialId' is not null or undefined
            if (trainingMetricsTrialId === null || trainingMetricsTrialId === undefined) {
                throw new RequiredError('trainingMetricsTrialId','Required parameter trainingMetricsTrialId was null or undefined when calling reportTrialTrainingMetrics.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialTrainingMetrics.');
            }
            const localVarPath = `/api/v1/trials/{trainingMetricsTrialId}/training_metrics`
                .replace(`{${"trainingMetricsTrialId"}}`, encodeURIComponent(String(trainingMetricsTrialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Record validation metrics.
         * @param {number} validationMetricsTrialId The trial associated with these metrics.
         * @param {V1TrialMetrics} body The training metrics to persist.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialValidationMetrics(validationMetricsTrialId: number, body: V1TrialMetrics, options: any = {}): FetchArgs {
            // verify required parameter 'validationMetricsTrialId' is not null or undefined
            if (validationMetricsTrialId === null || validationMetricsTrialId === undefined) {
                throw new RequiredError('validationMetricsTrialId','Required parameter validationMetricsTrialId was null or undefined when calling reportTrialValidationMetrics.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling reportTrialValidationMetrics.');
            }
            const localVarPath = `/api/v1/trials/{validationMetricsTrialId}/validation_metrics`
                .replace(`{${"validationMetricsTrialId"}}`, encodeURIComponent(String(validationMetricsTrialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unpause experiment associated with provided runs.
         * @param {V1ResumeRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        resumeRuns(body: V1ResumeRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling resumeRuns.');
            }
            const localVarPath = `/api/v1/runs/resume`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Start syncing and prepare to be able to report to a run. This should be called once per task that will report to the run.
         * @param {V1RunPrepareForReportingRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        runPrepareForReporting(body: V1RunPrepareForReportingRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling runPrepareForReporting.');
            }
            const localVarPath = `/api/v1/runs/start`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get experiments with grouping and search syntax
         * @param {V1SearchExperimentsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        searchExperiments(body: V1SearchExperimentsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling searchExperiments.');
            }
            const localVarPath = `/api/v1/experiments-search`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a list of runs.
         * @param {V1SearchRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        searchRuns(body: V1SearchRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling searchRuns.');
            }
            const localVarPath = `/api/v1/runs`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Start (an unmanaged) trial.
         * @param {number} trialId Trial id.
         * @param {V1StartTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        startTrial(trialId: number, body: V1StartTrialRequest, options: any = {}): FetchArgs {
            // verify required parameter 'trialId' is not null or undefined
            if (trialId === null || trialId === undefined) {
                throw new RequiredError('trialId','Required parameter trialId was null or undefined when calling startTrial.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling startTrial.');
            }
            const localVarPath = `/api/v1/trials/{trialId}/start`
                .replace(`{${"trialId"}}`, encodeURIComponent(String(trialId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a sample of the metrics over time for a sample of the trials.
         * @param {number} experimentId The id of the experiment.
         * @param {string} metricName A metric name.
         * @param {V1MetricType} [metricType] The type of metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {number} [maxTrials] Maximum number of trials to fetch data for.
         * @param {number} [maxDatapoints] Maximum number of initial / historical data points.
         * @param {number} [startBatches] Beginning of window (inclusive) to fetch data for.
         * @param {number} [endBatches] Ending of window (inclusive) to fetch data for.
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialsSample(experimentId: number, metricName: string, metricType?: V1MetricType, group?: string, maxTrials?: number, maxDatapoints?: number, startBatches?: number, endBatches?: number, periodSeconds?: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling trialsSample.');
            }
            // verify required parameter 'metricName' is not null or undefined
            if (metricName === null || metricName === undefined) {
                throw new RequiredError('metricName','Required parameter metricName was null or undefined when calling trialsSample.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/metrics-stream/trials-sample`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (metricName !== undefined) {
                localVarQueryParameter['metricName'] = metricName
            }
            
            if (metricType !== undefined) {
                localVarQueryParameter['metricType'] = metricType
            }
            
            if (group !== undefined) {
                localVarQueryParameter['group'] = group
            }
            
            if (maxTrials !== undefined) {
                localVarQueryParameter['maxTrials'] = maxTrials
            }
            
            if (maxDatapoints !== undefined) {
                localVarQueryParameter['maxDatapoints'] = maxDatapoints
            }
            
            if (startBatches !== undefined) {
                localVarQueryParameter['startBatches'] = startBatches
            }
            
            if (endBatches !== undefined) {
                localVarQueryParameter['endBatches'] = endBatches
            }
            
            if (periodSeconds !== undefined) {
                localVarQueryParameter['periodSeconds'] = periodSeconds
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Get a snapshot of a metric across all trials at a certain point of progress.
         * @param {number} experimentId The id of the experiment.
         * @param {string} metricName A metric name.
         * @param {number} batchesProcessed The point of progress at which to query metrics.
         * @param {V1MetricType} [metricType] The type of metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {number} [batchesMargin] A range either side of batches_processed to include near-misses.
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        trialsSnapshot(experimentId: number, metricName: string, batchesProcessed: number, metricType?: V1MetricType, group?: string, batchesMargin?: number, periodSeconds?: number, options: any = {}): FetchArgs {
            // verify required parameter 'experimentId' is not null or undefined
            if (experimentId === null || experimentId === undefined) {
                throw new RequiredError('experimentId','Required parameter experimentId was null or undefined when calling trialsSnapshot.');
            }
            // verify required parameter 'metricName' is not null or undefined
            if (metricName === null || metricName === undefined) {
                throw new RequiredError('metricName','Required parameter metricName was null or undefined when calling trialsSnapshot.');
            }
            // verify required parameter 'batchesProcessed' is not null or undefined
            if (batchesProcessed === null || batchesProcessed === undefined) {
                throw new RequiredError('batchesProcessed','Required parameter batchesProcessed was null or undefined when calling trialsSnapshot.');
            }
            const localVarPath = `/api/v1/experiments/{experimentId}/metrics-stream/trials-snapshot`
                .replace(`{${"experimentId"}}`, encodeURIComponent(String(experimentId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'GET', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            if (metricName !== undefined) {
                localVarQueryParameter['metricName'] = metricName
            }
            
            if (metricType !== undefined) {
                localVarQueryParameter['metricType'] = metricType
            }
            
            if (group !== undefined) {
                localVarQueryParameter['group'] = group
            }
            
            if (batchesProcessed !== undefined) {
                localVarQueryParameter['batchesProcessed'] = batchesProcessed
            }
            
            if (batchesMargin !== undefined) {
                localVarQueryParameter['batchesMargin'] = batchesMargin
            }
            
            if (periodSeconds !== undefined) {
                localVarQueryParameter['periodSeconds'] = periodSeconds
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unarchive runs.
         * @param {V1UnarchiveRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unarchiveRuns(body: V1UnarchiveRunsRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling unarchiveRuns.');
            }
            const localVarPath = `/api/v1/runs/unarchive`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unbind resource pool to workspace
         * @param {string} resourcePoolName The resource pool name.
         * @param {V1UnbindRPFromWorkspaceRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unbindRPFromWorkspace(resourcePoolName: string, body: V1UnbindRPFromWorkspaceRequest, options: any = {}): FetchArgs {
            // verify required parameter 'resourcePoolName' is not null or undefined
            if (resourcePoolName === null || resourcePoolName === undefined) {
                throw new RequiredError('resourcePoolName','Required parameter resourcePoolName was null or undefined when calling unbindRPFromWorkspace.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling unbindRPFromWorkspace.');
            }
            const localVarPath = `/api/v1/resource-pools/{resourcePoolName}/workspace-bindings`
                .replace(`{${"resourcePoolName"}}`, encodeURIComponent(String(resourcePoolName)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'DELETE', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Unpause generic task
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        unpauseGenericTask(taskId: string, options: any = {}): FetchArgs {
            // verify required parameter 'taskId' is not null or undefined
            if (taskId === null || taskId === undefined) {
                throw new RequiredError('taskId','Required parameter taskId was null or undefined when calling unpauseGenericTask.');
            }
            const localVarPath = `/api/v1/tasks/{taskId}/unpause`
                .replace(`{${"taskId"}}`, encodeURIComponent(String(taskId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Update group info.
         * @param {number} groupId The id of the group
         * @param {V1UpdateGroupRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        updateGroup(groupId: number, body: V1UpdateGroupRequest, options: any = {}): FetchArgs {
            // verify required parameter 'groupId' is not null or undefined
            if (groupId === null || groupId === undefined) {
                throw new RequiredError('groupId','Required parameter groupId was null or undefined when calling updateGroup.');
            }
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling updateGroup.');
            }
            const localVarPath = `/api/v1/groups/{groupId}`
                .replace(`{${"groupId"}}`, encodeURIComponent(String(groupId)));
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'PUT', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
        /**
         * 
         * @summary Control the job queues.
         * @param {V1UpdateJobQueueRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        updateJobQueue(body: V1UpdateJobQueueRequest, options: any = {}): FetchArgs {
            // verify required parameter 'body' is not null or undefined
            if (body === null || body === undefined) {
                throw new RequiredError('body','Required parameter body was null or undefined when calling updateJobQueue.');
            }
            const localVarPath = `/api/v1/job-queues`;
            const localVarUrlObj = new URL(localVarPath, BASE_PATH);
            const localVarRequestOptions = { method: 'POST', ...options };
            const localVarHeaderParameter = {} as any;
            const localVarQueryParameter = {} as any;
            
            // authentication BearerToken required
            if (configuration && configuration.apiKey) {
                const localVarApiKeyValue = typeof configuration.apiKey === 'function'
                    ? configuration.apiKey("Authorization")
                    : configuration.apiKey;
                localVarHeaderParameter["Authorization"] = localVarApiKeyValue;
            }
            
            localVarHeaderParameter['Content-Type'] = 'application/json';
            
            objToSearchParams(localVarQueryParameter, localVarUrlObj.searchParams);
            objToSearchParams(options.query || {}, localVarUrlObj.searchParams);
            localVarRequestOptions.headers = { ...localVarHeaderParameter, ...options.headers };
            localVarRequestOptions.body = JSON.stringify(body)
            
            return {
                url: `${localVarUrlObj.pathname}${localVarUrlObj.search}`,
                options: localVarRequestOptions,
            };
        },
    }
};

/**
 * InternalApi - functional programming interface
 * @export
 */
export const InternalApiFp = function (configuration?: Configuration) {
    return {
        /**
         * 
         * @summary Acknowledge the receipt of a signal to stop the given allocation early. This is used indicate and exit 0 isn't final; specifically, it is used for HP search directed early stops and preemption signals (not necessarily just scheduler preemption).
         * @param {string} allocationId The allocation that is acknowledging the request.
         * @param {V1AckAllocationPreemptionSignalRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        ackAllocationPreemptionSignal(allocationId: string, body: V1AckAllocationPreemptionSignalRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AckAllocationPreemptionSignalResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).ackAllocationPreemptionSignal(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary AllocationAllGather performs an all gather through the master. An allocation can only perform once all gather at a time.
         * @param {string} allocationId The ID of the allocation.
         * @param {V1AllocationAllGatherRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationAllGather(allocationId: string, body: V1AllocationAllGatherRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationAllGatherResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationAllGather(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Report the receipt of a signal to stop the given allocation early. This is used to communicate back from a SLURM job that it has been notified of a pending preememption. Upon a call to this API the RM should then trigger a checkpoint and immediate exit.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationPendingPreemptionSignalRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationPendingPreemptionSignal(allocationId: string, body: V1AllocationPendingPreemptionSignalRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationPendingPreemptionSignalResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationPendingPreemptionSignal(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Long poll preemption signals for the given allocation. If the allocation has been preempted when called, it will return so immediately. Otherwise, the connection will be kept open until the timeout is reached or the allocation is preempted.
         * @param {string} allocationId The id of the allocation.
         * @param {number} [timeoutSeconds] The timeout in seconds.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationPreemptionSignal(allocationId: string, timeoutSeconds?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationPreemptionSignalResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationPreemptionSignal(allocationId, timeoutSeconds, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Set allocation to ready state.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationReadyRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationReady(allocationId: string, body: V1AllocationReadyRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationReadyResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationReady(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Gather an allocation's rendezvous info. Blocks until all trial containers connect to gather their rendezvous information and responds to them all at once.
         * @param {string} allocationId The id of the allocation.
         * @param {string} resourcesId The id of the clump of resources.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationRendezvousInfo(allocationId: string, resourcesId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationRendezvousInfoResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationRendezvousInfo(allocationId, resourcesId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Set allocation to waiting state.
         * @param {string} allocationId The id of the allocation.
         * @param {V1AllocationWaitingRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        allocationWaiting(allocationId: string, body: V1AllocationWaitingRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AllocationWaitingResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).allocationWaiting(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Archive runs.
         * @param {V1ArchiveRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        archiveRuns(body: V1ArchiveRunsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ArchiveRunsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).archiveRuns(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Assign multiple users to multiple groups.
         * @param {V1AssignMultipleGroupsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        assignMultipleGroups(body: V1AssignMultipleGroupsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1AssignMultipleGroupsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).assignMultipleGroups(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Bind resource pool to workspace
         * @param {string} resourcePoolName The resource pool name.
         * @param {V1BindRPToWorkspaceRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        bindRPToWorkspace(resourcePoolName: string, body: V1BindRPToWorkspaceRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1BindRPToWorkspaceResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).bindRPToWorkspace(resourcePoolName, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Cleanup task logs according to the retention policy.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        cleanupLogs(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CleanupLogsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).cleanupLogs(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Continues an experiment either to make the existing experiment train longer or to retry it.
         * @param {V1ContinueExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        continueExperiment(body: V1ContinueExperimentRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ContinueExperimentResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).continueExperiment(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Create an experiment.
         * @param {V1CreateExperimentRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createExperiment(body: V1CreateExperimentRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CreateExperimentResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).createExperiment(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Create an experiment.
         * @param {V1CreateGenericTaskRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createGenericTask(body: V1CreateGenericTaskRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CreateGenericTaskResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).createGenericTask(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Create a group with optional members on creation.
         * @param {V1CreateGroupRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createGroup(body: V1CreateGroupRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CreateGroupResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).createGroup(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Create unmanaged trial.
         * @param {V1CreateTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        createTrial(body: V1CreateTrialRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1CreateTrialResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).createTrial(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Remove a group.
         * @param {number} groupId The id of the group that should be deleted.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteGroup(groupId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteGroupResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).deleteGroup(groupId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Delete a list of runs.
         * @param {V1DeleteRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        deleteRuns(body: V1DeleteRunsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1DeleteRunsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).deleteRuns(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the set of metric names recorded for a list of experiments.
         * @param {Array<number>} ids The ids for the experiments.
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        expMetricNames(ids: Array<number>, periodSeconds?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<StreamResultOfV1ExpMetricNamesResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).expMetricNames(ids, periodSeconds, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get details about an Allocation.
         * @param {string} allocationId The id of the allocation.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getAllocation(allocationId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetAllocationResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getAllocation(allocationId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the best searcher validation for an experiment by the given metric.
         * @param {number} experimentId The ID of the experiment.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getBestSearcherValidationMetric(experimentId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetBestSearcherValidationMetricResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getBestSearcherValidationMetric(experimentId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get task config
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGenericTaskConfig(taskId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetGenericTaskConfigResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getGenericTaskConfig(taskId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a group by id.
         * @param {number} groupId The id of the group to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGroup(groupId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetGroupResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getGroup(groupId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Search for groups with optional filters.
         * @param {V1GetGroupsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getGroups(body: V1GetGroupsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetGroupsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getGroups(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get job queue stats for a resource pool.
         * @param {Array<string>} [resourcePools] Filter the results based on a set of resource pools.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobQueueStats(resourcePools?: Array<string>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetJobQueueStatsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getJobQueueStats(resourcePools, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of jobs in queue.
         * @param {number} [offset] Pagination offset.
         * @param {number} [limit] Pagination limit.
         * @param {string} [resourcePool] The target resource-pool for agent resource manager.
         * @param {V1OrderBy} [orderBy] Order results in either ascending or descending order by the number of jobs ahead.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {Array<Jobv1State>} [states] Filter to jobs with states among those given.   - STATE_UNSPECIFIED: Unspecified state.  - STATE_QUEUED: Job is queued and waiting to be schedlued.  - STATE_SCHEDULED: Job is scheduled.  - STATE_SCHEDULED_BACKFILLED: Job is scheduled as a backfill.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobs(offset?: number, limit?: number, resourcePool?: string, orderBy?: V1OrderBy, states?: Array<Jobv1State>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetJobsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getJobs(offset, limit, resourcePool, orderBy, states, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of jobs in queue.
         * @param {number} [offset] Pagination offset.
         * @param {number} [limit] Pagination limit.
         * @param {string} [resourcePool] The target resource-pool for agent resource manager.
         * @param {V1OrderBy} [orderBy] Order results in either ascending or descending order by the number of jobs ahead.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {Array<Jobv1State>} [states] Filter to jobs with states among those given.   - STATE_UNSPECIFIED: Unspecified state.  - STATE_QUEUED: Job is queued and waiting to be schedlued.  - STATE_SCHEDULED: Job is scheduled.  - STATE_SCHEDULED_BACKFILLED: Job is scheduled as a backfill.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getJobsV2(offset?: number, limit?: number, resourcePool?: string, orderBy?: V1OrderBy, states?: Array<Jobv1State>, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetJobsV2Response> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getJobsV2(offset, limit, resourcePool, orderBy, states, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of all Kubernetes cluster names.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getKubernetesResourceManagers(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetKubernetesResourceManagersResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getKubernetesResourceManagers(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Retrieve run metadata values for a given string typed key within a given project.
         * @param {number} projectId Project id
         * @param {string} key Metadata key
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getMetadataValues(projectId: number, key: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetMetadataValuesResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getMetadataValues(projectId, key, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of columns for experiment list table.
         * @param {number} id The id of the project.
         * @param {V1TableType} [tableType] type of table for project columns.   - TABLE_TYPE_UNSPECIFIED: Unspecified table type.  - TABLE_TYPE_EXPERIMENT: experiment table.  - TABLE_TYPE_RUN: run table.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getProjectColumns(id: number, tableType?: V1TableType, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetProjectColumnsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getProjectColumns(id, tableType, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get metrics range for a project.
         * @param {number} id The id of the project.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getProjectNumericMetricsRange(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetProjectNumericMetricsRangeResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getProjectNumericMetricsRange(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a list of all resource pools from the cluster.
         * @param {number} [offset] Skip the number of resource pools before returning results. Negative values denote number of resource pools to skip from the end before returning results.
         * @param {number} [limit] Limit the number of resource pools. A value of 0 denotes no limit.
         * @param {boolean} [unbound] Indicate whether or not to return unbound pools only.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getResourcePools(offset?: number, limit?: number, unbound?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetResourcePoolsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getResourcePools(offset, limit, unbound, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get run metadata.
         * @param {number} runId The ID of the run to get metadata for.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getRunMetadata(runId: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetRunMetadataResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getRunMetadata(runId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary GetTaskAcceleratorData gets the accelerator data for each allocation associated with a task.
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTaskAcceleratorData(taskId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTaskAcceleratorDataResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTaskAcceleratorData(taskId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get telemetry information.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTelemetry(options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTelemetryResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTelemetry(options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get a single trial by external id.
         * @param {string} externalExperimentId External experiment id.
         * @param {string} externalTrialId External trial id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialByExternalID(externalExperimentId: string, externalTrialId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialByExternalIDResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTrialByExternalID(externalExperimentId, externalTrialId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Gets the metrics for all trials associated with this checkpoint
         * @param {string} checkpointUuid UUID of the checkpoint.
         * @param {V1TrialSourceInfoType} [trialSourceInfoType] Type of the TrialSourceInfo.   - TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED: The type is unspecified  - TRIAL_SOURCE_INFO_TYPE_INFERENCE: "Inference" Trial Source Info Type, used for batch inference  - TRIAL_SOURCE_INFO_TYPE_FINE_TUNING: "Fine Tuning" Trial Source Info Type, used in model hub
         * @param {string} [metricGroup] Metric Group string ("training", "validation", or anything else) (nil means all groups).
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialMetricsByCheckpoint(checkpointUuid: string, trialSourceInfoType?: V1TrialSourceInfoType, metricGroup?: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialMetricsByCheckpointResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTrialMetricsByCheckpoint(checkpointUuid, trialSourceInfoType, metricGroup, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Gets the metrics for all trials associated with this model version
         * @param {string} modelName The name of the model associated with the model version.
         * @param {number} modelVersionNum Sequential model version number.
         * @param {V1TrialSourceInfoType} [trialSourceInfoType] Type of the TrialSourceInfo.   - TRIAL_SOURCE_INFO_TYPE_UNSPECIFIED: The type is unspecified  - TRIAL_SOURCE_INFO_TYPE_INFERENCE: "Inference" Trial Source Info Type, used for batch inference  - TRIAL_SOURCE_INFO_TYPE_FINE_TUNING: "Fine Tuning" Trial Source Info Type, used in model hub
         * @param {string} [metricGroup] Metric Group string ("training", "validation", or anything else) (nil means all groups).
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialMetricsByModelVersion(modelName: string, modelVersionNum: number, trialSourceInfoType?: V1TrialSourceInfoType, metricGroup?: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialMetricsByModelVersionResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTrialMetricsByModelVersion(modelName, modelVersionNum, trialSourceInfoType, metricGroup, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the list of trials for an experiment.
         * @param {number} id The trial id.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialRemainingLogRetentionDays(id: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialRemainingLogRetentionDaysResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTrialRemainingLogRetentionDays(id, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the list of workloads for a trial.
         * @param {number} trialId Limit workloads to those that are owned by the specified trial.
         * @param {V1OrderBy} [orderBy] Order workloads in either ascending or descending order.   - ORDER_BY_UNSPECIFIED: Returns records in no specific order.  - ORDER_BY_ASC: Returns records in ascending order.  - ORDER_BY_DESC: Returns records in descending order.
         * @param {number} [offset] Skip the number of workloads before returning results. Negative values denote number of workloads to skip from the end before returning results.
         * @param {number} [limit] Limit the number of workloads. A value of 0 denotes no limit.
         * @param {string} [sortKey] Sort workloads by batches, a training metric, or a validation metric.
         * @param {GetTrialWorkloadsRequestFilterOption} [filter] Filter workloads with validation and/or checkpoint information.   - FILTER_OPTION_UNSPECIFIED: Any workload.  - FILTER_OPTION_CHECKPOINT: Only workloads with an associated checkpoint.  - FILTER_OPTION_VALIDATION: Only validation workloads.  - FILTER_OPTION_CHECKPOINT_OR_VALIDATION: Only validation workloads or ones with an associated checkpoint.
         * @param {boolean} [includeBatchMetrics] Include per-batch metrics.
         * @param {V1MetricType} [metricType] When sorting workloads by sort_key, specify training or validation form of a metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {boolean} [removeDeletedCheckpoints] Remove deleted checkpoints.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        getTrialWorkloads(trialId: number, orderBy?: V1OrderBy, offset?: number, limit?: number, sortKey?: string, filter?: GetTrialWorkloadsRequestFilterOption, includeBatchMetrics?: boolean, metricType?: V1MetricType, group?: string, removeDeletedCheckpoints?: boolean, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1GetTrialWorkloadsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).getTrialWorkloads(trialId, orderBy, offset, limit, sortKey, filter, includeBatchMetrics, metricType, group, removeDeletedCheckpoints, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Send notebook idle data to master
         * @param {string} notebookId The id of the notebook.
         * @param {V1IdleNotebookRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        idleNotebook(notebookId: string, body: V1IdleNotebookRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1IdleNotebookResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).idleNotebook(notebookId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill generic task
         * @param {string} taskId The id of the task.
         * @param {V1KillGenericTaskRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killGenericTask(taskId: string, body: V1KillGenericTaskRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillGenericTaskResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).killGenericTask(taskId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Kill runs.
         * @param {V1KillRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        killRuns(body: V1KillRunsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1KillRunsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).killRuns(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary List all resource pools, bound and unbound, available to a specific workspace
         * @param {number} workspaceId Workspace ID.
         * @param {number} [offset] The offset to use with pagination.
         * @param {number} [limit] The maximum number of results to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        listRPsBoundToWorkspace(workspaceId: number, offset?: number, limit?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ListRPsBoundToWorkspaceResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).listRPsBoundToWorkspace(workspaceId, offset, limit, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary List all workspaces bound to a specific resource pool
         * @param {string} resourcePoolName Resource pool name.
         * @param {number} [offset] The offset to use with pagination.
         * @param {number} [limit] The maximum number of results to return.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        listWorkspacesBoundToRP(resourcePoolName: string, offset?: number, limit?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ListWorkspacesBoundToRPResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).listWorkspacesBoundToRP(resourcePoolName, offset, limit, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Mark the given reservation (container, pod, etc) within an allocation as a daemon reservation. In the exit of a successful exit, Determined will wait for all resources to exit - unless they are marked as daemon resources, in which case Determined will clean them up regardless of exit status after all non-daemon resources have exited.
         * @param {string} allocationId The id of the allocation.
         * @param {string} resourcesId The id of the clump of resources to mark as daemon.
         * @param {V1MarkAllocationResourcesDaemonRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        markAllocationResourcesDaemon(allocationId: string, resourcesId: string, body: V1MarkAllocationResourcesDaemonRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1MarkAllocationResourcesDaemonResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).markAllocationResourcesDaemon(allocationId, resourcesId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Get the milestones (in batches processed) at which a metric is recorded by an experiment.
         * @param {number} experimentId The id of the experiment.
         * @param {string} metricName A metric name.
         * @param {V1MetricType} [metricType] The type of metric.   - METRIC_TYPE_UNSPECIFIED: Zero-value (not allowed).  - METRIC_TYPE_TRAINING: For metrics emitted during training.  - METRIC_TYPE_VALIDATION: For metrics emitted during validation.  - METRIC_TYPE_PROFILING: For metrics emitted during profiling.
         * @param {string} [group] Metric group (training, validation, etc).
         * @param {number} [periodSeconds] Seconds to wait when polling for updates.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        metricBatches(experimentId: number, metricName: string, metricType?: V1MetricType, group?: string, periodSeconds?: number, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<StreamResultOfV1MetricBatchesResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).metricBatches(experimentId, metricName, metricType, group, periodSeconds, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Move runs.
         * @param {V1MoveRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        moveRuns(body: V1MoveRunsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1MoveRunsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).moveRuns(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary NotifyContainterRunning is used to notify the master that the container is running.  On HPC, the launcher will report a state of "Running" as soon as Slurm starts the job, but the container may be in the process of getting pulled down from the Internet, so the experiment is not really considered to be in a "Running" state until all the containers that are part of the experiment are running and not being pulled.
         * @param {string} allocationId The ID of the allocation.
         * @param {V1NotifyContainerRunningRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        notifyContainerRunning(allocationId: string, body: V1NotifyContainerRunningRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1NotifyContainerRunningResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).notifyContainerRunning(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Overwrite resource pool - workspace bindings
         * @param {string} resourcePoolName The resource pool name.
         * @param {V1OverwriteRPWorkspaceBindingsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        overwriteRPWorkspaceBindings(resourcePoolName: string, body: V1OverwriteRPWorkspaceBindingsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1OverwriteRPWorkspaceBindingsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).overwriteRPWorkspaceBindings(resourcePoolName, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Update checkpoints. Won't modify checkpoint files.
         * @param {V1PatchCheckpointsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchCheckpoints(body: V1PatchCheckpointsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PatchCheckpointsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).patchCheckpoints(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Patch (an unmanaged) trial.
         * @param {number} trialId Trial id.
         * @param {V1PatchTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchTrial(trialId: number, body: V1PatchTrialRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PatchTrialResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).patchTrial(trialId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Patch multiple users' activation status.
         * @param {V1PatchUsersRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        patchUsers(body: V1PatchUsersRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PatchUsersResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).patchUsers(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Pause generic task
         * @param {string} taskId The id of the task.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseGenericTask(taskId: string, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PauseGenericTaskResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).pauseGenericTask(taskId, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Pause experiment associated with provided runs.
         * @param {V1PauseRunsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        pauseRuns(body: V1PauseRunsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PauseRunsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).pauseRuns(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary PostAllocationAcceleratorData sets the accelerator for a given allocation.
         * @param {string} allocationId The id of the allocation.
         * @param {V1PostAllocationAcceleratorDataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postAllocationAcceleratorData(allocationId: string, body: V1PostAllocationAcceleratorDataRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostAllocationAcceleratorDataResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postAllocationAcceleratorData(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary PostAllocationProxyAddress sets the proxy address to use when proxying to services provided by an allocation. Upon receipt, the master will also register any proxies specified by the task.
         * @param {string} allocationId The id of the allocation.
         * @param {V1PostAllocationProxyAddressRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postAllocationProxyAddress(allocationId: string, body: V1PostAllocationProxyAddressRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostAllocationProxyAddressResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postAllocationProxyAddress(allocationId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Update run metadata.
         * @param {number} runId The ID of the run to post metadata for.
         * @param {V1PostRunMetadataRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postRunMetadata(runId: number, body: V1PostRunMetadataRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostRunMetadataResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postRunMetadata(runId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Persist the given task logs.
         * @param {V1PostTaskLogsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTaskLogs(body: V1PostTaskLogsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostTaskLogsResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postTaskLogs(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Persist the given TrialProfilerMetricsBatch. The trial ID is in the labels.
         * @param {V1PostTrialProfilerMetricsBatchRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTrialProfilerMetricsBatch(body: V1PostTrialProfilerMetricsBatchRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostTrialProfilerMetricsBatchResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postTrialProfilerMetricsBatch(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary For bookkeeping, update trial runner metadata (currently just state).
         * @param {number} trialId The id of the trial.
         * @param {V1TrialRunnerMetadata} body The state for the trial runner.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        postTrialRunnerMetadata(trialId: number, body: V1TrialRunnerMetadata, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PostTrialRunnerMetadataResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).postTrialRunnerMetadata(trialId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Put an experiment by external id.
         * @param {string} externalExperimentId External experiment id.
         * @param {V1CreateExperimentRequest} body CreateExperimentRequest payload.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putExperiment(externalExperimentId: string, body: V1CreateExperimentRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutExperimentResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).putExperiment(externalExperimentId, body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Put a trial.
         * @param {V1PutTrialRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        putTrial(body: V1PutTrialRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1PutTrialResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).putTrial(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Record a checkpoint.
         * @param {V1Checkpoint} body The training metrics to persist.
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportCheckpoint(body: V1Checkpoint, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ReportCheckpointResponse> {
            const localVarFetchArgs = InternalApiFetchParamCreator(configuration).reportCheckpoint(body, options);
            return (fetch: FetchAPI = window.fetch, basePath: string = BASE_PATH) => {
                return fetch(basePath + localVarFetchArgs.url, localVarFetchArgs.options).then((response) => {
                    if (response.status >= 200 && response.status < 300) {
                        return response.json();
                    } else {
                        throw response;
                    }
                });
            };
        },
        /**
         * 
         * @summary Record metrics for specified trial.
         * @param {number} metricsTrialId The trial associated with these metrics.
         * @param {V1ReportTrialMetricsRequest} body
         * @param {*} [options] Override http request option.
         * @throws {RequiredError}
         */
        reportTrialMetrics(metricsTrialId: number, body: V1ReportTrialMetricsRequest, options?: any): (fetch?: FetchAPI, basePath?: string) => Promise<V1ReportTrialMetricsResponse> {
     