import Button from 'hew/Button';
import CodeSample from 'hew/CodeSample';
import Divider from 'hew/Divider';
import Form from 'hew/Form';
import { useTheme } from 'hew/Theme';
import { notification } from 'hew/Toast';
import { useObservable } from 'micro-observables';
import React, { useEffect, useMemo, useState } from 'react';
import { useLocation } from 'react-router-dom';

import LogoGoogle from 'assets/images/logo-sso-google-white.svg?url';
import LogoOkta from 'assets/images/logo-sso-okta-white.svg?url';
import DeterminedAuth from 'components/DeterminedAuth';
import Logo from 'components/Logo';
import Page from 'components/Page';
import PageMessage from 'components/PageMessage';
import useUI from 'components/ThemeProvider';
import { handleRelayState, samlUrl } from 'ee/SamlAuth';
import { globalStorage, sessionStorage } from 'globalStorage';
import useAuthCheck from 'hooks/useAuthCheck';
import usePolling from 'hooks/usePolling';
import { defaultRoute, rbacDefaultRoute } from 'routes';
import { routeAll } from 'routes/utils';
import authStore from 'stores/auth';
import determinedStore from 'stores/determinedInfo';
import { RecordKey } from 'types';
import { locationToPath, routeToReactUrl } from 'utils/routes';
import { capitalize } from 'utils/string';

import css from './SignIn.module.scss';

const logoConfig: Record<RecordKey, string> = {
  google: LogoGoogle,
  okta: LogoOkta,
};

const SignIn: React.FC = () => {
  const { actions: uiActions } = useUI();
  const location = useLocation();
  const isAuthChecked = useObservable(authStore.isChecked);
  const isAuthenticated = useObservable(authStore.isAuthenticated);
  const info = useObservable(determinedStore.info);
  const [canceler] = useState(new AbortController());
  const { rbacEnabled } = useObservable(determinedStore.info);
  const {
    themeSettings: { className: themeClass },
  } = useTheme();
  const queries = useMemo(() => new URLSearchParams(location.search), [location.search]);
  const ssoQueries = handleRelayState(queries);

  const externalAuthError = useMemo(() => {
    return isAuthChecked && !isAuthenticated && !info.externalLoginUri && queries.get('jwt');
  }, [isAuthChecked, isAuthenticated, info.externalLoginUri, queries]);

  /*
   * Check every so often to see if the user is authenticated.
   * For example, the user can authenticate in a different session,info
   * and this will pick up that auth and automatically redirect them into
   * their previous app. We don't run immediately because the router also
   * performs an auth check there as well upon the first page load.
   */
  usePolling(useAuthCheck(), { interval: 1000, runImmediately: false });

  /*
   * Check for when `isAuthenticated` becomes true and redirect
   * the user to the most recent requested page.
   */
  useEffect(() => {
    if (location.state != null) {
      sessionStorage.landingRedirect = locationToPath(location.state) ?? '';
    }

    if (isAuthenticated) {
      // Stop the spinner, prepping for user redirect.
      uiActions.hideSpinner();

      // Show auth token via notification if requested via query parameters.
      if (queries.get('cli') === 'true')
        notification.open({
          className: themeClass,
          description: <CodeSample text={globalStorage.authToken || 'Auth token not found.'} />,
          duration: 0,
          message: 'Your Determined Authentication Token',
        });

      // Reroute the authenticated user to the app.
      if (!queries.has('redirect')) {
        let path = sessionStorage.landingRedirect;
        sessionStorage.removeLandingRedirect();
        if (path === '') {
          path = globalStorage.landingRedirect;
          globalStorage.removeLandingRedirect();
        }
        routeToReactUrl(path || (rbacEnabled ? rbacDefaultRoute.path : defaultRoute.path));
      } else {
        routeAll(queries.get('redirect') || '');
      }
    } else if (isAuthChecked) {
      uiActions.hideSpinner();
    }
  }, [isAuthenticated, isAuthChecked, info, location, queries, uiActions, rbacEnabled, themeClass]);

  useEffect(() => {
    uiActions.hideChrome();
    return uiActions.showChrome;
  }, [uiActions]);

  // Stop the polling upon a dismount of this page.
  useEffect(() => {
    return () => canceler.abort();
  }, [canceler]);

  /*
   * Don't render sign in page if...
   *   1. jwt query param detected
   *   2. cluster has `externalLoginUri` defined
   *   3. authentication hasn't occurred yet
   *   4. a remote user token has expired
   *   5. an SSO providers is set to always redirect
   * This will prevent the form from showing for a split second when
   * accessing a page from the browser when the user is already verified.
   */
  const redirectToSSO =
    !queries.get('hard_logout') &&
    (queries.get('remote_expired') ||
      info.ssoProviders?.some((ssoProvider) => ssoProvider.alwaysRedirect));

  if (queries.has('jwt') || info.externalLoginUri || !isAuthChecked || redirectToSSO) return null;

  /*
   * An external auth error occurs when there are external auth urls,
   * auth fails with a jwt.
   */
  if (externalAuthError)
    return (
      <PageMessage title="Cluster Not Available">
        <p>Cluster is not ready. Please try again later.</p>
      </PageMessage>
    );

  return (
    <Page breadcrumb={[]} docTitle="Sign In" ignorePermissions noScroll>
      <div className={css.base}>
        <div className={css.content}>
          <Logo
            branding={info.branding}
            hasCustomLogo={info.hasCustomLogo}
            orientation="vertical"
          />
          <DeterminedAuth canceler={canceler} />
          {info.ssoProviders && info.ssoProviders.length > 0 && (
            <>
              <Divider>OR</Divider>
              <Form className={css.form} layout="vertical">
                <p>Alternatively, sign in with SSO</p>
                {info.ssoProviders.map((ssoProvider) => {
                  const key = ssoProvider.name.toLowerCase();
                  const logo = logoConfig[key] ? (
                    <img alt={key} className={css.ssoLogo} src={logoConfig[key]} />
                  ) : null;
                  return (
                    <Form.Item key={key}>
                      <a
                        className={css.ssoButton}
                        href={
                          ssoProvider.type === 'OIDC'
                            ? ssoProvider.ssoUrl
                            : samlUrl(ssoProvider.ssoUrl, ssoQueries.toString())
                        }>
                        <Button type="primary">
                          <div className={css.ssoProviderInfo}>
                            {logo}
                            <span>
                              {ssoProvider.name === key ? capitalize(key) : ssoProvider.name}
                            </span>
                          </div>
                        </Button>
                      </a>
                    </Form.Item>
                  );
                })}
              </Form>
            </>
          )}
        </div>
      </div>
    </Page>
  );
};

export default SignIn;
