import Message from 'hew/Message';
import Spinner from 'hew/Spinner';
import { Loadable } from 'hew/utils/loadable';
import _ from 'lodash';
import React, { useCallback, useEffect, useRef, useState } from 'react';
import { useNavigate, useParams } from 'react-router-dom';

import Page, { BreadCrumbRoute } from 'components/Page';
import { terminalRunStates } from 'constants/states';
import useFeature from 'hooks/useFeature';
import usePolling from 'hooks/usePolling';
import ExperimentDetailsHeader from 'pages/ExperimentDetails/ExperimentDetailsHeader';
import ExperimentMultiTrialTabs from 'pages/ExperimentDetails/ExperimentMultiTrialTabs';
import ExperimentSingleTrialTabs from 'pages/ExperimentDetails/ExperimentSingleTrialTabs';
import { TrialInfoBoxMultiTrial } from 'pages/TrialDetails/TrialInfoBox';
import { paths } from 'routes/utils';
import { getExperimentDetails } from 'services/api';
import userSettings from 'stores/userSettings';
import workspaceStore from 'stores/workspaces';
import { ExperimentBase, TrialItem, Workspace } from 'types';
import { isSingleTrialExperiment } from 'utils/experiment';
import { useObservable } from 'utils/observable';
import { isAborted, isNotFound } from 'utils/service';

type Params = {
  experimentId: string;
};

export const INVALID_ID_MESSAGE = 'Invalid Experiment ID';
export const ERROR_MESSAGE = 'Unable to fetch Experiment';

const settingsLoadedObs = userSettings.getAll().select((l) => l.isLoaded);

const ExperimentDetails: React.FC = () => {
  const { experimentId } = useParams<Params>();
  const [experiment, setExperiment] = useState<ExperimentBase>();
  const [trial, setTrial] = useState<TrialItem>();
  const [pageError, setPageError] = useState<Error>();
  const [isSingleTrial, setIsSingleTrial] = useState<boolean>();
  const pageRef = useRef<HTMLElement>(null);
  const canceler = useRef<AbortController>();
  const workspaces = Loadable.getOrElse([], useObservable(workspaceStore.workspaces));
  const id = parseInt(experimentId ?? '');
  const navigate = useNavigate();
  const f_flat_runs = useFeature().isOn('flat_runs');
  const isSettingsLoaded = useObservable(settingsLoadedObs);

  if (isSettingsLoaded && f_flat_runs) {
    navigate(paths.searchDetails(id), { replace: true });
  }

  const fetchExperimentDetails = useCallback(async () => {
    try {
      const newExperiment = await getExperimentDetails(
        { id },
        { signal: canceler.current?.signal },
      );
      setExperiment((prevExperiment) =>
        _.isEqual(prevExperiment, newExperiment) ? prevExperiment : newExperiment,
      );
      setIsSingleTrial(isSingleTrialExperiment(newExperiment));
    } catch (e) {
      if (!pageError && !isAborted(e)) setPageError(e as Error);
    }
  }, [id, pageError]);

  const { stopPolling } = usePolling(fetchExperimentDetails, { rerunOnNewFn: true });

  const handleSingleTrialUpdate = useCallback((trial: TrialItem) => {
    setTrial(trial);
  }, []);

  useEffect(() => {
    if (experiment && terminalRunStates.has(experiment.state)) {
      stopPolling();
    }
  }, [experiment, stopPolling]);

  useEffect(() => {
    fetchExperimentDetails();
  }, [fetchExperimentDetails]);

  useEffect(() => {
    canceler.current = new AbortController();
    return () => {
      canceler.current?.abort();
      canceler.current = undefined;
    };
  }, []);

  if (isNaN(id)) {
    return <Message title={`${INVALID_ID_MESSAGE} ${experimentId}`} />;
  } else if (pageError && !isNotFound(pageError)) {
    const message = `${ERROR_MESSAGE} ${experimentId}`;
    return <Message icon="warning" title={message} />;
  } else if (!pageError && (!experiment || isSingleTrial === undefined)) {
    return <Spinner spinning tip={`Loading experiment ${experimentId} details...`} />;
  }

  const workspaceName = workspaces.find((ws: Workspace) => ws.id === experiment?.workspaceId)?.name;

  const pageBreadcrumb: BreadCrumbRoute[] = [
    workspaceName && experiment?.workspaceId !== 1
      ? {
          breadcrumbName: workspaceName,
          path: paths.workspaceDetails(experiment?.workspaceId ?? 1),
        }
      : {
          breadcrumbName: 'Uncategorized Experiments',
          path: paths.projectDetails(1),
        },
  ];

  if (experiment?.projectName && experiment?.projectId && experiment?.projectId !== 1)
    pageBreadcrumb.push({
      breadcrumbName: experiment?.projectName ?? '',
      path: paths.projectDetails(experiment?.projectId),
    });

  pageBreadcrumb.push({
    breadcrumbName: experiment?.name ?? '',
    path: paths.experimentDetails(id),
  });

  return (
    <Page
      breadcrumb={pageBreadcrumb}
      containerRef={pageRef}
      headerComponent={
        experiment && (
          <ExperimentDetailsHeader
            experiment={experiment}
            fetchExperimentDetails={fetchExperimentDetails}
            trial={trial}
          />
        )
      }
      notFound={pageError && isNotFound(pageError)}
      stickyHeader
      title={`Experiment ${experimentId}`}>
      {experiment &&
        (isSingleTrial ? (
          <ExperimentSingleTrialTabs
            experiment={experiment}
            fetchExperimentDetails={fetchExperimentDetails}
            pageRef={pageRef}
            onTrialUpdate={handleSingleTrialUpdate}
          />
        ) : (
          <>
            <TrialInfoBoxMultiTrial experiment={experiment} />
            <ExperimentMultiTrialTabs
              experiment={experiment}
              fetchExperimentDetails={fetchExperimentDetails}
              pageRef={pageRef}
            />
          </>
        ))}
    </Page>
  );
};

export default ExperimentDetails;
