import Breadcrumb from 'hew/Breadcrumb';
import Card from 'hew/Card';
import Icon from 'hew/Icon';
import Message from 'hew/Message';
import Spinner from 'hew/Spinner';
import { Loadable } from 'hew/utils/loadable';
import React, { useCallback, useEffect, useState } from 'react';

import ExperimentIcons from 'components/ExperimentIcons';
import JupyterLabButton from 'components/JupyterLabButton';
import Link from 'components/Link';
import Page, { BreadCrumbRoute } from 'components/Page';
import ProjectCard from 'components/ProjectCard';
import Section from 'components/Section';
import ResponsiveTable from 'components/Table/ResponsiveTable';
import {
  experimentNameRenderer,
  relativeTimeRenderer,
  taskNameRenderer,
  taskTypeRenderer,
} from 'components/Table/Table';
import useFeature from 'hooks/useFeature';
import usePermissions from 'hooks/usePermissions';
import usePolling from 'hooks/usePolling';
import { paths } from 'routes/utils';
import {
  getCommands,
  getExperiments,
  getJupyterLabs,
  getProjectsByUserActivity,
  getShells,
  getTensorBoards,
} from 'services/api';
import userStore from 'stores/users';
import workspaceStore from 'stores/workspaces';
import { BulkExperimentItem, CommandTask, DetailedUser, Project } from 'types';
import handleError, { ErrorType } from 'utils/error';
import { useObservable } from 'utils/observable';
import { dateTimeStringSorter } from 'utils/sort';

import css from './Dashboard.module.scss';

const SUBMISSIONS_FETCH_LIMIT = 25;
const PROJECTS_FETCH_LIMIT = 5;
const DISPLAY_LIMIT = 25;

const Dashboard: React.FC = () => {
  const [experiments, setExperiments] = useState<BulkExperimentItem[]>([]);
  const [tasks, setTasks] = useState<CommandTask[]>([]);
  const [projects, setProjects] = useState<Project[]>([]);
  const [submissions, setSubmissions] = useState<Submission[]>([]);
  const [canceler] = useState(new AbortController());
  const [submissionsLoading, setSubmissionsLoading] = useState<boolean>(true);
  const [projectsLoading, setProjectsLoading] = useState<boolean>(true);
  const currentUser = Loadable.getOrElse(undefined, useObservable(userStore.currentUser));
  const workspaces = Loadable.getOrElse([], useObservable(workspaceStore.workspaces));
  const { canCreateNSC } = usePermissions();
  const f_flat_runs = useFeature().isOn('flat_runs');
  type Submission = BulkExperimentItem & CommandTask;

  const fetchTasks = useCallback(
    async (user: DetailedUser) => {
      const results = await Promise.allSettled([
        getCommands({
          limit: SUBMISSIONS_FETCH_LIMIT,
          orderBy: 'ORDER_BY_DESC',
          signal: canceler.signal,
          sortBy: 'SORT_BY_START_TIME',
          users: [user.id.toString()],
        }),
        getJupyterLabs({
          limit: SUBMISSIONS_FETCH_LIMIT,
          orderBy: 'ORDER_BY_DESC',
          signal: canceler.signal,
          sortBy: 'SORT_BY_START_TIME',
          users: [user.id.toString()],
        }),
        getShells({
          limit: SUBMISSIONS_FETCH_LIMIT,
          orderBy: 'ORDER_BY_DESC',
          signal: canceler.signal,
          sortBy: 'SORT_BY_START_TIME',
          users: [user.id.toString()],
        }),
        getTensorBoards({
          limit: SUBMISSIONS_FETCH_LIMIT,
          orderBy: 'ORDER_BY_DESC',
          signal: canceler.signal,
          sortBy: 'SORT_BY_START_TIME',
          users: [user.id.toString()],
        }),
      ]);
      const newTasks = results.reduce((acc, current) => {
        if (current.status === 'fulfilled') return acc.concat(current.value);
        return acc;
      }, [] as CommandTask[]);
      setTasks(newTasks);
    },
    [canceler],
  );

  const fetchExperiments = useCallback(
    async (user: DetailedUser) => {
      try {
        const response = await getExperiments(
          {
            limit: SUBMISSIONS_FETCH_LIMIT,
            orderBy: 'ORDER_BY_DESC',
            sortBy: 'SORT_BY_START_TIME',
            users: [user.id.toString()],
          },
          {
            signal: canceler.signal,
          },
        );
        setExperiments(response.experiments);
      } catch (e) {
        handleError(e, {
          publicSubject: `Error fetching ${f_flat_runs ? 'searches' : 'experiments'} for dashboard`,
          silent: false,
          type: ErrorType.Api,
        });
      }
    },
    [f_flat_runs, canceler],
  );

  const fetchProjects = useCallback(async () => {
    try {
      const projects = await getProjectsByUserActivity(
        { limit: PROJECTS_FETCH_LIMIT },
        {
          signal: canceler.signal,
        },
      );
      setProjects(projects);
      setProjectsLoading(false);
    } catch (e) {
      handleError(e, {
        publicSubject: 'Error fetching projects for dashboard',
        silent: false,
        type: ErrorType.Api,
      });
    }
  }, [canceler]);

  const fetchSubmissions = useCallback(async () => {
    if (!currentUser) return;
    await Promise.allSettled([fetchExperiments(currentUser), fetchTasks(currentUser)]);
    setSubmissionsLoading(false);
  }, [currentUser, fetchExperiments, fetchTasks]);

  const fetchAll = useCallback(() => {
    fetchProjects();
    fetchSubmissions();
  }, [fetchSubmissions, fetchProjects]);

  const { stopPolling } = usePolling(fetchAll, { rerunOnNewFn: true });

  useEffect(() => {
    setSubmissions(
      (experiments as Submission[])
        .concat(tasks as Submission[])
        .sort((a, b) => dateTimeStringSorter(b.startTime, a.startTime))
        .slice(0, DISPLAY_LIMIT),
    );
  }, [experiments, tasks]);

  useEffect(() => {
    return () => {
      canceler.abort();
      stopPolling();
    };
  }, [canceler, stopPolling]);

  const pageBreadCrumb: BreadCrumbRoute[] = [{ breadcrumbName: 'Home', path: paths.dashboard() }];

  if (projectsLoading && submissionsLoading) {
    return null;
  }

  return (
    <Page
      breadcrumb={pageBreadCrumb}
      options={<JupyterLabButton enabled={canCreateNSC} />}
      title="Home">
      {projectsLoading ? (
        <Section>
          <Spinner center spinning />
        </Section>
      ) : projects.length > 0 ? (
        // hide Projects header when empty:
        <Section title="Recently Viewed Projects">
          <Card.Group size="small" wrap={false}>
            {projects.map((project) => (
              <ProjectCard hideActionMenu key={project.id} project={project} showWorkspace />
            ))}
          </Card.Group>
        </Section>
      ) : null}
      {/* show Submissions header even when empty: */}
      <Section title="Your Recent Submissions">
        {submissionsLoading ? (
          <Spinner center spinning />
        ) : submissions.length > 0 ? (
          <ResponsiveTable<Submission>
            className={css.table}
            columns={[
              {
                dataIndex: 'state',
                render: (state) => {
                  return <ExperimentIcons state={state} />;
                },
                width: 1,
              },
              {
                dataIndex: 'projectId',
                render: (projectId, row, index) => {
                  if (projectId) {
                    return <Icon name="experiment" title={f_flat_runs ? 'Experiment' : 'Search'} />;
                  } else {
                    return taskTypeRenderer(row.type, row, index);
                  }
                },
                width: 1,
              },
              {
                dataIndex: 'name',
                render: (name, row, index) => {
                  if (row.projectId) {
                    return experimentNameRenderer(name, row);
                  } else {
                    return taskNameRenderer(row.id, row, index);
                  }
                },
              },
              {
                dataIndex: 'projectId',
                render: (projectId, row) => {
                  if (row.workspaceId && !row.projectId) {
                    // Tasks
                    return (
                      <Breadcrumb>
                        <Breadcrumb.Item>
                          <Link path={paths.workspaceDetails(row.workspaceId)}>
                            {workspaces.find((w) => w.id === row.workspaceId)?.name ||
                              String(row.workspaceId)}
                          </Link>
                        </Breadcrumb.Item>
                      </Breadcrumb>
                    );
                  } else if (row.workspaceId && row.projectId !== 1) {
                    return (
                      <Breadcrumb>
                        <Breadcrumb.Item>
                          <Link path={paths.workspaceDetails(row.workspaceId)}>
                            {row.workspaceName}
                          </Link>
                        </Breadcrumb.Item>
                        <Breadcrumb.Item>
                          <Link path={paths.projectDetails(projectId)}>{row.projectName}</Link>
                        </Breadcrumb.Item>
                      </Breadcrumb>
                    );
                  }
                  if (row.projectName) {
                    return (
                      <Breadcrumb>
                        <Breadcrumb.Item>
                          <Link path={paths.projectDetails(projectId)}>{row.projectName}</Link>
                        </Breadcrumb.Item>
                      </Breadcrumb>
                    );
                  }
                  return <></>;
                },
              },
              {
                dataIndex: 'startTime',
                render: relativeTimeRenderer,
              },
            ]}
            dataSource={submissions}
            loading={submissionsLoading}
            pagination={false}
            rowKey="id"
            showHeader={false}
            size="middle"
          />
        ) : (
          <Message
            description={
              <>
                Your recent {f_flat_runs ? 'searches' : 'experiments'} and tasks will show up here.{' '}
                <Link external path={paths.docs('/quickstart-mdldev.html')}>
                  Get started
                </Link>
              </>
            }
            icon="experiment"
            title="No submissions"
          />
        )}
      </Section>
    </Page>
  );
};

export default Dashboard;
