import Alert from 'hew/Alert';
import CodeEditor from 'hew/CodeEditor';
import { Modal } from 'hew/Modal';
import { Loadable, Loaded } from 'hew/utils/loadable';
import { Map } from 'immutable';
import { useMemoizedObservable } from 'micro-observables';
import React, { useCallback, useEffect, useMemo, useState } from 'react';

import useUI, { Mode } from 'components/ThemeProvider';
import userSettings from 'stores/userSettings';
import { Json } from 'types';
import { isJsonObject, isObject } from 'utils/data';
import handleError from 'utils/error';

interface Props {
  onSave?: () => void;
}

const UserSettingsModalComponent: React.FC<Props> = ({ onSave }: Props) => {
  const { actions: uiActions } = useUI();
  const [configError, setConfigError] = useState(false);
  const initialSettingsString = useMemoizedObservable<Loadable<string>>(
    () =>
      userSettings
        .getAll()
        .select((loadableState) =>
          Loadable.map(loadableState, (state) => JSON.stringify(state, undefined, ' ')),
        ),
    [],
  );
  const [editedSettingsString, setEditedSettingsString] =
    useState<Loadable<string>>(initialSettingsString);

  const editedSettings: Map<string, Json> | undefined = useMemo(
    () =>
      Loadable.match(editedSettingsString, {
        _: () => undefined,
        Loaded: (settingsString) => {
          try {
            const obj = JSON.parse(settingsString);
            setConfigError(false);
            return isObject(obj) ? Map<string, Json>(obj) : undefined;
          } catch {
            setConfigError(true);
            return;
          }
        },
      }),
    [editedSettingsString],
  );

  useEffect(() => {
    if (Loadable.isLoaded(editedSettingsString)) return;

    setEditedSettingsString(initialSettingsString);
  }, [editedSettingsString, initialSettingsString]);

  const handleSave = useCallback(() => {
    if (!editedSettings) return;

    onSave?.();
    userSettings.overwrite(editedSettings);

    // We have to special case the mode because otherwise ThemeProvider will revert the change.
    const theme = editedSettings.get('settings-theme');
    if (
      theme !== undefined &&
      isJsonObject(theme) &&
      Object.values(Mode).includes(theme.mode as Mode)
    ) {
      uiActions.setMode(theme.mode as Mode);
    }
  }, [editedSettings, onSave, uiActions]);

  const handleChange = useCallback((newSettings: string) => {
    setEditedSettingsString(Loaded(newSettings));
  }, []);

  return (
    <Modal
      cancel
      size="medium"
      submit={{
        disabled: editedSettings === undefined,
        handleError,
        handler: handleSave,
        text: 'Save Settings',
      }}
      title="Edit Raw Settings">
      <CodeEditor
        file={editedSettingsString}
        files={[
          {
            key: 'settings.json',
            title: 'settings.json',
          },
        ]}
        height="438px"
        onChange={handleChange}
        onError={handleError}
      />
      {configError && <Alert message="Invalid JSON" type="error" />}
    </Modal>
  );
};

export default UserSettingsModalComponent;
