CREATE TABLE storage_backend_shared_fs (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  host_path        TEXT NOT NULL,
  container_path   TEXT,
  checkpoint_path  TEXT,
  tensorboard_path TEXT,
  storage_path     TEXT,
  propagation      TEXT NOT NULL
  CONSTRAINT shared_fs_reserved_value CHECK (
    container_path != 'DeterminedReservedNullUniqueValue' AND
    checkpoint_path != 'DeterminedReservedNullUniqueValue' AND
    tensorboard_path != 'DeterminedReservedNullUniqueValue' AND
    storage_path != 'DeterminedReservedNullUniqueValue'
  )
);

-- Hack for lack of pre-postgres 15 NULLS NOT DISTINCT.
-- https://stackoverflow.com/questions/8289100/create-unique-constraint-with-null-columns
CREATE UNIQUE INDEX ix_storage_backend_unique_fs ON storage_backend_shared_fs (
  host_path,
  COALESCE(container_path, 'DeterminedReservedNullUniqueValue'),
  COALESCE(checkpoint_path, 'DeterminedReservedNullUniqueValue'),
  COALESCE(tensorboard_path, 'DeterminedReservedNullUniqueValue'),
  COALESCE(storage_path, 'DeterminedReservedNullUniqueValue'),
  propagation
);


CREATE TABLE storage_backend_s3 (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  bucket       TEXT NOT NULL,
  access_key   TEXT,
  secret_key   TEXT,
  endpoint_url TEXT,
  prefix       TEXT,
  CONSTRAINT s3_check_prefix CHECK (
    prefix != '..'
    AND NOT prefix LIKE '../%'
    AND NOT prefix LIKE '%/..'
    AND NOT prefix LIKE '%/../%'
  ),
  CONSTRAINT s3_reserved_value CHECK (
    bucket != 'DeterminedReservedNullUniqueValue' AND
    access_key != 'DeterminedReservedNullUniqueValue' AND
    secret_key != 'DeterminedReservedNullUniqueValue' AND
    endpoint_url != 'DeterminedReservedNullUniqueValue' AND
    prefix != 'DeterminedReservedNullUniqueValue'
  )
);

CREATE UNIQUE INDEX ix_storage_backend_unique_s3 ON storage_backend_s3 (
  bucket,
  COALESCE(access_key, 'DeterminedReservedNullUniqueValue'),
  COALESCE(secret_key, 'DeterminedReservedNullUniqueValue'),
  COALESCE(endpoint_url, 'DeterminedReservedNullUniqueValue'),
  COALESCE(prefix, 'DeterminedReservedNullUniqueValue')
);

CREATE TABLE storage_backend_gcs (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  bucket       TEXT NOT NULL,
  prefix       TEXT,
  CONSTRAINT gcs_check_prefix CHECK (
    prefix != '..'
    AND NOT prefix LIKE '../%'
    AND NOT prefix LIKE '%/..'
    AND NOT prefix LIKE '%/../%'
  ),
  CONSTRAINT gcs_reserved_value CHECK (
    bucket != 'DeterminedReservedNullUniqueValue' AND
    prefix != 'DeterminedReservedNullUniqueValue'
  )
);

CREATE UNIQUE INDEX ix_storage_backend_unique_gcs ON storage_backend_gcs (
  bucket,
  COALESCE(prefix, 'DeterminedReservedNullUniqueValue')
);

CREATE TABLE storage_backend_azure (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  container         TEXT NOT NULL,
  connection_string TEXT,
  account_url       TEXT,
  credential        TEXT,
  CONSTRAINT azure_dont_give_all CHECK (
    NOT (connection_string IS NOT NULL AND account_url IS NOT NULL) AND
    NOT (connection_string IS NOT NULL AND credential IS NOT NULL)
  ),
  CONSTRAINT azure_reserved_value CHECK (
    connection_string != 'DeterminedReservedNullUniqueValue' AND
    account_url != 'DeterminedReservedNullUniqueValue' AND
    credential != 'DeterminedReservedNullUniqueValue'
  )
);

CREATE UNIQUE INDEX ix_storage_backend_unique_azure ON storage_backend_azure (
  container,
  COALESCE(connection_string, 'DeterminedReservedNullUniqueValue'),
  COALESCE(account_url, 'DeterminedReservedNullUniqueValue'),
  COALESCE(credential, 'DeterminedReservedNullUniqueValue')
);

CREATE TABLE storage_backend_directory (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  container_path TEXT
);

CREATE UNIQUE INDEX ix_storage_backend_unique_directory ON storage_backend_directory (
  container_path
);

CREATE TABLE storage_backend (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,

  shared_fs_id integer UNIQUE REFERENCES storage_backend_shared_fs(id) ON DELETE CASCADE,
  s3_id        integer UNIQUE REFERENCES storage_backend_s3(id) ON DELETE CASCADE,
  gcs_id       integer UNIQUE REFERENCES storage_backend_gcs(id) ON DELETE CASCADE,
  azure_id     integer UNIQUE REFERENCES storage_backend_azure(id) ON DELETE CASCADE,
  directory_id integer UNIQUE REFERENCES storage_backend_directory(id) ON DELETE CASCADE,

  CONSTRAINT check_one_not_null
    CHECK (
      (shared_fs_id IS NOT NULL)::integer +
      (s3_id IS NOT NULL)::integer +
      (gcs_id IS NOT NULL)::integer +
      (azure_id IS NOT NULL)::integer +
      (directory_id IS NOT NULL)::integer = 1
    )
);

ALTER TABLE checkpoints_v2
  ADD COLUMN storage_id int REFERENCES storage_backend(id);

DROP VIEW proto_checkpoints_view;
DROP VIEW checkpoints_view;

CREATE OR REPLACE VIEW checkpoints_view AS
    SELECT
        c.id AS id,
        c.uuid AS uuid,
        c.task_id,
        c.allocation_id,
        c.report_time,
        c.state,
        c.resources,
        c.metadata,
        r.id AS trial_id,
        e.id AS experiment_id,
        e.config AS experiment_config,
        r.hparams AS hparams,
        s.metrics AS training_metrics,
        v.metrics->'validation_metrics' AS validation_metrics,
        (v.metrics->'validation_metrics'->>(e.config->'searcher'->>'metric'))::float8 AS searcher_metric,
        CAST(c.metadata->>'steps_completed' AS int) as steps_completed,
        c.size,
        c.storage_id
    FROM checkpoints_v2 AS c
    LEFT JOIN run_checkpoints AS rc on rc.checkpoint_id = c.uuid
    LEFT JOIN runs AS r on r.id = rc.run_id
    LEFT JOIN experiments AS e on r.experiment_id = e.id
    LEFT JOIN raw_validations AS v on CAST(c.metadata->>'steps_completed' AS int) = v.total_batches and r.id = v.trial_id AND NOT v.archived
    LEFT JOIN raw_steps AS s on CAST(c.metadata->>'steps_completed' AS int) = s.total_batches and r.id = s.trial_id AND NOT s.archived;

CREATE OR REPLACE VIEW proto_checkpoints_view AS
    SELECT
        c.uuid::text AS uuid,
        c.task_id,
        c.allocation_id,
        c.report_time as report_time,
        'STATE_' || c.state AS state,
        c.resources,
        c.metadata,
        c.storage_id,
        -- Build a training substruct for protobuf.
        jsonb_build_object(
            'trial_id', c.trial_id,
            'experiment_id', c.experiment_id,
            'experiment_config', c.experiment_config,
            'hparams', c.hparams,
            -- construct training metrics from the untyped jsonb deterministically, since older
            -- versions may have old keys (e.g., num_inputs) and our unmarshaling is strict.
            'training_metrics', jsonb_build_object(
                'avg_metrics', c.training_metrics->'avg_metrics',
                'batch_metrics', c.training_metrics->'batch_metrics'
            ),
            'validation_metrics', json_build_object('avg_metrics', c.validation_metrics),
            'searcher_metric', c.searcher_metric
        ) AS training
    FROM checkpoints_view AS c;
