ALTER table trials ADD COLUMN tags jsonb DEFAULT '{}';
-- if similar data structure were to be used where the actual values of the tags
-- was considered, index should not use jsonb_path_ops (just remove that part)
CREATE INDEX trials_tags_index ON trials USING GIN (tags jsonb_path_ops);
-- update trials set tags = '{}';

CREATE TABLE trials_collections (
    -- table to store a set of filters as defined in QueryFilters in api/trial.proto
    id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,	
    user_id integer REFERENCES public.users(id) ON DELETE CASCADE NOT NULL,
    project_id integer REFERENCES public.projects(id) ON DELETE CASCADE NOT NULL,
	  name text NOT NULL,
	  filters jsonb NOT NULL,
    sorter  jsonb NOT NULL,
  CONSTRAINT project_filter_name_uniq UNIQUE (project_id, name)
);

-- make analogous modifications to those found in reindex-steps pr
-- because we are joining validations on (trial_id, total_batches)
CREATE INDEX validations_archived ON raw_validations(archived);
ALTER TABLE raw_validations DROP CONSTRAINT validations_trial_id_run_id_total_batches_unique;
CREATE UNIQUE INDEX validations_trial_id_total_batches_run_id_unique ON raw_validations(trial_id, total_batches, trial_run_id);


CREATE AGGREGATE jsonb_collect(jsonb) (
  SFUNC = 'jsonb_concat',
  STYPE = jsonb,
  INITCOND = '{}'
);


-- aggregations should be mostly redundant.the only way they 
-- would get applied is when there are multiple trial_run_ids for a single 
-- (trial_id, total_batches) on steps or validations. in which case,
-- which trial_run_id is it appropriate to take the metrics from?
-- there have been situations where they differ.
CREATE OR REPLACE VIEW public.trials_augmented_view AS
  WITH b AS (
    select trial_id, max(total_batches) total_batches from steps group by trial_id
  )
  SELECT
      t.id AS trial_id,
      t.state AS state,
      t.hparams AS hparams,
      jsonb_collect(s.metrics->'avg_metrics') AS training_metrics,
      jsonb_collect(v.metrics->'validation_metrics') AS validation_metrics,
      t.tags AS tags,
      t.start_time AS start_time,
      t.end_time AS end_time,
      max(e.config->'searcher'->>'name') as searcher_type,
      max(e.id) AS experiment_id,
      max(e.config->>'name') AS experiment_name,
      max(e.config->>'description') AS experiment_description,
      -- there's only one
      jsonb_agg(e.config ->> 'labels'::text) AS experiment_labels,
      max(e.owner_id) AS user_id,
      max(e.project_id) AS project_id,
      max(p.workspace_id) AS workspace_id,
      -- temporary
      max(b.total_batches) as total_batches,
      max(e.config->'searcher'->>'metric') AS searcher_metric,
      max(v.metrics->'validation_metrics'->>(e.config->'searcher'->>'metric'))::float8 AS searcher_metric_value,
      max(CASE
          WHEN coalesce((config->'searcher'->>'smaller_is_better')::boolean, true) 
            THEN (v.metrics->'validation_metrics'->>(e.config->'searcher'->>'metric'))::float8
            ELSE -1.0 * (v.metrics->'validation_metrics'->>(e.config->'searcher'->>'metric'))::float8
      END) AS searcher_metric_loss
  FROM trials t
  LEFT JOIN experiments e ON t.experiment_id = e.id
  LEFT JOIN projects p ON e.project_id = p.id
  LEFT JOIN validations v ON t.id = v.trial_id AND v.id = t.best_validation_id
  LEFT JOIN steps s on t.id = s.trial_id AND v.total_batches = s.total_batches
  LEFT JOIN b on t.id = b.trial_id 
  GROUP BY t.id;

