package nprand

import (
	"runtime"
	"testing"

	"gotest.tools/assert"
)

func TestInt(t *testing.T) {
	state := State{}
	numbers := make([]int64, 100)

	// state = np.random.RandomState(42); [state.randint(0,10) for _ in range(100)]
	fromNumpy := []int64{
		6, 3, 7, 4, 6, 9, 2, 6, 7, 4, 3, 7, 7, 2, 5, 4, 1, 7, 5, 1, 4, 0, 9, 5,
		8, 0, 9, 2, 6, 3, 8, 2, 4, 2, 6, 4, 8, 6, 1, 3, 8, 1, 9, 8, 9, 4, 1, 3, 6, 7, 2, 0,
		3, 1, 7, 3, 1, 5, 5, 9, 3, 5, 1, 9, 1, 9, 3, 7, 6, 8, 7, 4, 1, 4, 7, 9, 8, 8, 0, 8,
		6, 8, 7, 0, 7, 7, 2, 0, 7, 2, 2, 0, 4, 9, 6, 9, 8, 6, 8, 7,
	}
	state.Seed(42)
	for i := 0; i < 100; i++ {
		numbers[i] = int64(state.Intn(10))
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(84); [state.randint(-10,10) for _ in range(100)]
	fromNumpy = []int64{
		0, 7, 4, 3, -10, 4, -9, 0, -6, -9, 3, -2, 4, -10, 6, 3, 4, -6, -6, 8,
		-8, 5, 2, 7, 6, 4, -1, 3, 4, -9, -1, 6, 0, -10, -7, -7, 5, -1, 1, 6, -6, 8, 7, 4, 8,
		3, -4, 3, 5, 7, 7, -5, 0, 7, 0, 7, -3, 7, 6, 0, -8, 2, -1, 9, -5, 8, 0, -8, 4, -6,
		-6, 4, 5, 6, -1, 5, 4, -5, 9, -3, 9, 3, -7, -3, 2, -10, -4, -4, -3, -7, -10, -1, 2,
		3, 7, -7, 9, 3, -3, -6,
	}
	state.Seed(84)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Int64(-10, 10)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(84); [state.randint(-1<<31,0) for _ in range(100)]
	fromNumpy = []int64{
		-1949739094, -1889440015, -557304999, -92369608, -1106734603,
		-792023684, -1595666226, -556143507, -42476992, -1173572370, -1586612255,
		-964641078, -1401923846, -1675861308, -1038033247, -302609543, -350473771,
		-1264057350, -1584761779, -1373692871, -900009860, -294849633, -20160184,
		-1277145106, -141715392, -1874287907, -870203906, -1026645712, -1495499283,
		-2075811538, -1994527420, -1265390140, -485217996, -2086920297, -581546094,
		-317141630, -255409577, -1585691976, -1692933765, -443246252, -1661197478,
		-16395473, -1380441972, -435460550, -1828895599, -539256336, -17468754, -1889309286,
		-617298583, -880583763, -689451826, -1784560387, -1554471455, -1182981281,
		-1247155991, -1902584624, -92655606, -38809728, -1553011389, -1055655010,
		-1918892157, -525616913, -1001868197, -1132423276, -808777507, -1063779991,
		-725856587, -1267904104, -1521144533, -1751799716, -769833477, -1190652400,
		-1455489500, -1630486926, -7115183, -839025857, -725714386, -1331852398,
		-1310642483, -1785995686, -1519011546, -273873292, -1345501619, -261646705,
		-23110767, -1663529679, -924461307, -1635027286, -1863191439, -1451066530,
		-1630766659, -1168826102, -1961241199, -1576305452, -1469514634, -536422233,
		-287701423, -543441744, -527355702, -1171398985,
	}
	state.Seed(84)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Int64(-1<<31, 0)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(999); [state.randint(0,1<<31) for _ in range(100)]
	fromNumpy = []int64{
		1303213504, 648491356, 118207333, 2129740257, 511579848, 922722529,
		599927259, 386407839, 390521049, 804619187, 1426898448, 254654445, 1835587189,
		1099784968, 233585288, 772563120, 550299205, 1045734978, 847796235, 2117402635,
		1245324437, 2044257239, 566464740, 45667066, 1472187878, 1046578482, 865691966,
		86866247, 890450790, 531089297, 143448766, 1627961829, 1757715571, 158804326,
		1740151796, 1784518187, 1118561824, 513479273, 2034776708, 1355844109, 1231448363,
		1023174740, 1079293000, 1741413351, 416998126, 513071387, 1771063218, 2020134935,
		1209630132, 734566484, 1676230277, 2145898795, 1593901905, 1948230623, 348945148,
		212852662, 248812458, 1525746666, 204158168, 1746543929, 1426801329, 1025677355,
		2032674049, 1659154666, 1229546481, 1765385422, 35465670, 886836995, 635443571,
		575167432, 843677548, 603107849, 1463810810, 2138371435, 16119886, 1740070928,
		1213847999, 1580413567, 1432275889, 14120762, 1880797727, 202916142, 2029808692,
		2105514700, 1429464078, 63090156, 282684464, 98305286, 1734275844, 18737751,
		1609725763, 876576180, 32218047, 931063065, 2052465899, 1243693175, 1370052523,
		421951454, 805483508, 810431131,
	}
	state.Seed(999)
	for i := 0; i < 100; i++ {
		numbers[i] = int64(state.Intn(1 << 31))
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(999); [state.randint(0,(1<<60) - 10) for _ in range(100)]
	fromNumpy = []int64{
		985573361606668636, 507696633659605473, 1044297212768526561,
		270824948556635551, 524353630052385715, 363874646093576173, 966257921317757704,
		1003241173559304368, 57674082469324354, 182492591293792267, 736941713441481687,
		127104525616599802, 558391267589981490, 259354168646269255, 365692508457912209,
		616107760249518565, 631801865632688486, 556366030186583595, 192500434720199273,
		668848886777940493, 677344430541006932, 23842123263237095, 638071811716995355,
		689129574837571607, 583635841650825300, 281825196932321579, 1081149036068578271,
		345786494363885494, 1068641371473120234, 876852658665301305, 363457523935778859,
		659818035441610986, 669175908144882894, 152323895815048963, 423366349407411144,
		164802963812037129, 522412035585406315, 69234384925319184, 601751443320550015,
		386970581231695674, 7514195811123502, 647511419134122700, 374893944993132012,
		61199025606231302, 531129004600453719, 1149111985455988148, 138375461284537625,
		744823383503487095, 119723257475004382, 760813464728219, 194578470907668852,
		1062731100845608487, 1136285328640490142, 832522939414883263, 794919625444963659,
		833467421569196804, 210139446049690552, 665443294671645006, 202972372447772219,
		452114442418257647, 740740317397480591, 1151980970831634503, 306698391028582512,
		678177886857409791, 385728481545834389, 701815967392886702, 353425416072798111,
		314861907771875183, 1011470953704266938, 184529915437511266, 212257564667705339,
		846029058392140643, 212989849129508674, 205688158248276755, 1102539287222379532,
		30079908690307457, 488868184377077413, 904388831225112971, 377934374090316401,
		126778098278560834, 271857906299531608, 649205888888739248, 967403490038220901,
		423495115321404439, 464293043144407109, 129734458599681725, 9627747890128734,
		1046366152860180964, 347754664926644360, 1104485654346920916, 593742220966272537,
		165017338713855666, 796208072377654984, 1079256488404811411, 394472078074915737,
		192618330110572681, 510511550966546207, 1115541434575891517, 435220280478597263,
		825240961408083308,
	}
	state.Seed(999)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Int64n((1 << 60) - 10)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(1234); [state.randint(-1<<31,1<<31) for _ in range(100)]
	fromNumpy = []int64{
		-1324913873, -10034477, 524453158, 1365105717, -267457332, 481516916,
		1225605784, 1165481977, 1202486927, 1549064881, -976707305, -1500502807, -960078694,
		-1294852066, 1296531115, 1353614494, 1967693548, -1465376900, 1614618394,
		-1648675466, -610670150, -2092046216, 4274032, -56549924, 787967301, -725783216,
		913548233, 1299827071, -557268746, -1725494786, 262835631, -1906993614, 13242096,
		-246262021, -2088348596, -2052376283, 1171781410, -898814042, 1643431418,
		-1089227581, -580310297, 1023433867, 495622784, 1671713513, -1823723701, 2092247376,
		-563396605, -1643068158, 1860322579, -456201332, 650164161, -203023925, -441511565,
		163843691, 1240086522, 1248212433, -786682866, -146731789, 292481479, -277745860,
		1585390058, 298408555, -274133074, 2015452060, 1297714257, -1973296297, -1530009843,
		206671988, 877294176, -160732276, 878670021, -530547715, -1207778703, -739111969,
		1824792549, 1346596511, -248503559, 629437436, 1757998681, -1943788460, -1890605011,
		2125826576, -1355976663, 812760904, -1944094280, 1844885740, 751107910, 1795796920,
		406410326, 2041405356, 143066074, -442373165, -1961408209, -1019513167, 263853074,
		-299997611, -731568466, 1136151010, 12742437, 428340589,
	}
	state.Seed(1234)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Int64(-1<<31, 1<<31)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(5678); [state.randint(-1<<62,1<<62) for _ in range(100)]
	fromNumpy = []int64{
		4414803568701470871, -3517195591178331246, 2143556465413097660,
		-4263680085131202307, -2799754469294632005, 3331749269463548063,
		-1316469376591590917, 650319681590087324, -3293875846254817521,
		-1204326396348558574, -2991546811827422339, -249610632523407040,
		-997624261146098477, 338332485979828751, -2075226260105136439, 856039155152523027,
		704531153692580888, 3078339553711573875, 712825593256167697, -2712134911757710223,
		3948708359832824309, -2086988552349359484, 4548125177112172322, 2667015340737807948,
		295790952562171881, -4268667518331709291, 1087796453590441589, -2201598230368574736,
		4308833325705366869, 3363509956910651886, -2411068331937295537, 460732619842180835,
		-2157416304997974473, -2028481099060538354, 917540470378496166,
		-1684860231456972856, 76777954168317433, -1982206562030750566, -1972781884142668845,
		410356051369168444, 3697982700245596759, 3907652030970494697, 3288585369142192908,
		2453248904467261911, -3252378920128550435, 1704135006835676901, 1022158508934051762,
		2149905273599743759, 4507537476424103854, -799215112688900954, 1197024621235705935,
		-2803081135463896219, 2692888393710604883, 3448022935336152472, 4479528627981058270,
		4245665342994048136, 2948896949706593217, -2449031627084370056, 224961478044949798,
		280769733153043695, 268591986528936111, 2650288621942662112, -3638546037447351518,
		2991541655020129410, 610073345447080706, -129588347920110307, 3253536686350629279,
		1836964314698344082, -4549139490522740929, -2688909701444867400,
		-2143978792506091669, -2551281432225897070, 1226871008445564645, 998386894459336,
		2534656328909165502, -2676733702653800013, -2011717662971794307,
		-3337556253839714968, -3793214837869184322, -4259348541758052036,
		927939317885344147, 1009661561780181160, -4392432387037735977, 1253893275397985437,
		-2548923824814692851, -234070172357999423, -2298728710066902935,
		4284768302329391453, -2319889914252708470, -97128387371406688, -633573805019059858,
		3695348830848024652, 3097801301592545549, 3513515901684684718, 44816373893023992,
		3015930157326236929, -1581234701281773930, -1796348888572582172, 997999778645771124,
		-2355187993426686211,
	}
	state.Seed(5678)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Int64(-1<<62, 1<<62)
	}
	assert.DeepEqual(t, fromNumpy, numbers)
}

func TestUniform(t *testing.T) {
	if runtime.GOARCH == "arm64" {
		t.Skip("doesn't work on arm64")
		return
	}

	state := State{}
	numbers := make([]float64, 100)

	// state = np.random.RandomState(1); [state.uniform(0, 1) for _ in range(100)]
	fromNumpy := []float64{
		0.417022004702574, 0.7203244934421581, 0.00011437481734488664,
		0.30233257263183977, 0.14675589081711304, 0.0923385947687978, 0.1862602113776709,
		0.34556072704304774, 0.39676747423066994, 0.538816734003357, 0.4191945144032948,
		0.6852195003967595, 0.20445224973151743, 0.8781174363909454, 0.027387593197926163,
		0.6704675101784022, 0.41730480236712697, 0.5586898284457517, 0.14038693859523377,
		0.1981014890848788, 0.8007445686755367, 0.9682615757193975, 0.31342417815924284,
		0.6923226156693141, 0.8763891522960383, 0.8946066635038473, 0.08504421136977791,
		0.03905478323288236, 0.1698304195645689, 0.8781425034294131, 0.0983468338330501,
		0.42110762500505217, 0.9578895301505019, 0.5331652849730171, 0.6918771139504734,
		0.31551563100606295, 0.6865009276815837, 0.8346256718973729, 0.018288277344191806,
		0.7501443149449675, 0.9888610889064947, 0.7481656543798394, 0.2804439920644052,
		0.7892793284514885, 0.10322600657764203, 0.44789352617590517, 0.9085955030930956,
		0.2936141483736795, 0.28777533858634874, 0.13002857211827767, 0.019366957870297075,
		0.678835532939891, 0.21162811600005904, 0.2655466593722262, 0.4915731592803383,
		0.053362545117080384, 0.5741176054920131, 0.14672857490581015, 0.5893055369032842,
		0.6997583600209312, 0.10233442882782584, 0.4140559878195683, 0.6944001577277451,
		0.41417926952690265, 0.04995345894608716, 0.5358964059155116, 0.6637946452197888,
		0.5148891120583086, 0.9445947559908133, 0.5865550405019929, 0.9034019152878835,
		0.13747470414623753, 0.13927634725075855, 0.8073912887095238, 0.3976768369855336,
		0.16535419711693278, 0.9275085803960339, 0.34776585974550656, 0.7508121031361555,
		0.7259979853504515, 0.8833060912058098, 0.6236722070556089, 0.7509424340273372,
		0.34889834197784253, 0.2699278917650261, 0.8958862181960668, 0.4280911898712949,
		0.9648400471483856, 0.6634414978184481, 0.6216957202091218, 0.11474597295337519,
		0.9494892587070712, 0.4499121334799405, 0.5783896143871318, 0.40813680276128117,
		0.2370269802430277, 0.9033795205622538, 0.5736794866722859, 0.00287032703115897,
		0.6171449136207239,
	}
	state.Seed(1)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(0, 1)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(2); [state.uniform(-1, 0) for _ in range(100)]
	fromNumpy = []float64{
		-0.5640050978579962, -0.9740737681721087, -0.45033752212129086,
		-0.5646776073817231, -0.579632197912511, -0.6696651789961259, -0.7953513659621575,
		-0.3807290336493363, -0.7003453263254769, -0.7331727248971334, -0.3788661672307051,
		-0.47085790572296093, -0.8654200546550664, -0.4864218787342536, -0.8155601343530847,
		-0.21466485218332654, -0.1460247073605112, -0.5057631626180722, -0.153438514642532,
		-0.920354522990939, -0.49475390987829604, -0.9347134956131219, -0.5718776724026106,
		-0.9034690843393874, -0.8728400282987225, -0.4032546910214042, -0.7739879993957641,
		-0.893054315690017, -0.7796937929294403, -0.6501737149967007, -0.5322125154176998,
		-0.7982567737350347, -0.3595932747850852, -0.5169301644482484, -0.49476327998145087,
		-0.6131073488814407, -0.20636254555842293, -0.41999582112219336,
		-0.8377014014976861, -0.29924765339284376, -0.035448919910744836,
		-0.49999163882978326, -0.11047993605385509, -0.6583863473288903,
		-0.43285587237229073, -0.57245403670389, -0.563252736973201, -0.2234408150028997,
		-0.4643958265023437, -0.046257773055133344, -0.45579183985189786,
		-0.9179050777249752, -0.6336575983249796, -0.14914949599549832, -0.5937249569520492,
		-0.9727976341051504, -0.7528227610026467, -0.9328556292630453,
		-0.006147988578727115, -0.029419686622826458, -0.1997416488674132,
		-0.3981828785945326, -0.23504013954831848, -0.8307745534158221, -0.7069767681805602,
		-0.4759331246996358, -0.6433757188772411, -0.9543210347546, -0.01684655464278728,
		-0.5586450806451533, -0.4959995606208474, -0.6764586824653012, -0.740255247252863,
		-0.6131101146792376, -0.16798310036549913, -0.2632529437128871, -0.6207894330583625,
		-0.9869826633054494, -0.20259506096740643, -0.7306112024036988,
		-0.41731511147096545, -0.9744490581503072, -0.3377979807316742, -0.6124765741267185,
		-0.5029262012654874, -0.5850941626253128, -0.6491280986983595, -0.44902209467558074,
		-0.027089310125175126, -0.8872237847619285, -0.6867414715702864,
		-0.9582022901846899, -0.26160024137909965, -0.34248761234095604,
		-0.7853642536000464, -0.5832465598046629, -0.356158065851805, -0.33851867328455865,
		-0.8295228665162293, -0.11834776425145144,
	}
	state.Seed(2)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(-1, 0)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(3); [state.uniform(-1, 1) for _ in range(100)]
	fromNumpy = []float64{
		0.10159580514915101, 0.41629564523620965, -0.41819052217411135,
		0.02165521039532603, 0.7858939086953094, 0.7925861778668761, -0.7488293790723275,
		-0.5855142437236265, -0.8970655933983402, -0.11838031269872706, -0.9402475782428661,
		-0.08633355121057784, 0.2982880952295215, -0.4430254347040494, 0.35250980396026255,
		0.18172563483270165, -0.9520362352456693, 0.11770817598176397, -0.4814951061850692,
		-0.1697976059798607, -0.4329498364573625, 0.38627583662599263, -0.11909256465852103,
		-0.6862645230500735, 0.08929803606368947, 0.5606295290227341, -0.3872729352476405,
		-0.5560842321356372, -0.22405748488870247, 0.8727672997208609, 0.9519908449458676,
		0.34476735182562734, 0.8056682170767961, 0.6915017425863585, -0.24401191734222172,
		-0.8155659822565515, 0.30682180504707235, 0.11568152477690186, -0.27687047387484376,
		-0.5498909903203362, -0.18696016724750275, -0.06211950260166188,
		-0.4615288434978775, -0.4164144515491348, -0.08462720135404411, 0.721067825893853,
		0.17250580893564282, -0.4330242751355804, -0.44404498675279735,
		-0.09075584895610644, -0.589179310790025, -0.5972425779138537, 0.028070119634883817,
		-0.8255412624383907, -0.032828935296826955, -0.27564757522433325,
		0.4153732436313422, 0.4934924462040091, 0.38218584371152864, 0.37836082750982825,
		-0.2527997517021434, 0.3362696094705351, -0.3203026720863642, 0.14558773973775008,
		-0.34838568387025104, -0.10970990147293591, -0.8769421383401215,
		-0.5146491556313237, 0.9432052123471353, -0.538831591289852, 0.38295502189466046,
		0.30095371642747915, 0.44787827887496423, -0.049822778731137385,
		0.19332754920372563, -0.8660611521343473, -0.8548757243555332, -0.6020479474058433,
		-0.6962780059829179, -0.7997913108371106, -0.7414122707822917, 0.10655546361530699,
		-0.6243703501522435, 0.9042024860624165, 0.36322355761485325, 0.08203934671954238,
		0.41436120161579537, -0.4722266580949732, 0.8534513685074556, 0.6783861155999413,
		0.45263899685607445, -0.03952008782917993, 0.6842063717532936, 0.4895046468487185,
		0.32065181279922506, 0.8279505340118756, 0.26733112757514954, -0.26811883029478123,
		0.1056891463008911, -0.6072388465997263,
	}
	state.Seed(3)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(-1, 1)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(4567); [state.uniform(0, 1.79e308) for _ in range(100)]
	fromNumpy = []float64{
		6.479293859233999e+307, 1.7621364278287492e+308,
		1.5558618834026866e+308, 1.2465238058185557e+308, 1.2664086344362704e+308,
		1.7596616585500137e+308, 4.640658285450111e+307, 2.4767665909316927e+307,
		5.121844149340489e+307, 4.730234674475894e+307, 1.4820632038969078e+308,
		8.108993809332149e+307, 1.217540281067502e+308, 1.6334087248914977e+308,
		2.687352544864222e+307, 1.1420001578939304e+307, 2.2995459342747167e+307,
		1.5789917711351067e+307, 1.430841682309823e+308, 5.714822407186646e+307,
		4.337774574696178e+307, 1.4310326917447495e+308, 1.0862129890982025e+308,
		1.592020263855846e+308, 2.817591201143198e+307, 2.161111839617282e+307,
		3.7381132111526487e+307, 4.3690334663253044e+307, 1.0601245888381637e+308,
		1.5830206481704367e+308, 1.4766185014048194e+308, 6.241980558104968e+307,
		4.655662876322328e+307, 1.0285210961417625e+308, 6.834868936315017e+307,
		1.6340701870874065e+308, 1.6718624264052181e+308, 1.081739278213566e+308,
		1.343300390345683e+308, 7.301846007809659e+307, 8.23944175006236e+305,
		1.7303724569825985e+308, 4.580817258547465e+307, 4.4463873423831547e+307,
		3.197705190209258e+307, 1.1485650042794282e+308, 1.3478133157601969e+308,
		1.1941155004403067e+308, 5.040977917734049e+307, 8.089950068250968e+307,
		8.488174438838402e+307, 1.452270410623104e+308, 1.2069054796824422e+308,
		5.163199853551888e+306, 1.7153971187069458e+308, 1.6120515830476463e+308,
		1.1377345571435526e+308, 6.658262195366134e+307, 2.9599993872356824e+307,
		1.5831122537446013e+308, 6.279582428532856e+307, 1.2348432261848676e+308,
		1.0435316009674476e+308, 4.1931095069621273e+307, 2.1127513334957228e+306,
		1.0453466985067643e+308, 1.9513623851748138e+307, 1.183452325170894e+308,
		4.62491903123592e+307, 9.472089582928704e+307, 1.7306785122564235e+308,
		1.3770427384505955e+308, 1.5641039371103278e+308, 1.1269993757512107e+308,
		3.610062998238717e+307, 9.421850937763314e+307, 5.981811494917679e+307,
		1.5558093608359316e+308, 1.561234301401435e+308, 1.296260274786705e+308,
		1.1930592443620075e+308, 7.772611806476275e+307, 2.0904807834935382e+307,
		6.604808196510698e+307, 8.292773069340692e+307, 6.395527495985944e+307,
		4.210923377790351e+307, 4.30953839867978e+307, 8.966735974187659e+307,
		1.7032418042780697e+308, 4.382992388824878e+307, 8.419916792510756e+307,
		2.9274289658914e+307, 7.320137441411957e+307, 1.753950949388129e+308,
		1.2144919408931684e+308, 7.448959537760188e+307, 1.1392406304957446e+307,
		9.246361076904628e+307, 1.7435837702205577e+308,
	}
	state.Seed(4567)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(0, 1.79e308)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(45678); [state.uniform(-1.79e308, 0) for _ in range(100)]
	fromNumpy = []float64{
		-4.2674056563823797e+307, -1.3074520956497302e+308,
		-1.4502807587073843e+308, -8.190131839321301e+307, -1.284438625211133e+307,
		-1.1421426785155382e+308, -7.825108326048907e+307, -8.284118961322912e+307,
		-6.477604666500911e+307, -6.846647217224101e+307, -4.706681614992132e+307,
		-1.5081132560608188e+308, -2.4185905828726717e+307, -6.67789975257644e+307,
		-1.017897704059691e+308, -1.3121604889443585e+308, -1.5543759216889496e+308,
		-1.4441729601490756e+308, -1.3096634311882269e+308, -9.440157820255456e+307,
		-3.060239307128759e+306, -1.1450797469257329e+308, -1.2136743972307815e+308,
		-3.760233229433685e+307, -4.085364329176201e+307, -7.641778665828278e+307,
		-1.5160698384689026e+308, -1.6673765475602206e+308, -9.160966833535706e+307,
		-1.6988878215090043e+308, -1.169578306861913e+308, -5.711943802110944e+307,
		-1.674834741475159e+308, -4.397316009442147e+307, -1.0877984071269499e+308,
		-5.650114405735717e+307, -1.69738027094174e+308, -8.657631420851135e+307,
		-6.105364783259194e+307, -1.1608864740451322e+308, -1.5852951260513606e+308,
		-1.376648261019796e+308, -1.7700161005860308e+308, -1.1875373486074923e+308,
		-1.7752303894771671e+308, -1.010582258434249e+308, -1.6236220383293869e+308,
		-1.5939456104532676e+308, -1.700985979450889e+308, -1.2264815987389536e+308,
		-4.491421392763089e+307, -5.172652110497964e+307, -1.69947400770436e+308,
		-1.6529863675821683e+308, -1.222444934440661e+308, -8.89138381495854e+307,
		-2.2018203806803574e+307, -1.2562464422650867e+308, -9.354300346931186e+307,
		-7.124673637608335e+307, -1.3646668372541863e+308, -2.2251610855944987e+307,
		-5.323358541917913e+307, -1.3215390380364379e+308, -6.972639517141035e+307,
		-7.420553942816136e+307, -9.358156062940017e+306, -1.4741233006157312e+308,
		-1.278441416483161e+308, -6.357245077791184e+307, -1.6182782308594372e+308,
		-7.362262468592758e+306, -1.1087741461986922e+308, -4.890338982015792e+307,
		-2.5402427403425164e+306, -1.3684589912232728e+308, -6.134845388872928e+307,
		-1.550031717088217e+307, -1.7855340922874402e+308, -1.319398539218355e+308,
		-5.894346613421058e+307, -3.039056289362901e+306, -4.863387584319833e+306,
		-8.387883584212694e+307, -1.5146631188587e+308, -1.2217936440645403e+308,
		-1.4866722223821753e+308, -1.3190990758316811e+308, -8.25456123108217e+305,
		-1.7455043463574984e+308, -3.319211448359854e+307, -8.820972108924404e+307,
		-4.548183219976137e+307, -1.1347843394084336e+308, -5.619057285296284e+307,
		-8.5120476306133e+306, -1.1944791024598815e+308, -3.897673198326657e+307,
		-9.74906175577547e+307, -8.477623785137812e+307,
	}
	state.Seed(45678)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(-1.79e308, 0)
	}
	assert.DeepEqual(t, fromNumpy, numbers)

	// state = np.random.RandomState(456789); [state.uniform(-8e307, 8e307) for _ in range(100)]
	fromNumpy = []float64{
		4.933395252151545e+307, 2.870311843912131e+307,
		-4.694530750584326e+307, 3.7108725123943595e+306, 7.41519531335662e+307,
		-7.335976971093971e+307, 1.287122765485954e+307, -2.690680304678878e+307,
		-3.7805396085215606e+306, -6.478630111711071e+307, -4.980627934744884e+307,
		-5.69295861734556e+307, -2.3379195190313664e+307, 6.72847584264828e+307,
		5.771103367074679e+306, 1.5193197432018074e+306, -6.68179173258199e+307,
		-1.295025920015057e+307, 5.675215371096981e+307, 2.2552592089179504e+307,
		-5.40801282740245e+307, 3.7810987860823145e+307, -4.660750006085056e+307,
		7.200549354703756e+307, 5.532631883047663e+307, -7.66098627974069e+307,
		7.762613699648142e+307, 5.531033749335768e+307, -1.4211005674378598e+307,
		1.9701925031039228e+307, -3.3786204249823136e+307, 6.018264944986112e+307,
		1.3371079711313676e+307, 2.3901492356155946e+307, 5.917418300683425e+307,
		6.812100788423331e+307, 2.8302826095968736e+307, 2.1864535488053306e+307,
		-2.1374242141009568e+307, 2.5792188707850927e+307, -5.124611035346168e+307,
		-6.428894004741567e+307, 2.000359247644964e+307, 1.3281022202920244e+307,
		2.0304924750351886e+307, 5.710174937750734e+307, 1.5732786343363795e+307,
		3.271896291899537e+307, -7.003142690861068e+307, 6.993715889043624e+307,
		-5.220765152030427e+307, 2.3171687769174634e+307, 4.72383679388924e+307,
		-6.77437998755628e+307, -4.670036576195282e+307, 4.371664550281262e+306,
		-2.744195447990275e+307, -2.6122830339661347e+307, -6.258525482733317e+307,
		-2.9050571114597506e+307, -1.9328250511050345e+307, 1.2293954571849298e+307,
		7.070507991887099e+307, 6.661290109307938e+307, 4.0047301333171193e+307,
		1.7329019690900744e+307, 6.398812023709227e+307, -7.053082330542008e+306,
		-3.9619950474572964e+307, -3.8209492899389817e+307, 5.3740392627313715e+307,
		-8.849484326546386e+306, -4.8635498365235845e+306, 5.241063941123018e+307,
		-2.462087937615219e+307, 7.714901549913985e+307, -1.0898353202755277e+307,
		3.332272046478825e+307, -6.806540404861073e+307, -4.892544785564114e+307,
		-2.6649111651376426e+307, 6.85234206154708e+307, 5.832816645758717e+307,
		-7.682350698490775e+307, 7.13859377527557e+306, -4.933255098127484e+307,
		4.26810351774149e+307, 6.322959336772563e+307, 9.500676975226379e+306,
		1.976943487413165e+307, -2.816348669603217e+307, -6.191825491496837e+307,
		1.0407273583635438e+307, -7.57876440408812e+307, 1.728305740135286e+307,
		6.379579456988475e+307, 3.5784892066286013e+307, 7.445704652601432e+307,
		-7.025928828016209e+307, -6.811420153811989e+307,
	}
	state.Seed(456789)
	for i := 0; i < 100; i++ {
		numbers[i] = state.Uniform(-8e307, 8e307)
	}
	assert.DeepEqual(t, fromNumpy, numbers)
}

// TestChoice tests the if naive implementation of `numpy.random.choice` reproduces the original.
func TestChoice(t *testing.T) {
	state := State{}
	choice := func(choices []int) int {
		return choices[state.Intn(len(choices))]
	}
	choices := make([]int, 100)
	vals := make([]int, 100)
	for i := range vals {
		vals[i] = i
	}

	// state = np.random.RandomState(123); [state.choice(list(range(100))) for _ in range(100)]
	fromNumpy := []int{
		66, 92, 98, 17, 83, 57, 86, 97, 96, 47, 73, 32, 46, 96, 25, 83, 78, 36,
		96, 80, 68, 49, 55, 67, 2, 84, 39, 66, 84, 47, 61, 48, 7, 99, 92, 52, 97, 85, 94,
		27, 34, 97, 76, 40, 3, 69, 64, 75, 34, 58, 10, 22, 77, 18, 15, 27, 30, 52, 70, 26,
		80, 6, 14, 75, 54, 71, 1, 43, 58, 55, 25, 50, 84, 56, 49, 12, 18, 81, 1, 51, 44, 48,
		56, 91, 49, 86, 3, 67, 11, 21, 89, 98, 3, 11, 3, 94, 6, 9, 87, 14,
	}
	state.Seed(123)
	for i := 0; i < 100; i++ {
		choices[i] = choice(vals)
	}
	assert.DeepEqual(t, fromNumpy, choices)
}
