package internal

import (
	"testing"

	"github.com/stretchr/testify/require"

	"github.com/determined-ai/determined/master/pkg/ptrs"
	"github.com/determined-ai/determined/master/pkg/schemas/expconf"
	"github.com/determined-ai/determined/master/pkg/searcher"
)

func TestShimExperimentSnapshotV4(t *testing.T) {
	//nolint:lll // Much more readable this way, than a hundreds of lines of formatted JSON.
	snapshot := []byte(`{"searcher_state": {"rand": {"key": [2970094109, 656686882, 618108684, 3428065983, 3347811667, 2888225350, 3059306387, 3429410465, 120474970, 2272301777, 985108865, 991558874, 3272543769, 1573748485, 942809215, 3888215743, 2210951765, 1718115507, 3963921664, 3557444060, 1499923783, 1829703377, 750493200, 3411092685, 589595500, 2596144409, 1879328096, 1280550458, 2715466210, 2544141428, 1543312021, 2997818084, 765128503, 536629897, 1001278031, 956821445, 2736363088, 947342293, 3898485884, 3425929255, 356849665, 3772908806, 3623557158, 2612581302, 3580922597, 217692149, 2059015628, 2096894728, 1031554394, 2656443257, 856730792, 2596152358, 3710976744, 3544672276, 1734608805, 481137749, 969621572, 754645566, 4116998116, 79516852, 3779707014, 84449565, 2299776967, 3049981717, 209980308, 730595889, 475796649, 176283770, 945344397, 3533051341, 1815446302, 3773966474, 2608085762, 3988702430, 1364580973, 954047394, 3612132653, 772541827, 644272441, 189803165, 931375382, 345347493, 1070927534, 1233280688, 2977342360, 964860546, 311939975, 4227569059, 1111154497, 2670796852, 2936307495, 1340846895, 2187526044, 4049656992, 2523649560, 3366534695, 4129435744, 2210183795, 1260247727, 3172172609, 845427714, 2412303263, 4029524955, 496281340, 1521525374, 1068147028, 715985502, 4093783277, 2159512480, 133552717, 436985375, 3034348399, 1875037974, 4060219881, 2429216519, 1706838315, 3704003030, 1782549491, 3768045061, 3989593374, 93865362, 1767865857, 1137597591, 252302268, 3084248212, 619916972, 4054361685, 2046158286, 1812194877, 3286982519, 3624326839, 2208625006, 3161233673, 3006516503, 2024883981, 1656495788, 587227161, 3021529118, 3172394998, 3398906615, 615744671, 3472224112, 212954520, 1118562041, 1921307781, 1197366600, 1484195533, 1200554730, 729716373, 1652264122, 1007315030, 2434524607, 585522965, 1141433277, 278327678, 3497727325, 356102370, 1394639384, 1857470125, 735035053, 2157227949, 2119739569, 171894210, 839804861, 1047866560, 420010537, 1667233883, 352520083, 1757099499, 2384088069, 251603325, 4218299275, 3730818711, 1003235663, 299940149, 2645812204, 2480727500, 1622974487, 1629312940, 3982198086, 2956489407, 2625953102, 1319536795, 42594099, 1983096735, 973960638, 3621842015, 3277003271, 2353057652, 181048897, 971058611, 596055263, 916085615, 181371615, 1523808158, 1627674241, 1944135612, 1287953114, 427398813, 3657156653, 2038254583, 1728892774, 2979810975, 1203810426, 870115982, 2342890482, 239526972, 161374077, 1298609642, 3361627384, 1120507760, 2857852335, 1817591908, 2278524994, 2381282828, 390752450, 2583666345, 2894166573, 1090621184, 1216468230, 2471775878, 4007164892, 4120390730, 2245347208, 1747432849, 2659194233, 3123421709, 1394627981, 3642599365, 447037831, 2948358439, 2071590290, 620060883, 1189294322, 865221550, 1072574172, 4239383229, 3932654708, 977824870, 4090083517, 996564545, 2215368799, 1639088756, 1063294874, 2589869379, 2931882145, 1119478531, 2913543435, 2863960600, 2220278034, 1514566588, 2171894117, 1040890154, 613591648, 4213726618, 3733950164, 3557521554, 451572785, 1963683489, 2910238750, 1724590625, 2276900333, 2786477774, 194531157, 2427295372, 2668452276, 3024832162, 4118825874, 2048453723, 602839057, 1287626269, 391207769, 1490752784, 631791025, 3012581026, 2165558496, 711793093, 394391602, 3030359078, 753489044, 1912791312, 805796394, 3405161241, 2387156388, 3012369288, 785079410, 1579291678, 2186594177, 1466653116, 2597054161, 2032480878, 2608593331, 245623596, 4030340391, 4102274438, 1820069747, 861878589, 1987396926, 1234162528, 4203068184, 4247911765, 675763300, 776602741, 2725860139, 1101969549, 2417348663, 3640520171, 4117232477, 1498968755, 1465742592, 978609897, 207663660, 3872867123, 2345883885, 3859129626, 2085742042, 2849699660, 766128209, 1220321488, 4036832842, 2625520262, 1672441713, 68307500, 1664835655, 4277760737, 2834495524, 1659112114, 4058377674, 3503936154, 3460939589, 1534417391, 546891454, 2119405354, 1342659996, 4063240713, 2470735676, 4263599371, 1004146457, 4110714736, 161171862, 3755105332, 2045654769, 3373580076, 3804959282, 172165195, 1513507373, 756508282, 2395802730, 1042880862, 424954450, 3528754016, 705074650, 1657675167, 3512260741, 1502017722, 3914739402, 2193244645, 2422500613, 99094108, 2136256596, 2765564582, 1469408647, 422762387, 4256532504, 2587902689, 322034733, 869598360, 1591826706, 3537194108, 432319981, 3238387140, 3998122956, 1638675667, 2820632381, 2704736212, 1001829652, 3842914244, 2163956861, 1247332102, 4161816793, 1047179830, 3332934466, 3373357404, 1755466786, 2066857771, 3041960543, 1984594045, 2873866111, 625922821, 2084260334, 647387503, 1795267011, 2734649093, 1429118207, 1763491842, 2499417212, 1322753865, 2546278618, 3069808828, 2758040961, 2095345089, 1117531393, 1079149953, 199028389, 5727143, 2626507809, 1394644924, 2371859674, 3378009021, 663183580, 1964126253, 4253076758, 1951862122, 3318205484, 246506769, 2065283265, 4062893977, 1357669276, 1910189941, 1768248614, 2642642591, 3082965658, 2821926795, 3387815148, 3811042618, 47443245, 1219528119, 2554982004, 3265451419, 2888244446, 4219157977, 970880074, 1607338684, 3191412771, 2649941819, 3075671217, 1402753481, 4075173340, 1295557797, 911791588, 847594689, 2359505701, 3661496760, 733404895, 249867897, 1640333750, 582965187, 1381547711, 2657453162, 1011026771, 2180307770, 52701968, 2169192963, 3746362727, 509530176, 4097693696, 406641367, 2522117714, 2016171603, 2069773173, 3084060900, 1332397153, 1412022588, 1875059352, 3676296777, 3375212542, 1306176142, 133024327, 2608708004, 139194617, 519502757, 1353538952, 3608390217, 3046145872, 1399364698, 4125351745, 572217286, 2142672378, 2889903141, 3434412210, 2393320012, 3956475466, 562179437, 3542566562, 3807446953, 1580557399, 3614175239, 3960865405, 3703717660, 2310530500, 1372501162, 3987983520, 2273295537, 3041676882, 1184238826, 1347025024, 1978741180, 1085872423, 3721856454, 4017856803, 3452381717, 2476724512, 3844971453, 3783643616, 1806640004, 945069700, 146817685, 3591154906, 3316795159, 3055211927, 3538095801, 2794589177, 2749831736, 3448467389, 2625023521, 183144251, 1181880302, 2668701387, 1211953638, 2740205019, 1424192308, 3005227360, 3518579311, 2030094765, 2163439322, 528876425, 713989960, 49680970, 3843683734, 2284197334, 754118913, 3660862775, 2811591585, 1900142815, 909665247, 3587945731, 1738555018, 4017103008, 363980482, 3596395463, 4036688775, 562009149, 652191781, 3255386890, 742878000, 9948449, 1293599923, 3198724758, 1824688996, 971993200, 545911088, 3111974979, 548678661, 3540201872, 347949716, 3899497825, 3919059349, 4023520489, 3906867502, 2926287461, 1975024135, 1961216548, 4241655703, 882861834, 447828602, 1073917811, 3204969134, 3637120523, 2561353660, 3087651888, 2625085299, 3045978593, 3296945077, 3353240266, 264317791, 3618427144, 3041663648, 3129441565, 1614617882, 742204376, 107307162, 4015043056, 164246162, 2971078786, 3420066832, 1704293461, 459244372, 744914593, 710366705, 2599447851, 3652089701, 712803605, 2980013293, 476563708, 1665840194, 1037190002, 3225477896, 1430942116, 2632665368, 3730881422, 1625879865, 363179597, 3829523707, 14110766, 1150504976, 2891936857, 256611346, 774302498, 184441060, 803067320, 3354307461, 2067530913, 466995748, 1177075098, 4068208859, 1923629384, 3949860772, 3830641659, 1017128665, 2972991357, 4238922926, 575232160, 101928834, 2405116568], "pos": 5}, "exits": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": 0}, "trials_created": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": true}, "trials_requested": 1, "search_method_state": {"created_trials": 1, "pending_trials": 1, "search_method_type": "single"}, "completed_operations": {}}, "trial_searcher_state": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": {"Op": {"Length": {"batches": 937}, "RequestID": "bb239c3a-c6b5-4ef3-ba7b-118038bd0d06"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1, "global_batch_size": 64}, "checkpoint": null, "request_id": "bb239c3a-c6b5-4ef3-ba7b-118038bd0d06", "trial_seed": 1757369869, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`)
	//nolint:lll // Same.
	newSnapshot := []byte(`{"searcher_state": {"rand": {"key": [2970094109, 656686882, 618108684, 3428065983, 3347811667, 2888225350, 3059306387, 3429410465, 120474970, 2272301777, 985108865, 991558874, 3272543769, 1573748485, 942809215, 3888215743, 2210951765, 1718115507, 3963921664, 3557444060, 1499923783, 1829703377, 750493200, 3411092685, 589595500, 2596144409, 1879328096, 1280550458, 2715466210, 2544141428, 1543312021, 2997818084, 765128503, 536629897, 1001278031, 956821445, 2736363088, 947342293, 3898485884, 3425929255, 356849665, 3772908806, 3623557158, 2612581302, 3580922597, 217692149, 2059015628, 2096894728, 1031554394, 2656443257, 856730792, 2596152358, 3710976744, 3544672276, 1734608805, 481137749, 969621572, 754645566, 4116998116, 79516852, 3779707014, 84449565, 2299776967, 3049981717, 209980308, 730595889, 475796649, 176283770, 945344397, 3533051341, 1815446302, 3773966474, 2608085762, 3988702430, 1364580973, 954047394, 3612132653, 772541827, 644272441, 189803165, 931375382, 345347493, 1070927534, 1233280688, 2977342360, 964860546, 311939975, 4227569059, 1111154497, 2670796852, 2936307495, 1340846895, 2187526044, 4049656992, 2523649560, 3366534695, 4129435744, 2210183795, 1260247727, 3172172609, 845427714, 2412303263, 4029524955, 496281340, 1521525374, 1068147028, 715985502, 4093783277, 2159512480, 133552717, 436985375, 3034348399, 1875037974, 4060219881, 2429216519, 1706838315, 3704003030, 1782549491, 3768045061, 3989593374, 93865362, 1767865857, 1137597591, 252302268, 3084248212, 619916972, 4054361685, 2046158286, 1812194877, 3286982519, 3624326839, 2208625006, 3161233673, 3006516503, 2024883981, 1656495788, 587227161, 3021529118, 3172394998, 3398906615, 615744671, 3472224112, 212954520, 1118562041, 1921307781, 1197366600, 1484195533, 1200554730, 729716373, 1652264122, 1007315030, 2434524607, 585522965, 1141433277, 278327678, 3497727325, 356102370, 1394639384, 1857470125, 735035053, 2157227949, 2119739569, 171894210, 839804861, 1047866560, 420010537, 1667233883, 352520083, 1757099499, 2384088069, 251603325, 4218299275, 3730818711, 1003235663, 299940149, 2645812204, 2480727500, 1622974487, 1629312940, 3982198086, 2956489407, 2625953102, 1319536795, 42594099, 1983096735, 973960638, 3621842015, 3277003271, 2353057652, 181048897, 971058611, 596055263, 916085615, 181371615, 1523808158, 1627674241, 1944135612, 1287953114, 427398813, 3657156653, 2038254583, 1728892774, 2979810975, 1203810426, 870115982, 2342890482, 239526972, 161374077, 1298609642, 3361627384, 1120507760, 2857852335, 1817591908, 2278524994, 2381282828, 390752450, 2583666345, 2894166573, 1090621184, 1216468230, 2471775878, 4007164892, 4120390730, 2245347208, 1747432849, 2659194233, 3123421709, 1394627981, 3642599365, 447037831, 2948358439, 2071590290, 620060883, 1189294322, 865221550, 1072574172, 4239383229, 3932654708, 977824870, 4090083517, 996564545, 2215368799, 1639088756, 1063294874, 2589869379, 2931882145, 1119478531, 2913543435, 2863960600, 2220278034, 1514566588, 2171894117, 1040890154, 613591648, 4213726618, 3733950164, 3557521554, 451572785, 1963683489, 2910238750, 1724590625, 2276900333, 2786477774, 194531157, 2427295372, 2668452276, 3024832162, 4118825874, 2048453723, 602839057, 1287626269, 391207769, 1490752784, 631791025, 3012581026, 2165558496, 711793093, 394391602, 3030359078, 753489044, 1912791312, 805796394, 3405161241, 2387156388, 3012369288, 785079410, 1579291678, 2186594177, 1466653116, 2597054161, 2032480878, 2608593331, 245623596, 4030340391, 4102274438, 1820069747, 861878589, 1987396926, 1234162528, 4203068184, 4247911765, 675763300, 776602741, 2725860139, 1101969549, 2417348663, 3640520171, 4117232477, 1498968755, 1465742592, 978609897, 207663660, 3872867123, 2345883885, 3859129626, 2085742042, 2849699660, 766128209, 1220321488, 4036832842, 2625520262, 1672441713, 68307500, 1664835655, 4277760737, 2834495524, 1659112114, 4058377674, 3503936154, 3460939589, 1534417391, 546891454, 2119405354, 1342659996, 4063240713, 2470735676, 4263599371, 1004146457, 4110714736, 161171862, 3755105332, 2045654769, 3373580076, 3804959282, 172165195, 1513507373, 756508282, 2395802730, 1042880862, 424954450, 3528754016, 705074650, 1657675167, 3512260741, 1502017722, 3914739402, 2193244645, 2422500613, 99094108, 2136256596, 2765564582, 1469408647, 422762387, 4256532504, 2587902689, 322034733, 869598360, 1591826706, 3537194108, 432319981, 3238387140, 3998122956, 1638675667, 2820632381, 2704736212, 1001829652, 3842914244, 2163956861, 1247332102, 4161816793, 1047179830, 3332934466, 3373357404, 1755466786, 2066857771, 3041960543, 1984594045, 2873866111, 625922821, 2084260334, 647387503, 1795267011, 2734649093, 1429118207, 1763491842, 2499417212, 1322753865, 2546278618, 3069808828, 2758040961, 2095345089, 1117531393, 1079149953, 199028389, 5727143, 2626507809, 1394644924, 2371859674, 3378009021, 663183580, 1964126253, 4253076758, 1951862122, 3318205484, 246506769, 2065283265, 4062893977, 1357669276, 1910189941, 1768248614, 2642642591, 3082965658, 2821926795, 3387815148, 3811042618, 47443245, 1219528119, 2554982004, 3265451419, 2888244446, 4219157977, 970880074, 1607338684, 3191412771, 2649941819, 3075671217, 1402753481, 4075173340, 1295557797, 911791588, 847594689, 2359505701, 3661496760, 733404895, 249867897, 1640333750, 582965187, 1381547711, 2657453162, 1011026771, 2180307770, 52701968, 2169192963, 3746362727, 509530176, 4097693696, 406641367, 2522117714, 2016171603, 2069773173, 3084060900, 1332397153, 1412022588, 1875059352, 3676296777, 3375212542, 1306176142, 133024327, 2608708004, 139194617, 519502757, 1353538952, 3608390217, 3046145872, 1399364698, 4125351745, 572217286, 2142672378, 2889903141, 3434412210, 2393320012, 3956475466, 562179437, 3542566562, 3807446953, 1580557399, 3614175239, 3960865405, 3703717660, 2310530500, 1372501162, 3987983520, 2273295537, 3041676882, 1184238826, 1347025024, 1978741180, 1085872423, 3721856454, 4017856803, 3452381717, 2476724512, 3844971453, 3783643616, 1806640004, 945069700, 146817685, 3591154906, 3316795159, 3055211927, 3538095801, 2794589177, 2749831736, 3448467389, 2625023521, 183144251, 1181880302, 2668701387, 1211953638, 2740205019, 1424192308, 3005227360, 3518579311, 2030094765, 2163439322, 528876425, 713989960, 49680970, 3843683734, 2284197334, 754118913, 3660862775, 2811591585, 1900142815, 909665247, 3587945731, 1738555018, 4017103008, 363980482, 3596395463, 4036688775, 562009149, 652191781, 3255386890, 742878000, 9948449, 1293599923, 3198724758, 1824688996, 971993200, 545911088, 3111974979, 548678661, 3540201872, 347949716, 3899497825, 3919059349, 4023520489, 3906867502, 2926287461, 1975024135, 1961216548, 4241655703, 882861834, 447828602, 1073917811, 3204969134, 3637120523, 2561353660, 3087651888, 2625085299, 3045978593, 3296945077, 3353240266, 264317791, 3618427144, 3041663648, 3129441565, 1614617882, 742204376, 107307162, 4015043056, 164246162, 2971078786, 3420066832, 1704293461, 459244372, 744914593, 710366705, 2599447851, 3652089701, 712803605, 2980013293, 476563708, 1665840194, 1037190002, 3225477896, 1430942116, 2632665368, 3730881422, 1625879865, 363179597, 3829523707, 14110766, 1150504976, 2891936857, 256611346, 774302498, 184441060, 803067320, 3354307461, 2067530913, 466995748, 1177075098, 4068208859, 1923629384, 3949860772, 3830641659, 1017128665, 2972991357, 4238922926, 575232160, 101928834, 2405116568], "pos": 5}, "exits": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": 0}, "trials_created": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": true}, "trials_requested": 1, "search_method_state": {"created_trials": 1, "pending_trials": 1, "search_method_type": "single"}, "completed_operations": {}}, "trial_searcher_state": {"bb239c3a-c6b5-4ef3-ba7b-118038bd0d06": {"Op": {"Length": 937, "RequestID": "bb239c3a-c6b5-4ef3-ba7b-118038bd0d06"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1, "global_batch_size": 64}, "checkpoint": null, "request_id": "bb239c3a-c6b5-4ef3-ba7b-118038bd0d06", "trial_seed": 1757369869, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`)
	actual, err := shimExperimentSnapshotV4(snapshot)
	require.NoError(t, err)
	require.JSONEq(t, string(newSnapshot), string(actual))
}

func TestShimExperimentSnapshotV5(t *testing.T) {
	cases := []struct {
		name       searcher.SearchMethodType
		v4Snapshot []byte
		v5Snapshot []byte
		err        string
	}{
		{
			name: searcher.SingleSearch,
			//nolint:lll
			v4Snapshot: []byte(`{"searcher_state": {"rand": {"key": [3922433409, 243371046, 1078118500, 751359450, 1341787537, 4110007575, 3830714619, 215586197, 202234178, 695788744, 568122858, 801842164, 915101998, 2089585430, 560765197, 132023793, 3625985341, 1033888033, 4275540481, 4081480301, 3803198840, 2062925182, 3815214353, 2217168443, 284556261, 4216276606, 3143802969, 4077949604, 1360661190, 1579111912, 3397581176, 2605001170, 1444943786, 1103161266, 2475001171, 586405187, 440758113, 69964893, 2302367679, 2318485289, 2699422072, 949642636, 992261637, 3418960830, 2747796370, 1988242324, 1336558633, 4144877284, 1447227155, 2130144917, 249701494, 845842848, 3562805067, 461807133, 316172300, 1019162321, 2042173338, 2357208493, 984738996, 2977486791, 4204759905, 3765157938, 1416537904, 673684512, 1014036428, 3291531021, 2149014833, 211182184, 1899637415, 2967746221, 1721885906, 226438235, 2148590681, 1789583513, 1586306530, 1162436037, 2949013157, 869268556, 2754664383, 190339737, 2683355822, 1858335276, 752629074, 3123663892, 3566745314, 2416019451, 2343621935, 4064029673, 2375292477, 4237178213, 733006903, 3966381652, 1902496609, 1636160960, 1441547075, 837638138, 885960515, 1113758007, 2646826291, 1038503594, 3726531118, 1965291352, 4027030242, 3233398909, 2964755703, 966551311, 1083992523, 1124285624, 3192801484, 2731176820, 2400133886, 2523660457, 1033111323, 2914428172, 1163901245, 1320025743, 2177720309, 1677889669, 2345935772, 2794475418, 9574722, 1397363731, 3692199984, 452561608, 112544138, 2256573095, 2829353338, 1987429988, 555809442, 1666429314, 215970428, 3154343711, 2051241582, 3877008887, 1816890945, 4013344821, 2102130895, 4186655536, 2673431404, 4026559573, 2176376772, 1221630340, 2467768183, 84713789, 3270911636, 1571685966, 1581830524, 4143291311, 2923146623, 3610913076, 2622548353, 464460930, 1486913384, 3667888493, 4266300151, 3341431843, 1965213295, 2565733412, 761474095, 761709082, 3177157908, 3043376221, 396311914, 3167326505, 1734174898, 1402058836, 2225699320, 1370103810, 1206643084, 3622576190, 109652665, 1635918058, 3535739916, 2701481795, 1919438907, 1981477202, 1360788607, 777792032, 4044337196, 2681268168, 2118070045, 1276795554, 3146333618, 746239793, 2520869193, 1894154101, 1005996243, 1898277441, 3296603528, 3591769481, 3145574469, 1323130515, 1238470035, 1076131551, 2785215479, 2287060195, 1378079485, 102967153, 2942614756, 1776625394, 938257756, 2470323776, 2927673692, 129462980, 221619173, 3277410069, 581765105, 1744873192, 1725783416, 2487492363, 3776340706, 848370815, 3284445102, 1648634245, 68108052, 3595739205, 2779510023, 2872989184, 1845125748, 121458802, 786888262, 3047839649, 2036720406, 1066086686, 3620519326, 2825286213, 1043364134, 745183959, 2008388075, 3781903549, 4088204083, 2422084247, 2245918792, 163997977, 4072121451, 377438777, 3434214799, 47957125, 901584641, 1663967634, 2224354240, 2140121772, 2239607504, 1296158592, 1144490013, 363777504, 352049636, 3537828370, 2018830719, 4106625267, 791310915, 1141549081, 383680238, 2211229427, 2955313933, 1786602475, 861117031, 1657038393, 49008346, 2757187031, 2415960764, 2063869385, 4250446539, 4072556442, 3080810042, 3746048829, 1207880898, 743812379, 1015188201, 1250574590, 2697317474, 1490831142, 3975889652, 3507228768, 3917477006, 922984658, 1548413943, 3607819493, 642573978, 2802934424, 1071690339, 2487490598, 353709112, 3041703309, 3991444003, 3766125511, 4022423108, 2255595963, 4244631415, 1858811130, 1272720251, 4254073557, 3076459717, 1790136700, 3582509861, 2769798461, 3229452726, 2752397444, 701766670, 1437507306, 1004148691, 578333132, 82009872, 3366243481, 1209720848, 1659428252, 789199905, 2021015352, 3384069524, 658409068, 1580305341, 4213315272, 4208459453, 2059532450, 893773418, 653840510, 1457966480, 393232134, 3136440598, 1431618763, 1172519077, 2575368795, 939388973, 1509101907, 988758589, 565340075, 2869588947, 1972475685, 4273757903, 204012057, 1101337211, 1533180088, 2801615526, 2440934744, 338200579, 3477442758, 2757306425, 2685737460, 3224302362, 1663459583, 696652877, 4112318289, 2675890357, 745289117, 2365755424, 2933967187, 4022266172, 1505363667, 2297045170, 570110611, 3346324247, 2491642679, 3975465941, 1581427559, 3057064987, 3791566228, 1635441872, 1271004942, 1877908946, 2574434132, 1044045440, 3454314545, 1253401130, 103952148, 3575409376, 490573770, 2501813399, 1298020930, 382668967, 2514416127, 3341542436, 1092494529, 195927818, 924628104, 3732801821, 1514695853, 1522275894, 219321854, 1556133239, 504225130, 1473692730, 406339414, 424484390, 4274713028, 2913973779, 4130369073, 1360365870, 628908339, 4154982639, 757598615, 2791015278, 2080857461, 4149657548, 3529030811, 2769339058, 429034912, 996084582, 1734359068, 1375161721, 156143303, 3037969436, 1190257791, 1676065026, 73997760, 2249183962, 3601724488, 4122290591, 2965613742, 1557037497, 4185416125, 3400325053, 3974330977, 380844763, 2773129515, 152044993, 1429397077, 4224338628, 1029309310, 1646236907, 3601261456, 148996457, 3350178033, 2474815495, 1410658357, 2550027794, 1767790115, 270701719, 3018598882, 4024035625, 887012213, 2098623801, 3632846028, 2921764843, 3962072584, 641857031, 2017495901, 263120295, 2019130833, 1014261205, 4186856768, 2015365872, 279028754, 72415546, 924570728, 3902648166, 1678286441, 2292588029, 2311635131, 2102542545, 2971228400, 1591614386, 434847124, 3404271484, 3154798571, 113413735, 2491451743, 876206292, 1926984841, 3365539483, 4251125583, 793584453, 301069822, 3419075736, 724765157, 3208860939, 374696415, 1335525536, 1919740580, 3663859906, 2324775696, 3864157940, 1565672574, 3802032297, 171709183, 1234689881, 3634865316, 2015586222, 3369223628, 2978260002, 279245834, 70045384, 214564933, 2965926443, 2361247591, 4165854731, 2521208355, 2800346995, 4049169329, 23045802, 1970506914, 3641829100, 1666315902, 2241077078, 271426571, 4088283675, 2162546067, 3127287949, 2587241655, 159689334, 3031939099, 86965855, 685758755, 1240649695, 3226456551, 2957980176, 3555426742, 428862619, 839793577, 773054281, 76694871, 1000736152, 1723860376, 2645296715, 1765515124, 2397562436, 3840534762, 3419177686, 2056708761, 2320819600, 722357493, 1951132286, 2697176849, 3418517148, 603726513, 2078863096, 2316090329, 2970470049, 462414015, 791905947, 1253829863, 78371665, 4043788288, 2830761235, 2395343283, 2760690656, 1435313866, 1069903560, 1768883836, 3478580525, 904619222, 279413896, 2319870694, 4228173768, 3232061024, 1484346351, 4004145551, 3963210362, 1575762141, 1425228619, 4101712633, 483124459, 1978185260, 2962076605, 340777240, 3916617987, 2077585685, 1670667332, 2219902641, 1768998827, 47882544, 38398050, 3045070584, 950674091, 2821356397, 2104948616, 1566315737, 865585600, 1541267679, 3879180856, 2839407679, 1159648826, 1505709233, 3176134596, 3648874638, 415873698, 3782961634, 1010479344, 3886069287, 2148727539, 3631608995, 377214383, 4046490475, 1123005720, 1622745139, 3490975330, 2511527452, 1283291646, 2881450334, 4188742159, 2510804289, 879103178, 2003656783, 2968076365, 4153230545, 1779731768, 2697187974, 938150976, 3424390191, 2962715395, 101985175, 2518895690, 2455009572, 3071599415, 1061702964, 2015878382, 2790438718, 3098661352, 105466182, 2218696853, 2782132502, 1059279116, 1168368222, 377291999, 548884237, 1330528867, 1028242923, 594901465, 2151845525, 1345962394, 829096669, 3769617076, 859176729, 1416740787, 2528276857], "pos": 5}, "exits": {}, "cancels": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": 0.75}, "trials_created": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": true}, "trials_requested": 1, "search_method_state": {"created_trials": 1, "pending_trials": 1, "search_method_type": "single"}, "completed_operations": {}}, "trial_searcher_state": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": {"Op": {"Length": 1000, "RequestID": "c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "checkpoint": null, "request_id": "c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae", "trial_seed": 530298166, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`),
			//nolint:lll
			v5Snapshot: []byte(`{"searcher_state": {"rand": {"key": [3922433409, 243371046, 1078118500, 751359450, 1341787537, 4110007575, 3830714619, 215586197, 202234178, 695788744, 568122858, 801842164, 915101998, 2089585430, 560765197, 132023793, 3625985341, 1033888033, 4275540481, 4081480301, 3803198840, 2062925182, 3815214353, 2217168443, 284556261, 4216276606, 3143802969, 4077949604, 1360661190, 1579111912, 3397581176, 2605001170, 1444943786, 1103161266, 2475001171, 586405187, 440758113, 69964893, 2302367679, 2318485289, 2699422072, 949642636, 992261637, 3418960830, 2747796370, 1988242324, 1336558633, 4144877284, 1447227155, 2130144917, 249701494, 845842848, 3562805067, 461807133, 316172300, 1019162321, 2042173338, 2357208493, 984738996, 2977486791, 4204759905, 3765157938, 1416537904, 673684512, 1014036428, 3291531021, 2149014833, 211182184, 1899637415, 2967746221, 1721885906, 226438235, 2148590681, 1789583513, 1586306530, 1162436037, 2949013157, 869268556, 2754664383, 190339737, 2683355822, 1858335276, 752629074, 3123663892, 3566745314, 2416019451, 2343621935, 4064029673, 2375292477, 4237178213, 733006903, 3966381652, 1902496609, 1636160960, 1441547075, 837638138, 885960515, 1113758007, 2646826291, 1038503594, 3726531118, 1965291352, 4027030242, 3233398909, 2964755703, 966551311, 1083992523, 1124285624, 3192801484, 2731176820, 2400133886, 2523660457, 1033111323, 2914428172, 1163901245, 1320025743, 2177720309, 1677889669, 2345935772, 2794475418, 9574722, 1397363731, 3692199984, 452561608, 112544138, 2256573095, 2829353338, 1987429988, 555809442, 1666429314, 215970428, 3154343711, 2051241582, 3877008887, 1816890945, 4013344821, 2102130895, 4186655536, 2673431404, 4026559573, 2176376772, 1221630340, 2467768183, 84713789, 3270911636, 1571685966, 1581830524, 4143291311, 2923146623, 3610913076, 2622548353, 464460930, 1486913384, 3667888493, 4266300151, 3341431843, 1965213295, 2565733412, 761474095, 761709082, 3177157908, 3043376221, 396311914, 3167326505, 1734174898, 1402058836, 2225699320, 1370103810, 1206643084, 3622576190, 109652665, 1635918058, 3535739916, 2701481795, 1919438907, 1981477202, 1360788607, 777792032, 4044337196, 2681268168, 2118070045, 1276795554, 3146333618, 746239793, 2520869193, 1894154101, 1005996243, 1898277441, 3296603528, 3591769481, 3145574469, 1323130515, 1238470035, 1076131551, 2785215479, 2287060195, 1378079485, 102967153, 2942614756, 1776625394, 938257756, 2470323776, 2927673692, 129462980, 221619173, 3277410069, 581765105, 1744873192, 1725783416, 2487492363, 3776340706, 848370815, 3284445102, 1648634245, 68108052, 3595739205, 2779510023, 2872989184, 1845125748, 121458802, 786888262, 3047839649, 2036720406, 1066086686, 3620519326, 2825286213, 1043364134, 745183959, 2008388075, 3781903549, 4088204083, 2422084247, 2245918792, 163997977, 4072121451, 377438777, 3434214799, 47957125, 901584641, 1663967634, 2224354240, 2140121772, 2239607504, 1296158592, 1144490013, 363777504, 352049636, 3537828370, 2018830719, 4106625267, 791310915, 1141549081, 383680238, 2211229427, 2955313933, 1786602475, 861117031, 1657038393, 49008346, 2757187031, 2415960764, 2063869385, 4250446539, 4072556442, 3080810042, 3746048829, 1207880898, 743812379, 1015188201, 1250574590, 2697317474, 1490831142, 3975889652, 3507228768, 3917477006, 922984658, 1548413943, 3607819493, 642573978, 2802934424, 1071690339, 2487490598, 353709112, 3041703309, 3991444003, 3766125511, 4022423108, 2255595963, 4244631415, 1858811130, 1272720251, 4254073557, 3076459717, 1790136700, 3582509861, 2769798461, 3229452726, 2752397444, 701766670, 1437507306, 1004148691, 578333132, 82009872, 3366243481, 1209720848, 1659428252, 789199905, 2021015352, 3384069524, 658409068, 1580305341, 4213315272, 4208459453, 2059532450, 893773418, 653840510, 1457966480, 393232134, 3136440598, 1431618763, 1172519077, 2575368795, 939388973, 1509101907, 988758589, 565340075, 2869588947, 1972475685, 4273757903, 204012057, 1101337211, 1533180088, 2801615526, 2440934744, 338200579, 3477442758, 2757306425, 2685737460, 3224302362, 1663459583, 696652877, 4112318289, 2675890357, 745289117, 2365755424, 2933967187, 4022266172, 1505363667, 2297045170, 570110611, 3346324247, 2491642679, 3975465941, 1581427559, 3057064987, 3791566228, 1635441872, 1271004942, 1877908946, 2574434132, 1044045440, 3454314545, 1253401130, 103952148, 3575409376, 490573770, 2501813399, 1298020930, 382668967, 2514416127, 3341542436, 1092494529, 195927818, 924628104, 3732801821, 1514695853, 1522275894, 219321854, 1556133239, 504225130, 1473692730, 406339414, 424484390, 4274713028, 2913973779, 4130369073, 1360365870, 628908339, 4154982639, 757598615, 2791015278, 2080857461, 4149657548, 3529030811, 2769339058, 429034912, 996084582, 1734359068, 1375161721, 156143303, 3037969436, 1190257791, 1676065026, 73997760, 2249183962, 3601724488, 4122290591, 2965613742, 1557037497, 4185416125, 3400325053, 3974330977, 380844763, 2773129515, 152044993, 1429397077, 4224338628, 1029309310, 1646236907, 3601261456, 148996457, 3350178033, 2474815495, 1410658357, 2550027794, 1767790115, 270701719, 3018598882, 4024035625, 887012213, 2098623801, 3632846028, 2921764843, 3962072584, 641857031, 2017495901, 263120295, 2019130833, 1014261205, 4186856768, 2015365872, 279028754, 72415546, 924570728, 3902648166, 1678286441, 2292588029, 2311635131, 2102542545, 2971228400, 1591614386, 434847124, 3404271484, 3154798571, 113413735, 2491451743, 876206292, 1926984841, 3365539483, 4251125583, 793584453, 301069822, 3419075736, 724765157, 3208860939, 374696415, 1335525536, 1919740580, 3663859906, 2324775696, 3864157940, 1565672574, 3802032297, 171709183, 1234689881, 3634865316, 2015586222, 3369223628, 2978260002, 279245834, 70045384, 214564933, 2965926443, 2361247591, 4165854731, 2521208355, 2800346995, 4049169329, 23045802, 1970506914, 3641829100, 1666315902, 2241077078, 271426571, 4088283675, 2162546067, 3127287949, 2587241655, 159689334, 3031939099, 86965855, 685758755, 1240649695, 3226456551, 2957980176, 3555426742, 428862619, 839793577, 773054281, 76694871, 1000736152, 1723860376, 2645296715, 1765515124, 2397562436, 3840534762, 3419177686, 2056708761, 2320819600, 722357493, 1951132286, 2697176849, 3418517148, 603726513, 2078863096, 2316090329, 2970470049, 462414015, 791905947, 1253829863, 78371665, 4043788288, 2830761235, 2395343283, 2760690656, 1435313866, 1069903560, 1768883836, 3478580525, 904619222, 279413896, 2319870694, 4228173768, 3232061024, 1484346351, 4004145551, 3963210362, 1575762141, 1425228619, 4101712633, 483124459, 1978185260, 2962076605, 340777240, 3916617987, 2077585685, 1670667332, 2219902641, 1768998827, 47882544, 38398050, 3045070584, 950674091, 2821356397, 2104948616, 1566315737, 865585600, 1541267679, 3879180856, 2839407679, 1159648826, 1505709233, 3176134596, 3648874638, 415873698, 3782961634, 1010479344, 3886069287, 2148727539, 3631608995, 377214383, 4046490475, 1123005720, 1622745139, 3490975330, 2511527452, 1283291646, 2881450334, 4188742159, 2510804289, 879103178, 2003656783, 2968076365, 4153230545, 1779731768, 2697187974, 938150976, 3424390191, 2962715395, 101985175, 2518895690, 2455009572, 3071599415, 1061702964, 2015878382, 2790438718, 3098661352, 105466182, 2218696853, 2782132502, 1059279116, 1168368222, 377291999, 548884237, 1330528867, 1028242923, 594901465, 2151845525, 1345962394, 829096669, 3769617076, 859176729, 1416740787, 2528276857], "pos": 5}, "exits": {}, "cancels": {}, "failures": {}, "trials_closed": {}, "trial_progress": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": 0.75}, "trials_created": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": true}, "trials_requested": 1, "search_method_state": {"created_trials": 1, "pending_trials": 1, "search_method_type": "single"}}, "trial_searcher_state": {"c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae": {"EarlyExitedByUserCode": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "request_id": "c7ffd20e-ec56-4e21-b6dd-2c2e6a27a9ae", "trial_seed": 530298166}, "EarlyStoppedBySearcher": false}}}`),
		},
		{
			name: searcher.AdaptiveASHASearch,
			//nolint:lll
			v4Snapshot: []byte(`{"searcher_state": {"rand": {"key": [1452545913, 2216706256, 2931285415, 1424553963, 2771469721, 1182268625, 2241039347, 812859712, 3441585930, 1306182607, 3485054815, 608308123, 112483053, 682642478, 1312525610, 4135551174, 1567793875, 708291338, 3829781031, 1127597650, 1379589895, 4278788276, 970175326, 4153583291, 3982059578, 1493128882, 1875069103, 3704826304, 2299697046, 3676722567, 3041672913, 3928015435, 3590950234, 1340137182, 1319309903, 945669253, 1297992086, 255387058, 3738978610, 1747520993, 1267854203, 1275083903, 1988788172, 181920895, 1106027729, 1025177369, 2440854937, 188734714, 2791495762, 379644148, 1364486454, 1214940208, 3655512912, 3005587086, 3108187149, 781287699, 4159116078, 4043987339, 1632188677, 1250292127, 3753687125, 1026997361, 3415512821, 3782772083, 1358650193, 2164683139, 564316438, 941046993, 3129882612, 2182601888, 1975366329, 3311855628, 1644174752, 2793779553, 2753842799, 3571302630, 887923171, 3588840141, 193784717, 2471237536, 1421225039, 341938740, 1192996113, 2218409675, 3991457045, 3826598018, 2275285112, 4011863454, 1177877850, 2698633618, 1138152393, 2202893314, 347707955, 2436375966, 1020461438, 792229146, 113320512, 2036693887, 1215276260, 1018416563, 1195865614, 3136674021, 4150724058, 1177584340, 3870966002, 2945594975, 3859621414, 1717325326, 2051142765, 1936740951, 3062042453, 1688513379, 910042307, 1750537331, 304051019, 2636291192, 2013521520, 4227041032, 102628453, 1359940869, 4245248134, 1993619173, 3500025022, 3924853543, 3578800279, 2421667083, 4060793652, 1094276667, 928577324, 3215001437, 2737811296, 3805282341, 408114954, 2248459851, 1476902951, 73200291, 3600387404, 422649030, 2174730155, 3307173078, 1507629695, 113607599, 4116987602, 3510362588, 1500789310, 2511817938, 3576571324, 1756177812, 2413290146, 3140611884, 1850078157, 2095279096, 635232750, 3799246263, 1381854103, 57954085, 69958379, 55700730, 2727860158, 646634153, 646716365, 3060248147, 2645894961, 521556301, 2332134955, 3223404494, 1278684995, 2435087898, 401816126, 1094921611, 563333291, 2556415388, 1739710028, 133744693, 3919682174, 4087793525, 768502435, 4228008790, 210686178, 4150681864, 4091917901, 1835561309, 2232363723, 3031311488, 1776937342, 1233396034, 3476217536, 3799316084, 864878498, 377881092, 3188094097, 2750744302, 3199804590, 1231139403, 2429436965, 2130823117, 757668219, 2740840838, 1377165079, 2589078863, 2810863562, 1162879957, 831096657, 3396087259, 3098412156, 4061324184, 3203871242, 1931886212, 2937742743, 907488495, 3996978148, 2997218253, 2004185458, 2516650587, 593077235, 3489768626, 2150154954, 2049860379, 2472886567, 1430759218, 2957836638, 435817594, 1082252350, 3951520482, 2400586842, 3456205344, 3356182591, 618991329, 2992469709, 1973188425, 450379261, 2798568635, 4037568720, 2839400952, 255422271, 1854524310, 4006718197, 3420074895, 2722893251, 3468530746, 1593574294, 3521350966, 3024147312, 1169215387, 3910087826, 443646771, 2925353376, 2069261626, 2548372079, 2863860503, 4132686053, 3433804206, 950538056, 3552376483, 1870093729, 3384688973, 1158025919, 493730904, 1789038843, 1829026938, 3596529436, 3639269430, 1817201820, 848183402, 537785361, 2081164395, 2475598070, 1575953173, 3376512332, 4036578374, 581826236, 3513291559, 2113632026, 2616353057, 3729275555, 1680494578, 1363619591, 692325427, 97273826, 2579406491, 1531446639, 2390954510, 1317588761, 536858982, 988517096, 2244293128, 2804565134, 3877515395, 839842752, 3405616275, 1039673240, 3962514502, 896670502, 1093259020, 2043255868, 3991535172, 296660859, 4163623869, 4045669283, 3512517932, 1859259214, 79788873, 217768790, 2455520809, 386000238, 3566902734, 2247723896, 674246520, 226356020, 1961300106, 3848127451, 353356745, 52039995, 2770049955, 3021363660, 1624062882, 2986876091, 448633518, 1629505280, 21883362, 1848415473, 756079215, 4190124084, 4131714971, 2392212283, 1308326958, 3371194070, 1147387144, 1285992797, 3405963410, 1892827156, 1967237941, 2398927758, 238263952, 2432436094, 1498668510, 858532034, 3668090456, 1838702366, 426802744, 1194191127, 2885124677, 1279814428, 2856963439, 2166294156, 2653671113, 26555231, 4188511414, 2257856313, 3262711136, 395413330, 2789270448, 4264752234, 1117258307, 2896009146, 1226138881, 3802856348, 2523573944, 4159838139, 1445150309, 264468203, 4150566522, 2240997763, 1509631376, 435995998, 4165403839, 3610262680, 1168629505, 4035006883, 2695323969, 1292644795, 2745853320, 225794602, 1356178611, 955998327, 2238901481, 1106469405, 826123461, 4155393474, 2635289592, 1224474211, 1824797825, 1233049504, 760946898, 1148631409, 3096376358, 4205086568, 2291321688, 537823608, 490724183, 966548030, 213314081, 3997861198, 3555542635, 3937802140, 3037829047, 3057237581, 149746191, 4198823951, 3419012586, 3058993263, 3099857571, 738118719, 2559925817, 429746199, 999300291, 3324548747, 599849818, 51382161, 1656669044, 1614461261, 3437196859, 2914200168, 2522112723, 1406081509, 1013688116, 2269547301, 1493949008, 352929430, 3057939739, 3841398959, 2865277247, 274596832, 3889831750, 4063920528, 322224127, 369484237, 1096644289, 2969910741, 355324547, 3017842962, 652957044, 2288093025, 1319424438, 3597823923, 3187158261, 2300967604, 2444733360, 3346034650, 3052890513, 3412771067, 1644673169, 2004833514, 4176603510, 812339001, 2377375086, 3657608513, 789432047, 1232357156, 311642210, 1135746832, 4278279012, 4187675947, 1484381964, 518250323, 1804044839, 749985106, 1247034074, 1322933383, 2558753984, 2148724241, 3355638721, 1239704585, 458131162, 1469044936, 1816687844, 3310401614, 711732101, 2570107777, 2195543406, 1685156591, 806200714, 485890833, 2324324855, 1361241033, 231249103, 3925784640, 2812874552, 105551818, 3360389267, 236876949, 3837288942, 2773521044, 3397769505, 396727058, 1340986334, 842480288, 3693769194, 1304819435, 3927626290, 1233405874, 2400596270, 3722895074, 4024237244, 1308945353, 2379151461, 1653639100, 2837868071, 3143239362, 3347100096, 505670669, 1220922352, 1683276312, 1409357641, 2864400686, 2154723844, 2865848502, 3711211457, 4174404079, 3760415152, 2635418164, 3817078233, 3797497846, 1843555871, 3097746780, 1631738968, 1273189302, 3422812667, 1884737974, 3996941605, 567541988, 2835894599, 2678807189, 2862594697, 259382272, 4246982418, 2015432239, 1584697754, 950481942, 2676080319, 3060794403, 3206445087, 1623919009, 3227302040, 3207120844, 2227286302, 1395885538, 786668072, 1111364168, 831375560, 2351912451, 1750459989, 1249540546, 1954378595, 377821224, 1223652079, 660872685, 3148523876, 636798872, 3792321202, 1157296821, 2728165650, 2125606106, 286143865, 1929354736, 3282533922, 2910169564, 3416956740, 3467743336, 3115919084, 1499088458, 2537111141, 2756648794, 1989501022, 1972571169, 1594350983, 1687019006, 2282666905, 2791303809, 816687306, 3206038097, 3432004981, 1128341472, 2489750111, 4212962683, 4150103461, 3277624383, 1740516965, 3298397586, 1066119190, 973732348, 1142153902, 2593584651, 1347071644, 476190584, 255904713, 2487444411, 4277502188, 503790438, 3701506354, 4080177843, 1973766567, 2765496076, 2445282555, 3253894374, 69025061, 1973540476, 3707131235, 2463125033, 3650253067, 1261214630, 1834547302, 1646383947, 1135075426, 4138467070, 1543536840, 2107446004, 379532649, 2859826137, 2562206910, 962843355, 1514927822, 1971937745, 693327781, 3542160189, 327935861, 853015903, 3483398805, 3644300960, 1581562223, 719777381, 2642808027, 4103170367, 1148912942], "pos": 32}, "exits": {}, "cancels": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"35cf42b8-36cf-49e3-b104-f3cce289d072": 0, "7ad68246-0a8a-4251-916d-db94cf525270": 0}, "trials_created": {"35cf42b8-36cf-49e3-b104-f3cce289d072": true, "7ad68246-0a8a-4251-916d-db94cf525270": true}, "trials_requested": 2, "search_method_state": {"trial_table": {"35cf42b8-36cf-49e3-b104-f3cce289d072": 1, "7ad68246-0a8a-4251-916d-db94cf525270": 0}, "sub_search_states": [{"rungs": [{"metrics": null, "start_trials": 0, "units_needed": 58, "promote_trials": 0, "outstanding_trials": 1}, {"metrics": null, "start_trials": 0, "units_needed": 292, "promote_trials": 0, "outstanding_trials": 0}, {"metrics": null, "start_trials": 0, "units_needed": 1229, "promote_trials": 0, "outstanding_trials": 0}], "trial_rungs": {"7ad68246-0a8a-4251-916d-db94cf525270": 0}, "closed_trials": {}, "invalid_trials": 0, "pending_trials": 0, "trials_completed": 0, "early_exit_trials": {}, "search_method_type": "asha"}, {"rungs": [{"metrics": null, "start_trials": 0, "units_needed": 234, "promote_trials": 0, "outstanding_trials": 1}, {"metrics": null, "start_trials": 0, "units_needed": 1171, "promote_trials": 0, "outstanding_trials": 0}], "trial_rungs": {"35cf42b8-36cf-49e3-b104-f3cce289d072": 0}, "closed_trials": {}, "invalid_trials": 0, "pending_trials": 0, "trials_completed": 0, "early_exit_trials": {}, "search_method_type": "asha"}], "search_method_type": "adaptive_asha", "sub_search_units_completed": [0, 0]}, "completed_operations": {}}, "trial_searcher_state": {"35cf42b8-36cf-49e3-b104-f3cce289d072": {"Op": {"Length": 234, "RequestID": "35cf42b8-36cf-49e3-b104-f3cce289d072"}, "Closed": false, "Create": {"hparams": {"dropout1": 0.4402283511805293, "dropout2": 0.44011555407446223, "n_filters1": 15, "n_filters2": 47, "learning_rate": 0.020523695929929888}, "checkpoint": null, "request_id": "35cf42b8-36cf-49e3-b104-f3cce289d072", "trial_seed": 897420172, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}, "7ad68246-0a8a-4251-916d-db94cf525270": {"Op": {"Length": 58, "RequestID": "7ad68246-0a8a-4251-916d-db94cf525270"}, "Closed": false, "Create": {"hparams": {"dropout1": 0.24506761071384414, "dropout2": 0.6338879717356689, "n_filters1": 25, "n_filters2": 32, "learning_rate": 0.9864141286469745}, "checkpoint": null, "request_id": "7ad68246-0a8a-4251-916d-db94cf525270", "trial_seed": 1221021447, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`),
			err:        "unsupported search_method_type",
		},
		{
			name: searcher.GridSearch,
			//nolint:lll
			v4Snapshot: []byte(`{"searcher_state": {"rand": {"key": [2076300081, 1180757701, 61390626, 2075653657, 3983276912, 2219280310, 1951783750, 3432627106, 1678661725, 3042313994, 2202725749, 2202588336, 4190101782, 1110057276, 3191526579, 4171973736, 4254763240, 2766827202, 1156928074, 1597039613, 3598380759, 1437190771, 1076126477, 2658152872, 3825090400, 3114761804, 915898666, 291106241, 3036411027, 2744580765, 820531638, 1883042039, 2922514564, 188829505, 1874867577, 640982739, 1714839269, 3901091477, 2804767841, 3254494107, 2484761704, 476134574, 990456848, 3870586239, 620062224, 1995301304, 2389729019, 2462435556, 2472017888, 2504467222, 2188233046, 2306262543, 3560836860, 1978513357, 606650471, 3854035095, 4245776466, 3871782493, 597013963, 2913535589, 1756877383, 1623386184, 496866040, 588131525, 4177340522, 2935519398, 850546169, 109126006, 858579993, 795071074, 3590755528, 1638945842, 1254896771, 2307161967, 3477967610, 692808483, 1899600971, 2769494850, 2779999007, 510215608, 4111810801, 4280921106, 1518864863, 834898192, 1306840028, 835688979, 459513373, 609148069, 3545906245, 1194162898, 258441420, 4179429231, 4022175812, 2648608732, 717531457, 306247172, 2976743141, 237988140, 592196390, 704362957, 7450542, 311689767, 3067438641, 808588439, 1235198799, 742758417, 2780949935, 3446973894, 3873685895, 2153250996, 4065466909, 1497342942, 3724631752, 4275880421, 272199235, 2252962393, 2056155089, 1877112438, 2110296751, 2590941302, 2013827341, 842823052, 1894032322, 1393960121, 3588305616, 2529793378, 3205436904, 596814183, 1254786875, 2966225931, 2121907724, 475901771, 555053869, 1400569825, 3688890119, 1353122628, 1459205986, 2300737010, 1481729865, 756275962, 599878589, 2046378230, 2769239562, 3613422680, 4252206344, 947287240, 2417250410, 1305902854, 2671572052, 1899272897, 3230713963, 2224033552, 1732656815, 1936213189, 443599218, 2770284889, 3459882565, 1332607404, 3278697474, 2224591365, 2257839008, 317213768, 2348128734, 1874716743, 456696413, 1116761881, 3309591553, 2862216319, 102982924, 2150630897, 3549437867, 2842597334, 4045646707, 302008588, 1654281218, 1727189467, 738749640, 3764450573, 3959398424, 3317500773, 3535585186, 3819002453, 3662808846, 1770928073, 3787632432, 2477652163, 1746899438, 3750614957, 761234677, 3689896539, 3520772878, 275147232, 337553768, 2986250282, 3506693941, 591880001, 1455806111, 3804664561, 2479592303, 4013399092, 726617980, 255395900, 178605610, 1293625725, 2646174944, 2376136796, 1142332741, 266983497, 999135041, 1646761432, 3287859002, 2993926759, 3462763364, 4227465400, 3355948815, 1238930500, 2174357849, 379232244, 2722994601, 2303499529, 974370129, 2577648277, 808892366, 3061424190, 4078456653, 1309753062, 2244949359, 371260896, 1136719967, 1708548536, 1658845169, 2768668945, 1531754301, 3191375913, 3511054906, 1062351502, 1744842885, 581610003, 1556303968, 867409273, 1667798951, 209299262, 437489551, 3229818234, 4190510574, 714265718, 2019898663, 564912797, 3376897769, 3559969466, 3938497845, 471145731, 349982265, 1552795612, 1447370340, 3534819495, 3702903169, 2270765654, 3192681951, 204048124, 3420549534, 3712568979, 3780914699, 2771260334, 2314887317, 745613193, 3577922416, 3094639701, 89439972, 1944758030, 1119237373, 586389801, 2520447175, 3625732341, 1427555151, 2087065690, 409476010, 187430597, 3753592208, 2951247134, 885626627, 844754868, 2092667268, 748208300, 3708044571, 3080007883, 3448755645, 3861156834, 1842920493, 818793362, 595126995, 2776784591, 62641362, 840212211, 3985028931, 387936511, 473253991, 1857032791, 4000006968, 4243356941, 1229644438, 153835145, 1283983784, 3918669237, 2721262354, 4294888081, 3113635576, 3580583332, 3521345732, 3528129211, 2061930144, 1634595536, 650979608, 2017813394, 1823871219, 3733362889, 3318357463, 2742200384, 763055433, 358005169, 768692017, 1050455834, 3453938424, 49062937, 3352742911, 3630652047, 2437806883, 1597018682, 2518758128, 1213648650, 4073021622, 4259149854, 3212854626, 79448901, 771981874, 3297404440, 3186097826, 319093164, 3890862606, 2245955576, 1497647520, 828363054, 3988483235, 3157718635, 562359205, 1350548803, 3372491415, 86446595, 1114830016, 2762338015, 1180274773, 2871068129, 2507166170, 2627076257, 1096162219, 2200646305, 3664591154, 3892273969, 476888795, 832867753, 4151853558, 2982123525, 182781907, 241410694, 1341125666, 4028887234, 3884607589, 2732864456, 1605421707, 2038450818, 3362242279, 4122980381, 2985487124, 110300201, 509696857, 4017443718, 1838466952, 2327354958, 2137521982, 138621377, 2133785874, 1413747039, 1739282333, 1675927427, 3185180235, 2373730108, 495353069, 1293977021, 1368037164, 2798684905, 81516419, 3857598893, 3495427721, 1760877692, 1152788660, 970145190, 3017785210, 1990290980, 67842654, 3925233768, 4002022873, 256245794, 1696909255, 738711838, 1446248938, 3868148475, 1939621088, 3658634988, 1784094744, 296610735, 1744996095, 2451305905, 2987883429, 2443943189, 1185159281, 4111976294, 2182983103, 3071467556, 1314957723, 152464689, 655243290, 1120785722, 1363764666, 3687005133, 4166824409, 4000596589, 2610383491, 3239976693, 4016033738, 1658070453, 1873771320, 1404781153, 3039196925, 3067316017, 2982654406, 373430827, 1067861532, 3675311637, 267701726, 3638688126, 2135521145, 1500865973, 1349464109, 3403519167, 2763769271, 4264496249, 1770838146, 1391852856, 2245935765, 2137032072, 1856750853, 688147071, 99608919, 3339156540, 2192512157, 2820609381, 2913853119, 2634309658, 1135799296, 1004115372, 2154400231, 248807841, 1855000157, 1129896866, 715899117, 923401956, 440983241, 3109426185, 477965580, 84854570, 1400593057, 1286402819, 3439792802, 318102008, 1067872774, 4240269308, 2549429311, 3855914333, 3795474048, 319956485, 4133374879, 1467447321, 78080933, 251366958, 3496186988, 3070794250, 2726275807, 2325947483, 53232900, 819143840, 2594863810, 55769358, 391403252, 866292794, 714727242, 2581427793, 698101228, 872117109, 997416829, 3565045060, 1611274371, 1562302439, 3417876422, 1392788396, 3206592320, 2207230324, 721665840, 2883852313, 1919434161, 395546903, 1837162976, 3429796583, 1522595013, 3348506059, 1175004850, 2074412352, 3758985171, 2415299591, 1703486181, 2304170834, 549422017, 2902550119, 2187986969, 3362538344, 768387705, 2456129962, 3429271129, 3706106775, 204072664, 806512945, 3311330988, 1474760185, 1814261184, 1577346575, 2078317992, 3122306338, 3467081158, 1225264382, 1847488518, 599526500, 2488492818, 1688305017, 438612976, 1607069832, 2927329539, 503082697, 1150118168, 1865053291, 2428734033, 4135113359, 821829536, 868562558, 413775071, 1334204977, 673408382, 1882374958, 4291776686, 4039700264, 911935763, 1482546356, 3253895898, 2457857766, 278986188, 1981961411, 1618768772, 658526763, 3374586925, 729845344, 1603013850, 1740519279, 2732939556, 821729772, 4152108407, 2159443430, 2359657320, 3741458889, 2625148830, 2740386969, 508381854, 4017968509, 3047356953, 3577840409, 4191139539, 592911320, 1632820963, 2552033020, 1212957884, 1014120341, 3310543727, 3779267868, 3583152261, 3680986715, 2995658876, 3137928161, 587029290, 2798799836, 4187466127, 594529572, 406159454, 4285083401, 2812893547, 841074247, 2953120157, 466989304, 2691320671, 3368132983, 3549982940, 3530994849, 3593311564, 2291771526, 3269554905, 4281627661, 1457258966, 3464984667, 1432592878, 1387812831, 154474021, 343300151, 1018325484, 2085104744, 4236418319, 2375494721, 1088217159, 389539389], "pos": 10}, "exits": {}, "cancels": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": 0, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": 0}, "trials_created": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": true, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": true}, "trials_requested": 2, "search_method_state": {"pending_trials": 2, "remaining_trials": [{"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 32, "learning_rate": 1}, {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 64, "learning_rate": 1}], "search_method_type": "grid"}, "completed_operations": {}}, "trial_searcher_state": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": {"Op": {"Length": 1, "RequestID": "b20fd10b-c039-45fa-b450-86e9ad91ec28"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 32, "learning_rate": 1}, "checkpoint": null, "request_id": "b20fd10b-c039-45fa-b450-86e9ad91ec28", "trial_seed": 1367408042, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": {"Op": {"Length": 1, "RequestID": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "checkpoint": null, "request_id": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e", "trial_seed": 1545095049, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`),
			//nolint:lll
			v5Snapshot: []byte(`{"searcher_state": {"rand": {"key": [2076300081, 1180757701, 61390626, 2075653657, 3983276912, 2219280310, 1951783750, 3432627106, 1678661725, 3042313994, 2202725749, 2202588336, 4190101782, 1110057276, 3191526579, 4171973736, 4254763240, 2766827202, 1156928074, 1597039613, 3598380759, 1437190771, 1076126477, 2658152872, 3825090400, 3114761804, 915898666, 291106241, 3036411027, 2744580765, 820531638, 1883042039, 2922514564, 188829505, 1874867577, 640982739, 1714839269, 3901091477, 2804767841, 3254494107, 2484761704, 476134574, 990456848, 3870586239, 620062224, 1995301304, 2389729019, 2462435556, 2472017888, 2504467222, 2188233046, 2306262543, 3560836860, 1978513357, 606650471, 3854035095, 4245776466, 3871782493, 597013963, 2913535589, 1756877383, 1623386184, 496866040, 588131525, 4177340522, 2935519398, 850546169, 109126006, 858579993, 795071074, 3590755528, 1638945842, 1254896771, 2307161967, 3477967610, 692808483, 1899600971, 2769494850, 2779999007, 510215608, 4111810801, 4280921106, 1518864863, 834898192, 1306840028, 835688979, 459513373, 609148069, 3545906245, 1194162898, 258441420, 4179429231, 4022175812, 2648608732, 717531457, 306247172, 2976743141, 237988140, 592196390, 704362957, 7450542, 311689767, 3067438641, 808588439, 1235198799, 742758417, 2780949935, 3446973894, 3873685895, 2153250996, 4065466909, 1497342942, 3724631752, 4275880421, 272199235, 2252962393, 2056155089, 1877112438, 2110296751, 2590941302, 2013827341, 842823052, 1894032322, 1393960121, 3588305616, 2529793378, 3205436904, 596814183, 1254786875, 2966225931, 2121907724, 475901771, 555053869, 1400569825, 3688890119, 1353122628, 1459205986, 2300737010, 1481729865, 756275962, 599878589, 2046378230, 2769239562, 3613422680, 4252206344, 947287240, 2417250410, 1305902854, 2671572052, 1899272897, 3230713963, 2224033552, 1732656815, 1936213189, 443599218, 2770284889, 3459882565, 1332607404, 3278697474, 2224591365, 2257839008, 317213768, 2348128734, 1874716743, 456696413, 1116761881, 3309591553, 2862216319, 102982924, 2150630897, 3549437867, 2842597334, 4045646707, 302008588, 1654281218, 1727189467, 738749640, 3764450573, 3959398424, 3317500773, 3535585186, 3819002453, 3662808846, 1770928073, 3787632432, 2477652163, 1746899438, 3750614957, 761234677, 3689896539, 3520772878, 275147232, 337553768, 2986250282, 3506693941, 591880001, 1455806111, 3804664561, 2479592303, 4013399092, 726617980, 255395900, 178605610, 1293625725, 2646174944, 2376136796, 1142332741, 266983497, 999135041, 1646761432, 3287859002, 2993926759, 3462763364, 4227465400, 3355948815, 1238930500, 2174357849, 379232244, 2722994601, 2303499529, 974370129, 2577648277, 808892366, 3061424190, 4078456653, 1309753062, 2244949359, 371260896, 1136719967, 1708548536, 1658845169, 2768668945, 1531754301, 3191375913, 3511054906, 1062351502, 1744842885, 581610003, 1556303968, 867409273, 1667798951, 209299262, 437489551, 3229818234, 4190510574, 714265718, 2019898663, 564912797, 3376897769, 3559969466, 3938497845, 471145731, 349982265, 1552795612, 1447370340, 3534819495, 3702903169, 2270765654, 3192681951, 204048124, 3420549534, 3712568979, 3780914699, 2771260334, 2314887317, 745613193, 3577922416, 3094639701, 89439972, 1944758030, 1119237373, 586389801, 2520447175, 3625732341, 1427555151, 2087065690, 409476010, 187430597, 3753592208, 2951247134, 885626627, 844754868, 2092667268, 748208300, 3708044571, 3080007883, 3448755645, 3861156834, 1842920493, 818793362, 595126995, 2776784591, 62641362, 840212211, 3985028931, 387936511, 473253991, 1857032791, 4000006968, 4243356941, 1229644438, 153835145, 1283983784, 3918669237, 2721262354, 4294888081, 3113635576, 3580583332, 3521345732, 3528129211, 2061930144, 1634595536, 650979608, 2017813394, 1823871219, 3733362889, 3318357463, 2742200384, 763055433, 358005169, 768692017, 1050455834, 3453938424, 49062937, 3352742911, 3630652047, 2437806883, 1597018682, 2518758128, 1213648650, 4073021622, 4259149854, 3212854626, 79448901, 771981874, 3297404440, 3186097826, 319093164, 3890862606, 2245955576, 1497647520, 828363054, 3988483235, 3157718635, 562359205, 1350548803, 3372491415, 86446595, 1114830016, 2762338015, 1180274773, 2871068129, 2507166170, 2627076257, 1096162219, 2200646305, 3664591154, 3892273969, 476888795, 832867753, 4151853558, 2982123525, 182781907, 241410694, 1341125666, 4028887234, 3884607589, 2732864456, 1605421707, 2038450818, 3362242279, 4122980381, 2985487124, 110300201, 509696857, 4017443718, 1838466952, 2327354958, 2137521982, 138621377, 2133785874, 1413747039, 1739282333, 1675927427, 3185180235, 2373730108, 495353069, 1293977021, 1368037164, 2798684905, 81516419, 3857598893, 3495427721, 1760877692, 1152788660, 970145190, 3017785210, 1990290980, 67842654, 3925233768, 4002022873, 256245794, 1696909255, 738711838, 1446248938, 3868148475, 1939621088, 3658634988, 1784094744, 296610735, 1744996095, 2451305905, 2987883429, 2443943189, 1185159281, 4111976294, 2182983103, 3071467556, 1314957723, 152464689, 655243290, 1120785722, 1363764666, 3687005133, 4166824409, 4000596589, 2610383491, 3239976693, 4016033738, 1658070453, 1873771320, 1404781153, 3039196925, 3067316017, 2982654406, 373430827, 1067861532, 3675311637, 267701726, 3638688126, 2135521145, 1500865973, 1349464109, 3403519167, 2763769271, 4264496249, 1770838146, 1391852856, 2245935765, 2137032072, 1856750853, 688147071, 99608919, 3339156540, 2192512157, 2820609381, 2913853119, 2634309658, 1135799296, 1004115372, 2154400231, 248807841, 1855000157, 1129896866, 715899117, 923401956, 440983241, 3109426185, 477965580, 84854570, 1400593057, 1286402819, 3439792802, 318102008, 1067872774, 4240269308, 2549429311, 3855914333, 3795474048, 319956485, 4133374879, 1467447321, 78080933, 251366958, 3496186988, 3070794250, 2726275807, 2325947483, 53232900, 819143840, 2594863810, 55769358, 391403252, 866292794, 714727242, 2581427793, 698101228, 872117109, 997416829, 3565045060, 1611274371, 1562302439, 3417876422, 1392788396, 3206592320, 2207230324, 721665840, 2883852313, 1919434161, 395546903, 1837162976, 3429796583, 1522595013, 3348506059, 1175004850, 2074412352, 3758985171, 2415299591, 1703486181, 2304170834, 549422017, 2902550119, 2187986969, 3362538344, 768387705, 2456129962, 3429271129, 3706106775, 204072664, 806512945, 3311330988, 1474760185, 1814261184, 1577346575, 2078317992, 3122306338, 3467081158, 1225264382, 1847488518, 599526500, 2488492818, 1688305017, 438612976, 1607069832, 2927329539, 503082697, 1150118168, 1865053291, 2428734033, 4135113359, 821829536, 868562558, 413775071, 1334204977, 673408382, 1882374958, 4291776686, 4039700264, 911935763, 1482546356, 3253895898, 2457857766, 278986188, 1981961411, 1618768772, 658526763, 3374586925, 729845344, 1603013850, 1740519279, 2732939556, 821729772, 4152108407, 2159443430, 2359657320, 3741458889, 2625148830, 2740386969, 508381854, 4017968509, 3047356953, 3577840409, 4191139539, 592911320, 1632820963, 2552033020, 1212957884, 1014120341, 3310543727, 3779267868, 3583152261, 3680986715, 2995658876, 3137928161, 587029290, 2798799836, 4187466127, 594529572, 406159454, 4285083401, 2812893547, 841074247, 2953120157, 466989304, 2691320671, 3368132983, 3549982940, 3530994849, 3593311564, 2291771526, 3269554905, 4281627661, 1457258966, 3464984667, 1432592878, 1387812831, 154474021, 343300151, 1018325484, 2085104744, 4236418319, 2375494721, 1088217159, 389539389], "pos": 10}, "exits": {}, "cancels": {}, "failures": {}, "trials_closed": {}, "trial_progress": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": 0, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": 0}, "trials_created": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": true, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": true}, "trials_requested": 2, "search_method_state": {"pending_trials": 2, "remaining_trials": [{"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 32, "learning_rate": 1}, {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 64, "learning_rate": 1}], "search_method_type": "grid"}}, "trial_searcher_state": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": {"EarlyExitedByUserCode": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 32, "learning_rate": 1}, "request_id": "b20fd10b-c039-45fa-b450-86e9ad91ec28", "trial_seed": 1367408042}, "EarlyStoppedBySearcher": false}, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": {"EarlyExitedByUserCode": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "request_id": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e", "trial_seed": 1545095049}, "EarlyStoppedBySearcher": false}}}`),
		},
		{
			name: searcher.RandomSearch,
			//nolint:lll
			v4Snapshot: []byte(`{"searcher_state": {"rand": {"key": [2076300081, 1180757701, 61390626, 2075653657, 3983276912, 2219280310, 1951783750, 3432627106, 1678661725, 3042313994, 2202725749, 2202588336, 4190101782, 1110057276, 3191526579, 4171973736, 4254763240, 2766827202, 1156928074, 1597039613, 3598380759, 1437190771, 1076126477, 2658152872, 3825090400, 3114761804, 915898666, 291106241, 3036411027, 2744580765, 820531638, 1883042039, 2922514564, 188829505, 1874867577, 640982739, 1714839269, 3901091477, 2804767841, 3254494107, 2484761704, 476134574, 990456848, 3870586239, 620062224, 1995301304, 2389729019, 2462435556, 2472017888, 2504467222, 2188233046, 2306262543, 3560836860, 1978513357, 606650471, 3854035095, 4245776466, 3871782493, 597013963, 2913535589, 1756877383, 1623386184, 496866040, 588131525, 4177340522, 2935519398, 850546169, 109126006, 858579993, 795071074, 3590755528, 1638945842, 1254896771, 2307161967, 3477967610, 692808483, 1899600971, 2769494850, 2779999007, 510215608, 4111810801, 4280921106, 1518864863, 834898192, 1306840028, 835688979, 459513373, 609148069, 3545906245, 1194162898, 258441420, 4179429231, 4022175812, 2648608732, 717531457, 306247172, 2976743141, 237988140, 592196390, 704362957, 7450542, 311689767, 3067438641, 808588439, 1235198799, 742758417, 2780949935, 3446973894, 3873685895, 2153250996, 4065466909, 1497342942, 3724631752, 4275880421, 272199235, 2252962393, 2056155089, 1877112438, 2110296751, 2590941302, 2013827341, 842823052, 1894032322, 1393960121, 3588305616, 2529793378, 3205436904, 596814183, 1254786875, 2966225931, 2121907724, 475901771, 555053869, 1400569825, 3688890119, 1353122628, 1459205986, 2300737010, 1481729865, 756275962, 599878589, 2046378230, 2769239562, 3613422680, 4252206344, 947287240, 2417250410, 1305902854, 2671572052, 1899272897, 3230713963, 2224033552, 1732656815, 1936213189, 443599218, 2770284889, 3459882565, 1332607404, 3278697474, 2224591365, 2257839008, 317213768, 2348128734, 1874716743, 456696413, 1116761881, 3309591553, 2862216319, 102982924, 2150630897, 3549437867, 2842597334, 4045646707, 302008588, 1654281218, 1727189467, 738749640, 3764450573, 3959398424, 3317500773, 3535585186, 3819002453, 3662808846, 1770928073, 3787632432, 2477652163, 1746899438, 3750614957, 761234677, 3689896539, 3520772878, 275147232, 337553768, 2986250282, 3506693941, 591880001, 1455806111, 3804664561, 2479592303, 4013399092, 726617980, 255395900, 178605610, 1293625725, 2646174944, 2376136796, 1142332741, 266983497, 999135041, 1646761432, 3287859002, 2993926759, 3462763364, 4227465400, 3355948815, 1238930500, 2174357849, 379232244, 2722994601, 2303499529, 974370129, 2577648277, 808892366, 3061424190, 4078456653, 1309753062, 2244949359, 371260896, 1136719967, 1708548536, 1658845169, 2768668945, 1531754301, 3191375913, 3511054906, 1062351502, 1744842885, 581610003, 1556303968, 867409273, 1667798951, 209299262, 437489551, 3229818234, 4190510574, 714265718, 2019898663, 564912797, 3376897769, 3559969466, 3938497845, 471145731, 349982265, 1552795612, 1447370340, 3534819495, 3702903169, 2270765654, 3192681951, 204048124, 3420549534, 3712568979, 3780914699, 2771260334, 2314887317, 745613193, 3577922416, 3094639701, 89439972, 1944758030, 1119237373, 586389801, 2520447175, 3625732341, 1427555151, 2087065690, 409476010, 187430597, 3753592208, 2951247134, 885626627, 844754868, 2092667268, 748208300, 3708044571, 3080007883, 3448755645, 3861156834, 1842920493, 818793362, 595126995, 2776784591, 62641362, 840212211, 3985028931, 387936511, 473253991, 1857032791, 4000006968, 4243356941, 1229644438, 153835145, 1283983784, 3918669237, 2721262354, 4294888081, 3113635576, 3580583332, 3521345732, 3528129211, 2061930144, 1634595536, 650979608, 2017813394, 1823871219, 3733362889, 3318357463, 2742200384, 763055433, 358005169, 768692017, 1050455834, 3453938424, 49062937, 3352742911, 3630652047, 2437806883, 1597018682, 2518758128, 1213648650, 4073021622, 4259149854, 3212854626, 79448901, 771981874, 3297404440, 3186097826, 319093164, 3890862606, 2245955576, 1497647520, 828363054, 3988483235, 3157718635, 562359205, 1350548803, 3372491415, 86446595, 1114830016, 2762338015, 1180274773, 2871068129, 2507166170, 2627076257, 1096162219, 2200646305, 3664591154, 3892273969, 476888795, 832867753, 4151853558, 2982123525, 182781907, 241410694, 1341125666, 4028887234, 3884607589, 2732864456, 1605421707, 2038450818, 3362242279, 4122980381, 2985487124, 110300201, 509696857, 4017443718, 1838466952, 2327354958, 2137521982, 138621377, 2133785874, 1413747039, 1739282333, 1675927427, 3185180235, 2373730108, 495353069, 1293977021, 1368037164, 2798684905, 81516419, 3857598893, 3495427721, 1760877692, 1152788660, 970145190, 3017785210, 1990290980, 67842654, 3925233768, 4002022873, 256245794, 1696909255, 738711838, 1446248938, 3868148475, 1939621088, 3658634988, 1784094744, 296610735, 1744996095, 2451305905, 2987883429, 2443943189, 1185159281, 4111976294, 2182983103, 3071467556, 1314957723, 152464689, 655243290, 1120785722, 1363764666, 3687005133, 4166824409, 4000596589, 2610383491, 3239976693, 4016033738, 1658070453, 1873771320, 1404781153, 3039196925, 3067316017, 2982654406, 373430827, 1067861532, 3675311637, 267701726, 3638688126, 2135521145, 1500865973, 1349464109, 3403519167, 2763769271, 4264496249, 1770838146, 1391852856, 2245935765, 2137032072, 1856750853, 688147071, 99608919, 3339156540, 2192512157, 2820609381, 2913853119, 2634309658, 1135799296, 1004115372, 2154400231, 248807841, 1855000157, 1129896866, 715899117, 923401956, 440983241, 3109426185, 477965580, 84854570, 1400593057, 1286402819, 3439792802, 318102008, 1067872774, 4240269308, 2549429311, 3855914333, 3795474048, 319956485, 4133374879, 1467447321, 78080933, 251366958, 3496186988, 3070794250, 2726275807, 2325947483, 53232900, 819143840, 2594863810, 55769358, 391403252, 866292794, 714727242, 2581427793, 698101228, 872117109, 997416829, 3565045060, 1611274371, 1562302439, 3417876422, 1392788396, 3206592320, 2207230324, 721665840, 2883852313, 1919434161, 395546903, 1837162976, 3429796583, 1522595013, 3348506059, 1175004850, 2074412352, 3758985171, 2415299591, 1703486181, 2304170834, 549422017, 2902550119, 2187986969, 3362538344, 768387705, 2456129962, 3429271129, 3706106775, 204072664, 806512945, 3311330988, 1474760185, 1814261184, 1577346575, 2078317992, 3122306338, 3467081158, 1225264382, 1847488518, 599526500, 2488492818, 1688305017, 438612976, 1607069832, 2927329539, 503082697, 1150118168, 1865053291, 2428734033, 4135113359, 821829536, 868562558, 413775071, 1334204977, 673408382, 1882374958, 4291776686, 4039700264, 911935763, 1482546356, 3253895898, 2457857766, 278986188, 1981961411, 1618768772, 658526763, 3374586925, 729845344, 1603013850, 1740519279, 2732939556, 821729772, 4152108407, 2159443430, 2359657320, 3741458889, 2625148830, 2740386969, 508381854, 4017968509, 3047356953, 3577840409, 4191139539, 592911320, 1632820963, 2552033020, 1212957884, 1014120341, 3310543727, 3779267868, 3583152261, 3680986715, 2995658876, 3137928161, 587029290, 2798799836, 4187466127, 594529572, 406159454, 4285083401, 2812893547, 841074247, 2953120157, 466989304, 2691320671, 3368132983, 3549982940, 3530994849, 3593311564, 2291771526, 3269554905, 4281627661, 1457258966, 3464984667, 1432592878, 1387812831, 154474021, 343300151, 1018325484, 2085104744, 4236418319, 2375494721, 1088217159, 389539389], "pos": 10}, "exits": {}, "cancels": {}, "failures": {}, "shutdown": false, "trials_closed": {}, "trial_progress": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": 0, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": 0}, "trials_created": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": true, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": true}, "trials_requested": 2, "search_method_state": {"pending_trials": 2, "remaining_trials": [{"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 32, "learning_rate": 1}, {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 64, "learning_rate": 1}], "search_method_type": "grid"}, "completed_operations": {}}, "trial_searcher_state": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": {"Op": {"Length": 1, "RequestID": "b20fd10b-c039-45fa-b450-86e9ad91ec28"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 32, "learning_rate": 1}, "checkpoint": null, "request_id": "b20fd10b-c039-45fa-b450-86e9ad91ec28", "trial_seed": 1367408042, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": {"Op": {"Length": 1, "RequestID": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e"}, "Closed": true, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "checkpoint": null, "request_id": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e", "trial_seed": 1545095049, "workload_sequencer_type": "TRIAL_WORKLOAD_SEQUENCER"}, "Complete": false}}}`),
			//nolint:lll
			v5Snapshot: []byte(`{"searcher_state": {"rand": {"key": [2076300081, 1180757701, 61390626, 2075653657, 3983276912, 2219280310, 1951783750, 3432627106, 1678661725, 3042313994, 2202725749, 2202588336, 4190101782, 1110057276, 3191526579, 4171973736, 4254763240, 2766827202, 1156928074, 1597039613, 3598380759, 1437190771, 1076126477, 2658152872, 3825090400, 3114761804, 915898666, 291106241, 3036411027, 2744580765, 820531638, 1883042039, 2922514564, 188829505, 1874867577, 640982739, 1714839269, 3901091477, 2804767841, 3254494107, 2484761704, 476134574, 990456848, 3870586239, 620062224, 1995301304, 2389729019, 2462435556, 2472017888, 2504467222, 2188233046, 2306262543, 3560836860, 1978513357, 606650471, 3854035095, 4245776466, 3871782493, 597013963, 2913535589, 1756877383, 1623386184, 496866040, 588131525, 4177340522, 2935519398, 850546169, 109126006, 858579993, 795071074, 3590755528, 1638945842, 1254896771, 2307161967, 3477967610, 692808483, 1899600971, 2769494850, 2779999007, 510215608, 4111810801, 4280921106, 1518864863, 834898192, 1306840028, 835688979, 459513373, 609148069, 3545906245, 1194162898, 258441420, 4179429231, 4022175812, 2648608732, 717531457, 306247172, 2976743141, 237988140, 592196390, 704362957, 7450542, 311689767, 3067438641, 808588439, 1235198799, 742758417, 2780949935, 3446973894, 3873685895, 2153250996, 4065466909, 1497342942, 3724631752, 4275880421, 272199235, 2252962393, 2056155089, 1877112438, 2110296751, 2590941302, 2013827341, 842823052, 1894032322, 1393960121, 3588305616, 2529793378, 3205436904, 596814183, 1254786875, 2966225931, 2121907724, 475901771, 555053869, 1400569825, 3688890119, 1353122628, 1459205986, 2300737010, 1481729865, 756275962, 599878589, 2046378230, 2769239562, 3613422680, 4252206344, 947287240, 2417250410, 1305902854, 2671572052, 1899272897, 3230713963, 2224033552, 1732656815, 1936213189, 443599218, 2770284889, 3459882565, 1332607404, 3278697474, 2224591365, 2257839008, 317213768, 2348128734, 1874716743, 456696413, 1116761881, 3309591553, 2862216319, 102982924, 2150630897, 3549437867, 2842597334, 4045646707, 302008588, 1654281218, 1727189467, 738749640, 3764450573, 3959398424, 3317500773, 3535585186, 3819002453, 3662808846, 1770928073, 3787632432, 2477652163, 1746899438, 3750614957, 761234677, 3689896539, 3520772878, 275147232, 337553768, 2986250282, 3506693941, 591880001, 1455806111, 3804664561, 2479592303, 4013399092, 726617980, 255395900, 178605610, 1293625725, 2646174944, 2376136796, 1142332741, 266983497, 999135041, 1646761432, 3287859002, 2993926759, 3462763364, 4227465400, 3355948815, 1238930500, 2174357849, 379232244, 2722994601, 2303499529, 974370129, 2577648277, 808892366, 3061424190, 4078456653, 1309753062, 2244949359, 371260896, 1136719967, 1708548536, 1658845169, 2768668945, 1531754301, 3191375913, 3511054906, 1062351502, 1744842885, 581610003, 1556303968, 867409273, 1667798951, 209299262, 437489551, 3229818234, 4190510574, 714265718, 2019898663, 564912797, 3376897769, 3559969466, 3938497845, 471145731, 349982265, 1552795612, 1447370340, 3534819495, 3702903169, 2270765654, 3192681951, 204048124, 3420549534, 3712568979, 3780914699, 2771260334, 2314887317, 745613193, 3577922416, 3094639701, 89439972, 1944758030, 1119237373, 586389801, 2520447175, 3625732341, 1427555151, 2087065690, 409476010, 187430597, 3753592208, 2951247134, 885626627, 844754868, 2092667268, 748208300, 3708044571, 3080007883, 3448755645, 3861156834, 1842920493, 818793362, 595126995, 2776784591, 62641362, 840212211, 3985028931, 387936511, 473253991, 1857032791, 4000006968, 4243356941, 1229644438, 153835145, 1283983784, 3918669237, 2721262354, 4294888081, 3113635576, 3580583332, 3521345732, 3528129211, 2061930144, 1634595536, 650979608, 2017813394, 1823871219, 3733362889, 3318357463, 2742200384, 763055433, 358005169, 768692017, 1050455834, 3453938424, 49062937, 3352742911, 3630652047, 2437806883, 1597018682, 2518758128, 1213648650, 4073021622, 4259149854, 3212854626, 79448901, 771981874, 3297404440, 3186097826, 319093164, 3890862606, 2245955576, 1497647520, 828363054, 3988483235, 3157718635, 562359205, 1350548803, 3372491415, 86446595, 1114830016, 2762338015, 1180274773, 2871068129, 2507166170, 2627076257, 1096162219, 2200646305, 3664591154, 3892273969, 476888795, 832867753, 4151853558, 2982123525, 182781907, 241410694, 1341125666, 4028887234, 3884607589, 2732864456, 1605421707, 2038450818, 3362242279, 4122980381, 2985487124, 110300201, 509696857, 4017443718, 1838466952, 2327354958, 2137521982, 138621377, 2133785874, 1413747039, 1739282333, 1675927427, 3185180235, 2373730108, 495353069, 1293977021, 1368037164, 2798684905, 81516419, 3857598893, 3495427721, 1760877692, 1152788660, 970145190, 3017785210, 1990290980, 67842654, 3925233768, 4002022873, 256245794, 1696909255, 738711838, 1446248938, 3868148475, 1939621088, 3658634988, 1784094744, 296610735, 1744996095, 2451305905, 2987883429, 2443943189, 1185159281, 4111976294, 2182983103, 3071467556, 1314957723, 152464689, 655243290, 1120785722, 1363764666, 3687005133, 4166824409, 4000596589, 2610383491, 3239976693, 4016033738, 1658070453, 1873771320, 1404781153, 3039196925, 3067316017, 2982654406, 373430827, 1067861532, 3675311637, 267701726, 3638688126, 2135521145, 1500865973, 1349464109, 3403519167, 2763769271, 4264496249, 1770838146, 1391852856, 2245935765, 2137032072, 1856750853, 688147071, 99608919, 3339156540, 2192512157, 2820609381, 2913853119, 2634309658, 1135799296, 1004115372, 2154400231, 248807841, 1855000157, 1129896866, 715899117, 923401956, 440983241, 3109426185, 477965580, 84854570, 1400593057, 1286402819, 3439792802, 318102008, 1067872774, 4240269308, 2549429311, 3855914333, 3795474048, 319956485, 4133374879, 1467447321, 78080933, 251366958, 3496186988, 3070794250, 2726275807, 2325947483, 53232900, 819143840, 2594863810, 55769358, 391403252, 866292794, 714727242, 2581427793, 698101228, 872117109, 997416829, 3565045060, 1611274371, 1562302439, 3417876422, 1392788396, 3206592320, 2207230324, 721665840, 2883852313, 1919434161, 395546903, 1837162976, 3429796583, 1522595013, 3348506059, 1175004850, 2074412352, 3758985171, 2415299591, 1703486181, 2304170834, 549422017, 2902550119, 2187986969, 3362538344, 768387705, 2456129962, 3429271129, 3706106775, 204072664, 806512945, 3311330988, 1474760185, 1814261184, 1577346575, 2078317992, 3122306338, 3467081158, 1225264382, 1847488518, 599526500, 2488492818, 1688305017, 438612976, 1607069832, 2927329539, 503082697, 1150118168, 1865053291, 2428734033, 4135113359, 821829536, 868562558, 413775071, 1334204977, 673408382, 1882374958, 4291776686, 4039700264, 911935763, 1482546356, 3253895898, 2457857766, 278986188, 1981961411, 1618768772, 658526763, 3374586925, 729845344, 1603013850, 1740519279, 2732939556, 821729772, 4152108407, 2159443430, 2359657320, 3741458889, 2625148830, 2740386969, 508381854, 4017968509, 3047356953, 3577840409, 4191139539, 592911320, 1632820963, 2552033020, 1212957884, 1014120341, 3310543727, 3779267868, 3583152261, 3680986715, 2995658876, 3137928161, 587029290, 2798799836, 4187466127, 594529572, 406159454, 4285083401, 2812893547, 841074247, 2953120157, 466989304, 2691320671, 3368132983, 3549982940, 3530994849, 3593311564, 2291771526, 3269554905, 4281627661, 1457258966, 3464984667, 1432592878, 1387812831, 154474021, 343300151, 1018325484, 2085104744, 4236418319, 2375494721, 1088217159, 389539389], "pos": 10}, "exits": {}, "cancels": {}, "failures": {}, "trials_closed": {}, "trial_progress": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": 0, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": 0}, "trials_created": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": true, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": true}, "trials_requested": 2, "search_method_state": {"pending_trials": 2, "remaining_trials": [{"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 32, "learning_rate": 1}, {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 16, "n_filters2": 64, "learning_rate": 1}], "search_method_type": "grid"}}, "trial_searcher_state": {"b20fd10b-c039-45fa-b450-86e9ad91ec28": {"EarlyExitedByUserCode": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 32, "learning_rate": 1}, "request_id": "b20fd10b-c039-45fa-b450-86e9ad91ec28", "trial_seed": 1367408042}, "EarlyStoppedBySearcher": false}, "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e": {"EarlyExitedByUserCode": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "request_id": "c8810eb9-937a-4c71-86e6-cefbc9f1ba8e", "trial_seed": 1545095049}, "EarlyStoppedBySearcher": false}}}`),
		},
	}
	for _, c := range cases {
		t.Run(string(c.name), func(t *testing.T) {
			snapshot, err := shimExperimentSnapshotV5(c.v4Snapshot)
			if c.err != "" {
				require.ErrorContains(t, err, c.err)
			} else {
				require.NoError(t, err)
				require.JSONEq(t, string(c.v5Snapshot), string(snapshot))
			}
		})
	}
}

func TestDeserExperimentSnapshotIntoCurrent(t *testing.T) {
	// This test tries to deserialize a copy of the current experiment snapshot.
	// If this test fails, it means there was a breaking change to snapshots which may've not
	// received a shim. Please ensure there is a shim and fix this test for the next time.
	tests := []struct {
		name       string
		experiment func() *internalExperiment
		snapshot   []byte
	}{
		{
			name: "single",
			experiment: func() *internalExperiment {
				e := &internalExperiment{}
				//nolint:exhaustruct
				single := expconf.SearcherConfig{
					//nolint:exhaustruct
					RawSingleConfig: &expconf.SingleConfig{},
				}
				sm := searcher.NewSearchMethod(single)
				e.searcher = searcher.NewSearcher(0, sm, expconf.Hyperparameters{})
				return e
			},
			//nolint:lll
			snapshot: []byte(`{"searcher_state": {"rand": {"key": [2458791418, 3116467974, 2738037530, 1223607127, 3286445278, 625897174, 1172106481, 2572993093, 2198533759, 3837319279, 89569234, 2840266316, 3281728377, 2462842803, 223147636, 1426784029, 117006397, 2636154877, 1449436706, 2927825553, 669826405, 4158003124, 3051998482, 3863884161, 4115545788, 3969262691, 1993859905, 526793478, 2862121258, 2585948009, 363338570, 2434066506, 2697236630, 3741159605, 4276159879, 2540043166, 1254547227, 281417924, 3524428006, 1717738180, 2788608558, 3067505898, 1428442263, 2529107117, 2141185930, 2917242155, 2718794995, 1103873205, 2335184097, 1489619463, 1859666767, 643801413, 3529330080, 1352766355, 2547570226, 2073329677, 2610277209, 435342246, 2505497031, 798656570, 3382074658, 4189489635, 28016400, 3296078941, 1945638788, 2826985424, 4265599268, 304154357, 932685664, 2758841683, 2455326991, 3980298044, 2314318323, 595095528, 4177520052, 3112160276, 1257280438, 1207886472, 2677553884, 3265570879, 2457548809, 2111634072, 3972389638, 665045651, 3994591990, 2042170295, 3594420958, 1611936557, 819891391, 3473558321, 426840388, 1314012383, 2561237332, 2272262342, 3134027931, 4294175632, 3732369287, 4128031240, 4147187676, 3149809049, 4144063892, 2345560497, 3738737588, 3840299650, 280313741, 101675464, 1393579054, 1505545589, 3277145020, 2188655472, 1421937881, 854289081, 2431554199, 1539816964, 83937993, 3224697403, 1610045703, 2816848158, 792619755, 934271931, 2710423316, 3868916984, 3853694682, 580166127, 3034428351, 1242892688, 859119565, 3802529696, 1000657147, 1905689709, 2988202998, 4019138435, 3698964560, 781687417, 2634708534, 1790387080, 3083588514, 2067517328, 3813894530, 1697102895, 1736661565, 1635070572, 2451931176, 2607824278, 2710884525, 3333385014, 2122182769, 2392051351, 2906588340, 4103192869, 1168352864, 1983143924, 710890467, 2699295676, 3119053614, 1518994382, 2466039509, 572476338, 4181949791, 3735635392, 742157348, 1157573401, 1972168146, 272316361, 1510876922, 1556419901, 941326744, 2682202561, 3173709824, 1829453176, 2251191850, 3456122487, 3173338821, 330462762, 1754886765, 250991513, 1585255942, 253192177, 3012909793, 3530990056, 2004128557, 514254595, 2445947804, 2613994303, 1602501425, 2012372011, 3584430451, 2144992891, 453151729, 418148689, 3200328899, 1320919208, 3350564767, 1251183192, 3290155726, 3287744578, 1064686988, 3961170913, 1980153684, 1339419052, 3783807526, 2183864424, 2594654516, 1832017181, 3373700328, 248263337, 4269409531, 3608097081, 2124157490, 3498209804, 3263200819, 3127724750, 1312779256, 1734613689, 3719011732, 3243076830, 335603414, 1448836682, 2863567879, 3171222479, 953744612, 1854108551, 396912697, 1471435093, 2536166045, 2432755996, 50855133, 1127984088, 2445499632, 682533110, 692746935, 1651664519, 2705150811, 4151653551, 1934542959, 3007387232, 2198726288, 3698559570, 3010880745, 3239971816, 168338881, 2794239687, 2901311416, 1552192572, 1131246424, 2788592201, 805537073, 2327010547, 1261208610, 3515399927, 3639687688, 151099323, 1666670090, 3955849057, 484700107, 3470434774, 2223492086, 90444150, 1572468487, 697895815, 2621785173, 4028029284, 1631608044, 1794520030, 1369705039, 4286067898, 1548006208, 1859332416, 3062951908, 343535845, 1027995635, 847936898, 1002776644, 1787482428, 1761964089, 1010518086, 2242475935, 3301831376, 3685964202, 2354379073, 3765363997, 3338328111, 2354888011, 3883114207, 936858297, 1841452303, 2180987975, 3833453400, 4253783174, 187816314, 1065449290, 828211291, 1227118119, 3312227455, 857392088, 3989703709, 2722876258, 2826056677, 2487224771, 1620253778, 4235150726, 1357697909, 2030894361, 3540876365, 2001105143, 203686501, 3114867920, 2041712100, 2614268670, 3094356533, 3358484761, 4080319699, 3980200285, 1516323641, 1169578821, 2585305656, 1638623928, 2071984054, 1179146259, 3142029384, 2910525849, 430984842, 1809790632, 1593515823, 2878348267, 4018175470, 1966255236, 1939592619, 2197525510, 4123890540, 4027942930, 2056710998, 1057194760, 3276845032, 618524839, 912386042, 1369305871, 4213981309, 4157357229, 1998049094, 4068346859, 283997778, 433297801, 2185564962, 2668307743, 3547263851, 2971812172, 145864931, 38002940, 2929551979, 2152716980, 2278120571, 1172785191, 1888308402, 930353176, 3711016582, 2308015795, 2305204206, 4068791645, 2473454079, 2501692185, 3674392733, 2375917712, 2557427355, 1990988406, 2226078296, 600963582, 347505141, 705969876, 3222111902, 237273550, 472225245, 1376144940, 3812100488, 4130033222, 1834636053, 2200817839, 3623132254, 2048566977, 872938564, 157673410, 2667298410, 1593933448, 2550146443, 3424352086, 3520633431, 3318249052, 1535053463, 1320135181, 1072090846, 3748951794, 3105511402, 2155614440, 3343066249, 1664809838, 2354003702, 577588884, 2764537545, 4260035710, 160815645, 3551374291, 2754442860, 1118743534, 1286742203, 3375424472, 1193358286, 1523416396, 2286745682, 3818424205, 3229692865, 1250603769, 3376938675, 1934656537, 3002173085, 1421535763, 2329166244, 1539829771, 3071809424, 3722361831, 1807071764, 2928747228, 3466233204, 1681608616, 1707233748, 532399055, 1655975870, 2192615935, 1181907517, 502315677, 195435504, 1068351185, 4253522108, 401515526, 2322634116, 3817476766, 2545684199, 367003362, 939004561, 1720493014, 3282529078, 916836710, 3271777100, 1051347323, 903520982, 1843390427, 903860254, 3163311743, 1498324049, 1602796155, 3993830274, 2222947632, 1594870809, 2262149984, 706505490, 1878448793, 297783572, 1922520975, 1809817994, 3378836142, 4283471362, 724529390, 3871255903, 2675377541, 625442236, 1396594762, 584050984, 3160502847, 3713812500, 3070779976, 2415860718, 2087253424, 2150659256, 2846376499, 642246206, 954003700, 790693801, 2852862948, 2069560311, 2716657131, 1777996881, 23021516, 394280992, 151827274, 554425914, 3093163480, 500362788, 4063471109, 1518232184, 3051194338, 1564237119, 1784240953, 1728907606, 2423906971, 3245861425, 1403272208, 705601793, 809561192, 2967681376, 1107800719, 3654973939, 3529963233, 1141430012, 171161121, 1222606508, 3325794587, 2933848069, 3319288855, 4076047601, 1422688234, 750895463, 723415161, 4110917996, 172378373, 614310717, 464190258, 2719742080, 2882190533, 260241577, 872187166, 51392255, 690678529, 1032247122, 177848862, 2631860912, 3876348701, 3971628423, 1411241180, 1525225113, 3429234786, 2902950094, 3480849833, 1067421076, 4184135694, 2905699039, 3922251469, 556665325, 1008854626, 1749376673, 1706526652, 1670818738, 1459937672, 1236710049, 2044381071, 28380869, 1088464802, 2500621248, 3319166637, 303350363, 4233871691, 462899686, 1864082083, 1633855571, 3234480516, 2411053733, 3986263103, 1992510755, 1986246923, 1418621100, 3772785621, 3118288321, 3069082678, 3575509950, 2661593874, 189268839, 1690530432, 1411957573, 4198485847, 1214217760, 4032838120, 647907711, 4245255521, 1940716058, 242619953, 2097141622, 1218025765, 1007790786, 3748834757, 8175890, 2456318442, 219038421, 4142868721, 2303296422, 465336410, 237135465, 2525403966, 2972194247, 672480967, 2623034704, 1120730289, 1018451191, 1318162784, 2171567834, 1931677349, 1300941830, 1091032044, 344455379, 3648398142, 1660289918, 1476172654, 1382274791, 376429449, 3753352837, 884844415, 3994051274, 3011131147, 3092988955, 1358346546, 3519843272, 3978487789, 3830946161, 1625675712, 3511881408, 2031293928, 2203163940, 3362955715, 4279271535, 4220311393, 2051555050, 996884877, 1530801618, 3353569995, 1741998127, 728272117, 724195433], "pos": 5}, "exits": {}, "cancels": {}, "failures": {}, "trials_closed": {}, "trial_progress": {"4244f6bb-a0b0-4876-9e75-819b7cbdcd95": 0}, "trials_created": {"4244f6bb-a0b0-4876-9e75-819b7cbdcd95": true}, "trials_requested": 1, "search_method_state": {"created_trials": 1, "pending_trials": 1, "search_method_type": "single"}}, "trial_searcher_state": {"4244f6bb-a0b0-4876-9e75-819b7cbdcd95": {"Closed": false, "Create": {"hparams": {"dropout1": 0.25, "dropout2": 0.5, "n_filters1": 32, "n_filters2": 64, "learning_rate": 1}, "request_id": "4244f6bb-a0b0-4876-9e75-819b7cbdcd95", "trial_seed": 1557607182}, "EarlyStoppedBySearcher": false}}}`),
		},
		{
			name: "asha",
			experiment: func() *internalExperiment {
				e := &internalExperiment{}
				//nolint:exhaustruct
				asha := expconf.SearcherConfig{
					//nolint:exhaustruct
					RawAsyncHalvingConfig: &expconf.AsyncHalvingConfig{
						RawNumRungs:   ptrs.Ptr(4),
						RawMaxTime:    ptrs.Ptr(937),
						RawTimeMetric: ptrs.Ptr("batches"),
						RawDivisor:    ptrs.Ptr[float64](4),
					},
					RawSmallerIsBetter: ptrs.Ptr(true),
					RawMetric:          ptrs.Ptr("loss"),
				}
				sm := searcher.NewSearchMethod(asha)
				e.searcher = searcher.NewSearcher(0, sm, expconf.Hyperparameters{})
				return e
			},
			//nolint:lll
			snapshot: []byte(`{"searcher_state": {"rand": {"key": [3912824393, 130359580, 1945038949, 683827302, 3610818264, 2838364317, 4181320722, 221936505, 3819130084, 2131487376, 3365651765, 920499941, 1270817540, 2246276083, 1885603276, 573420793, 583261746, 1955039589, 1811914457, 2837955835, 2508148103, 3642464710, 3458020788, 2195814166, 1386387121, 1852638509, 3909111177, 1735517492, 3892314499, 3707856689, 879481158, 1802646466, 116991403, 298788384, 1221574842, 3804792269, 2328567293, 3014632505, 2320044096, 1413171799, 1368251971, 4083772998, 2986640645, 1030832729, 2241679027, 2849293624, 1296881709, 2785559672, 1135156955, 2796434558, 1925391449, 1829990763, 3217631629, 3067008047, 3612907438, 1326801960, 2558425453, 4068998963, 3070803605, 2403866132, 3753353239, 3296906854, 464488764, 785445029, 141537657, 2465881066, 3135233249, 4043279941, 1278274609, 2486847651, 3954374223, 129103232, 1720962446, 3816310415, 1806398817, 1250007987, 969936705, 2336561042, 2031384508, 1285947737, 39769194, 1138144666, 993968588, 101542063, 3128650165, 2003415882, 140441857, 3432886469, 3489699300, 575264191, 3842163036, 2887600812, 2487983973, 1955108072, 1173170507, 4219701607, 1171912093, 2695607515, 1710258012, 2435659755, 2071371892, 3067907555, 544251540, 2010934296, 3117136901, 4036472942, 3059696740, 3788372660, 566955155, 358758290, 2882442037, 4169179923, 1160213813, 387790744, 594608752, 3632049166, 3841361487, 1667303281, 1460402258, 2438875135, 4278318681, 2281109706, 221829418, 3933060752, 4221766281, 3185559353, 70745549, 224504629, 2249432201, 156645497, 1758678503, 3375329816, 4241049874, 3873245768, 19688418, 1201216527, 3912743911, 216273652, 1882335310, 2852029249, 3415491706, 913228854, 2437734945, 3245489646, 1027437138, 1302726420, 1927993221, 118233091, 2085121918, 3536455915, 1939892139, 4010137323, 2303981927, 3716018075, 2265312706, 2949632039, 2447406304, 624960780, 4071396684, 3488406087, 3938075266, 757291956, 582212968, 3048403818, 2464054892, 4190155310, 292733183, 3094168255, 3626519501, 1066765321, 2996573857, 1225379640, 3774880867, 758941699, 116163606, 3942208179, 1042100701, 461769728, 3164917034, 1857854599, 2167218317, 4253533681, 3818830317, 3399752715, 2419846770, 4069369592, 2150336629, 1810772325, 3026418855, 4177626623, 4092551316, 3523400346, 4234812158, 713640060, 2681330789, 1467890933, 2993052514, 3162989269, 113784382, 3852222819, 4292344413, 1633710857, 3912716682, 6098202, 2978849649, 4160324727, 2926842096, 1514591911, 1081000496, 372322291, 3857309830, 4178509955, 669675724, 1711300651, 3716229730, 143240009, 4059287817, 3731622530, 496463408, 4088097703, 1794430305, 3973039871, 2939992759, 2712680772, 3534376933, 2786410924, 278436436, 3437770869, 2024532266, 2579067234, 4007664809, 664954588, 1789677620, 1276521848, 3683816415, 1548063198, 4095783552, 1459425342, 1799248342, 1609271699, 1512067210, 3465440954, 4269839847, 4233980673, 375224779, 3138146110, 3370149200, 579857917, 760314921, 3775887833, 4176789722, 2301024383, 1686052811, 2967171371, 4180069250, 1516946627, 705831480, 1611848722, 2806858519, 3504667135, 764796204, 4065414292, 2092577162, 1913641905, 2109472504, 1540233965, 1722162277, 3692095086, 2929243742, 3283756641, 1377766218, 699611861, 1212571847, 78849955, 698623272, 2796609094, 241440821, 426090509, 1958846217, 2444318395, 3236802781, 133530479, 1092024126, 4072615628, 2198341714, 282741196, 1398198553, 242935369, 2651474853, 677947580, 1708783244, 1906140963, 3673969094, 2279101198, 3916495828, 1917458424, 2415715383, 2959221265, 2938458868, 1115625347, 2902621834, 1038437194, 3503923682, 2953499342, 3582273746, 325962114, 707955326, 3027365402, 4165229536, 1899485827, 3813891089, 2681883220, 1184425310, 2323719917, 2457851599, 2520494097, 3105387047, 1535476439, 1564705883, 3867748520, 1195731676, 2921591976, 333612810, 1635420993, 4259291997, 1241352669, 3675691581, 352453891, 2248815017, 3137177282, 729048856, 3439156494, 4008821214, 4014395416, 440152211, 768552979, 3657002944, 2120438653, 4230634748, 1758565824, 782079134, 3695862927, 394672225, 1006358179, 183399528, 2883538266, 1469129531, 320543524, 892791693, 870595213, 3437448149, 808212320, 3050276488, 492820811, 4000264990, 3105300607, 133021706, 1603329269, 4075016600, 1820213035, 2998065090, 1632804974, 675976454, 1681042893, 1653548377, 3253586235, 1201835952, 2918720057, 1931586371, 3328503644, 403841940, 2742827784, 586210709, 3159629495, 246459628, 1721247097, 1312377931, 3239372286, 1679600923, 794123331, 1817944193, 3905102033, 3378552747, 2071157036, 2394010092, 2765024891, 966846871, 2805363146, 2459053889, 618201661, 1749046098, 568312106, 484146509, 541311246, 3718232983, 326564023, 284954267, 538365293, 1950605761, 943434222, 1820312121, 121100221, 2064278572, 2697100734, 102308777, 3114797650, 2729269456, 537270132, 1206836708, 2583134729, 4017452671, 1912421380, 1901938780, 2087021833, 3135764995, 1186775394, 26323543, 2703782091, 30908834, 901812067, 2118851464, 3029690482, 375825601, 2213919290, 2396825578, 3057698947, 2058071747, 1632052001, 544954234, 2757964379, 4281338914, 2452188298, 2530548058, 2140337096, 8135661, 1073358913, 1887116366, 1039964896, 3596599560, 3114419318, 1914007107, 3160591144, 554429067, 857043881, 3862692398, 2322654082, 4293541172, 703438619, 1995234082, 1658759338, 3008231258, 2598610424, 1965795485, 3566144875, 2921648686, 3345004424, 2078323309, 3324845404, 1698658304, 455090979, 361487935, 3873466677, 1296591922, 3439118724, 2030854673, 4221505583, 529693036, 1194187791, 2108178438, 3512528157, 3889045825, 1912433754, 3786020945, 3365110760, 4106573256, 3223175585, 1666849351, 1260427075, 2012436050, 1441286049, 3811351537, 3360856433, 260097501, 632249840, 3297189399, 4276078413, 200516346, 3427560446, 1180083550, 570419283, 3289677820, 557135629, 3537527213, 3760168670, 3350252912, 2643371920, 3193156919, 4182398732, 3395828475, 495148909, 3660199181, 3822738986, 2696324761, 1126852263, 2413756996, 2224145812, 763779427, 4236225820, 1512096030, 936890385, 3871755505, 3056164931, 1971858895, 2437295547, 1083615465, 2711940487, 1939973176, 611444976, 2577021608, 2690806091, 4001225725, 1295618973, 2739332287, 1543112445, 3309369740, 1403192430, 2634062471, 472690292, 2308009112, 2156438823, 568667603, 1171167051, 551717849, 2679282693, 1893880801, 1266209741, 2206656951, 3095140942, 3825923, 3559702505, 3008679460, 554453022, 395553371, 1655473374, 828559601, 4044140613, 1728590401, 3047261812, 1003908978, 2447662187, 1442086653, 1489454960, 2524559089, 2050535748, 3390209316, 1931858035, 720671506, 3793532296, 2486505886, 3776873526, 1029777346, 3472059923, 2093494001, 4001189252, 1295318492, 256747214, 2561538344, 820561233, 1492703895, 4281534291, 2679759074, 1296226481, 2068854739, 4204858459, 3826654958, 814564580, 2703682783, 511237125, 1886845754, 4054856321, 2444144699, 95504058, 685255946, 4196035992, 4009208239, 2844476321, 3882335574, 1524669498, 3071049109, 3939141547, 53659124, 2986167924, 4062365550, 3944273163, 858877051, 3811779921, 2800310021, 3471182814, 3978499914, 184095907, 3143372954, 978190594, 428108114, 3979094339, 504541629, 3012747542, 1929522274, 641465681, 389762475, 3863437281, 131374935, 695264856, 934006733, 4014903973, 2845408941, 2293374102, 3231336512, 516967894, 51360470, 3991740459, 2740043234, 2010491456, 3737653932, 4251501920], "pos": 142}, "exits": {}, "cancels": {}, "failures": {}, "trials_closed": {"1241288f-ef58-4d42-9287-31cfa20244c5": true, "5d32d462-b831-4437-84d5-9bf8b820e09e": true, "5d804239-7e15-49e3-b710-fd7c24509d44": true, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": true, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": true, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": true, "a3e23956-04ee-489d-9135-7186a7941d28": true, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": true, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": true, "d882fdea-d2be-453d-bb93-831af2dc317b": true}, "trial_progress": {"1241288f-ef58-4d42-9287-31cfa20244c5": 1, "5d32d462-b831-4437-84d5-9bf8b820e09e": 0.3333333333333333, "5d804239-7e15-49e3-b710-fd7c24509d44": 1, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": 1, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": 0.3333333333333333, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": 1, "a3e23956-04ee-489d-9135-7186a7941d28": 0.3333333333333333, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": 1, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": 0.3333333333333333, "d882fdea-d2be-453d-bb93-831af2dc317b": 1}, "trials_created": {"1241288f-ef58-4d42-9287-31cfa20244c5": true, "5d32d462-b831-4437-84d5-9bf8b820e09e": true, "5d804239-7e15-49e3-b710-fd7c24509d44": true, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": true, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": true, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": true, "a3e23956-04ee-489d-9135-7186a7941d28": true, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": true, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": true, "d882fdea-d2be-453d-bb93-831af2dc317b": true}, "trials_requested": 10, "search_method_state": {"trial_table": {"1241288f-ef58-4d42-9287-31cfa20244c5": 1, "5d32d462-b831-4437-84d5-9bf8b820e09e": 0, "5d804239-7e15-49e3-b710-fd7c24509d44": 1, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": 1, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": 0, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": 0, "a3e23956-04ee-489d-9135-7186a7941d28": 0, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": 0, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": 0, "d882fdea-d2be-453d-bb93-831af2dc317b": 0}, "sub_search_states": [{"rungs": [{"metrics": [{"metric": 0.09292137346198563, "request_id": "d882fdea-d2be-453d-bb93-831af2dc317b"}, {"metric": 0.14394278251002454, "request_id": "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec"}, {"metric": 0.169617546004522, "request_id": "98a8c8cd-2aaf-4fbe-a358-7735337e11cd"}, {"metric": 0.17236693170526698, "request_id": "a3e23956-04ee-489d-9135-7186a7941d28"}, {"metric": 0.1814434532918463, "request_id": "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8"}, {"metric": 0.19748935120026018, "request_id": "5d32d462-b831-4437-84d5-9bf8b820e09e"}, {"metric": 0.21989905646153887, "request_id": "872fb3e1-39b9-40f8-b5ce-12ca53c225ed"}], "units_needed": 225}, {"metrics": [{"metric": 0.05633601378523418, "request_id": "d882fdea-d2be-453d-bb93-831af2dc317b"}, {"metric": 0.09834996045635314, "request_id": "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8"}, {"metric": 0.10896268058612729, "request_id": "98a8c8cd-2aaf-4fbe-a358-7735337e11cd"}], "units_needed": 900}], "trial_rungs": {"5d32d462-b831-4437-84d5-9bf8b820e09e": 0, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": 0, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": 1, "a3e23956-04ee-489d-9135-7186a7941d28": 0, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": 1, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": 0, "d882fdea-d2be-453d-bb93-831af2dc317b": 1}, "invalid_trials": 0, "trials_completed": 7, "early_exit_trials": {}, "search_method_type": "asha"}, {"rungs": [{"metrics": [{"metric": 0.05686319968148741, "request_id": "763e2853-d3ec-4ea1-a325-050f5aef5b1a"}, {"metric": 0.09348985412422045, "request_id": "1241288f-ef58-4d42-9287-31cfa20244c5"}, {"metric": 0.21061281618442695, "request_id": "5d804239-7e15-49e3-b710-fd7c24509d44"}], "units_needed": 900}], "trial_rungs": {"1241288f-ef58-4d42-9287-31cfa20244c5": 0, "5d804239-7e15-49e3-b710-fd7c24509d44": 0, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": 0}, "invalid_trials": 0, "trials_completed": 3, "early_exit_trials": {}, "search_method_type": "asha"}], "search_method_type": "adaptive_asha"}}, "trial_searcher_state": {"1241288f-ef58-4d42-9287-31cfa20244c5": {"Closed": false, "Create": {"hparams": {"dropout1": 0.7856465654579918, "dropout2": 0.33680938844866426, "n_filters1": 28, "n_filters2": 40, "learning_rate": 0.33873975920890426}, "request_id": "1241288f-ef58-4d42-9287-31cfa20244c5", "trial_seed": 1053878338}, "EarlyStoppedBySearcher": true}, "5d32d462-b831-4437-84d5-9bf8b820e09e": {"Closed": false, "Create": {"hparams": {"dropout1": 0.3005242490944792, "dropout2": 0.39344632915675276, "n_filters1": 44, "n_filters2": 9, "learning_rate": 0.22728733457505207}, "request_id": "5d32d462-b831-4437-84d5-9bf8b820e09e", "trial_seed": 40569440}, "EarlyStoppedBySearcher": true}, "5d804239-7e15-49e3-b710-fd7c24509d44": {"Closed": false, "Create": {"hparams": {"dropout1": 0.7197975421763473, "dropout2": 0.718622666277326, "n_filters1": 62, "n_filters2": 9, "learning_rate": 0.3138344243067023}, "request_id": "5d804239-7e15-49e3-b710-fd7c24509d44", "trial_seed": 821874082}, "EarlyStoppedBySearcher": true}, "763e2853-d3ec-4ea1-a325-050f5aef5b1a": {"Closed": false, "Create": {"hparams": {"dropout1": 0.3797289112689676, "dropout2": 0.4162215587922132, "n_filters1": 12, "n_filters2": 42, "learning_rate": 0.8267899122382265}, "request_id": "763e2853-d3ec-4ea1-a325-050f5aef5b1a", "trial_seed": 905057022}, "EarlyStoppedBySearcher": true}, "872fb3e1-39b9-40f8-b5ce-12ca53c225ed": {"Closed": false, "Create": {"hparams": {"dropout1": 0.5123726165661315, "dropout2": 0.6988116676979639, "n_filters1": 57, "n_filters2": 41, "learning_rate": 0.14624646818760453}, "request_id": "872fb3e1-39b9-40f8-b5ce-12ca53c225ed", "trial_seed": 1848480811}, "EarlyStoppedBySearcher": true}, "98a8c8cd-2aaf-4fbe-a358-7735337e11cd": {"Closed": false, "Create": {"hparams": {"dropout1": 0.7577371982482018, "dropout2": 0.757670158564443, "n_filters1": 35, "n_filters2": 50, "learning_rate": 0.5384580326825059}, "request_id": "98a8c8cd-2aaf-4fbe-a358-7735337e11cd", "trial_seed": 1457684159}, "EarlyStoppedBySearcher": true}, "a3e23956-04ee-489d-9135-7186a7941d28": {"Closed": false, "Create": {"hparams": {"dropout1": 0.6717156035663113, "dropout2": 0.6826566058616935, "n_filters1": 34, "n_filters2": 38, "learning_rate": 0.36435611193497525}, "request_id": "a3e23956-04ee-489d-9135-7186a7941d28", "trial_seed": 283271754}, "EarlyStoppedBySearcher": true}, "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8": {"Closed": false, "Create": {"hparams": {"dropout1": 0.5394040074405178, "dropout2": 0.5360441307525875, "n_filters1": 22, "n_filters2": 20, "learning_rate": 0.30396393450440096}, "request_id": "b6f8eca1-9c4a-4f9d-9730-ded009bf51d8", "trial_seed": 1659377654}, "EarlyStoppedBySearcher": true}, "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec": {"Closed": false, "Create": {"hparams": {"dropout1": 0.2799191061847552, "dropout2": 0.394202279713878, "n_filters1": 40, "n_filters2": 65, "learning_rate": 0.14238235074491137}, "request_id": "cea8e3d2-d430-4c69-a8c8-e7d2c35dc5ec", "trial_seed": 229268612}, "EarlyStoppedBySearcher": true}, "d882fdea-d2be-453d-bb93-831af2dc317b": {"Closed": false, "Create": {"hparams": {"dropout1": 0.39100440701610784, "dropout2": 0.49850349318825615, "n_filters1": 59, "n_filters2": 71, "learning_rate": 0.33813778901495695}, "request_id": "d882fdea-d2be-453d-bb93-831af2dc317b", "trial_seed": 491875076}, "EarlyStoppedBySearcher": true}}}`),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			e := tt.experiment()
			require.NoError(t, e.restore(tt.snapshot))
		})
	}
}
