package main

import (
	"testing"

	k8sV1 "k8s.io/api/core/v1"
	metaV1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"google.golang.org/api/compute/v1"
	"gotest.tools/assert"

	"github.com/determined-ai/determined/master/internal/config"
	"github.com/determined-ai/determined/master/internal/config/provconfig"
	"github.com/determined-ai/determined/master/pkg/aproto"
	"github.com/determined-ai/determined/master/pkg/model"
	"github.com/determined-ai/determined/master/pkg/schemas"
)

func TestUnmarshalMasterConfigurationViaViper(t *testing.T) {
	raw := `
resource_pools:
  - pool_name: default
    provider:
      type: gcp
      base_config:
        disks:
          - mode: READ_ONLY
            boot: false
            initializeParams:
              sourceImage: projects/determined-ai/global/images/determined-agent
              diskSizeGb: "200"
              diskType: projects/determined-ai/zones/us-central1-a/diskTypes/pd-ssd
            autoDelete: true
    task_container_defaults:
      cpu_pod_spec:
        apiVersion: v1
        kind: Pod
        metadata:
          labels:
            "app.kubernetes.io/camelCase": "cpu-label"
        spec:
          containers:
            - name: determined-container
task_container_defaults:
  cpu_pod_spec:
    apiVersion: v1
    kind: Pod
    metadata:
      labels:
        "app.kubernetes.io/camelCase": "cpu-label"
    spec:
      containers:
        - name: determined-container
webhooks:
    signing_key: testWebhookSigningKey
`

	expected := config.DefaultConfig()
	expected.Webhooks.SigningKey = "testWebhookSigningKey"
	providerConf := provconfig.DefaultConfig()
	providerConf.GCP = provconfig.DefaultGCPClusterConfig()
	providerConf.GCP.BaseConfig = &compute.InstanceProperties{
		Disks: []*compute.AttachedDisk{
			{
				AutoDelete: true,
				Mode:       "READ_ONLY",
				InitializeParams: &compute.AttachedDiskInitializeParams{
					DiskSizeGb:  200,
					DiskType:    "projects/determined-ai/zones/us-central1-a/diskTypes/pd-ssd",
					SourceImage: "projects/determined-ai/global/images/determined-agent",
				},
			},
		},
	}
	expected.TaskContainerDefaults = model.TaskContainerDefaultsConfig{
		ShmSizeBytes:      4294967296,
		NetworkMode:       "bridge",
		PreemptionTimeout: model.DefaultPreemptionTimeout,
	}
	expected.TaskContainerDefaults.CPUPodSpec = &k8sV1.Pod{
		TypeMeta: metaV1.TypeMeta{
			Kind:       "Pod",
			APIVersion: "v1",
		},
		Spec: k8sV1.PodSpec{
			Containers: []k8sV1.Container{
				{
					Name: "determined-container",
				},
			},
		},
		ObjectMeta: metaV1.ObjectMeta{
			Labels: map[string]string{
				"app.kubernetes.io/camelCase": "cpu-label",
			},
		},
	}

	//nolint:staticcheck
	expected.RootPoolsInternal = []config.ResourcePoolConfig{
		{
			PoolName:                 "default",
			Provider:                 providerConf,
			MaxAuxContainersPerAgent: 100,
			AgentReconnectWait:       model.Duration(aproto.AgentReconnectWait),
			TaskContainerDefaults:    &expected.TaskContainerDefaults,
		},
	}

	err := expected.Resolve()
	assert.NilError(t, err)
	config, err := mergeConfigIntoViper([]byte(raw))
	assert.NilError(t, err)
	assert.DeepEqual(t, config, expected)
}

func TestUnmarshalMasterConfiguration(t *testing.T) {
	config, err := getConfig(v.AllSettings())
	assert.NilError(t, err)

	c := schemas.WithDefaults(config.CheckpointStorage)

	if f := c.SaveTrialBest(); f <= 0 {
		t.Errorf("SaveTrialBest %d <= 0", f)
	}
}

func TestApplyBackwardsCompatibility(t *testing.T) {
	type testcase struct {
		name     string
		before   map[string]interface{}
		expected map[string]interface{}
		err      error
	}
	tcs := []testcase{
		{
			before: map[string]interface{}{
				"scheduler": map[string]interface{}{
					"fit": "best",
				},
				"provisioner": map[string]interface{}{
					"max_idle_agent_period":     "30s",
					"max_agent_starting_period": "30s",
				},
			},
			expected: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type": "agent",
					"scheduler": map[string]interface{}{
						"fitting_policy": "best",
						"type":           "priority",
					},
					"default_cpu_resource_pool": "default",
					"default_gpu_resource_pool": "default",
				},
				"resource_pools": []map[string]interface{}{
					{
						"pool_name": "default",
						"provider": map[string]interface{}{
							"max_idle_agent_period":     "30s",
							"max_agent_starting_period": "30s",
						},
					},
				},
			},
		},
		{
			before: map[string]interface{}{
				"scheduler": map[string]interface{}{
					"fit": "best",
					"resource_provider": map[string]interface{}{
						"type":                "kubernetes",
						"master_service_name": "k8s-det",
					},
				},
			},
			expected: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type": "kubernetes",
					"scheduler": map[string]interface{}{
						"fitting_policy": "best",
						"type":           "priority",
					},
					"master_service_name": "k8s-det",
				},
			},
		},
		{
			name: "preemption timeout (__internal set)",
			before: map[string]interface{}{
				"__internal": map[string]interface{}{
					"audit_logging_enabled": false,
					"preemption_timeout":    "1h10m10s",
				},
				"task_container_defaults": map[string]interface{}{
					"shm_size_bytes": 4294967296,
				},
			},
			expected: map[string]interface{}{
				"__internal": map[string]interface{}{
					"audit_logging_enabled": false,
				},
				"task_container_defaults": map[string]interface{}{
					"shm_size_bytes":     4294967296,
					"preemption_timeout": 4210,
				},
			},
		},
		{
			name: "preemption timeout (both set)",
			before: map[string]interface{}{
				"__internal": map[string]interface{}{
					"audit_logging_enabled": false,
					"preemption_timeout":    "1h10m10s",
				},
				"task_container_defaults": map[string]interface{}{
					"shm_size_bytes":     4294967296,
					"preemption_timeout": 1000,
				},
			},
			expected: map[string]interface{}{
				"__internal": map[string]interface{}{
					"audit_logging_enabled": false,
				},
				"task_container_defaults": map[string]interface{}{
					"shm_size_bytes":     4294967296,
					"preemption_timeout": 1000,
				},
			},
		},
		{
			name: "determined master ip to host rename, ip set",
			before: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                 "kubernetes",
					"determined_master_ip": "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                 "kubernetes",
						"determined_master_ip": "10.0.0.2",
					},
				},
			},
			expected: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                   "kubernetes",
					"determined_master_host": "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                   "kubernetes",
						"determined_master_host": "10.0.0.2",
					},
				},
			},
		},
		{
			name: "determined master ip to host rename, both set",
			before: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                   "kubernetes",
					"determined_master_host": "10.0.0.1",
					"determined_master_ip":   "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                   "kubernetes",
						"determined_master_host": "10.0.0.2",
						"determined_master_ip":   "10.0.0.2",
					},
				},
			},
			expected: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                   "kubernetes",
					"determined_master_host": "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                   "kubernetes",
						"determined_master_host": "10.0.0.2",
					},
				},
			},
		},
		{
			name: "determined master ip to host rename, host set get left alone",
			before: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                   "kubernetes",
					"determined_master_host": "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                   "kubernetes",
						"determined_master_host": "10.0.0.2",
					},
				},
			},
			expected: map[string]interface{}{
				"resource_manager": map[string]interface{}{
					"type":                   "kubernetes",
					"determined_master_host": "10.0.0.1",
				},
				"additional_resource_managers": []map[string]any{
					{
						"type":                   "kubernetes",
						"determined_master_host": "10.0.0.2",
					},
				},
			},
		},
	}
	for ix := range tcs {
		tc := tcs[ix]
		t.Run(tc.name, func(t *testing.T) {
			after, err := applyBackwardsCompatibility(tc.before)
			assert.Equal(t, err, tc.err)
			assert.DeepEqual(t, after, tc.expected)
		})
	}
}
