import pytest
import responses

from determined.common import api
from determined.common.api import bindings, errors
from determined.common.experimental import workspace
from tests.fixtures import api_responses

_MASTER = "http://localhost:8080"


@pytest.fixture
def standard_session() -> api.Session:
    return api.Session(_MASTER, "username", "token", cert=None)


@pytest.fixture
def single_item_workspaces() -> bindings.v1GetWorkspacesResponse:
    sample_workspace = api_responses.sample_get_workspace().workspace
    single_item_pagination = bindings.v1Pagination(endIndex=1, startIndex=0, total=1)
    return bindings.v1GetWorkspacesResponse(
        workspaces=[sample_workspace], pagination=single_item_pagination
    )


@pytest.fixture
def single_item_rps_bound_to_workspace() -> bindings.v1ListRPsBoundToWorkspaceResponse:
    single_item_pagination = bindings.v1Pagination(endIndex=1, startIndex=0, total=1)
    return bindings.v1ListRPsBoundToWorkspaceResponse(
        resourcePools=["foo"], pagination=single_item_pagination
    )


@pytest.fixture
def multi_item_rps_bound_to_workspace() -> bindings.v1ListRPsBoundToWorkspaceResponse:
    multi_items_pagination = bindings.v1Pagination(endIndex=2, startIndex=0, total=2)
    return bindings.v1ListRPsBoundToWorkspaceResponse(
        resourcePools=["foo", "bar"], pagination=multi_items_pagination
    )


@responses.activate
def test_workspace_constructor_requires_exactly_one_of_id_or_name(
    standard_session: api.Session,
    single_item_workspaces: bindings.v1GetWorkspacesResponse,
) -> None:
    responses.get(f"{_MASTER}/api/v1/workspaces", json=single_item_workspaces.to_json())

    with pytest.raises(ValueError):
        workspace.Workspace(session=standard_session, workspace_id=1, workspace_name="foo")

    with pytest.raises(ValueError):
        workspace.Workspace(session=standard_session)

    workspace.Workspace(session=standard_session, workspace_id=1)
    workspace.Workspace(session=standard_session, workspace_name="foo")


@responses.activate
def test_workspace_constructor_errors_when_no_workspace_found(
    standard_session: api.Session,
) -> None:
    resp = bindings.v1GetWorkspacesResponse(
        workspaces=[], pagination=api_responses.empty_get_pagination()
    )

    responses.get(f"{_MASTER}/api/v1/workspaces", json=resp.to_json())
    with pytest.raises(errors.NotFoundException):
        workspace.Workspace(session=standard_session, workspace_name="not_found")


@responses.activate
def test_workspace_constructor_populates_id_from_name(
    standard_session: api.Session,
    single_item_workspaces: bindings.v1GetWorkspacesResponse,
) -> None:
    workspace_id = 1
    workspace_name = "foo"
    single_item_workspaces.workspaces[0].id = workspace_id
    single_item_workspaces.workspaces[0].name = workspace_name
    responses.get(f"{_MASTER}/api/v1/workspaces", json=single_item_workspaces.to_json())

    ws = workspace.Workspace(session=standard_session, workspace_name=workspace_name)
    assert ws.id == workspace_id


def test_workspace_constructor_doesnt_populate_name_from_id(standard_session: api.Session) -> None:
    ws = workspace.Workspace(session=standard_session, workspace_id=1)
    assert ws.name is None
