import argparse
import json
import os
import re
import subprocess
from typing import Any, Dict

import psycopg2
from psycopg2 import extensions, sql

# Database schema is described.
"""
CREATE TABLE perf_test_runs (
  id     integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  commit text NOT NULL,
  branch text NOT NULL,
  time   timestamptz NOT NULL DEFAULT NOW()
);

CREATE INDEX perf_test_runs_ix ON perf_test_runs(commit);
CREATE INDEX perf_test_brach_ix ON perf_test_runs(branch);
CREATE INDEX perf_test_time_ix ON perf_test_runs(time);

CREATE TABLE perf_tests (
  id integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  test_name text NOT NULL,
  run_id int REFERENCES perf_test_runs(id),
  avg    double precision,
  min    double precision,
  med    double precision,
  max    double precision,
  p90    double precision,
  p95    double precision,
  passes int NOT NULL DEFAULT 0,
  fails  int NOT NULL DEFAULT 0
);

CREATE INDEX perf_tests_test_name_ix ON perf_tests(test_name);
CREATE INDEX perf_tests_test_run_id_ix ON perf_tests(run_id);

CREATE TABLE migration_runs (
  id           integer PRIMARY KEY GENERATED BY DEFAULT AS IDENTITY,
  commit       text NOT NULL,
  branch       text NOT NULL,
  time         timestamptz NOT NULL DEFAULT NOW(),
  duration_ms  double precision NOT NULL,
  from_version BIGINT NOT NULL,
  to_version BIGINT NOT NULL
);

CREATE INDEX migration_runs_ix ON migration_runs(commit);
CREATE INDEX miration_branch_ix ON migration_runs(branch);
CREATE INDEX migration_time_ix ON migration_runs(time);
"""


def get_perf_run_info() -> Dict[str, Any]:
    try:
        commit = subprocess.check_output(
            ["git", "log", "-1", "--pretty=format:%H"], universal_newlines=True
        ).strip()
    except subprocess.CalledProcessError:
        commit = "unknown"

    branch = os.environ.get("CIRCLE_BRANCH", "unknown")

    out = {"commit": commit, "branch": branch}
    print("Debug adding:", out)

    return out


def insert_test_result_to_db(
    conn: extensions.connection, info: Dict[str, Any], test_cases: Dict[str, Any]
) -> None:
    with conn.cursor() as cursor:
        # Insert into perf_test_runs table
        run_sql_query = sql.SQL(
            """
            INSERT INTO perf_test_runs (commit, branch)
            VALUES ({commit}, {branch})
            RETURNING id;
        """
        ).format(commit=sql.Literal(info["commit"]), branch=sql.Literal(info["branch"]))
        cursor.execute(run_sql_query)
        run_id = cursor.fetchone()[0]

        # Insert into perf_tests table for each test case
        for test_case in test_cases.values():
            test_sql_query = sql.SQL(
                """
    INSERT INTO perf_tests (test_name, run_id, avg, min, med, max, p90, p95, passes, fails)
    VALUES ({test_name}, {run_id}, {avg}, {min}, {med}, {max}, {p90}, {p95}, {passes}, {fails});
            """
            ).format(
                test_name=sql.Literal(test_case["test_name"]),
                run_id=sql.Literal(run_id),
                avg=sql.Literal(test_case.get("avg", None)),
                min=sql.Literal(test_case.get("min", None)),
                med=sql.Literal(test_case.get("med", None)),
                max=sql.Literal(test_case.get("max", None)),
                p90=sql.Literal(test_case.get("p90", None)),
                p95=sql.Literal(test_case.get("p95", None)),
                passes=sql.Literal(test_case.get("passes", None)),
                fails=sql.Literal(test_case.get("fails", None)),
            )
            cursor.execute(test_sql_query)

    conn.commit()


def parse_test_run_results(file_name: str) -> Dict[str, Any]:
    with open(file_name) as f:
        summary = json.load(f)

    out = {}
    for m in summary.get("metrics", {}):
        match = re.search(r"{group: ::(.*?)}", m)
        if match:
            result = match.group(1)
            out[result] = {
                "test_name": result,
                "avg": summary["metrics"][m].get("avg", None),
                "min": summary["metrics"][m].get("min", None),
                "med": summary["metrics"][m].get("med", None),
                "max": summary["metrics"][m].get("max", None),
                "p90": summary["metrics"][m].get("p(90)", None),
                "p95": summary["metrics"][m].get("p(95)", None),
            }

    groups = summary.get("root_group", {}).get("groups", {})
    for g in groups:
        if g == "setup":
            continue

        if g not in out:
            out[g] = {
                "test_name": g,
            }
        out[g]["passes"] = groups[g].get("checks", {}).get("200 response", {}).get("passes", None)
        out[g]["fails"] = groups[g].get("checks", {}).get("200 response", {}).get("fails", None)

    print("Debug adding")
    for o in out:
        print(o, out[o])

    return out


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Parse test results path")
    parser.add_argument("json_k6_summary", help="File ")
    args = parser.parse_args()
    perf_tests = parse_test_run_results(args.json_k6_summary)

    perf_run = get_perf_run_info()

    db_params = {
        "host": os.environ["PERF_RESULT_DB_HOST"],
        "user": os.environ["PERF_RESULT_DB_USER"],
        "password": os.environ["PERF_RESULT_DB_PASS"],
        "dbname": "postgres",
    }

    connection = psycopg2.connect(**db_params)
    insert_test_result_to_db(connection, perf_run, perf_tests)
