// Copyright (C) Explorer++ Project
// SPDX-License-Identifier: GPL-3.0-only
// See LICENSE in the top level directory

#include "stdafx.h"
#include "ShellBrowserImpl.h"
#include "Config.h"
#include "DocumentServiceProvider.h"
#include "HistoryEntry.h"
#include "IconFetcher.h"
#include "ItemData.h"
#include "MainResource.h"
#include "ShellEnumerator.h"
#include "ShellNavigationController.h"
#include "ShellView.h"
#include "ViewModes.h"
#include "WebBrowserApp.h"
#include "../Helper/ListViewHelper.h"
#include "../Helper/Macros.h"
#include "../Helper/ShellHelper.h"
#include "../Helper/WinRTBaseWrapper.h"
#include <wil/com.h>
#include <propkey.h>
#include <propvarutil.h>
#include <list>

HRESULT ShellBrowserImpl::Navigate(NavigateParams &navigateParams)
{
	SetCursor(LoadCursor(nullptr, IDC_WAIT));

	auto resetCursor = wil::scope_exit([] { SetCursor(LoadCursor(nullptr, IDC_ARROW)); });

	m_navigationStartedSignal(navigateParams);

	std::vector<ItemInfo_t> items;
	HRESULT hr = PerformEnumeration(navigateParams, items);

	if (FAILED(hr))
	{
		m_navigationFailedSignal(navigateParams);
		return hr;
	}

	OnEnumerationCompleted(std::move(items), navigateParams);

	return hr;
}

void ShellBrowserImpl::PrepareToChangeFolders()
{
	if (m_bFolderVisited)
	{
		SaveColumnWidths();
	}

	ClearPendingResults();

	m_shellChangeWatcher.StopWatchingAll();

	StoreCurrentlySelectedItems();

	ListView_DeleteAllItems(m_hListView);

	if (m_bFolderVisited)
	{
		ResetFolderState();
	}
}

void ShellBrowserImpl::ClearPendingResults()
{
	m_columnThreadPool.clear_queue();
	m_columnResults.clear();

	m_iconFetcher->ClearQueue();

	m_thumbnailThreadPool.clear_queue();
	m_thumbnailResults.clear();

	m_infoTipsThreadPool.clear_queue();
	m_infoTipResults.clear();
}

void ShellBrowserImpl::ResetFolderState()
{
	ListView_SetImageList(m_hListView, nullptr, LVSIL_SMALL);
	ListView_SetImageList(m_hListView, nullptr, LVSIL_NORMAL);

	ListView_RemoveAllGroups(m_hListView);

	m_directoryState = DirectoryState();

	EnterCriticalSection(&m_csDirectoryAltered);
	m_AlteredList.clear();
	LeaveCriticalSection(&m_csDirectoryAltered);

	m_itemInfoMap.clear();

	m_renamedItemOldPidl.reset();
}

void ShellBrowserImpl::StoreCurrentlySelectedItems()
{
	auto *entry = m_navigationController->GetCurrentEntry();

	if (!entry)
	{
		return;
	}

	auto selectedItems = GetSelectedItemPidls();
	entry->SetSelectedItems(selectedItems);
}

HRESULT ShellBrowserImpl::PerformEnumeration(NavigateParams &navigateParams,
	std::vector<ShellBrowserImpl::ItemInfo_t> &items)
{
	// Note that although standard shortcuts (.lnk files) are currently handled outside this class,
	// symlinks and virtual link objects aren't, so they will be handled here.
	// Navigating to the target pidl is important for folders like the quick access folder. Although
	// navigating directly to a recent/pinned folder works as expected, directory monitoring doesn't
	// work. Presumably, that's because directory change notifications are only generated for the
	// original (target) directory. To have things work correctly, the navigation needs to proceed
	// to the original folder instead.
	// Note that this call simply retrieves the target item, but doesn't attempt to resolve it. That
	// matches the behavior of Explorer. For example, if a symlink to a directory is created and the
	// target directory is then removed, Explorer will try to navigate to the target, without
	// attempting to resolve the link.
	unique_pidl_absolute targetPidl;
	HRESULT hr = MaybeGetLinkTarget(navigateParams.pidl.Raw(), targetPidl);

	if (SUCCEEDED(hr))
	{
		navigateParams.pidl = targetPidl.get();
	}

	wil::com_ptr_nothrow<IShellFolder> parent;
	PCITEMID_CHILD child;
	RETURN_IF_FAILED(SHBindToParent(navigateParams.pidl.Raw(), IID_PPV_ARGS(&parent), &child));

	SFGAOF attr = SFGAO_FILESYSTEM;
	RETURN_IF_FAILED(parent->GetAttributesOf(1, &child, &attr));

	std::wstring parsingPath;
	RETURN_IF_FAILED(GetDisplayName(parent.get(), child, SHGDN_FORPARSING, parsingPath));

	RETURN_IF_FAILED(
		EnumerateFolder(navigateParams.pidl.Raw(), m_hOwner, m_folderSettings.showHidden, items));

	PrepareToChangeFolders();

	m_directoryState.pidlDirectory.reset(ILCloneFull(navigateParams.pidl.Raw()));
	m_directoryState.directory = parsingPath;
	m_directoryState.virtualFolder = WI_IsFlagClear(attr, SFGAO_FILESYSTEM);
	m_uniqueFolderId++;

	SetActiveColumnSet();
	VerifySortMode();
	SetViewModeInternal(m_folderSettings.viewMode);

	NotifyShellOfNavigation(navigateParams.pidl.Raw());

	m_navigationCommittedSignal(navigateParams);

	return S_OK;
}

HRESULT ShellBrowserImpl::EnumerateFolder(PCIDLIST_ABSOLUTE pidlDirectory, HWND owner,
	bool showHidden, std::vector<ShellBrowserImpl::ItemInfo_t> &items)
{
	wil::com_ptr_nothrow<IShellFolder> shellFolder;
	RETURN_IF_FAILED(BindToIdl(pidlDirectory, IID_PPV_ARGS(&shellFolder)));

	ShellEnumerator::Flags flags = ShellEnumerator::Flags::Standard;

	if (showHidden)
	{
		WI_SetFlag(flags, ShellEnumerator::Flags::IncludeHidden);
	}

	ShellEnumerator enumerator;
	std::vector<PidlChild> outputPidls;
	RETURN_IF_FAILED(enumerator.EnumerateDirectory(shellFolder.get(), owner, flags, outputPidls));

	for (const auto &pidl : outputPidls)
	{
		auto item = GetItemInformation(shellFolder.get(), pidlDirectory, pidl.Raw());

		if (item)
		{
			items.push_back(std::move(*item));
		}
	}

	return S_OK;
}

void ShellBrowserImpl::NotifyShellOfNavigation(PCIDLIST_ABSOLUTE pidl)
{
	if (m_config->replaceExplorerMode == +DefaultFileManager::ReplaceExplorerMode::None)
	{
		return;
	}

	HRESULT hr = RegisterShellWindowIfNecessary(pidl);

	if (FAILED(hr))
	{
		return;
	}

	wil::unique_variant pidlVariant;
	hr = InitVariantFromBuffer(pidl, ILGetSize(pidl), &pidlVariant);

	if (FAILED(hr))
	{
		return;
	}

	m_shellWindows->OnNavigate(m_shellWindowCookie.get(), &pidlVariant);
}

HRESULT ShellBrowserImpl::RegisterShellWindowIfNecessary(PCIDLIST_ABSOLUTE pidl)
{
	if (m_shellWindowRegistered)
	{
		return S_OK;
	}

	HRESULT hr = RegisterShellWindow(pidl);

	if (SUCCEEDED(hr))
	{
		m_shellWindowRegistered = true;
	}

	return hr;
}

// When the shell needs to find an existing window (to select a specific item), it will go through
// the following process:
//
// 1. Use IShellWindows::FindWindowSW() to query for a window of type SWC_BROWSER with the necessary
//    pidl.
// 2. Query for the IWebBrowserApp interface for that window (using the IDispatch interface that's
//    registered).
// 3. Call IWebBrowserApp::get_Document() to retrieve the IDispatch interface for the document.
// 4. Call QueryInterface() on the IDispatch interface to request IServiceProvider.
// 5. Call IServiceProvider::QueryService() to request an IID_IFolderView service and IID_IShellView
//    interface.
// 6. Call IShellView::SelectItem() to select the appropriate item.
//
// Note that the shell will also bring the window it finds to the foreground as part of this
// process.
//
// A similar process also occurs when simply searching for an existing shell window. For example, if
// the user double clicks the recycle bin icon on the desktop, any existing recycle bin window will
// be brought to the foreground if present.
HRESULT ShellBrowserImpl::RegisterShellWindow(PCIDLIST_ABSOLUTE pidl)
{
	if (!m_shellWindows)
	{
		return E_FAIL;
	}

	wil::unique_variant pidlVariant;
	RETURN_IF_FAILED(InitVariantFromBuffer(pidl, ILGetSize(pidl), &pidlVariant));

	wil::unique_variant empty;

	m_shellWindowCookie.associate(m_shellWindows.get());

	// Note that while the documentation states that the pidl variant (the second argument) must be
	// "of type VT_VARIANT | VT_BYREF", it appears that's not actually necessary (and would be more
	// complicated, since there would need to be two variants).
	// Also, an empty variant must be supplied for the third argument (in contrast to the
	// documentation, which states that null is allowed).
	// Additionally, the shell window type that's set is SWC_BROWSER, which the documentation states
	// is for "An Internet Explorer (Iexplore.exe) browser window.". While a value like SWC_3RDPARTY
	// or SWC_EXPLORER might make more sense, the shell looks for Explorer windows using the
	// SWC_BROWSER type. So for a window to be found, it needs to use that specific type.
	// Finally, there are two possible ways of supplying a pidl: using
	// IShellWindows::RegisterPending() or IShellWindows::OnNavigate(). The second
	// method is called during each navigation anyway, so it would be simpler to just use that,
	// however calling IShellWindows::RegisterPending() is necessary.
	// If a new instance is launched as part of a SHOpenFolderAndSelectItems() call, then no
	// selection request will be made unless IShellWindows::RegisterPending() is called. In this
	// case, it's redundant, since IShellWindows::OnNavigate() will be called anyway, which will
	// supply a pidl, but it appears that that's not enough.
	// Therefore, that's the only reason this method is called.
	RETURN_IF_FAILED(m_shellWindows->RegisterPending(GetCurrentThreadId(), &pidlVariant, &empty,
		SWC_BROWSER, &m_shellWindowCookie));

	auto document = winrt::make_self<DocumentServiceProvider>();
	auto shellView = winrt::make_self<ShellView>(weak_from_this(), m_tabNavigation, true);
	document->RegisterService(IID_IFolderView, shellView.get());

	auto browserApp = winrt::make_self<WebBrowserApp>(m_hOwner, document.get());

	// Registering the same window multiple times (ultimately with different pidls) is odd, but
	// appears to work fine (since a shell window is uniquely identified through a cookie, rather
	// than a window handle).
	// Registering each listview wouldn't work, as it wouldn't make sense for the shell to bring the
	// listview windows to the foreground or activate them (doing so would break the way tabs are
	// managed).
	// This has implications for the way the appropriate tab is selected. Since the shell will only
	// bring the top-level window to the foreground, the appropriate tab will be selected when the
	// item selection is set (via a call to the IShellView instance set up above).
	// Note that the cast from HWND to long causes warnings, but the warnings can be safely ignored.
	// Only the lower 32 bits of a handle are important (see the discussion at
	// https://stackoverflow.com/q/1822667).
	long registeredCookie;
#pragma warning(push)
#pragma warning(                                                                                   \
	disable : 4311 4302) // 'reinterpret_cast': pointer truncation from 'HWND' to 'long',
						 // 'reinterpret_cast': truncation from 'HWND' to 'long'
	RETURN_IF_FAILED(m_shellWindows->Register(browserApp.get(), reinterpret_cast<long>(m_hOwner),
		SWC_BROWSER, &registeredCookie));
#pragma warning(pop)

	// The call to RegisterPending() above is passed the thread ID. The call to Register() will use
	// that thread ID to link a pending window to the specified window handle. That means the cookie
	// values for the two calls should be the same - since they refer to the same window instance.
	DCHECK(registeredCookie == m_shellWindowCookie.get());

	return S_OK;
}

std::optional<int> ShellBrowserImpl::AddItemInternal(IShellFolder *shellFolder,
	PCIDLIST_ABSOLUTE pidlDirectory, PCITEMID_CHILD pidlChild, int itemIndex, BOOL setPosition)
{
	auto itemInfo = GetItemInformation(shellFolder, pidlDirectory, pidlChild);

	if (!itemInfo)
	{
		return std::nullopt;
	}

	return AddItemInternal(itemIndex, std::move(*itemInfo), setPosition);
}

int ShellBrowserImpl::AddItemInternal(int itemIndex, ItemInfo_t itemInfo, BOOL setPosition)
{
	int itemId = GenerateUniqueItemId();
	m_itemInfoMap.insert({ itemId, std::move(itemInfo) });

	AwaitingAdd_t awaitingAdd;

	if (itemIndex == -1)
	{
		awaitingAdd.iItem =
			m_directoryState.numItems + static_cast<int>(m_directoryState.awaitingAddList.size());
	}
	else
	{
		awaitingAdd.iItem = itemIndex;
	}

	awaitingAdd.iItemInternal = itemId;
	awaitingAdd.bPosition = setPosition;
	awaitingAdd.iAfter = itemIndex - 1;

	m_directoryState.awaitingAddList.push_back(awaitingAdd);

	return itemId;
}

std::optional<ShellBrowserImpl::ItemInfo_t> ShellBrowserImpl::GetItemInformation(
	IShellFolder *shellFolder, PCIDLIST_ABSOLUTE pidlDirectory, PCITEMID_CHILD pidlChild)
{
	ItemInfo_t itemInfo;

	unique_pidl_absolute pidlItem(ILCombine(pidlDirectory, pidlChild));

	itemInfo.pidlComplete.reset(ILCloneFull(pidlItem.get()));
	itemInfo.pridl.reset(ILCloneChild(pidlChild));

	std::wstring parsingName;
	HRESULT hr = GetDisplayName(shellFolder, pidlChild, SHGDN_FORPARSING, parsingName);

	if (FAILED(hr))
	{
		return std::nullopt;
	}

	itemInfo.parsingName = parsingName;

	ULONG attributes = SFGAO_FOLDER | SFGAO_FILESYSTEM;
	PCITEMID_CHILD items[] = { pidlChild };
	hr = shellFolder->GetAttributesOf(1, items, &attributes);

	if (FAILED(hr))
	{
		return std::nullopt;
	}

	SHGDNF displayNameFlags = SHGDN_INFOLDER;

	unique_pidl_absolute recycleBinPidl;
	hr = SHGetKnownFolderIDList(FOLDERID_RecycleBinFolder, KF_FLAG_DEFAULT, nullptr,
		wil::out_param(recycleBinPidl));

	bool isRecycleBin = SUCCEEDED(hr) && ArePidlsEquivalent(pidlDirectory, recycleBinPidl.get());

	// SHGDN_INFOLDER | SHGDN_FORPARSING is used to ensure that the name retrieved for a filesystem
	// file contains an extension, even if extensions are hidden in Windows Explorer. When using
	// SHGDN_INFOLDER by itself, the resulting name won't contain an extension if extensions are
	// hidden in Windows Explorer.
	// Note that the recycle bin is excluded here, as the parsing names for the items are completely
	// different to their regular display names.
	if (!isRecycleBin && WI_IsFlagSet(attributes, SFGAO_FILESYSTEM)
		&& WI_IsFlagClear(attributes, SFGAO_FOLDER))
	{
		WI_SetFlag(displayNameFlags, SHGDN_FORPARSING);
	}

	std::wstring displayName;
	hr = GetDisplayName(shellFolder, pidlChild, displayNameFlags, displayName);

	if (FAILED(hr))
	{
		return std::nullopt;
	}

	itemInfo.displayName = displayName;

	std::wstring editingName;
	hr = GetDisplayName(shellFolder, pidlChild, SHGDN_INFOLDER | SHGDN_FOREDITING, editingName);

	if (FAILED(hr))
	{
		return std::nullopt;
	}

	itemInfo.editingName = editingName;

	if (PathIsRoot(parsingName.c_str()))
	{
		itemInfo.bDrive = TRUE;
		StringCchCopy(itemInfo.szDrive, SIZEOF_ARRAY(itemInfo.szDrive), parsingName.c_str());
	}
	else
	{
		itemInfo.bDrive = FALSE;
	}

	WIN32_FIND_DATA wfd;
	hr = SHGetDataFromIDList(shellFolder, pidlChild, SHGDFIL_FINDDATA, &wfd, sizeof(wfd));

	if (FAILED(hr))
	{
		hr = ExtractFindDataUsingPropertyStore(shellFolder, pidlChild, wfd);
	}

	if (SUCCEEDED(hr))
	{
		itemInfo.wfd = wfd;
		itemInfo.isFindDataValid = true;
	}
	else
	{
		StringCchCopy(itemInfo.wfd.cFileName, SIZEOF_ARRAY(itemInfo.wfd.cFileName),
			displayName.c_str());

		if (WI_IsFlagSet(attributes, SFGAO_FOLDER))
		{
			WI_SetFlag(itemInfo.wfd.dwFileAttributes, FILE_ATTRIBUTE_DIRECTORY);
		}
	}

	return std::move(itemInfo);
}

HRESULT ShellBrowserImpl::ExtractFindDataUsingPropertyStore(IShellFolder *shellFolder,
	PCITEMID_CHILD pidlChild, WIN32_FIND_DATA &output)
{
	wil::com_ptr_nothrow<IPropertyStoreFactory> factory;
	HRESULT hr = shellFolder->BindToObject(pidlChild, nullptr, IID_PPV_ARGS(&factory));

	if (FAILED(hr))
	{
		return hr;
	}

	wil::com_ptr_nothrow<IPropertyStore> store;
	PROPERTYKEY keys[] = { PKEY_FindData };
	hr = factory->GetPropertyStoreForKeys(keys, SIZEOF_ARRAY(keys), GPS_FASTPROPERTIESONLY,
		IID_PPV_ARGS(&store));

	if (FAILED(hr))
	{
		return hr;
	}

	wil::unique_prop_variant findDataProp;
	hr = store->GetValue(PKEY_FindData, &findDataProp);

	if (FAILED(hr))
	{
		return hr;
	}

	if (PropVariantGetElementCount(findDataProp) != sizeof(WIN32_FIND_DATA))
	{
		return hr;
	}

	WIN32_FIND_DATA wfd;
	hr = PropVariantToBuffer(findDataProp, &wfd, sizeof(wfd));

	if (FAILED(hr))
	{
		return hr;
	}

	output = wfd;

	return hr;
}

void ShellBrowserImpl::OnEnumerationCompleted(std::vector<ShellBrowserImpl::ItemInfo_t> &&items,
	const NavigateParams &navigateParams)
{
	for (auto &item : items)
	{
		AddItemInternal(-1, std::move(item), FALSE);
	}

	/* Stop the list view from redrawing itself each time is inserted.
	Redrawing will be allowed once all items have being inserted.
	(reduces lag when a large number of items are going to be inserted). */
	SendMessage(m_hListView, WM_SETREDRAW, FALSE, NULL);

	InsertAwaitingItems();
	SortFolder();

	ListView_EnsureVisible(m_hListView, 0, FALSE);

	/* Allow the listview to redraw itself once again. */
	SendMessage(m_hListView, WM_SETREDRAW, TRUE, NULL);

	/* Set the focus back to the first item. */
	ListView_SetItemState(m_hListView, 0, LVIS_FOCUSED, LVIS_FOCUSED);

	// A history entry should be created when the navigation is committed, so there should always be
	// a current entry here, and that entry should be for the current navigation.
	auto currentEntry = m_navigationController->GetCurrentEntry();
	CHECK(currentEntry);
	DCHECK(currentEntry->GetPidl() == navigateParams.pidl);

	auto selectedItems = currentEntry->GetSelectedItems();
	SelectItems(selectedItems);

	if (navigateParams.navigationType == NavigationType::Up)
	{
		SelectItems({ navigateParams.originalPidl });
	}

	if (m_config->shellChangeNotificationType == ShellChangeNotificationType::All
		|| (m_config->shellChangeNotificationType == ShellChangeNotificationType::NonFilesystem
			&& m_directoryState.virtualFolder))
	{
		StartDirectoryMonitoring(m_directoryState.pidlDirectory.get());
	}

	m_bFolderVisited = TRUE;

	m_navigationCompletedSignal(navigateParams);
}

void ShellBrowserImpl::InsertAwaitingItems()
{
	int nPrevItems = ListView_GetItemCount(m_hListView);

	if (nPrevItems == 0 && m_directoryState.awaitingAddList.empty())
	{
		m_directoryState.numItems = 0;

		return;
	}

	/* Make the listview allocate space (for internal data structures)
	for all the items at once, rather than individually.
	Acts as a speed optimization. */
	ListView_SetItemCount(m_hListView, m_directoryState.awaitingAddList.size() + nPrevItems);

	if (m_folderSettings.autoArrange)
	{
		ListViewHelper::SetAutoArrange(m_hListView, FALSE);
	}

	int nAdded = 0;
	std::optional<int> itemToRename;

	for (const auto &awaitingItem : m_directoryState.awaitingAddList)
	{
		const auto &itemInfo = m_itemInfoMap.at(awaitingItem.iItemInternal);

		if (IsFileFiltered(itemInfo))
		{
			m_directoryState.filteredItemsList.insert(awaitingItem.iItemInternal);
			continue;
		}

		BasicItemInfo_t basicItemInfo = getBasicItemInfo(awaitingItem.iItemInternal);
		std::wstring filename = ProcessItemFileName(basicItemInfo, m_config->globalFolderSettings);

		LVITEM lv;
		lv.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM;

		if (m_folderSettings.showInGroups)
		{
			int groupId = DetermineItemGroup(awaitingItem.iItemInternal);

			lv.mask |= LVIF_GROUPID;
			lv.iGroupId = groupId;

			EnsureGroupExistsInListView(groupId);
		}

		lv.iItem = awaitingItem.iItem;
		lv.iSubItem = 0;

		auto firstColumn = GetFirstCheckedColumn();

		if ((m_folderSettings.viewMode == +ViewMode::Details)
			&& firstColumn.type != +ColumnType::Name)
		{
			lv.pszText = LPSTR_TEXTCALLBACK;
		}
		else
		{
			lv.pszText = filename.data();
		}

		lv.iImage = I_IMAGECALLBACK;
		lv.lParam = awaitingItem.iItemInternal;

		/* Insert the item into the list view control. */
		int iItemIndex = ListView_InsertItem(m_hListView, &lv);

		if (awaitingItem.bPosition && m_folderSettings.viewMode != +ViewMode::Details)
		{
			POINT ptItem;

			if (awaitingItem.iAfter != -1)
			{
				ListView_GetItemPosition(m_hListView, awaitingItem.iAfter, &ptItem);
			}
			else
			{
				ptItem.x = 0;
				ptItem.y = 0;
			}

			/* The item will end up in the position AFTER iAfter. */
			ListView_SetItemPosition32(m_hListView, iItemIndex, ptItem.x, ptItem.y);
		}

		if (m_folderSettings.viewMode == +ViewMode::Tiles)
		{
			SetTileViewItemInfo(iItemIndex, awaitingItem.iItemInternal);
		}

		if (m_directoryState.queuedRenameItem
			&& ArePidlsEquivalent(itemInfo.pidlComplete.get(),
				m_directoryState.queuedRenameItem.get()))
		{
			itemToRename = iItemIndex;
		}

		auto selectItr = std::find_if(m_directoryState.filesToSelect.begin(),
			m_directoryState.filesToSelect.end(), [&itemInfo](const auto &pidl)
			{ return ArePidlsEquivalent(pidl.Raw(), itemInfo.pidlComplete.get()); });

		if (selectItr != m_directoryState.filesToSelect.end())
		{
			ListViewHelper::SelectItem(m_hListView, iItemIndex, TRUE);

			int selectedCount = ListView_GetSelectedCount(m_hListView);

			if (selectedCount == 1)
			{
				ListViewHelper::FocusItem(m_hListView, iItemIndex, TRUE);
				ListView_EnsureVisible(m_hListView, iItemIndex, FALSE);
			}

			m_directoryState.filesToSelect.erase(selectItr);
		}

		/* If the file is marked as hidden, ghost it out. */
		if (itemInfo.wfd.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
		{
			ListView_SetItemState(m_hListView, iItemIndex, LVIS_CUT, LVIS_CUT);
		}

		/* Add the current file's size to the running size of the current directory. */
		/* A folder may or may not have 0 in its high file size member.
		It should either be zeroed, or never counted. */
		ULARGE_INTEGER ulFileSize;
		ulFileSize.LowPart = itemInfo.wfd.nFileSizeLow;
		ulFileSize.HighPart = itemInfo.wfd.nFileSizeHigh;

		m_directoryState.totalDirSize += ulFileSize.QuadPart;

		nAdded++;
	}

	if (m_folderSettings.autoArrange)
	{
		ListViewHelper::SetAutoArrange(m_hListView, TRUE);
	}

	m_directoryState.numItems = nPrevItems + nAdded;

	m_directoryState.awaitingAddList.clear();

	if (itemToRename)
	{
		m_directoryState.queuedRenameItem.reset();
		ListView_EditLabel(m_hListView, *itemToRename);
	}
}

BOOL ShellBrowserImpl::IsFileFiltered(const ItemInfo_t &itemInfo) const
{
	BOOL bHideSystemFile = FALSE;
	BOOL bFilenameFiltered = FALSE;

	if (m_folderSettings.applyFilter
		&& ((itemInfo.wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != FILE_ATTRIBUTE_DIRECTORY))
	{
		bFilenameFiltered = IsFilenameFiltered(itemInfo.displayName.c_str());
	}

	if (m_config->globalFolderSettings.hideSystemFiles)
	{
		bHideSystemFile =
			(itemInfo.wfd.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) == FILE_ATTRIBUTE_SYSTEM;
	}

	return bFilenameFiltered || bHideSystemFile;
}

void ShellBrowserImpl::RemoveItem(int iItemInternal)
{
	ULARGE_INTEGER ulFileSize;
	LVFINDINFO lvfi;
	int iItem;
	int nItems;

	if (iItemInternal == -1)
	{
		return;
	}

	/* Take the file size of the removed file away from the total
	directory size. */
	ulFileSize.LowPart = m_itemInfoMap.at(iItemInternal).wfd.nFileSizeLow;
	ulFileSize.HighPart = m_itemInfoMap.at(iItemInternal).wfd.nFileSizeHigh;

	m_directoryState.totalDirSize -= ulFileSize.QuadPart;

	/* Locate the item within the listview.
	Could use filename, providing removed
	items are always deleted before new
	items are inserted. */
	lvfi.flags = LVFI_PARAM;
	lvfi.lParam = iItemInternal;
	iItem = ListView_FindItem(m_hListView, -1, &lvfi);

	if (iItem != -1)
	{
		if (m_folderSettings.showInGroups)
		{
			auto groupId = GetItemGroupId(iItem);

			if (groupId)
			{
				OnItemRemovedFromGroup(*groupId);
			}
		}

		/* Remove the item from the listview. */
		ListView_DeleteItem(m_hListView, iItem);
	}

	m_directoryState.filteredItemsList.erase(iItemInternal);
	m_itemInfoMap.erase(iItemInternal);

	nItems = ListView_GetItemCount(m_hListView);

	m_directoryState.numItems--;
}

ShellNavigationController *ShellBrowserImpl::GetNavigationController() const
{
	return m_navigationController.get();
}

boost::signals2::connection ShellBrowserImpl::AddNavigationStartedObserver(
	const NavigationStartedSignal::slot_type &observer, boost::signals2::connect_position position)
{
	return m_navigationStartedSignal.connect(observer, position);
}

boost::signals2::connection ShellBrowserImpl::AddNavigationCommittedObserver(
	const NavigationCommittedSignal::slot_type &observer,
	boost::signals2::connect_position position)
{
	return m_navigationCommittedSignal.connect(observer, position);
}

boost::signals2::connection ShellBrowserImpl::AddNavigationCompletedObserver(
	const NavigationCompletedSignal::slot_type &observer,
	boost::signals2::connect_position position)
{
	return m_navigationCompletedSignal.connect(observer, position);
}

boost::signals2::connection ShellBrowserImpl::AddNavigationFailedObserver(
	const NavigationFailedSignal::slot_type &observer, boost::signals2::connect_position position)
{
	return m_navigationFailedSignal.connect(observer, position);
}
