# Building a custom pipeline

The **deep**doctection analyzer is an example of a Document Layout Analysis pipeline. In this tutorial we'll show you the concepts so that you can build a pipeline yourself and according the needs you have.


```python
from pathlib import Path
import deepdoctection as dd
```

The idea is not that difficult: There are **models** that fulfill a given task, there are **pipeline components** or **pipeline backbones** that invoke models and take care of pre- and post-processing results. There are also pipeline backbones that do not invoke models but only consolidate results. 

And there is the **pipeline** that puts everything together.

## Catalog and registries

You can get the essential information for pre-trained model from the `ModelCatalog`. 


```python
dd.print_model_infos(add_description=False,add_config=False,add_categories=False)
```

    [36m╒═════════════════════════════════════════════════════════════════════╕
    │ name                                                                │
    ╞═════════════════════════════════════════════════════════════════════╡
    │ layout/model-800000_inf_only.data-00000-of-00001                    │
    ├─────────────────────────────────────────────────────────────────────┤
    │ cell/model-1800000_inf_only.data-00000-of-00001                     │
    ├─────────────────────────────────────────────────────────────────────┤
    │ item/model-1620000_inf_only.data-00000-of-00001                     │
    ├─────────────────────────────────────────────────────────────────────┤
    │ item/model-1620000.data-00000-of-00001                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ layout/model-800000.data-00000-of-00001                             │
    ├─────────────────────────────────────────────────────────────────────┤
    │ cell/model-1800000.data-00000-of-00001                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ layout/d2_model_0829999_layout_inf_only.pt                          │
    ├─────────────────────────────────────────────────────────────────────┤
    │ layout/d2_model_0829999_layout.pth                                  │
    ├─────────────────────────────────────────────────────────────────────┤
    │ layout/d2_model_0829999_layout_inf_only.ts                          │
    ├─────────────────────────────────────────────────────────────────────┤
    │ cell/d2_model_1849999_cell_inf_only.pt                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ cell/d2_model_1849999_cell_inf_only.ts                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ cell/d2_model_1849999_cell.pth                                      │
    ├─────────────────────────────────────────────────────────────────────┤
    │ item/d2_model_1639999_item.pth                                      │
    ├─────────────────────────────────────────────────────────────────────┤
    │ item/d2_model_1639999_item_inf_only.pt                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ item/d2_model_1639999_item_inf_only.ts                              │
    ├─────────────────────────────────────────────────────────────────────┤
    │ nielsr/lilt-xlm-roberta-base/pytorch_model.bin                      │
    ├─────────────────────────────────────────────────────────────────────┤
    │ SCUT-DLVCLab/lilt-infoxlm-base/pytorch_model.bin                    │
    ├─────────────────────────────────────────────────────────────────────┤
    │ SCUT-DLVCLab/lilt-roberta-en-base/pytorch_model.bin                 │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/layoutlm-base-uncased/pytorch_model.bin                   │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/layoutlm-large-uncased/pytorch_model.bin                  │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/layoutlmv2-base-uncased/pytorch_model.bin                 │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/layoutxlm-base/pytorch_model.bin                          │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/layoutlmv3-base/pytorch_model.bin                         │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/table-transformer-detection/pytorch_model.bin             │
    ├─────────────────────────────────────────────────────────────────────┤
    │ microsoft/table-transformer-structure-recognition/pytorch_model.bin │
    ├─────────────────────────────────────────────────────────────────────┤
    │ doctr/db_resnet50/pt/db_resnet50-ac60cadc.pt                        │
    ├─────────────────────────────────────────────────────────────────────┤
    │ doctr/db_resnet50/tf/db_resnet50-adcafc63.zip                       │
    ├─────────────────────────────────────────────────────────────────────┤
    │ doctr/crnn_vgg16_bn/pt/crnn_vgg16_bn-9762b0b0.pt                    │
    ├─────────────────────────────────────────────────────────────────────┤
    │ doctr/crnn_vgg16_bn/tf/crnn_vgg16_bn-76b7f2c6.zip                   │
    ├─────────────────────────────────────────────────────────────────────┤
    │ FacebookAI/xlm-roberta-base/pytorch_model.bin                       │
    ├─────────────────────────────────────────────────────────────────────┤
    │ fasttext/lid.176.bin                                                │
    ╘═════════════════════════════════════════════════════════════════════╛[0m


Let's select fasttext language detector. We need the categories that the model predicts and the model wrapper. `fasttext/lid.176.bin` is just an artifact. 


```python
categories=dd.ModelCatalog.get_profile("fasttext/lid.176.bin").categories
categories_orig = dd.ModelCatalog.get_profile("fasttext/lid.176.bin").categories_orig
```


```python
dd.ModelCatalog.get_profile("fasttext/lid.176.bin").model_wrapper
```




    'FasttextLangDetector'



We can download `lid.176.bin` with help of the `ModelDownloadManager`.


```python
path_weights=dd.ModelDownloadManager.maybe_download_weights_and_configs("fasttext/lid.176.bin")
```

    [32m[0712 15:02.04 @fs.py:100][0m  [32mINF[0m  [97mFile lid.176.bin exists! Skip download.[0m


## Model wrapper

We know from the `ModelCatalog` which wrapper we must use for the fasttext model, namely `FasttextLangDetector`.


```python
fast_text = dd.FasttextLangDetector(path_weights, categories, categories_orig)
```

    Warning : `load_model` does not return WordVectorModel or SupervisedModel any more, but a `FastText` object which is very similar.


We are not done yet, because we still need to choose how to extract text. Let's simply stick to Tesseract and use the default english setting.


```python
tess_ocr_config_path = dd.get_configs_dir_path() / "dd/conf_tesseract.yaml"  # This file will be in your .cache if you ran the analyzer before. 
# Otherwise make sure to copy the file from 'configs/conf_tesseract.yaml'

tesseract_ocr = dd.TesseractOcrDetector(tess_ocr_config_path)
```

## Pipeline backbone

Similar to models we have a pipeline component registry. Having this starting point we can select the right backbone. Check the API documentation to see what the components are used for.


```python
dd.pipeline_component_registry.get_all()
```




    {'ImageCroppingService': deepdoctection.pipe.concurrency.MultiThreadPipelineComponent,
     'MatchingService': deepdoctection.pipe.common.MatchingService,
     'PageParsingService': deepdoctection.pipe.common.PageParsingService,
     'AnnotationNmsService': deepdoctection.pipe.common.AnnotationNmsService,
     'ImageParsingService': deepdoctection.pipe.common.ImageParsingService,
     'LanguageDetectionService': deepdoctection.pipe.language.LanguageDetectionService,
     'ImageLayoutService': deepdoctection.pipe.layout.ImageLayoutService,
     'LMTokenClassifierService': deepdoctection.pipe.lm.LMTokenClassifierService,
     'LMSequenceClassifierService': deepdoctection.pipe.lm.LMSequenceClassifierService,
     'TextOrderService': deepdoctection.pipe.order.TextOrderService,
     'TableSegmentationRefinementService': deepdoctection.pipe.refine.TableSegmentationRefinementService,
     'TableSegmentationService': deepdoctection.pipe.segment.TableSegmentationService,
     'SubImageLayoutService': deepdoctection.pipe.sub_layout.SubImageLayoutService,
     'TextExtractionService': deepdoctection.pipe.text.TextExtractionService,
     'SimpleTransformService': deepdoctection.pipe.transform.SimpleTransformService}



## Fasttext language detector


```python
lang_detect_comp = dd.LanguageDetectionService(fast_text,text_detector=tesseract_ocr)
```

We can now build our very simple pipeline.


```python
pipe_comp_list = [lang_detect_comp]
pipe = dd.DoctectionPipe(pipeline_component_list=pipe_comp_list)
```


```python
image_path = Path.cwd() / "pics/samples/sample_3" 
```

![title](./_imgs/sample_3.png)

When running the pipe, we get the language the document was written. 


```python
df = pipe.analyze(path=image_path)
df.reset_state()
dp = next(iter(df))
dp.language
```

    |                                                                                                                                                                                                                                                                                                                                  |1/?[00:00<00:00,1145.98it/s]
    [32m[0712 15:02.11 @doctectionpipe.py:84][0m  [32mINF[0m  [97mProcessing sample_3.png[0m
    [32m[0712 15:02.14 @context.py:133][0m  [32mINF[0m  [97mLanguageDetectionService total: 2.7886 sec.[0m





    <Languages.GERMAN>



When getting the text, the response is somewhat disappointing.


```python
dp.text
```




    ''



The reason for that is that `LanguageDetectionService` is not responsible for extracting text. It has an OCR model, but the output is only used as input feed to the language detector. The text however is not persisted. If we had added a `TextExtractionService` before `LanguageDetectionService` we could have omitted the OCR model in the `LanguageDetectionService`. 

## Tesseract OCR detector


```python
tesseract_ocr = dd.TesseractOcrDetector(tess_ocr_config_path.as_posix(),["LANGUAGES=deu"])
```


```python
tesseract_ocr.config
```




    {'LANGUAGES': 'deu', 'LINES': False, 'psm': 11}




```python
# setting run_time_ocr_language_selection=True will dynamically select the OCR model for text extraction based on 
# the predicted languages. This helps to get much improved OCR results, if you have documents with various languages.

text_comp = dd.TextExtractionService(tesseract_ocr, run_time_ocr_language_selection=True)
pipe_comp_list.append(text_comp)
```


```python
tesseract_ocr
```




    <deepdoctection.extern.tessocr.TesseractOcrDetector at 0x75b7cce7abb0>




```python
df = pipe.analyze(path=image_path)
df.reset_state()
dp = next(iter(df))
dp.text
```

    |                                                                                                                                                                                                                                                                                                                                   |1/?[00:00<00:00,901.03it/s]
    [32m[0712 15:02.16 @doctectionpipe.py:84][0m  [32mINF[0m  [97mProcessing sample_3.png[0m
    [32m[0712 15:02.19 @context.py:133][0m  [32mINF[0m  [97mLanguageDetectionService total: 2.7362 sec.[0m
    [32m[0712 15:02.21 @context.py:133][0m  [32mINF[0m  [97mTextExtractionService total: 2.7723 sec.[0m





    ''



This is something unexpected. Why don't we generate any text? We can clearly see that the `TextExtractionService` did its job.


```python
word_sample = dp.words[0]
len(dp.words), word_sample.characters, word_sample.bbox, word_sample.reading_order 
```




    (553, 'Anleihemärkte', [137.0, 158.0, 472.0, 195.0], None)



## Text ordering

The reason is, that we do not have inferred a reading order. If there is no reading order, there is no contiguous text. We treat text extraction as a character recognition problem only. If we want a reading order of predicted words, we need to do it ourself. So let's add the `TextOrderService`.


```python
order_comp = dd.TextOrderService(text_container=dd.LayoutType.WORD)
pipe_comp_list.append(order_comp)
```

At least, we got some text. The beginning sounds good. But once the text comes to the region where the second and third column also have text lines, the order service does not distinguish between columns. So we must identify columns. For that we use the layout analyzer.


```python
df = pipe.analyze(path=image_path)
df.reset_state()
dp = next(iter(df))
dp.text
```

    |                                                                                                                                                                                                                                                                                                                                  |1/?[00:00<00:00,1051.47it/s]
    [32m[0712 15:02.22 @doctectionpipe.py:84][0m  [32mINF[0m  [97mProcessing sample_3.png[0m
    [32m[0712 15:02.25 @context.py:133][0m  [32mINF[0m  [97mLanguageDetectionService total: 2.656 sec.[0m
    [32m[0712 15:02.27 @context.py:133][0m  [32mINF[0m  [97mTextExtractionService total: 2.9117 sec.[0m
    [32m[0712 15:02.28 @context.py:133][0m  [32mINF[0m  [97mTextOrderService total: 0.1003 sec.[0m





    'Anleihemärkte im Geschäftsjahr\nbis zum 31.12.2018\nSchwieriges Marktumfeld\nDie internationalen Anleihe-\nmärkte entwickelten sich im\nGeschäftsjahr 2018 unter-\nschiedlich und phasenweise\nsehr volatil. Dabei machte sich\nbei den Investoren zunehmend\nNervosität breit, was in steigen-\nden Risikoprämien zum Aus-\ndruck kam. Grund hierfür waren\nTurbulenzen auf der weltpoli-\ntischen Bühne, die die politi-\nschen Risiken erhöhten. Dazu\nzählten unter anderem populis-\ntische Strömungen nicht nur\nin den USA und Europa, auch\nin den Emerging Markets, wie\nzuletzt in Brasilien und Mexiko,\nwo Populisten in die Regie-\nrungen gewählt wurden. Der\neskalierende Handelskonflikt\nzwischen den USA einerseits\nsowie Europa und China ande-\nrerseits tat sein übriges. Zudem\nging Italien im Rahmen seiner\nHaushaltspolitik auf Konfronta-\ntionskurs zur Europäischen Uni-\non (EU). Darüber hinaus verun-\nsicherte weiterhin der drohende\nBrexit die Marktteilnehmer,\ninsbesondere dahingehend, ob\nder mögliche Austritt des Ver-\neinigten Königreiches aus der\nEU geordnet oder - ohne ein\nÜbereinkommen - ungeordnet\nvollzogen wird. Im Gegensatz\nzu den politischen Unsicher-\nheiten standen die bislang eher\nzuversichtlichen, konventionel-\nlen Wirtschaftsindikatoren. So\nexpandierte die Weltwirtschaft\nkräftig, wenngleich sich deren\nWachstum im Laufe der zwei-\nten Jahreshälfte 2018 etwas\nverlangsamte. Die Geldpolitik\nwar historisch gesehen immer\nnoch sehr locker, trotz der welt-\nweit sehr hohen Verschuldung\nund der Zinserhöhungen der\nUS-Notenbank.\nEntwicklung der Leitzinsen in den USA und im Euroraum\n%p.a.\nu\nu\nu\nu\nu\nu\nu\nu\nu\nu\n12/08 12/09 12/10 12/11 12/12 12/13 12/14 12/15 12/16 12/17 12/18\nBE Fed-Leitzins\nQuelle: Thomson Financial Datastream\nBE E28-Leitzins\nStand: 31.12.2018\n-1 u\nZinswende nach Rekordtiefs\nbei Anleiherenditen?\nIm Berichtszeitraum kam es an\nden Anleihemärkten - wenn\nauch uneinheitlich und unter-\nschiedlich stark ausgeprägt -\nunter Schwankungen zu stei-\ngenden Renditen auf teilweise\nimmer noch sehr niedrigem\nNiveau, begleitet von nachge-\nbenden Kursen. Dabei konnten\nsich die Zinsen vor allem in den\nUSA weiter von ihren histori-\nschen Tiefs lösen. Gleichzeitig\nwurde die Zentralbankdivergenz\nzwischen den USA und dem\nEuroraum immer deutlicher. An-\ngesichts des Wirtschaftsbooms\nin den USA hob die US-Noten-\nbank Fed im Berichtszeitraum\nden Leitzins in vier Schritten\nweiter um einen Prozentpunkt\nauf einen Korridor von 2,25% -\n2,50% p.a. an. Die Europäische\nZentralbank (EZB) hingegen\nhielt an ihrer Nullzinspolitik fest\nund die Bank of Japan beließ\nihren Leitzins bei -0,10% p.a.\nDie Fed begründete ihre Zinser-\nhöhungen mit der Wachstums-\nbeschleunigung und der Voll-\nbeschäftigung am Arbeitsmarkt\nin den USA. Zinserhöhungen\nermöglichten der US-Notenbank\neiner Überhitzung der US-Wirt-\nschaft vorzubeugen, die durch\ndie prozyklische expansive\nFiskalpolitik des US-Präsidenten\nDonald Trump in Form von\nSteuererleichterungen und einer\nErhöhung der Staatsausgaben\nnoch befeuert wurde. Vor die-\nsem Hintergrund verzeichneten\ndie US-Bondmärkte einen spür-\nbaren Renditeanstieg, der mit\nmerklichen Kursermäßigungen\neinherging. Per saldo stiegen\ndie Renditen zehnjähriger US-\nStaatsanleihen auf Jahressicht\nvon 2,4% p.a. auf 3,1% p.a.\nDiese Entwicklung in den USA\nhatte auf den Euroraum jedoch\nnur phasenweise und partiell,\ninsgesamt aber kaum einen\nzinstreibenden Effekt auf Staats-\nanleihen aus den europäischen\nKernmärkten wie beispielsweise\nDeutschland und Frankreich.\nSo gaben zehnjährige deutsche\nBundesanleihen im Jahresver-\nlauf 2018 unter Schwankungen\nper saldo sogar von 0,42% p.a.\nauf 0,25% p. a. nach. Vielmehr\nstanden die Anleihemärkte\nder Euroländer - insbeson-\ndere ab dem zweiten Quartal\n2018 - unter dem Einfluss der\npolitischen und wirtschaftlichen\nEntwicklung in der Eurozone,\nvor allem in den Ländern mit\nhoher Verschuldung und nied-\nrigem Wirtschaftswachstum.\nIn den Monaten Mai und Juni'



## Layout service

It now depends on whether we use Tensorflow or PyTorch. We opt for PyTorch, just because the model runs on a CPU.
Make sure, that the model has been loaded to your .cache.


```python
path_weights = dd.ModelCatalog.get_full_path_weights("layout/d2_model_0829999_layout_inf_only.pt")
path_configs = dd.ModelCatalog.get_full_path_configs("layout/d2_model_0829999_layout_inf_only.pt")
categories = dd.ModelCatalog.get_profile("layout/d2_model_0829999_layout_inf_only.pt").categories

categories
```




    {1: <LayoutType.TEXT>,
     2: <LayoutType.TITLE>,
     3: <LayoutType.LIST>,
     4: <LayoutType.TABLE>,
     5: <LayoutType.FIGURE>}




```python
layout_detector = dd.D2FrcnnDetector(path_configs,path_weights,categories,device="cpu")
layout_comp = dd.ImageLayoutService(layout_detector)
```

We need to make sure, that the `ImageLayoutService` has been invoked before `TextOrderService`.  


```python
pipe_comp_list.insert(0,layout_comp)
```


```python
df = pipe.analyze(path=image_path)
df.reset_state()
dp = next(iter(df))
dp.layouts
```

    |                                                                                                                                                                                                                                                                                                                                  |1/?[00:00<00:00,1078.50it/s]
    [32m[0712 15:02.33 @doctectionpipe.py:84][0m  [32mINF[0m  [97mProcessing sample_3.png[0m
    /home/janis/Documents/Repos/deepdoctection_pt/.venv/lib/python3.9/site-packages/torch/functional.py:504: UserWarning: torch.meshgrid: in an upcoming release, it will be required to pass the indexing argument. (Triggered internally at ../aten/src/ATen/native/TensorShape.cpp:3526.)
      return _VF.meshgrid(tensors, **kwargs)  # type: ignore[attr-defined]
    [32m[0712 15:02.35 @context.py:133][0m  [32mINF[0m  [97mImageLayoutService total: 1.7507 sec.[0m
    [32m[0712 15:02.37 @context.py:133][0m  [32mINF[0m  [97mLanguageDetectionService total: 2.623 sec.[0m
    [32m[0712 15:02.40 @context.py:133][0m  [32mINF[0m  [97mTextExtractionService total: 2.7651 sec.[0m
    [32m[0712 15:02.40 @context.py:133][0m  [32mINF[0m  [97mTextOrderService total: 0.1094 sec.[0m





    [Layout(active=True, _annotation_id='46fa3a5c-940d-3e1d-8222-5aefb83249cd', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TEXT>, _category_name=<LayoutType.TEXT>, category_id=1, score=0.9831379652023315, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='404a8222-3eb5-3bf3-9440-aaadec80998a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=5, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.1199951171875, uly=393.2299499511719, lrx=552.439453125, lry=2188.857666015625, height=1795.6277160644531, width=414.3194580078125)),
     Layout(active=True, _annotation_id='473634c1-12b0-3372-8e94-fecda44dee15', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TEXT>, _category_name=<LayoutType.TEXT>, category_id=1, score=0.9732131958007812, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2e8e2f35-16ee-3088-9503-ec83841f410d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=104, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=618.3727416992188, uly=872.4019165039062, lrx=1036.106689453125, lry=2191.201171875, height=1318.7992553710938, width=417.73394775390625)),
     Layout(active=True, _annotation_id='79e31718-5dce-3522-8ca9-14453e3f6ce7', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TITLE>, _category_name=<LayoutType.TITLE>, category_id=2, score=0.9648802280426025, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='088f0dfa-345e-33a3-87b1-477d2df1b206', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=1, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=135.84957885742188, uly=141.39149475097656, lrx=885.5208740234375, lry=274.1432189941406, height=132.75172424316406, width=749.6712951660156)),
     Layout(active=True, _annotation_id='3b7c8395-0e2d-39c8-abd0-84325a8e7f30', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TEXT>, _category_name=<LayoutType.TEXT>, category_id=1, score=0.9539077281951904, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='84dc05b9-213f-318d-ad1a-6ff1a0ad5644', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=66, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=1102.441650390625, uly=1358.8077392578125, lrx=1520.0098876953125, lry=2187.2880859375, height=828.4803466796875, width=417.5682373046875)),
     Layout(active=True, _annotation_id='d7e62e37-9aff-3e73-8725-e0a938069cbe', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TEXT>, _category_name=<LayoutType.TEXT>, category_id=1, score=0.9395560026168823, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='15280181-e186-3dd5-880e-0c0a32004791', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=60, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=1099.9849853515625, uly=851.8798217773438, lrx=1520.9630126953125, lry=1322.9498291015625, height=471.07000732421875, width=420.97802734375)),
     Layout(active=True, _annotation_id='c554fd1f-3cd4-3100-8c3f-a85f01258e1f', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.FIGURE>, _category_name=<LayoutType.FIGURE>, category_id=5, score=0.7482666373252869, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='90ae71c7-5dc6-3ce3-8f23-cef195779d85', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=55, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=628.3258666992188, uly=468.3048400878906, lrx=1522.9581298828125, lry=732.95068359375, height=264.6458435058594, width=894.6322631835938)),
     Layout(active=True, _annotation_id='0bfeb7ab-0717-358f-b911-16e74d852b42', service_id='c5a80ae0', model_id='3ecda8b7', session_id=None, category_name=<LayoutType.TEXT>, _category_name=<LayoutType.TEXT>, category_id=1, score=0.6195411682128906, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='05cd8d0c-5055-3455-a873-4d4bffc2f272', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=88, score=None, sub_categories={}, relationships={})}, relationships={}, bounding_box=BoundingBox(absolute_coords=True, ulx=615.2214965820312, uly=383.41668701171875, lrx=1244.1329345703125, lry=444.78387451171875, height=61.3671875, width=628.9114379882812)),
     Layout(active=True, _annotation_id='75b23a60-e8fe-3336-9927-b4d7088081da', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4f7e3218-e4c8-39e6-adf3-48afc3d73dea', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=54, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['eefd6d57-52c2-3850-8e70-011de105a3a4']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=2159.0, lrx=331.0, lry=2179.0, height=20.0, width=191.0)),
     Layout(active=True, _annotation_id='1750111e-60cc-3cb9-9c7a-92fb49e3fdbe', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='21057a4e-eed3-3ce5-a302-e9a99a313a67', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=150, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b972d4c2-dd1c-3f29-93b5-b30c2f853bb2', 'd9e16ac4-ba50-3db3-8cf7-e2acd41c2364', '055b0d62-cc7f-36b4-b363-ea859bcc8ce0']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=2159.0, lrx=958.0, lry=2184.0, height=25.0, width=337.0)),
     Layout(active=True, _annotation_id='38f99bc8-fba2-37ec-9594-3937ce3c7f58', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b4941569-f20a-309c-ae5c-390b5741dfef', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=87, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['73526780-87de-317e-9f49-06b4a3911acf', 'b3a1316e-c94d-3e94-b374-c77969ac6519', 'a277e654-2d71-31ed-ac2c-218199c17404', 'b2fde76b-8d70-316e-b788-793b7de431df', 'bea7c87b-67e1-3a8f-b25b-56a2b6efed88', '3b341ad2-a7e1-3520-9359-2b44811c7094']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=2159.0, lrx=1476.0, lry=2179.0, height=20.0, width=371.0)),
     Layout(active=True, _annotation_id='4bf94e02-8a3a-3114-b275-4376d882e280', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b35f8306-674a-3653-8303-abd9679519e9', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=53, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['408a1d78-7eaa-37f3-b5cd-64875967f2f4', '03516f46-1efe-3cdc-9e6f-0f7a8521fbbf', 'bc030ee0-3fe9-3dec-b9f4-f0aee5e6f606', '97d02e25-469e-3390-a2cf-9f66f1da4e52']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=2123.0, lrx=504.0, lry=2149.0, height=26.0, width=364.0)),
     Layout(active=True, _annotation_id='ed48f52d-ff41-3e4f-acb6-d5dc5732850c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='799a8047-9137-311f-a19b-f2c13d54f089', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=148, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4f16d8a0-316d-3ae3-b519-aa6f966491e5', 'bb35c879-0a95-3f1c-906f-7bdb6f792033', 'c335bb13-cd35-3e79-8955-5f9a13f97d40', '349374df-6de1-3723-9886-ef37b14e64b6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=2123.0, lrx=1005.0, lry=2149.0, height=26.0, width=384.0)),
     Layout(active=True, _annotation_id='54417aa1-6d16-3027-a312-b078fa4845a3', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='402f4f7b-ae21-3a86-b110-78e53807d97f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=149, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['9eaf20f2-108f-311d-95dd-7f7912c1aee0', '6009383f-b565-3c71-98be-f74647e520ca']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=2123.0, lrx=1470.0, lry=2149.0, height=26.0, width=366.0)),
     Layout(active=True, _annotation_id='4753bda3-129a-3030-8f12-b739aaa983a4', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='75df6087-0004-3cbb-8c92-130c5ee625f2', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=52, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['e7b50b1a-1e84-33ef-a702-905f20a172b6', 'd8c43fe6-bdc4-37ae-834d-24a686e23964', '33eb18cc-7969-35ce-9ea3-56ed98fa5c12', '620b50c8-a229-3b10-b987-913d0606cd96']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=2087.0, lrx=527.0, lry=2113.0, height=26.0, width=389.0)),
     Layout(active=True, _annotation_id='1f882958-acdd-3abb-896b-2c6ed76c20d8', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='daf56742-89d6-3a5e-9381-f6310e3a7489', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=146, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['cb9cb43d-7ba3-39a6-ad0d-b5f2ca9d5456', 'bbe48442-15d1-3086-9474-89c815c08854', '8fd331de-59ea-3d07-9ae4-e3ba162534c5', '52f4b35d-d964-30ed-aff6-a34cb5699ab4']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=2083.0, lrx=1011.0, lry=2113.0, height=30.0, width=390.0)),
     Layout(active=True, _annotation_id='ff2adfa9-bea2-33db-88ce-55a9061093e3', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1f4548bb-74e5-37dd-b528-2e709c8a1218', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=147, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b661f787-7f4f-3743-87d0-49f186842ee6', '6567f376-8c8f-35dd-af38-6063ae64932c', '5062611f-98b5-31c0-ad4a-454988e4bb14', 'e5ec434c-a594-3019-aeb7-0f248fad805b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=2087.0, lrx=1487.0, lry=2113.0, height=26.0, width=383.0)),
     Layout(active=True, _annotation_id='6080d2e9-12e9-3ca2-803d-68e4fb4384f7', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='484fea92-261e-37f7-bed9-1aeba0e4cc30', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=51, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['56e241ec-05ea-3027-b7f6-e5ff0c6221fa', '3b4f6b7e-1431-3a3c-9bf2-046409f1ed24', 'cf5bcb2b-a551-3c33-9b52-aa4497d2bc30', '6e5f174e-c29d-380c-bf2b-996c6527f298', 'c6795dba-e127-30f0-8840-1b85ca1be89d', '1e62668e-116c-3af8-97a9-efcfc77a9137']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=2051.0, lrx=545.0, lry=2074.0, height=23.0, width=405.0)),
     Layout(active=True, _annotation_id='8e261b04-8d2f-3b44-a5e7-e697c7ea262f', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='e713b4dd-b214-31a7-803c-f3312e8a059b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=144, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['2c3b0976-dbdb-31b9-98cc-fb326f3b4c50', '86ce7284-4558-3f53-bfbb-37702f0eec5c', 'a64597e1-c547-3093-8a26-6d8692be03af']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=2051.0, lrx=1032.0, lry=2077.0, height=26.0, width=411.0)),
     Layout(active=True, _annotation_id='03b4674f-8325-3c71-9dda-9152f62ad08d', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='759b9b0c-be1d-33fb-bfe5-0e4b1ffb4e0b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=145, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['28f6399f-0d08-3055-abfd-47d559db3b5a', '9fa812aa-f9f1-3320-bf1c-b04829b84f4b', 'e6fa3b5b-074b-3d54-a7dc-aa93a99af88b', '9f12508c-78e6-35dc-ac59-ae82a01d9f3a', '2fff6a06-f5ab-32cd-8dda-f1dc779ab635', '24f2ca42-fbc6-3c04-9a0b-fd5e3368b70d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=2051.0, lrx=1468.0, lry=2071.0, height=20.0, width=365.0)),
     Layout(active=True, _annotation_id='4aec3feb-07c5-33db-a3bb-c3420e8d7e5e', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='acc2b401-5cd1-38ec-800b-58974b778a7e', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=50, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['263b0a16-48c8-3d64-8fb7-df9edc491826', 'ff7c9672-829c-3c28-8f0b-cb8e5387388c', '107664a6-512e-304f-847c-3ff22dbf8b73', '09a99ee7-940e-3f20-8b0e-5a1b38d0d72c']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=2015.0, lrx=528.0, lry=2041.0, height=26.0, width=390.0)),
     Layout(active=True, _annotation_id='bb3d465a-5ed9-3149-9acf-101e8bf31ede', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3d634ecf-c02e-39e3-b3cb-b654ecfb17ab', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=142, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['dc06d845-08e1-3d83-ae14-b5dde1ed9df4', '6db5cfd1-3988-38f2-a7c6-ccfcadb8c105', '828b69e3-a596-3172-a072-a5abd6bc45db', '171b5495-160b-315c-9e88-62ad677fa8e9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=2015.0, lrx=983.0, lry=2041.0, height=26.0, width=361.0)),
     Layout(active=True, _annotation_id='56ba9989-e66f-328c-a88b-a8887414889a', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='320e1576-e8fc-30f8-ad21-a063aca541a1', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=143, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['29cc0b93-76ab-3833-a232-39671a05433e', 'af734fd2-e2c4-364c-85a7-299d47b635a0', '73147fb7-945c-3118-b84f-e7cdac33e9ce', 'c6548dfa-2692-377a-a6fe-fb799c68b0f3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=2015.0, lrx=1473.0, lry=2041.0, height=26.0, width=368.0)),
     Layout(active=True, _annotation_id='f5fc5cf1-1cc1-336a-bb25-c12a120eef4c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='9452aba9-5fdb-3921-92f5-52215fd8106a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=49, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['60c9d562-6d1f-369a-bca3-c70862464535', '24321259-3d76-3eeb-94c3-96663b53ebae', 'f69d6896-e14b-3076-aeab-fe0935eeab73']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1979.0, lrx=515.0, lry=2005.0, height=26.0, width=377.0)),
     Layout(active=True, _annotation_id='c1b7b8a3-a884-3902-8db3-8914e28902d1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='6d4db47e-2e5a-3072-93ca-b098fe394d3e', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=140, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['52d2f03b-b0df-36d6-a57b-96a4eee0af89', '3f388c9c-fa58-318a-821f-3c8f2b12de00', '85304cf5-e1b2-3883-be25-8a8823b08dd5']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1979.0, lrx=1024.0, lry=2005.0, height=26.0, width=402.0)),
     Layout(active=True, _annotation_id='ba67dab8-afc8-31cf-b7dd-953326956ea3', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='172323f8-5159-3ad8-b413-a3656439325a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=141, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['78b034cc-556f-37d8-b149-7fdcd9f9d3a9', '8a3da63f-762a-3ace-be0f-44db8949b63a', 'd1f6c275-5a66-3c9f-8a0c-202f90eda160']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1979.0, lrx=1509.0, lry=2004.0, height=25.0, width=405.0)),
     Layout(active=True, _annotation_id='a4aef346-0d30-3304-9006-a8ff130f8182', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='141f85b0-3c8a-3369-99df-9d4164586b74', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=48, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['3d9be59d-460b-3b75-8c19-065f8a6e5ee8', '48600898-4478-300b-8f91-4afcae064fc9', 'c078beba-9c1f-3d70-bdfd-f046fff02dcc', '36981c58-bb39-3309-8504-075cf209bf75']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1943.0, lrx=499.0, lry=1963.0, height=20.0, width=361.0)),
     Layout(active=True, _annotation_id='36523721-c10f-3c87-9e6f-a7b56f661fe9', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='9458cede-e7b0-3cee-9b5d-2db0a9dac8e2', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=138, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['96f887e5-84b2-376f-b043-470e41c428ce', 'e41cced4-a29f-3af8-a731-7f4b805a86ba', 'e56f546b-4566-3ae5-9803-bbebf7f1e980', 'e7926288-dfd5-36db-953e-9c2248f4bf8b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1943.0, lrx=993.0, lry=1969.0, height=26.0, width=371.0)),
     Layout(active=True, _annotation_id='068a1dcd-3fdf-3fac-9b6a-092d064ce49b', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1621afc1-c7d1-3066-a78b-63b73a8d06af', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=139, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['69be18a6-0e47-3474-8031-6d657e9841fb', '48834ee2-7ba2-3b7f-9e2c-bf91b92d214c', '3de3a460-13c8-32c8-9d64-90d07e7cd1db', '9130e32a-20b4-3f9a-8bd3-00aadccda2ba', 'f57547f0-0e96-3d46-861b-6a9a1bf0092c', 'd858daf6-6a47-37d0-8fe0-69a80e6170f2']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1943.0, lrx=1482.0, lry=1963.0, height=20.0, width=378.0)),
     Layout(active=True, _annotation_id='74f3d8ab-f087-3856-82ef-6535019a69c8', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3a78121d-c617-33cc-9180-20f7e2aa9055', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=47, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['0364cb8a-de45-327d-9864-ae7ceb07893a', '2cf20fb0-5f77-33d7-896d-d0077ddfb911', 'cb2f6c87-ae7c-352b-97ab-d23fbce8df7e', '58f046aa-aab4-3cd7-9328-f2dcaef02905', 'dbf7eaa8-7190-318b-8b3a-82ebf7766e97']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1906.0, lrx=521.0, lry=1926.0, height=20.0, width=383.0)),
     Layout(active=True, _annotation_id='1e1bbd02-e6e1-3736-a127-ca7e5a4792e7', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='19285cae-613c-319a-9ad0-817d63648496', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=136, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c08fda93-63d7-332b-8840-6134d4a5e19d', '066c9c47-6e82-375e-be02-4c7fb7f08a75', '39a22a82-a40e-39f6-a525-cb5ededde158', '94b9ff2f-c1be-3d51-8e1d-04a21f119893']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1906.0, lrx=1016.0, lry=1932.0, height=26.0, width=394.0)),
     Layout(active=True, _annotation_id='91e8246a-cccc-3a77-8073-2623cc960df5', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='ce4a5a05-de94-370a-bf84-31ce38dbe6a4', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=137, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4b46dc4e-ab38-3cf4-be60-9173f1139f37', '5e4493d3-4bb5-3274-9f28-bb67d05f8935', '31cb99ab-183c-3277-a89f-5305194bdf3f', '73d77e0c-95f9-3229-b00a-a81e33b92eee', '7f6fa480-e120-3c05-9d0d-88278a7abaf0']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1906.0, lrx=1468.0, lry=1926.0, height=20.0, width=365.0)),
     Layout(active=True, _annotation_id='e12d7299-e564-302c-bfaa-bfad82a13ebb', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='7b923bce-f512-348b-ac3b-595c104b0939', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=46, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['5940484a-2087-38ee-8f3f-84026fc76608', '3a7e74b1-67aa-3794-9052-d584eb3b13ef', '0706c096-74fa-3ef6-ae9a-f1e80387d008', '9c39af48-0107-3605-9d3f-66e045b26c16']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1870.0, lrx=525.0, lry=1896.0, height=26.0, width=385.0)),
     Layout(active=True, _annotation_id='274e581d-91fe-3019-93f1-219575d31fdb', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='a1fe7649-a958-35d0-b525-9aabe4d8f2b0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=134, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['1797913c-aa71-3aaa-88ec-6c89bce38da8', '74be4dca-6153-3d79-af5d-91a8fecb2507', 'd104bc78-7e5d-3af0-9c56-10dcb7246e98', 'dbc9363b-b39f-3d2f-887b-1f9bbd796fa2', 'c4c7d6b0-e1ab-3e93-a5be-158922c45cb3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=623.0, uly=1870.0, lrx=1020.0, lry=1896.0, height=26.0, width=397.0)),
     Layout(active=True, _annotation_id='ea1fd2b4-6e1b-3cdc-8c09-4ab9c9d5294d', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='46ed9fbd-7479-3533-b982-e6f15afdce7d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=135, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['97c99924-2b36-383e-a1de-1b97b28d2c17', '08367b2f-8d19-3780-9c08-9cdd34018039', 'b64c6a24-07e8-3ea2-b311-acbfce3b1dc2', '9880de06-e96a-3719-a51c-55df357010ea']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1870.0, lrx=1443.0, lry=1890.0, height=20.0, width=340.0)),
     Layout(active=True, _annotation_id='30995073-3ed5-3504-8f20-9413b0aedef6', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='958fc96f-4973-3742-9bcd-2641765b975f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=45, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['0beb9a99-761c-3c4d-8fe4-77a2bbeab7e6', '8d7a8781-c0cc-3bd3-84b5-d4ac4205d247', 'd81c38e4-46fd-323c-9821-7fd23096f92f']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1834.0, lrx=532.0, lry=1859.0, height=25.0, width=393.0)),
     Layout(active=True, _annotation_id='281665cd-259d-35ae-85c8-a1d08cb3d097', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='d8d5ef28-5cf8-3056-8820-18454314ca5a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=132, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c650f3ee-6db8-356b-8169-72b54371d4b2', 'e46e2650-acdd-3788-a2b9-b12a76448f67', '487fd672-cb8f-355d-af48-d23cca104af6', '5e76c39a-703a-3b4b-b1dd-efcc2193d8df', '70386fbe-f5bc-305b-9c75-3928f92bb13d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1834.0, lrx=987.0, lry=1859.0, height=25.0, width=365.0)),
     Layout(active=True, _annotation_id='748fb3ef-812e-367f-a1c7-56916a211f53', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3a1b4c1a-453b-3ba8-a6e6-47aa2fa02e86', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=133, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8e52bdbb-0fde-338d-a379-96765e68f9f5', '42a3daf6-fa41-37a3-97c0-1b10aaef50b4', '39c1a91d-ccf9-398b-8f01-2889ab2aef1e']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1834.0, lrx=1443.0, lry=1854.0, height=20.0, width=339.0)),
     Layout(active=True, _annotation_id='24258340-8afa-318d-99c7-c45160e7aaa6', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3ba22195-f282-3689-93f6-65f171a92b69', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=44, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4e23d43d-6e19-36b6-a8c7-5ed82f0212b1', '64661d0f-64ed-347d-a3ff-f6787e4a4428', 'd332cf8f-c0ca-32af-8bdf-41974b945ac1']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1798.0, lrx=526.0, lry=1818.0, height=20.0, width=386.0)),
     Layout(active=True, _annotation_id='0df10586-1639-3aca-a291-836bc45b5447', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='36f4e179-8898-3106-bc09-e9ad7c3cdab1', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=130, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['47f622b8-f5b4-3f08-aee5-ed9eee81d710', '3a13f3bd-bcc3-3558-a198-4ad95412f247', '1d183927-b908-3f25-a28c-d8c33971126f', 'ca2a53bc-fb12-338c-bb36-bfda2a1ce930', 'efadf2f0-e8b8-303f-be5c-ed11feb0f16b', '86266c99-17e6-3bdb-a7dd-e1d06d8e6876']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1798.0, lrx=991.0, lry=1823.0, height=25.0, width=369.0)),
     Layout(active=True, _annotation_id='2c3529f8-5a51-332d-98dd-2270252ce0f9', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='f665fdf6-7703-3111-832d-cfb1fa9758aa', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=131, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['86275b9f-9fc8-3998-b0cf-8eaf93d4c5d8', '9e8bdbfc-7a14-3425-9826-46f2fbf99959', '507c282f-c326-3443-835b-3d38d40221a3', '983d89d5-964a-3c80-88d1-8681551e8017', '3dcd63db-3fd7-3cb8-b548-55d1d1c331b0', '12d9e276-3ebc-3b5b-943f-318d05e56093']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1798.0, lrx=1492.0, lry=1823.0, height=25.0, width=389.0)),
     Layout(active=True, _annotation_id='f8776b67-5894-3627-98e7-a7cc4c8a1861', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='19411152-6ae4-3e3e-806e-2113d85e5d12', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=43, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['bc687244-cac4-3dc2-8ba0-723db7a20920', 'f0658c9a-0ce6-3374-a113-bde10e2157b9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1762.0, lrx=534.0, lry=1785.0, height=23.0, width=395.0)),
     Layout(active=True, _annotation_id='093e178c-9739-3d31-a216-818a7e0182a2', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='836aed92-3f9b-3b7e-9762-b5d08cd23ff8', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=128, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['90fdefee-ad65-38a3-af8a-905ec283277f', '359e66f7-bd62-3c9e-8873-0a9cb6c6c5c8', '1dc6f130-b490-3e83-b0f1-6dcb62b4d580', 'f5bfe210-a9fe-37f3-b389-b7b92101f31b', '5ba30c59-3501-3dfd-8db7-4382310d6c92']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1762.0, lrx=1021.0, lry=1787.0, height=25.0, width=399.0)),
     Layout(active=True, _annotation_id='cead95bd-da20-39f6-815a-40764a85ca1a', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5b251634-fd71-3b3f-8140-e7910221bc8c', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=129, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8a9771b5-51bd-37c8-a63e-ce5c93ffabf2', '237c690a-b5ab-3c75-a2e8-8007ea867d1c', '57590122-6ff6-3745-bd45-790f5f994fd7', '1ec1fdb9-1630-3bf7-b07d-b9ba3a3a79b0', 'cc53db2d-2dfe-37bd-8589-35892b7b0dea', 'ed7cbc38-5b4b-3de3-9411-9a08ea9b0433']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1762.0, lrx=1497.0, lry=1788.0, height=26.0, width=393.0)),
     Layout(active=True, _annotation_id='5d377714-8000-3a00-8af2-807ab68b173f', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1e5aa014-4a17-3120-980b-2852c81e3646', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=42, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['42e9dbb5-18e1-3d58-bd63-05bb74199013', '36f20b4f-788d-3bcb-827e-7d438815b225', '380c2e3c-a8a9-327b-a925-444dca80cc47', 'da6f1547-abe3-3f2c-8173-f27e1c43703e', '8aa68d96-319c-3bf6-b3e0-b33e16db15ea']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1726.0, lrx=542.0, lry=1752.0, height=26.0, width=402.0)),
     Layout(active=True, _annotation_id='53767d6a-a7ac-3e44-baab-cfd9f2c56447', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='9ae6049c-7082-3ca5-a9f2-25cf8acc1135', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=85, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['387e4b8e-11d9-3703-ace9-2c1b922f2018', 'b9f5dd5a-46d3-331b-8789-c4bf1fd91e8a', '4ba98b86-2d9b-320e-9d1b-48e3cfb3aebc']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1726.0, lrx=970.0, lry=1752.0, height=26.0, width=349.0)),
     Layout(active=True, _annotation_id='f128b384-1038-3195-a3c8-02433be3ffc7', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='735bd34a-b512-3a42-be40-5903953b9d94', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=86, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['710a7eb4-c5f4-3ba8-b0b9-146d7f282ab4', 'b261cbcb-7714-388c-a77e-6c89354449cf', 'f1003422-ad01-3faf-adc0-2e2079025a02', '10a6662d-adbd-3b92-ac0d-6eb2d2360d2a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1726.0, lrx=1498.0, lry=1752.0, height=26.0, width=394.0)),
     Layout(active=True, _annotation_id='01fa963f-fe28-3e63-b305-19554d7171d6', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='0098875c-bed4-35a1-80f6-9a2148a85ab7', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=41, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['95e443fd-2634-3cf9-9368-a54601f6fcc8', '6d507927-6026-3b09-9d2d-5009da4eba80', '4edbe5e7-3250-3a6e-9d45-6b9695c4cc53', 'b3e347b1-72ad-3f17-8699-96e7b6f47931']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1690.0, lrx=501.0, lry=1715.0, height=25.0, width=362.0)),
     Layout(active=True, _annotation_id='ebb908af-768b-3c44-946c-a185c281d42b', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1fef462b-74c9-3cf5-93d1-75258022d8cd', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=83, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a9e2db38-c502-3db8-8da3-0c433685e433', 'f186a78d-b93e-3030-a028-511c37666055', '182f957b-143a-3547-9bc1-5492f493ed79', '6d4293af-0902-3ba3-8154-2644cbf34e8a', 'b03f5ea5-bff9-3e53-aae0-588efa6ee248']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1690.0, lrx=1018.0, lry=1715.0, height=25.0, width=397.0)),
     Layout(active=True, _annotation_id='9703b872-f23e-3354-84c2-baa6135789ac', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='48b3cfea-06cf-3970-8ffa-0b8f98f118f0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=84, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['18ed3292-75c4-35fe-bfc5-f3840c44a300', '02bffeeb-d512-3f52-b37a-31484d362427', '2432d975-660f-3842-a6e1-910edd356875']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=1690.0, lrx=1484.0, lry=1710.0, height=20.0, width=379.0)),
     Layout(active=True, _annotation_id='5039e2a9-cd09-36bd-8076-0934a96fc7aa', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='dc3b73d6-c464-3584-8490-a38bf2b2d464', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=40, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['5453d664-d276-353f-903e-1885a5ad4fc8', 'a201696e-f0f9-3d81-9439-a556661cd147', '4c76725a-6c99-311c-8e06-4fbc3501aa79', 'd5fe89a2-f9de-399c-8004-f0e8f70b4ff8']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1654.0, lrx=517.0, lry=1680.0, height=26.0, width=379.0)),
     Layout(active=True, _annotation_id='15e993a2-8924-38f1-b21b-4f3e893791ee', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='41986dc3-40cc-349c-9d28-268ab7c240bc', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=81, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d74d4b2c-21de-39a9-b98e-ae3ccbaeb24b', '95e57c20-5a69-38f7-b8bb-5769546e213d', 'f3b6b7cc-5bf8-3daa-951f-f6eb896292af', 'f572e053-05fe-323a-9e62-d0f24563e921', 'df3bead2-c583-3a80-a314-025a55790e34', 'd4dd363d-15fb-399e-b2d3-29b04f5fcf3d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1654.0, lrx=1012.0, lry=1677.0, height=23.0, width=391.0)),
     Layout(active=True, _annotation_id='7a561cae-83c5-34d5-ba9d-b851db5075fc', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2407ca89-ff7b-3dca-b5c8-a09013321e01', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=82, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a10c458d-06ee-33e5-914b-9ff4ac1516c8', '35c3a8ae-7fb5-3fc9-8d44-c4b0d1621f0c', 'a0a3e292-586d-3385-a555-61ba798140d8', '5e986f90-7bbb-34b6-af5b-d57fb347c0d8']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1654.0, lrx=1502.0, lry=1680.0, height=26.0, width=398.0)),
     Layout(active=True, _annotation_id='7b298e50-715c-3a96-af43-2e49fb3e6b66', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='526d3223-e9f0-3b55-a8ef-7cd9d83c19da', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=39, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['cf6e1ba4-d884-3b28-a7a2-9908bc1ec723', '571462dc-cb7c-375b-84d5-b9ea5290da59', '3f7684dc-cc08-35ad-9d7a-253d6a82e873']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1614.0, lrx=531.0, lry=1644.0, height=30.0, width=391.0)),
     Layout(active=True, _annotation_id='3ebab06f-133b-33a7-bf4b-1a865a70fa45', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='12767b2d-49c0-3dc8-811a-39e054f124f7', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=79, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b83f6970-c9de-3c6b-bb80-0d3cd3747d9c', '25b283a0-53bb-3d12-9d38-1d3dacc848b7', '1545535c-036e-3b5c-8ae2-fc0a8b21e9ea', 'd531c49f-c4c9-365c-84e7-50286cc6ea57']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1618.0, lrx=1003.0, lry=1643.0, height=25.0, width=382.0)),
     Layout(active=True, _annotation_id='fdad3b8b-0d2f-332a-8969-aee9f19acd89', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b1a76464-2977-3e7b-a527-a09be8249eb4', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=80, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['0d84a85c-9f5f-3e60-a2a2-539e37b3aa5f', '2dcf119f-4c85-3ec8-9472-d63ba52122f0', 'a757e2ea-99ff-3a8b-8463-dbfd64a9ead3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=1618.0, lrx=1467.0, lry=1638.0, height=20.0, width=362.0)),
     Layout(active=True, _annotation_id='7f5a7056-e680-36d3-8c5b-7db72010ca41', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='fdb023c5-59ac-3e4c-8a79-58099f37322f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=38, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d3c79866-7c76-3bc0-bf0d-9968cf9efd99', 'b9f43364-c91b-3d63-81ee-a073074dcd20', 'ca804ae2-86b1-3d71-a1a1-7df657021f86', '05908812-cbd2-3ac5-b1cd-6cc7bfb1d752', '4b0bab53-bab1-3237-bf56-0792c70db22b', 'ab0b2350-6e94-3ff5-a614-a65103335141']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1581.0, lrx=504.0, lry=1607.0, height=26.0, width=364.0)),
     Layout(active=True, _annotation_id='ea3ff2d1-f30e-30d0-8b91-24e1f5e1347a', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2803d9ea-9b75-3262-9cf4-5023dcd2ab10', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=77, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8ef824e9-79fe-3521-914e-940c802ba2e9', 'fbef215a-3b97-3975-9186-531746aa600a', 'c6772ac4-010e-3aae-b9df-1728499bbf2e', 'b053cdcb-5410-35cc-876d-0ad8de724585', 'c30fbcb5-2654-3f93-9c1f-4167e2023256']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1581.0, lrx=978.0, lry=1601.0, height=20.0, width=357.0)),
     Layout(active=True, _annotation_id='cebaef88-965d-3aea-a216-94f15c7f0388', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='66437c8a-72a2-3745-82fa-5112ab60f447', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=78, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8698d915-41d1-3984-aa40-28f786b7ab11', '54da22a3-44f3-39fa-8e2a-b3dffbc969e6', 'c6a9668c-a224-3f05-bf0b-bb92a58aafd3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=1581.0, lrx=1511.0, lry=1606.0, height=25.0, width=406.0)),
     Layout(active=True, _annotation_id='e9f6c4e6-7eda-3516-91d4-650de63053c9', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1da2ed3e-4294-397d-b866-1a81b9764016', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=37, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8c095084-788b-38c1-85ad-8dd149452d8a', '5f420b5a-6b71-3197-9753-cdc7a2430dad', '0315c629-3e4b-3096-a1db-bf9430471f7a', '52a130c0-b221-3e4a-918c-8e99cc39c7ed']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1545.0, lrx=519.0, lry=1571.0, height=26.0, width=380.0)),
     Layout(active=True, _annotation_id='2ff52542-a4e8-300e-8e25-b7510d026743', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='14066261-4df7-3d2f-bcc2-10e5701b3866', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=75, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b0efaf0f-89d2-3915-8195-ca741fbe408e', '5e91e9d4-84eb-3069-8003-eb9769d55a7c', '6b05843a-3529-36c5-8929-ad76b8584d6c', '69f5ee78-c952-3125-8bf6-60e023df8ca0']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1545.0, lrx=998.0, lry=1565.0, height=20.0, width=376.0)),
     Layout(active=True, _annotation_id='4da23361-dd45-3ae4-9a2e-6c27011005b5', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='23baa0f8-8c42-3cd5-a6a2-f5d8c8daa7d7', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=76, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a5bb9294-c73b-30f3-96d3-9e1e67c85ad8', 'f8531af4-105a-36c8-b7ec-a90dac64b70d', 'd80858e1-ab71-3a0b-beed-e2ed36e108a4', 'a8165b6d-0033-3e28-b687-8bc3b32c15fe']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1545.0, lrx=1496.0, lry=1570.0, height=25.0, width=393.0)),
     Layout(active=True, _annotation_id='9076c5e3-3f0e-3487-add7-fed12f8e48a1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='f46a18e7-9fc4-3784-81e3-7a9810600de5', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=36, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['f587583e-8057-3af1-bd03-2f9d463f200c', '0219c89a-473a-3f78-a801-f58aeaee0450', '45967a9c-b637-3b92-9d27-be5dd8f45a40', '4a263ba1-606e-38a4-9930-c08df86a1eeb', 'f87f9618-9c1d-31ab-9e76-7a179ea003b7']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1509.0, lrx=521.0, lry=1535.0, height=26.0, width=382.0)),
     Layout(active=True, _annotation_id='121c1979-d69c-3710-b26f-8f71edc36ee8', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4c962b50-aad1-3502-a360-5c2395847f88', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=73, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['729aed2f-4377-3463-99a1-534f3372bc6b', '1b16db09-55a3-373a-a08b-15d53255f687', '4d483cea-131f-3da4-aced-c04df3dbe111', 'feefb923-9daa-36c9-8048-60a0164692ee', 'becd1d0a-c665-309a-838f-bb0620605f13', '76e24384-7c53-3c73-8356-1df38b455c7b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1509.0, lrx=1006.0, lry=1529.0, height=20.0, width=384.0)),
     Layout(active=True, _annotation_id='3f806ac5-2d8c-3181-95e4-4b5a5089a5db', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c8d3364d-7cf8-3789-a555-f1a8981616d5', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=74, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['234fb343-3bce-37c5-874c-9b0361437abc', '179e4af3-f1e3-34f6-81eb-09fa89c7528f', '6cba8b0f-3726-3384-a86e-9811e9452877', 'a1d4b87f-b7f2-36df-97e3-ef2a52a418e7']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1509.0, lrx=1515.0, lry=1529.0, height=20.0, width=412.0)),
     Layout(active=True, _annotation_id='e7863371-7618-30f1-a3ad-c9babb26d5ba', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='12afbbb3-a423-37e8-bc1f-ed1e1abfae1d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=35, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4b9b07e0-f0b3-3853-b224-ac51d096825c', 'cc836e6c-2875-32ad-89ab-710a644fdca1', '9008d62d-2a94-3fb0-be07-a9da444b891f']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1473.0, lrx=535.0, lry=1499.0, height=26.0, width=395.0)),
     Layout(active=True, _annotation_id='ca81f26a-de3b-30a6-889f-8090efcd06d6', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5eb19fe1-513d-33af-98b3-eb96d662f1da', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=71, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['bbd03fb3-4c81-3b08-9ca2-ed0d6040b9f1', '3d781e5a-f1dc-3899-90ef-ccf520ad77c3', '47ab7e65-1f70-32be-b47a-4de13445d0d9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1473.0, lrx=1020.0, lry=1499.0, height=26.0, width=399.0)),
     Layout(active=True, _annotation_id='6e7d7f5b-0b03-33be-b5aa-eb08c1bcdd1c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b384f6b9-c8cf-34d1-b493-5ccad00f9f57', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=72, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['0739897c-9727-3c45-ace6-e578aff3d9c6', '7b242daa-fbe8-3cb7-a9e7-0475d739f9ca', '72f4d314-dbf2-3c35-bf9f-178941bea502', '9cdb56d9-9110-3289-b553-ca58625bcba1']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1473.0, lrx=1454.0, lry=1499.0, height=26.0, width=350.0)),
     Layout(active=True, _annotation_id='8d0af1e3-7cd2-3de6-b646-28d0a0bdac90', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='cc00f43f-a364-33a6-8ed0-ff8f359860f0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=34, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4ba57a24-f21c-3cc4-8af3-8746ccdb4c98', 'f6cb5e11-31e7-30c0-8624-2f49dd2258f4', 'b4d36b33-c91f-3e40-b532-381cb57b04a3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1437.0, lrx=484.0, lry=1460.0, height=23.0, width=344.0)),
     Layout(active=True, _annotation_id='7b4cf7e1-9f49-3b02-af45-e79749aa8409', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='66a9db79-c2bf-30c8-b42c-86c0ba5970c8', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=69, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['6bb71ccf-47ff-30e3-9621-0a8ad1af250c', 'c61b82b3-d5d2-349c-8b38-b50676f88818', 'b749ed08-8651-3157-bfa1-43efc3553f5e', '419db6b8-79cc-310f-8ecd-115f22e25e03']}, bounding_box=BoundingBox(absolute_coords=True, ulx=623.0, uly=1437.0, lrx=1029.0, lry=1457.0, height=20.0, width=406.0)),
     Layout(active=True, _annotation_id='b0f0b86e-b572-35bd-b217-aa4915644ffb', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='d473ef72-c7d3-3a14-9293-16e0090bd763', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=70, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['f837dc62-24b6-3764-9102-3f8276a31953', '4fc65513-2936-3aa3-bb28-55a9675b0624', '15c39642-e6e1-33d3-8a79-c7180bd220af', 'f66a0a20-a59b-3058-b72a-7b8c02d41c2a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1437.0, lrx=1477.0, lry=1462.0, height=25.0, width=373.0)),
     Layout(active=True, _annotation_id='c3262754-b5d9-313f-9e10-0f804bfd6fba', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='de997a05-f567-3e55-bcd5-641fdad603b9', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=33, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['596a4967-2d5c-3b1b-9dd6-1fc57ba021d1', '6f46201d-5d32-3df7-929f-5cfb04bf0fe5', '6827de87-12a9-353c-8a14-e21915908f44', '89bb8167-dbb7-34fb-bd6f-8552829df50e']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1401.0, lrx=547.0, lry=1421.0, height=20.0, width=408.0)),
     Layout(active=True, _annotation_id='464d0bcc-8952-315b-a837-4cf512c76588', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='936fc572-97bb-3a47-92f9-56c3872e5386', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=67, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8c0a13e1-c96a-3b7d-a06b-3a35e7fdac76', '920a18d1-a758-3bf2-9593-fe503707d3b2', '6729eb7e-0e1e-38ea-868a-acaf53475a0d', '29a42558-ea19-3f14-8416-2e579fad0e31', '961ea2d9-b429-3f6d-ad1a-39a7348dce39']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1401.0, lrx=978.0, lry=1421.0, height=20.0, width=357.0)),
     Layout(active=True, _annotation_id='1e026d7e-4ac9-33d9-833f-ab75c3f6dd83', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1c15e577-a40f-3dbf-af84-38ad1aa163ac', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=68, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['5f7599f5-a346-3b96-b34f-eb91d89648fd', '47e87089-a041-3247-8a83-32dfe884abbe', 'c34a58af-1277-3e67-aad1-6eb110845353', '917905c8-0624-378a-9761-b9492d7e748f', 'c2899c52-e8eb-3e17-bca5-0150d897ef7e']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1401.0, lrx=1496.0, lry=1426.0, height=25.0, width=392.0)),
     Layout(active=True, _annotation_id='cf4d4cdb-362b-34e8-a124-29512740d684', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='73c9b56a-0d28-306b-9d8e-dba25acfa276', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=32, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a829a300-2d2e-3a00-ac00-d9fcb32b8e4f', 'eda4c95c-c02c-3dd8-983e-1e4a791f7c79', 'c677238b-3c0c-32a0-ab4d-89a90dd03a0d', '63f4e7cb-9c85-3f86-bdf9-8a54efbaddd8', '8c173265-2904-30b6-9a98-ecbd5ab48a3a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1365.0, lrx=534.0, lry=1389.0, height=24.0, width=395.0)),
     Layout(active=True, _annotation_id='311aff64-208d-3470-b4fa-47cff65cd9c0', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5d96bcfd-b318-3e5d-9493-ba820ea2d6b8', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=64, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['73ed3821-227d-3abd-8b9f-3c9f052ac561', '3923e616-df8a-3427-97d5-dde29de4589e', '7a964ef3-1aa8-3321-b478-3a90ce9fc712']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1365.0, lrx=1029.0, lry=1391.0, height=26.0, width=408.0)),
     Layout(active=True, _annotation_id='c029e905-4c0b-3763-a7c9-cad693446b6d', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='bc658b4b-536f-3e96-a6ff-058ba365b915', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=65, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['febadf72-3e88-3abb-be54-06dc2ff8b961', '9d4bc3e9-4458-33fe-b289-9bd2da9fdd3d', 'da4d1012-3691-3e82-a228-9095671af5a4', '949da3d8-c73f-38ec-97b7-5f9838f5c551', 'e13c0ec5-88f0-3703-b79a-057a76cbaec6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=1365.0, lrx=1489.0, lry=1391.0, height=26.0, width=384.0)),
     Layout(active=True, _annotation_id='c0dd1fb8-8be0-35a1-a0a5-2835b3d28b06', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b391af58-9e0d-3f40-b9c0-a72fe73cf867', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=31, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['006f8d19-97f9-3b93-b62c-0ebb73f03b24', 'd1e572bd-bef6-3178-b674-a2b5be6d69b3', '9e48f77f-5315-33f3-82af-4926388d9c6c', '4bc7887c-aa41-3148-a202-f1092878e9ca']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1329.0, lrx=546.0, lry=1354.0, height=25.0, width=408.0)),
     Layout(active=True, _annotation_id='b77e7483-0fb9-38d3-99e8-172725ac8ca2', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='041a4de6-eaec-336d-87f9-84f431ba4d99', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=63, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['01b0d929-8816-3371-9196-1367a05bce48', 'af6c3483-5817-32c3-ab89-50318976133f', '9ce8fb99-b9a3-3df9-962f-831089d8c922', '3db2bb14-4750-37ca-9c75-73a94bca8607']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1329.0, lrx=1001.0, lry=1355.0, height=26.0, width=380.0)),
     Layout(active=True, _annotation_id='0b1a98ac-f99d-3831-845f-9fea537c7d95', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='7411fe1e-aefb-334e-bff4-dfd18897437b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=30, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a450a0bc-3972-30bf-b6f3-94861e23f711', '8edbf352-0923-3823-a882-b1ad3ccba94b', 'a0269ad6-13aa-3ee4-aef8-05e4128a3ae6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1293.0, lrx=537.0, lry=1318.0, height=25.0, width=397.0)),
     Layout(active=True, _annotation_id='5d26c6af-e8a5-3682-8d14-8dfded3d588b', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='6c54853b-a31b-3afd-8ab1-b394772da87e', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=61, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c47b06b1-f4df-3e14-9ac9-3e5a4ce9d124', '2c542cd0-a1cb-3970-a70c-3fd0800dd5a7', '04134eba-9f9a-3196-8397-60295607e7f6', '2de85afe-f8c8-3e44-8545-52493ad8c2fd', '22827ea7-1b6b-30f1-97cd-645d63499204']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1293.0, lrx=986.0, lry=1313.0, height=20.0, width=364.0)),
     Layout(active=True, _annotation_id='a07c03af-2974-368b-b800-775d6bf74098', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='89fece89-8478-31b6-a396-ea4ebc5a6df1', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=62, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['39ebe51a-e4e7-3f8d-a5ec-7337ad331d2c', '1f3dafda-d23a-33ad-8fd5-2ae5e850c3ef', 'e51ad83d-2eb7-3610-9273-46175fdd7ab3', 'cb483f40-80f3-341a-8cc7-51941fbd6cae', '7c56238b-b623-3dda-a0c9-edf5bab818d1', '4feef5da-7a7c-358f-9432-d028113d35b9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1293.0, lrx=1455.0, lry=1318.0, height=25.0, width=352.0)),
     Layout(active=True, _annotation_id='edc765c5-6731-3a9f-a65a-90f5b86d1a56', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5df45c1a-733c-3007-a639-c2b131c557f6', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=29, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['0c0c7aa2-0ab4-325a-bfb2-08e28a4c9279', '119202b2-c00a-33a3-9e93-8ec0e2016a68', 'e6638e59-1d86-3cbf-907e-02f32ac832fc', '9debdf1f-f43f-3015-b342-fedf34f1d7cd', 'b65d1da5-ee8a-3ded-8547-83a8105fd479']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1256.0, lrx=522.0, lry=1282.0, height=26.0, width=383.0)),
     Layout(active=True, _annotation_id='a88f10c7-5fac-30c5-9cd9-f7b889741165', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c37e0c31-2491-3483-a978-6dd3556a8c7e', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=126, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b1d8900f-0523-3203-8ead-2a4e4bd09511', 'b56a5a46-003f-39ec-98c5-9f0c8e7284af', '85341111-43f5-3607-bde8-5be7485a7806', 'b27fc572-dd14-3e1d-8c7b-9fc37a2a7b1a', 'fd47d781-6443-3da3-a285-67086bb77b6c', '69db1f03-ec0b-3b5e-8e83-34eef6968952', '8ca20ec1-23cf-3be4-8ca3-2d8978188599']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1256.0, lrx=1020.0, lry=1276.0, height=20.0, width=399.0)),
     Layout(active=True, _annotation_id='87bb27c6-05eb-3425-82e1-87c5a5a908c8', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='14eaf693-7a7a-3bef-b0eb-8591df24c8e3', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=127, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['ead40fb9-e8ef-3a5a-b5d3-4c392b114df2', 'ab61cc5a-339d-3f95-95bd-3765fcb84862', '18ddba08-83e5-34bc-9352-5ef4920ad198']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1256.0, lrx=1491.0, lry=1276.0, height=20.0, width=387.0)),
     Layout(active=True, _annotation_id='a42d9f90-74f7-3e66-810b-f2cf7ef630cf', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='39dc8eda-8a9d-3728-ab62-cc8b0bab5f36', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=28, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['fa1605f4-f782-319b-a6bf-cda535ead5bb', '15b10336-fc4d-3242-b2ff-5b4e8e495a8c', '01b9ba15-d315-3b06-a79e-ef3e4e8853bd', 'd991b3e7-345c-3148-9154-660d02a94365', 'a4feec3f-9feb-368f-b105-ba75c1851eee']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1220.0, lrx=542.0, lry=1246.0, height=26.0, width=402.0)),
     Layout(active=True, _annotation_id='a50f529d-5d4f-390a-a7d6-39fd7e5f7a41', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='8aee8248-4ace-32e9-9b5e-e8624b1194c7', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=124, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b9898c34-adac-3d79-a9f6-8b341cff3954', '326c9b97-bcc6-36d9-9c51-5de0590a2a55', '7dc84b55-e008-30a5-a98d-86398b13d0d8', '271a7d9f-dc13-36f8-9013-67e1a6363b04']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1220.0, lrx=1010.0, lry=1240.0, height=20.0, width=388.0)),
     Layout(active=True, _annotation_id='45918f93-387c-3fd0-b941-2c3d36e39253', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b33958d5-025c-3a6c-8ec8-28106dde18f6', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=125, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d6c65ce6-b867-39b9-a2fa-d8489a9c38a2', 'b3e62866-02ae-3c3d-9f9e-f3f9a9fd5fb2', '442c20b9-e5d2-3d0f-8f3b-cdac59f8735c', '4e143b15-fe90-3537-951b-e3b0436ee5d8']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1220.0, lrx=1481.0, lry=1246.0, height=26.0, width=378.0)),
     Layout(active=True, _annotation_id='91297e13-aada-3281-8570-719b51daacb6', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='a5b4542a-3225-3c9b-a768-c6fc1f857083', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=27, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['ac3013d9-7c88-3a78-bffe-88410562c591', '5667f41a-a50f-3766-b65b-80be30728fb0', '08dd16f2-4a3c-3093-a19e-ddd5c74bc248', '149a6a6e-4883-3c4c-b369-1f25c62387fc', '0dada72a-bfc2-3c26-a729-cc66e01dd1a6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1184.0, lrx=531.0, lry=1209.0, height=25.0, width=392.0)),
     Layout(active=True, _annotation_id='cbe156d2-f824-3410-95e9-16ebe87ca598', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='f39ae3f4-3284-3c9f-9edf-1126ca62912e', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=122, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['fb50b831-5da1-3bed-842e-3d088b2ec4ef', 'b34f5ac3-d521-3768-a157-86ca092a313e', '6b176a95-d092-3339-aee3-fdd37446d92f', '174c9e94-7a36-3f19-8098-0dfa2b60f545']}, bounding_box=BoundingBox(absolute_coords=True, ulx=623.0, uly=1184.0, lrx=1000.0, lry=1210.0, height=26.0, width=377.0)),
     Layout(active=True, _annotation_id='300647ef-4077-3db7-a902-79873f1f9465', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2bfe656a-dfd2-3451-8b3a-aafbc5927723', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=123, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['5e9b9cd3-0cfd-3a8c-8098-d5c1041aa6cc', '591439ed-1b0d-3f47-9ae6-b3d285f4eee2', '87dec3f1-811b-3dc2-a583-3b574bc01d7b', '31b2fe24-f903-3e07-b001-e9ec8c801cd3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1184.0, lrx=1470.0, lry=1210.0, height=26.0, width=367.0)),
     Layout(active=True, _annotation_id='5e759caf-7505-3963-8d36-4d9cbd349488', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='bb1a3553-6a51-38c1-9fff-4cba74f4195a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=26, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['51049c17-4eb7-3fe5-ad4c-7f6a37f88003', '1bf4dc83-d65d-3d76-a740-f299ce3c55e6', '22ddfc9c-7a63-35a4-9477-171042ec0b10', 'e8529060-5c5b-30eb-a045-eeebc9429b97']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1148.0, lrx=508.0, lry=1168.0, height=20.0, width=369.0)),
     Layout(active=True, _annotation_id='e53ca2ea-2901-30fe-8b6a-6c7d436e01ec', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b9f6b354-be79-34c1-8a73-853f24860e0b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=120, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['6b0f11a7-0638-34e6-97e5-b26fc9f2b29f', '4da73e14-b4da-33c6-80e8-d4a0571713d2', '08cb7fe9-8a67-31f7-8ab8-79a565dc9816', '7c7e44a7-efb8-3b88-935c-da637bdbd61a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1148.0, lrx=971.0, lry=1174.0, height=26.0, width=349.0)),
     Layout(active=True, _annotation_id='c45ee45b-b60d-303a-ae65-691cacdbf10c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3d92de52-d83a-3a3a-9df0-3448121ee3f0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=121, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['db97f019-bbce-36e4-90b8-2d5c2cd8fd56', 'e8c28f7e-9751-3f69-afdb-34b154231659']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1148.0, lrx=1493.0, lry=1174.0, height=26.0, width=389.0)),
     Layout(active=True, _annotation_id='864c1b2a-c2d8-33fc-97a6-56f5cdf82d94', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='7d9ab63b-1a23-3020-8a5d-2d70f0103c6a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=25, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['2ff33aa9-ff8b-3d3b-a231-29ab956133dd', 'dd3e5877-8041-3f57-bae8-4f3fd6ad1b3f']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1112.0, lrx=508.0, lry=1132.0, height=20.0, width=369.0)),
     Layout(active=True, _annotation_id='db92a2df-abc6-35aa-842c-4e791b8e8d1f', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='399d6bc5-528c-3f91-b1c4-6801c5e40ad3', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=118, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['e2d6bc6b-0f03-3425-8efb-536a325fdee5', 'a9877410-48e1-3203-ac73-86a7bd77876f', 'd5b156fd-2ab2-3300-b93a-1af82a3f9789', 'bac979bc-b0ef-3921-b186-04588e401820']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1112.0, lrx=1005.0, lry=1138.0, height=26.0, width=384.0)),
     Layout(active=True, _annotation_id='b5240ab1-6472-3f13-8ec4-4035a87ed6be', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='19ca9f2a-3cec-39d8-a253-1593a8fd74e9', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=119, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b7163e52-9aa7-3659-b132-b7f1c930f1c8', '781e8f05-46bd-3c76-bb28-9cad43152c56', '0ba0a874-cf82-32e3-95ed-a8b25a15af37', '5554fd85-6d8a-349c-9310-a8d364453b03']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1112.0, lrx=1480.0, lry=1138.0, height=26.0, width=376.0)),
     Layout(active=True, _annotation_id='5c1f1ca6-3243-34de-b143-18eca01f9b76', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='af667c5f-52fd-3459-b50a-6c21679a7b5a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=24, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['56bfc7f0-2bc6-358a-a5f4-2676a61095c4', 'd19f2f7a-ca7b-3d92-b36e-354325e22a3e', 'a42fe6c3-e0ff-35be-9bf4-16cc19ea0214', 'e721eff1-f2ac-3cee-9e1c-23a41e37d355']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=1076.0, lrx=508.0, lry=1102.0, height=26.0, width=368.0)),
     Layout(active=True, _annotation_id='ee39db81-0ed5-3f32-a87d-df0e29b13ec1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='589120dc-0f59-347f-814f-2a565bf66925', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=116, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['afe104f9-5c5a-396d-8a52-e8003cce9817', '24e65a90-ca0b-3314-8f19-c049f5dfc26d', '6d79d70b-3de6-319c-a7b5-c954a27ca545', '81c3ad03-5b96-3214-87f5-30c29e9c2ce9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=1076.0, lrx=991.0, lry=1102.0, height=26.0, width=369.0)),
     Layout(active=True, _annotation_id='4a29972d-d082-30fe-be00-2a2ec1ffe5e1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='a4cb60d7-bacb-3bd2-ae4f-30ef0f96c4c0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=117, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['79cebb72-874d-3afc-b1d5-a8c8056a05c7', 'eeaf040c-664d-37a8-bace-f20186c2e448', '577921dd-720e-35e7-b8b7-ee2c31d02ea0', 'bb0e29a1-07ed-39cb-bcaf-8aeb3fdd09cc']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1103.0, uly=1076.0, lrx=1501.0, lry=1101.0, height=25.0, width=398.0)),
     Layout(active=True, _annotation_id='17e721a1-3bc6-3b34-9812-5ec89558c75a', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4437b667-e48a-3c16-966c-7e760bab9cbb', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=23, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['1beaa75e-29c9-3512-8257-d20e02a855ac', 'f2637a56-2092-33e0-a16b-78be667a1814', '525baca3-629c-3ae5-a9db-ad115bab343a', 'b00ea2fc-383a-3e9a-b55f-005405045b83', '6b9e8bbd-27af-3490-a769-572736d6fa62']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=1040.0, lrx=486.0, lry=1066.0, height=26.0, width=348.0)),
     Layout(active=True, _annotation_id='d39c07d7-ac86-3fea-921d-17b0350de7f0', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='f4c58251-3881-3f7b-98e3-4271ed33dbd3', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=114, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d6e33fa5-3327-3d2c-b1c4-6d74dc27618f', '1f515d01-705a-3aea-9f21-fb74342434c5', 'f6c34068-3602-3e93-8c21-98c13aac6c2e', '0f999828-9fee-3f87-81cc-e76aa501c4a4']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1040.0, lrx=993.0, lry=1066.0, height=26.0, width=372.0)),
     Layout(active=True, _annotation_id='43cb258d-c372-3529-a8ca-4ce976cbe5c9', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='25ff42b4-bdbe-3339-b1f2-3601c421b933', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=115, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['042659bb-e470-3496-8dd8-476077375571', '70c4f969-8eba-3ce5-b43d-02540feb8fbe', '4944ed36-2b30-310f-99c7-e31e14901d39']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1040.0, lrx=1501.0, lry=1066.0, height=26.0, width=397.0)),
     Layout(active=True, _annotation_id='61e31888-5d0e-3f1a-8331-69d12ff494ee', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='ebc133b7-7ad9-3d06-a2ef-832dfd942f68', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=22, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a0d77a30-8006-366c-b5c4-fa0263a6067a', '80744d2d-846b-3013-85f5-ae842527af2b', 'ca42f2e5-ecd3-35b6-8cac-1e70c39470ed', '61289768-206f-3b8c-87d2-c4ea80769fb7', '4bce24ce-39a0-37f9-b4f7-004dd5ce0f7d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=1004.0, lrx=535.0, lry=1027.0, height=23.0, width=396.0)),
     Layout(active=True, _annotation_id='8989f112-ea0c-3902-a1c8-816b9fc61d0d', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b1019d71-4278-32a8-989c-037b05454f2f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=112, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c5c889e9-9a4a-3e4e-81dd-db7caf2bba7c', '639684d2-98b5-345c-a91a-37b595bdd39f', '8ae1907a-c387-3fb6-a2f7-b688012d567c', '25121395-36d5-3291-8777-3dd1b24b2aa4']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=1004.0, lrx=994.0, lry=1024.0, height=20.0, width=373.0)),
     Layout(active=True, _annotation_id='a40bbd64-4261-31d0-9ef3-33d52ef202d4', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3753ab0f-d5f6-36fd-9521-48fd582c0fb1', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=113, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['e22e8411-1368-33e7-8335-07567cb83ddd', '15298bd8-426a-370b-8ac2-6f8ff337463c', '05159e26-ab31-3d89-838b-8c07bd3e7aa0', 'd65b4ccb-4cc7-32f1-9119-fbe569b8d488', '7ce76514-da6c-3ab0-b129-50e5fe3b3285']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=1004.0, lrx=1482.0, lry=1024.0, height=20.0, width=378.0)),
     Layout(active=True, _annotation_id='da3fa14b-641f-3249-8917-37d4d71c5993', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='26306472-4e72-3e5b-ab17-d047c1f1817d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=21, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['428b82da-a9f6-3ea6-892e-23bc2fcd5c5a', '612869ae-0f5d-3400-9ab8-0c5e8ab92616', '103183a6-b110-3e95-b3b7-8be63aa4efdd', '5967f9d3-1c36-3464-8450-9dd794dfb96d', '68d95bfc-922d-3070-be87-fc5e1c9c0b05']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=968.0, lrx=523.0, lry=994.0, height=26.0, width=383.0)),
     Layout(active=True, _annotation_id='6d9ba447-90c6-3e5f-9fb2-e32f38e4088c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='a43a3748-f80a-3bfb-96e7-336a960b6f3f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=110, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['9fef984e-d7af-3f5d-87c1-c681d69d7455', 'e48fb9aa-1656-38b0-8cd3-b3f6ad0d07aa', '04786561-b3c1-338c-af39-50c9d0f35520', 'a7d70ee0-34dc-3c3e-8cfc-f6e8e26f9eaf']}, bounding_box=BoundingBox(absolute_coords=True, ulx=621.0, uly=968.0, lrx=976.0, lry=988.0, height=20.0, width=355.0)),
     Layout(active=True, _annotation_id='014f02ba-00dc-3f91-bd1c-16da2a19e6f1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='355ed7eb-cc02-3157-a20b-1f02d1a0c66b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=111, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['5e57598e-cc35-31b5-891a-ef220d7add57', '21d68b9a-3b7d-3960-a4ea-0060c389257d', '6cbb1c4b-d850-3cc6-b03f-e02944478504']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=968.0, lrx=1486.0, lry=994.0, height=26.0, width=381.0)),
     Layout(active=True, _annotation_id='b7b1473f-e30b-3465-835d-b1e6257d1676', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='b8373b5b-7cd9-39e6-a0e7-9bb7cafd163a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=20, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['144adf46-31a1-313b-a953-45f02624e528', 'd03e649f-cf1b-3baa-9a15-155ea4e7bf71', 'c4f2a02a-a876-3776-9ea5-e3b7ef614667', 'acfef675-180a-329d-9deb-05e217aadf0f', '1079ab75-c3f5-31c4-8844-17312f6c60d3', '48750d58-04e9-343d-8398-6eee32d6621c']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=931.0, lrx=515.0, lry=956.0, height=25.0, width=375.0)),
     Layout(active=True, _annotation_id='6c37ab5b-f585-349e-8822-2428b3183ccb', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='f5836e3b-e03e-33fd-9641-20d4dbd825c5', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=108, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8d6e9ab7-0082-3ed9-8be9-29f3792faa92', '8e44dc3c-0e2e-3821-92b7-cd9a00c7e8fb', 'd681835d-34e1-3c94-acfa-bfbad10b6f6d', '586b1353-1c31-3b89-b21a-767f956dd418', 'd05b6185-b11b-35d4-ad93-90041c8e2c1d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=623.0, uly=931.0, lrx=1013.0, lry=951.0, height=20.0, width=390.0)),
     Layout(active=True, _annotation_id='b3ee3ddc-73b5-3226-b42d-6ccddbd83efe', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c33848d7-fe60-3850-b490-436ebb84ec67', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=109, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['a52c0f07-5936-3a99-9c9b-797904c556cd', '951b55bf-d5f5-3c8c-ae6f-227d19bee61a', '1ce7299d-4fd4-3f49-b1d2-0b11e063236d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1104.0, uly=931.0, lrx=1513.0, lry=957.0, height=26.0, width=409.0)),
     Layout(active=True, _annotation_id='010e49ec-4d00-3cae-83a9-a63909ca6d38', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='323d811b-903b-3d8f-99df-e6c0a75600c7', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=19, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['e5af2972-98a1-3f60-84e6-0876a361c8e1', '4b85a73f-c3c5-3d25-8aec-092bcebadf93', 'b90fe5d3-67df-3b7c-8901-509e3d527269', 'c4c6214f-c3c1-3be0-b3fa-4ba308100eb3']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=895.0, lrx=505.0, lry=921.0, height=26.0, width=367.0)),
     Layout(active=True, _annotation_id='4fe7df06-4cb0-300f-8ac2-c9e034b81f74', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='37a0caba-fdaa-3eb9-885c-9aeb2161c2ce', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=106, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['f0a430a8-448a-3fe7-b029-04fb6bbc6610', '3f562642-f7a9-3e46-b0c5-6fce924e3700']}, bounding_box=BoundingBox(absolute_coords=True, ulx=622.0, uly=893.0, lrx=908.0, lry=916.0, height=23.0, width=286.0)),
     Layout(active=True, _annotation_id='bd7baa97-fc74-3473-98de-fbc07c7045db', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='6d407a82-d850-378b-8d21-eca704b06531', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=107, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d9d26b61-ca01-3cd1-8a94-0706f5664e58', '9910279c-a48e-368c-92c0-1240426bd9cf', 'fedba8c5-c79c-3340-87e3-be21fd99c2c8', 'c2a0bcbc-e80a-3292-b5fd-0f114dcfced7', '44d5e3be-a293-3863-acaa-05c1e23ac212']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=895.0, lrx=1440.0, lry=920.0, height=25.0, width=335.0)),
     Layout(active=True, _annotation_id='8dcffcd9-ea3f-3b37-a773-25e1ea7c1719', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5ed188bb-f11f-303d-a729-92fec6836c87', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=18, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['67991b8d-7bfd-3ccb-a3c2-08a09e2e2d18', '1d8494cc-5936-38b4-8043-4e3f55922dac', '7eb764ae-83de-3216-ab42-37bdaed18c9f', '960a4e4c-43b5-3a46-a2c7-979ede870ba6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=859.0, lrx=539.0, lry=884.0, height=25.0, width=400.0)),
     Layout(active=True, _annotation_id='d9622815-84ed-36d4-9daa-e91a91499d02', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1831c112-1413-326e-bbab-51f69817a7c8', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=103, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4a0093b0-a758-389d-9de8-836d253af83c', '0fa65ee8-a875-38d7-b898-7a02d4fd8e51', 'b9b8f1e2-a599-3671-b99f-c90c134a59c6']}, bounding_box=BoundingBox(absolute_coords=True, ulx=620.0, uly=857.0, lrx=1020.0, lry=880.0, height=23.0, width=400.0)),
     Layout(active=True, _annotation_id='42dc36b4-7d7b-3cfd-90a7-6766ae096e23', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3dca890a-d859-3068-ab60-3b19b11253b0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=105, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['fc1cf4a5-0425-3938-9c83-5c4016bbea81', '7451658a-740d-3bc7-ade8-38444c078853', 'c9cb63d1-1ae5-34dd-9d8c-a6f9fbf8e5d2']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1105.0, uly=859.0, lrx=1509.0, lry=884.0, height=25.0, width=404.0)),
     Layout(active=True, _annotation_id='645745f5-e634-3b87-ac2e-276a977922d7', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1a599997-9338-3964-9fbe-ec2a28bfd71d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=17, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['bc0fedcd-83ce-3434-8fc5-da23e3ff179a', 'a7b8db6d-248b-377e-b649-de59ddde91af', '6b1805ad-61ad-3137-92c6-e1ae24cd9bee', 'f25c5543-ae91-3b98-8348-45e82c179fba']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=823.0, lrx=517.0, lry=843.0, height=20.0, width=378.0)),
     Layout(active=True, _annotation_id='bdd2feb8-d1da-3b65-b6a0-ea42b60f6a05', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='0a2a004b-ced1-33c5-adfe-bc2a135d99ba', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=16, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['04d47aaa-d356-3b1d-aabf-3bb92c8afbbc', '83f7c50d-d8ec-3dff-ba16-802418ac09a9', '770fdc0d-d423-3442-9b4a-294be35f140c', 'a1a8b8b6-7c28-317d-a351-e583a69310c6', '97fac9dd-f9e3-3409-9b2e-4047554a1bbd']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=787.0, lrx=503.0, lry=812.0, height=25.0, width=365.0)),
     Layout(active=True, _annotation_id='773b7559-4401-3b3b-a301-1079f6a129ed', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='d039d076-30a3-314c-9b14-73dc9ab46f4c', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=15, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['afdc1b17-17a8-3701-a002-38fde24d04de', '8c9f4ec1-4e18-3622-bb08-4d69659eddbe', 'e2b2a00e-8d5c-317c-8416-72c9086786b0', '8c50323d-7876-3f87-ac90-29600dfbb556']}, bounding_box=BoundingBox(absolute_coords=True, ulx=138.0, uly=751.0, lrx=511.0, lry=776.0, height=25.0, width=373.0)),
     Layout(active=True, _annotation_id='1ee9f802-1964-3147-b113-2d00b7302ba9', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4160cb2d-2fde-3f8a-aec2-53610cf9710b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=58, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['30458a32-3d9d-3d1a-aabb-84c180fbf0e3', '6fc68a43-b054-3669-be11-c0d8c236dd07']}, bounding_box=BoundingBox(absolute_coords=True, ulx=687.0, uly=753.0, lrx=828.0, lry=771.0, height=18.0, width=141.0)),
     Layout(active=True, _annotation_id='d86ece02-37c6-3cc1-8722-25d87c63224e', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4bbbfcac-537b-3d93-a4fd-62615f06c31f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=59, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b326e466-8330-3c3c-875f-1702e2387e82', '6f53b103-7525-36ae-8591-19bbd292c19b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1379.0, uly=757.0, lrx=1516.0, lry=769.0, height=12.0, width=137.0)),
     Layout(active=True, _annotation_id='16f89a69-1ff5-383a-8fdb-6ee41376010b', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='8bd14b03-7992-3338-967e-78a8c7e0530a', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=56, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['3da7d1ab-e7b1-3266-8113-aa7070736cb1', '619d965d-293b-3475-9ee8-e16d3a5ea1ae']}, bounding_box=BoundingBox(absolute_coords=True, ulx=687.0, uly=731.0, lrx=826.0, lry=749.0, height=18.0, width=139.0)),
     Layout(active=True, _annotation_id='fadd1ee3-af88-38b4-949f-0e0040f186de', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2a31c7e5-385f-3166-bd5c-9233c644dfe2', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=57, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['9328cd01-0fd3-3e8a-bfbb-5d8745377995', '59a5db38-40b6-3115-b125-ce6d41353ecc', '993b6506-5be4-330b-a676-112ff956ee39', '5a8efe73-407d-3ff3-914c-ecc846a7428c']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1219.0, uly=734.0, lrx=1516.0, lry=746.0, height=12.0, width=297.0)),
     Layout(active=True, _annotation_id='46674972-f5f1-3d5d-8f79-04e6a50fa609', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='11b72d1e-5429-3e0b-be71-53d4b47f914f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=14, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['08b4dcdc-1708-3def-8e3e-f9ed8af6ff01', '2dc92429-1061-33ba-a59b-e6bdc0b85f98', 'cebfcabe-4c0d-3b86-b894-78df0e9fdaeb', '6f29dc39-8fe4-32b9-9c05-ff7c9ebdbbf2', '2ef3dd03-ed31-380a-9b77-65fc11d92545']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=715.0, lrx=549.0, lry=735.0, height=20.0, width=410.0)),
     Layout(active=True, _annotation_id='9b8cba8c-c2f7-375d-8b54-48360d747397', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3926b51e-2d91-38c6-936b-1286e26fd7a8', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=102, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['ba7ab891-76c0-305b-97a1-e783f00a25e0', '0c5c7b61-c5c0-3b4d-860b-a5a9f4324150', '6dfb24b3-c18d-3877-9fc9-2112640af610', '4ee0fa38-98fd-3622-8fad-74744eb7fc30', '09a98fd9-76c3-3ecc-9457-682b70678c40', 'a488569d-f02f-3531-93ff-d3885b698f89', '227dc21f-9a41-30c2-8918-b459a498173f', '01269a7c-e052-3388-aaf7-f8a1445f271c', '235e94b8-0946-30c1-9258-ac98941bb507', '01f6cfd3-abff-3db9-b019-699d1821cf2d', 'ff21a2b5-45f0-3f34-8452-5714f660e8d9']}, bounding_box=BoundingBox(absolute_coords=True, ulx=685.0, uly=691.0, lrx=1518.0, lry=707.0, height=16.0, width=833.0)),
     Layout(active=True, _annotation_id='7c0647c2-2334-36c1-93dd-fa60d72c13cd', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='3f73ff78-7583-36fc-9372-c00325c22301', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=13, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['ebfe9887-20a2-314a-9563-8d70ba595ee2', '3eba73c0-6a0c-3153-9806-2d1bde712d2e', '5501081d-5d3b-3a0c-ad40-4971454e4d55', 'ffbb1d56-7496-3ab3-b0a1-b8eb99bfaeee']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=679.0, lrx=505.0, lry=704.0, height=25.0, width=366.0)),
     Layout(active=True, _annotation_id='24bd0172-e9bd-3af3-a65d-b046825cae1e', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='4bd6ea94-faaf-31f7-9e75-879e7379c4b4', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=91, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c4068615-1dab-310a-948d-fd97ea9cf5a7', '7a32254d-4fbb-3d9d-bd66-0ea9af3a404e']}, bounding_box=BoundingBox(absolute_coords=True, ulx=634.0, uly=666.0, lrx=712.0, lry=682.0, height=16.0, width=78.0)),
     Layout(active=True, _annotation_id='70c30314-68ee-3e73-8fe8-b22c43522287', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='6af892cf-bde5-381d-9f3b-f2cbeef1f9a3', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=92, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['803c8a07-cca8-345c-92db-1747ba96154c']}, bounding_box=BoundingBox(absolute_coords=True, ulx=784.0, uly=668.0, lrx=792.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='50666aa3-65fb-3a13-bbf8-d7030cd9e370', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='821eabe9-77b2-329f-b81d-125871be546c', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=93, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b6081abb-79e1-3d02-b3d6-1fce9597b9c5']}, bounding_box=BoundingBox(absolute_coords=True, ulx=863.0, uly=668.0, lrx=870.0, lry=678.0, height=10.0, width=7.0)),
     Layout(active=True, _annotation_id='4c22ee6d-1fa9-38de-b4a0-091425ce16ad', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c72bcc80-9aa3-3913-a4f1-ea04e3894e6d', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=94, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4ed72b2f-7fcd-375e-ba76-c1332711b67b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=941.0, uly=668.0, lrx=949.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='c677a4db-16cf-321e-9f65-99d77086614c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='02ae0854-3670-3826-86bc-a9100b5af1c6', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=95, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['050b1a85-1928-372f-83f4-22872ee93c3b']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1019.0, uly=668.0, lrx=1027.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='0af5d4f4-6590-353b-82e4-938ddf83aff0', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='883e62dc-1afc-31c1-badb-54fa5959cf5f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=96, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['51d3734e-abe8-3dd9-960c-437c5f1ff220']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1098.0, uly=668.0, lrx=1106.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='3f062967-e64c-35ec-b7a1-37aef18974d3', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='312a4403-de92-3434-8d98-d771c3510bb5', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=97, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['f339ec9d-cf83-3a7c-9b9b-2f892aacae01']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1180.0, uly=668.0, lrx=1188.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='89092f5b-eff2-33d8-a545-3797b2f9e50c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='6f1c4234-a7b8-3605-9b49-fcb9a1c2172b', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=98, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['6486264a-a5db-3418-a41f-2f13eb176b5d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1254.0, uly=668.0, lrx=1262.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='22207e8e-8646-3352-b59e-2396a59f0e06', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='a1d2cea9-2d81-3df4-a5bf-bbc42af8e5c4', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=99, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['4c5fdbc5-f8d6-3885-ae22-d3e9f8d1992a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1332.0, uly=668.0, lrx=1340.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='e15f18f4-a650-3c41-a04f-a5b00dedc9be', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c59f4609-150b-33d3-aa08-93703b7d2fa2', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=100, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['d66dffd9-66d8-3676-9d3e-964efae7f54f']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1411.0, uly=668.0, lrx=1418.0, lry=678.0, height=10.0, width=7.0)),
     Layout(active=True, _annotation_id='97755093-7c85-31d5-ba39-3546328ebf00', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='76163acc-9010-39ad-aed5-0babab826995', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=101, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['01850668-3ba2-319e-9375-58fe1ec979bd']}, bounding_box=BoundingBox(absolute_coords=True, ulx=1489.0, uly=668.0, lrx=1497.0, lry=678.0, height=10.0, width=8.0)),
     Layout(active=True, _annotation_id='742cd105-25a2-3c1e-a24e-6da27d057f9d', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='eef2ec77-11f0-3ffd-a6f3-2b0612d0e504', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=12, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['f4a992d4-40e5-38ca-b63c-66996b1faa50', 'a6b3431d-a918-3303-aeb9-86788249a745', 'ea1d6b2e-6971-37f1-9a49-b339fabc7b7e', '56f034b1-ace2-3a02-bea0-54cc657ead66', 'b2f5afe6-ae79-3700-9502-b11c2b6bc717']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=643.0, lrx=546.0, lry=669.0, height=26.0, width=406.0)),
     Layout(active=True, _annotation_id='7df693c0-3619-3a52-be0c-0ff09384dd46', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='125f9508-05a4-3ed0-b351-fc54867d94d0', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=11, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['812f35f0-a42d-3bce-bf4f-d8ecc9d3e00a', 'd3058f07-1625-35db-9ec8-bd7aa9d5aec2', '15c5ef1e-bd3d-3f25-b5a8-7ef1970d22c4', '5facf1ff-7566-3445-953e-ff05c94bf02d']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=606.0, lrx=533.0, lry=626.0, height=20.0, width=393.0)),
     Layout(active=True, _annotation_id='280452ae-efad-38cd-87cc-3de10c992c93', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='1b9b4d4d-8ed3-37a2-9f6b-75d53092d8e2', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=10, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['b9b0fde8-612a-3297-9c71-e101ae75e486', 'a1fbe69b-3431-3cff-a950-f23ab104338d', '436aff74-a8ab-36c4-8f57-369a1d3603e1', '53a673fd-7534-3678-8c8c-00cbbd4057ff', '310af418-bb65-38d8-a2ce-5169dbe4403a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=570.0, lrx=531.0, lry=590.0, height=20.0, width=392.0)),
     Layout(active=True, _annotation_id='a7385300-9485-34ba-92e0-bb3f870bfe87', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='7f577a48-2817-3e58-a562-be8de0f3b546', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=9, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['e5fd2c1a-acae-356c-b021-59a090be85b9', 'ef76209f-6aed-3c2a-a65c-82a3d05df846', '89ed54c9-df2f-3283-901f-b1a3ecf5bd08']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=534.0, lrx=500.0, lry=559.0, height=25.0, width=361.0)),
     Layout(active=True, _annotation_id='068422d0-edf9-3c53-96bb-08334d9b4505', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c2dbbf82-e128-36dc-b244-65d39fd3c8ba', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=8, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['357f87fc-a778-3c19-86e8-2ccf18cd9008', 'e8dfc1a8-9a34-33cd-823c-a7b89294bd06', '0b4dd9f9-d55f-3a70-b2b6-524417499c0a']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=498.0, lrx=468.0, lry=523.0, height=25.0, width=329.0)),
     Layout(active=True, _annotation_id='cce036c8-1fea-31ca-925c-7ed36764958b', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='c5144fe2-5520-3a59-8440-aa567a757bb3', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=7, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['2539b16f-f69d-3cdd-b382-103759e60118', 'cfce2813-c33d-3c4d-8151-554997126079', '33aee75a-9fb2-321e-bdc5-6d52519433ec', 'ca0439ba-271c-3ce4-acf0-c01b8e3fdbf2']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=462.0, lrx=500.0, lry=482.0, height=20.0, width=360.0)),
     Layout(active=True, _annotation_id='e9251e1e-906f-3f30-bb19-4fa8a832e0b1', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='e9755b75-eab1-3b7b-8644-2a1d21e12865', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=6, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['95fb9531-09ac-310a-895d-4dd7dc476468', '01ed7ca8-c189-3e76-bc6d-1d9b7e811840', '5b8f72dc-64dd-3a91-861e-09ac8ac9d437']}, bounding_box=BoundingBox(absolute_coords=True, ulx=140.0, uly=426.0, lrx=494.0, lry=446.0, height=20.0, width=354.0)),
     Layout(active=True, _annotation_id='a53e3876-5b4e-3d61-bd43-26a82f5e988f', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2b6e2acd-486e-38d3-a2c2-dc44eaf26863', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=90, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['c20c75be-c46f-36e8-8245-d01e34d3c29c']}, bounding_box=BoundingBox(absolute_coords=True, ulx=617.0, uly=422.0, lrx=682.0, lry=444.0, height=22.0, width=65.0)),
     Layout(active=True, _annotation_id='944d4f9e-0778-353b-a362-72f5a5f2c9a3', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='5f181ce0-38f7-3397-b113-14f88565ba8f', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=4, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['848e1f44-bfb8-3234-87d3-b955090bf302', '03bb45e9-574c-3478-bef1-86cfdc453512']}, bounding_box=BoundingBox(absolute_coords=True, ulx=139.0, uly=388.0, lrx=499.0, lry=416.0, height=28.0, width=360.0)),
     Layout(active=True, _annotation_id='27ee1f13-8ad3-30e3-96c9-9e3de9f2954c', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='2b75cc93-55ef-37ab-982e-5d299a1ea0b4', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=89, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['ddb59f28-252e-367b-bd82-269411334518', '041fb0dc-4dda-355c-9a83-d7140fdfa504', '5aa0a526-2ef6-3f54-8ac9-9f9385daf7e7', '58fa014a-059a-3e1c-9feb-411af26ef6d8', 'ab051da9-d07d-3c0e-a0d1-57589e9d6aeb', '2201390f-d283-3fa5-a5e7-a45dc7737309', '25732e60-3822-3c04-b0f2-71a352a94a0f', 'b5009670-2ec6-303e-aae4-654cab018adc', '1cafca51-b0ba-388f-9cc4-bbacadd20acf']}, bounding_box=BoundingBox(absolute_coords=True, ulx=620.0, uly=389.0, lrx=1238.0, lry=411.0, height=22.0, width=618.0)),
     Layout(active=True, _annotation_id='83fd5f27-960e-3a22-91c0-2c8c38435dbd', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='0a892e14-9074-3e57-ace7-810606f4aeac', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=3, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['492aba84-9c63-3102-b864-27735a6492f2', 'da233d1b-cf3c-369d-9c03-42a2354a277b', '6398aa8d-40af-3617-b6b6-4e226e1d2d44']}, bounding_box=BoundingBox(absolute_coords=True, ulx=141.0, uly=224.0, lrx=584.0, lry=261.0, height=37.0, width=443.0)),
     Layout(active=True, _annotation_id='9b9bc74c-7f31-3e2f-980c-27149aa98743', service_id='f10aa678', model_id=None, session_id=None, category_name=<LayoutType.LINE>, _category_name=<LayoutType.LINE>, category_id=1, score=None, sub_categories={<Relationships.READING_ORDER>: CategoryAnnotation(active=True, _annotation_id='fc6bf343-a0de-3b0d-8626-56475c9fba74', service_id='f10aa678', model_id=None, session_id=None, category_name=<Relationships.READING_ORDER>, _category_name=<Relationships.READING_ORDER>, category_id=2, score=None, sub_categories={}, relationships={})}, relationships={<Relationships.CHILD>: ['8ba65ea0-5d20-3317-bc6c-b2f46ea834eb', '5c1c1a0a-ce6b-3e7b-b315-c9072e77aa69', '1463b0f7-218f-39d6-985d-6703cf35fa16']}, bounding_box=BoundingBox(absolute_coords=True, ulx=137.0, uly=157.0, lrx=880.0, lry=203.0, height=46.0, width=743.0))]




```python
dp.text, dp.layouts[0].text
```




    ('\nAnleihemärkte im Geschäftsjahr\nbis zum 31.12.2018\nSchwieriges Marktumfeld\n\nDie internationalen Anleihe-\nmärkte entwickelten sich im\nGeschäftsjahr 2018 unter-\nschiedlich und phasenweise\nsehr volatil. Dabei machte sich\nbei den Investoren zunehmend\nNervosität breit, was in steigen-\nden Risikoprämien zum Aus-\ndruck kam. Grund hierfür waren\nTurbulenzen auf der weltpoli-\ntischen Bühne, die die politi-\nschen Risiken erhöhten. Dazu\nzählten unter anderem populis-\ntische Strömungen nicht nur\nin den USA und Europa, auch\nin den Emerging Markets, wie\nzuletzt in Brasilien und Mexiko,\nwo Populisten in die Regie-\nrungen gewählt wurden. Der\neskalierende Handelskonflikt\nzwischen den USA einerseits\nsowie Europa und China ande-\nrerseits tat sein übriges. Zudem\nging Italien im Rahmen seiner\nHaushaltspolitik auf Konfronta-\ntionskurs zur Europäischen Uni-\non (EU). Darüber hinaus verun-\nsicherte weiterhin der drohende\nBrexit die Marktteilnehmer,\ninsbesondere dahingehend, ob\nder mögliche Austritt des Ver-\neinigten Königreiches aus der\nEU geordnet oder - ohne ein\nÜbereinkommen - ungeordnet\nvollzogen wird. Im Gegensatz\nzu den politischen Unsicher-\nheiten standen die bislang eher\nzuversichtlichen, konventionel-\nlen Wirtschaftsindikatoren. So\nexpandierte die Weltwirtschaft\nkräftig, wenngleich sich deren\nWachstum im Laufe der zwei-\nten Jahreshälfte 2018 etwas\nverlangsamte. Die Geldpolitik\nwar historisch gesehen immer\nnoch sehr locker, trotz der welt-\nweit sehr hohen Verschuldung\nund der Zinserhöhungen der\nUS-Notenbank.\n\nBE Fed-Leitzins\nQuelle: Thomson Financial Datastream\nBE E28-Leitzins\nStand: 31.12.2018\n\nUSA weiter von ihren histori-\nvon 2,4% p.a. auf 3,1% p.a.\nschen Tiefs lösen. Gleichzeitig\nwurde die Zentralbankdivergenz\nDiese Entwicklung in den USA\n\nzwischen den USA und dem\nhatte auf den Euroraum jedoch\nEuroraum immer deutlicher. An-\nnur phasenweise und partiell,\ngesichts des Wirtschaftsbooms\ninsgesamt aber kaum einen\nin den USA hob die US-Noten-\nzinstreibenden Effekt auf Staats-\nbank Fed im Berichtszeitraum\nanleihen aus den europäischen\nden Leitzins in vier Schritten\nKernmärkten wie beispielsweise\nweiter um einen Prozentpunkt\nDeutschland und Frankreich.\nauf einen Korridor von 2,25% -\nSo gaben zehnjährige deutsche\n2,50% p.a. an. Die Europäische\nBundesanleihen im Jahresver-\nZentralbank (EZB) hingegen\nlauf 2018 unter Schwankungen\nIn den Monaten Mai und Juni\n\nEntwicklung der Leitzinsen in den USA und im Euroraum\n%p.a.\n-1 u\nu\nu\nu\nu\nu\nu\nu\nu\nu\nu\n12/08 12/09 12/10 12/11 12/12 12/13 12/14 12/15 12/16 12/17 12/18\nZinswende nach Rekordtiefs\n\nFiskalpolitik des US-Präsidenten\nbei Anleiherenditen?\nDonald Trump in Form von\nIm Berichtszeitraum kam es an\nSteuererleichterungen und einer\nden Anleihemärkten - wenn\nErhöhung der Staatsausgaben\nauch uneinheitlich und unter-\nnoch befeuert wurde. Vor die-\nschiedlich stark ausgeprägt -\nsem Hintergrund verzeichneten\nunter Schwankungen zu stei-\ndie US-Bondmärkte einen spür-\ngenden Renditen auf teilweise\nbaren Renditeanstieg, der mit\nimmer noch sehr niedrigem\nmerklichen Kursermäßigungen\nNiveau, begleitet von nachge-\neinherging. Per saldo stiegen\nbenden Kursen. Dabei konnten\ndie Renditen zehnjähriger US-\nsich die Zinsen vor allem in den\nStaatsanleihen auf Jahressicht\nhielt an ihrer Nullzinspolitik fest\nper saldo sogar von 0,42% p.a.\nund die Bank of Japan beließ\nauf 0,25% p. a. nach. Vielmehr\nihren Leitzins bei -0,10% p.a.\nstanden die Anleihemärkte\nDie Fed begründete ihre Zinser-\nder Euroländer - insbeson-\nhöhungen mit der Wachstums-\ndere ab dem zweiten Quartal\nbeschleunigung und der Voll-\n2018 - unter dem Einfluss der\nbeschäftigung am Arbeitsmarkt\npolitischen und wirtschaftlichen\nin den USA. Zinserhöhungen\nEntwicklung in der Eurozone,\nermöglichten der US-Notenbank\nvor allem in den Ländern mit\neiner Überhitzung der US-Wirt-\nhoher Verschuldung und nied-\nschaft vorzubeugen, die durch\nrigem Wirtschaftswachstum.\ndie prozyklische expansive',
     '')



Now this looks weird again, doesn't it? However the reason is still quite simple. We now get an empty text string because once we have a non-empty `dp.layouts` the routine responsible for creating `dp.text` will try to get the text from the `Layout`'s. But we haven't run any method that maps a `word` to some `Layout` object. We need to specify this by applying a `MatchingService`. We will also have to slightly change the configuration of the  `TextOrderService`.


```python
map_comp = dd.MatchingService(parent_categories=["text","title","list","table","figure"], child_categories=["word"],
                             matching_rule = 'ioa', threshold=0.6) # same setting as for the deepdoctection analyzer

order_comp = dd.TextOrderService(text_container=dd.LayoutType.WORD,
                                 floating_text_block_categories=["text","title","list", "figure"],
                                 text_block_categories=["text","title","list","table","figure"])

pipe_comp_list = [layout_comp, lang_detect_comp, text_comp, map_comp, order_comp]
pipe = dd.DoctectionPipe(pipeline_component_list=pipe_comp_list)
```


```python
df = pipe.analyze(path=image_path)
df.reset_state()
dp = next(iter(df))
dp.text
```

    |                                                                                                                                                                                                                                                                                                                                   |1/?[00:00<00:00,893.93it/s]
    [32m[0712 15:03.00 @doctectionpipe.py:84][0m  [32mINF[0m  [97mProcessing sample_3.png[0m
    [32m[0712 15:03.02 @context.py:133][0m  [32mINF[0m  [97mImageLayoutService total: 1.6847 sec.[0m
    [32m[0712 15:03.05 @context.py:133][0m  [32mINF[0m  [97mLanguageDetectionService total: 2.6838 sec.[0m
    [32m[0712 15:03.08 @context.py:133][0m  [32mINF[0m  [97mTextExtractionService total: 3.1799 sec.[0m
    [32m[0712 15:03.08 @context.py:133][0m  [32mINF[0m  [97mMatchingService total: 0.0041 sec.[0m
    [32m[0712 15:03.08 @context.py:133][0m  [32mINF[0m  [97mTextOrderService total: 0.0338 sec.[0m





    'Anleihemärkte im Geschäftsjahr bis zum 31.12.2018\nSchwieriges Marktumfeld Die internationalen Anleihe- märkte entwickelten sich im Geschäftsjahr 2018 unter- schiedlich und phasenweise sehr volatil. Dabei machte sich bei den Investoren zunehmend Nervosität breit, was in steigen- den Risikoprämien zum Aus- druck kam. Grund hierfür waren Turbulenzen auf der weltpoli- tischen Bühne, die die politi- schen Risiken erhöhten. Dazu zählten unter anderem populis- tische Strömungen nicht nur in den USA und Europa, auch in den Emerging Markets, wie zuletzt in Brasilien und Mexiko, wo Populisten in die Regie- rungen gewählt wurden. Der eskalierende Handelskonflikt zwischen den USA einerseits sowie Europa und China ande- rerseits tat sein übriges. Zudem ging Italien im Rahmen seiner Haushaltspolitik auf Konfronta- tionskurs zur Europäischen Uni- on (EU). Darüber hinaus verun- sicherte weiterhin der drohende Brexit die Marktteilnehmer, insbesondere dahingehend, ob der mögliche Austritt des Ver- einigten Königreiches aus der EU geordnet oder - ohne ein Übereinkommen - ungeordnet vollzogen wird. Im Gegensatz zu den politischen Unsicher- heiten standen die bislang eher zuversichtlichen, konventionel- len Wirtschaftsindikatoren. So expandierte die Weltwirtschaft kräftig, wenngleich sich deren Wachstum im Laufe der zwei- ten Jahreshälfte 2018 etwas verlangsamte. Die Geldpolitik war historisch gesehen immer noch sehr locker, trotz der welt- weit sehr hohen Verschuldung und der Zinserhöhungen der US-Notenbank.\nEntwicklung der Leitzinsen in den USA und im Euroraum %p.a.\nZinswende nach Rekordtiefs\nbei Anleiherenditen? Im Berichtszeitraum kam es an den Anleihemärkten - wenn auch uneinheitlich und unter- schiedlich stark ausgeprägt - unter Schwankungen zu stei- genden Renditen auf teilweise immer noch sehr niedrigem Niveau, begleitet von nachge- benden Kursen. Dabei konnten sich die Zinsen vor allem in den USA weiter von ihren histori- schen Tiefs lösen. Gleichzeitig wurde die Zentralbankdivergenz zwischen den USA und dem Euroraum immer deutlicher. An- gesichts des Wirtschaftsbooms in den USA hob die US-Noten- bank Fed im Berichtszeitraum den Leitzins in vier Schritten weiter um einen Prozentpunkt auf einen Korridor von 2,25% - 2,50% p.a. an. Die Europäische Zentralbank (EZB) hingegen hielt an ihrer Nullzinspolitik fest und die Bank of Japan beließ ihren Leitzins bei -0,10% p.a. Die Fed begründete ihre Zinser- höhungen mit der Wachstums- beschleunigung und der Voll- beschäftigung am Arbeitsmarkt in den USA. Zinserhöhungen ermöglichten der US-Notenbank einer Überhitzung der US-Wirt- schaft vorzubeugen, die durch die prozyklische expansive\n-1 u u u u u u u u u u u 12/08 12/09 12/10 12/11 12/12 12/13 12/14 12/15 12/16 12/17 12/18\nBE Fed-Leitzins\nQuelle: Thomson Financial Datastream\nBE E28-Leitzins\nStand: 31.12.2018\nFiskalpolitik des US-Präsidenten Donald Trump in Form von Steuererleichterungen und einer Erhöhung der Staatsausgaben noch befeuert wurde. Vor die- sem Hintergrund verzeichneten die US-Bondmärkte einen spür- baren Renditeanstieg, der mit merklichen Kursermäßigungen einherging. Per saldo stiegen die Renditen zehnjähriger US- Staatsanleihen auf Jahressicht von 2,4% p.a. auf 3,1% p.a.\nDiese Entwicklung in den USA hatte auf den Euroraum jedoch nur phasenweise und partiell, insgesamt aber kaum einen zinstreibenden Effekt auf Staats- anleihen aus den europäischen Kernmärkten wie beispielsweise Deutschland und Frankreich. So gaben zehnjährige deutsche Bundesanleihen im Jahresver- lauf 2018 unter Schwankungen per saldo sogar von 0,42% p.a. auf 0,25% p. a. nach. Vielmehr standen die Anleihemärkte der Euroländer - insbeson- dere ab dem zweiten Quartal 2018 - unter dem Einfluss der politischen und wirtschaftlichen Entwicklung in der Eurozone, vor allem in den Ländern mit hoher Verschuldung und nied- rigem Wirtschaftswachstum. In den Monaten Mai und Juni'



Finally, we got it!
