/*
 * DBeaver - Universal Database Manager
 * Copyright (C) 2010-2024 DBeaver Corp and others
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.cloudbeaver.model;

import org.jkiss.code.NotNull;
import org.jkiss.dbeaver.model.net.DBWHandlerConfiguration;
import org.jkiss.dbeaver.model.net.DBWHandlerType;
import org.jkiss.dbeaver.model.net.ssh.SSHConstants;
import org.jkiss.dbeaver.model.preferences.DBPPropertyDescriptor;
import org.jkiss.dbeaver.registry.RegistryConstants;
import org.jkiss.utils.CommonUtils;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Web network handler info
 */
public class WebNetworkHandlerConfig {

    private final DBWHandlerConfiguration configuration;

    public WebNetworkHandlerConfig(DBWHandlerConfiguration configuration) {
        this.configuration = configuration;
    }

    public DBWHandlerType getType() {
        return configuration.getType();
    }

    public boolean isSecured() {
        return configuration.isSecured();
    }

    @NotNull
    public String getId() {
        return configuration.getId();
    }

    public boolean isEnabled() {
        return configuration.isEnabled();
    }

    @Deprecated
    public SSHConstants.AuthType getAuthType() {
        return CommonUtils.valueOf(SSHConstants.AuthType.class, configuration.getStringProperty(SSHConstants.PROP_AUTH_TYPE), SSHConstants.AuthType.PASSWORD);
    }

    public String getUserName() {
        return configuration.getUserName();
    }

    public String getPassword() {
        return CommonUtils.isEmpty(configuration.getPassword()) ? null : "";
    }

    @Deprecated // use secure properties
    public String getKey() {
        return CommonUtils.isEmpty(configuration.getSecureProperty(SSHConstants.PROP_KEY_VALUE)) ? null : "";
    }

    public boolean isSavePassword() {
        return configuration.isSavePassword();
    }

    @NotNull
    public Map<String, Object> getProperties() {
        return configuration.getProperties();
    }

    @NotNull
    public Map<String, String> getSecureProperties() {
        Map<String, String> secureProperties = new LinkedHashMap<>(configuration.getSecureProperties());
        DBPPropertyDescriptor[] descriptor = configuration.getHandlerDescriptor().getHandlerProperties();
        for (DBPPropertyDescriptor p : descriptor) {
            if (p.hasFeature(RegistryConstants.ATTR_PASSWORD)) {
                secureProperties.computeIfPresent(p.getId(), (k, v) -> CommonUtils.isEmpty(v) ? null : WebConnectionInfo.SECURED_VALUE);
            }
        }
        return secureProperties;
    }

}
