/*
 * Copyright (c) 2021 Cyrille Rossant and contributors. All rights reserved.
 * Licensed under the MIT license. See LICENSE file in the project root for details.
 * SPDX-License-Identifier: MIT
 */

/*************************************************************************************************/
/*  Monoglyph                                                                                    */
/*************************************************************************************************/



/*************************************************************************************************/
/*  Includes                                                                                     */
/*************************************************************************************************/

#include "scene/visuals/monoglyph.h"
#include "datoviz.h"
#include "datoviz_types.h"
#include "fileio.h"
#include "request.h"
#include "scene/graphics.h"
#include "scene/viewset.h"
#include "scene/visual.h"



/*************************************************************************************************/
/*  Constants                                                                                    */
/*************************************************************************************************/

#define MAX_TEXT_LENGTH 65536



/*************************************************************************************************/
/*  Glyphs                                                                                       */
/*************************************************************************************************/

// see https://github.com/glumpy/glumpy/blob/master/glumpy/app/console.py#L12
// 96 * 6 = 576;
// each group of 6 bytes represents 1 glyph = 6 integers 0-255, which will be cast to floats.

const uint8_t __FONT_6x8__[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0xE3, 0x84, 0x10, 0x01, 0x00, //
    0x6D, 0xB4, 0x80, 0x00, 0x00, 0x00, 0x00, 0xA7, 0xCA, 0x29, 0xF2, 0x80, //
    0x20, 0xE4, 0x0C, 0x09, 0xC1, 0x00, 0x65, 0x90, 0x84, 0x21, 0x34, 0xC0, //
    0x21, 0x45, 0x08, 0x55, 0x23, 0x40, 0x30, 0xC2, 0x00, 0x00, 0x00, 0x00, //
    0x10, 0x82, 0x08, 0x20, 0x81, 0x00, 0x20, 0x41, 0x04, 0x10, 0x42, 0x00, //
    0x00, 0xA3, 0x9F, 0x38, 0xA0, 0x00, 0x00, 0x41, 0x1F, 0x10, 0x40, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0xC3, 0x08, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, //
    0x00, 0x00, 0x00, 0x00, 0xC3, 0x00, 0x00, 0x10, 0x84, 0x21, 0x00, 0x00, //
    0x39, 0x14, 0xD5, 0x65, 0x13, 0x80, 0x10, 0xC1, 0x04, 0x10, 0x43, 0x80, //
    0x39, 0x10, 0x46, 0x21, 0x07, 0xC0, 0x39, 0x10, 0x4E, 0x05, 0x13, 0x80, //
    0x08, 0x62, 0x92, 0x7C, 0x20, 0x80, 0x7D, 0x04, 0x1E, 0x05, 0x13, 0x80, //
    0x18, 0x84, 0x1E, 0x45, 0x13, 0x80, 0x7C, 0x10, 0x84, 0x20, 0x82, 0x00, //
    0x39, 0x14, 0x4E, 0x45, 0x13, 0x80, 0x39, 0x14, 0x4F, 0x04, 0x23, 0x00, //
    0x00, 0x03, 0x0C, 0x00, 0xC3, 0x00, 0x00, 0x03, 0x0C, 0x00, 0xC3, 0x08, //
    0x08, 0x42, 0x10, 0x20, 0x40, 0x80, 0x00, 0x07, 0xC0, 0x01, 0xF0, 0x00, //
    0x20, 0x40, 0x81, 0x08, 0x42, 0x00, 0x39, 0x10, 0x46, 0x10, 0x01, 0x00, //
    0x39, 0x15, 0xD5, 0x5D, 0x03, 0x80, 0x39, 0x14, 0x51, 0x7D, 0x14, 0x40, //
    0x79, 0x14, 0x5E, 0x45, 0x17, 0x80, 0x39, 0x14, 0x10, 0x41, 0x13, 0x80, //
    0x79, 0x14, 0x51, 0x45, 0x17, 0x80, 0x7D, 0x04, 0x1E, 0x41, 0x07, 0xC0, //
    0x7D, 0x04, 0x1E, 0x41, 0x04, 0x00, 0x39, 0x14, 0x17, 0x45, 0x13, 0xC0, //
    0x45, 0x14, 0x5F, 0x45, 0x14, 0x40, 0x38, 0x41, 0x04, 0x10, 0x43, 0x80, //
    0x04, 0x10, 0x41, 0x45, 0x13, 0x80, 0x45, 0x25, 0x18, 0x51, 0x24, 0x40, //
    0x41, 0x04, 0x10, 0x41, 0x07, 0xC0, 0x45, 0xB5, 0x51, 0x45, 0x14, 0x40, //
    0x45, 0x95, 0x53, 0x45, 0x14, 0x40, 0x39, 0x14, 0x51, 0x45, 0x13, 0x80, //
    0x79, 0x14, 0x5E, 0x41, 0x04, 0x00, 0x39, 0x14, 0x51, 0x55, 0x23, 0x40, //
    0x79, 0x14, 0x5E, 0x49, 0x14, 0x40, 0x39, 0x14, 0x0E, 0x05, 0x13, 0x80, //
    0x7C, 0x41, 0x04, 0x10, 0x41, 0x00, 0x45, 0x14, 0x51, 0x45, 0x13, 0x80, //
    0x45, 0x14, 0x51, 0x44, 0xA1, 0x00, 0x45, 0x15, 0x55, 0x55, 0x52, 0x80, //
    0x45, 0x12, 0x84, 0x29, 0x14, 0x40, 0x45, 0x14, 0x4A, 0x10, 0x41, 0x00, //
    0x78, 0x21, 0x08, 0x41, 0x07, 0x80, 0x38, 0x82, 0x08, 0x20, 0x83, 0x80, //
    0x01, 0x02, 0x04, 0x08, 0x10, 0x00, 0x38, 0x20, 0x82, 0x08, 0x23, 0x80, //
    0x10, 0xA4, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, //
    0x30, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x81, 0x3D, 0x13, 0xC0, //
    0x41, 0x07, 0x91, 0x45, 0x17, 0x80, 0x00, 0x03, 0x91, 0x41, 0x13, 0x80, //
    0x04, 0x13, 0xD1, 0x45, 0x13, 0xC0, 0x00, 0x03, 0x91, 0x79, 0x03, 0x80, //
    0x18, 0x82, 0x1E, 0x20, 0x82, 0x00, 0x00, 0x03, 0xD1, 0x44, 0xF0, 0x4E, //
    0x41, 0x07, 0x12, 0x49, 0x24, 0x80, 0x10, 0x01, 0x04, 0x10, 0x41, 0x80, //
    0x08, 0x01, 0x82, 0x08, 0x24, 0x8C, 0x41, 0x04, 0x94, 0x61, 0x44, 0x80, //
    0x10, 0x41, 0x04, 0x10, 0x41, 0x80, 0x00, 0x06, 0x95, 0x55, 0x14, 0x40, //
    0x00, 0x07, 0x12, 0x49, 0x24, 0x80, 0x00, 0x03, 0x91, 0x45, 0x13, 0x80, //
    0x00, 0x07, 0x91, 0x45, 0x17, 0x90, 0x00, 0x03, 0xD1, 0x45, 0x13, 0xC1, //
    0x00, 0x05, 0x89, 0x20, 0x87, 0x00, 0x00, 0x03, 0x90, 0x38, 0x13, 0x80, //
    0x00, 0x87, 0x88, 0x20, 0xA1, 0x00, 0x00, 0x04, 0x92, 0x49, 0x62, 0x80, //
    0x00, 0x04, 0x51, 0x44, 0xA1, 0x00, 0x00, 0x04, 0x51, 0x55, 0xF2, 0x80, //
    0x00, 0x04, 0x92, 0x31, 0x24, 0x80, 0x00, 0x04, 0x92, 0x48, 0xE1, 0x18, //
    0x00, 0x07, 0x82, 0x31, 0x07, 0x80, 0x18, 0x82, 0x18, 0x20, 0x81, 0x80, //
    0x10, 0x41, 0x00, 0x10, 0x41, 0x00, 0x30, 0x20, 0x83, 0x08, 0x23, 0x00, //
    0x29, 0x40, 0x00, 0x00, 0x00, 0x00, 0x10, 0xE6, 0xD1, 0x45, 0xF0, 0x00, //
};



/*************************************************************************************************/
/*  Functions                                                                                    */
/*************************************************************************************************/

DvzVisual* dvz_monoglyph(DvzBatch* batch, int flags)
{
    ANN(batch);

    DvzVisual* visual = dvz_visual(batch, DVZ_PRIMITIVE_TOPOLOGY_POINT_LIST, flags);
    ANN(visual);

    // Visual shaders.
    dvz_visual_shader(visual, "graphics_monoglyph");

    // Vertex attributes.
    dvz_visual_attr(visual, 0, FIELD(DvzMonoglyphVertex, pos), DVZ_FORMAT_R32G32B32_SFLOAT, 0);
    dvz_visual_attr(
        visual, 1, FIELD(DvzMonoglyphVertex, bytes_012), DVZ_FORMAT_R32G32B32_SFLOAT, 0);
    dvz_visual_attr(
        visual, 2, FIELD(DvzMonoglyphVertex, bytes_345), DVZ_FORMAT_R32G32B32_SFLOAT, 0);
    dvz_visual_attr(visual, 3, FIELD(DvzMonoglyphVertex, offset), DVZ_FORMAT_R32G32_SINT, 0);
    dvz_visual_attr(visual, 4, FIELD(DvzMonoglyphVertex, color), DVZ_FORMAT_R8G8B8A8_UNORM, 0);

    // Vertex stride.
    dvz_visual_stride(visual, 0, sizeof(DvzMonoglyphVertex));

    // Slots.
    dvz_visual_slot(visual, 0, DVZ_SLOT_DAT);
    dvz_visual_slot(visual, 1, DVZ_SLOT_DAT);
    dvz_visual_slot(visual, 2, DVZ_SLOT_DAT);

    // Params.
    DvzParams* params = dvz_visual_params(visual, 2, sizeof(DvzMonoglyphParams));
    dvz_params_attr(params, 0, FIELD(DvzMonoglyphParams, anchor));
    dvz_params_attr(params, 1, FIELD(DvzMonoglyphParams, size));

    return visual;
}



void dvz_monoglyph_alloc(DvzVisual* visual, uint32_t item_count)
{
    ANN(visual);
    log_debug("allocating the visual visual");

    DvzBatch* batch = visual->batch;
    ANN(batch);

    // Create the visual.
    dvz_visual_alloc(visual, item_count, item_count, 0);
}



void dvz_monoglyph_position(
    DvzVisual* visual, uint32_t first, uint32_t count, vec3* values, int flags)
{
    ANN(visual);
    dvz_visual_data(visual, 0, first, count, (void*)values);
}



void dvz_monoglyph_offset(
    DvzVisual* visual, uint32_t first, uint32_t count, ivec2* values, int flags)
{
    ANN(visual);
    dvz_visual_data(visual, 3, first, count, (void*)values);
}



void dvz_monoglyph_color(
    DvzVisual* visual, uint32_t first, uint32_t count, cvec4* values, int flags)
{
    ANN(visual);
    dvz_visual_data(visual, 4, first, count, (void*)values);
}



void dvz_monoglyph_glyph(DvzVisual* visual, uint32_t first, const char* text, int flags)
{
    ANN(visual);
    ANN(text);

    uint32_t count = strnlen(text, MAX_TEXT_LENGTH);
    ASSERT(count > 0);

    vec3* bytes_012 = (vec3*)calloc(count, sizeof(vec3));
    vec3* bytes_345 = (vec3*)calloc(count, sizeof(vec3));

    int32_t glyph = 0;
    for (uint32_t i = 0; i < count; i++)
    {
        glyph = (int32_t)text[i] - 32;
        ASSERT(glyph >= 0);
        ASSERT(glyph < 96);

        bytes_012[i][0] = (float)__FONT_6x8__[6 * glyph + 0];
        bytes_012[i][1] = (float)__FONT_6x8__[6 * glyph + 1];
        bytes_012[i][2] = (float)__FONT_6x8__[6 * glyph + 2];

        bytes_345[i][0] = (float)__FONT_6x8__[6 * glyph + 3];
        bytes_345[i][1] = (float)__FONT_6x8__[6 * glyph + 4];
        bytes_345[i][2] = (float)__FONT_6x8__[6 * glyph + 5];
    }

    dvz_visual_data(visual, 1, first, count, (void*)bytes_012);
    dvz_visual_data(visual, 2, first, count, (void*)bytes_345);

    FREE(bytes_012);
    FREE(bytes_345);
}



void dvz_monoglyph_anchor(DvzVisual* visual, vec2 anchor)
{
    ANN(visual);
    dvz_visual_param(visual, 2, 0, anchor);
}



void dvz_monoglyph_size(DvzVisual* visual, float size)
{
    ANN(visual);
    dvz_visual_param(visual, 2, 1, &size);
}



void dvz_monoglyph_textarea(
    DvzVisual* visual, vec3 position, cvec4 color, float size, const char* text)
{
    ANN(visual);

    // NOTE: count may contain an arbitrary number of newlines \n that we should subtract.
    uint32_t count = strnlen(text, MAX_TEXT_LENGTH);
    char* glyphs = (char*)calloc(count, sizeof(char));
    ivec2* offsets = (ivec2*)calloc(count, sizeof(ivec2));

    uint32_t new_lines = 0, k = 0, column = 0, row = 0;
    for (uint32_t i = 0; i < count; i++)
    {
        if (text[i] == '\n')
        {
            new_lines++;
            row++;
            column = 0;
        }
        else
        {
            glyphs[k] = text[i];
            offsets[k][0] = (int32_t)row;
            offsets[k][1] = (int32_t)column;
            column++;
            k++;
        }
    }

    ASSERT(new_lines < count);
    count = (uint32_t)(count - new_lines);

    dvz_monoglyph_alloc(visual, count);

    vec3* positions = dvz_mock_fixed(count, position);
    dvz_monoglyph_position(visual, 0, count, positions, 0);

    dvz_monoglyph_offset(visual, 0, count, offsets, 0);

    cvec4* colors = dvz_mock_monochrome(count, color);
    dvz_monoglyph_color(visual, 0, count, colors, 0);

    dvz_monoglyph_glyph(visual, 0, (const char*)glyphs, 0);

    dvz_monoglyph_size(visual, size);

    FREE(positions);
    FREE(offsets);
    FREE(colors);
    FREE(glyphs);
}
