# Colormaps

Datoviz natively includes a collection of common colormaps (continuous) and color palettes (discrete).
They come from:

* matplotlib
* bokeh
* colorcet
* [Kenneth Moreland's page](https://www.kennethmoreland.com/color-advice/)

| Name | Row, Col | Colormap |
| ---- | ---- | ---- |
| `binary` | 0, 0 | ![BINARY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAq0lEQVR4nO3WMQ7CMBQFwQ/3P7NNQyRERwQS0c40Vgr7pdzb3nvPzDwP55fOtda8sm//yvtn3zvuHd9n/8f+f+x/+s77vbP/Yf83+/cBAHIEAAAECQAACBIAABAkAAAgSAAAQJAAAIAgAQAAQQIAAIIEAAAECQAACBIAABAkAAAgSAAAQJAAAIAgAQAAQQIAAIIEAAAECQAACBIAABAkAAAgSAAAQJAAAICgB0NNvE2VurQFAAAAAElFTkSuQmCC) |
| `hsv` | 1, 0 | ![HSV](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA6UlEQVR4nO3WvW7CQBCF0cEY3v95g1lT7DaREiUSP1rrntOMKDxC9hTfaa/aq6rqUt31RfPV+ybf28bzbe1zH7Od+9zG/Krn5u3J5z+1913/s1V/kXut4/f3Oc9hTXqof+27j/c46wEc5FBP9z6X7ec56dc/zN61WlVVLbX9Mv/7YY9yqO/ZuxQAEEcAAEAgAQAAgQQAAAQSAAAQSAAAQCABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAQSAAAQSAAAQCABAACBBAAABBIAABBIAABAIAEAAIEeICeJRQb6LxwAAAAASUVORK5CYII=) |
| `cividis` | 2, 0 | ![CIVIDIS](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABZElEQVR4nO3WQXLCMAwFULn0JL1P738b0kWTMLYxCp3u9N4mxLJsJzPAb/H1vUVERPuIXy26+/O6j8c4nvS9O3/qO/bt2w6frb/e9vHbcd+27v5cvQ3X2Lrtxnob+tq+7vk0i/mP8X79sd6G843jq/ltce6875/G9w+reqz60vF2qS+yvuW5j3o/cd33et1I3sOxz/R+sr6pPpx39dzZ8yXnyt/b8/H5XO/1z/Utqf+xf7lO9j3NzpPtt6qv9k3mp+e7uu7ifp8//X4l8x/1i31J//i7u95/rL/e72OaN54nqV9d57zeF/3DeKzm35/PP+/7euz1xwvs/1eP/xsAoBABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoKAfTKVFSlIEa9AAAAAASUVORK5CYII=) |
| `inferno` | 3, 0 | ![INFERNO](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABjElEQVR4nO3Wy3LqMAwAUNmh///H+C4uDmOlLmm3OmeT0cOPwMCoRRwjIiKi/X9Ef0WvuPWUP5Z8m/nc19b+M96sm339Uj+WuKd4ru+R+nKc90nv08eaf9dTfnyqt1SfcX7u6rHm21o/v6WW+lM86+/+vG+k9X/t+zm+3Rfj1v7ve4zNvca9/Tf7nOde9s99+Zzxu3Xbvvw55PxI++T3S/XY9F/uscbXcz7E2/ulfJ/x89Z5v13f+mZ9f36//6X/+fqAPvT154/1mPXNusjr+vwbXtdFft+c3+yXz4+2WXf2R7r/mj/rKb7+AGd+xv1mnJ5tk0/P0Y9N/XjVP+WPlF/3G9t6itsj1R+pbxO3R+rP+a8l39pabynfNvl+1o8lnl8XAFCIAQAACjIAAEBBBgAAKMgAAAAFGQAAoCADAAAUZAAAgIIMAABQkAEAAAoyAABAQQYAACjIAAAABRkAAKAgAwAAFGQAAICCDAAAUJABAAAKMgAAQEEGAAAoyAAAAAUZAACgIAMAABT0D7GHbM280iuxAAAAAElFTkSuQmCC) |
| `magma` | 4, 0 | ![MAGMA](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABg0lEQVR4nO3WQY7DIAwAQEP//2b2sKEVTki6e/XMpbKxgbRR5RbxGhEREe33I/oRHXHra34TtyOOFOf1d/zOv67z277XZb7H/fo8p+dzY/ane+T8Nt7kR7rXPP9d//v9tpH3Sespzp9tl2+5Lo7nz3VxG/f5Grz7Yq2b67k+1vV5bs5/zsv7Xtfvz7+Pv6177hvf3esUj/tzjvXc/6nf9ef8Gn/uMdK++Tmu4/O+6z7tqW6T//x+Y90nfw95/RSv9zjX5Xse+T5SXyz5U3+/juOUT/dZ/xa39XGqi6VuPui+vqX89YvW8ouRXvSn9XmhlvOPn+mBTuf1+/6W13f1/Y99u/o1P/p1flcfX9f3tP993f/37Zd182cGAAoxAABAQQYAACjIAAAABRkAAKAgAwAAFGQAAICCDAAAUJABAAAKMgAAQEEGAAAoyAAAAAUZAACgIAMAABRkAACAggwAAFCQAQAACjIAAEBBBgAAKMgAAAAFGQAAoCADAAAUZAAAgIJ+ALgzLuV4sjd4AAAAAElFTkSuQmCC) |
| `plasma` | 5, 0 | ![PLASMA](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABcUlEQVR4nO3WQW7DIBAF0IFW6qbn7rFDF40beTCOmy7nvY0FzAC2lOi3z4+vERHxHi0iIt5Gj4iIvo3Tc5vvI42vPkecrrff8XFdW/S3Vf02P/b7tun8ND9yXSz67+NxsS7ds031+/HVumncFuttG480zuvbeOz2e+y7n891vW/1q7p0/pO61hfzz/ryedO98n3O+1vP3+OvfS8+p/7z9/j/+XHaH5fPjRf7fn+Ap3Vx8fus6ub7RDr/6rlxWB+L7577p/dcnP+sL58Xy/nj91z1z3Vxvr64ZyzOe9TH6Xo83W/RP80f143pDzif9/MYy/k4nl/VTevtcJ/R22H9tM/Ul+97u49vu9cEAAoRAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAr6BjS/nzfy6Uv9AAAAAElFTkSuQmCC) |
| `viridis` | 6, 0 | ![VIRIDIS](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABcUlEQVR4nO3WS67bMAwAQMq+Wo/Q+x+lfosqNkRZcRp0x5mNwY8oIYuA7Vf7fURERGvx97v1zxhHj9urb+v5FF/1+/P5nmnuav409/57PNRX54+zHl/WY5h/9re470vzcv8c5/mruZHOpTnbQ306//6e5/jT+ffxt/f87/mPffG+7/P88d35eOo/vnpHrl99i/r5jtX59/m26GtTnM5F7kt/a8v4Id/nblP/0fOR4ly/n7PqO7/xUE99W/vzYd/Yv/9zfbxnj/t7c35/nYuH+HVvum+O799xzR3ff/WP8+b+dN8qn96xeu8e+XcZf899ykfPxxif+ZbyPe75M+4dV//Wv72vbcMcAKAQCwAAFGQBAICCLAAAUJAFAAAKsgAAQEEWAAAoyAIAAAVZAACgIAsAABRkAQCAgiwAAFCQBQAACrIAAEBBFgAAKMgCAAAFWQAAoCALAAAUZAEAgIIsAABQkAUAAAqyAABAQRYAACjoB+XIiTft13keAAAAAElFTkSuQmCC) |
| `blues` | 7, 0 | ![BLUES](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABKUlEQVR4nO3WQW6DMBAFULf3v28XlUwWIVQxjExE2kX/exvLjBljQqT/8fW9LK21ttyHtg5tHbbrbZsf17dVF+vDdmX9p9+1+tnz/ff6+H56G7+H5bV6cf0xP18/fs6+e+6xXp1v7T/Ot/XH5+vbWJxz6DPtP7k+Psd0XXl/8Xzj+ynu3/0eY313jsl3Mtuv6lec42y/3XczOc98PjlXf9c+r+17ve/zH6+qn133rj6/tV9b+uOG53kf5tW6t8//ap/7/LMBAHEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAS6AbQUF/CjYfi3AAAAAElFTkSuQmCC) |
| `bugn` | 8, 0 | ![BUGN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABYklEQVR4nO3WwXaCMBAF0Gn//3+7aRMXitQJMcJ27l2IJJMhgJ7zvn5+/3pERO89IiLun7uevozz/cO64/meKs/3ubrvx7Hn6/fD+WGf6Xn1f9/u85N+zw0dj/e08XE+3+/sevm9rPrdj61vxzgcH+uOx7e+bVgfn9Wl8XVdvJ4v1i/rnv3buf2l+qHf8Pza++e3uM7n7+P9ffTHE5z176nPPp7v97VP7pvH27Rucp10f7O6/f2s+uZ1qb5N1k/OYzG/9zvXd3reVvtY9Zn97uNtXT9Zl/+fV/tsdcPvd9VneN+xNXoseG7w9djy+GrdYn7Wb7+xi+uu7fM7AIByBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCbqyNNKyrKFdFAAAAAElFTkSuQmCC) |
| `bupu` | 9, 0 | ![BUPU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABSElEQVR4nO3W0U7EIBAFUPT//9cXhX3YdiPDEthqosmc89IA04FWu7lvH59frZRSWmvlfi33azmc8/1wuX7OxPrV+tX9WukL988z6Rffx1C3Wr/ap3+e3T511ue3zj27fzjnq313nz885/Q8156zxHF9fv9Qd/atfeNZfauzPrv79/OxLu4/1k0+kHD+fzdfZ+f+2Xx836++l/h3+fZB/s3+Q308T9g/zofxY32oq8/rYp/4f7e736TPsm557n5cjw+3Pn5vZuPj+vgd6cfnegvjoX6yz3mO3b7LfVb3H/u+FwAgHQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgoRsq6gTl9AUZKwAAAABJRU5ErkJggg==) |
| `gnbu` | 10, 0 | ![GNBU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABWUlEQVR4nO3Wu27EIBAF0HH+/2PTpYkgRWyQiVnW2k015zTWYAPjl3S3r+/PGhFRa42IiBrHcTcZP+p+/rnxfv7e+LKf2ne61U+bN1zfzk/GF/vGsH6N633ev//j5zXb/+9zua7L4vzsvRx1Gfdb1W/av0z3i1Nd6vawrnvd1zv62s71ct51/XIft+ftdXsOk/tY3df5N5v3UYd503WGfRd9HX1P+5is04/7d1HOdftej+vK8N+M8yfr9H7qsM6z86+vL29ap/W3vL9JvZhXVutM72PS5+33cl1HaQ9oOJZjo+vx3sj1+Mvz2wv5p/5+jx8BAKQjAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACT0A5Sd9wcviAPsAAAAAElFTkSuQmCC) |
| `greens` | 11, 0 | ![GREENS](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABOUlEQVR4nO3WwVKDMBAG4NX3f13Hi5N40AYIjQFk9LDfd2lpNsuGtjP/y/vHW42IqLVGRESNr9eHdt3Wl5X1evv8ZJ+6evcfffbnfv75qP5w3cn9/dx/dv/uurTr8mN92e2/p75M5y2DfqPzlGPnbXN2cw3m759T3+e+/t25p/239fP+sbne9bu77nu933e6rp1/Vredp3/utdt/dj2m+++5z+/neL5eDt//XF3//V3t86g7/f31//flh7B5XqsG0Q2+rRvum6yP+i0Hu7jv2pyvAQCkIwAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAk9Am3n06sl1PtjAAAAABJRU5ErkJggg==) |
| `greys` | 12, 0 | ![GREYS](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAxklEQVR4nO3WQQ7CIBRF0a/7XzM4sQ5MjEYKbfPOmZBCoSSd3FvvvVdVPYeX1c9HfffT87f50fXV+/Za/3UcPefo/a21ofWrnrP3vlXnzT73qvf4957v87PfO8s9Zv3fs72/jfcCAOIIAAAIJAAAIJAAAIBAAgAAAgkAAAgkAAAgkAAAgEACAAACCQAACCQAACCQAACAQAIAAAIJAAAIJAAAIJAAAIBAAgAAAgkAAAgkAAAgkAAAgEACAAACCQAACCQAACDQAy1jp00Azd72AAAAAElFTkSuQmCC) |
| `oranges` | 13, 0 | ![ORANGES](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABLElEQVR4nO3Wy26DMBAFUNP//9Ouu7W7SEIUg2PSBHVxz9kgPxiPkZDu0n6+WymltNbKxfX5sXE5tv+supPx8N6tdvN1UO/g/Kbef5/Tvz+oc3h9MF9Pqvv2ubP5yfc9azzs+5y+2qDuOl9v/8l+H21Qv3X32Z4zqVsH9Y72X1/r8/5b7c/f3y/P12u//sc6g/ltn/vnzetf230clu6z3a4z3teND+9br7Ec3Pd8POp7fo/lYdx9nmG9V9fX8z5V5831Vi73/ioAQBwBAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIF+AbsO3OivfNX6AAAAAElFTkSuQmCC) |
| `orrd` | 14, 0 | ![ORRD](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPElEQVR4nO3WwU7DMAwG4MD7Pyp3bjEHKFPdWU3VISH5+y5T3MTJsmr63+LzI8YYI2KOb/HzEYvjeXF+sf7yum1879yx+r2u1nO/6pzl+W/uf3quqj729UO/F8073EMx//a87fnzc0SeN9O9zHRf82R+2edi39X9bvaJdJ8xn9/v43VK43yfs1j3u+82TPum59X63H/1fMd1++8b6Z4e9W2cnqd6zHwP+76R+o6iHvn3WT5Hel7UI703uf+s+p6M58X5f79+LK3fXov0L1yOX1X/L+vfBwDQjgAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQ0Bc7N+69+jNnaAAAAABJRU5ErkJggg==) |
| `pubu` | 15, 0 | ![PUBU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPElEQVR4nO3WQU7EMAwF0MD978sCKWHBpFXdcSeIIhZ+b1PiBsfNYvTfxsfnaK21MUb7fraDEQpzOUJhX89H1u+4cb3f9Tx5v+M8q/1GmCOvX9/b/n/HObJ+7bfnp+esnT/rPasnfZfrLfRfrM85e/Ldy/XtnKzv8+8/r6/r2bw9mWv1/J7d14vz473mc4V1fH/qu7pv7X38/vW5FveF96fvT+a4e52df/f61Txj/6PNGwobHvX4Q7L4fqv3H+7/437b79Ls83j22D/ex7318U/nzvV7AwDKEQAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAKEgAAoCABAAAK+gI/BfvQjJSq/AAAAABJRU5ErkJggg==) |
| `pubugn` | 16, 0 | ![PUBUGN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABS0lEQVR4nO3Wy07EMAwF0MD//y8LpJgFfajuhAQYxMLnbKo4qZN2qtF9ibf3aK21iGif17ZJ49gvX9ePy6Se+//9vscGad9BvV3n4zYfD9e31fWj51rcZ3Su23vN6xf79FF98BzL9Zb6L9b3c/bB8yzXj30e9+3p+c/66rrJ/elc/31/pPr+vd3ez+R9zdb9eH7wHeXvZPn8ab5HT/V93C/79G0c23wfrT/O2S/3jfY5+8bS+kjzw/0G9bzPPn/+gaRxHL/QdTxd/6w+s/t/2bcP7rv9kQ3qw/WTft/t++x+2/W1AQDlCAAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFfQB8L/e7ll55WAAAAABJRU5ErkJggg==) |
| `purples` | 17, 0 | ![PURPLES](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPUlEQVR4nO3WS27DMAwFQLX3v3Fbq6vGCW3GH6CrN7MTE1OUDAPv4/vrZ44xxpxzPFuXs6xfymOO7rnXQmn3tC7Pd/1L3818Z/tv+uz/Xp9v67Vve95yj119M8frxm29nqer1/71fN383f+urse95w/nuNm3X+/XH3Ms7/93ub5ce/7wPparc3XrUh/NfVztc/v97dc332nzfR7dV/d+H/3Le+rqa7+y39n60pxvM9f+vH/zrPs05zysd33O3UN/n9fq23s4u//Nueq+zfd02P/f+5y9n/315wAA4ggAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACPQLIcYaCatV3uIAAAAASUVORK5CYII=) |
| `rdpu` | 18, 0 | ![RDPU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABU0lEQVR4nO3WMXLDIBAFUJL73zZFOtZFJGW0GEuM3e17jQxaFlDh+V/x+xOttRYRrf39OD93w/vjxdr80Tfaed+1dcP81Xkn6+Ju3+n3uVk3vd+7+1zd9/Vz+P65X5/M5/p+0WdSn4871PfJ/O261/eJPN/jeX2+3z7uue92rzz/Zt//cy6uv9k39vFwj/Ox93tF/o49vZ/MD/V395/Uf2r/ftxrG6f7Hu+P+sn6oV/b+uX+5/F1XbvYZ3Lu3C99n77/D7d9fH7G9macX62Lyfxa3ep+8/XbOHLdbN1V3fN1+Zzj+Lz+bt2n9v1uAEA5AgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBDwy/v/HibkIUAAAAAElFTkSuQmCC) |
| `reds` | 19, 0 | ![REDS](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABO0lEQVR4nO3WQZLCIBAFUGbuf9jZuKVdjEkqJAS0tFz0exslDTRJSuv/xO0vSiklIkr5/1IeX/bjj18vX+l7uO+o1+Paq3eu1+t948n58+cZjEd9evVRv1692+/Vde855/H999ZP1l9d96X6ev91Py8G42X++rNq5x/mzfXpzR+vL3PrO9e3cuzb1PN6f7xs35lXr/ep033O92sf5+w+j23Wx7j9Lcfueh3U1/sv1/Vo6u1rbOvHfc/rx3M2/ZvP2tx/bfq8q96ea5sXnXWfrS/P7bcAAOkIAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAndATpu+rHSLZ//AAAAAElFTkSuQmCC) |
| `ylgn` | 20, 0 | ![YLGN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABTUlEQVR4nO3WQXLDIAwFUNr7X7PXQF3E9jQyGnucRRd6b+NCEIg0yfyviJ8YY4yIOV5ie85tFMv5fRwj123P+LC+Oj+e1q/XxdHnSPORVhfzx/hqfq+/2O+459Pzi3Nunj/zfrGen8U+5Xzqdxb3mLnPqr5ct77ncd5+n5v95fuc5tO5t+evzv2vfor3Mb9+Xfd+Tll/+v8U+5b3uFefP995/fGc68/V+fX8HOPvH1fr6tdv9nGxru6j2u/9/UnbLH538j7VOO37sP78O/Ww/vj+pEZmGp8arcbF/HxYV43L/j7r+3sAAO0IAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA0JAADQkAAAAA39AqjTJLufDnupAAAAAElFTkSuQmCC) |
| `ylgnbu` | 21, 0 | ![YLGNBU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABZElEQVR4nO3WQXKDMAwFUNH7X7CHCeoiQCd2BLSdrvTeIhTLsk0mSf+S+ZkREZlrRERkZMT21/N13W7ztH6MT/PO68f6kcW8e/X5XPu883oe68TrNV/v1zyv7/fHaXP523r/tW6x3t73GNb5vj7nPabxol6s8xj2K+tl/+t+8/i4/71zV/vV9eE5qvowPr7fd/tu71c8f9WX1XW9OW///qzv69f75M/6pn0u+otz7depYS3Gt/ulegPHL97wQVpuzpv2mc73vn9a/+I59np5rvmH4tb4dO7ph+l348sfzzWP7/fb/4fxGtV4vh2Pq76L/qqvXjdv7lesu53jIwCAdgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhgQAAGhIAACAhr4A4S/eu5DbpkEAAAAASUVORK5CYII=) |
| `ylorbr` | 22, 0 | ![YLORBR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABMklEQVR4nO3WS3KEMAwFQJP7XzPHCM4ihNQIFPOblbo3FELWCGoWb+r9s7fWWutz+9Ffr1m9LfUe66PnYd7h59l+o+fZ/strx+dxXr9Yv30+7P9UPb7n2n/2+hXmXp1z83cfuy7fY06+4xy/28n+03OT/8mmf761X1/7f9tez/dBfb2fw5xB/e9cmDOoj/aM73V4z2TvTf3ofbLH285d3Dv9m7bX+01f2z+Xnd/0reenf+e/f79pf148n9yn75vN2/RN+31P/26yx0cDAMoRAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAr6Bl1/3M5xHucsAAAAAElFTkSuQmCC) |
| `ylorrd` | 23, 0 | ![YLORRD](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABMUlEQVR4nO3WW26DMBAF0KH7X1rXZPcjgSZDLIxCH9Kc84MMN2MnatFdev/sERHRW9yut2VEXt+vay6vD3ODubO5t/d/vU+fPufJ+3neaJ+f2n/2d8z5fK42uL+u29G8q+asuWv36Vsu0uci5QfPfzu/fY9r8j0/b6P1bO6vP3duTm/re+B5HWm9fz3m+6/X3/sMcof7pTmzucH3Gr0Wdv82fXk81kNuSbn1+EvKpTm7/PPzt/dP+dn92/3c25/HyWs/mf8v89e5HwEAlKMAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFKQAAEBBCgAAFPQFtpSs5DFmoEIAAAAASUVORK5CYII=) |
| `afmhot` | 24, 0 | ![AFMHOT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAyUlEQVR4nO3WwQrCMBBF0Uny/9+sCxNEoTQIReWdsxlbHUKhyG1Vdauq6vWQOtucI/T7Pj+0OUd/vb877f/X/vr9+QHj4H77cG93/+rzr36OHz//+QLM2d6u1zz85zjZ391z/jfOX1cAQBABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAQSAAAQSAAAQCABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAS6A00eBCyOMjxCAAAAAElFTkSuQmCC) |
| `autumn` | 25, 0 | ![AUTUMN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAm0lEQVR4nO3WsQ3DMBAEwZP775mOVIADAYJ3JmFAPj/d62xn23Zt6fP2id7/+s78f84//Z/99tv/mv33twBAiAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAEPQF6MsDLUMRWpYAAAAASUVORK5CYII=) |
| `bone` | 26, 0 | ![BONE](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA+ElEQVR4nO3WSw7CMAwFQDf3vzMsKCAKVov4qbyZBREhdh0Q0puq6lBVVTWdXqduHY/357paO9d9flf/ZJ+2fmOfS/2Tc//8/uf6sbP7b/39Pn3/unlfY23eeR1NfXuuW+vmXDt/22cn87f9mudvvsfK/P/y/X16/vn/N5rzy/2xfP6X66/7761fn/c39X3fV+vntQCAOAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAh0BKR4ELNh0hOMAAAAASUVORK5CYII=) |
| `cool` | 27, 0 | ![COOL](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAu0lEQVR4nO3WsQqDQBRFwaf//81uGisrBYXAmWmWEHYvJM3ZZq01M7OtmfI557m/9f1xnvPR+y/v3925/Tte9t/6H+x/u//6e8fl88P79v9s/+k713sP79v/dn8fACBHAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAI+gHytewt1doO2AAAAABJRU5ErkJggg==) |
| `copper` | 28, 0 | ![COPPER](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA/0lEQVR4nO3Wyw6CMBAF0Kn//8+6MPUBwSJqY3LP2ZBmgJYZFrdV1bmqqtVVa/3antanaq/rfV3H6n196uuN/XbXN/Yb1Vfn3dhvVV+873B9sd/qvHvr/Xs3vm9vfdb8R/2eNf9Rv2fNf9TvWfMf9XvW/N/+P340/1G/Z81/+H9Mmv+9HzvvW9QfHnxe14f1220f1v91vy+dp88fAAgiAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACDQBRgAA3PGb1dcAAAAAElFTkSuQmCC) |
| `gist_heat` | 29, 0 | ![GIST_HEAT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA/0lEQVR4nO3WQQ6CMBQE0Bb0/kfWhUAisRYjGMy8tykt+f1pYTG1lHIrpZRaHoZpPOt8vX72+m/7/fq+9r6fo/t/2+/f57udf3oYhuf5sj6+fj/U9/VzXfN9p77Xv9bGvlv7d/bpnr9Rv1zsuJq3xtaH27pPq77bfz7IZdrn8jyfx/H6en25mEZ9s25j/eH9V/t83L9Tf/L+828AAAQRAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABDoDqkqBCy6v6d5AAAAAElFTkSuQmCC) |
| `gray` | 30, 0 | ![GRAY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAm0lEQVR4nO3WsQ1CMRAFwWf33zNEFEDwJcTOJA7s86V7tr227Zyz8vlx703ef/vO/H/OP/2f/fbb/zv77wCAHAEAAEECAACCBAAABAkAAAgSAAAQJAAAIEgAAECQAACAIAEAAEECAACCBAAABAkAAAgSAAAQJAAAIEgAAECQAACAIAEAAEECAACCBAAABAkAAAgSAAAQJAAAIOgNTv4ELNVOTwUAAAAASUVORK5CYII=) |
| `hot` | 31, 0 | ![HOT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAtUlEQVR4nO3WuQ6CABAA0UWs/P//1cKjoFELI2TeazYhLEcCySyXmevMzDp3p8dcN3N7/NPz7Nvf0/7y7kJHfbHa/lGec6/7rx/h/Ke5fnm++/9iPr8CACBEAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIugF61wQsmz5X8gAAAABJRU5ErkJggg==) |
| `pink` | 32, 0 | ![PINK](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABSklEQVR4nO3WTW7CMBAG0LGhcIve/5btJiFygjuBqupi3tsYT8Y/QQh97TPiKyLiertFRMTHOt7vw/x6uURExGUZH/Pex3FX72u9tXGePO9n51l9GVv2fLLfvv5X87Qv4tT6bF26funP9k/vMzk3e57d+919H/Xe1oblRcf+6G14vM3H/mhJ3zrszzvUk/PbuL7Nzn+x77Bucu5x/919p/vt+57Pt/Vn+7L7vLb+7H7rh3z9vn7yPvHL90nWz/rb9kNZxvF/7936Nu+Terbv7H7P9znb937/q/f+7/rP3/vj/zYAgHIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIK+Aeg7BCy8yYv5AAAAAElFTkSuQmCC) |
| `spring` | 33, 0 | ![SPRING](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAvElEQVR4nO3WMQrDMBQFwe/c/8x2irhyZYMMgZ1pRAjSg6TZ7ZjjmJmZ7XdkzznPz6rv9/Ocl95fvH935/bveNlf9T/Yf3d/+Xv75fPD+/b/bP/pO9d7D+/bf3X/MwBAjgAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAEPQFCrTsLf3G18cAAAAASUVORK5CYII=) |
| `summer` | 34, 0 | ![SUMMER](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAzUlEQVR4nO3WQQ6CMABE0SmH9RoeXTewUGOIJUTNvLcB0tBSWPBHrpdbkoyRJMl6qLveTpbZ8dUy3owfnf/k8fG0keXD9/eyzuT3mF7/2/fn8eTvnv/ofHvz/sh+7P+k9ffmtf9z15+83v4HAEARAQAAhQQAABQSAABQSAAAQCEBAACFBAAAFBIAAFBIAABAIQEAAIUEAAAUEgAAUEgAAEAhAQAAhQQAABQSAABQSAAAQCEBAACFBAAAFBIAAFBIAABAIQEAAIUEAAAUugN1NgOTM85YYAAAAABJRU5ErkJggg==) |
| `winter` | 35, 0 | ![WINTER](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAvUlEQVR4nO3WOw6DMBQAwQf3vzMpEgrSIAW5iHamQeZjYxp2mzmOmZnZ3ofZZpLj+Yz36PVzfNp//J7Ln/8+/2/v//T5xftfPd/j9W/2v3z9xevZ/8369n+979n85+8AAAgRAAAQJAAAIEgAAECQAACAIAEAAEECAACCBAAABAkAAAgSAAAQJAAAIEgAAECQAACAIAEAAEECAACCBAAABAkAAAgSAAAQJAAAIEgAAECQAACAIAEAAEECAACCXgmagq0pR9ZiAAAAAElFTkSuQmCC) |
| `wistia` | 36, 0 | ![WISTIA](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA+klEQVR4nO3WQQ6CMBAF0OqxPa/nsC4ElRpCU4mR/Pc2ZGqZmRgW/3Stl1pKKaXeysPjeap1WZepft6r0/lU16aef2/uz+8/++01b2V+77y1fbfmre07PG9jfu+83v/nNW9x/DavqZvjz/OmXj0f7P/x3p/vO9z/V3PbPht17z19D9G3/by6++6930Y9vOe39VH2HKzPBQCIIwAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAg0B00Yd6qfWl7tAAAAABJRU5ErkJggg==) |
| `brbg` | 37, 0 | ![BRBG](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABXElEQVR4nO3WW27DIBAFUMDtfrqY7n8nhn4ERzI2cpq+VM05P3RgxiRRGt38/vbaUkqp5JRO17Kvl5xP98d16X88/LzJ+dPP6wdlmyuf3M9/PN/rXJbbuiy7ugz18fzl/LyvZTKX+9z8fDI/3Hd4feOah/eXb2u6OL+aH8/TF+dn5y31L9x2Tyr/sm69qtvaWq/bbr+N+y2d1i1d9bX9vW1/z3Xf+XO3vrXenrT2/rXt69rG83Y6N+3r+3UyN+0bn3e4b+irQ98n17rV63Pz6zfN1+F91J9at8/xl+5Jvb6vbajv/xBP9h3622R/7H/w9fW+7dcAAAhEAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAhIAACAgAQAAAjoAy19TqpFKLs9AAAAAElFTkSuQmCC) |
| `bwr` | 38, 0 | ![BWR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAr0lEQVR4nO3WOw6DMBBAwXW4/5GBFKTPp4p4M80i2StRWW/NnOfMzOMxU55rXXPbmuff3rN/j/015/VxHP899/2a54//a/+z/Xf3nN/q/PUMAAAlAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABAkAAAgCABAABBAgAAggQAAAQJAAAIEgAAECQAACBIAABA0BO53AE9j9KUDgAAAABJRU5ErkJggg==) |
| `coolwarm` | 39, 0 | ![COOLWARM](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABcUlEQVR4nO3WMZKkMAwFUMGx9gxz6L3XbNuT0J5CYKCDjfRe0mVJlom6/vLn62+PiFiWJSIilnWNiIh13Z9zf0n9ddZ/vHfZ+uf10b95Zz28O7ufvz8+2nesx/n+957p/pi8t58f92fvbf2877Y+3jnfN5+L67mb+nyu39zvz+oxm0/7Y9KP/B39fO+Ye9Y/vDfrP6yvuR7t2f0831O9tw/P+3rkuX69J/L9nr/vfO69Nybn6d5xvp4be1veez4XLX/Pef3uXm/P5m7vbeeezu/+mD/sub5/OLe0L+2f7cvv5/nx2+/mJvf+U729rud/+2luq7e8/9X2916TfeP+tu97O//rn52/9/9XAEAhAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBP9DlVBoxSTBGAAAAAElFTkSuQmCC) |
| `piyg` | 40, 0 | ![PIYG](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABaklEQVR4nO3WS27CMBAAUCf0hD1Yb0zcBZ7QOBhCRFfz3saajwdHlaqZfqbvWkop8zSV2zl35+P85Z/7T9+bb/npa96elxfx233Th+c9zpf4nnnbt+bj3uH61OrzuXqbWwb3It7V4976PS2Oo69Po7g873t37sn+2p4R9+5nqw/yEdc+382rXf/rOY/n7X6n61+/oywtru1cyt+O2tXH/YN8fXfOuf6lXp+f5dqec7RveVqPeDhv13ds3nouES+b+Fof5+/3lm3f4fr2fdfBvX198I41H2f8neqLfFff5aN/GeQ/fL/l65ovt7idaz3iGvVRf/nwvHhfN3/QH33x7xQASMQCAAAJWQAAICELAAAkZAEAgIQsAACQkAUAABKyAABAQhYAAEjIAgAACVkAACAhCwAAJGQBAICELAAAkJAFAAASsgAAQEIWAABIyAIAAAlZAAAgIQsAACRkAQCAhCwAAJCQBQAAEvoFfpNN4jZwYxMAAAAASUVORK5CYII=) |
| `prgn` | 41, 0 | ![PRGN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABaklEQVR4nO3WQXKDMAwFUAEX6773P07sLmLIoIRQmu703sZjIcvOJvOnr/juERHzNEdExDyPNe+3+vKyvjz1Lydz9n3L0b2p7/R967rc12me7uuS6suob/25b//9qZ7m57mP+w/qp/en+XPq3+of9i1nfa/P/b1vervf6tP7vsP9dDL3n+6JdRnnY97v13Kkdz3Xr/U97vttX37Xvr7q0cbaIyKijbX3fX3r6z31p76Dc4+5ad7BuXZ4b54/+sf50zV+2Xe1/5/m3vrtvraxb2l/+v2gfnnOZ9+333Mb+62vp/36vaf9ev5if5q/9X34jjbe0dvZ2t5+b6fnr83765r+BgCACgQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAggQAAChIAACAgn4Adh5nnfMeCHUAAAAASUVORK5CYII=) |
| `puor` | 42, 0 | ![PUOR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABb0lEQVR4nO3WTW7DIBAGUOxk0Tv1cD25TRcBS0CQXVdVF/PehgzD4CT90bd8fX7klFJal5xea/rZul49n2/O3XxePVc26v5S625/fq7e29bLbL6sy7Df3rNM3t9wz/q4ta61fvT957W5fh3uae+bznVrquvybOtjv9bPrp70h/3J/X3/6v1dv37enNa2X36guauv9nO6Nzftl/tO59Lr9yy//jzTXl7UOu9dXV7sR93u93P7rD/cU/r7+/PD+xrqcq6+3/r8szr/8fnj+/vh+aPem/2h3k76td7+577c37u1dd5m/TJ/Ovd+fpy79tzTuePzlTrXejup6/mt+56unRvr3z239ut/CwAgEAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgIAEAAAISAAAgoG/yLXq2Mq3nPgAAAABJRU5ErkJggg==) |
| `rdbu` | 43, 0 | ![RDBU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABeElEQVR4nO3WMXLDIBAFUIQOkuPl/n0kUgisMTFGGhcp9r2GsAtr4slM/vKdvkpKKeV0WJe2Lv+y//QdOdc56zFpqY1lUM+tv3b9PKj393Jffz+/nT/r/b5/R346twzq0/Nd/3zn+vZevjhv/o7jc1Juazt3rd7mpbWvP9+b1j99R71f6t9dWvLLtQzqZ7+7n+/dH87p5k3fcezSXn/4sy/HD3tq+9Yvg/MX+7Vznp99XntfeTmvfc62t/6xbm1eHfjYP/p1/l5u9rt5+6De5kz76Xnf3RutP5P+Vn/x2Zzr826u3fdTuvVRL4N61z/r6fl8+XB+9/0P59Tvs+zb23Uf9uv97f396byb90fvaP9vAYBABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICABAAACEgAAICAfgEb4GHUi3UcIAAAAABJRU5ErkJggg==) |
| `rdgy` | 44, 0 | ![RDGY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABJUlEQVR4nO3WQW6DMBAF0LF9kByv9z8A0AWxIiwR0hBUtfPexvJ4bCyB0C9fcVsiImqsWulj+ZX52XvUej+nrSeV+0LZqde+3ob1ulMf99Wx/vz83v+oj/PxHnXTV3bqh/3D+uOe7em++uJ5x/dYnxO1j73vtXo/L9pY3+47rJ+9R99f6nase/P2Vl856hvX3+2L9XtbliUiIuZ53sz3xk/1Xf28/zL/6+M0TT+qX7Xv7LljX39PV8+vei/9LwAAJCIAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJPQNcOV+jGWL7lUAAAAASUVORK5CYII=) |
| `rdylbu` | 45, 0 | ![RDYLBU](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABZUlEQVR4nO3WQXLDIAwFUEzu0pv3mDZdFNwJjuI6aVd6b8MAQiZx2vnLZ/lopZSyLKWUUkqtL47vnv+jPktvcBhvS68b8/q0LjwX1j3uV4N+Z8+tt+kLmfbHvEzPP63f12vQJ5rX6fzFvpdf7Ks/hP99zv75xh/MUh+PdV4P6g91U324H/W5eK/yPbY+bu1+3tptmve6/dztbj2sO+sT7P/c6/n+GNet9PrWx7P5tbr1tO69/mN9HfMtGKf99bBerp0L9uf1Q/3puX6P/mLCzzPG/gW3fT04t97P26Hf9rBuHtuh3/N7tul+4wfXgv22/2Au1r24/9t7jRcz/gsAAIkIAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAkJAACQkAAAAAl9AV0fRTo/bIbEAAAAAElFTkSuQmCC) |
| `rdylgn` | 46, 0 | ![RDYLGN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPElEQVR4nO3W0XLCIBAF0A3+S/+8nyn0ocSOtBhN0vZhz3lhYBcw0XHu8h5vLSJiWSIiIkrZOR7df9I5Sz/g23hZet86Lw/7pvumfT+fVybnbd1bLsMLGerrPIb7N/tv62Vyzmxehv0vnjv9Yp9dP7r/nPXl64f2eDyr77fv62rUiIho7XOsfWwxzNf62P/P+8e+2towbqzHuP7X++77r7f16PMY5r1ex3qb9D9Zr7P6k59nVq/3z3Ud7nm5fuuLyb6d9Y371/fX6s7x6P6zzunPdf8vAACkIAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAk9AFA+FfdjBBvIQAAAABJRU5ErkJggg==) |
| `seismic` | 47, 0 | ![SEISMIC](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAxElEQVR4nO3WMRKCMBRF0R+E3bn/zQBqkVgwY2Gl6Dun+ZCEgqHgtqrrvaqq6lLHOdXrdef+6Vxr/X6ex+7YXpY6rL87Pf9bz0916xfbdo65rn3u+3f2z/LeH7ofX98Mnc+/BQAQRAAAQCABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAQSAAAQSAAAQCABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgR7D7sCu2SPbcQAAAABJRU5ErkJggg==) |
| `spectral` | 48, 0 | ![SPECTRAL](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABX0lEQVR4nO3WS3KDMBAFQAluloPmpqAsEFQxNiUHktV0b6ZGPxD24tXv+tVKKWWaStlq3eoc+n1+3vo59KP98+C8q36+ep+r83qtR99OfY19bW/3vaybwro6mL94zqfvtZ9T+v1K/y419Pt8PX6QwfpjfDqPH+dc9VPY/8tzp3DOFN53OP50/9+M19c/yvv6V+v++3llu18ra6/b/7C1Qb+vb7H/dF3oHz639PllK2U9ag3j9TS/hP7x+hLnnz0n3mcJdTi+3tx393kv4/0eS+/Xeq5LfT8+mr+7r3+Pu+e2fo+pXzjWulyMx/X9h7meH+2/GP90fq8FAEhHAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEjoB7IUEo1p93stAAAAAElFTkSuQmCC) |
| `twilight_shifted` | 49, 0 | ![TWILIGHT_SHIFTED](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABcklEQVR4nO3WQXKDMAwFUIsr9CBd9v4no4sAGYQV6Ex3em+TYEk2dOjkx/fXzzrGGDE26+tbbCux7pWj41Tf14/rPF/0xViK9b3/VY+Yz136cj3m/ce5l3oxF/t9pn1iPjfS3Ltvqz9cP+4v0v7l3Pn6mF/yfkXfsZ7qS1WP+fzyuS/Xy32PvnGz37l+2S/V35/jn/rm13HbX6xfnjvtdzv/t/4o6sdr//A+4uH+ua+qX9aX+b71/Of+6/z5fu5eu3zOfd9f+8995euZ+8p/p/nzLbf7FOvbDvm8fE7dX8yPPJ//Dk/rxXmjOL+ay+el6/z7VM29f3/y3LmePsq5MeZ9eX3/FutI1u15AIB2BAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGBAAAaEgAAICGfgHrUwjJR/jGugAAAABJRU5ErkJggg==) |
| `twilight` | 50, 0 | ![TWILIGHT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABZklEQVR4nO3WQVZDIQwFUOr+F+hUFwKd6LcEUtrjMPcOqh9CoJUe3+3r83u0B+N2/fbw+jCfjY95fKkb+/F0Xajvcf2Y6/pp3bXPfryH86zzYb/D/uf5ZL/lfEm/Q/36PrL9nu+znnfus9S11+rWn6/WvVkfztNeXNeSfdqpPnywed/9edb553XxnMv48gd53j/t29+sj/2zC5edo2f9sot7mP9n3Tj2SS7e+sVJ+h0uZE/WZX2vfn0/P+b5OD56P9T9PLf9/Fj27dt1p/FrvoX+bZ4ffx94GE/WtXC+FvZP14V9Yn34/7TWzRd+7Tv29bds/e9zm8dj/fXcpuePBgCUIwAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUdAe17Oi3VTcsBwAAAABJRU5ErkJggg==) |
| `brg` | 51, 0 | ![BRG](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAy0lEQVR4nO3WMQ7CMBBFwXVy/yM7psA0QVFQRAr4M80K2c4WNK9VjVFVtdRWybPNuYaef36vzzn+/P3Rvb77PX76fbXn2JaD+TpfT+7dNef+PvePi9/py8X3Yfvf7u3+/7PvfP3c/lv3zwEAJBEAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAQSAAAQSAAAQCABAACBBAAABBIAABBIAABAIAEAAIEEAAAEEgAAEEgAAEAgAQAAgQQAAAQSAAAQSAAAQKAHRaQAPadDlKgAAAAASUVORK5CYII=) |
| `cmrmap` | 52, 0 | ![CMRMAP](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABAElEQVR4nO3W0WqDQBQE0Ltr+v9fHM2Dm4AWUwOlUOecl2HjYCAEnVZVS1VVa7d6l72/v/5/e9Oa1TfZd+ejPNs72//r+/VqI+c125rT8zxyavMle9Ord9/m+Pz2Ot8v2Wvj+tKXWn+YZXNe+rw7X61XI9sujz6/Wm8UxvOwxvPwW/bpkr39+6Hq4P3Qvn7onb3fp73t9//29z7/HgBAEAMAAAIZAAAQyAAAgEAGAAAEMgAAIJABAACBDAAACGQAAEAgAwAAAhkAABDIAACAQAYAAAQyAAAgkAEAAIEMAAAIZAAAQCADAAACGQAAEMgAAIBABgAABDIAACCQAQAAgR7b9XwsZFE41AAAAABJRU5ErkJggg==) |
| `cubehelix` | 53, 0 | ![CUBEHELIX](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABp0lEQVR4nO3WS47jMAwFQFLp+x95PIvxB6KlzuxZtRH4qCiwgSjMiDgiIjIyIiJGjljVI84881xrXtYzHzUva5T92/5Vj3X/2ZdzPtb7YtR9v9fHXWc5N+d+2ffksenXPNb5Xce076qPV/+YHqfmz3nH9FhXvcvzPndd3997rs9rOpbrGH/muvbv13r8vsYu/+dT8s/1WF/qT/18ze/+Vee6/8pzmf+88msd6/yo+djsn3+vT/8znTNKvqvH9blNPvIz12XNzb6Mnym/9uWr/ln3cz4nan6uVx7nvlc+Sn2vY5kfY91/5ecP4rj7NV/Xr313P0te7p3c5KWu91u9f+o9dpT76pV/ua/+N3/ulelxvvbvv4F6D5b7JLPun8/Z7Xv9HcQmz5zzWOfPa7v+X7Ocm/Ma6zq3/VHOuX6vAEA7BgAAaMgAAAANGQAAoCEDAAA0ZAAAgIYMAADQkAEAABoyAABAQwYAAGjIAAAADRkAAKAhAwAANGQAAICGDAAA0JABAAAaMgAAQEMGAABoyAAAAA0ZAACgIQMAADRkAACAhgwAANDQX9AlQyw8XDUnAAAAAElFTkSuQmCC) |
| `flag` | 54, 0 | ![FLAG](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAADu0lEQVR4nO3W/09WZRjH8beRDbEohxMjGyaLUktZ5JcWFVgo2fpiWtZyxp9G5bRylrNFKDw+lGygpjMXado3E202Fef3QWQ/XJ97zziuv+D+vH657vNwP9d1znOuc7hm3IE7AHSvAIC3uyKufjXi3OcAODIeh33fK+6OONQTkVvHtNir+I3itwAs1ZGyo+y0K1a98WQs3l0X8UXteKQDgFM374vsx5X964jljyPeHvtVmfYp9iruB6CRmwCsLdR/WbG2bUEs3u/UH9ZHbIod56fmAND/s+rrMge2Rbx0/IIyDRTqx3EdfwPwSqG+qtGw5KFYbNEZdan+4thxdeajAJR+V/39utpPI54p35h2vZX68XtU8xsAHYX6+rVprp8Zi63a8bp2tMSOqfsXAzB4XvWHIvbtjDi6S4kYVEz3P/XDDwC06airEFvv1aJ7dcQNqr9KO+pWAnDokuoeVJUvI4706Pv/HNGirxDjhJfrKF13sQ/ZqE59R3XbtKMhdpy4XhVXdUz1v4qY+nDywiklKj4HZQAWcRuo9KHuMmsUZ3c0xuI97Vij+o9FX5ydrAWg/0R83KvLK6kPr/ykG0R/oX704TziB0x9mOqn4/pldbHYok/Wqf4T0YfjVQ1RT49bb0nVdkQcG7qqTKVC/ejDGs5Mu95UP/0eTQuqY5H68DXVXxZ3bKKmGYDyOWX/LuLezyOe3DOlTIOF+qkPRgF4SUfF56Bllhbd6tS31AfPKs5pBWD4orKPKPsXEQ/36PscKtRN5xFfeEZHxefghfT1zerUTTqz53Wm89sB+PGash9V9j0RBz+K+O9lNcj/9OHjTAJ3Pwfp/TCrc1EsUh+26041xp37Y2I2APtGlV3pS59EvHb6rDKlPkzvo+iLh7kCVN5/qX7qw7mt82LxgT7pVP3mOL44ox6AgV9UX6/d/u0R/xoZn1avUj/O5wHGgMr7P9VPfbhwYU0stqpT16v+07HjVnUTAOU/VT/14WcRT/dOKFNZcfr78B5OApX3Tqqf+uCpWi0+VKe+qR2tumMPtgBwIP6t0DesqPfwUb0PYLhQP/XjYZ2HmZmZZccDgJmZWYY8AJiZmWXIA4CZmVmGPACYmZllyAOAmZlZhjwAmJmZZcgDgJmZWYY8AJiZmWXIA4CZmVmGPACYmZllyAOAmZlZhjwAmJmZZcgDgJmZWYY8AJiZmWXIA4CZmVmGPACYmZllyAOAmZlZhjwAmJmZZcgDgJmZWYY8AJiZmWXIA4CZmVmGPACYmZll6D/Hk+AvjBmoQQAAAABJRU5ErkJggg==) |
| `gist_earth` | 55, 0 | ![GIST_EARTH](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABoElEQVR4nO3WQY7bMAwAQNJBLy32Jf3/95IeIjkwbTXZ9siZiyCKomRECJgR8YiIiPgdEREZX89p/BrzOf68jm8/nuEtx/w5zdtIn/M9ntfxMcatxnMRr/nH86Pc4xRf5df6n37XKr/Wz/Jdyzolnov8rPUX35XHfVHucapzqn/MO9XJ8YzGmHEv87I+80fea/5mfVWvnHeqk/fFvjf3PtW7L+p/eI9a59N7jPUtyziq3sYP83qGeb3+aV7M/GPe6/mX9cjrvPke9/nMH593qjPip3u+yf/XOntelLzVvfM6fxV/V2dxr/25zPs9RsLxmezzeHx33yJez1/V3/cd/zhyH58vIMcfxVbmOf5I9/ys68f4Vvfl9TnnOuUeM74t8ra/z7e57z/Hj+vcFvtvZf2b4zx/vlMAoBENAAA0pAEAgIY0AADQkAYAABrSAABAQxoAAGhIAwAADWkAAKAhDQAANKQBAICGNAAA0JAGAAAa0gAAQEMaAABoSAMAAA1pAACgIQ0AADSkAQCAhjQAANCQBgAAGtIAAEBDGgAAaOgPxxhGIPyNTukAAAAASUVORK5CYII=) |
| `gist_ncar` | 56, 0 | ![GIST_NCAR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABo0lEQVR4nO3WyXLTUBAF0FbiATaA+WJ+GhIZDyzUckrPlq0EWPU5m1u331O5ylN1F/HjHBER218x5EtM+qemj+dz86vnmvl+N2T/LXM3zX07X3qvuT++zpcx42P5deG9z5nrJldNtvOH987Z99n7zOyRPbKf53rmeX+/R9vzc4uft3s315fd22Tf5P3tMeeHmOR27MeZ+dib89nnn4d8anM1M3/3+XPmMFhnrrrs+UGv4n5feu+tH7If7val9676Mfsp85hv6GmIyPrP+mXeZe/+b3/N13s938+XB+dL7/2OaR7i9vyvz/N/8PA9B2Pumv5ovuy8z+9/n9/7Pn8Xff5OHvX9w3vD+9dvhtzM5unmfD329f17bXaX/+UuezeZdzmPpo/3rucxPZ+dT3v7fFzmzXMz8/H5pwAAyrEAAEBBFgAAKMgCAAAFWQAAoCALAAAUZAEAgIIsAABQkAUAAAqyAABAQRYAACjIAgAABVkAAKAgCwAAFGQBAICCLAAAUJAFAAAKsgAAQEEWAAAoyAIAAAVZAACgIAsAABRkAQCAgv4Apr7BInyhuScAAAAASUVORK5CYII=) |
| `gist_rainbow` | 57, 0 | ![GIST_RAINBOW](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA40lEQVR4nO3WXQqCQBiG0c+k1tL+l1eGXehNYST5F7zn3EwNMyiM2dP0de2rqqru9TreatZ88xjGyzh9runvn+a3Xvcv97Hyun4c309p5qmtti913fcDW3rg9k/Od+0wLn3A9/6hrLXvoOu347/Er6/dvV/fa+07+vqn6sZP2zwopwIA4ggAAAgkAAAgkAAAgEACAAACCQAACCQAACCQAACAQAIAAAIJAAAIJAAAIJAAAIBAAgAAAgkAAAgkAAAgkAAAgEACAAACCQAACCQAACCQAACAQAIAAAIJAAAIJAAAINATR/N27R9Elf4AAAAASUVORK5CYII=) |
| `gist_stern` | 58, 0 | ![GIST_STERN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABEklEQVR4nO3WQU7DMBAF0LHLBaB34f73SgQs6giFNk0QolL139uM7BnXkdXFb1X1WVX11nrdqueN/b3+1v7rqNOD67w3V+1S+ymq/z5qG7XGXB9z7Ud/rzr/bOc/Rp1v1qrpsu6Xdb+am+6e/2t/ub/3x9679KvmJ7+/jbr8H1526nquauz3db8f/J0a6+v5ve85re7v/dj3/r5/f+77/v+95/j7L+958P0233/sFwAQRwAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAoC8R+nEsORLd2wAAAABJRU5ErkJggg==) |
| `gnuplot2` | 59, 0 | ![GNUPLOT2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAuklEQVR4nO3WOxKCQABEweFjJve/rRqAgRthEel0J1MsbEq9KckzSZI13bsceyt7v3y8Hb/+dt0fzqfHvvO5XU9+96/352Nj7U/v+w+wHXu/+DyeX72/P88BAOoIAAAoJAAAoJAAAIBCAgAACgkAACgkAACgkAAAgEICAAAKCQAAKCQAAKCQAACAQgIAAAoJAAAoJAAAoJAAAIBCAgAACgkAACgkAACgkAAAgEICAAAKCQAAKCQAAKDQC4Z9gyzLXsu9AAAAAElFTkSuQmCC) |
| `gnuplot` | 60, 0 | ![GNUPLOT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABeUlEQVR4nO3Wu3IbMQwFUHDXjzL//6mx9UgRLj2Cl9LKmVQ4p4EBkJTtUXFbRFwjIn7FW0REvPf62utLvPf6dlPXVJd+7qvP+1l9fWrfRn9/Pu9fDva3dTZvsf7dt3V3PvajX3b3X/Olv7fszlu0Sd9rS33EpO+1xf581Gs/d73tZ7Vd0vyS5pf+8taf0/6c5ufdeYz5KZ0/pfmpn0/z0X/u9jHmn+ncbR/f9h9p/7G/7zW+3Uv7yXzro/1+cH9/P97d/o/pe7D90FKN/H3J89n5PI8nz8/my7+9M/rl/rlt//T9PP/pufx35t9n1s/uzd4/eu/gOw/vHewfnlsPvrvev//w3f/V589ff7g/2G91ew4AKEQAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKEgAAICCBAAAKOgPlZhALde9ajIAAAAASUVORK5CYII=) |
| `jet` | 61, 0 | ![JET](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA9ElEQVR4nO3WwU6DUBQE0KHF7/bDBeuCh4mY2IVVW+eczeTy4IYEFjMlz5ckSZ62yJzbzveyd8qXZvlYOX7beRm5JklOI8/vuZ3Pn66PPI/MuC/7fMzvnv/XPY/5vvt8dc865mXP15EZ51tOY86e62G+dv23n7v3fX/8HpeR47PnZZ+Xw7yf52MuN55/au8pAEAdBQAACikAAFBIAQCAQgoAABRSAACgkAIAAIUUAAAopAAAQCEFAAAKKQAAUEgBAIBCCgAAFFIAAKCQAgAAhRQAACikAABAIQUAAAopAABQSAEAgEIKAAAUUgAAoJACAACF3gAt17OuXVtfIAAAAABJRU5ErkJggg==) |
| `nipy_spectral` | 62, 0 | ![NIPY_SPECTRAL](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABQklEQVR4nO3WTU/bUBQE0LFBQkkjPv7/zyygVgWbxXuhwmooal7UxT1nM7rjRRaxpZmSrEmyyz4tvyVJ9pv73P4muyTJlClJMvecMn+p//388/5Xv5/6/fx+j+2XXKe5cO7yMfcZ29/0vNrkPLifl54vLadNjuqn1/6D21xO9H97fqJfnluuTx9zVL/+bHmp/+Pc/n///on+tp93+fN9bn9YW14fc9ncg/qrfh9f53lzj+7zcuF8PDO/f/58/dHyH7/mL/fHez1xj+pzOLS862/gw0PL+/uhefyMAIBCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICCDAAAKMgAAICC3gDkgJOTdyuWzAAAAABJRU5ErkJggg==) |
| `ocean` | 63, 0 | ![OCEAN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABH0lEQVR4nO3WQW6DMBAF0MHNvXvzko1Z1JVjaJJK1X9vYw3gGUyy+Ft91l5VVftWVVX11df/Vo/XL/erXrdJv+H+av+z/Z+9f3n+8fykz+nv3ffvkzln98/mb8Paxrrv39rk/mTfuL+1xZxZ/Ufz33b+V3+Pk/XV+VfnvHr+ol/r9cc2rOP1ZV19bY+fm1y/tcdz2rJf9T6L+UN9nP/263Mf9ffzr9/32vnH933V+X983yd//9ubzn/8vQGAIAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAt0BytFZLBXUQMEAAAAASUVORK5CYII=) |
| `prism` | 64, 0 | ![PRISM](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAADHklEQVR4nO3WyU8UURTF4SMSlAVC1IhjRFQcGBo0xiHGWfmXnacoajRAN3ODyNAbSYhDAlFZtIt3XkmXdHT/ft/mBFKv6la9utV3W1WqarNjzn7ngLPPecp5NMRyc8hR/3vEWfJZpyohF8qXJUnr43e84KoXFEIW2yVJB7z+hLPH5fVrxuW8lyQV9EKS1KQHPtIX2uU/fVqdy/191nk8xPreresvOie+hJybPyxJWpm65wNueMEFn6BLktSy2iDpz2M84+zTZ9dfdL6SJO3XYx/xRjVinfH59zt7nV3OIyHKjbX1xxz7FXJmKeTSzKAkaWPslg+84vrDibeXWzefNrtMj364nCnnW0lSt575iLgP30O05+qO9xH3IT4YP6hV79torv64D5Ph8Wn+Y3gwXyfuesE1LzjvBeHF3LMe/uz0+u7s8osuZ9jlvZQkteqRj5gI0ZirN78PPc6TIaoHa+uN9cf7GV8LObvQIkmqTId9qJbiPlx0hvvbUdkpKWsznXb26ZvLGHdZQ77Ppz7ivmp01Kk/9nM8sS9Uaa6tP9/P07GfZy9JktZiP4/EfvYFSu5nr6vfzx8kSQU9lyQ16aGPXA7Rkqs7/x7Fjc31c719yPr50yFJ0srkoBdc94LYz+FDF/u5w+vja1vQissZdXmvw/1m/TykGnHhQC7/0c+x7nr9vFgO36ON0m0vcD+PhBe0ody2+bTZZXr103VPupx3kqSev/o5vG/aV6f++B7l+7l16/qzfg6PT/NzuX4uup+H/eEudkiSdrufvc3Z57FfSy4n9HPB/dyW9XPok//uZ7+o1fB6qJSrP97PWN1+vukF7udR97N/KGM/x5/RPn8vBzTmDL8DnXriI+I+uJE6cnXn96FOP8e68/3cIAAAkBwGAAAAEsQAAABAghgAAABIEAMAAAAJYgAAACBBDAAAACSIAQAAgAQxAAAAkCAGAAAAEsQAAABAghgAAABIEAMAAAAJYgAAACBBDAAAACSIAQAAgAQxAAAAkCAGAAAAEsQAAABAghgAAABIEAMAAAAJYgAAACBBDAAAACToN9uVzYGmIKhPAAAAAElFTkSuQmCC) |
| `rainbow` | 65, 0 | ![RAINBOW](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABN0lEQVR4nO3WMZKDMAwFUEP23nvxxd4Ch0KMx0Amld5rPJIlkyrzl9/SWiml1Fcr+1nunT83548zfG/6zuz3PbxfZ/ujvdgP9ejd0d4xP5gb1nE/1KO9ft96vfa9tffXd3/9Un+J/X4ug71lsj+q43vLbD7snfoXz3Lz/mk965/OenFust8G97Hf6uR+8G48Z/ft4X39sF/D+3UwP+2Hug7mTvXVuVBve730uhzn3i9bqN/325fq+L3t4vzV+1P9dO/iu7O5v4d7p3c++9397xQAyEQAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASOgfdLd+PXmyIwYAAAAASUVORK5CYII=) |
| `terrain` | 66, 0 | ![TERRAIN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABUElEQVR4nO3WQW7CMBAF0LFNT8Rpe9WqJF0wKiI0JLSlUjXvbYxNPIODZP12PL7OERFT7xERMY9xNf75es/18bM602e95/SN/p7j278eW57j0FpEROTp4iXnh7Uxn1uujxxfNp5bq3ezf2ffS//8/bHRb6Xeav+d5/hv76/N5/k0nT9Mp/nr+ef6dP+53XW25td9TjnOj/abFvu/eY7TtNj/4Dlu+z9WZ73/vnOsvb/Wzvdaz3uv5z3Y855ree+Ncb3ee1/Mx9063933a/2Xz6+st9w3ss5lvlU/9+f341n9N+q0nI+9/0cAAOUIAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAV9AI3hAjv08nxOAAAAAElFTkSuQmCC) |
| `bkr` | 67, 0 | ![BKR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABX0lEQVR4nO3WQXbqMAwFUJlVdLvddgc1A0g4sS0IB2a6d5IfW5bVfAav/fz+9YiIS/+PiIg2PKf1WNeNz8j2Y9vvp+pjOreun99v/SPWddn5/Rnr+dLnfk9/qz6+XX+6br3fk+839zme71Nd0j+5d7xvmuNkn57M3cc+9/M9maMn3zFdn+Yf99ffPT/3vN+jTzLv/vs9vj+eN22r21eO7y2pa/d/XPb9tl5vxz6XF3X5fhv6vHdurk/2t/7jffvf0Yb3s/cl8w33nT+XzDHO+eH8Wb9xvq/3m9a389nvYLjn5f2v5n/+fff/l4/Xh/2z6+nv9twc43cCAAoRAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAq6AqFn9qWYX6QyAAAAAElFTkSuQmCC) |
| `bky` | 68, 0 | ![BKY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABQklEQVR4nO3WMXbEIAwFQNkHyOVz2xTLNl7nGVsLzqbTTMNDgIyp/vL1/dMiItb2iJlxSeafjrHPW1Lv9j+y88d533e877oeyb3G57fvxvtz2T3zcbTvXp822+f0H8f1Nvn9lq4f32vUb9RnPx9z98z6tejr/bsl7xijfd33T/dsx33xvv4778akvkR09djq2/oSl/tf9X38uL4c5uvo/N0+s/26fWvXb/4eyfdv71/+eG70X3P7Z9/v//u1y/rd9x3fq03ty9e388l/nM9f/1fWf/QOt9fjWH/dGwAoRAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAo6Amdgw6UQ1V7nQAAAABJRU5ErkJggg==) |
| `cet_d10` | 69, 0 | ![CET_D10](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABKklEQVR4nO3WQW7CMBAF0EloVaRK3P9o3CZdNFkw1PK00KrSvLcx+Sa2E1j8Ja7bFhERS0RExOVziPd9PO/52379eozL7fXLMe75KeXH9SnN53wt5mvKq/OP5kt1frDeMsrTfaN9cp7XK89/c73qfY+uN82L+83y8vwgn85Pnns4/8NzVvfL60VsN0OO766H+fZ1Xr6/eJ5y/l+e65feS/G58vbT9zJd98m/T/re/XmffI7i/X99jvp+OS/+L/Ix9w9rAADtKAAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANKQAA0JACAAANfQBj9GjFqYOVWgAAAABJRU5ErkJggg==) |
| `cet_d11` | 70, 0 | ![CET_D11](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABSUlEQVR4nO3WS3aEIBAFULTzGWT/W8n2yEAlp9EKGpNR3TvhAGWBdg/eVD5rLT94W8ePtey9rS/z1zYuHuv8ZR0fbSzdfBnn+rw+t7F089H6c79tPnXPTV39VM7VT/X4uajf1PWLzmn1NViP6nfrwX79Xb8yqq8X61vdYH9w3+ic0fkl+B7Rffr1+Nytb/SeV/sFfW7338pG79ufs86Hv3fXpw72+3NafXB+1/f8ff+4/uR7Rf/js++zO78eP/89P3lu0Cf6/e7f6/i7jfbPftfL+7v3C+5195zL9/if/dF3nwsAkI4AAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkJAAAAAJCQAAkNAX2mG8IRLoORQAAAAASUVORK5CYII=) |
| `cet_d8` | 71, 0 | ![CET_D8](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABaklEQVR4nO3WTVLDMAwGUKUtcBfufxdO05QFTmYq1/kBdnpv45GdKHbazHxTfH49IiIi2jC18WD93uqPVr9NP6u3Vt/a+rXV11RfWn05XLdtpPVRPeX1wfyU+vbr5+rY6Rc7ffr1pntOun8w39dx7vqTffL+Y3Du+Pc+Sx2b9+32T332r9uu19/vl9/ZX8833tfz/tbvbP1Z0vcXeT7Vj9f9zl+f6qN9d65b5x/b5x/3yc95fa7h/Wl+9B5yn73+R/vsnT+fa/nbDPef30P634z6Dc/XXf/6vnX/o/0M+gz3NxpT3+65J8fu/jlfN2+uX+a8Pj+PaT6W8d4O3vrF/WdY3jsAUIgAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUJAAAAAFCQAAUNA3ZrkTVeaI7LcAAAAASUVORK5CYII=) |
| `cet_d13` | 72, 0 | ![CET_D13](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABrUlEQVR4nO3WQY6kMAwAQAPzgH3I/mv+f95psgdCoyQdmLurLpEd20FqqeXlz9/vEhGxrFtERCzL2pzrGa/rx/tl2e7vp33HGZN3h77JnKF/Wnc/b3YfD3Hf38+ZxVd++ZiPoX65zZehr8vP4u79cr4znP33LvW5z+da69a1P6O537r7rY4/81ut32r919rd1/qvIa5nN6/Pj3X391e+PNSVyXtt3/ZQt3Xzru9r788567vvON8/V5zxfiRqHLWuxJHfyxHvZ1zPV2nPn1r3E33+/vy3/7Luad7+y/qH917767bvVV5d/JCv857ifRLv0/g4y97myzt/xvX33dt8dPno5vX5GPJ7kx/n7k3cz72+q7T3ff3QX5r+q66dU2rd1d/2XXNmfV3+Hbd94/f0c2f1n+9/Wx9DfXysH+No+qOrq38nAEAmFgAASMgCAAAJWQAAICELAAAkZAEAgIQsAACQkAUAABKyAABAQhYAAEjIAgAACVkAACAhCwAAJGQBAICELAAAkJAFAAASsgAAQEIWAABIyAIAAAlZAAAgIQsAACRkAQCAhCwAAJDQf+nNa3i+HI/hAAAAAElFTkSuQmCC) |
| `cet_d3` | 73, 0 | ![CET_D3](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABVUlEQVR4nO3WTW7CMBAGUJvue56etMe1u0AJspPBQGE1722ijP8dhL768/vdSynlUr/K8CzTe71M7df3utcv07iofa7X+/Nt85Sxf13Md+u/tdcn55v67+1T/32++/XjfYz7jOuP7eP5/U3jg/t4Wz34zi/Xp/U+Vl98x/i7v2tf8/1G7dPvZutXythvf5bhWfr4Xnuf2vtQf/S99Hatt/P6YVxr5+PnccF88Tzn6279on3f1lnM16L93j/XrX2xzuo8L47vi/v87/56tO/FefphP+9Zp0fnaY/eS7DePP5D7X0+16Ee3GM7n2f7fwAAEhEAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEvoDCdkJtV2eLXoAAAAASUVORK5CYII=) |
| `cet_d1a` | 74, 0 | ![CET_D1A](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABmUlEQVR4nO3WS3LbMAwAUJI6RM/Q4/X+I7ELU/IQMmMnWeK9DQY/yptkUP/8/ddLKaXU7RFGLCF+XG8/fKf9br4u5s+5VluYe+Qt9K/YHnEL9a3VuR7yrdYwV8Pc3G+3+fJyvi3m22KvvduL87e98uX3bvu39+b9Kw/z21UPsS7qy37/1VytX/ef740/l1t/3qu37y/6sb6ar+F3Xb+jT/lVj3kZeTlGPser3ldxf8TjkZdjD/VFf8Sy76/rY76Gfgl7cT6+F+fLx/PHt/b7am/EHup9n9+J+/2I83P9nL/yW/9NDPPHm7ll/9y/fb+/3o/fjXvX/Lx/1W95/7DeX9bPfI/98f0Ryn7m4+9ljD3zUqZ4q9ezfs711/U6948rX9Xn7zzf6XNe3uRhfvw7AQAycQAAQEIOAABIyAEAAAk5AAAgIQcAACTkAACAhBwAAJCQAwAAEnIAAEBCDgAASMgBAAAJOQAAICEHAAAk5AAAgIQcAACQkAMAABJyAABAQg4AAEjIAQAACTkAACAhBwAAJOQAAICE/gMI6DWtzKz3hAAAAABJRU5ErkJggg==) |
| `bjy` | 75, 0 | ![BJY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABaklEQVR4nO3WUVLCMBAG4A3oCTyJ93/yXAr4QIrTpDEtjk/7fS9Mtsk2KXT4y9v7xy0i4rVERES8xC0iIs51fC51fB/Gqakv49PjelN/XF/Xy6xe+9RhlLI9rwzWL/XS7KN0fZv6YH03b7r+ub4R2/XR+pjsKyb7Or6+Xbdv/7Fz/z/rB/O6+zfzdp6vu+/kfNGtP3aOrl/3HNpzTPpPnkccfB7j39lf5x1b3+73v+7ffp/T+3Xv7bLf7Xq3r8Pn/b3v6L0cnSN29o9hn2fnj97/7fnjfbbPbe99j/a57uxzHfRZ10s087r+bZ9rU19/Dq/HZd2/XGr9azWO8lnr98/lfxsASEQAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASEgAAICEBAAASOgbNlGcF4KJgK4AAAAASUVORK5CYII=) |
| `gwv` | 76, 0 | ![GWV](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABSElEQVR4nO3WO3KEMBAFQInNfR6f1MdFDrzgQuwgwOtouhOK0YdBELz6+fXRSillqo+yuZbuvk7d+M99XetTty4a7+v1eL9ln7KdXwf7/c5fxuvF/br563g3f93vuL4/j22fcf1cH9f769YH5/G2evCdb9e75/1bffAd4+/+t+dH/2P/v+z6Wvp5Xstu/fPalvFufiubeml9vbyuB9fW1+egHs1f6nM0vwX1dz2vXVof9xndt3vr5tG8drju+vu8Poez32vY33B+1P+gr+B99+8ZnNfp/drF593cJzinu+umAgCkIwAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAkJAAAQEICAAAk9A1KS8AjQwhkowAAAABJRU5ErkJggg==) |
| `bwy` | 77, 0 | ![BWY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABbElEQVR4nO3WSXKEMAwFUJm+VU6Z+yZpZ0PoYFqYDDu9t6E8C1NF/fbyeu8REbcWERFxi/XZenzvX7b+tT3MXy72L8N+y3rOsX//bLPxdF3frWtJHWP/eN5hPPrT+dN1Sb1tUu/x/Gv9bbjf4/h5fY/xfjo+r2Pc//z+Zuu3eqbnzN73b/u04V6y73bcp1+cl5yXjcf4nfpuw639NW9oR9zX5nm7f7X7/bx/ffbY9/dx/HDe8/22det+s30e48/n5/tk48n85P1+Wk9+T+v85H765DtcraPHtfrm9/4/93R4j6S+7Htl9zfvH97jcO+zusfvkez/63Ufk/79M/r7MP4WEY//FQBQiAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQ0CdIxzZ8BkpCbwAAAABJRU5ErkJggg==) |
| `cet_d12` | 78, 0 | ![CET_D12](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABYElEQVR4nO3WPW7DMAwGUClpUXTo/a/UW6mL4cC0WSlpN763GJRpWv4Zvt6+x2ittda2wy7U41h/bec/t/pjq9/3YzvU961+G8f6vp+/Xr/FetvGLZx/1Nn6Vo9j3ZP+uH6qkzl9cU4fydwwp2dzRta3NufZ+7d4/uX7x/0f19vqnBH7167bnyPZ/+z+bfLc2fUtee/z60L/6bmz/tA34vqkfrq/T/rD+X1+vCKsn/ZxfZ99fcTVftF1td+kL5mb7aOP3+c8+iZz4kryvk7zF99X3Gd237jPU5V89//+3tl+Xv0PHvP++D5Xv+Psv1x8z9k+p/9n8p1uDQAoRwAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAoSAAAgIIEAAAo6Ae9fXGdSnhJIwAAAABJRU5ErkJggg==) |
| `cet_r3` | 79, 0 | ![CET_R3](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAByklEQVR4nO3WQZLbIBAF0AY7qRwy9z9DKmNDFgJNQCbjqix5b4N/02DJC1enHz9/1YiI7/kRERHf2nq7tbXllJ8REZHPfKxxWY++2vJ1PfZLGutl7kvTvS33789pfL5by/c05lv+GPO5fgz35J77+037PUdb01zPr/fjPDf2f55/Djm3nKOtqbTnLS3XI0dbW/6sx1C/pRjq175ef7Mvpv7+/V/1RR3y6vvW9yz2z77p/ss9r/fn9/zqvjz9rqvnu+bx3vVz/fvc9f3S2N8+5UhTPt4w9VzvY73O6/1lPcpYj75f8pCj9HNzvec85fmeef+9nEp6sz8t9lfnp/7lPXPfu/e9vqde7nnvXM913j/z//a9XuvxN/VXvT1WqUOO2nNp+2fjWK/Plp9jntZaHmM+9x/TPWNe9dWyqH9xT2398/P2ep3fY3XPXC/z+cdUX60f7Wdpa/nd8rH2/w0AYCMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA0ZAABgQwYAANiQAQAANmQAAIANGQAAYEMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA0ZAABgQwYAANiQAQAANvQHpj0wGIdg8GgAAAAASUVORK5CYII=) |
| `cet_d9` | 80, 0 | ![CET_D9](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPklEQVR4nO3WQXbCMAwFQIXSm/SevXncBQ0UE2GHstPMxs+JLIts+MvX99oiIj6XiL/ruV8vS3ws9+u525+W/bpTPK87JWt/PqvL1iV7H5N1yfusfnno217ql+3Tunjx3GTd+Hk7VP/4nSbPHX6/P9dsn9fr3nPvuP7gd9/qo1vbZY227u6v67rVd8+3+vV+/1C3Pu/b97ue7/rm9yZ1Sf843P9N5wZzt9Hcg77t4FzD+373bXKe9s/7svnb4Pff5pubO71nss9033X/uwzr3nVu23fzbv8bAEAhAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBAgAAFCQAAEBBP95dH6qEN4tUAAAAAElFTkSuQmCC) |
| `cwr` | 81, 0 | ![CWR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABUElEQVR4nO3Wy3XCMBAF0BFxM+m/ijRCG8oC7BNJCNvAyWbu3RiN9RmLzSvfP9caEbFEiYiI5faIJaIZf0X33OqlGV8m89b69r6Utj6ZN9bL4/ru+tuPMpt/H2/vt3qZ1Lv5w/r7uu6c6fxt3N7L8fVtn/O+2n1Ld59768+eN/b//F7Kzr3s9bV33vH15WF9fs5aP3gvw77P/4eh/1lfp581/iq1G6/v13qd1I/O65+roV7bfg7O78/v65/eb5wf7TnT7z+337yvF7/jze8sR/va3a/dp77a18fu83/Oqe/2sb0+9752/V0CAEhHAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEhIAACAhAQAAEjoF4SRv1+amrwxAAAAAElFTkSuQmCC) |
| `cet_cbc1` | 82, 0 | ![CET_CBC1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABrUlEQVR4nO3WQXLrIAwAUIFzrn+8f9ieoYUuQjyDbJrseW+jkQCB0y5U/v3/6hERtZaIiDjKMz5e8azHlD/OPMVx7jjzmPMRj1TP/XP9zFO9vt1X5n0p1nQu97ucW9Trst/8PXV1fvGO1Xtq7bf9aulp/RlL5LxNMeJnxJH3n9vY24j9+++8jbynvM192rt9o29L53L/tuq/eFfL37OoX/ou7sv3tNX7L9+T713tv//eZf3ynnxPm/LW+sj7lPfx7zDSaK2MvEx5T3le/3x/LPbFvG/EnuuL/Non3xP350d9fc9iPfU7+19+x3w+70vvT+s993mtx/z7nXHUI+ptHmXEOOZ8xFKO27zUV33kr/VLfeT1+Oh83hcf9rvsO/PHVF/vu1+P+vf5yPeMvL9bT/We+r5bX79j/jvUM3/G118bANiIAQAANmQAAIANGQAAYEMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA0ZAABgQwYAANiQAQAANmQAAIANGQAAYEMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA39AqgwRuZOAhv0AAAAAElFTkSuQmCC) |
| `cet_cbc2` | 83, 0 | ![CET_CBC2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABr0lEQVR4nO3WQbLaMAwAUDuB+1+07QX4xF3Ezv8RmEy71XsbjWTZDoFhVP/8/tVKKaW0reye5Wfe2sifvfx1rm+x3mOvb6Me9rVYD+dsF31b+7/92+i7eM7X/e/vjedt8TnC53695/O92/aY9I08rj/6fRdxfM+llbOllFJKLese6/19XPa4HPXbni8xv4X+WF9P9bqE9aN/Pa/XsP/oD/m0HvdN6uPeyXPPnq/+a18Nny/unzznqJdYr6O+nmP/Xltdznn/3re2x9Zqzyex/13E/HlVH3krk/X6cf37/Mm50/Prxfq5Hu+b1sP611W9v5eRHzH0zeqPWX4Rx/sa//LR8Wuo53jv8Rbiffmc32Z5qK9HvZ3uHX1jfX3pf1+P/cvFOSMefbW9X6+T/ov74rnT9XDPa9/5/Sxh/xLOH3kd9ZGXUC8AQDoGAABIyAAAAAkZAAAgIQMAACRkAACAhAwAAJCQAQAAEjIAAEBCBgAASMgAAAAJGQAAICEDAAAkZAAAgIQMAACQkAEAABIyAABAQgYAAEjIAAAACRkAACAhAwAAJGQAAICEDAAAkNBfEjtABtjB5uIAAAAASUVORK5CYII=) |
| `cet_cbl1` | 84, 0 | ![CET_CBL1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABfUlEQVR4nO3WQXLDIAwFUGFfIPe/rLtIcAZh4rTTnd7bMAiJ0s6k+e3xeBwREa3tERERbXsur/XcRxv20xqr+mJu1b+YOxZzZ7214fxo131HjP1T3zk/nuefvzo/YjHX+xfzsfg9zn0+/7Zv+vuP862v0ffPvu1V31LfHu3y/L32V92c932M9X3qy/25Hmk+Pvbt7Uj11z6Ocf7sW/Sfc/H6+3zX3873Leqr/r6/mWvp5+d3tZt3t9t7+zvu5r97x+o99/eO75zf9909q3fO90a6N361vj9H+TzX23U9cv2/1u1jffo/MfVvn+uR78n913P5e2i13vXN59ffc3P//qf6et/78/w+9m93932u53un+9N5/9wBAIUIAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAX9AD5rTP83xUliAAAAAElFTkSuQmCC) |
| `cet_cbl2` | 85, 0 | ![CET_CBL2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABl0lEQVR4nO3WwZriIAwAYNDqXH3/Z521e5DWaSiDu9f8/yVfIFBqq6Y+Ho+1lFJqvZSfsXR5fYVynJ+N73k536906z6d3/atv86vk3Ot4Xzd+nCOdXDOfp/Bdbr7CusndePPdzI/eB7Tug/P0d3f/15nsG64/+Cc66wuPt/BezV93pPnPHr/4r7xHN0+3Xv9b/Wzz237Xl7C9/na8i0urW7Z8xhf294G+W3Ly3rMa8yfh/Fln3+exqX+ecWyjbc8zrd4/TC/1u+Qn8dLqLt08+f5KG7X3cdLrHuG/FhfP7xOXzfa5/fx2s1/h/nz+tG6eK7a7vcd1xZLqyst3/5nriHeWlxCfm8L72H8azD/Ncgn45f7ad2Wr+E6o7i28733ux3H97ic5u/1r3z7tQMAEtEAAEBCGgAASEgDAAAJaQAAICENAAAkpAEAgIQ0AACQkAYAABLSAABAQhoAAEhIAwAACWkAACAhDQAAJKQBAICENAAAkJAGAAAS0gAAQEIaAABISAMAAAlpAAAgIQ0AACSkAQCAhP4CDh5lFoEAsxQAAAAASUVORK5CYII=) |
| `cet_cbtc1` | 86, 0 | ![CET_CBTC1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABl0lEQVR4nO3WMa7bMAwAUMpJr9L7771Hb2Orw7dSiLaatFMBvrcYpEhJRoKE7fuPnz0i4tFaREQ8z+e3d88tUj5u+58tbuNnOu9xyUfc3etxyc/xY7Ff7rvE24d1Kb99WDfi7R/3u/Yt6iOm+vNj+l0/1qNHRETrOT7O57nQ53wcaf2MI62v8v2V739Zv9q/z3Uf7zOf3/P+PfWN+jf7rupX9+vv7vHKp7rF+/e8b4rH+vHq63Pc53j9PK8Tc34f+Vz/lY4997+tu6/fY7XPp3Vzfk/nL9/3vzl/cW6k5/gapHyPrx+Eo81xP39Pom1T3M848vp2n49tm/u2tH7pz32rc1Zxvvef+5br430XdTk+FutHji99bao7Ul+O90W85/VXXZvi8V7trG/jfyUAgHIMAABQkAEAAAoyAABAQQYAACjIAAAABRkAAKAgAwAAFGQAAICCDAAAUJABAAAKMgAAQEEGAAAoyAAAAAUZAACgIAMAABRkAACAggwAAFCQAQAACjIAAEBBBgAAKMgAAAAFGQAAoKBfZktY8LeLMhAAAAAASUVORK5CYII=) |
| `cet_cbtc2` | 87, 0 | ![CET_CBTC2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABiUlEQVR4nO3Wy3LCMAwFUDul//+/hcZdYDNjBZf0sdM5G3MVR8kkMKhePz5aKaXU1koppZS+1tLXfZ/qpfW89+OPvIfj39fbHvv9tk+4v9P74742H//l9drivtu/75+fX7yvw/Md/YaYo1rHhynXbc6lbqF+z2Ub9W3ef6iHvMW+c79j/1fXDef3/fVQX/V/fv+v96/6/7FPz23xvMa+NvbV+X3tod76++3fkrIf8t1ne54/+/doD/mxP+R4fH3+ufN+ep2x3k72fbX/9qLvLew7m6+P3Hqe6+vc+5VwP/fl8f7Gr3/8yvu3p1zGWsd6//De6+8j1+/zZZEvJ/PbIc/7Vvl4Xjy+qv81P69vJ/tsp4/H+vPrjfd5yH0d/+8jAwCJGAAAICEDAAAkZAAAgIQMAACQkAEAABIyAABAQgYAAEjIAAAACRkAACAhAwAAJGQAAICEDAAAkJABAAASMgAAQEIGAABIyAAAAAkZAAAgIQMAACRkAACAhAwAAJCQAQAAEjIAAEBCX9h0IysjK62yAAAAAElFTkSuQmCC) |
| `cet_cbtl1` | 88, 0 | ![CET_CBTL1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABv0lEQVR4nO3WQXLjIBAFUDBKZuv7n3QmccIsDKRoS5H2vLehftMikuKyO9/v95pSSiWnlFJKJc3rrdVvPac657A/rntZ6369Xuw/6qsX+1suB/Xc6/XgnJpDPjknhevS7/s5PFdOOfTt10fOPfdzLp6b9+svefyf47k57B/0nf2dcP3rc53c99W+8TzX+m55/3lf3vfVvqP7PXn+1/d8dk7ozwf5FvPFtcTcDhpfJM+ct7lee99Wpr46+sqUa8+h/j1ymXOZ83fZft3/Ougb+7f9/V7/6vWQf/Z7/7P+CH0jt/UR6o9w3Wfb/xz7rd76PkYuLc/1uP6LOff68z3/HfXbVB+5rR+t/mifo0er1/C5Ki1vbX1LT+8t/8lzfm951NNc7+vb2J9zrG/tC3jsj/p+Pqtv7XenjLzfV9JJbvfVr4+/xz99R+fV3f0Srovn9+8bAGAhBgAAWJABAAAWZAAAgAUZAABgQQYAAFiQAQAAFmQAAIAFGQAAYEEGAABYkAEAABZkAACABRkAAGBBBgAAWJABAAAWZAAAgAUZAABgQQYAAFiQAQAAFmQAAIAFGQAAYEEGAABYkAEAABb0H1mRXBo1LBe8AAAAAElFTkSuQmCC) |
| `bgy` | 89, 0 | ![BGY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABa0lEQVR4nO3WQXKDMAwFUJuw7JF79UAXATqWoSbtrnpvo5EtjJtMOr+Wj8+1lFJKmV6lbvXoa/l5f+9r29895/Z6PP/mfne/q7m/1nK+Hl7fzXf7F/X23Hqxvw7m1vO5vZ/i/mhuuZhfL+ZjXW7ODd7XPTe61/Lmc1fvjeu/7I9z2v1p6+etPi766aKf6rPtp3a9hvl6zMf9Z7Nfw3w97j3ql9P9Y737HM5r7T7PN7+3rdbR7+dm7c958/m7v/tfvm80H/+dPfZ/a6Gf9760/bw9OA/ntnp3vZyf28+Fex779XS+u1dp62P0XDi/X4/z7XndPUs4J/5d8flQv7/IErwWprgMAPx/AgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCAgAAJCQAAEBCX13PZ0KxoFhaAAAAAElFTkSuQmCC) |
| `bgyw` | 90, 0 | ![BGYW](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABjElEQVR4nO3WQXLbMAwFUFBOnZ6h982xY3UhkR5SYmS3S7y3wXwAkp1MMoPyJ77WiIiIJX6spex17E/mhzqb/2//X/fKe3tL3y+Xz0/q0ufje6724nQe7WsMe8N+aXmc19cNfw4tT2qbP/b6vdXblteaJ7XUXIZc50O/HPqP872ay+PN/mzef07MPn98/sX9KD+/5zh/nNdl0o9Jf69Ly+tkvr6XYzZfu8879ON8Pt+b1Mt5DP14bd76/fy2/2Md5+NzW+M25OfekNt7r/qz95SL/fN+me4vr81jsr/n+lypedhr/Vi6fNwfatu/df0Ycp0/+zV/7L/A2v/o9ur82a/515ZjyMP82e9zifvQv3c14n7R/9yf33PU/Hmao/zefswAANJxAABAQg4AAEjIAQAACTkAACAhBwAAJOQAAICEHAAAkJADAAAScgAAQEIOAABIyAEAAAk5AAAgIQcAACTkAACAhBwAAJCQAwAAEnIAAEBCDgAASMgBAAAJOQAAICEHAAAk5AAAgIT+AvpPZSuKInKNAAAAAElFTkSuQmCC) |
| `bmw` | 91, 0 | ![BMW](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABP0lEQVR4nO3WQW6DQAwF0BmSRm0XPV/vfxG6CEHCjcuQNurC720sG0MiEtDv7fw5t9Za61O7Wmo/te089oPzeL2WfM7wecn3+q/90fOz+/vtfsT7mMyzvfV4H5z3be3JfPT4unfr29j+unbt35f5xzJ+W/rXpb8s/WXtr15CPSf9Xj0d7eft/GiN/7a949n+3t7OU5ruTT3s9fvzKcx73EvqujcNzpM+7ve4P93f73H+YI2P595jm/YHXw/5D/ekmrxeDu89evy382fv/dV56XXmH/fmweveWgCgEAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgoC9dzQcV+479QgAAAABJRU5ErkJggg==) |
| `cet_c1` | 92, 0 | ![CET_C1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABuUlEQVR4nO3WQW7jMAwFUMrO0efOU0xtaRaWHEixm3St9zbEpyjXSIKC6evP3xIRkUqOvu5djpoj136tUbajtJyPHPte+1vXH+fKvl3PDffP871/zni/vLmfX+b7uTy8Vx7m8jjXctmG80/nr+duz+v30M7LkNt5idqvtaSfayy5q2Xtc9SczlxqLn0++9HnNYZ+/Nz/NN/045f349P+Ojz/Zm78+5/OP++lod/yclOPB5b2wOVR8zrUR1dzq/WFcjwu616u82s9nrO96W81n/3c919qHvvH/Pdw3vKzHp/H9835v/xzbvfO/lC3oe5DzXv9vGsue/3ia077dV3qvWfu69h/5qWfzze51rV+PktJXV4j9TW1HH1ernP7ua7LMDf8bH+b382t6a5//R5t/pGu589+m3t3ftv/7N7r/FDjXf/4f7tEqzHkfHkOAEzEAgAAE7IAAMCELAAAMCELAABMyAIAABOyAADAhCwAADAhCwAATMgCAAATsgAAwIQsAAAwIQsAAEzIAgAAE7IAAMCELAAAMCELAABMyAIAABOyAADAhCwAADAhCwAATMgCAAATsgAAwIT+A1PdMbMk5Q+IAAAAAElFTkSuQmCC) |
| `cet_c1s` | 93, 0 | ![CET_C1S](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABvElEQVR4nO3WQXLjIBAF0Eb2uXL/48w6kWAWArnAUuSZLe9tun7TKI7tpDp9ff0pERFLSRER8SjLXiP1NbUcfV7O85Jqfxnm2vl/5ru5R7rqn7+ONv9M5/NHv83dnV/2P7v3Pj/UuOuX/fePVmPIucup5Iu6dTlqjlz7tUZZ99Jy3nNsW+2vXX+cK9t6PjfcP863/jnj/XJzP7/N93N5eF15mMvjXMtlHc4/nT+fuzyvn0M7L0Nu5yVqv9aSfq+x5K6WR5+j5nTk+v1ZSp+PfvS5fkFf/fi9/2m+6Mc/3o9P++0P7WZu/Pmfzr/upaE//GN7q/sDS3vg8qz5MdRnV3Or9QXleJ7WrZzn97o/Z73przUf/dz332oe+/v8z3De8qvu78fPxfl3/j23e0d/qOtQt6Hmrb7fNZetfvA1p+28LvXeK/d17L/y0s/ni9xqAADTsQAAwIQsAAAwIQsAAEzIAgAAE7IAAMCELAAAMCELAABMyAIAABOyAADAhCwAADAhCwAATMgCAAATsgAAwIQsAAAwIQsAAEzIAgAAE7IAAMCELAAAMCELAABMyAIAABOyAADAhCwAADChv9kQMaYKnU5sAAAAAElFTkSuQmCC) |
| `cet_c2` | 94, 0 | ![CET_C2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABzUlEQVR4nO3WQU7EMAyFYSfpFbn/ERD7JmZROyhuMwNs/X8b6zlOBtBIuHx+fKmISKkqIiK1DhERKc2rhtyXLC3O96Uvcf7oz32/V9fz+M6t1s153dyP79/yZn5375fnZebzzTvXucb5Wbudr30t59Lf5eH9cr0zZO0P6Y+1e1bP19+p67C+ZVHr/7aK3V/zPB/y3Nf/9c/xt/l5L/bDO+fmndiPn3/O971f1v6Ifa/Vfo6QtVn/ysOzVR1t6es4rg+wLJaLV+t7ruF8ZvW8zt/vhXO7J6G/yxJy/Pk1zOlooV79ceu30F/n4vw91/W9Ht7v8XzN4nneq9Zf5+OchL7XEqu9Vy1X+960mO37ddj8zPZ9bGI51HYVOcrr3GrMVn1u5t38muvuHT/f9N/XMC/++8jrvuzO9fH82Mz/ZH19XtbzOrP/P7e+/3+ffasCAADSYQEAACAhFgAAABJiAQAAICEWAAAAEmIBAAAgIRYAAAASYgEAACAhFgAAABJiAQAAICEWAAAAEmIBAAAgIRYAAAASYgEAACAhFgAAABJiAQAAICEWAAAAEmIBAAAgIRYAAAASYgEAACAhFgAAABJiAQAAICEWAAAAEvoGEcBIcBlMb+YAAAAASUVORK5CYII=) |
| `cet_c4` | 95, 0 | ![CET_C4](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABtElEQVR4nO3WwZKcIBAGYFDz/k+Xqj3lRXaEHEQTGNk1ufb3Xbp+GvAwo9X518+PmlJKuZbU173PZe/W05n3PqfyGtaPnM68v/r9+6Tfcp2dP/vj+ba/Dvv/5OFcub+nTu4f19/vve8/Pvf67p4yubdfT6Xlq9ZjvU5qyl1JObfSFpblyK2mteWrrl0/b2u//la3Pk/3b+15z+5Js/uv/rB/md1z/9zr/DLb1+ervzzrj/el5ch1yGe/Lv35OtY81uP3KWdOLQ+11tyvt7yX3OX29/qr33Lp+9f6Pu5L93l/1h+ftw/7Z/dM6/5wX6uvWd77/Li21/GznLV2/c+xP8uttq/XdW/bdjlf9/ZWpy339cd3denr1r4Xb+v/WddJHuu2ft2f1eXhueuzt9b79cl9/9xf6pf7rn4++7Xv5yPnXLu8pHO9dPn83QGAQAwAABCQAQAAAjIAAEBABgAACMgAAAABGQAAICADAAAEZAAAgIAMAAAQkAEAAAIyAABAQAYAAAjIAAAAARkAACAgAwAABGQAAICADAAAEJABAAACMgAAQEAGAAAIyAAAAAEZAAAgoN8ITCvK41G7MAAAAABJRU5ErkJggg==) |
| `cet_c4s` | 96, 0 | ![CET_C4S](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABsUlEQVR4nO3WQXbcIBAFQGDGl8j1fPXcwoPIQkgxWNhKtl216fdpQIuR5nX+9f67pZRSSbtnHuvbT7WM9Znz9fp/1sciz/X5+L6/quXmuXLkR7teX9z3z/3Svt139vPRb2M/7znnNuSSjvVtzGnPuc21jnmrw3o6ch1z2l7T+p7Tketr3F8X/Z7b6vzRn8/3/W3a/zdP57bre9ri/nn9673X/dvnXj/dsy3uHdfT1vNZ9987tUVNeSipf8e51+MFy9MLm8/6GPq5f5Dn+pf6HPNy/7M/7949aXX/2Z/2l9U91889z5fVvjGf/XKvP9+Xyp7blI9+K+P5Ntc81/332Y7c//m3qbaWx/We65aH3F+vT/2et7F/rtd5X7rO9V5/fl6d9q/uWdZ6c1+vr1WuY75d++f4sR21Df2Pub/KvfZ/r/Pevu1UEgAQjgEAAAIyAABAQAYAAAjIAAAAARkAACAgAwAABGQAAICADAAAEJABAAACMgAAQEAGAAAIyAAAAAEZAAAgIAMAAARkAACAgAwAABCQAQAAAjIAAEBABgAACMgAAAABGQAAICADAAAE9AdDMCueOIHbwwAAAABJRU5ErkJggg==) |
| `cet_c5` | 97, 0 | ![CET_C5](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA/klEQVR4nO3WS26EMBAFwLYvze0hixmiBAXxCU4Wr2oB6nYbLI3EvDZN01JV1Vqrqqree32tR/dHzz227133Xz7n7vn23n/2vrXtL8vy49zav3qf13qev/cP6rNzl+ub53nqfH81N2r/v/d3fr+jfXffd3b96edcnV8d9S9bvx+fZdss3/sO7X3vRs2NWt/d19/rbez/6FP7XlcAIIoAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEAfI81qoI9IZO8AAAAASUVORK5CYII=) |
| `cet_i1` | 98, 0 | ![CET_I1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABOklEQVR4nO3WSVLDMBAF0FYOxdW5FHcwmzhgC9lyCMXiv7dxaeiWXBnqt7f3j6WqqtVS3591Mm5VU/se47Zfn6s/v89xn2qv6fP8+H6N7v2Px/Pn/NW9jz/H0/ed/h79bt/V+m59ebLuRfva4B7r53q5X9dn2+92WleD+nXfaH03f+936/rt9/3Xep2897p+tX62bjfu7rt91mj/sN/xfN9vrk9d3N++/UNd698m7//zvv68dnzeslt/nDOaPzt/MD/ot57/Vb2tH92zu8fonEef7fz6+wQAgggAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACPQJrvK9IRlUVWQAAAAASUVORK5CYII=) |
| `cet_i3` | 99, 0 | ![CET_I3](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABTElEQVR4nO3WQXrCIBAG0CG2J+j9D9ID9RzpwoqfxBHU7ua9DRImA+Rz8bev7589IuIzzj762CIi4rSf56e4Hbdkvv29t+3j88dj6/Xt9vm0/rbuWt/uvn9571ifPO/17e56Xj8Zk77ZObJzv1y3XN/e2j8W6/J++1Jd/NP3ufaZ7Jv0ib6+du427BeT9fzcs/Mm/ce+vSzpN+wzq0v3m+1z+R7J/Xvfw72z7zg+X+t3OF9yr3nd4/3SusX7jXWr5znMD//roe7pPs/de9bn6fW+3+VX8p1m63367vpr9/n/c5znWwAA5QgAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABQkAAFCQAAAABf0CZDZ4XPiPsegAAAAASUVORK5CYII=) |
| `cet_l10` | 100, 0 | ![CET_L10](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABOElEQVR4nO3WQXLDIAwFUMhdc57se1mySOlMwDJ243aj91YBZMkeu51f71+PVkoptbzU/qO08rbf14vzrtaz50PfOswdz4M+P/2Xddvzl/c93V8wb7ou6r9//fhe5rqT+0Gf27JPr1udr/oP67ZfFz3P0br4PKgf3mtZ1IfnQ4NVfZnmbn/Xy7pp7nh9G3c27zdqEM35df38QN/L4EUc3q/Dcv5Pc81+vai+7wdf9uk+7+vW/2I+7BOf34L9a58jmtP+ac7Rfu2P57TD9fvfUz8FABIRAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABJ6Ao+XLqt4s3gaAAAAAElFTkSuQmCC) |
| `cet_l11` | 101, 0 | ![CET_L11](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABPElEQVR4nO3WQXLCMAwFUDnX7CF6sd6RLtokUyfCbpKd3lvAYCzJmBn47fPr4xURsbQWERE/j/vz0r0er5/3OdS1a/Wj+a2rT/cl8/s++Zy5+un93Xlm7+Hu3Disd/XDe/rf/OO8yXN3DZ6et+1LPu/e53xeNie2fsm87HWb3H+4l/OL6usOhdmAvMHcgbbl5JP0F5X2aYP1vk+yv7Vk36h+0Hd6//v11/qLc7NPPNZnXV9+z3e3T3LOx/r9Pe9+rxf7HN6/dt7x/ME/zWBO/r28P+/6LgBQiAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQkAAAAAUJAABQ0DcE3wjVnE8rywAAAABJRU5ErkJggg==) |
| `cet_l12` | 102, 0 | ![CET_L12](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABL0lEQVR4nO3WS3LDIBAFQJL7H1YnCNm4SmXQGIlk97o3LjwYButT7+s4jt5aa7299LeP1otxK+t9UR9+X8xb9XHW+2afi/pQ2O2z3qdf1x/v0xf1VR/P1vlrH9N1u9vfsMD6fMP6Zf99Ub+773W96qe6f8f55/h9/tjndJ8W31f389h/eZ7FOarr+/R3q/PN5xj/r2G8ue7uuN737vVejYf1p+33rt/8Pvrc72qd3efg+fviun5O+/z83n6Pbr4Hqv3qfv533k+x/3cDAOIIAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAj0C3Xea6Qd2QBNAAAAAElFTkSuQmCC) |
| `cet_l16` | 103, 0 | ![CET_L16](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAB0UlEQVR4nO3W3XIbMQiG4Q82aZu0nbr3f61e9cBIqVgpP8e8zwkDAtnrGdvY7XZrkuT2JEly9XhIkswiKsenKdelL51v5rU5VzrPr5PPvxqbberpddpmrsmXfc083ePLesv1iOfoS+fec0tzto4+5xq5plwRWsr74yn1j/lt3xfzS70t+8xbmut9qT7yFs/VNv35/Jxex1LukXvMuZ9T/Uh57rNRvy/nx/nov6f3Mc9bmjO/x3PkuTl/O4/+MXdu6ut75fl9rO+5vs/1c799Pp/L8+f80fmufuhz/celvp7/uC/y3f1qU76r96/r6Huk/9X1br1/vY74ffBLv6X+1NfnZOmeFC/nnu5L9TR3rfv7c+P+uW6jb46W7s194//PPMVcj9z77378/13ic5w/R/5tGWXfI/8xRannr9H3EvnPyB+x6Vfkvx+5/ZEknYpof+O5AQBAOSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEH/AKzkhxoBEjQrAAAAAElFTkSuQmCC) |
| `cet_l17` | 104, 0 | ![CET_L17](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABkUlEQVR4nO3Wy27cMAwFUKn//7NdBW1m5C4i2R3ajDtAdzxnI+iK1mOQAOxj27bWWmvbs/09bnHe1nzcrK/vx+V+t3Utnh/OS9bbfMYxfzMf/2mfm3z93D3WjS18F79/zfvpu+u603h65/W43y+91xpbUtdu3hfW17njOr89J71nXG/X41zfTnmsj/sm6yHfTvXv7XPcqyd5OG/0JI/1PcmT+v3cnuTf122n+/fLd6z7x3cf93ldj+/N82Sf079FD+P1vdK6t7+P97upG9+fe/y5r3moi/n8OUa49/Hnuuav+xzf/9u5I6trr+ft8x7maT7nPdvnOn9m++zrSR7OW/s8474hf+zz9rL/mj+S+sdp/uVz5p8z/93W+BX8mvnHzH/O/GPmPxoAUI4GAAAK0gAAQEEaAAAoSAMAAAVpAACgIA0AABSkAQCAgjQAAFCQBgAACtIAAEBBGgAAKEgDAAAFaQAAoCANAAAUpAEAgII0AABQkAYAAArSAABAQRoAAChIAwAABWkAAKAgDQAAFPQHIZaiD2T8B4wAAAAASUVORK5CYII=) |
| `cet_l18` | 105, 0 | ![CET_L18](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABSUlEQVR4nO3W0W6DMAwFULP//+SN9KE0VdM6AW3SHnzOC4I4jpk0ere9tRYREe3nfon7Ndo+vW9HfcT4fD+eJ3X9nKNuuE/7L/bF8RrRz2+v8zzWh/o21Pf3TJ73feO5p/t+rs/mz/t9niudd3Xu2/zz9dX+9/XkfZL1P+9/9px/qmtX+y3r42J9sv/iPM/3SM5fzpftu9gv5vvasu/4vA82Xz953365fzXP+GfI+83Xx3+/9f5Fn7NzJJ+75bnJZ2Xdf5uup/1P18379897UvdcH+qG58PP5uNnuF/b9/Zy/xUAQDkCAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEECAAAUJAAAQEE30pmoPtvhI2kAAAAASUVORK5CYII=) |
| `cet_l19` | 106, 0 | ![CET_L19](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABYElEQVR4nO3WMZLDIAwAQHz//3LgijjOIKzYSavdIgyyJOCKG219jNFaa/vSnr+t7dvWw/76+7hV35N+6/d7/Y79Rd/3fvux7vN56buSeIv7sZ3nJX3uxuM59+PbaXx90Hl9/L6s2X2XB93rF98f/5BbFl/qYv5F/WXf+YKx73V8TGsen++zxrN3fe6f1ver8/vHc951z7ytZ99jvJ+en+Vn/de8fd/7eX18T4/v6yH/9Y8r6de/65/Xzee05R4hnuSt5yR5aXx8mZ/dY17Hz/HHdM7dPkteUv/ajzaf99of65G332eJz98v647v8bxHiJ+vbV//GgBQjgEAAAoyAABAQQYAACjIAAAABRkAAKAgAwAAFGQAAICCDAAAUJABAAAKMgAAQEEGAAAoyAAAAAUZAACgIAMAABRkAACAggwAAFCQAQAACjIAAEBBBgAAKMgAAAAFGQAAoCADAAAU9A/vibs9PPLKhQAAAABJRU5ErkJggg==) |
| `cet_l4` | 107, 0 | ![CET_L4](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABE0lEQVR4nO3WQW6DMBAFUJumvf+Nk0XDIkaWRxQW1X9vM/I4DEgQ6/fW2rO11n7ar71+v+tjqF9D3SbrVe3FfnV99rrb1v3aeZdf14u/K86b9vs1c2dzyvMW7+PP1y2e7/T+7P6z9zf2x+v70B/vuw397bNu7z96n9T9oDj0i+tDfzxg9v3ZwTPuFw+o6bxiPdyv/GEv5pY/1Iv3J/3D85w9qO4++O6a88/Wq/e1twGAIAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAr0ALKYDLRVdxgMAAAAASUVORK5CYII=) |
| `cet_l7` | 108, 0 | ![CET_L7](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABTElEQVR4nO3WQW6EMAwF0BiO1/ufhy4mUBHwhKm6qfzexkpsV6WdQT/a8rW11lpry6tEr9l5Vi/z66O9SO/v96/z66R/npv18+fL+vf3W0x+biS/V5yf46jj/XBefnvudbnMxemc3h970Z/mfN7763Huc31/Pc5Df6yR3bf7+za5v+y1pM765/rzX73fyz4Vn87ln654OLcl973GsBdDTe5jnNvrkswtyd7wNUnn+0LM5rMak37ytb32t0d74x88e82mr9/pa3V72//0tX7U+KO5tB/v++P907mn+zF+sJ/u/c/zfgQAChEAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAAChIAAKAgAQAACvoGoVsIFfDXCBIAAAAASUVORK5CYII=) |
| `cet_l8` | 109, 0 | ![CET_L8](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABeElEQVR4nO3WS47jIBAAUIrZ9wn7/sdoZhGbCAyy0zO7em9j1Q8SyYkqytd3K6WUErW8vJ4RMcRnPUoc9bP/iKe52uNx7szXGOdqn9vl61Cf++/r6/uv9dfzT//2Zzz3lSF/7T/qrUzzc36M5/r7niNuu7gN8e6cuV5bW5537V+ff86Pb8//yG/6jnw87YtNX0znxM3cEccU93N2/THFdVOv4znvvjbE/Z4658efcWzql7npWeb77vrrz+ZznPWHcXzWf50rj/rLpn/74j/tj9/m1/fFbj7W/dv7pvep12/OiW3f+vPHtq8Mfbf5Hsc/9t3kj7jFOn+dX8dP59s83/s/67u7r+3y7z+c4fyzDQBIxAIAAAlZAAAgIQsAACRkAQCAhCwAAJCQBQAAErIAAEBCFgAASMgCAAAJWQAAICELAAAkZAEAgIQsAACQkAUAABKyAABAQhYAAEjIAgAACVkAACAhCwAAJGQBAICELAAAkJAFAAAS+gtwxXVpiNAZsgAAAABJRU5ErkJggg==) |
| `cet_l9` | 110, 0 | ![CET_L9](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABs0lEQVR4nO3WS27kIBAA0ILOb5JR7n+HOWPas2iwBDZxsq73Nqg+QMuxnCrP8W+LiIhaIiKi3OKhraXlY5WvPW71fk6Job7Mlx/Wy3Tfoa9M/RfnLOtX+3/Xt/2wf9W31W1x7zbuq/2cuX8b84e+uT7my77ex7j21+be1m1Yy5yv96G/lHn/dP6cr/cpP/+u362xn/t1nj/0f53WL/N1jOd7jvsWvzsu6hfP4/C8Ynz+8zmrv9O8r07vyZ4/nD++HzXm92bqW9yzzC/PPc/fLvvP433f/jk6j289Hj9bh/i2qu/5cnHOXJ/i9r1b95VF33lcln11rK/2t7753rL3TedEv7d+e0457Kun+T0+1L9f+z/GQxw9/9TyT4v4eVijx/HS8i8t/zrG0eO3Vn9r+3r8p63vrf+xbuXjsUZby98Wf0ZExL18tucEAKRjAACAhAwAAJCQAQAAEjIAAEBCBgAASMgAAAAJGQAAICEDAAAkZAAAgIQMAACQkAEAABIyAABAQgYAAEjIAAAACRkAACAhAwAAJGQAAICEDAAAkJABAAASMgAAQEIGAABIyAAAAAn9B7BegBqkXS5lAAAAAElFTkSuQmCC) |
| `cet_r1` | 111, 0 | ![CET_R1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABvklEQVR4nO3WQY4bIRAFUKBt5Ti5/8GyG5ssGrCoNjNezI73NuhXAY2jiVQ5/f1XU0op3b7SuT7O9ThzaflPy/fjzMdY276WczlzDjm1nHouj6leQz3m69q+03LOjymXUf+a8tH2lTzvi3m1L8d6+G7sX96X59/R35fC/pFDf/z+Vn/dF+7J7/fl+O+S47ufU/1Y5CPXVj/XI/V6mvq3nlMK/bCOfg79/H1/US9pcc84l9v7v/9Oubx73ld+6d7y477FuRTeE78/6qdc53te9fc5rmnkOufn3E/Puf+q91znvKqv7mv9uqh/en+N91++2/d9+N4P762xvvi9dfGeS/9y/+L7l/fl9+9o9VrzvG/kMufa8/kXl0Puf4m5/0X2fbn3b3M/t7xYV/1RL71+f7s/lXjuPp2L94x+OJ/C+Us9nP+5/kv7Wq7hXB3nzrX/fwcANmIAAIANGQAAYEMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA0ZAABgQwYAANiQAQAANmQAAIANGQAAYEMGAADYkAEAADZkAACADRkAAGBDBgAA2JABAAA2ZAAAgA0ZAABgQ/8Bzem4t1Su2QsAAAAASUVORK5CYII=) |
| `cet_r2` | 112, 0 | ![CET_R2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABfklEQVR4nO3WQW7CMBAFUNtAe6De/1jdFdIFNpInWARVXc17G2vGYyepqPRr+freSimlnK7l6dpuoY77Ya7F/h/n6pv7dfGc0a+x//ye2vsfff3s+5deX3p97vWpr+feH3Xb1XN/1HVXz/3DdfiuWufvr/F7F2uc2+rPoXMlzNVdPffj+8e/y6mMfv+Z9rWVUZd5vS+l1Rr6c93KOFf7/Dx3uB7n433xPUK9uq/tnhfOHby37b437r+45/BzV/f/7X2Wz1ude9GvJfTvP59Hv45620LdB0J/1GXM3eb6sX9bnF/Oh3sX9+zq3b1vPj+eezn3X/eE9z46t6xfzL373Lgf7t1W+8v+sbntGvf7GvuhfpxbrOP/AQBIRAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABISAAAgIQEAABI6BdQK+lbwcvohQAAAABJRU5ErkJggg==) |
| `colorwheel` | 113, 0 | ![COLORWHEEL](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAB0UlEQVR4nO3WQXIcMQiFYSTNRXKB3P9MPkNLZNGgFPRo7GTL/22oB5I8drls2u9fXyoi0lcTEZGhd32tHrPKncVyquMu8mqf8+g5W/VzO5/Ox9xP7/j80P++pvPi34987stprm/nr8P5v1k/z1uc953v6t9/7xrmO48lIiLNc7fs/aEpz5Bl5PMz9CWff833fb/X4zy/86j9MO+H+/n9Rz6cP9374bztfH3zzj3XfH7XafPY13aF/ikv77f7nSWxv2S+rdOzer5/TlOX9S3b7+vUn1ax+zHv+ZL3ff2//rX+7fy+l/vpnevwTu7nr3/t973fYn/lvtdunyNlHda/8/JsVdcIfV32F8iyWG5ere+5p/nO6jmef95Lc7snqX/KknL+/JrO6Rqp3v316I/Uj+fy+Wfu8b2Z3p95HrN43vfsD/eM5/M5SX2vLVd7r1v2//d2CwAAVMICAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABTEAgAAQEEsAAAAFMQCAABAQSwAAAAUxAIAAEBBLAAAABT0B2g4R3WsBTWkAAAAAElFTkSuQmCC) |
| `fire` | 114, 0 | ![FIRE](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABRklEQVR4nO3Wy07DMBAFUCflKSEB5f+/FRZMFplqSFLoAs05m9HYjpVGdXKnMcbnGGM8jG+PUe+j3kU9bdS5qHl+2lif5/9dP637vdfdfN10cP0v9792v6PP8a/uZ+/zWdZdvU91X8X4xX55XdGX65b5ueijTqf1+NLPp3U/Up9rnh9p3/wiyevG/PN4uW7rRXR0fqvm33X0j7u37t3n2gN2sxfKxsHZPPDV9cV43q86EOX+VZ8PUuovxqsDFF/WOb6801KfYvw5+pd1nV/jht5i/D3qOepHzJ+Lur5ueRwAQCMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQEMCAAA0JAAAQENfGrEELExJpMQAAAAASUVORK5CYII=) |
| `isolum` | 115, 0 | ![ISOLUM](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABM0lEQVR4nO3WQXbDIAwFQJML96w5EV0kznsGy4CbVTWzSQxCiN0vP89atymvslLq4bvdj38Hde++5Uv9SlTXzX/et9y8vztXzuvH/dfm6ff3g6O6dn8L1hfvDfbn+5zPM3t+/D0352zf8fq2VP/Zr9f70TvC33o8N6zf62o0b9CnButd/cuje2dT1/ULvsP37Sb7ffZn75md9/p8P9+o/3XfrdsfzdXOc74+miM+t3rPcX1fud/v/W/1nq5/Gdxbbq13/rx+fG+7/jjvAgD8ZwIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQgIAACQkAABAQr8PrbFhiNpukQAAAABJRU5ErkJggg==) |
| `kb` | 116, 0 | ![KB](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABIklEQVR4nO3WwXKDIBQF0Ef+/5/bRTQdNRawZtG552wYH4JCNN5WVV9VVVVtaR7Pth7b46Vt+/rruJ2c197Pd7heW07f9+/Ht23/Um/7+uG8s/p23vaq18n5nflP+9fy6Ljq9O/razN3P6PrOOzL9D7/Pq5dHNf/PdbuyXGD93HY7+5+1kl9dD+uzffzfl18/jr7NP7cfWofbnoPL17ntufrw+vvzjdZ//u63z8Xw+/Vf1/37P/Ibet9fvfWry0AEEQAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQKBv1hECm0Y4FloAAAAASUVORK5CYII=) |
| `kbc` | 117, 0 | ![KBC](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABcklEQVR4nO3WS26FMAwAQJvX8/X+t0kXBSoMEdCtZzaR7Xwg9KnOiO8RERG5xK8yZl7nJ2NO4nP+c1Mv47jb99l+9fz581/Xo9THfk/LGh/XbfWRnxLf7FO+R52/f6/JmDXe3m/9nlnmzcff+cvrfDlnyUN9ns/rfIlji/c/0+O82IYyL7LES0zWHeuRdf5xv9ufzV18yuf1vKzzjvG4OW/U53667uVzPM1nzd/Us+z3tP52n7/zx7/e65S/u9dSr/l6/6Pk6/er9dm6WMaj8+p7zJ579t71vmO917ffb/+5rfOWdZ/Pmp+NX4/j8XL+GketXz9XXbf9tzjlZ+ujzrver+4zO3e7bgCgEQ0AADSkAQCAhjQAANCQBgAAGtIAAEBDGgAAaEgDAAANaQAAoCENAAA0pAEAgIY0AADQkAYAABrSAABAQxoAAGhIAwAADWkAAKAhDQAANKQBAICGNAAA0JAGAAAa0gAAQEMaAABo6AdUXCbVx2ogHAAAAABJRU5ErkJggg==) |
| `kg` | 118, 0 | ![KG](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAqklEQVR4nO3WwQ5EQBBAweb//3n3uhchEmG9qosMZprbW2bmM7+Wi6575289P3r/7Htn11fv+7fvuHvfW+a/7dy75jxlrv+wfvB6HQAgRwAAQJAAAIAgAQAAQQIAAIIEAAAECQAACBIAABAkAAAgSAAAQJAAAIAgAQAAQQIAAIIEAAAECQAACBIAABAkAAAgSAAAQJAAAIAgAQAAQQIAAIIEAAAECQAACPoCh7cBrihQjwIAAAAASUVORK5CYII=) |
| `kgy` | 119, 0 | ![KGY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABKElEQVR4nO3WS5KDIBQFULDjEnr/SzUT7apgEfwNuuqeMyEIPAQzuLX8zksppdSyWn/Upm2f/03vzevUK4N5h9vBuqW7brm238F9e+fs3d/59e3Fndz36PjBuufr1+/jpXwfP3ovd8/f1Lt+v/Xm+m382vmeP9f/uJc6PfW9B//H0fPpmfOM1tfdfg+fZ1f35nmmbfzc++zf63P91Ixv/Z+mfXX6r9L0e+06bx6Mb/15VL+t1523fNZt5/X6o+fb/k1/++wAQBABAAACCQAAEEgAAIBAAgAABBIAACCQAAAAgQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIHeaxIGGujrdn4AAAAASUVORK5CYII=) |
| `kr` | 120, 0 | ![KR](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAA8klEQVR4nO3WQQ6DIBAF0KH3v3O7aOqChmgR6+K/tzHKYEbE+FtVPauqHvW2HVt3PrjeBnWtq2/99W5+P96Ojs/WD+pqVN/N26v7Gj8572h/2/jOOgzv86c+W1cw2+fq97Z6He/u76rnub3+5r6Ofger3t+v382q/bj6uWf3+2X9LN4PV6/79DoO7vv5DwMAQQQAAAgkAABAIAEAAAIJAAAQSAAAgEACAAAEEgAAIJAAAACBBAAACCQAAEAgAQAAAgkAABBIAACAQAIAAAQSAAAgkAAAAIEEAAAIJAAAQCABAAACCQAAEEgAAIBAAgAABHoBLs8CPwhb2mEAAAAASUVORK5CYII=) |
| `black_body` | 121, 0 | ![BLACK_BODY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAABbElEQVR4nO3Wy1LDMAwFUDlQHsOC//9ZwoK4TJW4bmCpczYe2bKapgxzW0SsERFPrcXPGhER8ZzqZauXre79bXDeeh2pTvdGfdvxb/+ojvQccdx/nf/wnPv7LW6/727+yXln517fc97P89rkvNej9zSYP5s7+z679zG4/9fPPTt3eK+/5yWtbT3en51f56139yN/bp7X69N9k3p4/3Y/dvujOetgTq+/Htvf6vyDtf0fXOrLz53P+/Me7+f+2Zx9/3qqP//us77T88b/eP5XD/uG/8gmc2b3Pra2z23t9fu2vm3r62B9mayX23q5reN6fjm8tz8/3o98b5n1P7Yfy/Hz5LW/TgCgEAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgIAEAAAoSAACgoG80FiEsqKij2QAAAABJRU5ErkJggg==) |
| `kindlmann` | 122, 0 | ![KINDLMANN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAB7UlEQVR4nO3WPW4cMQyG4Y/y/NiAkSJtDpH7Xyplini9k0KUAnFGGAMp+T4NQYqSZnbhNU3SIUmrFknS0qKN+Ytehlhm0Wq0ULe7aGWyPtal2Fdz2fU+Te7T5N7Yf87D/Rb7ZuddP/e//ZP1cI78+5CN+dHjKkl6evy0TZL00C5J+uPxt71Jkn6V11p/fdb+/bMeuz1qXGs8PGr58Mdp+WPIj1YvY73nvV7vOVpufm+5idb2PcO+lj+v81P9qOdM6j3O6n295Xd9dd1O6+3+SX3aF9/v+nnN19t72u05N5/ff/eNz3Xu++Lnefr8W1SIrW/M599Du/cxxM37dj+3xdXjFvIWl0l91fX6Kddk3V9w6fXjsh7vOZ03yU/7v1jv58X12b1x/bChHuN21N+zop+SJLMfNeq7599qo717/uZ5i7vXd883j6vX/abSfmcnsYTf/xZLyHs9/J/qeYh9fwn7Yp9d9tmpr+V2eV77swAAAIkwAAAAkBADAAAACTEAAACQEAMAAAAJMQAAAJAQAwAAAAkxAAAAkBADAAAACTEAAACQEAMAAAAJMQAAAJAQAwAAAAkxAAAAkBADAAAACTEAAACQEAMAAAAJMQAAAJAQAwAAAAkxAAAAkBADAAAACTEAAACQEAMAAAAJ/QUx5Xks8SzuCQAAAABJRU5ErkJggg==) |
| `extended_kindlmann` | 123, 0 | ![EXTENDED_KINDLMANN](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAB/klEQVR4nO3WzZbTMAyG4U9O3QAb7v8y2QFtYxaWPceahha2ep+NRj/JOD1pj0xSk6SqiyTpMqLVJd+0SZKKx4+8981C7n2b+VqXPZ/T6M/6mA957M+8PM3H+TTyk/mP2M/T/HM41ONduyTpl/X4Q98kST/31i/bb/26+rvn9ea3XWMLucr9aWwzf/jn8DjJjzW3Y6m3kVsL/ZDHvucW++Vs/kW9xLl/i/bf17cQFZ5fJ3PPn+Pi9atfVz1eZt7/qIr1NR9vfQ35nNeb8y/q8b5z/mRu1GuzkK/96+h7/XqMeo/7YUt9H/1Hr+/+us5497q/9v510vXW61a+e/zaG/alx3L1WL3vJ9z8icr212j+PVcZ834/G/cb+fg/+5J/9PdwfbzuLK9hfpynhv7z+jy/jfNfwnXhcxh5mG+l/x4+fOzh7WMzj1pjWfNWzqItuUJ//gxbyE/qFudsvB/rvMX5N+ufzhH74f+env/Vfd7sf8pPnmMeZ0Zb8i18/8dtAQBAIiwAAAAkxAIAAEBCLAAAACTEAgAAQEIsAAAAJMQCAABAQiwAAAAkxAIAAEBCLAAAACTEAgAAQEIsAAAAJMQCAABAQiwAAAAkxAIAAEBCLAAAACTEAgAAQEIsAAAAJMQCAABAQiwAAAAkxAIAAEBCLAAAACTEAgAAQEJ/AIj2aSwKlTNZAAAAAElFTkSuQmCC) |
| `glasbey` | 176, 0 | ![GLASBEY](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFOUlEQVR4nO3W+5OXZRUA8M+iuCwFrCQxxUpm41DkZSshMAgN0gZQXAFBUmCR0KGAhcFcRQWxRUviprCIWlzCCxlGCi2YA4xAXGLGGAYhJiHGIESQ2TaZDYR+OP0X7/n88p15L8/7POc5z/mekn1cgHXfuQAeGFgCLly1DdQPvgHcP/2v4CczHwPPrjwMhi+K6692ugtY9bP4/bQRLKtdAjrdtAgcuFALut52E1hZ9SfQ4bXZYM7U/eCTGQ3gomE1cf3aGHZy8zhw6uY7QM221aD3uOdAVUWMv2FajP9m3YmY1rZ7wPN1Mc7504PAtU+tAS9trAfzrol5z9z/U3Dw9V7xu3Mr2Lox4rSpyybQ+cnvgYYBb8b3l/QHAzwRcbqsLZh/bm7Mt9dg0OeDDmDc2GbQ8cZycPKXk8DhGTHPHjUtQO3oHWDL8v9GXNbMBGtKzoDZ5S3BhUkLwN/Kvg6+dEXsZ839r4MRlVUx7+4vgObt98b3Jp4H7w2KALVbfRqsW3IEPNoYeXBvw2JQXf0iOHPseTB5w3KwY32s5+jESyJOY16JuFT+E+weehU4ceQ+8NT4avDq6JfAsIq4P+o/h0DZrBqw+M8Rj4Fl/wClXYeD4Yd+FO+N+kusv2kiGD059rFy6R5wnVvB/JoY/9NDP47r3ZvAE1P7xe/1T4Nd5ypBr4u+Cxrv6Azqroy8WtX2LNh3KuJ/pCn2rfrdyOPWTW+Db06J61veif0Z2foWcKhLxH1M40bQpTT2o+2dkRfdl0X+XNzjYjC3/+dAzyH7wOenRL727flJvPdRzHf3krfAg9NagW4vbADNe7eDL8+dCiYMLQWv9ekKnin/f77UjwfLL30DPDJhKeg05RqwrN9YMGNBrPuWGZEfqn4LHh85OZ5r+XI897XYj9XDI26Djse8F74f78/Z+g5o+XJvsPLyGKf1HyM/KltFXek2L+rR+Rt2gq+cioS4/V9x3p4eFfm3YvwAsKc64n62eQS456Et4OYVl0b8vrAi5j/75+CHj8W5/vjgh2Dj9lWgvin2veFbcT7Wb5oGhreJ+/vK24BeJfNjXgtqwN51kfczHow6sqoqvrfz7ciPc9MrwOaz3wZdK6JOVLy1Lsbf8w3Q3LM9uKs26lqLCdNj/INxXta2iXx5+OBJcFnrZ8GBsjjvj3aZEs83xH5evXYM6L0lzunyzfFcn35RVx++Pe4veP9XoO6+qB/HD98ISkdEvNpfEef5mf5xzua1i/pwrCzq/Wcm7AVD9h6NOLWKOrrZF8GsWZVg4JzYj/o7d4HLF98N3vtN1P/jkx8HHftG/Tx6IPZ3V2PU7drNp4H1Eb/rJ8U5qusWfxhvnD0Aqj4bcRs5+Grw5NAHQIsO3cHO338MKsZ+AN79dzuwNLbHyR/EOn+xMOrjK6ciH87vjvsf/mEYOLMj8n5mU9SNHu3jf2n3V8vi+tmoRw89EnlQvT7yveSSWH/H/beBNaWRv0evWwv+PmRUvL/oFDj9u8iH0vKPQOfFCyMed0feXFn3fXBrn1/HfI9FPp442Rc0nOkT65dSSimlwskGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAsoGIKWUUiqgbABSSimlAvofcV99Un2SM1EAAAAASUVORK5CYII=) |
| `glasbey_cool` | 177, 0 | ![GLASBEY_COOL](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFO0lEQVR4nO3W/3PXdR0A8AeexM0h7Xa3myDRlJsStyDajdCzjGB4CjS0HBiVdhipgO3KSM4OEvJsSGKikV+Q8xScxKSpLSOMrxm4M9gikcCciEDHEZwiNmHSD0//i/fz8cv79rnX+7Xn6/l8vV7vZ5/+t549C2dqvg7+V7ccXF23Fvxx2i6wdtJKsGPnUvD6/SNBw/q3wen6b4LWpU+Av7THuA3Lu8E924H+278Bxl9+C5i7Zh/omfefGLf6LvCz0jZw475zQdmX/wYap14f4257EvyhuREMrv4KmP9axPvWxg/BvqG7I/6TIyLejnXg9kk3R5wlF8d8G28FC6d/DeydMBWsHvtFcHbJ30Hnpz4GI0s6QNPimP/g4cHx3oElYMSoP8V6a/4Mxk6JdbQ9+yJo/lLk5+q77wbfHzEcfDxoTOSjbmDkofUA6D2yDIw5GPkb1jfyX3HlKnDfud3ghu5ecGTb0Fjngsjzq20RR3XDReDoziFgU0fUu3ZO/H7viXGgcUPUu3zYF0DdhP5gxj0Rb++3bot8bo46tN0U6/d4xD1v3EegprseTN/344hj0e/Alhe3gJf2nAYbZl8e77dti+emEnDTDz8H6ssrwLc/iPxdtijWc0lvNbhwYozvOf8asGRpvNfQbwH476EGsPXhTtAyIOZbu/E1cOUtkdetiw+DdzvXRL7GjQdzamM/bN/yTzCruhmsH3ovWHFz7ONn522O+BfXxnPgosjHxB3gxD/2x+8/OgIqDlwATra/Cu4/tgc09kQdL3hwNHhuQRUYW/oS+PeJ+H9/rZ8JvrO/EpTs3wh2j4x8ntn8AzDojk2g6kQ8H9gZcc767Cxw8NB80HTtvPi74z3wysOlYHlDxN/UEfme+cs4F+3PRx1HDBwF+lbGe02r4zzdWD831l0b+2PD4lYwuSzq031dF3hq7Z3gTrGupy9qB3ddGudi1aLYRx9WPQaaH7oi0rgg9n/TnKjTDbMfiHjW/AJce6om1v+vOLdVleeAn86OfbHzvnj+fFvcC0uPxv66qmsGWFP/FJiy4xHQWx71GVAd57+u4rtg1+lfgZbRA8C0t+O8n9xzEHStjnU/1L4LPNkSeVv1TsS1vOqr4N3jvwfH34l5Xq6M39/o3Av6vP8omFM3Fjy4Oeo+uu/roOGqOL/9PtMH3PFM3KMzP38SnJg7HQzfHfuvp+wSUNIV47pKzweTymL/rLw+5r+i9gzYU/1bcOzQhWDYY1H38k/2Z8exiPOcQb8BE8bVgbZHI7+XVsX98P6Y88Cb6/uBIROngKq3Pg32rXsDvDc88lNzLO6JFZN7Yj2741xUXnwcvPDy02BATxkY2Rxxn1oW4w+OmgQ+KIu8VKyIc1R5WdSpdlDcD6VDIq7rBh8DCx+PczL/aHxPWqbHvnvlk3WW741zOuKaYRHf9PgeLHxmGjhaGfu5vfUFcKg16rp+YOyf2qNxT/c0xvdpcl3fmH9lCxhfEvdwyeD4niyeEefgza64txqXxfxTJ8Q8H9XHvb513a/B4bbvgefGPA9uP+8noPNU1CdOQ0oppZQKJRuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQKKBuAlFJKqYCyAUgppZQK6P+EWH+ZpfQ5aAAAAABJRU5ErkJggg==) |
| `glasbey_dark` | 178, 0 | ![GLASBEY_DARK](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFN0lEQVR4nO3W+5OXZRUA8A+Lzg5jMqQy0EKb484G2EZeIKnFGEYwkssGFKSizDS1eSmLCQ2WIl00h7iJZGYiA66WrlwEcaUtaDY3CIiBxQ1C2nA20tnhEgOK7YTQD4f/4j2fX3b2+77f53Kec57v6XGAC9BUfQE8OKEHuFC5Hbzb/UXwiUNt4Lv188Hx6o3g5QG3g76LbgTHPjUZrHlwMRgw+peg58TfgRcnN8f7a5eCpbP/Du7/9AtgSccB0KN9fTy/LtbzrcfqwMrmx0C/li1g88+OgYXrPwa2LWqI7y99CHy0YARoGjgG1M4YC17Ych6UbN0MrqlpjffmxnyNUyIeEy0ANXN+Am5duQyMLDsMlpdPBNUHYt+bmh4H9Sv6gs++vwFcXb8OvLbxYnwH7gE7p88DA8e+AUZcCkzZ+wq449o/gg1/2A8WPxqfb9vZAb60Pc7p90/PBgenxHov3/A++GB1rLOh4W9g7pD4v1Gs4zett4IFNX3A/BMHI55d9WDfmRin/NgPQe9lj4JDkxaCh2bfBfY2/ArcNfMZsGvuCjB89RXg+LpG8PBtsb/3Tn4FVLTHvqfeNyri0ed60NxaAUoWzwVXfrwm3v9r7LPUN8FVD8R4vfveA8rWnwLVFRH30n7jwU9vjucd2y4BMz/cHfEb3RTPN3XFfOWRx9dPngQubYu/Xy6N/b507jrwp+5/gM43I68+2hTj9b/wPDh673/BN07Ee0emRfymHCwFN7bFQc87HHXw8MSopy/cEvvZ3RF51FnWCWY19ga93o7vPXn/3bHedTvBA1WRB7Xd70Y82vqD57bEeW0cFvn4/OU9wdoV28ChlZFHdy6vBPc+EuO8WDEHnG26L/Yx8BdgSe+I68iL90J71XPgRMtUUFdTDtY3XMyzM3E/zHh1GNjx+mVgz764F94YGfna1j4Y7G+KuhzVHOMPWxXjXfPziMv4kjift2t/BM7VTYvnN8XzM4Mjf/7c9QEo6xX1+rXhse6Wxqjzzqc+B6oOxD63tsS9NnZM1PGi2sivw6uWgzUHIn973hz7nbkr4t9n36CY+Hysp/uTb8U8p06BO6ui7pt3RJxLJsX6dlfF/2bEgQ+aPTP2ezTm+09l5Nu3l78GLpkX9VB1JO7VEd+ZADbc9g4Yej7uv3Fda8HOG6Kuy38d+d3eYxxY8Ejct7N2lMX8Q4eAcwt7gRua454evfkI+Nf/oi5fPh358fl34vOlg2Id01ZFXa6dHuM9PvxKcPeeONc5I0rAs90Rl2MrPwMui7Co3/oqeK9jNXhr80kwvzPu7xVT+4Gz+6fH/mpi/g/HnQVXtH4VdL7+FCjrH78LLaf7gL4ThoKSzohT56j4/Ti560nwvcpInCf+PQA0Tl0S6/pBnMOyc0dj3nsij2/aGPuu7fo+6DU17rnOV2KeQzWrQN32yKc3h0SePbM56rAirkd1bRGPCcfj3jh49p9gVtMd4Nq9PwZ7flsV62uNPKiujPz/emmcx+C/RNzHV7WDW/ZdHeOcXgNeOhX1fnvVcPB03/jdGVO1KOIipZRSSoWTDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgWUDUBKKaVUQNkApJRSSgX0fyszfZCmpNYLAAAAAElFTkSuQmCC) |
| `glasbey_hv` | 179, 0 | ![GLASBEY_HV](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFPUlEQVR4nO3W8Y/XdRkA8NeBdJKF2kkMAp0QwTE7ZKfUBEUCBkGgl4hM9EAqqBAmYAjdwQiJE4RrqUskDBmnEobsxC4mIAI3BwHmjDWQACW8JGe7VEbI4vrh4b/4PK9fvtv3+/28P8/neT/v5/OU9F//bhscfPlucGZyR7D8VBfQ4/NBYGXlCLBw08dg2K4HwXerHgd7dQKbTq4AN077G1g04gHw5tBF4MqdG8G4kqfBT3a8AWqfvAvMa6wAI5sq435jxoLTvdvAuqO/BVv2XwW+v/MdMP6PS8GQ3fG/mc0lse7Q50DFnCmgZRUw+/XloK3uE7Dvtk/Bt569Bpx4fyHo+3CsM/ntuL7X6HVgfp+z4K7N3wE1vfaDNacjjgmjbwEfbj4IWufOAw9+87/g4uIvgsc3DgM955eDFxp6RB62jwOnRvYFd1fWg0fvPQU6HbgDtJ96M5j4lxfA7u2xb+OPrI7nW/An8NjaO8G5ysjzkgmfgc/0Bw9VTgVrKy+tW1sFGsa3gB7Px34sf/JaMLZ3O1DZ/NOIf9kTYEVte/BBxZqIe/45sGnuu+Bs/QWwte5fYNHqWG/bvKiH8uPx+8qKGWDWlqOgurEDKJt+HJRWRJ637ou6Pd8p8rvxTORzeN+Ib/7EruCRG6J+rxt0Blw2rwaM6gN8+6pdoHZ/M3h0Q3zf8E5PsOD1E2DKgcGgbW3c94evbot4S2K9qRsPg9bXbgAHZ8Q+7xwYz9WlvhGc3vYYaK6Oev7zB5GvG2c8A4bdvx7Mbf0V6NbhqQjo3u/FdQOuB9fsirwu+0rU6+Rbq+M+b/4T7NvaDTzV68XIz2UR1+WDYp9rvhAJqGm4DVSVvwF+vX165G93rNOz6gi4/qa/gxH/eRa891Kc59E9/wfuuHg1qD70MhjXJ9bZ0Brh1731m4ijKfLUr33U6T3Ho0/sP3kAzDoWG7B/cb9YL9Kl3+CyiON09JlJVT8A/zg7B/y796X6+uVu0K5z3G/45Kij4yujb9Udij73VvlNoOTWUvDQj6L/lN43CRzpGOdi4dqooxU1sS+ffxp5a2z9Kvh9deT1nj2Rn3aTL4Kl98e+/XhSxLlsQHdwYcNWcN+clyL+TXHey2uuA81DrgCDOywGHXqNB79Y3RvMe+U8qP/ZR2BizXvg2qo4B4uuPAbGPBP3+8OWWjDhUh463h5xfjz7G6DsxffBmiXRb5cOjXoaWBJ5mlK/B/z1a61gztcjHycujzr8+c69YO6QmfH85waCLz08HDzdP87LjieiX55rjvfIhl0RV9mOqK8l6yeAxtL4HDVxAGj6MPr66aY7wclj0Y+mvRr9f1blZvDa+cj7Iy1N4ObnI7/bp8S5/qhr7NOwztEny+bEOhe674z/T43+3tQt8jT79riuYXise64t9v3Le0aCsevj/dC6KuI7vLkBbDwfeVrXMepl1Yp4D7QMi+eZ2SX25e3zdbHeKxHXrqFR94cORz9a9ECcs1s2xfth74JpoPuSUaD0uXgvXDwa/b97ZfTr302PPrO5pTPo+knk9+ox0T8rr4hzFl0ypZRSSoWSA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgWUA0BKKaVUQDkApJRSSgX0f4r7dqUkwFz3AAAAAElFTkSuQmCC) |
| `glasbey_light` | 180, 0 | ![GLASBEY_LIGHT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFM0lEQVR4nO3W/W9X5RUA8I8M1jEcaRBkkIVgmEqAiY0OlbUwQmDMDFEBqXOIIabAmMOhiDBMKmxixnTKLJMNRcSZRougI6USBWHMN9bxIikTRpCmcWO1unTQpgPpfjj+F/d8fvnmm3vv83Luec49FzXRDSNXA9uXdYOb694Ba6ePBfNfHB03nF4Plu9bB1a//DOwtfZq0FLyC1Dy5kNgg7ngmd+NACPO9AT7lzeBpvtjvLkffw/Ubq0CpRNngvJe20CfBT8A//1jD9D3UIzfvS/mm9Ujxh9//Cw4174ILPpSzLNp27fAJ/VDQUfHEjDqwUHgvV0/B5P2PwOGf14LHo9hzWjbC25o3QqOXHNxhGPEnNj/078GV5z+LOL15Etgz96ToL0mxp2+9ItAr3gF7P3arRGnMY+BuxtGxT7KVoGdXbvA5KqIt5HN4PvPHwerOr4Lrp0xGxz/bX/wjc5HY/6hS0Hpling+sdeBUsejXiUPhTjVI38ALyx4k8xXucE8Nm9j4O3TpSAgXP7gpaxQ0DPOZPBA+L6jgFHwKnFm8HBBTHeuuti3h4zYz0PTxwAblsUcT20JfLtaGk8P/1E79ifp8HaK64D9bMiTuNmPhzhmBMv6H+dkR+He3WB3rf+NP4Pqgfb1y8G9//mQfD3+T8C09pvAg1NT4HxnQMjDvNXgk+uivc5Yfu9Ea+bngCvdF8ExnQfA00/fDn22b8B/OXrMU/NxZ+CMzfG+lrWTwUVp98Hs9eOAcvuqgCVZRGXK6eUgtevLwen6saDzs3DwPmzsZ4/j4h82DYoft8pifPUd8YhcOK5A6B12VfAzuY4P5uHxfi1jbHOxZ8fBhf2Rd699fYtYPSaeM+bX/gJONI/8nPiJZeD+q6oC1WVN4N+X/4O+M+kf8Y+yv8KdtTHeVMT856qiffU3i/GWTr6kbh+yxqwe0PM//6Mj8HgBW+C/aULwZAFkZe9WzaAsncjz9re+yYYNufbcb1fNbjs03+D2zvuAuNGxfkecDCmvWrKLLC89j5QN64TdPeO9b1eEXl+5YE4v6v2DwWzp8c56XEwfndf+hGY+9omMLAq4v/2ylhPn7WN4PL77gAXzkadqzv5RV0ZFnm4pC3ifq407t+95GoweU/kcdOHZaDvmjjHg97YEuvv6gVefOEcuGF+nKuq2ta4f0Lk0+mGeWDomWvAA8MvgEeqngPle54AtZVRn348qw7sGnwJaGusibg9FXl8+NVYZ4dfggnVUa+a74j9tZ+M+nO0IurstEnx/NLmmHf2kDhnzV3Xgks33gMqqyN/lh9bAVpKPgLDJk8D7/aM8aetux2cb496/OzU+F8x4DYwpTXm21kd61u5ejBo6Iq6cqQr4tmnLerb8fM7wR+6473va4znSoZ/CH7fGPl0duGOiNvCv4GvvhZx+mBenNeS0W1gU1nkwT0HzoONY+O7MHZj1Jv29rj/7srIi+ZTkc/H5kU96fxXfOcGH43v20std4K+uyOvDo1qiX1etgDcWP0PUD71V+DO1jhvFTOjXg4ZWAmiGqSUUkqpULIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgLIBSCmllAooG4CUUkqpgP4P0O+AkxSlJR0AAAAASUVORK5CYII=) |
| `glasbey_warm` | 181, 0 | ![GLASBEY_WARM](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAFMklEQVR4nO3W/W9W5RkA4Eu0IOuy4LChFdJRCJE5BBy0wwKS4tyQqQQx6BQcQ0SJMxiGgLPsA11wm8bqxjRCFCI4hbm6IAuTmU4krAy7IukqG1Bqw5hZFRsnIBZwP9z7L859/dKc9+3znvM898e5z2vnM/jy/FPg5dr+oHnBfeDQtQ2gcebPwBPtwOIf/xvsGzAAbK39PVi5+k2wY/NE8Pmb5oMrDy6OhVvvAStGdIK55XeDhj9tBmsHPg82HHsGlJU3g46XusFHM14F1ZNXgs6lq8Cdb70W1+PLQEnTFWDw6kbw3Myt4HTHs2D+yjdA37b4/s+TW8GHk3aBg1WzQO3Zr4OLpsS++refH9vY3xe8U7oX/GDeo+Bf7VfHfk+vi/0Omg4+6bkdLD/8Orj7aB3Y9VotuMHR+Hdvg53bRoATO8eB97SAcb1LwOcmxXXLY3FebZ9NBqWf/gLcNPahuO/xOKdX3xwZ51kX8f1izVJQMyPu23LbP8Bls0aDo8sHgMol88CwmntjPydng+9veQQsvKUJvLgt4rusahpoOr0BTJ/7INjYENftuw6CBQ/H58N2LAeP9+sClw4dAj7tjvO65t5vgz33bQdTO4eB/bOWge6//weMP/wXsKliRux/+61gw8o14JbX47nWH4/nvbxsCqh9fChonXo/+Oami0DHt0pB8+y/xrlNrwebd0Qelb/3MRi3I+JburYNDNz0JbCuZD34dU3s75ObD4MLO28Dh7reAh9cMAoc+PlaMPGXcd20LfJrWtUfQZ/DEbdzPQ+DwX0iXj8qifwtPRX5UtY34ndq309j3Zyo0yHxR0XLILDlXOTfdcviXDtnNIC66yMfK5ZsA7sbou4GVV0GSobE7w8YsgIcWRPnXL6qBjQuij5RtqAEnF4adXn7VyKOW+ZEnU9ouyqe7xtxv6dbo35ueCF+5+VrYt1o1eDWiugvf/MA+Gr9cPDOP6Oe3vjh78Ds9yPOz5eNAVe0xz53n426XvpM1HXzgwtA28D/gksWTopzq38KzFwc+XqyZzfoGRvx2/j0WTB41AnwndXfBd03Rr86NC3q7+NjsX5A55m437A+YGTbDlB6+UbQ/+wfwJnjURftl54EA++JPBxUH5+PGXsHuHBMnKsR0TdfWnQtuPlUAzhR+iJoHdkDmhrfBcNLy0HJmIvBhMo45w9GRf/a/mzk1YrqqeB7T0Q+LZ99JVjSGvW+ti76Z8n7FXG/ip+A3jXRpw9UDQV7WiOvTIq8+UJX5N3ounngqtPrwc4TEa8RXZGg5+6IPF/1wup4znejv5+b2AEOvh3rHjk/1nnyt/H9gdjvQ83RP6b1nAc6y34V9+vsB9om3An6nXwM7DsSeXb1+I9A48J4b9S3RN945f4bweDer4Hq6ZH3HdVPgg87Ix+2NO7//77iHLp6oo4vqYz3R/fFd4HhXgFHTkTcK6dEvPcujPyeK96PexfF+7j1qci7uuHRd3rPRF2P+s0x8FxtxK/yruhHc9bF+6LjgXmxvjf+v/aCqLNTeyKuPWXR1yIrU0oppVQoOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRAOQCklFJKBZQDQEoppVRA/wONwnvPZDrtlwAAAABJRU5ErkJggg==) |
| `accent` | 240, 0 | ![ACCENT](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAvUlEQVR4nO3WIQECUQAFQTiLA0EW1K+AQRCCDNeCi4BHUQF1KcgBLXjizSRYudv5PX83xcbnkk6IGvtnOiFqGod0QtTpdkwnRL3uazohar0u6YSox+6cToia0gEAwP8ZAAAoZAAAoJABAIBCBgAAChkAAChkAACgkAEAgEIGAAAKGQAAKGQAAKCQAQCAQgYAAAoZAAAoZAAAoJABAIBCBgAAChkAAChkAACgkAEAgEIGAAAKGQAAKGQAAKDQD2uLC2FKNMt4AAAAAElFTkSuQmCC) |
| `dark2` | 240, 32 | ![DARK2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAwElEQVR4nO3WsREBAQBEUcwoQHwVaUciMieXCo0ehBJzYgVcAwrQAV3YYN+r4Id/OVz330Wxx/OUToi6bG7phKjdfU4nRI2Hczoh6rh+pxOipteQToiaPtt0QtQqHQAA/J8BAIBCBgAAChkAAChkAACgkAEAgEIGAAAKGQAAKGQAAKCQAQCAQgYAAAoZAAAoZAAAoJABAIBCBgAAChkAAChkAACgkAEAgEIGAAAKGQAAKGQAAKCQAQCAQgYAAAr9AIqXDmFcuCGjAAAAAElFTkSuQmCC) |
| `paired` | 240, 64 | ![PAIRED](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfgAAAAYCAYAAAAWG+9JAAAA5UlEQVR4nO3VsWmCARSFUeMYruEo6bKAlQPYJaRMEUTsRdAiZSZQSREQhDRZwCYzpDErvCLw+C/nTHCby3e3v1xvowFYvH11TyhZzb67J5Q8HbfdE0o+f+fdE0p+Hp+7J5RMdvfdE0rGh2X3hJLzdBh/37x8dE8oWZ8GkaPR+8Nr94SScfcAAOD/CTwABBJ4AAgk8AAQSOABIJDAA0AggQeAQAIPAIEEHgACCTwABBJ4AAgk8AAQSOABIJDAA0AggQeAQAIPAIEEHgACCTwABBJ4AAgk8AAQSOABIJDAA0AggQeAQH/4Exdh3Ns6eQAAAABJRU5ErkJggg==) |
| `pastel1` | 240, 96 | ![PASTEL1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfgAAAAYCAYAAAAWG+9JAAAAwklEQVR4nO3VoRIBARSGUbtBFCRbkLZKXlsSZG+hGUEXJIGXcOfO/HPOE3ztGz7n03cR7DIduxNKTdtHd0Kp9X3XnVBqv7p2J5Qa5rk7odTz9upOKLUZD90Jpd6bZXdCqbE7AAD4P4MHgEAGDwCBDB4AAhk8AAQyeAAIZPAAEMjgASCQwQNAIIMHgEAGDwCBDB4AAhk8AAQyeAAIZPAAEMjgASCQwQNAIIMHgEAGDwCBDB4AAhk8AAQyeAAIZPAAEOgHpjoMBcduiaQAAAAASUVORK5CYII=) |
| `pastel2` | 240, 128 | ![PASTEL2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAvElEQVR4nO3WsQ0BAQCF4XOVFTR6scJNYF1zsIBCItG4Ri7REbbwivd9E7zX/avj7fQdih3ma3pC1Hk9pSdE7V7v9ISoZX9PT4jafLr/L89tekLU5ZFekDWmBwAA/ycAAKCQAACAQgIAAAoJAAAoJAAAoJAAAIBCAgAACgkAACgkAACgkAAAgEICAAAKCQAAKCQAAKCQAACAQgIAAAoJAAAoJAAAoJAAAIBCAgAACgkAACgkAACgkAAAgEI/R1IQk6YxYmsAAAAASUVORK5CYII=) |
| `set1` | 240, 160 | ![SET1](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfgAAAAYCAYAAAAWG+9JAAAAx0lEQVR4nO3VsQ2BARRGUQwhIjZQ/TGGShSi0EmsYQaNRiLR2EltBTVLeHnJl3MmuN0dv+eL7yjY9njrTig1HS7dCaXW9313QqnTcOhOqHVedReUeu4+3QmlNstrd0Kpx+zVnVBq0h0AAPyfwQNAIIMHgEAGDwCBDB4AAhk8AAQyeAAIZPAAEMjgASCQwQNAIIMHgEAGDwCBDB4AAhk8AAQyeAAIZPAAEMjgASCQwQNAIIMHgEAGDwCBDB4AAhk8AAQyeAAI9APiBwv6iHBdQAAAAABJRU5ErkJggg==) |
| `set2` | 240, 192 | ![SET2](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAv0lEQVR4nO3WsQ1BAQBFUd8KBrDR7+lJjPObv4ExRKKVWECvoNDq2MIr3jkT3PIO+8vxuyg2X0/phKhpdUgnRG2fn3RC1Hmc0glRm+GeToh6vHbphKjbe51OiFqmAwCA/zMAAFDIAABAIQMAAIUMAAAUMgAAUMgAAEAhAwAAhQwAABQyAABQyAAAQCEDAACFDAAAFDIAAFDIAABAIQMAAIUMAAAUMgAAUMgAAEAhAwAAhQwAABQyAABQyAAAQKEf/3YPSAPiC4UAAAAASUVORK5CYII=) |
| `set3` | 240, 224 | ![SET3](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfgAAAAYCAYAAAAWG+9JAAAA5UlEQVR4nO3VIU4CABiGYYQAgeAt2ByFkTkAxyDbzBqNzsoN3DyAhUCTA5hJGNksdL3CH9j++e15TvC29+bl6/A7+Ace5ufuhJL9btadULL63HYnlDwvNt0JJU+j1+6Eko+7cXdCyfrnsTuh5Di9dCeUnN6+uxNKbu8n3Qkly+F7d0LJsDsAALg+gweAQAYPAIEMHgACGTwABDJ4AAhk8AAQyOABIJDBA0AggweAQAYPAIEMHgACGTwABDJ4AAhk8AAQyOABIJDBA0AggweAQAYPAIEMHgACGTwABDJ4AAhk8AAQ6A93NxKKHKW71QAAAABJRU5ErkJggg==) |
| `tab10` | 241, 0 | ![TAB10](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAf4AAAAYCAYAAAAbBZ8OAAAAz0lEQVR4nO3Vsa0BABSGUVSmeGIAI5C3hYRBjKDT6XQ6iU7LLDbQKiQsceMm/nMm+LpvONlc3oMfcR+vuhPKzKZ/3QllTttXd0KZ2/++O6HM87HrTiiznG66E8ocxtfuhDLzxbE7ocx6eO5OKDPqDgAAvsf4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAQxfgAIYvwAEMT4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAQxfgAIYvwAEMT4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAT5AHRhDtMxDx/yAAAAAElFTkSuQmCC) |
| `tab20` | 241, 32 | ![TAB20](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAABAElEQVR4nO3VIS7FAQDH8ccJBDPBvMYENpsgvHmbYP/XyDYHwGwKR6BJqqhpL7iBSQ4gmiNQOcWv/Pb5XOAbvwvj2/nfKOzheC+dGJ28bsQbo8l5PLH9PY83Lqdn8cbB8BRvbF1fxRuPH5vxxu5sNd54e76JN/aPZvHG+HMn3lieLsUb9+8v8cYwDPHGz+9pvDFev4g3Jl+H8cbd2kq8sRgvAABxhg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGggKEDQAFDB4AChg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGggKEDQAFDB4AChg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGgwD8K0RiMxqk33AAAAABJRU5ErkJggg==) |
| `tab20b` | 241, 64 | ![TAB20B](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAAA80lEQVR4nO3VoU0EURiF0QGFgLDjGLIG+SaQMAJBcLRBFUjaICCRqO0BSwkIioEqrrk5p4GbvJc/39H9w8vfFLYsa3piWi6v4xvb7Xl84/v3Kb4xxpLfuMlvLKdbfOPr9T2+sW0n8Y11n/+PaX6MT3w8f8Y3xtmc37i4im/Md/n7eDv8xDfWXf6txn7EN3Zb/gaP4wsAQJygA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACgg6ABQQNABoICgA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACgg6ABQQNABoICgA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACjwD92lD4EUAutZAAAAAElFTkSuQmCC) |
| `tab20c` | 241, 96 | ![TAB20C](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAAA8ElEQVR4nO3VvUlDAQCF0Wgr+ANKEJ7BRPQpKEGeYLB1KbFMl2Us7J3CSZxAp7gWl3MW+Mpv73H79TsLe7k/Tydm03AQb9yeHMYbl++LeGO+Wccbs+U/NFbP8cTT92e88bpcxRsPZ9fxxs1xvvGx/Yk37tbzeGNxdZpvXBzFG7vdW7wxTVO8MY5jvDEMQ7yxHy8AAHGGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaCAoQNAAUMHgAKGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaCAoQNAAUMHgAKGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaDAH3AeDrrA4hO7AAAAAElFTkSuQmCC) |
| `category10_10` | 241, 128 | ![CATEGORY10_10](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAf4AAAAYCAYAAAAbBZ8OAAAAz0lEQVR4nO3Vsa0BABSGUVSmeGIAI5C3hYRBjKDT6XQ6iU7LLDbQKiQsceMm/nMm+LpvONlc3oMfcR+vuhPKzKZ/3QllTttXd0KZ2/++O6HM87HrTiiznG66E8ocxtfuhDLzxbE7ocx6eO5OKDPqDgAAvsf4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAQxfgAIYvwAEMT4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAQxfgAIYvwAEMT4ASCI8QNAEOMHgCDGDwBBjB8Aghg/AAT5AHRhDtMxDx/yAAAAAElFTkSuQmCC) |
| `category20_20` | 241, 160 | ![CATEGORY20_20](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAABAElEQVR4nO3VIS7FAQDH8ccJBDPBvMYENpsgvHmbYP/XyDYHwGwKR6BJqqhpL7iBSQ4gmiNQOcWv/Pb5XOAbvwvj2/nfKOzheC+dGJ28bsQbo8l5PLH9PY83Lqdn8cbB8BRvbF1fxRuPH5vxxu5sNd54e76JN/aPZvHG+HMn3lieLsUb9+8v8cYwDPHGz+9pvDFev4g3Jl+H8cbd2kq8sRgvAABxhg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGggKEDQAFDB4AChg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGggKEDQAFDB4AChg4ABQwdAAoYOgAUMHQAKGDoAFDA0AGgwD8K0RiMxqk33AAAAABJRU5ErkJggg==) |
| `category20b_20` | 241, 192 | ![CATEGORY20B_20](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAAA80lEQVR4nO3VoU0EURiF0QGFgLDjGLIG+SaQMAJBcLRBFUjaICCRqO0BSwkIioEqrrk5p4GbvJc/39H9w8vfFLYsa3piWi6v4xvb7Xl84/v3Kb4xxpLfuMlvLKdbfOPr9T2+sW0n8Y11n/+PaX6MT3w8f8Y3xtmc37i4im/Md/n7eDv8xDfWXf6txn7EN3Zb/gaP4wsAQJygA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACgg6ABQQNABoICgA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACgg6ABQQNABoICgA0ABQQeAAoIOAAUEHQAKCDoAFBB0ACjwD92lD4EUAutZAAAAAElFTkSuQmCC) |
| `category20c_20` | 241, 224 | ![CATEGORY20C_20](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAfQAAAAYCAYAAAAMJw/HAAAA8ElEQVR4nO3VvUlDAQCF0Wgr+ANKEJ7BRPQpKEGeYLB1KbFMl2Us7J3CSZxAp7gWl3MW+Mpv73H79TsLe7k/Tydm03AQb9yeHMYbl++LeGO+Wccbs+U/NFbP8cTT92e88bpcxRsPZ9fxxs1xvvGx/Yk37tbzeGNxdZpvXBzFG7vdW7wxTVO8MY5jvDEMQ7yxHy8AAHGGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaCAoQNAAUMHgAKGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaCAoQNAAUMHgAKGDgAFDB0AChg6ABQwdAAoYOgAUMDQAaDAH3AeDrrA4hO7AAAAAElFTkSuQmCC) |
| `colorblind8` | 242, 0 | ![COLORBLIND8](data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAgAAAAAYCAYAAACMYkUlAAAAwUlEQVR4nO3WsQ1BAQBF0U+p+hWFGRSSP4FEq5RYQ6FVGkWiVBrICkq28Ip3zgS3vLPh/PwOxd6bQzoharHfpROixtc2nRB1XF7SCVHXxyqdEPVZ39MJUdPtlE6ImqcDAID/MwAAUMgAAEAhAwAAhQwAABQyAABQyAAAQCEDAACFDAAAFDIAAFDIAABAIQMAAIUMAAAUMgAAUMgAAEAhAwAAhQwAABQyAABQyAAAQCEDAACFDAAAFDIAAFDIAABAoR8t8govDegaBAAAAABJRU5ErkJggg==) |

