# Workflow Core 3.4.0

## Execute Workflow Middleware

These middleware get run after each workflow execution and can be used to perform additional actions or build metrics/statistics for all workflows in your app.

The following example illustrates how you can use a execute workflow middleware to build [prometheus](https://prometheus.io/) metrics.

Note that you use `WorkflowMiddlewarePhase.ExecuteWorkflow` to specify that it runs after each workflow execution.

**Important:** You should call `next` as part of the workflow middleware to ensure that the next workflow in the chain runs.

```cs
public class MetricsMiddleware : IWorkflowMiddleware
{
    private readonly ConcurrentHashSet<string>() _suspendedWorkflows =
        new ConcurrentHashSet<string>();

    private readonly Counter _completed;
    private readonly Counter _suspended;

    public MetricsMiddleware()
    {
        _completed = Prometheus.Metrics.CreateCounter(
            "workflow_completed", "Workflow completed");

        _suspended = Prometheus.Metrics.CreateCounter(
            "workflow_suspended", "Workflow suspended");
    }

    public WorkflowMiddlewarePhase Phase =>
        WorkflowMiddlewarePhase.ExecuteWorkflow;

    public Task HandleAsync(
        WorkflowInstance workflow,
        WorkflowDelegate next)
    {
        switch (workflow.Status)
        {
            case WorkflowStatus.Complete:
                TryDecrementSuspended(workflow);
                _completed.Inc();
                break;
            case WorkflowStatus.Suspended:
                _suspendedWorkflows.Add(workflow.Id);
                _suspended.Inc();
                break;
            default:
                TryDecrementSuspended(workflow);
                break;
        }

        return next();
    }

    private void TryDecrementSuspended(WorkflowInstance workflow)
    {
        if (_suspendedWorkflows.TryRemove(workflow.Id))
        {
            _suspended.Dec();
        }
    }
}
```