# Workflow Core 3.3.0

# Workflow Middleware

Workflows can be extended with Middleware that run before/after workflows start/complete as well as around workflow steps to provide flexibility in implementing cross-cutting concerns such as [log correlation](https://www.frakkingsweet.com/net-core-log-correlation-easy-access-to-headers/), [retries](https://docs.microsoft.com/en-us/dotnet/architecture/microservices/implement-resilient-applications/implement-http-call-retries-exponential-backoff-polly), and other use-cases.

This is done by implementing and registering `IWorkflowMiddleware` for workflows or `IWorkflowStepMiddleware` for steps.

## Step Middleware

Step middleware lets you run additional code around the execution of a given step and alter its behavior. Implementing a step middleware should look familiar to anyone familiar with [ASP.NET Core's middleware pipeline](https://docs.microsoft.com/en-us/aspnet/core/fundamentals/middleware/?view=aspnetcore-3.1) or [`HttpClient`'s `DelegatingHandler` middleware](https://docs.microsoft.com/en-us/aspnet/core/fundamentals/http-requests?view=aspnetcore-3.1#outgoing-request-middleware).

### Usage

First, create your own middleware class that implements `IWorkflowStepMiddleware`. Here's an example of a middleware that adds workflow ID and step ID to the log correlation context of every workflow step in your app.

**Important:** You must make sure to call `next()` as part of your middleware. If you do not do this, your step will never run.

```cs
public class LogCorrelationStepMiddleware : IWorkflowStepMiddleware
{
    private readonly ILogger<LogCorrelationStepMiddleware> _log;

    public LogCorrelationStepMiddleware(
        ILogger<LogCorrelationStepMiddleware> log)
    {
        _log = log;
    }

    public async Task<ExecutionResult> HandleAsync(
        IStepExecutionContext context,
        IStepBody body,
        WorkflowStepDelegate next)
    {
        var workflowId = context.Workflow.Id;
        var stepId = context.Step.Id;

        // Uses log scope to add a few attributes to the scope
        using (_log.BeginScope("{@WorkflowId}", workflowId))
        using (_log.BeginScope("{@StepId}", stepId))
        {
            // Calling next ensures step gets executed
            return await next();
        }
    }
}
```

Here's another example of a middleware that uses the [Polly](https://github.com/App-vNext/Polly) dotnet resiliency library to implement retries on workflow steps based off a custom retry policy.

```cs
public class PollyRetryStepMiddleware : IWorkflowStepMiddleware
{
    private const string StepContextKey = "WorkflowStepContext";
    private const int MaxRetries = 3;
    private readonly ILogger<PollyRetryStepMiddleware> _log;

    public PollyRetryMiddleware(ILogger<PollyRetryStepMiddleware> log)
    {
        _log = log;
    }

    // Consult Polly's docs for more information on how to build
    // retry policies:
    // https://github.com/App-vNext/Polly
    public IAsyncPolicy<ExecutionResult> GetRetryPolicy() =>
        Policy<ExecutionResult>
            .Handle<TimeoutException>()
            .RetryAsync(
                MaxRetries,
                (result, retryCount, context) =>
                    UpdateRetryCount(
                        result.Exception,
                        retryCount,
                        context[StepContextKey] as IStepExecutionContext)
            );

    public async Task<ExecutionResult> HandleAsync(
        IStepExecutionContext context,
        IStepBody body,
        WorkflowStepDelegate next
    )
    {
        return await GetRetryPolicy().ExecuteAsync(
            ctx => next(),
            // The step execution context gets passed down so that
            // the step is accessible within the retry policy
            new Dictionary<string, object>
            {
                { StepContextKey, context }
            });
    }

    private Task UpdateRetryCount(
        Exception exception,
        int retryCount,
        IStepExecutionContext stepContext)
    {
        var stepInstance = stepContext.ExecutionPointer;
        stepInstance.RetryCount = retryCount;
        return Task.CompletedTask;
    }
}
```

## Pre/Post Workflow Middleware

Workflow middleware run either before a workflow starts or after a workflow completes and can be used to hook into the workflow lifecycle or alter the workflow itself before it is started.

### Pre Workflow Middleware

These middleware get run before the workflow is started and can potentially alter properties on the `WorkflowInstance`.

The following example illustrates setting the `Description` property on the `WorkflowInstance` using a middleware that interprets the data on the passed workflow. This is useful in cases where you want the description of the workflow to be derived from the data passed to the workflow.

Note that you use `WorkflowMiddlewarePhase.PreWorkflow` to specify that it runs before the workflow starts.

**Important:** You should call `next` as part of the workflow middleware to ensure that the next workflow in the chain runs.

```cs
// AddDescriptionWorkflowMiddleware.cs
public class AddDescriptionWorkflowMiddleware : IWorkflowMiddleware
{
    public WorkflowMiddlewarePhase Phase =>
        WorkflowMiddlewarePhase.PreWorkflow;

    public Task HandleAsync(
        WorkflowInstance workflow,
        WorkflowDelegate next
    )
    {
        if (workflow.Data is IDescriptiveWorkflowParams descriptiveParams)
        {
            workflow.Description = descriptiveParams.Description;
        }

        return next();
    }
}

// IDescriptiveWorkflowParams.cs
public interface IDescriptiveWorkflowParams
{
    string Description { get; }
}

// MyWorkflowParams.cs
public MyWorkflowParams : IDescriptiveWorkflowParams
{
    public string Description => $"Run task '{TaskName}'";

    public string TaskName { get; set; }
}
```

### Exception Handling in Pre Workflow Middleware

Pre workflow middleware exception handling gets treated differently from post workflow middleware. Since the middleware runs before the workflow starts, any exceptions thrown within a pre workflow middleware will bubble up to the `StartWorkflow` method and it is up to the caller of `StartWorkflow` to handle the exception and act accordingly.

```cs
public async Task MyMethodThatStartsAWorkflow()
{
    try
    {
        await host.StartWorkflow("HelloWorld", 1, null);
    }
    catch(Exception ex)
    {
        // Handle the exception appropriately
    }
}
```

### Post Workflow Middleware

These middleware get run after the workflow has completed and can be used to perform additional actions for all workflows in your app.

The following example illustrates how you can use a post workflow middleware to print a summary of the workflow to console.

Note that you use `WorkflowMiddlewarePhase.PostWorkflow` to specify that it runs after the workflow completes.

**Important:** You should call `next` as part of the workflow middleware to ensure that the next workflow in the chain runs.

```cs
public class PrintWorkflowSummaryMiddleware : IWorkflowMiddleware
{
    private readonly ILogger<PrintWorkflowSummaryMiddleware> _log;

    public PrintWorkflowSummaryMiddleware(
        ILogger<PrintWorkflowSummaryMiddleware> log
    )
    {
        _log = log;
    }

    public WorkflowMiddlewarePhase Phase =>
        WorkflowMiddlewarePhase.PostWorkflow;

    public Task HandleAsync(
        WorkflowInstance workflow,
        WorkflowDelegate next
    )
    {
        if (!workflow.CompleteTime.HasValue)
        {
            return next();
        }

        var duration = workflow.CompleteTime.Value - workflow.CreateTime;
        _log.LogInformation($@"Workflow {workflow.Description} completed in {duration:g}");

        foreach (var step in workflow.ExecutionPointers)
        {
            var stepName = step.StepName;
            var stepDuration = (step.EndTime - step.StartTime) ?? TimeSpan.Zero;
            _log.LogInformation($"  - Step {stepName} completed in {stepDuration:g}");
        }

        return next();
    }
}
```

### Exception Handling in Post Workflow Middleware

Post workflow middleware exception handling gets treated differently from pre workflow middleware. At the time that the workflow completes, your workflow has ran already so an uncaught exception would be difficult to act on.

By default, if a workflow middleware throws an exception, it will be logged and the workflow will complete as normal. This behavior can be changed, however.

To override the default post workflow error handling for all workflows in your app, just register a new `IWorkflowMiddlewareErrorHandler` in the dependency injection framework with your custom behavior as follows.

```cs
// CustomMiddlewareErrorHandler.cs
public class CustomHandler : IWorkflowMiddlewareErrorHandler
{
    public Task HandleAsync(Exception ex)
    {
        // Handle your error asynchronously
    }
}

// Startup.cs
public void ConfigureServices(IServiceCollection services)
{
    // Other workflow configuration
    services.AddWorkflow();

    // Should go after .AddWorkflow()
    services.AddTransient<IWorkflowMiddlewareErrorHandler, CustomHandler>();
}
```

## Registering Middleware

In order for middleware to take effect, they must be registered with the built-in dependency injection framework using the convenience helpers.

**Note:** Middleware will be run in the order that they are registered with middleware that are registered earlier running earlier in the chain and finishing later in the chain. For pre/post workflow middleware, all pre middleware will be run before a workflow starts and all post middleware will be run after a workflow completes.

```cs
public class Startup
{
    public void ConfigureServices(IServiceCollection services)
    {
        ...

        // Add workflow middleware
        services.AddWorkflowMiddleware<AddDescriptionWorkflowMiddleware>();
        services.AddWorkflowMiddleware<PrintWorkflowSummaryMiddleware>();

        // Add step middleware
        services.AddWorkflowStepMiddleware<LogCorrelationStepMiddleware>();
        services.AddWorkflowStepMiddleware<PollyRetryMiddleware>();

        ...
    }
}
```

## More Information

See the [Workflow Middleware](https://github.com/danielgerlag/workflow-core/tree/master/src/samples/WorkflowCore.Sample19) sample for full examples of workflow middleware in action.


Many thanks to Danil Flores @dflor003 for this contribution!